package com.fr.scheduler.job;

import com.fr.log.FineLoggerFactory;
import com.fr.scheduler.cluster.FineQuartzClusterUtils;
import com.fr.third.v2.org.quartz.Job;
import com.fr.third.v2.org.quartz.JobExecutionContext;
import com.fr.third.v2.org.quartz.JobExecutionException;

import java.util.concurrent.locks.Lock;

/**
 * 分布式任务类.
 *
 * <p>
 * 集群模式下任务可能被多个节点取到，将自动使用{@link com.fr.scheduler.cluster.FineSchedulerClusterProvider#getLock(String)}来保证任务执行的唯一性.
 * </p>
 *
 * @author Cloud.Liu
 * created on 2020-09-09
 * @IncludeIntoJavadoc
 */
public abstract class FineJob implements Job {

    /**
     * 分布式任务.
     *
     * @param context JobExecutionContext
     * @throws JobExecutionException 任务执行失败
     */
    @Override
    public void execute(JobExecutionContext context) throws JobExecutionException {

        FineLoggerFactory.getLogger().info("FineJob started.");
        Lock lock = FineQuartzClusterUtils.getInstance().getClusterProvider().getLock(getTaskLockName(context));
        FineLoggerFactory.getLogger().info("FineJob got lock.");

        if (lock.tryLock()) {
            FineLoggerFactory.getLogger().info("FineJob tryLock() successfully.");
            try {
                run(context);
            } catch (Exception e) {
                FineLoggerFactory.getLogger().error("Error running FineJob.", e);
            } finally {
                FineLoggerFactory.getLogger().info("FineJob about to unlock.");
                lock.unlock();
                FineLoggerFactory.getLogger().info("FineJob finished.");
            }
        } else {
            FineLoggerFactory.getLogger().info("FineJob tryLock() failed.");
        }
    }

    /**
     * 任务业务逻辑.
     *
     * <p>
     * 需覆写.<br>
     * 使用{@link JobExecutionContext#getMergedJobDataMap()}获取任务携带的JobDataMap.
     * </p>
     *
     * @param context JobExecutionContext
     * @throws Exception exception
     */
    public abstract void run(JobExecutionContext context) throws Exception;

    /**
     * 获取任务锁名称.
     *
     * <p>
     * 默认为JobKey的名称.<br>
     * 该方法可能会被覆写.
     * </p>
     *
     * @param context {@link JobExecutionContext}
     * @return 任务锁名称
     */
    public String getTaskLockName(JobExecutionContext context) {
        return context.getJobDetail().getKey().getName();
    }
}
