package com.fr.scheduler.job;

import com.fr.log.FineLoggerFactory;
import com.fr.scheduler.cluster.FineQuartzClusterUtils;
import com.fr.scheduler.cluster.FineSchedulerClusterConstants;
import com.fr.scheduler.enums.SimpleMisfireInstType;
import com.fr.scheduler.tool.FineScheduler;
import com.fr.scheduler.tool.FineTriggerBuilder;
import com.fr.stable.StringUtils;
import com.fr.third.v2.org.quartz.JobDetail;
import com.fr.third.v2.org.quartz.JobExecutionContext;
import com.fr.third.v2.org.quartz.JobKey;
import com.fr.third.v2.org.quartz.Trigger;

import java.util.Arrays;
import java.util.Date;

/**
 * 用于实现多节点执行的分发任务类.
 *
 * @author Cloud.Liu
 * created on 2020-09-17
 */
public final class FineDispatchJob extends FineJob {

    @SuppressWarnings("unchecked")
    @Override
    public void run(JobExecutionContext context) throws Exception {

        String nodeToDispatch = context.getMergedJobDataMap().getString(FineSchedulerClusterConstants.NODE_TO_DISPATCH);
        String[] nodes;

        // 判断是否是分发给所有节点的任务
        if (StringUtils.equals(FineSchedulerClusterConstants.DISPATCH_TO_ALL_NODES, nodeToDispatch)) {
            // 分发给所有节点：运行时再决定要在哪些节点运行
            nodes = FineQuartzClusterUtils.getInstance().getClusterProvider().allNodeIds();
        } else {
            nodes = nodeToDispatch.split(FineSchedulerClusterConstants.SEPARATOR_NODE_ID);
        }

        Class<? extends FineJob> nodeJobClass = (Class<? extends FineJob>) context.getMergedJobDataMap().get(FineSchedulerClusterConstants.NODE_JOB_CLASS);

        JobDetail parentJobDetail = context.getJobDetail();
        String jobName = parentJobDetail.getKey().getName();

        FineLoggerFactory.getLogger().info("Dispatching job " + jobName + " to nodes: " + Arrays.toString(nodes));
        // 移除dispatch信息，恢复真实JobClass
        // 防止套娃
        parentJobDetail.getJobDataMap().remove(FineSchedulerClusterConstants.NODE_TO_DISPATCH);
        parentJobDetail.getJobDataMap().remove(FineSchedulerClusterConstants.NODE_JOB_CLASS);

        // 对每一个节点创建once、startNow、逾期不候的Job
        for (String node : nodes) {
            JobDetail childJobDetail = cloneJobDetailForNode(parentJobDetail, nodeJobClass, node);
            Trigger immediateTrigger = createImmediateTriggerForNode(parentJobDetail.getKey(), node);
            FineScheduler.getInstance().addJobRunOnSpecificNode(childJobDetail, immediateTrigger, node);
            FineLoggerFactory.getLogger().info("Job " + jobName + " dispatched for node " + node);
        }

        FineLoggerFactory.getLogger().info("Job " + jobName + " dispatched");
    }

    /**
     * 从分发Job的JobKey获取父任务JobKey.
     *
     * @param jobKey 分发JobKey
     * @return 父任务JobKey
     */
    public static JobKey retrieveRealJobKey(JobKey jobKey) {
        String name = jobKey.getName().substring(FineSchedulerClusterConstants.NODE_JOB_PREFIX.length(), jobKey.getName().indexOf(FineSchedulerClusterConstants.SEPARATOR_NODE_ID));
        String group = jobKey.getGroup().substring(FineSchedulerClusterConstants.NODE_JOB_PREFIX.length(), jobKey.getGroup().indexOf(FineSchedulerClusterConstants.SEPARATOR_NODE_ID));
        return new JobKey(name, group);
    }


    /**
     * 创建子节点的JobDetail.
     *
     * <p>
     * jobName被改为NODE_JOB_PREFIX + 原JobName + "_&_" + 节点ID
     * jobGroup被改为NODE_JOB_PREFIX + 原JobGroup + "_&_" + 节点ID
     * </p>
     *
     * @param jobDetail JobDetail
     * @param nodeId 子节点ID
     * @return 子节点的JobDetail
     */
    private static JobDetail cloneJobDetailForNode(JobDetail jobDetail, Class<? extends FineJob> nodeJobClass, String nodeId) {

        String name = FineSchedulerClusterConstants.NODE_JOB_PREFIX + jobDetail.getKey().getName() + FineSchedulerClusterConstants.SEPARATOR_NODE_ID + nodeId + FineSchedulerClusterConstants.SEPARATOR_NODE_ID + new Date().getTime();
        String group = FineSchedulerClusterConstants.NODE_JOB_PREFIX + jobDetail.getKey().getGroup() + FineSchedulerClusterConstants.SEPARATOR_NODE_ID + nodeId + FineSchedulerClusterConstants.SEPARATOR_NODE_ID + new Date().getTime();

        return jobDetail.getJobBuilder().withIdentity(name, group).ofType(nodeJobClass).build();
    }

    /**
     * 创建子节点的立即执行Trigger.
     *
     * <p>
     * triggerName被改为NODE_TRIGGER_PREFIX + 原TriggerName + "_&_" + 节点ID
     * triggerGroup被改为NODE_TRIGGER_PREFIX + 原TriggerGroup + "_&_" + 节点ID
     * </p>
     *
     * @param jobKey 父任务jobKey
     * @param nodeId 子节点ID
     * @return 子节点的JobDetail
     */
    private static Trigger createImmediateTriggerForNode(JobKey jobKey, String nodeId) {

        String name = FineSchedulerClusterConstants.NODE_TRIGGER_PREFIX + jobKey.getName() + FineSchedulerClusterConstants.SEPARATOR_NODE_ID + nodeId + FineSchedulerClusterConstants.SEPARATOR_NODE_ID + new Date().getTime();
        String group = FineSchedulerClusterConstants.NODE_TRIGGER_PREFIX + jobKey.getGroup() + FineSchedulerClusterConstants.SEPARATOR_NODE_ID + nodeId + FineSchedulerClusterConstants.SEPARATOR_NODE_ID + new Date().getTime();

        return FineTriggerBuilder.newTrigger()
                .withName(name)
                .withGroup(group)
                .startNow()
                .onlyOnce()
                .withSimpleMisfireInst(SimpleMisfireInstType.IGNORE_ALL_MISFIRES_AND_BACK_TO_SCHEDULE_WITH_PLANNED_COUNT)
                .build();
    }
}
