package com.fr.scheduler.enums;

import com.fr.third.v2.org.quartz.SimpleScheduleBuilder;

/**
 * 简单重复触发器{@link com.fr.scheduler.enums.FineTriggerType#SIMPLE}的Misfire策略.
 *
 * <p>
 * Misfire指任务由于关机等原因未能在该触发时触发.<br>
 * Misfire策略指定了重新上线发现Misfire后的处理策略.
 * </p>
 *
 * @author Cloud.Liu
 * created on 2020-10-21
 */
public enum SimpleMisfireInstType {

    /**
     * 默认策略.
     *
     * <p>
     * 简单重复类型触发器默认的Misfire策略.
     * </p>
     *
     * <pre>
     * 不重复：立即触发.<br>
     *  如：任务计划于9点执行，9点未执行，10:15发现，则触发时间为10:15.<br>
     * 重复特定次数：立即触发一次（无论Misfire多少次），并以当前时间点为开始时间点，以指定时间间隔为时间间隔重复触发，直到触发完本应当触发的所有次数. 最终触发总次数等于最初规划次数，后续触发时间点被同步修改.<br>
     *  如：任务计划于9~12的整点执行，9和10点未执行，10:15发现，则触发时间为10:15, 11:15, 12:15, 13:15.<br>
     * 永远重复：忽略所有Misfire，继续按最初规划执行. 后续触发时间点符合最初规划.<br>
     *  如：任务计划于9点及之后的整点执行，9和10点未执行，10:15发现，则触发时间为11:00, 12:00, 13:00...<br>
     * </pre>
     * @IncludeIntoJavadoc
     */
    DEFAULT() {
        @Override
        public void modifySimpleScheduleBuilder(SimpleScheduleBuilder builder) {
            // does nothing
        }
    },
    /**
     * FireNow策略.
     *
     * <pre>
     * 不重复：立即触发.<br>
     *  如：任务计划于9点执行，9点未执行，10:15发现，则触发时间为10:15.<br>
     * 重复特定次数：立即触发一次（无论Misfire多少次），并以当前时间点为开始时间点，以指定时间间隔为时间间隔重复触发，直到触发完剩余未触发的所有次数. 最终触发总次数小于等于最初规划次数，后续触发时间点被同步修改.<br>
     *  如：任务计划于9~12的整点执行，9和10点未执行，10:15发现，则触发时间为10:15, 11:15, 12:15.<br>
     * 永远重复：立即触发一次（无论Misfire多少次），并以当前时间点为开始时间点，以指定时间间隔为时间间隔重复触发. 后续触发时间点被同步修改.<br>
     *  如：任务计划于9点及之后的整点执行，9和10点未执行，10:15发现，则触发时间为10:15, 11:15, 12:15...<br>
     * </pre>
     * @IncludeIntoJavadoc
     */
    FIRE_ONE_MISFIRE_NOW_AND_RESCHEDULE_WITH_REMAINING_COUNT() {
        @Override
        public void modifySimpleScheduleBuilder(SimpleScheduleBuilder builder) {
            builder.withMisfireHandlingInstructionFireNow();
        }
    },
    /**
     * IgnoreMisfire策略.
     *
     * <pre>
     * 不重复：立即触发.<br>
     *  如：任务计划于9点执行，9点未执行，10:15发现，则触发时间为10:15.<br>
     * 重复特定次数：立即触发所有Misfire，并继续按最初规划执行. 最终触发总次数等于最初规划次数，后续触发时间点符合最初规划.<br>
     *  如：任务计划于9~12的整点执行，9和10点未执行，10:15发现，则触发时间为10:15（2次）, 11:00, 12:00.<br>
     * 永远重复：立即触发所有Misfire，并继续按最初规划执行. 后续触发时间点符合最初规划.<br>
     *  如：任务计划于9点及之后的整点执行，9和10点未执行，10:15发现，则触发时间为10:15（2次）, 11:00, 12:00...<br>
     * </pre>
     * @IncludeIntoJavadoc
     */
    FIRE_ALL_MISFIRES_NOW_AND_BACK_TO_SCHEDULE_WITH_REMAINING_COUNT() {
        @Override
        public void modifySimpleScheduleBuilder(SimpleScheduleBuilder builder) {
            builder.withMisfireHandlingInstructionIgnoreMisfires();
        }
    },
    /**
     * NextWithExistingCount策略.
     *
     * <pre>
     * 不重复：不触发.<br>
     *  如：10点要录制电视节目，11点发现10:00没有录制，但已经没有补录的必要了.<br>
     * 重复特定次数：不立即触发，并继续按最初规划执行，直到触发完本应当触发的所有次数. 最终触发总次数等于最初规划次数，后续触发时间点符合最初规划.<br>
     *  如：任务计划于9~12的整点执行，9和10点未执行，10:15发现，则触发时间为11:00, 12:00, 13:00, 14:00.<br>
     * 永远重复：不触发，并继续按最初规划执行. 后续触发时间点符合最初规划.<br>
     *  如：任务计划于9~12的整点执行，9和10点未执行，10:15发现，则触发时间为11:00, 12:00...<br>
     * </pre>
     * @IncludeIntoJavadoc
     */
    IGNORE_ALL_MISFIRES_AND_BACK_TO_SCHEDULE_WITH_PLANNED_COUNT() {
        @Override
        public void modifySimpleScheduleBuilder(SimpleScheduleBuilder builder) {
            builder.withMisfireHandlingInstructionNextWithExistingCount();
        }
    },
    /**
     * NextWithRemainingCount策略.
     *
     * <pre>
     * 不重复：不触发.<br>
     *  如：10点要录制电视节目，11点发现10:00没有录制，但已经没有补录的必要了.<br>
     * 重复特定次数：不触发，并继续按最初规划执行，直到触发完剩余未触发的所有次数. 最终触发总次数小于最初规划次数，后续触发时间点符合最初规划.<br>
     *  如：任务计划于9~12的整点执行，9和10点未执行，10:15发现，则触发时间为11:00, 12:00.<br>
     * 永远重复：不触发，并继续按最初规划执行. 后续触发时间点符合最初规划.<br>
     *  如：任务计划于9~12的整点执行，9和10点未执行，10:15发现，则触发时间为11:00, 12:00...<br>
     * </pre>
     * @IncludeIntoJavadoc
     */
    IGNORE_ALL_MISFIRES_AND_BACK_TO_SCHEDULE_WITH_REMAINING_COUNT() {
        @Override
        public void modifySimpleScheduleBuilder(SimpleScheduleBuilder builder) {
            builder.withMisfireHandlingInstructionNextWithRemainingCount();
        }
    },
    /**
     * NowWithExistingCount策略.
     *
     * <pre>
     * 不重复：立即触发.<br>
     *  如：任务计划于9点执行，9点未执行，10:15发现，则触发时间为10:15.<br>
     * 重复特定次数：立即触发一次（无论Misfire多少次），并以当前时间点为开始时间点，以指定时间间隔为时间间隔重复触发，直到触发完本应当触发的所有次数. 最终触发总次数等于最初规划次数，后续触发时间点被同步修改.<br>
     *  如：任务计划于9~12的整点执行，9和10点未执行，10:15发现，则触发时间为10:15, 11:15, 12:15, 13:15.<br>
     * 永远重复：立即触发一次（无论Misfire多少次），并以当前时间点为开始时间点，以指定时间间隔为时间间隔重复触发. 后续触发时间点被同步修改.<br>
     *  如：任务计划于9点及之后的整点执行，9和10点未执行，10:15发现，则触发时间为10:15, 11:15, 12:15...<br>
     * </pre>
     * @IncludeIntoJavadoc
     */
    FIRE_NOW_AND_RESCHEDULE_WITH_PLANNED_COUNT() {
        @Override
        public void modifySimpleScheduleBuilder(SimpleScheduleBuilder builder) {
            builder.withMisfireHandlingInstructionNowWithExistingCount();
        }
    },
    /**
     * NowWithRemainingCount策略.
     *
     * <pre>
     * 不重复：立即触发.<br>
     *  如：任务计划于9点执行，9点未执行，10:15发现，则触发时间为10:15.<br>
     * 重复特定次数：立即触发一次（无论Misfire多少次），并以当前时间点为开始时间点，以指定时间间隔为时间间隔重复触发，直到触发完剩余未触发的所有次数. 最终触发总次数小于等于最初规划次数，后续触发时间点被同步修改.<br>
     *  如：任务计划于9~12的整点执行，9和10点未执行，10:15发现，则触发时间为10:15, 11:15, 12:15.<br>
     * 永远重复：立即触发一次（无论Misfire多少次），并以当前时间点为开始时间点，以指定时间间隔为时间间隔重复触发. 后续触发时间点被同步修改.<br>
     *  如：任务计划于9点及之后的整点执行，9和10点未执行，10:15发现，则触发时间为10:15, 11:15, 12:15...<br>
     * </pre>
     * @IncludeIntoJavadoc
     */
    FIRE_NOW_AND_RESCHEDULE_WITH_REMAINING_COUNT() {
        @Override
        public void modifySimpleScheduleBuilder(SimpleScheduleBuilder builder) {
            builder.withMisfireHandlingInstructionNowWithRemainingCount();
        }
    };

    public abstract void modifySimpleScheduleBuilder(SimpleScheduleBuilder builder);
}
