package com.fr.scheduler.cluster;

import com.fr.scheduler.tool.FineScheduler;
import com.fr.scheduler.tool.FineSchedulerPropertiesBuilder;
import org.jetbrains.annotations.NotNull;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;

/**
 * fine-scheduler集群模式工具接口.
 *
 * <p>
 * 使用fine-scheduler集群模式需实现此接口.
 * </p>
 *
 * <pre>
 * 基于Redis的典型实现见com/fr/scheduler/cluster/ClusterImpl.java.
 * </pre>
 *
 * @author Cloud.Liu
 * created on 2020-09-17
 * @see FineSchedulerPropertiesBuilder {@link FineScheduler}初始化Properties创建工具类
 * @IncludeIntoJavadoc
 */
public interface FineSchedulerClusterProvider {

    /**
     * 初始化本节点.
     *
     * <p>
     * 与{@link FineSchedulerClusterProvider#notify(String, long)}一起，用于在集群下通知其他节点即将有任务执行.<br>
     * 如在A节点添加了B、C节点需要执行的任务，则会需要通知B、C节点有任务即将执行.<br>
     * </p>
     *
     * <p>
     * 实现应当至少包含以下两项内容：<br>
     * 1 将本节点ID注册到一个存放着所有节点ID的分布式容器中.<br>
     * 2 实现当其他节点调用{@link FineSchedulerClusterProvider#notify(java.lang.String, long)}对本节点进行通知时，调用{@link com.fr.scheduler.cluster.FineSchedulerClusterProvider#notifySelf(long)}方法.<br>
     * </p>
     *
     * <p>
     * 以上两项内容的典型实现如：<br>
     * 1 将本节点ID添加到Redis集合中.<br>
     * 2 向Redis订阅消息，消息内容应当包括节点ID与任务下次执行时间（见{@link FineSchedulerClusterProvider#notify(java.lang.String, long)}）.将节点ID与本节点ID进行比对，若相同，调用{@link com.fr.scheduler.cluster.FineSchedulerClusterProvider#notifySelf(long)}方法.<br>
     * </p>
     *
     * <p>
     * 这个方法在{@link com.fr.scheduler.tool.FineScheduler}以集群模式初始化时被自动调用。
     * </p>
     *
     * @param nodeId 节点ID
     */
    void register(String nodeId);

    /**
     * 关闭本节点.
     *
     * <p>
     * 与{@link FineSchedulerClusterProvider#register(String)}相对应.<br>
     * 如果{@code register}中进行了节点ID的添加和回调注册，本方法就应当删除节点ID并解注册.<br>
     * </p>
     *
     * @param nodeId 本节点ID
     */
    void unregister(String nodeId);

    /**
     * 通知其他节点.
     *
     * <p>
     * 在需要的时候，{@link com.fr.scheduler.tool.FineScheduler}会自动调用该方法，以通知其他节点有任务即将执行.<br>
     * </p>
     *
     * <p>
     * 这个方法的典型实现如：<br>
     * 向注册中心推送消息，消息内容包含要通知的节点ID {@code nodeId}和任务执行时间{@code time}.
     * </p>
     *
     * @param nodeId 要通知的节点ID
     * @param time 将要执行任务的执行时间
     */
    void notify(String nodeId, long time);

    /**
     * 提供分布式锁.
     *
     * <p>
     * 为了保证集群下任务不会被重复执行，需要提供分布式锁.<br>
     * 典型实现如基于Redis或ZooKeeper的分布式锁.<br>
     * </p>
     *
     * <p>
     * 实际只用到了{@code tryLock()}与{@code unlock()}方法.
     * </p>
     *
     * @param name 锁的名称
     * @return 锁
     */
    Lock getLock(String name);

    /**
     * 获取所有节点ID.
     *
     * <p>
     * 返回在{@link FineSchedulerClusterProvider#register(java.lang.String)}中注册的所有节点ID.
     * </p>
     *
     * @return 所有节点ID
     */
    String[] allNodeIds();

    /**
     * 通知本节点即将有任务执行.
     *
     * <p>
     * 供{@link FineSchedulerClusterProvider#register(String)}使用.
     * </p>
     *
     * @param time 任务执行时间
     */
    default void notifySelf(long time) {
        FineScheduler.getInstance().notifyScheduler(time);
    }

    /**
     * 默认实现.
     *
     * <p>
     * 注册与解注册没有实际操作。<br>
     * 集群锁获取永远成功。<br>
     * 获取集群所有节点返回空数组。<br>
     * </p>
     */
    FineSchedulerClusterProvider DEFAULT = new FineSchedulerClusterProvider() {

        @Override
        public void register(String nodeId) {

        }

        @Override
        public void unregister(String nodeId) {

        }

        @Override
        public void notify(String nodeId, long time) {

        }

        @Override
        public Lock getLock(String name) {

            return new Lock() {
                @Override
                public void lock() {

                }

                @Override
                public void lockInterruptibly() throws InterruptedException {

                }

                @Override
                public boolean tryLock() {
                    return true;
                }

                @Override
                public boolean tryLock(long time, @NotNull TimeUnit unit) throws InterruptedException {
                    return false;
                }

                @Override
                public void unlock() {

                }

                @NotNull
                @Override
                public Condition newCondition() {
                    return null;
                }
            };
        }

        @Override
        public String[] allNodeIds() {
            return new String[0];
        }
    };
}
