package com.fr.cache.list;

import com.fr.stable.CommonUtils;
import com.fr.stable.collections.map.IntMap;

import java.io.Serializable;
import java.util.Arrays;


/**
 * 整数链表，这个链表的所有元素都是整数
 */
public class IntList implements Cloneable, Serializable {
    private int[] data; // An array to store the numbers.
    private int size = 0;  // Index of next unused element of array


    private static final int CACHE_SIZE = 100;

    /**
     * 默认的构造函数，生成一个容量为8的整数链表
     */
    public IntList() {
        this(8);
    }

    /**
     * 生成一个指定容量的整数链表
     *
     * @param initialCapacity 容量
     */
    public IntList(int initialCapacity) {
        if (initialCapacity < 0)
            throw new IllegalArgumentException("Illegal Capacity: " +
                    initialCapacity);
        data = new int[initialCapacity];
    }

    /**
     * 获取整数链表指定位置的元素
     *
     * @param index 要获取元素的索引
     * @return 链表元素
     * @throws ArrayIndexOutOfBoundsException 指定的位置比整数链表容量大时抛出此异常
     */
    public int get(int index) throws ArrayIndexOutOfBoundsException {
        if (index >= size) {
            throw new ArrayIndexOutOfBoundsException(index);
        } else {
            return data[index];
        }
    }

    /**
     * 设置整数链表指定位置的元素为给定值
     *
     * @param index 指定的索引
     * @param value 要设置的值
     * @throws ArrayIndexOutOfBoundsException 指定的索引超过链表容量时抛出此异常
     */
    public void set(int index, int value) throws ArrayIndexOutOfBoundsException {
        if (index >= size) {
            throw new ArrayIndexOutOfBoundsException(index);
        } else {
            data[index] = value;
        }
    }

    /**
     * 添加一个元素到链表中，在必要的时候链表的容量会自动增加
     *
     * @param x 要添加到链表中的元素
     */
    public void add(int x) {
        ensureCapacity(size + 1);
        data[size++] = x;         // Store the int in it.
    }

    /**
     * 在链表的指定位置插入一个元素
     *
     * @param index 要插入元素的位置
     * @param x     要插入的元素
     */
    public void add(int index, int x) {
        if (index > size || index < 0)
            throw new IndexOutOfBoundsException("Index: " + index + ", Size: " + size);

        ensureCapacity(size + 1); // Increments modCount!!
        System.arraycopy(data, index, data, index + 1, size - index);
        data[index] = x;
        size++;
    }

    /**
     * 判断指定值是否包含在此整数链表中
     *
     * @param value 指定的值
     * @return 如果链表中包含此值则返回true，否则返回false
     */
    public boolean contain(int value) {
        return indexOf(value) >= 0;
    }

    /**
     * 获取指定值在链表中的位置，如果该值在链表中不存在则返回-1.
     *
     * @param value 指定的值
     * @return 值在链表中的位置
     */
    public int indexOf(int value) {
        for (int i = 0; i < size; i++) {
            if (data[i] == value) {
                return i;
            }
        }

        return -1;
    }

    /**
     * 获取链表的元素的个数
     *
     * @return 链表元素的个数
     */
    public int size() {
        return size;
    }

    /**
     * 移除链表中指定位置的元素
     *
     * @param index 要移除的元素在链表中的索引
     */
    public void remove(int index) {
        int numMoved = size - index - 1;
        if (numMoved > 0) {
            System.arraycopy(data, index + 1, data, index, numMoved);
        }
        --size;
    }

    /**
     * 移除整数链表中所有的相等等数字
     */
    public void removeEqual() {
        int[] newDataArray = this.toArray();

        this.clear();
        for (int i = 0; i < newDataArray.length; i++) {
            if (!this.contain(newDataArray[i])) {
                this.add(newDataArray[i]);
            }
        }
    }

    /**
     * 排序
     */
    public void sort() {
        this.trimToSize();

        Arrays.sort(this.data);
    }

    /**
     * 清楚整数链表中的所有元素
     */
    public void clear() {
        this.data = new int[0];
        this.size = 0;
    }

    /**
     * Trim to Size.
     */
    public void trimToSize() {
        this.data = this.toArray();
    }

    /**
     * 将链表转化成整数数组
     *
     * @return 整数数组
     */
    public int[] toArray() {
        int[] intArray = new int[size];

        System.arraycopy(data, 0, intArray, 0, size);

        return intArray;
    }

    /**
     * 将指定的新的整数链表中的所有元素插入到当前链表中指定位置
     *
     * @param index 要插入的位置
     * @param c     要插入的整数链表
     * @return 如果在当前链表中成功插入了元素，则返回true，否则返回false
     */
    public boolean addAll(int index, IntList c) {
        if (index > size || index < 0)
            throw new IndexOutOfBoundsException(
                    "Index: " + index + ", Size: " + size);
        int[] a = c.toArray();

        int numNew = a.length;
        ensureCapacity(size + numNew);  // Increments modCount

        int numMoved = size - index;
        if (numMoved > 0)
            System.arraycopy(data, index, data, index + numNew,
                    numMoved);

        System.arraycopy(a, 0, data, index, numNew);
        size += numNew;
        return numNew != 0;
    }

    /**
     * 将指定的整数列表的所有元素添加到当前列表中
     *
     * @param c 要添加的整数列表
     * @return 如果在当前列表添加了元素则返回true，否则返回false
     */
    public boolean addAll(IntList c) {
        int[] a = c.toArray();
        int numNew = a.length;
        ensureCapacity(size + numNew);  // Increments modCount
        System.arraycopy(a, 0, data, size, numNew);
        size += numNew;
        return numNew != 0;
    }

    /**
     * 判断当前对象是否和指定的对象相等
     *
     * @param o 指定的对象
     * @return 相等则返回true，不相等则返回false
     */
    public boolean equals(Object o) {
        if (!(o instanceof IntList)) {
            return false;
        }

        IntList that = (IntList) o;
        if (this.size != that.size) {
            return false;
        }

        for (int i = 0; i < this.size; i++) {
            if (this.data[i] != that.data[i]) {
                return false;
            }
        }

        return true;
    }

    /**
     * @return 克隆的对象
     * @throws CloneNotSupportedException cloneNotSupportedException
     */
    public Object clone() throws CloneNotSupportedException {
        IntList newIntList = (IntList) super.clone();
        newIntList.clear();

        for (int i = 0; i < this.size(); i++) {
            newIntList.add(this.get(i));
        }

        return newIntList;
    }

    /**
     * 表示该整数链表对象的字符串
     *
     * @return 链表对象的字符串表示
     */
    public String toString() {
        StringBuffer sb = new StringBuffer("[");
        for (int i = 0, len = size(); i < len; i++) {
            if (i > 0) {
                sb.append(',');
            }
            sb.append(get(i));
        }
        sb.append(']');

        return sb.toString();
    }

    /**
     * An internal method to change the allocated size of the array
     */
    private void ensureCapacity(int minCapacity) {
        int oldCapacity = this.data.length;
        if (minCapacity > oldCapacity) {
            int oldData[] = data;
            int newCapacity = (oldCapacity * 3) / 2 + 1;
            if (newCapacity < minCapacity)
                newCapacity = minCapacity;
            data = new int[newCapacity];
            System.arraycopy(oldData, 0, data, 0, size);
        }
    }

    /**
     * 将一个整数数组转化成一个整数链表
     *
     * @param array 要转化的整数数组
     * @return 整数数组转化成的整数链表
     */
    public static IntList asList(int[] array) {
        if (array == null) {
            throw new NullPointerException();
        }

        IntList list = new IntList();
        list.data = array;
        list.size = list.data.length;
        return list;
    }


    /**
     * 将使用特定字符分割的由整数、“-”字符以及分割符组成的字符串转化成整数链表
     *
     * @param str     待分析的字符串
     * @param decimal 分割字符串的分隔符
     * @return 整数链表
     */
    public static IntList toIntListDecimal(String str, char decimal) {
        IntList intList = new IntList();
        if (str == null) {
            return intList;
        }

        String[] rowIndexArray = CommonUtils.splitString(str, "" + decimal);
        for (int i = 0; i < rowIndexArray.length; i++) {
            String intTxt = rowIndexArray[i].trim();
            try {
                intList.add(Integer.parseInt(intTxt));
            } catch (Exception exp) {
                //分析 2-6
                String[] intArray = CommonUtils.splitString(intTxt, "-");
                int minInt = -1;
                int maxInt = -1;
                for (int j = 0; j < intArray.length; j++) {
                    int pInt = 0;
                    try {
                        pInt = Integer.parseInt(intArray[j]);
                    } catch (NumberFormatException numberFormatException) {
                        //do nothing.
                    }
                    if (j == 0) {
                        minInt = pInt;
                        maxInt = pInt;
                        continue;
                    }

                    minInt = Math.min(pInt, minInt);
                    maxInt = Math.max(pInt, maxInt);
                }

                for (int j = minInt; j <= maxInt; j++) {
                    intList.add(j);
                }
            }
        }

        return intList;
    }

    /**
     * 将使用特定字符分割的由整数、“-”字符以及分割符组成的字符串转化成整数数组
     *
     * @param str     待分析的字符串
     * @param decimal 分隔符
     * @return 整数数组
     */
    public static int[] toIntArrayDecimal(String str, char decimal) {
        return IntList.toIntListDecimal(str, decimal).toArray();
    }

    /**
     * 返回两个整数数组交集做组成的数组
     *
     * @param i1 数组1
     * @param i2 数组2
     * @return 交集所组成的数组
     */
    public static int[] unionArray(int[] i1, int[] i2) {
        //如果i1和i2中有一个是null就返回另一个
        if (i1 == null) {
            return i2 == null ? i2 : (int[]) i2.clone();
        }
        if (i2 == null) {
            return (int[]) i1.clone();
        }

        int[] array1 = (int[]) i1.clone();
        int[] array2 = (int[]) i2.clone();

        java.util.Arrays.sort(array1);
        java.util.Arrays.sort(array2);

        IntList res = new IntList();
        int i = 0, j = 0;
        while (i < array1.length || j < array2.length) {
            if (i == array1.length) {
                res.add(array2[j]);
                j++;
            } else if (j == array2.length) {
                res.add(array1[i]);
                i++;
            } else if (array1[i] < array2[j]) {
                res.add(array1[i]);
                i++;
            } else if (array1[i] > array2[j]) {
                res.add(array2[j]);
                j++;
            } else if (array1[i] == array2[j]) {
                res.add(array2[j]);
                i++;
                j++;
            }
        }

        return res.toArray();
    }

    /**
     * 给指定的整数列表排序
     *
     * @param list 待排序的整数列表
     */
    public static void sort(IntList list) {
        int[] array = list.toArray();
        Arrays.sort(array);
        list.data = array;
    }

    //缓存最近的CACHE_SIZE个
    private static IntMap<int[]> cacheMap = new IntMap<int[]>(CACHE_SIZE);


    /**
     * 从缓存中获取数组
     *
     * @param to 结束数字
     * @return 整数数组
     */
    public static int[] getRangeFromCache(int to) {
        int[] ret = getCache(to);
        if (ret == null) {
            ret = range(to);
            putCache(to, ret);
        }
        return ret;
    }

    private static int[] getCache(int to) {
        synchronized (cacheMap) {
            return cacheMap.get(to);
        }
    }

    //TODO 这些个代码都可以完全删掉暂时先这样处理，完全没有返回int数组的必要
    private static void putCache(int to, int[] cache) {
        synchronized (cacheMap) {
            if(cacheMap.size > 1000) {
                cacheMap.clear();
            }
            cacheMap.put(to, cache);
        }
    }

    /**
     * 生成一个0到指定结束数字的整数数组，数组的元素为0，1，2，......，to -1
     * TODO 何必生成int数组呢直接用描述不就好了 0-100， 直接就用一个含有100的对象表示，在对象里面做其他的运算效率多好
     * @param to 结束数字
     * @return 整数数组
     */
    public static int[] range(int to) {
        //如果只是单纯生成一个range(100)之类的数组,那就直接return个数组好了
        if (to > 0) {
            int[] array = new int[to];
            for (int i = 0; i < to; i++) {
                array[i] = i;
            }

            return array;
        }

        return range(0, to);
    }

    /**
     * 生成一个指定起始和结束数字的整数数组
     *
     * @param from 起始数字
     * @param to   结束数字
     * @return 整数数组
     */
    public static int[] range(int from, int to) {
        return range(from, to, 1);
    }

    /**
     * 生成一个指定起始和结束数以及步长的整数数组，数组的元素为from, from + step,  ......，from + n*step;
     *
     * @param from 起始数字
     * @param to   结束数字
     * @param step 步长
     * @return 整数数组
     */
    public static int[] range(int from, int to, int step) {
        IntList list = new IntList();
        if (step > 0) {
            while (from < to) {
                list.add(from);
                from = from + step;
            }
        } else {
            while (from > to) {
                list.add(from);
                from = from + step;
            }
        }

        return list.toArray();
    }

    /**
     * 反转数组
     *
     * @param list {@code IntList}
     * @return 反转后的 {@code IntList}
     */
    public static IntList reverse(IntList list) {
        int len = list.size();
        int[] list_array = list.toArray();
        int[] temp = new int[len];
        for (int i = 0; i < len; i++) {
            temp[i] = list_array[len - i - 1];
        }

        return asList(temp);
    }
}