/* ----------------------------------------------------------------------
 * Project:      CMSIS DSP Library
 * Title:        arm_common_tables.c
 * Description:  common tables like fft twiddle factors, Bitreverse, reciprocal etc
 *
 * $Date:        27. January 2017
 * $Revision:    V.1.5.1
 *
 * Target Processor: Cortex-M cores
 * -------------------------------------------------------------------- */
/*
 * Copyright (C) 2010-2017 ARM Limited or its affiliates. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "arm_math.h"
#include "arm_common_tables.h"

/**
 * @ingroup ComplexFFT
 */

/**
 * @addtogroup CFFT_CIFFT Complex FFT Tables
 * @{
 */

/**
* \par
* Pseudo code for Generation of Bit reversal Table is
* \par
* <pre>for(l=1;l <= N/4;l++)
* {
*   for(i=0;i<logN2;i++)
*   {
*     a[i]=l&(1<<i);
*   }
*   for(j=0; j<logN2; j++)
*   {
*     if (a[j]!=0)
*     y[l]+=(1<<((logN2-1)-j));
*   }
*   y[l] = y[l] >> 1;
*  } </pre>
* \par
* where N = 4096	logN2 = 12
* \par
* N is the maximum FFT Size supported
*/

/*
* @brief  Table for bit reversal process
*/
const uint16_t armBitRevTable[1024] = {
   0x400, 0x200, 0x600, 0x100, 0x500, 0x300, 0x700, 0x80, 0x480, 0x280,
   0x680, 0x180, 0x580, 0x380, 0x780, 0x40, 0x440, 0x240, 0x640, 0x140,
   0x540, 0x340, 0x740, 0xc0, 0x4c0, 0x2c0, 0x6c0, 0x1c0, 0x5c0, 0x3c0,
   0x7c0, 0x20, 0x420, 0x220, 0x620, 0x120, 0x520, 0x320, 0x720, 0xa0,
   0x4a0, 0x2a0, 0x6a0, 0x1a0, 0x5a0, 0x3a0, 0x7a0, 0x60, 0x460, 0x260,
   0x660, 0x160, 0x560, 0x360, 0x760, 0xe0, 0x4e0, 0x2e0, 0x6e0, 0x1e0,
   0x5e0, 0x3e0, 0x7e0, 0x10, 0x410, 0x210, 0x610, 0x110, 0x510, 0x310,
   0x710, 0x90, 0x490, 0x290, 0x690, 0x190, 0x590, 0x390, 0x790, 0x50,
   0x450, 0x250, 0x650, 0x150, 0x550, 0x350, 0x750, 0xd0, 0x4d0, 0x2d0,
   0x6d0, 0x1d0, 0x5d0, 0x3d0, 0x7d0, 0x30, 0x430, 0x230, 0x630, 0x130,
   0x530, 0x330, 0x730, 0xb0, 0x4b0, 0x2b0, 0x6b0, 0x1b0, 0x5b0, 0x3b0,
   0x7b0, 0x70, 0x470, 0x270, 0x670, 0x170, 0x570, 0x370, 0x770, 0xf0,
   0x4f0, 0x2f0, 0x6f0, 0x1f0, 0x5f0, 0x3f0, 0x7f0, 0x8, 0x408, 0x208,
   0x608, 0x108, 0x508, 0x308, 0x708, 0x88, 0x488, 0x288, 0x688, 0x188,
   0x588, 0x388, 0x788, 0x48, 0x448, 0x248, 0x648, 0x148, 0x548, 0x348,
   0x748, 0xc8, 0x4c8, 0x2c8, 0x6c8, 0x1c8, 0x5c8, 0x3c8, 0x7c8, 0x28,
   0x428, 0x228, 0x628, 0x128, 0x528, 0x328, 0x728, 0xa8, 0x4a8, 0x2a8,
   0x6a8, 0x1a8, 0x5a8, 0x3a8, 0x7a8, 0x68, 0x468, 0x268, 0x668, 0x168,
   0x568, 0x368, 0x768, 0xe8, 0x4e8, 0x2e8, 0x6e8, 0x1e8, 0x5e8, 0x3e8,
   0x7e8, 0x18, 0x418, 0x218, 0x618, 0x118, 0x518, 0x318, 0x718, 0x98,
   0x498, 0x298, 0x698, 0x198, 0x598, 0x398, 0x798, 0x58, 0x458, 0x258,
   0x658, 0x158, 0x558, 0x358, 0x758, 0xd8, 0x4d8, 0x2d8, 0x6d8, 0x1d8,
   0x5d8, 0x3d8, 0x7d8, 0x38, 0x438, 0x238, 0x638, 0x138, 0x538, 0x338,
   0x738, 0xb8, 0x4b8, 0x2b8, 0x6b8, 0x1b8, 0x5b8, 0x3b8, 0x7b8, 0x78,
   0x478, 0x278, 0x678, 0x178, 0x578, 0x378, 0x778, 0xf8, 0x4f8, 0x2f8,
   0x6f8, 0x1f8, 0x5f8, 0x3f8, 0x7f8, 0x4, 0x404, 0x204, 0x604, 0x104,
   0x504, 0x304, 0x704, 0x84, 0x484, 0x284, 0x684, 0x184, 0x584, 0x384,
   0x784, 0x44, 0x444, 0x244, 0x644, 0x144, 0x544, 0x344, 0x744, 0xc4,
   0x4c4, 0x2c4, 0x6c4, 0x1c4, 0x5c4, 0x3c4, 0x7c4, 0x24, 0x424, 0x224,
   0x624, 0x124, 0x524, 0x324, 0x724, 0xa4, 0x4a4, 0x2a4, 0x6a4, 0x1a4,
   0x5a4, 0x3a4, 0x7a4, 0x64, 0x464, 0x264, 0x664, 0x164, 0x564, 0x364,
   0x764, 0xe4, 0x4e4, 0x2e4, 0x6e4, 0x1e4, 0x5e4, 0x3e4, 0x7e4, 0x14,
   0x414, 0x214, 0x614, 0x114, 0x514, 0x314, 0x714, 0x94, 0x494, 0x294,
   0x694, 0x194, 0x594, 0x394, 0x794, 0x54, 0x454, 0x254, 0x654, 0x154,
   0x554, 0x354, 0x754, 0xd4, 0x4d4, 0x2d4, 0x6d4, 0x1d4, 0x5d4, 0x3d4,
   0x7d4, 0x34, 0x434, 0x234, 0x634, 0x134, 0x534, 0x334, 0x734, 0xb4,
   0x4b4, 0x2b4, 0x6b4, 0x1b4, 0x5b4, 0x3b4, 0x7b4, 0x74, 0x474, 0x274,
   0x674, 0x174, 0x574, 0x374, 0x774, 0xf4, 0x4f4, 0x2f4, 0x6f4, 0x1f4,
   0x5f4, 0x3f4, 0x7f4, 0xc, 0x40c, 0x20c, 0x60c, 0x10c, 0x50c, 0x30c,
   0x70c, 0x8c, 0x48c, 0x28c, 0x68c, 0x18c, 0x58c, 0x38c, 0x78c, 0x4c,
   0x44c, 0x24c, 0x64c, 0x14c, 0x54c, 0x34c, 0x74c, 0xcc, 0x4cc, 0x2cc,
   0x6cc, 0x1cc, 0x5cc, 0x3cc, 0x7cc, 0x2c, 0x42c, 0x22c, 0x62c, 0x12c,
   0x52c, 0x32c, 0x72c, 0xac, 0x4ac, 0x2ac, 0x6ac, 0x1ac, 0x5ac, 0x3ac,
   0x7ac, 0x6c, 0x46c, 0x26c, 0x66c, 0x16c, 0x56c, 0x36c, 0x76c, 0xec,
   0x4ec, 0x2ec, 0x6ec, 0x1ec, 0x5ec, 0x3ec, 0x7ec, 0x1c, 0x41c, 0x21c,
   0x61c, 0x11c, 0x51c, 0x31c, 0x71c, 0x9c, 0x49c, 0x29c, 0x69c, 0x19c,
   0x59c, 0x39c, 0x79c, 0x5c, 0x45c, 0x25c, 0x65c, 0x15c, 0x55c, 0x35c,
   0x75c, 0xdc, 0x4dc, 0x2dc, 0x6dc, 0x1dc, 0x5dc, 0x3dc, 0x7dc, 0x3c,
   0x43c, 0x23c, 0x63c, 0x13c, 0x53c, 0x33c, 0x73c, 0xbc, 0x4bc, 0x2bc,
   0x6bc, 0x1bc, 0x5bc, 0x3bc, 0x7bc, 0x7c, 0x47c, 0x27c, 0x67c, 0x17c,
   0x57c, 0x37c, 0x77c, 0xfc, 0x4fc, 0x2fc, 0x6fc, 0x1fc, 0x5fc, 0x3fc,
   0x7fc, 0x2, 0x402, 0x202, 0x602, 0x102, 0x502, 0x302, 0x702, 0x82,
   0x482, 0x282, 0x682, 0x182, 0x582, 0x382, 0x782, 0x42, 0x442, 0x242,
   0x642, 0x142, 0x542, 0x342, 0x742, 0xc2, 0x4c2, 0x2c2, 0x6c2, 0x1c2,
   0x5c2, 0x3c2, 0x7c2, 0x22, 0x422, 0x222, 0x622, 0x122, 0x522, 0x322,
   0x722, 0xa2, 0x4a2, 0x2a2, 0x6a2, 0x1a2, 0x5a2, 0x3a2, 0x7a2, 0x62,
   0x462, 0x262, 0x662, 0x162, 0x562, 0x362, 0x762, 0xe2, 0x4e2, 0x2e2,
   0x6e2, 0x1e2, 0x5e2, 0x3e2, 0x7e2, 0x12, 0x412, 0x212, 0x612, 0x112,
   0x512, 0x312, 0x712, 0x92, 0x492, 0x292, 0x692, 0x192, 0x592, 0x392,
   0x792, 0x52, 0x452, 0x252, 0x652, 0x152, 0x552, 0x352, 0x752, 0xd2,
   0x4d2, 0x2d2, 0x6d2, 0x1d2, 0x5d2, 0x3d2, 0x7d2, 0x32, 0x432, 0x232,
   0x632, 0x132, 0x532, 0x332, 0x732, 0xb2, 0x4b2, 0x2b2, 0x6b2, 0x1b2,
   0x5b2, 0x3b2, 0x7b2, 0x72, 0x472, 0x272, 0x672, 0x172, 0x572, 0x372,
   0x772, 0xf2, 0x4f2, 0x2f2, 0x6f2, 0x1f2, 0x5f2, 0x3f2, 0x7f2, 0xa,
   0x40a, 0x20a, 0x60a, 0x10a, 0x50a, 0x30a, 0x70a, 0x8a, 0x48a, 0x28a,
   0x68a, 0x18a, 0x58a, 0x38a, 0x78a, 0x4a, 0x44a, 0x24a, 0x64a, 0x14a,
   0x54a, 0x34a, 0x74a, 0xca, 0x4ca, 0x2ca, 0x6ca, 0x1ca, 0x5ca, 0x3ca,
   0x7ca, 0x2a, 0x42a, 0x22a, 0x62a, 0x12a, 0x52a, 0x32a, 0x72a, 0xaa,
   0x4aa, 0x2aa, 0x6aa, 0x1aa, 0x5aa, 0x3aa, 0x7aa, 0x6a, 0x46a, 0x26a,
   0x66a, 0x16a, 0x56a, 0x36a, 0x76a, 0xea, 0x4ea, 0x2ea, 0x6ea, 0x1ea,
   0x5ea, 0x3ea, 0x7ea, 0x1a, 0x41a, 0x21a, 0x61a, 0x11a, 0x51a, 0x31a,
   0x71a, 0x9a, 0x49a, 0x29a, 0x69a, 0x19a, 0x59a, 0x39a, 0x79a, 0x5a,
   0x45a, 0x25a, 0x65a, 0x15a, 0x55a, 0x35a, 0x75a, 0xda, 0x4da, 0x2da,
   0x6da, 0x1da, 0x5da, 0x3da, 0x7da, 0x3a, 0x43a, 0x23a, 0x63a, 0x13a,
   0x53a, 0x33a, 0x73a, 0xba, 0x4ba, 0x2ba, 0x6ba, 0x1ba, 0x5ba, 0x3ba,
   0x7ba, 0x7a, 0x47a, 0x27a, 0x67a, 0x17a, 0x57a, 0x37a, 0x77a, 0xfa,
   0x4fa, 0x2fa, 0x6fa, 0x1fa, 0x5fa, 0x3fa, 0x7fa, 0x6, 0x406, 0x206,
   0x606, 0x106, 0x506, 0x306, 0x706, 0x86, 0x486, 0x286, 0x686, 0x186,
   0x586, 0x386, 0x786, 0x46, 0x446, 0x246, 0x646, 0x146, 0x546, 0x346,
   0x746, 0xc6, 0x4c6, 0x2c6, 0x6c6, 0x1c6, 0x5c6, 0x3c6, 0x7c6, 0x26,
   0x426, 0x226, 0x626, 0x126, 0x526, 0x326, 0x726, 0xa6, 0x4a6, 0x2a6,
   0x6a6, 0x1a6, 0x5a6, 0x3a6, 0x7a6, 0x66, 0x466, 0x266, 0x666, 0x166,
   0x566, 0x366, 0x766, 0xe6, 0x4e6, 0x2e6, 0x6e6, 0x1e6, 0x5e6, 0x3e6,
   0x7e6, 0x16, 0x416, 0x216, 0x616, 0x116, 0x516, 0x316, 0x716, 0x96,
   0x496, 0x296, 0x696, 0x196, 0x596, 0x396, 0x796, 0x56, 0x456, 0x256,
   0x656, 0x156, 0x556, 0x356, 0x756, 0xd6, 0x4d6, 0x2d6, 0x6d6, 0x1d6,
   0x5d6, 0x3d6, 0x7d6, 0x36, 0x436, 0x236, 0x636, 0x136, 0x536, 0x336,
   0x736, 0xb6, 0x4b6, 0x2b6, 0x6b6, 0x1b6, 0x5b6, 0x3b6, 0x7b6, 0x76,
   0x476, 0x276, 0x676, 0x176, 0x576, 0x376, 0x776, 0xf6, 0x4f6, 0x2f6,
   0x6f6, 0x1f6, 0x5f6, 0x3f6, 0x7f6, 0xe, 0x40e, 0x20e, 0x60e, 0x10e,
   0x50e, 0x30e, 0x70e, 0x8e, 0x48e, 0x28e, 0x68e, 0x18e, 0x58e, 0x38e,
   0x78e, 0x4e, 0x44e, 0x24e, 0x64e, 0x14e, 0x54e, 0x34e, 0x74e, 0xce,
   0x4ce, 0x2ce, 0x6ce, 0x1ce, 0x5ce, 0x3ce, 0x7ce, 0x2e, 0x42e, 0x22e,
   0x62e, 0x12e, 0x52e, 0x32e, 0x72e, 0xae, 0x4ae, 0x2ae, 0x6ae, 0x1ae,
   0x5ae, 0x3ae, 0x7ae, 0x6e, 0x46e, 0x26e, 0x66e, 0x16e, 0x56e, 0x36e,
   0x76e, 0xee, 0x4ee, 0x2ee, 0x6ee, 0x1ee, 0x5ee, 0x3ee, 0x7ee, 0x1e,
   0x41e, 0x21e, 0x61e, 0x11e, 0x51e, 0x31e, 0x71e, 0x9e, 0x49e, 0x29e,
   0x69e, 0x19e, 0x59e, 0x39e, 0x79e, 0x5e, 0x45e, 0x25e, 0x65e, 0x15e,
   0x55e, 0x35e, 0x75e, 0xde, 0x4de, 0x2de, 0x6de, 0x1de, 0x5de, 0x3de,
   0x7de, 0x3e, 0x43e, 0x23e, 0x63e, 0x13e, 0x53e, 0x33e, 0x73e, 0xbe,
   0x4be, 0x2be, 0x6be, 0x1be, 0x5be, 0x3be, 0x7be, 0x7e, 0x47e, 0x27e,
   0x67e, 0x17e, 0x57e, 0x37e, 0x77e, 0xfe, 0x4fe, 0x2fe, 0x6fe, 0x1fe,
   0x5fe, 0x3fe, 0x7fe, 0x1
};


/*
* @brief  Floating-point Twiddle factors Table Generation
*/

/**
* \par
* Example code for Floating-point Twiddle factors Generation:
* \par
* <pre>for(i = 0; i< N/; i++)
* {
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 16	and PI = 3.14159265358979
* \par
* Cos and Sin values are in interleaved fashion
*
*/
const float32_t twiddleCoef_16[32] = {
    1.000000000f,  0.000000000f,
    0.923879533f,  0.382683432f,
    0.707106781f,  0.707106781f,
    0.382683432f,  0.923879533f,
    0.000000000f,  1.000000000f,
   -0.382683432f,  0.923879533f,
   -0.707106781f,  0.707106781f,
   -0.923879533f,  0.382683432f,
   -1.000000000f,  0.000000000f,
   -0.923879533f, -0.382683432f,
   -0.707106781f, -0.707106781f,
   -0.382683432f, -0.923879533f,
   -0.000000000f, -1.000000000f,
    0.382683432f, -0.923879533f,
    0.707106781f, -0.707106781f,
    0.923879533f, -0.382683432f
};

/**
* \par
* Example code for Floating-point Twiddle factors Generation:
* \par
* <pre>for(i = 0; i< N/; i++)
* {
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 32	and PI = 3.14159265358979
* \par
* Cos and Sin values are in interleaved fashion
*
*/
const float32_t twiddleCoef_32[64] = {
    1.000000000f,  0.000000000f,
    0.980785280f,  0.195090322f,
    0.923879533f,  0.382683432f,
    0.831469612f,  0.555570233f,
    0.707106781f,  0.707106781f,
    0.555570233f,  0.831469612f,
    0.382683432f,  0.923879533f,
    0.195090322f,  0.980785280f,
    0.000000000f,  1.000000000f,
   -0.195090322f,  0.980785280f,
   -0.382683432f,  0.923879533f,
   -0.555570233f,  0.831469612f,
   -0.707106781f,  0.707106781f,
   -0.831469612f,  0.555570233f,
   -0.923879533f,  0.382683432f,
   -0.980785280f,  0.195090322f,
   -1.000000000f,  0.000000000f,
   -0.980785280f, -0.195090322f,
   -0.923879533f, -0.382683432f,
   -0.831469612f, -0.555570233f,
   -0.707106781f, -0.707106781f,
   -0.555570233f, -0.831469612f,
   -0.382683432f, -0.923879533f,
   -0.195090322f, -0.980785280f,
   -0.000000000f, -1.000000000f,
    0.195090322f, -0.980785280f,
    0.382683432f, -0.923879533f,
    0.555570233f, -0.831469612f,
    0.707106781f, -0.707106781f,
    0.831469612f, -0.555570233f,
    0.923879533f, -0.382683432f,
    0.980785280f, -0.195090322f
};

/**
* \par
* Example code for Floating-point Twiddle factors Generation:
* \par
* <pre>for(i = 0; i< N/; i++)
* {
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 64	and PI = 3.14159265358979
* \par
* Cos and Sin values are in interleaved fashion
*
*/
const float32_t twiddleCoef_64[128] = {
    1.000000000f,  0.000000000f,
    0.995184727f,  0.098017140f,
    0.980785280f,  0.195090322f,
    0.956940336f,  0.290284677f,
    0.923879533f,  0.382683432f,
    0.881921264f,  0.471396737f,
    0.831469612f,  0.555570233f,
    0.773010453f,  0.634393284f,
    0.707106781f,  0.707106781f,
    0.634393284f,  0.773010453f,
    0.555570233f,  0.831469612f,
    0.471396737f,  0.881921264f,
    0.382683432f,  0.923879533f,
    0.290284677f,  0.956940336f,
    0.195090322f,  0.980785280f,
    0.098017140f,  0.995184727f,
    0.000000000f,  1.000000000f,
   -0.098017140f,  0.995184727f,
   -0.195090322f,  0.980785280f,
   -0.290284677f,  0.956940336f,
   -0.382683432f,  0.923879533f,
   -0.471396737f,  0.881921264f,
   -0.555570233f,  0.831469612f,
   -0.634393284f,  0.773010453f,
   -0.707106781f,  0.707106781f,
   -0.773010453f,  0.634393284f,
   -0.831469612f,  0.555570233f,
   -0.881921264f,  0.471396737f,
   -0.923879533f,  0.382683432f,
   -0.956940336f,  0.290284677f,
   -0.980785280f,  0.195090322f,
   -0.995184727f,  0.098017140f,
   -1.000000000f,  0.000000000f,
   -0.995184727f, -0.098017140f,
   -0.980785280f, -0.195090322f,
   -0.956940336f, -0.290284677f,
   -0.923879533f, -0.382683432f,
   -0.881921264f, -0.471396737f,
   -0.831469612f, -0.555570233f,
   -0.773010453f, -0.634393284f,
   -0.707106781f, -0.707106781f,
   -0.634393284f, -0.773010453f,
   -0.555570233f, -0.831469612f,
   -0.471396737f, -0.881921264f,
   -0.382683432f, -0.923879533f,
   -0.290284677f, -0.956940336f,
   -0.195090322f, -0.980785280f,
   -0.098017140f, -0.995184727f,
   -0.000000000f, -1.000000000f,
    0.098017140f, -0.995184727f,
    0.195090322f, -0.980785280f,
    0.290284677f, -0.956940336f,
    0.382683432f, -0.923879533f,
    0.471396737f, -0.881921264f,
    0.555570233f, -0.831469612f,
    0.634393284f, -0.773010453f,
    0.707106781f, -0.707106781f,
    0.773010453f, -0.634393284f,
    0.831469612f, -0.555570233f,
    0.881921264f, -0.471396737f,
    0.923879533f, -0.382683432f,
    0.956940336f, -0.290284677f,
    0.980785280f, -0.195090322f,
    0.995184727f, -0.098017140f
};

/**
* \par
* Example code for Floating-point Twiddle factors Generation:
* \par
* <pre>for(i = 0; i< N/; i++)
* {
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 128	and PI = 3.14159265358979
* \par
* Cos and Sin values are in interleaved fashion
*
*/

const float32_t twiddleCoef_128[256] = {
    1.000000000f,  0.000000000f,
    0.998795456f,  0.049067674f,
    0.995184727f,  0.098017140f,
    0.989176510f,  0.146730474f,
    0.980785280f,  0.195090322f,
    0.970031253f,  0.242980180f,
    0.956940336f,  0.290284677f,
    0.941544065f,  0.336889853f,
    0.923879533f,  0.382683432f,
    0.903989293f,  0.427555093f,
    0.881921264f,  0.471396737f,
    0.857728610f,  0.514102744f,
    0.831469612f,  0.555570233f,
    0.803207531f,  0.595699304f,
    0.773010453f,  0.634393284f,
    0.740951125f,  0.671558955f,
    0.707106781f,  0.707106781f,
    0.671558955f,  0.740951125f,
    0.634393284f,  0.773010453f,
    0.595699304f,  0.803207531f,
    0.555570233f,  0.831469612f,
    0.514102744f,  0.857728610f,
    0.471396737f,  0.881921264f,
    0.427555093f,  0.903989293f,
    0.382683432f,  0.923879533f,
    0.336889853f,  0.941544065f,
    0.290284677f,  0.956940336f,
    0.242980180f,  0.970031253f,
    0.195090322f,  0.980785280f,
    0.146730474f,  0.989176510f,
    0.098017140f,  0.995184727f,
    0.049067674f,  0.998795456f,
    0.000000000f,  1.000000000f,
   -0.049067674f,  0.998795456f,
   -0.098017140f,  0.995184727f,
   -0.146730474f,  0.989176510f,
   -0.195090322f,  0.980785280f,
   -0.242980180f,  0.970031253f,
   -0.290284677f,  0.956940336f,
   -0.336889853f,  0.941544065f,
   -0.382683432f,  0.923879533f,
   -0.427555093f,  0.903989293f,
   -0.471396737f,  0.881921264f,
   -0.514102744f,  0.857728610f,
   -0.555570233f,  0.831469612f,
   -0.595699304f,  0.803207531f,
   -0.634393284f,  0.773010453f,
   -0.671558955f,  0.740951125f,
   -0.707106781f,  0.707106781f,
   -0.740951125f,  0.671558955f,
   -0.773010453f,  0.634393284f,
   -0.803207531f,  0.595699304f,
   -0.831469612f,  0.555570233f,
   -0.857728610f,  0.514102744f,
   -0.881921264f,  0.471396737f,
   -0.903989293f,  0.427555093f,
   -0.923879533f,  0.382683432f,
   -0.941544065f,  0.336889853f,
   -0.956940336f,  0.290284677f,
   -0.970031253f,  0.242980180f,
   -0.980785280f,  0.195090322f,
   -0.989176510f,  0.146730474f,
   -0.995184727f,  0.098017140f,
   -0.998795456f,  0.049067674f,
   -1.000000000f,  0.000000000f,
   -0.998795456f, -0.049067674f,
   -0.995184727f, -0.098017140f,
   -0.989176510f, -0.146730474f,
   -0.980785280f, -0.195090322f,
   -0.970031253f, -0.242980180f,
   -0.956940336f, -0.290284677f,
   -0.941544065f, -0.336889853f,
   -0.923879533f, -0.382683432f,
   -0.903989293f, -0.427555093f,
   -0.881921264f, -0.471396737f,
   -0.857728610f, -0.514102744f,
   -0.831469612f, -0.555570233f,
   -0.803207531f, -0.595699304f,
   -0.773010453f, -0.634393284f,
   -0.740951125f, -0.671558955f,
   -0.707106781f, -0.707106781f,
   -0.671558955f, -0.740951125f,
   -0.634393284f, -0.773010453f,
   -0.595699304f, -0.803207531f,
   -0.555570233f, -0.831469612f,
   -0.514102744f, -0.857728610f,
   -0.471396737f, -0.881921264f,
   -0.427555093f, -0.903989293f,
   -0.382683432f, -0.923879533f,
   -0.336889853f, -0.941544065f,
   -0.290284677f, -0.956940336f,
   -0.242980180f, -0.970031253f,
   -0.195090322f, -0.980785280f,
   -0.146730474f, -0.989176510f,
   -0.098017140f, -0.995184727f,
   -0.049067674f, -0.998795456f,
   -0.000000000f, -1.000000000f,
    0.049067674f, -0.998795456f,
    0.098017140f, -0.995184727f,
    0.146730474f, -0.989176510f,
    0.195090322f, -0.980785280f,
    0.242980180f, -0.970031253f,
    0.290284677f, -0.956940336f,
    0.336889853f, -0.941544065f,
    0.382683432f, -0.923879533f,
    0.427555093f, -0.903989293f,
    0.471396737f, -0.881921264f,
    0.514102744f, -0.857728610f,
    0.555570233f, -0.831469612f,
    0.595699304f, -0.803207531f,
    0.634393284f, -0.773010453f,
    0.671558955f, -0.740951125f,
    0.707106781f, -0.707106781f,
    0.740951125f, -0.671558955f,
    0.773010453f, -0.634393284f,
    0.803207531f, -0.595699304f,
    0.831469612f, -0.555570233f,
    0.857728610f, -0.514102744f,
    0.881921264f, -0.471396737f,
    0.903989293f, -0.427555093f,
    0.923879533f, -0.382683432f,
    0.941544065f, -0.336889853f,
    0.956940336f, -0.290284677f,
    0.970031253f, -0.242980180f,
    0.980785280f, -0.195090322f,
    0.989176510f, -0.146730474f,
    0.995184727f, -0.098017140f,
    0.998795456f, -0.049067674f
};

/**
* \par
* Example code for Floating-point Twiddle factors Generation:
* \par
* <pre>for(i = 0; i< N/; i++)
* {
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 256	and PI = 3.14159265358979
* \par
* Cos and Sin values are in interleaved fashion
*
*/
const float32_t twiddleCoef_256[512] = {
    1.000000000f,  0.000000000f,
    0.999698819f,  0.024541229f,
    0.998795456f,  0.049067674f,
    0.997290457f,  0.073564564f,
    0.995184727f,  0.098017140f,
    0.992479535f,  0.122410675f,
    0.989176510f,  0.146730474f,
    0.985277642f,  0.170961889f,
    0.980785280f,  0.195090322f,
    0.975702130f,  0.219101240f,
    0.970031253f,  0.242980180f,
    0.963776066f,  0.266712757f,
    0.956940336f,  0.290284677f,
    0.949528181f,  0.313681740f,
    0.941544065f,  0.336889853f,
    0.932992799f,  0.359895037f,
    0.923879533f,  0.382683432f,
    0.914209756f,  0.405241314f,
    0.903989293f,  0.427555093f,
    0.893224301f,  0.449611330f,
    0.881921264f,  0.471396737f,
    0.870086991f,  0.492898192f,
    0.857728610f,  0.514102744f,
    0.844853565f,  0.534997620f,
    0.831469612f,  0.555570233f,
    0.817584813f,  0.575808191f,
    0.803207531f,  0.595699304f,
    0.788346428f,  0.615231591f,
    0.773010453f,  0.634393284f,
    0.757208847f,  0.653172843f,
    0.740951125f,  0.671558955f,
    0.724247083f,  0.689540545f,
    0.707106781f,  0.707106781f,
    0.689540545f,  0.724247083f,
    0.671558955f,  0.740951125f,
    0.653172843f,  0.757208847f,
    0.634393284f,  0.773010453f,
    0.615231591f,  0.788346428f,
    0.595699304f,  0.803207531f,
    0.575808191f,  0.817584813f,
    0.555570233f,  0.831469612f,
    0.534997620f,  0.844853565f,
    0.514102744f,  0.857728610f,
    0.492898192f,  0.870086991f,
    0.471396737f,  0.881921264f,
    0.449611330f,  0.893224301f,
    0.427555093f,  0.903989293f,
    0.405241314f,  0.914209756f,
    0.382683432f,  0.923879533f,
    0.359895037f,  0.932992799f,
    0.336889853f,  0.941544065f,
    0.313681740f,  0.949528181f,
    0.290284677f,  0.956940336f,
    0.266712757f,  0.963776066f,
    0.242980180f,  0.970031253f,
    0.219101240f,  0.975702130f,
    0.195090322f,  0.980785280f,
    0.170961889f,  0.985277642f,
    0.146730474f,  0.989176510f,
    0.122410675f,  0.992479535f,
    0.098017140f,  0.995184727f,
    0.073564564f,  0.997290457f,
    0.049067674f,  0.998795456f,
    0.024541229f,  0.999698819f,
    0.000000000f,  1.000000000f,
   -0.024541229f,  0.999698819f,
   -0.049067674f,  0.998795456f,
   -0.073564564f,  0.997290457f,
   -0.098017140f,  0.995184727f,
   -0.122410675f,  0.992479535f,
   -0.146730474f,  0.989176510f,
   -0.170961889f,  0.985277642f,
   -0.195090322f,  0.980785280f,
   -0.219101240f,  0.975702130f,
   -0.242980180f,  0.970031253f,
   -0.266712757f,  0.963776066f,
   -0.290284677f,  0.956940336f,
   -0.313681740f,  0.949528181f,
   -0.336889853f,  0.941544065f,
   -0.359895037f,  0.932992799f,
   -0.382683432f,  0.923879533f,
   -0.405241314f,  0.914209756f,
   -0.427555093f,  0.903989293f,
   -0.449611330f,  0.893224301f,
   -0.471396737f,  0.881921264f,
   -0.492898192f,  0.870086991f,
   -0.514102744f,  0.857728610f,
   -0.534997620f,  0.844853565f,
   -0.555570233f,  0.831469612f,
   -0.575808191f,  0.817584813f,
   -0.595699304f,  0.803207531f,
   -0.615231591f,  0.788346428f,
   -0.634393284f,  0.773010453f,
   -0.653172843f,  0.757208847f,
   -0.671558955f,  0.740951125f,
   -0.689540545f,  0.724247083f,
   -0.707106781f,  0.707106781f,
   -0.724247083f,  0.689540545f,
   -0.740951125f,  0.671558955f,
   -0.757208847f,  0.653172843f,
   -0.773010453f,  0.634393284f,
   -0.788346428f,  0.615231591f,
   -0.803207531f,  0.595699304f,
   -0.817584813f,  0.575808191f,
   -0.831469612f,  0.555570233f,
   -0.844853565f,  0.534997620f,
   -0.857728610f,  0.514102744f,
   -0.870086991f,  0.492898192f,
   -0.881921264f,  0.471396737f,
   -0.893224301f,  0.449611330f,
   -0.903989293f,  0.427555093f,
   -0.914209756f,  0.405241314f,
   -0.923879533f,  0.382683432f,
   -0.932992799f,  0.359895037f,
   -0.941544065f,  0.336889853f,
   -0.949528181f,  0.313681740f,
   -0.956940336f,  0.290284677f,
   -0.963776066f,  0.266712757f,
   -0.970031253f,  0.242980180f,
   -0.975702130f,  0.219101240f,
   -0.980785280f,  0.195090322f,
   -0.985277642f,  0.170961889f,
   -0.989176510f,  0.146730474f,
   -0.992479535f,  0.122410675f,
   -0.995184727f,  0.098017140f,
   -0.997290457f,  0.073564564f,
   -0.998795456f,  0.049067674f,
   -0.999698819f,  0.024541229f,
   -1.000000000f,  0.000000000f,
   -0.999698819f, -0.024541229f,
   -0.998795456f, -0.049067674f,
   -0.997290457f, -0.073564564f,
   -0.995184727f, -0.098017140f,
   -0.992479535f, -0.122410675f,
   -0.989176510f, -0.146730474f,
   -0.985277642f, -0.170961889f,
   -0.980785280f, -0.195090322f,
   -0.975702130f, -0.219101240f,
   -0.970031253f, -0.242980180f,
   -0.963776066f, -0.266712757f,
   -0.956940336f, -0.290284677f,
   -0.949528181f, -0.313681740f,
   -0.941544065f, -0.336889853f,
   -0.932992799f, -0.359895037f,
   -0.923879533f, -0.382683432f,
   -0.914209756f, -0.405241314f,
   -0.903989293f, -0.427555093f,
   -0.893224301f, -0.449611330f,
   -0.881921264f, -0.471396737f,
   -0.870086991f, -0.492898192f,
   -0.857728610f, -0.514102744f,
   -0.844853565f, -0.534997620f,
   -0.831469612f, -0.555570233f,
   -0.817584813f, -0.575808191f,
   -0.803207531f, -0.595699304f,
   -0.788346428f, -0.615231591f,
   -0.773010453f, -0.634393284f,
   -0.757208847f, -0.653172843f,
   -0.740951125f, -0.671558955f,
   -0.724247083f, -0.689540545f,
   -0.707106781f, -0.707106781f,
   -0.689540545f, -0.724247083f,
   -0.671558955f, -0.740951125f,
   -0.653172843f, -0.757208847f,
   -0.634393284f, -0.773010453f,
   -0.615231591f, -0.788346428f,
   -0.595699304f, -0.803207531f,
   -0.575808191f, -0.817584813f,
   -0.555570233f, -0.831469612f,
   -0.534997620f, -0.844853565f,
   -0.514102744f, -0.857728610f,
   -0.492898192f, -0.870086991f,
   -0.471396737f, -0.881921264f,
   -0.449611330f, -0.893224301f,
   -0.427555093f, -0.903989293f,
   -0.405241314f, -0.914209756f,
   -0.382683432f, -0.923879533f,
   -0.359895037f, -0.932992799f,
   -0.336889853f, -0.941544065f,
   -0.313681740f, -0.949528181f,
   -0.290284677f, -0.956940336f,
   -0.266712757f, -0.963776066f,
   -0.242980180f, -0.970031253f,
   -0.219101240f, -0.975702130f,
   -0.195090322f, -0.980785280f,
   -0.170961889f, -0.985277642f,
   -0.146730474f, -0.989176510f,
   -0.122410675f, -0.992479535f,
   -0.098017140f, -0.995184727f,
   -0.073564564f, -0.997290457f,
   -0.049067674f, -0.998795456f,
   -0.024541229f, -0.999698819f,
   -0.000000000f, -1.000000000f,
    0.024541229f, -0.999698819f,
    0.049067674f, -0.998795456f,
    0.073564564f, -0.997290457f,
    0.098017140f, -0.995184727f,
    0.122410675f, -0.992479535f,
    0.146730474f, -0.989176510f,
    0.170961889f, -0.985277642f,
    0.195090322f, -0.980785280f,
    0.219101240f, -0.975702130f,
    0.242980180f, -0.970031253f,
    0.266712757f, -0.963776066f,
    0.290284677f, -0.956940336f,
    0.313681740f, -0.949528181f,
    0.336889853f, -0.941544065f,
    0.359895037f, -0.932992799f,
    0.382683432f, -0.923879533f,
    0.405241314f, -0.914209756f,
    0.427555093f, -0.903989293f,
    0.449611330f, -0.893224301f,
    0.471396737f, -0.881921264f,
    0.492898192f, -0.870086991f,
    0.514102744f, -0.857728610f,
    0.534997620f, -0.844853565f,
    0.555570233f, -0.831469612f,
    0.575808191f, -0.817584813f,
    0.595699304f, -0.803207531f,
    0.615231591f, -0.788346428f,
    0.634393284f, -0.773010453f,
    0.653172843f, -0.757208847f,
    0.671558955f, -0.740951125f,
    0.689540545f, -0.724247083f,
    0.707106781f, -0.707106781f,
    0.724247083f, -0.689540545f,
    0.740951125f, -0.671558955f,
    0.757208847f, -0.653172843f,
    0.773010453f, -0.634393284f,
    0.788346428f, -0.615231591f,
    0.803207531f, -0.595699304f,
    0.817584813f, -0.575808191f,
    0.831469612f, -0.555570233f,
    0.844853565f, -0.534997620f,
    0.857728610f, -0.514102744f,
    0.870086991f, -0.492898192f,
    0.881921264f, -0.471396737f,
    0.893224301f, -0.449611330f,
    0.903989293f, -0.427555093f,
    0.914209756f, -0.405241314f,
    0.923879533f, -0.382683432f,
    0.932992799f, -0.359895037f,
    0.941544065f, -0.336889853f,
    0.949528181f, -0.313681740f,
    0.956940336f, -0.290284677f,
    0.963776066f, -0.266712757f,
    0.970031253f, -0.242980180f,
    0.975702130f, -0.219101240f,
    0.980785280f, -0.195090322f,
    0.985277642f, -0.170961889f,
    0.989176510f, -0.146730474f,
    0.992479535f, -0.122410675f,
    0.995184727f, -0.098017140f,
    0.997290457f, -0.073564564f,
    0.998795456f, -0.049067674f,
    0.999698819f, -0.024541229f
};

/**
* \par
* Example code for Floating-point Twiddle factors Generation:
* \par
* <pre>for(i = 0; i< N/; i++)
* {
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 512	and PI = 3.14159265358979
* \par
* Cos and Sin values are in interleaved fashion
*
*/
const float32_t twiddleCoef_512[1024] = {
    1.000000000f,  0.000000000f,
    0.999924702f,  0.012271538f,
    0.999698819f,  0.024541229f,
    0.999322385f,  0.036807223f,
    0.998795456f,  0.049067674f,
    0.998118113f,  0.061320736f,
    0.997290457f,  0.073564564f,
    0.996312612f,  0.085797312f,
    0.995184727f,  0.098017140f,
    0.993906970f,  0.110222207f,
    0.992479535f,  0.122410675f,
    0.990902635f,  0.134580709f,
    0.989176510f,  0.146730474f,
    0.987301418f,  0.158858143f,
    0.985277642f,  0.170961889f,
    0.983105487f,  0.183039888f,
    0.980785280f,  0.195090322f,
    0.978317371f,  0.207111376f,
    0.975702130f,  0.219101240f,
    0.972939952f,  0.231058108f,
    0.970031253f,  0.242980180f,
    0.966976471f,  0.254865660f,
    0.963776066f,  0.266712757f,
    0.960430519f,  0.278519689f,
    0.956940336f,  0.290284677f,
    0.953306040f,  0.302005949f,
    0.949528181f,  0.313681740f,
    0.945607325f,  0.325310292f,
    0.941544065f,  0.336889853f,
    0.937339012f,  0.348418680f,
    0.932992799f,  0.359895037f,
    0.928506080f,  0.371317194f,
    0.923879533f,  0.382683432f,
    0.919113852f,  0.393992040f,
    0.914209756f,  0.405241314f,
    0.909167983f,  0.416429560f,
    0.903989293f,  0.427555093f,
    0.898674466f,  0.438616239f,
    0.893224301f,  0.449611330f,
    0.887639620f,  0.460538711f,
    0.881921264f,  0.471396737f,
    0.876070094f,  0.482183772f,
    0.870086991f,  0.492898192f,
    0.863972856f,  0.503538384f,
    0.857728610f,  0.514102744f,
    0.851355193f,  0.524589683f,
    0.844853565f,  0.534997620f,
    0.838224706f,  0.545324988f,
    0.831469612f,  0.555570233f,
    0.824589303f,  0.565731811f,
    0.817584813f,  0.575808191f,
    0.810457198f,  0.585797857f,
    0.803207531f,  0.595699304f,
    0.795836905f,  0.605511041f,
    0.788346428f,  0.615231591f,
    0.780737229f,  0.624859488f,
    0.773010453f,  0.634393284f,
    0.765167266f,  0.643831543f,
    0.757208847f,  0.653172843f,
    0.749136395f,  0.662415778f,
    0.740951125f,  0.671558955f,
    0.732654272f,  0.680600998f,
    0.724247083f,  0.689540545f,
    0.715730825f,  0.698376249f,
    0.707106781f,  0.707106781f,
    0.698376249f,  0.715730825f,
    0.689540545f,  0.724247083f,
    0.680600998f,  0.732654272f,
    0.671558955f,  0.740951125f,
    0.662415778f,  0.749136395f,
    0.653172843f,  0.757208847f,
    0.643831543f,  0.765167266f,
    0.634393284f,  0.773010453f,
    0.624859488f,  0.780737229f,
    0.615231591f,  0.788346428f,
    0.605511041f,  0.795836905f,
    0.595699304f,  0.803207531f,
    0.585797857f,  0.810457198f,
    0.575808191f,  0.817584813f,
    0.565731811f,  0.824589303f,
    0.555570233f,  0.831469612f,
    0.545324988f,  0.838224706f,
    0.534997620f,  0.844853565f,
    0.524589683f,  0.851355193f,
    0.514102744f,  0.857728610f,
    0.503538384f,  0.863972856f,
    0.492898192f,  0.870086991f,
    0.482183772f,  0.876070094f,
    0.471396737f,  0.881921264f,
    0.460538711f,  0.887639620f,
    0.449611330f,  0.893224301f,
    0.438616239f,  0.898674466f,
    0.427555093f,  0.903989293f,
    0.416429560f,  0.909167983f,
    0.405241314f,  0.914209756f,
    0.393992040f,  0.919113852f,
    0.382683432f,  0.923879533f,
    0.371317194f,  0.928506080f,
    0.359895037f,  0.932992799f,
    0.348418680f,  0.937339012f,
    0.336889853f,  0.941544065f,
    0.325310292f,  0.945607325f,
    0.313681740f,  0.949528181f,
    0.302005949f,  0.953306040f,
    0.290284677f,  0.956940336f,
    0.278519689f,  0.960430519f,
    0.266712757f,  0.963776066f,
    0.254865660f,  0.966976471f,
    0.242980180f,  0.970031253f,
    0.231058108f,  0.972939952f,
    0.219101240f,  0.975702130f,
    0.207111376f,  0.978317371f,
    0.195090322f,  0.980785280f,
    0.183039888f,  0.983105487f,
    0.170961889f,  0.985277642f,
    0.158858143f,  0.987301418f,
    0.146730474f,  0.989176510f,
    0.134580709f,  0.990902635f,
    0.122410675f,  0.992479535f,
    0.110222207f,  0.993906970f,
    0.098017140f,  0.995184727f,
    0.085797312f,  0.996312612f,
    0.073564564f,  0.997290457f,
    0.061320736f,  0.998118113f,
    0.049067674f,  0.998795456f,
    0.036807223f,  0.999322385f,
    0.024541229f,  0.999698819f,
    0.012271538f,  0.999924702f,
    0.000000000f,  1.000000000f,
   -0.012271538f,  0.999924702f,
   -0.024541229f,  0.999698819f,
   -0.036807223f,  0.999322385f,
   -0.049067674f,  0.998795456f,
   -0.061320736f,  0.998118113f,
   -0.073564564f,  0.997290457f,
   -0.085797312f,  0.996312612f,
   -0.098017140f,  0.995184727f,
   -0.110222207f,  0.993906970f,
   -0.122410675f,  0.992479535f,
   -0.134580709f,  0.990902635f,
   -0.146730474f,  0.989176510f,
   -0.158858143f,  0.987301418f,
   -0.170961889f,  0.985277642f,
   -0.183039888f,  0.983105487f,
   -0.195090322f,  0.980785280f,
   -0.207111376f,  0.978317371f,
   -0.219101240f,  0.975702130f,
   -0.231058108f,  0.972939952f,
   -0.242980180f,  0.970031253f,
   -0.254865660f,  0.966976471f,
   -0.266712757f,  0.963776066f,
   -0.278519689f,  0.960430519f,
   -0.290284677f,  0.956940336f,
   -0.302005949f,  0.953306040f,
   -0.313681740f,  0.949528181f,
   -0.325310292f,  0.945607325f,
   -0.336889853f,  0.941544065f,
   -0.348418680f,  0.937339012f,
   -0.359895037f,  0.932992799f,
   -0.371317194f,  0.928506080f,
   -0.382683432f,  0.923879533f,
   -0.393992040f,  0.919113852f,
   -0.405241314f,  0.914209756f,
   -0.416429560f,  0.909167983f,
   -0.427555093f,  0.903989293f,
   -0.438616239f,  0.898674466f,
   -0.449611330f,  0.893224301f,
   -0.460538711f,  0.887639620f,
   -0.471396737f,  0.881921264f,
   -0.482183772f,  0.876070094f,
   -0.492898192f,  0.870086991f,
   -0.503538384f,  0.863972856f,
   -0.514102744f,  0.857728610f,
   -0.524589683f,  0.851355193f,
   -0.534997620f,  0.844853565f,
   -0.545324988f,  0.838224706f,
   -0.555570233f,  0.831469612f,
   -0.565731811f,  0.824589303f,
   -0.575808191f,  0.817584813f,
   -0.585797857f,  0.810457198f,
   -0.595699304f,  0.803207531f,
   -0.605511041f,  0.795836905f,
   -0.615231591f,  0.788346428f,
   -0.624859488f,  0.780737229f,
   -0.634393284f,  0.773010453f,
   -0.643831543f,  0.765167266f,
   -0.653172843f,  0.757208847f,
   -0.662415778f,  0.749136395f,
   -0.671558955f,  0.740951125f,
   -0.680600998f,  0.732654272f,
   -0.689540545f,  0.724247083f,
   -0.698376249f,  0.715730825f,
   -0.707106781f,  0.707106781f,
   -0.715730825f,  0.698376249f,
   -0.724247083f,  0.689540545f,
   -0.732654272f,  0.680600998f,
   -0.740951125f,  0.671558955f,
   -0.749136395f,  0.662415778f,
   -0.757208847f,  0.653172843f,
   -0.765167266f,  0.643831543f,
   -0.773010453f,  0.634393284f,
   -0.780737229f,  0.624859488f,
   -0.788346428f,  0.615231591f,
   -0.795836905f,  0.605511041f,
   -0.803207531f,  0.595699304f,
   -0.810457198f,  0.585797857f,
   -0.817584813f,  0.575808191f,
   -0.824589303f,  0.565731811f,
   -0.831469612f,  0.555570233f,
   -0.838224706f,  0.545324988f,
   -0.844853565f,  0.534997620f,
   -0.851355193f,  0.524589683f,
   -0.857728610f,  0.514102744f,
   -0.863972856f,  0.503538384f,
   -0.870086991f,  0.492898192f,
   -0.876070094f,  0.482183772f,
   -0.881921264f,  0.471396737f,
   -0.887639620f,  0.460538711f,
   -0.893224301f,  0.449611330f,
   -0.898674466f,  0.438616239f,
   -0.903989293f,  0.427555093f,
   -0.909167983f,  0.416429560f,
   -0.914209756f,  0.405241314f,
   -0.919113852f,  0.393992040f,
   -0.923879533f,  0.382683432f,
   -0.928506080f,  0.371317194f,
   -0.932992799f,  0.359895037f,
   -0.937339012f,  0.348418680f,
   -0.941544065f,  0.336889853f,
   -0.945607325f,  0.325310292f,
   -0.949528181f,  0.313681740f,
   -0.953306040f,  0.302005949f,
   -0.956940336f,  0.290284677f,
   -0.960430519f,  0.278519689f,
   -0.963776066f,  0.266712757f,
   -0.966976471f,  0.254865660f,
   -0.970031253f,  0.242980180f,
   -0.972939952f,  0.231058108f,
   -0.975702130f,  0.219101240f,
   -0.978317371f,  0.207111376f,
   -0.980785280f,  0.195090322f,
   -0.983105487f,  0.183039888f,
   -0.985277642f,  0.170961889f,
   -0.987301418f,  0.158858143f,
   -0.989176510f,  0.146730474f,
   -0.990902635f,  0.134580709f,
   -0.992479535f,  0.122410675f,
   -0.993906970f,  0.110222207f,
   -0.995184727f,  0.098017140f,
   -0.996312612f,  0.085797312f,
   -0.997290457f,  0.073564564f,
   -0.998118113f,  0.061320736f,
   -0.998795456f,  0.049067674f,
   -0.999322385f,  0.036807223f,
   -0.999698819f,  0.024541229f,
   -0.999924702f,  0.012271538f,
   -1.000000000f,  0.000000000f,
   -0.999924702f, -0.012271538f,
   -0.999698819f, -0.024541229f,
   -0.999322385f, -0.036807223f,
   -0.998795456f, -0.049067674f,
   -0.998118113f, -0.061320736f,
   -0.997290457f, -0.073564564f,
   -0.996312612f, -0.085797312f,
   -0.995184727f, -0.098017140f,
   -0.993906970f, -0.110222207f,
   -0.992479535f, -0.122410675f,
   -0.990902635f, -0.134580709f,
   -0.989176510f, -0.146730474f,
   -0.987301418f, -0.158858143f,
   -0.985277642f, -0.170961889f,
   -0.983105487f, -0.183039888f,
   -0.980785280f, -0.195090322f,
   -0.978317371f, -0.207111376f,
   -0.975702130f, -0.219101240f,
   -0.972939952f, -0.231058108f,
   -0.970031253f, -0.242980180f,
   -0.966976471f, -0.254865660f,
   -0.963776066f, -0.266712757f,
   -0.960430519f, -0.278519689f,
   -0.956940336f, -0.290284677f,
   -0.953306040f, -0.302005949f,
   -0.949528181f, -0.313681740f,
   -0.945607325f, -0.325310292f,
   -0.941544065f, -0.336889853f,
   -0.937339012f, -0.348418680f,
   -0.932992799f, -0.359895037f,
   -0.928506080f, -0.371317194f,
   -0.923879533f, -0.382683432f,
   -0.919113852f, -0.393992040f,
   -0.914209756f, -0.405241314f,
   -0.909167983f, -0.416429560f,
   -0.903989293f, -0.427555093f,
   -0.898674466f, -0.438616239f,
   -0.893224301f, -0.449611330f,
   -0.887639620f, -0.460538711f,
   -0.881921264f, -0.471396737f,
   -0.876070094f, -0.482183772f,
   -0.870086991f, -0.492898192f,
   -0.863972856f, -0.503538384f,
   -0.857728610f, -0.514102744f,
   -0.851355193f, -0.524589683f,
   -0.844853565f, -0.534997620f,
   -0.838224706f, -0.545324988f,
   -0.831469612f, -0.555570233f,
   -0.824589303f, -0.565731811f,
   -0.817584813f, -0.575808191f,
   -0.810457198f, -0.585797857f,
   -0.803207531f, -0.595699304f,
   -0.795836905f, -0.605511041f,
   -0.788346428f, -0.615231591f,
   -0.780737229f, -0.624859488f,
   -0.773010453f, -0.634393284f,
   -0.765167266f, -0.643831543f,
   -0.757208847f, -0.653172843f,
   -0.749136395f, -0.662415778f,
   -0.740951125f, -0.671558955f,
   -0.732654272f, -0.680600998f,
   -0.724247083f, -0.689540545f,
   -0.715730825f, -0.698376249f,
   -0.707106781f, -0.707106781f,
   -0.698376249f, -0.715730825f,
   -0.689540545f, -0.724247083f,
   -0.680600998f, -0.732654272f,
   -0.671558955f, -0.740951125f,
   -0.662415778f, -0.749136395f,
   -0.653172843f, -0.757208847f,
   -0.643831543f, -0.765167266f,
   -0.634393284f, -0.773010453f,
   -0.624859488f, -0.780737229f,
   -0.615231591f, -0.788346428f,
   -0.605511041f, -0.795836905f,
   -0.595699304f, -0.803207531f,
   -0.585797857f, -0.810457198f,
   -0.575808191f, -0.817584813f,
   -0.565731811f, -0.824589303f,
   -0.555570233f, -0.831469612f,
   -0.545324988f, -0.838224706f,
   -0.534997620f, -0.844853565f,
   -0.524589683f, -0.851355193f,
   -0.514102744f, -0.857728610f,
   -0.503538384f, -0.863972856f,
   -0.492898192f, -0.870086991f,
   -0.482183772f, -0.876070094f,
   -0.471396737f, -0.881921264f,
   -0.460538711f, -0.887639620f,
   -0.449611330f, -0.893224301f,
   -0.438616239f, -0.898674466f,
   -0.427555093f, -0.903989293f,
   -0.416429560f, -0.909167983f,
   -0.405241314f, -0.914209756f,
   -0.393992040f, -0.919113852f,
   -0.382683432f, -0.923879533f,
   -0.371317194f, -0.928506080f,
   -0.359895037f, -0.932992799f,
   -0.348418680f, -0.937339012f,
   -0.336889853f, -0.941544065f,
   -0.325310292f, -0.945607325f,
   -0.313681740f, -0.949528181f,
   -0.302005949f, -0.953306040f,
   -0.290284677f, -0.956940336f,
   -0.278519689f, -0.960430519f,
   -0.266712757f, -0.963776066f,
   -0.254865660f, -0.966976471f,
   -0.242980180f, -0.970031253f,
   -0.231058108f, -0.972939952f,
   -0.219101240f, -0.975702130f,
   -0.207111376f, -0.978317371f,
   -0.195090322f, -0.980785280f,
   -0.183039888f, -0.983105487f,
   -0.170961889f, -0.985277642f,
   -0.158858143f, -0.987301418f,
   -0.146730474f, -0.989176510f,
   -0.134580709f, -0.990902635f,
   -0.122410675f, -0.992479535f,
   -0.110222207f, -0.993906970f,
   -0.098017140f, -0.995184727f,
   -0.085797312f, -0.996312612f,
   -0.073564564f, -0.997290457f,
   -0.061320736f, -0.998118113f,
   -0.049067674f, -0.998795456f,
   -0.036807223f, -0.999322385f,
   -0.024541229f, -0.999698819f,
   -0.012271538f, -0.999924702f,
   -0.000000000f, -1.000000000f,
    0.012271538f, -0.999924702f,
    0.024541229f, -0.999698819f,
    0.036807223f, -0.999322385f,
    0.049067674f, -0.998795456f,
    0.061320736f, -0.998118113f,
    0.073564564f, -0.997290457f,
    0.085797312f, -0.996312612f,
    0.098017140f, -0.995184727f,
    0.110222207f, -0.993906970f,
    0.122410675f, -0.992479535f,
    0.134580709f, -0.990902635f,
    0.146730474f, -0.989176510f,
    0.158858143f, -0.987301418f,
    0.170961889f, -0.985277642f,
    0.183039888f, -0.983105487f,
    0.195090322f, -0.980785280f,
    0.207111376f, -0.978317371f,
    0.219101240f, -0.975702130f,
    0.231058108f, -0.972939952f,
    0.242980180f, -0.970031253f,
    0.254865660f, -0.966976471f,
    0.266712757f, -0.963776066f,
    0.278519689f, -0.960430519f,
    0.290284677f, -0.956940336f,
    0.302005949f, -0.953306040f,
    0.313681740f, -0.949528181f,
    0.325310292f, -0.945607325f,
    0.336889853f, -0.941544065f,
    0.348418680f, -0.937339012f,
    0.359895037f, -0.932992799f,
    0.371317194f, -0.928506080f,
    0.382683432f, -0.923879533f,
    0.393992040f, -0.919113852f,
    0.405241314f, -0.914209756f,
    0.416429560f, -0.909167983f,
    0.427555093f, -0.903989293f,
    0.438616239f, -0.898674466f,
    0.449611330f, -0.893224301f,
    0.460538711f, -0.887639620f,
    0.471396737f, -0.881921264f,
    0.482183772f, -0.876070094f,
    0.492898192f, -0.870086991f,
    0.503538384f, -0.863972856f,
    0.514102744f, -0.857728610f,
    0.524589683f, -0.851355193f,
    0.534997620f, -0.844853565f,
    0.545324988f, -0.838224706f,
    0.555570233f, -0.831469612f,
    0.565731811f, -0.824589303f,
    0.575808191f, -0.817584813f,
    0.585797857f, -0.810457198f,
    0.595699304f, -0.803207531f,
    0.605511041f, -0.795836905f,
    0.615231591f, -0.788346428f,
    0.624859488f, -0.780737229f,
    0.634393284f, -0.773010453f,
    0.643831543f, -0.765167266f,
    0.653172843f, -0.757208847f,
    0.662415778f, -0.749136395f,
    0.671558955f, -0.740951125f,
    0.680600998f, -0.732654272f,
    0.689540545f, -0.724247083f,
    0.698376249f, -0.715730825f,
    0.707106781f, -0.707106781f,
    0.715730825f, -0.698376249f,
    0.724247083f, -0.689540545f,
    0.732654272f, -0.680600998f,
    0.740951125f, -0.671558955f,
    0.749136395f, -0.662415778f,
    0.757208847f, -0.653172843f,
    0.765167266f, -0.643831543f,
    0.773010453f, -0.634393284f,
    0.780737229f, -0.624859488f,
    0.788346428f, -0.615231591f,
    0.795836905f, -0.605511041f,
    0.803207531f, -0.595699304f,
    0.810457198f, -0.585797857f,
    0.817584813f, -0.575808191f,
    0.824589303f, -0.565731811f,
    0.831469612f, -0.555570233f,
    0.838224706f, -0.545324988f,
    0.844853565f, -0.534997620f,
    0.851355193f, -0.524589683f,
    0.857728610f, -0.514102744f,
    0.863972856f, -0.503538384f,
    0.870086991f, -0.492898192f,
    0.876070094f, -0.482183772f,
    0.881921264f, -0.471396737f,
    0.887639620f, -0.460538711f,
    0.893224301f, -0.449611330f,
    0.898674466f, -0.438616239f,
    0.903989293f, -0.427555093f,
    0.909167983f, -0.416429560f,
    0.914209756f, -0.405241314f,
    0.919113852f, -0.393992040f,
    0.923879533f, -0.382683432f,
    0.928506080f, -0.371317194f,
    0.932992799f, -0.359895037f,
    0.937339012f, -0.348418680f,
    0.941544065f, -0.336889853f,
    0.945607325f, -0.325310292f,
    0.949528181f, -0.313681740f,
    0.953306040f, -0.302005949f,
    0.956940336f, -0.290284677f,
    0.960430519f, -0.278519689f,
    0.963776066f, -0.266712757f,
    0.966976471f, -0.254865660f,
    0.970031253f, -0.242980180f,
    0.972939952f, -0.231058108f,
    0.975702130f, -0.219101240f,
    0.978317371f, -0.207111376f,
    0.980785280f, -0.195090322f,
    0.983105487f, -0.183039888f,
    0.985277642f, -0.170961889f,
    0.987301418f, -0.158858143f,
    0.989176510f, -0.146730474f,
    0.990902635f, -0.134580709f,
    0.992479535f, -0.122410675f,
    0.993906970f, -0.110222207f,
    0.995184727f, -0.098017140f,
    0.996312612f, -0.085797312f,
    0.997290457f, -0.073564564f,
    0.998118113f, -0.061320736f,
    0.998795456f, -0.049067674f,
    0.999322385f, -0.036807223f,
    0.999698819f, -0.024541229f,
    0.999924702f, -0.012271538f
};
/**
* \par
* Example code for Floating-point Twiddle factors Generation:
* \par
* <pre>for(i = 0; i< N/; i++)
* {
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 1024	and PI = 3.14159265358979
* \par
* Cos and Sin values are in interleaved fashion
*
*/
#if 0  /* AmebaD already define this array in DROM */
const float32_t twiddleCoef_1024[2048] = {
    1.000000000f,  0.000000000f,
    0.999981175f,  0.006135885f,
    0.999924702f,  0.012271538f,
    0.999830582f,  0.018406730f,
    0.999698819f,  0.024541229f,
    0.999529418f,  0.030674803f,
    0.999322385f,  0.036807223f,
    0.999077728f,  0.042938257f,
    0.998795456f,  0.049067674f,
    0.998475581f,  0.055195244f,
    0.998118113f,  0.061320736f,
    0.997723067f,  0.067443920f,
    0.997290457f,  0.073564564f,
    0.996820299f,  0.079682438f,
    0.996312612f,  0.085797312f,
    0.995767414f,  0.091908956f,
    0.995184727f,  0.098017140f,
    0.994564571f,  0.104121634f,
    0.993906970f,  0.110222207f,
    0.993211949f,  0.116318631f,
    0.992479535f,  0.122410675f,
    0.991709754f,  0.128498111f,
    0.990902635f,  0.134580709f,
    0.990058210f,  0.140658239f,
    0.989176510f,  0.146730474f,
    0.988257568f,  0.152797185f,
    0.987301418f,  0.158858143f,
    0.986308097f,  0.164913120f,
    0.985277642f,  0.170961889f,
    0.984210092f,  0.177004220f,
    0.983105487f,  0.183039888f,
    0.981963869f,  0.189068664f,
    0.980785280f,  0.195090322f,
    0.979569766f,  0.201104635f,
    0.978317371f,  0.207111376f,
    0.977028143f,  0.213110320f,
    0.975702130f,  0.219101240f,
    0.974339383f,  0.225083911f,
    0.972939952f,  0.231058108f,
    0.971503891f,  0.237023606f,
    0.970031253f,  0.242980180f,
    0.968522094f,  0.248927606f,
    0.966976471f,  0.254865660f,
    0.965394442f,  0.260794118f,
    0.963776066f,  0.266712757f,
    0.962121404f,  0.272621355f,
    0.960430519f,  0.278519689f,
    0.958703475f,  0.284407537f,
    0.956940336f,  0.290284677f,
    0.955141168f,  0.296150888f,
    0.953306040f,  0.302005949f,
    0.951435021f,  0.307849640f,
    0.949528181f,  0.313681740f,
    0.947585591f,  0.319502031f,
    0.945607325f,  0.325310292f,
    0.943593458f,  0.331106306f,
    0.941544065f,  0.336889853f,
    0.939459224f,  0.342660717f,
    0.937339012f,  0.348418680f,
    0.935183510f,  0.354163525f,
    0.932992799f,  0.359895037f,
    0.930766961f,  0.365612998f,
    0.928506080f,  0.371317194f,
    0.926210242f,  0.377007410f,
    0.923879533f,  0.382683432f,
    0.921514039f,  0.388345047f,
    0.919113852f,  0.393992040f,
    0.916679060f,  0.399624200f,
    0.914209756f,  0.405241314f,
    0.911706032f,  0.410843171f,
    0.909167983f,  0.416429560f,
    0.906595705f,  0.422000271f,
    0.903989293f,  0.427555093f,
    0.901348847f,  0.433093819f,
    0.898674466f,  0.438616239f,
    0.895966250f,  0.444122145f,
    0.893224301f,  0.449611330f,
    0.890448723f,  0.455083587f,
    0.887639620f,  0.460538711f,
    0.884797098f,  0.465976496f,
    0.881921264f,  0.471396737f,
    0.879012226f,  0.476799230f,
    0.876070094f,  0.482183772f,
    0.873094978f,  0.487550160f,
    0.870086991f,  0.492898192f,
    0.867046246f,  0.498227667f,
    0.863972856f,  0.503538384f,
    0.860866939f,  0.508830143f,
    0.857728610f,  0.514102744f,
    0.854557988f,  0.519355990f,
    0.851355193f,  0.524589683f,
    0.848120345f,  0.529803625f,
    0.844853565f,  0.534997620f,
    0.841554977f,  0.540171473f,
    0.838224706f,  0.545324988f,
    0.834862875f,  0.550457973f,
    0.831469612f,  0.555570233f,
    0.828045045f,  0.560661576f,
    0.824589303f,  0.565731811f,
    0.821102515f,  0.570780746f,
    0.817584813f,  0.575808191f,
    0.814036330f,  0.580813958f,
    0.810457198f,  0.585797857f,
    0.806847554f,  0.590759702f,
    0.803207531f,  0.595699304f,
    0.799537269f,  0.600616479f,
    0.795836905f,  0.605511041f,
    0.792106577f,  0.610382806f,
    0.788346428f,  0.615231591f,
    0.784556597f,  0.620057212f,
    0.780737229f,  0.624859488f,
    0.776888466f,  0.629638239f,
    0.773010453f,  0.634393284f,
    0.769103338f,  0.639124445f,
    0.765167266f,  0.643831543f,
    0.761202385f,  0.648514401f,
    0.757208847f,  0.653172843f,
    0.753186799f,  0.657806693f,
    0.749136395f,  0.662415778f,
    0.745057785f,  0.666999922f,
    0.740951125f,  0.671558955f,
    0.736816569f,  0.676092704f,
    0.732654272f,  0.680600998f,
    0.728464390f,  0.685083668f,
    0.724247083f,  0.689540545f,
    0.720002508f,  0.693971461f,
    0.715730825f,  0.698376249f,
    0.711432196f,  0.702754744f,
    0.707106781f,  0.707106781f,
    0.702754744f,  0.711432196f,
    0.698376249f,  0.715730825f,
    0.693971461f,  0.720002508f,
    0.689540545f,  0.724247083f,
    0.685083668f,  0.728464390f,
    0.680600998f,  0.732654272f,
    0.676092704f,  0.736816569f,
    0.671558955f,  0.740951125f,
    0.666999922f,  0.745057785f,
    0.662415778f,  0.749136395f,
    0.657806693f,  0.753186799f,
    0.653172843f,  0.757208847f,
    0.648514401f,  0.761202385f,
    0.643831543f,  0.765167266f,
    0.639124445f,  0.769103338f,
    0.634393284f,  0.773010453f,
    0.629638239f,  0.776888466f,
    0.624859488f,  0.780737229f,
    0.620057212f,  0.784556597f,
    0.615231591f,  0.788346428f,
    0.610382806f,  0.792106577f,
    0.605511041f,  0.795836905f,
    0.600616479f,  0.799537269f,
    0.595699304f,  0.803207531f,
    0.590759702f,  0.806847554f,
    0.585797857f,  0.810457198f,
    0.580813958f,  0.814036330f,
    0.575808191f,  0.817584813f,
    0.570780746f,  0.821102515f,
    0.565731811f,  0.824589303f,
    0.560661576f,  0.828045045f,
    0.555570233f,  0.831469612f,
    0.550457973f,  0.834862875f,
    0.545324988f,  0.838224706f,
    0.540171473f,  0.841554977f,
    0.534997620f,  0.844853565f,
    0.529803625f,  0.848120345f,
    0.524589683f,  0.851355193f,
    0.519355990f,  0.854557988f,
    0.514102744f,  0.857728610f,
    0.508830143f,  0.860866939f,
    0.503538384f,  0.863972856f,
    0.498227667f,  0.867046246f,
    0.492898192f,  0.870086991f,
    0.487550160f,  0.873094978f,
    0.482183772f,  0.876070094f,
    0.476799230f,  0.879012226f,
    0.471396737f,  0.881921264f,
    0.465976496f,  0.884797098f,
    0.460538711f,  0.887639620f,
    0.455083587f,  0.890448723f,
    0.449611330f,  0.893224301f,
    0.444122145f,  0.895966250f,
    0.438616239f,  0.898674466f,
    0.433093819f,  0.901348847f,
    0.427555093f,  0.903989293f,
    0.422000271f,  0.906595705f,
    0.416429560f,  0.909167983f,
    0.410843171f,  0.911706032f,
    0.405241314f,  0.914209756f,
    0.399624200f,  0.916679060f,
    0.393992040f,  0.919113852f,
    0.388345047f,  0.921514039f,
    0.382683432f,  0.923879533f,
    0.377007410f,  0.926210242f,
    0.371317194f,  0.928506080f,
    0.365612998f,  0.930766961f,
    0.359895037f,  0.932992799f,
    0.354163525f,  0.935183510f,
    0.348418680f,  0.937339012f,
    0.342660717f,  0.939459224f,
    0.336889853f,  0.941544065f,
    0.331106306f,  0.943593458f,
    0.325310292f,  0.945607325f,
    0.319502031f,  0.947585591f,
    0.313681740f,  0.949528181f,
    0.307849640f,  0.951435021f,
    0.302005949f,  0.953306040f,
    0.296150888f,  0.955141168f,
    0.290284677f,  0.956940336f,
    0.284407537f,  0.958703475f,
    0.278519689f,  0.960430519f,
    0.272621355f,  0.962121404f,
    0.266712757f,  0.963776066f,
    0.260794118f,  0.965394442f,
    0.254865660f,  0.966976471f,
    0.248927606f,  0.968522094f,
    0.242980180f,  0.970031253f,
    0.237023606f,  0.971503891f,
    0.231058108f,  0.972939952f,
    0.225083911f,  0.974339383f,
    0.219101240f,  0.975702130f,
    0.213110320f,  0.977028143f,
    0.207111376f,  0.978317371f,
    0.201104635f,  0.979569766f,
    0.195090322f,  0.980785280f,
    0.189068664f,  0.981963869f,
    0.183039888f,  0.983105487f,
    0.177004220f,  0.984210092f,
    0.170961889f,  0.985277642f,
    0.164913120f,  0.986308097f,
    0.158858143f,  0.987301418f,
    0.152797185f,  0.988257568f,
    0.146730474f,  0.989176510f,
    0.140658239f,  0.990058210f,
    0.134580709f,  0.990902635f,
    0.128498111f,  0.991709754f,
    0.122410675f,  0.992479535f,
    0.116318631f,  0.993211949f,
    0.110222207f,  0.993906970f,
    0.104121634f,  0.994564571f,
    0.098017140f,  0.995184727f,
    0.091908956f,  0.995767414f,
    0.085797312f,  0.996312612f,
    0.079682438f,  0.996820299f,
    0.073564564f,  0.997290457f,
    0.067443920f,  0.997723067f,
    0.061320736f,  0.998118113f,
    0.055195244f,  0.998475581f,
    0.049067674f,  0.998795456f,
    0.042938257f,  0.999077728f,
    0.036807223f,  0.999322385f,
    0.030674803f,  0.999529418f,
    0.024541229f,  0.999698819f,
    0.018406730f,  0.999830582f,
    0.012271538f,  0.999924702f,
    0.006135885f,  0.999981175f,
    0.000000000f,  1.000000000f,
   -0.006135885f,  0.999981175f,
   -0.012271538f,  0.999924702f,
   -0.018406730f,  0.999830582f,
   -0.024541229f,  0.999698819f,
   -0.030674803f,  0.999529418f,
   -0.036807223f,  0.999322385f,
   -0.042938257f,  0.999077728f,
   -0.049067674f,  0.998795456f,
   -0.055195244f,  0.998475581f,
   -0.061320736f,  0.998118113f,
   -0.067443920f,  0.997723067f,
   -0.073564564f,  0.997290457f,
   -0.079682438f,  0.996820299f,
   -0.085797312f,  0.996312612f,
   -0.091908956f,  0.995767414f,
   -0.098017140f,  0.995184727f,
   -0.104121634f,  0.994564571f,
   -0.110222207f,  0.993906970f,
   -0.116318631f,  0.993211949f,
   -0.122410675f,  0.992479535f,
   -0.128498111f,  0.991709754f,
   -0.134580709f,  0.990902635f,
   -0.140658239f,  0.990058210f,
   -0.146730474f,  0.989176510f,
   -0.152797185f,  0.988257568f,
   -0.158858143f,  0.987301418f,
   -0.164913120f,  0.986308097f,
   -0.170961889f,  0.985277642f,
   -0.177004220f,  0.984210092f,
   -0.183039888f,  0.983105487f,
   -0.189068664f,  0.981963869f,
   -0.195090322f,  0.980785280f,
   -0.201104635f,  0.979569766f,
   -0.207111376f,  0.978317371f,
   -0.213110320f,  0.977028143f,
   -0.219101240f,  0.975702130f,
   -0.225083911f,  0.974339383f,
   -0.231058108f,  0.972939952f,
   -0.237023606f,  0.971503891f,
   -0.242980180f,  0.970031253f,
   -0.248927606f,  0.968522094f,
   -0.254865660f,  0.966976471f,
   -0.260794118f,  0.965394442f,
   -0.266712757f,  0.963776066f,
   -0.272621355f,  0.962121404f,
   -0.278519689f,  0.960430519f,
   -0.284407537f,  0.958703475f,
   -0.290284677f,  0.956940336f,
   -0.296150888f,  0.955141168f,
   -0.302005949f,  0.953306040f,
   -0.307849640f,  0.951435021f,
   -0.313681740f,  0.949528181f,
   -0.319502031f,  0.947585591f,
   -0.325310292f,  0.945607325f,
   -0.331106306f,  0.943593458f,
   -0.336889853f,  0.941544065f,
   -0.342660717f,  0.939459224f,
   -0.348418680f,  0.937339012f,
   -0.354163525f,  0.935183510f,
   -0.359895037f,  0.932992799f,
   -0.365612998f,  0.930766961f,
   -0.371317194f,  0.928506080f,
   -0.377007410f,  0.926210242f,
   -0.382683432f,  0.923879533f,
   -0.388345047f,  0.921514039f,
   -0.393992040f,  0.919113852f,
   -0.399624200f,  0.916679060f,
   -0.405241314f,  0.914209756f,
   -0.410843171f,  0.911706032f,
   -0.416429560f,  0.909167983f,
   -0.422000271f,  0.906595705f,
   -0.427555093f,  0.903989293f,
   -0.433093819f,  0.901348847f,
   -0.438616239f,  0.898674466f,
   -0.444122145f,  0.895966250f,
   -0.449611330f,  0.893224301f,
   -0.455083587f,  0.890448723f,
   -0.460538711f,  0.887639620f,
   -0.465976496f,  0.884797098f,
   -0.471396737f,  0.881921264f,
   -0.476799230f,  0.879012226f,
   -0.482183772f,  0.876070094f,
   -0.487550160f,  0.873094978f,
   -0.492898192f,  0.870086991f,
   -0.498227667f,  0.867046246f,
   -0.503538384f,  0.863972856f,
   -0.508830143f,  0.860866939f,
   -0.514102744f,  0.857728610f,
   -0.519355990f,  0.854557988f,
   -0.524589683f,  0.851355193f,
   -0.529803625f,  0.848120345f,
   -0.534997620f,  0.844853565f,
   -0.540171473f,  0.841554977f,
   -0.545324988f,  0.838224706f,
   -0.550457973f,  0.834862875f,
   -0.555570233f,  0.831469612f,
   -0.560661576f,  0.828045045f,
   -0.565731811f,  0.824589303f,
   -0.570780746f,  0.821102515f,
   -0.575808191f,  0.817584813f,
   -0.580813958f,  0.814036330f,
   -0.585797857f,  0.810457198f,
   -0.590759702f,  0.806847554f,
   -0.595699304f,  0.803207531f,
   -0.600616479f,  0.799537269f,
   -0.605511041f,  0.795836905f,
   -0.610382806f,  0.792106577f,
   -0.615231591f,  0.788346428f,
   -0.620057212f,  0.784556597f,
   -0.624859488f,  0.780737229f,
   -0.629638239f,  0.776888466f,
   -0.634393284f,  0.773010453f,
   -0.639124445f,  0.769103338f,
   -0.643831543f,  0.765167266f,
   -0.648514401f,  0.761202385f,
   -0.653172843f,  0.757208847f,
   -0.657806693f,  0.753186799f,
   -0.662415778f,  0.749136395f,
   -0.666999922f,  0.745057785f,
   -0.671558955f,  0.740951125f,
   -0.676092704f,  0.736816569f,
   -0.680600998f,  0.732654272f,
   -0.685083668f,  0.728464390f,
   -0.689540545f,  0.724247083f,
   -0.693971461f,  0.720002508f,
   -0.698376249f,  0.715730825f,
   -0.702754744f,  0.711432196f,
   -0.707106781f,  0.707106781f,
   -0.711432196f,  0.702754744f,
   -0.715730825f,  0.698376249f,
   -0.720002508f,  0.693971461f,
   -0.724247083f,  0.689540545f,
   -0.728464390f,  0.685083668f,
   -0.732654272f,  0.680600998f,
   -0.736816569f,  0.676092704f,
   -0.740951125f,  0.671558955f,
   -0.745057785f,  0.666999922f,
   -0.749136395f,  0.662415778f,
   -0.753186799f,  0.657806693f,
   -0.757208847f,  0.653172843f,
   -0.761202385f,  0.648514401f,
   -0.765167266f,  0.643831543f,
   -0.769103338f,  0.639124445f,
   -0.773010453f,  0.634393284f,
   -0.776888466f,  0.629638239f,
   -0.780737229f,  0.624859488f,
   -0.784556597f,  0.620057212f,
   -0.788346428f,  0.615231591f,
   -0.792106577f,  0.610382806f,
   -0.795836905f,  0.605511041f,
   -0.799537269f,  0.600616479f,
   -0.803207531f,  0.595699304f,
   -0.806847554f,  0.590759702f,
   -0.810457198f,  0.585797857f,
   -0.814036330f,  0.580813958f,
   -0.817584813f,  0.575808191f,
   -0.821102515f,  0.570780746f,
   -0.824589303f,  0.565731811f,
   -0.828045045f,  0.560661576f,
   -0.831469612f,  0.555570233f,
   -0.834862875f,  0.550457973f,
   -0.838224706f,  0.545324988f,
   -0.841554977f,  0.540171473f,
   -0.844853565f,  0.534997620f,
   -0.848120345f,  0.529803625f,
   -0.851355193f,  0.524589683f,
   -0.854557988f,  0.519355990f,
   -0.857728610f,  0.514102744f,
   -0.860866939f,  0.508830143f,
   -0.863972856f,  0.503538384f,
   -0.867046246f,  0.498227667f,
   -0.870086991f,  0.492898192f,
   -0.873094978f,  0.487550160f,
   -0.876070094f,  0.482183772f,
   -0.879012226f,  0.476799230f,
   -0.881921264f,  0.471396737f,
   -0.884797098f,  0.465976496f,
   -0.887639620f,  0.460538711f,
   -0.890448723f,  0.455083587f,
   -0.893224301f,  0.449611330f,
   -0.895966250f,  0.444122145f,
   -0.898674466f,  0.438616239f,
   -0.901348847f,  0.433093819f,
   -0.903989293f,  0.427555093f,
   -0.906595705f,  0.422000271f,
   -0.909167983f,  0.416429560f,
   -0.911706032f,  0.410843171f,
   -0.914209756f,  0.405241314f,
   -0.916679060f,  0.399624200f,
   -0.919113852f,  0.393992040f,
   -0.921514039f,  0.388345047f,
   -0.923879533f,  0.382683432f,
   -0.926210242f,  0.377007410f,
   -0.928506080f,  0.371317194f,
   -0.930766961f,  0.365612998f,
   -0.932992799f,  0.359895037f,
   -0.935183510f,  0.354163525f,
   -0.937339012f,  0.348418680f,
   -0.939459224f,  0.342660717f,
   -0.941544065f,  0.336889853f,
   -0.943593458f,  0.331106306f,
   -0.945607325f,  0.325310292f,
   -0.947585591f,  0.319502031f,
   -0.949528181f,  0.313681740f,
   -0.951435021f,  0.307849640f,
   -0.953306040f,  0.302005949f,
   -0.955141168f,  0.296150888f,
   -0.956940336f,  0.290284677f,
   -0.958703475f,  0.284407537f,
   -0.960430519f,  0.278519689f,
   -0.962121404f,  0.272621355f,
   -0.963776066f,  0.266712757f,
   -0.965394442f,  0.260794118f,
   -0.966976471f,  0.254865660f,
   -0.968522094f,  0.248927606f,
   -0.970031253f,  0.242980180f,
   -0.971503891f,  0.237023606f,
   -0.972939952f,  0.231058108f,
   -0.974339383f,  0.225083911f,
   -0.975702130f,  0.219101240f,
   -0.977028143f,  0.213110320f,
   -0.978317371f,  0.207111376f,
   -0.979569766f,  0.201104635f,
   -0.980785280f,  0.195090322f,
   -0.981963869f,  0.189068664f,
   -0.983105487f,  0.183039888f,
   -0.984210092f,  0.177004220f,
   -0.985277642f,  0.170961889f,
   -0.986308097f,  0.164913120f,
   -0.987301418f,  0.158858143f,
   -0.988257568f,  0.152797185f,
   -0.989176510f,  0.146730474f,
   -0.990058210f,  0.140658239f,
   -0.990902635f,  0.134580709f,
   -0.991709754f,  0.128498111f,
   -0.992479535f,  0.122410675f,
   -0.993211949f,  0.116318631f,
   -0.993906970f,  0.110222207f,
   -0.994564571f,  0.104121634f,
   -0.995184727f,  0.098017140f,
   -0.995767414f,  0.091908956f,
   -0.996312612f,  0.085797312f,
   -0.996820299f,  0.079682438f,
   -0.997290457f,  0.073564564f,
   -0.997723067f,  0.067443920f,
   -0.998118113f,  0.061320736f,
   -0.998475581f,  0.055195244f,
   -0.998795456f,  0.049067674f,
   -0.999077728f,  0.042938257f,
   -0.999322385f,  0.036807223f,
   -0.999529418f,  0.030674803f,
   -0.999698819f,  0.024541229f,
   -0.999830582f,  0.018406730f,
   -0.999924702f,  0.012271538f,
   -0.999981175f,  0.006135885f,
   -1.000000000f,  0.000000000f,
   -0.999981175f, -0.006135885f,
   -0.999924702f, -0.012271538f,
   -0.999830582f, -0.018406730f,
   -0.999698819f, -0.024541229f,
   -0.999529418f, -0.030674803f,
   -0.999322385f, -0.036807223f,
   -0.999077728f, -0.042938257f,
   -0.998795456f, -0.049067674f,
   -0.998475581f, -0.055195244f,
   -0.998118113f, -0.061320736f,
   -0.997723067f, -0.067443920f,
   -0.997290457f, -0.073564564f,
   -0.996820299f, -0.079682438f,
   -0.996312612f, -0.085797312f,
   -0.995767414f, -0.091908956f,
   -0.995184727f, -0.098017140f,
   -0.994564571f, -0.104121634f,
   -0.993906970f, -0.110222207f,
   -0.993211949f, -0.116318631f,
   -0.992479535f, -0.122410675f,
   -0.991709754f, -0.128498111f,
   -0.990902635f, -0.134580709f,
   -0.990058210f, -0.140658239f,
   -0.989176510f, -0.146730474f,
   -0.988257568f, -0.152797185f,
   -0.987301418f, -0.158858143f,
   -0.986308097f, -0.164913120f,
   -0.985277642f, -0.170961889f,
   -0.984210092f, -0.177004220f,
   -0.983105487f, -0.183039888f,
   -0.981963869f, -0.189068664f,
   -0.980785280f, -0.195090322f,
   -0.979569766f, -0.201104635f,
   -0.978317371f, -0.207111376f,
   -0.977028143f, -0.213110320f,
   -0.975702130f, -0.219101240f,
   -0.974339383f, -0.225083911f,
   -0.972939952f, -0.231058108f,
   -0.971503891f, -0.237023606f,
   -0.970031253f, -0.242980180f,
   -0.968522094f, -0.248927606f,
   -0.966976471f, -0.254865660f,
   -0.965394442f, -0.260794118f,
   -0.963776066f, -0.266712757f,
   -0.962121404f, -0.272621355f,
   -0.960430519f, -0.278519689f,
   -0.958703475f, -0.284407537f,
   -0.956940336f, -0.290284677f,
   -0.955141168f, -0.296150888f,
   -0.953306040f, -0.302005949f,
   -0.951435021f, -0.307849640f,
   -0.949528181f, -0.313681740f,
   -0.947585591f, -0.319502031f,
   -0.945607325f, -0.325310292f,
   -0.943593458f, -0.331106306f,
   -0.941544065f, -0.336889853f,
   -0.939459224f, -0.342660717f,
   -0.937339012f, -0.348418680f,
   -0.935183510f, -0.354163525f,
   -0.932992799f, -0.359895037f,
   -0.930766961f, -0.365612998f,
   -0.928506080f, -0.371317194f,
   -0.926210242f, -0.377007410f,
   -0.923879533f, -0.382683432f,
   -0.921514039f, -0.388345047f,
   -0.919113852f, -0.393992040f,
   -0.916679060f, -0.399624200f,
   -0.914209756f, -0.405241314f,
   -0.911706032f, -0.410843171f,
   -0.909167983f, -0.416429560f,
   -0.906595705f, -0.422000271f,
   -0.903989293f, -0.427555093f,
   -0.901348847f, -0.433093819f,
   -0.898674466f, -0.438616239f,
   -0.895966250f, -0.444122145f,
   -0.893224301f, -0.449611330f,
   -0.890448723f, -0.455083587f,
   -0.887639620f, -0.460538711f,
   -0.884797098f, -0.465976496f,
   -0.881921264f, -0.471396737f,
   -0.879012226f, -0.476799230f,
   -0.876070094f, -0.482183772f,
   -0.873094978f, -0.487550160f,
   -0.870086991f, -0.492898192f,
   -0.867046246f, -0.498227667f,
   -0.863972856f, -0.503538384f,
   -0.860866939f, -0.508830143f,
   -0.857728610f, -0.514102744f,
   -0.854557988f, -0.519355990f,
   -0.851355193f, -0.524589683f,
   -0.848120345f, -0.529803625f,
   -0.844853565f, -0.534997620f,
   -0.841554977f, -0.540171473f,
   -0.838224706f, -0.545324988f,
   -0.834862875f, -0.550457973f,
   -0.831469612f, -0.555570233f,
   -0.828045045f, -0.560661576f,
   -0.824589303f, -0.565731811f,
   -0.821102515f, -0.570780746f,
   -0.817584813f, -0.575808191f,
   -0.814036330f, -0.580813958f,
   -0.810457198f, -0.585797857f,
   -0.806847554f, -0.590759702f,
   -0.803207531f, -0.595699304f,
   -0.799537269f, -0.600616479f,
   -0.795836905f, -0.605511041f,
   -0.792106577f, -0.610382806f,
   -0.788346428f, -0.615231591f,
   -0.784556597f, -0.620057212f,
   -0.780737229f, -0.624859488f,
   -0.776888466f, -0.629638239f,
   -0.773010453f, -0.634393284f,
   -0.769103338f, -0.639124445f,
   -0.765167266f, -0.643831543f,
   -0.761202385f, -0.648514401f,
   -0.757208847f, -0.653172843f,
   -0.753186799f, -0.657806693f,
   -0.749136395f, -0.662415778f,
   -0.745057785f, -0.666999922f,
   -0.740951125f, -0.671558955f,
   -0.736816569f, -0.676092704f,
   -0.732654272f, -0.680600998f,
   -0.728464390f, -0.685083668f,
   -0.724247083f, -0.689540545f,
   -0.720002508f, -0.693971461f,
   -0.715730825f, -0.698376249f,
   -0.711432196f, -0.702754744f,
   -0.707106781f, -0.707106781f,
   -0.702754744f, -0.711432196f,
   -0.698376249f, -0.715730825f,
   -0.693971461f, -0.720002508f,
   -0.689540545f, -0.724247083f,
   -0.685083668f, -0.728464390f,
   -0.680600998f, -0.732654272f,
   -0.676092704f, -0.736816569f,
   -0.671558955f, -0.740951125f,
   -0.666999922f, -0.745057785f,
   -0.662415778f, -0.749136395f,
   -0.657806693f, -0.753186799f,
   -0.653172843f, -0.757208847f,
   -0.648514401f, -0.761202385f,
   -0.643831543f, -0.765167266f,
   -0.639124445f, -0.769103338f,
   -0.634393284f, -0.773010453f,
   -0.629638239f, -0.776888466f,
   -0.624859488f, -0.780737229f,
   -0.620057212f, -0.784556597f,
   -0.615231591f, -0.788346428f,
   -0.610382806f, -0.792106577f,
   -0.605511041f, -0.795836905f,
   -0.600616479f, -0.799537269f,
   -0.595699304f, -0.803207531f,
   -0.590759702f, -0.806847554f,
   -0.585797857f, -0.810457198f,
   -0.580813958f, -0.814036330f,
   -0.575808191f, -0.817584813f,
   -0.570780746f, -0.821102515f,
   -0.565731811f, -0.824589303f,
   -0.560661576f, -0.828045045f,
   -0.555570233f, -0.831469612f,
   -0.550457973f, -0.834862875f,
   -0.545324988f, -0.838224706f,
   -0.540171473f, -0.841554977f,
   -0.534997620f, -0.844853565f,
   -0.529803625f, -0.848120345f,
   -0.524589683f, -0.851355193f,
   -0.519355990f, -0.854557988f,
   -0.514102744f, -0.857728610f,
   -0.508830143f, -0.860866939f,
   -0.503538384f, -0.863972856f,
   -0.498227667f, -0.867046246f,
   -0.492898192f, -0.870086991f,
   -0.487550160f, -0.873094978f,
   -0.482183772f, -0.876070094f,
   -0.476799230f, -0.879012226f,
   -0.471396737f, -0.881921264f,
   -0.465976496f, -0.884797098f,
   -0.460538711f, -0.887639620f,
   -0.455083587f, -0.890448723f,
   -0.449611330f, -0.893224301f,
   -0.444122145f, -0.895966250f,
   -0.438616239f, -0.898674466f,
   -0.433093819f, -0.901348847f,
   -0.427555093f, -0.903989293f,
   -0.422000271f, -0.906595705f,
   -0.416429560f, -0.909167983f,
   -0.410843171f, -0.911706032f,
   -0.405241314f, -0.914209756f,
   -0.399624200f, -0.916679060f,
   -0.393992040f, -0.919113852f,
   -0.388345047f, -0.921514039f,
   -0.382683432f, -0.923879533f,
   -0.377007410f, -0.926210242f,
   -0.371317194f, -0.928506080f,
   -0.365612998f, -0.930766961f,
   -0.359895037f, -0.932992799f,
   -0.354163525f, -0.935183510f,
   -0.348418680f, -0.937339012f,
   -0.342660717f, -0.939459224f,
   -0.336889853f, -0.941544065f,
   -0.331106306f, -0.943593458f,
   -0.325310292f, -0.945607325f,
   -0.319502031f, -0.947585591f,
   -0.313681740f, -0.949528181f,
   -0.307849640f, -0.951435021f,
   -0.302005949f, -0.953306040f,
   -0.296150888f, -0.955141168f,
   -0.290284677f, -0.956940336f,
   -0.284407537f, -0.958703475f,
   -0.278519689f, -0.960430519f,
   -0.272621355f, -0.962121404f,
   -0.266712757f, -0.963776066f,
   -0.260794118f, -0.965394442f,
   -0.254865660f, -0.966976471f,
   -0.248927606f, -0.968522094f,
   -0.242980180f, -0.970031253f,
   -0.237023606f, -0.971503891f,
   -0.231058108f, -0.972939952f,
   -0.225083911f, -0.974339383f,
   -0.219101240f, -0.975702130f,
   -0.213110320f, -0.977028143f,
   -0.207111376f, -0.978317371f,
   -0.201104635f, -0.979569766f,
   -0.195090322f, -0.980785280f,
   -0.189068664f, -0.981963869f,
   -0.183039888f, -0.983105487f,
   -0.177004220f, -0.984210092f,
   -0.170961889f, -0.985277642f,
   -0.164913120f, -0.986308097f,
   -0.158858143f, -0.987301418f,
   -0.152797185f, -0.988257568f,
   -0.146730474f, -0.989176510f,
   -0.140658239f, -0.990058210f,
   -0.134580709f, -0.990902635f,
   -0.128498111f, -0.991709754f,
   -0.122410675f, -0.992479535f,
   -0.116318631f, -0.993211949f,
   -0.110222207f, -0.993906970f,
   -0.104121634f, -0.994564571f,
   -0.098017140f, -0.995184727f,
   -0.091908956f, -0.995767414f,
   -0.085797312f, -0.996312612f,
   -0.079682438f, -0.996820299f,
   -0.073564564f, -0.997290457f,
   -0.067443920f, -0.997723067f,
   -0.061320736f, -0.998118113f,
   -0.055195244f, -0.998475581f,
   -0.049067674f, -0.998795456f,
   -0.042938257f, -0.999077728f,
   -0.036807223f, -0.999322385f,
   -0.030674803f, -0.999529418f,
   -0.024541229f, -0.999698819f,
   -0.018406730f, -0.999830582f,
   -0.012271538f, -0.999924702f,
   -0.006135885f, -0.999981175f,
   -0.000000000f, -1.000000000f,
    0.006135885f, -0.999981175f,
    0.012271538f, -0.999924702f,
    0.018406730f, -0.999830582f,
    0.024541229f, -0.999698819f,
    0.030674803f, -0.999529418f,
    0.036807223f, -0.999322385f,
    0.042938257f, -0.999077728f,
    0.049067674f, -0.998795456f,
    0.055195244f, -0.998475581f,
    0.061320736f, -0.998118113f,
    0.067443920f, -0.997723067f,
    0.073564564f, -0.997290457f,
    0.079682438f, -0.996820299f,
    0.085797312f, -0.996312612f,
    0.091908956f, -0.995767414f,
    0.098017140f, -0.995184727f,
    0.104121634f, -0.994564571f,
    0.110222207f, -0.993906970f,
    0.116318631f, -0.993211949f,
    0.122410675f, -0.992479535f,
    0.128498111f, -0.991709754f,
    0.134580709f, -0.990902635f,
    0.140658239f, -0.990058210f,
    0.146730474f, -0.989176510f,
    0.152797185f, -0.988257568f,
    0.158858143f, -0.987301418f,
    0.164913120f, -0.986308097f,
    0.170961889f, -0.985277642f,
    0.177004220f, -0.984210092f,
    0.183039888f, -0.983105487f,
    0.189068664f, -0.981963869f,
    0.195090322f, -0.980785280f,
    0.201104635f, -0.979569766f,
    0.207111376f, -0.978317371f,
    0.213110320f, -0.977028143f,
    0.219101240f, -0.975702130f,
    0.225083911f, -0.974339383f,
    0.231058108f, -0.972939952f,
    0.237023606f, -0.971503891f,
    0.242980180f, -0.970031253f,
    0.248927606f, -0.968522094f,
    0.254865660f, -0.966976471f,
    0.260794118f, -0.965394442f,
    0.266712757f, -0.963776066f,
    0.272621355f, -0.962121404f,
    0.278519689f, -0.960430519f,
    0.284407537f, -0.958703475f,
    0.290284677f, -0.956940336f,
    0.296150888f, -0.955141168f,
    0.302005949f, -0.953306040f,
    0.307849640f, -0.951435021f,
    0.313681740f, -0.949528181f,
    0.319502031f, -0.947585591f,
    0.325310292f, -0.945607325f,
    0.331106306f, -0.943593458f,
    0.336889853f, -0.941544065f,
    0.342660717f, -0.939459224f,
    0.348418680f, -0.937339012f,
    0.354163525f, -0.935183510f,
    0.359895037f, -0.932992799f,
    0.365612998f, -0.930766961f,
    0.371317194f, -0.928506080f,
    0.377007410f, -0.926210242f,
    0.382683432f, -0.923879533f,
    0.388345047f, -0.921514039f,
    0.393992040f, -0.919113852f,
    0.399624200f, -0.916679060f,
    0.405241314f, -0.914209756f,
    0.410843171f, -0.911706032f,
    0.416429560f, -0.909167983f,
    0.422000271f, -0.906595705f,
    0.427555093f, -0.903989293f,
    0.433093819f, -0.901348847f,
    0.438616239f, -0.898674466f,
    0.444122145f, -0.895966250f,
    0.449611330f, -0.893224301f,
    0.455083587f, -0.890448723f,
    0.460538711f, -0.887639620f,
    0.465976496f, -0.884797098f,
    0.471396737f, -0.881921264f,
    0.476799230f, -0.879012226f,
    0.482183772f, -0.876070094f,
    0.487550160f, -0.873094978f,
    0.492898192f, -0.870086991f,
    0.498227667f, -0.867046246f,
    0.503538384f, -0.863972856f,
    0.508830143f, -0.860866939f,
    0.514102744f, -0.857728610f,
    0.519355990f, -0.854557988f,
    0.524589683f, -0.851355193f,
    0.529803625f, -0.848120345f,
    0.534997620f, -0.844853565f,
    0.540171473f, -0.841554977f,
    0.545324988f, -0.838224706f,
    0.550457973f, -0.834862875f,
    0.555570233f, -0.831469612f,
    0.560661576f, -0.828045045f,
    0.565731811f, -0.824589303f,
    0.570780746f, -0.821102515f,
    0.575808191f, -0.817584813f,
    0.580813958f, -0.814036330f,
    0.585797857f, -0.810457198f,
    0.590759702f, -0.806847554f,
    0.595699304f, -0.803207531f,
    0.600616479f, -0.799537269f,
    0.605511041f, -0.795836905f,
    0.610382806f, -0.792106577f,
    0.615231591f, -0.788346428f,
    0.620057212f, -0.784556597f,
    0.624859488f, -0.780737229f,
    0.629638239f, -0.776888466f,
    0.634393284f, -0.773010453f,
    0.639124445f, -0.769103338f,
    0.643831543f, -0.765167266f,
    0.648514401f, -0.761202385f,
    0.653172843f, -0.757208847f,
    0.657806693f, -0.753186799f,
    0.662415778f, -0.749136395f,
    0.666999922f, -0.745057785f,
    0.671558955f, -0.740951125f,
    0.676092704f, -0.736816569f,
    0.680600998f, -0.732654272f,
    0.685083668f, -0.728464390f,
    0.689540545f, -0.724247083f,
    0.693971461f, -0.720002508f,
    0.698376249f, -0.715730825f,
    0.702754744f, -0.711432196f,
    0.707106781f, -0.707106781f,
    0.711432196f, -0.702754744f,
    0.715730825f, -0.698376249f,
    0.720002508f, -0.693971461f,
    0.724247083f, -0.689540545f,
    0.728464390f, -0.685083668f,
    0.732654272f, -0.680600998f,
    0.736816569f, -0.676092704f,
    0.740951125f, -0.671558955f,
    0.745057785f, -0.666999922f,
    0.749136395f, -0.662415778f,
    0.753186799f, -0.657806693f,
    0.757208847f, -0.653172843f,
    0.761202385f, -0.648514401f,
    0.765167266f, -0.643831543f,
    0.769103338f, -0.639124445f,
    0.773010453f, -0.634393284f,
    0.776888466f, -0.629638239f,
    0.780737229f, -0.624859488f,
    0.784556597f, -0.620057212f,
    0.788346428f, -0.615231591f,
    0.792106577f, -0.610382806f,
    0.795836905f, -0.605511041f,
    0.799537269f, -0.600616479f,
    0.803207531f, -0.595699304f,
    0.806847554f, -0.590759702f,
    0.810457198f, -0.585797857f,
    0.814036330f, -0.580813958f,
    0.817584813f, -0.575808191f,
    0.821102515f, -0.570780746f,
    0.824589303f, -0.565731811f,
    0.828045045f, -0.560661576f,
    0.831469612f, -0.555570233f,
    0.834862875f, -0.550457973f,
    0.838224706f, -0.545324988f,
    0.841554977f, -0.540171473f,
    0.844853565f, -0.534997620f,
    0.848120345f, -0.529803625f,
    0.851355193f, -0.524589683f,
    0.854557988f, -0.519355990f,
    0.857728610f, -0.514102744f,
    0.860866939f, -0.508830143f,
    0.863972856f, -0.503538384f,
    0.867046246f, -0.498227667f,
    0.870086991f, -0.492898192f,
    0.873094978f, -0.487550160f,
    0.876070094f, -0.482183772f,
    0.879012226f, -0.476799230f,
    0.881921264f, -0.471396737f,
    0.884797098f, -0.465976496f,
    0.887639620f, -0.460538711f,
    0.890448723f, -0.455083587f,
    0.893224301f, -0.449611330f,
    0.895966250f, -0.444122145f,
    0.898674466f, -0.438616239f,
    0.901348847f, -0.433093819f,
    0.903989293f, -0.427555093f,
    0.906595705f, -0.422000271f,
    0.909167983f, -0.416429560f,
    0.911706032f, -0.410843171f,
    0.914209756f, -0.405241314f,
    0.916679060f, -0.399624200f,
    0.919113852f, -0.393992040f,
    0.921514039f, -0.388345047f,
    0.923879533f, -0.382683432f,
    0.926210242f, -0.377007410f,
    0.928506080f, -0.371317194f,
    0.930766961f, -0.365612998f,
    0.932992799f, -0.359895037f,
    0.935183510f, -0.354163525f,
    0.937339012f, -0.348418680f,
    0.939459224f, -0.342660717f,
    0.941544065f, -0.336889853f,
    0.943593458f, -0.331106306f,
    0.945607325f, -0.325310292f,
    0.947585591f, -0.319502031f,
    0.949528181f, -0.313681740f,
    0.951435021f, -0.307849640f,
    0.953306040f, -0.302005949f,
    0.955141168f, -0.296150888f,
    0.956940336f, -0.290284677f,
    0.958703475f, -0.284407537f,
    0.960430519f, -0.278519689f,
    0.962121404f, -0.272621355f,
    0.963776066f, -0.266712757f,
    0.965394442f, -0.260794118f,
    0.966976471f, -0.254865660f,
    0.968522094f, -0.248927606f,
    0.970031253f, -0.242980180f,
    0.971503891f, -0.237023606f,
    0.972939952f, -0.231058108f,
    0.974339383f, -0.225083911f,
    0.975702130f, -0.219101240f,
    0.977028143f, -0.213110320f,
    0.978317371f, -0.207111376f,
    0.979569766f, -0.201104635f,
    0.980785280f, -0.195090322f,
    0.981963869f, -0.189068664f,
    0.983105487f, -0.183039888f,
    0.984210092f, -0.177004220f,
    0.985277642f, -0.170961889f,
    0.986308097f, -0.164913120f,
    0.987301418f, -0.158858143f,
    0.988257568f, -0.152797185f,
    0.989176510f, -0.146730474f,
    0.990058210f, -0.140658239f,
    0.990902635f, -0.134580709f,
    0.991709754f, -0.128498111f,
    0.992479535f, -0.122410675f,
    0.993211949f, -0.116318631f,
    0.993906970f, -0.110222207f,
    0.994564571f, -0.104121634f,
    0.995184727f, -0.098017140f,
    0.995767414f, -0.091908956f,
    0.996312612f, -0.085797312f,
    0.996820299f, -0.079682438f,
    0.997290457f, -0.073564564f,
    0.997723067f, -0.067443920f,
    0.998118113f, -0.061320736f,
    0.998475581f, -0.055195244f,
    0.998795456f, -0.049067674f,
    0.999077728f, -0.042938257f,
    0.999322385f, -0.036807223f,
    0.999529418f, -0.030674803f,
    0.999698819f, -0.024541229f,
    0.999830582f, -0.018406730f,
    0.999924702f, -0.012271538f,
    0.999981175f, -0.006135885f
};
#endif
/**
* \par
* Example code for Floating-point Twiddle factors Generation:
* \par
* <pre>for(i = 0; i< N/; i++)
* {
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 2048	and PI = 3.14159265358979
* \par
* Cos and Sin values are in interleaved fashion
*
*/
const float32_t twiddleCoef_2048[4096] = {
    1.000000000f,  0.000000000f,
    0.999995294f,  0.003067957f,
    0.999981175f,  0.006135885f,
    0.999957645f,  0.009203755f,
    0.999924702f,  0.012271538f,
    0.999882347f,  0.015339206f,
    0.999830582f,  0.018406730f,
    0.999769405f,  0.021474080f,
    0.999698819f,  0.024541229f,
    0.999618822f,  0.027608146f,
    0.999529418f,  0.030674803f,
    0.999430605f,  0.033741172f,
    0.999322385f,  0.036807223f,
    0.999204759f,  0.039872928f,
    0.999077728f,  0.042938257f,
    0.998941293f,  0.046003182f,
    0.998795456f,  0.049067674f,
    0.998640218f,  0.052131705f,
    0.998475581f,  0.055195244f,
    0.998301545f,  0.058258265f,
    0.998118113f,  0.061320736f,
    0.997925286f,  0.064382631f,
    0.997723067f,  0.067443920f,
    0.997511456f,  0.070504573f,
    0.997290457f,  0.073564564f,
    0.997060070f,  0.076623861f,
    0.996820299f,  0.079682438f,
    0.996571146f,  0.082740265f,
    0.996312612f,  0.085797312f,
    0.996044701f,  0.088853553f,
    0.995767414f,  0.091908956f,
    0.995480755f,  0.094963495f,
    0.995184727f,  0.098017140f,
    0.994879331f,  0.101069863f,
    0.994564571f,  0.104121634f,
    0.994240449f,  0.107172425f,
    0.993906970f,  0.110222207f,
    0.993564136f,  0.113270952f,
    0.993211949f,  0.116318631f,
    0.992850414f,  0.119365215f,
    0.992479535f,  0.122410675f,
    0.992099313f,  0.125454983f,
    0.991709754f,  0.128498111f,
    0.991310860f,  0.131540029f,
    0.990902635f,  0.134580709f,
    0.990485084f,  0.137620122f,
    0.990058210f,  0.140658239f,
    0.989622017f,  0.143695033f,
    0.989176510f,  0.146730474f,
    0.988721692f,  0.149764535f,
    0.988257568f,  0.152797185f,
    0.987784142f,  0.155828398f,
    0.987301418f,  0.158858143f,
    0.986809402f,  0.161886394f,
    0.986308097f,  0.164913120f,
    0.985797509f,  0.167938295f,
    0.985277642f,  0.170961889f,
    0.984748502f,  0.173983873f,
    0.984210092f,  0.177004220f,
    0.983662419f,  0.180022901f,
    0.983105487f,  0.183039888f,
    0.982539302f,  0.186055152f,
    0.981963869f,  0.189068664f,
    0.981379193f,  0.192080397f,
    0.980785280f,  0.195090322f,
    0.980182136f,  0.198098411f,
    0.979569766f,  0.201104635f,
    0.978948175f,  0.204108966f,
    0.978317371f,  0.207111376f,
    0.977677358f,  0.210111837f,
    0.977028143f,  0.213110320f,
    0.976369731f,  0.216106797f,
    0.975702130f,  0.219101240f,
    0.975025345f,  0.222093621f,
    0.974339383f,  0.225083911f,
    0.973644250f,  0.228072083f,
    0.972939952f,  0.231058108f,
    0.972226497f,  0.234041959f,
    0.971503891f,  0.237023606f,
    0.970772141f,  0.240003022f,
    0.970031253f,  0.242980180f,
    0.969281235f,  0.245955050f,
    0.968522094f,  0.248927606f,
    0.967753837f,  0.251897818f,
    0.966976471f,  0.254865660f,
    0.966190003f,  0.257831102f,
    0.965394442f,  0.260794118f,
    0.964589793f,  0.263754679f,
    0.963776066f,  0.266712757f,
    0.962953267f,  0.269668326f,
    0.962121404f,  0.272621355f,
    0.961280486f,  0.275571819f,
    0.960430519f,  0.278519689f,
    0.959571513f,  0.281464938f,
    0.958703475f,  0.284407537f,
    0.957826413f,  0.287347460f,
    0.956940336f,  0.290284677f,
    0.956045251f,  0.293219163f,
    0.955141168f,  0.296150888f,
    0.954228095f,  0.299079826f,
    0.953306040f,  0.302005949f,
    0.952375013f,  0.304929230f,
    0.951435021f,  0.307849640f,
    0.950486074f,  0.310767153f,
    0.949528181f,  0.313681740f,
    0.948561350f,  0.316593376f,
    0.947585591f,  0.319502031f,
    0.946600913f,  0.322407679f,
    0.945607325f,  0.325310292f,
    0.944604837f,  0.328209844f,
    0.943593458f,  0.331106306f,
    0.942573198f,  0.333999651f,
    0.941544065f,  0.336889853f,
    0.940506071f,  0.339776884f,
    0.939459224f,  0.342660717f,
    0.938403534f,  0.345541325f,
    0.937339012f,  0.348418680f,
    0.936265667f,  0.351292756f,
    0.935183510f,  0.354163525f,
    0.934092550f,  0.357030961f,
    0.932992799f,  0.359895037f,
    0.931884266f,  0.362755724f,
    0.930766961f,  0.365612998f,
    0.929640896f,  0.368466830f,
    0.928506080f,  0.371317194f,
    0.927362526f,  0.374164063f,
    0.926210242f,  0.377007410f,
    0.925049241f,  0.379847209f,
    0.923879533f,  0.382683432f,
    0.922701128f,  0.385516054f,
    0.921514039f,  0.388345047f,
    0.920318277f,  0.391170384f,
    0.919113852f,  0.393992040f,
    0.917900776f,  0.396809987f,
    0.916679060f,  0.399624200f,
    0.915448716f,  0.402434651f,
    0.914209756f,  0.405241314f,
    0.912962190f,  0.408044163f,
    0.911706032f,  0.410843171f,
    0.910441292f,  0.413638312f,
    0.909167983f,  0.416429560f,
    0.907886116f,  0.419216888f,
    0.906595705f,  0.422000271f,
    0.905296759f,  0.424779681f,
    0.903989293f,  0.427555093f,
    0.902673318f,  0.430326481f,
    0.901348847f,  0.433093819f,
    0.900015892f,  0.435857080f,
    0.898674466f,  0.438616239f,
    0.897324581f,  0.441371269f,
    0.895966250f,  0.444122145f,
    0.894599486f,  0.446868840f,
    0.893224301f,  0.449611330f,
    0.891840709f,  0.452349587f,
    0.890448723f,  0.455083587f,
    0.889048356f,  0.457813304f,
    0.887639620f,  0.460538711f,
    0.886222530f,  0.463259784f,
    0.884797098f,  0.465976496f,
    0.883363339f,  0.468688822f,
    0.881921264f,  0.471396737f,
    0.880470889f,  0.474100215f,
    0.879012226f,  0.476799230f,
    0.877545290f,  0.479493758f,
    0.876070094f,  0.482183772f,
    0.874586652f,  0.484869248f,
    0.873094978f,  0.487550160f,
    0.871595087f,  0.490226483f,
    0.870086991f,  0.492898192f,
    0.868570706f,  0.495565262f,
    0.867046246f,  0.498227667f,
    0.865513624f,  0.500885383f,
    0.863972856f,  0.503538384f,
    0.862423956f,  0.506186645f,
    0.860866939f,  0.508830143f,
    0.859301818f,  0.511468850f,
    0.857728610f,  0.514102744f,
    0.856147328f,  0.516731799f,
    0.854557988f,  0.519355990f,
    0.852960605f,  0.521975293f,
    0.851355193f,  0.524589683f,
    0.849741768f,  0.527199135f,
    0.848120345f,  0.529803625f,
    0.846490939f,  0.532403128f,
    0.844853565f,  0.534997620f,
    0.843208240f,  0.537587076f,
    0.841554977f,  0.540171473f,
    0.839893794f,  0.542750785f,
    0.838224706f,  0.545324988f,
    0.836547727f,  0.547894059f,
    0.834862875f,  0.550457973f,
    0.833170165f,  0.553016706f,
    0.831469612f,  0.555570233f,
    0.829761234f,  0.558118531f,
    0.828045045f,  0.560661576f,
    0.826321063f,  0.563199344f,
    0.824589303f,  0.565731811f,
    0.822849781f,  0.568258953f,
    0.821102515f,  0.570780746f,
    0.819347520f,  0.573297167f,
    0.817584813f,  0.575808191f,
    0.815814411f,  0.578313796f,
    0.814036330f,  0.580813958f,
    0.812250587f,  0.583308653f,
    0.810457198f,  0.585797857f,
    0.808656182f,  0.588281548f,
    0.806847554f,  0.590759702f,
    0.805031331f,  0.593232295f,
    0.803207531f,  0.595699304f,
    0.801376172f,  0.598160707f,
    0.799537269f,  0.600616479f,
    0.797690841f,  0.603066599f,
    0.795836905f,  0.605511041f,
    0.793975478f,  0.607949785f,
    0.792106577f,  0.610382806f,
    0.790230221f,  0.612810082f,
    0.788346428f,  0.615231591f,
    0.786455214f,  0.617647308f,
    0.784556597f,  0.620057212f,
    0.782650596f,  0.622461279f,
    0.780737229f,  0.624859488f,
    0.778816512f,  0.627251815f,
    0.776888466f,  0.629638239f,
    0.774953107f,  0.632018736f,
    0.773010453f,  0.634393284f,
    0.771060524f,  0.636761861f,
    0.769103338f,  0.639124445f,
    0.767138912f,  0.641481013f,
    0.765167266f,  0.643831543f,
    0.763188417f,  0.646176013f,
    0.761202385f,  0.648514401f,
    0.759209189f,  0.650846685f,
    0.757208847f,  0.653172843f,
    0.755201377f,  0.655492853f,
    0.753186799f,  0.657806693f,
    0.751165132f,  0.660114342f,
    0.749136395f,  0.662415778f,
    0.747100606f,  0.664710978f,
    0.745057785f,  0.666999922f,
    0.743007952f,  0.669282588f,
    0.740951125f,  0.671558955f,
    0.738887324f,  0.673829000f,
    0.736816569f,  0.676092704f,
    0.734738878f,  0.678350043f,
    0.732654272f,  0.680600998f,
    0.730562769f,  0.682845546f,
    0.728464390f,  0.685083668f,
    0.726359155f,  0.687315341f,
    0.724247083f,  0.689540545f,
    0.722128194f,  0.691759258f,
    0.720002508f,  0.693971461f,
    0.717870045f,  0.696177131f,
    0.715730825f,  0.698376249f,
    0.713584869f,  0.700568794f,
    0.711432196f,  0.702754744f,
    0.709272826f,  0.704934080f,
    0.707106781f,  0.707106781f,
    0.704934080f,  0.709272826f,
    0.702754744f,  0.711432196f,
    0.700568794f,  0.713584869f,
    0.698376249f,  0.715730825f,
    0.696177131f,  0.717870045f,
    0.693971461f,  0.720002508f,
    0.691759258f,  0.722128194f,
    0.689540545f,  0.724247083f,
    0.687315341f,  0.726359155f,
    0.685083668f,  0.728464390f,
    0.682845546f,  0.730562769f,
    0.680600998f,  0.732654272f,
    0.678350043f,  0.734738878f,
    0.676092704f,  0.736816569f,
    0.673829000f,  0.738887324f,
    0.671558955f,  0.740951125f,
    0.669282588f,  0.743007952f,
    0.666999922f,  0.745057785f,
    0.664710978f,  0.747100606f,
    0.662415778f,  0.749136395f,
    0.660114342f,  0.751165132f,
    0.657806693f,  0.753186799f,
    0.655492853f,  0.755201377f,
    0.653172843f,  0.757208847f,
    0.650846685f,  0.759209189f,
    0.648514401f,  0.761202385f,
    0.646176013f,  0.763188417f,
    0.643831543f,  0.765167266f,
    0.641481013f,  0.767138912f,
    0.639124445f,  0.769103338f,
    0.636761861f,  0.771060524f,
    0.634393284f,  0.773010453f,
    0.632018736f,  0.774953107f,
    0.629638239f,  0.776888466f,
    0.627251815f,  0.778816512f,
    0.624859488f,  0.780737229f,
    0.622461279f,  0.782650596f,
    0.620057212f,  0.784556597f,
    0.617647308f,  0.786455214f,
    0.615231591f,  0.788346428f,
    0.612810082f,  0.790230221f,
    0.610382806f,  0.792106577f,
    0.607949785f,  0.793975478f,
    0.605511041f,  0.795836905f,
    0.603066599f,  0.797690841f,
    0.600616479f,  0.799537269f,
    0.598160707f,  0.801376172f,
    0.595699304f,  0.803207531f,
    0.593232295f,  0.805031331f,
    0.590759702f,  0.806847554f,
    0.588281548f,  0.808656182f,
    0.585797857f,  0.810457198f,
    0.583308653f,  0.812250587f,
    0.580813958f,  0.814036330f,
    0.578313796f,  0.815814411f,
    0.575808191f,  0.817584813f,
    0.573297167f,  0.819347520f,
    0.570780746f,  0.821102515f,
    0.568258953f,  0.822849781f,
    0.565731811f,  0.824589303f,
    0.563199344f,  0.826321063f,
    0.560661576f,  0.828045045f,
    0.558118531f,  0.829761234f,
    0.555570233f,  0.831469612f,
    0.553016706f,  0.833170165f,
    0.550457973f,  0.834862875f,
    0.547894059f,  0.836547727f,
    0.545324988f,  0.838224706f,
    0.542750785f,  0.839893794f,
    0.540171473f,  0.841554977f,
    0.537587076f,  0.843208240f,
    0.534997620f,  0.844853565f,
    0.532403128f,  0.846490939f,
    0.529803625f,  0.848120345f,
    0.527199135f,  0.849741768f,
    0.524589683f,  0.851355193f,
    0.521975293f,  0.852960605f,
    0.519355990f,  0.854557988f,
    0.516731799f,  0.856147328f,
    0.514102744f,  0.857728610f,
    0.511468850f,  0.859301818f,
    0.508830143f,  0.860866939f,
    0.506186645f,  0.862423956f,
    0.503538384f,  0.863972856f,
    0.500885383f,  0.865513624f,
    0.498227667f,  0.867046246f,
    0.495565262f,  0.868570706f,
    0.492898192f,  0.870086991f,
    0.490226483f,  0.871595087f,
    0.487550160f,  0.873094978f,
    0.484869248f,  0.874586652f,
    0.482183772f,  0.876070094f,
    0.479493758f,  0.877545290f,
    0.476799230f,  0.879012226f,
    0.474100215f,  0.880470889f,
    0.471396737f,  0.881921264f,
    0.468688822f,  0.883363339f,
    0.465976496f,  0.884797098f,
    0.463259784f,  0.886222530f,
    0.460538711f,  0.887639620f,
    0.457813304f,  0.889048356f,
    0.455083587f,  0.890448723f,
    0.452349587f,  0.891840709f,
    0.449611330f,  0.893224301f,
    0.446868840f,  0.894599486f,
    0.444122145f,  0.895966250f,
    0.441371269f,  0.897324581f,
    0.438616239f,  0.898674466f,
    0.435857080f,  0.900015892f,
    0.433093819f,  0.901348847f,
    0.430326481f,  0.902673318f,
    0.427555093f,  0.903989293f,
    0.424779681f,  0.905296759f,
    0.422000271f,  0.906595705f,
    0.419216888f,  0.907886116f,
    0.416429560f,  0.909167983f,
    0.413638312f,  0.910441292f,
    0.410843171f,  0.911706032f,
    0.408044163f,  0.912962190f,
    0.405241314f,  0.914209756f,
    0.402434651f,  0.915448716f,
    0.399624200f,  0.916679060f,
    0.396809987f,  0.917900776f,
    0.393992040f,  0.919113852f,
    0.391170384f,  0.920318277f,
    0.388345047f,  0.921514039f,
    0.385516054f,  0.922701128f,
    0.382683432f,  0.923879533f,
    0.379847209f,  0.925049241f,
    0.377007410f,  0.926210242f,
    0.374164063f,  0.927362526f,
    0.371317194f,  0.928506080f,
    0.368466830f,  0.929640896f,
    0.365612998f,  0.930766961f,
    0.362755724f,  0.931884266f,
    0.359895037f,  0.932992799f,
    0.357030961f,  0.934092550f,
    0.354163525f,  0.935183510f,
    0.351292756f,  0.936265667f,
    0.348418680f,  0.937339012f,
    0.345541325f,  0.938403534f,
    0.342660717f,  0.939459224f,
    0.339776884f,  0.940506071f,
    0.336889853f,  0.941544065f,
    0.333999651f,  0.942573198f,
    0.331106306f,  0.943593458f,
    0.328209844f,  0.944604837f,
    0.325310292f,  0.945607325f,
    0.322407679f,  0.946600913f,
    0.319502031f,  0.947585591f,
    0.316593376f,  0.948561350f,
    0.313681740f,  0.949528181f,
    0.310767153f,  0.950486074f,
    0.307849640f,  0.951435021f,
    0.304929230f,  0.952375013f,
    0.302005949f,  0.953306040f,
    0.299079826f,  0.954228095f,
    0.296150888f,  0.955141168f,
    0.293219163f,  0.956045251f,
    0.290284677f,  0.956940336f,
    0.287347460f,  0.957826413f,
    0.284407537f,  0.958703475f,
    0.281464938f,  0.959571513f,
    0.278519689f,  0.960430519f,
    0.275571819f,  0.961280486f,
    0.272621355f,  0.962121404f,
    0.269668326f,  0.962953267f,
    0.266712757f,  0.963776066f,
    0.263754679f,  0.964589793f,
    0.260794118f,  0.965394442f,
    0.257831102f,  0.966190003f,
    0.254865660f,  0.966976471f,
    0.251897818f,  0.967753837f,
    0.248927606f,  0.968522094f,
    0.245955050f,  0.969281235f,
    0.242980180f,  0.970031253f,
    0.240003022f,  0.970772141f,
    0.237023606f,  0.971503891f,
    0.234041959f,  0.972226497f,
    0.231058108f,  0.972939952f,
    0.228072083f,  0.973644250f,
    0.225083911f,  0.974339383f,
    0.222093621f,  0.975025345f,
    0.219101240f,  0.975702130f,
    0.216106797f,  0.976369731f,
    0.213110320f,  0.977028143f,
    0.210111837f,  0.977677358f,
    0.207111376f,  0.978317371f,
    0.204108966f,  0.978948175f,
    0.201104635f,  0.979569766f,
    0.198098411f,  0.980182136f,
    0.195090322f,  0.980785280f,
    0.192080397f,  0.981379193f,
    0.189068664f,  0.981963869f,
    0.186055152f,  0.982539302f,
    0.183039888f,  0.983105487f,
    0.180022901f,  0.983662419f,
    0.177004220f,  0.984210092f,
    0.173983873f,  0.984748502f,
    0.170961889f,  0.985277642f,
    0.167938295f,  0.985797509f,
    0.164913120f,  0.986308097f,
    0.161886394f,  0.986809402f,
    0.158858143f,  0.987301418f,
    0.155828398f,  0.987784142f,
    0.152797185f,  0.988257568f,
    0.149764535f,  0.988721692f,
    0.146730474f,  0.989176510f,
    0.143695033f,  0.989622017f,
    0.140658239f,  0.990058210f,
    0.137620122f,  0.990485084f,
    0.134580709f,  0.990902635f,
    0.131540029f,  0.991310860f,
    0.128498111f,  0.991709754f,
    0.125454983f,  0.992099313f,
    0.122410675f,  0.992479535f,
    0.119365215f,  0.992850414f,
    0.116318631f,  0.993211949f,
    0.113270952f,  0.993564136f,
    0.110222207f,  0.993906970f,
    0.107172425f,  0.994240449f,
    0.104121634f,  0.994564571f,
    0.101069863f,  0.994879331f,
    0.098017140f,  0.995184727f,
    0.094963495f,  0.995480755f,
    0.091908956f,  0.995767414f,
    0.088853553f,  0.996044701f,
    0.085797312f,  0.996312612f,
    0.082740265f,  0.996571146f,
    0.079682438f,  0.996820299f,
    0.076623861f,  0.997060070f,
    0.073564564f,  0.997290457f,
    0.070504573f,  0.997511456f,
    0.067443920f,  0.997723067f,
    0.064382631f,  0.997925286f,
    0.061320736f,  0.998118113f,
    0.058258265f,  0.998301545f,
    0.055195244f,  0.998475581f,
    0.052131705f,  0.998640218f,
    0.049067674f,  0.998795456f,
    0.046003182f,  0.998941293f,
    0.042938257f,  0.999077728f,
    0.039872928f,  0.999204759f,
    0.036807223f,  0.999322385f,
    0.033741172f,  0.999430605f,
    0.030674803f,  0.999529418f,
    0.027608146f,  0.999618822f,
    0.024541229f,  0.999698819f,
    0.021474080f,  0.999769405f,
    0.018406730f,  0.999830582f,
    0.015339206f,  0.999882347f,
    0.012271538f,  0.999924702f,
    0.009203755f,  0.999957645f,
    0.006135885f,  0.999981175f,
    0.003067957f,  0.999995294f,
    0.000000000f,  1.000000000f,
   -0.003067957f,  0.999995294f,
   -0.006135885f,  0.999981175f,
   -0.009203755f,  0.999957645f,
   -0.012271538f,  0.999924702f,
   -0.015339206f,  0.999882347f,
   -0.018406730f,  0.999830582f,
   -0.021474080f,  0.999769405f,
   -0.024541229f,  0.999698819f,
   -0.027608146f,  0.999618822f,
   -0.030674803f,  0.999529418f,
   -0.033741172f,  0.999430605f,
   -0.036807223f,  0.999322385f,
   -0.039872928f,  0.999204759f,
   -0.042938257f,  0.999077728f,
   -0.046003182f,  0.998941293f,
   -0.049067674f,  0.998795456f,
   -0.052131705f,  0.998640218f,
   -0.055195244f,  0.998475581f,
   -0.058258265f,  0.998301545f,
   -0.061320736f,  0.998118113f,
   -0.064382631f,  0.997925286f,
   -0.067443920f,  0.997723067f,
   -0.070504573f,  0.997511456f,
   -0.073564564f,  0.997290457f,
   -0.076623861f,  0.997060070f,
   -0.079682438f,  0.996820299f,
   -0.082740265f,  0.996571146f,
   -0.085797312f,  0.996312612f,
   -0.088853553f,  0.996044701f,
   -0.091908956f,  0.995767414f,
   -0.094963495f,  0.995480755f,
   -0.098017140f,  0.995184727f,
   -0.101069863f,  0.994879331f,
   -0.104121634f,  0.994564571f,
   -0.107172425f,  0.994240449f,
   -0.110222207f,  0.993906970f,
   -0.113270952f,  0.993564136f,
   -0.116318631f,  0.993211949f,
   -0.119365215f,  0.992850414f,
   -0.122410675f,  0.992479535f,
   -0.125454983f,  0.992099313f,
   -0.128498111f,  0.991709754f,
   -0.131540029f,  0.991310860f,
   -0.134580709f,  0.990902635f,
   -0.137620122f,  0.990485084f,
   -0.140658239f,  0.990058210f,
   -0.143695033f,  0.989622017f,
   -0.146730474f,  0.989176510f,
   -0.149764535f,  0.988721692f,
   -0.152797185f,  0.988257568f,
   -0.155828398f,  0.987784142f,
   -0.158858143f,  0.987301418f,
   -0.161886394f,  0.986809402f,
   -0.164913120f,  0.986308097f,
   -0.167938295f,  0.985797509f,
   -0.170961889f,  0.985277642f,
   -0.173983873f,  0.984748502f,
   -0.177004220f,  0.984210092f,
   -0.180022901f,  0.983662419f,
   -0.183039888f,  0.983105487f,
   -0.186055152f,  0.982539302f,
   -0.189068664f,  0.981963869f,
   -0.192080397f,  0.981379193f,
   -0.195090322f,  0.980785280f,
   -0.198098411f,  0.980182136f,
   -0.201104635f,  0.979569766f,
   -0.204108966f,  0.978948175f,
   -0.207111376f,  0.978317371f,
   -0.210111837f,  0.977677358f,
   -0.213110320f,  0.977028143f,
   -0.216106797f,  0.976369731f,
   -0.219101240f,  0.975702130f,
   -0.222093621f,  0.975025345f,
   -0.225083911f,  0.974339383f,
   -0.228072083f,  0.973644250f,
   -0.231058108f,  0.972939952f,
   -0.234041959f,  0.972226497f,
   -0.237023606f,  0.971503891f,
   -0.240003022f,  0.970772141f,
   -0.242980180f,  0.970031253f,
   -0.245955050f,  0.969281235f,
   -0.248927606f,  0.968522094f,
   -0.251897818f,  0.967753837f,
   -0.254865660f,  0.966976471f,
   -0.257831102f,  0.966190003f,
   -0.260794118f,  0.965394442f,
   -0.263754679f,  0.964589793f,
   -0.266712757f,  0.963776066f,
   -0.269668326f,  0.962953267f,
   -0.272621355f,  0.962121404f,
   -0.275571819f,  0.961280486f,
   -0.278519689f,  0.960430519f,
   -0.281464938f,  0.959571513f,
   -0.284407537f,  0.958703475f,
   -0.287347460f,  0.957826413f,
   -0.290284677f,  0.956940336f,
   -0.293219163f,  0.956045251f,
   -0.296150888f,  0.955141168f,
   -0.299079826f,  0.954228095f,
   -0.302005949f,  0.953306040f,
   -0.304929230f,  0.952375013f,
   -0.307849640f,  0.951435021f,
   -0.310767153f,  0.950486074f,
   -0.313681740f,  0.949528181f,
   -0.316593376f,  0.948561350f,
   -0.319502031f,  0.947585591f,
   -0.322407679f,  0.946600913f,
   -0.325310292f,  0.945607325f,
   -0.328209844f,  0.944604837f,
   -0.331106306f,  0.943593458f,
   -0.333999651f,  0.942573198f,
   -0.336889853f,  0.941544065f,
   -0.339776884f,  0.940506071f,
   -0.342660717f,  0.939459224f,
   -0.345541325f,  0.938403534f,
   -0.348418680f,  0.937339012f,
   -0.351292756f,  0.936265667f,
   -0.354163525f,  0.935183510f,
   -0.357030961f,  0.934092550f,
   -0.359895037f,  0.932992799f,
   -0.362755724f,  0.931884266f,
   -0.365612998f,  0.930766961f,
   -0.368466830f,  0.929640896f,
   -0.371317194f,  0.928506080f,
   -0.374164063f,  0.927362526f,
   -0.377007410f,  0.926210242f,
   -0.379847209f,  0.925049241f,
   -0.382683432f,  0.923879533f,
   -0.385516054f,  0.922701128f,
   -0.388345047f,  0.921514039f,
   -0.391170384f,  0.920318277f,
   -0.393992040f,  0.919113852f,
   -0.396809987f,  0.917900776f,
   -0.399624200f,  0.916679060f,
   -0.402434651f,  0.915448716f,
   -0.405241314f,  0.914209756f,
   -0.408044163f,  0.912962190f,
   -0.410843171f,  0.911706032f,
   -0.413638312f,  0.910441292f,
   -0.416429560f,  0.909167983f,
   -0.419216888f,  0.907886116f,
   -0.422000271f,  0.906595705f,
   -0.424779681f,  0.905296759f,
   -0.427555093f,  0.903989293f,
   -0.430326481f,  0.902673318f,
   -0.433093819f,  0.901348847f,
   -0.435857080f,  0.900015892f,
   -0.438616239f,  0.898674466f,
   -0.441371269f,  0.897324581f,
   -0.444122145f,  0.895966250f,
   -0.446868840f,  0.894599486f,
   -0.449611330f,  0.893224301f,
   -0.452349587f,  0.891840709f,
   -0.455083587f,  0.890448723f,
   -0.457813304f,  0.889048356f,
   -0.460538711f,  0.887639620f,
   -0.463259784f,  0.886222530f,
   -0.465976496f,  0.884797098f,
   -0.468688822f,  0.883363339f,
   -0.471396737f,  0.881921264f,
   -0.474100215f,  0.880470889f,
   -0.476799230f,  0.879012226f,
   -0.479493758f,  0.877545290f,
   -0.482183772f,  0.876070094f,
   -0.484869248f,  0.874586652f,
   -0.487550160f,  0.873094978f,
   -0.490226483f,  0.871595087f,
   -0.492898192f,  0.870086991f,
   -0.495565262f,  0.868570706f,
   -0.498227667f,  0.867046246f,
   -0.500885383f,  0.865513624f,
   -0.503538384f,  0.863972856f,
   -0.506186645f,  0.862423956f,
   -0.508830143f,  0.860866939f,
   -0.511468850f,  0.859301818f,
   -0.514102744f,  0.857728610f,
   -0.516731799f,  0.856147328f,
   -0.519355990f,  0.854557988f,
   -0.521975293f,  0.852960605f,
   -0.524589683f,  0.851355193f,
   -0.527199135f,  0.849741768f,
   -0.529803625f,  0.848120345f,
   -0.532403128f,  0.846490939f,
   -0.534997620f,  0.844853565f,
   -0.537587076f,  0.843208240f,
   -0.540171473f,  0.841554977f,
   -0.542750785f,  0.839893794f,
   -0.545324988f,  0.838224706f,
   -0.547894059f,  0.836547727f,
   -0.550457973f,  0.834862875f,
   -0.553016706f,  0.833170165f,
   -0.555570233f,  0.831469612f,
   -0.558118531f,  0.829761234f,
   -0.560661576f,  0.828045045f,
   -0.563199344f,  0.826321063f,
   -0.565731811f,  0.824589303f,
   -0.568258953f,  0.822849781f,
   -0.570780746f,  0.821102515f,
   -0.573297167f,  0.819347520f,
   -0.575808191f,  0.817584813f,
   -0.578313796f,  0.815814411f,
   -0.580813958f,  0.814036330f,
   -0.583308653f,  0.812250587f,
   -0.585797857f,  0.810457198f,
   -0.588281548f,  0.808656182f,
   -0.590759702f,  0.806847554f,
   -0.593232295f,  0.805031331f,
   -0.595699304f,  0.803207531f,
   -0.598160707f,  0.801376172f,
   -0.600616479f,  0.799537269f,
   -0.603066599f,  0.797690841f,
   -0.605511041f,  0.795836905f,
   -0.607949785f,  0.793975478f,
   -0.610382806f,  0.792106577f,
   -0.612810082f,  0.790230221f,
   -0.615231591f,  0.788346428f,
   -0.617647308f,  0.786455214f,
   -0.620057212f,  0.784556597f,
   -0.622461279f,  0.782650596f,
   -0.624859488f,  0.780737229f,
   -0.627251815f,  0.778816512f,
   -0.629638239f,  0.776888466f,
   -0.632018736f,  0.774953107f,
   -0.634393284f,  0.773010453f,
   -0.636761861f,  0.771060524f,
   -0.639124445f,  0.769103338f,
   -0.641481013f,  0.767138912f,
   -0.643831543f,  0.765167266f,
   -0.646176013f,  0.763188417f,
   -0.648514401f,  0.761202385f,
   -0.650846685f,  0.759209189f,
   -0.653172843f,  0.757208847f,
   -0.655492853f,  0.755201377f,
   -0.657806693f,  0.753186799f,
   -0.660114342f,  0.751165132f,
   -0.662415778f,  0.749136395f,
   -0.664710978f,  0.747100606f,
   -0.666999922f,  0.745057785f,
   -0.669282588f,  0.743007952f,
   -0.671558955f,  0.740951125f,
   -0.673829000f,  0.738887324f,
   -0.676092704f,  0.736816569f,
   -0.678350043f,  0.734738878f,
   -0.680600998f,  0.732654272f,
   -0.682845546f,  0.730562769f,
   -0.685083668f,  0.728464390f,
   -0.687315341f,  0.726359155f,
   -0.689540545f,  0.724247083f,
   -0.691759258f,  0.722128194f,
   -0.693971461f,  0.720002508f,
   -0.696177131f,  0.717870045f,
   -0.698376249f,  0.715730825f,
   -0.700568794f,  0.713584869f,
   -0.702754744f,  0.711432196f,
   -0.704934080f,  0.709272826f,
   -0.707106781f,  0.707106781f,
   -0.709272826f,  0.704934080f,
   -0.711432196f,  0.702754744f,
   -0.713584869f,  0.700568794f,
   -0.715730825f,  0.698376249f,
   -0.717870045f,  0.696177131f,
   -0.720002508f,  0.693971461f,
   -0.722128194f,  0.691759258f,
   -0.724247083f,  0.689540545f,
   -0.726359155f,  0.687315341f,
   -0.728464390f,  0.685083668f,
   -0.730562769f,  0.682845546f,
   -0.732654272f,  0.680600998f,
   -0.734738878f,  0.678350043f,
   -0.736816569f,  0.676092704f,
   -0.738887324f,  0.673829000f,
   -0.740951125f,  0.671558955f,
   -0.743007952f,  0.669282588f,
   -0.745057785f,  0.666999922f,
   -0.747100606f,  0.664710978f,
   -0.749136395f,  0.662415778f,
   -0.751165132f,  0.660114342f,
   -0.753186799f,  0.657806693f,
   -0.755201377f,  0.655492853f,
   -0.757208847f,  0.653172843f,
   -0.759209189f,  0.650846685f,
   -0.761202385f,  0.648514401f,
   -0.763188417f,  0.646176013f,
   -0.765167266f,  0.643831543f,
   -0.767138912f,  0.641481013f,
   -0.769103338f,  0.639124445f,
   -0.771060524f,  0.636761861f,
   -0.773010453f,  0.634393284f,
   -0.774953107f,  0.632018736f,
   -0.776888466f,  0.629638239f,
   -0.778816512f,  0.627251815f,
   -0.780737229f,  0.624859488f,
   -0.782650596f,  0.622461279f,
   -0.784556597f,  0.620057212f,
   -0.786455214f,  0.617647308f,
   -0.788346428f,  0.615231591f,
   -0.790230221f,  0.612810082f,
   -0.792106577f,  0.610382806f,
   -0.793975478f,  0.607949785f,
   -0.795836905f,  0.605511041f,
   -0.797690841f,  0.603066599f,
   -0.799537269f,  0.600616479f,
   -0.801376172f,  0.598160707f,
   -0.803207531f,  0.595699304f,
   -0.805031331f,  0.593232295f,
   -0.806847554f,  0.590759702f,
   -0.808656182f,  0.588281548f,
   -0.810457198f,  0.585797857f,
   -0.812250587f,  0.583308653f,
   -0.814036330f,  0.580813958f,
   -0.815814411f,  0.578313796f,
   -0.817584813f,  0.575808191f,
   -0.819347520f,  0.573297167f,
   -0.821102515f,  0.570780746f,
   -0.822849781f,  0.568258953f,
   -0.824589303f,  0.565731811f,
   -0.826321063f,  0.563199344f,
   -0.828045045f,  0.560661576f,
   -0.829761234f,  0.558118531f,
   -0.831469612f,  0.555570233f,
   -0.833170165f,  0.553016706f,
   -0.834862875f,  0.550457973f,
   -0.836547727f,  0.547894059f,
   -0.838224706f,  0.545324988f,
   -0.839893794f,  0.542750785f,
   -0.841554977f,  0.540171473f,
   -0.843208240f,  0.537587076f,
   -0.844853565f,  0.534997620f,
   -0.846490939f,  0.532403128f,
   -0.848120345f,  0.529803625f,
   -0.849741768f,  0.527199135f,
   -0.851355193f,  0.524589683f,
   -0.852960605f,  0.521975293f,
   -0.854557988f,  0.519355990f,
   -0.856147328f,  0.516731799f,
   -0.857728610f,  0.514102744f,
   -0.859301818f,  0.511468850f,
   -0.860866939f,  0.508830143f,
   -0.862423956f,  0.506186645f,
   -0.863972856f,  0.503538384f,
   -0.865513624f,  0.500885383f,
   -0.867046246f,  0.498227667f,
   -0.868570706f,  0.495565262f,
   -0.870086991f,  0.492898192f,
   -0.871595087f,  0.490226483f,
   -0.873094978f,  0.487550160f,
   -0.874586652f,  0.484869248f,
   -0.876070094f,  0.482183772f,
   -0.877545290f,  0.479493758f,
   -0.879012226f,  0.476799230f,
   -0.880470889f,  0.474100215f,
   -0.881921264f,  0.471396737f,
   -0.883363339f,  0.468688822f,
   -0.884797098f,  0.465976496f,
   -0.886222530f,  0.463259784f,
   -0.887639620f,  0.460538711f,
   -0.889048356f,  0.457813304f,
   -0.890448723f,  0.455083587f,
   -0.891840709f,  0.452349587f,
   -0.893224301f,  0.449611330f,
   -0.894599486f,  0.446868840f,
   -0.895966250f,  0.444122145f,
   -0.897324581f,  0.441371269f,
   -0.898674466f,  0.438616239f,
   -0.900015892f,  0.435857080f,
   -0.901348847f,  0.433093819f,
   -0.902673318f,  0.430326481f,
   -0.903989293f,  0.427555093f,
   -0.905296759f,  0.424779681f,
   -0.906595705f,  0.422000271f,
   -0.907886116f,  0.419216888f,
   -0.909167983f,  0.416429560f,
   -0.910441292f,  0.413638312f,
   -0.911706032f,  0.410843171f,
   -0.912962190f,  0.408044163f,
   -0.914209756f,  0.405241314f,
   -0.915448716f,  0.402434651f,
   -0.916679060f,  0.399624200f,
   -0.917900776f,  0.396809987f,
   -0.919113852f,  0.393992040f,
   -0.920318277f,  0.391170384f,
   -0.921514039f,  0.388345047f,
   -0.922701128f,  0.385516054f,
   -0.923879533f,  0.382683432f,
   -0.925049241f,  0.379847209f,
   -0.926210242f,  0.377007410f,
   -0.927362526f,  0.374164063f,
   -0.928506080f,  0.371317194f,
   -0.929640896f,  0.368466830f,
   -0.930766961f,  0.365612998f,
   -0.931884266f,  0.362755724f,
   -0.932992799f,  0.359895037f,
   -0.934092550f,  0.357030961f,
   -0.935183510f,  0.354163525f,
   -0.936265667f,  0.351292756f,
   -0.937339012f,  0.348418680f,
   -0.938403534f,  0.345541325f,
   -0.939459224f,  0.342660717f,
   -0.940506071f,  0.339776884f,
   -0.941544065f,  0.336889853f,
   -0.942573198f,  0.333999651f,
   -0.943593458f,  0.331106306f,
   -0.944604837f,  0.328209844f,
   -0.945607325f,  0.325310292f,
   -0.946600913f,  0.322407679f,
   -0.947585591f,  0.319502031f,
   -0.948561350f,  0.316593376f,
   -0.949528181f,  0.313681740f,
   -0.950486074f,  0.310767153f,
   -0.951435021f,  0.307849640f,
   -0.952375013f,  0.304929230f,
   -0.953306040f,  0.302005949f,
   -0.954228095f,  0.299079826f,
   -0.955141168f,  0.296150888f,
   -0.956045251f,  0.293219163f,
   -0.956940336f,  0.290284677f,
   -0.957826413f,  0.287347460f,
   -0.958703475f,  0.284407537f,
   -0.959571513f,  0.281464938f,
   -0.960430519f,  0.278519689f,
   -0.961280486f,  0.275571819f,
   -0.962121404f,  0.272621355f,
   -0.962953267f,  0.269668326f,
   -0.963776066f,  0.266712757f,
   -0.964589793f,  0.263754679f,
   -0.965394442f,  0.260794118f,
   -0.966190003f,  0.257831102f,
   -0.966976471f,  0.254865660f,
   -0.967753837f,  0.251897818f,
   -0.968522094f,  0.248927606f,
   -0.969281235f,  0.245955050f,
   -0.970031253f,  0.242980180f,
   -0.970772141f,  0.240003022f,
   -0.971503891f,  0.237023606f,
   -0.972226497f,  0.234041959f,
   -0.972939952f,  0.231058108f,
   -0.973644250f,  0.228072083f,
   -0.974339383f,  0.225083911f,
   -0.975025345f,  0.222093621f,
   -0.975702130f,  0.219101240f,
   -0.976369731f,  0.216106797f,
   -0.977028143f,  0.213110320f,
   -0.977677358f,  0.210111837f,
   -0.978317371f,  0.207111376f,
   -0.978948175f,  0.204108966f,
   -0.979569766f,  0.201104635f,
   -0.980182136f,  0.198098411f,
   -0.980785280f,  0.195090322f,
   -0.981379193f,  0.192080397f,
   -0.981963869f,  0.189068664f,
   -0.982539302f,  0.186055152f,
   -0.983105487f,  0.183039888f,
   -0.983662419f,  0.180022901f,
   -0.984210092f,  0.177004220f,
   -0.984748502f,  0.173983873f,
   -0.985277642f,  0.170961889f,
   -0.985797509f,  0.167938295f,
   -0.986308097f,  0.164913120f,
   -0.986809402f,  0.161886394f,
   -0.987301418f,  0.158858143f,
   -0.987784142f,  0.155828398f,
   -0.988257568f,  0.152797185f,
   -0.988721692f,  0.149764535f,
   -0.989176510f,  0.146730474f,
   -0.989622017f,  0.143695033f,
   -0.990058210f,  0.140658239f,
   -0.990485084f,  0.137620122f,
   -0.990902635f,  0.134580709f,
   -0.991310860f,  0.131540029f,
   -0.991709754f,  0.128498111f,
   -0.992099313f,  0.125454983f,
   -0.992479535f,  0.122410675f,
   -0.992850414f,  0.119365215f,
   -0.993211949f,  0.116318631f,
   -0.993564136f,  0.113270952f,
   -0.993906970f,  0.110222207f,
   -0.994240449f,  0.107172425f,
   -0.994564571f,  0.104121634f,
   -0.994879331f,  0.101069863f,
   -0.995184727f,  0.098017140f,
   -0.995480755f,  0.094963495f,
   -0.995767414f,  0.091908956f,
   -0.996044701f,  0.088853553f,
   -0.996312612f,  0.085797312f,
   -0.996571146f,  0.082740265f,
   -0.996820299f,  0.079682438f,
   -0.997060070f,  0.076623861f,
   -0.997290457f,  0.073564564f,
   -0.997511456f,  0.070504573f,
   -0.997723067f,  0.067443920f,
   -0.997925286f,  0.064382631f,
   -0.998118113f,  0.061320736f,
   -0.998301545f,  0.058258265f,
   -0.998475581f,  0.055195244f,
   -0.998640218f,  0.052131705f,
   -0.998795456f,  0.049067674f,
   -0.998941293f,  0.046003182f,
   -0.999077728f,  0.042938257f,
   -0.999204759f,  0.039872928f,
   -0.999322385f,  0.036807223f,
   -0.999430605f,  0.033741172f,
   -0.999529418f,  0.030674803f,
   -0.999618822f,  0.027608146f,
   -0.999698819f,  0.024541229f,
   -0.999769405f,  0.021474080f,
   -0.999830582f,  0.018406730f,
   -0.999882347f,  0.015339206f,
   -0.999924702f,  0.012271538f,
   -0.999957645f,  0.009203755f,
   -0.999981175f,  0.006135885f,
   -0.999995294f,  0.003067957f,
   -1.000000000f,  0.000000000f,
   -0.999995294f, -0.003067957f,
   -0.999981175f, -0.006135885f,
   -0.999957645f, -0.009203755f,
   -0.999924702f, -0.012271538f,
   -0.999882347f, -0.015339206f,
   -0.999830582f, -0.018406730f,
   -0.999769405f, -0.021474080f,
   -0.999698819f, -0.024541229f,
   -0.999618822f, -0.027608146f,
   -0.999529418f, -0.030674803f,
   -0.999430605f, -0.033741172f,
   -0.999322385f, -0.036807223f,
   -0.999204759f, -0.039872928f,
   -0.999077728f, -0.042938257f,
   -0.998941293f, -0.046003182f,
   -0.998795456f, -0.049067674f,
   -0.998640218f, -0.052131705f,
   -0.998475581f, -0.055195244f,
   -0.998301545f, -0.058258265f,
   -0.998118113f, -0.061320736f,
   -0.997925286f, -0.064382631f,
   -0.997723067f, -0.067443920f,
   -0.997511456f, -0.070504573f,
   -0.997290457f, -0.073564564f,
   -0.997060070f, -0.076623861f,
   -0.996820299f, -0.079682438f,
   -0.996571146f, -0.082740265f,
   -0.996312612f, -0.085797312f,
   -0.996044701f, -0.088853553f,
   -0.995767414f, -0.091908956f,
   -0.995480755f, -0.094963495f,
   -0.995184727f, -0.098017140f,
   -0.994879331f, -0.101069863f,
   -0.994564571f, -0.104121634f,
   -0.994240449f, -0.107172425f,
   -0.993906970f, -0.110222207f,
   -0.993564136f, -0.113270952f,
   -0.993211949f, -0.116318631f,
   -0.992850414f, -0.119365215f,
   -0.992479535f, -0.122410675f,
   -0.992099313f, -0.125454983f,
   -0.991709754f, -0.128498111f,
   -0.991310860f, -0.131540029f,
   -0.990902635f, -0.134580709f,
   -0.990485084f, -0.137620122f,
   -0.990058210f, -0.140658239f,
   -0.989622017f, -0.143695033f,
   -0.989176510f, -0.146730474f,
   -0.988721692f, -0.149764535f,
   -0.988257568f, -0.152797185f,
   -0.987784142f, -0.155828398f,
   -0.987301418f, -0.158858143f,
   -0.986809402f, -0.161886394f,
   -0.986308097f, -0.164913120f,
   -0.985797509f, -0.167938295f,
   -0.985277642f, -0.170961889f,
   -0.984748502f, -0.173983873f,
   -0.984210092f, -0.177004220f,
   -0.983662419f, -0.180022901f,
   -0.983105487f, -0.183039888f,
   -0.982539302f, -0.186055152f,
   -0.981963869f, -0.189068664f,
   -0.981379193f, -0.192080397f,
   -0.980785280f, -0.195090322f,
   -0.980182136f, -0.198098411f,
   -0.979569766f, -0.201104635f,
   -0.978948175f, -0.204108966f,
   -0.978317371f, -0.207111376f,
   -0.977677358f, -0.210111837f,
   -0.977028143f, -0.213110320f,
   -0.976369731f, -0.216106797f,
   -0.975702130f, -0.219101240f,
   -0.975025345f, -0.222093621f,
   -0.974339383f, -0.225083911f,
   -0.973644250f, -0.228072083f,
   -0.972939952f, -0.231058108f,
   -0.972226497f, -0.234041959f,
   -0.971503891f, -0.237023606f,
   -0.970772141f, -0.240003022f,
   -0.970031253f, -0.242980180f,
   -0.969281235f, -0.245955050f,
   -0.968522094f, -0.248927606f,
   -0.967753837f, -0.251897818f,
   -0.966976471f, -0.254865660f,
   -0.966190003f, -0.257831102f,
   -0.965394442f, -0.260794118f,
   -0.964589793f, -0.263754679f,
   -0.963776066f, -0.266712757f,
   -0.962953267f, -0.269668326f,
   -0.962121404f, -0.272621355f,
   -0.961280486f, -0.275571819f,
   -0.960430519f, -0.278519689f,
   -0.959571513f, -0.281464938f,
   -0.958703475f, -0.284407537f,
   -0.957826413f, -0.287347460f,
   -0.956940336f, -0.290284677f,
   -0.956045251f, -0.293219163f,
   -0.955141168f, -0.296150888f,
   -0.954228095f, -0.299079826f,
   -0.953306040f, -0.302005949f,
   -0.952375013f, -0.304929230f,
   -0.951435021f, -0.307849640f,
   -0.950486074f, -0.310767153f,
   -0.949528181f, -0.313681740f,
   -0.948561350f, -0.316593376f,
   -0.947585591f, -0.319502031f,
   -0.946600913f, -0.322407679f,
   -0.945607325f, -0.325310292f,
   -0.944604837f, -0.328209844f,
   -0.943593458f, -0.331106306f,
   -0.942573198f, -0.333999651f,
   -0.941544065f, -0.336889853f,
   -0.940506071f, -0.339776884f,
   -0.939459224f, -0.342660717f,
   -0.938403534f, -0.345541325f,
   -0.937339012f, -0.348418680f,
   -0.936265667f, -0.351292756f,
   -0.935183510f, -0.354163525f,
   -0.934092550f, -0.357030961f,
   -0.932992799f, -0.359895037f,
   -0.931884266f, -0.362755724f,
   -0.930766961f, -0.365612998f,
   -0.929640896f, -0.368466830f,
   -0.928506080f, -0.371317194f,
   -0.927362526f, -0.374164063f,
   -0.926210242f, -0.377007410f,
   -0.925049241f, -0.379847209f,
   -0.923879533f, -0.382683432f,
   -0.922701128f, -0.385516054f,
   -0.921514039f, -0.388345047f,
   -0.920318277f, -0.391170384f,
   -0.919113852f, -0.393992040f,
   -0.917900776f, -0.396809987f,
   -0.916679060f, -0.399624200f,
   -0.915448716f, -0.402434651f,
   -0.914209756f, -0.405241314f,
   -0.912962190f, -0.408044163f,
   -0.911706032f, -0.410843171f,
   -0.910441292f, -0.413638312f,
   -0.909167983f, -0.416429560f,
   -0.907886116f, -0.419216888f,
   -0.906595705f, -0.422000271f,
   -0.905296759f, -0.424779681f,
   -0.903989293f, -0.427555093f,
   -0.902673318f, -0.430326481f,
   -0.901348847f, -0.433093819f,
   -0.900015892f, -0.435857080f,
   -0.898674466f, -0.438616239f,
   -0.897324581f, -0.441371269f,
   -0.895966250f, -0.444122145f,
   -0.894599486f, -0.446868840f,
   -0.893224301f, -0.449611330f,
   -0.891840709f, -0.452349587f,
   -0.890448723f, -0.455083587f,
   -0.889048356f, -0.457813304f,
   -0.887639620f, -0.460538711f,
   -0.886222530f, -0.463259784f,
   -0.884797098f, -0.465976496f,
   -0.883363339f, -0.468688822f,
   -0.881921264f, -0.471396737f,
   -0.880470889f, -0.474100215f,
   -0.879012226f, -0.476799230f,
   -0.877545290f, -0.479493758f,
   -0.876070094f, -0.482183772f,
   -0.874586652f, -0.484869248f,
   -0.873094978f, -0.487550160f,
   -0.871595087f, -0.490226483f,
   -0.870086991f, -0.492898192f,
   -0.868570706f, -0.495565262f,
   -0.867046246f, -0.498227667f,
   -0.865513624f, -0.500885383f,
   -0.863972856f, -0.503538384f,
   -0.862423956f, -0.506186645f,
   -0.860866939f, -0.508830143f,
   -0.859301818f, -0.511468850f,
   -0.857728610f, -0.514102744f,
   -0.856147328f, -0.516731799f,
   -0.854557988f, -0.519355990f,
   -0.852960605f, -0.521975293f,
   -0.851355193f, -0.524589683f,
   -0.849741768f, -0.527199135f,
   -0.848120345f, -0.529803625f,
   -0.846490939f, -0.532403128f,
   -0.844853565f, -0.534997620f,
   -0.843208240f, -0.537587076f,
   -0.841554977f, -0.540171473f,
   -0.839893794f, -0.542750785f,
   -0.838224706f, -0.545324988f,
   -0.836547727f, -0.547894059f,
   -0.834862875f, -0.550457973f,
   -0.833170165f, -0.553016706f,
   -0.831469612f, -0.555570233f,
   -0.829761234f, -0.558118531f,
   -0.828045045f, -0.560661576f,
   -0.826321063f, -0.563199344f,
   -0.824589303f, -0.565731811f,
   -0.822849781f, -0.568258953f,
   -0.821102515f, -0.570780746f,
   -0.819347520f, -0.573297167f,
   -0.817584813f, -0.575808191f,
   -0.815814411f, -0.578313796f,
   -0.814036330f, -0.580813958f,
   -0.812250587f, -0.583308653f,
   -0.810457198f, -0.585797857f,
   -0.808656182f, -0.588281548f,
   -0.806847554f, -0.590759702f,
   -0.805031331f, -0.593232295f,
   -0.803207531f, -0.595699304f,
   -0.801376172f, -0.598160707f,
   -0.799537269f, -0.600616479f,
   -0.797690841f, -0.603066599f,
   -0.795836905f, -0.605511041f,
   -0.793975478f, -0.607949785f,
   -0.792106577f, -0.610382806f,
   -0.790230221f, -0.612810082f,
   -0.788346428f, -0.615231591f,
   -0.786455214f, -0.617647308f,
   -0.784556597f, -0.620057212f,
   -0.782650596f, -0.622461279f,
   -0.780737229f, -0.624859488f,
   -0.778816512f, -0.627251815f,
   -0.776888466f, -0.629638239f,
   -0.774953107f, -0.632018736f,
   -0.773010453f, -0.634393284f,
   -0.771060524f, -0.636761861f,
   -0.769103338f, -0.639124445f,
   -0.767138912f, -0.641481013f,
   -0.765167266f, -0.643831543f,
   -0.763188417f, -0.646176013f,
   -0.761202385f, -0.648514401f,
   -0.759209189f, -0.650846685f,
   -0.757208847f, -0.653172843f,
   -0.755201377f, -0.655492853f,
   -0.753186799f, -0.657806693f,
   -0.751165132f, -0.660114342f,
   -0.749136395f, -0.662415778f,
   -0.747100606f, -0.664710978f,
   -0.745057785f, -0.666999922f,
   -0.743007952f, -0.669282588f,
   -0.740951125f, -0.671558955f,
   -0.738887324f, -0.673829000f,
   -0.736816569f, -0.676092704f,
   -0.734738878f, -0.678350043f,
   -0.732654272f, -0.680600998f,
   -0.730562769f, -0.682845546f,
   -0.728464390f, -0.685083668f,
   -0.726359155f, -0.687315341f,
   -0.724247083f, -0.689540545f,
   -0.722128194f, -0.691759258f,
   -0.720002508f, -0.693971461f,
   -0.717870045f, -0.696177131f,
   -0.715730825f, -0.698376249f,
   -0.713584869f, -0.700568794f,
   -0.711432196f, -0.702754744f,
   -0.709272826f, -0.704934080f,
   -0.707106781f, -0.707106781f,
   -0.704934080f, -0.709272826f,
   -0.702754744f, -0.711432196f,
   -0.700568794f, -0.713584869f,
   -0.698376249f, -0.715730825f,
   -0.696177131f, -0.717870045f,
   -0.693971461f, -0.720002508f,
   -0.691759258f, -0.722128194f,
   -0.689540545f, -0.724247083f,
   -0.687315341f, -0.726359155f,
   -0.685083668f, -0.728464390f,
   -0.682845546f, -0.730562769f,
   -0.680600998f, -0.732654272f,
   -0.678350043f, -0.734738878f,
   -0.676092704f, -0.736816569f,
   -0.673829000f, -0.738887324f,
   -0.671558955f, -0.740951125f,
   -0.669282588f, -0.743007952f,
   -0.666999922f, -0.745057785f,
   -0.664710978f, -0.747100606f,
   -0.662415778f, -0.749136395f,
   -0.660114342f, -0.751165132f,
   -0.657806693f, -0.753186799f,
   -0.655492853f, -0.755201377f,
   -0.653172843f, -0.757208847f,
   -0.650846685f, -0.759209189f,
   -0.648514401f, -0.761202385f,
   -0.646176013f, -0.763188417f,
   -0.643831543f, -0.765167266f,
   -0.641481013f, -0.767138912f,
   -0.639124445f, -0.769103338f,
   -0.636761861f, -0.771060524f,
   -0.634393284f, -0.773010453f,
   -0.632018736f, -0.774953107f,
   -0.629638239f, -0.776888466f,
   -0.627251815f, -0.778816512f,
   -0.624859488f, -0.780737229f,
   -0.622461279f, -0.782650596f,
   -0.620057212f, -0.784556597f,
   -0.617647308f, -0.786455214f,
   -0.615231591f, -0.788346428f,
   -0.612810082f, -0.790230221f,
   -0.610382806f, -0.792106577f,
   -0.607949785f, -0.793975478f,
   -0.605511041f, -0.795836905f,
   -0.603066599f, -0.797690841f,
   -0.600616479f, -0.799537269f,
   -0.598160707f, -0.801376172f,
   -0.595699304f, -0.803207531f,
   -0.593232295f, -0.805031331f,
   -0.590759702f, -0.806847554f,
   -0.588281548f, -0.808656182f,
   -0.585797857f, -0.810457198f,
   -0.583308653f, -0.812250587f,
   -0.580813958f, -0.814036330f,
   -0.578313796f, -0.815814411f,
   -0.575808191f, -0.817584813f,
   -0.573297167f, -0.819347520f,
   -0.570780746f, -0.821102515f,
   -0.568258953f, -0.822849781f,
   -0.565731811f, -0.824589303f,
   -0.563199344f, -0.826321063f,
   -0.560661576f, -0.828045045f,
   -0.558118531f, -0.829761234f,
   -0.555570233f, -0.831469612f,
   -0.553016706f, -0.833170165f,
   -0.550457973f, -0.834862875f,
   -0.547894059f, -0.836547727f,
   -0.545324988f, -0.838224706f,
   -0.542750785f, -0.839893794f,
   -0.540171473f, -0.841554977f,
   -0.537587076f, -0.843208240f,
   -0.534997620f, -0.844853565f,
   -0.532403128f, -0.846490939f,
   -0.529803625f, -0.848120345f,
   -0.527199135f, -0.849741768f,
   -0.524589683f, -0.851355193f,
   -0.521975293f, -0.852960605f,
   -0.519355990f, -0.854557988f,
   -0.516731799f, -0.856147328f,
   -0.514102744f, -0.857728610f,
   -0.511468850f, -0.859301818f,
   -0.508830143f, -0.860866939f,
   -0.506186645f, -0.862423956f,
   -0.503538384f, -0.863972856f,
   -0.500885383f, -0.865513624f,
   -0.498227667f, -0.867046246f,
   -0.495565262f, -0.868570706f,
   -0.492898192f, -0.870086991f,
   -0.490226483f, -0.871595087f,
   -0.487550160f, -0.873094978f,
   -0.484869248f, -0.874586652f,
   -0.482183772f, -0.876070094f,
   -0.479493758f, -0.877545290f,
   -0.476799230f, -0.879012226f,
   -0.474100215f, -0.880470889f,
   -0.471396737f, -0.881921264f,
   -0.468688822f, -0.883363339f,
   -0.465976496f, -0.884797098f,
   -0.463259784f, -0.886222530f,
   -0.460538711f, -0.887639620f,
   -0.457813304f, -0.889048356f,
   -0.455083587f, -0.890448723f,
   -0.452349587f, -0.891840709f,
   -0.449611330f, -0.893224301f,
   -0.446868840f, -0.894599486f,
   -0.444122145f, -0.895966250f,
   -0.441371269f, -0.897324581f,
   -0.438616239f, -0.898674466f,
   -0.435857080f, -0.900015892f,
   -0.433093819f, -0.901348847f,
   -0.430326481f, -0.902673318f,
   -0.427555093f, -0.903989293f,
   -0.424779681f, -0.905296759f,
   -0.422000271f, -0.906595705f,
   -0.419216888f, -0.907886116f,
   -0.416429560f, -0.909167983f,
   -0.413638312f, -0.910441292f,
   -0.410843171f, -0.911706032f,
   -0.408044163f, -0.912962190f,
   -0.405241314f, -0.914209756f,
   -0.402434651f, -0.915448716f,
   -0.399624200f, -0.916679060f,
   -0.396809987f, -0.917900776f,
   -0.393992040f, -0.919113852f,
   -0.391170384f, -0.920318277f,
   -0.388345047f, -0.921514039f,
   -0.385516054f, -0.922701128f,
   -0.382683432f, -0.923879533f,
   -0.379847209f, -0.925049241f,
   -0.377007410f, -0.926210242f,
   -0.374164063f, -0.927362526f,
   -0.371317194f, -0.928506080f,
   -0.368466830f, -0.929640896f,
   -0.365612998f, -0.930766961f,
   -0.362755724f, -0.931884266f,
   -0.359895037f, -0.932992799f,
   -0.357030961f, -0.934092550f,
   -0.354163525f, -0.935183510f,
   -0.351292756f, -0.936265667f,
   -0.348418680f, -0.937339012f,
   -0.345541325f, -0.938403534f,
   -0.342660717f, -0.939459224f,
   -0.339776884f, -0.940506071f,
   -0.336889853f, -0.941544065f,
   -0.333999651f, -0.942573198f,
   -0.331106306f, -0.943593458f,
   -0.328209844f, -0.944604837f,
   -0.325310292f, -0.945607325f,
   -0.322407679f, -0.946600913f,
   -0.319502031f, -0.947585591f,
   -0.316593376f, -0.948561350f,
   -0.313681740f, -0.949528181f,
   -0.310767153f, -0.950486074f,
   -0.307849640f, -0.951435021f,
   -0.304929230f, -0.952375013f,
   -0.302005949f, -0.953306040f,
   -0.299079826f, -0.954228095f,
   -0.296150888f, -0.955141168f,
   -0.293219163f, -0.956045251f,
   -0.290284677f, -0.956940336f,
   -0.287347460f, -0.957826413f,
   -0.284407537f, -0.958703475f,
   -0.281464938f, -0.959571513f,
   -0.278519689f, -0.960430519f,
   -0.275571819f, -0.961280486f,
   -0.272621355f, -0.962121404f,
   -0.269668326f, -0.962953267f,
   -0.266712757f, -0.963776066f,
   -0.263754679f, -0.964589793f,
   -0.260794118f, -0.965394442f,
   -0.257831102f, -0.966190003f,
   -0.254865660f, -0.966976471f,
   -0.251897818f, -0.967753837f,
   -0.248927606f, -0.968522094f,
   -0.245955050f, -0.969281235f,
   -0.242980180f, -0.970031253f,
   -0.240003022f, -0.970772141f,
   -0.237023606f, -0.971503891f,
   -0.234041959f, -0.972226497f,
   -0.231058108f, -0.972939952f,
   -0.228072083f, -0.973644250f,
   -0.225083911f, -0.974339383f,
   -0.222093621f, -0.975025345f,
   -0.219101240f, -0.975702130f,
   -0.216106797f, -0.976369731f,
   -0.213110320f, -0.977028143f,
   -0.210111837f, -0.977677358f,
   -0.207111376f, -0.978317371f,
   -0.204108966f, -0.978948175f,
   -0.201104635f, -0.979569766f,
   -0.198098411f, -0.980182136f,
   -0.195090322f, -0.980785280f,
   -0.192080397f, -0.981379193f,
   -0.189068664f, -0.981963869f,
   -0.186055152f, -0.982539302f,
   -0.183039888f, -0.983105487f,
   -0.180022901f, -0.983662419f,
   -0.177004220f, -0.984210092f,
   -0.173983873f, -0.984748502f,
   -0.170961889f, -0.985277642f,
   -0.167938295f, -0.985797509f,
   -0.164913120f, -0.986308097f,
   -0.161886394f, -0.986809402f,
   -0.158858143f, -0.987301418f,
   -0.155828398f, -0.987784142f,
   -0.152797185f, -0.988257568f,
   -0.149764535f, -0.988721692f,
   -0.146730474f, -0.989176510f,
   -0.143695033f, -0.989622017f,
   -0.140658239f, -0.990058210f,
   -0.137620122f, -0.990485084f,
   -0.134580709f, -0.990902635f,
   -0.131540029f, -0.991310860f,
   -0.128498111f, -0.991709754f,
   -0.125454983f, -0.992099313f,
   -0.122410675f, -0.992479535f,
   -0.119365215f, -0.992850414f,
   -0.116318631f, -0.993211949f,
   -0.113270952f, -0.993564136f,
   -0.110222207f, -0.993906970f,
   -0.107172425f, -0.994240449f,
   -0.104121634f, -0.994564571f,
   -0.101069863f, -0.994879331f,
   -0.098017140f, -0.995184727f,
   -0.094963495f, -0.995480755f,
   -0.091908956f, -0.995767414f,
   -0.088853553f, -0.996044701f,
   -0.085797312f, -0.996312612f,
   -0.082740265f, -0.996571146f,
   -0.079682438f, -0.996820299f,
   -0.076623861f, -0.997060070f,
   -0.073564564f, -0.997290457f,
   -0.070504573f, -0.997511456f,
   -0.067443920f, -0.997723067f,
   -0.064382631f, -0.997925286f,
   -0.061320736f, -0.998118113f,
   -0.058258265f, -0.998301545f,
   -0.055195244f, -0.998475581f,
   -0.052131705f, -0.998640218f,
   -0.049067674f, -0.998795456f,
   -0.046003182f, -0.998941293f,
   -0.042938257f, -0.999077728f,
   -0.039872928f, -0.999204759f,
   -0.036807223f, -0.999322385f,
   -0.033741172f, -0.999430605f,
   -0.030674803f, -0.999529418f,
   -0.027608146f, -0.999618822f,
   -0.024541229f, -0.999698819f,
   -0.021474080f, -0.999769405f,
   -0.018406730f, -0.999830582f,
   -0.015339206f, -0.999882347f,
   -0.012271538f, -0.999924702f,
   -0.009203755f, -0.999957645f,
   -0.006135885f, -0.999981175f,
   -0.003067957f, -0.999995294f,
   -0.000000000f, -1.000000000f,
    0.003067957f, -0.999995294f,
    0.006135885f, -0.999981175f,
    0.009203755f, -0.999957645f,
    0.012271538f, -0.999924702f,
    0.015339206f, -0.999882347f,
    0.018406730f, -0.999830582f,
    0.021474080f, -0.999769405f,
    0.024541229f, -0.999698819f,
    0.027608146f, -0.999618822f,
    0.030674803f, -0.999529418f,
    0.033741172f, -0.999430605f,
    0.036807223f, -0.999322385f,
    0.039872928f, -0.999204759f,
    0.042938257f, -0.999077728f,
    0.046003182f, -0.998941293f,
    0.049067674f, -0.998795456f,
    0.052131705f, -0.998640218f,
    0.055195244f, -0.998475581f,
    0.058258265f, -0.998301545f,
    0.061320736f, -0.998118113f,
    0.064382631f, -0.997925286f,
    0.067443920f, -0.997723067f,
    0.070504573f, -0.997511456f,
    0.073564564f, -0.997290457f,
    0.076623861f, -0.997060070f,
    0.079682438f, -0.996820299f,
    0.082740265f, -0.996571146f,
    0.085797312f, -0.996312612f,
    0.088853553f, -0.996044701f,
    0.091908956f, -0.995767414f,
    0.094963495f, -0.995480755f,
    0.098017140f, -0.995184727f,
    0.101069863f, -0.994879331f,
    0.104121634f, -0.994564571f,
    0.107172425f, -0.994240449f,
    0.110222207f, -0.993906970f,
    0.113270952f, -0.993564136f,
    0.116318631f, -0.993211949f,
    0.119365215f, -0.992850414f,
    0.122410675f, -0.992479535f,
    0.125454983f, -0.992099313f,
    0.128498111f, -0.991709754f,
    0.131540029f, -0.991310860f,
    0.134580709f, -0.990902635f,
    0.137620122f, -0.990485084f,
    0.140658239f, -0.990058210f,
    0.143695033f, -0.989622017f,
    0.146730474f, -0.989176510f,
    0.149764535f, -0.988721692f,
    0.152797185f, -0.988257568f,
    0.155828398f, -0.987784142f,
    0.158858143f, -0.987301418f,
    0.161886394f, -0.986809402f,
    0.164913120f, -0.986308097f,
    0.167938295f, -0.985797509f,
    0.170961889f, -0.985277642f,
    0.173983873f, -0.984748502f,
    0.177004220f, -0.984210092f,
    0.180022901f, -0.983662419f,
    0.183039888f, -0.983105487f,
    0.186055152f, -0.982539302f,
    0.189068664f, -0.981963869f,
    0.192080397f, -0.981379193f,
    0.195090322f, -0.980785280f,
    0.198098411f, -0.980182136f,
    0.201104635f, -0.979569766f,
    0.204108966f, -0.978948175f,
    0.207111376f, -0.978317371f,
    0.210111837f, -0.977677358f,
    0.213110320f, -0.977028143f,
    0.216106797f, -0.976369731f,
    0.219101240f, -0.975702130f,
    0.222093621f, -0.975025345f,
    0.225083911f, -0.974339383f,
    0.228072083f, -0.973644250f,
    0.231058108f, -0.972939952f,
    0.234041959f, -0.972226497f,
    0.237023606f, -0.971503891f,
    0.240003022f, -0.970772141f,
    0.242980180f, -0.970031253f,
    0.245955050f, -0.969281235f,
    0.248927606f, -0.968522094f,
    0.251897818f, -0.967753837f,
    0.254865660f, -0.966976471f,
    0.257831102f, -0.966190003f,
    0.260794118f, -0.965394442f,
    0.263754679f, -0.964589793f,
    0.266712757f, -0.963776066f,
    0.269668326f, -0.962953267f,
    0.272621355f, -0.962121404f,
    0.275571819f, -0.961280486f,
    0.278519689f, -0.960430519f,
    0.281464938f, -0.959571513f,
    0.284407537f, -0.958703475f,
    0.287347460f, -0.957826413f,
    0.290284677f, -0.956940336f,
    0.293219163f, -0.956045251f,
    0.296150888f, -0.955141168f,
    0.299079826f, -0.954228095f,
    0.302005949f, -0.953306040f,
    0.304929230f, -0.952375013f,
    0.307849640f, -0.951435021f,
    0.310767153f, -0.950486074f,
    0.313681740f, -0.949528181f,
    0.316593376f, -0.948561350f,
    0.319502031f, -0.947585591f,
    0.322407679f, -0.946600913f,
    0.325310292f, -0.945607325f,
    0.328209844f, -0.944604837f,
    0.331106306f, -0.943593458f,
    0.333999651f, -0.942573198f,
    0.336889853f, -0.941544065f,
    0.339776884f, -0.940506071f,
    0.342660717f, -0.939459224f,
    0.345541325f, -0.938403534f,
    0.348418680f, -0.937339012f,
    0.351292756f, -0.936265667f,
    0.354163525f, -0.935183510f,
    0.357030961f, -0.934092550f,
    0.359895037f, -0.932992799f,
    0.362755724f, -0.931884266f,
    0.365612998f, -0.930766961f,
    0.368466830f, -0.929640896f,
    0.371317194f, -0.928506080f,
    0.374164063f, -0.927362526f,
    0.377007410f, -0.926210242f,
    0.379847209f, -0.925049241f,
    0.382683432f, -0.923879533f,
    0.385516054f, -0.922701128f,
    0.388345047f, -0.921514039f,
    0.391170384f, -0.920318277f,
    0.393992040f, -0.919113852f,
    0.396809987f, -0.917900776f,
    0.399624200f, -0.916679060f,
    0.402434651f, -0.915448716f,
    0.405241314f, -0.914209756f,
    0.408044163f, -0.912962190f,
    0.410843171f, -0.911706032f,
    0.413638312f, -0.910441292f,
    0.416429560f, -0.909167983f,
    0.419216888f, -0.907886116f,
    0.422000271f, -0.906595705f,
    0.424779681f, -0.905296759f,
    0.427555093f, -0.903989293f,
    0.430326481f, -0.902673318f,
    0.433093819f, -0.901348847f,
    0.435857080f, -0.900015892f,
    0.438616239f, -0.898674466f,
    0.441371269f, -0.897324581f,
    0.444122145f, -0.895966250f,
    0.446868840f, -0.894599486f,
    0.449611330f, -0.893224301f,
    0.452349587f, -0.891840709f,
    0.455083587f, -0.890448723f,
    0.457813304f, -0.889048356f,
    0.460538711f, -0.887639620f,
    0.463259784f, -0.886222530f,
    0.465976496f, -0.884797098f,
    0.468688822f, -0.883363339f,
    0.471396737f, -0.881921264f,
    0.474100215f, -0.880470889f,
    0.476799230f, -0.879012226f,
    0.479493758f, -0.877545290f,
    0.482183772f, -0.876070094f,
    0.484869248f, -0.874586652f,
    0.487550160f, -0.873094978f,
    0.490226483f, -0.871595087f,
    0.492898192f, -0.870086991f,
    0.495565262f, -0.868570706f,
    0.498227667f, -0.867046246f,
    0.500885383f, -0.865513624f,
    0.503538384f, -0.863972856f,
    0.506186645f, -0.862423956f,
    0.508830143f, -0.860866939f,
    0.511468850f, -0.859301818f,
    0.514102744f, -0.857728610f,
    0.516731799f, -0.856147328f,
    0.519355990f, -0.854557988f,
    0.521975293f, -0.852960605f,
    0.524589683f, -0.851355193f,
    0.527199135f, -0.849741768f,
    0.529803625f, -0.848120345f,
    0.532403128f, -0.846490939f,
    0.534997620f, -0.844853565f,
    0.537587076f, -0.843208240f,
    0.540171473f, -0.841554977f,
    0.542750785f, -0.839893794f,
    0.545324988f, -0.838224706f,
    0.547894059f, -0.836547727f,
    0.550457973f, -0.834862875f,
    0.553016706f, -0.833170165f,
    0.555570233f, -0.831469612f,
    0.558118531f, -0.829761234f,
    0.560661576f, -0.828045045f,
    0.563199344f, -0.826321063f,
    0.565731811f, -0.824589303f,
    0.568258953f, -0.822849781f,
    0.570780746f, -0.821102515f,
    0.573297167f, -0.819347520f,
    0.575808191f, -0.817584813f,
    0.578313796f, -0.815814411f,
    0.580813958f, -0.814036330f,
    0.583308653f, -0.812250587f,
    0.585797857f, -0.810457198f,
    0.588281548f, -0.808656182f,
    0.590759702f, -0.806847554f,
    0.593232295f, -0.805031331f,
    0.595699304f, -0.803207531f,
    0.598160707f, -0.801376172f,
    0.600616479f, -0.799537269f,
    0.603066599f, -0.797690841f,
    0.605511041f, -0.795836905f,
    0.607949785f, -0.793975478f,
    0.610382806f, -0.792106577f,
    0.612810082f, -0.790230221f,
    0.615231591f, -0.788346428f,
    0.617647308f, -0.786455214f,
    0.620057212f, -0.784556597f,
    0.622461279f, -0.782650596f,
    0.624859488f, -0.780737229f,
    0.627251815f, -0.778816512f,
    0.629638239f, -0.776888466f,
    0.632018736f, -0.774953107f,
    0.634393284f, -0.773010453f,
    0.636761861f, -0.771060524f,
    0.639124445f, -0.769103338f,
    0.641481013f, -0.767138912f,
    0.643831543f, -0.765167266f,
    0.646176013f, -0.763188417f,
    0.648514401f, -0.761202385f,
    0.650846685f, -0.759209189f,
    0.653172843f, -0.757208847f,
    0.655492853f, -0.755201377f,
    0.657806693f, -0.753186799f,
    0.660114342f, -0.751165132f,
    0.662415778f, -0.749136395f,
    0.664710978f, -0.747100606f,
    0.666999922f, -0.745057785f,
    0.669282588f, -0.743007952f,
    0.671558955f, -0.740951125f,
    0.673829000f, -0.738887324f,
    0.676092704f, -0.736816569f,
    0.678350043f, -0.734738878f,
    0.680600998f, -0.732654272f,
    0.682845546f, -0.730562769f,
    0.685083668f, -0.728464390f,
    0.687315341f, -0.726359155f,
    0.689540545f, -0.724247083f,
    0.691759258f, -0.722128194f,
    0.693971461f, -0.720002508f,
    0.696177131f, -0.717870045f,
    0.698376249f, -0.715730825f,
    0.700568794f, -0.713584869f,
    0.702754744f, -0.711432196f,
    0.704934080f, -0.709272826f,
    0.707106781f, -0.707106781f,
    0.709272826f, -0.704934080f,
    0.711432196f, -0.702754744f,
    0.713584869f, -0.700568794f,
    0.715730825f, -0.698376249f,
    0.717870045f, -0.696177131f,
    0.720002508f, -0.693971461f,
    0.722128194f, -0.691759258f,
    0.724247083f, -0.689540545f,
    0.726359155f, -0.687315341f,
    0.728464390f, -0.685083668f,
    0.730562769f, -0.682845546f,
    0.732654272f, -0.680600998f,
    0.734738878f, -0.678350043f,
    0.736816569f, -0.676092704f,
    0.738887324f, -0.673829000f,
    0.740951125f, -0.671558955f,
    0.743007952f, -0.669282588f,
    0.745057785f, -0.666999922f,
    0.747100606f, -0.664710978f,
    0.749136395f, -0.662415778f,
    0.751165132f, -0.660114342f,
    0.753186799f, -0.657806693f,
    0.755201377f, -0.655492853f,
    0.757208847f, -0.653172843f,
    0.759209189f, -0.650846685f,
    0.761202385f, -0.648514401f,
    0.763188417f, -0.646176013f,
    0.765167266f, -0.643831543f,
    0.767138912f, -0.641481013f,
    0.769103338f, -0.639124445f,
    0.771060524f, -0.636761861f,
    0.773010453f, -0.634393284f,
    0.774953107f, -0.632018736f,
    0.776888466f, -0.629638239f,
    0.778816512f, -0.627251815f,
    0.780737229f, -0.624859488f,
    0.782650596f, -0.622461279f,
    0.784556597f, -0.620057212f,
    0.786455214f, -0.617647308f,
    0.788346428f, -0.615231591f,
    0.790230221f, -0.612810082f,
    0.792106577f, -0.610382806f,
    0.793975478f, -0.607949785f,
    0.795836905f, -0.605511041f,
    0.797690841f, -0.603066599f,
    0.799537269f, -0.600616479f,
    0.801376172f, -0.598160707f,
    0.803207531f, -0.595699304f,
    0.805031331f, -0.593232295f,
    0.806847554f, -0.590759702f,
    0.808656182f, -0.588281548f,
    0.810457198f, -0.585797857f,
    0.812250587f, -0.583308653f,
    0.814036330f, -0.580813958f,
    0.815814411f, -0.578313796f,
    0.817584813f, -0.575808191f,
    0.819347520f, -0.573297167f,
    0.821102515f, -0.570780746f,
    0.822849781f, -0.568258953f,
    0.824589303f, -0.565731811f,
    0.826321063f, -0.563199344f,
    0.828045045f, -0.560661576f,
    0.829761234f, -0.558118531f,
    0.831469612f, -0.555570233f,
    0.833170165f, -0.553016706f,
    0.834862875f, -0.550457973f,
    0.836547727f, -0.547894059f,
    0.838224706f, -0.545324988f,
    0.839893794f, -0.542750785f,
    0.841554977f, -0.540171473f,
    0.843208240f, -0.537587076f,
    0.844853565f, -0.534997620f,
    0.846490939f, -0.532403128f,
    0.848120345f, -0.529803625f,
    0.849741768f, -0.527199135f,
    0.851355193f, -0.524589683f,
    0.852960605f, -0.521975293f,
    0.854557988f, -0.519355990f,
    0.856147328f, -0.516731799f,
    0.857728610f, -0.514102744f,
    0.859301818f, -0.511468850f,
    0.860866939f, -0.508830143f,
    0.862423956f, -0.506186645f,
    0.863972856f, -0.503538384f,
    0.865513624f, -0.500885383f,
    0.867046246f, -0.498227667f,
    0.868570706f, -0.495565262f,
    0.870086991f, -0.492898192f,
    0.871595087f, -0.490226483f,
    0.873094978f, -0.487550160f,
    0.874586652f, -0.484869248f,
    0.876070094f, -0.482183772f,
    0.877545290f, -0.479493758f,
    0.879012226f, -0.476799230f,
    0.880470889f, -0.474100215f,
    0.881921264f, -0.471396737f,
    0.883363339f, -0.468688822f,
    0.884797098f, -0.465976496f,
    0.886222530f, -0.463259784f,
    0.887639620f, -0.460538711f,
    0.889048356f, -0.457813304f,
    0.890448723f, -0.455083587f,
    0.891840709f, -0.452349587f,
    0.893224301f, -0.449611330f,
    0.894599486f, -0.446868840f,
    0.895966250f, -0.444122145f,
    0.897324581f, -0.441371269f,
    0.898674466f, -0.438616239f,
    0.900015892f, -0.435857080f,
    0.901348847f, -0.433093819f,
    0.902673318f, -0.430326481f,
    0.903989293f, -0.427555093f,
    0.905296759f, -0.424779681f,
    0.906595705f, -0.422000271f,
    0.907886116f, -0.419216888f,
    0.909167983f, -0.416429560f,
    0.910441292f, -0.413638312f,
    0.911706032f, -0.410843171f,
    0.912962190f, -0.408044163f,
    0.914209756f, -0.405241314f,
    0.915448716f, -0.402434651f,
    0.916679060f, -0.399624200f,
    0.917900776f, -0.396809987f,
    0.919113852f, -0.393992040f,
    0.920318277f, -0.391170384f,
    0.921514039f, -0.388345047f,
    0.922701128f, -0.385516054f,
    0.923879533f, -0.382683432f,
    0.925049241f, -0.379847209f,
    0.926210242f, -0.377007410f,
    0.927362526f, -0.374164063f,
    0.928506080f, -0.371317194f,
    0.929640896f, -0.368466830f,
    0.930766961f, -0.365612998f,
    0.931884266f, -0.362755724f,
    0.932992799f, -0.359895037f,
    0.934092550f, -0.357030961f,
    0.935183510f, -0.354163525f,
    0.936265667f, -0.351292756f,
    0.937339012f, -0.348418680f,
    0.938403534f, -0.345541325f,
    0.939459224f, -0.342660717f,
    0.940506071f, -0.339776884f,
    0.941544065f, -0.336889853f,
    0.942573198f, -0.333999651f,
    0.943593458f, -0.331106306f,
    0.944604837f, -0.328209844f,
    0.945607325f, -0.325310292f,
    0.946600913f, -0.322407679f,
    0.947585591f, -0.319502031f,
    0.948561350f, -0.316593376f,
    0.949528181f, -0.313681740f,
    0.950486074f, -0.310767153f,
    0.951435021f, -0.307849640f,
    0.952375013f, -0.304929230f,
    0.953306040f, -0.302005949f,
    0.954228095f, -0.299079826f,
    0.955141168f, -0.296150888f,
    0.956045251f, -0.293219163f,
    0.956940336f, -0.290284677f,
    0.957826413f, -0.287347460f,
    0.958703475f, -0.284407537f,
    0.959571513f, -0.281464938f,
    0.960430519f, -0.278519689f,
    0.961280486f, -0.275571819f,
    0.962121404f, -0.272621355f,
    0.962953267f, -0.269668326f,
    0.963776066f, -0.266712757f,
    0.964589793f, -0.263754679f,
    0.965394442f, -0.260794118f,
    0.966190003f, -0.257831102f,
    0.966976471f, -0.254865660f,
    0.967753837f, -0.251897818f,
    0.968522094f, -0.248927606f,
    0.969281235f, -0.245955050f,
    0.970031253f, -0.242980180f,
    0.970772141f, -0.240003022f,
    0.971503891f, -0.237023606f,
    0.972226497f, -0.234041959f,
    0.972939952f, -0.231058108f,
    0.973644250f, -0.228072083f,
    0.974339383f, -0.225083911f,
    0.975025345f, -0.222093621f,
    0.975702130f, -0.219101240f,
    0.976369731f, -0.216106797f,
    0.977028143f, -0.213110320f,
    0.977677358f, -0.210111837f,
    0.978317371f, -0.207111376f,
    0.978948175f, -0.204108966f,
    0.979569766f, -0.201104635f,
    0.980182136f, -0.198098411f,
    0.980785280f, -0.195090322f,
    0.981379193f, -0.192080397f,
    0.981963869f, -0.189068664f,
    0.982539302f, -0.186055152f,
    0.983105487f, -0.183039888f,
    0.983662419f, -0.180022901f,
    0.984210092f, -0.177004220f,
    0.984748502f, -0.173983873f,
    0.985277642f, -0.170961889f,
    0.985797509f, -0.167938295f,
    0.986308097f, -0.164913120f,
    0.986809402f, -0.161886394f,
    0.987301418f, -0.158858143f,
    0.987784142f, -0.155828398f,
    0.988257568f, -0.152797185f,
    0.988721692f, -0.149764535f,
    0.989176510f, -0.146730474f,
    0.989622017f, -0.143695033f,
    0.990058210f, -0.140658239f,
    0.990485084f, -0.137620122f,
    0.990902635f, -0.134580709f,
    0.991310860f, -0.131540029f,
    0.991709754f, -0.128498111f,
    0.992099313f, -0.125454983f,
    0.992479535f, -0.122410675f,
    0.992850414f, -0.119365215f,
    0.993211949f, -0.116318631f,
    0.993564136f, -0.113270952f,
    0.993906970f, -0.110222207f,
    0.994240449f, -0.107172425f,
    0.994564571f, -0.104121634f,
    0.994879331f, -0.101069863f,
    0.995184727f, -0.098017140f,
    0.995480755f, -0.094963495f,
    0.995767414f, -0.091908956f,
    0.996044701f, -0.088853553f,
    0.996312612f, -0.085797312f,
    0.996571146f, -0.082740265f,
    0.996820299f, -0.079682438f,
    0.997060070f, -0.076623861f,
    0.997290457f, -0.073564564f,
    0.997511456f, -0.070504573f,
    0.997723067f, -0.067443920f,
    0.997925286f, -0.064382631f,
    0.998118113f, -0.061320736f,
    0.998301545f, -0.058258265f,
    0.998475581f, -0.055195244f,
    0.998640218f, -0.052131705f,
    0.998795456f, -0.049067674f,
    0.998941293f, -0.046003182f,
    0.999077728f, -0.042938257f,
    0.999204759f, -0.039872928f,
    0.999322385f, -0.036807223f,
    0.999430605f, -0.033741172f,
    0.999529418f, -0.030674803f,
    0.999618822f, -0.027608146f,
    0.999698819f, -0.024541229f,
    0.999769405f, -0.021474080f,
    0.999830582f, -0.018406730f,
    0.999882347f, -0.015339206f,
    0.999924702f, -0.012271538f,
    0.999957645f, -0.009203755f,
    0.999981175f, -0.006135885f,
    0.999995294f, -0.003067957f
};

/**
* \par
* Example code for Floating-point Twiddle factors Generation:
* \par
* <pre>for(i = 0; i< N/; i++)
* {
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 4096	and PI = 3.14159265358979
* \par
* Cos and Sin values are in interleaved fashion
*
*/
const float32_t twiddleCoef_4096[8192] = {
    1.000000000f,  0.000000000f,
    0.999998823f,  0.001533980f,
    0.999995294f,  0.003067957f,
    0.999989411f,  0.004601926f,
    0.999981175f,  0.006135885f,
    0.999970586f,  0.007669829f,
    0.999957645f,  0.009203755f,
    0.999942350f,  0.010737659f,
    0.999924702f,  0.012271538f,
    0.999904701f,  0.013805389f,
    0.999882347f,  0.015339206f,
    0.999857641f,  0.016872988f,
    0.999830582f,  0.018406730f,
    0.999801170f,  0.019940429f,
    0.999769405f,  0.021474080f,
    0.999735288f,  0.023007681f,
    0.999698819f,  0.024541229f,
    0.999659997f,  0.026074718f,
    0.999618822f,  0.027608146f,
    0.999575296f,  0.029141509f,
    0.999529418f,  0.030674803f,
    0.999481187f,  0.032208025f,
    0.999430605f,  0.033741172f,
    0.999377670f,  0.035274239f,
    0.999322385f,  0.036807223f,
    0.999264747f,  0.038340120f,
    0.999204759f,  0.039872928f,
    0.999142419f,  0.041405641f,
    0.999077728f,  0.042938257f,
    0.999010686f,  0.044470772f,
    0.998941293f,  0.046003182f,
    0.998869550f,  0.047535484f,
    0.998795456f,  0.049067674f,
    0.998719012f,  0.050599749f,
    0.998640218f,  0.052131705f,
    0.998559074f,  0.053663538f,
    0.998475581f,  0.055195244f,
    0.998389737f,  0.056726821f,
    0.998301545f,  0.058258265f,
    0.998211003f,  0.059789571f,
    0.998118113f,  0.061320736f,
    0.998022874f,  0.062851758f,
    0.997925286f,  0.064382631f,
    0.997825350f,  0.065913353f,
    0.997723067f,  0.067443920f,
    0.997618435f,  0.068974328f,
    0.997511456f,  0.070504573f,
    0.997402130f,  0.072034653f,
    0.997290457f,  0.073564564f,
    0.997176437f,  0.075094301f,
    0.997060070f,  0.076623861f,
    0.996941358f,  0.078153242f,
    0.996820299f,  0.079682438f,
    0.996696895f,  0.081211447f,
    0.996571146f,  0.082740265f,
    0.996443051f,  0.084268888f,
    0.996312612f,  0.085797312f,
    0.996179829f,  0.087325535f,
    0.996044701f,  0.088853553f,
    0.995907229f,  0.090381361f,
    0.995767414f,  0.091908956f,
    0.995625256f,  0.093436336f,
    0.995480755f,  0.094963495f,
    0.995333912f,  0.096490431f,
    0.995184727f,  0.098017140f,
    0.995033199f,  0.099543619f,
    0.994879331f,  0.101069863f,
    0.994723121f,  0.102595869f,
    0.994564571f,  0.104121634f,
    0.994403680f,  0.105647154f,
    0.994240449f,  0.107172425f,
    0.994074879f,  0.108697444f,
    0.993906970f,  0.110222207f,
    0.993736722f,  0.111746711f,
    0.993564136f,  0.113270952f,
    0.993389211f,  0.114794927f,
    0.993211949f,  0.116318631f,
    0.993032350f,  0.117842062f,
    0.992850414f,  0.119365215f,
    0.992666142f,  0.120888087f,
    0.992479535f,  0.122410675f,
    0.992290591f,  0.123932975f,
    0.992099313f,  0.125454983f,
    0.991905700f,  0.126976696f,
    0.991709754f,  0.128498111f,
    0.991511473f,  0.130019223f,
    0.991310860f,  0.131540029f,
    0.991107914f,  0.133060525f,
    0.990902635f,  0.134580709f,
    0.990695025f,  0.136100575f,
    0.990485084f,  0.137620122f,
    0.990272812f,  0.139139344f,
    0.990058210f,  0.140658239f,
    0.989841278f,  0.142176804f,
    0.989622017f,  0.143695033f,
    0.989400428f,  0.145212925f,
    0.989176510f,  0.146730474f,
    0.988950265f,  0.148247679f,
    0.988721692f,  0.149764535f,
    0.988490793f,  0.151281038f,
    0.988257568f,  0.152797185f,
    0.988022017f,  0.154312973f,
    0.987784142f,  0.155828398f,
    0.987543942f,  0.157343456f,
    0.987301418f,  0.158858143f,
    0.987056571f,  0.160372457f,
    0.986809402f,  0.161886394f,
    0.986559910f,  0.163399949f,
    0.986308097f,  0.164913120f,
    0.986053963f,  0.166425904f,
    0.985797509f,  0.167938295f,
    0.985538735f,  0.169450291f,
    0.985277642f,  0.170961889f,
    0.985014231f,  0.172473084f,
    0.984748502f,  0.173983873f,
    0.984480455f,  0.175494253f,
    0.984210092f,  0.177004220f,
    0.983937413f,  0.178513771f,
    0.983662419f,  0.180022901f,
    0.983385110f,  0.181531608f,
    0.983105487f,  0.183039888f,
    0.982823551f,  0.184547737f,
    0.982539302f,  0.186055152f,
    0.982252741f,  0.187562129f,
    0.981963869f,  0.189068664f,
    0.981672686f,  0.190574755f,
    0.981379193f,  0.192080397f,
    0.981083391f,  0.193585587f,
    0.980785280f,  0.195090322f,
    0.980484862f,  0.196594598f,
    0.980182136f,  0.198098411f,
    0.979877104f,  0.199601758f,
    0.979569766f,  0.201104635f,
    0.979260123f,  0.202607039f,
    0.978948175f,  0.204108966f,
    0.978633924f,  0.205610413f,
    0.978317371f,  0.207111376f,
    0.977998515f,  0.208611852f,
    0.977677358f,  0.210111837f,
    0.977353900f,  0.211611327f,
    0.977028143f,  0.213110320f,
    0.976700086f,  0.214608811f,
    0.976369731f,  0.216106797f,
    0.976037079f,  0.217604275f,
    0.975702130f,  0.219101240f,
    0.975364885f,  0.220597690f,
    0.975025345f,  0.222093621f,
    0.974683511f,  0.223589029f,
    0.974339383f,  0.225083911f,
    0.973992962f,  0.226578264f,
    0.973644250f,  0.228072083f,
    0.973293246f,  0.229565366f,
    0.972939952f,  0.231058108f,
    0.972584369f,  0.232550307f,
    0.972226497f,  0.234041959f,
    0.971866337f,  0.235533059f,
    0.971503891f,  0.237023606f,
    0.971139158f,  0.238513595f,
    0.970772141f,  0.240003022f,
    0.970402839f,  0.241491885f,
    0.970031253f,  0.242980180f,
    0.969657385f,  0.244467903f,
    0.969281235f,  0.245955050f,
    0.968902805f,  0.247441619f,
    0.968522094f,  0.248927606f,
    0.968139105f,  0.250413007f,
    0.967753837f,  0.251897818f,
    0.967366292f,  0.253382037f,
    0.966976471f,  0.254865660f,
    0.966584374f,  0.256348682f,
    0.966190003f,  0.257831102f,
    0.965793359f,  0.259312915f,
    0.965394442f,  0.260794118f,
    0.964993253f,  0.262274707f,
    0.964589793f,  0.263754679f,
    0.964184064f,  0.265234030f,
    0.963776066f,  0.266712757f,
    0.963365800f,  0.268190857f,
    0.962953267f,  0.269668326f,
    0.962538468f,  0.271145160f,
    0.962121404f,  0.272621355f,
    0.961702077f,  0.274096910f,
    0.961280486f,  0.275571819f,
    0.960856633f,  0.277046080f,
    0.960430519f,  0.278519689f,
    0.960002146f,  0.279992643f,
    0.959571513f,  0.281464938f,
    0.959138622f,  0.282936570f,
    0.958703475f,  0.284407537f,
    0.958266071f,  0.285877835f,
    0.957826413f,  0.287347460f,
    0.957384501f,  0.288816408f,
    0.956940336f,  0.290284677f,
    0.956493919f,  0.291752263f,
    0.956045251f,  0.293219163f,
    0.955594334f,  0.294685372f,
    0.955141168f,  0.296150888f,
    0.954685755f,  0.297615707f,
    0.954228095f,  0.299079826f,
    0.953768190f,  0.300543241f,
    0.953306040f,  0.302005949f,
    0.952841648f,  0.303467947f,
    0.952375013f,  0.304929230f,
    0.951906137f,  0.306389795f,
    0.951435021f,  0.307849640f,
    0.950961666f,  0.309308760f,
    0.950486074f,  0.310767153f,
    0.950008245f,  0.312224814f,
    0.949528181f,  0.313681740f,
    0.949045882f,  0.315137929f,
    0.948561350f,  0.316593376f,
    0.948074586f,  0.318048077f,
    0.947585591f,  0.319502031f,
    0.947094366f,  0.320955232f,
    0.946600913f,  0.322407679f,
    0.946105232f,  0.323859367f,
    0.945607325f,  0.325310292f,
    0.945107193f,  0.326760452f,
    0.944604837f,  0.328209844f,
    0.944100258f,  0.329658463f,
    0.943593458f,  0.331106306f,
    0.943084437f,  0.332553370f,
    0.942573198f,  0.333999651f,
    0.942059740f,  0.335445147f,
    0.941544065f,  0.336889853f,
    0.941026175f,  0.338333767f,
    0.940506071f,  0.339776884f,
    0.939983753f,  0.341219202f,
    0.939459224f,  0.342660717f,
    0.938932484f,  0.344101426f,
    0.938403534f,  0.345541325f,
    0.937872376f,  0.346980411f,
    0.937339012f,  0.348418680f,
    0.936803442f,  0.349856130f,
    0.936265667f,  0.351292756f,
    0.935725689f,  0.352728556f,
    0.935183510f,  0.354163525f,
    0.934639130f,  0.355597662f,
    0.934092550f,  0.357030961f,
    0.933543773f,  0.358463421f,
    0.932992799f,  0.359895037f,
    0.932439629f,  0.361325806f,
    0.931884266f,  0.362755724f,
    0.931326709f,  0.364184790f,
    0.930766961f,  0.365612998f,
    0.930205023f,  0.367040346f,
    0.929640896f,  0.368466830f,
    0.929074581f,  0.369892447f,
    0.928506080f,  0.371317194f,
    0.927935395f,  0.372741067f,
    0.927362526f,  0.374164063f,
    0.926787474f,  0.375586178f,
    0.926210242f,  0.377007410f,
    0.925630831f,  0.378427755f,
    0.925049241f,  0.379847209f,
    0.924465474f,  0.381265769f,
    0.923879533f,  0.382683432f,
    0.923291417f,  0.384100195f,
    0.922701128f,  0.385516054f,
    0.922108669f,  0.386931006f,
    0.921514039f,  0.388345047f,
    0.920917242f,  0.389758174f,
    0.920318277f,  0.391170384f,
    0.919717146f,  0.392581674f,
    0.919113852f,  0.393992040f,
    0.918508394f,  0.395401479f,
    0.917900776f,  0.396809987f,
    0.917290997f,  0.398217562f,
    0.916679060f,  0.399624200f,
    0.916064966f,  0.401029897f,
    0.915448716f,  0.402434651f,
    0.914830312f,  0.403838458f,
    0.914209756f,  0.405241314f,
    0.913587048f,  0.406643217f,
    0.912962190f,  0.408044163f,
    0.912335185f,  0.409444149f,
    0.911706032f,  0.410843171f,
    0.911074734f,  0.412241227f,
    0.910441292f,  0.413638312f,
    0.909805708f,  0.415034424f,
    0.909167983f,  0.416429560f,
    0.908528119f,  0.417823716f,
    0.907886116f,  0.419216888f,
    0.907241978f,  0.420609074f,
    0.906595705f,  0.422000271f,
    0.905947298f,  0.423390474f,
    0.905296759f,  0.424779681f,
    0.904644091f,  0.426167889f,
    0.903989293f,  0.427555093f,
    0.903332368f,  0.428941292f,
    0.902673318f,  0.430326481f,
    0.902012144f,  0.431710658f,
    0.901348847f,  0.433093819f,
    0.900683429f,  0.434475961f,
    0.900015892f,  0.435857080f,
    0.899346237f,  0.437237174f,
    0.898674466f,  0.438616239f,
    0.898000580f,  0.439994271f,
    0.897324581f,  0.441371269f,
    0.896646470f,  0.442747228f,
    0.895966250f,  0.444122145f,
    0.895283921f,  0.445496017f,
    0.894599486f,  0.446868840f,
    0.893912945f,  0.448240612f,
    0.893224301f,  0.449611330f,
    0.892533555f,  0.450980989f,
    0.891840709f,  0.452349587f,
    0.891145765f,  0.453717121f,
    0.890448723f,  0.455083587f,
    0.889749586f,  0.456448982f,
    0.889048356f,  0.457813304f,
    0.888345033f,  0.459176548f,
    0.887639620f,  0.460538711f,
    0.886932119f,  0.461899791f,
    0.886222530f,  0.463259784f,
    0.885510856f,  0.464618686f,
    0.884797098f,  0.465976496f,
    0.884081259f,  0.467333209f,
    0.883363339f,  0.468688822f,
    0.882643340f,  0.470043332f,
    0.881921264f,  0.471396737f,
    0.881197113f,  0.472749032f,
    0.880470889f,  0.474100215f,
    0.879742593f,  0.475450282f,
    0.879012226f,  0.476799230f,
    0.878279792f,  0.478147056f,
    0.877545290f,  0.479493758f,
    0.876808724f,  0.480839331f,
    0.876070094f,  0.482183772f,
    0.875329403f,  0.483527079f,
    0.874586652f,  0.484869248f,
    0.873841843f,  0.486210276f,
    0.873094978f,  0.487550160f,
    0.872346059f,  0.488888897f,
    0.871595087f,  0.490226483f,
    0.870842063f,  0.491562916f,
    0.870086991f,  0.492898192f,
    0.869329871f,  0.494232309f,
    0.868570706f,  0.495565262f,
    0.867809497f,  0.496897049f,
    0.867046246f,  0.498227667f,
    0.866280954f,  0.499557113f,
    0.865513624f,  0.500885383f,
    0.864744258f,  0.502212474f,
    0.863972856f,  0.503538384f,
    0.863199422f,  0.504863109f,
    0.862423956f,  0.506186645f,
    0.861646461f,  0.507508991f,
    0.860866939f,  0.508830143f,
    0.860085390f,  0.510150097f,
    0.859301818f,  0.511468850f,
    0.858516224f,  0.512786401f,
    0.857728610f,  0.514102744f,
    0.856938977f,  0.515417878f,
    0.856147328f,  0.516731799f,
    0.855353665f,  0.518044504f,
    0.854557988f,  0.519355990f,
    0.853760301f,  0.520666254f,
    0.852960605f,  0.521975293f,
    0.852158902f,  0.523283103f,
    0.851355193f,  0.524589683f,
    0.850549481f,  0.525895027f,
    0.849741768f,  0.527199135f,
    0.848932055f,  0.528502002f,
    0.848120345f,  0.529803625f,
    0.847306639f,  0.531104001f,
    0.846490939f,  0.532403128f,
    0.845673247f,  0.533701002f,
    0.844853565f,  0.534997620f,
    0.844031895f,  0.536292979f,
    0.843208240f,  0.537587076f,
    0.842382600f,  0.538879909f,
    0.841554977f,  0.540171473f,
    0.840725375f,  0.541461766f,
    0.839893794f,  0.542750785f,
    0.839060237f,  0.544038527f,
    0.838224706f,  0.545324988f,
    0.837387202f,  0.546610167f,
    0.836547727f,  0.547894059f,
    0.835706284f,  0.549176662f,
    0.834862875f,  0.550457973f,
    0.834017501f,  0.551737988f,
    0.833170165f,  0.553016706f,
    0.832320868f,  0.554294121f,
    0.831469612f,  0.555570233f,
    0.830616400f,  0.556845037f,
    0.829761234f,  0.558118531f,
    0.828904115f,  0.559390712f,
    0.828045045f,  0.560661576f,
    0.827184027f,  0.561931121f,
    0.826321063f,  0.563199344f,
    0.825456154f,  0.564466242f,
    0.824589303f,  0.565731811f,
    0.823720511f,  0.566996049f,
    0.822849781f,  0.568258953f,
    0.821977115f,  0.569520519f,
    0.821102515f,  0.570780746f,
    0.820225983f,  0.572039629f,
    0.819347520f,  0.573297167f,
    0.818467130f,  0.574553355f,
    0.817584813f,  0.575808191f,
    0.816700573f,  0.577061673f,
    0.815814411f,  0.578313796f,
    0.814926329f,  0.579564559f,
    0.814036330f,  0.580813958f,
    0.813144415f,  0.582061990f,
    0.812250587f,  0.583308653f,
    0.811354847f,  0.584553943f,
    0.810457198f,  0.585797857f,
    0.809557642f,  0.587040394f,
    0.808656182f,  0.588281548f,
    0.807752818f,  0.589521319f,
    0.806847554f,  0.590759702f,
    0.805940391f,  0.591996695f,
    0.805031331f,  0.593232295f,
    0.804120377f,  0.594466499f,
    0.803207531f,  0.595699304f,
    0.802292796f,  0.596930708f,
    0.801376172f,  0.598160707f,
    0.800457662f,  0.599389298f,
    0.799537269f,  0.600616479f,
    0.798614995f,  0.601842247f,
    0.797690841f,  0.603066599f,
    0.796764810f,  0.604289531f,
    0.795836905f,  0.605511041f,
    0.794907126f,  0.606731127f,
    0.793975478f,  0.607949785f,
    0.793041960f,  0.609167012f,
    0.792106577f,  0.610382806f,
    0.791169330f,  0.611597164f,
    0.790230221f,  0.612810082f,
    0.789289253f,  0.614021559f,
    0.788346428f,  0.615231591f,
    0.787401747f,  0.616440175f,
    0.786455214f,  0.617647308f,
    0.785506830f,  0.618852988f,
    0.784556597f,  0.620057212f,
    0.783604519f,  0.621259977f,
    0.782650596f,  0.622461279f,
    0.781694832f,  0.623661118f,
    0.780737229f,  0.624859488f,
    0.779777788f,  0.626056388f,
    0.778816512f,  0.627251815f,
    0.777853404f,  0.628445767f,
    0.776888466f,  0.629638239f,
    0.775921699f,  0.630829230f,
    0.774953107f,  0.632018736f,
    0.773982691f,  0.633206755f,
    0.773010453f,  0.634393284f,
    0.772036397f,  0.635578320f,
    0.771060524f,  0.636761861f,
    0.770082837f,  0.637943904f,
    0.769103338f,  0.639124445f,
    0.768122029f,  0.640303482f,
    0.767138912f,  0.641481013f,
    0.766153990f,  0.642657034f,
    0.765167266f,  0.643831543f,
    0.764178741f,  0.645004537f,
    0.763188417f,  0.646176013f,
    0.762196298f,  0.647345969f,
    0.761202385f,  0.648514401f,
    0.760206682f,  0.649681307f,
    0.759209189f,  0.650846685f,
    0.758209910f,  0.652010531f,
    0.757208847f,  0.653172843f,
    0.756206001f,  0.654333618f,
    0.755201377f,  0.655492853f,
    0.754194975f,  0.656650546f,
    0.753186799f,  0.657806693f,
    0.752176850f,  0.658961293f,
    0.751165132f,  0.660114342f,
    0.750151646f,  0.661265838f,
    0.749136395f,  0.662415778f,
    0.748119380f,  0.663564159f,
    0.747100606f,  0.664710978f,
    0.746080074f,  0.665856234f,
    0.745057785f,  0.666999922f,
    0.744033744f,  0.668142041f,
    0.743007952f,  0.669282588f,
    0.741980412f,  0.670421560f,
    0.740951125f,  0.671558955f,
    0.739920095f,  0.672694769f,
    0.738887324f,  0.673829000f,
    0.737852815f,  0.674961646f,
    0.736816569f,  0.676092704f,
    0.735778589f,  0.677222170f,
    0.734738878f,  0.678350043f,
    0.733697438f,  0.679476320f,
    0.732654272f,  0.680600998f,
    0.731609381f,  0.681724074f,
    0.730562769f,  0.682845546f,
    0.729514438f,  0.683965412f,
    0.728464390f,  0.685083668f,
    0.727412629f,  0.686200312f,
    0.726359155f,  0.687315341f,
    0.725303972f,  0.688428753f,
    0.724247083f,  0.689540545f,
    0.723188489f,  0.690650714f,
    0.722128194f,  0.691759258f,
    0.721066199f,  0.692866175f,
    0.720002508f,  0.693971461f,
    0.718937122f,  0.695075114f,
    0.717870045f,  0.696177131f,
    0.716801279f,  0.697277511f,
    0.715730825f,  0.698376249f,
    0.714658688f,  0.699473345f,
    0.713584869f,  0.700568794f,
    0.712509371f,  0.701662595f,
    0.711432196f,  0.702754744f,
    0.710353347f,  0.703845241f,
    0.709272826f,  0.704934080f,
    0.708190637f,  0.706021261f,
    0.707106781f,  0.707106781f,
    0.706021261f,  0.708190637f,
    0.704934080f,  0.709272826f,
    0.703845241f,  0.710353347f,
    0.702754744f,  0.711432196f,
    0.701662595f,  0.712509371f,
    0.700568794f,  0.713584869f,
    0.699473345f,  0.714658688f,
    0.698376249f,  0.715730825f,
    0.697277511f,  0.716801279f,
    0.696177131f,  0.717870045f,
    0.695075114f,  0.718937122f,
    0.693971461f,  0.720002508f,
    0.692866175f,  0.721066199f,
    0.691759258f,  0.722128194f,
    0.690650714f,  0.723188489f,
    0.689540545f,  0.724247083f,
    0.688428753f,  0.725303972f,
    0.687315341f,  0.726359155f,
    0.686200312f,  0.727412629f,
    0.685083668f,  0.728464390f,
    0.683965412f,  0.729514438f,
    0.682845546f,  0.730562769f,
    0.681724074f,  0.731609381f,
    0.680600998f,  0.732654272f,
    0.679476320f,  0.733697438f,
    0.678350043f,  0.734738878f,
    0.677222170f,  0.735778589f,
    0.676092704f,  0.736816569f,
    0.674961646f,  0.737852815f,
    0.673829000f,  0.738887324f,
    0.672694769f,  0.739920095f,
    0.671558955f,  0.740951125f,
    0.670421560f,  0.741980412f,
    0.669282588f,  0.743007952f,
    0.668142041f,  0.744033744f,
    0.666999922f,  0.745057785f,
    0.665856234f,  0.746080074f,
    0.664710978f,  0.747100606f,
    0.663564159f,  0.748119380f,
    0.662415778f,  0.749136395f,
    0.661265838f,  0.750151646f,
    0.660114342f,  0.751165132f,
    0.658961293f,  0.752176850f,
    0.657806693f,  0.753186799f,
    0.656650546f,  0.754194975f,
    0.655492853f,  0.755201377f,
    0.654333618f,  0.756206001f,
    0.653172843f,  0.757208847f,
    0.652010531f,  0.758209910f,
    0.650846685f,  0.759209189f,
    0.649681307f,  0.760206682f,
    0.648514401f,  0.761202385f,
    0.647345969f,  0.762196298f,
    0.646176013f,  0.763188417f,
    0.645004537f,  0.764178741f,
    0.643831543f,  0.765167266f,
    0.642657034f,  0.766153990f,
    0.641481013f,  0.767138912f,
    0.640303482f,  0.768122029f,
    0.639124445f,  0.769103338f,
    0.637943904f,  0.770082837f,
    0.636761861f,  0.771060524f,
    0.635578320f,  0.772036397f,
    0.634393284f,  0.773010453f,
    0.633206755f,  0.773982691f,
    0.632018736f,  0.774953107f,
    0.630829230f,  0.775921699f,
    0.629638239f,  0.776888466f,
    0.628445767f,  0.777853404f,
    0.627251815f,  0.778816512f,
    0.626056388f,  0.779777788f,
    0.624859488f,  0.780737229f,
    0.623661118f,  0.781694832f,
    0.622461279f,  0.782650596f,
    0.621259977f,  0.783604519f,
    0.620057212f,  0.784556597f,
    0.618852988f,  0.785506830f,
    0.617647308f,  0.786455214f,
    0.616440175f,  0.787401747f,
    0.615231591f,  0.788346428f,
    0.614021559f,  0.789289253f,
    0.612810082f,  0.790230221f,
    0.611597164f,  0.791169330f,
    0.610382806f,  0.792106577f,
    0.609167012f,  0.793041960f,
    0.607949785f,  0.793975478f,
    0.606731127f,  0.794907126f,
    0.605511041f,  0.795836905f,
    0.604289531f,  0.796764810f,
    0.603066599f,  0.797690841f,
    0.601842247f,  0.798614995f,
    0.600616479f,  0.799537269f,
    0.599389298f,  0.800457662f,
    0.598160707f,  0.801376172f,
    0.596930708f,  0.802292796f,
    0.595699304f,  0.803207531f,
    0.594466499f,  0.804120377f,
    0.593232295f,  0.805031331f,
    0.591996695f,  0.805940391f,
    0.590759702f,  0.806847554f,
    0.589521319f,  0.807752818f,
    0.588281548f,  0.808656182f,
    0.587040394f,  0.809557642f,
    0.585797857f,  0.810457198f,
    0.584553943f,  0.811354847f,
    0.583308653f,  0.812250587f,
    0.582061990f,  0.813144415f,
    0.580813958f,  0.814036330f,
    0.579564559f,  0.814926329f,
    0.578313796f,  0.815814411f,
    0.577061673f,  0.816700573f,
    0.575808191f,  0.817584813f,
    0.574553355f,  0.818467130f,
    0.573297167f,  0.819347520f,
    0.572039629f,  0.820225983f,
    0.570780746f,  0.821102515f,
    0.569520519f,  0.821977115f,
    0.568258953f,  0.822849781f,
    0.566996049f,  0.823720511f,
    0.565731811f,  0.824589303f,
    0.564466242f,  0.825456154f,
    0.563199344f,  0.826321063f,
    0.561931121f,  0.827184027f,
    0.560661576f,  0.828045045f,
    0.559390712f,  0.828904115f,
    0.558118531f,  0.829761234f,
    0.556845037f,  0.830616400f,
    0.555570233f,  0.831469612f,
    0.554294121f,  0.832320868f,
    0.553016706f,  0.833170165f,
    0.551737988f,  0.834017501f,
    0.550457973f,  0.834862875f,
    0.549176662f,  0.835706284f,
    0.547894059f,  0.836547727f,
    0.546610167f,  0.837387202f,
    0.545324988f,  0.838224706f,
    0.544038527f,  0.839060237f,
    0.542750785f,  0.839893794f,
    0.541461766f,  0.840725375f,
    0.540171473f,  0.841554977f,
    0.538879909f,  0.842382600f,
    0.537587076f,  0.843208240f,
    0.536292979f,  0.844031895f,
    0.534997620f,  0.844853565f,
    0.533701002f,  0.845673247f,
    0.532403128f,  0.846490939f,
    0.531104001f,  0.847306639f,
    0.529803625f,  0.848120345f,
    0.528502002f,  0.848932055f,
    0.527199135f,  0.849741768f,
    0.525895027f,  0.850549481f,
    0.524589683f,  0.851355193f,
    0.523283103f,  0.852158902f,
    0.521975293f,  0.852960605f,
    0.520666254f,  0.853760301f,
    0.519355990f,  0.854557988f,
    0.518044504f,  0.855353665f,
    0.516731799f,  0.856147328f,
    0.515417878f,  0.856938977f,
    0.514102744f,  0.857728610f,
    0.512786401f,  0.858516224f,
    0.511468850f,  0.859301818f,
    0.510150097f,  0.860085390f,
    0.508830143f,  0.860866939f,
    0.507508991f,  0.861646461f,
    0.506186645f,  0.862423956f,
    0.504863109f,  0.863199422f,
    0.503538384f,  0.863972856f,
    0.502212474f,  0.864744258f,
    0.500885383f,  0.865513624f,
    0.499557113f,  0.866280954f,
    0.498227667f,  0.867046246f,
    0.496897049f,  0.867809497f,
    0.495565262f,  0.868570706f,
    0.494232309f,  0.869329871f,
    0.492898192f,  0.870086991f,
    0.491562916f,  0.870842063f,
    0.490226483f,  0.871595087f,
    0.488888897f,  0.872346059f,
    0.487550160f,  0.873094978f,
    0.486210276f,  0.873841843f,
    0.484869248f,  0.874586652f,
    0.483527079f,  0.875329403f,
    0.482183772f,  0.876070094f,
    0.480839331f,  0.876808724f,
    0.479493758f,  0.877545290f,
    0.478147056f,  0.878279792f,
    0.476799230f,  0.879012226f,
    0.475450282f,  0.879742593f,
    0.474100215f,  0.880470889f,
    0.472749032f,  0.881197113f,
    0.471396737f,  0.881921264f,
    0.470043332f,  0.882643340f,
    0.468688822f,  0.883363339f,
    0.467333209f,  0.884081259f,
    0.465976496f,  0.884797098f,
    0.464618686f,  0.885510856f,
    0.463259784f,  0.886222530f,
    0.461899791f,  0.886932119f,
    0.460538711f,  0.887639620f,
    0.459176548f,  0.888345033f,
    0.457813304f,  0.889048356f,
    0.456448982f,  0.889749586f,
    0.455083587f,  0.890448723f,
    0.453717121f,  0.891145765f,
    0.452349587f,  0.891840709f,
    0.450980989f,  0.892533555f,
    0.449611330f,  0.893224301f,
    0.448240612f,  0.893912945f,
    0.446868840f,  0.894599486f,
    0.445496017f,  0.895283921f,
    0.444122145f,  0.895966250f,
    0.442747228f,  0.896646470f,
    0.441371269f,  0.897324581f,
    0.439994271f,  0.898000580f,
    0.438616239f,  0.898674466f,
    0.437237174f,  0.899346237f,
    0.435857080f,  0.900015892f,
    0.434475961f,  0.900683429f,
    0.433093819f,  0.901348847f,
    0.431710658f,  0.902012144f,
    0.430326481f,  0.902673318f,
    0.428941292f,  0.903332368f,
    0.427555093f,  0.903989293f,
    0.426167889f,  0.904644091f,
    0.424779681f,  0.905296759f,
    0.423390474f,  0.905947298f,
    0.422000271f,  0.906595705f,
    0.420609074f,  0.907241978f,
    0.419216888f,  0.907886116f,
    0.417823716f,  0.908528119f,
    0.416429560f,  0.909167983f,
    0.415034424f,  0.909805708f,
    0.413638312f,  0.910441292f,
    0.412241227f,  0.911074734f,
    0.410843171f,  0.911706032f,
    0.409444149f,  0.912335185f,
    0.408044163f,  0.912962190f,
    0.406643217f,  0.913587048f,
    0.405241314f,  0.914209756f,
    0.403838458f,  0.914830312f,
    0.402434651f,  0.915448716f,
    0.401029897f,  0.916064966f,
    0.399624200f,  0.916679060f,
    0.398217562f,  0.917290997f,
    0.396809987f,  0.917900776f,
    0.395401479f,  0.918508394f,
    0.393992040f,  0.919113852f,
    0.392581674f,  0.919717146f,
    0.391170384f,  0.920318277f,
    0.389758174f,  0.920917242f,
    0.388345047f,  0.921514039f,
    0.386931006f,  0.922108669f,
    0.385516054f,  0.922701128f,
    0.384100195f,  0.923291417f,
    0.382683432f,  0.923879533f,
    0.381265769f,  0.924465474f,
    0.379847209f,  0.925049241f,
    0.378427755f,  0.925630831f,
    0.377007410f,  0.926210242f,
    0.375586178f,  0.926787474f,
    0.374164063f,  0.927362526f,
    0.372741067f,  0.927935395f,
    0.371317194f,  0.928506080f,
    0.369892447f,  0.929074581f,
    0.368466830f,  0.929640896f,
    0.367040346f,  0.930205023f,
    0.365612998f,  0.930766961f,
    0.364184790f,  0.931326709f,
    0.362755724f,  0.931884266f,
    0.361325806f,  0.932439629f,
    0.359895037f,  0.932992799f,
    0.358463421f,  0.933543773f,
    0.357030961f,  0.934092550f,
    0.355597662f,  0.934639130f,
    0.354163525f,  0.935183510f,
    0.352728556f,  0.935725689f,
    0.351292756f,  0.936265667f,
    0.349856130f,  0.936803442f,
    0.348418680f,  0.937339012f,
    0.346980411f,  0.937872376f,
    0.345541325f,  0.938403534f,
    0.344101426f,  0.938932484f,
    0.342660717f,  0.939459224f,
    0.341219202f,  0.939983753f,
    0.339776884f,  0.940506071f,
    0.338333767f,  0.941026175f,
    0.336889853f,  0.941544065f,
    0.335445147f,  0.942059740f,
    0.333999651f,  0.942573198f,
    0.332553370f,  0.943084437f,
    0.331106306f,  0.943593458f,
    0.329658463f,  0.944100258f,
    0.328209844f,  0.944604837f,
    0.326760452f,  0.945107193f,
    0.325310292f,  0.945607325f,
    0.323859367f,  0.946105232f,
    0.322407679f,  0.946600913f,
    0.320955232f,  0.947094366f,
    0.319502031f,  0.947585591f,
    0.318048077f,  0.948074586f,
    0.316593376f,  0.948561350f,
    0.315137929f,  0.949045882f,
    0.313681740f,  0.949528181f,
    0.312224814f,  0.950008245f,
    0.310767153f,  0.950486074f,
    0.309308760f,  0.950961666f,
    0.307849640f,  0.951435021f,
    0.306389795f,  0.951906137f,
    0.304929230f,  0.952375013f,
    0.303467947f,  0.952841648f,
    0.302005949f,  0.953306040f,
    0.300543241f,  0.953768190f,
    0.299079826f,  0.954228095f,
    0.297615707f,  0.954685755f,
    0.296150888f,  0.955141168f,
    0.294685372f,  0.955594334f,
    0.293219163f,  0.956045251f,
    0.291752263f,  0.956493919f,
    0.290284677f,  0.956940336f,
    0.288816408f,  0.957384501f,
    0.287347460f,  0.957826413f,
    0.285877835f,  0.958266071f,
    0.284407537f,  0.958703475f,
    0.282936570f,  0.959138622f,
    0.281464938f,  0.959571513f,
    0.279992643f,  0.960002146f,
    0.278519689f,  0.960430519f,
    0.277046080f,  0.960856633f,
    0.275571819f,  0.961280486f,
    0.274096910f,  0.961702077f,
    0.272621355f,  0.962121404f,
    0.271145160f,  0.962538468f,
    0.269668326f,  0.962953267f,
    0.268190857f,  0.963365800f,
    0.266712757f,  0.963776066f,
    0.265234030f,  0.964184064f,
    0.263754679f,  0.964589793f,
    0.262274707f,  0.964993253f,
    0.260794118f,  0.965394442f,
    0.259312915f,  0.965793359f,
    0.257831102f,  0.966190003f,
    0.256348682f,  0.966584374f,
    0.254865660f,  0.966976471f,
    0.253382037f,  0.967366292f,
    0.251897818f,  0.967753837f,
    0.250413007f,  0.968139105f,
    0.248927606f,  0.968522094f,
    0.247441619f,  0.968902805f,
    0.245955050f,  0.969281235f,
    0.244467903f,  0.969657385f,
    0.242980180f,  0.970031253f,
    0.241491885f,  0.970402839f,
    0.240003022f,  0.970772141f,
    0.238513595f,  0.971139158f,
    0.237023606f,  0.971503891f,
    0.235533059f,  0.971866337f,
    0.234041959f,  0.972226497f,
    0.232550307f,  0.972584369f,
    0.231058108f,  0.972939952f,
    0.229565366f,  0.973293246f,
    0.228072083f,  0.973644250f,
    0.226578264f,  0.973992962f,
    0.225083911f,  0.974339383f,
    0.223589029f,  0.974683511f,
    0.222093621f,  0.975025345f,
    0.220597690f,  0.975364885f,
    0.219101240f,  0.975702130f,
    0.217604275f,  0.976037079f,
    0.216106797f,  0.976369731f,
    0.214608811f,  0.976700086f,
    0.213110320f,  0.977028143f,
    0.211611327f,  0.977353900f,
    0.210111837f,  0.977677358f,
    0.208611852f,  0.977998515f,
    0.207111376f,  0.978317371f,
    0.205610413f,  0.978633924f,
    0.204108966f,  0.978948175f,
    0.202607039f,  0.979260123f,
    0.201104635f,  0.979569766f,
    0.199601758f,  0.979877104f,
    0.198098411f,  0.980182136f,
    0.196594598f,  0.980484862f,
    0.195090322f,  0.980785280f,
    0.193585587f,  0.981083391f,
    0.192080397f,  0.981379193f,
    0.190574755f,  0.981672686f,
    0.189068664f,  0.981963869f,
    0.187562129f,  0.982252741f,
    0.186055152f,  0.982539302f,
    0.184547737f,  0.982823551f,
    0.183039888f,  0.983105487f,
    0.181531608f,  0.983385110f,
    0.180022901f,  0.983662419f,
    0.178513771f,  0.983937413f,
    0.177004220f,  0.984210092f,
    0.175494253f,  0.984480455f,
    0.173983873f,  0.984748502f,
    0.172473084f,  0.985014231f,
    0.170961889f,  0.985277642f,
    0.169450291f,  0.985538735f,
    0.167938295f,  0.985797509f,
    0.166425904f,  0.986053963f,
    0.164913120f,  0.986308097f,
    0.163399949f,  0.986559910f,
    0.161886394f,  0.986809402f,
    0.160372457f,  0.987056571f,
    0.158858143f,  0.987301418f,
    0.157343456f,  0.987543942f,
    0.155828398f,  0.987784142f,
    0.154312973f,  0.988022017f,
    0.152797185f,  0.988257568f,
    0.151281038f,  0.988490793f,
    0.149764535f,  0.988721692f,
    0.148247679f,  0.988950265f,
    0.146730474f,  0.989176510f,
    0.145212925f,  0.989400428f,
    0.143695033f,  0.989622017f,
    0.142176804f,  0.989841278f,
    0.140658239f,  0.990058210f,
    0.139139344f,  0.990272812f,
    0.137620122f,  0.990485084f,
    0.136100575f,  0.990695025f,
    0.134580709f,  0.990902635f,
    0.133060525f,  0.991107914f,
    0.131540029f,  0.991310860f,
    0.130019223f,  0.991511473f,
    0.128498111f,  0.991709754f,
    0.126976696f,  0.991905700f,
    0.125454983f,  0.992099313f,
    0.123932975f,  0.992290591f,
    0.122410675f,  0.992479535f,
    0.120888087f,  0.992666142f,
    0.119365215f,  0.992850414f,
    0.117842062f,  0.993032350f,
    0.116318631f,  0.993211949f,
    0.114794927f,  0.993389211f,
    0.113270952f,  0.993564136f,
    0.111746711f,  0.993736722f,
    0.110222207f,  0.993906970f,
    0.108697444f,  0.994074879f,
    0.107172425f,  0.994240449f,
    0.105647154f,  0.994403680f,
    0.104121634f,  0.994564571f,
    0.102595869f,  0.994723121f,
    0.101069863f,  0.994879331f,
    0.099543619f,  0.995033199f,
    0.098017140f,  0.995184727f,
    0.096490431f,  0.995333912f,
    0.094963495f,  0.995480755f,
    0.093436336f,  0.995625256f,
    0.091908956f,  0.995767414f,
    0.090381361f,  0.995907229f,
    0.088853553f,  0.996044701f,
    0.087325535f,  0.996179829f,
    0.085797312f,  0.996312612f,
    0.084268888f,  0.996443051f,
    0.082740265f,  0.996571146f,
    0.081211447f,  0.996696895f,
    0.079682438f,  0.996820299f,
    0.078153242f,  0.996941358f,
    0.076623861f,  0.997060070f,
    0.075094301f,  0.997176437f,
    0.073564564f,  0.997290457f,
    0.072034653f,  0.997402130f,
    0.070504573f,  0.997511456f,
    0.068974328f,  0.997618435f,
    0.067443920f,  0.997723067f,
    0.065913353f,  0.997825350f,
    0.064382631f,  0.997925286f,
    0.062851758f,  0.998022874f,
    0.061320736f,  0.998118113f,
    0.059789571f,  0.998211003f,
    0.058258265f,  0.998301545f,
    0.056726821f,  0.998389737f,
    0.055195244f,  0.998475581f,
    0.053663538f,  0.998559074f,
    0.052131705f,  0.998640218f,
    0.050599749f,  0.998719012f,
    0.049067674f,  0.998795456f,
    0.047535484f,  0.998869550f,
    0.046003182f,  0.998941293f,
    0.044470772f,  0.999010686f,
    0.042938257f,  0.999077728f,
    0.041405641f,  0.999142419f,
    0.039872928f,  0.999204759f,
    0.038340120f,  0.999264747f,
    0.036807223f,  0.999322385f,
    0.035274239f,  0.999377670f,
    0.033741172f,  0.999430605f,
    0.032208025f,  0.999481187f,
    0.030674803f,  0.999529418f,
    0.029141509f,  0.999575296f,
    0.027608146f,  0.999618822f,
    0.026074718f,  0.999659997f,
    0.024541229f,  0.999698819f,
    0.023007681f,  0.999735288f,
    0.021474080f,  0.999769405f,
    0.019940429f,  0.999801170f,
    0.018406730f,  0.999830582f,
    0.016872988f,  0.999857641f,
    0.015339206f,  0.999882347f,
    0.013805389f,  0.999904701f,
    0.012271538f,  0.999924702f,
    0.010737659f,  0.999942350f,
    0.009203755f,  0.999957645f,
    0.007669829f,  0.999970586f,
    0.006135885f,  0.999981175f,
    0.004601926f,  0.999989411f,
    0.003067957f,  0.999995294f,
    0.001533980f,  0.999998823f,
    0.000000000f,  1.000000000f,
   -0.001533980f,  0.999998823f,
   -0.003067957f,  0.999995294f,
   -0.004601926f,  0.999989411f,
   -0.006135885f,  0.999981175f,
   -0.007669829f,  0.999970586f,
   -0.009203755f,  0.999957645f,
   -0.010737659f,  0.999942350f,
   -0.012271538f,  0.999924702f,
   -0.013805389f,  0.999904701f,
   -0.015339206f,  0.999882347f,
   -0.016872988f,  0.999857641f,
   -0.018406730f,  0.999830582f,
   -0.019940429f,  0.999801170f,
   -0.021474080f,  0.999769405f,
   -0.023007681f,  0.999735288f,
   -0.024541229f,  0.999698819f,
   -0.026074718f,  0.999659997f,
   -0.027608146f,  0.999618822f,
   -0.029141509f,  0.999575296f,
   -0.030674803f,  0.999529418f,
   -0.032208025f,  0.999481187f,
   -0.033741172f,  0.999430605f,
   -0.035274239f,  0.999377670f,
   -0.036807223f,  0.999322385f,
   -0.038340120f,  0.999264747f,
   -0.039872928f,  0.999204759f,
   -0.041405641f,  0.999142419f,
   -0.042938257f,  0.999077728f,
   -0.044470772f,  0.999010686f,
   -0.046003182f,  0.998941293f,
   -0.047535484f,  0.998869550f,
   -0.049067674f,  0.998795456f,
   -0.050599749f,  0.998719012f,
   -0.052131705f,  0.998640218f,
   -0.053663538f,  0.998559074f,
   -0.055195244f,  0.998475581f,
   -0.056726821f,  0.998389737f,
   -0.058258265f,  0.998301545f,
   -0.059789571f,  0.998211003f,
   -0.061320736f,  0.998118113f,
   -0.062851758f,  0.998022874f,
   -0.064382631f,  0.997925286f,
   -0.065913353f,  0.997825350f,
   -0.067443920f,  0.997723067f,
   -0.068974328f,  0.997618435f,
   -0.070504573f,  0.997511456f,
   -0.072034653f,  0.997402130f,
   -0.073564564f,  0.997290457f,
   -0.075094301f,  0.997176437f,
   -0.076623861f,  0.997060070f,
   -0.078153242f,  0.996941358f,
   -0.079682438f,  0.996820299f,
   -0.081211447f,  0.996696895f,
   -0.082740265f,  0.996571146f,
   -0.084268888f,  0.996443051f,
   -0.085797312f,  0.996312612f,
   -0.087325535f,  0.996179829f,
   -0.088853553f,  0.996044701f,
   -0.090381361f,  0.995907229f,
   -0.091908956f,  0.995767414f,
   -0.093436336f,  0.995625256f,
   -0.094963495f,  0.995480755f,
   -0.096490431f,  0.995333912f,
   -0.098017140f,  0.995184727f,
   -0.099543619f,  0.995033199f,
   -0.101069863f,  0.994879331f,
   -0.102595869f,  0.994723121f,
   -0.104121634f,  0.994564571f,
   -0.105647154f,  0.994403680f,
   -0.107172425f,  0.994240449f,
   -0.108697444f,  0.994074879f,
   -0.110222207f,  0.993906970f,
   -0.111746711f,  0.993736722f,
   -0.113270952f,  0.993564136f,
   -0.114794927f,  0.993389211f,
   -0.116318631f,  0.993211949f,
   -0.117842062f,  0.993032350f,
   -0.119365215f,  0.992850414f,
   -0.120888087f,  0.992666142f,
   -0.122410675f,  0.992479535f,
   -0.123932975f,  0.992290591f,
   -0.125454983f,  0.992099313f,
   -0.126976696f,  0.991905700f,
   -0.128498111f,  0.991709754f,
   -0.130019223f,  0.991511473f,
   -0.131540029f,  0.991310860f,
   -0.133060525f,  0.991107914f,
   -0.134580709f,  0.990902635f,
   -0.136100575f,  0.990695025f,
   -0.137620122f,  0.990485084f,
   -0.139139344f,  0.990272812f,
   -0.140658239f,  0.990058210f,
   -0.142176804f,  0.989841278f,
   -0.143695033f,  0.989622017f,
   -0.145212925f,  0.989400428f,
   -0.146730474f,  0.989176510f,
   -0.148247679f,  0.988950265f,
   -0.149764535f,  0.988721692f,
   -0.151281038f,  0.988490793f,
   -0.152797185f,  0.988257568f,
   -0.154312973f,  0.988022017f,
   -0.155828398f,  0.987784142f,
   -0.157343456f,  0.987543942f,
   -0.158858143f,  0.987301418f,
   -0.160372457f,  0.987056571f,
   -0.161886394f,  0.986809402f,
   -0.163399949f,  0.986559910f,
   -0.164913120f,  0.986308097f,
   -0.166425904f,  0.986053963f,
   -0.167938295f,  0.985797509f,
   -0.169450291f,  0.985538735f,
   -0.170961889f,  0.985277642f,
   -0.172473084f,  0.985014231f,
   -0.173983873f,  0.984748502f,
   -0.175494253f,  0.984480455f,
   -0.177004220f,  0.984210092f,
   -0.178513771f,  0.983937413f,
   -0.180022901f,  0.983662419f,
   -0.181531608f,  0.983385110f,
   -0.183039888f,  0.983105487f,
   -0.184547737f,  0.982823551f,
   -0.186055152f,  0.982539302f,
   -0.187562129f,  0.982252741f,
   -0.189068664f,  0.981963869f,
   -0.190574755f,  0.981672686f,
   -0.192080397f,  0.981379193f,
   -0.193585587f,  0.981083391f,
   -0.195090322f,  0.980785280f,
   -0.196594598f,  0.980484862f,
   -0.198098411f,  0.980182136f,
   -0.199601758f,  0.979877104f,
   -0.201104635f,  0.979569766f,
   -0.202607039f,  0.979260123f,
   -0.204108966f,  0.978948175f,
   -0.205610413f,  0.978633924f,
   -0.207111376f,  0.978317371f,
   -0.208611852f,  0.977998515f,
   -0.210111837f,  0.977677358f,
   -0.211611327f,  0.977353900f,
   -0.213110320f,  0.977028143f,
   -0.214608811f,  0.976700086f,
   -0.216106797f,  0.976369731f,
   -0.217604275f,  0.976037079f,
   -0.219101240f,  0.975702130f,
   -0.220597690f,  0.975364885f,
   -0.222093621f,  0.975025345f,
   -0.223589029f,  0.974683511f,
   -0.225083911f,  0.974339383f,
   -0.226578264f,  0.973992962f,
   -0.228072083f,  0.973644250f,
   -0.229565366f,  0.973293246f,
   -0.231058108f,  0.972939952f,
   -0.232550307f,  0.972584369f,
   -0.234041959f,  0.972226497f,
   -0.235533059f,  0.971866337f,
   -0.237023606f,  0.971503891f,
   -0.238513595f,  0.971139158f,
   -0.240003022f,  0.970772141f,
   -0.241491885f,  0.970402839f,
   -0.242980180f,  0.970031253f,
   -0.244467903f,  0.969657385f,
   -0.245955050f,  0.969281235f,
   -0.247441619f,  0.968902805f,
   -0.248927606f,  0.968522094f,
   -0.250413007f,  0.968139105f,
   -0.251897818f,  0.967753837f,
   -0.253382037f,  0.967366292f,
   -0.254865660f,  0.966976471f,
   -0.256348682f,  0.966584374f,
   -0.257831102f,  0.966190003f,
   -0.259312915f,  0.965793359f,
   -0.260794118f,  0.965394442f,
   -0.262274707f,  0.964993253f,
   -0.263754679f,  0.964589793f,
   -0.265234030f,  0.964184064f,
   -0.266712757f,  0.963776066f,
   -0.268190857f,  0.963365800f,
   -0.269668326f,  0.962953267f,
   -0.271145160f,  0.962538468f,
   -0.272621355f,  0.962121404f,
   -0.274096910f,  0.961702077f,
   -0.275571819f,  0.961280486f,
   -0.277046080f,  0.960856633f,
   -0.278519689f,  0.960430519f,
   -0.279992643f,  0.960002146f,
   -0.281464938f,  0.959571513f,
   -0.282936570f,  0.959138622f,
   -0.284407537f,  0.958703475f,
   -0.285877835f,  0.958266071f,
   -0.287347460f,  0.957826413f,
   -0.288816408f,  0.957384501f,
   -0.290284677f,  0.956940336f,
   -0.291752263f,  0.956493919f,
   -0.293219163f,  0.956045251f,
   -0.294685372f,  0.955594334f,
   -0.296150888f,  0.955141168f,
   -0.297615707f,  0.954685755f,
   -0.299079826f,  0.954228095f,
   -0.300543241f,  0.953768190f,
   -0.302005949f,  0.953306040f,
   -0.303467947f,  0.952841648f,
   -0.304929230f,  0.952375013f,
   -0.306389795f,  0.951906137f,
   -0.307849640f,  0.951435021f,
   -0.309308760f,  0.950961666f,
   -0.310767153f,  0.950486074f,
   -0.312224814f,  0.950008245f,
   -0.313681740f,  0.949528181f,
   -0.315137929f,  0.949045882f,
   -0.316593376f,  0.948561350f,
   -0.318048077f,  0.948074586f,
   -0.319502031f,  0.947585591f,
   -0.320955232f,  0.947094366f,
   -0.322407679f,  0.946600913f,
   -0.323859367f,  0.946105232f,
   -0.325310292f,  0.945607325f,
   -0.326760452f,  0.945107193f,
   -0.328209844f,  0.944604837f,
   -0.329658463f,  0.944100258f,
   -0.331106306f,  0.943593458f,
   -0.332553370f,  0.943084437f,
   -0.333999651f,  0.942573198f,
   -0.335445147f,  0.942059740f,
   -0.336889853f,  0.941544065f,
   -0.338333767f,  0.941026175f,
   -0.339776884f,  0.940506071f,
   -0.341219202f,  0.939983753f,
   -0.342660717f,  0.939459224f,
   -0.344101426f,  0.938932484f,
   -0.345541325f,  0.938403534f,
   -0.346980411f,  0.937872376f,
   -0.348418680f,  0.937339012f,
   -0.349856130f,  0.936803442f,
   -0.351292756f,  0.936265667f,
   -0.352728556f,  0.935725689f,
   -0.354163525f,  0.935183510f,
   -0.355597662f,  0.934639130f,
   -0.357030961f,  0.934092550f,
   -0.358463421f,  0.933543773f,
   -0.359895037f,  0.932992799f,
   -0.361325806f,  0.932439629f,
   -0.362755724f,  0.931884266f,
   -0.364184790f,  0.931326709f,
   -0.365612998f,  0.930766961f,
   -0.367040346f,  0.930205023f,
   -0.368466830f,  0.929640896f,
   -0.369892447f,  0.929074581f,
   -0.371317194f,  0.928506080f,
   -0.372741067f,  0.927935395f,
   -0.374164063f,  0.927362526f,
   -0.375586178f,  0.926787474f,
   -0.377007410f,  0.926210242f,
   -0.378427755f,  0.925630831f,
   -0.379847209f,  0.925049241f,
   -0.381265769f,  0.924465474f,
   -0.382683432f,  0.923879533f,
   -0.384100195f,  0.923291417f,
   -0.385516054f,  0.922701128f,
   -0.386931006f,  0.922108669f,
   -0.388345047f,  0.921514039f,
   -0.389758174f,  0.920917242f,
   -0.391170384f,  0.920318277f,
   -0.392581674f,  0.919717146f,
   -0.393992040f,  0.919113852f,
   -0.395401479f,  0.918508394f,
   -0.396809987f,  0.917900776f,
   -0.398217562f,  0.917290997f,
   -0.399624200f,  0.916679060f,
   -0.401029897f,  0.916064966f,
   -0.402434651f,  0.915448716f,
   -0.403838458f,  0.914830312f,
   -0.405241314f,  0.914209756f,
   -0.406643217f,  0.913587048f,
   -0.408044163f,  0.912962190f,
   -0.409444149f,  0.912335185f,
   -0.410843171f,  0.911706032f,
   -0.412241227f,  0.911074734f,
   -0.413638312f,  0.910441292f,
   -0.415034424f,  0.909805708f,
   -0.416429560f,  0.909167983f,
   -0.417823716f,  0.908528119f,
   -0.419216888f,  0.907886116f,
   -0.420609074f,  0.907241978f,
   -0.422000271f,  0.906595705f,
   -0.423390474f,  0.905947298f,
   -0.424779681f,  0.905296759f,
   -0.426167889f,  0.904644091f,
   -0.427555093f,  0.903989293f,
   -0.428941292f,  0.903332368f,
   -0.430326481f,  0.902673318f,
   -0.431710658f,  0.902012144f,
   -0.433093819f,  0.901348847f,
   -0.434475961f,  0.900683429f,
   -0.435857080f,  0.900015892f,
   -0.437237174f,  0.899346237f,
   -0.438616239f,  0.898674466f,
   -0.439994271f,  0.898000580f,
   -0.441371269f,  0.897324581f,
   -0.442747228f,  0.896646470f,
   -0.444122145f,  0.895966250f,
   -0.445496017f,  0.895283921f,
   -0.446868840f,  0.894599486f,
   -0.448240612f,  0.893912945f,
   -0.449611330f,  0.893224301f,
   -0.450980989f,  0.892533555f,
   -0.452349587f,  0.891840709f,
   -0.453717121f,  0.891145765f,
   -0.455083587f,  0.890448723f,
   -0.456448982f,  0.889749586f,
   -0.457813304f,  0.889048356f,
   -0.459176548f,  0.888345033f,
   -0.460538711f,  0.887639620f,
   -0.461899791f,  0.886932119f,
   -0.463259784f,  0.886222530f,
   -0.464618686f,  0.885510856f,
   -0.465976496f,  0.884797098f,
   -0.467333209f,  0.884081259f,
   -0.468688822f,  0.883363339f,
   -0.470043332f,  0.882643340f,
   -0.471396737f,  0.881921264f,
   -0.472749032f,  0.881197113f,
   -0.474100215f,  0.880470889f,
   -0.475450282f,  0.879742593f,
   -0.476799230f,  0.879012226f,
   -0.478147056f,  0.878279792f,
   -0.479493758f,  0.877545290f,
   -0.480839331f,  0.876808724f,
   -0.482183772f,  0.876070094f,
   -0.483527079f,  0.875329403f,
   -0.484869248f,  0.874586652f,
   -0.486210276f,  0.873841843f,
   -0.487550160f,  0.873094978f,
   -0.488888897f,  0.872346059f,
   -0.490226483f,  0.871595087f,
   -0.491562916f,  0.870842063f,
   -0.492898192f,  0.870086991f,
   -0.494232309f,  0.869329871f,
   -0.495565262f,  0.868570706f,
   -0.496897049f,  0.867809497f,
   -0.498227667f,  0.867046246f,
   -0.499557113f,  0.866280954f,
   -0.500885383f,  0.865513624f,
   -0.502212474f,  0.864744258f,
   -0.503538384f,  0.863972856f,
   -0.504863109f,  0.863199422f,
   -0.506186645f,  0.862423956f,
   -0.507508991f,  0.861646461f,
   -0.508830143f,  0.860866939f,
   -0.510150097f,  0.860085390f,
   -0.511468850f,  0.859301818f,
   -0.512786401f,  0.858516224f,
   -0.514102744f,  0.857728610f,
   -0.515417878f,  0.856938977f,
   -0.516731799f,  0.856147328f,
   -0.518044504f,  0.855353665f,
   -0.519355990f,  0.854557988f,
   -0.520666254f,  0.853760301f,
   -0.521975293f,  0.852960605f,
   -0.523283103f,  0.852158902f,
   -0.524589683f,  0.851355193f,
   -0.525895027f,  0.850549481f,
   -0.527199135f,  0.849741768f,
   -0.528502002f,  0.848932055f,
   -0.529803625f,  0.848120345f,
   -0.531104001f,  0.847306639f,
   -0.532403128f,  0.846490939f,
   -0.533701002f,  0.845673247f,
   -0.534997620f,  0.844853565f,
   -0.536292979f,  0.844031895f,
   -0.537587076f,  0.843208240f,
   -0.538879909f,  0.842382600f,
   -0.540171473f,  0.841554977f,
   -0.541461766f,  0.840725375f,
   -0.542750785f,  0.839893794f,
   -0.544038527f,  0.839060237f,
   -0.545324988f,  0.838224706f,
   -0.546610167f,  0.837387202f,
   -0.547894059f,  0.836547727f,
   -0.549176662f,  0.835706284f,
   -0.550457973f,  0.834862875f,
   -0.551737988f,  0.834017501f,
   -0.553016706f,  0.833170165f,
   -0.554294121f,  0.832320868f,
   -0.555570233f,  0.831469612f,
   -0.556845037f,  0.830616400f,
   -0.558118531f,  0.829761234f,
   -0.559390712f,  0.828904115f,
   -0.560661576f,  0.828045045f,
   -0.561931121f,  0.827184027f,
   -0.563199344f,  0.826321063f,
   -0.564466242f,  0.825456154f,
   -0.565731811f,  0.824589303f,
   -0.566996049f,  0.823720511f,
   -0.568258953f,  0.822849781f,
   -0.569520519f,  0.821977115f,
   -0.570780746f,  0.821102515f,
   -0.572039629f,  0.820225983f,
   -0.573297167f,  0.819347520f,
   -0.574553355f,  0.818467130f,
   -0.575808191f,  0.817584813f,
   -0.577061673f,  0.816700573f,
   -0.578313796f,  0.815814411f,
   -0.579564559f,  0.814926329f,
   -0.580813958f,  0.814036330f,
   -0.582061990f,  0.813144415f,
   -0.583308653f,  0.812250587f,
   -0.584553943f,  0.811354847f,
   -0.585797857f,  0.810457198f,
   -0.587040394f,  0.809557642f,
   -0.588281548f,  0.808656182f,
   -0.589521319f,  0.807752818f,
   -0.590759702f,  0.806847554f,
   -0.591996695f,  0.805940391f,
   -0.593232295f,  0.805031331f,
   -0.594466499f,  0.804120377f,
   -0.595699304f,  0.803207531f,
   -0.596930708f,  0.802292796f,
   -0.598160707f,  0.801376172f,
   -0.599389298f,  0.800457662f,
   -0.600616479f,  0.799537269f,
   -0.601842247f,  0.798614995f,
   -0.603066599f,  0.797690841f,
   -0.604289531f,  0.796764810f,
   -0.605511041f,  0.795836905f,
   -0.606731127f,  0.794907126f,
   -0.607949785f,  0.793975478f,
   -0.609167012f,  0.793041960f,
   -0.610382806f,  0.792106577f,
   -0.611597164f,  0.791169330f,
   -0.612810082f,  0.790230221f,
   -0.614021559f,  0.789289253f,
   -0.615231591f,  0.788346428f,
   -0.616440175f,  0.787401747f,
   -0.617647308f,  0.786455214f,
   -0.618852988f,  0.785506830f,
   -0.620057212f,  0.784556597f,
   -0.621259977f,  0.783604519f,
   -0.622461279f,  0.782650596f,
   -0.623661118f,  0.781694832f,
   -0.624859488f,  0.780737229f,
   -0.626056388f,  0.779777788f,
   -0.627251815f,  0.778816512f,
   -0.628445767f,  0.777853404f,
   -0.629638239f,  0.776888466f,
   -0.630829230f,  0.775921699f,
   -0.632018736f,  0.774953107f,
   -0.633206755f,  0.773982691f,
   -0.634393284f,  0.773010453f,
   -0.635578320f,  0.772036397f,
   -0.636761861f,  0.771060524f,
   -0.637943904f,  0.770082837f,
   -0.639124445f,  0.769103338f,
   -0.640303482f,  0.768122029f,
   -0.641481013f,  0.767138912f,
   -0.642657034f,  0.766153990f,
   -0.643831543f,  0.765167266f,
   -0.645004537f,  0.764178741f,
   -0.646176013f,  0.763188417f,
   -0.647345969f,  0.762196298f,
   -0.648514401f,  0.761202385f,
   -0.649681307f,  0.760206682f,
   -0.650846685f,  0.759209189f,
   -0.652010531f,  0.758209910f,
   -0.653172843f,  0.757208847f,
   -0.654333618f,  0.756206001f,
   -0.655492853f,  0.755201377f,
   -0.656650546f,  0.754194975f,
   -0.657806693f,  0.753186799f,
   -0.658961293f,  0.752176850f,
   -0.660114342f,  0.751165132f,
   -0.661265838f,  0.750151646f,
   -0.662415778f,  0.749136395f,
   -0.663564159f,  0.748119380f,
   -0.664710978f,  0.747100606f,
   -0.665856234f,  0.746080074f,
   -0.666999922f,  0.745057785f,
   -0.668142041f,  0.744033744f,
   -0.669282588f,  0.743007952f,
   -0.670421560f,  0.741980412f,
   -0.671558955f,  0.740951125f,
   -0.672694769f,  0.739920095f,
   -0.673829000f,  0.738887324f,
   -0.674961646f,  0.737852815f,
   -0.676092704f,  0.736816569f,
   -0.677222170f,  0.735778589f,
   -0.678350043f,  0.734738878f,
   -0.679476320f,  0.733697438f,
   -0.680600998f,  0.732654272f,
   -0.681724074f,  0.731609381f,
   -0.682845546f,  0.730562769f,
   -0.683965412f,  0.729514438f,
   -0.685083668f,  0.728464390f,
   -0.686200312f,  0.727412629f,
   -0.687315341f,  0.726359155f,
   -0.688428753f,  0.725303972f,
   -0.689540545f,  0.724247083f,
   -0.690650714f,  0.723188489f,
   -0.691759258f,  0.722128194f,
   -0.692866175f,  0.721066199f,
   -0.693971461f,  0.720002508f,
   -0.695075114f,  0.718937122f,
   -0.696177131f,  0.717870045f,
   -0.697277511f,  0.716801279f,
   -0.698376249f,  0.715730825f,
   -0.699473345f,  0.714658688f,
   -0.700568794f,  0.713584869f,
   -0.701662595f,  0.712509371f,
   -0.702754744f,  0.711432196f,
   -0.703845241f,  0.710353347f,
   -0.704934080f,  0.709272826f,
   -0.706021261f,  0.708190637f,
   -0.707106781f,  0.707106781f,
   -0.708190637f,  0.706021261f,
   -0.709272826f,  0.704934080f,
   -0.710353347f,  0.703845241f,
   -0.711432196f,  0.702754744f,
   -0.712509371f,  0.701662595f,
   -0.713584869f,  0.700568794f,
   -0.714658688f,  0.699473345f,
   -0.715730825f,  0.698376249f,
   -0.716801279f,  0.697277511f,
   -0.717870045f,  0.696177131f,
   -0.718937122f,  0.695075114f,
   -0.720002508f,  0.693971461f,
   -0.721066199f,  0.692866175f,
   -0.722128194f,  0.691759258f,
   -0.723188489f,  0.690650714f,
   -0.724247083f,  0.689540545f,
   -0.725303972f,  0.688428753f,
   -0.726359155f,  0.687315341f,
   -0.727412629f,  0.686200312f,
   -0.728464390f,  0.685083668f,
   -0.729514438f,  0.683965412f,
   -0.730562769f,  0.682845546f,
   -0.731609381f,  0.681724074f,
   -0.732654272f,  0.680600998f,
   -0.733697438f,  0.679476320f,
   -0.734738878f,  0.678350043f,
   -0.735778589f,  0.677222170f,
   -0.736816569f,  0.676092704f,
   -0.737852815f,  0.674961646f,
   -0.738887324f,  0.673829000f,
   -0.739920095f,  0.672694769f,
   -0.740951125f,  0.671558955f,
   -0.741980412f,  0.670421560f,
   -0.743007952f,  0.669282588f,
   -0.744033744f,  0.668142041f,
   -0.745057785f,  0.666999922f,
   -0.746080074f,  0.665856234f,
   -0.747100606f,  0.664710978f,
   -0.748119380f,  0.663564159f,
   -0.749136395f,  0.662415778f,
   -0.750151646f,  0.661265838f,
   -0.751165132f,  0.660114342f,
   -0.752176850f,  0.658961293f,
   -0.753186799f,  0.657806693f,
   -0.754194975f,  0.656650546f,
   -0.755201377f,  0.655492853f,
   -0.756206001f,  0.654333618f,
   -0.757208847f,  0.653172843f,
   -0.758209910f,  0.652010531f,
   -0.759209189f,  0.650846685f,
   -0.760206682f,  0.649681307f,
   -0.761202385f,  0.648514401f,
   -0.762196298f,  0.647345969f,
   -0.763188417f,  0.646176013f,
   -0.764178741f,  0.645004537f,
   -0.765167266f,  0.643831543f,
   -0.766153990f,  0.642657034f,
   -0.767138912f,  0.641481013f,
   -0.768122029f,  0.640303482f,
   -0.769103338f,  0.639124445f,
   -0.770082837f,  0.637943904f,
   -0.771060524f,  0.636761861f,
   -0.772036397f,  0.635578320f,
   -0.773010453f,  0.634393284f,
   -0.773982691f,  0.633206755f,
   -0.774953107f,  0.632018736f,
   -0.775921699f,  0.630829230f,
   -0.776888466f,  0.629638239f,
   -0.777853404f,  0.628445767f,
   -0.778816512f,  0.627251815f,
   -0.779777788f,  0.626056388f,
   -0.780737229f,  0.624859488f,
   -0.781694832f,  0.623661118f,
   -0.782650596f,  0.622461279f,
   -0.783604519f,  0.621259977f,
   -0.784556597f,  0.620057212f,
   -0.785506830f,  0.618852988f,
   -0.786455214f,  0.617647308f,
   -0.787401747f,  0.616440175f,
   -0.788346428f,  0.615231591f,
   -0.789289253f,  0.614021559f,
   -0.790230221f,  0.612810082f,
   -0.791169330f,  0.611597164f,
   -0.792106577f,  0.610382806f,
   -0.793041960f,  0.609167012f,
   -0.793975478f,  0.607949785f,
   -0.794907126f,  0.606731127f,
   -0.795836905f,  0.605511041f,
   -0.796764810f,  0.604289531f,
   -0.797690841f,  0.603066599f,
   -0.798614995f,  0.601842247f,
   -0.799537269f,  0.600616479f,
   -0.800457662f,  0.599389298f,
   -0.801376172f,  0.598160707f,
   -0.802292796f,  0.596930708f,
   -0.803207531f,  0.595699304f,
   -0.804120377f,  0.594466499f,
   -0.805031331f,  0.593232295f,
   -0.805940391f,  0.591996695f,
   -0.806847554f,  0.590759702f,
   -0.807752818f,  0.589521319f,
   -0.808656182f,  0.588281548f,
   -0.809557642f,  0.587040394f,
   -0.810457198f,  0.585797857f,
   -0.811354847f,  0.584553943f,
   -0.812250587f,  0.583308653f,
   -0.813144415f,  0.582061990f,
   -0.814036330f,  0.580813958f,
   -0.814926329f,  0.579564559f,
   -0.815814411f,  0.578313796f,
   -0.816700573f,  0.577061673f,
   -0.817584813f,  0.575808191f,
   -0.818467130f,  0.574553355f,
   -0.819347520f,  0.573297167f,
   -0.820225983f,  0.572039629f,
   -0.821102515f,  0.570780746f,
   -0.821977115f,  0.569520519f,
   -0.822849781f,  0.568258953f,
   -0.823720511f,  0.566996049f,
   -0.824589303f,  0.565731811f,
   -0.825456154f,  0.564466242f,
   -0.826321063f,  0.563199344f,
   -0.827184027f,  0.561931121f,
   -0.828045045f,  0.560661576f,
   -0.828904115f,  0.559390712f,
   -0.829761234f,  0.558118531f,
   -0.830616400f,  0.556845037f,
   -0.831469612f,  0.555570233f,
   -0.832320868f,  0.554294121f,
   -0.833170165f,  0.553016706f,
   -0.834017501f,  0.551737988f,
   -0.834862875f,  0.550457973f,
   -0.835706284f,  0.549176662f,
   -0.836547727f,  0.547894059f,
   -0.837387202f,  0.546610167f,
   -0.838224706f,  0.545324988f,
   -0.839060237f,  0.544038527f,
   -0.839893794f,  0.542750785f,
   -0.840725375f,  0.541461766f,
   -0.841554977f,  0.540171473f,
   -0.842382600f,  0.538879909f,
   -0.843208240f,  0.537587076f,
   -0.844031895f,  0.536292979f,
   -0.844853565f,  0.534997620f,
   -0.845673247f,  0.533701002f,
   -0.846490939f,  0.532403128f,
   -0.847306639f,  0.531104001f,
   -0.848120345f,  0.529803625f,
   -0.848932055f,  0.528502002f,
   -0.849741768f,  0.527199135f,
   -0.850549481f,  0.525895027f,
   -0.851355193f,  0.524589683f,
   -0.852158902f,  0.523283103f,
   -0.852960605f,  0.521975293f,
   -0.853760301f,  0.520666254f,
   -0.854557988f,  0.519355990f,
   -0.855353665f,  0.518044504f,
   -0.856147328f,  0.516731799f,
   -0.856938977f,  0.515417878f,
   -0.857728610f,  0.514102744f,
   -0.858516224f,  0.512786401f,
   -0.859301818f,  0.511468850f,
   -0.860085390f,  0.510150097f,
   -0.860866939f,  0.508830143f,
   -0.861646461f,  0.507508991f,
   -0.862423956f,  0.506186645f,
   -0.863199422f,  0.504863109f,
   -0.863972856f,  0.503538384f,
   -0.864744258f,  0.502212474f,
   -0.865513624f,  0.500885383f,
   -0.866280954f,  0.499557113f,
   -0.867046246f,  0.498227667f,
   -0.867809497f,  0.496897049f,
   -0.868570706f,  0.495565262f,
   -0.869329871f,  0.494232309f,
   -0.870086991f,  0.492898192f,
   -0.870842063f,  0.491562916f,
   -0.871595087f,  0.490226483f,
   -0.872346059f,  0.488888897f,
   -0.873094978f,  0.487550160f,
   -0.873841843f,  0.486210276f,
   -0.874586652f,  0.484869248f,
   -0.875329403f,  0.483527079f,
   -0.876070094f,  0.482183772f,
   -0.876808724f,  0.480839331f,
   -0.877545290f,  0.479493758f,
   -0.878279792f,  0.478147056f,
   -0.879012226f,  0.476799230f,
   -0.879742593f,  0.475450282f,
   -0.880470889f,  0.474100215f,
   -0.881197113f,  0.472749032f,
   -0.881921264f,  0.471396737f,
   -0.882643340f,  0.470043332f,
   -0.883363339f,  0.468688822f,
   -0.884081259f,  0.467333209f,
   -0.884797098f,  0.465976496f,
   -0.885510856f,  0.464618686f,
   -0.886222530f,  0.463259784f,
   -0.886932119f,  0.461899791f,
   -0.887639620f,  0.460538711f,
   -0.888345033f,  0.459176548f,
   -0.889048356f,  0.457813304f,
   -0.889749586f,  0.456448982f,
   -0.890448723f,  0.455083587f,
   -0.891145765f,  0.453717121f,
   -0.891840709f,  0.452349587f,
   -0.892533555f,  0.450980989f,
   -0.893224301f,  0.449611330f,
   -0.893912945f,  0.448240612f,
   -0.894599486f,  0.446868840f,
   -0.895283921f,  0.445496017f,
   -0.895966250f,  0.444122145f,
   -0.896646470f,  0.442747228f,
   -0.897324581f,  0.441371269f,
   -0.898000580f,  0.439994271f,
   -0.898674466f,  0.438616239f,
   -0.899346237f,  0.437237174f,
   -0.900015892f,  0.435857080f,
   -0.900683429f,  0.434475961f,
   -0.901348847f,  0.433093819f,
   -0.902012144f,  0.431710658f,
   -0.902673318f,  0.430326481f,
   -0.903332368f,  0.428941292f,
   -0.903989293f,  0.427555093f,
   -0.904644091f,  0.426167889f,
   -0.905296759f,  0.424779681f,
   -0.905947298f,  0.423390474f,
   -0.906595705f,  0.422000271f,
   -0.907241978f,  0.420609074f,
   -0.907886116f,  0.419216888f,
   -0.908528119f,  0.417823716f,
   -0.909167983f,  0.416429560f,
   -0.909805708f,  0.415034424f,
   -0.910441292f,  0.413638312f,
   -0.911074734f,  0.412241227f,
   -0.911706032f,  0.410843171f,
   -0.912335185f,  0.409444149f,
   -0.912962190f,  0.408044163f,
   -0.913587048f,  0.406643217f,
   -0.914209756f,  0.405241314f,
   -0.914830312f,  0.403838458f,
   -0.915448716f,  0.402434651f,
   -0.916064966f,  0.401029897f,
   -0.916679060f,  0.399624200f,
   -0.917290997f,  0.398217562f,
   -0.917900776f,  0.396809987f,
   -0.918508394f,  0.395401479f,
   -0.919113852f,  0.393992040f,
   -0.919717146f,  0.392581674f,
   -0.920318277f,  0.391170384f,
   -0.920917242f,  0.389758174f,
   -0.921514039f,  0.388345047f,
   -0.922108669f,  0.386931006f,
   -0.922701128f,  0.385516054f,
   -0.923291417f,  0.384100195f,
   -0.923879533f,  0.382683432f,
   -0.924465474f,  0.381265769f,
   -0.925049241f,  0.379847209f,
   -0.925630831f,  0.378427755f,
   -0.926210242f,  0.377007410f,
   -0.926787474f,  0.375586178f,
   -0.927362526f,  0.374164063f,
   -0.927935395f,  0.372741067f,
   -0.928506080f,  0.371317194f,
   -0.929074581f,  0.369892447f,
   -0.929640896f,  0.368466830f,
   -0.930205023f,  0.367040346f,
   -0.930766961f,  0.365612998f,
   -0.931326709f,  0.364184790f,
   -0.931884266f,  0.362755724f,
   -0.932439629f,  0.361325806f,
   -0.932992799f,  0.359895037f,
   -0.933543773f,  0.358463421f,
   -0.934092550f,  0.357030961f,
   -0.934639130f,  0.355597662f,
   -0.935183510f,  0.354163525f,
   -0.935725689f,  0.352728556f,
   -0.936265667f,  0.351292756f,
   -0.936803442f,  0.349856130f,
   -0.937339012f,  0.348418680f,
   -0.937872376f,  0.346980411f,
   -0.938403534f,  0.345541325f,
   -0.938932484f,  0.344101426f,
   -0.939459224f,  0.342660717f,
   -0.939983753f,  0.341219202f,
   -0.940506071f,  0.339776884f,
   -0.941026175f,  0.338333767f,
   -0.941544065f,  0.336889853f,
   -0.942059740f,  0.335445147f,
   -0.942573198f,  0.333999651f,
   -0.943084437f,  0.332553370f,
   -0.943593458f,  0.331106306f,
   -0.944100258f,  0.329658463f,
   -0.944604837f,  0.328209844f,
   -0.945107193f,  0.326760452f,
   -0.945607325f,  0.325310292f,
   -0.946105232f,  0.323859367f,
   -0.946600913f,  0.322407679f,
   -0.947094366f,  0.320955232f,
   -0.947585591f,  0.319502031f,
   -0.948074586f,  0.318048077f,
   -0.948561350f,  0.316593376f,
   -0.949045882f,  0.315137929f,
   -0.949528181f,  0.313681740f,
   -0.950008245f,  0.312224814f,
   -0.950486074f,  0.310767153f,
   -0.950961666f,  0.309308760f,
   -0.951435021f,  0.307849640f,
   -0.951906137f,  0.306389795f,
   -0.952375013f,  0.304929230f,
   -0.952841648f,  0.303467947f,
   -0.953306040f,  0.302005949f,
   -0.953768190f,  0.300543241f,
   -0.954228095f,  0.299079826f,
   -0.954685755f,  0.297615707f,
   -0.955141168f,  0.296150888f,
   -0.955594334f,  0.294685372f,
   -0.956045251f,  0.293219163f,
   -0.956493919f,  0.291752263f,
   -0.956940336f,  0.290284677f,
   -0.957384501f,  0.288816408f,
   -0.957826413f,  0.287347460f,
   -0.958266071f,  0.285877835f,
   -0.958703475f,  0.284407537f,
   -0.959138622f,  0.282936570f,
   -0.959571513f,  0.281464938f,
   -0.960002146f,  0.279992643f,
   -0.960430519f,  0.278519689f,
   -0.960856633f,  0.277046080f,
   -0.961280486f,  0.275571819f,
   -0.961702077f,  0.274096910f,
   -0.962121404f,  0.272621355f,
   -0.962538468f,  0.271145160f,
   -0.962953267f,  0.269668326f,
   -0.963365800f,  0.268190857f,
   -0.963776066f,  0.266712757f,
   -0.964184064f,  0.265234030f,
   -0.964589793f,  0.263754679f,
   -0.964993253f,  0.262274707f,
   -0.965394442f,  0.260794118f,
   -0.965793359f,  0.259312915f,
   -0.966190003f,  0.257831102f,
   -0.966584374f,  0.256348682f,
   -0.966976471f,  0.254865660f,
   -0.967366292f,  0.253382037f,
   -0.967753837f,  0.251897818f,
   -0.968139105f,  0.250413007f,
   -0.968522094f,  0.248927606f,
   -0.968902805f,  0.247441619f,
   -0.969281235f,  0.245955050f,
   -0.969657385f,  0.244467903f,
   -0.970031253f,  0.242980180f,
   -0.970402839f,  0.241491885f,
   -0.970772141f,  0.240003022f,
   -0.971139158f,  0.238513595f,
   -0.971503891f,  0.237023606f,
   -0.971866337f,  0.235533059f,
   -0.972226497f,  0.234041959f,
   -0.972584369f,  0.232550307f,
   -0.972939952f,  0.231058108f,
   -0.973293246f,  0.229565366f,
   -0.973644250f,  0.228072083f,
   -0.973992962f,  0.226578264f,
   -0.974339383f,  0.225083911f,
   -0.974683511f,  0.223589029f,
   -0.975025345f,  0.222093621f,
   -0.975364885f,  0.220597690f,
   -0.975702130f,  0.219101240f,
   -0.976037079f,  0.217604275f,
   -0.976369731f,  0.216106797f,
   -0.976700086f,  0.214608811f,
   -0.977028143f,  0.213110320f,
   -0.977353900f,  0.211611327f,
   -0.977677358f,  0.210111837f,
   -0.977998515f,  0.208611852f,
   -0.978317371f,  0.207111376f,
   -0.978633924f,  0.205610413f,
   -0.978948175f,  0.204108966f,
   -0.979260123f,  0.202607039f,
   -0.979569766f,  0.201104635f,
   -0.979877104f,  0.199601758f,
   -0.980182136f,  0.198098411f,
   -0.980484862f,  0.196594598f,
   -0.980785280f,  0.195090322f,
   -0.981083391f,  0.193585587f,
   -0.981379193f,  0.192080397f,
   -0.981672686f,  0.190574755f,
   -0.981963869f,  0.189068664f,
   -0.982252741f,  0.187562129f,
   -0.982539302f,  0.186055152f,
   -0.982823551f,  0.184547737f,
   -0.983105487f,  0.183039888f,
   -0.983385110f,  0.181531608f,
   -0.983662419f,  0.180022901f,
   -0.983937413f,  0.178513771f,
   -0.984210092f,  0.177004220f,
   -0.984480455f,  0.175494253f,
   -0.984748502f,  0.173983873f,
   -0.985014231f,  0.172473084f,
   -0.985277642f,  0.170961889f,
   -0.985538735f,  0.169450291f,
   -0.985797509f,  0.167938295f,
   -0.986053963f,  0.166425904f,
   -0.986308097f,  0.164913120f,
   -0.986559910f,  0.163399949f,
   -0.986809402f,  0.161886394f,
   -0.987056571f,  0.160372457f,
   -0.987301418f,  0.158858143f,
   -0.987543942f,  0.157343456f,
   -0.987784142f,  0.155828398f,
   -0.988022017f,  0.154312973f,
   -0.988257568f,  0.152797185f,
   -0.988490793f,  0.151281038f,
   -0.988721692f,  0.149764535f,
   -0.988950265f,  0.148247679f,
   -0.989176510f,  0.146730474f,
   -0.989400428f,  0.145212925f,
   -0.989622017f,  0.143695033f,
   -0.989841278f,  0.142176804f,
   -0.990058210f,  0.140658239f,
   -0.990272812f,  0.139139344f,
   -0.990485084f,  0.137620122f,
   -0.990695025f,  0.136100575f,
   -0.990902635f,  0.134580709f,
   -0.991107914f,  0.133060525f,
   -0.991310860f,  0.131540029f,
   -0.991511473f,  0.130019223f,
   -0.991709754f,  0.128498111f,
   -0.991905700f,  0.126976696f,
   -0.992099313f,  0.125454983f,
   -0.992290591f,  0.123932975f,
   -0.992479535f,  0.122410675f,
   -0.992666142f,  0.120888087f,
   -0.992850414f,  0.119365215f,
   -0.993032350f,  0.117842062f,
   -0.993211949f,  0.116318631f,
   -0.993389211f,  0.114794927f,
   -0.993564136f,  0.113270952f,
   -0.993736722f,  0.111746711f,
   -0.993906970f,  0.110222207f,
   -0.994074879f,  0.108697444f,
   -0.994240449f,  0.107172425f,
   -0.994403680f,  0.105647154f,
   -0.994564571f,  0.104121634f,
   -0.994723121f,  0.102595869f,
   -0.994879331f,  0.101069863f,
   -0.995033199f,  0.099543619f,
   -0.995184727f,  0.098017140f,
   -0.995333912f,  0.096490431f,
   -0.995480755f,  0.094963495f,
   -0.995625256f,  0.093436336f,
   -0.995767414f,  0.091908956f,
   -0.995907229f,  0.090381361f,
   -0.996044701f,  0.088853553f,
   -0.996179829f,  0.087325535f,
   -0.996312612f,  0.085797312f,
   -0.996443051f,  0.084268888f,
   -0.996571146f,  0.082740265f,
   -0.996696895f,  0.081211447f,
   -0.996820299f,  0.079682438f,
   -0.996941358f,  0.078153242f,
   -0.997060070f,  0.076623861f,
   -0.997176437f,  0.075094301f,
   -0.997290457f,  0.073564564f,
   -0.997402130f,  0.072034653f,
   -0.997511456f,  0.070504573f,
   -0.997618435f,  0.068974328f,
   -0.997723067f,  0.067443920f,
   -0.997825350f,  0.065913353f,
   -0.997925286f,  0.064382631f,
   -0.998022874f,  0.062851758f,
   -0.998118113f,  0.061320736f,
   -0.998211003f,  0.059789571f,
   -0.998301545f,  0.058258265f,
   -0.998389737f,  0.056726821f,
   -0.998475581f,  0.055195244f,
   -0.998559074f,  0.053663538f,
   -0.998640218f,  0.052131705f,
   -0.998719012f,  0.050599749f,
   -0.998795456f,  0.049067674f,
   -0.998869550f,  0.047535484f,
   -0.998941293f,  0.046003182f,
   -0.999010686f,  0.044470772f,
   -0.999077728f,  0.042938257f,
   -0.999142419f,  0.041405641f,
   -0.999204759f,  0.039872928f,
   -0.999264747f,  0.038340120f,
   -0.999322385f,  0.036807223f,
   -0.999377670f,  0.035274239f,
   -0.999430605f,  0.033741172f,
   -0.999481187f,  0.032208025f,
   -0.999529418f,  0.030674803f,
   -0.999575296f,  0.029141509f,
   -0.999618822f,  0.027608146f,
   -0.999659997f,  0.026074718f,
   -0.999698819f,  0.024541229f,
   -0.999735288f,  0.023007681f,
   -0.999769405f,  0.021474080f,
   -0.999801170f,  0.019940429f,
   -0.999830582f,  0.018406730f,
   -0.999857641f,  0.016872988f,
   -0.999882347f,  0.015339206f,
   -0.999904701f,  0.013805389f,
   -0.999924702f,  0.012271538f,
   -0.999942350f,  0.010737659f,
   -0.999957645f,  0.009203755f,
   -0.999970586f,  0.007669829f,
   -0.999981175f,  0.006135885f,
   -0.999989411f,  0.004601926f,
   -0.999995294f,  0.003067957f,
   -0.999998823f,  0.001533980f,
   -1.000000000f,  0.000000000f,
   -0.999998823f, -0.001533980f,
   -0.999995294f, -0.003067957f,
   -0.999989411f, -0.004601926f,
   -0.999981175f, -0.006135885f,
   -0.999970586f, -0.007669829f,
   -0.999957645f, -0.009203755f,
   -0.999942350f, -0.010737659f,
   -0.999924702f, -0.012271538f,
   -0.999904701f, -0.013805389f,
   -0.999882347f, -0.015339206f,
   -0.999857641f, -0.016872988f,
   -0.999830582f, -0.018406730f,
   -0.999801170f, -0.019940429f,
   -0.999769405f, -0.021474080f,
   -0.999735288f, -0.023007681f,
   -0.999698819f, -0.024541229f,
   -0.999659997f, -0.026074718f,
   -0.999618822f, -0.027608146f,
   -0.999575296f, -0.029141509f,
   -0.999529418f, -0.030674803f,
   -0.999481187f, -0.032208025f,
   -0.999430605f, -0.033741172f,
   -0.999377670f, -0.035274239f,
   -0.999322385f, -0.036807223f,
   -0.999264747f, -0.038340120f,
   -0.999204759f, -0.039872928f,
   -0.999142419f, -0.041405641f,
   -0.999077728f, -0.042938257f,
   -0.999010686f, -0.044470772f,
   -0.998941293f, -0.046003182f,
   -0.998869550f, -0.047535484f,
   -0.998795456f, -0.049067674f,
   -0.998719012f, -0.050599749f,
   -0.998640218f, -0.052131705f,
   -0.998559074f, -0.053663538f,
   -0.998475581f, -0.055195244f,
   -0.998389737f, -0.056726821f,
   -0.998301545f, -0.058258265f,
   -0.998211003f, -0.059789571f,
   -0.998118113f, -0.061320736f,
   -0.998022874f, -0.062851758f,
   -0.997925286f, -0.064382631f,
   -0.997825350f, -0.065913353f,
   -0.997723067f, -0.067443920f,
   -0.997618435f, -0.068974328f,
   -0.997511456f, -0.070504573f,
   -0.997402130f, -0.072034653f,
   -0.997290457f, -0.073564564f,
   -0.997176437f, -0.075094301f,
   -0.997060070f, -0.076623861f,
   -0.996941358f, -0.078153242f,
   -0.996820299f, -0.079682438f,
   -0.996696895f, -0.081211447f,
   -0.996571146f, -0.082740265f,
   -0.996443051f, -0.084268888f,
   -0.996312612f, -0.085797312f,
   -0.996179829f, -0.087325535f,
   -0.996044701f, -0.088853553f,
   -0.995907229f, -0.090381361f,
   -0.995767414f, -0.091908956f,
   -0.995625256f, -0.093436336f,
   -0.995480755f, -0.094963495f,
   -0.995333912f, -0.096490431f,
   -0.995184727f, -0.098017140f,
   -0.995033199f, -0.099543619f,
   -0.994879331f, -0.101069863f,
   -0.994723121f, -0.102595869f,
   -0.994564571f, -0.104121634f,
   -0.994403680f, -0.105647154f,
   -0.994240449f, -0.107172425f,
   -0.994074879f, -0.108697444f,
   -0.993906970f, -0.110222207f,
   -0.993736722f, -0.111746711f,
   -0.993564136f, -0.113270952f,
   -0.993389211f, -0.114794927f,
   -0.993211949f, -0.116318631f,
   -0.993032350f, -0.117842062f,
   -0.992850414f, -0.119365215f,
   -0.992666142f, -0.120888087f,
   -0.992479535f, -0.122410675f,
   -0.992290591f, -0.123932975f,
   -0.992099313f, -0.125454983f,
   -0.991905700f, -0.126976696f,
   -0.991709754f, -0.128498111f,
   -0.991511473f, -0.130019223f,
   -0.991310860f, -0.131540029f,
   -0.991107914f, -0.133060525f,
   -0.990902635f, -0.134580709f,
   -0.990695025f, -0.136100575f,
   -0.990485084f, -0.137620122f,
   -0.990272812f, -0.139139344f,
   -0.990058210f, -0.140658239f,
   -0.989841278f, -0.142176804f,
   -0.989622017f, -0.143695033f,
   -0.989400428f, -0.145212925f,
   -0.989176510f, -0.146730474f,
   -0.988950265f, -0.148247679f,
   -0.988721692f, -0.149764535f,
   -0.988490793f, -0.151281038f,
   -0.988257568f, -0.152797185f,
   -0.988022017f, -0.154312973f,
   -0.987784142f, -0.155828398f,
   -0.987543942f, -0.157343456f,
   -0.987301418f, -0.158858143f,
   -0.987056571f, -0.160372457f,
   -0.986809402f, -0.161886394f,
   -0.986559910f, -0.163399949f,
   -0.986308097f, -0.164913120f,
   -0.986053963f, -0.166425904f,
   -0.985797509f, -0.167938295f,
   -0.985538735f, -0.169450291f,
   -0.985277642f, -0.170961889f,
   -0.985014231f, -0.172473084f,
   -0.984748502f, -0.173983873f,
   -0.984480455f, -0.175494253f,
   -0.984210092f, -0.177004220f,
   -0.983937413f, -0.178513771f,
   -0.983662419f, -0.180022901f,
   -0.983385110f, -0.181531608f,
   -0.983105487f, -0.183039888f,
   -0.982823551f, -0.184547737f,
   -0.982539302f, -0.186055152f,
   -0.982252741f, -0.187562129f,
   -0.981963869f, -0.189068664f,
   -0.981672686f, -0.190574755f,
   -0.981379193f, -0.192080397f,
   -0.981083391f, -0.193585587f,
   -0.980785280f, -0.195090322f,
   -0.980484862f, -0.196594598f,
   -0.980182136f, -0.198098411f,
   -0.979877104f, -0.199601758f,
   -0.979569766f, -0.201104635f,
   -0.979260123f, -0.202607039f,
   -0.978948175f, -0.204108966f,
   -0.978633924f, -0.205610413f,
   -0.978317371f, -0.207111376f,
   -0.977998515f, -0.208611852f,
   -0.977677358f, -0.210111837f,
   -0.977353900f, -0.211611327f,
   -0.977028143f, -0.213110320f,
   -0.976700086f, -0.214608811f,
   -0.976369731f, -0.216106797f,
   -0.976037079f, -0.217604275f,
   -0.975702130f, -0.219101240f,
   -0.975364885f, -0.220597690f,
   -0.975025345f, -0.222093621f,
   -0.974683511f, -0.223589029f,
   -0.974339383f, -0.225083911f,
   -0.973992962f, -0.226578264f,
   -0.973644250f, -0.228072083f,
   -0.973293246f, -0.229565366f,
   -0.972939952f, -0.231058108f,
   -0.972584369f, -0.232550307f,
   -0.972226497f, -0.234041959f,
   -0.971866337f, -0.235533059f,
   -0.971503891f, -0.237023606f,
   -0.971139158f, -0.238513595f,
   -0.970772141f, -0.240003022f,
   -0.970402839f, -0.241491885f,
   -0.970031253f, -0.242980180f,
   -0.969657385f, -0.244467903f,
   -0.969281235f, -0.245955050f,
   -0.968902805f, -0.247441619f,
   -0.968522094f, -0.248927606f,
   -0.968139105f, -0.250413007f,
   -0.967753837f, -0.251897818f,
   -0.967366292f, -0.253382037f,
   -0.966976471f, -0.254865660f,
   -0.966584374f, -0.256348682f,
   -0.966190003f, -0.257831102f,
   -0.965793359f, -0.259312915f,
   -0.965394442f, -0.260794118f,
   -0.964993253f, -0.262274707f,
   -0.964589793f, -0.263754679f,
   -0.964184064f, -0.265234030f,
   -0.963776066f, -0.266712757f,
   -0.963365800f, -0.268190857f,
   -0.962953267f, -0.269668326f,
   -0.962538468f, -0.271145160f,
   -0.962121404f, -0.272621355f,
   -0.961702077f, -0.274096910f,
   -0.961280486f, -0.275571819f,
   -0.960856633f, -0.277046080f,
   -0.960430519f, -0.278519689f,
   -0.960002146f, -0.279992643f,
   -0.959571513f, -0.281464938f,
   -0.959138622f, -0.282936570f,
   -0.958703475f, -0.284407537f,
   -0.958266071f, -0.285877835f,
   -0.957826413f, -0.287347460f,
   -0.957384501f, -0.288816408f,
   -0.956940336f, -0.290284677f,
   -0.956493919f, -0.291752263f,
   -0.956045251f, -0.293219163f,
   -0.955594334f, -0.294685372f,
   -0.955141168f, -0.296150888f,
   -0.954685755f, -0.297615707f,
   -0.954228095f, -0.299079826f,
   -0.953768190f, -0.300543241f,
   -0.953306040f, -0.302005949f,
   -0.952841648f, -0.303467947f,
   -0.952375013f, -0.304929230f,
   -0.951906137f, -0.306389795f,
   -0.951435021f, -0.307849640f,
   -0.950961666f, -0.309308760f,
   -0.950486074f, -0.310767153f,
   -0.950008245f, -0.312224814f,
   -0.949528181f, -0.313681740f,
   -0.949045882f, -0.315137929f,
   -0.948561350f, -0.316593376f,
   -0.948074586f, -0.318048077f,
   -0.947585591f, -0.319502031f,
   -0.947094366f, -0.320955232f,
   -0.946600913f, -0.322407679f,
   -0.946105232f, -0.323859367f,
   -0.945607325f, -0.325310292f,
   -0.945107193f, -0.326760452f,
   -0.944604837f, -0.328209844f,
   -0.944100258f, -0.329658463f,
   -0.943593458f, -0.331106306f,
   -0.943084437f, -0.332553370f,
   -0.942573198f, -0.333999651f,
   -0.942059740f, -0.335445147f,
   -0.941544065f, -0.336889853f,
   -0.941026175f, -0.338333767f,
   -0.940506071f, -0.339776884f,
   -0.939983753f, -0.341219202f,
   -0.939459224f, -0.342660717f,
   -0.938932484f, -0.344101426f,
   -0.938403534f, -0.345541325f,
   -0.937872376f, -0.346980411f,
   -0.937339012f, -0.348418680f,
   -0.936803442f, -0.349856130f,
   -0.936265667f, -0.351292756f,
   -0.935725689f, -0.352728556f,
   -0.935183510f, -0.354163525f,
   -0.934639130f, -0.355597662f,
   -0.934092550f, -0.357030961f,
   -0.933543773f, -0.358463421f,
   -0.932992799f, -0.359895037f,
   -0.932439629f, -0.361325806f,
   -0.931884266f, -0.362755724f,
   -0.931326709f, -0.364184790f,
   -0.930766961f, -0.365612998f,
   -0.930205023f, -0.367040346f,
   -0.929640896f, -0.368466830f,
   -0.929074581f, -0.369892447f,
   -0.928506080f, -0.371317194f,
   -0.927935395f, -0.372741067f,
   -0.927362526f, -0.374164063f,
   -0.926787474f, -0.375586178f,
   -0.926210242f, -0.377007410f,
   -0.925630831f, -0.378427755f,
   -0.925049241f, -0.379847209f,
   -0.924465474f, -0.381265769f,
   -0.923879533f, -0.382683432f,
   -0.923291417f, -0.384100195f,
   -0.922701128f, -0.385516054f,
   -0.922108669f, -0.386931006f,
   -0.921514039f, -0.388345047f,
   -0.920917242f, -0.389758174f,
   -0.920318277f, -0.391170384f,
   -0.919717146f, -0.392581674f,
   -0.919113852f, -0.393992040f,
   -0.918508394f, -0.395401479f,
   -0.917900776f, -0.396809987f,
   -0.917290997f, -0.398217562f,
   -0.916679060f, -0.399624200f,
   -0.916064966f, -0.401029897f,
   -0.915448716f, -0.402434651f,
   -0.914830312f, -0.403838458f,
   -0.914209756f, -0.405241314f,
   -0.913587048f, -0.406643217f,
   -0.912962190f, -0.408044163f,
   -0.912335185f, -0.409444149f,
   -0.911706032f, -0.410843171f,
   -0.911074734f, -0.412241227f,
   -0.910441292f, -0.413638312f,
   -0.909805708f, -0.415034424f,
   -0.909167983f, -0.416429560f,
   -0.908528119f, -0.417823716f,
   -0.907886116f, -0.419216888f,
   -0.907241978f, -0.420609074f,
   -0.906595705f, -0.422000271f,
   -0.905947298f, -0.423390474f,
   -0.905296759f, -0.424779681f,
   -0.904644091f, -0.426167889f,
   -0.903989293f, -0.427555093f,
   -0.903332368f, -0.428941292f,
   -0.902673318f, -0.430326481f,
   -0.902012144f, -0.431710658f,
   -0.901348847f, -0.433093819f,
   -0.900683429f, -0.434475961f,
   -0.900015892f, -0.435857080f,
   -0.899346237f, -0.437237174f,
   -0.898674466f, -0.438616239f,
   -0.898000580f, -0.439994271f,
   -0.897324581f, -0.441371269f,
   -0.896646470f, -0.442747228f,
   -0.895966250f, -0.444122145f,
   -0.895283921f, -0.445496017f,
   -0.894599486f, -0.446868840f,
   -0.893912945f, -0.448240612f,
   -0.893224301f, -0.449611330f,
   -0.892533555f, -0.450980989f,
   -0.891840709f, -0.452349587f,
   -0.891145765f, -0.453717121f,
   -0.890448723f, -0.455083587f,
   -0.889749586f, -0.456448982f,
   -0.889048356f, -0.457813304f,
   -0.888345033f, -0.459176548f,
   -0.887639620f, -0.460538711f,
   -0.886932119f, -0.461899791f,
   -0.886222530f, -0.463259784f,
   -0.885510856f, -0.464618686f,
   -0.884797098f, -0.465976496f,
   -0.884081259f, -0.467333209f,
   -0.883363339f, -0.468688822f,
   -0.882643340f, -0.470043332f,
   -0.881921264f, -0.471396737f,
   -0.881197113f, -0.472749032f,
   -0.880470889f, -0.474100215f,
   -0.879742593f, -0.475450282f,
   -0.879012226f, -0.476799230f,
   -0.878279792f, -0.478147056f,
   -0.877545290f, -0.479493758f,
   -0.876808724f, -0.480839331f,
   -0.876070094f, -0.482183772f,
   -0.875329403f, -0.483527079f,
   -0.874586652f, -0.484869248f,
   -0.873841843f, -0.486210276f,
   -0.873094978f, -0.487550160f,
   -0.872346059f, -0.488888897f,
   -0.871595087f, -0.490226483f,
   -0.870842063f, -0.491562916f,
   -0.870086991f, -0.492898192f,
   -0.869329871f, -0.494232309f,
   -0.868570706f, -0.495565262f,
   -0.867809497f, -0.496897049f,
   -0.867046246f, -0.498227667f,
   -0.866280954f, -0.499557113f,
   -0.865513624f, -0.500885383f,
   -0.864744258f, -0.502212474f,
   -0.863972856f, -0.503538384f,
   -0.863199422f, -0.504863109f,
   -0.862423956f, -0.506186645f,
   -0.861646461f, -0.507508991f,
   -0.860866939f, -0.508830143f,
   -0.860085390f, -0.510150097f,
   -0.859301818f, -0.511468850f,
   -0.858516224f, -0.512786401f,
   -0.857728610f, -0.514102744f,
   -0.856938977f, -0.515417878f,
   -0.856147328f, -0.516731799f,
   -0.855353665f, -0.518044504f,
   -0.854557988f, -0.519355990f,
   -0.853760301f, -0.520666254f,
   -0.852960605f, -0.521975293f,
   -0.852158902f, -0.523283103f,
   -0.851355193f, -0.524589683f,
   -0.850549481f, -0.525895027f,
   -0.849741768f, -0.527199135f,
   -0.848932055f, -0.528502002f,
   -0.848120345f, -0.529803625f,
   -0.847306639f, -0.531104001f,
   -0.846490939f, -0.532403128f,
   -0.845673247f, -0.533701002f,
   -0.844853565f, -0.534997620f,
   -0.844031895f, -0.536292979f,
   -0.843208240f, -0.537587076f,
   -0.842382600f, -0.538879909f,
   -0.841554977f, -0.540171473f,
   -0.840725375f, -0.541461766f,
   -0.839893794f, -0.542750785f,
   -0.839060237f, -0.544038527f,
   -0.838224706f, -0.545324988f,
   -0.837387202f, -0.546610167f,
   -0.836547727f, -0.547894059f,
   -0.835706284f, -0.549176662f,
   -0.834862875f, -0.550457973f,
   -0.834017501f, -0.551737988f,
   -0.833170165f, -0.553016706f,
   -0.832320868f, -0.554294121f,
   -0.831469612f, -0.555570233f,
   -0.830616400f, -0.556845037f,
   -0.829761234f, -0.558118531f,
   -0.828904115f, -0.559390712f,
   -0.828045045f, -0.560661576f,
   -0.827184027f, -0.561931121f,
   -0.826321063f, -0.563199344f,
   -0.825456154f, -0.564466242f,
   -0.824589303f, -0.565731811f,
   -0.823720511f, -0.566996049f,
   -0.822849781f, -0.568258953f,
   -0.821977115f, -0.569520519f,
   -0.821102515f, -0.570780746f,
   -0.820225983f, -0.572039629f,
   -0.819347520f, -0.573297167f,
   -0.818467130f, -0.574553355f,
   -0.817584813f, -0.575808191f,
   -0.816700573f, -0.577061673f,
   -0.815814411f, -0.578313796f,
   -0.814926329f, -0.579564559f,
   -0.814036330f, -0.580813958f,
   -0.813144415f, -0.582061990f,
   -0.812250587f, -0.583308653f,
   -0.811354847f, -0.584553943f,
   -0.810457198f, -0.585797857f,
   -0.809557642f, -0.587040394f,
   -0.808656182f, -0.588281548f,
   -0.807752818f, -0.589521319f,
   -0.806847554f, -0.590759702f,
   -0.805940391f, -0.591996695f,
   -0.805031331f, -0.593232295f,
   -0.804120377f, -0.594466499f,
   -0.803207531f, -0.595699304f,
   -0.802292796f, -0.596930708f,
   -0.801376172f, -0.598160707f,
   -0.800457662f, -0.599389298f,
   -0.799537269f, -0.600616479f,
   -0.798614995f, -0.601842247f,
   -0.797690841f, -0.603066599f,
   -0.796764810f, -0.604289531f,
   -0.795836905f, -0.605511041f,
   -0.794907126f, -0.606731127f,
   -0.793975478f, -0.607949785f,
   -0.793041960f, -0.609167012f,
   -0.792106577f, -0.610382806f,
   -0.791169330f, -0.611597164f,
   -0.790230221f, -0.612810082f,
   -0.789289253f, -0.614021559f,
   -0.788346428f, -0.615231591f,
   -0.787401747f, -0.616440175f,
   -0.786455214f, -0.617647308f,
   -0.785506830f, -0.618852988f,
   -0.784556597f, -0.620057212f,
   -0.783604519f, -0.621259977f,
   -0.782650596f, -0.622461279f,
   -0.781694832f, -0.623661118f,
   -0.780737229f, -0.624859488f,
   -0.779777788f, -0.626056388f,
   -0.778816512f, -0.627251815f,
   -0.777853404f, -0.628445767f,
   -0.776888466f, -0.629638239f,
   -0.775921699f, -0.630829230f,
   -0.774953107f, -0.632018736f,
   -0.773982691f, -0.633206755f,
   -0.773010453f, -0.634393284f,
   -0.772036397f, -0.635578320f,
   -0.771060524f, -0.636761861f,
   -0.770082837f, -0.637943904f,
   -0.769103338f, -0.639124445f,
   -0.768122029f, -0.640303482f,
   -0.767138912f, -0.641481013f,
   -0.766153990f, -0.642657034f,
   -0.765167266f, -0.643831543f,
   -0.764178741f, -0.645004537f,
   -0.763188417f, -0.646176013f,
   -0.762196298f, -0.647345969f,
   -0.761202385f, -0.648514401f,
   -0.760206682f, -0.649681307f,
   -0.759209189f, -0.650846685f,
   -0.758209910f, -0.652010531f,
   -0.757208847f, -0.653172843f,
   -0.756206001f, -0.654333618f,
   -0.755201377f, -0.655492853f,
   -0.754194975f, -0.656650546f,
   -0.753186799f, -0.657806693f,
   -0.752176850f, -0.658961293f,
   -0.751165132f, -0.660114342f,
   -0.750151646f, -0.661265838f,
   -0.749136395f, -0.662415778f,
   -0.748119380f, -0.663564159f,
   -0.747100606f, -0.664710978f,
   -0.746080074f, -0.665856234f,
   -0.745057785f, -0.666999922f,
   -0.744033744f, -0.668142041f,
   -0.743007952f, -0.669282588f,
   -0.741980412f, -0.670421560f,
   -0.740951125f, -0.671558955f,
   -0.739920095f, -0.672694769f,
   -0.738887324f, -0.673829000f,
   -0.737852815f, -0.674961646f,
   -0.736816569f, -0.676092704f,
   -0.735778589f, -0.677222170f,
   -0.734738878f, -0.678350043f,
   -0.733697438f, -0.679476320f,
   -0.732654272f, -0.680600998f,
   -0.731609381f, -0.681724074f,
   -0.730562769f, -0.682845546f,
   -0.729514438f, -0.683965412f,
   -0.728464390f, -0.685083668f,
   -0.727412629f, -0.686200312f,
   -0.726359155f, -0.687315341f,
   -0.725303972f, -0.688428753f,
   -0.724247083f, -0.689540545f,
   -0.723188489f, -0.690650714f,
   -0.722128194f, -0.691759258f,
   -0.721066199f, -0.692866175f,
   -0.720002508f, -0.693971461f,
   -0.718937122f, -0.695075114f,
   -0.717870045f, -0.696177131f,
   -0.716801279f, -0.697277511f,
   -0.715730825f, -0.698376249f,
   -0.714658688f, -0.699473345f,
   -0.713584869f, -0.700568794f,
   -0.712509371f, -0.701662595f,
   -0.711432196f, -0.702754744f,
   -0.710353347f, -0.703845241f,
   -0.709272826f, -0.704934080f,
   -0.708190637f, -0.706021261f,
   -0.707106781f, -0.707106781f,
   -0.706021261f, -0.708190637f,
   -0.704934080f, -0.709272826f,
   -0.703845241f, -0.710353347f,
   -0.702754744f, -0.711432196f,
   -0.701662595f, -0.712509371f,
   -0.700568794f, -0.713584869f,
   -0.699473345f, -0.714658688f,
   -0.698376249f, -0.715730825f,
   -0.697277511f, -0.716801279f,
   -0.696177131f, -0.717870045f,
   -0.695075114f, -0.718937122f,
   -0.693971461f, -0.720002508f,
   -0.692866175f, -0.721066199f,
   -0.691759258f, -0.722128194f,
   -0.690650714f, -0.723188489f,
   -0.689540545f, -0.724247083f,
   -0.688428753f, -0.725303972f,
   -0.687315341f, -0.726359155f,
   -0.686200312f, -0.727412629f,
   -0.685083668f, -0.728464390f,
   -0.683965412f, -0.729514438f,
   -0.682845546f, -0.730562769f,
   -0.681724074f, -0.731609381f,
   -0.680600998f, -0.732654272f,
   -0.679476320f, -0.733697438f,
   -0.678350043f, -0.734738878f,
   -0.677222170f, -0.735778589f,
   -0.676092704f, -0.736816569f,
   -0.674961646f, -0.737852815f,
   -0.673829000f, -0.738887324f,
   -0.672694769f, -0.739920095f,
   -0.671558955f, -0.740951125f,
   -0.670421560f, -0.741980412f,
   -0.669282588f, -0.743007952f,
   -0.668142041f, -0.744033744f,
   -0.666999922f, -0.745057785f,
   -0.665856234f, -0.746080074f,
   -0.664710978f, -0.747100606f,
   -0.663564159f, -0.748119380f,
   -0.662415778f, -0.749136395f,
   -0.661265838f, -0.750151646f,
   -0.660114342f, -0.751165132f,
   -0.658961293f, -0.752176850f,
   -0.657806693f, -0.753186799f,
   -0.656650546f, -0.754194975f,
   -0.655492853f, -0.755201377f,
   -0.654333618f, -0.756206001f,
   -0.653172843f, -0.757208847f,
   -0.652010531f, -0.758209910f,
   -0.650846685f, -0.759209189f,
   -0.649681307f, -0.760206682f,
   -0.648514401f, -0.761202385f,
   -0.647345969f, -0.762196298f,
   -0.646176013f, -0.763188417f,
   -0.645004537f, -0.764178741f,
   -0.643831543f, -0.765167266f,
   -0.642657034f, -0.766153990f,
   -0.641481013f, -0.767138912f,
   -0.640303482f, -0.768122029f,
   -0.639124445f, -0.769103338f,
   -0.637943904f, -0.770082837f,
   -0.636761861f, -0.771060524f,
   -0.635578320f, -0.772036397f,
   -0.634393284f, -0.773010453f,
   -0.633206755f, -0.773982691f,
   -0.632018736f, -0.774953107f,
   -0.630829230f, -0.775921699f,
   -0.629638239f, -0.776888466f,
   -0.628445767f, -0.777853404f,
   -0.627251815f, -0.778816512f,
   -0.626056388f, -0.779777788f,
   -0.624859488f, -0.780737229f,
   -0.623661118f, -0.781694832f,
   -0.622461279f, -0.782650596f,
   -0.621259977f, -0.783604519f,
   -0.620057212f, -0.784556597f,
   -0.618852988f, -0.785506830f,
   -0.617647308f, -0.786455214f,
   -0.616440175f, -0.787401747f,
   -0.615231591f, -0.788346428f,
   -0.614021559f, -0.789289253f,
   -0.612810082f, -0.790230221f,
   -0.611597164f, -0.791169330f,
   -0.610382806f, -0.792106577f,
   -0.609167012f, -0.793041960f,
   -0.607949785f, -0.793975478f,
   -0.606731127f, -0.794907126f,
   -0.605511041f, -0.795836905f,
   -0.604289531f, -0.796764810f,
   -0.603066599f, -0.797690841f,
   -0.601842247f, -0.798614995f,
   -0.600616479f, -0.799537269f,
   -0.599389298f, -0.800457662f,
   -0.598160707f, -0.801376172f,
   -0.596930708f, -0.802292796f,
   -0.595699304f, -0.803207531f,
   -0.594466499f, -0.804120377f,
   -0.593232295f, -0.805031331f,
   -0.591996695f, -0.805940391f,
   -0.590759702f, -0.806847554f,
   -0.589521319f, -0.807752818f,
   -0.588281548f, -0.808656182f,
   -0.587040394f, -0.809557642f,
   -0.585797857f, -0.810457198f,
   -0.584553943f, -0.811354847f,
   -0.583308653f, -0.812250587f,
   -0.582061990f, -0.813144415f,
   -0.580813958f, -0.814036330f,
   -0.579564559f, -0.814926329f,
   -0.578313796f, -0.815814411f,
   -0.577061673f, -0.816700573f,
   -0.575808191f, -0.817584813f,
   -0.574553355f, -0.818467130f,
   -0.573297167f, -0.819347520f,
   -0.572039629f, -0.820225983f,
   -0.570780746f, -0.821102515f,
   -0.569520519f, -0.821977115f,
   -0.568258953f, -0.822849781f,
   -0.566996049f, -0.823720511f,
   -0.565731811f, -0.824589303f,
   -0.564466242f, -0.825456154f,
   -0.563199344f, -0.826321063f,
   -0.561931121f, -0.827184027f,
   -0.560661576f, -0.828045045f,
   -0.559390712f, -0.828904115f,
   -0.558118531f, -0.829761234f,
   -0.556845037f, -0.830616400f,
   -0.555570233f, -0.831469612f,
   -0.554294121f, -0.832320868f,
   -0.553016706f, -0.833170165f,
   -0.551737988f, -0.834017501f,
   -0.550457973f, -0.834862875f,
   -0.549176662f, -0.835706284f,
   -0.547894059f, -0.836547727f,
   -0.546610167f, -0.837387202f,
   -0.545324988f, -0.838224706f,
   -0.544038527f, -0.839060237f,
   -0.542750785f, -0.839893794f,
   -0.541461766f, -0.840725375f,
   -0.540171473f, -0.841554977f,
   -0.538879909f, -0.842382600f,
   -0.537587076f, -0.843208240f,
   -0.536292979f, -0.844031895f,
   -0.534997620f, -0.844853565f,
   -0.533701002f, -0.845673247f,
   -0.532403128f, -0.846490939f,
   -0.531104001f, -0.847306639f,
   -0.529803625f, -0.848120345f,
   -0.528502002f, -0.848932055f,
   -0.527199135f, -0.849741768f,
   -0.525895027f, -0.850549481f,
   -0.524589683f, -0.851355193f,
   -0.523283103f, -0.852158902f,
   -0.521975293f, -0.852960605f,
   -0.520666254f, -0.853760301f,
   -0.519355990f, -0.854557988f,
   -0.518044504f, -0.855353665f,
   -0.516731799f, -0.856147328f,
   -0.515417878f, -0.856938977f,
   -0.514102744f, -0.857728610f,
   -0.512786401f, -0.858516224f,
   -0.511468850f, -0.859301818f,
   -0.510150097f, -0.860085390f,
   -0.508830143f, -0.860866939f,
   -0.507508991f, -0.861646461f,
   -0.506186645f, -0.862423956f,
   -0.504863109f, -0.863199422f,
   -0.503538384f, -0.863972856f,
   -0.502212474f, -0.864744258f,
   -0.500885383f, -0.865513624f,
   -0.499557113f, -0.866280954f,
   -0.498227667f, -0.867046246f,
   -0.496897049f, -0.867809497f,
   -0.495565262f, -0.868570706f,
   -0.494232309f, -0.869329871f,
   -0.492898192f, -0.870086991f,
   -0.491562916f, -0.870842063f,
   -0.490226483f, -0.871595087f,
   -0.488888897f, -0.872346059f,
   -0.487550160f, -0.873094978f,
   -0.486210276f, -0.873841843f,
   -0.484869248f, -0.874586652f,
   -0.483527079f, -0.875329403f,
   -0.482183772f, -0.876070094f,
   -0.480839331f, -0.876808724f,
   -0.479493758f, -0.877545290f,
   -0.478147056f, -0.878279792f,
   -0.476799230f, -0.879012226f,
   -0.475450282f, -0.879742593f,
   -0.474100215f, -0.880470889f,
   -0.472749032f, -0.881197113f,
   -0.471396737f, -0.881921264f,
   -0.470043332f, -0.882643340f,
   -0.468688822f, -0.883363339f,
   -0.467333209f, -0.884081259f,
   -0.465976496f, -0.884797098f,
   -0.464618686f, -0.885510856f,
   -0.463259784f, -0.886222530f,
   -0.461899791f, -0.886932119f,
   -0.460538711f, -0.887639620f,
   -0.459176548f, -0.888345033f,
   -0.457813304f, -0.889048356f,
   -0.456448982f, -0.889749586f,
   -0.455083587f, -0.890448723f,
   -0.453717121f, -0.891145765f,
   -0.452349587f, -0.891840709f,
   -0.450980989f, -0.892533555f,
   -0.449611330f, -0.893224301f,
   -0.448240612f, -0.893912945f,
   -0.446868840f, -0.894599486f,
   -0.445496017f, -0.895283921f,
   -0.444122145f, -0.895966250f,
   -0.442747228f, -0.896646470f,
   -0.441371269f, -0.897324581f,
   -0.439994271f, -0.898000580f,
   -0.438616239f, -0.898674466f,
   -0.437237174f, -0.899346237f,
   -0.435857080f, -0.900015892f,
   -0.434475961f, -0.900683429f,
   -0.433093819f, -0.901348847f,
   -0.431710658f, -0.902012144f,
   -0.430326481f, -0.902673318f,
   -0.428941292f, -0.903332368f,
   -0.427555093f, -0.903989293f,
   -0.426167889f, -0.904644091f,
   -0.424779681f, -0.905296759f,
   -0.423390474f, -0.905947298f,
   -0.422000271f, -0.906595705f,
   -0.420609074f, -0.907241978f,
   -0.419216888f, -0.907886116f,
   -0.417823716f, -0.908528119f,
   -0.416429560f, -0.909167983f,
   -0.415034424f, -0.909805708f,
   -0.413638312f, -0.910441292f,
   -0.412241227f, -0.911074734f,
   -0.410843171f, -0.911706032f,
   -0.409444149f, -0.912335185f,
   -0.408044163f, -0.912962190f,
   -0.406643217f, -0.913587048f,
   -0.405241314f, -0.914209756f,
   -0.403838458f, -0.914830312f,
   -0.402434651f, -0.915448716f,
   -0.401029897f, -0.916064966f,
   -0.399624200f, -0.916679060f,
   -0.398217562f, -0.917290997f,
   -0.396809987f, -0.917900776f,
   -0.395401479f, -0.918508394f,
   -0.393992040f, -0.919113852f,
   -0.392581674f, -0.919717146f,
   -0.391170384f, -0.920318277f,
   -0.389758174f, -0.920917242f,
   -0.388345047f, -0.921514039f,
   -0.386931006f, -0.922108669f,
   -0.385516054f, -0.922701128f,
   -0.384100195f, -0.923291417f,
   -0.382683432f, -0.923879533f,
   -0.381265769f, -0.924465474f,
   -0.379847209f, -0.925049241f,
   -0.378427755f, -0.925630831f,
   -0.377007410f, -0.926210242f,
   -0.375586178f, -0.926787474f,
   -0.374164063f, -0.927362526f,
   -0.372741067f, -0.927935395f,
   -0.371317194f, -0.928506080f,
   -0.369892447f, -0.929074581f,
   -0.368466830f, -0.929640896f,
   -0.367040346f, -0.930205023f,
   -0.365612998f, -0.930766961f,
   -0.364184790f, -0.931326709f,
   -0.362755724f, -0.931884266f,
   -0.361325806f, -0.932439629f,
   -0.359895037f, -0.932992799f,
   -0.358463421f, -0.933543773f,
   -0.357030961f, -0.934092550f,
   -0.355597662f, -0.934639130f,
   -0.354163525f, -0.935183510f,
   -0.352728556f, -0.935725689f,
   -0.351292756f, -0.936265667f,
   -0.349856130f, -0.936803442f,
   -0.348418680f, -0.937339012f,
   -0.346980411f, -0.937872376f,
   -0.345541325f, -0.938403534f,
   -0.344101426f, -0.938932484f,
   -0.342660717f, -0.939459224f,
   -0.341219202f, -0.939983753f,
   -0.339776884f, -0.940506071f,
   -0.338333767f, -0.941026175f,
   -0.336889853f, -0.941544065f,
   -0.335445147f, -0.942059740f,
   -0.333999651f, -0.942573198f,
   -0.332553370f, -0.943084437f,
   -0.331106306f, -0.943593458f,
   -0.329658463f, -0.944100258f,
   -0.328209844f, -0.944604837f,
   -0.326760452f, -0.945107193f,
   -0.325310292f, -0.945607325f,
   -0.323859367f, -0.946105232f,
   -0.322407679f, -0.946600913f,
   -0.320955232f, -0.947094366f,
   -0.319502031f, -0.947585591f,
   -0.318048077f, -0.948074586f,
   -0.316593376f, -0.948561350f,
   -0.315137929f, -0.949045882f,
   -0.313681740f, -0.949528181f,
   -0.312224814f, -0.950008245f,
   -0.310767153f, -0.950486074f,
   -0.309308760f, -0.950961666f,
   -0.307849640f, -0.951435021f,
   -0.306389795f, -0.951906137f,
   -0.304929230f, -0.952375013f,
   -0.303467947f, -0.952841648f,
   -0.302005949f, -0.953306040f,
   -0.300543241f, -0.953768190f,
   -0.299079826f, -0.954228095f,
   -0.297615707f, -0.954685755f,
   -0.296150888f, -0.955141168f,
   -0.294685372f, -0.955594334f,
   -0.293219163f, -0.956045251f,
   -0.291752263f, -0.956493919f,
   -0.290284677f, -0.956940336f,
   -0.288816408f, -0.957384501f,
   -0.287347460f, -0.957826413f,
   -0.285877835f, -0.958266071f,
   -0.284407537f, -0.958703475f,
   -0.282936570f, -0.959138622f,
   -0.281464938f, -0.959571513f,
   -0.279992643f, -0.960002146f,
   -0.278519689f, -0.960430519f,
   -0.277046080f, -0.960856633f,
   -0.275571819f, -0.961280486f,
   -0.274096910f, -0.961702077f,
   -0.272621355f, -0.962121404f,
   -0.271145160f, -0.962538468f,
   -0.269668326f, -0.962953267f,
   -0.268190857f, -0.963365800f,
   -0.266712757f, -0.963776066f,
   -0.265234030f, -0.964184064f,
   -0.263754679f, -0.964589793f,
   -0.262274707f, -0.964993253f,
   -0.260794118f, -0.965394442f,
   -0.259312915f, -0.965793359f,
   -0.257831102f, -0.966190003f,
   -0.256348682f, -0.966584374f,
   -0.254865660f, -0.966976471f,
   -0.253382037f, -0.967366292f,
   -0.251897818f, -0.967753837f,
   -0.250413007f, -0.968139105f,
   -0.248927606f, -0.968522094f,
   -0.247441619f, -0.968902805f,
   -0.245955050f, -0.969281235f,
   -0.244467903f, -0.969657385f,
   -0.242980180f, -0.970031253f,
   -0.241491885f, -0.970402839f,
   -0.240003022f, -0.970772141f,
   -0.238513595f, -0.971139158f,
   -0.237023606f, -0.971503891f,
   -0.235533059f, -0.971866337f,
   -0.234041959f, -0.972226497f,
   -0.232550307f, -0.972584369f,
   -0.231058108f, -0.972939952f,
   -0.229565366f, -0.973293246f,
   -0.228072083f, -0.973644250f,
   -0.226578264f, -0.973992962f,
   -0.225083911f, -0.974339383f,
   -0.223589029f, -0.974683511f,
   -0.222093621f, -0.975025345f,
   -0.220597690f, -0.975364885f,
   -0.219101240f, -0.975702130f,
   -0.217604275f, -0.976037079f,
   -0.216106797f, -0.976369731f,
   -0.214608811f, -0.976700086f,
   -0.213110320f, -0.977028143f,
   -0.211611327f, -0.977353900f,
   -0.210111837f, -0.977677358f,
   -0.208611852f, -0.977998515f,
   -0.207111376f, -0.978317371f,
   -0.205610413f, -0.978633924f,
   -0.204108966f, -0.978948175f,
   -0.202607039f, -0.979260123f,
   -0.201104635f, -0.979569766f,
   -0.199601758f, -0.979877104f,
   -0.198098411f, -0.980182136f,
   -0.196594598f, -0.980484862f,
   -0.195090322f, -0.980785280f,
   -0.193585587f, -0.981083391f,
   -0.192080397f, -0.981379193f,
   -0.190574755f, -0.981672686f,
   -0.189068664f, -0.981963869f,
   -0.187562129f, -0.982252741f,
   -0.186055152f, -0.982539302f,
   -0.184547737f, -0.982823551f,
   -0.183039888f, -0.983105487f,
   -0.181531608f, -0.983385110f,
   -0.180022901f, -0.983662419f,
   -0.178513771f, -0.983937413f,
   -0.177004220f, -0.984210092f,
   -0.175494253f, -0.984480455f,
   -0.173983873f, -0.984748502f,
   -0.172473084f, -0.985014231f,
   -0.170961889f, -0.985277642f,
   -0.169450291f, -0.985538735f,
   -0.167938295f, -0.985797509f,
   -0.166425904f, -0.986053963f,
   -0.164913120f, -0.986308097f,
   -0.163399949f, -0.986559910f,
   -0.161886394f, -0.986809402f,
   -0.160372457f, -0.987056571f,
   -0.158858143f, -0.987301418f,
   -0.157343456f, -0.987543942f,
   -0.155828398f, -0.987784142f,
   -0.154312973f, -0.988022017f,
   -0.152797185f, -0.988257568f,
   -0.151281038f, -0.988490793f,
   -0.149764535f, -0.988721692f,
   -0.148247679f, -0.988950265f,
   -0.146730474f, -0.989176510f,
   -0.145212925f, -0.989400428f,
   -0.143695033f, -0.989622017f,
   -0.142176804f, -0.989841278f,
   -0.140658239f, -0.990058210f,
   -0.139139344f, -0.990272812f,
   -0.137620122f, -0.990485084f,
   -0.136100575f, -0.990695025f,
   -0.134580709f, -0.990902635f,
   -0.133060525f, -0.991107914f,
   -0.131540029f, -0.991310860f,
   -0.130019223f, -0.991511473f,
   -0.128498111f, -0.991709754f,
   -0.126976696f, -0.991905700f,
   -0.125454983f, -0.992099313f,
   -0.123932975f, -0.992290591f,
   -0.122410675f, -0.992479535f,
   -0.120888087f, -0.992666142f,
   -0.119365215f, -0.992850414f,
   -0.117842062f, -0.993032350f,
   -0.116318631f, -0.993211949f,
   -0.114794927f, -0.993389211f,
   -0.113270952f, -0.993564136f,
   -0.111746711f, -0.993736722f,
   -0.110222207f, -0.993906970f,
   -0.108697444f, -0.994074879f,
   -0.107172425f, -0.994240449f,
   -0.105647154f, -0.994403680f,
   -0.104121634f, -0.994564571f,
   -0.102595869f, -0.994723121f,
   -0.101069863f, -0.994879331f,
   -0.099543619f, -0.995033199f,
   -0.098017140f, -0.995184727f,
   -0.096490431f, -0.995333912f,
   -0.094963495f, -0.995480755f,
   -0.093436336f, -0.995625256f,
   -0.091908956f, -0.995767414f,
   -0.090381361f, -0.995907229f,
   -0.088853553f, -0.996044701f,
   -0.087325535f, -0.996179829f,
   -0.085797312f, -0.996312612f,
   -0.084268888f, -0.996443051f,
   -0.082740265f, -0.996571146f,
   -0.081211447f, -0.996696895f,
   -0.079682438f, -0.996820299f,
   -0.078153242f, -0.996941358f,
   -0.076623861f, -0.997060070f,
   -0.075094301f, -0.997176437f,
   -0.073564564f, -0.997290457f,
   -0.072034653f, -0.997402130f,
   -0.070504573f, -0.997511456f,
   -0.068974328f, -0.997618435f,
   -0.067443920f, -0.997723067f,
   -0.065913353f, -0.997825350f,
   -0.064382631f, -0.997925286f,
   -0.062851758f, -0.998022874f,
   -0.061320736f, -0.998118113f,
   -0.059789571f, -0.998211003f,
   -0.058258265f, -0.998301545f,
   -0.056726821f, -0.998389737f,
   -0.055195244f, -0.998475581f,
   -0.053663538f, -0.998559074f,
   -0.052131705f, -0.998640218f,
   -0.050599749f, -0.998719012f,
   -0.049067674f, -0.998795456f,
   -0.047535484f, -0.998869550f,
   -0.046003182f, -0.998941293f,
   -0.044470772f, -0.999010686f,
   -0.042938257f, -0.999077728f,
   -0.041405641f, -0.999142419f,
   -0.039872928f, -0.999204759f,
   -0.038340120f, -0.999264747f,
   -0.036807223f, -0.999322385f,
   -0.035274239f, -0.999377670f,
   -0.033741172f, -0.999430605f,
   -0.032208025f, -0.999481187f,
   -0.030674803f, -0.999529418f,
   -0.029141509f, -0.999575296f,
   -0.027608146f, -0.999618822f,
   -0.026074718f, -0.999659997f,
   -0.024541229f, -0.999698819f,
   -0.023007681f, -0.999735288f,
   -0.021474080f, -0.999769405f,
   -0.019940429f, -0.999801170f,
   -0.018406730f, -0.999830582f,
   -0.016872988f, -0.999857641f,
   -0.015339206f, -0.999882347f,
   -0.013805389f, -0.999904701f,
   -0.012271538f, -0.999924702f,
   -0.010737659f, -0.999942350f,
   -0.009203755f, -0.999957645f,
   -0.007669829f, -0.999970586f,
   -0.006135885f, -0.999981175f,
   -0.004601926f, -0.999989411f,
   -0.003067957f, -0.999995294f,
   -0.001533980f, -0.999998823f,
   -0.000000000f, -1.000000000f,
    0.001533980f, -0.999998823f,
    0.003067957f, -0.999995294f,
    0.004601926f, -0.999989411f,
    0.006135885f, -0.999981175f,
    0.007669829f, -0.999970586f,
    0.009203755f, -0.999957645f,
    0.010737659f, -0.999942350f,
    0.012271538f, -0.999924702f,
    0.013805389f, -0.999904701f,
    0.015339206f, -0.999882347f,
    0.016872988f, -0.999857641f,
    0.018406730f, -0.999830582f,
    0.019940429f, -0.999801170f,
    0.021474080f, -0.999769405f,
    0.023007681f, -0.999735288f,
    0.024541229f, -0.999698819f,
    0.026074718f, -0.999659997f,
    0.027608146f, -0.999618822f,
    0.029141509f, -0.999575296f,
    0.030674803f, -0.999529418f,
    0.032208025f, -0.999481187f,
    0.033741172f, -0.999430605f,
    0.035274239f, -0.999377670f,
    0.036807223f, -0.999322385f,
    0.038340120f, -0.999264747f,
    0.039872928f, -0.999204759f,
    0.041405641f, -0.999142419f,
    0.042938257f, -0.999077728f,
    0.044470772f, -0.999010686f,
    0.046003182f, -0.998941293f,
    0.047535484f, -0.998869550f,
    0.049067674f, -0.998795456f,
    0.050599749f, -0.998719012f,
    0.052131705f, -0.998640218f,
    0.053663538f, -0.998559074f,
    0.055195244f, -0.998475581f,
    0.056726821f, -0.998389737f,
    0.058258265f, -0.998301545f,
    0.059789571f, -0.998211003f,
    0.061320736f, -0.998118113f,
    0.062851758f, -0.998022874f,
    0.064382631f, -0.997925286f,
    0.065913353f, -0.997825350f,
    0.067443920f, -0.997723067f,
    0.068974328f, -0.997618435f,
    0.070504573f, -0.997511456f,
    0.072034653f, -0.997402130f,
    0.073564564f, -0.997290457f,
    0.075094301f, -0.997176437f,
    0.076623861f, -0.997060070f,
    0.078153242f, -0.996941358f,
    0.079682438f, -0.996820299f,
    0.081211447f, -0.996696895f,
    0.082740265f, -0.996571146f,
    0.084268888f, -0.996443051f,
    0.085797312f, -0.996312612f,
    0.087325535f, -0.996179829f,
    0.088853553f, -0.996044701f,
    0.090381361f, -0.995907229f,
    0.091908956f, -0.995767414f,
    0.093436336f, -0.995625256f,
    0.094963495f, -0.995480755f,
    0.096490431f, -0.995333912f,
    0.098017140f, -0.995184727f,
    0.099543619f, -0.995033199f,
    0.101069863f, -0.994879331f,
    0.102595869f, -0.994723121f,
    0.104121634f, -0.994564571f,
    0.105647154f, -0.994403680f,
    0.107172425f, -0.994240449f,
    0.108697444f, -0.994074879f,
    0.110222207f, -0.993906970f,
    0.111746711f, -0.993736722f,
    0.113270952f, -0.993564136f,
    0.114794927f, -0.993389211f,
    0.116318631f, -0.993211949f,
    0.117842062f, -0.993032350f,
    0.119365215f, -0.992850414f,
    0.120888087f, -0.992666142f,
    0.122410675f, -0.992479535f,
    0.123932975f, -0.992290591f,
    0.125454983f, -0.992099313f,
    0.126976696f, -0.991905700f,
    0.128498111f, -0.991709754f,
    0.130019223f, -0.991511473f,
    0.131540029f, -0.991310860f,
    0.133060525f, -0.991107914f,
    0.134580709f, -0.990902635f,
    0.136100575f, -0.990695025f,
    0.137620122f, -0.990485084f,
    0.139139344f, -0.990272812f,
    0.140658239f, -0.990058210f,
    0.142176804f, -0.989841278f,
    0.143695033f, -0.989622017f,
    0.145212925f, -0.989400428f,
    0.146730474f, -0.989176510f,
    0.148247679f, -0.988950265f,
    0.149764535f, -0.988721692f,
    0.151281038f, -0.988490793f,
    0.152797185f, -0.988257568f,
    0.154312973f, -0.988022017f,
    0.155828398f, -0.987784142f,
    0.157343456f, -0.987543942f,
    0.158858143f, -0.987301418f,
    0.160372457f, -0.987056571f,
    0.161886394f, -0.986809402f,
    0.163399949f, -0.986559910f,
    0.164913120f, -0.986308097f,
    0.166425904f, -0.986053963f,
    0.167938295f, -0.985797509f,
    0.169450291f, -0.985538735f,
    0.170961889f, -0.985277642f,
    0.172473084f, -0.985014231f,
    0.173983873f, -0.984748502f,
    0.175494253f, -0.984480455f,
    0.177004220f, -0.984210092f,
    0.178513771f, -0.983937413f,
    0.180022901f, -0.983662419f,
    0.181531608f, -0.983385110f,
    0.183039888f, -0.983105487f,
    0.184547737f, -0.982823551f,
    0.186055152f, -0.982539302f,
    0.187562129f, -0.982252741f,
    0.189068664f, -0.981963869f,
    0.190574755f, -0.981672686f,
    0.192080397f, -0.981379193f,
    0.193585587f, -0.981083391f,
    0.195090322f, -0.980785280f,
    0.196594598f, -0.980484862f,
    0.198098411f, -0.980182136f,
    0.199601758f, -0.979877104f,
    0.201104635f, -0.979569766f,
    0.202607039f, -0.979260123f,
    0.204108966f, -0.978948175f,
    0.205610413f, -0.978633924f,
    0.207111376f, -0.978317371f,
    0.208611852f, -0.977998515f,
    0.210111837f, -0.977677358f,
    0.211611327f, -0.977353900f,
    0.213110320f, -0.977028143f,
    0.214608811f, -0.976700086f,
    0.216106797f, -0.976369731f,
    0.217604275f, -0.976037079f,
    0.219101240f, -0.975702130f,
    0.220597690f, -0.975364885f,
    0.222093621f, -0.975025345f,
    0.223589029f, -0.974683511f,
    0.225083911f, -0.974339383f,
    0.226578264f, -0.973992962f,
    0.228072083f, -0.973644250f,
    0.229565366f, -0.973293246f,
    0.231058108f, -0.972939952f,
    0.232550307f, -0.972584369f,
    0.234041959f, -0.972226497f,
    0.235533059f, -0.971866337f,
    0.237023606f, -0.971503891f,
    0.238513595f, -0.971139158f,
    0.240003022f, -0.970772141f,
    0.241491885f, -0.970402839f,
    0.242980180f, -0.970031253f,
    0.244467903f, -0.969657385f,
    0.245955050f, -0.969281235f,
    0.247441619f, -0.968902805f,
    0.248927606f, -0.968522094f,
    0.250413007f, -0.968139105f,
    0.251897818f, -0.967753837f,
    0.253382037f, -0.967366292f,
    0.254865660f, -0.966976471f,
    0.256348682f, -0.966584374f,
    0.257831102f, -0.966190003f,
    0.259312915f, -0.965793359f,
    0.260794118f, -0.965394442f,
    0.262274707f, -0.964993253f,
    0.263754679f, -0.964589793f,
    0.265234030f, -0.964184064f,
    0.266712757f, -0.963776066f,
    0.268190857f, -0.963365800f,
    0.269668326f, -0.962953267f,
    0.271145160f, -0.962538468f,
    0.272621355f, -0.962121404f,
    0.274096910f, -0.961702077f,
    0.275571819f, -0.961280486f,
    0.277046080f, -0.960856633f,
    0.278519689f, -0.960430519f,
    0.279992643f, -0.960002146f,
    0.281464938f, -0.959571513f,
    0.282936570f, -0.959138622f,
    0.284407537f, -0.958703475f,
    0.285877835f, -0.958266071f,
    0.287347460f, -0.957826413f,
    0.288816408f, -0.957384501f,
    0.290284677f, -0.956940336f,
    0.291752263f, -0.956493919f,
    0.293219163f, -0.956045251f,
    0.294685372f, -0.955594334f,
    0.296150888f, -0.955141168f,
    0.297615707f, -0.954685755f,
    0.299079826f, -0.954228095f,
    0.300543241f, -0.953768190f,
    0.302005949f, -0.953306040f,
    0.303467947f, -0.952841648f,
    0.304929230f, -0.952375013f,
    0.306389795f, -0.951906137f,
    0.307849640f, -0.951435021f,
    0.309308760f, -0.950961666f,
    0.310767153f, -0.950486074f,
    0.312224814f, -0.950008245f,
    0.313681740f, -0.949528181f,
    0.315137929f, -0.949045882f,
    0.316593376f, -0.948561350f,
    0.318048077f, -0.948074586f,
    0.319502031f, -0.947585591f,
    0.320955232f, -0.947094366f,
    0.322407679f, -0.946600913f,
    0.323859367f, -0.946105232f,
    0.325310292f, -0.945607325f,
    0.326760452f, -0.945107193f,
    0.328209844f, -0.944604837f,
    0.329658463f, -0.944100258f,
    0.331106306f, -0.943593458f,
    0.332553370f, -0.943084437f,
    0.333999651f, -0.942573198f,
    0.335445147f, -0.942059740f,
    0.336889853f, -0.941544065f,
    0.338333767f, -0.941026175f,
    0.339776884f, -0.940506071f,
    0.341219202f, -0.939983753f,
    0.342660717f, -0.939459224f,
    0.344101426f, -0.938932484f,
    0.345541325f, -0.938403534f,
    0.346980411f, -0.937872376f,
    0.348418680f, -0.937339012f,
    0.349856130f, -0.936803442f,
    0.351292756f, -0.936265667f,
    0.352728556f, -0.935725689f,
    0.354163525f, -0.935183510f,
    0.355597662f, -0.934639130f,
    0.357030961f, -0.934092550f,
    0.358463421f, -0.933543773f,
    0.359895037f, -0.932992799f,
    0.361325806f, -0.932439629f,
    0.362755724f, -0.931884266f,
    0.364184790f, -0.931326709f,
    0.365612998f, -0.930766961f,
    0.367040346f, -0.930205023f,
    0.368466830f, -0.929640896f,
    0.369892447f, -0.929074581f,
    0.371317194f, -0.928506080f,
    0.372741067f, -0.927935395f,
    0.374164063f, -0.927362526f,
    0.375586178f, -0.926787474f,
    0.377007410f, -0.926210242f,
    0.378427755f, -0.925630831f,
    0.379847209f, -0.925049241f,
    0.381265769f, -0.924465474f,
    0.382683432f, -0.923879533f,
    0.384100195f, -0.923291417f,
    0.385516054f, -0.922701128f,
    0.386931006f, -0.922108669f,
    0.388345047f, -0.921514039f,
    0.389758174f, -0.920917242f,
    0.391170384f, -0.920318277f,
    0.392581674f, -0.919717146f,
    0.393992040f, -0.919113852f,
    0.395401479f, -0.918508394f,
    0.396809987f, -0.917900776f,
    0.398217562f, -0.917290997f,
    0.399624200f, -0.916679060f,
    0.401029897f, -0.916064966f,
    0.402434651f, -0.915448716f,
    0.403838458f, -0.914830312f,
    0.405241314f, -0.914209756f,
    0.406643217f, -0.913587048f,
    0.408044163f, -0.912962190f,
    0.409444149f, -0.912335185f,
    0.410843171f, -0.911706032f,
    0.412241227f, -0.911074734f,
    0.413638312f, -0.910441292f,
    0.415034424f, -0.909805708f,
    0.416429560f, -0.909167983f,
    0.417823716f, -0.908528119f,
    0.419216888f, -0.907886116f,
    0.420609074f, -0.907241978f,
    0.422000271f, -0.906595705f,
    0.423390474f, -0.905947298f,
    0.424779681f, -0.905296759f,
    0.426167889f, -0.904644091f,
    0.427555093f, -0.903989293f,
    0.428941292f, -0.903332368f,
    0.430326481f, -0.902673318f,
    0.431710658f, -0.902012144f,
    0.433093819f, -0.901348847f,
    0.434475961f, -0.900683429f,
    0.435857080f, -0.900015892f,
    0.437237174f, -0.899346237f,
    0.438616239f, -0.898674466f,
    0.439994271f, -0.898000580f,
    0.441371269f, -0.897324581f,
    0.442747228f, -0.896646470f,
    0.444122145f, -0.895966250f,
    0.445496017f, -0.895283921f,
    0.446868840f, -0.894599486f,
    0.448240612f, -0.893912945f,
    0.449611330f, -0.893224301f,
    0.450980989f, -0.892533555f,
    0.452349587f, -0.891840709f,
    0.453717121f, -0.891145765f,
    0.455083587f, -0.890448723f,
    0.456448982f, -0.889749586f,
    0.457813304f, -0.889048356f,
    0.459176548f, -0.888345033f,
    0.460538711f, -0.887639620f,
    0.461899791f, -0.886932119f,
    0.463259784f, -0.886222530f,
    0.464618686f, -0.885510856f,
    0.465976496f, -0.884797098f,
    0.467333209f, -0.884081259f,
    0.468688822f, -0.883363339f,
    0.470043332f, -0.882643340f,
    0.471396737f, -0.881921264f,
    0.472749032f, -0.881197113f,
    0.474100215f, -0.880470889f,
    0.475450282f, -0.879742593f,
    0.476799230f, -0.879012226f,
    0.478147056f, -0.878279792f,
    0.479493758f, -0.877545290f,
    0.480839331f, -0.876808724f,
    0.482183772f, -0.876070094f,
    0.483527079f, -0.875329403f,
    0.484869248f, -0.874586652f,
    0.486210276f, -0.873841843f,
    0.487550160f, -0.873094978f,
    0.488888897f, -0.872346059f,
    0.490226483f, -0.871595087f,
    0.491562916f, -0.870842063f,
    0.492898192f, -0.870086991f,
    0.494232309f, -0.869329871f,
    0.495565262f, -0.868570706f,
    0.496897049f, -0.867809497f,
    0.498227667f, -0.867046246f,
    0.499557113f, -0.866280954f,
    0.500885383f, -0.865513624f,
    0.502212474f, -0.864744258f,
    0.503538384f, -0.863972856f,
    0.504863109f, -0.863199422f,
    0.506186645f, -0.862423956f,
    0.507508991f, -0.861646461f,
    0.508830143f, -0.860866939f,
    0.510150097f, -0.860085390f,
    0.511468850f, -0.859301818f,
    0.512786401f, -0.858516224f,
    0.514102744f, -0.857728610f,
    0.515417878f, -0.856938977f,
    0.516731799f, -0.856147328f,
    0.518044504f, -0.855353665f,
    0.519355990f, -0.854557988f,
    0.520666254f, -0.853760301f,
    0.521975293f, -0.852960605f,
    0.523283103f, -0.852158902f,
    0.524589683f, -0.851355193f,
    0.525895027f, -0.850549481f,
    0.527199135f, -0.849741768f,
    0.528502002f, -0.848932055f,
    0.529803625f, -0.848120345f,
    0.531104001f, -0.847306639f,
    0.532403128f, -0.846490939f,
    0.533701002f, -0.845673247f,
    0.534997620f, -0.844853565f,
    0.536292979f, -0.844031895f,
    0.537587076f, -0.843208240f,
    0.538879909f, -0.842382600f,
    0.540171473f, -0.841554977f,
    0.541461766f, -0.840725375f,
    0.542750785f, -0.839893794f,
    0.544038527f, -0.839060237f,
    0.545324988f, -0.838224706f,
    0.546610167f, -0.837387202f,
    0.547894059f, -0.836547727f,
    0.549176662f, -0.835706284f,
    0.550457973f, -0.834862875f,
    0.551737988f, -0.834017501f,
    0.553016706f, -0.833170165f,
    0.554294121f, -0.832320868f,
    0.555570233f, -0.831469612f,
    0.556845037f, -0.830616400f,
    0.558118531f, -0.829761234f,
    0.559390712f, -0.828904115f,
    0.560661576f, -0.828045045f,
    0.561931121f, -0.827184027f,
    0.563199344f, -0.826321063f,
    0.564466242f, -0.825456154f,
    0.565731811f, -0.824589303f,
    0.566996049f, -0.823720511f,
    0.568258953f, -0.822849781f,
    0.569520519f, -0.821977115f,
    0.570780746f, -0.821102515f,
    0.572039629f, -0.820225983f,
    0.573297167f, -0.819347520f,
    0.574553355f, -0.818467130f,
    0.575808191f, -0.817584813f,
    0.577061673f, -0.816700573f,
    0.578313796f, -0.815814411f,
    0.579564559f, -0.814926329f,
    0.580813958f, -0.814036330f,
    0.582061990f, -0.813144415f,
    0.583308653f, -0.812250587f,
    0.584553943f, -0.811354847f,
    0.585797857f, -0.810457198f,
    0.587040394f, -0.809557642f,
    0.588281548f, -0.808656182f,
    0.589521319f, -0.807752818f,
    0.590759702f, -0.806847554f,
    0.591996695f, -0.805940391f,
    0.593232295f, -0.805031331f,
    0.594466499f, -0.804120377f,
    0.595699304f, -0.803207531f,
    0.596930708f, -0.802292796f,
    0.598160707f, -0.801376172f,
    0.599389298f, -0.800457662f,
    0.600616479f, -0.799537269f,
    0.601842247f, -0.798614995f,
    0.603066599f, -0.797690841f,
    0.604289531f, -0.796764810f,
    0.605511041f, -0.795836905f,
    0.606731127f, -0.794907126f,
    0.607949785f, -0.793975478f,
    0.609167012f, -0.793041960f,
    0.610382806f, -0.792106577f,
    0.611597164f, -0.791169330f,
    0.612810082f, -0.790230221f,
    0.614021559f, -0.789289253f,
    0.615231591f, -0.788346428f,
    0.616440175f, -0.787401747f,
    0.617647308f, -0.786455214f,
    0.618852988f, -0.785506830f,
    0.620057212f, -0.784556597f,
    0.621259977f, -0.783604519f,
    0.622461279f, -0.782650596f,
    0.623661118f, -0.781694832f,
    0.624859488f, -0.780737229f,
    0.626056388f, -0.779777788f,
    0.627251815f, -0.778816512f,
    0.628445767f, -0.777853404f,
    0.629638239f, -0.776888466f,
    0.630829230f, -0.775921699f,
    0.632018736f, -0.774953107f,
    0.633206755f, -0.773982691f,
    0.634393284f, -0.773010453f,
    0.635578320f, -0.772036397f,
    0.636761861f, -0.771060524f,
    0.637943904f, -0.770082837f,
    0.639124445f, -0.769103338f,
    0.640303482f, -0.768122029f,
    0.641481013f, -0.767138912f,
    0.642657034f, -0.766153990f,
    0.643831543f, -0.765167266f,
    0.645004537f, -0.764178741f,
    0.646176013f, -0.763188417f,
    0.647345969f, -0.762196298f,
    0.648514401f, -0.761202385f,
    0.649681307f, -0.760206682f,
    0.650846685f, -0.759209189f,
    0.652010531f, -0.758209910f,
    0.653172843f, -0.757208847f,
    0.654333618f, -0.756206001f,
    0.655492853f, -0.755201377f,
    0.656650546f, -0.754194975f,
    0.657806693f, -0.753186799f,
    0.658961293f, -0.752176850f,
    0.660114342f, -0.751165132f,
    0.661265838f, -0.750151646f,
    0.662415778f, -0.749136395f,
    0.663564159f, -0.748119380f,
    0.664710978f, -0.747100606f,
    0.665856234f, -0.746080074f,
    0.666999922f, -0.745057785f,
    0.668142041f, -0.744033744f,
    0.669282588f, -0.743007952f,
    0.670421560f, -0.741980412f,
    0.671558955f, -0.740951125f,
    0.672694769f, -0.739920095f,
    0.673829000f, -0.738887324f,
    0.674961646f, -0.737852815f,
    0.676092704f, -0.736816569f,
    0.677222170f, -0.735778589f,
    0.678350043f, -0.734738878f,
    0.679476320f, -0.733697438f,
    0.680600998f, -0.732654272f,
    0.681724074f, -0.731609381f,
    0.682845546f, -0.730562769f,
    0.683965412f, -0.729514438f,
    0.685083668f, -0.728464390f,
    0.686200312f, -0.727412629f,
    0.687315341f, -0.726359155f,
    0.688428753f, -0.725303972f,
    0.689540545f, -0.724247083f,
    0.690650714f, -0.723188489f,
    0.691759258f, -0.722128194f,
    0.692866175f, -0.721066199f,
    0.693971461f, -0.720002508f,
    0.695075114f, -0.718937122f,
    0.696177131f, -0.717870045f,
    0.697277511f, -0.716801279f,
    0.698376249f, -0.715730825f,
    0.699473345f, -0.714658688f,
    0.700568794f, -0.713584869f,
    0.701662595f, -0.712509371f,
    0.702754744f, -0.711432196f,
    0.703845241f, -0.710353347f,
    0.704934080f, -0.709272826f,
    0.706021261f, -0.708190637f,
    0.707106781f, -0.707106781f,
    0.708190637f, -0.706021261f,
    0.709272826f, -0.704934080f,
    0.710353347f, -0.703845241f,
    0.711432196f, -0.702754744f,
    0.712509371f, -0.701662595f,
    0.713584869f, -0.700568794f,
    0.714658688f, -0.699473345f,
    0.715730825f, -0.698376249f,
    0.716801279f, -0.697277511f,
    0.717870045f, -0.696177131f,
    0.718937122f, -0.695075114f,
    0.720002508f, -0.693971461f,
    0.721066199f, -0.692866175f,
    0.722128194f, -0.691759258f,
    0.723188489f, -0.690650714f,
    0.724247083f, -0.689540545f,
    0.725303972f, -0.688428753f,
    0.726359155f, -0.687315341f,
    0.727412629f, -0.686200312f,
    0.728464390f, -0.685083668f,
    0.729514438f, -0.683965412f,
    0.730562769f, -0.682845546f,
    0.731609381f, -0.681724074f,
    0.732654272f, -0.680600998f,
    0.733697438f, -0.679476320f,
    0.734738878f, -0.678350043f,
    0.735778589f, -0.677222170f,
    0.736816569f, -0.676092704f,
    0.737852815f, -0.674961646f,
    0.738887324f, -0.673829000f,
    0.739920095f, -0.672694769f,
    0.740951125f, -0.671558955f,
    0.741980412f, -0.670421560f,
    0.743007952f, -0.669282588f,
    0.744033744f, -0.668142041f,
    0.745057785f, -0.666999922f,
    0.746080074f, -0.665856234f,
    0.747100606f, -0.664710978f,
    0.748119380f, -0.663564159f,
    0.749136395f, -0.662415778f,
    0.750151646f, -0.661265838f,
    0.751165132f, -0.660114342f,
    0.752176850f, -0.658961293f,
    0.753186799f, -0.657806693f,
    0.754194975f, -0.656650546f,
    0.755201377f, -0.655492853f,
    0.756206001f, -0.654333618f,
    0.757208847f, -0.653172843f,
    0.758209910f, -0.652010531f,
    0.759209189f, -0.650846685f,
    0.760206682f, -0.649681307f,
    0.761202385f, -0.648514401f,
    0.762196298f, -0.647345969f,
    0.763188417f, -0.646176013f,
    0.764178741f, -0.645004537f,
    0.765167266f, -0.643831543f,
    0.766153990f, -0.642657034f,
    0.767138912f, -0.641481013f,
    0.768122029f, -0.640303482f,
    0.769103338f, -0.639124445f,
    0.770082837f, -0.637943904f,
    0.771060524f, -0.636761861f,
    0.772036397f, -0.635578320f,
    0.773010453f, -0.634393284f,
    0.773982691f, -0.633206755f,
    0.774953107f, -0.632018736f,
    0.775921699f, -0.630829230f,
    0.776888466f, -0.629638239f,
    0.777853404f, -0.628445767f,
    0.778816512f, -0.627251815f,
    0.779777788f, -0.626056388f,
    0.780737229f, -0.624859488f,
    0.781694832f, -0.623661118f,
    0.782650596f, -0.622461279f,
    0.783604519f, -0.621259977f,
    0.784556597f, -0.620057212f,
    0.785506830f, -0.618852988f,
    0.786455214f, -0.617647308f,
    0.787401747f, -0.616440175f,
    0.788346428f, -0.615231591f,
    0.789289253f, -0.614021559f,
    0.790230221f, -0.612810082f,
    0.791169330f, -0.611597164f,
    0.792106577f, -0.610382806f,
    0.793041960f, -0.609167012f,
    0.793975478f, -0.607949785f,
    0.794907126f, -0.606731127f,
    0.795836905f, -0.605511041f,
    0.796764810f, -0.604289531f,
    0.797690841f, -0.603066599f,
    0.798614995f, -0.601842247f,
    0.799537269f, -0.600616479f,
    0.800457662f, -0.599389298f,
    0.801376172f, -0.598160707f,
    0.802292796f, -0.596930708f,
    0.803207531f, -0.595699304f,
    0.804120377f, -0.594466499f,
    0.805031331f, -0.593232295f,
    0.805940391f, -0.591996695f,
    0.806847554f, -0.590759702f,
    0.807752818f, -0.589521319f,
    0.808656182f, -0.588281548f,
    0.809557642f, -0.587040394f,
    0.810457198f, -0.585797857f,
    0.811354847f, -0.584553943f,
    0.812250587f, -0.583308653f,
    0.813144415f, -0.582061990f,
    0.814036330f, -0.580813958f,
    0.814926329f, -0.579564559f,
    0.815814411f, -0.578313796f,
    0.816700573f, -0.577061673f,
    0.817584813f, -0.575808191f,
    0.818467130f, -0.574553355f,
    0.819347520f, -0.573297167f,
    0.820225983f, -0.572039629f,
    0.821102515f, -0.570780746f,
    0.821977115f, -0.569520519f,
    0.822849781f, -0.568258953f,
    0.823720511f, -0.566996049f,
    0.824589303f, -0.565731811f,
    0.825456154f, -0.564466242f,
    0.826321063f, -0.563199344f,
    0.827184027f, -0.561931121f,
    0.828045045f, -0.560661576f,
    0.828904115f, -0.559390712f,
    0.829761234f, -0.558118531f,
    0.830616400f, -0.556845037f,
    0.831469612f, -0.555570233f,
    0.832320868f, -0.554294121f,
    0.833170165f, -0.553016706f,
    0.834017501f, -0.551737988f,
    0.834862875f, -0.550457973f,
    0.835706284f, -0.549176662f,
    0.836547727f, -0.547894059f,
    0.837387202f, -0.546610167f,
    0.838224706f, -0.545324988f,
    0.839060237f, -0.544038527f,
    0.839893794f, -0.542750785f,
    0.840725375f, -0.541461766f,
    0.841554977f, -0.540171473f,
    0.842382600f, -0.538879909f,
    0.843208240f, -0.537587076f,
    0.844031895f, -0.536292979f,
    0.844853565f, -0.534997620f,
    0.845673247f, -0.533701002f,
    0.846490939f, -0.532403128f,
    0.847306639f, -0.531104001f,
    0.848120345f, -0.529803625f,
    0.848932055f, -0.528502002f,
    0.849741768f, -0.527199135f,
    0.850549481f, -0.525895027f,
    0.851355193f, -0.524589683f,
    0.852158902f, -0.523283103f,
    0.852960605f, -0.521975293f,
    0.853760301f, -0.520666254f,
    0.854557988f, -0.519355990f,
    0.855353665f, -0.518044504f,
    0.856147328f, -0.516731799f,
    0.856938977f, -0.515417878f,
    0.857728610f, -0.514102744f,
    0.858516224f, -0.512786401f,
    0.859301818f, -0.511468850f,
    0.860085390f, -0.510150097f,
    0.860866939f, -0.508830143f,
    0.861646461f, -0.507508991f,
    0.862423956f, -0.506186645f,
    0.863199422f, -0.504863109f,
    0.863972856f, -0.503538384f,
    0.864744258f, -0.502212474f,
    0.865513624f, -0.500885383f,
    0.866280954f, -0.499557113f,
    0.867046246f, -0.498227667f,
    0.867809497f, -0.496897049f,
    0.868570706f, -0.495565262f,
    0.869329871f, -0.494232309f,
    0.870086991f, -0.492898192f,
    0.870842063f, -0.491562916f,
    0.871595087f, -0.490226483f,
    0.872346059f, -0.488888897f,
    0.873094978f, -0.487550160f,
    0.873841843f, -0.486210276f,
    0.874586652f, -0.484869248f,
    0.875329403f, -0.483527079f,
    0.876070094f, -0.482183772f,
    0.876808724f, -0.480839331f,
    0.877545290f, -0.479493758f,
    0.878279792f, -0.478147056f,
    0.879012226f, -0.476799230f,
    0.879742593f, -0.475450282f,
    0.880470889f, -0.474100215f,
    0.881197113f, -0.472749032f,
    0.881921264f, -0.471396737f,
    0.882643340f, -0.470043332f,
    0.883363339f, -0.468688822f,
    0.884081259f, -0.467333209f,
    0.884797098f, -0.465976496f,
    0.885510856f, -0.464618686f,
    0.886222530f, -0.463259784f,
    0.886932119f, -0.461899791f,
    0.887639620f, -0.460538711f,
    0.888345033f, -0.459176548f,
    0.889048356f, -0.457813304f,
    0.889749586f, -0.456448982f,
    0.890448723f, -0.455083587f,
    0.891145765f, -0.453717121f,
    0.891840709f, -0.452349587f,
    0.892533555f, -0.450980989f,
    0.893224301f, -0.449611330f,
    0.893912945f, -0.448240612f,
    0.894599486f, -0.446868840f,
    0.895283921f, -0.445496017f,
    0.895966250f, -0.444122145f,
    0.896646470f, -0.442747228f,
    0.897324581f, -0.441371269f,
    0.898000580f, -0.439994271f,
    0.898674466f, -0.438616239f,
    0.899346237f, -0.437237174f,
    0.900015892f, -0.435857080f,
    0.900683429f, -0.434475961f,
    0.901348847f, -0.433093819f,
    0.902012144f, -0.431710658f,
    0.902673318f, -0.430326481f,
    0.903332368f, -0.428941292f,
    0.903989293f, -0.427555093f,
    0.904644091f, -0.426167889f,
    0.905296759f, -0.424779681f,
    0.905947298f, -0.423390474f,
    0.906595705f, -0.422000271f,
    0.907241978f, -0.420609074f,
    0.907886116f, -0.419216888f,
    0.908528119f, -0.417823716f,
    0.909167983f, -0.416429560f,
    0.909805708f, -0.415034424f,
    0.910441292f, -0.413638312f,
    0.911074734f, -0.412241227f,
    0.911706032f, -0.410843171f,
    0.912335185f, -0.409444149f,
    0.912962190f, -0.408044163f,
    0.913587048f, -0.406643217f,
    0.914209756f, -0.405241314f,
    0.914830312f, -0.403838458f,
    0.915448716f, -0.402434651f,
    0.916064966f, -0.401029897f,
    0.916679060f, -0.399624200f,
    0.917290997f, -0.398217562f,
    0.917900776f, -0.396809987f,
    0.918508394f, -0.395401479f,
    0.919113852f, -0.393992040f,
    0.919717146f, -0.392581674f,
    0.920318277f, -0.391170384f,
    0.920917242f, -0.389758174f,
    0.921514039f, -0.388345047f,
    0.922108669f, -0.386931006f,
    0.922701128f, -0.385516054f,
    0.923291417f, -0.384100195f,
    0.923879533f, -0.382683432f,
    0.924465474f, -0.381265769f,
    0.925049241f, -0.379847209f,
    0.925630831f, -0.378427755f,
    0.926210242f, -0.377007410f,
    0.926787474f, -0.375586178f,
    0.927362526f, -0.374164063f,
    0.927935395f, -0.372741067f,
    0.928506080f, -0.371317194f,
    0.929074581f, -0.369892447f,
    0.929640896f, -0.368466830f,
    0.930205023f, -0.367040346f,
    0.930766961f, -0.365612998f,
    0.931326709f, -0.364184790f,
    0.931884266f, -0.362755724f,
    0.932439629f, -0.361325806f,
    0.932992799f, -0.359895037f,
    0.933543773f, -0.358463421f,
    0.934092550f, -0.357030961f,
    0.934639130f, -0.355597662f,
    0.935183510f, -0.354163525f,
    0.935725689f, -0.352728556f,
    0.936265667f, -0.351292756f,
    0.936803442f, -0.349856130f,
    0.937339012f, -0.348418680f,
    0.937872376f, -0.346980411f,
    0.938403534f, -0.345541325f,
    0.938932484f, -0.344101426f,
    0.939459224f, -0.342660717f,
    0.939983753f, -0.341219202f,
    0.940506071f, -0.339776884f,
    0.941026175f, -0.338333767f,
    0.941544065f, -0.336889853f,
    0.942059740f, -0.335445147f,
    0.942573198f, -0.333999651f,
    0.943084437f, -0.332553370f,
    0.943593458f, -0.331106306f,
    0.944100258f, -0.329658463f,
    0.944604837f, -0.328209844f,
    0.945107193f, -0.326760452f,
    0.945607325f, -0.325310292f,
    0.946105232f, -0.323859367f,
    0.946600913f, -0.322407679f,
    0.947094366f, -0.320955232f,
    0.947585591f, -0.319502031f,
    0.948074586f, -0.318048077f,
    0.948561350f, -0.316593376f,
    0.949045882f, -0.315137929f,
    0.949528181f, -0.313681740f,
    0.950008245f, -0.312224814f,
    0.950486074f, -0.310767153f,
    0.950961666f, -0.309308760f,
    0.951435021f, -0.307849640f,
    0.951906137f, -0.306389795f,
    0.952375013f, -0.304929230f,
    0.952841648f, -0.303467947f,
    0.953306040f, -0.302005949f,
    0.953768190f, -0.300543241f,
    0.954228095f, -0.299079826f,
    0.954685755f, -0.297615707f,
    0.955141168f, -0.296150888f,
    0.955594334f, -0.294685372f,
    0.956045251f, -0.293219163f,
    0.956493919f, -0.291752263f,
    0.956940336f, -0.290284677f,
    0.957384501f, -0.288816408f,
    0.957826413f, -0.287347460f,
    0.958266071f, -0.285877835f,
    0.958703475f, -0.284407537f,
    0.959138622f, -0.282936570f,
    0.959571513f, -0.281464938f,
    0.960002146f, -0.279992643f,
    0.960430519f, -0.278519689f,
    0.960856633f, -0.277046080f,
    0.961280486f, -0.275571819f,
    0.961702077f, -0.274096910f,
    0.962121404f, -0.272621355f,
    0.962538468f, -0.271145160f,
    0.962953267f, -0.269668326f,
    0.963365800f, -0.268190857f,
    0.963776066f, -0.266712757f,
    0.964184064f, -0.265234030f,
    0.964589793f, -0.263754679f,
    0.964993253f, -0.262274707f,
    0.965394442f, -0.260794118f,
    0.965793359f, -0.259312915f,
    0.966190003f, -0.257831102f,
    0.966584374f, -0.256348682f,
    0.966976471f, -0.254865660f,
    0.967366292f, -0.253382037f,
    0.967753837f, -0.251897818f,
    0.968139105f, -0.250413007f,
    0.968522094f, -0.248927606f,
    0.968902805f, -0.247441619f,
    0.969281235f, -0.245955050f,
    0.969657385f, -0.244467903f,
    0.970031253f, -0.242980180f,
    0.970402839f, -0.241491885f,
    0.970772141f, -0.240003022f,
    0.971139158f, -0.238513595f,
    0.971503891f, -0.237023606f,
    0.971866337f, -0.235533059f,
    0.972226497f, -0.234041959f,
    0.972584369f, -0.232550307f,
    0.972939952f, -0.231058108f,
    0.973293246f, -0.229565366f,
    0.973644250f, -0.228072083f,
    0.973992962f, -0.226578264f,
    0.974339383f, -0.225083911f,
    0.974683511f, -0.223589029f,
    0.975025345f, -0.222093621f,
    0.975364885f, -0.220597690f,
    0.975702130f, -0.219101240f,
    0.976037079f, -0.217604275f,
    0.976369731f, -0.216106797f,
    0.976700086f, -0.214608811f,
    0.977028143f, -0.213110320f,
    0.977353900f, -0.211611327f,
    0.977677358f, -0.210111837f,
    0.977998515f, -0.208611852f,
    0.978317371f, -0.207111376f,
    0.978633924f, -0.205610413f,
    0.978948175f, -0.204108966f,
    0.979260123f, -0.202607039f,
    0.979569766f, -0.201104635f,
    0.979877104f, -0.199601758f,
    0.980182136f, -0.198098411f,
    0.980484862f, -0.196594598f,
    0.980785280f, -0.195090322f,
    0.981083391f, -0.193585587f,
    0.981379193f, -0.192080397f,
    0.981672686f, -0.190574755f,
    0.981963869f, -0.189068664f,
    0.982252741f, -0.187562129f,
    0.982539302f, -0.186055152f,
    0.982823551f, -0.184547737f,
    0.983105487f, -0.183039888f,
    0.983385110f, -0.181531608f,
    0.983662419f, -0.180022901f,
    0.983937413f, -0.178513771f,
    0.984210092f, -0.177004220f,
    0.984480455f, -0.175494253f,
    0.984748502f, -0.173983873f,
    0.985014231f, -0.172473084f,
    0.985277642f, -0.170961889f,
    0.985538735f, -0.169450291f,
    0.985797509f, -0.167938295f,
    0.986053963f, -0.166425904f,
    0.986308097f, -0.164913120f,
    0.986559910f, -0.163399949f,
    0.986809402f, -0.161886394f,
    0.987056571f, -0.160372457f,
    0.987301418f, -0.158858143f,
    0.987543942f, -0.157343456f,
    0.987784142f, -0.155828398f,
    0.988022017f, -0.154312973f,
    0.988257568f, -0.152797185f,
    0.988490793f, -0.151281038f,
    0.988721692f, -0.149764535f,
    0.988950265f, -0.148247679f,
    0.989176510f, -0.146730474f,
    0.989400428f, -0.145212925f,
    0.989622017f, -0.143695033f,
    0.989841278f, -0.142176804f,
    0.990058210f, -0.140658239f,
    0.990272812f, -0.139139344f,
    0.990485084f, -0.137620122f,
    0.990695025f, -0.136100575f,
    0.990902635f, -0.134580709f,
    0.991107914f, -0.133060525f,
    0.991310860f, -0.131540029f,
    0.991511473f, -0.130019223f,
    0.991709754f, -0.128498111f,
    0.991905700f, -0.126976696f,
    0.992099313f, -0.125454983f,
    0.992290591f, -0.123932975f,
    0.992479535f, -0.122410675f,
    0.992666142f, -0.120888087f,
    0.992850414f, -0.119365215f,
    0.993032350f, -0.117842062f,
    0.993211949f, -0.116318631f,
    0.993389211f, -0.114794927f,
    0.993564136f, -0.113270952f,
    0.993736722f, -0.111746711f,
    0.993906970f, -0.110222207f,
    0.994074879f, -0.108697444f,
    0.994240449f, -0.107172425f,
    0.994403680f, -0.105647154f,
    0.994564571f, -0.104121634f,
    0.994723121f, -0.102595869f,
    0.994879331f, -0.101069863f,
    0.995033199f, -0.099543619f,
    0.995184727f, -0.098017140f,
    0.995333912f, -0.096490431f,
    0.995480755f, -0.094963495f,
    0.995625256f, -0.093436336f,
    0.995767414f, -0.091908956f,
    0.995907229f, -0.090381361f,
    0.996044701f, -0.088853553f,
    0.996179829f, -0.087325535f,
    0.996312612f, -0.085797312f,
    0.996443051f, -0.084268888f,
    0.996571146f, -0.082740265f,
    0.996696895f, -0.081211447f,
    0.996820299f, -0.079682438f,
    0.996941358f, -0.078153242f,
    0.997060070f, -0.076623861f,
    0.997176437f, -0.075094301f,
    0.997290457f, -0.073564564f,
    0.997402130f, -0.072034653f,
    0.997511456f, -0.070504573f,
    0.997618435f, -0.068974328f,
    0.997723067f, -0.067443920f,
    0.997825350f, -0.065913353f,
    0.997925286f, -0.064382631f,
    0.998022874f, -0.062851758f,
    0.998118113f, -0.061320736f,
    0.998211003f, -0.059789571f,
    0.998301545f, -0.058258265f,
    0.998389737f, -0.056726821f,
    0.998475581f, -0.055195244f,
    0.998559074f, -0.053663538f,
    0.998640218f, -0.052131705f,
    0.998719012f, -0.050599749f,
    0.998795456f, -0.049067674f,
    0.998869550f, -0.047535484f,
    0.998941293f, -0.046003182f,
    0.999010686f, -0.044470772f,
    0.999077728f, -0.042938257f,
    0.999142419f, -0.041405641f,
    0.999204759f, -0.039872928f,
    0.999264747f, -0.038340120f,
    0.999322385f, -0.036807223f,
    0.999377670f, -0.035274239f,
    0.999430605f, -0.033741172f,
    0.999481187f, -0.032208025f,
    0.999529418f, -0.030674803f,
    0.999575296f, -0.029141509f,
    0.999618822f, -0.027608146f,
    0.999659997f, -0.026074718f,
    0.999698819f, -0.024541229f,
    0.999735288f, -0.023007681f,
    0.999769405f, -0.021474080f,
    0.999801170f, -0.019940429f,
    0.999830582f, -0.018406730f,
    0.999857641f, -0.016872988f,
    0.999882347f, -0.015339206f,
    0.999904701f, -0.013805389f,
    0.999924702f, -0.012271538f,
    0.999942350f, -0.010737659f,
    0.999957645f, -0.009203755f,
    0.999970586f, -0.007669829f,
    0.999981175f, -0.006135885f,
    0.999989411f, -0.004601926f,
    0.999995294f, -0.003067957f,
    0.999998823f, -0.001533980f
};

/*
* @brief  Q31 Twiddle factors Table
*/


/**
* \par
* Example code for Q31 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefQ31[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefQ31[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 16	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to Q31(Fixed point 1.31):
*	round(twiddleCoefQ31(i) * pow(2, 31))
*
*/
const q31_t twiddleCoef_16_q31[24] = {
    (q31_t)0x7FFFFFFF, (q31_t)0x00000000,
    (q31_t)0x7641AF3C, (q31_t)0x30FBC54D,
    (q31_t)0x5A82799A, (q31_t)0x5A82799A,
    (q31_t)0x30FBC54D, (q31_t)0x7641AF3C,
    (q31_t)0x00000000, (q31_t)0x7FFFFFFF,
    (q31_t)0xCF043AB2, (q31_t)0x7641AF3C,
    (q31_t)0xA57D8666, (q31_t)0x5A82799A,
    (q31_t)0x89BE50C3, (q31_t)0x30FBC54D,
    (q31_t)0x80000000, (q31_t)0x00000000,
    (q31_t)0x89BE50C3, (q31_t)0xCF043AB2,
    (q31_t)0xA57D8666, (q31_t)0xA57D8666,
    (q31_t)0xCF043AB2, (q31_t)0x89BE50C3
};

/**
* \par
* Example code for Q31 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefQ31[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefQ31[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 32	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to Q31(Fixed point 1.31):
*	round(twiddleCoefQ31(i) * pow(2, 31))
*
*/
const q31_t twiddleCoef_32_q31[48] = {
    (q31_t)0x7FFFFFFF, (q31_t)0x00000000,
    (q31_t)0x7D8A5F3F, (q31_t)0x18F8B83C,
    (q31_t)0x7641AF3C, (q31_t)0x30FBC54D,
    (q31_t)0x6A6D98A4, (q31_t)0x471CECE6,
    (q31_t)0x5A82799A, (q31_t)0x5A82799A,
    (q31_t)0x471CECE6, (q31_t)0x6A6D98A4,
    (q31_t)0x30FBC54D, (q31_t)0x7641AF3C,
    (q31_t)0x18F8B83C, (q31_t)0x7D8A5F3F,
    (q31_t)0x00000000, (q31_t)0x7FFFFFFF,
    (q31_t)0xE70747C3, (q31_t)0x7D8A5F3F,
    (q31_t)0xCF043AB2, (q31_t)0x7641AF3C,
    (q31_t)0xB8E31319, (q31_t)0x6A6D98A4,
    (q31_t)0xA57D8666, (q31_t)0x5A82799A,
    (q31_t)0x9592675B, (q31_t)0x471CECE6,
    (q31_t)0x89BE50C3, (q31_t)0x30FBC54D,
    (q31_t)0x8275A0C0, (q31_t)0x18F8B83C,
    (q31_t)0x80000000, (q31_t)0x00000000,
    (q31_t)0x8275A0C0, (q31_t)0xE70747C3,
    (q31_t)0x89BE50C3, (q31_t)0xCF043AB2,
    (q31_t)0x9592675B, (q31_t)0xB8E31319,
    (q31_t)0xA57D8666, (q31_t)0xA57D8666,
    (q31_t)0xB8E31319, (q31_t)0x9592675B,
    (q31_t)0xCF043AB2, (q31_t)0x89BE50C3,
    (q31_t)0xE70747C3, (q31_t)0x8275A0C0
};

/**
* \par
* Example code for Q31 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefQ31[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefQ31[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 64	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to Q31(Fixed point 1.31):
*	round(twiddleCoefQ31(i) * pow(2, 31))
*
*/
const q31_t twiddleCoef_64_q31[96] = {
	(q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7F62368F,
	(q31_t)0x0C8BD35E, (q31_t)0x7D8A5F3F, (q31_t)0x18F8B83C,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x7641AF3C,
	(q31_t)0x30FBC54D, (q31_t)0x70E2CBC6, (q31_t)0x3C56BA70,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x62F201AC,
	(q31_t)0x5133CC94, (q31_t)0x5A82799A, (q31_t)0x5A82799A,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x471CECE6,
	(q31_t)0x6A6D98A4, (q31_t)0x3C56BA70, (q31_t)0x70E2CBC6,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x25280C5D,
	(q31_t)0x7A7D055B, (q31_t)0x18F8B83C, (q31_t)0x7D8A5F3F,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x00000000,
	(q31_t)0x7FFFFFFF, (q31_t)0xF3742CA1, (q31_t)0x7F62368F,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xDAD7F3A2,
	(q31_t)0x7A7D055B, (q31_t)0xCF043AB2, (q31_t)0x7641AF3C,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xB8E31319,
	(q31_t)0x6A6D98A4, (q31_t)0xAECC336B, (q31_t)0x62F201AC,
	(q31_t)0xA57D8666, (q31_t)0x5A82799A, (q31_t)0x9D0DFE53,
	(q31_t)0x5133CC94, (q31_t)0x9592675B, (q31_t)0x471CECE6,
	(q31_t)0x8F1D343A, (q31_t)0x3C56BA70, (q31_t)0x89BE50C3,
	(q31_t)0x30FBC54D, (q31_t)0x8582FAA4, (q31_t)0x25280C5D,
	(q31_t)0x8275A0C0, (q31_t)0x18F8B83C, (q31_t)0x809DC970,
	(q31_t)0x0C8BD35E, (q31_t)0x80000000, (q31_t)0x00000000,
	(q31_t)0x809DC970, (q31_t)0xF3742CA1, (q31_t)0x8275A0C0,
	(q31_t)0xE70747C3, (q31_t)0x8582FAA4, (q31_t)0xDAD7F3A2,
	(q31_t)0x89BE50C3, (q31_t)0xCF043AB2, (q31_t)0x8F1D343A,
	(q31_t)0xC3A9458F, (q31_t)0x9592675B, (q31_t)0xB8E31319,
	(q31_t)0x9D0DFE53, (q31_t)0xAECC336B, (q31_t)0xA57D8666,
	(q31_t)0xA57D8666, (q31_t)0xAECC336B, (q31_t)0x9D0DFE53,
	(q31_t)0xB8E31319, (q31_t)0x9592675B, (q31_t)0xC3A9458F,
	(q31_t)0x8F1D343A, (q31_t)0xCF043AB2, (q31_t)0x89BE50C3,
	(q31_t)0xDAD7F3A2, (q31_t)0x8582FAA4, (q31_t)0xE70747C3,
	(q31_t)0x8275A0C0, (q31_t)0xF3742CA1, (q31_t)0x809DC970
};

/**
* \par
* Example code for Q31 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefQ31[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefQ31[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 128	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to Q31(Fixed point 1.31):
*	round(twiddleCoefQ31(i) * pow(2, 31))
*
*/
const q31_t twiddleCoef_128_q31[192] = {
	(q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7FD8878D,
	(q31_t)0x0647D97C, (q31_t)0x7F62368F, (q31_t)0x0C8BD35E,
	(q31_t)0x7E9D55FC, (q31_t)0x12C8106E, (q31_t)0x7D8A5F3F,
	(q31_t)0x18F8B83C, (q31_t)0x7C29FBEE, (q31_t)0x1F19F97B,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x78848413,
	(q31_t)0x2B1F34EB, (q31_t)0x7641AF3C, (q31_t)0x30FBC54D,
	(q31_t)0x73B5EBD0, (q31_t)0x36BA2013, (q31_t)0x70E2CBC6,
	(q31_t)0x3C56BA70, (q31_t)0x6DCA0D14, (q31_t)0x41CE1E64,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x66CF811F,
	(q31_t)0x4C3FDFF3, (q31_t)0x62F201AC, (q31_t)0x5133CC94,
	(q31_t)0x5ED77C89, (q31_t)0x55F5A4D2, (q31_t)0x5A82799A,
	(q31_t)0x5A82799A, (q31_t)0x55F5A4D2, (q31_t)0x5ED77C89,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x4C3FDFF3,
	(q31_t)0x66CF811F, (q31_t)0x471CECE6, (q31_t)0x6A6D98A4,
	(q31_t)0x41CE1E64, (q31_t)0x6DCA0D14, (q31_t)0x3C56BA70,
	(q31_t)0x70E2CBC6, (q31_t)0x36BA2013, (q31_t)0x73B5EBD0,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x2B1F34EB,
	(q31_t)0x78848413, (q31_t)0x25280C5D, (q31_t)0x7A7D055B,
	(q31_t)0x1F19F97B, (q31_t)0x7C29FBEE, (q31_t)0x18F8B83C,
	(q31_t)0x7D8A5F3F, (q31_t)0x12C8106E, (q31_t)0x7E9D55FC,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x0647D97C,
	(q31_t)0x7FD8878D, (q31_t)0x00000000, (q31_t)0x7FFFFFFF,
	(q31_t)0xF9B82683, (q31_t)0x7FD8878D, (q31_t)0xF3742CA1,
	(q31_t)0x7F62368F, (q31_t)0xED37EF91, (q31_t)0x7E9D55FC,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xE0E60684,
	(q31_t)0x7C29FBEE, (q31_t)0xDAD7F3A2, (q31_t)0x7A7D055B,
	(q31_t)0xD4E0CB14, (q31_t)0x78848413, (q31_t)0xCF043AB2,
	(q31_t)0x7641AF3C, (q31_t)0xC945DFEC, (q31_t)0x73B5EBD0,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xBE31E19B,
	(q31_t)0x6DCA0D14, (q31_t)0xB8E31319, (q31_t)0x6A6D98A4,
	(q31_t)0xB3C0200C, (q31_t)0x66CF811F, (q31_t)0xAECC336B,
	(q31_t)0x62F201AC, (q31_t)0xAA0A5B2D, (q31_t)0x5ED77C89,
	(q31_t)0xA57D8666, (q31_t)0x5A82799A, (q31_t)0xA1288376,
	(q31_t)0x55F5A4D2, (q31_t)0x9D0DFE53, (q31_t)0x5133CC94,
	(q31_t)0x99307EE0, (q31_t)0x4C3FDFF3, (q31_t)0x9592675B,
	(q31_t)0x471CECE6, (q31_t)0x9235F2EB, (q31_t)0x41CE1E64,
	(q31_t)0x8F1D343A, (q31_t)0x3C56BA70, (q31_t)0x8C4A142F,
	(q31_t)0x36BA2013, (q31_t)0x89BE50C3, (q31_t)0x30FBC54D,
	(q31_t)0x877B7BEC, (q31_t)0x2B1F34EB, (q31_t)0x8582FAA4,
	(q31_t)0x25280C5D, (q31_t)0x83D60411, (q31_t)0x1F19F97B,
	(q31_t)0x8275A0C0, (q31_t)0x18F8B83C, (q31_t)0x8162AA03,
	(q31_t)0x12C8106E, (q31_t)0x809DC970, (q31_t)0x0C8BD35E,
	(q31_t)0x80277872, (q31_t)0x0647D97C, (q31_t)0x80000000,
	(q31_t)0x00000000, (q31_t)0x80277872, (q31_t)0xF9B82683,
	(q31_t)0x809DC970, (q31_t)0xF3742CA1, (q31_t)0x8162AA03,
	(q31_t)0xED37EF91, (q31_t)0x8275A0C0, (q31_t)0xE70747C3,
	(q31_t)0x83D60411, (q31_t)0xE0E60684, (q31_t)0x8582FAA4,
	(q31_t)0xDAD7F3A2, (q31_t)0x877B7BEC, (q31_t)0xD4E0CB14,
	(q31_t)0x89BE50C3, (q31_t)0xCF043AB2, (q31_t)0x8C4A142F,
	(q31_t)0xC945DFEC, (q31_t)0x8F1D343A, (q31_t)0xC3A9458F,
	(q31_t)0x9235F2EB, (q31_t)0xBE31E19B, (q31_t)0x9592675B,
	(q31_t)0xB8E31319, (q31_t)0x99307EE0, (q31_t)0xB3C0200C,
	(q31_t)0x9D0DFE53, (q31_t)0xAECC336B, (q31_t)0xA1288376,
	(q31_t)0xAA0A5B2D, (q31_t)0xA57D8666, (q31_t)0xA57D8666,
	(q31_t)0xAA0A5B2D, (q31_t)0xA1288376, (q31_t)0xAECC336B,
	(q31_t)0x9D0DFE53, (q31_t)0xB3C0200C, (q31_t)0x99307EE0,
	(q31_t)0xB8E31319, (q31_t)0x9592675B, (q31_t)0xBE31E19B,
	(q31_t)0x9235F2EB, (q31_t)0xC3A9458F, (q31_t)0x8F1D343A,
	(q31_t)0xC945DFEC, (q31_t)0x8C4A142F, (q31_t)0xCF043AB2,
	(q31_t)0x89BE50C3, (q31_t)0xD4E0CB14, (q31_t)0x877B7BEC,
	(q31_t)0xDAD7F3A2, (q31_t)0x8582FAA4, (q31_t)0xE0E60684,
	(q31_t)0x83D60411, (q31_t)0xE70747C3, (q31_t)0x8275A0C0,
	(q31_t)0xED37EF91, (q31_t)0x8162AA03, (q31_t)0xF3742CA1,
	(q31_t)0x809DC970, (q31_t)0xF9B82683, (q31_t)0x80277872
};

/**
* \par
* Example code for Q31 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefQ31[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefQ31[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 256	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to Q31(Fixed point 1.31):
*	round(twiddleCoefQ31(i) * pow(2, 31))
*
*/
const q31_t twiddleCoef_256_q31[384] = {
	(q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7FF62182,
	(q31_t)0x03242ABF, (q31_t)0x7FD8878D, (q31_t)0x0647D97C,
	(q31_t)0x7FA736B4, (q31_t)0x096A9049, (q31_t)0x7F62368F,
	(q31_t)0x0C8BD35E, (q31_t)0x7F0991C3, (q31_t)0x0FAB272B,
	(q31_t)0x7E9D55FC, (q31_t)0x12C8106E, (q31_t)0x7E1D93E9,
	(q31_t)0x15E21444, (q31_t)0x7D8A5F3F, (q31_t)0x18F8B83C,
	(q31_t)0x7CE3CEB1, (q31_t)0x1C0B826A, (q31_t)0x7C29FBEE,
	(q31_t)0x1F19F97B, (q31_t)0x7B5D039D, (q31_t)0x2223A4C5,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x798A23B1,
	(q31_t)0x2826B928, (q31_t)0x78848413, (q31_t)0x2B1F34EB,
	(q31_t)0x776C4EDB, (q31_t)0x2E110A62, (q31_t)0x7641AF3C,
	(q31_t)0x30FBC54D, (q31_t)0x7504D345, (q31_t)0x33DEF287,
	(q31_t)0x73B5EBD0, (q31_t)0x36BA2013, (q31_t)0x72552C84,
	(q31_t)0x398CDD32, (q31_t)0x70E2CBC6, (q31_t)0x3C56BA70,
	(q31_t)0x6F5F02B1, (q31_t)0x3F1749B7, (q31_t)0x6DCA0D14,
	(q31_t)0x41CE1E64, (q31_t)0x6C242960, (q31_t)0x447ACD50,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x68A69E81,
	(q31_t)0x49B41533, (q31_t)0x66CF811F, (q31_t)0x4C3FDFF3,
	(q31_t)0x64E88926, (q31_t)0x4EBFE8A4, (q31_t)0x62F201AC,
	(q31_t)0x5133CC94, (q31_t)0x60EC3830, (q31_t)0x539B2AEF,
	(q31_t)0x5ED77C89, (q31_t)0x55F5A4D2, (q31_t)0x5CB420DF,
	(q31_t)0x5842DD54, (q31_t)0x5A82799A, (q31_t)0x5A82799A,
	(q31_t)0x5842DD54, (q31_t)0x5CB420DF, (q31_t)0x55F5A4D2,
	(q31_t)0x5ED77C89, (q31_t)0x539B2AEF, (q31_t)0x60EC3830,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x4EBFE8A4,
	(q31_t)0x64E88926, (q31_t)0x4C3FDFF3, (q31_t)0x66CF811F,
	(q31_t)0x49B41533, (q31_t)0x68A69E81, (q31_t)0x471CECE6,
	(q31_t)0x6A6D98A4, (q31_t)0x447ACD50, (q31_t)0x6C242960,
	(q31_t)0x41CE1E64, (q31_t)0x6DCA0D14, (q31_t)0x3F1749B7,
	(q31_t)0x6F5F02B1, (q31_t)0x3C56BA70, (q31_t)0x70E2CBC6,
	(q31_t)0x398CDD32, (q31_t)0x72552C84, (q31_t)0x36BA2013,
	(q31_t)0x73B5EBD0, (q31_t)0x33DEF287, (q31_t)0x7504D345,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x2E110A62,
	(q31_t)0x776C4EDB, (q31_t)0x2B1F34EB, (q31_t)0x78848413,
	(q31_t)0x2826B928, (q31_t)0x798A23B1, (q31_t)0x25280C5D,
	(q31_t)0x7A7D055B, (q31_t)0x2223A4C5, (q31_t)0x7B5D039D,
	(q31_t)0x1F19F97B, (q31_t)0x7C29FBEE, (q31_t)0x1C0B826A,
	(q31_t)0x7CE3CEB1, (q31_t)0x18F8B83C, (q31_t)0x7D8A5F3F,
	(q31_t)0x15E21444, (q31_t)0x7E1D93E9, (q31_t)0x12C8106E,
	(q31_t)0x7E9D55FC, (q31_t)0x0FAB272B, (q31_t)0x7F0991C3,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x096A9049,
	(q31_t)0x7FA736B4, (q31_t)0x0647D97C, (q31_t)0x7FD8878D,
	(q31_t)0x03242ABF, (q31_t)0x7FF62182, (q31_t)0x00000000,
	(q31_t)0x7FFFFFFF, (q31_t)0xFCDBD541, (q31_t)0x7FF62182,
	(q31_t)0xF9B82683, (q31_t)0x7FD8878D, (q31_t)0xF6956FB6,
	(q31_t)0x7FA736B4, (q31_t)0xF3742CA1, (q31_t)0x7F62368F,
	(q31_t)0xF054D8D4, (q31_t)0x7F0991C3, (q31_t)0xED37EF91,
	(q31_t)0x7E9D55FC, (q31_t)0xEA1DEBBB, (q31_t)0x7E1D93E9,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xE3F47D95,
	(q31_t)0x7CE3CEB1, (q31_t)0xE0E60684, (q31_t)0x7C29FBEE,
	(q31_t)0xDDDC5B3A, (q31_t)0x7B5D039D, (q31_t)0xDAD7F3A2,
	(q31_t)0x7A7D055B, (q31_t)0xD7D946D7, (q31_t)0x798A23B1,
	(q31_t)0xD4E0CB14, (q31_t)0x78848413, (q31_t)0xD1EEF59E,
	(q31_t)0x776C4EDB, (q31_t)0xCF043AB2, (q31_t)0x7641AF3C,
	(q31_t)0xCC210D78, (q31_t)0x7504D345, (q31_t)0xC945DFEC,
	(q31_t)0x73B5EBD0, (q31_t)0xC67322CD, (q31_t)0x72552C84,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xC0E8B648,
	(q31_t)0x6F5F02B1, (q31_t)0xBE31E19B, (q31_t)0x6DCA0D14,
	(q31_t)0xBB8532AF, (q31_t)0x6C242960, (q31_t)0xB8E31319,
	(q31_t)0x6A6D98A4, (q31_t)0xB64BEACC, (q31_t)0x68A69E81,
	(q31_t)0xB3C0200C, (q31_t)0x66CF811F, (q31_t)0xB140175B,
	(q31_t)0x64E88926, (q31_t)0xAECC336B, (q31_t)0x62F201AC,
	(q31_t)0xAC64D510, (q31_t)0x60EC3830, (q31_t)0xAA0A5B2D,
	(q31_t)0x5ED77C89, (q31_t)0xA7BD22AB, (q31_t)0x5CB420DF,
	(q31_t)0xA57D8666, (q31_t)0x5A82799A, (q31_t)0xA34BDF20,
	(q31_t)0x5842DD54, (q31_t)0xA1288376, (q31_t)0x55F5A4D2,
	(q31_t)0x9F13C7D0, (q31_t)0x539B2AEF, (q31_t)0x9D0DFE53,
	(q31_t)0x5133CC94, (q31_t)0x9B1776D9, (q31_t)0x4EBFE8A4,
	(q31_t)0x99307EE0, (q31_t)0x4C3FDFF3, (q31_t)0x9759617E,
	(q31_t)0x49B41533, (q31_t)0x9592675B, (q31_t)0x471CECE6,
	(q31_t)0x93DBD69F, (q31_t)0x447ACD50, (q31_t)0x9235F2EB,
	(q31_t)0x41CE1E64, (q31_t)0x90A0FD4E, (q31_t)0x3F1749B7,
	(q31_t)0x8F1D343A, (q31_t)0x3C56BA70, (q31_t)0x8DAAD37B,
	(q31_t)0x398CDD32, (q31_t)0x8C4A142F, (q31_t)0x36BA2013,
	(q31_t)0x8AFB2CBA, (q31_t)0x33DEF287, (q31_t)0x89BE50C3,
	(q31_t)0x30FBC54D, (q31_t)0x8893B124, (q31_t)0x2E110A62,
	(q31_t)0x877B7BEC, (q31_t)0x2B1F34EB, (q31_t)0x8675DC4E,
	(q31_t)0x2826B928, (q31_t)0x8582FAA4, (q31_t)0x25280C5D,
	(q31_t)0x84A2FC62, (q31_t)0x2223A4C5, (q31_t)0x83D60411,
	(q31_t)0x1F19F97B, (q31_t)0x831C314E, (q31_t)0x1C0B826A,
	(q31_t)0x8275A0C0, (q31_t)0x18F8B83C, (q31_t)0x81E26C16,
	(q31_t)0x15E21444, (q31_t)0x8162AA03, (q31_t)0x12C8106E,
	(q31_t)0x80F66E3C, (q31_t)0x0FAB272B, (q31_t)0x809DC970,
	(q31_t)0x0C8BD35E, (q31_t)0x8058C94C, (q31_t)0x096A9049,
	(q31_t)0x80277872, (q31_t)0x0647D97C, (q31_t)0x8009DE7D,
	(q31_t)0x03242ABF, (q31_t)0x80000000, (q31_t)0x00000000,
	(q31_t)0x8009DE7D, (q31_t)0xFCDBD541, (q31_t)0x80277872,
	(q31_t)0xF9B82683, (q31_t)0x8058C94C, (q31_t)0xF6956FB6,
	(q31_t)0x809DC970, (q31_t)0xF3742CA1, (q31_t)0x80F66E3C,
	(q31_t)0xF054D8D4, (q31_t)0x8162AA03, (q31_t)0xED37EF91,
	(q31_t)0x81E26C16, (q31_t)0xEA1DEBBB, (q31_t)0x8275A0C0,
	(q31_t)0xE70747C3, (q31_t)0x831C314E, (q31_t)0xE3F47D95,
	(q31_t)0x83D60411, (q31_t)0xE0E60684, (q31_t)0x84A2FC62,
	(q31_t)0xDDDC5B3A, (q31_t)0x8582FAA4, (q31_t)0xDAD7F3A2,
	(q31_t)0x8675DC4E, (q31_t)0xD7D946D7, (q31_t)0x877B7BEC,
	(q31_t)0xD4E0CB14, (q31_t)0x8893B124, (q31_t)0xD1EEF59E,
	(q31_t)0x89BE50C3, (q31_t)0xCF043AB2, (q31_t)0x8AFB2CBA,
	(q31_t)0xCC210D78, (q31_t)0x8C4A142F, (q31_t)0xC945DFEC,
	(q31_t)0x8DAAD37B, (q31_t)0xC67322CD, (q31_t)0x8F1D343A,
	(q31_t)0xC3A9458F, (q31_t)0x90A0FD4E, (q31_t)0xC0E8B648,
	(q31_t)0x9235F2EB, (q31_t)0xBE31E19B, (q31_t)0x93DBD69F,
	(q31_t)0xBB8532AF, (q31_t)0x9592675B, (q31_t)0xB8E31319,
	(q31_t)0x9759617E, (q31_t)0xB64BEACC, (q31_t)0x99307EE0,
	(q31_t)0xB3C0200C, (q31_t)0x9B1776D9, (q31_t)0xB140175B,
	(q31_t)0x9D0DFE53, (q31_t)0xAECC336B, (q31_t)0x9F13C7D0,
	(q31_t)0xAC64D510, (q31_t)0xA1288376, (q31_t)0xAA0A5B2D,
	(q31_t)0xA34BDF20, (q31_t)0xA7BD22AB, (q31_t)0xA57D8666,
	(q31_t)0xA57D8666, (q31_t)0xA7BD22AB, (q31_t)0xA34BDF20,
	(q31_t)0xAA0A5B2D, (q31_t)0xA1288376, (q31_t)0xAC64D510,
	(q31_t)0x9F13C7D0, (q31_t)0xAECC336B, (q31_t)0x9D0DFE53,
	(q31_t)0xB140175B, (q31_t)0x9B1776D9, (q31_t)0xB3C0200C,
	(q31_t)0x99307EE0, (q31_t)0xB64BEACC, (q31_t)0x9759617E,
	(q31_t)0xB8E31319, (q31_t)0x9592675B, (q31_t)0xBB8532AF,
	(q31_t)0x93DBD69F, (q31_t)0xBE31E19B, (q31_t)0x9235F2EB,
	(q31_t)0xC0E8B648, (q31_t)0x90A0FD4E, (q31_t)0xC3A9458F,
	(q31_t)0x8F1D343A, (q31_t)0xC67322CD, (q31_t)0x8DAAD37B,
	(q31_t)0xC945DFEC, (q31_t)0x8C4A142F, (q31_t)0xCC210D78,
	(q31_t)0x8AFB2CBA, (q31_t)0xCF043AB2, (q31_t)0x89BE50C3,
	(q31_t)0xD1EEF59E, (q31_t)0x8893B124, (q31_t)0xD4E0CB14,
	(q31_t)0x877B7BEC, (q31_t)0xD7D946D7, (q31_t)0x8675DC4E,
	(q31_t)0xDAD7F3A2, (q31_t)0x8582FAA4, (q31_t)0xDDDC5B3A,
	(q31_t)0x84A2FC62, (q31_t)0xE0E60684, (q31_t)0x83D60411,
	(q31_t)0xE3F47D95, (q31_t)0x831C314E, (q31_t)0xE70747C3,
	(q31_t)0x8275A0C0, (q31_t)0xEA1DEBBB, (q31_t)0x81E26C16,
	(q31_t)0xED37EF91, (q31_t)0x8162AA03, (q31_t)0xF054D8D4,
	(q31_t)0x80F66E3C, (q31_t)0xF3742CA1, (q31_t)0x809DC970,
	(q31_t)0xF6956FB6, (q31_t)0x8058C94C, (q31_t)0xF9B82683,
	(q31_t)0x80277872, (q31_t)0xFCDBD541, (q31_t)0x8009DE7D
};

/**
* \par
* Example code for Q31 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefQ31[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefQ31[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 512	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to Q31(Fixed point 1.31):
*	round(twiddleCoefQ31(i) * pow(2, 31))
*
*/
const q31_t twiddleCoef_512_q31[768] = {
    (q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7FFD885A,
	(q31_t)0x01921D1F, (q31_t)0x7FF62182, (q31_t)0x03242ABF,
	(q31_t)0x7FE9CBC0, (q31_t)0x04B6195D, (q31_t)0x7FD8878D,
	(q31_t)0x0647D97C, (q31_t)0x7FC25596, (q31_t)0x07D95B9E,
	(q31_t)0x7FA736B4, (q31_t)0x096A9049, (q31_t)0x7F872BF3,
	(q31_t)0x0AFB6805, (q31_t)0x7F62368F, (q31_t)0x0C8BD35E,
	(q31_t)0x7F3857F5, (q31_t)0x0E1BC2E3, (q31_t)0x7F0991C3,
	(q31_t)0x0FAB272B, (q31_t)0x7ED5E5C6, (q31_t)0x1139F0CE,
	(q31_t)0x7E9D55FC, (q31_t)0x12C8106E, (q31_t)0x7E5FE493,
	(q31_t)0x145576B1, (q31_t)0x7E1D93E9, (q31_t)0x15E21444,
	(q31_t)0x7DD6668E, (q31_t)0x176DD9DE, (q31_t)0x7D8A5F3F,
	(q31_t)0x18F8B83C, (q31_t)0x7D3980EC, (q31_t)0x1A82A025,
	(q31_t)0x7CE3CEB1, (q31_t)0x1C0B826A, (q31_t)0x7C894BDD,
	(q31_t)0x1D934FE5, (q31_t)0x7C29FBEE, (q31_t)0x1F19F97B,
	(q31_t)0x7BC5E28F, (q31_t)0x209F701C, (q31_t)0x7B5D039D,
	(q31_t)0x2223A4C5, (q31_t)0x7AEF6323, (q31_t)0x23A6887E,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x7A05EEAD,
	(q31_t)0x26A82185, (q31_t)0x798A23B1, (q31_t)0x2826B928,
	(q31_t)0x7909A92C, (q31_t)0x29A3C484, (q31_t)0x78848413,
	(q31_t)0x2B1F34EB, (q31_t)0x77FAB988, (q31_t)0x2C98FBBA,
	(q31_t)0x776C4EDB, (q31_t)0x2E110A62, (q31_t)0x76D94988,
	(q31_t)0x2F875262, (q31_t)0x7641AF3C, (q31_t)0x30FBC54D,
	(q31_t)0x75A585CF, (q31_t)0x326E54C7, (q31_t)0x7504D345,
	(q31_t)0x33DEF287, (q31_t)0x745F9DD1, (q31_t)0x354D9056,
	(q31_t)0x73B5EBD0, (q31_t)0x36BA2013, (q31_t)0x7307C3D0,
	(q31_t)0x382493B0, (q31_t)0x72552C84, (q31_t)0x398CDD32,
	(q31_t)0x719E2CD2, (q31_t)0x3AF2EEB7, (q31_t)0x70E2CBC6,
	(q31_t)0x3C56BA70, (q31_t)0x70231099, (q31_t)0x3DB832A5,
	(q31_t)0x6F5F02B1, (q31_t)0x3F1749B7, (q31_t)0x6E96A99C,
	(q31_t)0x4073F21D, (q31_t)0x6DCA0D14, (q31_t)0x41CE1E64,
	(q31_t)0x6CF934FB, (q31_t)0x4325C135, (q31_t)0x6C242960,
	(q31_t)0x447ACD50, (q31_t)0x6B4AF278, (q31_t)0x45CD358F,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x698C246C,
	(q31_t)0x4869E664, (q31_t)0x68A69E81, (q31_t)0x49B41533,
	(q31_t)0x67BD0FBC, (q31_t)0x4AFB6C97, (q31_t)0x66CF811F,
	(q31_t)0x4C3FDFF3, (q31_t)0x65DDFBD3, (q31_t)0x4D8162C4,
	(q31_t)0x64E88926, (q31_t)0x4EBFE8A4, (q31_t)0x63EF328F,
	(q31_t)0x4FFB654D, (q31_t)0x62F201AC, (q31_t)0x5133CC94,
	(q31_t)0x61F1003E, (q31_t)0x5269126E, (q31_t)0x60EC3830,
	(q31_t)0x539B2AEF, (q31_t)0x5FE3B38D, (q31_t)0x54CA0A4A,
	(q31_t)0x5ED77C89, (q31_t)0x55F5A4D2, (q31_t)0x5DC79D7C,
	(q31_t)0x571DEEF9, (q31_t)0x5CB420DF, (q31_t)0x5842DD54,
	(q31_t)0x5B9D1153, (q31_t)0x59646497, (q31_t)0x5A82799A,
	(q31_t)0x5A82799A, (q31_t)0x59646497, (q31_t)0x5B9D1153,
	(q31_t)0x5842DD54, (q31_t)0x5CB420DF, (q31_t)0x571DEEF9,
	(q31_t)0x5DC79D7C, (q31_t)0x55F5A4D2, (q31_t)0x5ED77C89,
	(q31_t)0x54CA0A4A, (q31_t)0x5FE3B38D, (q31_t)0x539B2AEF,
	(q31_t)0x60EC3830, (q31_t)0x5269126E, (q31_t)0x61F1003E,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x4FFB654D,
	(q31_t)0x63EF328F, (q31_t)0x4EBFE8A4, (q31_t)0x64E88926,
	(q31_t)0x4D8162C4, (q31_t)0x65DDFBD3, (q31_t)0x4C3FDFF3,
	(q31_t)0x66CF811F, (q31_t)0x4AFB6C97, (q31_t)0x67BD0FBC,
	(q31_t)0x49B41533, (q31_t)0x68A69E81, (q31_t)0x4869E664,
	(q31_t)0x698C246C, (q31_t)0x471CECE6, (q31_t)0x6A6D98A4,
	(q31_t)0x45CD358F, (q31_t)0x6B4AF278, (q31_t)0x447ACD50,
	(q31_t)0x6C242960, (q31_t)0x4325C135, (q31_t)0x6CF934FB,
	(q31_t)0x41CE1E64, (q31_t)0x6DCA0D14, (q31_t)0x4073F21D,
	(q31_t)0x6E96A99C, (q31_t)0x3F1749B7, (q31_t)0x6F5F02B1,
	(q31_t)0x3DB832A5, (q31_t)0x70231099, (q31_t)0x3C56BA70,
	(q31_t)0x70E2CBC6, (q31_t)0x3AF2EEB7, (q31_t)0x719E2CD2,
	(q31_t)0x398CDD32, (q31_t)0x72552C84, (q31_t)0x382493B0,
	(q31_t)0x7307C3D0, (q31_t)0x36BA2013, (q31_t)0x73B5EBD0,
	(q31_t)0x354D9056, (q31_t)0x745F9DD1, (q31_t)0x33DEF287,
	(q31_t)0x7504D345, (q31_t)0x326E54C7, (q31_t)0x75A585CF,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x2F875262,
	(q31_t)0x76D94988, (q31_t)0x2E110A62, (q31_t)0x776C4EDB,
	(q31_t)0x2C98FBBA, (q31_t)0x77FAB988, (q31_t)0x2B1F34EB,
	(q31_t)0x78848413, (q31_t)0x29A3C484, (q31_t)0x7909A92C,
	(q31_t)0x2826B928, (q31_t)0x798A23B1, (q31_t)0x26A82185,
	(q31_t)0x7A05EEAD, (q31_t)0x25280C5D, (q31_t)0x7A7D055B,
	(q31_t)0x23A6887E, (q31_t)0x7AEF6323, (q31_t)0x2223A4C5,
	(q31_t)0x7B5D039D, (q31_t)0x209F701C, (q31_t)0x7BC5E28F,
	(q31_t)0x1F19F97B, (q31_t)0x7C29FBEE, (q31_t)0x1D934FE5,
	(q31_t)0x7C894BDD, (q31_t)0x1C0B826A, (q31_t)0x7CE3CEB1,
	(q31_t)0x1A82A025, (q31_t)0x7D3980EC, (q31_t)0x18F8B83C,
	(q31_t)0x7D8A5F3F, (q31_t)0x176DD9DE, (q31_t)0x7DD6668E,
	(q31_t)0x15E21444, (q31_t)0x7E1D93E9, (q31_t)0x145576B1,
	(q31_t)0x7E5FE493, (q31_t)0x12C8106E, (q31_t)0x7E9D55FC,
	(q31_t)0x1139F0CE, (q31_t)0x7ED5E5C6, (q31_t)0x0FAB272B,
	(q31_t)0x7F0991C3, (q31_t)0x0E1BC2E3, (q31_t)0x7F3857F5,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x0AFB6805,
	(q31_t)0x7F872BF3, (q31_t)0x096A9049, (q31_t)0x7FA736B4,
	(q31_t)0x07D95B9E, (q31_t)0x7FC25596, (q31_t)0x0647D97C,
	(q31_t)0x7FD8878D, (q31_t)0x04B6195D, (q31_t)0x7FE9CBC0,
	(q31_t)0x03242ABF, (q31_t)0x7FF62182, (q31_t)0x01921D1F,
	(q31_t)0x7FFD885A, (q31_t)0x00000000, (q31_t)0x7FFFFFFF,
	(q31_t)0xFE6DE2E0, (q31_t)0x7FFD885A, (q31_t)0xFCDBD541,
	(q31_t)0x7FF62182, (q31_t)0xFB49E6A2, (q31_t)0x7FE9CBC0,
	(q31_t)0xF9B82683, (q31_t)0x7FD8878D, (q31_t)0xF826A461,
	(q31_t)0x7FC25596, (q31_t)0xF6956FB6, (q31_t)0x7FA736B4,
	(q31_t)0xF50497FA, (q31_t)0x7F872BF3, (q31_t)0xF3742CA1,
	(q31_t)0x7F62368F, (q31_t)0xF1E43D1C, (q31_t)0x7F3857F5,
	(q31_t)0xF054D8D4, (q31_t)0x7F0991C3, (q31_t)0xEEC60F31,
	(q31_t)0x7ED5E5C6, (q31_t)0xED37EF91, (q31_t)0x7E9D55FC,
	(q31_t)0xEBAA894E, (q31_t)0x7E5FE493, (q31_t)0xEA1DEBBB,
	(q31_t)0x7E1D93E9, (q31_t)0xE8922621, (q31_t)0x7DD6668E,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xE57D5FDA,
	(q31_t)0x7D3980EC, (q31_t)0xE3F47D95, (q31_t)0x7CE3CEB1,
	(q31_t)0xE26CB01A, (q31_t)0x7C894BDD, (q31_t)0xE0E60684,
	(q31_t)0x7C29FBEE, (q31_t)0xDF608FE3, (q31_t)0x7BC5E28F,
	(q31_t)0xDDDC5B3A, (q31_t)0x7B5D039D, (q31_t)0xDC597781,
	(q31_t)0x7AEF6323, (q31_t)0xDAD7F3A2, (q31_t)0x7A7D055B,
	(q31_t)0xD957DE7A, (q31_t)0x7A05EEAD, (q31_t)0xD7D946D7,
	(q31_t)0x798A23B1, (q31_t)0xD65C3B7B, (q31_t)0x7909A92C,
	(q31_t)0xD4E0CB14, (q31_t)0x78848413, (q31_t)0xD3670445,
	(q31_t)0x77FAB988, (q31_t)0xD1EEF59E, (q31_t)0x776C4EDB,
	(q31_t)0xD078AD9D, (q31_t)0x76D94988, (q31_t)0xCF043AB2,
	(q31_t)0x7641AF3C, (q31_t)0xCD91AB38, (q31_t)0x75A585CF,
	(q31_t)0xCC210D78, (q31_t)0x7504D345, (q31_t)0xCAB26FA9,
	(q31_t)0x745F9DD1, (q31_t)0xC945DFEC, (q31_t)0x73B5EBD0,
	(q31_t)0xC7DB6C50, (q31_t)0x7307C3D0, (q31_t)0xC67322CD,
	(q31_t)0x72552C84, (q31_t)0xC50D1148, (q31_t)0x719E2CD2,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xC247CD5A,
	(q31_t)0x70231099, (q31_t)0xC0E8B648, (q31_t)0x6F5F02B1,
	(q31_t)0xBF8C0DE2, (q31_t)0x6E96A99C, (q31_t)0xBE31E19B,
	(q31_t)0x6DCA0D14, (q31_t)0xBCDA3ECA, (q31_t)0x6CF934FB,
	(q31_t)0xBB8532AF, (q31_t)0x6C242960, (q31_t)0xBA32CA70,
	(q31_t)0x6B4AF278, (q31_t)0xB8E31319, (q31_t)0x6A6D98A4,
	(q31_t)0xB796199B, (q31_t)0x698C246C, (q31_t)0xB64BEACC,
	(q31_t)0x68A69E81, (q31_t)0xB5049368, (q31_t)0x67BD0FBC,
	(q31_t)0xB3C0200C, (q31_t)0x66CF811F, (q31_t)0xB27E9D3B,
	(q31_t)0x65DDFBD3, (q31_t)0xB140175B, (q31_t)0x64E88926,
	(q31_t)0xB0049AB2, (q31_t)0x63EF328F, (q31_t)0xAECC336B,
	(q31_t)0x62F201AC, (q31_t)0xAD96ED91, (q31_t)0x61F1003E,
	(q31_t)0xAC64D510, (q31_t)0x60EC3830, (q31_t)0xAB35F5B5,
	(q31_t)0x5FE3B38D, (q31_t)0xAA0A5B2D, (q31_t)0x5ED77C89,
	(q31_t)0xA8E21106, (q31_t)0x5DC79D7C, (q31_t)0xA7BD22AB,
	(q31_t)0x5CB420DF, (q31_t)0xA69B9B68, (q31_t)0x5B9D1153,
	(q31_t)0xA57D8666, (q31_t)0x5A82799A, (q31_t)0xA462EEAC,
	(q31_t)0x59646497, (q31_t)0xA34BDF20, (q31_t)0x5842DD54,
	(q31_t)0xA2386283, (q31_t)0x571DEEF9, (q31_t)0xA1288376,
	(q31_t)0x55F5A4D2, (q31_t)0xA01C4C72, (q31_t)0x54CA0A4A,
	(q31_t)0x9F13C7D0, (q31_t)0x539B2AEF, (q31_t)0x9E0EFFC1,
	(q31_t)0x5269126E, (q31_t)0x9D0DFE53, (q31_t)0x5133CC94,
	(q31_t)0x9C10CD70, (q31_t)0x4FFB654D, (q31_t)0x9B1776D9,
	(q31_t)0x4EBFE8A4, (q31_t)0x9A22042C, (q31_t)0x4D8162C4,
	(q31_t)0x99307EE0, (q31_t)0x4C3FDFF3, (q31_t)0x9842F043,
	(q31_t)0x4AFB6C97, (q31_t)0x9759617E, (q31_t)0x49B41533,
	(q31_t)0x9673DB94, (q31_t)0x4869E664, (q31_t)0x9592675B,
	(q31_t)0x471CECE6, (q31_t)0x94B50D87, (q31_t)0x45CD358F,
	(q31_t)0x93DBD69F, (q31_t)0x447ACD50, (q31_t)0x9306CB04,
	(q31_t)0x4325C135, (q31_t)0x9235F2EB, (q31_t)0x41CE1E64,
	(q31_t)0x91695663, (q31_t)0x4073F21D, (q31_t)0x90A0FD4E,
	(q31_t)0x3F1749B7, (q31_t)0x8FDCEF66, (q31_t)0x3DB832A5,
	(q31_t)0x8F1D343A, (q31_t)0x3C56BA70, (q31_t)0x8E61D32D,
	(q31_t)0x3AF2EEB7, (q31_t)0x8DAAD37B, (q31_t)0x398CDD32,
	(q31_t)0x8CF83C30, (q31_t)0x382493B0, (q31_t)0x8C4A142F,
	(q31_t)0x36BA2013, (q31_t)0x8BA0622F, (q31_t)0x354D9056,
	(q31_t)0x8AFB2CBA, (q31_t)0x33DEF287, (q31_t)0x8A5A7A30,
	(q31_t)0x326E54C7, (q31_t)0x89BE50C3, (q31_t)0x30FBC54D,
	(q31_t)0x8926B677, (q31_t)0x2F875262, (q31_t)0x8893B124,
	(q31_t)0x2E110A62, (q31_t)0x88054677, (q31_t)0x2C98FBBA,
	(q31_t)0x877B7BEC, (q31_t)0x2B1F34EB, (q31_t)0x86F656D3,
	(q31_t)0x29A3C484, (q31_t)0x8675DC4E, (q31_t)0x2826B928,
	(q31_t)0x85FA1152, (q31_t)0x26A82185, (q31_t)0x8582FAA4,
	(q31_t)0x25280C5D, (q31_t)0x85109CDC, (q31_t)0x23A6887E,
	(q31_t)0x84A2FC62, (q31_t)0x2223A4C5, (q31_t)0x843A1D70,
	(q31_t)0x209F701C, (q31_t)0x83D60411, (q31_t)0x1F19F97B,
	(q31_t)0x8376B422, (q31_t)0x1D934FE5, (q31_t)0x831C314E,
	(q31_t)0x1C0B826A, (q31_t)0x82C67F13, (q31_t)0x1A82A025,
	(q31_t)0x8275A0C0, (q31_t)0x18F8B83C, (q31_t)0x82299971,
	(q31_t)0x176DD9DE, (q31_t)0x81E26C16, (q31_t)0x15E21444,
	(q31_t)0x81A01B6C, (q31_t)0x145576B1, (q31_t)0x8162AA03,
	(q31_t)0x12C8106E, (q31_t)0x812A1A39, (q31_t)0x1139F0CE,
	(q31_t)0x80F66E3C, (q31_t)0x0FAB272B, (q31_t)0x80C7A80A,
	(q31_t)0x0E1BC2E3, (q31_t)0x809DC970, (q31_t)0x0C8BD35E,
	(q31_t)0x8078D40D, (q31_t)0x0AFB6805, (q31_t)0x8058C94C,
	(q31_t)0x096A9049, (q31_t)0x803DAA69, (q31_t)0x07D95B9E,
	(q31_t)0x80277872, (q31_t)0x0647D97C, (q31_t)0x80163440,
	(q31_t)0x04B6195D, (q31_t)0x8009DE7D, (q31_t)0x03242ABF,
	(q31_t)0x800277A5, (q31_t)0x01921D1F, (q31_t)0x80000000,
	(q31_t)0x00000000, (q31_t)0x800277A5, (q31_t)0xFE6DE2E0,
	(q31_t)0x8009DE7D, (q31_t)0xFCDBD541, (q31_t)0x80163440,
	(q31_t)0xFB49E6A2, (q31_t)0x80277872, (q31_t)0xF9B82683,
	(q31_t)0x803DAA69, (q31_t)0xF826A461, (q31_t)0x8058C94C,
	(q31_t)0xF6956FB6, (q31_t)0x8078D40D, (q31_t)0xF50497FA,
	(q31_t)0x809DC970, (q31_t)0xF3742CA1, (q31_t)0x80C7A80A,
	(q31_t)0xF1E43D1C, (q31_t)0x80F66E3C, (q31_t)0xF054D8D4,
	(q31_t)0x812A1A39, (q31_t)0xEEC60F31, (q31_t)0x8162AA03,
	(q31_t)0xED37EF91, (q31_t)0x81A01B6C, (q31_t)0xEBAA894E,
	(q31_t)0x81E26C16, (q31_t)0xEA1DEBBB, (q31_t)0x82299971,
	(q31_t)0xE8922621, (q31_t)0x8275A0C0, (q31_t)0xE70747C3,
	(q31_t)0x82C67F13, (q31_t)0xE57D5FDA, (q31_t)0x831C314E,
	(q31_t)0xE3F47D95, (q31_t)0x8376B422, (q31_t)0xE26CB01A,
	(q31_t)0x83D60411, (q31_t)0xE0E60684, (q31_t)0x843A1D70,
	(q31_t)0xDF608FE3, (q31_t)0x84A2FC62, (q31_t)0xDDDC5B3A,
	(q31_t)0x85109CDC, (q31_t)0xDC597781, (q31_t)0x8582FAA4,
	(q31_t)0xDAD7F3A2, (q31_t)0x85FA1152, (q31_t)0xD957DE7A,
	(q31_t)0x8675DC4E, (q31_t)0xD7D946D7, (q31_t)0x86F656D3,
	(q31_t)0xD65C3B7B, (q31_t)0x877B7BEC, (q31_t)0xD4E0CB14,
	(q31_t)0x88054677, (q31_t)0xD3670445, (q31_t)0x8893B124,
	(q31_t)0xD1EEF59E, (q31_t)0x8926B677, (q31_t)0xD078AD9D,
	(q31_t)0x89BE50C3, (q31_t)0xCF043AB2, (q31_t)0x8A5A7A30,
	(q31_t)0xCD91AB38, (q31_t)0x8AFB2CBA, (q31_t)0xCC210D78,
	(q31_t)0x8BA0622F, (q31_t)0xCAB26FA9, (q31_t)0x8C4A142F,
	(q31_t)0xC945DFEC, (q31_t)0x8CF83C30, (q31_t)0xC7DB6C50,
	(q31_t)0x8DAAD37B, (q31_t)0xC67322CD, (q31_t)0x8E61D32D,
	(q31_t)0xC50D1148, (q31_t)0x8F1D343A, (q31_t)0xC3A9458F,
	(q31_t)0x8FDCEF66, (q31_t)0xC247CD5A, (q31_t)0x90A0FD4E,
	(q31_t)0xC0E8B648, (q31_t)0x91695663, (q31_t)0xBF8C0DE2,
	(q31_t)0x9235F2EB, (q31_t)0xBE31E19B, (q31_t)0x9306CB04,
	(q31_t)0xBCDA3ECA, (q31_t)0x93DBD69F, (q31_t)0xBB8532AF,
	(q31_t)0x94B50D87, (q31_t)0xBA32CA70, (q31_t)0x9592675B,
	(q31_t)0xB8E31319, (q31_t)0x9673DB94, (q31_t)0xB796199B,
	(q31_t)0x9759617E, (q31_t)0xB64BEACC, (q31_t)0x9842F043,
	(q31_t)0xB5049368, (q31_t)0x99307EE0, (q31_t)0xB3C0200C,
	(q31_t)0x9A22042C, (q31_t)0xB27E9D3B, (q31_t)0x9B1776D9,
	(q31_t)0xB140175B, (q31_t)0x9C10CD70, (q31_t)0xB0049AB2,
	(q31_t)0x9D0DFE53, (q31_t)0xAECC336B, (q31_t)0x9E0EFFC1,
	(q31_t)0xAD96ED91, (q31_t)0x9F13C7D0, (q31_t)0xAC64D510,
	(q31_t)0xA01C4C72, (q31_t)0xAB35F5B5, (q31_t)0xA1288376,
	(q31_t)0xAA0A5B2D, (q31_t)0xA2386283, (q31_t)0xA8E21106,
	(q31_t)0xA34BDF20, (q31_t)0xA7BD22AB, (q31_t)0xA462EEAC,
	(q31_t)0xA69B9B68, (q31_t)0xA57D8666, (q31_t)0xA57D8666,
	(q31_t)0xA69B9B68, (q31_t)0xA462EEAC, (q31_t)0xA7BD22AB,
	(q31_t)0xA34BDF20, (q31_t)0xA8E21106, (q31_t)0xA2386283,
	(q31_t)0xAA0A5B2D, (q31_t)0xA1288376, (q31_t)0xAB35F5B5,
	(q31_t)0xA01C4C72, (q31_t)0xAC64D510, (q31_t)0x9F13C7D0,
	(q31_t)0xAD96ED91, (q31_t)0x9E0EFFC1, (q31_t)0xAECC336B,
	(q31_t)0x9D0DFE53, (q31_t)0xB0049AB2, (q31_t)0x9C10CD70,
	(q31_t)0xB140175B, (q31_t)0x9B1776D9, (q31_t)0xB27E9D3B,
	(q31_t)0x9A22042C, (q31_t)0xB3C0200C, (q31_t)0x99307EE0,
	(q31_t)0xB5049368, (q31_t)0x9842F043, (q31_t)0xB64BEACC,
	(q31_t)0x9759617E, (q31_t)0xB796199B, (q31_t)0x9673DB94,
	(q31_t)0xB8E31319, (q31_t)0x9592675B, (q31_t)0xBA32CA70,
	(q31_t)0x94B50D87, (q31_t)0xBB8532AF, (q31_t)0x93DBD69F,
	(q31_t)0xBCDA3ECA, (q31_t)0x9306CB04, (q31_t)0xBE31E19B,
	(q31_t)0x9235F2EB, (q31_t)0xBF8C0DE2, (q31_t)0x91695663,
	(q31_t)0xC0E8B648, (q31_t)0x90A0FD4E, (q31_t)0xC247CD5A,
	(q31_t)0x8FDCEF66, (q31_t)0xC3A9458F, (q31_t)0x8F1D343A,
	(q31_t)0xC50D1148, (q31_t)0x8E61D32D, (q31_t)0xC67322CD,
	(q31_t)0x8DAAD37B, (q31_t)0xC7DB6C50, (q31_t)0x8CF83C30,
	(q31_t)0xC945DFEC, (q31_t)0x8C4A142F, (q31_t)0xCAB26FA9,
	(q31_t)0x8BA0622F, (q31_t)0xCC210D78, (q31_t)0x8AFB2CBA,
	(q31_t)0xCD91AB38, (q31_t)0x8A5A7A30, (q31_t)0xCF043AB2,
	(q31_t)0x89BE50C3, (q31_t)0xD078AD9D, (q31_t)0x8926B677,
	(q31_t)0xD1EEF59E, (q31_t)0x8893B124, (q31_t)0xD3670445,
	(q31_t)0x88054677, (q31_t)0xD4E0CB14, (q31_t)0x877B7BEC,
	(q31_t)0xD65C3B7B, (q31_t)0x86F656D3, (q31_t)0xD7D946D7,
	(q31_t)0x8675DC4E, (q31_t)0xD957DE7A, (q31_t)0x85FA1152,
	(q31_t)0xDAD7F3A2, (q31_t)0x8582FAA4, (q31_t)0xDC597781,
	(q31_t)0x85109CDC, (q31_t)0xDDDC5B3A, (q31_t)0x84A2FC62,
	(q31_t)0xDF608FE3, (q31_t)0x843A1D70, (q31_t)0xE0E60684,
	(q31_t)0x83D60411, (q31_t)0xE26CB01A, (q31_t)0x8376B422,
	(q31_t)0xE3F47D95, (q31_t)0x831C314E, (q31_t)0xE57D5FDA,
	(q31_t)0x82C67F13, (q31_t)0xE70747C3, (q31_t)0x8275A0C0,
	(q31_t)0xE8922621, (q31_t)0x82299971, (q31_t)0xEA1DEBBB,
	(q31_t)0x81E26C16, (q31_t)0xEBAA894E, (q31_t)0x81A01B6C,
	(q31_t)0xED37EF91, (q31_t)0x8162AA03, (q31_t)0xEEC60F31,
	(q31_t)0x812A1A39, (q31_t)0xF054D8D4, (q31_t)0x80F66E3C,
	(q31_t)0xF1E43D1C, (q31_t)0x80C7A80A, (q31_t)0xF3742CA1,
	(q31_t)0x809DC970, (q31_t)0xF50497FA, (q31_t)0x8078D40D,
	(q31_t)0xF6956FB6, (q31_t)0x8058C94C, (q31_t)0xF826A461,
	(q31_t)0x803DAA69, (q31_t)0xF9B82683, (q31_t)0x80277872,
	(q31_t)0xFB49E6A2, (q31_t)0x80163440, (q31_t)0xFCDBD541,
	(q31_t)0x8009DE7D, (q31_t)0xFE6DE2E0, (q31_t)0x800277A5
};

/**
* \par
* Example code for Q31 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefQ31[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefQ31[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 1024	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to Q31(Fixed point 1.31):
*	round(twiddleCoefQ31(i) * pow(2, 31))
*
*/
const q31_t twiddleCoef_1024_q31[1536] = {
	(q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7FFF6216,
	(q31_t)0x00C90F88, (q31_t)0x7FFD885A, (q31_t)0x01921D1F,
	(q31_t)0x7FFA72D1, (q31_t)0x025B26D7, (q31_t)0x7FF62182,
	(q31_t)0x03242ABF, (q31_t)0x7FF09477, (q31_t)0x03ED26E6,
	(q31_t)0x7FE9CBC0, (q31_t)0x04B6195D, (q31_t)0x7FE1C76B,
	(q31_t)0x057F0034, (q31_t)0x7FD8878D, (q31_t)0x0647D97C,
	(q31_t)0x7FCE0C3E, (q31_t)0x0710A344, (q31_t)0x7FC25596,
	(q31_t)0x07D95B9E, (q31_t)0x7FB563B2, (q31_t)0x08A2009A,
	(q31_t)0x7FA736B4, (q31_t)0x096A9049, (q31_t)0x7F97CEBC,
	(q31_t)0x0A3308BC, (q31_t)0x7F872BF3, (q31_t)0x0AFB6805,
	(q31_t)0x7F754E7F, (q31_t)0x0BC3AC35, (q31_t)0x7F62368F,
	(q31_t)0x0C8BD35E, (q31_t)0x7F4DE450, (q31_t)0x0D53DB92,
	(q31_t)0x7F3857F5, (q31_t)0x0E1BC2E3, (q31_t)0x7F2191B4,
	(q31_t)0x0EE38765, (q31_t)0x7F0991C3, (q31_t)0x0FAB272B,
	(q31_t)0x7EF0585F, (q31_t)0x1072A047, (q31_t)0x7ED5E5C6,
	(q31_t)0x1139F0CE, (q31_t)0x7EBA3A39, (q31_t)0x120116D4,
	(q31_t)0x7E9D55FC, (q31_t)0x12C8106E, (q31_t)0x7E7F3956,
	(q31_t)0x138EDBB0, (q31_t)0x7E5FE493, (q31_t)0x145576B1,
	(q31_t)0x7E3F57FE, (q31_t)0x151BDF85, (q31_t)0x7E1D93E9,
	(q31_t)0x15E21444, (q31_t)0x7DFA98A7, (q31_t)0x16A81305,
	(q31_t)0x7DD6668E, (q31_t)0x176DD9DE, (q31_t)0x7DB0FDF7,
	(q31_t)0x183366E8, (q31_t)0x7D8A5F3F, (q31_t)0x18F8B83C,
	(q31_t)0x7D628AC5, (q31_t)0x19BDCBF2, (q31_t)0x7D3980EC,
	(q31_t)0x1A82A025, (q31_t)0x7D0F4218, (q31_t)0x1B4732EF,
	(q31_t)0x7CE3CEB1, (q31_t)0x1C0B826A, (q31_t)0x7CB72724,
	(q31_t)0x1CCF8CB3, (q31_t)0x7C894BDD, (q31_t)0x1D934FE5,
	(q31_t)0x7C5A3D4F, (q31_t)0x1E56CA1E, (q31_t)0x7C29FBEE,
	(q31_t)0x1F19F97B, (q31_t)0x7BF88830, (q31_t)0x1FDCDC1A,
	(q31_t)0x7BC5E28F, (q31_t)0x209F701C, (q31_t)0x7B920B89,
	(q31_t)0x2161B39F, (q31_t)0x7B5D039D, (q31_t)0x2223A4C5,
	(q31_t)0x7B26CB4F, (q31_t)0x22E541AE, (q31_t)0x7AEF6323,
	(q31_t)0x23A6887E, (q31_t)0x7AB6CBA3, (q31_t)0x24677757,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x7A4210D8,
	(q31_t)0x25E845B5, (q31_t)0x7A05EEAD, (q31_t)0x26A82185,
	(q31_t)0x79C89F6D, (q31_t)0x27679DF4, (q31_t)0x798A23B1,
	(q31_t)0x2826B928, (q31_t)0x794A7C11, (q31_t)0x28E5714A,
	(q31_t)0x7909A92C, (q31_t)0x29A3C484, (q31_t)0x78C7ABA1,
	(q31_t)0x2A61B101, (q31_t)0x78848413, (q31_t)0x2B1F34EB,
	(q31_t)0x78403328, (q31_t)0x2BDC4E6F, (q31_t)0x77FAB988,
	(q31_t)0x2C98FBBA, (q31_t)0x77B417DF, (q31_t)0x2D553AFB,
	(q31_t)0x776C4EDB, (q31_t)0x2E110A62, (q31_t)0x77235F2D,
	(q31_t)0x2ECC681E, (q31_t)0x76D94988, (q31_t)0x2F875262,
	(q31_t)0x768E0EA5, (q31_t)0x3041C760, (q31_t)0x7641AF3C,
	(q31_t)0x30FBC54D, (q31_t)0x75F42C0A, (q31_t)0x31B54A5D,
	(q31_t)0x75A585CF, (q31_t)0x326E54C7, (q31_t)0x7555BD4B,
	(q31_t)0x3326E2C2, (q31_t)0x7504D345, (q31_t)0x33DEF287,
	(q31_t)0x74B2C883, (q31_t)0x3496824F, (q31_t)0x745F9DD1,
	(q31_t)0x354D9056, (q31_t)0x740B53FA, (q31_t)0x36041AD9,
	(q31_t)0x73B5EBD0, (q31_t)0x36BA2013, (q31_t)0x735F6626,
	(q31_t)0x376F9E46, (q31_t)0x7307C3D0, (q31_t)0x382493B0,
	(q31_t)0x72AF05A6, (q31_t)0x38D8FE93, (q31_t)0x72552C84,
	(q31_t)0x398CDD32, (q31_t)0x71FA3948, (q31_t)0x3A402DD1,
	(q31_t)0x719E2CD2, (q31_t)0x3AF2EEB7, (q31_t)0x71410804,
	(q31_t)0x3BA51E29, (q31_t)0x70E2CBC6, (q31_t)0x3C56BA70,
	(q31_t)0x708378FE, (q31_t)0x3D07C1D5, (q31_t)0x70231099,
	(q31_t)0x3DB832A5, (q31_t)0x6FC19385, (q31_t)0x3E680B2C,
	(q31_t)0x6F5F02B1, (q31_t)0x3F1749B7, (q31_t)0x6EFB5F12,
	(q31_t)0x3FC5EC97, (q31_t)0x6E96A99C, (q31_t)0x4073F21D,
	(q31_t)0x6E30E349, (q31_t)0x4121589A, (q31_t)0x6DCA0D14,
	(q31_t)0x41CE1E64, (q31_t)0x6D6227FA, (q31_t)0x427A41D0,
	(q31_t)0x6CF934FB, (q31_t)0x4325C135, (q31_t)0x6C8F351C,
	(q31_t)0x43D09AEC, (q31_t)0x6C242960, (q31_t)0x447ACD50,
	(q31_t)0x6BB812D0, (q31_t)0x452456BC, (q31_t)0x6B4AF278,
	(q31_t)0x45CD358F, (q31_t)0x6ADCC964, (q31_t)0x46756827,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x69FD614A,
	(q31_t)0x47C3C22E, (q31_t)0x698C246C, (q31_t)0x4869E664,
	(q31_t)0x6919E320, (q31_t)0x490F57EE, (q31_t)0x68A69E81,
	(q31_t)0x49B41533, (q31_t)0x683257AA, (q31_t)0x4A581C9D,
	(q31_t)0x67BD0FBC, (q31_t)0x4AFB6C97, (q31_t)0x6746C7D7,
	(q31_t)0x4B9E038F, (q31_t)0x66CF811F, (q31_t)0x4C3FDFF3,
	(q31_t)0x66573CBB, (q31_t)0x4CE10034, (q31_t)0x65DDFBD3,
	(q31_t)0x4D8162C4, (q31_t)0x6563BF92, (q31_t)0x4E210617,
	(q31_t)0x64E88926, (q31_t)0x4EBFE8A4, (q31_t)0x646C59BF,
	(q31_t)0x4F5E08E3, (q31_t)0x63EF328F, (q31_t)0x4FFB654D,
	(q31_t)0x637114CC, (q31_t)0x5097FC5E, (q31_t)0x62F201AC,
	(q31_t)0x5133CC94, (q31_t)0x6271FA69, (q31_t)0x51CED46E,
	(q31_t)0x61F1003E, (q31_t)0x5269126E, (q31_t)0x616F146B,
	(q31_t)0x53028517, (q31_t)0x60EC3830, (q31_t)0x539B2AEF,
	(q31_t)0x60686CCE, (q31_t)0x5433027D, (q31_t)0x5FE3B38D,
	(q31_t)0x54CA0A4A, (q31_t)0x5F5E0DB3, (q31_t)0x556040E2,
	(q31_t)0x5ED77C89, (q31_t)0x55F5A4D2, (q31_t)0x5E50015D,
	(q31_t)0x568A34A9, (q31_t)0x5DC79D7C, (q31_t)0x571DEEF9,
	(q31_t)0x5D3E5236, (q31_t)0x57B0D256, (q31_t)0x5CB420DF,
	(q31_t)0x5842DD54, (q31_t)0x5C290ACC, (q31_t)0x58D40E8C,
	(q31_t)0x5B9D1153, (q31_t)0x59646497, (q31_t)0x5B1035CF,
	(q31_t)0x59F3DE12, (q31_t)0x5A82799A, (q31_t)0x5A82799A,
	(q31_t)0x59F3DE12, (q31_t)0x5B1035CF, (q31_t)0x59646497,
	(q31_t)0x5B9D1153, (q31_t)0x58D40E8C, (q31_t)0x5C290ACC,
	(q31_t)0x5842DD54, (q31_t)0x5CB420DF, (q31_t)0x57B0D256,
	(q31_t)0x5D3E5236, (q31_t)0x571DEEF9, (q31_t)0x5DC79D7C,
	(q31_t)0x568A34A9, (q31_t)0x5E50015D, (q31_t)0x55F5A4D2,
	(q31_t)0x5ED77C89, (q31_t)0x556040E2, (q31_t)0x5F5E0DB3,
	(q31_t)0x54CA0A4A, (q31_t)0x5FE3B38D, (q31_t)0x5433027D,
	(q31_t)0x60686CCE, (q31_t)0x539B2AEF, (q31_t)0x60EC3830,
	(q31_t)0x53028517, (q31_t)0x616F146B, (q31_t)0x5269126E,
	(q31_t)0x61F1003E, (q31_t)0x51CED46E, (q31_t)0x6271FA69,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x5097FC5E,
	(q31_t)0x637114CC, (q31_t)0x4FFB654D, (q31_t)0x63EF328F,
	(q31_t)0x4F5E08E3, (q31_t)0x646C59BF, (q31_t)0x4EBFE8A4,
	(q31_t)0x64E88926, (q31_t)0x4E210617, (q31_t)0x6563BF92,
	(q31_t)0x4D8162C4, (q31_t)0x65DDFBD3, (q31_t)0x4CE10034,
	(q31_t)0x66573CBB, (q31_t)0x4C3FDFF3, (q31_t)0x66CF811F,
	(q31_t)0x4B9E038F, (q31_t)0x6746C7D7, (q31_t)0x4AFB6C97,
	(q31_t)0x67BD0FBC, (q31_t)0x4A581C9D, (q31_t)0x683257AA,
	(q31_t)0x49B41533, (q31_t)0x68A69E81, (q31_t)0x490F57EE,
	(q31_t)0x6919E320, (q31_t)0x4869E664, (q31_t)0x698C246C,
	(q31_t)0x47C3C22E, (q31_t)0x69FD614A, (q31_t)0x471CECE6,
	(q31_t)0x6A6D98A4, (q31_t)0x46756827, (q31_t)0x6ADCC964,
	(q31_t)0x45CD358F, (q31_t)0x6B4AF278, (q31_t)0x452456BC,
	(q31_t)0x6BB812D0, (q31_t)0x447ACD50, (q31_t)0x6C242960,
	(q31_t)0x43D09AEC, (q31_t)0x6C8F351C, (q31_t)0x4325C135,
	(q31_t)0x6CF934FB, (q31_t)0x427A41D0, (q31_t)0x6D6227FA,
	(q31_t)0x41CE1E64, (q31_t)0x6DCA0D14, (q31_t)0x4121589A,
	(q31_t)0x6E30E349, (q31_t)0x4073F21D, (q31_t)0x6E96A99C,
	(q31_t)0x3FC5EC97, (q31_t)0x6EFB5F12, (q31_t)0x3F1749B7,
	(q31_t)0x6F5F02B1, (q31_t)0x3E680B2C, (q31_t)0x6FC19385,
	(q31_t)0x3DB832A5, (q31_t)0x70231099, (q31_t)0x3D07C1D5,
	(q31_t)0x708378FE, (q31_t)0x3C56BA70, (q31_t)0x70E2CBC6,
	(q31_t)0x3BA51E29, (q31_t)0x71410804, (q31_t)0x3AF2EEB7,
	(q31_t)0x719E2CD2, (q31_t)0x3A402DD1, (q31_t)0x71FA3948,
	(q31_t)0x398CDD32, (q31_t)0x72552C84, (q31_t)0x38D8FE93,
	(q31_t)0x72AF05A6, (q31_t)0x382493B0, (q31_t)0x7307C3D0,
	(q31_t)0x376F9E46, (q31_t)0x735F6626, (q31_t)0x36BA2013,
	(q31_t)0x73B5EBD0, (q31_t)0x36041AD9, (q31_t)0x740B53FA,
	(q31_t)0x354D9056, (q31_t)0x745F9DD1, (q31_t)0x3496824F,
	(q31_t)0x74B2C883, (q31_t)0x33DEF287, (q31_t)0x7504D345,
	(q31_t)0x3326E2C2, (q31_t)0x7555BD4B, (q31_t)0x326E54C7,
	(q31_t)0x75A585CF, (q31_t)0x31B54A5D, (q31_t)0x75F42C0A,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x3041C760,
	(q31_t)0x768E0EA5, (q31_t)0x2F875262, (q31_t)0x76D94988,
	(q31_t)0x2ECC681E, (q31_t)0x77235F2D, (q31_t)0x2E110A62,
	(q31_t)0x776C4EDB, (q31_t)0x2D553AFB, (q31_t)0x77B417DF,
	(q31_t)0x2C98FBBA, (q31_t)0x77FAB988, (q31_t)0x2BDC4E6F,
	(q31_t)0x78403328, (q31_t)0x2B1F34EB, (q31_t)0x78848413,
	(q31_t)0x2A61B101, (q31_t)0x78C7ABA1, (q31_t)0x29A3C484,
	(q31_t)0x7909A92C, (q31_t)0x28E5714A, (q31_t)0x794A7C11,
	(q31_t)0x2826B928, (q31_t)0x798A23B1, (q31_t)0x27679DF4,
	(q31_t)0x79C89F6D, (q31_t)0x26A82185, (q31_t)0x7A05EEAD,
	(q31_t)0x25E845B5, (q31_t)0x7A4210D8, (q31_t)0x25280C5D,
	(q31_t)0x7A7D055B, (q31_t)0x24677757, (q31_t)0x7AB6CBA3,
	(q31_t)0x23A6887E, (q31_t)0x7AEF6323, (q31_t)0x22E541AE,
	(q31_t)0x7B26CB4F, (q31_t)0x2223A4C5, (q31_t)0x7B5D039D,
	(q31_t)0x2161B39F, (q31_t)0x7B920B89, (q31_t)0x209F701C,
	(q31_t)0x7BC5E28F, (q31_t)0x1FDCDC1A, (q31_t)0x7BF88830,
	(q31_t)0x1F19F97B, (q31_t)0x7C29FBEE, (q31_t)0x1E56CA1E,
	(q31_t)0x7C5A3D4F, (q31_t)0x1D934FE5, (q31_t)0x7C894BDD,
	(q31_t)0x1CCF8CB3, (q31_t)0x7CB72724, (q31_t)0x1C0B826A,
	(q31_t)0x7CE3CEB1, (q31_t)0x1B4732EF, (q31_t)0x7D0F4218,
	(q31_t)0x1A82A025, (q31_t)0x7D3980EC, (q31_t)0x19BDCBF2,
	(q31_t)0x7D628AC5, (q31_t)0x18F8B83C, (q31_t)0x7D8A5F3F,
	(q31_t)0x183366E8, (q31_t)0x7DB0FDF7, (q31_t)0x176DD9DE,
	(q31_t)0x7DD6668E, (q31_t)0x16A81305, (q31_t)0x7DFA98A7,
	(q31_t)0x15E21444, (q31_t)0x7E1D93E9, (q31_t)0x151BDF85,
	(q31_t)0x7E3F57FE, (q31_t)0x145576B1, (q31_t)0x7E5FE493,
	(q31_t)0x138EDBB0, (q31_t)0x7E7F3956, (q31_t)0x12C8106E,
	(q31_t)0x7E9D55FC, (q31_t)0x120116D4, (q31_t)0x7EBA3A39,
	(q31_t)0x1139F0CE, (q31_t)0x7ED5E5C6, (q31_t)0x1072A047,
	(q31_t)0x7EF0585F, (q31_t)0x0FAB272B, (q31_t)0x7F0991C3,
	(q31_t)0x0EE38765, (q31_t)0x7F2191B4, (q31_t)0x0E1BC2E3,
	(q31_t)0x7F3857F5, (q31_t)0x0D53DB92, (q31_t)0x7F4DE450,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x0BC3AC35,
	(q31_t)0x7F754E7F, (q31_t)0x0AFB6805, (q31_t)0x7F872BF3,
	(q31_t)0x0A3308BC, (q31_t)0x7F97CEBC, (q31_t)0x096A9049,
	(q31_t)0x7FA736B4, (q31_t)0x08A2009A, (q31_t)0x7FB563B2,
	(q31_t)0x07D95B9E, (q31_t)0x7FC25596, (q31_t)0x0710A344,
	(q31_t)0x7FCE0C3E, (q31_t)0x0647D97C, (q31_t)0x7FD8878D,
	(q31_t)0x057F0034, (q31_t)0x7FE1C76B, (q31_t)0x04B6195D,
	(q31_t)0x7FE9CBC0, (q31_t)0x03ED26E6, (q31_t)0x7FF09477,
	(q31_t)0x03242ABF, (q31_t)0x7FF62182, (q31_t)0x025B26D7,
	(q31_t)0x7FFA72D1, (q31_t)0x01921D1F, (q31_t)0x7FFD885A,
	(q31_t)0x00C90F88, (q31_t)0x7FFF6216, (q31_t)0x00000000,
	(q31_t)0x7FFFFFFF, (q31_t)0xFF36F078, (q31_t)0x7FFF6216,
	(q31_t)0xFE6DE2E0, (q31_t)0x7FFD885A, (q31_t)0xFDA4D928,
	(q31_t)0x7FFA72D1, (q31_t)0xFCDBD541, (q31_t)0x7FF62182,
	(q31_t)0xFC12D919, (q31_t)0x7FF09477, (q31_t)0xFB49E6A2,
	(q31_t)0x7FE9CBC0, (q31_t)0xFA80FFCB, (q31_t)0x7FE1C76B,
	(q31_t)0xF9B82683, (q31_t)0x7FD8878D, (q31_t)0xF8EF5CBB,
	(q31_t)0x7FCE0C3E, (q31_t)0xF826A461, (q31_t)0x7FC25596,
	(q31_t)0xF75DFF65, (q31_t)0x7FB563B2, (q31_t)0xF6956FB6,
	(q31_t)0x7FA736B4, (q31_t)0xF5CCF743, (q31_t)0x7F97CEBC,
	(q31_t)0xF50497FA, (q31_t)0x7F872BF3, (q31_t)0xF43C53CA,
	(q31_t)0x7F754E7F, (q31_t)0xF3742CA1, (q31_t)0x7F62368F,
	(q31_t)0xF2AC246D, (q31_t)0x7F4DE450, (q31_t)0xF1E43D1C,
	(q31_t)0x7F3857F5, (q31_t)0xF11C789A, (q31_t)0x7F2191B4,
	(q31_t)0xF054D8D4, (q31_t)0x7F0991C3, (q31_t)0xEF8D5FB8,
	(q31_t)0x7EF0585F, (q31_t)0xEEC60F31, (q31_t)0x7ED5E5C6,
	(q31_t)0xEDFEE92B, (q31_t)0x7EBA3A39, (q31_t)0xED37EF91,
	(q31_t)0x7E9D55FC, (q31_t)0xEC71244F, (q31_t)0x7E7F3956,
	(q31_t)0xEBAA894E, (q31_t)0x7E5FE493, (q31_t)0xEAE4207A,
	(q31_t)0x7E3F57FE, (q31_t)0xEA1DEBBB, (q31_t)0x7E1D93E9,
	(q31_t)0xE957ECFB, (q31_t)0x7DFA98A7, (q31_t)0xE8922621,
	(q31_t)0x7DD6668E, (q31_t)0xE7CC9917, (q31_t)0x7DB0FDF7,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xE642340D,
	(q31_t)0x7D628AC5, (q31_t)0xE57D5FDA, (q31_t)0x7D3980EC,
	(q31_t)0xE4B8CD10, (q31_t)0x7D0F4218, (q31_t)0xE3F47D95,
	(q31_t)0x7CE3CEB1, (q31_t)0xE330734C, (q31_t)0x7CB72724,
	(q31_t)0xE26CB01A, (q31_t)0x7C894BDD, (q31_t)0xE1A935E1,
	(q31_t)0x7C5A3D4F, (q31_t)0xE0E60684, (q31_t)0x7C29FBEE,
	(q31_t)0xE02323E5, (q31_t)0x7BF88830, (q31_t)0xDF608FE3,
	(q31_t)0x7BC5E28F, (q31_t)0xDE9E4C60, (q31_t)0x7B920B89,
	(q31_t)0xDDDC5B3A, (q31_t)0x7B5D039D, (q31_t)0xDD1ABE51,
	(q31_t)0x7B26CB4F, (q31_t)0xDC597781, (q31_t)0x7AEF6323,
	(q31_t)0xDB9888A8, (q31_t)0x7AB6CBA3, (q31_t)0xDAD7F3A2,
	(q31_t)0x7A7D055B, (q31_t)0xDA17BA4A, (q31_t)0x7A4210D8,
	(q31_t)0xD957DE7A, (q31_t)0x7A05EEAD, (q31_t)0xD898620C,
	(q31_t)0x79C89F6D, (q31_t)0xD7D946D7, (q31_t)0x798A23B1,
	(q31_t)0xD71A8EB5, (q31_t)0x794A7C11, (q31_t)0xD65C3B7B,
	(q31_t)0x7909A92C, (q31_t)0xD59E4EFE, (q31_t)0x78C7ABA1,
	(q31_t)0xD4E0CB14, (q31_t)0x78848413, (q31_t)0xD423B190,
	(q31_t)0x78403328, (q31_t)0xD3670445, (q31_t)0x77FAB988,
	(q31_t)0xD2AAC504, (q31_t)0x77B417DF, (q31_t)0xD1EEF59E,
	(q31_t)0x776C4EDB, (q31_t)0xD13397E1, (q31_t)0x77235F2D,
	(q31_t)0xD078AD9D, (q31_t)0x76D94988, (q31_t)0xCFBE389F,
	(q31_t)0x768E0EA5, (q31_t)0xCF043AB2, (q31_t)0x7641AF3C,
	(q31_t)0xCE4AB5A2, (q31_t)0x75F42C0A, (q31_t)0xCD91AB38,
	(q31_t)0x75A585CF, (q31_t)0xCCD91D3D, (q31_t)0x7555BD4B,
	(q31_t)0xCC210D78, (q31_t)0x7504D345, (q31_t)0xCB697DB0,
	(q31_t)0x74B2C883, (q31_t)0xCAB26FA9, (q31_t)0x745F9DD1,
	(q31_t)0xC9FBE527, (q31_t)0x740B53FA, (q31_t)0xC945DFEC,
	(q31_t)0x73B5EBD0, (q31_t)0xC89061BA, (q31_t)0x735F6626,
	(q31_t)0xC7DB6C50, (q31_t)0x7307C3D0, (q31_t)0xC727016C,
	(q31_t)0x72AF05A6, (q31_t)0xC67322CD, (q31_t)0x72552C84,
	(q31_t)0xC5BFD22E, (q31_t)0x71FA3948, (q31_t)0xC50D1148,
	(q31_t)0x719E2CD2, (q31_t)0xC45AE1D7, (q31_t)0x71410804,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xC2F83E2A,
	(q31_t)0x708378FE, (q31_t)0xC247CD5A, (q31_t)0x70231099,
	(q31_t)0xC197F4D3, (q31_t)0x6FC19385, (q31_t)0xC0E8B648,
	(q31_t)0x6F5F02B1, (q31_t)0xC03A1368, (q31_t)0x6EFB5F12,
	(q31_t)0xBF8C0DE2, (q31_t)0x6E96A99C, (q31_t)0xBEDEA765,
	(q31_t)0x6E30E349, (q31_t)0xBE31E19B, (q31_t)0x6DCA0D14,
	(q31_t)0xBD85BE2F, (q31_t)0x6D6227FA, (q31_t)0xBCDA3ECA,
	(q31_t)0x6CF934FB, (q31_t)0xBC2F6513, (q31_t)0x6C8F351C,
	(q31_t)0xBB8532AF, (q31_t)0x6C242960, (q31_t)0xBADBA943,
	(q31_t)0x6BB812D0, (q31_t)0xBA32CA70, (q31_t)0x6B4AF278,
	(q31_t)0xB98A97D8, (q31_t)0x6ADCC964, (q31_t)0xB8E31319,
	(q31_t)0x6A6D98A4, (q31_t)0xB83C3DD1, (q31_t)0x69FD614A,
	(q31_t)0xB796199B, (q31_t)0x698C246C, (q31_t)0xB6F0A811,
	(q31_t)0x6919E320, (q31_t)0xB64BEACC, (q31_t)0x68A69E81,
	(q31_t)0xB5A7E362, (q31_t)0x683257AA, (q31_t)0xB5049368,
	(q31_t)0x67BD0FBC, (q31_t)0xB461FC70, (q31_t)0x6746C7D7,
	(q31_t)0xB3C0200C, (q31_t)0x66CF811F, (q31_t)0xB31EFFCB,
	(q31_t)0x66573CBB, (q31_t)0xB27E9D3B, (q31_t)0x65DDFBD3,
	(q31_t)0xB1DEF9E8, (q31_t)0x6563BF92, (q31_t)0xB140175B,
	(q31_t)0x64E88926, (q31_t)0xB0A1F71C, (q31_t)0x646C59BF,
	(q31_t)0xB0049AB2, (q31_t)0x63EF328F, (q31_t)0xAF6803A1,
	(q31_t)0x637114CC, (q31_t)0xAECC336B, (q31_t)0x62F201AC,
	(q31_t)0xAE312B91, (q31_t)0x6271FA69, (q31_t)0xAD96ED91,
	(q31_t)0x61F1003E, (q31_t)0xACFD7AE8, (q31_t)0x616F146B,
	(q31_t)0xAC64D510, (q31_t)0x60EC3830, (q31_t)0xABCCFD82,
	(q31_t)0x60686CCE, (q31_t)0xAB35F5B5, (q31_t)0x5FE3B38D,
	(q31_t)0xAA9FBF1D, (q31_t)0x5F5E0DB3, (q31_t)0xAA0A5B2D,
	(q31_t)0x5ED77C89, (q31_t)0xA975CB56, (q31_t)0x5E50015D,
	(q31_t)0xA8E21106, (q31_t)0x5DC79D7C, (q31_t)0xA84F2DA9,
	(q31_t)0x5D3E5236, (q31_t)0xA7BD22AB, (q31_t)0x5CB420DF,
	(q31_t)0xA72BF173, (q31_t)0x5C290ACC, (q31_t)0xA69B9B68,
	(q31_t)0x5B9D1153, (q31_t)0xA60C21ED, (q31_t)0x5B1035CF,
	(q31_t)0xA57D8666, (q31_t)0x5A82799A, (q31_t)0xA4EFCA31,
	(q31_t)0x59F3DE12, (q31_t)0xA462EEAC, (q31_t)0x59646497,
	(q31_t)0xA3D6F533, (q31_t)0x58D40E8C, (q31_t)0xA34BDF20,
	(q31_t)0x5842DD54, (q31_t)0xA2C1ADC9, (q31_t)0x57B0D256,
	(q31_t)0xA2386283, (q31_t)0x571DEEF9, (q31_t)0xA1AFFEA2,
	(q31_t)0x568A34A9, (q31_t)0xA1288376, (q31_t)0x55F5A4D2,
	(q31_t)0xA0A1F24C, (q31_t)0x556040E2, (q31_t)0xA01C4C72,
	(q31_t)0x54CA0A4A, (q31_t)0x9F979331, (q31_t)0x5433027D,
	(q31_t)0x9F13C7D0, (q31_t)0x539B2AEF, (q31_t)0x9E90EB94,
	(q31_t)0x53028517, (q31_t)0x9E0EFFC1, (q31_t)0x5269126E,
	(q31_t)0x9D8E0596, (q31_t)0x51CED46E, (q31_t)0x9D0DFE53,
	(q31_t)0x5133CC94, (q31_t)0x9C8EEB33, (q31_t)0x5097FC5E,
	(q31_t)0x9C10CD70, (q31_t)0x4FFB654D, (q31_t)0x9B93A640,
	(q31_t)0x4F5E08E3, (q31_t)0x9B1776D9, (q31_t)0x4EBFE8A4,
	(q31_t)0x9A9C406D, (q31_t)0x4E210617, (q31_t)0x9A22042C,
	(q31_t)0x4D8162C4, (q31_t)0x99A8C344, (q31_t)0x4CE10034,
	(q31_t)0x99307EE0, (q31_t)0x4C3FDFF3, (q31_t)0x98B93828,
	(q31_t)0x4B9E038F, (q31_t)0x9842F043, (q31_t)0x4AFB6C97,
	(q31_t)0x97CDA855, (q31_t)0x4A581C9D, (q31_t)0x9759617E,
	(q31_t)0x49B41533, (q31_t)0x96E61CDF, (q31_t)0x490F57EE,
	(q31_t)0x9673DB94, (q31_t)0x4869E664, (q31_t)0x96029EB5,
	(q31_t)0x47C3C22E, (q31_t)0x9592675B, (q31_t)0x471CECE6,
	(q31_t)0x9523369B, (q31_t)0x46756827, (q31_t)0x94B50D87,
	(q31_t)0x45CD358F, (q31_t)0x9447ED2F, (q31_t)0x452456BC,
	(q31_t)0x93DBD69F, (q31_t)0x447ACD50, (q31_t)0x9370CAE4,
	(q31_t)0x43D09AEC, (q31_t)0x9306CB04, (q31_t)0x4325C135,
	(q31_t)0x929DD805, (q31_t)0x427A41D0, (q31_t)0x9235F2EB,
	(q31_t)0x41CE1E64, (q31_t)0x91CF1CB6, (q31_t)0x4121589A,
	(q31_t)0x91695663, (q31_t)0x4073F21D, (q31_t)0x9104A0ED,
	(q31_t)0x3FC5EC97, (q31_t)0x90A0FD4E, (q31_t)0x3F1749B7,
	(q31_t)0x903E6C7A, (q31_t)0x3E680B2C, (q31_t)0x8FDCEF66,
	(q31_t)0x3DB832A5, (q31_t)0x8F7C8701, (q31_t)0x3D07C1D5,
	(q31_t)0x8F1D343A, (q31_t)0x3C56BA70, (q31_t)0x8EBEF7FB,
	(q31_t)0x3BA51E29, (q31_t)0x8E61D32D, (q31_t)0x3AF2EEB7,
	(q31_t)0x8E05C6B7, (q31_t)0x3A402DD1, (q31_t)0x8DAAD37B,
	(q31_t)0x398CDD32, (q31_t)0x8D50FA59, (q31_t)0x38D8FE93,
	(q31_t)0x8CF83C30, (q31_t)0x382493B0, (q31_t)0x8CA099D9,
	(q31_t)0x376F9E46, (q31_t)0x8C4A142F, (q31_t)0x36BA2013,
	(q31_t)0x8BF4AC05, (q31_t)0x36041AD9, (q31_t)0x8BA0622F,
	(q31_t)0x354D9056, (q31_t)0x8B4D377C, (q31_t)0x3496824F,
	(q31_t)0x8AFB2CBA, (q31_t)0x33DEF287, (q31_t)0x8AAA42B4,
	(q31_t)0x3326E2C2, (q31_t)0x8A5A7A30, (q31_t)0x326E54C7,
	(q31_t)0x8A0BD3F5, (q31_t)0x31B54A5D, (q31_t)0x89BE50C3,
	(q31_t)0x30FBC54D, (q31_t)0x8971F15A, (q31_t)0x3041C760,
	(q31_t)0x8926B677, (q31_t)0x2F875262, (q31_t)0x88DCA0D3,
	(q31_t)0x2ECC681E, (q31_t)0x8893B124, (q31_t)0x2E110A62,
	(q31_t)0x884BE820, (q31_t)0x2D553AFB, (q31_t)0x88054677,
	(q31_t)0x2C98FBBA, (q31_t)0x87BFCCD7, (q31_t)0x2BDC4E6F,
	(q31_t)0x877B7BEC, (q31_t)0x2B1F34EB, (q31_t)0x8738545E,
	(q31_t)0x2A61B101, (q31_t)0x86F656D3, (q31_t)0x29A3C484,
	(q31_t)0x86B583EE, (q31_t)0x28E5714A, (q31_t)0x8675DC4E,
	(q31_t)0x2826B928, (q31_t)0x86376092, (q31_t)0x27679DF4,
	(q31_t)0x85FA1152, (q31_t)0x26A82185, (q31_t)0x85BDEF27,
	(q31_t)0x25E845B5, (q31_t)0x8582FAA4, (q31_t)0x25280C5D,
	(q31_t)0x8549345C, (q31_t)0x24677757, (q31_t)0x85109CDC,
	(q31_t)0x23A6887E, (q31_t)0x84D934B0, (q31_t)0x22E541AE,
	(q31_t)0x84A2FC62, (q31_t)0x2223A4C5, (q31_t)0x846DF476,
	(q31_t)0x2161B39F, (q31_t)0x843A1D70, (q31_t)0x209F701C,
	(q31_t)0x840777CF, (q31_t)0x1FDCDC1A, (q31_t)0x83D60411,
	(q31_t)0x1F19F97B, (q31_t)0x83A5C2B0, (q31_t)0x1E56CA1E,
	(q31_t)0x8376B422, (q31_t)0x1D934FE5, (q31_t)0x8348D8DB,
	(q31_t)0x1CCF8CB3, (q31_t)0x831C314E, (q31_t)0x1C0B826A,
	(q31_t)0x82F0BDE8, (q31_t)0x1B4732EF, (q31_t)0x82C67F13,
	(q31_t)0x1A82A025, (q31_t)0x829D753A, (q31_t)0x19BDCBF2,
	(q31_t)0x8275A0C0, (q31_t)0x18F8B83C, (q31_t)0x824F0208,
	(q31_t)0x183366E8, (q31_t)0x82299971, (q31_t)0x176DD9DE,
	(q31_t)0x82056758, (q31_t)0x16A81305, (q31_t)0x81E26C16,
	(q31_t)0x15E21444, (q31_t)0x81C0A801, (q31_t)0x151BDF85,
	(q31_t)0x81A01B6C, (q31_t)0x145576B1, (q31_t)0x8180C6A9,
	(q31_t)0x138EDBB0, (q31_t)0x8162AA03, (q31_t)0x12C8106E,
	(q31_t)0x8145C5C6, (q31_t)0x120116D4, (q31_t)0x812A1A39,
	(q31_t)0x1139F0CE, (q31_t)0x810FA7A0, (q31_t)0x1072A047,
	(q31_t)0x80F66E3C, (q31_t)0x0FAB272B, (q31_t)0x80DE6E4C,
	(q31_t)0x0EE38765, (q31_t)0x80C7A80A, (q31_t)0x0E1BC2E3,
	(q31_t)0x80B21BAF, (q31_t)0x0D53DB92, (q31_t)0x809DC970,
	(q31_t)0x0C8BD35E, (q31_t)0x808AB180, (q31_t)0x0BC3AC35,
	(q31_t)0x8078D40D, (q31_t)0x0AFB6805, (q31_t)0x80683143,
	(q31_t)0x0A3308BC, (q31_t)0x8058C94C, (q31_t)0x096A9049,
	(q31_t)0x804A9C4D, (q31_t)0x08A2009A, (q31_t)0x803DAA69,
	(q31_t)0x07D95B9E, (q31_t)0x8031F3C1, (q31_t)0x0710A344,
	(q31_t)0x80277872, (q31_t)0x0647D97C, (q31_t)0x801E3894,
	(q31_t)0x057F0034, (q31_t)0x80163440, (q31_t)0x04B6195D,
	(q31_t)0x800F6B88, (q31_t)0x03ED26E6, (q31_t)0x8009DE7D,
	(q31_t)0x03242ABF, (q31_t)0x80058D2E, (q31_t)0x025B26D7,
	(q31_t)0x800277A5, (q31_t)0x01921D1F, (q31_t)0x80009DE9,
	(q31_t)0x00C90F88, (q31_t)0x80000000, (q31_t)0x00000000,
	(q31_t)0x80009DE9, (q31_t)0xFF36F078, (q31_t)0x800277A5,
	(q31_t)0xFE6DE2E0, (q31_t)0x80058D2E, (q31_t)0xFDA4D928,
	(q31_t)0x8009DE7D, (q31_t)0xFCDBD541, (q31_t)0x800F6B88,
	(q31_t)0xFC12D919, (q31_t)0x80163440, (q31_t)0xFB49E6A2,
	(q31_t)0x801E3894, (q31_t)0xFA80FFCB, (q31_t)0x80277872,
	(q31_t)0xF9B82683, (q31_t)0x8031F3C1, (q31_t)0xF8EF5CBB,
	(q31_t)0x803DAA69, (q31_t)0xF826A461, (q31_t)0x804A9C4D,
	(q31_t)0xF75DFF65, (q31_t)0x8058C94C, (q31_t)0xF6956FB6,
	(q31_t)0x80683143, (q31_t)0xF5CCF743, (q31_t)0x8078D40D,
	(q31_t)0xF50497FA, (q31_t)0x808AB180, (q31_t)0xF43C53CA,
	(q31_t)0x809DC970, (q31_t)0xF3742CA1, (q31_t)0x80B21BAF,
	(q31_t)0xF2AC246D, (q31_t)0x80C7A80A, (q31_t)0xF1E43D1C,
	(q31_t)0x80DE6E4C, (q31_t)0xF11C789A, (q31_t)0x80F66E3C,
	(q31_t)0xF054D8D4, (q31_t)0x810FA7A0, (q31_t)0xEF8D5FB8,
	(q31_t)0x812A1A39, (q31_t)0xEEC60F31, (q31_t)0x8145C5C6,
	(q31_t)0xEDFEE92B, (q31_t)0x8162AA03, (q31_t)0xED37EF91,
	(q31_t)0x8180C6A9, (q31_t)0xEC71244F, (q31_t)0x81A01B6C,
	(q31_t)0xEBAA894E, (q31_t)0x81C0A801, (q31_t)0xEAE4207A,
	(q31_t)0x81E26C16, (q31_t)0xEA1DEBBB, (q31_t)0x82056758,
	(q31_t)0xE957ECFB, (q31_t)0x82299971, (q31_t)0xE8922621,
	(q31_t)0x824F0208, (q31_t)0xE7CC9917, (q31_t)0x8275A0C0,
	(q31_t)0xE70747C3, (q31_t)0x829D753A, (q31_t)0xE642340D,
	(q31_t)0x82C67F13, (q31_t)0xE57D5FDA, (q31_t)0x82F0BDE8,
	(q31_t)0xE4B8CD10, (q31_t)0x831C314E, (q31_t)0xE3F47D95,
	(q31_t)0x8348D8DB, (q31_t)0xE330734C, (q31_t)0x8376B422,
	(q31_t)0xE26CB01A, (q31_t)0x83A5C2B0, (q31_t)0xE1A935E1,
	(q31_t)0x83D60411, (q31_t)0xE0E60684, (q31_t)0x840777CF,
	(q31_t)0xE02323E5, (q31_t)0x843A1D70, (q31_t)0xDF608FE3,
	(q31_t)0x846DF476, (q31_t)0xDE9E4C60, (q31_t)0x84A2FC62,
	(q31_t)0xDDDC5B3A, (q31_t)0x84D934B0, (q31_t)0xDD1ABE51,
	(q31_t)0x85109CDC, (q31_t)0xDC597781, (q31_t)0x8549345C,
	(q31_t)0xDB9888A8, (q31_t)0x8582FAA4, (q31_t)0xDAD7F3A2,
	(q31_t)0x85BDEF27, (q31_t)0xDA17BA4A, (q31_t)0x85FA1152,
	(q31_t)0xD957DE7A, (q31_t)0x86376092, (q31_t)0xD898620C,
	(q31_t)0x8675DC4E, (q31_t)0xD7D946D7, (q31_t)0x86B583EE,
	(q31_t)0xD71A8EB5, (q31_t)0x86F656D3, (q31_t)0xD65C3B7B,
	(q31_t)0x8738545E, (q31_t)0xD59E4EFE, (q31_t)0x877B7BEC,
	(q31_t)0xD4E0CB14, (q31_t)0x87BFCCD7, (q31_t)0xD423B190,
	(q31_t)0x88054677, (q31_t)0xD3670445, (q31_t)0x884BE820,
	(q31_t)0xD2AAC504, (q31_t)0x8893B124, (q31_t)0xD1EEF59E,
	(q31_t)0x88DCA0D3, (q31_t)0xD13397E1, (q31_t)0x8926B677,
	(q31_t)0xD078AD9D, (q31_t)0x8971F15A, (q31_t)0xCFBE389F,
	(q31_t)0x89BE50C3, (q31_t)0xCF043AB2, (q31_t)0x8A0BD3F5,
	(q31_t)0xCE4AB5A2, (q31_t)0x8A5A7A30, (q31_t)0xCD91AB38,
	(q31_t)0x8AAA42B4, (q31_t)0xCCD91D3D, (q31_t)0x8AFB2CBA,
	(q31_t)0xCC210D78, (q31_t)0x8B4D377C, (q31_t)0xCB697DB0,
	(q31_t)0x8BA0622F, (q31_t)0xCAB26FA9, (q31_t)0x8BF4AC05,
	(q31_t)0xC9FBE527, (q31_t)0x8C4A142F, (q31_t)0xC945DFEC,
	(q31_t)0x8CA099D9, (q31_t)0xC89061BA, (q31_t)0x8CF83C30,
	(q31_t)0xC7DB6C50, (q31_t)0x8D50FA59, (q31_t)0xC727016C,
	(q31_t)0x8DAAD37B, (q31_t)0xC67322CD, (q31_t)0x8E05C6B7,
	(q31_t)0xC5BFD22E, (q31_t)0x8E61D32D, (q31_t)0xC50D1148,
	(q31_t)0x8EBEF7FB, (q31_t)0xC45AE1D7, (q31_t)0x8F1D343A,
	(q31_t)0xC3A9458F, (q31_t)0x8F7C8701, (q31_t)0xC2F83E2A,
	(q31_t)0x8FDCEF66, (q31_t)0xC247CD5A, (q31_t)0x903E6C7A,
	(q31_t)0xC197F4D3, (q31_t)0x90A0FD4E, (q31_t)0xC0E8B648,
	(q31_t)0x9104A0ED, (q31_t)0xC03A1368, (q31_t)0x91695663,
	(q31_t)0xBF8C0DE2, (q31_t)0x91CF1CB6, (q31_t)0xBEDEA765,
	(q31_t)0x9235F2EB, (q31_t)0xBE31E19B, (q31_t)0x929DD805,
	(q31_t)0xBD85BE2F, (q31_t)0x9306CB04, (q31_t)0xBCDA3ECA,
	(q31_t)0x9370CAE4, (q31_t)0xBC2F6513, (q31_t)0x93DBD69F,
	(q31_t)0xBB8532AF, (q31_t)0x9447ED2F, (q31_t)0xBADBA943,
	(q31_t)0x94B50D87, (q31_t)0xBA32CA70, (q31_t)0x9523369B,
	(q31_t)0xB98A97D8, (q31_t)0x9592675B, (q31_t)0xB8E31319,
	(q31_t)0x96029EB5, (q31_t)0xB83C3DD1, (q31_t)0x9673DB94,
	(q31_t)0xB796199B, (q31_t)0x96E61CDF, (q31_t)0xB6F0A811,
	(q31_t)0x9759617E, (q31_t)0xB64BEACC, (q31_t)0x97CDA855,
	(q31_t)0xB5A7E362, (q31_t)0x9842F043, (q31_t)0xB5049368,
	(q31_t)0x98B93828, (q31_t)0xB461FC70, (q31_t)0x99307EE0,
	(q31_t)0xB3C0200C, (q31_t)0x99A8C344, (q31_t)0xB31EFFCB,
	(q31_t)0x9A22042C, (q31_t)0xB27E9D3B, (q31_t)0x9A9C406D,
	(q31_t)0xB1DEF9E8, (q31_t)0x9B1776D9, (q31_t)0xB140175B,
	(q31_t)0x9B93A640, (q31_t)0xB0A1F71C, (q31_t)0x9C10CD70,
	(q31_t)0xB0049AB2, (q31_t)0x9C8EEB33, (q31_t)0xAF6803A1,
	(q31_t)0x9D0DFE53, (q31_t)0xAECC336B, (q31_t)0x9D8E0596,
	(q31_t)0xAE312B91, (q31_t)0x9E0EFFC1, (q31_t)0xAD96ED91,
	(q31_t)0x9E90EB94, (q31_t)0xACFD7AE8, (q31_t)0x9F13C7D0,
	(q31_t)0xAC64D510, (q31_t)0x9F979331, (q31_t)0xABCCFD82,
	(q31_t)0xA01C4C72, (q31_t)0xAB35F5B5, (q31_t)0xA0A1F24C,
	(q31_t)0xAA9FBF1D, (q31_t)0xA1288376, (q31_t)0xAA0A5B2D,
	(q31_t)0xA1AFFEA2, (q31_t)0xA975CB56, (q31_t)0xA2386283,
	(q31_t)0xA8E21106, (q31_t)0xA2C1ADC9, (q31_t)0xA84F2DA9,
	(q31_t)0xA34BDF20, (q31_t)0xA7BD22AB, (q31_t)0xA3D6F533,
	(q31_t)0xA72BF173, (q31_t)0xA462EEAC, (q31_t)0xA69B9B68,
	(q31_t)0xA4EFCA31, (q31_t)0xA60C21ED, (q31_t)0xA57D8666,
	(q31_t)0xA57D8666, (q31_t)0xA60C21ED, (q31_t)0xA4EFCA31,
	(q31_t)0xA69B9B68, (q31_t)0xA462EEAC, (q31_t)0xA72BF173,
	(q31_t)0xA3D6F533, (q31_t)0xA7BD22AB, (q31_t)0xA34BDF20,
	(q31_t)0xA84F2DA9, (q31_t)0xA2C1ADC9, (q31_t)0xA8E21106,
	(q31_t)0xA2386283, (q31_t)0xA975CB56, (q31_t)0xA1AFFEA2,
	(q31_t)0xAA0A5B2D, (q31_t)0xA1288376, (q31_t)0xAA9FBF1D,
	(q31_t)0xA0A1F24C, (q31_t)0xAB35F5B5, (q31_t)0xA01C4C72,
	(q31_t)0xABCCFD82, (q31_t)0x9F979331, (q31_t)0xAC64D510,
	(q31_t)0x9F13C7D0, (q31_t)0xACFD7AE8, (q31_t)0x9E90EB94,
	(q31_t)0xAD96ED91, (q31_t)0x9E0EFFC1, (q31_t)0xAE312B91,
	(q31_t)0x9D8E0596, (q31_t)0xAECC336B, (q31_t)0x9D0DFE53,
	(q31_t)0xAF6803A1, (q31_t)0x9C8EEB33, (q31_t)0xB0049AB2,
	(q31_t)0x9C10CD70, (q31_t)0xB0A1F71C, (q31_t)0x9B93A640,
	(q31_t)0xB140175B, (q31_t)0x9B1776D9, (q31_t)0xB1DEF9E8,
	(q31_t)0x9A9C406D, (q31_t)0xB27E9D3B, (q31_t)0x9A22042C,
	(q31_t)0xB31EFFCB, (q31_t)0x99A8C344, (q31_t)0xB3C0200C,
	(q31_t)0x99307EE0, (q31_t)0xB461FC70, (q31_t)0x98B93828,
	(q31_t)0xB5049368, (q31_t)0x9842F043, (q31_t)0xB5A7E362,
	(q31_t)0x97CDA855, (q31_t)0xB64BEACC, (q31_t)0x9759617E,
	(q31_t)0xB6F0A811, (q31_t)0x96E61CDF, (q31_t)0xB796199B,
	(q31_t)0x9673DB94, (q31_t)0xB83C3DD1, (q31_t)0x96029EB5,
	(q31_t)0xB8E31319, (q31_t)0x9592675B, (q31_t)0xB98A97D8,
	(q31_t)0x9523369B, (q31_t)0xBA32CA70, (q31_t)0x94B50D87,
	(q31_t)0xBADBA943, (q31_t)0x9447ED2F, (q31_t)0xBB8532AF,
	(q31_t)0x93DBD69F, (q31_t)0xBC2F6513, (q31_t)0x9370CAE4,
	(q31_t)0xBCDA3ECA, (q31_t)0x9306CB04, (q31_t)0xBD85BE2F,
	(q31_t)0x929DD805, (q31_t)0xBE31E19B, (q31_t)0x9235F2EB,
	(q31_t)0xBEDEA765, (q31_t)0x91CF1CB6, (q31_t)0xBF8C0DE2,
	(q31_t)0x91695663, (q31_t)0xC03A1368, (q31_t)0x9104A0ED,
	(q31_t)0xC0E8B648, (q31_t)0x90A0FD4E, (q31_t)0xC197F4D3,
	(q31_t)0x903E6C7A, (q31_t)0xC247CD5A, (q31_t)0x8FDCEF66,
	(q31_t)0xC2F83E2A, (q31_t)0x8F7C8701, (q31_t)0xC3A9458F,
	(q31_t)0x8F1D343A, (q31_t)0xC45AE1D7, (q31_t)0x8EBEF7FB,
	(q31_t)0xC50D1148, (q31_t)0x8E61D32D, (q31_t)0xC5BFD22E,
	(q31_t)0x8E05C6B7, (q31_t)0xC67322CD, (q31_t)0x8DAAD37B,
	(q31_t)0xC727016C, (q31_t)0x8D50FA59, (q31_t)0xC7DB6C50,
	(q31_t)0x8CF83C30, (q31_t)0xC89061BA, (q31_t)0x8CA099D9,
	(q31_t)0xC945DFEC, (q31_t)0x8C4A142F, (q31_t)0xC9FBE527,
	(q31_t)0x8BF4AC05, (q31_t)0xCAB26FA9, (q31_t)0x8BA0622F,
	(q31_t)0xCB697DB0, (q31_t)0x8B4D377C, (q31_t)0xCC210D78,
	(q31_t)0x8AFB2CBA, (q31_t)0xCCD91D3D, (q31_t)0x8AAA42B4,
	(q31_t)0xCD91AB38, (q31_t)0x8A5A7A30, (q31_t)0xCE4AB5A2,
	(q31_t)0x8A0BD3F5, (q31_t)0xCF043AB2, (q31_t)0x89BE50C3,
	(q31_t)0xCFBE389F, (q31_t)0x8971F15A, (q31_t)0xD078AD9D,
	(q31_t)0x8926B677, (q31_t)0xD13397E1, (q31_t)0x88DCA0D3,
	(q31_t)0xD1EEF59E, (q31_t)0x8893B124, (q31_t)0xD2AAC504,
	(q31_t)0x884BE820, (q31_t)0xD3670445, (q31_t)0x88054677,
	(q31_t)0xD423B190, (q31_t)0x87BFCCD7, (q31_t)0xD4E0CB14,
	(q31_t)0x877B7BEC, (q31_t)0xD59E4EFE, (q31_t)0x8738545E,
	(q31_t)0xD65C3B7B, (q31_t)0x86F656D3, (q31_t)0xD71A8EB5,
	(q31_t)0x86B583EE, (q31_t)0xD7D946D7, (q31_t)0x8675DC4E,
	(q31_t)0xD898620C, (q31_t)0x86376092, (q31_t)0xD957DE7A,
	(q31_t)0x85FA1152, (q31_t)0xDA17BA4A, (q31_t)0x85BDEF27,
	(q31_t)0xDAD7F3A2, (q31_t)0x8582FAA4, (q31_t)0xDB9888A8,
	(q31_t)0x8549345C, (q31_t)0xDC597781, (q31_t)0x85109CDC,
	(q31_t)0xDD1ABE51, (q31_t)0x84D934B0, (q31_t)0xDDDC5B3A,
	(q31_t)0x84A2FC62, (q31_t)0xDE9E4C60, (q31_t)0x846DF476,
	(q31_t)0xDF608FE3, (q31_t)0x843A1D70, (q31_t)0xE02323E5,
	(q31_t)0x840777CF, (q31_t)0xE0E60684, (q31_t)0x83D60411,
	(q31_t)0xE1A935E1, (q31_t)0x83A5C2B0, (q31_t)0xE26CB01A,
	(q31_t)0x8376B422, (q31_t)0xE330734C, (q31_t)0x8348D8DB,
	(q31_t)0xE3F47D95, (q31_t)0x831C314E, (q31_t)0xE4B8CD10,
	(q31_t)0x82F0BDE8, (q31_t)0xE57D5FDA, (q31_t)0x82C67F13,
	(q31_t)0xE642340D, (q31_t)0x829D753A, (q31_t)0xE70747C3,
	(q31_t)0x8275A0C0, (q31_t)0xE7CC9917, (q31_t)0x824F0208,
	(q31_t)0xE8922621, (q31_t)0x82299971, (q31_t)0xE957ECFB,
	(q31_t)0x82056758, (q31_t)0xEA1DEBBB, (q31_t)0x81E26C16,
	(q31_t)0xEAE4207A, (q31_t)0x81C0A801, (q31_t)0xEBAA894E,
	(q31_t)0x81A01B6C, (q31_t)0xEC71244F, (q31_t)0x8180C6A9,
	(q31_t)0xED37EF91, (q31_t)0x8162AA03, (q31_t)0xEDFEE92B,
	(q31_t)0x8145C5C6, (q31_t)0xEEC60F31, (q31_t)0x812A1A39,
	(q31_t)0xEF8D5FB8, (q31_t)0x810FA7A0, (q31_t)0xF054D8D4,
	(q31_t)0x80F66E3C, (q31_t)0xF11C789A, (q31_t)0x80DE6E4C,
	(q31_t)0xF1E43D1C, (q31_t)0x80C7A80A, (q31_t)0xF2AC246D,
	(q31_t)0x80B21BAF, (q31_t)0xF3742CA1, (q31_t)0x809DC970,
	(q31_t)0xF43C53CA, (q31_t)0x808AB180, (q31_t)0xF50497FA,
	(q31_t)0x8078D40D, (q31_t)0xF5CCF743, (q31_t)0x80683143,
	(q31_t)0xF6956FB6, (q31_t)0x8058C94C, (q31_t)0xF75DFF65,
	(q31_t)0x804A9C4D, (q31_t)0xF826A461, (q31_t)0x803DAA69,
	(q31_t)0xF8EF5CBB, (q31_t)0x8031F3C1, (q31_t)0xF9B82683,
	(q31_t)0x80277872, (q31_t)0xFA80FFCB, (q31_t)0x801E3894,
	(q31_t)0xFB49E6A2, (q31_t)0x80163440, (q31_t)0xFC12D919,
	(q31_t)0x800F6B88, (q31_t)0xFCDBD541, (q31_t)0x8009DE7D,
	(q31_t)0xFDA4D928, (q31_t)0x80058D2E, (q31_t)0xFE6DE2E0,
	(q31_t)0x800277A5, (q31_t)0xFF36F078, (q31_t)0x80009DE9
};

/**
* \par
* Example code for Q31 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefQ31[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefQ31[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 2048	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to Q31(Fixed point 1.31):
*	round(twiddleCoefQ31(i) * pow(2, 31))
*
*/
const q31_t twiddleCoef_2048_q31[3072] = {
	(q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7FFFD885,
	(q31_t)0x006487E3, (q31_t)0x7FFF6216, (q31_t)0x00C90F88,
	(q31_t)0x7FFE9CB2, (q31_t)0x012D96B0, (q31_t)0x7FFD885A,
	(q31_t)0x01921D1F, (q31_t)0x7FFC250F, (q31_t)0x01F6A296,
	(q31_t)0x7FFA72D1, (q31_t)0x025B26D7, (q31_t)0x7FF871A1,
	(q31_t)0x02BFA9A4, (q31_t)0x7FF62182, (q31_t)0x03242ABF,
	(q31_t)0x7FF38273, (q31_t)0x0388A9E9, (q31_t)0x7FF09477,
	(q31_t)0x03ED26E6, (q31_t)0x7FED5790, (q31_t)0x0451A176,
	(q31_t)0x7FE9CBC0, (q31_t)0x04B6195D, (q31_t)0x7FE5F108,
	(q31_t)0x051A8E5C, (q31_t)0x7FE1C76B, (q31_t)0x057F0034,
	(q31_t)0x7FDD4EEC, (q31_t)0x05E36EA9, (q31_t)0x7FD8878D,
	(q31_t)0x0647D97C, (q31_t)0x7FD37152, (q31_t)0x06AC406F,
	(q31_t)0x7FCE0C3E, (q31_t)0x0710A344, (q31_t)0x7FC85853,
	(q31_t)0x077501BE, (q31_t)0x7FC25596, (q31_t)0x07D95B9E,
	(q31_t)0x7FBC040A, (q31_t)0x083DB0A7, (q31_t)0x7FB563B2,
	(q31_t)0x08A2009A, (q31_t)0x7FAE7494, (q31_t)0x09064B3A,
	(q31_t)0x7FA736B4, (q31_t)0x096A9049, (q31_t)0x7F9FAA15,
	(q31_t)0x09CECF89, (q31_t)0x7F97CEBC, (q31_t)0x0A3308BC,
	(q31_t)0x7F8FA4AF, (q31_t)0x0A973BA5, (q31_t)0x7F872BF3,
	(q31_t)0x0AFB6805, (q31_t)0x7F7E648B, (q31_t)0x0B5F8D9F,
	(q31_t)0x7F754E7F, (q31_t)0x0BC3AC35, (q31_t)0x7F6BE9D4,
	(q31_t)0x0C27C389, (q31_t)0x7F62368F, (q31_t)0x0C8BD35E,
	(q31_t)0x7F5834B6, (q31_t)0x0CEFDB75, (q31_t)0x7F4DE450,
	(q31_t)0x0D53DB92, (q31_t)0x7F434563, (q31_t)0x0DB7D376,
	(q31_t)0x7F3857F5, (q31_t)0x0E1BC2E3, (q31_t)0x7F2D1C0E,
	(q31_t)0x0E7FA99D, (q31_t)0x7F2191B4, (q31_t)0x0EE38765,
	(q31_t)0x7F15B8EE, (q31_t)0x0F475BFE, (q31_t)0x7F0991C3,
	(q31_t)0x0FAB272B, (q31_t)0x7EFD1C3C, (q31_t)0x100EE8AD,
	(q31_t)0x7EF0585F, (q31_t)0x1072A047, (q31_t)0x7EE34635,
	(q31_t)0x10D64DBC, (q31_t)0x7ED5E5C6, (q31_t)0x1139F0CE,
	(q31_t)0x7EC8371A, (q31_t)0x119D8940, (q31_t)0x7EBA3A39,
	(q31_t)0x120116D4, (q31_t)0x7EABEF2C, (q31_t)0x1264994E,
	(q31_t)0x7E9D55FC, (q31_t)0x12C8106E, (q31_t)0x7E8E6EB1,
	(q31_t)0x132B7BF9, (q31_t)0x7E7F3956, (q31_t)0x138EDBB0,
	(q31_t)0x7E6FB5F3, (q31_t)0x13F22F57, (q31_t)0x7E5FE493,
	(q31_t)0x145576B1, (q31_t)0x7E4FC53E, (q31_t)0x14B8B17F,
	(q31_t)0x7E3F57FE, (q31_t)0x151BDF85, (q31_t)0x7E2E9CDF,
	(q31_t)0x157F0086, (q31_t)0x7E1D93E9, (q31_t)0x15E21444,
	(q31_t)0x7E0C3D29, (q31_t)0x16451A83, (q31_t)0x7DFA98A7,
	(q31_t)0x16A81305, (q31_t)0x7DE8A670, (q31_t)0x170AFD8D,
	(q31_t)0x7DD6668E, (q31_t)0x176DD9DE, (q31_t)0x7DC3D90D,
	(q31_t)0x17D0A7BB, (q31_t)0x7DB0FDF7, (q31_t)0x183366E8,
	(q31_t)0x7D9DD55A, (q31_t)0x18961727, (q31_t)0x7D8A5F3F,
	(q31_t)0x18F8B83C, (q31_t)0x7D769BB5, (q31_t)0x195B49E9,
	(q31_t)0x7D628AC5, (q31_t)0x19BDCBF2, (q31_t)0x7D4E2C7E,
	(q31_t)0x1A203E1B, (q31_t)0x7D3980EC, (q31_t)0x1A82A025,
	(q31_t)0x7D24881A, (q31_t)0x1AE4F1D6, (q31_t)0x7D0F4218,
	(q31_t)0x1B4732EF, (q31_t)0x7CF9AEF0, (q31_t)0x1BA96334,
	(q31_t)0x7CE3CEB1, (q31_t)0x1C0B826A, (q31_t)0x7CCDA168,
	(q31_t)0x1C6D9053, (q31_t)0x7CB72724, (q31_t)0x1CCF8CB3,
	(q31_t)0x7CA05FF1, (q31_t)0x1D31774D, (q31_t)0x7C894BDD,
	(q31_t)0x1D934FE5, (q31_t)0x7C71EAF8, (q31_t)0x1DF5163F,
	(q31_t)0x7C5A3D4F, (q31_t)0x1E56CA1E, (q31_t)0x7C4242F2,
	(q31_t)0x1EB86B46, (q31_t)0x7C29FBEE, (q31_t)0x1F19F97B,
	(q31_t)0x7C116853, (q31_t)0x1F7B7480, (q31_t)0x7BF88830,
	(q31_t)0x1FDCDC1A, (q31_t)0x7BDF5B94, (q31_t)0x203E300D,
	(q31_t)0x7BC5E28F, (q31_t)0x209F701C, (q31_t)0x7BAC1D31,
	(q31_t)0x21009C0B, (q31_t)0x7B920B89, (q31_t)0x2161B39F,
	(q31_t)0x7B77ADA8, (q31_t)0x21C2B69C, (q31_t)0x7B5D039D,
	(q31_t)0x2223A4C5, (q31_t)0x7B420D7A, (q31_t)0x22847DDF,
	(q31_t)0x7B26CB4F, (q31_t)0x22E541AE, (q31_t)0x7B0B3D2C,
	(q31_t)0x2345EFF7, (q31_t)0x7AEF6323, (q31_t)0x23A6887E,
	(q31_t)0x7AD33D45, (q31_t)0x24070B07, (q31_t)0x7AB6CBA3,
	(q31_t)0x24677757, (q31_t)0x7A9A0E4F, (q31_t)0x24C7CD32,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x7A5FB0D8,
	(q31_t)0x2588349D, (q31_t)0x7A4210D8, (q31_t)0x25E845B5,
	(q31_t)0x7A24256E, (q31_t)0x26483F6C, (q31_t)0x7A05EEAD,
	(q31_t)0x26A82185, (q31_t)0x79E76CA6, (q31_t)0x2707EBC6,
	(q31_t)0x79C89F6D, (q31_t)0x27679DF4, (q31_t)0x79A98715,
	(q31_t)0x27C737D2, (q31_t)0x798A23B1, (q31_t)0x2826B928,
	(q31_t)0x796A7554, (q31_t)0x288621B9, (q31_t)0x794A7C11,
	(q31_t)0x28E5714A, (q31_t)0x792A37FE, (q31_t)0x2944A7A2,
	(q31_t)0x7909A92C, (q31_t)0x29A3C484, (q31_t)0x78E8CFB1,
	(q31_t)0x2A02C7B8, (q31_t)0x78C7ABA1, (q31_t)0x2A61B101,
	(q31_t)0x78A63D10, (q31_t)0x2AC08025, (q31_t)0x78848413,
	(q31_t)0x2B1F34EB, (q31_t)0x786280BF, (q31_t)0x2B7DCF17,
	(q31_t)0x78403328, (q31_t)0x2BDC4E6F, (q31_t)0x781D9B64,
	(q31_t)0x2C3AB2B9, (q31_t)0x77FAB988, (q31_t)0x2C98FBBA,
	(q31_t)0x77D78DAA, (q31_t)0x2CF72939, (q31_t)0x77B417DF,
	(q31_t)0x2D553AFB, (q31_t)0x7790583D, (q31_t)0x2DB330C7,
	(q31_t)0x776C4EDB, (q31_t)0x2E110A62, (q31_t)0x7747FBCE,
	(q31_t)0x2E6EC792, (q31_t)0x77235F2D, (q31_t)0x2ECC681E,
	(q31_t)0x76FE790E, (q31_t)0x2F29EBCC, (q31_t)0x76D94988,
	(q31_t)0x2F875262, (q31_t)0x76B3D0B3, (q31_t)0x2FE49BA6,
	(q31_t)0x768E0EA5, (q31_t)0x3041C760, (q31_t)0x76680376,
	(q31_t)0x309ED555, (q31_t)0x7641AF3C, (q31_t)0x30FBC54D,
	(q31_t)0x761B1211, (q31_t)0x3158970D, (q31_t)0x75F42C0A,
	(q31_t)0x31B54A5D, (q31_t)0x75CCFD42, (q31_t)0x3211DF03,
	(q31_t)0x75A585CF, (q31_t)0x326E54C7, (q31_t)0x757DC5CA,
	(q31_t)0x32CAAB6F, (q31_t)0x7555BD4B, (q31_t)0x3326E2C2,
	(q31_t)0x752D6C6C, (q31_t)0x3382FA88, (q31_t)0x7504D345,
	(q31_t)0x33DEF287, (q31_t)0x74DBF1EF, (q31_t)0x343ACA87,
	(q31_t)0x74B2C883, (q31_t)0x3496824F, (q31_t)0x7489571B,
	(q31_t)0x34F219A7, (q31_t)0x745F9DD1, (q31_t)0x354D9056,
	(q31_t)0x74359CBD, (q31_t)0x35A8E624, (q31_t)0x740B53FA,
	(q31_t)0x36041AD9, (q31_t)0x73E0C3A3, (q31_t)0x365F2E3B,
	(q31_t)0x73B5EBD0, (q31_t)0x36BA2013, (q31_t)0x738ACC9E,
	(q31_t)0x3714F02A, (q31_t)0x735F6626, (q31_t)0x376F9E46,
	(q31_t)0x7333B883, (q31_t)0x37CA2A30, (q31_t)0x7307C3D0,
	(q31_t)0x382493B0, (q31_t)0x72DB8828, (q31_t)0x387EDA8E,
	(q31_t)0x72AF05A6, (q31_t)0x38D8FE93, (q31_t)0x72823C66,
	(q31_t)0x3932FF87, (q31_t)0x72552C84, (q31_t)0x398CDD32,
	(q31_t)0x7227D61C, (q31_t)0x39E6975D, (q31_t)0x71FA3948,
	(q31_t)0x3A402DD1, (q31_t)0x71CC5626, (q31_t)0x3A99A057,
	(q31_t)0x719E2CD2, (q31_t)0x3AF2EEB7, (q31_t)0x716FBD68,
	(q31_t)0x3B4C18BA, (q31_t)0x71410804, (q31_t)0x3BA51E29,
	(q31_t)0x71120CC5, (q31_t)0x3BFDFECD, (q31_t)0x70E2CBC6,
	(q31_t)0x3C56BA70, (q31_t)0x70B34524, (q31_t)0x3CAF50DA,
	(q31_t)0x708378FE, (q31_t)0x3D07C1D5, (q31_t)0x70536771,
	(q31_t)0x3D600D2B, (q31_t)0x70231099, (q31_t)0x3DB832A5,
	(q31_t)0x6FF27496, (q31_t)0x3E10320D, (q31_t)0x6FC19385,
	(q31_t)0x3E680B2C, (q31_t)0x6F906D84, (q31_t)0x3EBFBDCC,
	(q31_t)0x6F5F02B1, (q31_t)0x3F1749B7, (q31_t)0x6F2D532C,
	(q31_t)0x3F6EAEB8, (q31_t)0x6EFB5F12, (q31_t)0x3FC5EC97,
	(q31_t)0x6EC92682, (q31_t)0x401D0320, (q31_t)0x6E96A99C,
	(q31_t)0x4073F21D, (q31_t)0x6E63E87F, (q31_t)0x40CAB957,
	(q31_t)0x6E30E349, (q31_t)0x4121589A, (q31_t)0x6DFD9A1B,
	(q31_t)0x4177CFB0, (q31_t)0x6DCA0D14, (q31_t)0x41CE1E64,
	(q31_t)0x6D963C54, (q31_t)0x42244480, (q31_t)0x6D6227FA,
	(q31_t)0x427A41D0, (q31_t)0x6D2DD027, (q31_t)0x42D0161E,
	(q31_t)0x6CF934FB, (q31_t)0x4325C135, (q31_t)0x6CC45697,
	(q31_t)0x437B42E1, (q31_t)0x6C8F351C, (q31_t)0x43D09AEC,
	(q31_t)0x6C59D0A9, (q31_t)0x4425C923, (q31_t)0x6C242960,
	(q31_t)0x447ACD50, (q31_t)0x6BEE3F62, (q31_t)0x44CFA73F,
	(q31_t)0x6BB812D0, (q31_t)0x452456BC, (q31_t)0x6B81A3CD,
	(q31_t)0x4578DB93, (q31_t)0x6B4AF278, (q31_t)0x45CD358F,
	(q31_t)0x6B13FEF5, (q31_t)0x4621647C, (q31_t)0x6ADCC964,
	(q31_t)0x46756827, (q31_t)0x6AA551E8, (q31_t)0x46C9405C,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x6A359DB9,
	(q31_t)0x47706D93, (q31_t)0x69FD614A, (q31_t)0x47C3C22E,
	(q31_t)0x69C4E37A, (q31_t)0x4816EA85, (q31_t)0x698C246C,
	(q31_t)0x4869E664, (q31_t)0x69532442, (q31_t)0x48BCB598,
	(q31_t)0x6919E320, (q31_t)0x490F57EE, (q31_t)0x68E06129,
	(q31_t)0x4961CD32, (q31_t)0x68A69E81, (q31_t)0x49B41533,
	(q31_t)0x686C9B4B, (q31_t)0x4A062FBD, (q31_t)0x683257AA,
	(q31_t)0x4A581C9D, (q31_t)0x67F7D3C4, (q31_t)0x4AA9DBA1,
	(q31_t)0x67BD0FBC, (q31_t)0x4AFB6C97, (q31_t)0x67820BB6,
	(q31_t)0x4B4CCF4D, (q31_t)0x6746C7D7, (q31_t)0x4B9E038F,
	(q31_t)0x670B4443, (q31_t)0x4BEF092D, (q31_t)0x66CF811F,
	(q31_t)0x4C3FDFF3, (q31_t)0x66937E90, (q31_t)0x4C9087B1,
	(q31_t)0x66573CBB, (q31_t)0x4CE10034, (q31_t)0x661ABBC5,
	(q31_t)0x4D31494B, (q31_t)0x65DDFBD3, (q31_t)0x4D8162C4,
	(q31_t)0x65A0FD0B, (q31_t)0x4DD14C6E, (q31_t)0x6563BF92,
	(q31_t)0x4E210617, (q31_t)0x6526438E, (q31_t)0x4E708F8F,
	(q31_t)0x64E88926, (q31_t)0x4EBFE8A4, (q31_t)0x64AA907F,
	(q31_t)0x4F0F1126, (q31_t)0x646C59BF, (q31_t)0x4F5E08E3,
	(q31_t)0x642DE50D, (q31_t)0x4FACCFAB, (q31_t)0x63EF328F,
	(q31_t)0x4FFB654D, (q31_t)0x63B0426D, (q31_t)0x5049C999,
	(q31_t)0x637114CC, (q31_t)0x5097FC5E, (q31_t)0x6331A9D4,
	(q31_t)0x50E5FD6C, (q31_t)0x62F201AC, (q31_t)0x5133CC94,
	(q31_t)0x62B21C7B, (q31_t)0x518169A4, (q31_t)0x6271FA69,
	(q31_t)0x51CED46E, (q31_t)0x62319B9D, (q31_t)0x521C0CC1,
	(q31_t)0x61F1003E, (q31_t)0x5269126E, (q31_t)0x61B02876,
	(q31_t)0x52B5E545, (q31_t)0x616F146B, (q31_t)0x53028517,
	(q31_t)0x612DC446, (q31_t)0x534EF1B5, (q31_t)0x60EC3830,
	(q31_t)0x539B2AEF, (q31_t)0x60AA704F, (q31_t)0x53E73097,
	(q31_t)0x60686CCE, (q31_t)0x5433027D, (q31_t)0x60262DD5,
	(q31_t)0x547EA073, (q31_t)0x5FE3B38D, (q31_t)0x54CA0A4A,
	(q31_t)0x5FA0FE1E, (q31_t)0x55153FD4, (q31_t)0x5F5E0DB3,
	(q31_t)0x556040E2, (q31_t)0x5F1AE273, (q31_t)0x55AB0D46,
	(q31_t)0x5ED77C89, (q31_t)0x55F5A4D2, (q31_t)0x5E93DC1F,
	(q31_t)0x56400757, (q31_t)0x5E50015D, (q31_t)0x568A34A9,
	(q31_t)0x5E0BEC6E, (q31_t)0x56D42C99, (q31_t)0x5DC79D7C,
	(q31_t)0x571DEEF9, (q31_t)0x5D8314B0, (q31_t)0x57677B9D,
	(q31_t)0x5D3E5236, (q31_t)0x57B0D256, (q31_t)0x5CF95638,
	(q31_t)0x57F9F2F7, (q31_t)0x5CB420DF, (q31_t)0x5842DD54,
	(q31_t)0x5C6EB258, (q31_t)0x588B913F, (q31_t)0x5C290ACC,
	(q31_t)0x58D40E8C, (q31_t)0x5BE32A67, (q31_t)0x591C550E,
	(q31_t)0x5B9D1153, (q31_t)0x59646497, (q31_t)0x5B56BFBD,
	(q31_t)0x59AC3CFD, (q31_t)0x5B1035CF, (q31_t)0x59F3DE12,
	(q31_t)0x5AC973B4, (q31_t)0x5A3B47AA, (q31_t)0x5A82799A,
	(q31_t)0x5A82799A, (q31_t)0x5A3B47AA, (q31_t)0x5AC973B4,
	(q31_t)0x59F3DE12, (q31_t)0x5B1035CF, (q31_t)0x59AC3CFD,
	(q31_t)0x5B56BFBD, (q31_t)0x59646497, (q31_t)0x5B9D1153,
	(q31_t)0x591C550E, (q31_t)0x5BE32A67, (q31_t)0x58D40E8C,
	(q31_t)0x5C290ACC, (q31_t)0x588B913F, (q31_t)0x5C6EB258,
	(q31_t)0x5842DD54, (q31_t)0x5CB420DF, (q31_t)0x57F9F2F7,
	(q31_t)0x5CF95638, (q31_t)0x57B0D256, (q31_t)0x5D3E5236,
	(q31_t)0x57677B9D, (q31_t)0x5D8314B0, (q31_t)0x571DEEF9,
	(q31_t)0x5DC79D7C, (q31_t)0x56D42C99, (q31_t)0x5E0BEC6E,
	(q31_t)0x568A34A9, (q31_t)0x5E50015D, (q31_t)0x56400757,
	(q31_t)0x5E93DC1F, (q31_t)0x55F5A4D2, (q31_t)0x5ED77C89,
	(q31_t)0x55AB0D46, (q31_t)0x5F1AE273, (q31_t)0x556040E2,
	(q31_t)0x5F5E0DB3, (q31_t)0x55153FD4, (q31_t)0x5FA0FE1E,
	(q31_t)0x54CA0A4A, (q31_t)0x5FE3B38D, (q31_t)0x547EA073,
	(q31_t)0x60262DD5, (q31_t)0x5433027D, (q31_t)0x60686CCE,
	(q31_t)0x53E73097, (q31_t)0x60AA704F, (q31_t)0x539B2AEF,
	(q31_t)0x60EC3830, (q31_t)0x534EF1B5, (q31_t)0x612DC446,
	(q31_t)0x53028517, (q31_t)0x616F146B, (q31_t)0x52B5E545,
	(q31_t)0x61B02876, (q31_t)0x5269126E, (q31_t)0x61F1003E,
	(q31_t)0x521C0CC1, (q31_t)0x62319B9D, (q31_t)0x51CED46E,
	(q31_t)0x6271FA69, (q31_t)0x518169A4, (q31_t)0x62B21C7B,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x50E5FD6C,
	(q31_t)0x6331A9D4, (q31_t)0x5097FC5E, (q31_t)0x637114CC,
	(q31_t)0x5049C999, (q31_t)0x63B0426D, (q31_t)0x4FFB654D,
	(q31_t)0x63EF328F, (q31_t)0x4FACCFAB, (q31_t)0x642DE50D,
	(q31_t)0x4F5E08E3, (q31_t)0x646C59BF, (q31_t)0x4F0F1126,
	(q31_t)0x64AA907F, (q31_t)0x4EBFE8A4, (q31_t)0x64E88926,
	(q31_t)0x4E708F8F, (q31_t)0x6526438E, (q31_t)0x4E210617,
	(q31_t)0x6563BF92, (q31_t)0x4DD14C6E, (q31_t)0x65A0FD0B,
	(q31_t)0x4D8162C4, (q31_t)0x65DDFBD3, (q31_t)0x4D31494B,
	(q31_t)0x661ABBC5, (q31_t)0x4CE10034, (q31_t)0x66573CBB,
	(q31_t)0x4C9087B1, (q31_t)0x66937E90, (q31_t)0x4C3FDFF3,
	(q31_t)0x66CF811F, (q31_t)0x4BEF092D, (q31_t)0x670B4443,
	(q31_t)0x4B9E038F, (q31_t)0x6746C7D7, (q31_t)0x4B4CCF4D,
	(q31_t)0x67820BB6, (q31_t)0x4AFB6C97, (q31_t)0x67BD0FBC,
	(q31_t)0x4AA9DBA1, (q31_t)0x67F7D3C4, (q31_t)0x4A581C9D,
	(q31_t)0x683257AA, (q31_t)0x4A062FBD, (q31_t)0x686C9B4B,
	(q31_t)0x49B41533, (q31_t)0x68A69E81, (q31_t)0x4961CD32,
	(q31_t)0x68E06129, (q31_t)0x490F57EE, (q31_t)0x6919E320,
	(q31_t)0x48BCB598, (q31_t)0x69532442, (q31_t)0x4869E664,
	(q31_t)0x698C246C, (q31_t)0x4816EA85, (q31_t)0x69C4E37A,
	(q31_t)0x47C3C22E, (q31_t)0x69FD614A, (q31_t)0x47706D93,
	(q31_t)0x6A359DB9, (q31_t)0x471CECE6, (q31_t)0x6A6D98A4,
	(q31_t)0x46C9405C, (q31_t)0x6AA551E8, (q31_t)0x46756827,
	(q31_t)0x6ADCC964, (q31_t)0x4621647C, (q31_t)0x6B13FEF5,
	(q31_t)0x45CD358F, (q31_t)0x6B4AF278, (q31_t)0x4578DB93,
	(q31_t)0x6B81A3CD, (q31_t)0x452456BC, (q31_t)0x6BB812D0,
	(q31_t)0x44CFA73F, (q31_t)0x6BEE3F62, (q31_t)0x447ACD50,
	(q31_t)0x6C242960, (q31_t)0x4425C923, (q31_t)0x6C59D0A9,
	(q31_t)0x43D09AEC, (q31_t)0x6C8F351C, (q31_t)0x437B42E1,
	(q31_t)0x6CC45697, (q31_t)0x4325C135, (q31_t)0x6CF934FB,
	(q31_t)0x42D0161E, (q31_t)0x6D2DD027, (q31_t)0x427A41D0,
	(q31_t)0x6D6227FA, (q31_t)0x42244480, (q31_t)0x6D963C54,
	(q31_t)0x41CE1E64, (q31_t)0x6DCA0D14, (q31_t)0x4177CFB0,
	(q31_t)0x6DFD9A1B, (q31_t)0x4121589A, (q31_t)0x6E30E349,
	(q31_t)0x40CAB957, (q31_t)0x6E63E87F, (q31_t)0x4073F21D,
	(q31_t)0x6E96A99C, (q31_t)0x401D0320, (q31_t)0x6EC92682,
	(q31_t)0x3FC5EC97, (q31_t)0x6EFB5F12, (q31_t)0x3F6EAEB8,
	(q31_t)0x6F2D532C, (q31_t)0x3F1749B7, (q31_t)0x6F5F02B1,
	(q31_t)0x3EBFBDCC, (q31_t)0x6F906D84, (q31_t)0x3E680B2C,
	(q31_t)0x6FC19385, (q31_t)0x3E10320D, (q31_t)0x6FF27496,
	(q31_t)0x3DB832A5, (q31_t)0x70231099, (q31_t)0x3D600D2B,
	(q31_t)0x70536771, (q31_t)0x3D07C1D5, (q31_t)0x708378FE,
	(q31_t)0x3CAF50DA, (q31_t)0x70B34524, (q31_t)0x3C56BA70,
	(q31_t)0x70E2CBC6, (q31_t)0x3BFDFECD, (q31_t)0x71120CC5,
	(q31_t)0x3BA51E29, (q31_t)0x71410804, (q31_t)0x3B4C18BA,
	(q31_t)0x716FBD68, (q31_t)0x3AF2EEB7, (q31_t)0x719E2CD2,
	(q31_t)0x3A99A057, (q31_t)0x71CC5626, (q31_t)0x3A402DD1,
	(q31_t)0x71FA3948, (q31_t)0x39E6975D, (q31_t)0x7227D61C,
	(q31_t)0x398CDD32, (q31_t)0x72552C84, (q31_t)0x3932FF87,
	(q31_t)0x72823C66, (q31_t)0x38D8FE93, (q31_t)0x72AF05A6,
	(q31_t)0x387EDA8E, (q31_t)0x72DB8828, (q31_t)0x382493B0,
	(q31_t)0x7307C3D0, (q31_t)0x37CA2A30, (q31_t)0x7333B883,
	(q31_t)0x376F9E46, (q31_t)0x735F6626, (q31_t)0x3714F02A,
	(q31_t)0x738ACC9E, (q31_t)0x36BA2013, (q31_t)0x73B5EBD0,
	(q31_t)0x365F2E3B, (q31_t)0x73E0C3A3, (q31_t)0x36041AD9,
	(q31_t)0x740B53FA, (q31_t)0x35A8E624, (q31_t)0x74359CBD,
	(q31_t)0x354D9056, (q31_t)0x745F9DD1, (q31_t)0x34F219A7,
	(q31_t)0x7489571B, (q31_t)0x3496824F, (q31_t)0x74B2C883,
	(q31_t)0x343ACA87, (q31_t)0x74DBF1EF, (q31_t)0x33DEF287,
	(q31_t)0x7504D345, (q31_t)0x3382FA88, (q31_t)0x752D6C6C,
	(q31_t)0x3326E2C2, (q31_t)0x7555BD4B, (q31_t)0x32CAAB6F,
	(q31_t)0x757DC5CA, (q31_t)0x326E54C7, (q31_t)0x75A585CF,
	(q31_t)0x3211DF03, (q31_t)0x75CCFD42, (q31_t)0x31B54A5D,
	(q31_t)0x75F42C0A, (q31_t)0x3158970D, (q31_t)0x761B1211,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x309ED555,
	(q31_t)0x76680376, (q31_t)0x3041C760, (q31_t)0x768E0EA5,
	(q31_t)0x2FE49BA6, (q31_t)0x76B3D0B3, (q31_t)0x2F875262,
	(q31_t)0x76D94988, (q31_t)0x2F29EBCC, (q31_t)0x76FE790E,
	(q31_t)0x2ECC681E, (q31_t)0x77235F2D, (q31_t)0x2E6EC792,
	(q31_t)0x7747FBCE, (q31_t)0x2E110A62, (q31_t)0x776C4EDB,
	(q31_t)0x2DB330C7, (q31_t)0x7790583D, (q31_t)0x2D553AFB,
	(q31_t)0x77B417DF, (q31_t)0x2CF72939, (q31_t)0x77D78DAA,
	(q31_t)0x2C98FBBA, (q31_t)0x77FAB988, (q31_t)0x2C3AB2B9,
	(q31_t)0x781D9B64, (q31_t)0x2BDC4E6F, (q31_t)0x78403328,
	(q31_t)0x2B7DCF17, (q31_t)0x786280BF, (q31_t)0x2B1F34EB,
	(q31_t)0x78848413, (q31_t)0x2AC08025, (q31_t)0x78A63D10,
	(q31_t)0x2A61B101, (q31_t)0x78C7ABA1, (q31_t)0x2A02C7B8,
	(q31_t)0x78E8CFB1, (q31_t)0x29A3C484, (q31_t)0x7909A92C,
	(q31_t)0x2944A7A2, (q31_t)0x792A37FE, (q31_t)0x28E5714A,
	(q31_t)0x794A7C11, (q31_t)0x288621B9, (q31_t)0x796A7554,
	(q31_t)0x2826B928, (q31_t)0x798A23B1, (q31_t)0x27C737D2,
	(q31_t)0x79A98715, (q31_t)0x27679DF4, (q31_t)0x79C89F6D,
	(q31_t)0x2707EBC6, (q31_t)0x79E76CA6, (q31_t)0x26A82185,
	(q31_t)0x7A05EEAD, (q31_t)0x26483F6C, (q31_t)0x7A24256E,
	(q31_t)0x25E845B5, (q31_t)0x7A4210D8, (q31_t)0x2588349D,
	(q31_t)0x7A5FB0D8, (q31_t)0x25280C5D, (q31_t)0x7A7D055B,
	(q31_t)0x24C7CD32, (q31_t)0x7A9A0E4F, (q31_t)0x24677757,
	(q31_t)0x7AB6CBA3, (q31_t)0x24070B07, (q31_t)0x7AD33D45,
	(q31_t)0x23A6887E, (q31_t)0x7AEF6323, (q31_t)0x2345EFF7,
	(q31_t)0x7B0B3D2C, (q31_t)0x22E541AE, (q31_t)0x7B26CB4F,
	(q31_t)0x22847DDF, (q31_t)0x7B420D7A, (q31_t)0x2223A4C5,
	(q31_t)0x7B5D039D, (q31_t)0x21C2B69C, (q31_t)0x7B77ADA8,
	(q31_t)0x2161B39F, (q31_t)0x7B920B89, (q31_t)0x21009C0B,
	(q31_t)0x7BAC1D31, (q31_t)0x209F701C, (q31_t)0x7BC5E28F,
	(q31_t)0x203E300D, (q31_t)0x7BDF5B94, (q31_t)0x1FDCDC1A,
	(q31_t)0x7BF88830, (q31_t)0x1F7B7480, (q31_t)0x7C116853,
	(q31_t)0x1F19F97B, (q31_t)0x7C29FBEE, (q31_t)0x1EB86B46,
	(q31_t)0x7C4242F2, (q31_t)0x1E56CA1E, (q31_t)0x7C5A3D4F,
	(q31_t)0x1DF5163F, (q31_t)0x7C71EAF8, (q31_t)0x1D934FE5,
	(q31_t)0x7C894BDD, (q31_t)0x1D31774D, (q31_t)0x7CA05FF1,
	(q31_t)0x1CCF8CB3, (q31_t)0x7CB72724, (q31_t)0x1C6D9053,
	(q31_t)0x7CCDA168, (q31_t)0x1C0B826A, (q31_t)0x7CE3CEB1,
	(q31_t)0x1BA96334, (q31_t)0x7CF9AEF0, (q31_t)0x1B4732EF,
	(q31_t)0x7D0F4218, (q31_t)0x1AE4F1D6, (q31_t)0x7D24881A,
	(q31_t)0x1A82A025, (q31_t)0x7D3980EC, (q31_t)0x1A203E1B,
	(q31_t)0x7D4E2C7E, (q31_t)0x19BDCBF2, (q31_t)0x7D628AC5,
	(q31_t)0x195B49E9, (q31_t)0x7D769BB5, (q31_t)0x18F8B83C,
	(q31_t)0x7D8A5F3F, (q31_t)0x18961727, (q31_t)0x7D9DD55A,
	(q31_t)0x183366E8, (q31_t)0x7DB0FDF7, (q31_t)0x17D0A7BB,
	(q31_t)0x7DC3D90D, (q31_t)0x176DD9DE, (q31_t)0x7DD6668E,
	(q31_t)0x170AFD8D, (q31_t)0x7DE8A670, (q31_t)0x16A81305,
	(q31_t)0x7DFA98A7, (q31_t)0x16451A83, (q31_t)0x7E0C3D29,
	(q31_t)0x15E21444, (q31_t)0x7E1D93E9, (q31_t)0x157F0086,
	(q31_t)0x7E2E9CDF, (q31_t)0x151BDF85, (q31_t)0x7E3F57FE,
	(q31_t)0x14B8B17F, (q31_t)0x7E4FC53E, (q31_t)0x145576B1,
	(q31_t)0x7E5FE493, (q31_t)0x13F22F57, (q31_t)0x7E6FB5F3,
	(q31_t)0x138EDBB0, (q31_t)0x7E7F3956, (q31_t)0x132B7BF9,
	(q31_t)0x7E8E6EB1, (q31_t)0x12C8106E, (q31_t)0x7E9D55FC,
	(q31_t)0x1264994E, (q31_t)0x7EABEF2C, (q31_t)0x120116D4,
	(q31_t)0x7EBA3A39, (q31_t)0x119D8940, (q31_t)0x7EC8371A,
	(q31_t)0x1139F0CE, (q31_t)0x7ED5E5C6, (q31_t)0x10D64DBC,
	(q31_t)0x7EE34635, (q31_t)0x1072A047, (q31_t)0x7EF0585F,
	(q31_t)0x100EE8AD, (q31_t)0x7EFD1C3C, (q31_t)0x0FAB272B,
	(q31_t)0x7F0991C3, (q31_t)0x0F475BFE, (q31_t)0x7F15B8EE,
	(q31_t)0x0EE38765, (q31_t)0x7F2191B4, (q31_t)0x0E7FA99D,
	(q31_t)0x7F2D1C0E, (q31_t)0x0E1BC2E3, (q31_t)0x7F3857F5,
	(q31_t)0x0DB7D376, (q31_t)0x7F434563, (q31_t)0x0D53DB92,
	(q31_t)0x7F4DE450, (q31_t)0x0CEFDB75, (q31_t)0x7F5834B6,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x0C27C389,
	(q31_t)0x7F6BE9D4, (q31_t)0x0BC3AC35, (q31_t)0x7F754E7F,
	(q31_t)0x0B5F8D9F, (q31_t)0x7F7E648B, (q31_t)0x0AFB6805,
	(q31_t)0x7F872BF3, (q31_t)0x0A973BA5, (q31_t)0x7F8FA4AF,
	(q31_t)0x0A3308BC, (q31_t)0x7F97CEBC, (q31_t)0x09CECF89,
	(q31_t)0x7F9FAA15, (q31_t)0x096A9049, (q31_t)0x7FA736B4,
	(q31_t)0x09064B3A, (q31_t)0x7FAE7494, (q31_t)0x08A2009A,
	(q31_t)0x7FB563B2, (q31_t)0x083DB0A7, (q31_t)0x7FBC040A,
	(q31_t)0x07D95B9E, (q31_t)0x7FC25596, (q31_t)0x077501BE,
	(q31_t)0x7FC85853, (q31_t)0x0710A344, (q31_t)0x7FCE0C3E,
	(q31_t)0x06AC406F, (q31_t)0x7FD37152, (q31_t)0x0647D97C,
	(q31_t)0x7FD8878D, (q31_t)0x05E36EA9, (q31_t)0x7FDD4EEC,
	(q31_t)0x057F0034, (q31_t)0x7FE1C76B, (q31_t)0x051A8E5C,
	(q31_t)0x7FE5F108, (q31_t)0x04B6195D, (q31_t)0x7FE9CBC0,
	(q31_t)0x0451A176, (q31_t)0x7FED5790, (q31_t)0x03ED26E6,
	(q31_t)0x7FF09477, (q31_t)0x0388A9E9, (q31_t)0x7FF38273,
	(q31_t)0x03242ABF, (q31_t)0x7FF62182, (q31_t)0x02BFA9A4,
	(q31_t)0x7FF871A1, (q31_t)0x025B26D7, (q31_t)0x7FFA72D1,
	(q31_t)0x01F6A296, (q31_t)0x7FFC250F, (q31_t)0x01921D1F,
	(q31_t)0x7FFD885A, (q31_t)0x012D96B0, (q31_t)0x7FFE9CB2,
	(q31_t)0x00C90F88, (q31_t)0x7FFF6216, (q31_t)0x006487E3,
	(q31_t)0x7FFFD885, (q31_t)0x00000000, (q31_t)0x7FFFFFFF,
	(q31_t)0xFF9B781D, (q31_t)0x7FFFD885, (q31_t)0xFF36F078,
	(q31_t)0x7FFF6216, (q31_t)0xFED2694F, (q31_t)0x7FFE9CB2,
	(q31_t)0xFE6DE2E0, (q31_t)0x7FFD885A, (q31_t)0xFE095D69,
	(q31_t)0x7FFC250F, (q31_t)0xFDA4D928, (q31_t)0x7FFA72D1,
	(q31_t)0xFD40565B, (q31_t)0x7FF871A1, (q31_t)0xFCDBD541,
	(q31_t)0x7FF62182, (q31_t)0xFC775616, (q31_t)0x7FF38273,
	(q31_t)0xFC12D919, (q31_t)0x7FF09477, (q31_t)0xFBAE5E89,
	(q31_t)0x7FED5790, (q31_t)0xFB49E6A2, (q31_t)0x7FE9CBC0,
	(q31_t)0xFAE571A4, (q31_t)0x7FE5F108, (q31_t)0xFA80FFCB,
	(q31_t)0x7FE1C76B, (q31_t)0xFA1C9156, (q31_t)0x7FDD4EEC,
	(q31_t)0xF9B82683, (q31_t)0x7FD8878D, (q31_t)0xF953BF90,
	(q31_t)0x7FD37152, (q31_t)0xF8EF5CBB, (q31_t)0x7FCE0C3E,
	(q31_t)0xF88AFE41, (q31_t)0x7FC85853, (q31_t)0xF826A461,
	(q31_t)0x7FC25596, (q31_t)0xF7C24F58, (q31_t)0x7FBC040A,
	(q31_t)0xF75DFF65, (q31_t)0x7FB563B2, (q31_t)0xF6F9B4C5,
	(q31_t)0x7FAE7494, (q31_t)0xF6956FB6, (q31_t)0x7FA736B4,
	(q31_t)0xF6313076, (q31_t)0x7F9FAA15, (q31_t)0xF5CCF743,
	(q31_t)0x7F97CEBC, (q31_t)0xF568C45A, (q31_t)0x7F8FA4AF,
	(q31_t)0xF50497FA, (q31_t)0x7F872BF3, (q31_t)0xF4A07260,
	(q31_t)0x7F7E648B, (q31_t)0xF43C53CA, (q31_t)0x7F754E7F,
	(q31_t)0xF3D83C76, (q31_t)0x7F6BE9D4, (q31_t)0xF3742CA1,
	(q31_t)0x7F62368F, (q31_t)0xF310248A, (q31_t)0x7F5834B6,
	(q31_t)0xF2AC246D, (q31_t)0x7F4DE450, (q31_t)0xF2482C89,
	(q31_t)0x7F434563, (q31_t)0xF1E43D1C, (q31_t)0x7F3857F5,
	(q31_t)0xF1805662, (q31_t)0x7F2D1C0E, (q31_t)0xF11C789A,
	(q31_t)0x7F2191B4, (q31_t)0xF0B8A401, (q31_t)0x7F15B8EE,
	(q31_t)0xF054D8D4, (q31_t)0x7F0991C3, (q31_t)0xEFF11752,
	(q31_t)0x7EFD1C3C, (q31_t)0xEF8D5FB8, (q31_t)0x7EF0585F,
	(q31_t)0xEF29B243, (q31_t)0x7EE34635, (q31_t)0xEEC60F31,
	(q31_t)0x7ED5E5C6, (q31_t)0xEE6276BF, (q31_t)0x7EC8371A,
	(q31_t)0xEDFEE92B, (q31_t)0x7EBA3A39, (q31_t)0xED9B66B2,
	(q31_t)0x7EABEF2C, (q31_t)0xED37EF91, (q31_t)0x7E9D55FC,
	(q31_t)0xECD48406, (q31_t)0x7E8E6EB1, (q31_t)0xEC71244F,
	(q31_t)0x7E7F3956, (q31_t)0xEC0DD0A8, (q31_t)0x7E6FB5F3,
	(q31_t)0xEBAA894E, (q31_t)0x7E5FE493, (q31_t)0xEB474E80,
	(q31_t)0x7E4FC53E, (q31_t)0xEAE4207A, (q31_t)0x7E3F57FE,
	(q31_t)0xEA80FF79, (q31_t)0x7E2E9CDF, (q31_t)0xEA1DEBBB,
	(q31_t)0x7E1D93E9, (q31_t)0xE9BAE57C, (q31_t)0x7E0C3D29,
	(q31_t)0xE957ECFB, (q31_t)0x7DFA98A7, (q31_t)0xE8F50273,
	(q31_t)0x7DE8A670, (q31_t)0xE8922621, (q31_t)0x7DD6668E,
	(q31_t)0xE82F5844, (q31_t)0x7DC3D90D, (q31_t)0xE7CC9917,
	(q31_t)0x7DB0FDF7, (q31_t)0xE769E8D8, (q31_t)0x7D9DD55A,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xE6A4B616,
	(q31_t)0x7D769BB5, (q31_t)0xE642340D, (q31_t)0x7D628AC5,
	(q31_t)0xE5DFC1E4, (q31_t)0x7D4E2C7E, (q31_t)0xE57D5FDA,
	(q31_t)0x7D3980EC, (q31_t)0xE51B0E2A, (q31_t)0x7D24881A,
	(q31_t)0xE4B8CD10, (q31_t)0x7D0F4218, (q31_t)0xE4569CCB,
	(q31_t)0x7CF9AEF0, (q31_t)0xE3F47D95, (q31_t)0x7CE3CEB1,
	(q31_t)0xE3926FAC, (q31_t)0x7CCDA168, (q31_t)0xE330734C,
	(q31_t)0x7CB72724, (q31_t)0xE2CE88B2, (q31_t)0x7CA05FF1,
	(q31_t)0xE26CB01A, (q31_t)0x7C894BDD, (q31_t)0xE20AE9C1,
	(q31_t)0x7C71EAF8, (q31_t)0xE1A935E1, (q31_t)0x7C5A3D4F,
	(q31_t)0xE14794B9, (q31_t)0x7C4242F2, (q31_t)0xE0E60684,
	(q31_t)0x7C29FBEE, (q31_t)0xE0848B7F, (q31_t)0x7C116853,
	(q31_t)0xE02323E5, (q31_t)0x7BF88830, (q31_t)0xDFC1CFF2,
	(q31_t)0x7BDF5B94, (q31_t)0xDF608FE3, (q31_t)0x7BC5E28F,
	(q31_t)0xDEFF63F4, (q31_t)0x7BAC1D31, (q31_t)0xDE9E4C60,
	(q31_t)0x7B920B89, (q31_t)0xDE3D4963, (q31_t)0x7B77ADA8,
	(q31_t)0xDDDC5B3A, (q31_t)0x7B5D039D, (q31_t)0xDD7B8220,
	(q31_t)0x7B420D7A, (q31_t)0xDD1ABE51, (q31_t)0x7B26CB4F,
	(q31_t)0xDCBA1008, (q31_t)0x7B0B3D2C, (q31_t)0xDC597781,
	(q31_t)0x7AEF6323, (q31_t)0xDBF8F4F8, (q31_t)0x7AD33D45,
	(q31_t)0xDB9888A8, (q31_t)0x7AB6CBA3, (q31_t)0xDB3832CD,
	(q31_t)0x7A9A0E4F, (q31_t)0xDAD7F3A2, (q31_t)0x7A7D055B,
	(q31_t)0xDA77CB62, (q31_t)0x7A5FB0D8, (q31_t)0xDA17BA4A,
	(q31_t)0x7A4210D8, (q31_t)0xD9B7C093, (q31_t)0x7A24256E,
	(q31_t)0xD957DE7A, (q31_t)0x7A05EEAD, (q31_t)0xD8F81439,
	(q31_t)0x79E76CA6, (q31_t)0xD898620C, (q31_t)0x79C89F6D,
	(q31_t)0xD838C82D, (q31_t)0x79A98715, (q31_t)0xD7D946D7,
	(q31_t)0x798A23B1, (q31_t)0xD779DE46, (q31_t)0x796A7554,
	(q31_t)0xD71A8EB5, (q31_t)0x794A7C11, (q31_t)0xD6BB585D,
	(q31_t)0x792A37FE, (q31_t)0xD65C3B7B, (q31_t)0x7909A92C,
	(q31_t)0xD5FD3847, (q31_t)0x78E8CFB1, (q31_t)0xD59E4EFE,
	(q31_t)0x78C7ABA1, (q31_t)0xD53F7FDA, (q31_t)0x78A63D10,
	(q31_t)0xD4E0CB14, (q31_t)0x78848413, (q31_t)0xD48230E8,
	(q31_t)0x786280BF, (q31_t)0xD423B190, (q31_t)0x78403328,
	(q31_t)0xD3C54D46, (q31_t)0x781D9B64, (q31_t)0xD3670445,
	(q31_t)0x77FAB988, (q31_t)0xD308D6C6, (q31_t)0x77D78DAA,
	(q31_t)0xD2AAC504, (q31_t)0x77B417DF, (q31_t)0xD24CCF38,
	(q31_t)0x7790583D, (q31_t)0xD1EEF59E, (q31_t)0x776C4EDB,
	(q31_t)0xD191386D, (q31_t)0x7747FBCE, (q31_t)0xD13397E1,
	(q31_t)0x77235F2D, (q31_t)0xD0D61433, (q31_t)0x76FE790E,
	(q31_t)0xD078AD9D, (q31_t)0x76D94988, (q31_t)0xD01B6459,
	(q31_t)0x76B3D0B3, (q31_t)0xCFBE389F, (q31_t)0x768E0EA5,
	(q31_t)0xCF612AAA, (q31_t)0x76680376, (q31_t)0xCF043AB2,
	(q31_t)0x7641AF3C, (q31_t)0xCEA768F2, (q31_t)0x761B1211,
	(q31_t)0xCE4AB5A2, (q31_t)0x75F42C0A, (q31_t)0xCDEE20FC,
	(q31_t)0x75CCFD42, (q31_t)0xCD91AB38, (q31_t)0x75A585CF,
	(q31_t)0xCD355490, (q31_t)0x757DC5CA, (q31_t)0xCCD91D3D,
	(q31_t)0x7555BD4B, (q31_t)0xCC7D0577, (q31_t)0x752D6C6C,
	(q31_t)0xCC210D78, (q31_t)0x7504D345, (q31_t)0xCBC53578,
	(q31_t)0x74DBF1EF, (q31_t)0xCB697DB0, (q31_t)0x74B2C883,
	(q31_t)0xCB0DE658, (q31_t)0x7489571B, (q31_t)0xCAB26FA9,
	(q31_t)0x745F9DD1, (q31_t)0xCA5719DB, (q31_t)0x74359CBD,
	(q31_t)0xC9FBE527, (q31_t)0x740B53FA, (q31_t)0xC9A0D1C4,
	(q31_t)0x73E0C3A3, (q31_t)0xC945DFEC, (q31_t)0x73B5EBD0,
	(q31_t)0xC8EB0FD6, (q31_t)0x738ACC9E, (q31_t)0xC89061BA,
	(q31_t)0x735F6626, (q31_t)0xC835D5D0, (q31_t)0x7333B883,
	(q31_t)0xC7DB6C50, (q31_t)0x7307C3D0, (q31_t)0xC7812571,
	(q31_t)0x72DB8828, (q31_t)0xC727016C, (q31_t)0x72AF05A6,
	(q31_t)0xC6CD0079, (q31_t)0x72823C66, (q31_t)0xC67322CD,
	(q31_t)0x72552C84, (q31_t)0xC61968A2, (q31_t)0x7227D61C,
	(q31_t)0xC5BFD22E, (q31_t)0x71FA3948, (q31_t)0xC5665FA8,
	(q31_t)0x71CC5626, (q31_t)0xC50D1148, (q31_t)0x719E2CD2,
	(q31_t)0xC4B3E746, (q31_t)0x716FBD68, (q31_t)0xC45AE1D7,
	(q31_t)0x71410804, (q31_t)0xC4020132, (q31_t)0x71120CC5,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xC350AF25,
	(q31_t)0x70B34524, (q31_t)0xC2F83E2A, (q31_t)0x708378FE,
	(q31_t)0xC29FF2D4, (q31_t)0x70536771, (q31_t)0xC247CD5A,
	(q31_t)0x70231099, (q31_t)0xC1EFCDF2, (q31_t)0x6FF27496,
	(q31_t)0xC197F4D3, (q31_t)0x6FC19385, (q31_t)0xC1404233,
	(q31_t)0x6F906D84, (q31_t)0xC0E8B648, (q31_t)0x6F5F02B1,
	(q31_t)0xC0915147, (q31_t)0x6F2D532C, (q31_t)0xC03A1368,
	(q31_t)0x6EFB5F12, (q31_t)0xBFE2FCDF, (q31_t)0x6EC92682,
	(q31_t)0xBF8C0DE2, (q31_t)0x6E96A99C, (q31_t)0xBF3546A8,
	(q31_t)0x6E63E87F, (q31_t)0xBEDEA765, (q31_t)0x6E30E349,
	(q31_t)0xBE88304F, (q31_t)0x6DFD9A1B, (q31_t)0xBE31E19B,
	(q31_t)0x6DCA0D14, (q31_t)0xBDDBBB7F, (q31_t)0x6D963C54,
	(q31_t)0xBD85BE2F, (q31_t)0x6D6227FA, (q31_t)0xBD2FE9E1,
	(q31_t)0x6D2DD027, (q31_t)0xBCDA3ECA, (q31_t)0x6CF934FB,
	(q31_t)0xBC84BD1E, (q31_t)0x6CC45697, (q31_t)0xBC2F6513,
	(q31_t)0x6C8F351C, (q31_t)0xBBDA36DC, (q31_t)0x6C59D0A9,
	(q31_t)0xBB8532AF, (q31_t)0x6C242960, (q31_t)0xBB3058C0,
	(q31_t)0x6BEE3F62, (q31_t)0xBADBA943, (q31_t)0x6BB812D0,
	(q31_t)0xBA87246C, (q31_t)0x6B81A3CD, (q31_t)0xBA32CA70,
	(q31_t)0x6B4AF278, (q31_t)0xB9DE9B83, (q31_t)0x6B13FEF5,
	(q31_t)0xB98A97D8, (q31_t)0x6ADCC964, (q31_t)0xB936BFA3,
	(q31_t)0x6AA551E8, (q31_t)0xB8E31319, (q31_t)0x6A6D98A4,
	(q31_t)0xB88F926C, (q31_t)0x6A359DB9, (q31_t)0xB83C3DD1,
	(q31_t)0x69FD614A, (q31_t)0xB7E9157A, (q31_t)0x69C4E37A,
	(q31_t)0xB796199B, (q31_t)0x698C246C, (q31_t)0xB7434A67,
	(q31_t)0x69532442, (q31_t)0xB6F0A811, (q31_t)0x6919E320,
	(q31_t)0xB69E32CD, (q31_t)0x68E06129, (q31_t)0xB64BEACC,
	(q31_t)0x68A69E81, (q31_t)0xB5F9D042, (q31_t)0x686C9B4B,
	(q31_t)0xB5A7E362, (q31_t)0x683257AA, (q31_t)0xB556245E,
	(q31_t)0x67F7D3C4, (q31_t)0xB5049368, (q31_t)0x67BD0FBC,
	(q31_t)0xB4B330B2, (q31_t)0x67820BB6, (q31_t)0xB461FC70,
	(q31_t)0x6746C7D7, (q31_t)0xB410F6D2, (q31_t)0x670B4443,
	(q31_t)0xB3C0200C, (q31_t)0x66CF811F, (q31_t)0xB36F784E,
	(q31_t)0x66937E90, (q31_t)0xB31EFFCB, (q31_t)0x66573CBB,
	(q31_t)0xB2CEB6B5, (q31_t)0x661ABBC5, (q31_t)0xB27E9D3B,
	(q31_t)0x65DDFBD3, (q31_t)0xB22EB392, (q31_t)0x65A0FD0B,
	(q31_t)0xB1DEF9E8, (q31_t)0x6563BF92, (q31_t)0xB18F7070,
	(q31_t)0x6526438E, (q31_t)0xB140175B, (q31_t)0x64E88926,
	(q31_t)0xB0F0EEDA, (q31_t)0x64AA907F, (q31_t)0xB0A1F71C,
	(q31_t)0x646C59BF, (q31_t)0xB0533055, (q31_t)0x642DE50D,
	(q31_t)0xB0049AB2, (q31_t)0x63EF328F, (q31_t)0xAFB63667,
	(q31_t)0x63B0426D, (q31_t)0xAF6803A1, (q31_t)0x637114CC,
	(q31_t)0xAF1A0293, (q31_t)0x6331A9D4, (q31_t)0xAECC336B,
	(q31_t)0x62F201AC, (q31_t)0xAE7E965B, (q31_t)0x62B21C7B,
	(q31_t)0xAE312B91, (q31_t)0x6271FA69, (q31_t)0xADE3F33E,
	(q31_t)0x62319B9D, (q31_t)0xAD96ED91, (q31_t)0x61F1003E,
	(q31_t)0xAD4A1ABA, (q31_t)0x61B02876, (q31_t)0xACFD7AE8,
	(q31_t)0x616F146B, (q31_t)0xACB10E4A, (q31_t)0x612DC446,
	(q31_t)0xAC64D510, (q31_t)0x60EC3830, (q31_t)0xAC18CF68,
	(q31_t)0x60AA704F, (q31_t)0xABCCFD82, (q31_t)0x60686CCE,
	(q31_t)0xAB815F8C, (q31_t)0x60262DD5, (q31_t)0xAB35F5B5,
	(q31_t)0x5FE3B38D, (q31_t)0xAAEAC02B, (q31_t)0x5FA0FE1E,
	(q31_t)0xAA9FBF1D, (q31_t)0x5F5E0DB3, (q31_t)0xAA54F2B9,
	(q31_t)0x5F1AE273, (q31_t)0xAA0A5B2D, (q31_t)0x5ED77C89,
	(q31_t)0xA9BFF8A8, (q31_t)0x5E93DC1F, (q31_t)0xA975CB56,
	(q31_t)0x5E50015D, (q31_t)0xA92BD366, (q31_t)0x5E0BEC6E,
	(q31_t)0xA8E21106, (q31_t)0x5DC79D7C, (q31_t)0xA8988463,
	(q31_t)0x5D8314B0, (q31_t)0xA84F2DA9, (q31_t)0x5D3E5236,
	(q31_t)0xA8060D08, (q31_t)0x5CF95638, (q31_t)0xA7BD22AB,
	(q31_t)0x5CB420DF, (q31_t)0xA7746EC0, (q31_t)0x5C6EB258,
	(q31_t)0xA72BF173, (q31_t)0x5C290ACC, (q31_t)0xA6E3AAF2,
	(q31_t)0x5BE32A67, (q31_t)0xA69B9B68, (q31_t)0x5B9D1153,
	(q31_t)0xA653C302, (q31_t)0x5B56BFBD, (q31_t)0xA60C21ED,
	(q31_t)0x5B1035CF, (q31_t)0xA5C4B855, (q31_t)0x5AC973B4,
	(q31_t)0xA57D8666, (q31_t)0x5A82799A, (q31_t)0xA5368C4B,
	(q31_t)0x5A3B47AA, (q31_t)0xA4EFCA31, (q31_t)0x59F3DE12,
	(q31_t)0xA4A94042, (q31_t)0x59AC3CFD, (q31_t)0xA462EEAC,
	(q31_t)0x59646497, (q31_t)0xA41CD598, (q31_t)0x591C550E,
	(q31_t)0xA3D6F533, (q31_t)0x58D40E8C, (q31_t)0xA3914DA7,
	(q31_t)0x588B913F, (q31_t)0xA34BDF20, (q31_t)0x5842DD54,
	(q31_t)0xA306A9C7, (q31_t)0x57F9F2F7, (q31_t)0xA2C1ADC9,
	(q31_t)0x57B0D256, (q31_t)0xA27CEB4F, (q31_t)0x57677B9D,
	(q31_t)0xA2386283, (q31_t)0x571DEEF9, (q31_t)0xA1F41391,
	(q31_t)0x56D42C99, (q31_t)0xA1AFFEA2, (q31_t)0x568A34A9,
	(q31_t)0xA16C23E1, (q31_t)0x56400757, (q31_t)0xA1288376,
	(q31_t)0x55F5A4D2, (q31_t)0xA0E51D8C, (q31_t)0x55AB0D46,
	(q31_t)0xA0A1F24C, (q31_t)0x556040E2, (q31_t)0xA05F01E1,
	(q31_t)0x55153FD4, (q31_t)0xA01C4C72, (q31_t)0x54CA0A4A,
	(q31_t)0x9FD9D22A, (q31_t)0x547EA073, (q31_t)0x9F979331,
	(q31_t)0x5433027D, (q31_t)0x9F558FB0, (q31_t)0x53E73097,
	(q31_t)0x9F13C7D0, (q31_t)0x539B2AEF, (q31_t)0x9ED23BB9,
	(q31_t)0x534EF1B5, (q31_t)0x9E90EB94, (q31_t)0x53028517,
	(q31_t)0x9E4FD789, (q31_t)0x52B5E545, (q31_t)0x9E0EFFC1,
	(q31_t)0x5269126E, (q31_t)0x9DCE6462, (q31_t)0x521C0CC1,
	(q31_t)0x9D8E0596, (q31_t)0x51CED46E, (q31_t)0x9D4DE384,
	(q31_t)0x518169A4, (q31_t)0x9D0DFE53, (q31_t)0x5133CC94,
	(q31_t)0x9CCE562B, (q31_t)0x50E5FD6C, (q31_t)0x9C8EEB33,
	(q31_t)0x5097FC5E, (q31_t)0x9C4FBD92, (q31_t)0x5049C999,
	(q31_t)0x9C10CD70, (q31_t)0x4FFB654D, (q31_t)0x9BD21AF2,
	(q31_t)0x4FACCFAB, (q31_t)0x9B93A640, (q31_t)0x4F5E08E3,
	(q31_t)0x9B556F80, (q31_t)0x4F0F1126, (q31_t)0x9B1776D9,
	(q31_t)0x4EBFE8A4, (q31_t)0x9AD9BC71, (q31_t)0x4E708F8F,
	(q31_t)0x9A9C406D, (q31_t)0x4E210617, (q31_t)0x9A5F02F5,
	(q31_t)0x4DD14C6E, (q31_t)0x9A22042C, (q31_t)0x4D8162C4,
	(q31_t)0x99E5443A, (q31_t)0x4D31494B, (q31_t)0x99A8C344,
	(q31_t)0x4CE10034, (q31_t)0x996C816F, (q31_t)0x4C9087B1,
	(q31_t)0x99307EE0, (q31_t)0x4C3FDFF3, (q31_t)0x98F4BBBC,
	(q31_t)0x4BEF092D, (q31_t)0x98B93828, (q31_t)0x4B9E038F,
	(q31_t)0x987DF449, (q31_t)0x4B4CCF4D, (q31_t)0x9842F043,
	(q31_t)0x4AFB6C97, (q31_t)0x98082C3B, (q31_t)0x4AA9DBA1,
	(q31_t)0x97CDA855, (q31_t)0x4A581C9D, (q31_t)0x979364B5,
	(q31_t)0x4A062FBD, (q31_t)0x9759617E, (q31_t)0x49B41533,
	(q31_t)0x971F9ED6, (q31_t)0x4961CD32, (q31_t)0x96E61CDF,
	(q31_t)0x490F57EE, (q31_t)0x96ACDBBD, (q31_t)0x48BCB598,
	(q31_t)0x9673DB94, (q31_t)0x4869E664, (q31_t)0x963B1C85,
	(q31_t)0x4816EA85, (q31_t)0x96029EB5, (q31_t)0x47C3C22E,
	(q31_t)0x95CA6246, (q31_t)0x47706D93, (q31_t)0x9592675B,
	(q31_t)0x471CECE6, (q31_t)0x955AAE17, (q31_t)0x46C9405C,
	(q31_t)0x9523369B, (q31_t)0x46756827, (q31_t)0x94EC010B,
	(q31_t)0x4621647C, (q31_t)0x94B50D87, (q31_t)0x45CD358F,
	(q31_t)0x947E5C32, (q31_t)0x4578DB93, (q31_t)0x9447ED2F,
	(q31_t)0x452456BC, (q31_t)0x9411C09D, (q31_t)0x44CFA73F,
	(q31_t)0x93DBD69F, (q31_t)0x447ACD50, (q31_t)0x93A62F56,
	(q31_t)0x4425C923, (q31_t)0x9370CAE4, (q31_t)0x43D09AEC,
	(q31_t)0x933BA968, (q31_t)0x437B42E1, (q31_t)0x9306CB04,
	(q31_t)0x4325C135, (q31_t)0x92D22FD8, (q31_t)0x42D0161E,
	(q31_t)0x929DD805, (q31_t)0x427A41D0, (q31_t)0x9269C3AC,
	(q31_t)0x42244480, (q31_t)0x9235F2EB, (q31_t)0x41CE1E64,
	(q31_t)0x920265E4, (q31_t)0x4177CFB0, (q31_t)0x91CF1CB6,
	(q31_t)0x4121589A, (q31_t)0x919C1780, (q31_t)0x40CAB957,
	(q31_t)0x91695663, (q31_t)0x4073F21D, (q31_t)0x9136D97D,
	(q31_t)0x401D0320, (q31_t)0x9104A0ED, (q31_t)0x3FC5EC97,
	(q31_t)0x90D2ACD3, (q31_t)0x3F6EAEB8, (q31_t)0x90A0FD4E,
	(q31_t)0x3F1749B7, (q31_t)0x906F927B, (q31_t)0x3EBFBDCC,
	(q31_t)0x903E6C7A, (q31_t)0x3E680B2C, (q31_t)0x900D8B69,
	(q31_t)0x3E10320D, (q31_t)0x8FDCEF66, (q31_t)0x3DB832A5,
	(q31_t)0x8FAC988E, (q31_t)0x3D600D2B, (q31_t)0x8F7C8701,
	(q31_t)0x3D07C1D5, (q31_t)0x8F4CBADB, (q31_t)0x3CAF50DA,
	(q31_t)0x8F1D343A, (q31_t)0x3C56BA70, (q31_t)0x8EEDF33B,
	(q31_t)0x3BFDFECD, (q31_t)0x8EBEF7FB, (q31_t)0x3BA51E29,
	(q31_t)0x8E904298, (q31_t)0x3B4C18BA, (q31_t)0x8E61D32D,
	(q31_t)0x3AF2EEB7, (q31_t)0x8E33A9D9, (q31_t)0x3A99A057,
	(q31_t)0x8E05C6B7, (q31_t)0x3A402DD1, (q31_t)0x8DD829E4,
	(q31_t)0x39E6975D, (q31_t)0x8DAAD37B, (q31_t)0x398CDD32,
	(q31_t)0x8D7DC399, (q31_t)0x3932FF87, (q31_t)0x8D50FA59,
	(q31_t)0x38D8FE93, (q31_t)0x8D2477D8, (q31_t)0x387EDA8E,
	(q31_t)0x8CF83C30, (q31_t)0x382493B0, (q31_t)0x8CCC477D,
	(q31_t)0x37CA2A30, (q31_t)0x8CA099D9, (q31_t)0x376F9E46,
	(q31_t)0x8C753361, (q31_t)0x3714F02A, (q31_t)0x8C4A142F,
	(q31_t)0x36BA2013, (q31_t)0x8C1F3C5C, (q31_t)0x365F2E3B,
	(q31_t)0x8BF4AC05, (q31_t)0x36041AD9, (q31_t)0x8BCA6342,
	(q31_t)0x35A8E624, (q31_t)0x8BA0622F, (q31_t)0x354D9056,
	(q31_t)0x8B76A8E4, (q31_t)0x34F219A7, (q31_t)0x8B4D377C,
	(q31_t)0x3496824F, (q31_t)0x8B240E10, (q31_t)0x343ACA87,
	(q31_t)0x8AFB2CBA, (q31_t)0x33DEF287, (q31_t)0x8AD29393,
	(q31_t)0x3382FA88, (q31_t)0x8AAA42B4, (q31_t)0x3326E2C2,
	(q31_t)0x8A823A35, (q31_t)0x32CAAB6F, (q31_t)0x8A5A7A30,
	(q31_t)0x326E54C7, (q31_t)0x8A3302BD, (q31_t)0x3211DF03,
	(q31_t)0x8A0BD3F5, (q31_t)0x31B54A5D, (q31_t)0x89E4EDEE,
	(q31_t)0x3158970D, (q31_t)0x89BE50C3, (q31_t)0x30FBC54D,
	(q31_t)0x8997FC89, (q31_t)0x309ED555, (q31_t)0x8971F15A,
	(q31_t)0x3041C760, (q31_t)0x894C2F4C, (q31_t)0x2FE49BA6,
	(q31_t)0x8926B677, (q31_t)0x2F875262, (q31_t)0x890186F1,
	(q31_t)0x2F29EBCC, (q31_t)0x88DCA0D3, (q31_t)0x2ECC681E,
	(q31_t)0x88B80431, (q31_t)0x2E6EC792, (q31_t)0x8893B124,
	(q31_t)0x2E110A62, (q31_t)0x886FA7C2, (q31_t)0x2DB330C7,
	(q31_t)0x884BE820, (q31_t)0x2D553AFB, (q31_t)0x88287255,
	(q31_t)0x2CF72939, (q31_t)0x88054677, (q31_t)0x2C98FBBA,
	(q31_t)0x87E2649B, (q31_t)0x2C3AB2B9, (q31_t)0x87BFCCD7,
	(q31_t)0x2BDC4E6F, (q31_t)0x879D7F40, (q31_t)0x2B7DCF17,
	(q31_t)0x877B7BEC, (q31_t)0x2B1F34EB, (q31_t)0x8759C2EF,
	(q31_t)0x2AC08025, (q31_t)0x8738545E, (q31_t)0x2A61B101,
	(q31_t)0x8717304E, (q31_t)0x2A02C7B8, (q31_t)0x86F656D3,
	(q31_t)0x29A3C484, (q31_t)0x86D5C802, (q31_t)0x2944A7A2,
	(q31_t)0x86B583EE, (q31_t)0x28E5714A, (q31_t)0x86958AAB,
	(q31_t)0x288621B9, (q31_t)0x8675DC4E, (q31_t)0x2826B928,
	(q31_t)0x865678EA, (q31_t)0x27C737D2, (q31_t)0x86376092,
	(q31_t)0x27679DF4, (q31_t)0x86189359, (q31_t)0x2707EBC6,
	(q31_t)0x85FA1152, (q31_t)0x26A82185, (q31_t)0x85DBDA91,
	(q31_t)0x26483F6C, (q31_t)0x85BDEF27, (q31_t)0x25E845B5,
	(q31_t)0x85A04F28, (q31_t)0x2588349D, (q31_t)0x8582FAA4,
	(q31_t)0x25280C5D, (q31_t)0x8565F1B0, (q31_t)0x24C7CD32,
	(q31_t)0x8549345C, (q31_t)0x24677757, (q31_t)0x852CC2BA,
	(q31_t)0x24070B07, (q31_t)0x85109CDC, (q31_t)0x23A6887E,
	(q31_t)0x84F4C2D3, (q31_t)0x2345EFF7, (q31_t)0x84D934B0,
	(q31_t)0x22E541AE, (q31_t)0x84BDF285, (q31_t)0x22847DDF,
	(q31_t)0x84A2FC62, (q31_t)0x2223A4C5, (q31_t)0x84885257,
	(q31_t)0x21C2B69C, (q31_t)0x846DF476, (q31_t)0x2161B39F,
	(q31_t)0x8453E2CE, (q31_t)0x21009C0B, (q31_t)0x843A1D70,
	(q31_t)0x209F701C, (q31_t)0x8420A46B, (q31_t)0x203E300D,
	(q31_t)0x840777CF, (q31_t)0x1FDCDC1A, (q31_t)0x83EE97AC,
	(q31_t)0x1F7B7480, (q31_t)0x83D60411, (q31_t)0x1F19F97B,
	(q31_t)0x83BDBD0D, (q31_t)0x1EB86B46, (q31_t)0x83A5C2B0,
	(q31_t)0x1E56CA1E, (q31_t)0x838E1507, (q31_t)0x1DF5163F,
	(q31_t)0x8376B422, (q31_t)0x1D934FE5, (q31_t)0x835FA00E,
	(q31_t)0x1D31774D, (q31_t)0x8348D8DB, (q31_t)0x1CCF8CB3,
	(q31_t)0x83325E97, (q31_t)0x1C6D9053, (q31_t)0x831C314E,
	(q31_t)0x1C0B826A, (q31_t)0x8306510F, (q31_t)0x1BA96334,
	(q31_t)0x82F0BDE8, (q31_t)0x1B4732EF, (q31_t)0x82DB77E5,
	(q31_t)0x1AE4F1D6, (q31_t)0x82C67F13, (q31_t)0x1A82A025,
	(q31_t)0x82B1D381, (q31_t)0x1A203E1B, (q31_t)0x829D753A,
	(q31_t)0x19BDCBF2, (q31_t)0x8289644A, (q31_t)0x195B49E9,
	(q31_t)0x8275A0C0, (q31_t)0x18F8B83C, (q31_t)0x82622AA5,
	(q31_t)0x18961727, (q31_t)0x824F0208, (q31_t)0x183366E8,
	(q31_t)0x823C26F2, (q31_t)0x17D0A7BB, (q31_t)0x82299971,
	(q31_t)0x176DD9DE, (q31_t)0x8217598F, (q31_t)0x170AFD8D,
	(q31_t)0x82056758, (q31_t)0x16A81305, (q31_t)0x81F3C2D7,
	(q31_t)0x16451A83, (q31_t)0x81E26C16, (q31_t)0x15E21444,
	(q31_t)0x81D16320, (q31_t)0x157F0086, (q31_t)0x81C0A801,
	(q31_t)0x151BDF85, (q31_t)0x81B03AC1, (q31_t)0x14B8B17F,
	(q31_t)0x81A01B6C, (q31_t)0x145576B1, (q31_t)0x81904A0C,
	(q31_t)0x13F22F57, (q31_t)0x8180C6A9, (q31_t)0x138EDBB0,
	(q31_t)0x8171914E, (q31_t)0x132B7BF9, (q31_t)0x8162AA03,
	(q31_t)0x12C8106E, (q31_t)0x815410D3, (q31_t)0x1264994E,
	(q31_t)0x8145C5C6, (q31_t)0x120116D4, (q31_t)0x8137C8E6,
	(q31_t)0x119D8940, (q31_t)0x812A1A39, (q31_t)0x1139F0CE,
	(q31_t)0x811CB9CA, (q31_t)0x10D64DBC, (q31_t)0x810FA7A0,
	(q31_t)0x1072A047, (q31_t)0x8102E3C3, (q31_t)0x100EE8AD,
	(q31_t)0x80F66E3C, (q31_t)0x0FAB272B, (q31_t)0x80EA4712,
	(q31_t)0x0F475BFE, (q31_t)0x80DE6E4C, (q31_t)0x0EE38765,
	(q31_t)0x80D2E3F1, (q31_t)0x0E7FA99D, (q31_t)0x80C7A80A,
	(q31_t)0x0E1BC2E3, (q31_t)0x80BCBA9C, (q31_t)0x0DB7D376,
	(q31_t)0x80B21BAF, (q31_t)0x0D53DB92, (q31_t)0x80A7CB49,
	(q31_t)0x0CEFDB75, (q31_t)0x809DC970, (q31_t)0x0C8BD35E,
	(q31_t)0x8094162B, (q31_t)0x0C27C389, (q31_t)0x808AB180,
	(q31_t)0x0BC3AC35, (q31_t)0x80819B74, (q31_t)0x0B5F8D9F,
	(q31_t)0x8078D40D, (q31_t)0x0AFB6805, (q31_t)0x80705B50,
	(q31_t)0x0A973BA5, (q31_t)0x80683143, (q31_t)0x0A3308BC,
	(q31_t)0x806055EA, (q31_t)0x09CECF89, (q31_t)0x8058C94C,
	(q31_t)0x096A9049, (q31_t)0x80518B6B, (q31_t)0x09064B3A,
	(q31_t)0x804A9C4D, (q31_t)0x08A2009A, (q31_t)0x8043FBF6,
	(q31_t)0x083DB0A7, (q31_t)0x803DAA69, (q31_t)0x07D95B9E,
	(q31_t)0x8037A7AC, (q31_t)0x077501BE, (q31_t)0x8031F3C1,
	(q31_t)0x0710A344, (q31_t)0x802C8EAD, (q31_t)0x06AC406F,
	(q31_t)0x80277872, (q31_t)0x0647D97C, (q31_t)0x8022B113,
	(q31_t)0x05E36EA9, (q31_t)0x801E3894, (q31_t)0x057F0034,
	(q31_t)0x801A0EF7, (q31_t)0x051A8E5C, (q31_t)0x80163440,
	(q31_t)0x04B6195D, (q31_t)0x8012A86F, (q31_t)0x0451A176,
	(q31_t)0x800F6B88, (q31_t)0x03ED26E6, (q31_t)0x800C7D8C,
	(q31_t)0x0388A9E9, (q31_t)0x8009DE7D, (q31_t)0x03242ABF,
	(q31_t)0x80078E5E, (q31_t)0x02BFA9A4, (q31_t)0x80058D2E,
	(q31_t)0x025B26D7, (q31_t)0x8003DAF0, (q31_t)0x01F6A296,
	(q31_t)0x800277A5, (q31_t)0x01921D1F, (q31_t)0x8001634D,
	(q31_t)0x012D96B0, (q31_t)0x80009DE9, (q31_t)0x00C90F88,
	(q31_t)0x8000277A, (q31_t)0x006487E3, (q31_t)0x80000000,
	(q31_t)0x00000000, (q31_t)0x8000277A, (q31_t)0xFF9B781D,
	(q31_t)0x80009DE9, (q31_t)0xFF36F078, (q31_t)0x8001634D,
	(q31_t)0xFED2694F, (q31_t)0x800277A5, (q31_t)0xFE6DE2E0,
	(q31_t)0x8003DAF0, (q31_t)0xFE095D69, (q31_t)0x80058D2E,
	(q31_t)0xFDA4D928, (q31_t)0x80078E5E, (q31_t)0xFD40565B,
	(q31_t)0x8009DE7D, (q31_t)0xFCDBD541, (q31_t)0x800C7D8C,
	(q31_t)0xFC775616, (q31_t)0x800F6B88, (q31_t)0xFC12D919,
	(q31_t)0x8012A86F, (q31_t)0xFBAE5E89, (q31_t)0x80163440,
	(q31_t)0xFB49E6A2, (q31_t)0x801A0EF7, (q31_t)0xFAE571A4,
	(q31_t)0x801E3894, (q31_t)0xFA80FFCB, (q31_t)0x8022B113,
	(q31_t)0xFA1C9156, (q31_t)0x80277872, (q31_t)0xF9B82683,
	(q31_t)0x802C8EAD, (q31_t)0xF953BF90, (q31_t)0x8031F3C1,
	(q31_t)0xF8EF5CBB, (q31_t)0x8037A7AC, (q31_t)0xF88AFE41,
	(q31_t)0x803DAA69, (q31_t)0xF826A461, (q31_t)0x8043FBF6,
	(q31_t)0xF7C24F58, (q31_t)0x804A9C4D, (q31_t)0xF75DFF65,
	(q31_t)0x80518B6B, (q31_t)0xF6F9B4C5, (q31_t)0x8058C94C,
	(q31_t)0xF6956FB6, (q31_t)0x806055EA, (q31_t)0xF6313076,
	(q31_t)0x80683143, (q31_t)0xF5CCF743, (q31_t)0x80705B50,
	(q31_t)0xF568C45A, (q31_t)0x8078D40D, (q31_t)0xF50497FA,
	(q31_t)0x80819B74, (q31_t)0xF4A07260, (q31_t)0x808AB180,
	(q31_t)0xF43C53CA, (q31_t)0x8094162B, (q31_t)0xF3D83C76,
	(q31_t)0x809DC970, (q31_t)0xF3742CA1, (q31_t)0x80A7CB49,
	(q31_t)0xF310248A, (q31_t)0x80B21BAF, (q31_t)0xF2AC246D,
	(q31_t)0x80BCBA9C, (q31_t)0xF2482C89, (q31_t)0x80C7A80A,
	(q31_t)0xF1E43D1C, (q31_t)0x80D2E3F1, (q31_t)0xF1805662,
	(q31_t)0x80DE6E4C, (q31_t)0xF11C789A, (q31_t)0x80EA4712,
	(q31_t)0xF0B8A401, (q31_t)0x80F66E3C, (q31_t)0xF054D8D4,
	(q31_t)0x8102E3C3, (q31_t)0xEFF11752, (q31_t)0x810FA7A0,
	(q31_t)0xEF8D5FB8, (q31_t)0x811CB9CA, (q31_t)0xEF29B243,
	(q31_t)0x812A1A39, (q31_t)0xEEC60F31, (q31_t)0x8137C8E6,
	(q31_t)0xEE6276BF, (q31_t)0x8145C5C6, (q31_t)0xEDFEE92B,
	(q31_t)0x815410D3, (q31_t)0xED9B66B2, (q31_t)0x8162AA03,
	(q31_t)0xED37EF91, (q31_t)0x8171914E, (q31_t)0xECD48406,
	(q31_t)0x8180C6A9, (q31_t)0xEC71244F, (q31_t)0x81904A0C,
	(q31_t)0xEC0DD0A8, (q31_t)0x81A01B6C, (q31_t)0xEBAA894E,
	(q31_t)0x81B03AC1, (q31_t)0xEB474E80, (q31_t)0x81C0A801,
	(q31_t)0xEAE4207A, (q31_t)0x81D16320, (q31_t)0xEA80FF79,
	(q31_t)0x81E26C16, (q31_t)0xEA1DEBBB, (q31_t)0x81F3C2D7,
	(q31_t)0xE9BAE57C, (q31_t)0x82056758, (q31_t)0xE957ECFB,
	(q31_t)0x8217598F, (q31_t)0xE8F50273, (q31_t)0x82299971,
	(q31_t)0xE8922621, (q31_t)0x823C26F2, (q31_t)0xE82F5844,
	(q31_t)0x824F0208, (q31_t)0xE7CC9917, (q31_t)0x82622AA5,
	(q31_t)0xE769E8D8, (q31_t)0x8275A0C0, (q31_t)0xE70747C3,
	(q31_t)0x8289644A, (q31_t)0xE6A4B616, (q31_t)0x829D753A,
	(q31_t)0xE642340D, (q31_t)0x82B1D381, (q31_t)0xE5DFC1E4,
	(q31_t)0x82C67F13, (q31_t)0xE57D5FDA, (q31_t)0x82DB77E5,
	(q31_t)0xE51B0E2A, (q31_t)0x82F0BDE8, (q31_t)0xE4B8CD10,
	(q31_t)0x8306510F, (q31_t)0xE4569CCB, (q31_t)0x831C314E,
	(q31_t)0xE3F47D95, (q31_t)0x83325E97, (q31_t)0xE3926FAC,
	(q31_t)0x8348D8DB, (q31_t)0xE330734C, (q31_t)0x835FA00E,
	(q31_t)0xE2CE88B2, (q31_t)0x8376B422, (q31_t)0xE26CB01A,
	(q31_t)0x838E1507, (q31_t)0xE20AE9C1, (q31_t)0x83A5C2B0,
	(q31_t)0xE1A935E1, (q31_t)0x83BDBD0D, (q31_t)0xE14794B9,
	(q31_t)0x83D60411, (q31_t)0xE0E60684, (q31_t)0x83EE97AC,
	(q31_t)0xE0848B7F, (q31_t)0x840777CF, (q31_t)0xE02323E5,
	(q31_t)0x8420A46B, (q31_t)0xDFC1CFF2, (q31_t)0x843A1D70,
	(q31_t)0xDF608FE3, (q31_t)0x8453E2CE, (q31_t)0xDEFF63F4,
	(q31_t)0x846DF476, (q31_t)0xDE9E4C60, (q31_t)0x84885257,
	(q31_t)0xDE3D4963, (q31_t)0x84A2FC62, (q31_t)0xDDDC5B3A,
	(q31_t)0x84BDF285, (q31_t)0xDD7B8220, (q31_t)0x84D934B0,
	(q31_t)0xDD1ABE51, (q31_t)0x84F4C2D3, (q31_t)0xDCBA1008,
	(q31_t)0x85109CDC, (q31_t)0xDC597781, (q31_t)0x852CC2BA,
	(q31_t)0xDBF8F4F8, (q31_t)0x8549345C, (q31_t)0xDB9888A8,
	(q31_t)0x8565F1B0, (q31_t)0xDB3832CD, (q31_t)0x8582FAA4,
	(q31_t)0xDAD7F3A2, (q31_t)0x85A04F28, (q31_t)0xDA77CB62,
	(q31_t)0x85BDEF27, (q31_t)0xDA17BA4A, (q31_t)0x85DBDA91,
	(q31_t)0xD9B7C093, (q31_t)0x85FA1152, (q31_t)0xD957DE7A,
	(q31_t)0x86189359, (q31_t)0xD8F81439, (q31_t)0x86376092,
	(q31_t)0xD898620C, (q31_t)0x865678EA, (q31_t)0xD838C82D,
	(q31_t)0x8675DC4E, (q31_t)0xD7D946D7, (q31_t)0x86958AAB,
	(q31_t)0xD779DE46, (q31_t)0x86B583EE, (q31_t)0xD71A8EB5,
	(q31_t)0x86D5C802, (q31_t)0xD6BB585D, (q31_t)0x86F656D3,
	(q31_t)0xD65C3B7B, (q31_t)0x8717304E, (q31_t)0xD5FD3847,
	(q31_t)0x8738545E, (q31_t)0xD59E4EFE, (q31_t)0x8759C2EF,
	(q31_t)0xD53F7FDA, (q31_t)0x877B7BEC, (q31_t)0xD4E0CB14,
	(q31_t)0x879D7F40, (q31_t)0xD48230E8, (q31_t)0x87BFCCD7,
	(q31_t)0xD423B190, (q31_t)0x87E2649B, (q31_t)0xD3C54D46,
	(q31_t)0x88054677, (q31_t)0xD3670445, (q31_t)0x88287255,
	(q31_t)0xD308D6C6, (q31_t)0x884BE820, (q31_t)0xD2AAC504,
	(q31_t)0x886FA7C2, (q31_t)0xD24CCF38, (q31_t)0x8893B124,
	(q31_t)0xD1EEF59E, (q31_t)0x88B80431, (q31_t)0xD191386D,
	(q31_t)0x88DCA0D3, (q31_t)0xD13397E1, (q31_t)0x890186F1,
	(q31_t)0xD0D61433, (q31_t)0x8926B677, (q31_t)0xD078AD9D,
	(q31_t)0x894C2F4C, (q31_t)0xD01B6459, (q31_t)0x8971F15A,
	(q31_t)0xCFBE389F, (q31_t)0x8997FC89, (q31_t)0xCF612AAA,
	(q31_t)0x89BE50C3, (q31_t)0xCF043AB2, (q31_t)0x89E4EDEE,
	(q31_t)0xCEA768F2, (q31_t)0x8A0BD3F5, (q31_t)0xCE4AB5A2,
	(q31_t)0x8A3302BD, (q31_t)0xCDEE20FC, (q31_t)0x8A5A7A30,
	(q31_t)0xCD91AB38, (q31_t)0x8A823A35, (q31_t)0xCD355490,
	(q31_t)0x8AAA42B4, (q31_t)0xCCD91D3D, (q31_t)0x8AD29393,
	(q31_t)0xCC7D0577, (q31_t)0x8AFB2CBA, (q31_t)0xCC210D78,
	(q31_t)0x8B240E10, (q31_t)0xCBC53578, (q31_t)0x8B4D377C,
	(q31_t)0xCB697DB0, (q31_t)0x8B76A8E4, (q31_t)0xCB0DE658,
	(q31_t)0x8BA0622F, (q31_t)0xCAB26FA9, (q31_t)0x8BCA6342,
	(q31_t)0xCA5719DB, (q31_t)0x8BF4AC05, (q31_t)0xC9FBE527,
	(q31_t)0x8C1F3C5C, (q31_t)0xC9A0D1C4, (q31_t)0x8C4A142F,
	(q31_t)0xC945DFEC, (q31_t)0x8C753361, (q31_t)0xC8EB0FD6,
	(q31_t)0x8CA099D9, (q31_t)0xC89061BA, (q31_t)0x8CCC477D,
	(q31_t)0xC835D5D0, (q31_t)0x8CF83C30, (q31_t)0xC7DB6C50,
	(q31_t)0x8D2477D8, (q31_t)0xC7812571, (q31_t)0x8D50FA59,
	(q31_t)0xC727016C, (q31_t)0x8D7DC399, (q31_t)0xC6CD0079,
	(q31_t)0x8DAAD37B, (q31_t)0xC67322CD, (q31_t)0x8DD829E4,
	(q31_t)0xC61968A2, (q31_t)0x8E05C6B7, (q31_t)0xC5BFD22E,
	(q31_t)0x8E33A9D9, (q31_t)0xC5665FA8, (q31_t)0x8E61D32D,
	(q31_t)0xC50D1148, (q31_t)0x8E904298, (q31_t)0xC4B3E746,
	(q31_t)0x8EBEF7FB, (q31_t)0xC45AE1D7, (q31_t)0x8EEDF33B,
	(q31_t)0xC4020132, (q31_t)0x8F1D343A, (q31_t)0xC3A9458F,
	(q31_t)0x8F4CBADB, (q31_t)0xC350AF25, (q31_t)0x8F7C8701,
	(q31_t)0xC2F83E2A, (q31_t)0x8FAC988E, (q31_t)0xC29FF2D4,
	(q31_t)0x8FDCEF66, (q31_t)0xC247CD5A, (q31_t)0x900D8B69,
	(q31_t)0xC1EFCDF2, (q31_t)0x903E6C7A, (q31_t)0xC197F4D3,
	(q31_t)0x906F927B, (q31_t)0xC1404233, (q31_t)0x90A0FD4E,
	(q31_t)0xC0E8B648, (q31_t)0x90D2ACD3, (q31_t)0xC0915147,
	(q31_t)0x9104A0ED, (q31_t)0xC03A1368, (q31_t)0x9136D97D,
	(q31_t)0xBFE2FCDF, (q31_t)0x91695663, (q31_t)0xBF8C0DE2,
	(q31_t)0x919C1780, (q31_t)0xBF3546A8, (q31_t)0x91CF1CB6,
	(q31_t)0xBEDEA765, (q31_t)0x920265E4, (q31_t)0xBE88304F,
	(q31_t)0x9235F2EB, (q31_t)0xBE31E19B, (q31_t)0x9269C3AC,
	(q31_t)0xBDDBBB7F, (q31_t)0x929DD805, (q31_t)0xBD85BE2F,
	(q31_t)0x92D22FD8, (q31_t)0xBD2FE9E1, (q31_t)0x9306CB04,
	(q31_t)0xBCDA3ECA, (q31_t)0x933BA968, (q31_t)0xBC84BD1E,
	(q31_t)0x9370CAE4, (q31_t)0xBC2F6513, (q31_t)0x93A62F56,
	(q31_t)0xBBDA36DC, (q31_t)0x93DBD69F, (q31_t)0xBB8532AF,
	(q31_t)0x9411C09D, (q31_t)0xBB3058C0, (q31_t)0x9447ED2F,
	(q31_t)0xBADBA943, (q31_t)0x947E5C32, (q31_t)0xBA87246C,
	(q31_t)0x94B50D87, (q31_t)0xBA32CA70, (q31_t)0x94EC010B,
	(q31_t)0xB9DE9B83, (q31_t)0x9523369B, (q31_t)0xB98A97D8,
	(q31_t)0x955AAE17, (q31_t)0xB936BFA3, (q31_t)0x9592675B,
	(q31_t)0xB8E31319, (q31_t)0x95CA6246, (q31_t)0xB88F926C,
	(q31_t)0x96029EB5, (q31_t)0xB83C3DD1, (q31_t)0x963B1C85,
	(q31_t)0xB7E9157A, (q31_t)0x9673DB94, (q31_t)0xB796199B,
	(q31_t)0x96ACDBBD, (q31_t)0xB7434A67, (q31_t)0x96E61CDF,
	(q31_t)0xB6F0A811, (q31_t)0x971F9ED6, (q31_t)0xB69E32CD,
	(q31_t)0x9759617E, (q31_t)0xB64BEACC, (q31_t)0x979364B5,
	(q31_t)0xB5F9D042, (q31_t)0x97CDA855, (q31_t)0xB5A7E362,
	(q31_t)0x98082C3B, (q31_t)0xB556245E, (q31_t)0x9842F043,
	(q31_t)0xB5049368, (q31_t)0x987DF449, (q31_t)0xB4B330B2,
	(q31_t)0x98B93828, (q31_t)0xB461FC70, (q31_t)0x98F4BBBC,
	(q31_t)0xB410F6D2, (q31_t)0x99307EE0, (q31_t)0xB3C0200C,
	(q31_t)0x996C816F, (q31_t)0xB36F784E, (q31_t)0x99A8C344,
	(q31_t)0xB31EFFCB, (q31_t)0x99E5443A, (q31_t)0xB2CEB6B5,
	(q31_t)0x9A22042C, (q31_t)0xB27E9D3B, (q31_t)0x9A5F02F5,
	(q31_t)0xB22EB392, (q31_t)0x9A9C406D, (q31_t)0xB1DEF9E8,
	(q31_t)0x9AD9BC71, (q31_t)0xB18F7070, (q31_t)0x9B1776D9,
	(q31_t)0xB140175B, (q31_t)0x9B556F80, (q31_t)0xB0F0EEDA,
	(q31_t)0x9B93A640, (q31_t)0xB0A1F71C, (q31_t)0x9BD21AF2,
	(q31_t)0xB0533055, (q31_t)0x9C10CD70, (q31_t)0xB0049AB2,
	(q31_t)0x9C4FBD92, (q31_t)0xAFB63667, (q31_t)0x9C8EEB33,
	(q31_t)0xAF6803A1, (q31_t)0x9CCE562B, (q31_t)0xAF1A0293,
	(q31_t)0x9D0DFE53, (q31_t)0xAECC336B, (q31_t)0x9D4DE384,
	(q31_t)0xAE7E965B, (q31_t)0x9D8E0596, (q31_t)0xAE312B91,
	(q31_t)0x9DCE6462, (q31_t)0xADE3F33E, (q31_t)0x9E0EFFC1,
	(q31_t)0xAD96ED91, (q31_t)0x9E4FD789, (q31_t)0xAD4A1ABA,
	(q31_t)0x9E90EB94, (q31_t)0xACFD7AE8, (q31_t)0x9ED23BB9,
	(q31_t)0xACB10E4A, (q31_t)0x9F13C7D0, (q31_t)0xAC64D510,
	(q31_t)0x9F558FB0, (q31_t)0xAC18CF68, (q31_t)0x9F979331,
	(q31_t)0xABCCFD82, (q31_t)0x9FD9D22A, (q31_t)0xAB815F8C,
	(q31_t)0xA01C4C72, (q31_t)0xAB35F5B5, (q31_t)0xA05F01E1,
	(q31_t)0xAAEAC02B, (q31_t)0xA0A1F24C, (q31_t)0xAA9FBF1D,
	(q31_t)0xA0E51D8C, (q31_t)0xAA54F2B9, (q31_t)0xA1288376,
	(q31_t)0xAA0A5B2D, (q31_t)0xA16C23E1, (q31_t)0xA9BFF8A8,
	(q31_t)0xA1AFFEA2, (q31_t)0xA975CB56, (q31_t)0xA1F41391,
	(q31_t)0xA92BD366, (q31_t)0xA2386283, (q31_t)0xA8E21106,
	(q31_t)0xA27CEB4F, (q31_t)0xA8988463, (q31_t)0xA2C1ADC9,
	(q31_t)0xA84F2DA9, (q31_t)0xA306A9C7, (q31_t)0xA8060D08,
	(q31_t)0xA34BDF20, (q31_t)0xA7BD22AB, (q31_t)0xA3914DA7,
	(q31_t)0xA7746EC0, (q31_t)0xA3D6F533, (q31_t)0xA72BF173,
	(q31_t)0xA41CD598, (q31_t)0xA6E3AAF2, (q31_t)0xA462EEAC,
	(q31_t)0xA69B9B68, (q31_t)0xA4A94042, (q31_t)0xA653C302,
	(q31_t)0xA4EFCA31, (q31_t)0xA60C21ED, (q31_t)0xA5368C4B,
	(q31_t)0xA5C4B855, (q31_t)0xA57D8666, (q31_t)0xA57D8666,
	(q31_t)0xA5C4B855, (q31_t)0xA5368C4B, (q31_t)0xA60C21ED,
	(q31_t)0xA4EFCA31, (q31_t)0xA653C302, (q31_t)0xA4A94042,
	(q31_t)0xA69B9B68, (q31_t)0xA462EEAC, (q31_t)0xA6E3AAF2,
	(q31_t)0xA41CD598, (q31_t)0xA72BF173, (q31_t)0xA3D6F533,
	(q31_t)0xA7746EC0, (q31_t)0xA3914DA7, (q31_t)0xA7BD22AB,
	(q31_t)0xA34BDF20, (q31_t)0xA8060D08, (q31_t)0xA306A9C7,
	(q31_t)0xA84F2DA9, (q31_t)0xA2C1ADC9, (q31_t)0xA8988463,
	(q31_t)0xA27CEB4F, (q31_t)0xA8E21106, (q31_t)0xA2386283,
	(q31_t)0xA92BD366, (q31_t)0xA1F41391, (q31_t)0xA975CB56,
	(q31_t)0xA1AFFEA2, (q31_t)0xA9BFF8A8, (q31_t)0xA16C23E1,
	(q31_t)0xAA0A5B2D, (q31_t)0xA1288376, (q31_t)0xAA54F2B9,
	(q31_t)0xA0E51D8C, (q31_t)0xAA9FBF1D, (q31_t)0xA0A1F24C,
	(q31_t)0xAAEAC02B, (q31_t)0xA05F01E1, (q31_t)0xAB35F5B5,
	(q31_t)0xA01C4C72, (q31_t)0xAB815F8C, (q31_t)0x9FD9D22A,
	(q31_t)0xABCCFD82, (q31_t)0x9F979331, (q31_t)0xAC18CF68,
	(q31_t)0x9F558FB0, (q31_t)0xAC64D510, (q31_t)0x9F13C7D0,
	(q31_t)0xACB10E4A, (q31_t)0x9ED23BB9, (q31_t)0xACFD7AE8,
	(q31_t)0x9E90EB94, (q31_t)0xAD4A1ABA, (q31_t)0x9E4FD789,
	(q31_t)0xAD96ED91, (q31_t)0x9E0EFFC1, (q31_t)0xADE3F33E,
	(q31_t)0x9DCE6462, (q31_t)0xAE312B91, (q31_t)0x9D8E0596,
	(q31_t)0xAE7E965B, (q31_t)0x9D4DE384, (q31_t)0xAECC336B,
	(q31_t)0x9D0DFE53, (q31_t)0xAF1A0293, (q31_t)0x9CCE562B,
	(q31_t)0xAF6803A1, (q31_t)0x9C8EEB33, (q31_t)0xAFB63667,
	(q31_t)0x9C4FBD92, (q31_t)0xB0049AB2, (q31_t)0x9C10CD70,
	(q31_t)0xB0533055, (q31_t)0x9BD21AF2, (q31_t)0xB0A1F71C,
	(q31_t)0x9B93A640, (q31_t)0xB0F0EEDA, (q31_t)0x9B556F80,
	(q31_t)0xB140175B, (q31_t)0x9B1776D9, (q31_t)0xB18F7070,
	(q31_t)0x9AD9BC71, (q31_t)0xB1DEF9E8, (q31_t)0x9A9C406D,
	(q31_t)0xB22EB392, (q31_t)0x9A5F02F5, (q31_t)0xB27E9D3B,
	(q31_t)0x9A22042C, (q31_t)0xB2CEB6B5, (q31_t)0x99E5443A,
	(q31_t)0xB31EFFCB, (q31_t)0x99A8C344, (q31_t)0xB36F784E,
	(q31_t)0x996C816F, (q31_t)0xB3C0200C, (q31_t)0x99307EE0,
	(q31_t)0xB410F6D2, (q31_t)0x98F4BBBC, (q31_t)0xB461FC70,
	(q31_t)0x98B93828, (q31_t)0xB4B330B2, (q31_t)0x987DF449,
	(q31_t)0xB5049368, (q31_t)0x9842F043, (q31_t)0xB556245E,
	(q31_t)0x98082C3B, (q31_t)0xB5A7E362, (q31_t)0x97CDA855,
	(q31_t)0xB5F9D042, (q31_t)0x979364B5, (q31_t)0xB64BEACC,
	(q31_t)0x9759617E, (q31_t)0xB69E32CD, (q31_t)0x971F9ED6,
	(q31_t)0xB6F0A811, (q31_t)0x96E61CDF, (q31_t)0xB7434A67,
	(q31_t)0x96ACDBBD, (q31_t)0xB796199B, (q31_t)0x9673DB94,
	(q31_t)0xB7E9157A, (q31_t)0x963B1C85, (q31_t)0xB83C3DD1,
	(q31_t)0x96029EB5, (q31_t)0xB88F926C, (q31_t)0x95CA6246,
	(q31_t)0xB8E31319, (q31_t)0x9592675B, (q31_t)0xB936BFA3,
	(q31_t)0x955AAE17, (q31_t)0xB98A97D8, (q31_t)0x9523369B,
	(q31_t)0xB9DE9B83, (q31_t)0x94EC010B, (q31_t)0xBA32CA70,
	(q31_t)0x94B50D87, (q31_t)0xBA87246C, (q31_t)0x947E5C32,
	(q31_t)0xBADBA943, (q31_t)0x9447ED2F, (q31_t)0xBB3058C0,
	(q31_t)0x9411C09D, (q31_t)0xBB8532AF, (q31_t)0x93DBD69F,
	(q31_t)0xBBDA36DC, (q31_t)0x93A62F56, (q31_t)0xBC2F6513,
	(q31_t)0x9370CAE4, (q31_t)0xBC84BD1E, (q31_t)0x933BA968,
	(q31_t)0xBCDA3ECA, (q31_t)0x9306CB04, (q31_t)0xBD2FE9E1,
	(q31_t)0x92D22FD8, (q31_t)0xBD85BE2F, (q31_t)0x929DD805,
	(q31_t)0xBDDBBB7F, (q31_t)0x9269C3AC, (q31_t)0xBE31E19B,
	(q31_t)0x9235F2EB, (q31_t)0xBE88304F, (q31_t)0x920265E4,
	(q31_t)0xBEDEA765, (q31_t)0x91CF1CB6, (q31_t)0xBF3546A8,
	(q31_t)0x919C1780, (q31_t)0xBF8C0DE2, (q31_t)0x91695663,
	(q31_t)0xBFE2FCDF, (q31_t)0x9136D97D, (q31_t)0xC03A1368,
	(q31_t)0x9104A0ED, (q31_t)0xC0915147, (q31_t)0x90D2ACD3,
	(q31_t)0xC0E8B648, (q31_t)0x90A0FD4E, (q31_t)0xC1404233,
	(q31_t)0x906F927B, (q31_t)0xC197F4D3, (q31_t)0x903E6C7A,
	(q31_t)0xC1EFCDF2, (q31_t)0x900D8B69, (q31_t)0xC247CD5A,
	(q31_t)0x8FDCEF66, (q31_t)0xC29FF2D4, (q31_t)0x8FAC988E,
	(q31_t)0xC2F83E2A, (q31_t)0x8F7C8701, (q31_t)0xC350AF25,
	(q31_t)0x8F4CBADB, (q31_t)0xC3A9458F, (q31_t)0x8F1D343A,
	(q31_t)0xC4020132, (q31_t)0x8EEDF33B, (q31_t)0xC45AE1D7,
	(q31_t)0x8EBEF7FB, (q31_t)0xC4B3E746, (q31_t)0x8E904298,
	(q31_t)0xC50D1148, (q31_t)0x8E61D32D, (q31_t)0xC5665FA8,
	(q31_t)0x8E33A9D9, (q31_t)0xC5BFD22E, (q31_t)0x8E05C6B7,
	(q31_t)0xC61968A2, (q31_t)0x8DD829E4, (q31_t)0xC67322CD,
	(q31_t)0x8DAAD37B, (q31_t)0xC6CD0079, (q31_t)0x8D7DC399,
	(q31_t)0xC727016C, (q31_t)0x8D50FA59, (q31_t)0xC7812571,
	(q31_t)0x8D2477D8, (q31_t)0xC7DB6C50, (q31_t)0x8CF83C30,
	(q31_t)0xC835D5D0, (q31_t)0x8CCC477D, (q31_t)0xC89061BA,
	(q31_t)0x8CA099D9, (q31_t)0xC8EB0FD6, (q31_t)0x8C753361,
	(q31_t)0xC945DFEC, (q31_t)0x8C4A142F, (q31_t)0xC9A0D1C4,
	(q31_t)0x8C1F3C5C, (q31_t)0xC9FBE527, (q31_t)0x8BF4AC05,
	(q31_t)0xCA5719DB, (q31_t)0x8BCA6342, (q31_t)0xCAB26FA9,
	(q31_t)0x8BA0622F, (q31_t)0xCB0DE658, (q31_t)0x8B76A8E4,
	(q31_t)0xCB697DB0, (q31_t)0x8B4D377C, (q31_t)0xCBC53578,
	(q31_t)0x8B240E10, (q31_t)0xCC210D78, (q31_t)0x8AFB2CBA,
	(q31_t)0xCC7D0577, (q31_t)0x8AD29393, (q31_t)0xCCD91D3D,
	(q31_t)0x8AAA42B4, (q31_t)0xCD355490, (q31_t)0x8A823A35,
	(q31_t)0xCD91AB38, (q31_t)0x8A5A7A30, (q31_t)0xCDEE20FC,
	(q31_t)0x8A3302BD, (q31_t)0xCE4AB5A2, (q31_t)0x8A0BD3F5,
	(q31_t)0xCEA768F2, (q31_t)0x89E4EDEE, (q31_t)0xCF043AB2,
	(q31_t)0x89BE50C3, (q31_t)0xCF612AAA, (q31_t)0x8997FC89,
	(q31_t)0xCFBE389F, (q31_t)0x8971F15A, (q31_t)0xD01B6459,
	(q31_t)0x894C2F4C, (q31_t)0xD078AD9D, (q31_t)0x8926B677,
	(q31_t)0xD0D61433, (q31_t)0x890186F1, (q31_t)0xD13397E1,
	(q31_t)0x88DCA0D3, (q31_t)0xD191386D, (q31_t)0x88B80431,
	(q31_t)0xD1EEF59E, (q31_t)0x8893B124, (q31_t)0xD24CCF38,
	(q31_t)0x886FA7C2, (q31_t)0xD2AAC504, (q31_t)0x884BE820,
	(q31_t)0xD308D6C6, (q31_t)0x88287255, (q31_t)0xD3670445,
	(q31_t)0x88054677, (q31_t)0xD3C54D46, (q31_t)0x87E2649B,
	(q31_t)0xD423B190, (q31_t)0x87BFCCD7, (q31_t)0xD48230E8,
	(q31_t)0x879D7F40, (q31_t)0xD4E0CB14, (q31_t)0x877B7BEC,
	(q31_t)0xD53F7FDA, (q31_t)0x8759C2EF, (q31_t)0xD59E4EFE,
	(q31_t)0x8738545E, (q31_t)0xD5FD3847, (q31_t)0x8717304E,
	(q31_t)0xD65C3B7B, (q31_t)0x86F656D3, (q31_t)0xD6BB585D,
	(q31_t)0x86D5C802, (q31_t)0xD71A8EB5, (q31_t)0x86B583EE,
	(q31_t)0xD779DE46, (q31_t)0x86958AAB, (q31_t)0xD7D946D7,
	(q31_t)0x8675DC4E, (q31_t)0xD838C82D, (q31_t)0x865678EA,
	(q31_t)0xD898620C, (q31_t)0x86376092, (q31_t)0xD8F81439,
	(q31_t)0x86189359, (q31_t)0xD957DE7A, (q31_t)0x85FA1152,
	(q31_t)0xD9B7C093, (q31_t)0x85DBDA91, (q31_t)0xDA17BA4A,
	(q31_t)0x85BDEF27, (q31_t)0xDA77CB62, (q31_t)0x85A04F28,
	(q31_t)0xDAD7F3A2, (q31_t)0x8582FAA4, (q31_t)0xDB3832CD,
	(q31_t)0x8565F1B0, (q31_t)0xDB9888A8, (q31_t)0x8549345C,
	(q31_t)0xDBF8F4F8, (q31_t)0x852CC2BA, (q31_t)0xDC597781,
	(q31_t)0x85109CDC, (q31_t)0xDCBA1008, (q31_t)0x84F4C2D3,
	(q31_t)0xDD1ABE51, (q31_t)0x84D934B0, (q31_t)0xDD7B8220,
	(q31_t)0x84BDF285, (q31_t)0xDDDC5B3A, (q31_t)0x84A2FC62,
	(q31_t)0xDE3D4963, (q31_t)0x84885257, (q31_t)0xDE9E4C60,
	(q31_t)0x846DF476, (q31_t)0xDEFF63F4, (q31_t)0x8453E2CE,
	(q31_t)0xDF608FE3, (q31_t)0x843A1D70, (q31_t)0xDFC1CFF2,
	(q31_t)0x8420A46B, (q31_t)0xE02323E5, (q31_t)0x840777CF,
	(q31_t)0xE0848B7F, (q31_t)0x83EE97AC, (q31_t)0xE0E60684,
	(q31_t)0x83D60411, (q31_t)0xE14794B9, (q31_t)0x83BDBD0D,
	(q31_t)0xE1A935E1, (q31_t)0x83A5C2B0, (q31_t)0xE20AE9C1,
	(q31_t)0x838E1507, (q31_t)0xE26CB01A, (q31_t)0x8376B422,
	(q31_t)0xE2CE88B2, (q31_t)0x835FA00E, (q31_t)0xE330734C,
	(q31_t)0x8348D8DB, (q31_t)0xE3926FAC, (q31_t)0x83325E97,
	(q31_t)0xE3F47D95, (q31_t)0x831C314E, (q31_t)0xE4569CCB,
	(q31_t)0x8306510F, (q31_t)0xE4B8CD10, (q31_t)0x82F0BDE8,
	(q31_t)0xE51B0E2A, (q31_t)0x82DB77E5, (q31_t)0xE57D5FDA,
	(q31_t)0x82C67F13, (q31_t)0xE5DFC1E4, (q31_t)0x82B1D381,
	(q31_t)0xE642340D, (q31_t)0x829D753A, (q31_t)0xE6A4B616,
	(q31_t)0x8289644A, (q31_t)0xE70747C3, (q31_t)0x8275A0C0,
	(q31_t)0xE769E8D8, (q31_t)0x82622AA5, (q31_t)0xE7CC9917,
	(q31_t)0x824F0208, (q31_t)0xE82F5844, (q31_t)0x823C26F2,
	(q31_t)0xE8922621, (q31_t)0x82299971, (q31_t)0xE8F50273,
	(q31_t)0x8217598F, (q31_t)0xE957ECFB, (q31_t)0x82056758,
	(q31_t)0xE9BAE57C, (q31_t)0x81F3C2D7, (q31_t)0xEA1DEBBB,
	(q31_t)0x81E26C16, (q31_t)0xEA80FF79, (q31_t)0x81D16320,
	(q31_t)0xEAE4207A, (q31_t)0x81C0A801, (q31_t)0xEB474E80,
	(q31_t)0x81B03AC1, (q31_t)0xEBAA894E, (q31_t)0x81A01B6C,
	(q31_t)0xEC0DD0A8, (q31_t)0x81904A0C, (q31_t)0xEC71244F,
	(q31_t)0x8180C6A9, (q31_t)0xECD48406, (q31_t)0x8171914E,
	(q31_t)0xED37EF91, (q31_t)0x8162AA03, (q31_t)0xED9B66B2,
	(q31_t)0x815410D3, (q31_t)0xEDFEE92B, (q31_t)0x8145C5C6,
	(q31_t)0xEE6276BF, (q31_t)0x8137C8E6, (q31_t)0xEEC60F31,
	(q31_t)0x812A1A39, (q31_t)0xEF29B243, (q31_t)0x811CB9CA,
	(q31_t)0xEF8D5FB8, (q31_t)0x810FA7A0, (q31_t)0xEFF11752,
	(q31_t)0x8102E3C3, (q31_t)0xF054D8D4, (q31_t)0x80F66E3C,
	(q31_t)0xF0B8A401, (q31_t)0x80EA4712, (q31_t)0xF11C789A,
	(q31_t)0x80DE6E4C, (q31_t)0xF1805662, (q31_t)0x80D2E3F1,
	(q31_t)0xF1E43D1C, (q31_t)0x80C7A80A, (q31_t)0xF2482C89,
	(q31_t)0x80BCBA9C, (q31_t)0xF2AC246D, (q31_t)0x80B21BAF,
	(q31_t)0xF310248A, (q31_t)0x80A7CB49, (q31_t)0xF3742CA1,
	(q31_t)0x809DC970, (q31_t)0xF3D83C76, (q31_t)0x8094162B,
	(q31_t)0xF43C53CA, (q31_t)0x808AB180, (q31_t)0xF4A07260,
	(q31_t)0x80819B74, (q31_t)0xF50497FA, (q31_t)0x8078D40D,
	(q31_t)0xF568C45A, (q31_t)0x80705B50, (q31_t)0xF5CCF743,
	(q31_t)0x80683143, (q31_t)0xF6313076, (q31_t)0x806055EA,
	(q31_t)0xF6956FB6, (q31_t)0x8058C94C, (q31_t)0xF6F9B4C5,
	(q31_t)0x80518B6B, (q31_t)0xF75DFF65, (q31_t)0x804A9C4D,
	(q31_t)0xF7C24F58, (q31_t)0x8043FBF6, (q31_t)0xF826A461,
	(q31_t)0x803DAA69, (q31_t)0xF88AFE41, (q31_t)0x8037A7AC,
	(q31_t)0xF8EF5CBB, (q31_t)0x8031F3C1, (q31_t)0xF953BF90,
	(q31_t)0x802C8EAD, (q31_t)0xF9B82683, (q31_t)0x80277872,
	(q31_t)0xFA1C9156, (q31_t)0x8022B113, (q31_t)0xFA80FFCB,
	(q31_t)0x801E3894, (q31_t)0xFAE571A4, (q31_t)0x801A0EF7,
	(q31_t)0xFB49E6A2, (q31_t)0x80163440, (q31_t)0xFBAE5E89,
	(q31_t)0x8012A86F, (q31_t)0xFC12D919, (q31_t)0x800F6B88,
	(q31_t)0xFC775616, (q31_t)0x800C7D8C, (q31_t)0xFCDBD541,
	(q31_t)0x8009DE7D, (q31_t)0xFD40565B, (q31_t)0x80078E5E,
	(q31_t)0xFDA4D928, (q31_t)0x80058D2E, (q31_t)0xFE095D69,
	(q31_t)0x8003DAF0, (q31_t)0xFE6DE2E0, (q31_t)0x800277A5,
	(q31_t)0xFED2694F, (q31_t)0x8001634D, (q31_t)0xFF36F078,
	(q31_t)0x80009DE9, (q31_t)0xFF9B781D, (q31_t)0x8000277A
};

/**
* \par
* Example code for Q31 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefQ31[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefQ31[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 4096	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to Q31(Fixed point 1.31):
*	round(twiddleCoefQ31(i) * pow(2, 31))
*
*/
const q31_t twiddleCoef_4096_q31[6144] =
{
	(q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7FFFF621,
	(q31_t)0x003243F5, (q31_t)0x7FFFD885, (q31_t)0x006487E3,
	(q31_t)0x7FFFA72C, (q31_t)0x0096CBC1, (q31_t)0x7FFF6216,
	(q31_t)0x00C90F88, (q31_t)0x7FFF0942, (q31_t)0x00FB532F,
	(q31_t)0x7FFE9CB2, (q31_t)0x012D96B0, (q31_t)0x7FFE1C64,
	(q31_t)0x015FDA03, (q31_t)0x7FFD885A, (q31_t)0x01921D1F,
	(q31_t)0x7FFCE093, (q31_t)0x01C45FFE, (q31_t)0x7FFC250F,
	(q31_t)0x01F6A296, (q31_t)0x7FFB55CE, (q31_t)0x0228E4E1,
	(q31_t)0x7FFA72D1, (q31_t)0x025B26D7, (q31_t)0x7FF97C17,
	(q31_t)0x028D6870, (q31_t)0x7FF871A1, (q31_t)0x02BFA9A4,
	(q31_t)0x7FF7536F, (q31_t)0x02F1EA6B, (q31_t)0x7FF62182,
	(q31_t)0x03242ABF, (q31_t)0x7FF4DBD8, (q31_t)0x03566A96,
	(q31_t)0x7FF38273, (q31_t)0x0388A9E9, (q31_t)0x7FF21553,
	(q31_t)0x03BAE8B1, (q31_t)0x7FF09477, (q31_t)0x03ED26E6,
	(q31_t)0x7FEEFFE1, (q31_t)0x041F647F, (q31_t)0x7FED5790,
	(q31_t)0x0451A176, (q31_t)0x7FEB9B85, (q31_t)0x0483DDC3,
	(q31_t)0x7FE9CBC0, (q31_t)0x04B6195D, (q31_t)0x7FE7E840,
	(q31_t)0x04E8543D, (q31_t)0x7FE5F108, (q31_t)0x051A8E5C,
	(q31_t)0x7FE3E616, (q31_t)0x054CC7B0, (q31_t)0x7FE1C76B,
	(q31_t)0x057F0034, (q31_t)0x7FDF9508, (q31_t)0x05B137DF,
	(q31_t)0x7FDD4EEC, (q31_t)0x05E36EA9, (q31_t)0x7FDAF518,
	(q31_t)0x0615A48A, (q31_t)0x7FD8878D, (q31_t)0x0647D97C,
	(q31_t)0x7FD6064B, (q31_t)0x067A0D75, (q31_t)0x7FD37152,
	(q31_t)0x06AC406F, (q31_t)0x7FD0C8A3, (q31_t)0x06DE7261,
	(q31_t)0x7FCE0C3E, (q31_t)0x0710A344, (q31_t)0x7FCB3C23,
	(q31_t)0x0742D310, (q31_t)0x7FC85853, (q31_t)0x077501BE,
	(q31_t)0x7FC560CF, (q31_t)0x07A72F45, (q31_t)0x7FC25596,
	(q31_t)0x07D95B9E, (q31_t)0x7FBF36A9, (q31_t)0x080B86C1,
	(q31_t)0x7FBC040A, (q31_t)0x083DB0A7, (q31_t)0x7FB8BDB7,
	(q31_t)0x086FD947, (q31_t)0x7FB563B2, (q31_t)0x08A2009A,
	(q31_t)0x7FB1F5FC, (q31_t)0x08D42698, (q31_t)0x7FAE7494,
	(q31_t)0x09064B3A, (q31_t)0x7FAADF7C, (q31_t)0x09386E77,
	(q31_t)0x7FA736B4, (q31_t)0x096A9049, (q31_t)0x7FA37A3C,
	(q31_t)0x099CB0A7, (q31_t)0x7F9FAA15, (q31_t)0x09CECF89,
	(q31_t)0x7F9BC63F, (q31_t)0x0A00ECE8, (q31_t)0x7F97CEBC,
	(q31_t)0x0A3308BC, (q31_t)0x7F93C38C, (q31_t)0x0A6522FE,
	(q31_t)0x7F8FA4AF, (q31_t)0x0A973BA5, (q31_t)0x7F8B7226,
	(q31_t)0x0AC952AA, (q31_t)0x7F872BF3, (q31_t)0x0AFB6805,
	(q31_t)0x7F82D214, (q31_t)0x0B2D7BAE, (q31_t)0x7F7E648B,
	(q31_t)0x0B5F8D9F, (q31_t)0x7F79E35A, (q31_t)0x0B919DCE,
	(q31_t)0x7F754E7F, (q31_t)0x0BC3AC35, (q31_t)0x7F70A5FD,
	(q31_t)0x0BF5B8CB, (q31_t)0x7F6BE9D4, (q31_t)0x0C27C389,
	(q31_t)0x7F671A04, (q31_t)0x0C59CC67, (q31_t)0x7F62368F,
	(q31_t)0x0C8BD35E, (q31_t)0x7F5D3F75, (q31_t)0x0CBDD865,
	(q31_t)0x7F5834B6, (q31_t)0x0CEFDB75, (q31_t)0x7F531654,
	(q31_t)0x0D21DC87, (q31_t)0x7F4DE450, (q31_t)0x0D53DB92,
	(q31_t)0x7F489EAA, (q31_t)0x0D85D88F, (q31_t)0x7F434563,
	(q31_t)0x0DB7D376, (q31_t)0x7F3DD87C, (q31_t)0x0DE9CC3F,
	(q31_t)0x7F3857F5, (q31_t)0x0E1BC2E3, (q31_t)0x7F32C3D0,
	(q31_t)0x0E4DB75B, (q31_t)0x7F2D1C0E, (q31_t)0x0E7FA99D,
	(q31_t)0x7F2760AF, (q31_t)0x0EB199A3, (q31_t)0x7F2191B4,
	(q31_t)0x0EE38765, (q31_t)0x7F1BAF1E, (q31_t)0x0F1572DC,
	(q31_t)0x7F15B8EE, (q31_t)0x0F475BFE, (q31_t)0x7F0FAF24,
	(q31_t)0x0F7942C6, (q31_t)0x7F0991C3, (q31_t)0x0FAB272B,
	(q31_t)0x7F0360CB, (q31_t)0x0FDD0925, (q31_t)0x7EFD1C3C,
	(q31_t)0x100EE8AD, (q31_t)0x7EF6C418, (q31_t)0x1040C5BB,
	(q31_t)0x7EF0585F, (q31_t)0x1072A047, (q31_t)0x7EE9D913,
	(q31_t)0x10A4784A, (q31_t)0x7EE34635, (q31_t)0x10D64DBC,
	(q31_t)0x7EDC9FC6, (q31_t)0x11082096, (q31_t)0x7ED5E5C6,
	(q31_t)0x1139F0CE, (q31_t)0x7ECF1837, (q31_t)0x116BBE5F,
	(q31_t)0x7EC8371A, (q31_t)0x119D8940, (q31_t)0x7EC1426F,
	(q31_t)0x11CF516A, (q31_t)0x7EBA3A39, (q31_t)0x120116D4,
	(q31_t)0x7EB31E77, (q31_t)0x1232D978, (q31_t)0x7EABEF2C,
	(q31_t)0x1264994E, (q31_t)0x7EA4AC58, (q31_t)0x1296564D,
	(q31_t)0x7E9D55FC, (q31_t)0x12C8106E, (q31_t)0x7E95EC19,
	(q31_t)0x12F9C7AA, (q31_t)0x7E8E6EB1, (q31_t)0x132B7BF9,
	(q31_t)0x7E86DDC5, (q31_t)0x135D2D53, (q31_t)0x7E7F3956,
	(q31_t)0x138EDBB0, (q31_t)0x7E778165, (q31_t)0x13C0870A,
	(q31_t)0x7E6FB5F3, (q31_t)0x13F22F57, (q31_t)0x7E67D702,
	(q31_t)0x1423D492, (q31_t)0x7E5FE493, (q31_t)0x145576B1,
	(q31_t)0x7E57DEA6, (q31_t)0x148715AD, (q31_t)0x7E4FC53E,
	(q31_t)0x14B8B17F, (q31_t)0x7E47985B, (q31_t)0x14EA4A1F,
	(q31_t)0x7E3F57FE, (q31_t)0x151BDF85, (q31_t)0x7E37042A,
	(q31_t)0x154D71AA, (q31_t)0x7E2E9CDF, (q31_t)0x157F0086,
	(q31_t)0x7E26221E, (q31_t)0x15B08C11, (q31_t)0x7E1D93E9,
	(q31_t)0x15E21444, (q31_t)0x7E14F242, (q31_t)0x16139917,
	(q31_t)0x7E0C3D29, (q31_t)0x16451A83, (q31_t)0x7E03749F,
	(q31_t)0x1676987F, (q31_t)0x7DFA98A7, (q31_t)0x16A81305,
	(q31_t)0x7DF1A942, (q31_t)0x16D98A0C, (q31_t)0x7DE8A670,
	(q31_t)0x170AFD8D, (q31_t)0x7DDF9034, (q31_t)0x173C6D80,
	(q31_t)0x7DD6668E, (q31_t)0x176DD9DE, (q31_t)0x7DCD2981,
	(q31_t)0x179F429F, (q31_t)0x7DC3D90D, (q31_t)0x17D0A7BB,
	(q31_t)0x7DBA7534, (q31_t)0x1802092C, (q31_t)0x7DB0FDF7,
	(q31_t)0x183366E8, (q31_t)0x7DA77359, (q31_t)0x1864C0E9,
	(q31_t)0x7D9DD55A, (q31_t)0x18961727, (q31_t)0x7D9423FB,
	(q31_t)0x18C7699B, (q31_t)0x7D8A5F3F, (q31_t)0x18F8B83C,
	(q31_t)0x7D808727, (q31_t)0x192A0303, (q31_t)0x7D769BB5,
	(q31_t)0x195B49E9, (q31_t)0x7D6C9CE9, (q31_t)0x198C8CE6,
	(q31_t)0x7D628AC5, (q31_t)0x19BDCBF2, (q31_t)0x7D58654C,
	(q31_t)0x19EF0706, (q31_t)0x7D4E2C7E, (q31_t)0x1A203E1B,
	(q31_t)0x7D43E05E, (q31_t)0x1A517127, (q31_t)0x7D3980EC,
	(q31_t)0x1A82A025, (q31_t)0x7D2F0E2A, (q31_t)0x1AB3CB0C,
	(q31_t)0x7D24881A, (q31_t)0x1AE4F1D6, (q31_t)0x7D19EEBE,
	(q31_t)0x1B161479, (q31_t)0x7D0F4218, (q31_t)0x1B4732EF,
	(q31_t)0x7D048228, (q31_t)0x1B784D30, (q31_t)0x7CF9AEF0,
	(q31_t)0x1BA96334, (q31_t)0x7CEEC873, (q31_t)0x1BDA74F5,
	(q31_t)0x7CE3CEB1, (q31_t)0x1C0B826A, (q31_t)0x7CD8C1AD,
	(q31_t)0x1C3C8B8C, (q31_t)0x7CCDA168, (q31_t)0x1C6D9053,
	(q31_t)0x7CC26DE5, (q31_t)0x1C9E90B8, (q31_t)0x7CB72724,
	(q31_t)0x1CCF8CB3, (q31_t)0x7CABCD27, (q31_t)0x1D00843C,
	(q31_t)0x7CA05FF1, (q31_t)0x1D31774D, (q31_t)0x7C94DF82,
	(q31_t)0x1D6265DD, (q31_t)0x7C894BDD, (q31_t)0x1D934FE5,
	(q31_t)0x7C7DA504, (q31_t)0x1DC4355D, (q31_t)0x7C71EAF8,
	(q31_t)0x1DF5163F, (q31_t)0x7C661DBB, (q31_t)0x1E25F281,
	(q31_t)0x7C5A3D4F, (q31_t)0x1E56CA1E, (q31_t)0x7C4E49B6,
	(q31_t)0x1E879D0C, (q31_t)0x7C4242F2, (q31_t)0x1EB86B46,
	(q31_t)0x7C362904, (q31_t)0x1EE934C2, (q31_t)0x7C29FBEE,
	(q31_t)0x1F19F97B, (q31_t)0x7C1DBBB2, (q31_t)0x1F4AB967,
	(q31_t)0x7C116853, (q31_t)0x1F7B7480, (q31_t)0x7C0501D1,
	(q31_t)0x1FAC2ABF, (q31_t)0x7BF88830, (q31_t)0x1FDCDC1A,
	(q31_t)0x7BEBFB70, (q31_t)0x200D888C, (q31_t)0x7BDF5B94,
	(q31_t)0x203E300D, (q31_t)0x7BD2A89E, (q31_t)0x206ED295,
	(q31_t)0x7BC5E28F, (q31_t)0x209F701C, (q31_t)0x7BB9096A,
	(q31_t)0x20D0089B, (q31_t)0x7BAC1D31, (q31_t)0x21009C0B,
	(q31_t)0x7B9F1DE5, (q31_t)0x21312A65, (q31_t)0x7B920B89,
	(q31_t)0x2161B39F, (q31_t)0x7B84E61E, (q31_t)0x219237B4,
	(q31_t)0x7B77ADA8, (q31_t)0x21C2B69C, (q31_t)0x7B6A6227,
	(q31_t)0x21F3304E, (q31_t)0x7B5D039D, (q31_t)0x2223A4C5,
	(q31_t)0x7B4F920E, (q31_t)0x225413F8, (q31_t)0x7B420D7A,
	(q31_t)0x22847DDF, (q31_t)0x7B3475E4, (q31_t)0x22B4E274,
	(q31_t)0x7B26CB4F, (q31_t)0x22E541AE, (q31_t)0x7B190DBB,
	(q31_t)0x23159B87, (q31_t)0x7B0B3D2C, (q31_t)0x2345EFF7,
	(q31_t)0x7AFD59A3, (q31_t)0x23763EF7, (q31_t)0x7AEF6323,
	(q31_t)0x23A6887E, (q31_t)0x7AE159AE, (q31_t)0x23D6CC86,
	(q31_t)0x7AD33D45, (q31_t)0x24070B07, (q31_t)0x7AC50DEB,
	(q31_t)0x243743FA, (q31_t)0x7AB6CBA3, (q31_t)0x24677757,
	(q31_t)0x7AA8766E, (q31_t)0x2497A517, (q31_t)0x7A9A0E4F,
	(q31_t)0x24C7CD32, (q31_t)0x7A8B9348, (q31_t)0x24F7EFA1,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x7A6E648A,
	(q31_t)0x2558235E, (q31_t)0x7A5FB0D8, (q31_t)0x2588349D,
	(q31_t)0x7A50EA46, (q31_t)0x25B84012, (q31_t)0x7A4210D8,
	(q31_t)0x25E845B5, (q31_t)0x7A33248F, (q31_t)0x26184581,
	(q31_t)0x7A24256E, (q31_t)0x26483F6C, (q31_t)0x7A151377,
	(q31_t)0x26783370, (q31_t)0x7A05EEAD, (q31_t)0x26A82185,
	(q31_t)0x79F6B711, (q31_t)0x26D809A5, (q31_t)0x79E76CA6,
	(q31_t)0x2707EBC6, (q31_t)0x79D80F6F, (q31_t)0x2737C7E3,
	(q31_t)0x79C89F6D, (q31_t)0x27679DF4, (q31_t)0x79B91CA4,
	(q31_t)0x27976DF1, (q31_t)0x79A98715, (q31_t)0x27C737D2,
	(q31_t)0x7999DEC3, (q31_t)0x27F6FB92, (q31_t)0x798A23B1,
	(q31_t)0x2826B928, (q31_t)0x797A55E0, (q31_t)0x2856708C,
	(q31_t)0x796A7554, (q31_t)0x288621B9, (q31_t)0x795A820E,
	(q31_t)0x28B5CCA5, (q31_t)0x794A7C11, (q31_t)0x28E5714A,
	(q31_t)0x793A6360, (q31_t)0x29150FA1, (q31_t)0x792A37FE,
	(q31_t)0x2944A7A2, (q31_t)0x7919F9EB, (q31_t)0x29743945,
	(q31_t)0x7909A92C, (q31_t)0x29A3C484, (q31_t)0x78F945C3,
	(q31_t)0x29D34958, (q31_t)0x78E8CFB1, (q31_t)0x2A02C7B8,
	(q31_t)0x78D846FB, (q31_t)0x2A323F9D, (q31_t)0x78C7ABA1,
	(q31_t)0x2A61B101, (q31_t)0x78B6FDA8, (q31_t)0x2A911BDB,
	(q31_t)0x78A63D10, (q31_t)0x2AC08025, (q31_t)0x789569DE,
	(q31_t)0x2AEFDDD8, (q31_t)0x78848413, (q31_t)0x2B1F34EB,
	(q31_t)0x78738BB3, (q31_t)0x2B4E8558, (q31_t)0x786280BF,
	(q31_t)0x2B7DCF17, (q31_t)0x7851633B, (q31_t)0x2BAD1221,
	(q31_t)0x78403328, (q31_t)0x2BDC4E6F, (q31_t)0x782EF08B,
	(q31_t)0x2C0B83F9, (q31_t)0x781D9B64, (q31_t)0x2C3AB2B9,
	(q31_t)0x780C33B8, (q31_t)0x2C69DAA6, (q31_t)0x77FAB988,
	(q31_t)0x2C98FBBA, (q31_t)0x77E92CD8, (q31_t)0x2CC815ED,
	(q31_t)0x77D78DAA, (q31_t)0x2CF72939, (q31_t)0x77C5DC01,
	(q31_t)0x2D263595, (q31_t)0x77B417DF, (q31_t)0x2D553AFB,
	(q31_t)0x77A24148, (q31_t)0x2D843963, (q31_t)0x7790583D,
	(q31_t)0x2DB330C7, (q31_t)0x777E5CC3, (q31_t)0x2DE2211E,
	(q31_t)0x776C4EDB, (q31_t)0x2E110A62, (q31_t)0x775A2E88,
	(q31_t)0x2E3FEC8B, (q31_t)0x7747FBCE, (q31_t)0x2E6EC792,
	(q31_t)0x7735B6AE, (q31_t)0x2E9D9B70, (q31_t)0x77235F2D,
	(q31_t)0x2ECC681E, (q31_t)0x7710F54B, (q31_t)0x2EFB2D94,
	(q31_t)0x76FE790E, (q31_t)0x2F29EBCC, (q31_t)0x76EBEA77,
	(q31_t)0x2F58A2BD, (q31_t)0x76D94988, (q31_t)0x2F875262,
	(q31_t)0x76C69646, (q31_t)0x2FB5FAB2, (q31_t)0x76B3D0B3,
	(q31_t)0x2FE49BA6, (q31_t)0x76A0F8D2, (q31_t)0x30133538,
	(q31_t)0x768E0EA5, (q31_t)0x3041C760, (q31_t)0x767B1230,
	(q31_t)0x30705217, (q31_t)0x76680376, (q31_t)0x309ED555,
	(q31_t)0x7654E279, (q31_t)0x30CD5114, (q31_t)0x7641AF3C,
	(q31_t)0x30FBC54D, (q31_t)0x762E69C3, (q31_t)0x312A31F8,
	(q31_t)0x761B1211, (q31_t)0x3158970D, (q31_t)0x7607A827,
	(q31_t)0x3186F487, (q31_t)0x75F42C0A, (q31_t)0x31B54A5D,
	(q31_t)0x75E09DBD, (q31_t)0x31E39889, (q31_t)0x75CCFD42,
	(q31_t)0x3211DF03, (q31_t)0x75B94A9C, (q31_t)0x32401DC5,
	(q31_t)0x75A585CF, (q31_t)0x326E54C7, (q31_t)0x7591AEDD,
	(q31_t)0x329C8402, (q31_t)0x757DC5CA, (q31_t)0x32CAAB6F,
	(q31_t)0x7569CA98, (q31_t)0x32F8CB07, (q31_t)0x7555BD4B,
	(q31_t)0x3326E2C2, (q31_t)0x75419DE6, (q31_t)0x3354F29A,
	(q31_t)0x752D6C6C, (q31_t)0x3382FA88, (q31_t)0x751928E0,
	(q31_t)0x33B0FA84, (q31_t)0x7504D345, (q31_t)0x33DEF287,
	(q31_t)0x74F06B9E, (q31_t)0x340CE28A, (q31_t)0x74DBF1EF,
	(q31_t)0x343ACA87, (q31_t)0x74C7663A, (q31_t)0x3468AA76,
	(q31_t)0x74B2C883, (q31_t)0x3496824F, (q31_t)0x749E18CD,
	(q31_t)0x34C4520D, (q31_t)0x7489571B, (q31_t)0x34F219A7,
	(q31_t)0x74748371, (q31_t)0x351FD917, (q31_t)0x745F9DD1,
	(q31_t)0x354D9056, (q31_t)0x744AA63E, (q31_t)0x357B3F5D,
	(q31_t)0x74359CBD, (q31_t)0x35A8E624, (q31_t)0x74208150,
	(q31_t)0x35D684A5, (q31_t)0x740B53FA, (q31_t)0x36041AD9,
	(q31_t)0x73F614C0, (q31_t)0x3631A8B7, (q31_t)0x73E0C3A3,
	(q31_t)0x365F2E3B, (q31_t)0x73CB60A7, (q31_t)0x368CAB5C,
	(q31_t)0x73B5EBD0, (q31_t)0x36BA2013, (q31_t)0x73A06522,
	(q31_t)0x36E78C5A, (q31_t)0x738ACC9E, (q31_t)0x3714F02A,
	(q31_t)0x73752249, (q31_t)0x37424B7A, (q31_t)0x735F6626,
	(q31_t)0x376F9E46, (q31_t)0x73499838, (q31_t)0x379CE884,
	(q31_t)0x7333B883, (q31_t)0x37CA2A30, (q31_t)0x731DC709,
	(q31_t)0x37F76340, (q31_t)0x7307C3D0, (q31_t)0x382493B0,
	(q31_t)0x72F1AED8, (q31_t)0x3851BB76, (q31_t)0x72DB8828,
	(q31_t)0x387EDA8E, (q31_t)0x72C54FC0, (q31_t)0x38ABF0EF,
	(q31_t)0x72AF05A6, (q31_t)0x38D8FE93, (q31_t)0x7298A9DC,
	(q31_t)0x39060372, (q31_t)0x72823C66, (q31_t)0x3932FF87,
	(q31_t)0x726BBD48, (q31_t)0x395FF2C9, (q31_t)0x72552C84,
	(q31_t)0x398CDD32, (q31_t)0x723E8A1F, (q31_t)0x39B9BEBB,
	(q31_t)0x7227D61C, (q31_t)0x39E6975D, (q31_t)0x7211107D,
	(q31_t)0x3A136712, (q31_t)0x71FA3948, (q31_t)0x3A402DD1,
	(q31_t)0x71E3507F, (q31_t)0x3A6CEB95, (q31_t)0x71CC5626,
	(q31_t)0x3A99A057, (q31_t)0x71B54A40, (q31_t)0x3AC64C0F,
	(q31_t)0x719E2CD2, (q31_t)0x3AF2EEB7, (q31_t)0x7186FDDE,
	(q31_t)0x3B1F8847, (q31_t)0x716FBD68, (q31_t)0x3B4C18BA,
	(q31_t)0x71586B73, (q31_t)0x3B78A007, (q31_t)0x71410804,
	(q31_t)0x3BA51E29, (q31_t)0x7129931E, (q31_t)0x3BD19317,
	(q31_t)0x71120CC5, (q31_t)0x3BFDFECD, (q31_t)0x70FA74FB,
	(q31_t)0x3C2A6142, (q31_t)0x70E2CBC6, (q31_t)0x3C56BA70,
	(q31_t)0x70CB1127, (q31_t)0x3C830A4F, (q31_t)0x70B34524,
	(q31_t)0x3CAF50DA, (q31_t)0x709B67C0, (q31_t)0x3CDB8E09,
	(q31_t)0x708378FE, (q31_t)0x3D07C1D5, (q31_t)0x706B78E3,
	(q31_t)0x3D33EC39, (q31_t)0x70536771, (q31_t)0x3D600D2B,
	(q31_t)0x703B44AC, (q31_t)0x3D8C24A7, (q31_t)0x70231099,
	(q31_t)0x3DB832A5, (q31_t)0x700ACB3B, (q31_t)0x3DE4371F,
	(q31_t)0x6FF27496, (q31_t)0x3E10320D, (q31_t)0x6FDA0CAD,
	(q31_t)0x3E3C2369, (q31_t)0x6FC19385, (q31_t)0x3E680B2C,
	(q31_t)0x6FA90920, (q31_t)0x3E93E94F, (q31_t)0x6F906D84,
	(q31_t)0x3EBFBDCC, (q31_t)0x6F77C0B3, (q31_t)0x3EEB889C,
	(q31_t)0x6F5F02B1, (q31_t)0x3F1749B7, (q31_t)0x6F463383,
	(q31_t)0x3F430118, (q31_t)0x6F2D532C, (q31_t)0x3F6EAEB8,
	(q31_t)0x6F1461AF, (q31_t)0x3F9A528F, (q31_t)0x6EFB5F12,
	(q31_t)0x3FC5EC97, (q31_t)0x6EE24B57, (q31_t)0x3FF17CCA,
	(q31_t)0x6EC92682, (q31_t)0x401D0320, (q31_t)0x6EAFF098,
	(q31_t)0x40487F93, (q31_t)0x6E96A99C, (q31_t)0x4073F21D,
	(q31_t)0x6E7D5193, (q31_t)0x409F5AB6, (q31_t)0x6E63E87F,
	(q31_t)0x40CAB957, (q31_t)0x6E4A6E65, (q31_t)0x40F60DFB,
	(q31_t)0x6E30E349, (q31_t)0x4121589A, (q31_t)0x6E17472F,
	(q31_t)0x414C992E, (q31_t)0x6DFD9A1B, (q31_t)0x4177CFB0,
	(q31_t)0x6DE3DC11, (q31_t)0x41A2FC1A, (q31_t)0x6DCA0D14,
	(q31_t)0x41CE1E64, (q31_t)0x6DB02D29, (q31_t)0x41F93688,
	(q31_t)0x6D963C54, (q31_t)0x42244480, (q31_t)0x6D7C3A98,
	(q31_t)0x424F4845, (q31_t)0x6D6227FA, (q31_t)0x427A41D0,
	(q31_t)0x6D48047E, (q31_t)0x42A5311A, (q31_t)0x6D2DD027,
	(q31_t)0x42D0161E, (q31_t)0x6D138AFA, (q31_t)0x42FAF0D4,
	(q31_t)0x6CF934FB, (q31_t)0x4325C135, (q31_t)0x6CDECE2E,
	(q31_t)0x4350873C, (q31_t)0x6CC45697, (q31_t)0x437B42E1,
	(q31_t)0x6CA9CE3A, (q31_t)0x43A5F41E, (q31_t)0x6C8F351C,
	(q31_t)0x43D09AEC, (q31_t)0x6C748B3F, (q31_t)0x43FB3745,
	(q31_t)0x6C59D0A9, (q31_t)0x4425C923, (q31_t)0x6C3F055D,
	(q31_t)0x4450507E, (q31_t)0x6C242960, (q31_t)0x447ACD50,
	(q31_t)0x6C093CB6, (q31_t)0x44A53F93, (q31_t)0x6BEE3F62,
	(q31_t)0x44CFA73F, (q31_t)0x6BD3316A, (q31_t)0x44FA044F,
	(q31_t)0x6BB812D0, (q31_t)0x452456BC, (q31_t)0x6B9CE39B,
	(q31_t)0x454E9E80, (q31_t)0x6B81A3CD, (q31_t)0x4578DB93,
	(q31_t)0x6B66536A, (q31_t)0x45A30DF0, (q31_t)0x6B4AF278,
	(q31_t)0x45CD358F, (q31_t)0x6B2F80FA, (q31_t)0x45F7526B,
	(q31_t)0x6B13FEF5, (q31_t)0x4621647C, (q31_t)0x6AF86C6C,
	(q31_t)0x464B6BBD, (q31_t)0x6ADCC964, (q31_t)0x46756827,
	(q31_t)0x6AC115E1, (q31_t)0x469F59B4, (q31_t)0x6AA551E8,
	(q31_t)0x46C9405C, (q31_t)0x6A897D7D, (q31_t)0x46F31C1A,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x6A51A361,
	(q31_t)0x4746B2BC, (q31_t)0x6A359DB9, (q31_t)0x47706D93,
	(q31_t)0x6A1987B0, (q31_t)0x479A1D66, (q31_t)0x69FD614A,
	(q31_t)0x47C3C22E, (q31_t)0x69E12A8C, (q31_t)0x47ED5BE6,
	(q31_t)0x69C4E37A, (q31_t)0x4816EA85, (q31_t)0x69A88C18,
	(q31_t)0x48406E07, (q31_t)0x698C246C, (q31_t)0x4869E664,
	(q31_t)0x696FAC78, (q31_t)0x48935397, (q31_t)0x69532442,
	(q31_t)0x48BCB598, (q31_t)0x69368BCE, (q31_t)0x48E60C62,
	(q31_t)0x6919E320, (q31_t)0x490F57EE, (q31_t)0x68FD2A3D,
	(q31_t)0x49389836, (q31_t)0x68E06129, (q31_t)0x4961CD32,
	(q31_t)0x68C387E9, (q31_t)0x498AF6DE, (q31_t)0x68A69E81,
	(q31_t)0x49B41533, (q31_t)0x6889A4F5, (q31_t)0x49DD282A,
	(q31_t)0x686C9B4B, (q31_t)0x4A062FBD, (q31_t)0x684F8186,
	(q31_t)0x4A2F2BE5, (q31_t)0x683257AA, (q31_t)0x4A581C9D,
	(q31_t)0x68151DBE, (q31_t)0x4A8101DE, (q31_t)0x67F7D3C4,
	(q31_t)0x4AA9DBA1, (q31_t)0x67DA79C2, (q31_t)0x4AD2A9E1,
	(q31_t)0x67BD0FBC, (q31_t)0x4AFB6C97, (q31_t)0x679F95B7,
	(q31_t)0x4B2423BD, (q31_t)0x67820BB6, (q31_t)0x4B4CCF4D,
	(q31_t)0x676471C0, (q31_t)0x4B756F3F, (q31_t)0x6746C7D7,
	(q31_t)0x4B9E038F, (q31_t)0x67290E02, (q31_t)0x4BC68C36,
	(q31_t)0x670B4443, (q31_t)0x4BEF092D, (q31_t)0x66ED6AA1,
	(q31_t)0x4C177A6E, (q31_t)0x66CF811F, (q31_t)0x4C3FDFF3,
	(q31_t)0x66B187C3, (q31_t)0x4C6839B6, (q31_t)0x66937E90,
	(q31_t)0x4C9087B1, (q31_t)0x6675658C, (q31_t)0x4CB8C9DD,
	(q31_t)0x66573CBB, (q31_t)0x4CE10034, (q31_t)0x66390422,
	(q31_t)0x4D092AB0, (q31_t)0x661ABBC5, (q31_t)0x4D31494B,
	(q31_t)0x65FC63A9, (q31_t)0x4D595BFE, (q31_t)0x65DDFBD3,
	(q31_t)0x4D8162C4, (q31_t)0x65BF8447, (q31_t)0x4DA95D96,
	(q31_t)0x65A0FD0B, (q31_t)0x4DD14C6E, (q31_t)0x65826622,
	(q31_t)0x4DF92F45, (q31_t)0x6563BF92, (q31_t)0x4E210617,
	(q31_t)0x6545095F, (q31_t)0x4E48D0DC, (q31_t)0x6526438E,
	(q31_t)0x4E708F8F, (q31_t)0x65076E24, (q31_t)0x4E984229,
	(q31_t)0x64E88926, (q31_t)0x4EBFE8A4, (q31_t)0x64C99498,
	(q31_t)0x4EE782FA, (q31_t)0x64AA907F, (q31_t)0x4F0F1126,
	(q31_t)0x648B7CDF, (q31_t)0x4F369320, (q31_t)0x646C59BF,
	(q31_t)0x4F5E08E3, (q31_t)0x644D2722, (q31_t)0x4F857268,
	(q31_t)0x642DE50D, (q31_t)0x4FACCFAB, (q31_t)0x640E9385,
	(q31_t)0x4FD420A3, (q31_t)0x63EF328F, (q31_t)0x4FFB654D,
	(q31_t)0x63CFC230, (q31_t)0x50229DA0, (q31_t)0x63B0426D,
	(q31_t)0x5049C999, (q31_t)0x6390B34A, (q31_t)0x5070E92F,
	(q31_t)0x637114CC, (q31_t)0x5097FC5E, (q31_t)0x635166F8,
	(q31_t)0x50BF031F, (q31_t)0x6331A9D4, (q31_t)0x50E5FD6C,
	(q31_t)0x6311DD63, (q31_t)0x510CEB40, (q31_t)0x62F201AC,
	(q31_t)0x5133CC94, (q31_t)0x62D216B2, (q31_t)0x515AA162,
	(q31_t)0x62B21C7B, (q31_t)0x518169A4, (q31_t)0x6292130C,
	(q31_t)0x51A82555, (q31_t)0x6271FA69, (q31_t)0x51CED46E,
	(q31_t)0x6251D297, (q31_t)0x51F576E9, (q31_t)0x62319B9D,
	(q31_t)0x521C0CC1, (q31_t)0x6211557D, (q31_t)0x524295EF,
	(q31_t)0x61F1003E, (q31_t)0x5269126E, (q31_t)0x61D09BE5,
	(q31_t)0x528F8237, (q31_t)0x61B02876, (q31_t)0x52B5E545,
	(q31_t)0x618FA5F6, (q31_t)0x52DC3B92, (q31_t)0x616F146B,
	(q31_t)0x53028517, (q31_t)0x614E73D9, (q31_t)0x5328C1D0,
	(q31_t)0x612DC446, (q31_t)0x534EF1B5, (q31_t)0x610D05B7,
	(q31_t)0x537514C1, (q31_t)0x60EC3830, (q31_t)0x539B2AEF,
	(q31_t)0x60CB5BB6, (q31_t)0x53C13438, (q31_t)0x60AA704F,
	(q31_t)0x53E73097, (q31_t)0x60897600, (q31_t)0x540D2005,
	(q31_t)0x60686CCE, (q31_t)0x5433027D, (q31_t)0x604754BE,
	(q31_t)0x5458D7F9, (q31_t)0x60262DD5, (q31_t)0x547EA073,
	(q31_t)0x6004F818, (q31_t)0x54A45BE5, (q31_t)0x5FE3B38D,
	(q31_t)0x54CA0A4A, (q31_t)0x5FC26038, (q31_t)0x54EFAB9C,
	(q31_t)0x5FA0FE1E, (q31_t)0x55153FD4, (q31_t)0x5F7F8D46,
	(q31_t)0x553AC6ED, (q31_t)0x5F5E0DB3, (q31_t)0x556040E2,
	(q31_t)0x5F3C7F6B, (q31_t)0x5585ADAC, (q31_t)0x5F1AE273,
	(q31_t)0x55AB0D46, (q31_t)0x5EF936D1, (q31_t)0x55D05FAA,
	(q31_t)0x5ED77C89, (q31_t)0x55F5A4D2, (q31_t)0x5EB5B3A1,
	(q31_t)0x561ADCB8, (q31_t)0x5E93DC1F, (q31_t)0x56400757,
	(q31_t)0x5E71F606, (q31_t)0x566524AA, (q31_t)0x5E50015D,
	(q31_t)0x568A34A9, (q31_t)0x5E2DFE28, (q31_t)0x56AF3750,
	(q31_t)0x5E0BEC6E, (q31_t)0x56D42C99, (q31_t)0x5DE9CC32,
	(q31_t)0x56F9147E, (q31_t)0x5DC79D7C, (q31_t)0x571DEEF9,
	(q31_t)0x5DA5604E, (q31_t)0x5742BC05, (q31_t)0x5D8314B0,
	(q31_t)0x57677B9D, (q31_t)0x5D60BAA6, (q31_t)0x578C2DB9,
	(q31_t)0x5D3E5236, (q31_t)0x57B0D256, (q31_t)0x5D1BDB65,
	(q31_t)0x57D5696C, (q31_t)0x5CF95638, (q31_t)0x57F9F2F7,
	(q31_t)0x5CD6C2B4, (q31_t)0x581E6EF1, (q31_t)0x5CB420DF,
	(q31_t)0x5842DD54, (q31_t)0x5C9170BF, (q31_t)0x58673E1B,
	(q31_t)0x5C6EB258, (q31_t)0x588B913F, (q31_t)0x5C4BE5B0,
	(q31_t)0x58AFD6BC, (q31_t)0x5C290ACC, (q31_t)0x58D40E8C,
	(q31_t)0x5C0621B2, (q31_t)0x58F838A9, (q31_t)0x5BE32A67,
	(q31_t)0x591C550E, (q31_t)0x5BC024F0, (q31_t)0x594063B4,
	(q31_t)0x5B9D1153, (q31_t)0x59646497, (q31_t)0x5B79EF96,
	(q31_t)0x598857B1, (q31_t)0x5B56BFBD, (q31_t)0x59AC3CFD,
	(q31_t)0x5B3381CE, (q31_t)0x59D01474, (q31_t)0x5B1035CF,
	(q31_t)0x59F3DE12, (q31_t)0x5AECDBC4, (q31_t)0x5A1799D0,
	(q31_t)0x5AC973B4, (q31_t)0x5A3B47AA, (q31_t)0x5AA5FDA4,
	(q31_t)0x5A5EE79A, (q31_t)0x5A82799A, (q31_t)0x5A82799A,
	(q31_t)0x5A5EE79A, (q31_t)0x5AA5FDA4, (q31_t)0x5A3B47AA,
	(q31_t)0x5AC973B4, (q31_t)0x5A1799D0, (q31_t)0x5AECDBC4,
	(q31_t)0x59F3DE12, (q31_t)0x5B1035CF, (q31_t)0x59D01474,
	(q31_t)0x5B3381CE, (q31_t)0x59AC3CFD, (q31_t)0x5B56BFBD,
	(q31_t)0x598857B1, (q31_t)0x5B79EF96, (q31_t)0x59646497,
	(q31_t)0x5B9D1153, (q31_t)0x594063B4, (q31_t)0x5BC024F0,
	(q31_t)0x591C550E, (q31_t)0x5BE32A67, (q31_t)0x58F838A9,
	(q31_t)0x5C0621B2, (q31_t)0x58D40E8C, (q31_t)0x5C290ACC,
	(q31_t)0x58AFD6BC, (q31_t)0x5C4BE5B0, (q31_t)0x588B913F,
	(q31_t)0x5C6EB258, (q31_t)0x58673E1B, (q31_t)0x5C9170BF,
	(q31_t)0x5842DD54, (q31_t)0x5CB420DF, (q31_t)0x581E6EF1,
	(q31_t)0x5CD6C2B4, (q31_t)0x57F9F2F7, (q31_t)0x5CF95638,
	(q31_t)0x57D5696C, (q31_t)0x5D1BDB65, (q31_t)0x57B0D256,
	(q31_t)0x5D3E5236, (q31_t)0x578C2DB9, (q31_t)0x5D60BAA6,
	(q31_t)0x57677B9D, (q31_t)0x5D8314B0, (q31_t)0x5742BC05,
	(q31_t)0x5DA5604E, (q31_t)0x571DEEF9, (q31_t)0x5DC79D7C,
	(q31_t)0x56F9147E, (q31_t)0x5DE9CC32, (q31_t)0x56D42C99,
	(q31_t)0x5E0BEC6E, (q31_t)0x56AF3750, (q31_t)0x5E2DFE28,
	(q31_t)0x568A34A9, (q31_t)0x5E50015D, (q31_t)0x566524AA,
	(q31_t)0x5E71F606, (q31_t)0x56400757, (q31_t)0x5E93DC1F,
	(q31_t)0x561ADCB8, (q31_t)0x5EB5B3A1, (q31_t)0x55F5A4D2,
	(q31_t)0x5ED77C89, (q31_t)0x55D05FAA, (q31_t)0x5EF936D1,
	(q31_t)0x55AB0D46, (q31_t)0x5F1AE273, (q31_t)0x5585ADAC,
	(q31_t)0x5F3C7F6B, (q31_t)0x556040E2, (q31_t)0x5F5E0DB3,
	(q31_t)0x553AC6ED, (q31_t)0x5F7F8D46, (q31_t)0x55153FD4,
	(q31_t)0x5FA0FE1E, (q31_t)0x54EFAB9C, (q31_t)0x5FC26038,
	(q31_t)0x54CA0A4A, (q31_t)0x5FE3B38D, (q31_t)0x54A45BE5,
	(q31_t)0x6004F818, (q31_t)0x547EA073, (q31_t)0x60262DD5,
	(q31_t)0x5458D7F9, (q31_t)0x604754BE, (q31_t)0x5433027D,
	(q31_t)0x60686CCE, (q31_t)0x540D2005, (q31_t)0x60897600,
	(q31_t)0x53E73097, (q31_t)0x60AA704F, (q31_t)0x53C13438,
	(q31_t)0x60CB5BB6, (q31_t)0x539B2AEF, (q31_t)0x60EC3830,
	(q31_t)0x537514C1, (q31_t)0x610D05B7, (q31_t)0x534EF1B5,
	(q31_t)0x612DC446, (q31_t)0x5328C1D0, (q31_t)0x614E73D9,
	(q31_t)0x53028517, (q31_t)0x616F146B, (q31_t)0x52DC3B92,
	(q31_t)0x618FA5F6, (q31_t)0x52B5E545, (q31_t)0x61B02876,
	(q31_t)0x528F8237, (q31_t)0x61D09BE5, (q31_t)0x5269126E,
	(q31_t)0x61F1003E, (q31_t)0x524295EF, (q31_t)0x6211557D,
	(q31_t)0x521C0CC1, (q31_t)0x62319B9D, (q31_t)0x51F576E9,
	(q31_t)0x6251D297, (q31_t)0x51CED46E, (q31_t)0x6271FA69,
	(q31_t)0x51A82555, (q31_t)0x6292130C, (q31_t)0x518169A4,
	(q31_t)0x62B21C7B, (q31_t)0x515AA162, (q31_t)0x62D216B2,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x510CEB40,
	(q31_t)0x6311DD63, (q31_t)0x50E5FD6C, (q31_t)0x6331A9D4,
	(q31_t)0x50BF031F, (q31_t)0x635166F8, (q31_t)0x5097FC5E,
	(q31_t)0x637114CC, (q31_t)0x5070E92F, (q31_t)0x6390B34A,
	(q31_t)0x5049C999, (q31_t)0x63B0426D, (q31_t)0x50229DA0,
	(q31_t)0x63CFC230, (q31_t)0x4FFB654D, (q31_t)0x63EF328F,
	(q31_t)0x4FD420A3, (q31_t)0x640E9385, (q31_t)0x4FACCFAB,
	(q31_t)0x642DE50D, (q31_t)0x4F857268, (q31_t)0x644D2722,
	(q31_t)0x4F5E08E3, (q31_t)0x646C59BF, (q31_t)0x4F369320,
	(q31_t)0x648B7CDF, (q31_t)0x4F0F1126, (q31_t)0x64AA907F,
	(q31_t)0x4EE782FA, (q31_t)0x64C99498, (q31_t)0x4EBFE8A4,
	(q31_t)0x64E88926, (q31_t)0x4E984229, (q31_t)0x65076E24,
	(q31_t)0x4E708F8F, (q31_t)0x6526438E, (q31_t)0x4E48D0DC,
	(q31_t)0x6545095F, (q31_t)0x4E210617, (q31_t)0x6563BF92,
	(q31_t)0x4DF92F45, (q31_t)0x65826622, (q31_t)0x4DD14C6E,
	(q31_t)0x65A0FD0B, (q31_t)0x4DA95D96, (q31_t)0x65BF8447,
	(q31_t)0x4D8162C4, (q31_t)0x65DDFBD3, (q31_t)0x4D595BFE,
	(q31_t)0x65FC63A9, (q31_t)0x4D31494B, (q31_t)0x661ABBC5,
	(q31_t)0x4D092AB0, (q31_t)0x66390422, (q31_t)0x4CE10034,
	(q31_t)0x66573CBB, (q31_t)0x4CB8C9DD, (q31_t)0x6675658C,
	(q31_t)0x4C9087B1, (q31_t)0x66937E90, (q31_t)0x4C6839B6,
	(q31_t)0x66B187C3, (q31_t)0x4C3FDFF3, (q31_t)0x66CF811F,
	(q31_t)0x4C177A6E, (q31_t)0x66ED6AA1, (q31_t)0x4BEF092D,
	(q31_t)0x670B4443, (q31_t)0x4BC68C36, (q31_t)0x67290E02,
	(q31_t)0x4B9E038F, (q31_t)0x6746C7D7, (q31_t)0x4B756F3F,
	(q31_t)0x676471C0, (q31_t)0x4B4CCF4D, (q31_t)0x67820BB6,
	(q31_t)0x4B2423BD, (q31_t)0x679F95B7, (q31_t)0x4AFB6C97,
	(q31_t)0x67BD0FBC, (q31_t)0x4AD2A9E1, (q31_t)0x67DA79C2,
	(q31_t)0x4AA9DBA1, (q31_t)0x67F7D3C4, (q31_t)0x4A8101DE,
	(q31_t)0x68151DBE, (q31_t)0x4A581C9D, (q31_t)0x683257AA,
	(q31_t)0x4A2F2BE5, (q31_t)0x684F8186, (q31_t)0x4A062FBD,
	(q31_t)0x686C9B4B, (q31_t)0x49DD282A, (q31_t)0x6889A4F5,
	(q31_t)0x49B41533, (q31_t)0x68A69E81, (q31_t)0x498AF6DE,
	(q31_t)0x68C387E9, (q31_t)0x4961CD32, (q31_t)0x68E06129,
	(q31_t)0x49389836, (q31_t)0x68FD2A3D, (q31_t)0x490F57EE,
	(q31_t)0x6919E320, (q31_t)0x48E60C62, (q31_t)0x69368BCE,
	(q31_t)0x48BCB598, (q31_t)0x69532442, (q31_t)0x48935397,
	(q31_t)0x696FAC78, (q31_t)0x4869E664, (q31_t)0x698C246C,
	(q31_t)0x48406E07, (q31_t)0x69A88C18, (q31_t)0x4816EA85,
	(q31_t)0x69C4E37A, (q31_t)0x47ED5BE6, (q31_t)0x69E12A8C,
	(q31_t)0x47C3C22E, (q31_t)0x69FD614A, (q31_t)0x479A1D66,
	(q31_t)0x6A1987B0, (q31_t)0x47706D93, (q31_t)0x6A359DB9,
	(q31_t)0x4746B2BC, (q31_t)0x6A51A361, (q31_t)0x471CECE6,
	(q31_t)0x6A6D98A4, (q31_t)0x46F31C1A, (q31_t)0x6A897D7D,
	(q31_t)0x46C9405C, (q31_t)0x6AA551E8, (q31_t)0x469F59B4,
	(q31_t)0x6AC115E1, (q31_t)0x46756827, (q31_t)0x6ADCC964,
	(q31_t)0x464B6BBD, (q31_t)0x6AF86C6C, (q31_t)0x4621647C,
	(q31_t)0x6B13FEF5, (q31_t)0x45F7526B, (q31_t)0x6B2F80FA,
	(q31_t)0x45CD358F, (q31_t)0x6B4AF278, (q31_t)0x45A30DF0,
	(q31_t)0x6B66536A, (q31_t)0x4578DB93, (q31_t)0x6B81A3CD,
	(q31_t)0x454E9E80, (q31_t)0x6B9CE39B, (q31_t)0x452456BC,
	(q31_t)0x6BB812D0, (q31_t)0x44FA044F, (q31_t)0x6BD3316A,
	(q31_t)0x44CFA73F, (q31_t)0x6BEE3F62, (q31_t)0x44A53F93,
	(q31_t)0x6C093CB6, (q31_t)0x447ACD50, (q31_t)0x6C242960,
	(q31_t)0x4450507E, (q31_t)0x6C3F055D, (q31_t)0x4425C923,
	(q31_t)0x6C59D0A9, (q31_t)0x43FB3745, (q31_t)0x6C748B3F,
	(q31_t)0x43D09AEC, (q31_t)0x6C8F351C, (q31_t)0x43A5F41E,
	(q31_t)0x6CA9CE3A, (q31_t)0x437B42E1, (q31_t)0x6CC45697,
	(q31_t)0x4350873C, (q31_t)0x6CDECE2E, (q31_t)0x4325C135,
	(q31_t)0x6CF934FB, (q31_t)0x42FAF0D4, (q31_t)0x6D138AFA,
	(q31_t)0x42D0161E, (q31_t)0x6D2DD027, (q31_t)0x42A5311A,
	(q31_t)0x6D48047E, (q31_t)0x427A41D0, (q31_t)0x6D6227FA,
	(q31_t)0x424F4845, (q31_t)0x6D7C3A98, (q31_t)0x42244480,
	(q31_t)0x6D963C54, (q31_t)0x41F93688, (q31_t)0x6DB02D29,
	(q31_t)0x41CE1E64, (q31_t)0x6DCA0D14, (q31_t)0x41A2FC1A,
	(q31_t)0x6DE3DC11, (q31_t)0x4177CFB0, (q31_t)0x6DFD9A1B,
	(q31_t)0x414C992E, (q31_t)0x6E17472F, (q31_t)0x4121589A,
	(q31_t)0x6E30E349, (q31_t)0x40F60DFB, (q31_t)0x6E4A6E65,
	(q31_t)0x40CAB957, (q31_t)0x6E63E87F, (q31_t)0x409F5AB6,
	(q31_t)0x6E7D5193, (q31_t)0x4073F21D, (q31_t)0x6E96A99C,
	(q31_t)0x40487F93, (q31_t)0x6EAFF098, (q31_t)0x401D0320,
	(q31_t)0x6EC92682, (q31_t)0x3FF17CCA, (q31_t)0x6EE24B57,
	(q31_t)0x3FC5EC97, (q31_t)0x6EFB5F12, (q31_t)0x3F9A528F,
	(q31_t)0x6F1461AF, (q31_t)0x3F6EAEB8, (q31_t)0x6F2D532C,
	(q31_t)0x3F430118, (q31_t)0x6F463383, (q31_t)0x3F1749B7,
	(q31_t)0x6F5F02B1, (q31_t)0x3EEB889C, (q31_t)0x6F77C0B3,
	(q31_t)0x3EBFBDCC, (q31_t)0x6F906D84, (q31_t)0x3E93E94F,
	(q31_t)0x6FA90920, (q31_t)0x3E680B2C, (q31_t)0x6FC19385,
	(q31_t)0x3E3C2369, (q31_t)0x6FDA0CAD, (q31_t)0x3E10320D,
	(q31_t)0x6FF27496, (q31_t)0x3DE4371F, (q31_t)0x700ACB3B,
	(q31_t)0x3DB832A5, (q31_t)0x70231099, (q31_t)0x3D8C24A7,
	(q31_t)0x703B44AC, (q31_t)0x3D600D2B, (q31_t)0x70536771,
	(q31_t)0x3D33EC39, (q31_t)0x706B78E3, (q31_t)0x3D07C1D5,
	(q31_t)0x708378FE, (q31_t)0x3CDB8E09, (q31_t)0x709B67C0,
	(q31_t)0x3CAF50DA, (q31_t)0x70B34524, (q31_t)0x3C830A4F,
	(q31_t)0x70CB1127, (q31_t)0x3C56BA70, (q31_t)0x70E2CBC6,
	(q31_t)0x3C2A6142, (q31_t)0x70FA74FB, (q31_t)0x3BFDFECD,
	(q31_t)0x71120CC5, (q31_t)0x3BD19317, (q31_t)0x7129931E,
	(q31_t)0x3BA51E29, (q31_t)0x71410804, (q31_t)0x3B78A007,
	(q31_t)0x71586B73, (q31_t)0x3B4C18BA, (q31_t)0x716FBD68,
	(q31_t)0x3B1F8847, (q31_t)0x7186FDDE, (q31_t)0x3AF2EEB7,
	(q31_t)0x719E2CD2, (q31_t)0x3AC64C0F, (q31_t)0x71B54A40,
	(q31_t)0x3A99A057, (q31_t)0x71CC5626, (q31_t)0x3A6CEB95,
	(q31_t)0x71E3507F, (q31_t)0x3A402DD1, (q31_t)0x71FA3948,
	(q31_t)0x3A136712, (q31_t)0x7211107D, (q31_t)0x39E6975D,
	(q31_t)0x7227D61C, (q31_t)0x39B9BEBB, (q31_t)0x723E8A1F,
	(q31_t)0x398CDD32, (q31_t)0x72552C84, (q31_t)0x395FF2C9,
	(q31_t)0x726BBD48, (q31_t)0x3932FF87, (q31_t)0x72823C66,
	(q31_t)0x39060372, (q31_t)0x7298A9DC, (q31_t)0x38D8FE93,
	(q31_t)0x72AF05A6, (q31_t)0x38ABF0EF, (q31_t)0x72C54FC0,
	(q31_t)0x387EDA8E, (q31_t)0x72DB8828, (q31_t)0x3851BB76,
	(q31_t)0x72F1AED8, (q31_t)0x382493B0, (q31_t)0x7307C3D0,
	(q31_t)0x37F76340, (q31_t)0x731DC709, (q31_t)0x37CA2A30,
	(q31_t)0x7333B883, (q31_t)0x379CE884, (q31_t)0x73499838,
	(q31_t)0x376F9E46, (q31_t)0x735F6626, (q31_t)0x37424B7A,
	(q31_t)0x73752249, (q31_t)0x3714F02A, (q31_t)0x738ACC9E,
	(q31_t)0x36E78C5A, (q31_t)0x73A06522, (q31_t)0x36BA2013,
	(q31_t)0x73B5EBD0, (q31_t)0x368CAB5C, (q31_t)0x73CB60A7,
	(q31_t)0x365F2E3B, (q31_t)0x73E0C3A3, (q31_t)0x3631A8B7,
	(q31_t)0x73F614C0, (q31_t)0x36041AD9, (q31_t)0x740B53FA,
	(q31_t)0x35D684A5, (q31_t)0x74208150, (q31_t)0x35A8E624,
	(q31_t)0x74359CBD, (q31_t)0x357B3F5D, (q31_t)0x744AA63E,
	(q31_t)0x354D9056, (q31_t)0x745F9DD1, (q31_t)0x351FD917,
	(q31_t)0x74748371, (q31_t)0x34F219A7, (q31_t)0x7489571B,
	(q31_t)0x34C4520D, (q31_t)0x749E18CD, (q31_t)0x3496824F,
	(q31_t)0x74B2C883, (q31_t)0x3468AA76, (q31_t)0x74C7663A,
	(q31_t)0x343ACA87, (q31_t)0x74DBF1EF, (q31_t)0x340CE28A,
	(q31_t)0x74F06B9E, (q31_t)0x33DEF287, (q31_t)0x7504D345,
	(q31_t)0x33B0FA84, (q31_t)0x751928E0, (q31_t)0x3382FA88,
	(q31_t)0x752D6C6C, (q31_t)0x3354F29A, (q31_t)0x75419DE6,
	(q31_t)0x3326E2C2, (q31_t)0x7555BD4B, (q31_t)0x32F8CB07,
	(q31_t)0x7569CA98, (q31_t)0x32CAAB6F, (q31_t)0x757DC5CA,
	(q31_t)0x329C8402, (q31_t)0x7591AEDD, (q31_t)0x326E54C7,
	(q31_t)0x75A585CF, (q31_t)0x32401DC5, (q31_t)0x75B94A9C,
	(q31_t)0x3211DF03, (q31_t)0x75CCFD42, (q31_t)0x31E39889,
	(q31_t)0x75E09DBD, (q31_t)0x31B54A5D, (q31_t)0x75F42C0A,
	(q31_t)0x3186F487, (q31_t)0x7607A827, (q31_t)0x3158970D,
	(q31_t)0x761B1211, (q31_t)0x312A31F8, (q31_t)0x762E69C3,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x30CD5114,
	(q31_t)0x7654E279, (q31_t)0x309ED555, (q31_t)0x76680376,
	(q31_t)0x30705217, (q31_t)0x767B1230, (q31_t)0x3041C760,
	(q31_t)0x768E0EA5, (q31_t)0x30133538, (q31_t)0x76A0F8D2,
	(q31_t)0x2FE49BA6, (q31_t)0x76B3D0B3, (q31_t)0x2FB5FAB2,
	(q31_t)0x76C69646, (q31_t)0x2F875262, (q31_t)0x76D94988,
	(q31_t)0x2F58A2BD, (q31_t)0x76EBEA77, (q31_t)0x2F29EBCC,
	(q31_t)0x76FE790E, (q31_t)0x2EFB2D94, (q31_t)0x7710F54B,
	(q31_t)0x2ECC681E, (q31_t)0x77235F2D, (q31_t)0x2E9D9B70,
	(q31_t)0x7735B6AE, (q31_t)0x2E6EC792, (q31_t)0x7747FBCE,
	(q31_t)0x2E3FEC8B, (q31_t)0x775A2E88, (q31_t)0x2E110A62,
	(q31_t)0x776C4EDB, (q31_t)0x2DE2211E, (q31_t)0x777E5CC3,
	(q31_t)0x2DB330C7, (q31_t)0x7790583D, (q31_t)0x2D843963,
	(q31_t)0x77A24148, (q31_t)0x2D553AFB, (q31_t)0x77B417DF,
	(q31_t)0x2D263595, (q31_t)0x77C5DC01, (q31_t)0x2CF72939,
	(q31_t)0x77D78DAA, (q31_t)0x2CC815ED, (q31_t)0x77E92CD8,
	(q31_t)0x2C98FBBA, (q31_t)0x77FAB988, (q31_t)0x2C69DAA6,
	(q31_t)0x780C33B8, (q31_t)0x2C3AB2B9, (q31_t)0x781D9B64,
	(q31_t)0x2C0B83F9, (q31_t)0x782EF08B, (q31_t)0x2BDC4E6F,
	(q31_t)0x78403328, (q31_t)0x2BAD1221, (q31_t)0x7851633B,
	(q31_t)0x2B7DCF17, (q31_t)0x786280BF, (q31_t)0x2B4E8558,
	(q31_t)0x78738BB3, (q31_t)0x2B1F34EB, (q31_t)0x78848413,
	(q31_t)0x2AEFDDD8, (q31_t)0x789569DE, (q31_t)0x2AC08025,
	(q31_t)0x78A63D10, (q31_t)0x2A911BDB, (q31_t)0x78B6FDA8,
	(q31_t)0x2A61B101, (q31_t)0x78C7ABA1, (q31_t)0x2A323F9D,
	(q31_t)0x78D846FB, (q31_t)0x2A02C7B8, (q31_t)0x78E8CFB1,
	(q31_t)0x29D34958, (q31_t)0x78F945C3, (q31_t)0x29A3C484,
	(q31_t)0x7909A92C, (q31_t)0x29743945, (q31_t)0x7919F9EB,
	(q31_t)0x2944A7A2, (q31_t)0x792A37FE, (q31_t)0x29150FA1,
	(q31_t)0x793A6360, (q31_t)0x28E5714A, (q31_t)0x794A7C11,
	(q31_t)0x28B5CCA5, (q31_t)0x795A820E, (q31_t)0x288621B9,
	(q31_t)0x796A7554, (q31_t)0x2856708C, (q31_t)0x797A55E0,
	(q31_t)0x2826B928, (q31_t)0x798A23B1, (q31_t)0x27F6FB92,
	(q31_t)0x7999DEC3, (q31_t)0x27C737D2, (q31_t)0x79A98715,
	(q31_t)0x27976DF1, (q31_t)0x79B91CA4, (q31_t)0x27679DF4,
	(q31_t)0x79C89F6D, (q31_t)0x2737C7E3, (q31_t)0x79D80F6F,
	(q31_t)0x2707EBC6, (q31_t)0x79E76CA6, (q31_t)0x26D809A5,
	(q31_t)0x79F6B711, (q31_t)0x26A82185, (q31_t)0x7A05EEAD,
	(q31_t)0x26783370, (q31_t)0x7A151377, (q31_t)0x26483F6C,
	(q31_t)0x7A24256E, (q31_t)0x26184581, (q31_t)0x7A33248F,
	(q31_t)0x25E845B5, (q31_t)0x7A4210D8, (q31_t)0x25B84012,
	(q31_t)0x7A50EA46, (q31_t)0x2588349D, (q31_t)0x7A5FB0D8,
	(q31_t)0x2558235E, (q31_t)0x7A6E648A, (q31_t)0x25280C5D,
	(q31_t)0x7A7D055B, (q31_t)0x24F7EFA1, (q31_t)0x7A8B9348,
	(q31_t)0x24C7CD32, (q31_t)0x7A9A0E4F, (q31_t)0x2497A517,
	(q31_t)0x7AA8766E, (q31_t)0x24677757, (q31_t)0x7AB6CBA3,
	(q31_t)0x243743FA, (q31_t)0x7AC50DEB, (q31_t)0x24070B07,
	(q31_t)0x7AD33D45, (q31_t)0x23D6CC86, (q31_t)0x7AE159AE,
	(q31_t)0x23A6887E, (q31_t)0x7AEF6323, (q31_t)0x23763EF7,
	(q31_t)0x7AFD59A3, (q31_t)0x2345EFF7, (q31_t)0x7B0B3D2C,
	(q31_t)0x23159B87, (q31_t)0x7B190DBB, (q31_t)0x22E541AE,
	(q31_t)0x7B26CB4F, (q31_t)0x22B4E274, (q31_t)0x7B3475E4,
	(q31_t)0x22847DDF, (q31_t)0x7B420D7A, (q31_t)0x225413F8,
	(q31_t)0x7B4F920E, (q31_t)0x2223A4C5, (q31_t)0x7B5D039D,
	(q31_t)0x21F3304E, (q31_t)0x7B6A6227, (q31_t)0x21C2B69C,
	(q31_t)0x7B77ADA8, (q31_t)0x219237B4, (q31_t)0x7B84E61E,
	(q31_t)0x2161B39F, (q31_t)0x7B920B89, (q31_t)0x21312A65,
	(q31_t)0x7B9F1DE5, (q31_t)0x21009C0B, (q31_t)0x7BAC1D31,
	(q31_t)0x20D0089B, (q31_t)0x7BB9096A, (q31_t)0x209F701C,
	(q31_t)0x7BC5E28F, (q31_t)0x206ED295, (q31_t)0x7BD2A89E,
	(q31_t)0x203E300D, (q31_t)0x7BDF5B94, (q31_t)0x200D888C,
	(q31_t)0x7BEBFB70, (q31_t)0x1FDCDC1A, (q31_t)0x7BF88830,
	(q31_t)0x1FAC2ABF, (q31_t)0x7C0501D1, (q31_t)0x1F7B7480,
	(q31_t)0x7C116853, (q31_t)0x1F4AB967, (q31_t)0x7C1DBBB2,
	(q31_t)0x1F19F97B, (q31_t)0x7C29FBEE, (q31_t)0x1EE934C2,
	(q31_t)0x7C362904, (q31_t)0x1EB86B46, (q31_t)0x7C4242F2,
	(q31_t)0x1E879D0C, (q31_t)0x7C4E49B6, (q31_t)0x1E56CA1E,
	(q31_t)0x7C5A3D4F, (q31_t)0x1E25F281, (q31_t)0x7C661DBB,
	(q31_t)0x1DF5163F, (q31_t)0x7C71EAF8, (q31_t)0x1DC4355D,
	(q31_t)0x7C7DA504, (q31_t)0x1D934FE5, (q31_t)0x7C894BDD,
	(q31_t)0x1D6265DD, (q31_t)0x7C94DF82, (q31_t)0x1D31774D,
	(q31_t)0x7CA05FF1, (q31_t)0x1D00843C, (q31_t)0x7CABCD27,
	(q31_t)0x1CCF8CB3, (q31_t)0x7CB72724, (q31_t)0x1C9E90B8,
	(q31_t)0x7CC26DE5, (q31_t)0x1C6D9053, (q31_t)0x7CCDA168,
	(q31_t)0x1C3C8B8C, (q31_t)0x7CD8C1AD, (q31_t)0x1C0B826A,
	(q31_t)0x7CE3CEB1, (q31_t)0x1BDA74F5, (q31_t)0x7CEEC873,
	(q31_t)0x1BA96334, (q31_t)0x7CF9AEF0, (q31_t)0x1B784D30,
	(q31_t)0x7D048228, (q31_t)0x1B4732EF, (q31_t)0x7D0F4218,
	(q31_t)0x1B161479, (q31_t)0x7D19EEBE, (q31_t)0x1AE4F1D6,
	(q31_t)0x7D24881A, (q31_t)0x1AB3CB0C, (q31_t)0x7D2F0E2A,
	(q31_t)0x1A82A025, (q31_t)0x7D3980EC, (q31_t)0x1A517127,
	(q31_t)0x7D43E05E, (q31_t)0x1A203E1B, (q31_t)0x7D4E2C7E,
	(q31_t)0x19EF0706, (q31_t)0x7D58654C, (q31_t)0x19BDCBF2,
	(q31_t)0x7D628AC5, (q31_t)0x198C8CE6, (q31_t)0x7D6C9CE9,
	(q31_t)0x195B49E9, (q31_t)0x7D769BB5, (q31_t)0x192A0303,
	(q31_t)0x7D808727, (q31_t)0x18F8B83C, (q31_t)0x7D8A5F3F,
	(q31_t)0x18C7699B, (q31_t)0x7D9423FB, (q31_t)0x18961727,
	(q31_t)0x7D9DD55A, (q31_t)0x1864C0E9, (q31_t)0x7DA77359,
	(q31_t)0x183366E8, (q31_t)0x7DB0FDF7, (q31_t)0x1802092C,
	(q31_t)0x7DBA7534, (q31_t)0x17D0A7BB, (q31_t)0x7DC3D90D,
	(q31_t)0x179F429F, (q31_t)0x7DCD2981, (q31_t)0x176DD9DE,
	(q31_t)0x7DD6668E, (q31_t)0x173C6D80, (q31_t)0x7DDF9034,
	(q31_t)0x170AFD8D, (q31_t)0x7DE8A670, (q31_t)0x16D98A0C,
	(q31_t)0x7DF1A942, (q31_t)0x16A81305, (q31_t)0x7DFA98A7,
	(q31_t)0x1676987F, (q31_t)0x7E03749F, (q31_t)0x16451A83,
	(q31_t)0x7E0C3D29, (q31_t)0x16139917, (q31_t)0x7E14F242,
	(q31_t)0x15E21444, (q31_t)0x7E1D93E9, (q31_t)0x15B08C11,
	(q31_t)0x7E26221E, (q31_t)0x157F0086, (q31_t)0x7E2E9CDF,
	(q31_t)0x154D71AA, (q31_t)0x7E37042A, (q31_t)0x151BDF85,
	(q31_t)0x7E3F57FE, (q31_t)0x14EA4A1F, (q31_t)0x7E47985B,
	(q31_t)0x14B8B17F, (q31_t)0x7E4FC53E, (q31_t)0x148715AD,
	(q31_t)0x7E57DEA6, (q31_t)0x145576B1, (q31_t)0x7E5FE493,
	(q31_t)0x1423D492, (q31_t)0x7E67D702, (q31_t)0x13F22F57,
	(q31_t)0x7E6FB5F3, (q31_t)0x13C0870A, (q31_t)0x7E778165,
	(q31_t)0x138EDBB0, (q31_t)0x7E7F3956, (q31_t)0x135D2D53,
	(q31_t)0x7E86DDC5, (q31_t)0x132B7BF9, (q31_t)0x7E8E6EB1,
	(q31_t)0x12F9C7AA, (q31_t)0x7E95EC19, (q31_t)0x12C8106E,
	(q31_t)0x7E9D55FC, (q31_t)0x1296564D, (q31_t)0x7EA4AC58,
	(q31_t)0x1264994E, (q31_t)0x7EABEF2C, (q31_t)0x1232D978,
	(q31_t)0x7EB31E77, (q31_t)0x120116D4, (q31_t)0x7EBA3A39,
	(q31_t)0x11CF516A, (q31_t)0x7EC1426F, (q31_t)0x119D8940,
	(q31_t)0x7EC8371A, (q31_t)0x116BBE5F, (q31_t)0x7ECF1837,
	(q31_t)0x1139F0CE, (q31_t)0x7ED5E5C6, (q31_t)0x11082096,
	(q31_t)0x7EDC9FC6, (q31_t)0x10D64DBC, (q31_t)0x7EE34635,
	(q31_t)0x10A4784A, (q31_t)0x7EE9D913, (q31_t)0x1072A047,
	(q31_t)0x7EF0585F, (q31_t)0x1040C5BB, (q31_t)0x7EF6C418,
	(q31_t)0x100EE8AD, (q31_t)0x7EFD1C3C, (q31_t)0x0FDD0925,
	(q31_t)0x7F0360CB, (q31_t)0x0FAB272B, (q31_t)0x7F0991C3,
	(q31_t)0x0F7942C6, (q31_t)0x7F0FAF24, (q31_t)0x0F475BFE,
	(q31_t)0x7F15B8EE, (q31_t)0x0F1572DC, (q31_t)0x7F1BAF1E,
	(q31_t)0x0EE38765, (q31_t)0x7F2191B4, (q31_t)0x0EB199A3,
	(q31_t)0x7F2760AF, (q31_t)0x0E7FA99D, (q31_t)0x7F2D1C0E,
	(q31_t)0x0E4DB75B, (q31_t)0x7F32C3D0, (q31_t)0x0E1BC2E3,
	(q31_t)0x7F3857F5, (q31_t)0x0DE9CC3F, (q31_t)0x7F3DD87C,
	(q31_t)0x0DB7D376, (q31_t)0x7F434563, (q31_t)0x0D85D88F,
	(q31_t)0x7F489EAA, (q31_t)0x0D53DB92, (q31_t)0x7F4DE450,
	(q31_t)0x0D21DC87, (q31_t)0x7F531654, (q31_t)0x0CEFDB75,
	(q31_t)0x7F5834B6, (q31_t)0x0CBDD865, (q31_t)0x7F5D3F75,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x0C59CC67,
	(q31_t)0x7F671A04, (q31_t)0x0C27C389, (q31_t)0x7F6BE9D4,
	(q31_t)0x0BF5B8CB, (q31_t)0x7F70A5FD, (q31_t)0x0BC3AC35,
	(q31_t)0x7F754E7F, (q31_t)0x0B919DCE, (q31_t)0x7F79E35A,
	(q31_t)0x0B5F8D9F, (q31_t)0x7F7E648B, (q31_t)0x0B2D7BAE,
	(q31_t)0x7F82D214, (q31_t)0x0AFB6805, (q31_t)0x7F872BF3,
	(q31_t)0x0AC952AA, (q31_t)0x7F8B7226, (q31_t)0x0A973BA5,
	(q31_t)0x7F8FA4AF, (q31_t)0x0A6522FE, (q31_t)0x7F93C38C,
	(q31_t)0x0A3308BC, (q31_t)0x7F97CEBC, (q31_t)0x0A00ECE8,
	(q31_t)0x7F9BC63F, (q31_t)0x09CECF89, (q31_t)0x7F9FAA15,
	(q31_t)0x099CB0A7, (q31_t)0x7FA37A3C, (q31_t)0x096A9049,
	(q31_t)0x7FA736B4, (q31_t)0x09386E77, (q31_t)0x7FAADF7C,
	(q31_t)0x09064B3A, (q31_t)0x7FAE7494, (q31_t)0x08D42698,
	(q31_t)0x7FB1F5FC, (q31_t)0x08A2009A, (q31_t)0x7FB563B2,
	(q31_t)0x086FD947, (q31_t)0x7FB8BDB7, (q31_t)0x083DB0A7,
	(q31_t)0x7FBC040A, (q31_t)0x080B86C1, (q31_t)0x7FBF36A9,
	(q31_t)0x07D95B9E, (q31_t)0x7FC25596, (q31_t)0x07A72F45,
	(q31_t)0x7FC560CF, (q31_t)0x077501BE, (q31_t)0x7FC85853,
	(q31_t)0x0742D310, (q31_t)0x7FCB3C23, (q31_t)0x0710A344,
	(q31_t)0x7FCE0C3E, (q31_t)0x06DE7261, (q31_t)0x7FD0C8A3,
	(q31_t)0x06AC406F, (q31_t)0x7FD37152, (q31_t)0x067A0D75,
	(q31_t)0x7FD6064B, (q31_t)0x0647D97C, (q31_t)0x7FD8878D,
	(q31_t)0x0615A48A, (q31_t)0x7FDAF518, (q31_t)0x05E36EA9,
	(q31_t)0x7FDD4EEC, (q31_t)0x05B137DF, (q31_t)0x7FDF9508,
	(q31_t)0x057F0034, (q31_t)0x7FE1C76B, (q31_t)0x054CC7B0,
	(q31_t)0x7FE3E616, (q31_t)0x051A8E5C, (q31_t)0x7FE5F108,
	(q31_t)0x04E8543D, (q31_t)0x7FE7E840, (q31_t)0x04B6195D,
	(q31_t)0x7FE9CBC0, (q31_t)0x0483DDC3, (q31_t)0x7FEB9B85,
	(q31_t)0x0451A176, (q31_t)0x7FED5790, (q31_t)0x041F647F,
	(q31_t)0x7FEEFFE1, (q31_t)0x03ED26E6, (q31_t)0x7FF09477,
	(q31_t)0x03BAE8B1, (q31_t)0x7FF21553, (q31_t)0x0388A9E9,
	(q31_t)0x7FF38273, (q31_t)0x03566A96, (q31_t)0x7FF4DBD8,
	(q31_t)0x03242ABF, (q31_t)0x7FF62182, (q31_t)0x02F1EA6B,
	(q31_t)0x7FF7536F, (q31_t)0x02BFA9A4, (q31_t)0x7FF871A1,
	(q31_t)0x028D6870, (q31_t)0x7FF97C17, (q31_t)0x025B26D7,
	(q31_t)0x7FFA72D1, (q31_t)0x0228E4E1, (q31_t)0x7FFB55CE,
	(q31_t)0x01F6A296, (q31_t)0x7FFC250F, (q31_t)0x01C45FFE,
	(q31_t)0x7FFCE093, (q31_t)0x01921D1F, (q31_t)0x7FFD885A,
	(q31_t)0x015FDA03, (q31_t)0x7FFE1C64, (q31_t)0x012D96B0,
	(q31_t)0x7FFE9CB2, (q31_t)0x00FB532F, (q31_t)0x7FFF0942,
	(q31_t)0x00C90F88, (q31_t)0x7FFF6216, (q31_t)0x0096CBC1,
	(q31_t)0x7FFFA72C, (q31_t)0x006487E3, (q31_t)0x7FFFD885,
	(q31_t)0x003243F5, (q31_t)0x7FFFF621, (q31_t)0x00000000,
	(q31_t)0x7FFFFFFF, (q31_t)0xFFCDBC0A, (q31_t)0x7FFFF621,
	(q31_t)0xFF9B781D, (q31_t)0x7FFFD885, (q31_t)0xFF69343E,
	(q31_t)0x7FFFA72C, (q31_t)0xFF36F078, (q31_t)0x7FFF6216,
	(q31_t)0xFF04ACD0, (q31_t)0x7FFF0942, (q31_t)0xFED2694F,
	(q31_t)0x7FFE9CB2, (q31_t)0xFEA025FC, (q31_t)0x7FFE1C64,
	(q31_t)0xFE6DE2E0, (q31_t)0x7FFD885A, (q31_t)0xFE3BA001,
	(q31_t)0x7FFCE093, (q31_t)0xFE095D69, (q31_t)0x7FFC250F,
	(q31_t)0xFDD71B1E, (q31_t)0x7FFB55CE, (q31_t)0xFDA4D928,
	(q31_t)0x7FFA72D1, (q31_t)0xFD72978F, (q31_t)0x7FF97C17,
	(q31_t)0xFD40565B, (q31_t)0x7FF871A1, (q31_t)0xFD0E1594,
	(q31_t)0x7FF7536F, (q31_t)0xFCDBD541, (q31_t)0x7FF62182,
	(q31_t)0xFCA99569, (q31_t)0x7FF4DBD8, (q31_t)0xFC775616,
	(q31_t)0x7FF38273, (q31_t)0xFC45174E, (q31_t)0x7FF21553,
	(q31_t)0xFC12D919, (q31_t)0x7FF09477, (q31_t)0xFBE09B80,
	(q31_t)0x7FEEFFE1, (q31_t)0xFBAE5E89, (q31_t)0x7FED5790,
	(q31_t)0xFB7C223C, (q31_t)0x7FEB9B85, (q31_t)0xFB49E6A2,
	(q31_t)0x7FE9CBC0, (q31_t)0xFB17ABC2, (q31_t)0x7FE7E840,
	(q31_t)0xFAE571A4, (q31_t)0x7FE5F108, (q31_t)0xFAB3384F,
	(q31_t)0x7FE3E616, (q31_t)0xFA80FFCB, (q31_t)0x7FE1C76B,
	(q31_t)0xFA4EC820, (q31_t)0x7FDF9508, (q31_t)0xFA1C9156,
	(q31_t)0x7FDD4EEC, (q31_t)0xF9EA5B75, (q31_t)0x7FDAF518,
	(q31_t)0xF9B82683, (q31_t)0x7FD8878D, (q31_t)0xF985F28A,
	(q31_t)0x7FD6064B, (q31_t)0xF953BF90, (q31_t)0x7FD37152,
	(q31_t)0xF9218D9E, (q31_t)0x7FD0C8A3, (q31_t)0xF8EF5CBB,
	(q31_t)0x7FCE0C3E, (q31_t)0xF8BD2CEF, (q31_t)0x7FCB3C23,
	(q31_t)0xF88AFE41, (q31_t)0x7FC85853, (q31_t)0xF858D0BA,
	(q31_t)0x7FC560CF, (q31_t)0xF826A461, (q31_t)0x7FC25596,
	(q31_t)0xF7F4793E, (q31_t)0x7FBF36A9, (q31_t)0xF7C24F58,
	(q31_t)0x7FBC040A, (q31_t)0xF79026B8, (q31_t)0x7FB8BDB7,
	(q31_t)0xF75DFF65, (q31_t)0x7FB563B2, (q31_t)0xF72BD967,
	(q31_t)0x7FB1F5FC, (q31_t)0xF6F9B4C5, (q31_t)0x7FAE7494,
	(q31_t)0xF6C79188, (q31_t)0x7FAADF7C, (q31_t)0xF6956FB6,
	(q31_t)0x7FA736B4, (q31_t)0xF6634F58, (q31_t)0x7FA37A3C,
	(q31_t)0xF6313076, (q31_t)0x7F9FAA15, (q31_t)0xF5FF1317,
	(q31_t)0x7F9BC63F, (q31_t)0xF5CCF743, (q31_t)0x7F97CEBC,
	(q31_t)0xF59ADD01, (q31_t)0x7F93C38C, (q31_t)0xF568C45A,
	(q31_t)0x7F8FA4AF, (q31_t)0xF536AD55, (q31_t)0x7F8B7226,
	(q31_t)0xF50497FA, (q31_t)0x7F872BF3, (q31_t)0xF4D28451,
	(q31_t)0x7F82D214, (q31_t)0xF4A07260, (q31_t)0x7F7E648B,
	(q31_t)0xF46E6231, (q31_t)0x7F79E35A, (q31_t)0xF43C53CA,
	(q31_t)0x7F754E7F, (q31_t)0xF40A4734, (q31_t)0x7F70A5FD,
	(q31_t)0xF3D83C76, (q31_t)0x7F6BE9D4, (q31_t)0xF3A63398,
	(q31_t)0x7F671A04, (q31_t)0xF3742CA1, (q31_t)0x7F62368F,
	(q31_t)0xF342279A, (q31_t)0x7F5D3F75, (q31_t)0xF310248A,
	(q31_t)0x7F5834B6, (q31_t)0xF2DE2378, (q31_t)0x7F531654,
	(q31_t)0xF2AC246D, (q31_t)0x7F4DE450, (q31_t)0xF27A2770,
	(q31_t)0x7F489EAA, (q31_t)0xF2482C89, (q31_t)0x7F434563,
	(q31_t)0xF21633C0, (q31_t)0x7F3DD87C, (q31_t)0xF1E43D1C,
	(q31_t)0x7F3857F5, (q31_t)0xF1B248A5, (q31_t)0x7F32C3D0,
	(q31_t)0xF1805662, (q31_t)0x7F2D1C0E, (q31_t)0xF14E665C,
	(q31_t)0x7F2760AF, (q31_t)0xF11C789A, (q31_t)0x7F2191B4,
	(q31_t)0xF0EA8D23, (q31_t)0x7F1BAF1E, (q31_t)0xF0B8A401,
	(q31_t)0x7F15B8EE, (q31_t)0xF086BD39, (q31_t)0x7F0FAF24,
	(q31_t)0xF054D8D4, (q31_t)0x7F0991C3, (q31_t)0xF022F6DA,
	(q31_t)0x7F0360CB, (q31_t)0xEFF11752, (q31_t)0x7EFD1C3C,
	(q31_t)0xEFBF3A44, (q31_t)0x7EF6C418, (q31_t)0xEF8D5FB8,
	(q31_t)0x7EF0585F, (q31_t)0xEF5B87B5, (q31_t)0x7EE9D913,
	(q31_t)0xEF29B243, (q31_t)0x7EE34635, (q31_t)0xEEF7DF6A,
	(q31_t)0x7EDC9FC6, (q31_t)0xEEC60F31, (q31_t)0x7ED5E5C6,
	(q31_t)0xEE9441A0, (q31_t)0x7ECF1837, (q31_t)0xEE6276BF,
	(q31_t)0x7EC8371A, (q31_t)0xEE30AE95, (q31_t)0x7EC1426F,
	(q31_t)0xEDFEE92B, (q31_t)0x7EBA3A39, (q31_t)0xEDCD2687,
	(q31_t)0x7EB31E77, (q31_t)0xED9B66B2, (q31_t)0x7EABEF2C,
	(q31_t)0xED69A9B2, (q31_t)0x7EA4AC58, (q31_t)0xED37EF91,
	(q31_t)0x7E9D55FC, (q31_t)0xED063855, (q31_t)0x7E95EC19,
	(q31_t)0xECD48406, (q31_t)0x7E8E6EB1, (q31_t)0xECA2D2AC,
	(q31_t)0x7E86DDC5, (q31_t)0xEC71244F, (q31_t)0x7E7F3956,
	(q31_t)0xEC3F78F5, (q31_t)0x7E778165, (q31_t)0xEC0DD0A8,
	(q31_t)0x7E6FB5F3, (q31_t)0xEBDC2B6D, (q31_t)0x7E67D702,
	(q31_t)0xEBAA894E, (q31_t)0x7E5FE493, (q31_t)0xEB78EA52,
	(q31_t)0x7E57DEA6, (q31_t)0xEB474E80, (q31_t)0x7E4FC53E,
	(q31_t)0xEB15B5E0, (q31_t)0x7E47985B, (q31_t)0xEAE4207A,
	(q31_t)0x7E3F57FE, (q31_t)0xEAB28E55, (q31_t)0x7E37042A,
	(q31_t)0xEA80FF79, (q31_t)0x7E2E9CDF, (q31_t)0xEA4F73EE,
	(q31_t)0x7E26221E, (q31_t)0xEA1DEBBB, (q31_t)0x7E1D93E9,
	(q31_t)0xE9EC66E8, (q31_t)0x7E14F242, (q31_t)0xE9BAE57C,
	(q31_t)0x7E0C3D29, (q31_t)0xE9896780, (q31_t)0x7E03749F,
	(q31_t)0xE957ECFB, (q31_t)0x7DFA98A7, (q31_t)0xE92675F4,
	(q31_t)0x7DF1A942, (q31_t)0xE8F50273, (q31_t)0x7DE8A670,
	(q31_t)0xE8C3927F, (q31_t)0x7DDF9034, (q31_t)0xE8922621,
	(q31_t)0x7DD6668E, (q31_t)0xE860BD60, (q31_t)0x7DCD2981,
	(q31_t)0xE82F5844, (q31_t)0x7DC3D90D, (q31_t)0xE7FDF6D3,
	(q31_t)0x7DBA7534, (q31_t)0xE7CC9917, (q31_t)0x7DB0FDF7,
	(q31_t)0xE79B3F16, (q31_t)0x7DA77359, (q31_t)0xE769E8D8,
	(q31_t)0x7D9DD55A, (q31_t)0xE7389664, (q31_t)0x7D9423FB,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xE6D5FCFC,
	(q31_t)0x7D808727, (q31_t)0xE6A4B616, (q31_t)0x7D769BB5,
	(q31_t)0xE6737319, (q31_t)0x7D6C9CE9, (q31_t)0xE642340D,
	(q31_t)0x7D628AC5, (q31_t)0xE610F8F9, (q31_t)0x7D58654C,
	(q31_t)0xE5DFC1E4, (q31_t)0x7D4E2C7E, (q31_t)0xE5AE8ED8,
	(q31_t)0x7D43E05E, (q31_t)0xE57D5FDA, (q31_t)0x7D3980EC,
	(q31_t)0xE54C34F3, (q31_t)0x7D2F0E2A, (q31_t)0xE51B0E2A,
	(q31_t)0x7D24881A, (q31_t)0xE4E9EB86, (q31_t)0x7D19EEBE,
	(q31_t)0xE4B8CD10, (q31_t)0x7D0F4218, (q31_t)0xE487B2CF,
	(q31_t)0x7D048228, (q31_t)0xE4569CCB, (q31_t)0x7CF9AEF0,
	(q31_t)0xE4258B0A, (q31_t)0x7CEEC873, (q31_t)0xE3F47D95,
	(q31_t)0x7CE3CEB1, (q31_t)0xE3C37473, (q31_t)0x7CD8C1AD,
	(q31_t)0xE3926FAC, (q31_t)0x7CCDA168, (q31_t)0xE3616F47,
	(q31_t)0x7CC26DE5, (q31_t)0xE330734C, (q31_t)0x7CB72724,
	(q31_t)0xE2FF7BC3, (q31_t)0x7CABCD27, (q31_t)0xE2CE88B2,
	(q31_t)0x7CA05FF1, (q31_t)0xE29D9A22, (q31_t)0x7C94DF82,
	(q31_t)0xE26CB01A, (q31_t)0x7C894BDD, (q31_t)0xE23BCAA2,
	(q31_t)0x7C7DA504, (q31_t)0xE20AE9C1, (q31_t)0x7C71EAF8,
	(q31_t)0xE1DA0D7E, (q31_t)0x7C661DBB, (q31_t)0xE1A935E1,
	(q31_t)0x7C5A3D4F, (q31_t)0xE17862F3, (q31_t)0x7C4E49B6,
	(q31_t)0xE14794B9, (q31_t)0x7C4242F2, (q31_t)0xE116CB3D,
	(q31_t)0x7C362904, (q31_t)0xE0E60684, (q31_t)0x7C29FBEE,
	(q31_t)0xE0B54698, (q31_t)0x7C1DBBB2, (q31_t)0xE0848B7F,
	(q31_t)0x7C116853, (q31_t)0xE053D541, (q31_t)0x7C0501D1,
	(q31_t)0xE02323E5, (q31_t)0x7BF88830, (q31_t)0xDFF27773,
	(q31_t)0x7BEBFB70, (q31_t)0xDFC1CFF2, (q31_t)0x7BDF5B94,
	(q31_t)0xDF912D6A, (q31_t)0x7BD2A89E, (q31_t)0xDF608FE3,
	(q31_t)0x7BC5E28F, (q31_t)0xDF2FF764, (q31_t)0x7BB9096A,
	(q31_t)0xDEFF63F4, (q31_t)0x7BAC1D31, (q31_t)0xDECED59B,
	(q31_t)0x7B9F1DE5, (q31_t)0xDE9E4C60, (q31_t)0x7B920B89,
	(q31_t)0xDE6DC84B, (q31_t)0x7B84E61E, (q31_t)0xDE3D4963,
	(q31_t)0x7B77ADA8, (q31_t)0xDE0CCFB1, (q31_t)0x7B6A6227,
	(q31_t)0xDDDC5B3A, (q31_t)0x7B5D039D, (q31_t)0xDDABEC07,
	(q31_t)0x7B4F920E, (q31_t)0xDD7B8220, (q31_t)0x7B420D7A,
	(q31_t)0xDD4B1D8B, (q31_t)0x7B3475E4, (q31_t)0xDD1ABE51,
	(q31_t)0x7B26CB4F, (q31_t)0xDCEA6478, (q31_t)0x7B190DBB,
	(q31_t)0xDCBA1008, (q31_t)0x7B0B3D2C, (q31_t)0xDC89C108,
	(q31_t)0x7AFD59A3, (q31_t)0xDC597781, (q31_t)0x7AEF6323,
	(q31_t)0xDC293379, (q31_t)0x7AE159AE, (q31_t)0xDBF8F4F8,
	(q31_t)0x7AD33D45, (q31_t)0xDBC8BC05, (q31_t)0x7AC50DEB,
	(q31_t)0xDB9888A8, (q31_t)0x7AB6CBA3, (q31_t)0xDB685AE8,
	(q31_t)0x7AA8766E, (q31_t)0xDB3832CD, (q31_t)0x7A9A0E4F,
	(q31_t)0xDB08105E, (q31_t)0x7A8B9348, (q31_t)0xDAD7F3A2,
	(q31_t)0x7A7D055B, (q31_t)0xDAA7DCA1, (q31_t)0x7A6E648A,
	(q31_t)0xDA77CB62, (q31_t)0x7A5FB0D8, (q31_t)0xDA47BFED,
	(q31_t)0x7A50EA46, (q31_t)0xDA17BA4A, (q31_t)0x7A4210D8,
	(q31_t)0xD9E7BA7E, (q31_t)0x7A33248F, (q31_t)0xD9B7C093,
	(q31_t)0x7A24256E, (q31_t)0xD987CC8F, (q31_t)0x7A151377,
	(q31_t)0xD957DE7A, (q31_t)0x7A05EEAD, (q31_t)0xD927F65B,
	(q31_t)0x79F6B711, (q31_t)0xD8F81439, (q31_t)0x79E76CA6,
	(q31_t)0xD8C8381C, (q31_t)0x79D80F6F, (q31_t)0xD898620C,
	(q31_t)0x79C89F6D, (q31_t)0xD868920F, (q31_t)0x79B91CA4,
	(q31_t)0xD838C82D, (q31_t)0x79A98715, (q31_t)0xD809046D,
	(q31_t)0x7999DEC3, (q31_t)0xD7D946D7, (q31_t)0x798A23B1,
	(q31_t)0xD7A98F73, (q31_t)0x797A55E0, (q31_t)0xD779DE46,
	(q31_t)0x796A7554, (q31_t)0xD74A335A, (q31_t)0x795A820E,
	(q31_t)0xD71A8EB5, (q31_t)0x794A7C11, (q31_t)0xD6EAF05E,
	(q31_t)0x793A6360, (q31_t)0xD6BB585D, (q31_t)0x792A37FE,
	(q31_t)0xD68BC6BA, (q31_t)0x7919F9EB, (q31_t)0xD65C3B7B,
	(q31_t)0x7909A92C, (q31_t)0xD62CB6A7, (q31_t)0x78F945C3,
	(q31_t)0xD5FD3847, (q31_t)0x78E8CFB1, (q31_t)0xD5CDC062,
	(q31_t)0x78D846FB, (q31_t)0xD59E4EFE, (q31_t)0x78C7ABA1,
	(q31_t)0xD56EE424, (q31_t)0x78B6FDA8, (q31_t)0xD53F7FDA,
	(q31_t)0x78A63D10, (q31_t)0xD5102227, (q31_t)0x789569DE,
	(q31_t)0xD4E0CB14, (q31_t)0x78848413, (q31_t)0xD4B17AA7,
	(q31_t)0x78738BB3, (q31_t)0xD48230E8, (q31_t)0x786280BF,
	(q31_t)0xD452EDDE, (q31_t)0x7851633B, (q31_t)0xD423B190,
	(q31_t)0x78403328, (q31_t)0xD3F47C06, (q31_t)0x782EF08B,
	(q31_t)0xD3C54D46, (q31_t)0x781D9B64, (q31_t)0xD3962559,
	(q31_t)0x780C33B8, (q31_t)0xD3670445, (q31_t)0x77FAB988,
	(q31_t)0xD337EA12, (q31_t)0x77E92CD8, (q31_t)0xD308D6C6,
	(q31_t)0x77D78DAA, (q31_t)0xD2D9CA6A, (q31_t)0x77C5DC01,
	(q31_t)0xD2AAC504, (q31_t)0x77B417DF, (q31_t)0xD27BC69C,
	(q31_t)0x77A24148, (q31_t)0xD24CCF38, (q31_t)0x7790583D,
	(q31_t)0xD21DDEE1, (q31_t)0x777E5CC3, (q31_t)0xD1EEF59E,
	(q31_t)0x776C4EDB, (q31_t)0xD1C01374, (q31_t)0x775A2E88,
	(q31_t)0xD191386D, (q31_t)0x7747FBCE, (q31_t)0xD162648F,
	(q31_t)0x7735B6AE, (q31_t)0xD13397E1, (q31_t)0x77235F2D,
	(q31_t)0xD104D26B, (q31_t)0x7710F54B, (q31_t)0xD0D61433,
	(q31_t)0x76FE790E, (q31_t)0xD0A75D42, (q31_t)0x76EBEA77,
	(q31_t)0xD078AD9D, (q31_t)0x76D94988, (q31_t)0xD04A054D,
	(q31_t)0x76C69646, (q31_t)0xD01B6459, (q31_t)0x76B3D0B3,
	(q31_t)0xCFECCAC7, (q31_t)0x76A0F8D2, (q31_t)0xCFBE389F,
	(q31_t)0x768E0EA5, (q31_t)0xCF8FADE8, (q31_t)0x767B1230,
	(q31_t)0xCF612AAA, (q31_t)0x76680376, (q31_t)0xCF32AEEB,
	(q31_t)0x7654E279, (q31_t)0xCF043AB2, (q31_t)0x7641AF3C,
	(q31_t)0xCED5CE08, (q31_t)0x762E69C3, (q31_t)0xCEA768F2,
	(q31_t)0x761B1211, (q31_t)0xCE790B78, (q31_t)0x7607A827,
	(q31_t)0xCE4AB5A2, (q31_t)0x75F42C0A, (q31_t)0xCE1C6776,
	(q31_t)0x75E09DBD, (q31_t)0xCDEE20FC, (q31_t)0x75CCFD42,
	(q31_t)0xCDBFE23A, (q31_t)0x75B94A9C, (q31_t)0xCD91AB38,
	(q31_t)0x75A585CF, (q31_t)0xCD637BFD, (q31_t)0x7591AEDD,
	(q31_t)0xCD355490, (q31_t)0x757DC5CA, (q31_t)0xCD0734F8,
	(q31_t)0x7569CA98, (q31_t)0xCCD91D3D, (q31_t)0x7555BD4B,
	(q31_t)0xCCAB0D65, (q31_t)0x75419DE6, (q31_t)0xCC7D0577,
	(q31_t)0x752D6C6C, (q31_t)0xCC4F057B, (q31_t)0x751928E0,
	(q31_t)0xCC210D78, (q31_t)0x7504D345, (q31_t)0xCBF31D75,
	(q31_t)0x74F06B9E, (q31_t)0xCBC53578, (q31_t)0x74DBF1EF,
	(q31_t)0xCB975589, (q31_t)0x74C7663A, (q31_t)0xCB697DB0,
	(q31_t)0x74B2C883, (q31_t)0xCB3BADF2, (q31_t)0x749E18CD,
	(q31_t)0xCB0DE658, (q31_t)0x7489571B, (q31_t)0xCAE026E8,
	(q31_t)0x74748371, (q31_t)0xCAB26FA9, (q31_t)0x745F9DD1,
	(q31_t)0xCA84C0A2, (q31_t)0x744AA63E, (q31_t)0xCA5719DB,
	(q31_t)0x74359CBD, (q31_t)0xCA297B5A, (q31_t)0x74208150,
	(q31_t)0xC9FBE527, (q31_t)0x740B53FA, (q31_t)0xC9CE5748,
	(q31_t)0x73F614C0, (q31_t)0xC9A0D1C4, (q31_t)0x73E0C3A3,
	(q31_t)0xC97354A3, (q31_t)0x73CB60A7, (q31_t)0xC945DFEC,
	(q31_t)0x73B5EBD0, (q31_t)0xC91873A5, (q31_t)0x73A06522,
	(q31_t)0xC8EB0FD6, (q31_t)0x738ACC9E, (q31_t)0xC8BDB485,
	(q31_t)0x73752249, (q31_t)0xC89061BA, (q31_t)0x735F6626,
	(q31_t)0xC863177B, (q31_t)0x73499838, (q31_t)0xC835D5D0,
	(q31_t)0x7333B883, (q31_t)0xC8089CBF, (q31_t)0x731DC709,
	(q31_t)0xC7DB6C50, (q31_t)0x7307C3D0, (q31_t)0xC7AE4489,
	(q31_t)0x72F1AED8, (q31_t)0xC7812571, (q31_t)0x72DB8828,
	(q31_t)0xC7540F10, (q31_t)0x72C54FC0, (q31_t)0xC727016C,
	(q31_t)0x72AF05A6, (q31_t)0xC6F9FC8D, (q31_t)0x7298A9DC,
	(q31_t)0xC6CD0079, (q31_t)0x72823C66, (q31_t)0xC6A00D36,
	(q31_t)0x726BBD48, (q31_t)0xC67322CD, (q31_t)0x72552C84,
	(q31_t)0xC6464144, (q31_t)0x723E8A1F, (q31_t)0xC61968A2,
	(q31_t)0x7227D61C, (q31_t)0xC5EC98ED, (q31_t)0x7211107D,
	(q31_t)0xC5BFD22E, (q31_t)0x71FA3948, (q31_t)0xC593146A,
	(q31_t)0x71E3507F, (q31_t)0xC5665FA8, (q31_t)0x71CC5626,
	(q31_t)0xC539B3F0, (q31_t)0x71B54A40, (q31_t)0xC50D1148,
	(q31_t)0x719E2CD2, (q31_t)0xC4E077B8, (q31_t)0x7186FDDE,
	(q31_t)0xC4B3E746, (q31_t)0x716FBD68, (q31_t)0xC4875FF8,
	(q31_t)0x71586B73, (q31_t)0xC45AE1D7, (q31_t)0x71410804,
	(q31_t)0xC42E6CE8, (q31_t)0x7129931E, (q31_t)0xC4020132,
	(q31_t)0x71120CC5, (q31_t)0xC3D59EBD, (q31_t)0x70FA74FB,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xC37CF5B0,
	(q31_t)0x70CB1127, (q31_t)0xC350AF25, (q31_t)0x70B34524,
	(q31_t)0xC32471F6, (q31_t)0x709B67C0, (q31_t)0xC2F83E2A,
	(q31_t)0x708378FE, (q31_t)0xC2CC13C7, (q31_t)0x706B78E3,
	(q31_t)0xC29FF2D4, (q31_t)0x70536771, (q31_t)0xC273DB58,
	(q31_t)0x703B44AC, (q31_t)0xC247CD5A, (q31_t)0x70231099,
	(q31_t)0xC21BC8E0, (q31_t)0x700ACB3B, (q31_t)0xC1EFCDF2,
	(q31_t)0x6FF27496, (q31_t)0xC1C3DC96, (q31_t)0x6FDA0CAD,
	(q31_t)0xC197F4D3, (q31_t)0x6FC19385, (q31_t)0xC16C16B0,
	(q31_t)0x6FA90920, (q31_t)0xC1404233, (q31_t)0x6F906D84,
	(q31_t)0xC1147763, (q31_t)0x6F77C0B3, (q31_t)0xC0E8B648,
	(q31_t)0x6F5F02B1, (q31_t)0xC0BCFEE7, (q31_t)0x6F463383,
	(q31_t)0xC0915147, (q31_t)0x6F2D532C, (q31_t)0xC065AD70,
	(q31_t)0x6F1461AF, (q31_t)0xC03A1368, (q31_t)0x6EFB5F12,
	(q31_t)0xC00E8335, (q31_t)0x6EE24B57, (q31_t)0xBFE2FCDF,
	(q31_t)0x6EC92682, (q31_t)0xBFB7806C, (q31_t)0x6EAFF098,
	(q31_t)0xBF8C0DE2, (q31_t)0x6E96A99C, (q31_t)0xBF60A54A,
	(q31_t)0x6E7D5193, (q31_t)0xBF3546A8, (q31_t)0x6E63E87F,
	(q31_t)0xBF09F204, (q31_t)0x6E4A6E65, (q31_t)0xBEDEA765,
	(q31_t)0x6E30E349, (q31_t)0xBEB366D1, (q31_t)0x6E17472F,
	(q31_t)0xBE88304F, (q31_t)0x6DFD9A1B, (q31_t)0xBE5D03E5,
	(q31_t)0x6DE3DC11, (q31_t)0xBE31E19B, (q31_t)0x6DCA0D14,
	(q31_t)0xBE06C977, (q31_t)0x6DB02D29, (q31_t)0xBDDBBB7F,
	(q31_t)0x6D963C54, (q31_t)0xBDB0B7BA, (q31_t)0x6D7C3A98,
	(q31_t)0xBD85BE2F, (q31_t)0x6D6227FA, (q31_t)0xBD5ACEE5,
	(q31_t)0x6D48047E, (q31_t)0xBD2FE9E1, (q31_t)0x6D2DD027,
	(q31_t)0xBD050F2C, (q31_t)0x6D138AFA, (q31_t)0xBCDA3ECA,
	(q31_t)0x6CF934FB, (q31_t)0xBCAF78C3, (q31_t)0x6CDECE2E,
	(q31_t)0xBC84BD1E, (q31_t)0x6CC45697, (q31_t)0xBC5A0BE1,
	(q31_t)0x6CA9CE3A, (q31_t)0xBC2F6513, (q31_t)0x6C8F351C,
	(q31_t)0xBC04C8BA, (q31_t)0x6C748B3F, (q31_t)0xBBDA36DC,
	(q31_t)0x6C59D0A9, (q31_t)0xBBAFAF81, (q31_t)0x6C3F055D,
	(q31_t)0xBB8532AF, (q31_t)0x6C242960, (q31_t)0xBB5AC06C,
	(q31_t)0x6C093CB6, (q31_t)0xBB3058C0, (q31_t)0x6BEE3F62,
	(q31_t)0xBB05FBB0, (q31_t)0x6BD3316A, (q31_t)0xBADBA943,
	(q31_t)0x6BB812D0, (q31_t)0xBAB1617F, (q31_t)0x6B9CE39B,
	(q31_t)0xBA87246C, (q31_t)0x6B81A3CD, (q31_t)0xBA5CF210,
	(q31_t)0x6B66536A, (q31_t)0xBA32CA70, (q31_t)0x6B4AF278,
	(q31_t)0xBA08AD94, (q31_t)0x6B2F80FA, (q31_t)0xB9DE9B83,
	(q31_t)0x6B13FEF5, (q31_t)0xB9B49442, (q31_t)0x6AF86C6C,
	(q31_t)0xB98A97D8, (q31_t)0x6ADCC964, (q31_t)0xB960A64B,
	(q31_t)0x6AC115E1, (q31_t)0xB936BFA3, (q31_t)0x6AA551E8,
	(q31_t)0xB90CE3E6, (q31_t)0x6A897D7D, (q31_t)0xB8E31319,
	(q31_t)0x6A6D98A4, (q31_t)0xB8B94D44, (q31_t)0x6A51A361,
	(q31_t)0xB88F926C, (q31_t)0x6A359DB9, (q31_t)0xB865E299,
	(q31_t)0x6A1987B0, (q31_t)0xB83C3DD1, (q31_t)0x69FD614A,
	(q31_t)0xB812A419, (q31_t)0x69E12A8C, (q31_t)0xB7E9157A,
	(q31_t)0x69C4E37A, (q31_t)0xB7BF91F8, (q31_t)0x69A88C18,
	(q31_t)0xB796199B, (q31_t)0x698C246C, (q31_t)0xB76CAC68,
	(q31_t)0x696FAC78, (q31_t)0xB7434A67, (q31_t)0x69532442,
	(q31_t)0xB719F39D, (q31_t)0x69368BCE, (q31_t)0xB6F0A811,
	(q31_t)0x6919E320, (q31_t)0xB6C767CA, (q31_t)0x68FD2A3D,
	(q31_t)0xB69E32CD, (q31_t)0x68E06129, (q31_t)0xB6750921,
	(q31_t)0x68C387E9, (q31_t)0xB64BEACC, (q31_t)0x68A69E81,
	(q31_t)0xB622D7D5, (q31_t)0x6889A4F5, (q31_t)0xB5F9D042,
	(q31_t)0x686C9B4B, (q31_t)0xB5D0D41A, (q31_t)0x684F8186,
	(q31_t)0xB5A7E362, (q31_t)0x683257AA, (q31_t)0xB57EFE21,
	(q31_t)0x68151DBE, (q31_t)0xB556245E, (q31_t)0x67F7D3C4,
	(q31_t)0xB52D561E, (q31_t)0x67DA79C2, (q31_t)0xB5049368,
	(q31_t)0x67BD0FBC, (q31_t)0xB4DBDC42, (q31_t)0x679F95B7,
	(q31_t)0xB4B330B2, (q31_t)0x67820BB6, (q31_t)0xB48A90C0,
	(q31_t)0x676471C0, (q31_t)0xB461FC70, (q31_t)0x6746C7D7,
	(q31_t)0xB43973C9, (q31_t)0x67290E02, (q31_t)0xB410F6D2,
	(q31_t)0x670B4443, (q31_t)0xB3E88591, (q31_t)0x66ED6AA1,
	(q31_t)0xB3C0200C, (q31_t)0x66CF811F, (q31_t)0xB397C649,
	(q31_t)0x66B187C3, (q31_t)0xB36F784E, (q31_t)0x66937E90,
	(q31_t)0xB3473622, (q31_t)0x6675658C, (q31_t)0xB31EFFCB,
	(q31_t)0x66573CBB, (q31_t)0xB2F6D54F, (q31_t)0x66390422,
	(q31_t)0xB2CEB6B5, (q31_t)0x661ABBC5, (q31_t)0xB2A6A401,
	(q31_t)0x65FC63A9, (q31_t)0xB27E9D3B, (q31_t)0x65DDFBD3,
	(q31_t)0xB256A26A, (q31_t)0x65BF8447, (q31_t)0xB22EB392,
	(q31_t)0x65A0FD0B, (q31_t)0xB206D0BA, (q31_t)0x65826622,
	(q31_t)0xB1DEF9E8, (q31_t)0x6563BF92, (q31_t)0xB1B72F23,
	(q31_t)0x6545095F, (q31_t)0xB18F7070, (q31_t)0x6526438E,
	(q31_t)0xB167BDD6, (q31_t)0x65076E24, (q31_t)0xB140175B,
	(q31_t)0x64E88926, (q31_t)0xB1187D05, (q31_t)0x64C99498,
	(q31_t)0xB0F0EEDA, (q31_t)0x64AA907F, (q31_t)0xB0C96CDF,
	(q31_t)0x648B7CDF, (q31_t)0xB0A1F71C, (q31_t)0x646C59BF,
	(q31_t)0xB07A8D97, (q31_t)0x644D2722, (q31_t)0xB0533055,
	(q31_t)0x642DE50D, (q31_t)0xB02BDF5C, (q31_t)0x640E9385,
	(q31_t)0xB0049AB2, (q31_t)0x63EF328F, (q31_t)0xAFDD625F,
	(q31_t)0x63CFC230, (q31_t)0xAFB63667, (q31_t)0x63B0426D,
	(q31_t)0xAF8F16D0, (q31_t)0x6390B34A, (q31_t)0xAF6803A1,
	(q31_t)0x637114CC, (q31_t)0xAF40FCE0, (q31_t)0x635166F8,
	(q31_t)0xAF1A0293, (q31_t)0x6331A9D4, (q31_t)0xAEF314BF,
	(q31_t)0x6311DD63, (q31_t)0xAECC336B, (q31_t)0x62F201AC,
	(q31_t)0xAEA55E9D, (q31_t)0x62D216B2, (q31_t)0xAE7E965B,
	(q31_t)0x62B21C7B, (q31_t)0xAE57DAAA, (q31_t)0x6292130C,
	(q31_t)0xAE312B91, (q31_t)0x6271FA69, (q31_t)0xAE0A8916,
	(q31_t)0x6251D297, (q31_t)0xADE3F33E, (q31_t)0x62319B9D,
	(q31_t)0xADBD6A10, (q31_t)0x6211557D, (q31_t)0xAD96ED91,
	(q31_t)0x61F1003E, (q31_t)0xAD707DC8, (q31_t)0x61D09BE5,
	(q31_t)0xAD4A1ABA, (q31_t)0x61B02876, (q31_t)0xAD23C46D,
	(q31_t)0x618FA5F6, (q31_t)0xACFD7AE8, (q31_t)0x616F146B,
	(q31_t)0xACD73E30, (q31_t)0x614E73D9, (q31_t)0xACB10E4A,
	(q31_t)0x612DC446, (q31_t)0xAC8AEB3E, (q31_t)0x610D05B7,
	(q31_t)0xAC64D510, (q31_t)0x60EC3830, (q31_t)0xAC3ECBC7,
	(q31_t)0x60CB5BB6, (q31_t)0xAC18CF68, (q31_t)0x60AA704F,
	(q31_t)0xABF2DFFA, (q31_t)0x60897600, (q31_t)0xABCCFD82,
	(q31_t)0x60686CCE, (q31_t)0xABA72806, (q31_t)0x604754BE,
	(q31_t)0xAB815F8C, (q31_t)0x60262DD5, (q31_t)0xAB5BA41A,
	(q31_t)0x6004F818, (q31_t)0xAB35F5B5, (q31_t)0x5FE3B38D,
	(q31_t)0xAB105464, (q31_t)0x5FC26038, (q31_t)0xAAEAC02B,
	(q31_t)0x5FA0FE1E, (q31_t)0xAAC53912, (q31_t)0x5F7F8D46,
	(q31_t)0xAA9FBF1D, (q31_t)0x5F5E0DB3, (q31_t)0xAA7A5253,
	(q31_t)0x5F3C7F6B, (q31_t)0xAA54F2B9, (q31_t)0x5F1AE273,
	(q31_t)0xAA2FA055, (q31_t)0x5EF936D1, (q31_t)0xAA0A5B2D,
	(q31_t)0x5ED77C89, (q31_t)0xA9E52347, (q31_t)0x5EB5B3A1,
	(q31_t)0xA9BFF8A8, (q31_t)0x5E93DC1F, (q31_t)0xA99ADB56,
	(q31_t)0x5E71F606, (q31_t)0xA975CB56, (q31_t)0x5E50015D,
	(q31_t)0xA950C8AF, (q31_t)0x5E2DFE28, (q31_t)0xA92BD366,
	(q31_t)0x5E0BEC6E, (q31_t)0xA906EB81, (q31_t)0x5DE9CC32,
	(q31_t)0xA8E21106, (q31_t)0x5DC79D7C, (q31_t)0xA8BD43FA,
	(q31_t)0x5DA5604E, (q31_t)0xA8988463, (q31_t)0x5D8314B0,
	(q31_t)0xA873D246, (q31_t)0x5D60BAA6, (q31_t)0xA84F2DA9,
	(q31_t)0x5D3E5236, (q31_t)0xA82A9693, (q31_t)0x5D1BDB65,
	(q31_t)0xA8060D08, (q31_t)0x5CF95638, (q31_t)0xA7E1910E,
	(q31_t)0x5CD6C2B4, (q31_t)0xA7BD22AB, (q31_t)0x5CB420DF,
	(q31_t)0xA798C1E4, (q31_t)0x5C9170BF, (q31_t)0xA7746EC0,
	(q31_t)0x5C6EB258, (q31_t)0xA7502943, (q31_t)0x5C4BE5B0,
	(q31_t)0xA72BF173, (q31_t)0x5C290ACC, (q31_t)0xA707C756,
	(q31_t)0x5C0621B2, (q31_t)0xA6E3AAF2, (q31_t)0x5BE32A67,
	(q31_t)0xA6BF9C4B, (q31_t)0x5BC024F0, (q31_t)0xA69B9B68,
	(q31_t)0x5B9D1153, (q31_t)0xA677A84E, (q31_t)0x5B79EF96,
	(q31_t)0xA653C302, (q31_t)0x5B56BFBD, (q31_t)0xA62FEB8B,
	(q31_t)0x5B3381CE, (q31_t)0xA60C21ED, (q31_t)0x5B1035CF,
	(q31_t)0xA5E8662F, (q31_t)0x5AECDBC4, (q31_t)0xA5C4B855,
	(q31_t)0x5AC973B4, (q31_t)0xA5A11865, (q31_t)0x5AA5FDA4,
	(q31_t)0xA57D8666, (q31_t)0x5A82799A, (q31_t)0xA55A025B,
	(q31_t)0x5A5EE79A, (q31_t)0xA5368C4B, (q31_t)0x5A3B47AA,
	(q31_t)0xA513243B, (q31_t)0x5A1799D0, (q31_t)0xA4EFCA31,
	(q31_t)0x59F3DE12, (q31_t)0xA4CC7E31, (q31_t)0x59D01474,
	(q31_t)0xA4A94042, (q31_t)0x59AC3CFD, (q31_t)0xA4861069,
	(q31_t)0x598857B1, (q31_t)0xA462EEAC, (q31_t)0x59646497,
	(q31_t)0xA43FDB0F, (q31_t)0x594063B4, (q31_t)0xA41CD598,
	(q31_t)0x591C550E, (q31_t)0xA3F9DE4D, (q31_t)0x58F838A9,
	(q31_t)0xA3D6F533, (q31_t)0x58D40E8C, (q31_t)0xA3B41A4F,
	(q31_t)0x58AFD6BC, (q31_t)0xA3914DA7, (q31_t)0x588B913F,
	(q31_t)0xA36E8F40, (q31_t)0x58673E1B, (q31_t)0xA34BDF20,
	(q31_t)0x5842DD54, (q31_t)0xA3293D4B, (q31_t)0x581E6EF1,
	(q31_t)0xA306A9C7, (q31_t)0x57F9F2F7, (q31_t)0xA2E4249A,
	(q31_t)0x57D5696C, (q31_t)0xA2C1ADC9, (q31_t)0x57B0D256,
	(q31_t)0xA29F4559, (q31_t)0x578C2DB9, (q31_t)0xA27CEB4F,
	(q31_t)0x57677B9D, (q31_t)0xA25A9FB1, (q31_t)0x5742BC05,
	(q31_t)0xA2386283, (q31_t)0x571DEEF9, (q31_t)0xA21633CD,
	(q31_t)0x56F9147E, (q31_t)0xA1F41391, (q31_t)0x56D42C99,
	(q31_t)0xA1D201D7, (q31_t)0x56AF3750, (q31_t)0xA1AFFEA2,
	(q31_t)0x568A34A9, (q31_t)0xA18E09F9, (q31_t)0x566524AA,
	(q31_t)0xA16C23E1, (q31_t)0x56400757, (q31_t)0xA14A4C5E,
	(q31_t)0x561ADCB8, (q31_t)0xA1288376, (q31_t)0x55F5A4D2,
	(q31_t)0xA106C92E, (q31_t)0x55D05FAA, (q31_t)0xA0E51D8C,
	(q31_t)0x55AB0D46, (q31_t)0xA0C38094, (q31_t)0x5585ADAC,
	(q31_t)0xA0A1F24C, (q31_t)0x556040E2, (q31_t)0xA08072BA,
	(q31_t)0x553AC6ED, (q31_t)0xA05F01E1, (q31_t)0x55153FD4,
	(q31_t)0xA03D9FC7, (q31_t)0x54EFAB9C, (q31_t)0xA01C4C72,
	(q31_t)0x54CA0A4A, (q31_t)0x9FFB07E7, (q31_t)0x54A45BE5,
	(q31_t)0x9FD9D22A, (q31_t)0x547EA073, (q31_t)0x9FB8AB41,
	(q31_t)0x5458D7F9, (q31_t)0x9F979331, (q31_t)0x5433027D,
	(q31_t)0x9F7689FF, (q31_t)0x540D2005, (q31_t)0x9F558FB0,
	(q31_t)0x53E73097, (q31_t)0x9F34A449, (q31_t)0x53C13438,
	(q31_t)0x9F13C7D0, (q31_t)0x539B2AEF, (q31_t)0x9EF2FA48,
	(q31_t)0x537514C1, (q31_t)0x9ED23BB9, (q31_t)0x534EF1B5,
	(q31_t)0x9EB18C26, (q31_t)0x5328C1D0, (q31_t)0x9E90EB94,
	(q31_t)0x53028517, (q31_t)0x9E705A09, (q31_t)0x52DC3B92,
	(q31_t)0x9E4FD789, (q31_t)0x52B5E545, (q31_t)0x9E2F641A,
	(q31_t)0x528F8237, (q31_t)0x9E0EFFC1, (q31_t)0x5269126E,
	(q31_t)0x9DEEAA82, (q31_t)0x524295EF, (q31_t)0x9DCE6462,
	(q31_t)0x521C0CC1, (q31_t)0x9DAE2D68, (q31_t)0x51F576E9,
	(q31_t)0x9D8E0596, (q31_t)0x51CED46E, (q31_t)0x9D6DECF4,
	(q31_t)0x51A82555, (q31_t)0x9D4DE384, (q31_t)0x518169A4,
	(q31_t)0x9D2DE94D, (q31_t)0x515AA162, (q31_t)0x9D0DFE53,
	(q31_t)0x5133CC94, (q31_t)0x9CEE229C, (q31_t)0x510CEB40,
	(q31_t)0x9CCE562B, (q31_t)0x50E5FD6C, (q31_t)0x9CAE9907,
	(q31_t)0x50BF031F, (q31_t)0x9C8EEB33, (q31_t)0x5097FC5E,
	(q31_t)0x9C6F4CB5, (q31_t)0x5070E92F, (q31_t)0x9C4FBD92,
	(q31_t)0x5049C999, (q31_t)0x9C303DCF, (q31_t)0x50229DA0,
	(q31_t)0x9C10CD70, (q31_t)0x4FFB654D, (q31_t)0x9BF16C7A,
	(q31_t)0x4FD420A3, (q31_t)0x9BD21AF2, (q31_t)0x4FACCFAB,
	(q31_t)0x9BB2D8DD, (q31_t)0x4F857268, (q31_t)0x9B93A640,
	(q31_t)0x4F5E08E3, (q31_t)0x9B748320, (q31_t)0x4F369320,
	(q31_t)0x9B556F80, (q31_t)0x4F0F1126, (q31_t)0x9B366B67,
	(q31_t)0x4EE782FA, (q31_t)0x9B1776D9, (q31_t)0x4EBFE8A4,
	(q31_t)0x9AF891DB, (q31_t)0x4E984229, (q31_t)0x9AD9BC71,
	(q31_t)0x4E708F8F, (q31_t)0x9ABAF6A0, (q31_t)0x4E48D0DC,
	(q31_t)0x9A9C406D, (q31_t)0x4E210617, (q31_t)0x9A7D99DD,
	(q31_t)0x4DF92F45, (q31_t)0x9A5F02F5, (q31_t)0x4DD14C6E,
	(q31_t)0x9A407BB8, (q31_t)0x4DA95D96, (q31_t)0x9A22042C,
	(q31_t)0x4D8162C4, (q31_t)0x9A039C56, (q31_t)0x4D595BFE,
	(q31_t)0x99E5443A, (q31_t)0x4D31494B, (q31_t)0x99C6FBDE,
	(q31_t)0x4D092AB0, (q31_t)0x99A8C344, (q31_t)0x4CE10034,
	(q31_t)0x998A9A73, (q31_t)0x4CB8C9DD, (q31_t)0x996C816F,
	(q31_t)0x4C9087B1, (q31_t)0x994E783C, (q31_t)0x4C6839B6,
	(q31_t)0x99307EE0, (q31_t)0x4C3FDFF3, (q31_t)0x9912955E,
	(q31_t)0x4C177A6E, (q31_t)0x98F4BBBC, (q31_t)0x4BEF092D,
	(q31_t)0x98D6F1FE, (q31_t)0x4BC68C36, (q31_t)0x98B93828,
	(q31_t)0x4B9E038F, (q31_t)0x989B8E3F, (q31_t)0x4B756F3F,
	(q31_t)0x987DF449, (q31_t)0x4B4CCF4D, (q31_t)0x98606A48,
	(q31_t)0x4B2423BD, (q31_t)0x9842F043, (q31_t)0x4AFB6C97,
	(q31_t)0x9825863D, (q31_t)0x4AD2A9E1, (q31_t)0x98082C3B,
	(q31_t)0x4AA9DBA1, (q31_t)0x97EAE241, (q31_t)0x4A8101DE,
	(q31_t)0x97CDA855, (q31_t)0x4A581C9D, (q31_t)0x97B07E7A,
	(q31_t)0x4A2F2BE5, (q31_t)0x979364B5, (q31_t)0x4A062FBD,
	(q31_t)0x97765B0A, (q31_t)0x49DD282A, (q31_t)0x9759617E,
	(q31_t)0x49B41533, (q31_t)0x973C7816, (q31_t)0x498AF6DE,
	(q31_t)0x971F9ED6, (q31_t)0x4961CD32, (q31_t)0x9702D5C2,
	(q31_t)0x49389836, (q31_t)0x96E61CDF, (q31_t)0x490F57EE,
	(q31_t)0x96C97431, (q31_t)0x48E60C62, (q31_t)0x96ACDBBD,
	(q31_t)0x48BCB598, (q31_t)0x96905387, (q31_t)0x48935397,
	(q31_t)0x9673DB94, (q31_t)0x4869E664, (q31_t)0x965773E7,
	(q31_t)0x48406E07, (q31_t)0x963B1C85, (q31_t)0x4816EA85,
	(q31_t)0x961ED573, (q31_t)0x47ED5BE6, (q31_t)0x96029EB5,
	(q31_t)0x47C3C22E, (q31_t)0x95E6784F, (q31_t)0x479A1D66,
	(q31_t)0x95CA6246, (q31_t)0x47706D93, (q31_t)0x95AE5C9E,
	(q31_t)0x4746B2BC, (q31_t)0x9592675B, (q31_t)0x471CECE6,
	(q31_t)0x95768282, (q31_t)0x46F31C1A, (q31_t)0x955AAE17,
	(q31_t)0x46C9405C, (q31_t)0x953EEA1E, (q31_t)0x469F59B4,
	(q31_t)0x9523369B, (q31_t)0x46756827, (q31_t)0x95079393,
	(q31_t)0x464B6BBD, (q31_t)0x94EC010B, (q31_t)0x4621647C,
	(q31_t)0x94D07F05, (q31_t)0x45F7526B, (q31_t)0x94B50D87,
	(q31_t)0x45CD358F, (q31_t)0x9499AC95, (q31_t)0x45A30DF0,
	(q31_t)0x947E5C32, (q31_t)0x4578DB93, (q31_t)0x94631C64,
	(q31_t)0x454E9E80, (q31_t)0x9447ED2F, (q31_t)0x452456BC,
	(q31_t)0x942CCE95, (q31_t)0x44FA044F, (q31_t)0x9411C09D,
	(q31_t)0x44CFA73F, (q31_t)0x93F6C34A, (q31_t)0x44A53F93,
	(q31_t)0x93DBD69F, (q31_t)0x447ACD50, (q31_t)0x93C0FAA2,
	(q31_t)0x4450507E, (q31_t)0x93A62F56, (q31_t)0x4425C923,
	(q31_t)0x938B74C0, (q31_t)0x43FB3745, (q31_t)0x9370CAE4,
	(q31_t)0x43D09AEC, (q31_t)0x935631C5, (q31_t)0x43A5F41E,
	(q31_t)0x933BA968, (q31_t)0x437B42E1, (q31_t)0x932131D1,
	(q31_t)0x4350873C, (q31_t)0x9306CB04, (q31_t)0x4325C135,
	(q31_t)0x92EC7505, (q31_t)0x42FAF0D4, (q31_t)0x92D22FD8,
	(q31_t)0x42D0161E, (q31_t)0x92B7FB82, (q31_t)0x42A5311A,
	(q31_t)0x929DD805, (q31_t)0x427A41D0, (q31_t)0x9283C567,
	(q31_t)0x424F4845, (q31_t)0x9269C3AC, (q31_t)0x42244480,
	(q31_t)0x924FD2D6, (q31_t)0x41F93688, (q31_t)0x9235F2EB,
	(q31_t)0x41CE1E64, (q31_t)0x921C23EE, (q31_t)0x41A2FC1A,
	(q31_t)0x920265E4, (q31_t)0x4177CFB0, (q31_t)0x91E8B8D0,
	(q31_t)0x414C992E, (q31_t)0x91CF1CB6, (q31_t)0x4121589A,
	(q31_t)0x91B5919A, (q31_t)0x40F60DFB, (q31_t)0x919C1780,
	(q31_t)0x40CAB957, (q31_t)0x9182AE6C, (q31_t)0x409F5AB6,
	(q31_t)0x91695663, (q31_t)0x4073F21D, (q31_t)0x91500F67,
	(q31_t)0x40487F93, (q31_t)0x9136D97D, (q31_t)0x401D0320,
	(q31_t)0x911DB4A8, (q31_t)0x3FF17CCA, (q31_t)0x9104A0ED,
	(q31_t)0x3FC5EC97, (q31_t)0x90EB9E50, (q31_t)0x3F9A528F,
	(q31_t)0x90D2ACD3, (q31_t)0x3F6EAEB8, (q31_t)0x90B9CC7C,
	(q31_t)0x3F430118, (q31_t)0x90A0FD4E, (q31_t)0x3F1749B7,
	(q31_t)0x90883F4C, (q31_t)0x3EEB889C, (q31_t)0x906F927B,
	(q31_t)0x3EBFBDCC, (q31_t)0x9056F6DF, (q31_t)0x3E93E94F,
	(q31_t)0x903E6C7A, (q31_t)0x3E680B2C, (q31_t)0x9025F352,
	(q31_t)0x3E3C2369, (q31_t)0x900D8B69, (q31_t)0x3E10320D,
	(q31_t)0x8FF534C4, (q31_t)0x3DE4371F, (q31_t)0x8FDCEF66,
	(q31_t)0x3DB832A5, (q31_t)0x8FC4BB53, (q31_t)0x3D8C24A7,
	(q31_t)0x8FAC988E, (q31_t)0x3D600D2B, (q31_t)0x8F94871D,
	(q31_t)0x3D33EC39, (q31_t)0x8F7C8701, (q31_t)0x3D07C1D5,
	(q31_t)0x8F64983F, (q31_t)0x3CDB8E09, (q31_t)0x8F4CBADB,
	(q31_t)0x3CAF50DA, (q31_t)0x8F34EED8, (q31_t)0x3C830A4F,
	(q31_t)0x8F1D343A, (q31_t)0x3C56BA70, (q31_t)0x8F058B04,
	(q31_t)0x3C2A6142, (q31_t)0x8EEDF33B, (q31_t)0x3BFDFECD,
	(q31_t)0x8ED66CE1, (q31_t)0x3BD19317, (q31_t)0x8EBEF7FB,
	(q31_t)0x3BA51E29, (q31_t)0x8EA7948C, (q31_t)0x3B78A007,
	(q31_t)0x8E904298, (q31_t)0x3B4C18BA, (q31_t)0x8E790222,
	(q31_t)0x3B1F8847, (q31_t)0x8E61D32D, (q31_t)0x3AF2EEB7,
	(q31_t)0x8E4AB5BF, (q31_t)0x3AC64C0F, (q31_t)0x8E33A9D9,
	(q31_t)0x3A99A057, (q31_t)0x8E1CAF80, (q31_t)0x3A6CEB95,
	(q31_t)0x8E05C6B7, (q31_t)0x3A402DD1, (q31_t)0x8DEEEF82,
	(q31_t)0x3A136712, (q31_t)0x8DD829E4, (q31_t)0x39E6975D,
	(q31_t)0x8DC175E0, (q31_t)0x39B9BEBB, (q31_t)0x8DAAD37B,
	(q31_t)0x398CDD32, (q31_t)0x8D9442B7, (q31_t)0x395FF2C9,
	(q31_t)0x8D7DC399, (q31_t)0x3932FF87, (q31_t)0x8D675623,
	(q31_t)0x39060372, (q31_t)0x8D50FA59, (q31_t)0x38D8FE93,
	(q31_t)0x8D3AB03F, (q31_t)0x38ABF0EF, (q31_t)0x8D2477D8,
	(q31_t)0x387EDA8E, (q31_t)0x8D0E5127, (q31_t)0x3851BB76,
	(q31_t)0x8CF83C30, (q31_t)0x382493B0, (q31_t)0x8CE238F6,
	(q31_t)0x37F76340, (q31_t)0x8CCC477D, (q31_t)0x37CA2A30,
	(q31_t)0x8CB667C7, (q31_t)0x379CE884, (q31_t)0x8CA099D9,
	(q31_t)0x376F9E46, (q31_t)0x8C8ADDB6, (q31_t)0x37424B7A,
	(q31_t)0x8C753361, (q31_t)0x3714F02A, (q31_t)0x8C5F9ADD,
	(q31_t)0x36E78C5A, (q31_t)0x8C4A142F, (q31_t)0x36BA2013,
	(q31_t)0x8C349F58, (q31_t)0x368CAB5C, (q31_t)0x8C1F3C5C,
	(q31_t)0x365F2E3B, (q31_t)0x8C09EB40, (q31_t)0x3631A8B7,
	(q31_t)0x8BF4AC05, (q31_t)0x36041AD9, (q31_t)0x8BDF7EAF,
	(q31_t)0x35D684A5, (q31_t)0x8BCA6342, (q31_t)0x35A8E624,
	(q31_t)0x8BB559C1, (q31_t)0x357B3F5D, (q31_t)0x8BA0622F,
	(q31_t)0x354D9056, (q31_t)0x8B8B7C8F, (q31_t)0x351FD917,
	(q31_t)0x8B76A8E4, (q31_t)0x34F219A7, (q31_t)0x8B61E732,
	(q31_t)0x34C4520D, (q31_t)0x8B4D377C, (q31_t)0x3496824F,
	(q31_t)0x8B3899C5, (q31_t)0x3468AA76, (q31_t)0x8B240E10,
	(q31_t)0x343ACA87, (q31_t)0x8B0F9461, (q31_t)0x340CE28A,
	(q31_t)0x8AFB2CBA, (q31_t)0x33DEF287, (q31_t)0x8AE6D71F,
	(q31_t)0x33B0FA84, (q31_t)0x8AD29393, (q31_t)0x3382FA88,
	(q31_t)0x8ABE6219, (q31_t)0x3354F29A, (q31_t)0x8AAA42B4,
	(q31_t)0x3326E2C2, (q31_t)0x8A963567, (q31_t)0x32F8CB07,
	(q31_t)0x8A823A35, (q31_t)0x32CAAB6F, (q31_t)0x8A6E5122,
	(q31_t)0x329C8402, (q31_t)0x8A5A7A30, (q31_t)0x326E54C7,
	(q31_t)0x8A46B563, (q31_t)0x32401DC5, (q31_t)0x8A3302BD,
	(q31_t)0x3211DF03, (q31_t)0x8A1F6242, (q31_t)0x31E39889,
	(q31_t)0x8A0BD3F5, (q31_t)0x31B54A5D, (q31_t)0x89F857D8,
	(q31_t)0x3186F487, (q31_t)0x89E4EDEE, (q31_t)0x3158970D,
	(q31_t)0x89D1963C, (q31_t)0x312A31F8, (q31_t)0x89BE50C3,
	(q31_t)0x30FBC54D, (q31_t)0x89AB1D86, (q31_t)0x30CD5114,
	(q31_t)0x8997FC89, (q31_t)0x309ED555, (q31_t)0x8984EDCF,
	(q31_t)0x30705217, (q31_t)0x8971F15A, (q31_t)0x3041C760,
	(q31_t)0x895F072D, (q31_t)0x30133538, (q31_t)0x894C2F4C,
	(q31_t)0x2FE49BA6, (q31_t)0x893969B9, (q31_t)0x2FB5FAB2,
	(q31_t)0x8926B677, (q31_t)0x2F875262, (q31_t)0x89141589,
	(q31_t)0x2F58A2BD, (q31_t)0x890186F1, (q31_t)0x2F29EBCC,
	(q31_t)0x88EF0AB4, (q31_t)0x2EFB2D94, (q31_t)0x88DCA0D3,
	(q31_t)0x2ECC681E, (q31_t)0x88CA4951, (q31_t)0x2E9D9B70,
	(q31_t)0x88B80431, (q31_t)0x2E6EC792, (q31_t)0x88A5D177,
	(q31_t)0x2E3FEC8B, (q31_t)0x8893B124, (q31_t)0x2E110A62,
	(q31_t)0x8881A33C, (q31_t)0x2DE2211E, (q31_t)0x886FA7C2,
	(q31_t)0x2DB330C7, (q31_t)0x885DBEB7, (q31_t)0x2D843963,
	(q31_t)0x884BE820, (q31_t)0x2D553AFB, (q31_t)0x883A23FE,
	(q31_t)0x2D263595, (q31_t)0x88287255, (q31_t)0x2CF72939,
	(q31_t)0x8816D327, (q31_t)0x2CC815ED, (q31_t)0x88054677,
	(q31_t)0x2C98FBBA, (q31_t)0x87F3CC47, (q31_t)0x2C69DAA6,
	(q31_t)0x87E2649B, (q31_t)0x2C3AB2B9, (q31_t)0x87D10F75,
	(q31_t)0x2C0B83F9, (q31_t)0x87BFCCD7, (q31_t)0x2BDC4E6F,
	(q31_t)0x87AE9CC5, (q31_t)0x2BAD1221, (q31_t)0x879D7F40,
	(q31_t)0x2B7DCF17, (q31_t)0x878C744C, (q31_t)0x2B4E8558,
	(q31_t)0x877B7BEC, (q31_t)0x2B1F34EB, (q31_t)0x876A9621,
	(q31_t)0x2AEFDDD8, (q31_t)0x8759C2EF, (q31_t)0x2AC08025,
	(q31_t)0x87490257, (q31_t)0x2A911BDB, (q31_t)0x8738545E,
	(q31_t)0x2A61B101, (q31_t)0x8727B904, (q31_t)0x2A323F9D,
	(q31_t)0x8717304E, (q31_t)0x2A02C7B8, (q31_t)0x8706BA3C,
	(q31_t)0x29D34958, (q31_t)0x86F656D3, (q31_t)0x29A3C484,
	(q31_t)0x86E60614, (q31_t)0x29743945, (q31_t)0x86D5C802,
	(q31_t)0x2944A7A2, (q31_t)0x86C59C9F, (q31_t)0x29150FA1,
	(q31_t)0x86B583EE, (q31_t)0x28E5714A, (q31_t)0x86A57DF1,
	(q31_t)0x28B5CCA5, (q31_t)0x86958AAB, (q31_t)0x288621B9,
	(q31_t)0x8685AA1F, (q31_t)0x2856708C, (q31_t)0x8675DC4E,
	(q31_t)0x2826B928, (q31_t)0x8666213C, (q31_t)0x27F6FB92,
	(q31_t)0x865678EA, (q31_t)0x27C737D2, (q31_t)0x8646E35B,
	(q31_t)0x27976DF1, (q31_t)0x86376092, (q31_t)0x27679DF4,
	(q31_t)0x8627F090, (q31_t)0x2737C7E3, (q31_t)0x86189359,
	(q31_t)0x2707EBC6, (q31_t)0x860948EE, (q31_t)0x26D809A5,
	(q31_t)0x85FA1152, (q31_t)0x26A82185, (q31_t)0x85EAEC88,
	(q31_t)0x26783370, (q31_t)0x85DBDA91, (q31_t)0x26483F6C,
	(q31_t)0x85CCDB70, (q31_t)0x26184581, (q31_t)0x85BDEF27,
	(q31_t)0x25E845B5, (q31_t)0x85AF15B9, (q31_t)0x25B84012,
	(q31_t)0x85A04F28, (q31_t)0x2588349D, (q31_t)0x85919B75,
	(q31_t)0x2558235E, (q31_t)0x8582FAA4, (q31_t)0x25280C5D,
	(q31_t)0x85746CB7, (q31_t)0x24F7EFA1, (q31_t)0x8565F1B0,
	(q31_t)0x24C7CD32, (q31_t)0x85578991, (q31_t)0x2497A517,
	(q31_t)0x8549345C, (q31_t)0x24677757, (q31_t)0x853AF214,
	(q31_t)0x243743FA, (q31_t)0x852CC2BA, (q31_t)0x24070B07,
	(q31_t)0x851EA652, (q31_t)0x23D6CC86, (q31_t)0x85109CDC,
	(q31_t)0x23A6887E, (q31_t)0x8502A65C, (q31_t)0x23763EF7,
	(q31_t)0x84F4C2D3, (q31_t)0x2345EFF7, (q31_t)0x84E6F244,
	(q31_t)0x23159B87, (q31_t)0x84D934B0, (q31_t)0x22E541AE,
	(q31_t)0x84CB8A1B, (q31_t)0x22B4E274, (q31_t)0x84BDF285,
	(q31_t)0x22847DDF, (q31_t)0x84B06DF1, (q31_t)0x225413F8,
	(q31_t)0x84A2FC62, (q31_t)0x2223A4C5, (q31_t)0x84959DD9,
	(q31_t)0x21F3304E, (q31_t)0x84885257, (q31_t)0x21C2B69C,
	(q31_t)0x847B19E1, (q31_t)0x219237B4, (q31_t)0x846DF476,
	(q31_t)0x2161B39F, (q31_t)0x8460E21A, (q31_t)0x21312A65,
	(q31_t)0x8453E2CE, (q31_t)0x21009C0B, (q31_t)0x8446F695,
	(q31_t)0x20D0089B, (q31_t)0x843A1D70, (q31_t)0x209F701C,
	(q31_t)0x842D5761, (q31_t)0x206ED295, (q31_t)0x8420A46B,
	(q31_t)0x203E300D, (q31_t)0x8414048F, (q31_t)0x200D888C,
	(q31_t)0x840777CF, (q31_t)0x1FDCDC1A, (q31_t)0x83FAFE2E,
	(q31_t)0x1FAC2ABF, (q31_t)0x83EE97AC, (q31_t)0x1F7B7480,
	(q31_t)0x83E2444D, (q31_t)0x1F4AB967, (q31_t)0x83D60411,
	(q31_t)0x1F19F97B, (q31_t)0x83C9D6FB, (q31_t)0x1EE934C2,
	(q31_t)0x83BDBD0D, (q31_t)0x1EB86B46, (q31_t)0x83B1B649,
	(q31_t)0x1E879D0C, (q31_t)0x83A5C2B0, (q31_t)0x1E56CA1E,
	(q31_t)0x8399E244, (q31_t)0x1E25F281, (q31_t)0x838E1507,
	(q31_t)0x1DF5163F, (q31_t)0x83825AFB, (q31_t)0x1DC4355D,
	(q31_t)0x8376B422, (q31_t)0x1D934FE5, (q31_t)0x836B207D,
	(q31_t)0x1D6265DD, (q31_t)0x835FA00E, (q31_t)0x1D31774D,
	(q31_t)0x835432D8, (q31_t)0x1D00843C, (q31_t)0x8348D8DB,
	(q31_t)0x1CCF8CB3, (q31_t)0x833D921A, (q31_t)0x1C9E90B8,
	(q31_t)0x83325E97, (q31_t)0x1C6D9053, (q31_t)0x83273E52,
	(q31_t)0x1C3C8B8C, (q31_t)0x831C314E, (q31_t)0x1C0B826A,
	(q31_t)0x8311378C, (q31_t)0x1BDA74F5, (q31_t)0x8306510F,
	(q31_t)0x1BA96334, (q31_t)0x82FB7DD8, (q31_t)0x1B784D30,
	(q31_t)0x82F0BDE8, (q31_t)0x1B4732EF, (q31_t)0x82E61141,
	(q31_t)0x1B161479, (q31_t)0x82DB77E5, (q31_t)0x1AE4F1D6,
	(q31_t)0x82D0F1D5, (q31_t)0x1AB3CB0C, (q31_t)0x82C67F13,
	(q31_t)0x1A82A025, (q31_t)0x82BC1FA1, (q31_t)0x1A517127,
	(q31_t)0x82B1D381, (q31_t)0x1A203E1B, (q31_t)0x82A79AB3,
	(q31_t)0x19EF0706, (q31_t)0x829D753A, (q31_t)0x19BDCBF2,
	(q31_t)0x82936316, (q31_t)0x198C8CE6, (q31_t)0x8289644A,
	(q31_t)0x195B49E9, (q31_t)0x827F78D8, (q31_t)0x192A0303,
	(q31_t)0x8275A0C0, (q31_t)0x18F8B83C, (q31_t)0x826BDC04,
	(q31_t)0x18C7699B, (q31_t)0x82622AA5, (q31_t)0x18961727,
	(q31_t)0x82588CA6, (q31_t)0x1864C0E9, (q31_t)0x824F0208,
	(q31_t)0x183366E8, (q31_t)0x82458ACB, (q31_t)0x1802092C,
	(q31_t)0x823C26F2, (q31_t)0x17D0A7BB, (q31_t)0x8232D67E,
	(q31_t)0x179F429F, (q31_t)0x82299971, (q31_t)0x176DD9DE,
	(q31_t)0x82206FCB, (q31_t)0x173C6D80, (q31_t)0x8217598F,
	(q31_t)0x170AFD8D, (q31_t)0x820E56BE, (q31_t)0x16D98A0C,
	(q31_t)0x82056758, (q31_t)0x16A81305, (q31_t)0x81FC8B60,
	(q31_t)0x1676987F, (q31_t)0x81F3C2D7, (q31_t)0x16451A83,
	(q31_t)0x81EB0DBD, (q31_t)0x16139917, (q31_t)0x81E26C16,
	(q31_t)0x15E21444, (q31_t)0x81D9DDE1, (q31_t)0x15B08C11,
	(q31_t)0x81D16320, (q31_t)0x157F0086, (q31_t)0x81C8FBD5,
	(q31_t)0x154D71AA, (q31_t)0x81C0A801, (q31_t)0x151BDF85,
	(q31_t)0x81B867A4, (q31_t)0x14EA4A1F, (q31_t)0x81B03AC1,
	(q31_t)0x14B8B17F, (q31_t)0x81A82159, (q31_t)0x148715AD,
	(q31_t)0x81A01B6C, (q31_t)0x145576B1, (q31_t)0x819828FD,
	(q31_t)0x1423D492, (q31_t)0x81904A0C, (q31_t)0x13F22F57,
	(q31_t)0x81887E9A, (q31_t)0x13C0870A, (q31_t)0x8180C6A9,
	(q31_t)0x138EDBB0, (q31_t)0x8179223A, (q31_t)0x135D2D53,
	(q31_t)0x8171914E, (q31_t)0x132B7BF9, (q31_t)0x816A13E6,
	(q31_t)0x12F9C7AA, (q31_t)0x8162AA03, (q31_t)0x12C8106E,
	(q31_t)0x815B53A8, (q31_t)0x1296564D, (q31_t)0x815410D3,
	(q31_t)0x1264994E, (q31_t)0x814CE188, (q31_t)0x1232D978,
	(q31_t)0x8145C5C6, (q31_t)0x120116D4, (q31_t)0x813EBD90,
	(q31_t)0x11CF516A, (q31_t)0x8137C8E6, (q31_t)0x119D8940,
	(q31_t)0x8130E7C8, (q31_t)0x116BBE5F, (q31_t)0x812A1A39,
	(q31_t)0x1139F0CE, (q31_t)0x81236039, (q31_t)0x11082096,
	(q31_t)0x811CB9CA, (q31_t)0x10D64DBC, (q31_t)0x811626EC,
	(q31_t)0x10A4784A, (q31_t)0x810FA7A0, (q31_t)0x1072A047,
	(q31_t)0x81093BE8, (q31_t)0x1040C5BB, (q31_t)0x8102E3C3,
	(q31_t)0x100EE8AD, (q31_t)0x80FC9F35, (q31_t)0x0FDD0925,
	(q31_t)0x80F66E3C, (q31_t)0x0FAB272B, (q31_t)0x80F050DB,
	(q31_t)0x0F7942C6, (q31_t)0x80EA4712, (q31_t)0x0F475BFE,
	(q31_t)0x80E450E2, (q31_t)0x0F1572DC, (q31_t)0x80DE6E4C,
	(q31_t)0x0EE38765, (q31_t)0x80D89F51, (q31_t)0x0EB199A3,
	(q31_t)0x80D2E3F1, (q31_t)0x0E7FA99D, (q31_t)0x80CD3C2F,
	(q31_t)0x0E4DB75B, (q31_t)0x80C7A80A, (q31_t)0x0E1BC2E3,
	(q31_t)0x80C22783, (q31_t)0x0DE9CC3F, (q31_t)0x80BCBA9C,
	(q31_t)0x0DB7D376, (q31_t)0x80B76155, (q31_t)0x0D85D88F,
	(q31_t)0x80B21BAF, (q31_t)0x0D53DB92, (q31_t)0x80ACE9AB,
	(q31_t)0x0D21DC87, (q31_t)0x80A7CB49, (q31_t)0x0CEFDB75,
	(q31_t)0x80A2C08B, (q31_t)0x0CBDD865, (q31_t)0x809DC970,
	(q31_t)0x0C8BD35E, (q31_t)0x8098E5FB, (q31_t)0x0C59CC67,
	(q31_t)0x8094162B, (q31_t)0x0C27C389, (q31_t)0x808F5A02,
	(q31_t)0x0BF5B8CB, (q31_t)0x808AB180, (q31_t)0x0BC3AC35,
	(q31_t)0x80861CA5, (q31_t)0x0B919DCE, (q31_t)0x80819B74,
	(q31_t)0x0B5F8D9F, (q31_t)0x807D2DEB, (q31_t)0x0B2D7BAE,
	(q31_t)0x8078D40D, (q31_t)0x0AFB6805, (q31_t)0x80748DD9,
	(q31_t)0x0AC952AA, (q31_t)0x80705B50, (q31_t)0x0A973BA5,
	(q31_t)0x806C3C73, (q31_t)0x0A6522FE, (q31_t)0x80683143,
	(q31_t)0x0A3308BC, (q31_t)0x806439C0, (q31_t)0x0A00ECE8,
	(q31_t)0x806055EA, (q31_t)0x09CECF89, (q31_t)0x805C85C3,
	(q31_t)0x099CB0A7, (q31_t)0x8058C94C, (q31_t)0x096A9049,
	(q31_t)0x80552083, (q31_t)0x09386E77, (q31_t)0x80518B6B,
	(q31_t)0x09064B3A, (q31_t)0x804E0A03, (q31_t)0x08D42698,
	(q31_t)0x804A9C4D, (q31_t)0x08A2009A, (q31_t)0x80474248,
	(q31_t)0x086FD947, (q31_t)0x8043FBF6, (q31_t)0x083DB0A7,
	(q31_t)0x8040C956, (q31_t)0x080B86C1, (q31_t)0x803DAA69,
	(q31_t)0x07D95B9E, (q31_t)0x803A9F31, (q31_t)0x07A72F45,
	(q31_t)0x8037A7AC, (q31_t)0x077501BE, (q31_t)0x8034C3DC,
	(q31_t)0x0742D310, (q31_t)0x8031F3C1, (q31_t)0x0710A344,
	(q31_t)0x802F375C, (q31_t)0x06DE7261, (q31_t)0x802C8EAD,
	(q31_t)0x06AC406F, (q31_t)0x8029F9B4, (q31_t)0x067A0D75,
	(q31_t)0x80277872, (q31_t)0x0647D97C, (q31_t)0x80250AE7,
	(q31_t)0x0615A48A, (q31_t)0x8022B113, (q31_t)0x05E36EA9,
	(q31_t)0x80206AF8, (q31_t)0x05B137DF, (q31_t)0x801E3894,
	(q31_t)0x057F0034, (q31_t)0x801C19E9, (q31_t)0x054CC7B0,
	(q31_t)0x801A0EF7, (q31_t)0x051A8E5C, (q31_t)0x801817BF,
	(q31_t)0x04E8543D, (q31_t)0x80163440, (q31_t)0x04B6195D,
	(q31_t)0x8014647A, (q31_t)0x0483DDC3, (q31_t)0x8012A86F,
	(q31_t)0x0451A176, (q31_t)0x8011001E, (q31_t)0x041F647F,
	(q31_t)0x800F6B88, (q31_t)0x03ED26E6, (q31_t)0x800DEAAC,
	(q31_t)0x03BAE8B1, (q31_t)0x800C7D8C, (q31_t)0x0388A9E9,
	(q31_t)0x800B2427, (q31_t)0x03566A96, (q31_t)0x8009DE7D,
	(q31_t)0x03242ABF, (q31_t)0x8008AC90, (q31_t)0x02F1EA6B,
	(q31_t)0x80078E5E, (q31_t)0x02BFA9A4, (q31_t)0x800683E8,
	(q31_t)0x028D6870, (q31_t)0x80058D2E, (q31_t)0x025B26D7,
	(q31_t)0x8004AA31, (q31_t)0x0228E4E1, (q31_t)0x8003DAF0,
	(q31_t)0x01F6A296, (q31_t)0x80031F6C, (q31_t)0x01C45FFE,
	(q31_t)0x800277A5, (q31_t)0x01921D1F, (q31_t)0x8001E39B,
	(q31_t)0x015FDA03, (q31_t)0x8001634D, (q31_t)0x012D96B0,
	(q31_t)0x8000F6BD, (q31_t)0x00FB532F, (q31_t)0x80009DE9,
	(q31_t)0x00C90F88, (q31_t)0x800058D3, (q31_t)0x0096CBC1,
	(q31_t)0x8000277A, (q31_t)0x006487E3, (q31_t)0x800009DE,
	(q31_t)0x003243F5, (q31_t)0x80000000, (q31_t)0x00000000,
	(q31_t)0x800009DE, (q31_t)0xFFCDBC0A, (q31_t)0x8000277A,
	(q31_t)0xFF9B781D, (q31_t)0x800058D3, (q31_t)0xFF69343E,
	(q31_t)0x80009DE9, (q31_t)0xFF36F078, (q31_t)0x8000F6BD,
	(q31_t)0xFF04ACD0, (q31_t)0x8001634D, (q31_t)0xFED2694F,
	(q31_t)0x8001E39B, (q31_t)0xFEA025FC, (q31_t)0x800277A5,
	(q31_t)0xFE6DE2E0, (q31_t)0x80031F6C, (q31_t)0xFE3BA001,
	(q31_t)0x8003DAF0, (q31_t)0xFE095D69, (q31_t)0x8004AA31,
	(q31_t)0xFDD71B1E, (q31_t)0x80058D2E, (q31_t)0xFDA4D928,
	(q31_t)0x800683E8, (q31_t)0xFD72978F, (q31_t)0x80078E5E,
	(q31_t)0xFD40565B, (q31_t)0x8008AC90, (q31_t)0xFD0E1594,
	(q31_t)0x8009DE7D, (q31_t)0xFCDBD541, (q31_t)0x800B2427,
	(q31_t)0xFCA99569, (q31_t)0x800C7D8C, (q31_t)0xFC775616,
	(q31_t)0x800DEAAC, (q31_t)0xFC45174E, (q31_t)0x800F6B88,
	(q31_t)0xFC12D919, (q31_t)0x8011001E, (q31_t)0xFBE09B80,
	(q31_t)0x8012A86F, (q31_t)0xFBAE5E89, (q31_t)0x8014647A,
	(q31_t)0xFB7C223C, (q31_t)0x80163440, (q31_t)0xFB49E6A2,
	(q31_t)0x801817BF, (q31_t)0xFB17ABC2, (q31_t)0x801A0EF7,
	(q31_t)0xFAE571A4, (q31_t)0x801C19E9, (q31_t)0xFAB3384F,
	(q31_t)0x801E3894, (q31_t)0xFA80FFCB, (q31_t)0x80206AF8,
	(q31_t)0xFA4EC820, (q31_t)0x8022B113, (q31_t)0xFA1C9156,
	(q31_t)0x80250AE7, (q31_t)0xF9EA5B75, (q31_t)0x80277872,
	(q31_t)0xF9B82683, (q31_t)0x8029F9B4, (q31_t)0xF985F28A,
	(q31_t)0x802C8EAD, (q31_t)0xF953BF90, (q31_t)0x802F375C,
	(q31_t)0xF9218D9E, (q31_t)0x8031F3C1, (q31_t)0xF8EF5CBB,
	(q31_t)0x8034C3DC, (q31_t)0xF8BD2CEF, (q31_t)0x8037A7AC,
	(q31_t)0xF88AFE41, (q31_t)0x803A9F31, (q31_t)0xF858D0BA,
	(q31_t)0x803DAA69, (q31_t)0xF826A461, (q31_t)0x8040C956,
	(q31_t)0xF7F4793E, (q31_t)0x8043FBF6, (q31_t)0xF7C24F58,
	(q31_t)0x80474248, (q31_t)0xF79026B8, (q31_t)0x804A9C4D,
	(q31_t)0xF75DFF65, (q31_t)0x804E0A03, (q31_t)0xF72BD967,
	(q31_t)0x80518B6B, (q31_t)0xF6F9B4C5, (q31_t)0x80552083,
	(q31_t)0xF6C79188, (q31_t)0x8058C94C, (q31_t)0xF6956FB6,
	(q31_t)0x805C85C3, (q31_t)0xF6634F58, (q31_t)0x806055EA,
	(q31_t)0xF6313076, (q31_t)0x806439C0, (q31_t)0xF5FF1317,
	(q31_t)0x80683143, (q31_t)0xF5CCF743, (q31_t)0x806C3C73,
	(q31_t)0xF59ADD01, (q31_t)0x80705B50, (q31_t)0xF568C45A,
	(q31_t)0x80748DD9, (q31_t)0xF536AD55, (q31_t)0x8078D40D,
	(q31_t)0xF50497FA, (q31_t)0x807D2DEB, (q31_t)0xF4D28451,
	(q31_t)0x80819B74, (q31_t)0xF4A07260, (q31_t)0x80861CA5,
	(q31_t)0xF46E6231, (q31_t)0x808AB180, (q31_t)0xF43C53CA,
	(q31_t)0x808F5A02, (q31_t)0xF40A4734, (q31_t)0x8094162B,
	(q31_t)0xF3D83C76, (q31_t)0x8098E5FB, (q31_t)0xF3A63398,
	(q31_t)0x809DC970, (q31_t)0xF3742CA1, (q31_t)0x80A2C08B,
	(q31_t)0xF342279A, (q31_t)0x80A7CB49, (q31_t)0xF310248A,
	(q31_t)0x80ACE9AB, (q31_t)0xF2DE2378, (q31_t)0x80B21BAF,
	(q31_t)0xF2AC246D, (q31_t)0x80B76155, (q31_t)0xF27A2770,
	(q31_t)0x80BCBA9C, (q31_t)0xF2482C89, (q31_t)0x80C22783,
	(q31_t)0xF21633C0, (q31_t)0x80C7A80A, (q31_t)0xF1E43D1C,
	(q31_t)0x80CD3C2F, (q31_t)0xF1B248A5, (q31_t)0x80D2E3F1,
	(q31_t)0xF1805662, (q31_t)0x80D89F51, (q31_t)0xF14E665C,
	(q31_t)0x80DE6E4C, (q31_t)0xF11C789A, (q31_t)0x80E450E2,
	(q31_t)0xF0EA8D23, (q31_t)0x80EA4712, (q31_t)0xF0B8A401,
	(q31_t)0x80F050DB, (q31_t)0xF086BD39, (q31_t)0x80F66E3C,
	(q31_t)0xF054D8D4, (q31_t)0x80FC9F35, (q31_t)0xF022F6DA,
	(q31_t)0x8102E3C3, (q31_t)0xEFF11752, (q31_t)0x81093BE8,
	(q31_t)0xEFBF3A44, (q31_t)0x810FA7A0, (q31_t)0xEF8D5FB8,
	(q31_t)0x811626EC, (q31_t)0xEF5B87B5, (q31_t)0x811CB9CA,
	(q31_t)0xEF29B243, (q31_t)0x81236039, (q31_t)0xEEF7DF6A,
	(q31_t)0x812A1A39, (q31_t)0xEEC60F31, (q31_t)0x8130E7C8,
	(q31_t)0xEE9441A0, (q31_t)0x8137C8E6, (q31_t)0xEE6276BF,
	(q31_t)0x813EBD90, (q31_t)0xEE30AE95, (q31_t)0x8145C5C6,
	(q31_t)0xEDFEE92B, (q31_t)0x814CE188, (q31_t)0xEDCD2687,
	(q31_t)0x815410D3, (q31_t)0xED9B66B2, (q31_t)0x815B53A8,
	(q31_t)0xED69A9B2, (q31_t)0x8162AA03, (q31_t)0xED37EF91,
	(q31_t)0x816A13E6, (q31_t)0xED063855, (q31_t)0x8171914E,
	(q31_t)0xECD48406, (q31_t)0x8179223A, (q31_t)0xECA2D2AC,
	(q31_t)0x8180C6A9, (q31_t)0xEC71244F, (q31_t)0x81887E9A,
	(q31_t)0xEC3F78F5, (q31_t)0x81904A0C, (q31_t)0xEC0DD0A8,
	(q31_t)0x819828FD, (q31_t)0xEBDC2B6D, (q31_t)0x81A01B6C,
	(q31_t)0xEBAA894E, (q31_t)0x81A82159, (q31_t)0xEB78EA52,
	(q31_t)0x81B03AC1, (q31_t)0xEB474E80, (q31_t)0x81B867A4,
	(q31_t)0xEB15B5E0, (q31_t)0x81C0A801, (q31_t)0xEAE4207A,
	(q31_t)0x81C8FBD5, (q31_t)0xEAB28E55, (q31_t)0x81D16320,
	(q31_t)0xEA80FF79, (q31_t)0x81D9DDE1, (q31_t)0xEA4F73EE,
	(q31_t)0x81E26C16, (q31_t)0xEA1DEBBB, (q31_t)0x81EB0DBD,
	(q31_t)0xE9EC66E8, (q31_t)0x81F3C2D7, (q31_t)0xE9BAE57C,
	(q31_t)0x81FC8B60, (q31_t)0xE9896780, (q31_t)0x82056758,
	(q31_t)0xE957ECFB, (q31_t)0x820E56BE, (q31_t)0xE92675F4,
	(q31_t)0x8217598F, (q31_t)0xE8F50273, (q31_t)0x82206FCB,
	(q31_t)0xE8C3927F, (q31_t)0x82299971, (q31_t)0xE8922621,
	(q31_t)0x8232D67E, (q31_t)0xE860BD60, (q31_t)0x823C26F2,
	(q31_t)0xE82F5844, (q31_t)0x82458ACB, (q31_t)0xE7FDF6D3,
	(q31_t)0x824F0208, (q31_t)0xE7CC9917, (q31_t)0x82588CA6,
	(q31_t)0xE79B3F16, (q31_t)0x82622AA5, (q31_t)0xE769E8D8,
	(q31_t)0x826BDC04, (q31_t)0xE7389664, (q31_t)0x8275A0C0,
	(q31_t)0xE70747C3, (q31_t)0x827F78D8, (q31_t)0xE6D5FCFC,
	(q31_t)0x8289644A, (q31_t)0xE6A4B616, (q31_t)0x82936316,
	(q31_t)0xE6737319, (q31_t)0x829D753A, (q31_t)0xE642340D,
	(q31_t)0x82A79AB3, (q31_t)0xE610F8F9, (q31_t)0x82B1D381,
	(q31_t)0xE5DFC1E4, (q31_t)0x82BC1FA1, (q31_t)0xE5AE8ED8,
	(q31_t)0x82C67F13, (q31_t)0xE57D5FDA, (q31_t)0x82D0F1D5,
	(q31_t)0xE54C34F3, (q31_t)0x82DB77E5, (q31_t)0xE51B0E2A,
	(q31_t)0x82E61141, (q31_t)0xE4E9EB86, (q31_t)0x82F0BDE8,
	(q31_t)0xE4B8CD10, (q31_t)0x82FB7DD8, (q31_t)0xE487B2CF,
	(q31_t)0x8306510F, (q31_t)0xE4569CCB, (q31_t)0x8311378C,
	(q31_t)0xE4258B0A, (q31_t)0x831C314E, (q31_t)0xE3F47D95,
	(q31_t)0x83273E52, (q31_t)0xE3C37473, (q31_t)0x83325E97,
	(q31_t)0xE3926FAC, (q31_t)0x833D921A, (q31_t)0xE3616F47,
	(q31_t)0x8348D8DB, (q31_t)0xE330734C, (q31_t)0x835432D8,
	(q31_t)0xE2FF7BC3, (q31_t)0x835FA00E, (q31_t)0xE2CE88B2,
	(q31_t)0x836B207D, (q31_t)0xE29D9A22, (q31_t)0x8376B422,
	(q31_t)0xE26CB01A, (q31_t)0x83825AFB, (q31_t)0xE23BCAA2,
	(q31_t)0x838E1507, (q31_t)0xE20AE9C1, (q31_t)0x8399E244,
	(q31_t)0xE1DA0D7E, (q31_t)0x83A5C2B0, (q31_t)0xE1A935E1,
	(q31_t)0x83B1B649, (q31_t)0xE17862F3, (q31_t)0x83BDBD0D,
	(q31_t)0xE14794B9, (q31_t)0x83C9D6FB, (q31_t)0xE116CB3D,
	(q31_t)0x83D60411, (q31_t)0xE0E60684, (q31_t)0x83E2444D,
	(q31_t)0xE0B54698, (q31_t)0x83EE97AC, (q31_t)0xE0848B7F,
	(q31_t)0x83FAFE2E, (q31_t)0xE053D541, (q31_t)0x840777CF,
	(q31_t)0xE02323E5, (q31_t)0x8414048F, (q31_t)0xDFF27773,
	(q31_t)0x8420A46B, (q31_t)0xDFC1CFF2, (q31_t)0x842D5761,
	(q31_t)0xDF912D6A, (q31_t)0x843A1D70, (q31_t)0xDF608FE3,
	(q31_t)0x8446F695, (q31_t)0xDF2FF764, (q31_t)0x8453E2CE,
	(q31_t)0xDEFF63F4, (q31_t)0x8460E21A, (q31_t)0xDECED59B,
	(q31_t)0x846DF476, (q31_t)0xDE9E4C60, (q31_t)0x847B19E1,
	(q31_t)0xDE6DC84B, (q31_t)0x84885257, (q31_t)0xDE3D4963,
	(q31_t)0x84959DD9, (q31_t)0xDE0CCFB1, (q31_t)0x84A2FC62,
	(q31_t)0xDDDC5B3A, (q31_t)0x84B06DF1, (q31_t)0xDDABEC07,
	(q31_t)0x84BDF285, (q31_t)0xDD7B8220, (q31_t)0x84CB8A1B,
	(q31_t)0xDD4B1D8B, (q31_t)0x84D934B0, (q31_t)0xDD1ABE51,
	(q31_t)0x84E6F244, (q31_t)0xDCEA6478, (q31_t)0x84F4C2D3,
	(q31_t)0xDCBA1008, (q31_t)0x8502A65C, (q31_t)0xDC89C108,
	(q31_t)0x85109CDC, (q31_t)0xDC597781, (q31_t)0x851EA652,
	(q31_t)0xDC293379, (q31_t)0x852CC2BA, (q31_t)0xDBF8F4F8,
	(q31_t)0x853AF214, (q31_t)0xDBC8BC05, (q31_t)0x8549345C,
	(q31_t)0xDB9888A8, (q31_t)0x85578991, (q31_t)0xDB685AE8,
	(q31_t)0x8565F1B0, (q31_t)0xDB3832CD, (q31_t)0x85746CB7,
	(q31_t)0xDB08105E, (q31_t)0x8582FAA4, (q31_t)0xDAD7F3A2,
	(q31_t)0x85919B75, (q31_t)0xDAA7DCA1, (q31_t)0x85A04F28,
	(q31_t)0xDA77CB62, (q31_t)0x85AF15B9, (q31_t)0xDA47BFED,
	(q31_t)0x85BDEF27, (q31_t)0xDA17BA4A, (q31_t)0x85CCDB70,
	(q31_t)0xD9E7BA7E, (q31_t)0x85DBDA91, (q31_t)0xD9B7C093,
	(q31_t)0x85EAEC88, (q31_t)0xD987CC8F, (q31_t)0x85FA1152,
	(q31_t)0xD957DE7A, (q31_t)0x860948EE, (q31_t)0xD927F65B,
	(q31_t)0x86189359, (q31_t)0xD8F81439, (q31_t)0x8627F090,
	(q31_t)0xD8C8381C, (q31_t)0x86376092, (q31_t)0xD898620C,
	(q31_t)0x8646E35B, (q31_t)0xD868920F, (q31_t)0x865678EA,
	(q31_t)0xD838C82D, (q31_t)0x8666213C, (q31_t)0xD809046D,
	(q31_t)0x8675DC4E, (q31_t)0xD7D946D7, (q31_t)0x8685AA1F,
	(q31_t)0xD7A98F73, (q31_t)0x86958AAB, (q31_t)0xD779DE46,
	(q31_t)0x86A57DF1, (q31_t)0xD74A335A, (q31_t)0x86B583EE,
	(q31_t)0xD71A8EB5, (q31_t)0x86C59C9F, (q31_t)0xD6EAF05E,
	(q31_t)0x86D5C802, (q31_t)0xD6BB585D, (q31_t)0x86E60614,
	(q31_t)0xD68BC6BA, (q31_t)0x86F656D3, (q31_t)0xD65C3B7B,
	(q31_t)0x8706BA3C, (q31_t)0xD62CB6A7, (q31_t)0x8717304E,
	(q31_t)0xD5FD3847, (q31_t)0x8727B904, (q31_t)0xD5CDC062,
	(q31_t)0x8738545E, (q31_t)0xD59E4EFE, (q31_t)0x87490257,
	(q31_t)0xD56EE424, (q31_t)0x8759C2EF, (q31_t)0xD53F7FDA,
	(q31_t)0x876A9621, (q31_t)0xD5102227, (q31_t)0x877B7BEC,
	(q31_t)0xD4E0CB14, (q31_t)0x878C744C, (q31_t)0xD4B17AA7,
	(q31_t)0x879D7F40, (q31_t)0xD48230E8, (q31_t)0x87AE9CC5,
	(q31_t)0xD452EDDE, (q31_t)0x87BFCCD7, (q31_t)0xD423B190,
	(q31_t)0x87D10F75, (q31_t)0xD3F47C06, (q31_t)0x87E2649B,
	(q31_t)0xD3C54D46, (q31_t)0x87F3CC47, (q31_t)0xD3962559,
	(q31_t)0x88054677, (q31_t)0xD3670445, (q31_t)0x8816D327,
	(q31_t)0xD337EA12, (q31_t)0x88287255, (q31_t)0xD308D6C6,
	(q31_t)0x883A23FE, (q31_t)0xD2D9CA6A, (q31_t)0x884BE820,
	(q31_t)0xD2AAC504, (q31_t)0x885DBEB7, (q31_t)0xD27BC69C,
	(q31_t)0x886FA7C2, (q31_t)0xD24CCF38, (q31_t)0x8881A33C,
	(q31_t)0xD21DDEE1, (q31_t)0x8893B124, (q31_t)0xD1EEF59E,
	(q31_t)0x88A5D177, (q31_t)0xD1C01374, (q31_t)0x88B80431,
	(q31_t)0xD191386D, (q31_t)0x88CA4951, (q31_t)0xD162648F,
	(q31_t)0x88DCA0D3, (q31_t)0xD13397E1, (q31_t)0x88EF0AB4,
	(q31_t)0xD104D26B, (q31_t)0x890186F1, (q31_t)0xD0D61433,
	(q31_t)0x89141589, (q31_t)0xD0A75D42, (q31_t)0x8926B677,
	(q31_t)0xD078AD9D, (q31_t)0x893969B9, (q31_t)0xD04A054D,
	(q31_t)0x894C2F4C, (q31_t)0xD01B6459, (q31_t)0x895F072D,
	(q31_t)0xCFECCAC7, (q31_t)0x8971F15A, (q31_t)0xCFBE389F,
	(q31_t)0x8984EDCF, (q31_t)0xCF8FADE8, (q31_t)0x8997FC89,
	(q31_t)0xCF612AAA, (q31_t)0x89AB1D86, (q31_t)0xCF32AEEB,
	(q31_t)0x89BE50C3, (q31_t)0xCF043AB2, (q31_t)0x89D1963C,
	(q31_t)0xCED5CE08, (q31_t)0x89E4EDEE, (q31_t)0xCEA768F2,
	(q31_t)0x89F857D8, (q31_t)0xCE790B78, (q31_t)0x8A0BD3F5,
	(q31_t)0xCE4AB5A2, (q31_t)0x8A1F6242, (q31_t)0xCE1C6776,
	(q31_t)0x8A3302BD, (q31_t)0xCDEE20FC, (q31_t)0x8A46B563,
	(q31_t)0xCDBFE23A, (q31_t)0x8A5A7A30, (q31_t)0xCD91AB38,
	(q31_t)0x8A6E5122, (q31_t)0xCD637BFD, (q31_t)0x8A823A35,
	(q31_t)0xCD355490, (q31_t)0x8A963567, (q31_t)0xCD0734F8,
	(q31_t)0x8AAA42B4, (q31_t)0xCCD91D3D, (q31_t)0x8ABE6219,
	(q31_t)0xCCAB0D65, (q31_t)0x8AD29393, (q31_t)0xCC7D0577,
	(q31_t)0x8AE6D71F, (q31_t)0xCC4F057B, (q31_t)0x8AFB2CBA,
	(q31_t)0xCC210D78, (q31_t)0x8B0F9461, (q31_t)0xCBF31D75,
	(q31_t)0x8B240E10, (q31_t)0xCBC53578, (q31_t)0x8B3899C5,
	(q31_t)0xCB975589, (q31_t)0x8B4D377C, (q31_t)0xCB697DB0,
	(q31_t)0x8B61E732, (q31_t)0xCB3BADF2, (q31_t)0x8B76A8E4,
	(q31_t)0xCB0DE658, (q31_t)0x8B8B7C8F, (q31_t)0xCAE026E8,
	(q31_t)0x8BA0622F, (q31_t)0xCAB26FA9, (q31_t)0x8BB559C1,
	(q31_t)0xCA84C0A2, (q31_t)0x8BCA6342, (q31_t)0xCA5719DB,
	(q31_t)0x8BDF7EAF, (q31_t)0xCA297B5A, (q31_t)0x8BF4AC05,
	(q31_t)0xC9FBE527, (q31_t)0x8C09EB40, (q31_t)0xC9CE5748,
	(q31_t)0x8C1F3C5C, (q31_t)0xC9A0D1C4, (q31_t)0x8C349F58,
	(q31_t)0xC97354A3, (q31_t)0x8C4A142F, (q31_t)0xC945DFEC,
	(q31_t)0x8C5F9ADD, (q31_t)0xC91873A5, (q31_t)0x8C753361,
	(q31_t)0xC8EB0FD6, (q31_t)0x8C8ADDB6, (q31_t)0xC8BDB485,
	(q31_t)0x8CA099D9, (q31_t)0xC89061BA, (q31_t)0x8CB667C7,
	(q31_t)0xC863177B, (q31_t)0x8CCC477D, (q31_t)0xC835D5D0,
	(q31_t)0x8CE238F6, (q31_t)0xC8089CBF, (q31_t)0x8CF83C30,
	(q31_t)0xC7DB6C50, (q31_t)0x8D0E5127, (q31_t)0xC7AE4489,
	(q31_t)0x8D2477D8, (q31_t)0xC7812571, (q31_t)0x8D3AB03F,
	(q31_t)0xC7540F10, (q31_t)0x8D50FA59, (q31_t)0xC727016C,
	(q31_t)0x8D675623, (q31_t)0xC6F9FC8D, (q31_t)0x8D7DC399,
	(q31_t)0xC6CD0079, (q31_t)0x8D9442B7, (q31_t)0xC6A00D36,
	(q31_t)0x8DAAD37B, (q31_t)0xC67322CD, (q31_t)0x8DC175E0,
	(q31_t)0xC6464144, (q31_t)0x8DD829E4, (q31_t)0xC61968A2,
	(q31_t)0x8DEEEF82, (q31_t)0xC5EC98ED, (q31_t)0x8E05C6B7,
	(q31_t)0xC5BFD22E, (q31_t)0x8E1CAF80, (q31_t)0xC593146A,
	(q31_t)0x8E33A9D9, (q31_t)0xC5665FA8, (q31_t)0x8E4AB5BF,
	(q31_t)0xC539B3F0, (q31_t)0x8E61D32D, (q31_t)0xC50D1148,
	(q31_t)0x8E790222, (q31_t)0xC4E077B8, (q31_t)0x8E904298,
	(q31_t)0xC4B3E746, (q31_t)0x8EA7948C, (q31_t)0xC4875FF8,
	(q31_t)0x8EBEF7FB, (q31_t)0xC45AE1D7, (q31_t)0x8ED66CE1,
	(q31_t)0xC42E6CE8, (q31_t)0x8EEDF33B, (q31_t)0xC4020132,
	(q31_t)0x8F058B04, (q31_t)0xC3D59EBD, (q31_t)0x8F1D343A,
	(q31_t)0xC3A9458F, (q31_t)0x8F34EED8, (q31_t)0xC37CF5B0,
	(q31_t)0x8F4CBADB, (q31_t)0xC350AF25, (q31_t)0x8F64983F,
	(q31_t)0xC32471F6, (q31_t)0x8F7C8701, (q31_t)0xC2F83E2A,
	(q31_t)0x8F94871D, (q31_t)0xC2CC13C7, (q31_t)0x8FAC988E,
	(q31_t)0xC29FF2D4, (q31_t)0x8FC4BB53, (q31_t)0xC273DB58,
	(q31_t)0x8FDCEF66, (q31_t)0xC247CD5A, (q31_t)0x8FF534C4,
	(q31_t)0xC21BC8E0, (q31_t)0x900D8B69, (q31_t)0xC1EFCDF2,
	(q31_t)0x9025F352, (q31_t)0xC1C3DC96, (q31_t)0x903E6C7A,
	(q31_t)0xC197F4D3, (q31_t)0x9056F6DF, (q31_t)0xC16C16B0,
	(q31_t)0x906F927B, (q31_t)0xC1404233, (q31_t)0x90883F4C,
	(q31_t)0xC1147763, (q31_t)0x90A0FD4E, (q31_t)0xC0E8B648,
	(q31_t)0x90B9CC7C, (q31_t)0xC0BCFEE7, (q31_t)0x90D2ACD3,
	(q31_t)0xC0915147, (q31_t)0x90EB9E50, (q31_t)0xC065AD70,
	(q31_t)0x9104A0ED, (q31_t)0xC03A1368, (q31_t)0x911DB4A8,
	(q31_t)0xC00E8335, (q31_t)0x9136D97D, (q31_t)0xBFE2FCDF,
	(q31_t)0x91500F67, (q31_t)0xBFB7806C, (q31_t)0x91695663,
	(q31_t)0xBF8C0DE2, (q31_t)0x9182AE6C, (q31_t)0xBF60A54A,
	(q31_t)0x919C1780, (q31_t)0xBF3546A8, (q31_t)0x91B5919A,
	(q31_t)0xBF09F204, (q31_t)0x91CF1CB6, (q31_t)0xBEDEA765,
	(q31_t)0x91E8B8D0, (q31_t)0xBEB366D1, (q31_t)0x920265E4,
	(q31_t)0xBE88304F, (q31_t)0x921C23EE, (q31_t)0xBE5D03E5,
	(q31_t)0x9235F2EB, (q31_t)0xBE31E19B, (q31_t)0x924FD2D6,
	(q31_t)0xBE06C977, (q31_t)0x9269C3AC, (q31_t)0xBDDBBB7F,
	(q31_t)0x9283C567, (q31_t)0xBDB0B7BA, (q31_t)0x929DD805,
	(q31_t)0xBD85BE2F, (q31_t)0x92B7FB82, (q31_t)0xBD5ACEE5,
	(q31_t)0x92D22FD8, (q31_t)0xBD2FE9E1, (q31_t)0x92EC7505,
	(q31_t)0xBD050F2C, (q31_t)0x9306CB04, (q31_t)0xBCDA3ECA,
	(q31_t)0x932131D1, (q31_t)0xBCAF78C3, (q31_t)0x933BA968,
	(q31_t)0xBC84BD1E, (q31_t)0x935631C5, (q31_t)0xBC5A0BE1,
	(q31_t)0x9370CAE4, (q31_t)0xBC2F6513, (q31_t)0x938B74C0,
	(q31_t)0xBC04C8BA, (q31_t)0x93A62F56, (q31_t)0xBBDA36DC,
	(q31_t)0x93C0FAA2, (q31_t)0xBBAFAF81, (q31_t)0x93DBD69F,
	(q31_t)0xBB8532AF, (q31_t)0x93F6C34A, (q31_t)0xBB5AC06C,
	(q31_t)0x9411C09D, (q31_t)0xBB3058C0, (q31_t)0x942CCE95,
	(q31_t)0xBB05FBB0, (q31_t)0x9447ED2F, (q31_t)0xBADBA943,
	(q31_t)0x94631C64, (q31_t)0xBAB1617F, (q31_t)0x947E5C32,
	(q31_t)0xBA87246C, (q31_t)0x9499AC95, (q31_t)0xBA5CF210,
	(q31_t)0x94B50D87, (q31_t)0xBA32CA70, (q31_t)0x94D07F05,
	(q31_t)0xBA08AD94, (q31_t)0x94EC010B, (q31_t)0xB9DE9B83,
	(q31_t)0x95079393, (q31_t)0xB9B49442, (q31_t)0x9523369B,
	(q31_t)0xB98A97D8, (q31_t)0x953EEA1E, (q31_t)0xB960A64B,
	(q31_t)0x955AAE17, (q31_t)0xB936BFA3, (q31_t)0x95768282,
	(q31_t)0xB90CE3E6, (q31_t)0x9592675B, (q31_t)0xB8E31319,
	(q31_t)0x95AE5C9E, (q31_t)0xB8B94D44, (q31_t)0x95CA6246,
	(q31_t)0xB88F926C, (q31_t)0x95E6784F, (q31_t)0xB865E299,
	(q31_t)0x96029EB5, (q31_t)0xB83C3DD1, (q31_t)0x961ED573,
	(q31_t)0xB812A419, (q31_t)0x963B1C85, (q31_t)0xB7E9157A,
	(q31_t)0x965773E7, (q31_t)0xB7BF91F8, (q31_t)0x9673DB94,
	(q31_t)0xB796199B, (q31_t)0x96905387, (q31_t)0xB76CAC68,
	(q31_t)0x96ACDBBD, (q31_t)0xB7434A67, (q31_t)0x96C97431,
	(q31_t)0xB719F39D, (q31_t)0x96E61CDF, (q31_t)0xB6F0A811,
	(q31_t)0x9702D5C2, (q31_t)0xB6C767CA, (q31_t)0x971F9ED6,
	(q31_t)0xB69E32CD, (q31_t)0x973C7816, (q31_t)0xB6750921,
	(q31_t)0x9759617E, (q31_t)0xB64BEACC, (q31_t)0x97765B0A,
	(q31_t)0xB622D7D5, (q31_t)0x979364B5, (q31_t)0xB5F9D042,
	(q31_t)0x97B07E7A, (q31_t)0xB5D0D41A, (q31_t)0x97CDA855,
	(q31_t)0xB5A7E362, (q31_t)0x97EAE241, (q31_t)0xB57EFE21,
	(q31_t)0x98082C3B, (q31_t)0xB556245E, (q31_t)0x9825863D,
	(q31_t)0xB52D561E, (q31_t)0x9842F043, (q31_t)0xB5049368,
	(q31_t)0x98606A48, (q31_t)0xB4DBDC42, (q31_t)0x987DF449,
	(q31_t)0xB4B330B2, (q31_t)0x989B8E3F, (q31_t)0xB48A90C0,
	(q31_t)0x98B93828, (q31_t)0xB461FC70, (q31_t)0x98D6F1FE,
	(q31_t)0xB43973C9, (q31_t)0x98F4BBBC, (q31_t)0xB410F6D2,
	(q31_t)0x9912955E, (q31_t)0xB3E88591, (q31_t)0x99307EE0,
	(q31_t)0xB3C0200C, (q31_t)0x994E783C, (q31_t)0xB397C649,
	(q31_t)0x996C816F, (q31_t)0xB36F784E, (q31_t)0x998A9A73,
	(q31_t)0xB3473622, (q31_t)0x99A8C344, (q31_t)0xB31EFFCB,
	(q31_t)0x99C6FBDE, (q31_t)0xB2F6D54F, (q31_t)0x99E5443A,
	(q31_t)0xB2CEB6B5, (q31_t)0x9A039C56, (q31_t)0xB2A6A401,
	(q31_t)0x9A22042C, (q31_t)0xB27E9D3B, (q31_t)0x9A407BB8,
	(q31_t)0xB256A26A, (q31_t)0x9A5F02F5, (q31_t)0xB22EB392,
	(q31_t)0x9A7D99DD, (q31_t)0xB206D0BA, (q31_t)0x9A9C406D,
	(q31_t)0xB1DEF9E8, (q31_t)0x9ABAF6A0, (q31_t)0xB1B72F23,
	(q31_t)0x9AD9BC71, (q31_t)0xB18F7070, (q31_t)0x9AF891DB,
	(q31_t)0xB167BDD6, (q31_t)0x9B1776D9, (q31_t)0xB140175B,
	(q31_t)0x9B366B67, (q31_t)0xB1187D05, (q31_t)0x9B556F80,
	(q31_t)0xB0F0EEDA, (q31_t)0x9B748320, (q31_t)0xB0C96CDF,
	(q31_t)0x9B93A640, (q31_t)0xB0A1F71C, (q31_t)0x9BB2D8DD,
	(q31_t)0xB07A8D97, (q31_t)0x9BD21AF2, (q31_t)0xB0533055,
	(q31_t)0x9BF16C7A, (q31_t)0xB02BDF5C, (q31_t)0x9C10CD70,
	(q31_t)0xB0049AB2, (q31_t)0x9C303DCF, (q31_t)0xAFDD625F,
	(q31_t)0x9C4FBD92, (q31_t)0xAFB63667, (q31_t)0x9C6F4CB5,
	(q31_t)0xAF8F16D0, (q31_t)0x9C8EEB33, (q31_t)0xAF6803A1,
	(q31_t)0x9CAE9907, (q31_t)0xAF40FCE0, (q31_t)0x9CCE562B,
	(q31_t)0xAF1A0293, (q31_t)0x9CEE229C, (q31_t)0xAEF314BF,
	(q31_t)0x9D0DFE53, (q31_t)0xAECC336B, (q31_t)0x9D2DE94D,
	(q31_t)0xAEA55E9D, (q31_t)0x9D4DE384, (q31_t)0xAE7E965B,
	(q31_t)0x9D6DECF4, (q31_t)0xAE57DAAA, (q31_t)0x9D8E0596,
	(q31_t)0xAE312B91, (q31_t)0x9DAE2D68, (q31_t)0xAE0A8916,
	(q31_t)0x9DCE6462, (q31_t)0xADE3F33E, (q31_t)0x9DEEAA82,
	(q31_t)0xADBD6A10, (q31_t)0x9E0EFFC1, (q31_t)0xAD96ED91,
	(q31_t)0x9E2F641A, (q31_t)0xAD707DC8, (q31_t)0x9E4FD789,
	(q31_t)0xAD4A1ABA, (q31_t)0x9E705A09, (q31_t)0xAD23C46D,
	(q31_t)0x9E90EB94, (q31_t)0xACFD7AE8, (q31_t)0x9EB18C26,
	(q31_t)0xACD73E30, (q31_t)0x9ED23BB9, (q31_t)0xACB10E4A,
	(q31_t)0x9EF2FA48, (q31_t)0xAC8AEB3E, (q31_t)0x9F13C7D0,
	(q31_t)0xAC64D510, (q31_t)0x9F34A449, (q31_t)0xAC3ECBC7,
	(q31_t)0x9F558FB0, (q31_t)0xAC18CF68, (q31_t)0x9F7689FF,
	(q31_t)0xABF2DFFA, (q31_t)0x9F979331, (q31_t)0xABCCFD82,
	(q31_t)0x9FB8AB41, (q31_t)0xABA72806, (q31_t)0x9FD9D22A,
	(q31_t)0xAB815F8C, (q31_t)0x9FFB07E7, (q31_t)0xAB5BA41A,
	(q31_t)0xA01C4C72, (q31_t)0xAB35F5B5, (q31_t)0xA03D9FC7,
	(q31_t)0xAB105464, (q31_t)0xA05F01E1, (q31_t)0xAAEAC02B,
	(q31_t)0xA08072BA, (q31_t)0xAAC53912, (q31_t)0xA0A1F24C,
	(q31_t)0xAA9FBF1D, (q31_t)0xA0C38094, (q31_t)0xAA7A5253,
	(q31_t)0xA0E51D8C, (q31_t)0xAA54F2B9, (q31_t)0xA106C92E,
	(q31_t)0xAA2FA055, (q31_t)0xA1288376, (q31_t)0xAA0A5B2D,
	(q31_t)0xA14A4C5E, (q31_t)0xA9E52347, (q31_t)0xA16C23E1,
	(q31_t)0xA9BFF8A8, (q31_t)0xA18E09F9, (q31_t)0xA99ADB56,
	(q31_t)0xA1AFFEA2, (q31_t)0xA975CB56, (q31_t)0xA1D201D7,
	(q31_t)0xA950C8AF, (q31_t)0xA1F41391, (q31_t)0xA92BD366,
	(q31_t)0xA21633CD, (q31_t)0xA906EB81, (q31_t)0xA2386283,
	(q31_t)0xA8E21106, (q31_t)0xA25A9FB1, (q31_t)0xA8BD43FA,
	(q31_t)0xA27CEB4F, (q31_t)0xA8988463, (q31_t)0xA29F4559,
	(q31_t)0xA873D246, (q31_t)0xA2C1ADC9, (q31_t)0xA84F2DA9,
	(q31_t)0xA2E4249A, (q31_t)0xA82A9693, (q31_t)0xA306A9C7,
	(q31_t)0xA8060D08, (q31_t)0xA3293D4B, (q31_t)0xA7E1910E,
	(q31_t)0xA34BDF20, (q31_t)0xA7BD22AB, (q31_t)0xA36E8F40,
	(q31_t)0xA798C1E4, (q31_t)0xA3914DA7, (q31_t)0xA7746EC0,
	(q31_t)0xA3B41A4F, (q31_t)0xA7502943, (q31_t)0xA3D6F533,
	(q31_t)0xA72BF173, (q31_t)0xA3F9DE4D, (q31_t)0xA707C756,
	(q31_t)0xA41CD598, (q31_t)0xA6E3AAF2, (q31_t)0xA43FDB0F,
	(q31_t)0xA6BF9C4B, (q31_t)0xA462EEAC, (q31_t)0xA69B9B68,
	(q31_t)0xA4861069, (q31_t)0xA677A84E, (q31_t)0xA4A94042,
	(q31_t)0xA653C302, (q31_t)0xA4CC7E31, (q31_t)0xA62FEB8B,
	(q31_t)0xA4EFCA31, (q31_t)0xA60C21ED, (q31_t)0xA513243B,
	(q31_t)0xA5E8662F, (q31_t)0xA5368C4B, (q31_t)0xA5C4B855,
	(q31_t)0xA55A025B, (q31_t)0xA5A11865, (q31_t)0xA57D8666,
	(q31_t)0xA57D8666, (q31_t)0xA5A11865, (q31_t)0xA55A025B,
	(q31_t)0xA5C4B855, (q31_t)0xA5368C4B, (q31_t)0xA5E8662F,
	(q31_t)0xA513243B, (q31_t)0xA60C21ED, (q31_t)0xA4EFCA31,
	(q31_t)0xA62FEB8B, (q31_t)0xA4CC7E31, (q31_t)0xA653C302,
	(q31_t)0xA4A94042, (q31_t)0xA677A84E, (q31_t)0xA4861069,
	(q31_t)0xA69B9B68, (q31_t)0xA462EEAC, (q31_t)0xA6BF9C4B,
	(q31_t)0xA43FDB0F, (q31_t)0xA6E3AAF2, (q31_t)0xA41CD598,
	(q31_t)0xA707C756, (q31_t)0xA3F9DE4D, (q31_t)0xA72BF173,
	(q31_t)0xA3D6F533, (q31_t)0xA7502943, (q31_t)0xA3B41A4F,
	(q31_t)0xA7746EC0, (q31_t)0xA3914DA7, (q31_t)0xA798C1E4,
	(q31_t)0xA36E8F40, (q31_t)0xA7BD22AB, (q31_t)0xA34BDF20,
	(q31_t)0xA7E1910E, (q31_t)0xA3293D4B, (q31_t)0xA8060D08,
	(q31_t)0xA306A9C7, (q31_t)0xA82A9693, (q31_t)0xA2E4249A,
	(q31_t)0xA84F2DA9, (q31_t)0xA2C1ADC9, (q31_t)0xA873D246,
	(q31_t)0xA29F4559, (q31_t)0xA8988463, (q31_t)0xA27CEB4F,
	(q31_t)0xA8BD43FA, (q31_t)0xA25A9FB1, (q31_t)0xA8E21106,
	(q31_t)0xA2386283, (q31_t)0xA906EB81, (q31_t)0xA21633CD,
	(q31_t)0xA92BD366, (q31_t)0xA1F41391, (q31_t)0xA950C8AF,
	(q31_t)0xA1D201D7, (q31_t)0xA975CB56, (q31_t)0xA1AFFEA2,
	(q31_t)0xA99ADB56, (q31_t)0xA18E09F9, (q31_t)0xA9BFF8A8,
	(q31_t)0xA16C23E1, (q31_t)0xA9E52347, (q31_t)0xA14A4C5E,
	(q31_t)0xAA0A5B2D, (q31_t)0xA1288376, (q31_t)0xAA2FA055,
	(q31_t)0xA106C92E, (q31_t)0xAA54F2B9, (q31_t)0xA0E51D8C,
	(q31_t)0xAA7A5253, (q31_t)0xA0C38094, (q31_t)0xAA9FBF1D,
	(q31_t)0xA0A1F24C, (q31_t)0xAAC53912, (q31_t)0xA08072BA,
	(q31_t)0xAAEAC02B, (q31_t)0xA05F01E1, (q31_t)0xAB105464,
	(q31_t)0xA03D9FC7, (q31_t)0xAB35F5B5, (q31_t)0xA01C4C72,
	(q31_t)0xAB5BA41A, (q31_t)0x9FFB07E7, (q31_t)0xAB815F8C,
	(q31_t)0x9FD9D22A, (q31_t)0xABA72806, (q31_t)0x9FB8AB41,
	(q31_t)0xABCCFD82, (q31_t)0x9F979331, (q31_t)0xABF2DFFA,
	(q31_t)0x9F7689FF, (q31_t)0xAC18CF68, (q31_t)0x9F558FB0,
	(q31_t)0xAC3ECBC7, (q31_t)0x9F34A449, (q31_t)0xAC64D510,
	(q31_t)0x9F13C7D0, (q31_t)0xAC8AEB3E, (q31_t)0x9EF2FA48,
	(q31_t)0xACB10E4A, (q31_t)0x9ED23BB9, (q31_t)0xACD73E30,
	(q31_t)0x9EB18C26, (q31_t)0xACFD7AE8, (q31_t)0x9E90EB94,
	(q31_t)0xAD23C46D, (q31_t)0x9E705A09, (q31_t)0xAD4A1ABA,
	(q31_t)0x9E4FD789, (q31_t)0xAD707DC8, (q31_t)0x9E2F641A,
	(q31_t)0xAD96ED91, (q31_t)0x9E0EFFC1, (q31_t)0xADBD6A10,
	(q31_t)0x9DEEAA82, (q31_t)0xADE3F33E, (q31_t)0x9DCE6462,
	(q31_t)0xAE0A8916, (q31_t)0x9DAE2D68, (q31_t)0xAE312B91,
	(q31_t)0x9D8E0596, (q31_t)0xAE57DAAA, (q31_t)0x9D6DECF4,
	(q31_t)0xAE7E965B, (q31_t)0x9D4DE384, (q31_t)0xAEA55E9D,
	(q31_t)0x9D2DE94D, (q31_t)0xAECC336B, (q31_t)0x9D0DFE53,
	(q31_t)0xAEF314BF, (q31_t)0x9CEE229C, (q31_t)0xAF1A0293,
	(q31_t)0x9CCE562B, (q31_t)0xAF40FCE0, (q31_t)0x9CAE9907,
	(q31_t)0xAF6803A1, (q31_t)0x9C8EEB33, (q31_t)0xAF8F16D0,
	(q31_t)0x9C6F4CB5, (q31_t)0xAFB63667, (q31_t)0x9C4FBD92,
	(q31_t)0xAFDD625F, (q31_t)0x9C303DCF, (q31_t)0xB0049AB2,
	(q31_t)0x9C10CD70, (q31_t)0xB02BDF5C, (q31_t)0x9BF16C7A,
	(q31_t)0xB0533055, (q31_t)0x9BD21AF2, (q31_t)0xB07A8D97,
	(q31_t)0x9BB2D8DD, (q31_t)0xB0A1F71C, (q31_t)0x9B93A640,
	(q31_t)0xB0C96CDF, (q31_t)0x9B748320, (q31_t)0xB0F0EEDA,
	(q31_t)0x9B556F80, (q31_t)0xB1187D05, (q31_t)0x9B366B67,
	(q31_t)0xB140175B, (q31_t)0x9B1776D9, (q31_t)0xB167BDD6,
	(q31_t)0x9AF891DB, (q31_t)0xB18F7070, (q31_t)0x9AD9BC71,
	(q31_t)0xB1B72F23, (q31_t)0x9ABAF6A0, (q31_t)0xB1DEF9E8,
	(q31_t)0x9A9C406D, (q31_t)0xB206D0BA, (q31_t)0x9A7D99DD,
	(q31_t)0xB22EB392, (q31_t)0x9A5F02F5, (q31_t)0xB256A26A,
	(q31_t)0x9A407BB8, (q31_t)0xB27E9D3B, (q31_t)0x9A22042C,
	(q31_t)0xB2A6A401, (q31_t)0x9A039C56, (q31_t)0xB2CEB6B5,
	(q31_t)0x99E5443A, (q31_t)0xB2F6D54F, (q31_t)0x99C6FBDE,
	(q31_t)0xB31EFFCB, (q31_t)0x99A8C344, (q31_t)0xB3473622,
	(q31_t)0x998A9A73, (q31_t)0xB36F784E, (q31_t)0x996C816F,
	(q31_t)0xB397C649, (q31_t)0x994E783C, (q31_t)0xB3C0200C,
	(q31_t)0x99307EE0, (q31_t)0xB3E88591, (q31_t)0x9912955E,
	(q31_t)0xB410F6D2, (q31_t)0x98F4BBBC, (q31_t)0xB43973C9,
	(q31_t)0x98D6F1FE, (q31_t)0xB461FC70, (q31_t)0x98B93828,
	(q31_t)0xB48A90C0, (q31_t)0x989B8E3F, (q31_t)0xB4B330B2,
	(q31_t)0x987DF449, (q31_t)0xB4DBDC42, (q31_t)0x98606A48,
	(q31_t)0xB5049368, (q31_t)0x9842F043, (q31_t)0xB52D561E,
	(q31_t)0x9825863D, (q31_t)0xB556245E, (q31_t)0x98082C3B,
	(q31_t)0xB57EFE21, (q31_t)0x97EAE241, (q31_t)0xB5A7E362,
	(q31_t)0x97CDA855, (q31_t)0xB5D0D41A, (q31_t)0x97B07E7A,
	(q31_t)0xB5F9D042, (q31_t)0x979364B5, (q31_t)0xB622D7D5,
	(q31_t)0x97765B0A, (q31_t)0xB64BEACC, (q31_t)0x9759617E,
	(q31_t)0xB6750921, (q31_t)0x973C7816, (q31_t)0xB69E32CD,
	(q31_t)0x971F9ED6, (q31_t)0xB6C767CA, (q31_t)0x9702D5C2,
	(q31_t)0xB6F0A811, (q31_t)0x96E61CDF, (q31_t)0xB719F39D,
	(q31_t)0x96C97431, (q31_t)0xB7434A67, (q31_t)0x96ACDBBD,
	(q31_t)0xB76CAC68, (q31_t)0x96905387, (q31_t)0xB796199B,
	(q31_t)0x9673DB94, (q31_t)0xB7BF91F8, (q31_t)0x965773E7,
	(q31_t)0xB7E9157A, (q31_t)0x963B1C85, (q31_t)0xB812A419,
	(q31_t)0x961ED573, (q31_t)0xB83C3DD1, (q31_t)0x96029EB5,
	(q31_t)0xB865E299, (q31_t)0x95E6784F, (q31_t)0xB88F926C,
	(q31_t)0x95CA6246, (q31_t)0xB8B94D44, (q31_t)0x95AE5C9E,
	(q31_t)0xB8E31319, (q31_t)0x9592675B, (q31_t)0xB90CE3E6,
	(q31_t)0x95768282, (q31_t)0xB936BFA3, (q31_t)0x955AAE17,
	(q31_t)0xB960A64B, (q31_t)0x953EEA1E, (q31_t)0xB98A97D8,
	(q31_t)0x9523369B, (q31_t)0xB9B49442, (q31_t)0x95079393,
	(q31_t)0xB9DE9B83, (q31_t)0x94EC010B, (q31_t)0xBA08AD94,
	(q31_t)0x94D07F05, (q31_t)0xBA32CA70, (q31_t)0x94B50D87,
	(q31_t)0xBA5CF210, (q31_t)0x9499AC95, (q31_t)0xBA87246C,
	(q31_t)0x947E5C32, (q31_t)0xBAB1617F, (q31_t)0x94631C64,
	(q31_t)0xBADBA943, (q31_t)0x9447ED2F, (q31_t)0xBB05FBB0,
	(q31_t)0x942CCE95, (q31_t)0xBB3058C0, (q31_t)0x9411C09D,
	(q31_t)0xBB5AC06C, (q31_t)0x93F6C34A, (q31_t)0xBB8532AF,
	(q31_t)0x93DBD69F, (q31_t)0xBBAFAF81, (q31_t)0x93C0FAA2,
	(q31_t)0xBBDA36DC, (q31_t)0x93A62F56, (q31_t)0xBC04C8BA,
	(q31_t)0x938B74C0, (q31_t)0xBC2F6513, (q31_t)0x9370CAE4,
	(q31_t)0xBC5A0BE1, (q31_t)0x935631C5, (q31_t)0xBC84BD1E,
	(q31_t)0x933BA968, (q31_t)0xBCAF78C3, (q31_t)0x932131D1,
	(q31_t)0xBCDA3ECA, (q31_t)0x9306CB04, (q31_t)0xBD050F2C,
	(q31_t)0x92EC7505, (q31_t)0xBD2FE9E1, (q31_t)0x92D22FD8,
	(q31_t)0xBD5ACEE5, (q31_t)0x92B7FB82, (q31_t)0xBD85BE2F,
	(q31_t)0x929DD805, (q31_t)0xBDB0B7BA, (q31_t)0x9283C567,
	(q31_t)0xBDDBBB7F, (q31_t)0x9269C3AC, (q31_t)0xBE06C977,
	(q31_t)0x924FD2D6, (q31_t)0xBE31E19B, (q31_t)0x9235F2EB,
	(q31_t)0xBE5D03E5, (q31_t)0x921C23EE, (q31_t)0xBE88304F,
	(q31_t)0x920265E4, (q31_t)0xBEB366D1, (q31_t)0x91E8B8D0,
	(q31_t)0xBEDEA765, (q31_t)0x91CF1CB6, (q31_t)0xBF09F204,
	(q31_t)0x91B5919A, (q31_t)0xBF3546A8, (q31_t)0x919C1780,
	(q31_t)0xBF60A54A, (q31_t)0x9182AE6C, (q31_t)0xBF8C0DE2,
	(q31_t)0x91695663, (q31_t)0xBFB7806C, (q31_t)0x91500F67,
	(q31_t)0xBFE2FCDF, (q31_t)0x9136D97D, (q31_t)0xC00E8335,
	(q31_t)0x911DB4A8, (q31_t)0xC03A1368, (q31_t)0x9104A0ED,
	(q31_t)0xC065AD70, (q31_t)0x90EB9E50, (q31_t)0xC0915147,
	(q31_t)0x90D2ACD3, (q31_t)0xC0BCFEE7, (q31_t)0x90B9CC7C,
	(q31_t)0xC0E8B648, (q31_t)0x90A0FD4E, (q31_t)0xC1147763,
	(q31_t)0x90883F4C, (q31_t)0xC1404233, (q31_t)0x906F927B,
	(q31_t)0xC16C16B0, (q31_t)0x9056F6DF, (q31_t)0xC197F4D3,
	(q31_t)0x903E6C7A, (q31_t)0xC1C3DC96, (q31_t)0x9025F352,
	(q31_t)0xC1EFCDF2, (q31_t)0x900D8B69, (q31_t)0xC21BC8E0,
	(q31_t)0x8FF534C4, (q31_t)0xC247CD5A, (q31_t)0x8FDCEF66,
	(q31_t)0xC273DB58, (q31_t)0x8FC4BB53, (q31_t)0xC29FF2D4,
	(q31_t)0x8FAC988E, (q31_t)0xC2CC13C7, (q31_t)0x8F94871D,
	(q31_t)0xC2F83E2A, (q31_t)0x8F7C8701, (q31_t)0xC32471F6,
	(q31_t)0x8F64983F, (q31_t)0xC350AF25, (q31_t)0x8F4CBADB,
	(q31_t)0xC37CF5B0, (q31_t)0x8F34EED8, (q31_t)0xC3A9458F,
	(q31_t)0x8F1D343A, (q31_t)0xC3D59EBD, (q31_t)0x8F058B04,
	(q31_t)0xC4020132, (q31_t)0x8EEDF33B, (q31_t)0xC42E6CE8,
	(q31_t)0x8ED66CE1, (q31_t)0xC45AE1D7, (q31_t)0x8EBEF7FB,
	(q31_t)0xC4875FF8, (q31_t)0x8EA7948C, (q31_t)0xC4B3E746,
	(q31_t)0x8E904298, (q31_t)0xC4E077B8, (q31_t)0x8E790222,
	(q31_t)0xC50D1148, (q31_t)0x8E61D32D, (q31_t)0xC539B3F0,
	(q31_t)0x8E4AB5BF, (q31_t)0xC5665FA8, (q31_t)0x8E33A9D9,
	(q31_t)0xC593146A, (q31_t)0x8E1CAF80, (q31_t)0xC5BFD22E,
	(q31_t)0x8E05C6B7, (q31_t)0xC5EC98ED, (q31_t)0x8DEEEF82,
	(q31_t)0xC61968A2, (q31_t)0x8DD829E4, (q31_t)0xC6464144,
	(q31_t)0x8DC175E0, (q31_t)0xC67322CD, (q31_t)0x8DAAD37B,
	(q31_t)0xC6A00D36, (q31_t)0x8D9442B7, (q31_t)0xC6CD0079,
	(q31_t)0x8D7DC399, (q31_t)0xC6F9FC8D, (q31_t)0x8D675623,
	(q31_t)0xC727016C, (q31_t)0x8D50FA59, (q31_t)0xC7540F10,
	(q31_t)0x8D3AB03F, (q31_t)0xC7812571, (q31_t)0x8D2477D8,
	(q31_t)0xC7AE4489, (q31_t)0x8D0E5127, (q31_t)0xC7DB6C50,
	(q31_t)0x8CF83C30, (q31_t)0xC8089CBF, (q31_t)0x8CE238F6,
	(q31_t)0xC835D5D0, (q31_t)0x8CCC477D, (q31_t)0xC863177B,
	(q31_t)0x8CB667C7, (q31_t)0xC89061BA, (q31_t)0x8CA099D9,
	(q31_t)0xC8BDB485, (q31_t)0x8C8ADDB6, (q31_t)0xC8EB0FD6,
	(q31_t)0x8C753361, (q31_t)0xC91873A5, (q31_t)0x8C5F9ADD,
	(q31_t)0xC945DFEC, (q31_t)0x8C4A142F, (q31_t)0xC97354A3,
	(q31_t)0x8C349F58, (q31_t)0xC9A0D1C4, (q31_t)0x8C1F3C5C,
	(q31_t)0xC9CE5748, (q31_t)0x8C09EB40, (q31_t)0xC9FBE527,
	(q31_t)0x8BF4AC05, (q31_t)0xCA297B5A, (q31_t)0x8BDF7EAF,
	(q31_t)0xCA5719DB, (q31_t)0x8BCA6342, (q31_t)0xCA84C0A2,
	(q31_t)0x8BB559C1, (q31_t)0xCAB26FA9, (q31_t)0x8BA0622F,
	(q31_t)0xCAE026E8, (q31_t)0x8B8B7C8F, (q31_t)0xCB0DE658,
	(q31_t)0x8B76A8E4, (q31_t)0xCB3BADF2, (q31_t)0x8B61E732,
	(q31_t)0xCB697DB0, (q31_t)0x8B4D377C, (q31_t)0xCB975589,
	(q31_t)0x8B3899C5, (q31_t)0xCBC53578, (q31_t)0x8B240E10,
	(q31_t)0xCBF31D75, (q31_t)0x8B0F9461, (q31_t)0xCC210D78,
	(q31_t)0x8AFB2CBA, (q31_t)0xCC4F057B, (q31_t)0x8AE6D71F,
	(q31_t)0xCC7D0577, (q31_t)0x8AD29393, (q31_t)0xCCAB0D65,
	(q31_t)0x8ABE6219, (q31_t)0xCCD91D3D, (q31_t)0x8AAA42B4,
	(q31_t)0xCD0734F8, (q31_t)0x8A963567, (q31_t)0xCD355490,
	(q31_t)0x8A823A35, (q31_t)0xCD637BFD, (q31_t)0x8A6E5122,
	(q31_t)0xCD91AB38, (q31_t)0x8A5A7A30, (q31_t)0xCDBFE23A,
	(q31_t)0x8A46B563, (q31_t)0xCDEE20FC, (q31_t)0x8A3302BD,
	(q31_t)0xCE1C6776, (q31_t)0x8A1F6242, (q31_t)0xCE4AB5A2,
	(q31_t)0x8A0BD3F5, (q31_t)0xCE790B78, (q31_t)0x89F857D8,
	(q31_t)0xCEA768F2, (q31_t)0x89E4EDEE, (q31_t)0xCED5CE08,
	(q31_t)0x89D1963C, (q31_t)0xCF043AB2, (q31_t)0x89BE50C3,
	(q31_t)0xCF32AEEB, (q31_t)0x89AB1D86, (q31_t)0xCF612AAA,
	(q31_t)0x8997FC89, (q31_t)0xCF8FADE8, (q31_t)0x8984EDCF,
	(q31_t)0xCFBE389F, (q31_t)0x8971F15A, (q31_t)0xCFECCAC7,
	(q31_t)0x895F072D, (q31_t)0xD01B6459, (q31_t)0x894C2F4C,
	(q31_t)0xD04A054D, (q31_t)0x893969B9, (q31_t)0xD078AD9D,
	(q31_t)0x8926B677, (q31_t)0xD0A75D42, (q31_t)0x89141589,
	(q31_t)0xD0D61433, (q31_t)0x890186F1, (q31_t)0xD104D26B,
	(q31_t)0x88EF0AB4, (q31_t)0xD13397E1, (q31_t)0x88DCA0D3,
	(q31_t)0xD162648F, (q31_t)0x88CA4951, (q31_t)0xD191386D,
	(q31_t)0x88B80431, (q31_t)0xD1C01374, (q31_t)0x88A5D177,
	(q31_t)0xD1EEF59E, (q31_t)0x8893B124, (q31_t)0xD21DDEE1,
	(q31_t)0x8881A33C, (q31_t)0xD24CCF38, (q31_t)0x886FA7C2,
	(q31_t)0xD27BC69C, (q31_t)0x885DBEB7, (q31_t)0xD2AAC504,
	(q31_t)0x884BE820, (q31_t)0xD2D9CA6A, (q31_t)0x883A23FE,
	(q31_t)0xD308D6C6, (q31_t)0x88287255, (q31_t)0xD337EA12,
	(q31_t)0x8816D327, (q31_t)0xD3670445, (q31_t)0x88054677,
	(q31_t)0xD3962559, (q31_t)0x87F3CC47, (q31_t)0xD3C54D46,
	(q31_t)0x87E2649B, (q31_t)0xD3F47C06, (q31_t)0x87D10F75,
	(q31_t)0xD423B190, (q31_t)0x87BFCCD7, (q31_t)0xD452EDDE,
	(q31_t)0x87AE9CC5, (q31_t)0xD48230E8, (q31_t)0x879D7F40,
	(q31_t)0xD4B17AA7, (q31_t)0x878C744C, (q31_t)0xD4E0CB14,
	(q31_t)0x877B7BEC, (q31_t)0xD5102227, (q31_t)0x876A9621,
	(q31_t)0xD53F7FDA, (q31_t)0x8759C2EF, (q31_t)0xD56EE424,
	(q31_t)0x87490257, (q31_t)0xD59E4EFE, (q31_t)0x8738545E,
	(q31_t)0xD5CDC062, (q31_t)0x8727B904, (q31_t)0xD5FD3847,
	(q31_t)0x8717304E, (q31_t)0xD62CB6A7, (q31_t)0x8706BA3C,
	(q31_t)0xD65C3B7B, (q31_t)0x86F656D3, (q31_t)0xD68BC6BA,
	(q31_t)0x86E60614, (q31_t)0xD6BB585D, (q31_t)0x86D5C802,
	(q31_t)0xD6EAF05E, (q31_t)0x86C59C9F, (q31_t)0xD71A8EB5,
	(q31_t)0x86B583EE, (q31_t)0xD74A335A, (q31_t)0x86A57DF1,
	(q31_t)0xD779DE46, (q31_t)0x86958AAB, (q31_t)0xD7A98F73,
	(q31_t)0x8685AA1F, (q31_t)0xD7D946D7, (q31_t)0x8675DC4E,
	(q31_t)0xD809046D, (q31_t)0x8666213C, (q31_t)0xD838C82D,
	(q31_t)0x865678EA, (q31_t)0xD868920F, (q31_t)0x8646E35B,
	(q31_t)0xD898620C, (q31_t)0x86376092, (q31_t)0xD8C8381C,
	(q31_t)0x8627F090, (q31_t)0xD8F81439, (q31_t)0x86189359,
	(q31_t)0xD927F65B, (q31_t)0x860948EE, (q31_t)0xD957DE7A,
	(q31_t)0x85FA1152, (q31_t)0xD987CC8F, (q31_t)0x85EAEC88,
	(q31_t)0xD9B7C093, (q31_t)0x85DBDA91, (q31_t)0xD9E7BA7E,
	(q31_t)0x85CCDB70, (q31_t)0xDA17BA4A, (q31_t)0x85BDEF27,
	(q31_t)0xDA47BFED, (q31_t)0x85AF15B9, (q31_t)0xDA77CB62,
	(q31_t)0x85A04F28, (q31_t)0xDAA7DCA1, (q31_t)0x85919B75,
	(q31_t)0xDAD7F3A2, (q31_t)0x8582FAA4, (q31_t)0xDB08105E,
	(q31_t)0x85746CB7, (q31_t)0xDB3832CD, (q31_t)0x8565F1B0,
	(q31_t)0xDB685AE8, (q31_t)0x85578991, (q31_t)0xDB9888A8,
	(q31_t)0x8549345C, (q31_t)0xDBC8BC05, (q31_t)0x853AF214,
	(q31_t)0xDBF8F4F8, (q31_t)0x852CC2BA, (q31_t)0xDC293379,
	(q31_t)0x851EA652, (q31_t)0xDC597781, (q31_t)0x85109CDC,
	(q31_t)0xDC89C108, (q31_t)0x8502A65C, (q31_t)0xDCBA1008,
	(q31_t)0x84F4C2D3, (q31_t)0xDCEA6478, (q31_t)0x84E6F244,
	(q31_t)0xDD1ABE51, (q31_t)0x84D934B0, (q31_t)0xDD4B1D8B,
	(q31_t)0x84CB8A1B, (q31_t)0xDD7B8220, (q31_t)0x84BDF285,
	(q31_t)0xDDABEC07, (q31_t)0x84B06DF1, (q31_t)0xDDDC5B3A,
	(q31_t)0x84A2FC62, (q31_t)0xDE0CCFB1, (q31_t)0x84959DD9,
	(q31_t)0xDE3D4963, (q31_t)0x84885257, (q31_t)0xDE6DC84B,
	(q31_t)0x847B19E1, (q31_t)0xDE9E4C60, (q31_t)0x846DF476,
	(q31_t)0xDECED59B, (q31_t)0x8460E21A, (q31_t)0xDEFF63F4,
	(q31_t)0x8453E2CE, (q31_t)0xDF2FF764, (q31_t)0x8446F695,
	(q31_t)0xDF608FE3, (q31_t)0x843A1D70, (q31_t)0xDF912D6A,
	(q31_t)0x842D5761, (q31_t)0xDFC1CFF2, (q31_t)0x8420A46B,
	(q31_t)0xDFF27773, (q31_t)0x8414048F, (q31_t)0xE02323E5,
	(q31_t)0x840777CF, (q31_t)0xE053D541, (q31_t)0x83FAFE2E,
	(q31_t)0xE0848B7F, (q31_t)0x83EE97AC, (q31_t)0xE0B54698,
	(q31_t)0x83E2444D, (q31_t)0xE0E60684, (q31_t)0x83D60411,
	(q31_t)0xE116CB3D, (q31_t)0x83C9D6FB, (q31_t)0xE14794B9,
	(q31_t)0x83BDBD0D, (q31_t)0xE17862F3, (q31_t)0x83B1B649,
	(q31_t)0xE1A935E1, (q31_t)0x83A5C2B0, (q31_t)0xE1DA0D7E,
	(q31_t)0x8399E244, (q31_t)0xE20AE9C1, (q31_t)0x838E1507,
	(q31_t)0xE23BCAA2, (q31_t)0x83825AFB, (q31_t)0xE26CB01A,
	(q31_t)0x8376B422, (q31_t)0xE29D9A22, (q31_t)0x836B207D,
	(q31_t)0xE2CE88B2, (q31_t)0x835FA00E, (q31_t)0xE2FF7BC3,
	(q31_t)0x835432D8, (q31_t)0xE330734C, (q31_t)0x8348D8DB,
	(q31_t)0xE3616F47, (q31_t)0x833D921A, (q31_t)0xE3926FAC,
	(q31_t)0x83325E97, (q31_t)0xE3C37473, (q31_t)0x83273E52,
	(q31_t)0xE3F47D95, (q31_t)0x831C314E, (q31_t)0xE4258B0A,
	(q31_t)0x8311378C, (q31_t)0xE4569CCB, (q31_t)0x8306510F,
	(q31_t)0xE487B2CF, (q31_t)0x82FB7DD8, (q31_t)0xE4B8CD10,
	(q31_t)0x82F0BDE8, (q31_t)0xE4E9EB86, (q31_t)0x82E61141,
	(q31_t)0xE51B0E2A, (q31_t)0x82DB77E5, (q31_t)0xE54C34F3,
	(q31_t)0x82D0F1D5, (q31_t)0xE57D5FDA, (q31_t)0x82C67F13,
	(q31_t)0xE5AE8ED8, (q31_t)0x82BC1FA1, (q31_t)0xE5DFC1E4,
	(q31_t)0x82B1D381, (q31_t)0xE610F8F9, (q31_t)0x82A79AB3,
	(q31_t)0xE642340D, (q31_t)0x829D753A, (q31_t)0xE6737319,
	(q31_t)0x82936316, (q31_t)0xE6A4B616, (q31_t)0x8289644A,
	(q31_t)0xE6D5FCFC, (q31_t)0x827F78D8, (q31_t)0xE70747C3,
	(q31_t)0x8275A0C0, (q31_t)0xE7389664, (q31_t)0x826BDC04,
	(q31_t)0xE769E8D8, (q31_t)0x82622AA5, (q31_t)0xE79B3F16,
	(q31_t)0x82588CA6, (q31_t)0xE7CC9917, (q31_t)0x824F0208,
	(q31_t)0xE7FDF6D3, (q31_t)0x82458ACB, (q31_t)0xE82F5844,
	(q31_t)0x823C26F2, (q31_t)0xE860BD60, (q31_t)0x8232D67E,
	(q31_t)0xE8922621, (q31_t)0x82299971, (q31_t)0xE8C3927F,
	(q31_t)0x82206FCB, (q31_t)0xE8F50273, (q31_t)0x8217598F,
	(q31_t)0xE92675F4, (q31_t)0x820E56BE, (q31_t)0xE957ECFB,
	(q31_t)0x82056758, (q31_t)0xE9896780, (q31_t)0x81FC8B60,
	(q31_t)0xE9BAE57C, (q31_t)0x81F3C2D7, (q31_t)0xE9EC66E8,
	(q31_t)0x81EB0DBD, (q31_t)0xEA1DEBBB, (q31_t)0x81E26C16,
	(q31_t)0xEA4F73EE, (q31_t)0x81D9DDE1, (q31_t)0xEA80FF79,
	(q31_t)0x81D16320, (q31_t)0xEAB28E55, (q31_t)0x81C8FBD5,
	(q31_t)0xEAE4207A, (q31_t)0x81C0A801, (q31_t)0xEB15B5E0,
	(q31_t)0x81B867A4, (q31_t)0xEB474E80, (q31_t)0x81B03AC1,
	(q31_t)0xEB78EA52, (q31_t)0x81A82159, (q31_t)0xEBAA894E,
	(q31_t)0x81A01B6C, (q31_t)0xEBDC2B6D, (q31_t)0x819828FD,
	(q31_t)0xEC0DD0A8, (q31_t)0x81904A0C, (q31_t)0xEC3F78F5,
	(q31_t)0x81887E9A, (q31_t)0xEC71244F, (q31_t)0x8180C6A9,
	(q31_t)0xECA2D2AC, (q31_t)0x8179223A, (q31_t)0xECD48406,
	(q31_t)0x8171914E, (q31_t)0xED063855, (q31_t)0x816A13E6,
	(q31_t)0xED37EF91, (q31_t)0x8162AA03, (q31_t)0xED69A9B2,
	(q31_t)0x815B53A8, (q31_t)0xED9B66B2, (q31_t)0x815410D3,
	(q31_t)0xEDCD2687, (q31_t)0x814CE188, (q31_t)0xEDFEE92B,
	(q31_t)0x8145C5C6, (q31_t)0xEE30AE95, (q31_t)0x813EBD90,
	(q31_t)0xEE6276BF, (q31_t)0x8137C8E6, (q31_t)0xEE9441A0,
	(q31_t)0x8130E7C8, (q31_t)0xEEC60F31, (q31_t)0x812A1A39,
	(q31_t)0xEEF7DF6A, (q31_t)0x81236039, (q31_t)0xEF29B243,
	(q31_t)0x811CB9CA, (q31_t)0xEF5B87B5, (q31_t)0x811626EC,
	(q31_t)0xEF8D5FB8, (q31_t)0x810FA7A0, (q31_t)0xEFBF3A44,
	(q31_t)0x81093BE8, (q31_t)0xEFF11752, (q31_t)0x8102E3C3,
	(q31_t)0xF022F6DA, (q31_t)0x80FC9F35, (q31_t)0xF054D8D4,
	(q31_t)0x80F66E3C, (q31_t)0xF086BD39, (q31_t)0x80F050DB,
	(q31_t)0xF0B8A401, (q31_t)0x80EA4712, (q31_t)0xF0EA8D23,
	(q31_t)0x80E450E2, (q31_t)0xF11C789A, (q31_t)0x80DE6E4C,
	(q31_t)0xF14E665C, (q31_t)0x80D89F51, (q31_t)0xF1805662,
	(q31_t)0x80D2E3F1, (q31_t)0xF1B248A5, (q31_t)0x80CD3C2F,
	(q31_t)0xF1E43D1C, (q31_t)0x80C7A80A, (q31_t)0xF21633C0,
	(q31_t)0x80C22783, (q31_t)0xF2482C89, (q31_t)0x80BCBA9C,
	(q31_t)0xF27A2770, (q31_t)0x80B76155, (q31_t)0xF2AC246D,
	(q31_t)0x80B21BAF, (q31_t)0xF2DE2378, (q31_t)0x80ACE9AB,
	(q31_t)0xF310248A, (q31_t)0x80A7CB49, (q31_t)0xF342279A,
	(q31_t)0x80A2C08B, (q31_t)0xF3742CA1, (q31_t)0x809DC970,
	(q31_t)0xF3A63398, (q31_t)0x8098E5FB, (q31_t)0xF3D83C76,
	(q31_t)0x8094162B, (q31_t)0xF40A4734, (q31_t)0x808F5A02,
	(q31_t)0xF43C53CA, (q31_t)0x808AB180, (q31_t)0xF46E6231,
	(q31_t)0x80861CA5, (q31_t)0xF4A07260, (q31_t)0x80819B74,
	(q31_t)0xF4D28451, (q31_t)0x807D2DEB, (q31_t)0xF50497FA,
	(q31_t)0x8078D40D, (q31_t)0xF536AD55, (q31_t)0x80748DD9,
	(q31_t)0xF568C45A, (q31_t)0x80705B50, (q31_t)0xF59ADD01,
	(q31_t)0x806C3C73, (q31_t)0xF5CCF743, (q31_t)0x80683143,
	(q31_t)0xF5FF1317, (q31_t)0x806439C0, (q31_t)0xF6313076,
	(q31_t)0x806055EA, (q31_t)0xF6634F58, (q31_t)0x805C85C3,
	(q31_t)0xF6956FB6, (q31_t)0x8058C94C, (q31_t)0xF6C79188,
	(q31_t)0x80552083, (q31_t)0xF6F9B4C5, (q31_t)0x80518B6B,
	(q31_t)0xF72BD967, (q31_t)0x804E0A03, (q31_t)0xF75DFF65,
	(q31_t)0x804A9C4D, (q31_t)0xF79026B8, (q31_t)0x80474248,
	(q31_t)0xF7C24F58, (q31_t)0x8043FBF6, (q31_t)0xF7F4793E,
	(q31_t)0x8040C956, (q31_t)0xF826A461, (q31_t)0x803DAA69,
	(q31_t)0xF858D0BA, (q31_t)0x803A9F31, (q31_t)0xF88AFE41,
	(q31_t)0x8037A7AC, (q31_t)0xF8BD2CEF, (q31_t)0x8034C3DC,
	(q31_t)0xF8EF5CBB, (q31_t)0x8031F3C1, (q31_t)0xF9218D9E,
	(q31_t)0x802F375C, (q31_t)0xF953BF90, (q31_t)0x802C8EAD,
	(q31_t)0xF985F28A, (q31_t)0x8029F9B4, (q31_t)0xF9B82683,
	(q31_t)0x80277872, (q31_t)0xF9EA5B75, (q31_t)0x80250AE7,
	(q31_t)0xFA1C9156, (q31_t)0x8022B113, (q31_t)0xFA4EC820,
	(q31_t)0x80206AF8, (q31_t)0xFA80FFCB, (q31_t)0x801E3894,
	(q31_t)0xFAB3384F, (q31_t)0x801C19E9, (q31_t)0xFAE571A4,
	(q31_t)0x801A0EF7, (q31_t)0xFB17ABC2, (q31_t)0x801817BF,
	(q31_t)0xFB49E6A2, (q31_t)0x80163440, (q31_t)0xFB7C223C,
	(q31_t)0x8014647A, (q31_t)0xFBAE5E89, (q31_t)0x8012A86F,
	(q31_t)0xFBE09B80, (q31_t)0x8011001E, (q31_t)0xFC12D919,
	(q31_t)0x800F6B88, (q31_t)0xFC45174E, (q31_t)0x800DEAAC,
	(q31_t)0xFC775616, (q31_t)0x800C7D8C, (q31_t)0xFCA99569,
	(q31_t)0x800B2427, (q31_t)0xFCDBD541, (q31_t)0x8009DE7D,
	(q31_t)0xFD0E1594, (q31_t)0x8008AC90, (q31_t)0xFD40565B,
	(q31_t)0x80078E5E, (q31_t)0xFD72978F, (q31_t)0x800683E8,
	(q31_t)0xFDA4D928, (q31_t)0x80058D2E, (q31_t)0xFDD71B1E,
	(q31_t)0x8004AA31, (q31_t)0xFE095D69, (q31_t)0x8003DAF0,
	(q31_t)0xFE3BA001, (q31_t)0x80031F6C, (q31_t)0xFE6DE2E0,
	(q31_t)0x800277A5, (q31_t)0xFEA025FC, (q31_t)0x8001E39B,
	(q31_t)0xFED2694F, (q31_t)0x8001634D, (q31_t)0xFF04ACD0,
	(q31_t)0x8000F6BD, (q31_t)0xFF36F078, (q31_t)0x80009DE9,
	(q31_t)0xFF69343E, (q31_t)0x800058D3, (q31_t)0xFF9B781D,
	(q31_t)0x8000277A, (q31_t)0xFFCDBC0A, (q31_t)0x800009DE
};



/*
* @brief  q15 Twiddle factors Table
*/


/**
* \par
* Example code for q15 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefq15[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefq15[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 16	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to q15(Fixed point 1.15):
*	round(twiddleCoefq15(i) * pow(2, 15))
*
*/
const q15_t twiddleCoef_16_q15[24] = {
    (q15_t)0x7FFF, (q15_t)0x0000,
    (q15_t)0x7641, (q15_t)0x30FB,
    (q15_t)0x5A82, (q15_t)0x5A82,
    (q15_t)0x30FB, (q15_t)0x7641,
    (q15_t)0x0000, (q15_t)0x7FFF,
    (q15_t)0xCF04, (q15_t)0x7641,
    (q15_t)0xA57D, (q15_t)0x5A82,
    (q15_t)0x89BE, (q15_t)0x30FB,
    (q15_t)0x8000, (q15_t)0x0000,
    (q15_t)0x89BE, (q15_t)0xCF04,
    (q15_t)0xA57D, (q15_t)0xA57D,
    (q15_t)0xCF04, (q15_t)0x89BE
};

/**
* \par
* Example code for q15 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefq15[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefq15[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 32	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to q15(Fixed point 1.15):
*	round(twiddleCoefq15(i) * pow(2, 15))
*
*/
const q15_t twiddleCoef_32_q15[48] = {
    (q15_t)0x7FFF, (q15_t)0x0000,
    (q15_t)0x7D8A, (q15_t)0x18F8,
    (q15_t)0x7641, (q15_t)0x30FB,
    (q15_t)0x6A6D, (q15_t)0x471C,
    (q15_t)0x5A82, (q15_t)0x5A82,
    (q15_t)0x471C, (q15_t)0x6A6D,
    (q15_t)0x30FB, (q15_t)0x7641,
    (q15_t)0x18F8, (q15_t)0x7D8A,
    (q15_t)0x0000, (q15_t)0x7FFF,
    (q15_t)0xE707, (q15_t)0x7D8A,
    (q15_t)0xCF04, (q15_t)0x7641,
    (q15_t)0xB8E3, (q15_t)0x6A6D,
    (q15_t)0xA57D, (q15_t)0x5A82,
    (q15_t)0x9592, (q15_t)0x471C,
    (q15_t)0x89BE, (q15_t)0x30FB,
    (q15_t)0x8275, (q15_t)0x18F8,
    (q15_t)0x8000, (q15_t)0x0000,
    (q15_t)0x8275, (q15_t)0xE707,
    (q15_t)0x89BE, (q15_t)0xCF04,
    (q15_t)0x9592, (q15_t)0xB8E3,
    (q15_t)0xA57D, (q15_t)0xA57D,
    (q15_t)0xB8E3, (q15_t)0x9592,
    (q15_t)0xCF04, (q15_t)0x89BE,
    (q15_t)0xE707, (q15_t)0x8275
};

/**
* \par
* Example code for q15 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefq15[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefq15[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 64	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to q15(Fixed point 1.15):
*	round(twiddleCoefq15(i) * pow(2, 15))
*
*/
const q15_t twiddleCoef_64_q15[96] = {
	(q15_t)0x7FFF, (q15_t)0x0000, (q15_t)0x7F62, (q15_t)0x0C8B,
	(q15_t)0x7D8A, (q15_t)0x18F8, (q15_t)0x7A7D, (q15_t)0x2528,
	(q15_t)0x7641, (q15_t)0x30FB, (q15_t)0x70E2, (q15_t)0x3C56,
	(q15_t)0x6A6D, (q15_t)0x471C, (q15_t)0x62F2, (q15_t)0x5133,
	(q15_t)0x5A82, (q15_t)0x5A82, (q15_t)0x5133, (q15_t)0x62F2,
	(q15_t)0x471C, (q15_t)0x6A6D, (q15_t)0x3C56, (q15_t)0x70E2,
	(q15_t)0x30FB, (q15_t)0x7641, (q15_t)0x2528, (q15_t)0x7A7D,
	(q15_t)0x18F8, (q15_t)0x7D8A, (q15_t)0x0C8B, (q15_t)0x7F62,
	(q15_t)0x0000, (q15_t)0x7FFF, (q15_t)0xF374, (q15_t)0x7F62,
	(q15_t)0xE707, (q15_t)0x7D8A, (q15_t)0xDAD7, (q15_t)0x7A7D,
	(q15_t)0xCF04, (q15_t)0x7641, (q15_t)0xC3A9, (q15_t)0x70E2,
	(q15_t)0xB8E3, (q15_t)0x6A6D, (q15_t)0xAECC, (q15_t)0x62F2,
	(q15_t)0xA57D, (q15_t)0x5A82, (q15_t)0x9D0D, (q15_t)0x5133,
	(q15_t)0x9592, (q15_t)0x471C, (q15_t)0x8F1D, (q15_t)0x3C56,
	(q15_t)0x89BE, (q15_t)0x30FB, (q15_t)0x8582, (q15_t)0x2528,
	(q15_t)0x8275, (q15_t)0x18F8, (q15_t)0x809D, (q15_t)0x0C8B,
	(q15_t)0x8000, (q15_t)0x0000, (q15_t)0x809D, (q15_t)0xF374,
	(q15_t)0x8275, (q15_t)0xE707, (q15_t)0x8582, (q15_t)0xDAD7,
	(q15_t)0x89BE, (q15_t)0xCF04, (q15_t)0x8F1D, (q15_t)0xC3A9,
	(q15_t)0x9592, (q15_t)0xB8E3, (q15_t)0x9D0D, (q15_t)0xAECC,
	(q15_t)0xA57D, (q15_t)0xA57D, (q15_t)0xAECC, (q15_t)0x9D0D,
	(q15_t)0xB8E3, (q15_t)0x9592, (q15_t)0xC3A9, (q15_t)0x8F1D,
	(q15_t)0xCF04, (q15_t)0x89BE, (q15_t)0xDAD7, (q15_t)0x8582,
	(q15_t)0xE707, (q15_t)0x8275, (q15_t)0xF374, (q15_t)0x809D
};

/**
* \par
* Example code for q15 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefq15[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefq15[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 128	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to q15(Fixed point 1.15):
*	round(twiddleCoefq15(i) * pow(2, 15))
*
*/
const q15_t twiddleCoef_128_q15[192] = {
	(q15_t)0x7FFF, (q15_t)0x0000, (q15_t)0x7FD8, (q15_t)0x0647,
	(q15_t)0x7F62, (q15_t)0x0C8B, (q15_t)0x7E9D, (q15_t)0x12C8,
	(q15_t)0x7D8A, (q15_t)0x18F8, (q15_t)0x7C29, (q15_t)0x1F19,
	(q15_t)0x7A7D, (q15_t)0x2528, (q15_t)0x7884, (q15_t)0x2B1F,
	(q15_t)0x7641, (q15_t)0x30FB, (q15_t)0x73B5, (q15_t)0x36BA,
	(q15_t)0x70E2, (q15_t)0x3C56, (q15_t)0x6DCA, (q15_t)0x41CE,
	(q15_t)0x6A6D, (q15_t)0x471C, (q15_t)0x66CF, (q15_t)0x4C3F,
	(q15_t)0x62F2, (q15_t)0x5133, (q15_t)0x5ED7, (q15_t)0x55F5,
	(q15_t)0x5A82, (q15_t)0x5A82, (q15_t)0x55F5, (q15_t)0x5ED7,
	(q15_t)0x5133, (q15_t)0x62F2, (q15_t)0x4C3F, (q15_t)0x66CF,
	(q15_t)0x471C, (q15_t)0x6A6D, (q15_t)0x41CE, (q15_t)0x6DCA,
	(q15_t)0x3C56, (q15_t)0x70E2, (q15_t)0x36BA, (q15_t)0x73B5,
	(q15_t)0x30FB, (q15_t)0x7641, (q15_t)0x2B1F, (q15_t)0x7884,
	(q15_t)0x2528, (q15_t)0x7A7D, (q15_t)0x1F19, (q15_t)0x7C29,
	(q15_t)0x18F8, (q15_t)0x7D8A, (q15_t)0x12C8, (q15_t)0x7E9D,
	(q15_t)0x0C8B, (q15_t)0x7F62, (q15_t)0x0647, (q15_t)0x7FD8,
	(q15_t)0x0000, (q15_t)0x7FFF, (q15_t)0xF9B8, (q15_t)0x7FD8,
	(q15_t)0xF374, (q15_t)0x7F62, (q15_t)0xED37, (q15_t)0x7E9D,
	(q15_t)0xE707, (q15_t)0x7D8A, (q15_t)0xE0E6, (q15_t)0x7C29,
	(q15_t)0xDAD7, (q15_t)0x7A7D, (q15_t)0xD4E0, (q15_t)0x7884,
	(q15_t)0xCF04, (q15_t)0x7641, (q15_t)0xC945, (q15_t)0x73B5,
	(q15_t)0xC3A9, (q15_t)0x70E2, (q15_t)0xBE31, (q15_t)0x6DCA,
	(q15_t)0xB8E3, (q15_t)0x6A6D, (q15_t)0xB3C0, (q15_t)0x66CF,
	(q15_t)0xAECC, (q15_t)0x62F2, (q15_t)0xAA0A, (q15_t)0x5ED7,
	(q15_t)0xA57D, (q15_t)0x5A82, (q15_t)0xA128, (q15_t)0x55F5,
	(q15_t)0x9D0D, (q15_t)0x5133, (q15_t)0x9930, (q15_t)0x4C3F,
	(q15_t)0x9592, (q15_t)0x471C, (q15_t)0x9235, (q15_t)0x41CE,
	(q15_t)0x8F1D, (q15_t)0x3C56, (q15_t)0x8C4A, (q15_t)0x36BA,
	(q15_t)0x89BE, (q15_t)0x30FB, (q15_t)0x877B, (q15_t)0x2B1F,
	(q15_t)0x8582, (q15_t)0x2528, (q15_t)0x83D6, (q15_t)0x1F19,
	(q15_t)0x8275, (q15_t)0x18F8, (q15_t)0x8162, (q15_t)0x12C8,
	(q15_t)0x809D, (q15_t)0x0C8B, (q15_t)0x8027, (q15_t)0x0647,
	(q15_t)0x8000, (q15_t)0x0000, (q15_t)0x8027, (q15_t)0xF9B8,
	(q15_t)0x809D, (q15_t)0xF374, (q15_t)0x8162, (q15_t)0xED37,
	(q15_t)0x8275, (q15_t)0xE707, (q15_t)0x83D6, (q15_t)0xE0E6,
	(q15_t)0x8582, (q15_t)0xDAD7, (q15_t)0x877B, (q15_t)0xD4E0,
	(q15_t)0x89BE, (q15_t)0xCF04, (q15_t)0x8C4A, (q15_t)0xC945,
	(q15_t)0x8F1D, (q15_t)0xC3A9, (q15_t)0x9235, (q15_t)0xBE31,
	(q15_t)0x9592, (q15_t)0xB8E3, (q15_t)0x9930, (q15_t)0xB3C0,
	(q15_t)0x9D0D, (q15_t)0xAECC, (q15_t)0xA128, (q15_t)0xAA0A,
	(q15_t)0xA57D, (q15_t)0xA57D, (q15_t)0xAA0A, (q15_t)0xA128,
	(q15_t)0xAECC, (q15_t)0x9D0D, (q15_t)0xB3C0, (q15_t)0x9930,
	(q15_t)0xB8E3, (q15_t)0x9592, (q15_t)0xBE31, (q15_t)0x9235,
	(q15_t)0xC3A9, (q15_t)0x8F1D, (q15_t)0xC945, (q15_t)0x8C4A,
	(q15_t)0xCF04, (q15_t)0x89BE, (q15_t)0xD4E0, (q15_t)0x877B,
	(q15_t)0xDAD7, (q15_t)0x8582, (q15_t)0xE0E6, (q15_t)0x83D6,
	(q15_t)0xE707, (q15_t)0x8275, (q15_t)0xED37, (q15_t)0x8162,
	(q15_t)0xF374, (q15_t)0x809D, (q15_t)0xF9B8, (q15_t)0x8027
};

/**
* \par
* Example code for q15 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefq15[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefq15[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 256	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to q15(Fixed point 1.15):
*	round(twiddleCoefq15(i) * pow(2, 15))
*
*/
const q15_t twiddleCoef_256_q15[384] = {
	(q15_t)0x7FFF, (q15_t)0x0000, (q15_t)0x7FF6, (q15_t)0x0324,
	(q15_t)0x7FD8, (q15_t)0x0647, (q15_t)0x7FA7, (q15_t)0x096A,
	(q15_t)0x7F62, (q15_t)0x0C8B, (q15_t)0x7F09, (q15_t)0x0FAB,
	(q15_t)0x7E9D, (q15_t)0x12C8, (q15_t)0x7E1D, (q15_t)0x15E2,
	(q15_t)0x7D8A, (q15_t)0x18F8, (q15_t)0x7CE3, (q15_t)0x1C0B,
	(q15_t)0x7C29, (q15_t)0x1F19, (q15_t)0x7B5D, (q15_t)0x2223,
	(q15_t)0x7A7D, (q15_t)0x2528, (q15_t)0x798A, (q15_t)0x2826,
	(q15_t)0x7884, (q15_t)0x2B1F, (q15_t)0x776C, (q15_t)0x2E11,
	(q15_t)0x7641, (q15_t)0x30FB, (q15_t)0x7504, (q15_t)0x33DE,
	(q15_t)0x73B5, (q15_t)0x36BA, (q15_t)0x7255, (q15_t)0x398C,
	(q15_t)0x70E2, (q15_t)0x3C56, (q15_t)0x6F5F, (q15_t)0x3F17,
	(q15_t)0x6DCA, (q15_t)0x41CE, (q15_t)0x6C24, (q15_t)0x447A,
	(q15_t)0x6A6D, (q15_t)0x471C, (q15_t)0x68A6, (q15_t)0x49B4,
	(q15_t)0x66CF, (q15_t)0x4C3F, (q15_t)0x64E8, (q15_t)0x4EBF,
	(q15_t)0x62F2, (q15_t)0x5133, (q15_t)0x60EC, (q15_t)0x539B,
	(q15_t)0x5ED7, (q15_t)0x55F5, (q15_t)0x5CB4, (q15_t)0x5842,
	(q15_t)0x5A82, (q15_t)0x5A82, (q15_t)0x5842, (q15_t)0x5CB4,
	(q15_t)0x55F5, (q15_t)0x5ED7, (q15_t)0x539B, (q15_t)0x60EC,
	(q15_t)0x5133, (q15_t)0x62F2, (q15_t)0x4EBF, (q15_t)0x64E8,
	(q15_t)0x4C3F, (q15_t)0x66CF, (q15_t)0x49B4, (q15_t)0x68A6,
	(q15_t)0x471C, (q15_t)0x6A6D, (q15_t)0x447A, (q15_t)0x6C24,
	(q15_t)0x41CE, (q15_t)0x6DCA, (q15_t)0x3F17, (q15_t)0x6F5F,
	(q15_t)0x3C56, (q15_t)0x70E2, (q15_t)0x398C, (q15_t)0x7255,
	(q15_t)0x36BA, (q15_t)0x73B5, (q15_t)0x33DE, (q15_t)0x7504,
	(q15_t)0x30FB, (q15_t)0x7641, (q15_t)0x2E11, (q15_t)0x776C,
	(q15_t)0x2B1F, (q15_t)0x7884, (q15_t)0x2826, (q15_t)0x798A,
	(q15_t)0x2528, (q15_t)0x7A7D, (q15_t)0x2223, (q15_t)0x7B5D,
	(q15_t)0x1F19, (q15_t)0x7C29, (q15_t)0x1C0B, (q15_t)0x7CE3,
	(q15_t)0x18F8, (q15_t)0x7D8A, (q15_t)0x15E2, (q15_t)0x7E1D,
	(q15_t)0x12C8, (q15_t)0x7E9D, (q15_t)0x0FAB, (q15_t)0x7F09,
	(q15_t)0x0C8B, (q15_t)0x7F62, (q15_t)0x096A, (q15_t)0x7FA7,
	(q15_t)0x0647, (q15_t)0x7FD8, (q15_t)0x0324, (q15_t)0x7FF6,
	(q15_t)0x0000, (q15_t)0x7FFF, (q15_t)0xFCDB, (q15_t)0x7FF6,
	(q15_t)0xF9B8, (q15_t)0x7FD8, (q15_t)0xF695, (q15_t)0x7FA7,
	(q15_t)0xF374, (q15_t)0x7F62, (q15_t)0xF054, (q15_t)0x7F09,
	(q15_t)0xED37, (q15_t)0x7E9D, (q15_t)0xEA1D, (q15_t)0x7E1D,
	(q15_t)0xE707, (q15_t)0x7D8A, (q15_t)0xE3F4, (q15_t)0x7CE3,
	(q15_t)0xE0E6, (q15_t)0x7C29, (q15_t)0xDDDC, (q15_t)0x7B5D,
	(q15_t)0xDAD7, (q15_t)0x7A7D, (q15_t)0xD7D9, (q15_t)0x798A,
	(q15_t)0xD4E0, (q15_t)0x7884, (q15_t)0xD1EE, (q15_t)0x776C,
	(q15_t)0xCF04, (q15_t)0x7641, (q15_t)0xCC21, (q15_t)0x7504,
	(q15_t)0xC945, (q15_t)0x73B5, (q15_t)0xC673, (q15_t)0x7255,
	(q15_t)0xC3A9, (q15_t)0x70E2, (q15_t)0xC0E8, (q15_t)0x6F5F,
	(q15_t)0xBE31, (q15_t)0x6DCA, (q15_t)0xBB85, (q15_t)0x6C24,
	(q15_t)0xB8E3, (q15_t)0x6A6D, (q15_t)0xB64B, (q15_t)0x68A6,
	(q15_t)0xB3C0, (q15_t)0x66CF, (q15_t)0xB140, (q15_t)0x64E8,
	(q15_t)0xAECC, (q15_t)0x62F2, (q15_t)0xAC64, (q15_t)0x60EC,
	(q15_t)0xAA0A, (q15_t)0x5ED7, (q15_t)0xA7BD, (q15_t)0x5CB4,
	(q15_t)0xA57D, (q15_t)0x5A82, (q15_t)0xA34B, (q15_t)0x5842,
	(q15_t)0xA128, (q15_t)0x55F5, (q15_t)0x9F13, (q15_t)0x539B,
	(q15_t)0x9D0D, (q15_t)0x5133, (q15_t)0x9B17, (q15_t)0x4EBF,
	(q15_t)0x9930, (q15_t)0x4C3F, (q15_t)0x9759, (q15_t)0x49B4,
	(q15_t)0x9592, (q15_t)0x471C, (q15_t)0x93DB, (q15_t)0x447A,
	(q15_t)0x9235, (q15_t)0x41CE, (q15_t)0x90A0, (q15_t)0x3F17,
	(q15_t)0x8F1D, (q15_t)0x3C56, (q15_t)0x8DAA, (q15_t)0x398C,
	(q15_t)0x8C4A, (q15_t)0x36BA, (q15_t)0x8AFB, (q15_t)0x33DE,
	(q15_t)0x89BE, (q15_t)0x30FB, (q15_t)0x8893, (q15_t)0x2E11,
	(q15_t)0x877B, (q15_t)0x2B1F, (q15_t)0x8675, (q15_t)0x2826,
	(q15_t)0x8582, (q15_t)0x2528, (q15_t)0x84A2, (q15_t)0x2223,
	(q15_t)0x83D6, (q15_t)0x1F19, (q15_t)0x831C, (q15_t)0x1C0B,
	(q15_t)0x8275, (q15_t)0x18F8, (q15_t)0x81E2, (q15_t)0x15E2,
	(q15_t)0x8162, (q15_t)0x12C8, (q15_t)0x80F6, (q15_t)0x0FAB,
	(q15_t)0x809D, (q15_t)0x0C8B, (q15_t)0x8058, (q15_t)0x096A,
	(q15_t)0x8027, (q15_t)0x0647, (q15_t)0x8009, (q15_t)0x0324,
	(q15_t)0x8000, (q15_t)0x0000, (q15_t)0x8009, (q15_t)0xFCDB,
	(q15_t)0x8027, (q15_t)0xF9B8, (q15_t)0x8058, (q15_t)0xF695,
	(q15_t)0x809D, (q15_t)0xF374, (q15_t)0x80F6, (q15_t)0xF054,
	(q15_t)0x8162, (q15_t)0xED37, (q15_t)0x81E2, (q15_t)0xEA1D,
	(q15_t)0x8275, (q15_t)0xE707, (q15_t)0x831C, (q15_t)0xE3F4,
	(q15_t)0x83D6, (q15_t)0xE0E6, (q15_t)0x84A2, (q15_t)0xDDDC,
	(q15_t)0x8582, (q15_t)0xDAD7, (q15_t)0x8675, (q15_t)0xD7D9,
	(q15_t)0x877B, (q15_t)0xD4E0, (q15_t)0x8893, (q15_t)0xD1EE,
	(q15_t)0x89BE, (q15_t)0xCF04, (q15_t)0x8AFB, (q15_t)0xCC21,
	(q15_t)0x8C4A, (q15_t)0xC945, (q15_t)0x8DAA, (q15_t)0xC673,
	(q15_t)0x8F1D, (q15_t)0xC3A9, (q15_t)0x90A0, (q15_t)0xC0E8,
	(q15_t)0x9235, (q15_t)0xBE31, (q15_t)0x93DB, (q15_t)0xBB85,
	(q15_t)0x9592, (q15_t)0xB8E3, (q15_t)0x9759, (q15_t)0xB64B,
	(q15_t)0x9930, (q15_t)0xB3C0, (q15_t)0x9B17, (q15_t)0xB140,
	(q15_t)0x9D0D, (q15_t)0xAECC, (q15_t)0x9F13, (q15_t)0xAC64,
	(q15_t)0xA128, (q15_t)0xAA0A, (q15_t)0xA34B, (q15_t)0xA7BD,
	(q15_t)0xA57D, (q15_t)0xA57D, (q15_t)0xA7BD, (q15_t)0xA34B,
	(q15_t)0xAA0A, (q15_t)0xA128, (q15_t)0xAC64, (q15_t)0x9F13,
	(q15_t)0xAECC, (q15_t)0x9D0D, (q15_t)0xB140, (q15_t)0x9B17,
	(q15_t)0xB3C0, (q15_t)0x9930, (q15_t)0xB64B, (q15_t)0x9759,
	(q15_t)0xB8E3, (q15_t)0x9592, (q15_t)0xBB85, (q15_t)0x93DB,
	(q15_t)0xBE31, (q15_t)0x9235, (q15_t)0xC0E8, (q15_t)0x90A0,
	(q15_t)0xC3A9, (q15_t)0x8F1D, (q15_t)0xC673, (q15_t)0x8DAA,
	(q15_t)0xC945, (q15_t)0x8C4A, (q15_t)0xCC21, (q15_t)0x8AFB,
	(q15_t)0xCF04, (q15_t)0x89BE, (q15_t)0xD1EE, (q15_t)0x8893,
	(q15_t)0xD4E0, (q15_t)0x877B, (q15_t)0xD7D9, (q15_t)0x8675,
	(q15_t)0xDAD7, (q15_t)0x8582, (q15_t)0xDDDC, (q15_t)0x84A2,
	(q15_t)0xE0E6, (q15_t)0x83D6, (q15_t)0xE3F4, (q15_t)0x831C,
	(q15_t)0xE707, (q15_t)0x8275, (q15_t)0xEA1D, (q15_t)0x81E2,
	(q15_t)0xED37, (q15_t)0x8162, (q15_t)0xF054, (q15_t)0x80F6,
	(q15_t)0xF374, (q15_t)0x809D, (q15_t)0xF695, (q15_t)0x8058,
	(q15_t)0xF9B8, (q15_t)0x8027, (q15_t)0xFCDB, (q15_t)0x8009
};

/**
* \par
* Example code for q15 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefq15[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefq15[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 512	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to q15(Fixed point 1.15):
*	round(twiddleCoefq15(i) * pow(2, 15))
*
*/
const q15_t twiddleCoef_512_q15[768] = {
	(q15_t)0x7FFF, (q15_t)0x0000, (q15_t)0x7FFD, (q15_t)0x0192,
	(q15_t)0x7FF6, (q15_t)0x0324, (q15_t)0x7FE9, (q15_t)0x04B6,
	(q15_t)0x7FD8, (q15_t)0x0647, (q15_t)0x7FC2, (q15_t)0x07D9,
	(q15_t)0x7FA7, (q15_t)0x096A, (q15_t)0x7F87, (q15_t)0x0AFB,
	(q15_t)0x7F62, (q15_t)0x0C8B, (q15_t)0x7F38, (q15_t)0x0E1B,
	(q15_t)0x7F09, (q15_t)0x0FAB, (q15_t)0x7ED5, (q15_t)0x1139,
	(q15_t)0x7E9D, (q15_t)0x12C8, (q15_t)0x7E5F, (q15_t)0x1455,
	(q15_t)0x7E1D, (q15_t)0x15E2, (q15_t)0x7DD6, (q15_t)0x176D,
	(q15_t)0x7D8A, (q15_t)0x18F8, (q15_t)0x7D39, (q15_t)0x1A82,
	(q15_t)0x7CE3, (q15_t)0x1C0B, (q15_t)0x7C89, (q15_t)0x1D93,
	(q15_t)0x7C29, (q15_t)0x1F19, (q15_t)0x7BC5, (q15_t)0x209F,
	(q15_t)0x7B5D, (q15_t)0x2223, (q15_t)0x7AEF, (q15_t)0x23A6,
	(q15_t)0x7A7D, (q15_t)0x2528, (q15_t)0x7A05, (q15_t)0x26A8,
	(q15_t)0x798A, (q15_t)0x2826, (q15_t)0x7909, (q15_t)0x29A3,
	(q15_t)0x7884, (q15_t)0x2B1F, (q15_t)0x77FA, (q15_t)0x2C98,
	(q15_t)0x776C, (q15_t)0x2E11, (q15_t)0x76D9, (q15_t)0x2F87,
	(q15_t)0x7641, (q15_t)0x30FB, (q15_t)0x75A5, (q15_t)0x326E,
	(q15_t)0x7504, (q15_t)0x33DE, (q15_t)0x745F, (q15_t)0x354D,
	(q15_t)0x73B5, (q15_t)0x36BA, (q15_t)0x7307, (q15_t)0x3824,
	(q15_t)0x7255, (q15_t)0x398C, (q15_t)0x719E, (q15_t)0x3AF2,
	(q15_t)0x70E2, (q15_t)0x3C56, (q15_t)0x7023, (q15_t)0x3DB8,
	(q15_t)0x6F5F, (q15_t)0x3F17, (q15_t)0x6E96, (q15_t)0x4073,
	(q15_t)0x6DCA, (q15_t)0x41CE, (q15_t)0x6CF9, (q15_t)0x4325,
	(q15_t)0x6C24, (q15_t)0x447A, (q15_t)0x6B4A, (q15_t)0x45CD,
	(q15_t)0x6A6D, (q15_t)0x471C, (q15_t)0x698C, (q15_t)0x4869,
	(q15_t)0x68A6, (q15_t)0x49B4, (q15_t)0x67BD, (q15_t)0x4AFB,
	(q15_t)0x66CF, (q15_t)0x4C3F, (q15_t)0x65DD, (q15_t)0x4D81,
	(q15_t)0x64E8, (q15_t)0x4EBF, (q15_t)0x63EF, (q15_t)0x4FFB,
	(q15_t)0x62F2, (q15_t)0x5133, (q15_t)0x61F1, (q15_t)0x5269,
	(q15_t)0x60EC, (q15_t)0x539B, (q15_t)0x5FE3, (q15_t)0x54CA,
	(q15_t)0x5ED7, (q15_t)0x55F5, (q15_t)0x5DC7, (q15_t)0x571D,
	(q15_t)0x5CB4, (q15_t)0x5842, (q15_t)0x5B9D, (q15_t)0x5964,
	(q15_t)0x5A82, (q15_t)0x5A82, (q15_t)0x5964, (q15_t)0x5B9D,
	(q15_t)0x5842, (q15_t)0x5CB4, (q15_t)0x571D, (q15_t)0x5DC7,
	(q15_t)0x55F5, (q15_t)0x5ED7, (q15_t)0x54CA, (q15_t)0x5FE3,
	(q15_t)0x539B, (q15_t)0x60EC, (q15_t)0x5269, (q15_t)0x61F1,
	(q15_t)0x5133, (q15_t)0x62F2, (q15_t)0x4FFB, (q15_t)0x63EF,
	(q15_t)0x4EBF, (q15_t)0x64E8, (q15_t)0x4D81, (q15_t)0x65DD,
	(q15_t)0x4C3F, (q15_t)0x66CF, (q15_t)0x4AFB, (q15_t)0x67BD,
	(q15_t)0x49B4, (q15_t)0x68A6, (q15_t)0x4869, (q15_t)0x698C,
	(q15_t)0x471C, (q15_t)0x6A6D, (q15_t)0x45CD, (q15_t)0x6B4A,
	(q15_t)0x447A, (q15_t)0x6C24, (q15_t)0x4325, (q15_t)0x6CF9,
	(q15_t)0x41CE, (q15_t)0x6DCA, (q15_t)0x4073, (q15_t)0x6E96,
	(q15_t)0x3F17, (q15_t)0x6F5F, (q15_t)0x3DB8, (q15_t)0x7023,
	(q15_t)0x3C56, (q15_t)0x70E2, (q15_t)0x3AF2, (q15_t)0x719E,
	(q15_t)0x398C, (q15_t)0x7255, (q15_t)0x3824, (q15_t)0x7307,
	(q15_t)0x36BA, (q15_t)0x73B5, (q15_t)0x354D, (q15_t)0x745F,
	(q15_t)0x33DE, (q15_t)0x7504, (q15_t)0x326E, (q15_t)0x75A5,
	(q15_t)0x30FB, (q15_t)0x7641, (q15_t)0x2F87, (q15_t)0x76D9,
	(q15_t)0x2E11, (q15_t)0x776C, (q15_t)0x2C98, (q15_t)0x77FA,
	(q15_t)0x2B1F, (q15_t)0x7884, (q15_t)0x29A3, (q15_t)0x7909,
	(q15_t)0x2826, (q15_t)0x798A, (q15_t)0x26A8, (q15_t)0x7A05,
	(q15_t)0x2528, (q15_t)0x7A7D, (q15_t)0x23A6, (q15_t)0x7AEF,
	(q15_t)0x2223, (q15_t)0x7B5D, (q15_t)0x209F, (q15_t)0x7BC5,
	(q15_t)0x1F19, (q15_t)0x7C29, (q15_t)0x1D93, (q15_t)0x7C89,
	(q15_t)0x1C0B, (q15_t)0x7CE3, (q15_t)0x1A82, (q15_t)0x7D39,
	(q15_t)0x18F8, (q15_t)0x7D8A, (q15_t)0x176D, (q15_t)0x7DD6,
	(q15_t)0x15E2, (q15_t)0x7E1D, (q15_t)0x1455, (q15_t)0x7E5F,
	(q15_t)0x12C8, (q15_t)0x7E9D, (q15_t)0x1139, (q15_t)0x7ED5,
	(q15_t)0x0FAB, (q15_t)0x7F09, (q15_t)0x0E1B, (q15_t)0x7F38,
	(q15_t)0x0C8B, (q15_t)0x7F62, (q15_t)0x0AFB, (q15_t)0x7F87,
	(q15_t)0x096A, (q15_t)0x7FA7, (q15_t)0x07D9, (q15_t)0x7FC2,
	(q15_t)0x0647, (q15_t)0x7FD8, (q15_t)0x04B6, (q15_t)0x7FE9,
	(q15_t)0x0324, (q15_t)0x7FF6, (q15_t)0x0192, (q15_t)0x7FFD,
	(q15_t)0x0000, (q15_t)0x7FFF, (q15_t)0xFE6D, (q15_t)0x7FFD,
	(q15_t)0xFCDB, (q15_t)0x7FF6, (q15_t)0xFB49, (q15_t)0x7FE9,
	(q15_t)0xF9B8, (q15_t)0x7FD8, (q15_t)0xF826, (q15_t)0x7FC2,
	(q15_t)0xF695, (q15_t)0x7FA7, (q15_t)0xF504, (q15_t)0x7F87,
	(q15_t)0xF374, (q15_t)0x7F62, (q15_t)0xF1E4, (q15_t)0x7F38,
	(q15_t)0xF054, (q15_t)0x7F09, (q15_t)0xEEC6, (q15_t)0x7ED5,
	(q15_t)0xED37, (q15_t)0x7E9D, (q15_t)0xEBAA, (q15_t)0x7E5F,
	(q15_t)0xEA1D, (q15_t)0x7E1D, (q15_t)0xE892, (q15_t)0x7DD6,
	(q15_t)0xE707, (q15_t)0x7D8A, (q15_t)0xE57D, (q15_t)0x7D39,
	(q15_t)0xE3F4, (q15_t)0x7CE3, (q15_t)0xE26C, (q15_t)0x7C89,
	(q15_t)0xE0E6, (q15_t)0x7C29, (q15_t)0xDF60, (q15_t)0x7BC5,
	(q15_t)0xDDDC, (q15_t)0x7B5D, (q15_t)0xDC59, (q15_t)0x7AEF,
	(q15_t)0xDAD7, (q15_t)0x7A7D, (q15_t)0xD957, (q15_t)0x7A05,
	(q15_t)0xD7D9, (q15_t)0x798A, (q15_t)0xD65C, (q15_t)0x7909,
	(q15_t)0xD4E0, (q15_t)0x7884, (q15_t)0xD367, (q15_t)0x77FA,
	(q15_t)0xD1EE, (q15_t)0x776C, (q15_t)0xD078, (q15_t)0x76D9,
	(q15_t)0xCF04, (q15_t)0x7641, (q15_t)0xCD91, (q15_t)0x75A5,
	(q15_t)0xCC21, (q15_t)0x7504, (q15_t)0xCAB2, (q15_t)0x745F,
	(q15_t)0xC945, (q15_t)0x73B5, (q15_t)0xC7DB, (q15_t)0x7307,
	(q15_t)0xC673, (q15_t)0x7255, (q15_t)0xC50D, (q15_t)0x719E,
	(q15_t)0xC3A9, (q15_t)0x70E2, (q15_t)0xC247, (q15_t)0x7023,
	(q15_t)0xC0E8, (q15_t)0x6F5F, (q15_t)0xBF8C, (q15_t)0x6E96,
	(q15_t)0xBE31, (q15_t)0x6DCA, (q15_t)0xBCDA, (q15_t)0x6CF9,
	(q15_t)0xBB85, (q15_t)0x6C24, (q15_t)0xBA32, (q15_t)0x6B4A,
	(q15_t)0xB8E3, (q15_t)0x6A6D, (q15_t)0xB796, (q15_t)0x698C,
	(q15_t)0xB64B, (q15_t)0x68A6, (q15_t)0xB504, (q15_t)0x67BD,
	(q15_t)0xB3C0, (q15_t)0x66CF, (q15_t)0xB27E, (q15_t)0x65DD,
	(q15_t)0xB140, (q15_t)0x64E8, (q15_t)0xB004, (q15_t)0x63EF,
	(q15_t)0xAECC, (q15_t)0x62F2, (q15_t)0xAD96, (q15_t)0x61F1,
	(q15_t)0xAC64, (q15_t)0x60EC, (q15_t)0xAB35, (q15_t)0x5FE3,
	(q15_t)0xAA0A, (q15_t)0x5ED7, (q15_t)0xA8E2, (q15_t)0x5DC7,
	(q15_t)0xA7BD, (q15_t)0x5CB4, (q15_t)0xA69B, (q15_t)0x5B9D,
	(q15_t)0xA57D, (q15_t)0x5A82, (q15_t)0xA462, (q15_t)0x5964,
	(q15_t)0xA34B, (q15_t)0x5842, (q15_t)0xA238, (q15_t)0x571D,
	(q15_t)0xA128, (q15_t)0x55F5, (q15_t)0xA01C, (q15_t)0x54CA,
	(q15_t)0x9F13, (q15_t)0x539B, (q15_t)0x9E0E, (q15_t)0x5269,
	(q15_t)0x9D0D, (q15_t)0x5133, (q15_t)0x9C10, (q15_t)0x4FFB,
	(q15_t)0x9B17, (q15_t)0x4EBF, (q15_t)0x9A22, (q15_t)0x4D81,
	(q15_t)0x9930, (q15_t)0x4C3F, (q15_t)0x9842, (q15_t)0x4AFB,
	(q15_t)0x9759, (q15_t)0x49B4, (q15_t)0x9673, (q15_t)0x4869,
	(q15_t)0x9592, (q15_t)0x471C, (q15_t)0x94B5, (q15_t)0x45CD,
	(q15_t)0x93DB, (q15_t)0x447A, (q15_t)0x9306, (q15_t)0x4325,
	(q15_t)0x9235, (q15_t)0x41CE, (q15_t)0x9169, (q15_t)0x4073,
	(q15_t)0x90A0, (q15_t)0x3F17, (q15_t)0x8FDC, (q15_t)0x3DB8,
	(q15_t)0x8F1D, (q15_t)0x3C56, (q15_t)0x8E61, (q15_t)0x3AF2,
	(q15_t)0x8DAA, (q15_t)0x398C, (q15_t)0x8CF8, (q15_t)0x3824,
	(q15_t)0x8C4A, (q15_t)0x36BA, (q15_t)0x8BA0, (q15_t)0x354D,
	(q15_t)0x8AFB, (q15_t)0x33DE, (q15_t)0x8A5A, (q15_t)0x326E,
	(q15_t)0x89BE, (q15_t)0x30FB, (q15_t)0x8926, (q15_t)0x2F87,
	(q15_t)0x8893, (q15_t)0x2E11, (q15_t)0x8805, (q15_t)0x2C98,
	(q15_t)0x877B, (q15_t)0x2B1F, (q15_t)0x86F6, (q15_t)0x29A3,
	(q15_t)0x8675, (q15_t)0x2826, (q15_t)0x85FA, (q15_t)0x26A8,
	(q15_t)0x8582, (q15_t)0x2528, (q15_t)0x8510, (q15_t)0x23A6,
	(q15_t)0x84A2, (q15_t)0x2223, (q15_t)0x843A, (q15_t)0x209F,
	(q15_t)0x83D6, (q15_t)0x1F19, (q15_t)0x8376, (q15_t)0x1D93,
	(q15_t)0x831C, (q15_t)0x1C0B, (q15_t)0x82C6, (q15_t)0x1A82,
	(q15_t)0x8275, (q15_t)0x18F8, (q15_t)0x8229, (q15_t)0x176D,
	(q15_t)0x81E2, (q15_t)0x15E2, (q15_t)0x81A0, (q15_t)0x1455,
	(q15_t)0x8162, (q15_t)0x12C8, (q15_t)0x812A, (q15_t)0x1139,
	(q15_t)0x80F6, (q15_t)0x0FAB, (q15_t)0x80C7, (q15_t)0x0E1B,
	(q15_t)0x809D, (q15_t)0x0C8B, (q15_t)0x8078, (q15_t)0x0AFB,
	(q15_t)0x8058, (q15_t)0x096A, (q15_t)0x803D, (q15_t)0x07D9,
	(q15_t)0x8027, (q15_t)0x0647, (q15_t)0x8016, (q15_t)0x04B6,
	(q15_t)0x8009, (q15_t)0x0324, (q15_t)0x8002, (q15_t)0x0192,
	(q15_t)0x8000, (q15_t)0x0000, (q15_t)0x8002, (q15_t)0xFE6D,
	(q15_t)0x8009, (q15_t)0xFCDB, (q15_t)0x8016, (q15_t)0xFB49,
	(q15_t)0x8027, (q15_t)0xF9B8, (q15_t)0x803D, (q15_t)0xF826,
	(q15_t)0x8058, (q15_t)0xF695, (q15_t)0x8078, (q15_t)0xF504,
	(q15_t)0x809D, (q15_t)0xF374, (q15_t)0x80C7, (q15_t)0xF1E4,
	(q15_t)0x80F6, (q15_t)0xF054, (q15_t)0x812A, (q15_t)0xEEC6,
	(q15_t)0x8162, (q15_t)0xED37, (q15_t)0x81A0, (q15_t)0xEBAA,
	(q15_t)0x81E2, (q15_t)0xEA1D, (q15_t)0x8229, (q15_t)0xE892,
	(q15_t)0x8275, (q15_t)0xE707, (q15_t)0x82C6, (q15_t)0xE57D,
	(q15_t)0x831C, (q15_t)0xE3F4, (q15_t)0x8376, (q15_t)0xE26C,
	(q15_t)0x83D6, (q15_t)0xE0E6, (q15_t)0x843A, (q15_t)0xDF60,
	(q15_t)0x84A2, (q15_t)0xDDDC, (q15_t)0x8510, (q15_t)0xDC59,
	(q15_t)0x8582, (q15_t)0xDAD7, (q15_t)0x85FA, (q15_t)0xD957,
	(q15_t)0x8675, (q15_t)0xD7D9, (q15_t)0x86F6, (q15_t)0xD65C,
	(q15_t)0x877B, (q15_t)0xD4E0, (q15_t)0x8805, (q15_t)0xD367,
	(q15_t)0x8893, (q15_t)0xD1EE, (q15_t)0x8926, (q15_t)0xD078,
	(q15_t)0x89BE, (q15_t)0xCF04, (q15_t)0x8A5A, (q15_t)0xCD91,
	(q15_t)0x8AFB, (q15_t)0xCC21, (q15_t)0x8BA0, (q15_t)0xCAB2,
	(q15_t)0x8C4A, (q15_t)0xC945, (q15_t)0x8CF8, (q15_t)0xC7DB,
	(q15_t)0x8DAA, (q15_t)0xC673, (q15_t)0x8E61, (q15_t)0xC50D,
	(q15_t)0x8F1D, (q15_t)0xC3A9, (q15_t)0x8FDC, (q15_t)0xC247,
	(q15_t)0x90A0, (q15_t)0xC0E8, (q15_t)0x9169, (q15_t)0xBF8C,
	(q15_t)0x9235, (q15_t)0xBE31, (q15_t)0x9306, (q15_t)0xBCDA,
	(q15_t)0x93DB, (q15_t)0xBB85, (q15_t)0x94B5, (q15_t)0xBA32,
	(q15_t)0x9592, (q15_t)0xB8E3, (q15_t)0x9673, (q15_t)0xB796,
	(q15_t)0x9759, (q15_t)0xB64B, (q15_t)0x9842, (q15_t)0xB504,
	(q15_t)0x9930, (q15_t)0xB3C0, (q15_t)0x9A22, (q15_t)0xB27E,
	(q15_t)0x9B17, (q15_t)0xB140, (q15_t)0x9C10, (q15_t)0xB004,
	(q15_t)0x9D0D, (q15_t)0xAECC, (q15_t)0x9E0E, (q15_t)0xAD96,
	(q15_t)0x9F13, (q15_t)0xAC64, (q15_t)0xA01C, (q15_t)0xAB35,
	(q15_t)0xA128, (q15_t)0xAA0A, (q15_t)0xA238, (q15_t)0xA8E2,
	(q15_t)0xA34B, (q15_t)0xA7BD, (q15_t)0xA462, (q15_t)0xA69B,
	(q15_t)0xA57D, (q15_t)0xA57D, (q15_t)0xA69B, (q15_t)0xA462,
	(q15_t)0xA7BD, (q15_t)0xA34B, (q15_t)0xA8E2, (q15_t)0xA238,
	(q15_t)0xAA0A, (q15_t)0xA128, (q15_t)0xAB35, (q15_t)0xA01C,
	(q15_t)0xAC64, (q15_t)0x9F13, (q15_t)0xAD96, (q15_t)0x9E0E,
	(q15_t)0xAECC, (q15_t)0x9D0D, (q15_t)0xB004, (q15_t)0x9C10,
	(q15_t)0xB140, (q15_t)0x9B17, (q15_t)0xB27E, (q15_t)0x9A22,
	(q15_t)0xB3C0, (q15_t)0x9930, (q15_t)0xB504, (q15_t)0x9842,
	(q15_t)0xB64B, (q15_t)0x9759, (q15_t)0xB796, (q15_t)0x9673,
	(q15_t)0xB8E3, (q15_t)0x9592, (q15_t)0xBA32, (q15_t)0x94B5,
	(q15_t)0xBB85, (q15_t)0x93DB, (q15_t)0xBCDA, (q15_t)0x9306,
	(q15_t)0xBE31, (q15_t)0x9235, (q15_t)0xBF8C, (q15_t)0x9169,
	(q15_t)0xC0E8, (q15_t)0x90A0, (q15_t)0xC247, (q15_t)0x8FDC,
	(q15_t)0xC3A9, (q15_t)0x8F1D, (q15_t)0xC50D, (q15_t)0x8E61,
	(q15_t)0xC673, (q15_t)0x8DAA, (q15_t)0xC7DB, (q15_t)0x8CF8,
	(q15_t)0xC945, (q15_t)0x8C4A, (q15_t)0xCAB2, (q15_t)0x8BA0,
	(q15_t)0xCC21, (q15_t)0x8AFB, (q15_t)0xCD91, (q15_t)0x8A5A,
	(q15_t)0xCF04, (q15_t)0x89BE, (q15_t)0xD078, (q15_t)0x8926,
	(q15_t)0xD1EE, (q15_t)0x8893, (q15_t)0xD367, (q15_t)0x8805,
	(q15_t)0xD4E0, (q15_t)0x877B, (q15_t)0xD65C, (q15_t)0x86F6,
	(q15_t)0xD7D9, (q15_t)0x8675, (q15_t)0xD957, (q15_t)0x85FA,
	(q15_t)0xDAD7, (q15_t)0x8582, (q15_t)0xDC59, (q15_t)0x8510,
	(q15_t)0xDDDC, (q15_t)0x84A2, (q15_t)0xDF60, (q15_t)0x843A,
	(q15_t)0xE0E6, (q15_t)0x83D6, (q15_t)0xE26C, (q15_t)0x8376,
	(q15_t)0xE3F4, (q15_t)0x831C, (q15_t)0xE57D, (q15_t)0x82C6,
	(q15_t)0xE707, (q15_t)0x8275, (q15_t)0xE892, (q15_t)0x8229,
	(q15_t)0xEA1D, (q15_t)0x81E2, (q15_t)0xEBAA, (q15_t)0x81A0,
	(q15_t)0xED37, (q15_t)0x8162, (q15_t)0xEEC6, (q15_t)0x812A,
	(q15_t)0xF054, (q15_t)0x80F6, (q15_t)0xF1E4, (q15_t)0x80C7,
	(q15_t)0xF374, (q15_t)0x809D, (q15_t)0xF504, (q15_t)0x8078,
	(q15_t)0xF695, (q15_t)0x8058, (q15_t)0xF826, (q15_t)0x803D,
	(q15_t)0xF9B8, (q15_t)0x8027, (q15_t)0xFB49, (q15_t)0x8016,
	(q15_t)0xFCDB, (q15_t)0x8009, (q15_t)0xFE6D, (q15_t)0x8002
};

/**
* \par
* Example code for q15 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefq15[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefq15[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 1024	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to q15(Fixed point 1.15):
*	round(twiddleCoefq15(i) * pow(2, 15))
*
*/
const q15_t twiddleCoef_1024_q15[1536] = {
	(q15_t)0x7FFF, (q15_t)0x0000, (q15_t)0x7FFF, (q15_t)0x00C9,
	(q15_t)0x7FFD, (q15_t)0x0192, (q15_t)0x7FFA, (q15_t)0x025B,
	(q15_t)0x7FF6, (q15_t)0x0324, (q15_t)0x7FF0, (q15_t)0x03ED,
	(q15_t)0x7FE9, (q15_t)0x04B6, (q15_t)0x7FE1, (q15_t)0x057F,
	(q15_t)0x7FD8, (q15_t)0x0647, (q15_t)0x7FCE, (q15_t)0x0710,
	(q15_t)0x7FC2, (q15_t)0x07D9, (q15_t)0x7FB5, (q15_t)0x08A2,
	(q15_t)0x7FA7, (q15_t)0x096A, (q15_t)0x7F97, (q15_t)0x0A33,
	(q15_t)0x7F87, (q15_t)0x0AFB, (q15_t)0x7F75, (q15_t)0x0BC3,
	(q15_t)0x7F62, (q15_t)0x0C8B, (q15_t)0x7F4D, (q15_t)0x0D53,
	(q15_t)0x7F38, (q15_t)0x0E1B, (q15_t)0x7F21, (q15_t)0x0EE3,
	(q15_t)0x7F09, (q15_t)0x0FAB, (q15_t)0x7EF0, (q15_t)0x1072,
	(q15_t)0x7ED5, (q15_t)0x1139, (q15_t)0x7EBA, (q15_t)0x1201,
	(q15_t)0x7E9D, (q15_t)0x12C8, (q15_t)0x7E7F, (q15_t)0x138E,
	(q15_t)0x7E5F, (q15_t)0x1455, (q15_t)0x7E3F, (q15_t)0x151B,
	(q15_t)0x7E1D, (q15_t)0x15E2, (q15_t)0x7DFA, (q15_t)0x16A8,
	(q15_t)0x7DD6, (q15_t)0x176D, (q15_t)0x7DB0, (q15_t)0x1833,
	(q15_t)0x7D8A, (q15_t)0x18F8, (q15_t)0x7D62, (q15_t)0x19BD,
	(q15_t)0x7D39, (q15_t)0x1A82, (q15_t)0x7D0F, (q15_t)0x1B47,
	(q15_t)0x7CE3, (q15_t)0x1C0B, (q15_t)0x7CB7, (q15_t)0x1CCF,
	(q15_t)0x7C89, (q15_t)0x1D93, (q15_t)0x7C5A, (q15_t)0x1E56,
	(q15_t)0x7C29, (q15_t)0x1F19, (q15_t)0x7BF8, (q15_t)0x1FDC,
	(q15_t)0x7BC5, (q15_t)0x209F, (q15_t)0x7B92, (q15_t)0x2161,
	(q15_t)0x7B5D, (q15_t)0x2223, (q15_t)0x7B26, (q15_t)0x22E5,
	(q15_t)0x7AEF, (q15_t)0x23A6, (q15_t)0x7AB6, (q15_t)0x2467,
	(q15_t)0x7A7D, (q15_t)0x2528, (q15_t)0x7A42, (q15_t)0x25E8,
	(q15_t)0x7A05, (q15_t)0x26A8, (q15_t)0x79C8, (q15_t)0x2767,
	(q15_t)0x798A, (q15_t)0x2826, (q15_t)0x794A, (q15_t)0x28E5,
	(q15_t)0x7909, (q15_t)0x29A3, (q15_t)0x78C7, (q15_t)0x2A61,
	(q15_t)0x7884, (q15_t)0x2B1F, (q15_t)0x7840, (q15_t)0x2BDC,
	(q15_t)0x77FA, (q15_t)0x2C98, (q15_t)0x77B4, (q15_t)0x2D55,
	(q15_t)0x776C, (q15_t)0x2E11, (q15_t)0x7723, (q15_t)0x2ECC,
	(q15_t)0x76D9, (q15_t)0x2F87, (q15_t)0x768E, (q15_t)0x3041,
	(q15_t)0x7641, (q15_t)0x30FB, (q15_t)0x75F4, (q15_t)0x31B5,
	(q15_t)0x75A5, (q15_t)0x326E, (q15_t)0x7555, (q15_t)0x3326,
	(q15_t)0x7504, (q15_t)0x33DE, (q15_t)0x74B2, (q15_t)0x3496,
	(q15_t)0x745F, (q15_t)0x354D, (q15_t)0x740B, (q15_t)0x3604,
	(q15_t)0x73B5, (q15_t)0x36BA, (q15_t)0x735F, (q15_t)0x376F,
	(q15_t)0x7307, (q15_t)0x3824, (q15_t)0x72AF, (q15_t)0x38D8,
	(q15_t)0x7255, (q15_t)0x398C, (q15_t)0x71FA, (q15_t)0x3A40,
	(q15_t)0x719E, (q15_t)0x3AF2, (q15_t)0x7141, (q15_t)0x3BA5,
	(q15_t)0x70E2, (q15_t)0x3C56, (q15_t)0x7083, (q15_t)0x3D07,
	(q15_t)0x7023, (q15_t)0x3DB8, (q15_t)0x6FC1, (q15_t)0x3E68,
	(q15_t)0x6F5F, (q15_t)0x3F17, (q15_t)0x6EFB, (q15_t)0x3FC5,
	(q15_t)0x6E96, (q15_t)0x4073, (q15_t)0x6E30, (q15_t)0x4121,
	(q15_t)0x6DCA, (q15_t)0x41CE, (q15_t)0x6D62, (q15_t)0x427A,
	(q15_t)0x6CF9, (q15_t)0x4325, (q15_t)0x6C8F, (q15_t)0x43D0,
	(q15_t)0x6C24, (q15_t)0x447A, (q15_t)0x6BB8, (q15_t)0x4524,
	(q15_t)0x6B4A, (q15_t)0x45CD, (q15_t)0x6ADC, (q15_t)0x4675,
	(q15_t)0x6A6D, (q15_t)0x471C, (q15_t)0x69FD, (q15_t)0x47C3,
	(q15_t)0x698C, (q15_t)0x4869, (q15_t)0x6919, (q15_t)0x490F,
	(q15_t)0x68A6, (q15_t)0x49B4, (q15_t)0x6832, (q15_t)0x4A58,
	(q15_t)0x67BD, (q15_t)0x4AFB, (q15_t)0x6746, (q15_t)0x4B9E,
	(q15_t)0x66CF, (q15_t)0x4C3F, (q15_t)0x6657, (q15_t)0x4CE1,
	(q15_t)0x65DD, (q15_t)0x4D81, (q15_t)0x6563, (q15_t)0x4E21,
	(q15_t)0x64E8, (q15_t)0x4EBF, (q15_t)0x646C, (q15_t)0x4F5E,
	(q15_t)0x63EF, (q15_t)0x4FFB, (q15_t)0x6371, (q15_t)0x5097,
	(q15_t)0x62F2, (q15_t)0x5133, (q15_t)0x6271, (q15_t)0x51CE,
	(q15_t)0x61F1, (q15_t)0x5269, (q15_t)0x616F, (q15_t)0x5302,
	(q15_t)0x60EC, (q15_t)0x539B, (q15_t)0x6068, (q15_t)0x5433,
	(q15_t)0x5FE3, (q15_t)0x54CA, (q15_t)0x5F5E, (q15_t)0x5560,
	(q15_t)0x5ED7, (q15_t)0x55F5, (q15_t)0x5E50, (q15_t)0x568A,
	(q15_t)0x5DC7, (q15_t)0x571D, (q15_t)0x5D3E, (q15_t)0x57B0,
	(q15_t)0x5CB4, (q15_t)0x5842, (q15_t)0x5C29, (q15_t)0x58D4,
	(q15_t)0x5B9D, (q15_t)0x5964, (q15_t)0x5B10, (q15_t)0x59F3,
	(q15_t)0x5A82, (q15_t)0x5A82, (q15_t)0x59F3, (q15_t)0x5B10,
	(q15_t)0x5964, (q15_t)0x5B9D, (q15_t)0x58D4, (q15_t)0x5C29,
	(q15_t)0x5842, (q15_t)0x5CB4, (q15_t)0x57B0, (q15_t)0x5D3E,
	(q15_t)0x571D, (q15_t)0x5DC7, (q15_t)0x568A, (q15_t)0x5E50,
	(q15_t)0x55F5, (q15_t)0x5ED7, (q15_t)0x5560, (q15_t)0x5F5E,
	(q15_t)0x54CA, (q15_t)0x5FE3, (q15_t)0x5433, (q15_t)0x6068,
	(q15_t)0x539B, (q15_t)0x60EC, (q15_t)0x5302, (q15_t)0x616F,
	(q15_t)0x5269, (q15_t)0x61F1, (q15_t)0x51CE, (q15_t)0x6271,
	(q15_t)0x5133, (q15_t)0x62F2, (q15_t)0x5097, (q15_t)0x6371,
	(q15_t)0x4FFB, (q15_t)0x63EF, (q15_t)0x4F5E, (q15_t)0x646C,
	(q15_t)0x4EBF, (q15_t)0x64E8, (q15_t)0x4E21, (q15_t)0x6563,
	(q15_t)0x4D81, (q15_t)0x65DD, (q15_t)0x4CE1, (q15_t)0x6657,
	(q15_t)0x4C3F, (q15_t)0x66CF, (q15_t)0x4B9E, (q15_t)0x6746,
	(q15_t)0x4AFB, (q15_t)0x67BD, (q15_t)0x4A58, (q15_t)0x6832,
	(q15_t)0x49B4, (q15_t)0x68A6, (q15_t)0x490F, (q15_t)0x6919,
	(q15_t)0x4869, (q15_t)0x698C, (q15_t)0x47C3, (q15_t)0x69FD,
	(q15_t)0x471C, (q15_t)0x6A6D, (q15_t)0x4675, (q15_t)0x6ADC,
	(q15_t)0x45CD, (q15_t)0x6B4A, (q15_t)0x4524, (q15_t)0x6BB8,
	(q15_t)0x447A, (q15_t)0x6C24, (q15_t)0x43D0, (q15_t)0x6C8F,
	(q15_t)0x4325, (q15_t)0x6CF9, (q15_t)0x427A, (q15_t)0x6D62,
	(q15_t)0x41CE, (q15_t)0x6DCA, (q15_t)0x4121, (q15_t)0x6E30,
	(q15_t)0x4073, (q15_t)0x6E96, (q15_t)0x3FC5, (q15_t)0x6EFB,
	(q15_t)0x3F17, (q15_t)0x6F5F, (q15_t)0x3E68, (q15_t)0x6FC1,
	(q15_t)0x3DB8, (q15_t)0x7023, (q15_t)0x3D07, (q15_t)0x7083,
	(q15_t)0x3C56, (q15_t)0x70E2, (q15_t)0x3BA5, (q15_t)0x7141,
	(q15_t)0x3AF2, (q15_t)0x719E, (q15_t)0x3A40, (q15_t)0x71FA,
	(q15_t)0x398C, (q15_t)0x7255, (q15_t)0x38D8, (q15_t)0x72AF,
	(q15_t)0x3824, (q15_t)0x7307, (q15_t)0x376F, (q15_t)0x735F,
	(q15_t)0x36BA, (q15_t)0x73B5, (q15_t)0x3604, (q15_t)0x740B,
	(q15_t)0x354D, (q15_t)0x745F, (q15_t)0x3496, (q15_t)0x74B2,
	(q15_t)0x33DE, (q15_t)0x7504, (q15_t)0x3326, (q15_t)0x7555,
	(q15_t)0x326E, (q15_t)0x75A5, (q15_t)0x31B5, (q15_t)0x75F4,
	(q15_t)0x30FB, (q15_t)0x7641, (q15_t)0x3041, (q15_t)0x768E,
	(q15_t)0x2F87, (q15_t)0x76D9, (q15_t)0x2ECC, (q15_t)0x7723,
	(q15_t)0x2E11, (q15_t)0x776C, (q15_t)0x2D55, (q15_t)0x77B4,
	(q15_t)0x2C98, (q15_t)0x77FA, (q15_t)0x2BDC, (q15_t)0x7840,
	(q15_t)0x2B1F, (q15_t)0x7884, (q15_t)0x2A61, (q15_t)0x78C7,
	(q15_t)0x29A3, (q15_t)0x7909, (q15_t)0x28E5, (q15_t)0x794A,
	(q15_t)0x2826, (q15_t)0x798A, (q15_t)0x2767, (q15_t)0x79C8,
	(q15_t)0x26A8, (q15_t)0x7A05, (q15_t)0x25E8, (q15_t)0x7A42,
	(q15_t)0x2528, (q15_t)0x7A7D, (q15_t)0x2467, (q15_t)0x7AB6,
	(q15_t)0x23A6, (q15_t)0x7AEF, (q15_t)0x22E5, (q15_t)0x7B26,
	(q15_t)0x2223, (q15_t)0x7B5D, (q15_t)0x2161, (q15_t)0x7B92,
	(q15_t)0x209F, (q15_t)0x7BC5, (q15_t)0x1FDC, (q15_t)0x7BF8,
	(q15_t)0x1F19, (q15_t)0x7C29, (q15_t)0x1E56, (q15_t)0x7C5A,
	(q15_t)0x1D93, (q15_t)0x7C89, (q15_t)0x1CCF, (q15_t)0x7CB7,
	(q15_t)0x1C0B, (q15_t)0x7CE3, (q15_t)0x1B47, (q15_t)0x7D0F,
	(q15_t)0x1A82, (q15_t)0x7D39, (q15_t)0x19BD, (q15_t)0x7D62,
	(q15_t)0x18F8, (q15_t)0x7D8A, (q15_t)0x1833, (q15_t)0x7DB0,
	(q15_t)0x176D, (q15_t)0x7DD6, (q15_t)0x16A8, (q15_t)0x7DFA,
	(q15_t)0x15E2, (q15_t)0x7E1D, (q15_t)0x151B, (q15_t)0x7E3F,
	(q15_t)0x1455, (q15_t)0x7E5F, (q15_t)0x138E, (q15_t)0x7E7F,
	(q15_t)0x12C8, (q15_t)0x7E9D, (q15_t)0x1201, (q15_t)0x7EBA,
	(q15_t)0x1139, (q15_t)0x7ED5, (q15_t)0x1072, (q15_t)0x7EF0,
	(q15_t)0x0FAB, (q15_t)0x7F09, (q15_t)0x0EE3, (q15_t)0x7F21,
	(q15_t)0x0E1B, (q15_t)0x7F38, (q15_t)0x0D53, (q15_t)0x7F4D,
	(q15_t)0x0C8B, (q15_t)0x7F62, (q15_t)0x0BC3, (q15_t)0x7F75,
	(q15_t)0x0AFB, (q15_t)0x7F87, (q15_t)0x0A33, (q15_t)0x7F97,
	(q15_t)0x096A, (q15_t)0x7FA7, (q15_t)0x08A2, (q15_t)0x7FB5,
	(q15_t)0x07D9, (q15_t)0x7FC2, (q15_t)0x0710, (q15_t)0x7FCE,
	(q15_t)0x0647, (q15_t)0x7FD8, (q15_t)0x057F, (q15_t)0x7FE1,
	(q15_t)0x04B6, (q15_t)0x7FE9, (q15_t)0x03ED, (q15_t)0x7FF0,
	(q15_t)0x0324, (q15_t)0x7FF6, (q15_t)0x025B, (q15_t)0x7FFA,
	(q15_t)0x0192, (q15_t)0x7FFD, (q15_t)0x00C9, (q15_t)0x7FFF,
	(q15_t)0x0000, (q15_t)0x7FFF, (q15_t)0xFF36, (q15_t)0x7FFF,
	(q15_t)0xFE6D, (q15_t)0x7FFD, (q15_t)0xFDA4, (q15_t)0x7FFA,
	(q15_t)0xFCDB, (q15_t)0x7FF6, (q15_t)0xFC12, (q15_t)0x7FF0,
	(q15_t)0xFB49, (q15_t)0x7FE9, (q15_t)0xFA80, (q15_t)0x7FE1,
	(q15_t)0xF9B8, (q15_t)0x7FD8, (q15_t)0xF8EF, (q15_t)0x7FCE,
	(q15_t)0xF826, (q15_t)0x7FC2, (q15_t)0xF75D, (q15_t)0x7FB5,
	(q15_t)0xF695, (q15_t)0x7FA7, (q15_t)0xF5CC, (q15_t)0x7F97,
	(q15_t)0xF504, (q15_t)0x7F87, (q15_t)0xF43C, (q15_t)0x7F75,
	(q15_t)0xF374, (q15_t)0x7F62, (q15_t)0xF2AC, (q15_t)0x7F4D,
	(q15_t)0xF1E4, (q15_t)0x7F38, (q15_t)0xF11C, (q15_t)0x7F21,
	(q15_t)0xF054, (q15_t)0x7F09, (q15_t)0xEF8D, (q15_t)0x7EF0,
	(q15_t)0xEEC6, (q15_t)0x7ED5, (q15_t)0xEDFE, (q15_t)0x7EBA,
	(q15_t)0xED37, (q15_t)0x7E9D, (q15_t)0xEC71, (q15_t)0x7E7F,
	(q15_t)0xEBAA, (q15_t)0x7E5F, (q15_t)0xEAE4, (q15_t)0x7E3F,
	(q15_t)0xEA1D, (q15_t)0x7E1D, (q15_t)0xE957, (q15_t)0x7DFA,
	(q15_t)0xE892, (q15_t)0x7DD6, (q15_t)0xE7CC, (q15_t)0x7DB0,
	(q15_t)0xE707, (q15_t)0x7D8A, (q15_t)0xE642, (q15_t)0x7D62,
	(q15_t)0xE57D, (q15_t)0x7D39, (q15_t)0xE4B8, (q15_t)0x7D0F,
	(q15_t)0xE3F4, (q15_t)0x7CE3, (q15_t)0xE330, (q15_t)0x7CB7,
	(q15_t)0xE26C, (q15_t)0x7C89, (q15_t)0xE1A9, (q15_t)0x7C5A,
	(q15_t)0xE0E6, (q15_t)0x7C29, (q15_t)0xE023, (q15_t)0x7BF8,
	(q15_t)0xDF60, (q15_t)0x7BC5, (q15_t)0xDE9E, (q15_t)0x7B92,
	(q15_t)0xDDDC, (q15_t)0x7B5D, (q15_t)0xDD1A, (q15_t)0x7B26,
	(q15_t)0xDC59, (q15_t)0x7AEF, (q15_t)0xDB98, (q15_t)0x7AB6,
	(q15_t)0xDAD7, (q15_t)0x7A7D, (q15_t)0xDA17, (q15_t)0x7A42,
	(q15_t)0xD957, (q15_t)0x7A05, (q15_t)0xD898, (q15_t)0x79C8,
	(q15_t)0xD7D9, (q15_t)0x798A, (q15_t)0xD71A, (q15_t)0x794A,
	(q15_t)0xD65C, (q15_t)0x7909, (q15_t)0xD59E, (q15_t)0x78C7,
	(q15_t)0xD4E0, (q15_t)0x7884, (q15_t)0xD423, (q15_t)0x7840,
	(q15_t)0xD367, (q15_t)0x77FA, (q15_t)0xD2AA, (q15_t)0x77B4,
	(q15_t)0xD1EE, (q15_t)0x776C, (q15_t)0xD133, (q15_t)0x7723,
	(q15_t)0xD078, (q15_t)0x76D9, (q15_t)0xCFBE, (q15_t)0x768E,
	(q15_t)0xCF04, (q15_t)0x7641, (q15_t)0xCE4A, (q15_t)0x75F4,
	(q15_t)0xCD91, (q15_t)0x75A5, (q15_t)0xCCD9, (q15_t)0x7555,
	(q15_t)0xCC21, (q15_t)0x7504, (q15_t)0xCB69, (q15_t)0x74B2,
	(q15_t)0xCAB2, (q15_t)0x745F, (q15_t)0xC9FB, (q15_t)0x740B,
	(q15_t)0xC945, (q15_t)0x73B5, (q15_t)0xC890, (q15_t)0x735F,
	(q15_t)0xC7DB, (q15_t)0x7307, (q15_t)0xC727, (q15_t)0x72AF,
	(q15_t)0xC673, (q15_t)0x7255, (q15_t)0xC5BF, (q15_t)0x71FA,
	(q15_t)0xC50D, (q15_t)0x719E, (q15_t)0xC45A, (q15_t)0x7141,
	(q15_t)0xC3A9, (q15_t)0x70E2, (q15_t)0xC2F8, (q15_t)0x7083,
	(q15_t)0xC247, (q15_t)0x7023, (q15_t)0xC197, (q15_t)0x6FC1,
	(q15_t)0xC0E8, (q15_t)0x6F5F, (q15_t)0xC03A, (q15_t)0x6EFB,
	(q15_t)0xBF8C, (q15_t)0x6E96, (q15_t)0xBEDE, (q15_t)0x6E30,
	(q15_t)0xBE31, (q15_t)0x6DCA, (q15_t)0xBD85, (q15_t)0x6D62,
	(q15_t)0xBCDA, (q15_t)0x6CF9, (q15_t)0xBC2F, (q15_t)0x6C8F,
	(q15_t)0xBB85, (q15_t)0x6C24, (q15_t)0xBADB, (q15_t)0x6BB8,
	(q15_t)0xBA32, (q15_t)0x6B4A, (q15_t)0xB98A, (q15_t)0x6ADC,
	(q15_t)0xB8E3, (q15_t)0x6A6D, (q15_t)0xB83C, (q15_t)0x69FD,
	(q15_t)0xB796, (q15_t)0x698C, (q15_t)0xB6F0, (q15_t)0x6919,
	(q15_t)0xB64B, (q15_t)0x68A6, (q15_t)0xB5A7, (q15_t)0x6832,
	(q15_t)0xB504, (q15_t)0x67BD, (q15_t)0xB461, (q15_t)0x6746,
	(q15_t)0xB3C0, (q15_t)0x66CF, (q15_t)0xB31E, (q15_t)0x6657,
	(q15_t)0xB27E, (q15_t)0x65DD, (q15_t)0xB1DE, (q15_t)0x6563,
	(q15_t)0xB140, (q15_t)0x64E8, (q15_t)0xB0A1, (q15_t)0x646C,
	(q15_t)0xB004, (q15_t)0x63EF, (q15_t)0xAF68, (q15_t)0x6371,
	(q15_t)0xAECC, (q15_t)0x62F2, (q15_t)0xAE31, (q15_t)0x6271,
	(q15_t)0xAD96, (q15_t)0x61F1, (q15_t)0xACFD, (q15_t)0x616F,
	(q15_t)0xAC64, (q15_t)0x60EC, (q15_t)0xABCC, (q15_t)0x6068,
	(q15_t)0xAB35, (q15_t)0x5FE3, (q15_t)0xAA9F, (q15_t)0x5F5E,
	(q15_t)0xAA0A, (q15_t)0x5ED7, (q15_t)0xA975, (q15_t)0x5E50,
	(q15_t)0xA8E2, (q15_t)0x5DC7, (q15_t)0xA84F, (q15_t)0x5D3E,
	(q15_t)0xA7BD, (q15_t)0x5CB4, (q15_t)0xA72B, (q15_t)0x5C29,
	(q15_t)0xA69B, (q15_t)0x5B9D, (q15_t)0xA60C, (q15_t)0x5B10,
	(q15_t)0xA57D, (q15_t)0x5A82, (q15_t)0xA4EF, (q15_t)0x59F3,
	(q15_t)0xA462, (q15_t)0x5964, (q15_t)0xA3D6, (q15_t)0x58D4,
	(q15_t)0xA34B, (q15_t)0x5842, (q15_t)0xA2C1, (q15_t)0x57B0,
	(q15_t)0xA238, (q15_t)0x571D, (q15_t)0xA1AF, (q15_t)0x568A,
	(q15_t)0xA128, (q15_t)0x55F5, (q15_t)0xA0A1, (q15_t)0x5560,
	(q15_t)0xA01C, (q15_t)0x54CA, (q15_t)0x9F97, (q15_t)0x5433,
	(q15_t)0x9F13, (q15_t)0x539B, (q15_t)0x9E90, (q15_t)0x5302,
	(q15_t)0x9E0E, (q15_t)0x5269, (q15_t)0x9D8E, (q15_t)0x51CE,
	(q15_t)0x9D0D, (q15_t)0x5133, (q15_t)0x9C8E, (q15_t)0x5097,
	(q15_t)0x9C10, (q15_t)0x4FFB, (q15_t)0x9B93, (q15_t)0x4F5E,
	(q15_t)0x9B17, (q15_t)0x4EBF, (q15_t)0x9A9C, (q15_t)0x4E21,
	(q15_t)0x9A22, (q15_t)0x4D81, (q15_t)0x99A8, (q15_t)0x4CE1,
	(q15_t)0x9930, (q15_t)0x4C3F, (q15_t)0x98B9, (q15_t)0x4B9E,
	(q15_t)0x9842, (q15_t)0x4AFB, (q15_t)0x97CD, (q15_t)0x4A58,
	(q15_t)0x9759, (q15_t)0x49B4, (q15_t)0x96E6, (q15_t)0x490F,
	(q15_t)0x9673, (q15_t)0x4869, (q15_t)0x9602, (q15_t)0x47C3,
	(q15_t)0x9592, (q15_t)0x471C, (q15_t)0x9523, (q15_t)0x4675,
	(q15_t)0x94B5, (q15_t)0x45CD, (q15_t)0x9447, (q15_t)0x4524,
	(q15_t)0x93DB, (q15_t)0x447A, (q15_t)0x9370, (q15_t)0x43D0,
	(q15_t)0x9306, (q15_t)0x4325, (q15_t)0x929D, (q15_t)0x427A,
	(q15_t)0x9235, (q15_t)0x41CE, (q15_t)0x91CF, (q15_t)0x4121,
	(q15_t)0x9169, (q15_t)0x4073, (q15_t)0x9104, (q15_t)0x3FC5,
	(q15_t)0x90A0, (q15_t)0x3F17, (q15_t)0x903E, (q15_t)0x3E68,
	(q15_t)0x8FDC, (q15_t)0x3DB8, (q15_t)0x8F7C, (q15_t)0x3D07,
	(q15_t)0x8F1D, (q15_t)0x3C56, (q15_t)0x8EBE, (q15_t)0x3BA5,
	(q15_t)0x8E61, (q15_t)0x3AF2, (q15_t)0x8E05, (q15_t)0x3A40,
	(q15_t)0x8DAA, (q15_t)0x398C, (q15_t)0x8D50, (q15_t)0x38D8,
	(q15_t)0x8CF8, (q15_t)0x3824, (q15_t)0x8CA0, (q15_t)0x376F,
	(q15_t)0x8C4A, (q15_t)0x36BA, (q15_t)0x8BF4, (q15_t)0x3604,
	(q15_t)0x8BA0, (q15_t)0x354D, (q15_t)0x8B4D, (q15_t)0x3496,
	(q15_t)0x8AFB, (q15_t)0x33DE, (q15_t)0x8AAA, (q15_t)0x3326,
	(q15_t)0x8A5A, (q15_t)0x326E, (q15_t)0x8A0B, (q15_t)0x31B5,
	(q15_t)0x89BE, (q15_t)0x30FB, (q15_t)0x8971, (q15_t)0x3041,
	(q15_t)0x8926, (q15_t)0x2F87, (q15_t)0x88DC, (q15_t)0x2ECC,
	(q15_t)0x8893, (q15_t)0x2E11, (q15_t)0x884B, (q15_t)0x2D55,
	(q15_t)0x8805, (q15_t)0x2C98, (q15_t)0x87BF, (q15_t)0x2BDC,
	(q15_t)0x877B, (q15_t)0x2B1F, (q15_t)0x8738, (q15_t)0x2A61,
	(q15_t)0x86F6, (q15_t)0x29A3, (q15_t)0x86B5, (q15_t)0x28E5,
	(q15_t)0x8675, (q15_t)0x2826, (q15_t)0x8637, (q15_t)0x2767,
	(q15_t)0x85FA, (q15_t)0x26A8, (q15_t)0x85BD, (q15_t)0x25E8,
	(q15_t)0x8582, (q15_t)0x2528, (q15_t)0x8549, (q15_t)0x2467,
	(q15_t)0x8510, (q15_t)0x23A6, (q15_t)0x84D9, (q15_t)0x22E5,
	(q15_t)0x84A2, (q15_t)0x2223, (q15_t)0x846D, (q15_t)0x2161,
	(q15_t)0x843A, (q15_t)0x209F, (q15_t)0x8407, (q15_t)0x1FDC,
	(q15_t)0x83D6, (q15_t)0x1F19, (q15_t)0x83A5, (q15_t)0x1E56,
	(q15_t)0x8376, (q15_t)0x1D93, (q15_t)0x8348, (q15_t)0x1CCF,
	(q15_t)0x831C, (q15_t)0x1C0B, (q15_t)0x82F0, (q15_t)0x1B47,
	(q15_t)0x82C6, (q15_t)0x1A82, (q15_t)0x829D, (q15_t)0x19BD,
	(q15_t)0x8275, (q15_t)0x18F8, (q15_t)0x824F, (q15_t)0x1833,
	(q15_t)0x8229, (q15_t)0x176D, (q15_t)0x8205, (q15_t)0x16A8,
	(q15_t)0x81E2, (q15_t)0x15E2, (q15_t)0x81C0, (q15_t)0x151B,
	(q15_t)0x81A0, (q15_t)0x1455, (q15_t)0x8180, (q15_t)0x138E,
	(q15_t)0x8162, (q15_t)0x12C8, (q15_t)0x8145, (q15_t)0x1201,
	(q15_t)0x812A, (q15_t)0x1139, (q15_t)0x810F, (q15_t)0x1072,
	(q15_t)0x80F6, (q15_t)0x0FAB, (q15_t)0x80DE, (q15_t)0x0EE3,
	(q15_t)0x80C7, (q15_t)0x0E1B, (q15_t)0x80B2, (q15_t)0x0D53,
	(q15_t)0x809D, (q15_t)0x0C8B, (q15_t)0x808A, (q15_t)0x0BC3,
	(q15_t)0x8078, (q15_t)0x0AFB, (q15_t)0x8068, (q15_t)0x0A33,
	(q15_t)0x8058, (q15_t)0x096A, (q15_t)0x804A, (q15_t)0x08A2,
	(q15_t)0x803D, (q15_t)0x07D9, (q15_t)0x8031, (q15_t)0x0710,
	(q15_t)0x8027, (q15_t)0x0647, (q15_t)0x801E, (q15_t)0x057F,
	(q15_t)0x8016, (q15_t)0x04B6, (q15_t)0x800F, (q15_t)0x03ED,
	(q15_t)0x8009, (q15_t)0x0324, (q15_t)0x8005, (q15_t)0x025B,
	(q15_t)0x8002, (q15_t)0x0192, (q15_t)0x8000, (q15_t)0x00C9,
	(q15_t)0x8000, (q15_t)0x0000, (q15_t)0x8000, (q15_t)0xFF36,
	(q15_t)0x8002, (q15_t)0xFE6D, (q15_t)0x8005, (q15_t)0xFDA4,
	(q15_t)0x8009, (q15_t)0xFCDB, (q15_t)0x800F, (q15_t)0xFC12,
	(q15_t)0x8016, (q15_t)0xFB49, (q15_t)0x801E, (q15_t)0xFA80,
	(q15_t)0x8027, (q15_t)0xF9B8, (q15_t)0x8031, (q15_t)0xF8EF,
	(q15_t)0x803D, (q15_t)0xF826, (q15_t)0x804A, (q15_t)0xF75D,
	(q15_t)0x8058, (q15_t)0xF695, (q15_t)0x8068, (q15_t)0xF5CC,
	(q15_t)0x8078, (q15_t)0xF504, (q15_t)0x808A, (q15_t)0xF43C,
	(q15_t)0x809D, (q15_t)0xF374, (q15_t)0x80B2, (q15_t)0xF2AC,
	(q15_t)0x80C7, (q15_t)0xF1E4, (q15_t)0x80DE, (q15_t)0xF11C,
	(q15_t)0x80F6, (q15_t)0xF054, (q15_t)0x810F, (q15_t)0xEF8D,
	(q15_t)0x812A, (q15_t)0xEEC6, (q15_t)0x8145, (q15_t)0xEDFE,
	(q15_t)0x8162, (q15_t)0xED37, (q15_t)0x8180, (q15_t)0xEC71,
	(q15_t)0x81A0, (q15_t)0xEBAA, (q15_t)0x81C0, (q15_t)0xEAE4,
	(q15_t)0x81E2, (q15_t)0xEA1D, (q15_t)0x8205, (q15_t)0xE957,
	(q15_t)0x8229, (q15_t)0xE892, (q15_t)0x824F, (q15_t)0xE7CC,
	(q15_t)0x8275, (q15_t)0xE707, (q15_t)0x829D, (q15_t)0xE642,
	(q15_t)0x82C6, (q15_t)0xE57D, (q15_t)0x82F0, (q15_t)0xE4B8,
	(q15_t)0x831C, (q15_t)0xE3F4, (q15_t)0x8348, (q15_t)0xE330,
	(q15_t)0x8376, (q15_t)0xE26C, (q15_t)0x83A5, (q15_t)0xE1A9,
	(q15_t)0x83D6, (q15_t)0xE0E6, (q15_t)0x8407, (q15_t)0xE023,
	(q15_t)0x843A, (q15_t)0xDF60, (q15_t)0x846D, (q15_t)0xDE9E,
	(q15_t)0x84A2, (q15_t)0xDDDC, (q15_t)0x84D9, (q15_t)0xDD1A,
	(q15_t)0x8510, (q15_t)0xDC59, (q15_t)0x8549, (q15_t)0xDB98,
	(q15_t)0x8582, (q15_t)0xDAD7, (q15_t)0x85BD, (q15_t)0xDA17,
	(q15_t)0x85FA, (q15_t)0xD957, (q15_t)0x8637, (q15_t)0xD898,
	(q15_t)0x8675, (q15_t)0xD7D9, (q15_t)0x86B5, (q15_t)0xD71A,
	(q15_t)0x86F6, (q15_t)0xD65C, (q15_t)0x8738, (q15_t)0xD59E,
	(q15_t)0x877B, (q15_t)0xD4E0, (q15_t)0x87BF, (q15_t)0xD423,
	(q15_t)0x8805, (q15_t)0xD367, (q15_t)0x884B, (q15_t)0xD2AA,
	(q15_t)0x8893, (q15_t)0xD1EE, (q15_t)0x88DC, (q15_t)0xD133,
	(q15_t)0x8926, (q15_t)0xD078, (q15_t)0x8971, (q15_t)0xCFBE,
	(q15_t)0x89BE, (q15_t)0xCF04, (q15_t)0x8A0B, (q15_t)0xCE4A,
	(q15_t)0x8A5A, (q15_t)0xCD91, (q15_t)0x8AAA, (q15_t)0xCCD9,
	(q15_t)0x8AFB, (q15_t)0xCC21, (q15_t)0x8B4D, (q15_t)0xCB69,
	(q15_t)0x8BA0, (q15_t)0xCAB2, (q15_t)0x8BF4, (q15_t)0xC9FB,
	(q15_t)0x8C4A, (q15_t)0xC945, (q15_t)0x8CA0, (q15_t)0xC890,
	(q15_t)0x8CF8, (q15_t)0xC7DB, (q15_t)0x8D50, (q15_t)0xC727,
	(q15_t)0x8DAA, (q15_t)0xC673, (q15_t)0x8E05, (q15_t)0xC5BF,
	(q15_t)0x8E61, (q15_t)0xC50D, (q15_t)0x8EBE, (q15_t)0xC45A,
	(q15_t)0x8F1D, (q15_t)0xC3A9, (q15_t)0x8F7C, (q15_t)0xC2F8,
	(q15_t)0x8FDC, (q15_t)0xC247, (q15_t)0x903E, (q15_t)0xC197,
	(q15_t)0x90A0, (q15_t)0xC0E8, (q15_t)0x9104, (q15_t)0xC03A,
	(q15_t)0x9169, (q15_t)0xBF8C, (q15_t)0x91CF, (q15_t)0xBEDE,
	(q15_t)0x9235, (q15_t)0xBE31, (q15_t)0x929D, (q15_t)0xBD85,
	(q15_t)0x9306, (q15_t)0xBCDA, (q15_t)0x9370, (q15_t)0xBC2F,
	(q15_t)0x93DB, (q15_t)0xBB85, (q15_t)0x9447, (q15_t)0xBADB,
	(q15_t)0x94B5, (q15_t)0xBA32, (q15_t)0x9523, (q15_t)0xB98A,
	(q15_t)0x9592, (q15_t)0xB8E3, (q15_t)0x9602, (q15_t)0xB83C,
	(q15_t)0x9673, (q15_t)0xB796, (q15_t)0x96E6, (q15_t)0xB6F0,
	(q15_t)0x9759, (q15_t)0xB64B, (q15_t)0x97CD, (q15_t)0xB5A7,
	(q15_t)0x9842, (q15_t)0xB504, (q15_t)0x98B9, (q15_t)0xB461,
	(q15_t)0x9930, (q15_t)0xB3C0, (q15_t)0x99A8, (q15_t)0xB31E,
	(q15_t)0x9A22, (q15_t)0xB27E, (q15_t)0x9A9C, (q15_t)0xB1DE,
	(q15_t)0x9B17, (q15_t)0xB140, (q15_t)0x9B93, (q15_t)0xB0A1,
	(q15_t)0x9C10, (q15_t)0xB004, (q15_t)0x9C8E, (q15_t)0xAF68,
	(q15_t)0x9D0D, (q15_t)0xAECC, (q15_t)0x9D8E, (q15_t)0xAE31,
	(q15_t)0x9E0E, (q15_t)0xAD96, (q15_t)0x9E90, (q15_t)0xACFD,
	(q15_t)0x9F13, (q15_t)0xAC64, (q15_t)0x9F97, (q15_t)0xABCC,
	(q15_t)0xA01C, (q15_t)0xAB35, (q15_t)0xA0A1, (q15_t)0xAA9F,
	(q15_t)0xA128, (q15_t)0xAA0A, (q15_t)0xA1AF, (q15_t)0xA975,
	(q15_t)0xA238, (q15_t)0xA8E2, (q15_t)0xA2C1, (q15_t)0xA84F,
	(q15_t)0xA34B, (q15_t)0xA7BD, (q15_t)0xA3D6, (q15_t)0xA72B,
	(q15_t)0xA462, (q15_t)0xA69B, (q15_t)0xA4EF, (q15_t)0xA60C,
	(q15_t)0xA57D, (q15_t)0xA57D, (q15_t)0xA60C, (q15_t)0xA4EF,
	(q15_t)0xA69B, (q15_t)0xA462, (q15_t)0xA72B, (q15_t)0xA3D6,
	(q15_t)0xA7BD, (q15_t)0xA34B, (q15_t)0xA84F, (q15_t)0xA2C1,
	(q15_t)0xA8E2, (q15_t)0xA238, (q15_t)0xA975, (q15_t)0xA1AF,
	(q15_t)0xAA0A, (q15_t)0xA128, (q15_t)0xAA9F, (q15_t)0xA0A1,
	(q15_t)0xAB35, (q15_t)0xA01C, (q15_t)0xABCC, (q15_t)0x9F97,
	(q15_t)0xAC64, (q15_t)0x9F13, (q15_t)0xACFD, (q15_t)0x9E90,
	(q15_t)0xAD96, (q15_t)0x9E0E, (q15_t)0xAE31, (q15_t)0x9D8E,
	(q15_t)0xAECC, (q15_t)0x9D0D, (q15_t)0xAF68, (q15_t)0x9C8E,
	(q15_t)0xB004, (q15_t)0x9C10, (q15_t)0xB0A1, (q15_t)0x9B93,
	(q15_t)0xB140, (q15_t)0x9B17, (q15_t)0xB1DE, (q15_t)0x9A9C,
	(q15_t)0xB27E, (q15_t)0x9A22, (q15_t)0xB31E, (q15_t)0x99A8,
	(q15_t)0xB3C0, (q15_t)0x9930, (q15_t)0xB461, (q15_t)0x98B9,
	(q15_t)0xB504, (q15_t)0x9842, (q15_t)0xB5A7, (q15_t)0x97CD,
	(q15_t)0xB64B, (q15_t)0x9759, (q15_t)0xB6F0, (q15_t)0x96E6,
	(q15_t)0xB796, (q15_t)0x9673, (q15_t)0xB83C, (q15_t)0x9602,
	(q15_t)0xB8E3, (q15_t)0x9592, (q15_t)0xB98A, (q15_t)0x9523,
	(q15_t)0xBA32, (q15_t)0x94B5, (q15_t)0xBADB, (q15_t)0x9447,
	(q15_t)0xBB85, (q15_t)0x93DB, (q15_t)0xBC2F, (q15_t)0x9370,
	(q15_t)0xBCDA, (q15_t)0x9306, (q15_t)0xBD85, (q15_t)0x929D,
	(q15_t)0xBE31, (q15_t)0x9235, (q15_t)0xBEDE, (q15_t)0x91CF,
	(q15_t)0xBF8C, (q15_t)0x9169, (q15_t)0xC03A, (q15_t)0x9104,
	(q15_t)0xC0E8, (q15_t)0x90A0, (q15_t)0xC197, (q15_t)0x903E,
	(q15_t)0xC247, (q15_t)0x8FDC, (q15_t)0xC2F8, (q15_t)0x8F7C,
	(q15_t)0xC3A9, (q15_t)0x8F1D, (q15_t)0xC45A, (q15_t)0x8EBE,
	(q15_t)0xC50D, (q15_t)0x8E61, (q15_t)0xC5BF, (q15_t)0x8E05,
	(q15_t)0xC673, (q15_t)0x8DAA, (q15_t)0xC727, (q15_t)0x8D50,
	(q15_t)0xC7DB, (q15_t)0x8CF8, (q15_t)0xC890, (q15_t)0x8CA0,
	(q15_t)0xC945, (q15_t)0x8C4A, (q15_t)0xC9FB, (q15_t)0x8BF4,
	(q15_t)0xCAB2, (q15_t)0x8BA0, (q15_t)0xCB69, (q15_t)0x8B4D,
	(q15_t)0xCC21, (q15_t)0x8AFB, (q15_t)0xCCD9, (q15_t)0x8AAA,
	(q15_t)0xCD91, (q15_t)0x8A5A, (q15_t)0xCE4A, (q15_t)0x8A0B,
	(q15_t)0xCF04, (q15_t)0x89BE, (q15_t)0xCFBE, (q15_t)0x8971,
	(q15_t)0xD078, (q15_t)0x8926, (q15_t)0xD133, (q15_t)0x88DC,
	(q15_t)0xD1EE, (q15_t)0x8893, (q15_t)0xD2AA, (q15_t)0x884B,
	(q15_t)0xD367, (q15_t)0x8805, (q15_t)0xD423, (q15_t)0x87BF,
	(q15_t)0xD4E0, (q15_t)0x877B, (q15_t)0xD59E, (q15_t)0x8738,
	(q15_t)0xD65C, (q15_t)0x86F6, (q15_t)0xD71A, (q15_t)0x86B5,
	(q15_t)0xD7D9, (q15_t)0x8675, (q15_t)0xD898, (q15_t)0x8637,
	(q15_t)0xD957, (q15_t)0x85FA, (q15_t)0xDA17, (q15_t)0x85BD,
	(q15_t)0xDAD7, (q15_t)0x8582, (q15_t)0xDB98, (q15_t)0x8549,
	(q15_t)0xDC59, (q15_t)0x8510, (q15_t)0xDD1A, (q15_t)0x84D9,
	(q15_t)0xDDDC, (q15_t)0x84A2, (q15_t)0xDE9E, (q15_t)0x846D,
	(q15_t)0xDF60, (q15_t)0x843A, (q15_t)0xE023, (q15_t)0x8407,
	(q15_t)0xE0E6, (q15_t)0x83D6, (q15_t)0xE1A9, (q15_t)0x83A5,
	(q15_t)0xE26C, (q15_t)0x8376, (q15_t)0xE330, (q15_t)0x8348,
	(q15_t)0xE3F4, (q15_t)0x831C, (q15_t)0xE4B8, (q15_t)0x82F0,
	(q15_t)0xE57D, (q15_t)0x82C6, (q15_t)0xE642, (q15_t)0x829D,
	(q15_t)0xE707, (q15_t)0x8275, (q15_t)0xE7CC, (q15_t)0x824F,
	(q15_t)0xE892, (q15_t)0x8229, (q15_t)0xE957, (q15_t)0x8205,
	(q15_t)0xEA1D, (q15_t)0x81E2, (q15_t)0xEAE4, (q15_t)0x81C0,
	(q15_t)0xEBAA, (q15_t)0x81A0, (q15_t)0xEC71, (q15_t)0x8180,
	(q15_t)0xED37, (q15_t)0x8162, (q15_t)0xEDFE, (q15_t)0x8145,
	(q15_t)0xEEC6, (q15_t)0x812A, (q15_t)0xEF8D, (q15_t)0x810F,
	(q15_t)0xF054, (q15_t)0x80F6, (q15_t)0xF11C, (q15_t)0x80DE,
	(q15_t)0xF1E4, (q15_t)0x80C7, (q15_t)0xF2AC, (q15_t)0x80B2,
	(q15_t)0xF374, (q15_t)0x809D, (q15_t)0xF43C, (q15_t)0x808A,
	(q15_t)0xF504, (q15_t)0x8078, (q15_t)0xF5CC, (q15_t)0x8068,
	(q15_t)0xF695, (q15_t)0x8058, (q15_t)0xF75D, (q15_t)0x804A,
	(q15_t)0xF826, (q15_t)0x803D, (q15_t)0xF8EF, (q15_t)0x8031,
	(q15_t)0xF9B8, (q15_t)0x8027, (q15_t)0xFA80, (q15_t)0x801E,
	(q15_t)0xFB49, (q15_t)0x8016, (q15_t)0xFC12, (q15_t)0x800F,
	(q15_t)0xFCDB, (q15_t)0x8009, (q15_t)0xFDA4, (q15_t)0x8005,
	(q15_t)0xFE6D, (q15_t)0x8002, (q15_t)0xFF36, (q15_t)0x8000
};

/**
* \par
* Example code for q15 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefq15[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefq15[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 2048	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to q15(Fixed point 1.15):
*	round(twiddleCoefq15(i) * pow(2, 15))
*
*/
const q15_t twiddleCoef_2048_q15[3072] = {
	(q15_t)0x7FFF, (q15_t)0x0000, (q15_t)0x7FFF, (q15_t)0x0064,
	(q15_t)0x7FFF, (q15_t)0x00C9, (q15_t)0x7FFE, (q15_t)0x012D,
	(q15_t)0x7FFD, (q15_t)0x0192, (q15_t)0x7FFC, (q15_t)0x01F6,
	(q15_t)0x7FFA, (q15_t)0x025B, (q15_t)0x7FF8, (q15_t)0x02BF,
	(q15_t)0x7FF6, (q15_t)0x0324, (q15_t)0x7FF3, (q15_t)0x0388,
	(q15_t)0x7FF0, (q15_t)0x03ED, (q15_t)0x7FED, (q15_t)0x0451,
	(q15_t)0x7FE9, (q15_t)0x04B6, (q15_t)0x7FE5, (q15_t)0x051A,
	(q15_t)0x7FE1, (q15_t)0x057F, (q15_t)0x7FDD, (q15_t)0x05E3,
	(q15_t)0x7FD8, (q15_t)0x0647, (q15_t)0x7FD3, (q15_t)0x06AC,
	(q15_t)0x7FCE, (q15_t)0x0710, (q15_t)0x7FC8, (q15_t)0x0775,
	(q15_t)0x7FC2, (q15_t)0x07D9, (q15_t)0x7FBC, (q15_t)0x083D,
	(q15_t)0x7FB5, (q15_t)0x08A2, (q15_t)0x7FAE, (q15_t)0x0906,
	(q15_t)0x7FA7, (q15_t)0x096A, (q15_t)0x7F9F, (q15_t)0x09CE,
	(q15_t)0x7F97, (q15_t)0x0A33, (q15_t)0x7F8F, (q15_t)0x0A97,
	(q15_t)0x7F87, (q15_t)0x0AFB, (q15_t)0x7F7E, (q15_t)0x0B5F,
	(q15_t)0x7F75, (q15_t)0x0BC3, (q15_t)0x7F6B, (q15_t)0x0C27,
	(q15_t)0x7F62, (q15_t)0x0C8B, (q15_t)0x7F58, (q15_t)0x0CEF,
	(q15_t)0x7F4D, (q15_t)0x0D53, (q15_t)0x7F43, (q15_t)0x0DB7,
	(q15_t)0x7F38, (q15_t)0x0E1B, (q15_t)0x7F2D, (q15_t)0x0E7F,
	(q15_t)0x7F21, (q15_t)0x0EE3, (q15_t)0x7F15, (q15_t)0x0F47,
	(q15_t)0x7F09, (q15_t)0x0FAB, (q15_t)0x7EFD, (q15_t)0x100E,
	(q15_t)0x7EF0, (q15_t)0x1072, (q15_t)0x7EE3, (q15_t)0x10D6,
	(q15_t)0x7ED5, (q15_t)0x1139, (q15_t)0x7EC8, (q15_t)0x119D,
	(q15_t)0x7EBA, (q15_t)0x1201, (q15_t)0x7EAB, (q15_t)0x1264,
	(q15_t)0x7E9D, (q15_t)0x12C8, (q15_t)0x7E8E, (q15_t)0x132B,
	(q15_t)0x7E7F, (q15_t)0x138E, (q15_t)0x7E6F, (q15_t)0x13F2,
	(q15_t)0x7E5F, (q15_t)0x1455, (q15_t)0x7E4F, (q15_t)0x14B8,
	(q15_t)0x7E3F, (q15_t)0x151B, (q15_t)0x7E2E, (q15_t)0x157F,
	(q15_t)0x7E1D, (q15_t)0x15E2, (q15_t)0x7E0C, (q15_t)0x1645,
	(q15_t)0x7DFA, (q15_t)0x16A8, (q15_t)0x7DE8, (q15_t)0x170A,
	(q15_t)0x7DD6, (q15_t)0x176D, (q15_t)0x7DC3, (q15_t)0x17D0,
	(q15_t)0x7DB0, (q15_t)0x1833, (q15_t)0x7D9D, (q15_t)0x1896,
	(q15_t)0x7D8A, (q15_t)0x18F8, (q15_t)0x7D76, (q15_t)0x195B,
	(q15_t)0x7D62, (q15_t)0x19BD, (q15_t)0x7D4E, (q15_t)0x1A20,
	(q15_t)0x7D39, (q15_t)0x1A82, (q15_t)0x7D24, (q15_t)0x1AE4,
	(q15_t)0x7D0F, (q15_t)0x1B47, (q15_t)0x7CF9, (q15_t)0x1BA9,
	(q15_t)0x7CE3, (q15_t)0x1C0B, (q15_t)0x7CCD, (q15_t)0x1C6D,
	(q15_t)0x7CB7, (q15_t)0x1CCF, (q15_t)0x7CA0, (q15_t)0x1D31,
	(q15_t)0x7C89, (q15_t)0x1D93, (q15_t)0x7C71, (q15_t)0x1DF5,
	(q15_t)0x7C5A, (q15_t)0x1E56, (q15_t)0x7C42, (q15_t)0x1EB8,
	(q15_t)0x7C29, (q15_t)0x1F19, (q15_t)0x7C11, (q15_t)0x1F7B,
	(q15_t)0x7BF8, (q15_t)0x1FDC, (q15_t)0x7BDF, (q15_t)0x203E,
	(q15_t)0x7BC5, (q15_t)0x209F, (q15_t)0x7BAC, (q15_t)0x2100,
	(q15_t)0x7B92, (q15_t)0x2161, (q15_t)0x7B77, (q15_t)0x21C2,
	(q15_t)0x7B5D, (q15_t)0x2223, (q15_t)0x7B42, (q15_t)0x2284,
	(q15_t)0x7B26, (q15_t)0x22E5, (q15_t)0x7B0B, (q15_t)0x2345,
	(q15_t)0x7AEF, (q15_t)0x23A6, (q15_t)0x7AD3, (q15_t)0x2407,
	(q15_t)0x7AB6, (q15_t)0x2467, (q15_t)0x7A9A, (q15_t)0x24C7,
	(q15_t)0x7A7D, (q15_t)0x2528, (q15_t)0x7A5F, (q15_t)0x2588,
	(q15_t)0x7A42, (q15_t)0x25E8, (q15_t)0x7A24, (q15_t)0x2648,
	(q15_t)0x7A05, (q15_t)0x26A8, (q15_t)0x79E7, (q15_t)0x2707,
	(q15_t)0x79C8, (q15_t)0x2767, (q15_t)0x79A9, (q15_t)0x27C7,
	(q15_t)0x798A, (q15_t)0x2826, (q15_t)0x796A, (q15_t)0x2886,
	(q15_t)0x794A, (q15_t)0x28E5, (q15_t)0x792A, (q15_t)0x2944,
	(q15_t)0x7909, (q15_t)0x29A3, (q15_t)0x78E8, (q15_t)0x2A02,
	(q15_t)0x78C7, (q15_t)0x2A61, (q15_t)0x78A6, (q15_t)0x2AC0,
	(q15_t)0x7884, (q15_t)0x2B1F, (q15_t)0x7862, (q15_t)0x2B7D,
	(q15_t)0x7840, (q15_t)0x2BDC, (q15_t)0x781D, (q15_t)0x2C3A,
	(q15_t)0x77FA, (q15_t)0x2C98, (q15_t)0x77D7, (q15_t)0x2CF7,
	(q15_t)0x77B4, (q15_t)0x2D55, (q15_t)0x7790, (q15_t)0x2DB3,
	(q15_t)0x776C, (q15_t)0x2E11, (q15_t)0x7747, (q15_t)0x2E6E,
	(q15_t)0x7723, (q15_t)0x2ECC, (q15_t)0x76FE, (q15_t)0x2F29,
	(q15_t)0x76D9, (q15_t)0x2F87, (q15_t)0x76B3, (q15_t)0x2FE4,
	(q15_t)0x768E, (q15_t)0x3041, (q15_t)0x7668, (q15_t)0x309E,
	(q15_t)0x7641, (q15_t)0x30FB, (q15_t)0x761B, (q15_t)0x3158,
	(q15_t)0x75F4, (q15_t)0x31B5, (q15_t)0x75CC, (q15_t)0x3211,
	(q15_t)0x75A5, (q15_t)0x326E, (q15_t)0x757D, (q15_t)0x32CA,
	(q15_t)0x7555, (q15_t)0x3326, (q15_t)0x752D, (q15_t)0x3382,
	(q15_t)0x7504, (q15_t)0x33DE, (q15_t)0x74DB, (q15_t)0x343A,
	(q15_t)0x74B2, (q15_t)0x3496, (q15_t)0x7489, (q15_t)0x34F2,
	(q15_t)0x745F, (q15_t)0x354D, (q15_t)0x7435, (q15_t)0x35A8,
	(q15_t)0x740B, (q15_t)0x3604, (q15_t)0x73E0, (q15_t)0x365F,
	(q15_t)0x73B5, (q15_t)0x36BA, (q15_t)0x738A, (q15_t)0x3714,
	(q15_t)0x735F, (q15_t)0x376F, (q15_t)0x7333, (q15_t)0x37CA,
	(q15_t)0x7307, (q15_t)0x3824, (q15_t)0x72DB, (q15_t)0x387E,
	(q15_t)0x72AF, (q15_t)0x38D8, (q15_t)0x7282, (q15_t)0x3932,
	(q15_t)0x7255, (q15_t)0x398C, (q15_t)0x7227, (q15_t)0x39E6,
	(q15_t)0x71FA, (q15_t)0x3A40, (q15_t)0x71CC, (q15_t)0x3A99,
	(q15_t)0x719E, (q15_t)0x3AF2, (q15_t)0x716F, (q15_t)0x3B4C,
	(q15_t)0x7141, (q15_t)0x3BA5, (q15_t)0x7112, (q15_t)0x3BFD,
	(q15_t)0x70E2, (q15_t)0x3C56, (q15_t)0x70B3, (q15_t)0x3CAF,
	(q15_t)0x7083, (q15_t)0x3D07, (q15_t)0x7053, (q15_t)0x3D60,
	(q15_t)0x7023, (q15_t)0x3DB8, (q15_t)0x6FF2, (q15_t)0x3E10,
	(q15_t)0x6FC1, (q15_t)0x3E68, (q15_t)0x6F90, (q15_t)0x3EBF,
	(q15_t)0x6F5F, (q15_t)0x3F17, (q15_t)0x6F2D, (q15_t)0x3F6E,
	(q15_t)0x6EFB, (q15_t)0x3FC5, (q15_t)0x6EC9, (q15_t)0x401D,
	(q15_t)0x6E96, (q15_t)0x4073, (q15_t)0x6E63, (q15_t)0x40CA,
	(q15_t)0x6E30, (q15_t)0x4121, (q15_t)0x6DFD, (q15_t)0x4177,
	(q15_t)0x6DCA, (q15_t)0x41CE, (q15_t)0x6D96, (q15_t)0x4224,
	(q15_t)0x6D62, (q15_t)0x427A, (q15_t)0x6D2D, (q15_t)0x42D0,
	(q15_t)0x6CF9, (q15_t)0x4325, (q15_t)0x6CC4, (q15_t)0x437B,
	(q15_t)0x6C8F, (q15_t)0x43D0, (q15_t)0x6C59, (q15_t)0x4425,
	(q15_t)0x6C24, (q15_t)0x447A, (q15_t)0x6BEE, (q15_t)0x44CF,
	(q15_t)0x6BB8, (q15_t)0x4524, (q15_t)0x6B81, (q15_t)0x4578,
	(q15_t)0x6B4A, (q15_t)0x45CD, (q15_t)0x6B13, (q15_t)0x4621,
	(q15_t)0x6ADC, (q15_t)0x4675, (q15_t)0x6AA5, (q15_t)0x46C9,
	(q15_t)0x6A6D, (q15_t)0x471C, (q15_t)0x6A35, (q15_t)0x4770,
	(q15_t)0x69FD, (q15_t)0x47C3, (q15_t)0x69C4, (q15_t)0x4816,
	(q15_t)0x698C, (q15_t)0x4869, (q15_t)0x6953, (q15_t)0x48BC,
	(q15_t)0x6919, (q15_t)0x490F, (q15_t)0x68E0, (q15_t)0x4961,
	(q15_t)0x68A6, (q15_t)0x49B4, (q15_t)0x686C, (q15_t)0x4A06,
	(q15_t)0x6832, (q15_t)0x4A58, (q15_t)0x67F7, (q15_t)0x4AA9,
	(q15_t)0x67BD, (q15_t)0x4AFB, (q15_t)0x6782, (q15_t)0x4B4C,
	(q15_t)0x6746, (q15_t)0x4B9E, (q15_t)0x670B, (q15_t)0x4BEF,
	(q15_t)0x66CF, (q15_t)0x4C3F, (q15_t)0x6693, (q15_t)0x4C90,
	(q15_t)0x6657, (q15_t)0x4CE1, (q15_t)0x661A, (q15_t)0x4D31,
	(q15_t)0x65DD, (q15_t)0x4D81, (q15_t)0x65A0, (q15_t)0x4DD1,
	(q15_t)0x6563, (q15_t)0x4E21, (q15_t)0x6526, (q15_t)0x4E70,
	(q15_t)0x64E8, (q15_t)0x4EBF, (q15_t)0x64AA, (q15_t)0x4F0F,
	(q15_t)0x646C, (q15_t)0x4F5E, (q15_t)0x642D, (q15_t)0x4FAC,
	(q15_t)0x63EF, (q15_t)0x4FFB, (q15_t)0x63B0, (q15_t)0x5049,
	(q15_t)0x6371, (q15_t)0x5097, (q15_t)0x6331, (q15_t)0x50E5,
	(q15_t)0x62F2, (q15_t)0x5133, (q15_t)0x62B2, (q15_t)0x5181,
	(q15_t)0x6271, (q15_t)0x51CE, (q15_t)0x6231, (q15_t)0x521C,
	(q15_t)0x61F1, (q15_t)0x5269, (q15_t)0x61B0, (q15_t)0x52B5,
	(q15_t)0x616F, (q15_t)0x5302, (q15_t)0x612D, (q15_t)0x534E,
	(q15_t)0x60EC, (q15_t)0x539B, (q15_t)0x60AA, (q15_t)0x53E7,
	(q15_t)0x6068, (q15_t)0x5433, (q15_t)0x6026, (q15_t)0x547E,
	(q15_t)0x5FE3, (q15_t)0x54CA, (q15_t)0x5FA0, (q15_t)0x5515,
	(q15_t)0x5F5E, (q15_t)0x5560, (q15_t)0x5F1A, (q15_t)0x55AB,
	(q15_t)0x5ED7, (q15_t)0x55F5, (q15_t)0x5E93, (q15_t)0x5640,
	(q15_t)0x5E50, (q15_t)0x568A, (q15_t)0x5E0B, (q15_t)0x56D4,
	(q15_t)0x5DC7, (q15_t)0x571D, (q15_t)0x5D83, (q15_t)0x5767,
	(q15_t)0x5D3E, (q15_t)0x57B0, (q15_t)0x5CF9, (q15_t)0x57F9,
	(q15_t)0x5CB4, (q15_t)0x5842, (q15_t)0x5C6E, (q15_t)0x588B,
	(q15_t)0x5C29, (q15_t)0x58D4, (q15_t)0x5BE3, (q15_t)0x591C,
	(q15_t)0x5B9D, (q15_t)0x5964, (q15_t)0x5B56, (q15_t)0x59AC,
	(q15_t)0x5B10, (q15_t)0x59F3, (q15_t)0x5AC9, (q15_t)0x5A3B,
	(q15_t)0x5A82, (q15_t)0x5A82, (q15_t)0x5A3B, (q15_t)0x5AC9,
	(q15_t)0x59F3, (q15_t)0x5B10, (q15_t)0x59AC, (q15_t)0x5B56,
	(q15_t)0x5964, (q15_t)0x5B9D, (q15_t)0x591C, (q15_t)0x5BE3,
	(q15_t)0x58D4, (q15_t)0x5C29, (q15_t)0x588B, (q15_t)0x5C6E,
	(q15_t)0x5842, (q15_t)0x5CB4, (q15_t)0x57F9, (q15_t)0x5CF9,
	(q15_t)0x57B0, (q15_t)0x5D3E, (q15_t)0x5767, (q15_t)0x5D83,
	(q15_t)0x571D, (q15_t)0x5DC7, (q15_t)0x56D4, (q15_t)0x5E0B,
	(q15_t)0x568A, (q15_t)0x5E50, (q15_t)0x5640, (q15_t)0x5E93,
	(q15_t)0x55F5, (q15_t)0x5ED7, (q15_t)0x55AB, (q15_t)0x5F1A,
	(q15_t)0x5560, (q15_t)0x5F5E, (q15_t)0x5515, (q15_t)0x5FA0,
	(q15_t)0x54CA, (q15_t)0x5FE3, (q15_t)0x547E, (q15_t)0x6026,
	(q15_t)0x5433, (q15_t)0x6068, (q15_t)0x53E7, (q15_t)0x60AA,
	(q15_t)0x539B, (q15_t)0x60EC, (q15_t)0x534E, (q15_t)0x612D,
	(q15_t)0x5302, (q15_t)0x616F, (q15_t)0x52B5, (q15_t)0x61B0,
	(q15_t)0x5269, (q15_t)0x61F1, (q15_t)0x521C, (q15_t)0x6231,
	(q15_t)0x51CE, (q15_t)0x6271, (q15_t)0x5181, (q15_t)0x62B2,
	(q15_t)0x5133, (q15_t)0x62F2, (q15_t)0x50E5, (q15_t)0x6331,
	(q15_t)0x5097, (q15_t)0x6371, (q15_t)0x5049, (q15_t)0x63B0,
	(q15_t)0x4FFB, (q15_t)0x63EF, (q15_t)0x4FAC, (q15_t)0x642D,
	(q15_t)0x4F5E, (q15_t)0x646C, (q15_t)0x4F0F, (q15_t)0x64AA,
	(q15_t)0x4EBF, (q15_t)0x64E8, (q15_t)0x4E70, (q15_t)0x6526,
	(q15_t)0x4E21, (q15_t)0x6563, (q15_t)0x4DD1, (q15_t)0x65A0,
	(q15_t)0x4D81, (q15_t)0x65DD, (q15_t)0x4D31, (q15_t)0x661A,
	(q15_t)0x4CE1, (q15_t)0x6657, (q15_t)0x4C90, (q15_t)0x6693,
	(q15_t)0x4C3F, (q15_t)0x66CF, (q15_t)0x4BEF, (q15_t)0x670B,
	(q15_t)0x4B9E, (q15_t)0x6746, (q15_t)0x4B4C, (q15_t)0x6782,
	(q15_t)0x4AFB, (q15_t)0x67BD, (q15_t)0x4AA9, (q15_t)0x67F7,
	(q15_t)0x4A58, (q15_t)0x6832, (q15_t)0x4A06, (q15_t)0x686C,
	(q15_t)0x49B4, (q15_t)0x68A6, (q15_t)0x4961, (q15_t)0x68E0,
	(q15_t)0x490F, (q15_t)0x6919, (q15_t)0x48BC, (q15_t)0x6953,
	(q15_t)0x4869, (q15_t)0x698C, (q15_t)0x4816, (q15_t)0x69C4,
	(q15_t)0x47C3, (q15_t)0x69FD, (q15_t)0x4770, (q15_t)0x6A35,
	(q15_t)0x471C, (q15_t)0x6A6D, (q15_t)0x46C9, (q15_t)0x6AA5,
	(q15_t)0x4675, (q15_t)0x6ADC, (q15_t)0x4621, (q15_t)0x6B13,
	(q15_t)0x45CD, (q15_t)0x6B4A, (q15_t)0x4578, (q15_t)0x6B81,
	(q15_t)0x4524, (q15_t)0x6BB8, (q15_t)0x44CF, (q15_t)0x6BEE,
	(q15_t)0x447A, (q15_t)0x6C24, (q15_t)0x4425, (q15_t)0x6C59,
	(q15_t)0x43D0, (q15_t)0x6C8F, (q15_t)0x437B, (q15_t)0x6CC4,
	(q15_t)0x4325, (q15_t)0x6CF9, (q15_t)0x42D0, (q15_t)0x6D2D,
	(q15_t)0x427A, (q15_t)0x6D62, (q15_t)0x4224, (q15_t)0x6D96,
	(q15_t)0x41CE, (q15_t)0x6DCA, (q15_t)0x4177, (q15_t)0x6DFD,
	(q15_t)0x4121, (q15_t)0x6E30, (q15_t)0x40CA, (q15_t)0x6E63,
	(q15_t)0x4073, (q15_t)0x6E96, (q15_t)0x401D, (q15_t)0x6EC9,
	(q15_t)0x3FC5, (q15_t)0x6EFB, (q15_t)0x3F6E, (q15_t)0x6F2D,
	(q15_t)0x3F17, (q15_t)0x6F5F, (q15_t)0x3EBF, (q15_t)0x6F90,
	(q15_t)0x3E68, (q15_t)0x6FC1, (q15_t)0x3E10, (q15_t)0x6FF2,
	(q15_t)0x3DB8, (q15_t)0x7023, (q15_t)0x3D60, (q15_t)0x7053,
	(q15_t)0x3D07, (q15_t)0x7083, (q15_t)0x3CAF, (q15_t)0x70B3,
	(q15_t)0x3C56, (q15_t)0x70E2, (q15_t)0x3BFD, (q15_t)0x7112,
	(q15_t)0x3BA5, (q15_t)0x7141, (q15_t)0x3B4C, (q15_t)0x716F,
	(q15_t)0x3AF2, (q15_t)0x719E, (q15_t)0x3A99, (q15_t)0x71CC,
	(q15_t)0x3A40, (q15_t)0x71FA, (q15_t)0x39E6, (q15_t)0x7227,
	(q15_t)0x398C, (q15_t)0x7255, (q15_t)0x3932, (q15_t)0x7282,
	(q15_t)0x38D8, (q15_t)0x72AF, (q15_t)0x387E, (q15_t)0x72DB,
	(q15_t)0x3824, (q15_t)0x7307, (q15_t)0x37CA, (q15_t)0x7333,
	(q15_t)0x376F, (q15_t)0x735F, (q15_t)0x3714, (q15_t)0x738A,
	(q15_t)0x36BA, (q15_t)0x73B5, (q15_t)0x365F, (q15_t)0x73E0,
	(q15_t)0x3604, (q15_t)0x740B, (q15_t)0x35A8, (q15_t)0x7435,
	(q15_t)0x354D, (q15_t)0x745F, (q15_t)0x34F2, (q15_t)0x7489,
	(q15_t)0x3496, (q15_t)0x74B2, (q15_t)0x343A, (q15_t)0x74DB,
	(q15_t)0x33DE, (q15_t)0x7504, (q15_t)0x3382, (q15_t)0x752D,
	(q15_t)0x3326, (q15_t)0x7555, (q15_t)0x32CA, (q15_t)0x757D,
	(q15_t)0x326E, (q15_t)0x75A5, (q15_t)0x3211, (q15_t)0x75CC,
	(q15_t)0x31B5, (q15_t)0x75F4, (q15_t)0x3158, (q15_t)0x761B,
	(q15_t)0x30FB, (q15_t)0x7641, (q15_t)0x309E, (q15_t)0x7668,
	(q15_t)0x3041, (q15_t)0x768E, (q15_t)0x2FE4, (q15_t)0x76B3,
	(q15_t)0x2F87, (q15_t)0x76D9, (q15_t)0x2F29, (q15_t)0x76FE,
	(q15_t)0x2ECC, (q15_t)0x7723, (q15_t)0x2E6E, (q15_t)0x7747,
	(q15_t)0x2E11, (q15_t)0x776C, (q15_t)0x2DB3, (q15_t)0x7790,
	(q15_t)0x2D55, (q15_t)0x77B4, (q15_t)0x2CF7, (q15_t)0x77D7,
	(q15_t)0x2C98, (q15_t)0x77FA, (q15_t)0x2C3A, (q15_t)0x781D,
	(q15_t)0x2BDC, (q15_t)0x7840, (q15_t)0x2B7D, (q15_t)0x7862,
	(q15_t)0x2B1F, (q15_t)0x7884, (q15_t)0x2AC0, (q15_t)0x78A6,
	(q15_t)0x2A61, (q15_t)0x78C7, (q15_t)0x2A02, (q15_t)0x78E8,
	(q15_t)0x29A3, (q15_t)0x7909, (q15_t)0x2944, (q15_t)0x792A,
	(q15_t)0x28E5, (q15_t)0x794A, (q15_t)0x2886, (q15_t)0x796A,
	(q15_t)0x2826, (q15_t)0x798A, (q15_t)0x27C7, (q15_t)0x79A9,
	(q15_t)0x2767, (q15_t)0x79C8, (q15_t)0x2707, (q15_t)0x79E7,
	(q15_t)0x26A8, (q15_t)0x7A05, (q15_t)0x2648, (q15_t)0x7A24,
	(q15_t)0x25E8, (q15_t)0x7A42, (q15_t)0x2588, (q15_t)0x7A5F,
	(q15_t)0x2528, (q15_t)0x7A7D, (q15_t)0x24C7, (q15_t)0x7A9A,
	(q15_t)0x2467, (q15_t)0x7AB6, (q15_t)0x2407, (q15_t)0x7AD3,
	(q15_t)0x23A6, (q15_t)0x7AEF, (q15_t)0x2345, (q15_t)0x7B0B,
	(q15_t)0x22E5, (q15_t)0x7B26, (q15_t)0x2284, (q15_t)0x7B42,
	(q15_t)0x2223, (q15_t)0x7B5D, (q15_t)0x21C2, (q15_t)0x7B77,
	(q15_t)0x2161, (q15_t)0x7B92, (q15_t)0x2100, (q15_t)0x7BAC,
	(q15_t)0x209F, (q15_t)0x7BC5, (q15_t)0x203E, (q15_t)0x7BDF,
	(q15_t)0x1FDC, (q15_t)0x7BF8, (q15_t)0x1F7B, (q15_t)0x7C11,
	(q15_t)0x1F19, (q15_t)0x7C29, (q15_t)0x1EB8, (q15_t)0x7C42,
	(q15_t)0x1E56, (q15_t)0x7C5A, (q15_t)0x1DF5, (q15_t)0x7C71,
	(q15_t)0x1D93, (q15_t)0x7C89, (q15_t)0x1D31, (q15_t)0x7CA0,
	(q15_t)0x1CCF, (q15_t)0x7CB7, (q15_t)0x1C6D, (q15_t)0x7CCD,
	(q15_t)0x1C0B, (q15_t)0x7CE3, (q15_t)0x1BA9, (q15_t)0x7CF9,
	(q15_t)0x1B47, (q15_t)0x7D0F, (q15_t)0x1AE4, (q15_t)0x7D24,
	(q15_t)0x1A82, (q15_t)0x7D39, (q15_t)0x1A20, (q15_t)0x7D4E,
	(q15_t)0x19BD, (q15_t)0x7D62, (q15_t)0x195B, (q15_t)0x7D76,
	(q15_t)0x18F8, (q15_t)0x7D8A, (q15_t)0x1896, (q15_t)0x7D9D,
	(q15_t)0x1833, (q15_t)0x7DB0, (q15_t)0x17D0, (q15_t)0x7DC3,
	(q15_t)0x176D, (q15_t)0x7DD6, (q15_t)0x170A, (q15_t)0x7DE8,
	(q15_t)0x16A8, (q15_t)0x7DFA, (q15_t)0x1645, (q15_t)0x7E0C,
	(q15_t)0x15E2, (q15_t)0x7E1D, (q15_t)0x157F, (q15_t)0x7E2E,
	(q15_t)0x151B, (q15_t)0x7E3F, (q15_t)0x14B8, (q15_t)0x7E4F,
	(q15_t)0x1455, (q15_t)0x7E5F, (q15_t)0x13F2, (q15_t)0x7E6F,
	(q15_t)0x138E, (q15_t)0x7E7F, (q15_t)0x132B, (q15_t)0x7E8E,
	(q15_t)0x12C8, (q15_t)0x7E9D, (q15_t)0x1264, (q15_t)0x7EAB,
	(q15_t)0x1201, (q15_t)0x7EBA, (q15_t)0x119D, (q15_t)0x7EC8,
	(q15_t)0x1139, (q15_t)0x7ED5, (q15_t)0x10D6, (q15_t)0x7EE3,
	(q15_t)0x1072, (q15_t)0x7EF0, (q15_t)0x100E, (q15_t)0x7EFD,
	(q15_t)0x0FAB, (q15_t)0x7F09, (q15_t)0x0F47, (q15_t)0x7F15,
	(q15_t)0x0EE3, (q15_t)0x7F21, (q15_t)0x0E7F, (q15_t)0x7F2D,
	(q15_t)0x0E1B, (q15_t)0x7F38, (q15_t)0x0DB7, (q15_t)0x7F43,
	(q15_t)0x0D53, (q15_t)0x7F4D, (q15_t)0x0CEF, (q15_t)0x7F58,
	(q15_t)0x0C8B, (q15_t)0x7F62, (q15_t)0x0C27, (q15_t)0x7F6B,
	(q15_t)0x0BC3, (q15_t)0x7F75, (q15_t)0x0B5F, (q15_t)0x7F7E,
	(q15_t)0x0AFB, (q15_t)0x7F87, (q15_t)0x0A97, (q15_t)0x7F8F,
	(q15_t)0x0A33, (q15_t)0x7F97, (q15_t)0x09CE, (q15_t)0x7F9F,
	(q15_t)0x096A, (q15_t)0x7FA7, (q15_t)0x0906, (q15_t)0x7FAE,
	(q15_t)0x08A2, (q15_t)0x7FB5, (q15_t)0x083D, (q15_t)0x7FBC,
	(q15_t)0x07D9, (q15_t)0x7FC2, (q15_t)0x0775, (q15_t)0x7FC8,
	(q15_t)0x0710, (q15_t)0x7FCE, (q15_t)0x06AC, (q15_t)0x7FD3,
	(q15_t)0x0647, (q15_t)0x7FD8, (q15_t)0x05E3, (q15_t)0x7FDD,
	(q15_t)0x057F, (q15_t)0x7FE1, (q15_t)0x051A, (q15_t)0x7FE5,
	(q15_t)0x04B6, (q15_t)0x7FE9, (q15_t)0x0451, (q15_t)0x7FED,
	(q15_t)0x03ED, (q15_t)0x7FF0, (q15_t)0x0388, (q15_t)0x7FF3,
	(q15_t)0x0324, (q15_t)0x7FF6, (q15_t)0x02BF, (q15_t)0x7FF8,
	(q15_t)0x025B, (q15_t)0x7FFA, (q15_t)0x01F6, (q15_t)0x7FFC,
	(q15_t)0x0192, (q15_t)0x7FFD, (q15_t)0x012D, (q15_t)0x7FFE,
	(q15_t)0x00C9, (q15_t)0x7FFF, (q15_t)0x0064, (q15_t)0x7FFF,
	(q15_t)0x0000, (q15_t)0x7FFF, (q15_t)0xFF9B, (q15_t)0x7FFF,
	(q15_t)0xFF36, (q15_t)0x7FFF, (q15_t)0xFED2, (q15_t)0x7FFE,
	(q15_t)0xFE6D, (q15_t)0x7FFD, (q15_t)0xFE09, (q15_t)0x7FFC,
	(q15_t)0xFDA4, (q15_t)0x7FFA, (q15_t)0xFD40, (q15_t)0x7FF8,
	(q15_t)0xFCDB, (q15_t)0x7FF6, (q15_t)0xFC77, (q15_t)0x7FF3,
	(q15_t)0xFC12, (q15_t)0x7FF0, (q15_t)0xFBAE, (q15_t)0x7FED,
	(q15_t)0xFB49, (q15_t)0x7FE9, (q15_t)0xFAE5, (q15_t)0x7FE5,
	(q15_t)0xFA80, (q15_t)0x7FE1, (q15_t)0xFA1C, (q15_t)0x7FDD,
	(q15_t)0xF9B8, (q15_t)0x7FD8, (q15_t)0xF953, (q15_t)0x7FD3,
	(q15_t)0xF8EF, (q15_t)0x7FCE, (q15_t)0xF88A, (q15_t)0x7FC8,
	(q15_t)0xF826, (q15_t)0x7FC2, (q15_t)0xF7C2, (q15_t)0x7FBC,
	(q15_t)0xF75D, (q15_t)0x7FB5, (q15_t)0xF6F9, (q15_t)0x7FAE,
	(q15_t)0xF695, (q15_t)0x7FA7, (q15_t)0xF631, (q15_t)0x7F9F,
	(q15_t)0xF5CC, (q15_t)0x7F97, (q15_t)0xF568, (q15_t)0x7F8F,
	(q15_t)0xF504, (q15_t)0x7F87, (q15_t)0xF4A0, (q15_t)0x7F7E,
	(q15_t)0xF43C, (q15_t)0x7F75, (q15_t)0xF3D8, (q15_t)0x7F6B,
	(q15_t)0xF374, (q15_t)0x7F62, (q15_t)0xF310, (q15_t)0x7F58,
	(q15_t)0xF2AC, (q15_t)0x7F4D, (q15_t)0xF248, (q15_t)0x7F43,
	(q15_t)0xF1E4, (q15_t)0x7F38, (q15_t)0xF180, (q15_t)0x7F2D,
	(q15_t)0xF11C, (q15_t)0x7F21, (q15_t)0xF0B8, (q15_t)0x7F15,
	(q15_t)0xF054, (q15_t)0x7F09, (q15_t)0xEFF1, (q15_t)0x7EFD,
	(q15_t)0xEF8D, (q15_t)0x7EF0, (q15_t)0xEF29, (q15_t)0x7EE3,
	(q15_t)0xEEC6, (q15_t)0x7ED5, (q15_t)0xEE62, (q15_t)0x7EC8,
	(q15_t)0xEDFE, (q15_t)0x7EBA, (q15_t)0xED9B, (q15_t)0x7EAB,
	(q15_t)0xED37, (q15_t)0x7E9D, (q15_t)0xECD4, (q15_t)0x7E8E,
	(q15_t)0xEC71, (q15_t)0x7E7F, (q15_t)0xEC0D, (q15_t)0x7E6F,
	(q15_t)0xEBAA, (q15_t)0x7E5F, (q15_t)0xEB47, (q15_t)0x7E4F,
	(q15_t)0xEAE4, (q15_t)0x7E3F, (q15_t)0xEA80, (q15_t)0x7E2E,
	(q15_t)0xEA1D, (q15_t)0x7E1D, (q15_t)0xE9BA, (q15_t)0x7E0C,
	(q15_t)0xE957, (q15_t)0x7DFA, (q15_t)0xE8F5, (q15_t)0x7DE8,
	(q15_t)0xE892, (q15_t)0x7DD6, (q15_t)0xE82F, (q15_t)0x7DC3,
	(q15_t)0xE7CC, (q15_t)0x7DB0, (q15_t)0xE769, (q15_t)0x7D9D,
	(q15_t)0xE707, (q15_t)0x7D8A, (q15_t)0xE6A4, (q15_t)0x7D76,
	(q15_t)0xE642, (q15_t)0x7D62, (q15_t)0xE5DF, (q15_t)0x7D4E,
	(q15_t)0xE57D, (q15_t)0x7D39, (q15_t)0xE51B, (q15_t)0x7D24,
	(q15_t)0xE4B8, (q15_t)0x7D0F, (q15_t)0xE456, (q15_t)0x7CF9,
	(q15_t)0xE3F4, (q15_t)0x7CE3, (q15_t)0xE392, (q15_t)0x7CCD,
	(q15_t)0xE330, (q15_t)0x7CB7, (q15_t)0xE2CE, (q15_t)0x7CA0,
	(q15_t)0xE26C, (q15_t)0x7C89, (q15_t)0xE20A, (q15_t)0x7C71,
	(q15_t)0xE1A9, (q15_t)0x7C5A, (q15_t)0xE147, (q15_t)0x7C42,
	(q15_t)0xE0E6, (q15_t)0x7C29, (q15_t)0xE084, (q15_t)0x7C11,
	(q15_t)0xE023, (q15_t)0x7BF8, (q15_t)0xDFC1, (q15_t)0x7BDF,
	(q15_t)0xDF60, (q15_t)0x7BC5, (q15_t)0xDEFF, (q15_t)0x7BAC,
	(q15_t)0xDE9E, (q15_t)0x7B92, (q15_t)0xDE3D, (q15_t)0x7B77,
	(q15_t)0xDDDC, (q15_t)0x7B5D, (q15_t)0xDD7B, (q15_t)0x7B42,
	(q15_t)0xDD1A, (q15_t)0x7B26, (q15_t)0xDCBA, (q15_t)0x7B0B,
	(q15_t)0xDC59, (q15_t)0x7AEF, (q15_t)0xDBF8, (q15_t)0x7AD3,
	(q15_t)0xDB98, (q15_t)0x7AB6, (q15_t)0xDB38, (q15_t)0x7A9A,
	(q15_t)0xDAD7, (q15_t)0x7A7D, (q15_t)0xDA77, (q15_t)0x7A5F,
	(q15_t)0xDA17, (q15_t)0x7A42, (q15_t)0xD9B7, (q15_t)0x7A24,
	(q15_t)0xD957, (q15_t)0x7A05, (q15_t)0xD8F8, (q15_t)0x79E7,
	(q15_t)0xD898, (q15_t)0x79C8, (q15_t)0xD838, (q15_t)0x79A9,
	(q15_t)0xD7D9, (q15_t)0x798A, (q15_t)0xD779, (q15_t)0x796A,
	(q15_t)0xD71A, (q15_t)0x794A, (q15_t)0xD6BB, (q15_t)0x792A,
	(q15_t)0xD65C, (q15_t)0x7909, (q15_t)0xD5FD, (q15_t)0x78E8,
	(q15_t)0xD59E, (q15_t)0x78C7, (q15_t)0xD53F, (q15_t)0x78A6,
	(q15_t)0xD4E0, (q15_t)0x7884, (q15_t)0xD482, (q15_t)0x7862,
	(q15_t)0xD423, (q15_t)0x7840, (q15_t)0xD3C5, (q15_t)0x781D,
	(q15_t)0xD367, (q15_t)0x77FA, (q15_t)0xD308, (q15_t)0x77D7,
	(q15_t)0xD2AA, (q15_t)0x77B4, (q15_t)0xD24C, (q15_t)0x7790,
	(q15_t)0xD1EE, (q15_t)0x776C, (q15_t)0xD191, (q15_t)0x7747,
	(q15_t)0xD133, (q15_t)0x7723, (q15_t)0xD0D6, (q15_t)0x76FE,
	(q15_t)0xD078, (q15_t)0x76D9, (q15_t)0xD01B, (q15_t)0x76B3,
	(q15_t)0xCFBE, (q15_t)0x768E, (q15_t)0xCF61, (q15_t)0x7668,
	(q15_t)0xCF04, (q15_t)0x7641, (q15_t)0xCEA7, (q15_t)0x761B,
	(q15_t)0xCE4A, (q15_t)0x75F4, (q15_t)0xCDEE, (q15_t)0x75CC,
	(q15_t)0xCD91, (q15_t)0x75A5, (q15_t)0xCD35, (q15_t)0x757D,
	(q15_t)0xCCD9, (q15_t)0x7555, (q15_t)0xCC7D, (q15_t)0x752D,
	(q15_t)0xCC21, (q15_t)0x7504, (q15_t)0xCBC5, (q15_t)0x74DB,
	(q15_t)0xCB69, (q15_t)0x74B2, (q15_t)0xCB0D, (q15_t)0x7489,
	(q15_t)0xCAB2, (q15_t)0x745F, (q15_t)0xCA57, (q15_t)0x7435,
	(q15_t)0xC9FB, (q15_t)0x740B, (q15_t)0xC9A0, (q15_t)0x73E0,
	(q15_t)0xC945, (q15_t)0x73B5, (q15_t)0xC8EB, (q15_t)0x738A,
	(q15_t)0xC890, (q15_t)0x735F, (q15_t)0xC835, (q15_t)0x7333,
	(q15_t)0xC7DB, (q15_t)0x7307, (q15_t)0xC781, (q15_t)0x72DB,
	(q15_t)0xC727, (q15_t)0x72AF, (q15_t)0xC6CD, (q15_t)0x7282,
	(q15_t)0xC673, (q15_t)0x7255, (q15_t)0xC619, (q15_t)0x7227,
	(q15_t)0xC5BF, (q15_t)0x71FA, (q15_t)0xC566, (q15_t)0x71CC,
	(q15_t)0xC50D, (q15_t)0x719E, (q15_t)0xC4B3, (q15_t)0x716F,
	(q15_t)0xC45A, (q15_t)0x7141, (q15_t)0xC402, (q15_t)0x7112,
	(q15_t)0xC3A9, (q15_t)0x70E2, (q15_t)0xC350, (q15_t)0x70B3,
	(q15_t)0xC2F8, (q15_t)0x7083, (q15_t)0xC29F, (q15_t)0x7053,
	(q15_t)0xC247, (q15_t)0x7023, (q15_t)0xC1EF, (q15_t)0x6FF2,
	(q15_t)0xC197, (q15_t)0x6FC1, (q15_t)0xC140, (q15_t)0x6F90,
	(q15_t)0xC0E8, (q15_t)0x6F5F, (q15_t)0xC091, (q15_t)0x6F2D,
	(q15_t)0xC03A, (q15_t)0x6EFB, (q15_t)0xBFE2, (q15_t)0x6EC9,
	(q15_t)0xBF8C, (q15_t)0x6E96, (q15_t)0xBF35, (q15_t)0x6E63,
	(q15_t)0xBEDE, (q15_t)0x6E30, (q15_t)0xBE88, (q15_t)0x6DFD,
	(q15_t)0xBE31, (q15_t)0x6DCA, (q15_t)0xBDDB, (q15_t)0x6D96,
	(q15_t)0xBD85, (q15_t)0x6D62, (q15_t)0xBD2F, (q15_t)0x6D2D,
	(q15_t)0xBCDA, (q15_t)0x6CF9, (q15_t)0xBC84, (q15_t)0x6CC4,
	(q15_t)0xBC2F, (q15_t)0x6C8F, (q15_t)0xBBDA, (q15_t)0x6C59,
	(q15_t)0xBB85, (q15_t)0x6C24, (q15_t)0xBB30, (q15_t)0x6BEE,
	(q15_t)0xBADB, (q15_t)0x6BB8, (q15_t)0xBA87, (q15_t)0x6B81,
	(q15_t)0xBA32, (q15_t)0x6B4A, (q15_t)0xB9DE, (q15_t)0x6B13,
	(q15_t)0xB98A, (q15_t)0x6ADC, (q15_t)0xB936, (q15_t)0x6AA5,
	(q15_t)0xB8E3, (q15_t)0x6A6D, (q15_t)0xB88F, (q15_t)0x6A35,
	(q15_t)0xB83C, (q15_t)0x69FD, (q15_t)0xB7E9, (q15_t)0x69C4,
	(q15_t)0xB796, (q15_t)0x698C, (q15_t)0xB743, (q15_t)0x6953,
	(q15_t)0xB6F0, (q15_t)0x6919, (q15_t)0xB69E, (q15_t)0x68E0,
	(q15_t)0xB64B, (q15_t)0x68A6, (q15_t)0xB5F9, (q15_t)0x686C,
	(q15_t)0xB5A7, (q15_t)0x6832, (q15_t)0xB556, (q15_t)0x67F7,
	(q15_t)0xB504, (q15_t)0x67BD, (q15_t)0xB4B3, (q15_t)0x6782,
	(q15_t)0xB461, (q15_t)0x6746, (q15_t)0xB410, (q15_t)0x670B,
	(q15_t)0xB3C0, (q15_t)0x66CF, (q15_t)0xB36F, (q15_t)0x6693,
	(q15_t)0xB31E, (q15_t)0x6657, (q15_t)0xB2CE, (q15_t)0x661A,
	(q15_t)0xB27E, (q15_t)0x65DD, (q15_t)0xB22E, (q15_t)0x65A0,
	(q15_t)0xB1DE, (q15_t)0x6563, (q15_t)0xB18F, (q15_t)0x6526,
	(q15_t)0xB140, (q15_t)0x64E8, (q15_t)0xB0F0, (q15_t)0x64AA,
	(q15_t)0xB0A1, (q15_t)0x646C, (q15_t)0xB053, (q15_t)0x642D,
	(q15_t)0xB004, (q15_t)0x63EF, (q15_t)0xAFB6, (q15_t)0x63B0,
	(q15_t)0xAF68, (q15_t)0x6371, (q15_t)0xAF1A, (q15_t)0x6331,
	(q15_t)0xAECC, (q15_t)0x62F2, (q15_t)0xAE7E, (q15_t)0x62B2,
	(q15_t)0xAE31, (q15_t)0x6271, (q15_t)0xADE3, (q15_t)0x6231,
	(q15_t)0xAD96, (q15_t)0x61F1, (q15_t)0xAD4A, (q15_t)0x61B0,
	(q15_t)0xACFD, (q15_t)0x616F, (q15_t)0xACB1, (q15_t)0x612D,
	(q15_t)0xAC64, (q15_t)0x60EC, (q15_t)0xAC18, (q15_t)0x60AA,
	(q15_t)0xABCC, (q15_t)0x6068, (q15_t)0xAB81, (q15_t)0x6026,
	(q15_t)0xAB35, (q15_t)0x5FE3, (q15_t)0xAAEA, (q15_t)0x5FA0,
	(q15_t)0xAA9F, (q15_t)0x5F5E, (q15_t)0xAA54, (q15_t)0x5F1A,
	(q15_t)0xAA0A, (q15_t)0x5ED7, (q15_t)0xA9BF, (q15_t)0x5E93,
	(q15_t)0xA975, (q15_t)0x5E50, (q15_t)0xA92B, (q15_t)0x5E0B,
	(q15_t)0xA8E2, (q15_t)0x5DC7, (q15_t)0xA898, (q15_t)0x5D83,
	(q15_t)0xA84F, (q15_t)0x5D3E, (q15_t)0xA806, (q15_t)0x5CF9,
	(q15_t)0xA7BD, (q15_t)0x5CB4, (q15_t)0xA774, (q15_t)0x5C6E,
	(q15_t)0xA72B, (q15_t)0x5C29, (q15_t)0xA6E3, (q15_t)0x5BE3,
	(q15_t)0xA69B, (q15_t)0x5B9D, (q15_t)0xA653, (q15_t)0x5B56,
	(q15_t)0xA60C, (q15_t)0x5B10, (q15_t)0xA5C4, (q15_t)0x5AC9,
	(q15_t)0xA57D, (q15_t)0x5A82, (q15_t)0xA536, (q15_t)0x5A3B,
	(q15_t)0xA4EF, (q15_t)0x59F3, (q15_t)0xA4A9, (q15_t)0x59AC,
	(q15_t)0xA462, (q15_t)0x5964, (q15_t)0xA41C, (q15_t)0x591C,
	(q15_t)0xA3D6, (q15_t)0x58D4, (q15_t)0xA391, (q15_t)0x588B,
	(q15_t)0xA34B, (q15_t)0x5842, (q15_t)0xA306, (q15_t)0x57F9,
	(q15_t)0xA2C1, (q15_t)0x57B0, (q15_t)0xA27C, (q15_t)0x5767,
	(q15_t)0xA238, (q15_t)0x571D, (q15_t)0xA1F4, (q15_t)0x56D4,
	(q15_t)0xA1AF, (q15_t)0x568A, (q15_t)0xA16C, (q15_t)0x5640,
	(q15_t)0xA128, (q15_t)0x55F5, (q15_t)0xA0E5, (q15_t)0x55AB,
	(q15_t)0xA0A1, (q15_t)0x5560, (q15_t)0xA05F, (q15_t)0x5515,
	(q15_t)0xA01C, (q15_t)0x54CA, (q15_t)0x9FD9, (q15_t)0x547E,
	(q15_t)0x9F97, (q15_t)0x5433, (q15_t)0x9F55, (q15_t)0x53E7,
	(q15_t)0x9F13, (q15_t)0x539B, (q15_t)0x9ED2, (q15_t)0x534E,
	(q15_t)0x9E90, (q15_t)0x5302, (q15_t)0x9E4F, (q15_t)0x52B5,
	(q15_t)0x9E0E, (q15_t)0x5269, (q15_t)0x9DCE, (q15_t)0x521C,
	(q15_t)0x9D8E, (q15_t)0x51CE, (q15_t)0x9D4D, (q15_t)0x5181,
	(q15_t)0x9D0D, (q15_t)0x5133, (q15_t)0x9CCE, (q15_t)0x50E5,
	(q15_t)0x9C8E, (q15_t)0x5097, (q15_t)0x9C4F, (q15_t)0x5049,
	(q15_t)0x9C10, (q15_t)0x4FFB, (q15_t)0x9BD2, (q15_t)0x4FAC,
	(q15_t)0x9B93, (q15_t)0x4F5E, (q15_t)0x9B55, (q15_t)0x4F0F,
	(q15_t)0x9B17, (q15_t)0x4EBF, (q15_t)0x9AD9, (q15_t)0x4E70,
	(q15_t)0x9A9C, (q15_t)0x4E21, (q15_t)0x9A5F, (q15_t)0x4DD1,
	(q15_t)0x9A22, (q15_t)0x4D81, (q15_t)0x99E5, (q15_t)0x4D31,
	(q15_t)0x99A8, (q15_t)0x4CE1, (q15_t)0x996C, (q15_t)0x4C90,
	(q15_t)0x9930, (q15_t)0x4C3F, (q15_t)0x98F4, (q15_t)0x4BEF,
	(q15_t)0x98B9, (q15_t)0x4B9E, (q15_t)0x987D, (q15_t)0x4B4C,
	(q15_t)0x9842, (q15_t)0x4AFB, (q15_t)0x9808, (q15_t)0x4AA9,
	(q15_t)0x97CD, (q15_t)0x4A58, (q15_t)0x9793, (q15_t)0x4A06,
	(q15_t)0x9759, (q15_t)0x49B4, (q15_t)0x971F, (q15_t)0x4961,
	(q15_t)0x96E6, (q15_t)0x490F, (q15_t)0x96AC, (q15_t)0x48BC,
	(q15_t)0x9673, (q15_t)0x4869, (q15_t)0x963B, (q15_t)0x4816,
	(q15_t)0x9602, (q15_t)0x47C3, (q15_t)0x95CA, (q15_t)0x4770,
	(q15_t)0x9592, (q15_t)0x471C, (q15_t)0x955A, (q15_t)0x46C9,
	(q15_t)0x9523, (q15_t)0x4675, (q15_t)0x94EC, (q15_t)0x4621,
	(q15_t)0x94B5, (q15_t)0x45CD, (q15_t)0x947E, (q15_t)0x4578,
	(q15_t)0x9447, (q15_t)0x4524, (q15_t)0x9411, (q15_t)0x44CF,
	(q15_t)0x93DB, (q15_t)0x447A, (q15_t)0x93A6, (q15_t)0x4425,
	(q15_t)0x9370, (q15_t)0x43D0, (q15_t)0x933B, (q15_t)0x437B,
	(q15_t)0x9306, (q15_t)0x4325, (q15_t)0x92D2, (q15_t)0x42D0,
	(q15_t)0x929D, (q15_t)0x427A, (q15_t)0x9269, (q15_t)0x4224,
	(q15_t)0x9235, (q15_t)0x41CE, (q15_t)0x9202, (q15_t)0x4177,
	(q15_t)0x91CF, (q15_t)0x4121, (q15_t)0x919C, (q15_t)0x40CA,
	(q15_t)0x9169, (q15_t)0x4073, (q15_t)0x9136, (q15_t)0x401D,
	(q15_t)0x9104, (q15_t)0x3FC5, (q15_t)0x90D2, (q15_t)0x3F6E,
	(q15_t)0x90A0, (q15_t)0x3F17, (q15_t)0x906F, (q15_t)0x3EBF,
	(q15_t)0x903E, (q15_t)0x3E68, (q15_t)0x900D, (q15_t)0x3E10,
	(q15_t)0x8FDC, (q15_t)0x3DB8, (q15_t)0x8FAC, (q15_t)0x3D60,
	(q15_t)0x8F7C, (q15_t)0x3D07, (q15_t)0x8F4C, (q15_t)0x3CAF,
	(q15_t)0x8F1D, (q15_t)0x3C56, (q15_t)0x8EED, (q15_t)0x3BFD,
	(q15_t)0x8EBE, (q15_t)0x3BA5, (q15_t)0x8E90, (q15_t)0x3B4C,
	(q15_t)0x8E61, (q15_t)0x3AF2, (q15_t)0x8E33, (q15_t)0x3A99,
	(q15_t)0x8E05, (q15_t)0x3A40, (q15_t)0x8DD8, (q15_t)0x39E6,
	(q15_t)0x8DAA, (q15_t)0x398C, (q15_t)0x8D7D, (q15_t)0x3932,
	(q15_t)0x8D50, (q15_t)0x38D8, (q15_t)0x8D24, (q15_t)0x387E,
	(q15_t)0x8CF8, (q15_t)0x3824, (q15_t)0x8CCC, (q15_t)0x37CA,
	(q15_t)0x8CA0, (q15_t)0x376F, (q15_t)0x8C75, (q15_t)0x3714,
	(q15_t)0x8C4A, (q15_t)0x36BA, (q15_t)0x8C1F, (q15_t)0x365F,
	(q15_t)0x8BF4, (q15_t)0x3604, (q15_t)0x8BCA, (q15_t)0x35A8,
	(q15_t)0x8BA0, (q15_t)0x354D, (q15_t)0x8B76, (q15_t)0x34F2,
	(q15_t)0x8B4D, (q15_t)0x3496, (q15_t)0x8B24, (q15_t)0x343A,
	(q15_t)0x8AFB, (q15_t)0x33DE, (q15_t)0x8AD2, (q15_t)0x3382,
	(q15_t)0x8AAA, (q15_t)0x3326, (q15_t)0x8A82, (q15_t)0x32CA,
	(q15_t)0x8A5A, (q15_t)0x326E, (q15_t)0x8A33, (q15_t)0x3211,
	(q15_t)0x8A0B, (q15_t)0x31B5, (q15_t)0x89E4, (q15_t)0x3158,
	(q15_t)0x89BE, (q15_t)0x30FB, (q15_t)0x8997, (q15_t)0x309E,
	(q15_t)0x8971, (q15_t)0x3041, (q15_t)0x894C, (q15_t)0x2FE4,
	(q15_t)0x8926, (q15_t)0x2F87, (q15_t)0x8901, (q15_t)0x2F29,
	(q15_t)0x88DC, (q15_t)0x2ECC, (q15_t)0x88B8, (q15_t)0x2E6E,
	(q15_t)0x8893, (q15_t)0x2E11, (q15_t)0x886F, (q15_t)0x2DB3,
	(q15_t)0x884B, (q15_t)0x2D55, (q15_t)0x8828, (q15_t)0x2CF7,
	(q15_t)0x8805, (q15_t)0x2C98, (q15_t)0x87E2, (q15_t)0x2C3A,
	(q15_t)0x87BF, (q15_t)0x2BDC, (q15_t)0x879D, (q15_t)0x2B7D,
	(q15_t)0x877B, (q15_t)0x2B1F, (q15_t)0x8759, (q15_t)0x2AC0,
	(q15_t)0x8738, (q15_t)0x2A61, (q15_t)0x8717, (q15_t)0x2A02,
	(q15_t)0x86F6, (q15_t)0x29A3, (q15_t)0x86D5, (q15_t)0x2944,
	(q15_t)0x86B5, (q15_t)0x28E5, (q15_t)0x8695, (q15_t)0x2886,
	(q15_t)0x8675, (q15_t)0x2826, (q15_t)0x8656, (q15_t)0x27C7,
	(q15_t)0x8637, (q15_t)0x2767, (q15_t)0x8618, (q15_t)0x2707,
	(q15_t)0x85FA, (q15_t)0x26A8, (q15_t)0x85DB, (q15_t)0x2648,
	(q15_t)0x85BD, (q15_t)0x25E8, (q15_t)0x85A0, (q15_t)0x2588,
	(q15_t)0x8582, (q15_t)0x2528, (q15_t)0x8565, (q15_t)0x24C7,
	(q15_t)0x8549, (q15_t)0x2467, (q15_t)0x852C, (q15_t)0x2407,
	(q15_t)0x8510, (q15_t)0x23A6, (q15_t)0x84F4, (q15_t)0x2345,
	(q15_t)0x84D9, (q15_t)0x22E5, (q15_t)0x84BD, (q15_t)0x2284,
	(q15_t)0x84A2, (q15_t)0x2223, (q15_t)0x8488, (q15_t)0x21C2,
	(q15_t)0x846D, (q15_t)0x2161, (q15_t)0x8453, (q15_t)0x2100,
	(q15_t)0x843A, (q15_t)0x209F, (q15_t)0x8420, (q15_t)0x203E,
	(q15_t)0x8407, (q15_t)0x1FDC, (q15_t)0x83EE, (q15_t)0x1F7B,
	(q15_t)0x83D6, (q15_t)0x1F19, (q15_t)0x83BD, (q15_t)0x1EB8,
	(q15_t)0x83A5, (q15_t)0x1E56, (q15_t)0x838E, (q15_t)0x1DF5,
	(q15_t)0x8376, (q15_t)0x1D93, (q15_t)0x835F, (q15_t)0x1D31,
	(q15_t)0x8348, (q15_t)0x1CCF, (q15_t)0x8332, (q15_t)0x1C6D,
	(q15_t)0x831C, (q15_t)0x1C0B, (q15_t)0x8306, (q15_t)0x1BA9,
	(q15_t)0x82F0, (q15_t)0x1B47, (q15_t)0x82DB, (q15_t)0x1AE4,
	(q15_t)0x82C6, (q15_t)0x1A82, (q15_t)0x82B1, (q15_t)0x1A20,
	(q15_t)0x829D, (q15_t)0x19BD, (q15_t)0x8289, (q15_t)0x195B,
	(q15_t)0x8275, (q15_t)0x18F8, (q15_t)0x8262, (q15_t)0x1896,
	(q15_t)0x824F, (q15_t)0x1833, (q15_t)0x823C, (q15_t)0x17D0,
	(q15_t)0x8229, (q15_t)0x176D, (q15_t)0x8217, (q15_t)0x170A,
	(q15_t)0x8205, (q15_t)0x16A8, (q15_t)0x81F3, (q15_t)0x1645,
	(q15_t)0x81E2, (q15_t)0x15E2, (q15_t)0x81D1, (q15_t)0x157F,
	(q15_t)0x81C0, (q15_t)0x151B, (q15_t)0x81B0, (q15_t)0x14B8,
	(q15_t)0x81A0, (q15_t)0x1455, (q15_t)0x8190, (q15_t)0x13F2,
	(q15_t)0x8180, (q15_t)0x138E, (q15_t)0x8171, (q15_t)0x132B,
	(q15_t)0x8162, (q15_t)0x12C8, (q15_t)0x8154, (q15_t)0x1264,
	(q15_t)0x8145, (q15_t)0x1201, (q15_t)0x8137, (q15_t)0x119D,
	(q15_t)0x812A, (q15_t)0x1139, (q15_t)0x811C, (q15_t)0x10D6,
	(q15_t)0x810F, (q15_t)0x1072, (q15_t)0x8102, (q15_t)0x100E,
	(q15_t)0x80F6, (q15_t)0x0FAB, (q15_t)0x80EA, (q15_t)0x0F47,
	(q15_t)0x80DE, (q15_t)0x0EE3, (q15_t)0x80D2, (q15_t)0x0E7F,
	(q15_t)0x80C7, (q15_t)0x0E1B, (q15_t)0x80BC, (q15_t)0x0DB7,
	(q15_t)0x80B2, (q15_t)0x0D53, (q15_t)0x80A7, (q15_t)0x0CEF,
	(q15_t)0x809D, (q15_t)0x0C8B, (q15_t)0x8094, (q15_t)0x0C27,
	(q15_t)0x808A, (q15_t)0x0BC3, (q15_t)0x8081, (q15_t)0x0B5F,
	(q15_t)0x8078, (q15_t)0x0AFB, (q15_t)0x8070, (q15_t)0x0A97,
	(q15_t)0x8068, (q15_t)0x0A33, (q15_t)0x8060, (q15_t)0x09CE,
	(q15_t)0x8058, (q15_t)0x096A, (q15_t)0x8051, (q15_t)0x0906,
	(q15_t)0x804A, (q15_t)0x08A2, (q15_t)0x8043, (q15_t)0x083D,
	(q15_t)0x803D, (q15_t)0x07D9, (q15_t)0x8037, (q15_t)0x0775,
	(q15_t)0x8031, (q15_t)0x0710, (q15_t)0x802C, (q15_t)0x06AC,
	(q15_t)0x8027, (q15_t)0x0647, (q15_t)0x8022, (q15_t)0x05E3,
	(q15_t)0x801E, (q15_t)0x057F, (q15_t)0x801A, (q15_t)0x051A,
	(q15_t)0x8016, (q15_t)0x04B6, (q15_t)0x8012, (q15_t)0x0451,
	(q15_t)0x800F, (q15_t)0x03ED, (q15_t)0x800C, (q15_t)0x0388,
	(q15_t)0x8009, (q15_t)0x0324, (q15_t)0x8007, (q15_t)0x02BF,
	(q15_t)0x8005, (q15_t)0x025B, (q15_t)0x8003, (q15_t)0x01F6,
	(q15_t)0x8002, (q15_t)0x0192, (q15_t)0x8001, (q15_t)0x012D,
	(q15_t)0x8000, (q15_t)0x00C9, (q15_t)0x8000, (q15_t)0x0064,
	(q15_t)0x8000, (q15_t)0x0000, (q15_t)0x8000, (q15_t)0xFF9B,
	(q15_t)0x8000, (q15_t)0xFF36, (q15_t)0x8001, (q15_t)0xFED2,
	(q15_t)0x8002, (q15_t)0xFE6D, (q15_t)0x8003, (q15_t)0xFE09,
	(q15_t)0x8005, (q15_t)0xFDA4, (q15_t)0x8007, (q15_t)0xFD40,
	(q15_t)0x8009, (q15_t)0xFCDB, (q15_t)0x800C, (q15_t)0xFC77,
	(q15_t)0x800F, (q15_t)0xFC12, (q15_t)0x8012, (q15_t)0xFBAE,
	(q15_t)0x8016, (q15_t)0xFB49, (q15_t)0x801A, (q15_t)0xFAE5,
	(q15_t)0x801E, (q15_t)0xFA80, (q15_t)0x8022, (q15_t)0xFA1C,
	(q15_t)0x8027, (q15_t)0xF9B8, (q15_t)0x802C, (q15_t)0xF953,
	(q15_t)0x8031, (q15_t)0xF8EF, (q15_t)0x8037, (q15_t)0xF88A,
	(q15_t)0x803D, (q15_t)0xF826, (q15_t)0x8043, (q15_t)0xF7C2,
	(q15_t)0x804A, (q15_t)0xF75D, (q15_t)0x8051, (q15_t)0xF6F9,
	(q15_t)0x8058, (q15_t)0xF695, (q15_t)0x8060, (q15_t)0xF631,
	(q15_t)0x8068, (q15_t)0xF5CC, (q15_t)0x8070, (q15_t)0xF568,
	(q15_t)0x8078, (q15_t)0xF504, (q15_t)0x8081, (q15_t)0xF4A0,
	(q15_t)0x808A, (q15_t)0xF43C, (q15_t)0x8094, (q15_t)0xF3D8,
	(q15_t)0x809D, (q15_t)0xF374, (q15_t)0x80A7, (q15_t)0xF310,
	(q15_t)0x80B2, (q15_t)0xF2AC, (q15_t)0x80BC, (q15_t)0xF248,
	(q15_t)0x80C7, (q15_t)0xF1E4, (q15_t)0x80D2, (q15_t)0xF180,
	(q15_t)0x80DE, (q15_t)0xF11C, (q15_t)0x80EA, (q15_t)0xF0B8,
	(q15_t)0x80F6, (q15_t)0xF054, (q15_t)0x8102, (q15_t)0xEFF1,
	(q15_t)0x810F, (q15_t)0xEF8D, (q15_t)0x811C, (q15_t)0xEF29,
	(q15_t)0x812A, (q15_t)0xEEC6, (q15_t)0x8137, (q15_t)0xEE62,
	(q15_t)0x8145, (q15_t)0xEDFE, (q15_t)0x8154, (q15_t)0xED9B,
	(q15_t)0x8162, (q15_t)0xED37, (q15_t)0x8171, (q15_t)0xECD4,
	(q15_t)0x8180, (q15_t)0xEC71, (q15_t)0x8190, (q15_t)0xEC0D,
	(q15_t)0x81A0, (q15_t)0xEBAA, (q15_t)0x81B0, (q15_t)0xEB47,
	(q15_t)0x81C0, (q15_t)0xEAE4, (q15_t)0x81D1, (q15_t)0xEA80,
	(q15_t)0x81E2, (q15_t)0xEA1D, (q15_t)0x81F3, (q15_t)0xE9BA,
	(q15_t)0x8205, (q15_t)0xE957, (q15_t)0x8217, (q15_t)0xE8F5,
	(q15_t)0x8229, (q15_t)0xE892, (q15_t)0x823C, (q15_t)0xE82F,
	(q15_t)0x824F, (q15_t)0xE7CC, (q15_t)0x8262, (q15_t)0xE769,
	(q15_t)0x8275, (q15_t)0xE707, (q15_t)0x8289, (q15_t)0xE6A4,
	(q15_t)0x829D, (q15_t)0xE642, (q15_t)0x82B1, (q15_t)0xE5DF,
	(q15_t)0x82C6, (q15_t)0xE57D, (q15_t)0x82DB, (q15_t)0xE51B,
	(q15_t)0x82F0, (q15_t)0xE4B8, (q15_t)0x8306, (q15_t)0xE456,
	(q15_t)0x831C, (q15_t)0xE3F4, (q15_t)0x8332, (q15_t)0xE392,
	(q15_t)0x8348, (q15_t)0xE330, (q15_t)0x835F, (q15_t)0xE2CE,
	(q15_t)0x8376, (q15_t)0xE26C, (q15_t)0x838E, (q15_t)0xE20A,
	(q15_t)0x83A5, (q15_t)0xE1A9, (q15_t)0x83BD, (q15_t)0xE147,
	(q15_t)0x83D6, (q15_t)0xE0E6, (q15_t)0x83EE, (q15_t)0xE084,
	(q15_t)0x8407, (q15_t)0xE023, (q15_t)0x8420, (q15_t)0xDFC1,
	(q15_t)0x843A, (q15_t)0xDF60, (q15_t)0x8453, (q15_t)0xDEFF,
	(q15_t)0x846D, (q15_t)0xDE9E, (q15_t)0x8488, (q15_t)0xDE3D,
	(q15_t)0x84A2, (q15_t)0xDDDC, (q15_t)0x84BD, (q15_t)0xDD7B,
	(q15_t)0x84D9, (q15_t)0xDD1A, (q15_t)0x84F4, (q15_t)0xDCBA,
	(q15_t)0x8510, (q15_t)0xDC59, (q15_t)0x852C, (q15_t)0xDBF8,
	(q15_t)0x8549, (q15_t)0xDB98, (q15_t)0x8565, (q15_t)0xDB38,
	(q15_t)0x8582, (q15_t)0xDAD7, (q15_t)0x85A0, (q15_t)0xDA77,
	(q15_t)0x85BD, (q15_t)0xDA17, (q15_t)0x85DB, (q15_t)0xD9B7,
	(q15_t)0x85FA, (q15_t)0xD957, (q15_t)0x8618, (q15_t)0xD8F8,
	(q15_t)0x8637, (q15_t)0xD898, (q15_t)0x8656, (q15_t)0xD838,
	(q15_t)0x8675, (q15_t)0xD7D9, (q15_t)0x8695, (q15_t)0xD779,
	(q15_t)0x86B5, (q15_t)0xD71A, (q15_t)0x86D5, (q15_t)0xD6BB,
	(q15_t)0x86F6, (q15_t)0xD65C, (q15_t)0x8717, (q15_t)0xD5FD,
	(q15_t)0x8738, (q15_t)0xD59E, (q15_t)0x8759, (q15_t)0xD53F,
	(q15_t)0x877B, (q15_t)0xD4E0, (q15_t)0x879D, (q15_t)0xD482,
	(q15_t)0x87BF, (q15_t)0xD423, (q15_t)0x87E2, (q15_t)0xD3C5,
	(q15_t)0x8805, (q15_t)0xD367, (q15_t)0x8828, (q15_t)0xD308,
	(q15_t)0x884B, (q15_t)0xD2AA, (q15_t)0x886F, (q15_t)0xD24C,
	(q15_t)0x8893, (q15_t)0xD1EE, (q15_t)0x88B8, (q15_t)0xD191,
	(q15_t)0x88DC, (q15_t)0xD133, (q15_t)0x8901, (q15_t)0xD0D6,
	(q15_t)0x8926, (q15_t)0xD078, (q15_t)0x894C, (q15_t)0xD01B,
	(q15_t)0x8971, (q15_t)0xCFBE, (q15_t)0x8997, (q15_t)0xCF61,
	(q15_t)0x89BE, (q15_t)0xCF04, (q15_t)0x89E4, (q15_t)0xCEA7,
	(q15_t)0x8A0B, (q15_t)0xCE4A, (q15_t)0x8A33, (q15_t)0xCDEE,
	(q15_t)0x8A5A, (q15_t)0xCD91, (q15_t)0x8A82, (q15_t)0xCD35,
	(q15_t)0x8AAA, (q15_t)0xCCD9, (q15_t)0x8AD2, (q15_t)0xCC7D,
	(q15_t)0x8AFB, (q15_t)0xCC21, (q15_t)0x8B24, (q15_t)0xCBC5,
	(q15_t)0x8B4D, (q15_t)0xCB69, (q15_t)0x8B76, (q15_t)0xCB0D,
	(q15_t)0x8BA0, (q15_t)0xCAB2, (q15_t)0x8BCA, (q15_t)0xCA57,
	(q15_t)0x8BF4, (q15_t)0xC9FB, (q15_t)0x8C1F, (q15_t)0xC9A0,
	(q15_t)0x8C4A, (q15_t)0xC945, (q15_t)0x8C75, (q15_t)0xC8EB,
	(q15_t)0x8CA0, (q15_t)0xC890, (q15_t)0x8CCC, (q15_t)0xC835,
	(q15_t)0x8CF8, (q15_t)0xC7DB, (q15_t)0x8D24, (q15_t)0xC781,
	(q15_t)0x8D50, (q15_t)0xC727, (q15_t)0x8D7D, (q15_t)0xC6CD,
	(q15_t)0x8DAA, (q15_t)0xC673, (q15_t)0x8DD8, (q15_t)0xC619,
	(q15_t)0x8E05, (q15_t)0xC5BF, (q15_t)0x8E33, (q15_t)0xC566,
	(q15_t)0x8E61, (q15_t)0xC50D, (q15_t)0x8E90, (q15_t)0xC4B3,
	(q15_t)0x8EBE, (q15_t)0xC45A, (q15_t)0x8EED, (q15_t)0xC402,
	(q15_t)0x8F1D, (q15_t)0xC3A9, (q15_t)0x8F4C, (q15_t)0xC350,
	(q15_t)0x8F7C, (q15_t)0xC2F8, (q15_t)0x8FAC, (q15_t)0xC29F,
	(q15_t)0x8FDC, (q15_t)0xC247, (q15_t)0x900D, (q15_t)0xC1EF,
	(q15_t)0x903E, (q15_t)0xC197, (q15_t)0x906F, (q15_t)0xC140,
	(q15_t)0x90A0, (q15_t)0xC0E8, (q15_t)0x90D2, (q15_t)0xC091,
	(q15_t)0x9104, (q15_t)0xC03A, (q15_t)0x9136, (q15_t)0xBFE2,
	(q15_t)0x9169, (q15_t)0xBF8C, (q15_t)0x919C, (q15_t)0xBF35,
	(q15_t)0x91CF, (q15_t)0xBEDE, (q15_t)0x9202, (q15_t)0xBE88,
	(q15_t)0x9235, (q15_t)0xBE31, (q15_t)0x9269, (q15_t)0xBDDB,
	(q15_t)0x929D, (q15_t)0xBD85, (q15_t)0x92D2, (q15_t)0xBD2F,
	(q15_t)0x9306, (q15_t)0xBCDA, (q15_t)0x933B, (q15_t)0xBC84,
	(q15_t)0x9370, (q15_t)0xBC2F, (q15_t)0x93A6, (q15_t)0xBBDA,
	(q15_t)0x93DB, (q15_t)0xBB85, (q15_t)0x9411, (q15_t)0xBB30,
	(q15_t)0x9447, (q15_t)0xBADB, (q15_t)0x947E, (q15_t)0xBA87,
	(q15_t)0x94B5, (q15_t)0xBA32, (q15_t)0x94EC, (q15_t)0xB9DE,
	(q15_t)0x9523, (q15_t)0xB98A, (q15_t)0x955A, (q15_t)0xB936,
	(q15_t)0x9592, (q15_t)0xB8E3, (q15_t)0x95CA, (q15_t)0xB88F,
	(q15_t)0x9602, (q15_t)0xB83C, (q15_t)0x963B, (q15_t)0xB7E9,
	(q15_t)0x9673, (q15_t)0xB796, (q15_t)0x96AC, (q15_t)0xB743,
	(q15_t)0x96E6, (q15_t)0xB6F0, (q15_t)0x971F, (q15_t)0xB69E,
	(q15_t)0x9759, (q15_t)0xB64B, (q15_t)0x9793, (q15_t)0xB5F9,
	(q15_t)0x97CD, (q15_t)0xB5A7, (q15_t)0x9808, (q15_t)0xB556,
	(q15_t)0x9842, (q15_t)0xB504, (q15_t)0x987D, (q15_t)0xB4B3,
	(q15_t)0x98B9, (q15_t)0xB461, (q15_t)0x98F4, (q15_t)0xB410,
	(q15_t)0x9930, (q15_t)0xB3C0, (q15_t)0x996C, (q15_t)0xB36F,
	(q15_t)0x99A8, (q15_t)0xB31E, (q15_t)0x99E5, (q15_t)0xB2CE,
	(q15_t)0x9A22, (q15_t)0xB27E, (q15_t)0x9A5F, (q15_t)0xB22E,
	(q15_t)0x9A9C, (q15_t)0xB1DE, (q15_t)0x9AD9, (q15_t)0xB18F,
	(q15_t)0x9B17, (q15_t)0xB140, (q15_t)0x9B55, (q15_t)0xB0F0,
	(q15_t)0x9B93, (q15_t)0xB0A1, (q15_t)0x9BD2, (q15_t)0xB053,
	(q15_t)0x9C10, (q15_t)0xB004, (q15_t)0x9C4F, (q15_t)0xAFB6,
	(q15_t)0x9C8E, (q15_t)0xAF68, (q15_t)0x9CCE, (q15_t)0xAF1A,
	(q15_t)0x9D0D, (q15_t)0xAECC, (q15_t)0x9D4D, (q15_t)0xAE7E,
	(q15_t)0x9D8E, (q15_t)0xAE31, (q15_t)0x9DCE, (q15_t)0xADE3,
	(q15_t)0x9E0E, (q15_t)0xAD96, (q15_t)0x9E4F, (q15_t)0xAD4A,
	(q15_t)0x9E90, (q15_t)0xACFD, (q15_t)0x9ED2, (q15_t)0xACB1,
	(q15_t)0x9F13, (q15_t)0xAC64, (q15_t)0x9F55, (q15_t)0xAC18,
	(q15_t)0x9F97, (q15_t)0xABCC, (q15_t)0x9FD9, (q15_t)0xAB81,
	(q15_t)0xA01C, (q15_t)0xAB35, (q15_t)0xA05F, (q15_t)0xAAEA,
	(q15_t)0xA0A1, (q15_t)0xAA9F, (q15_t)0xA0E5, (q15_t)0xAA54,
	(q15_t)0xA128, (q15_t)0xAA0A, (q15_t)0xA16C, (q15_t)0xA9BF,
	(q15_t)0xA1AF, (q15_t)0xA975, (q15_t)0xA1F4, (q15_t)0xA92B,
	(q15_t)0xA238, (q15_t)0xA8E2, (q15_t)0xA27C, (q15_t)0xA898,
	(q15_t)0xA2C1, (q15_t)0xA84F, (q15_t)0xA306, (q15_t)0xA806,
	(q15_t)0xA34B, (q15_t)0xA7BD, (q15_t)0xA391, (q15_t)0xA774,
	(q15_t)0xA3D6, (q15_t)0xA72B, (q15_t)0xA41C, (q15_t)0xA6E3,
	(q15_t)0xA462, (q15_t)0xA69B, (q15_t)0xA4A9, (q15_t)0xA653,
	(q15_t)0xA4EF, (q15_t)0xA60C, (q15_t)0xA536, (q15_t)0xA5C4,
	(q15_t)0xA57D, (q15_t)0xA57D, (q15_t)0xA5C4, (q15_t)0xA536,
	(q15_t)0xA60C, (q15_t)0xA4EF, (q15_t)0xA653, (q15_t)0xA4A9,
	(q15_t)0xA69B, (q15_t)0xA462, (q15_t)0xA6E3, (q15_t)0xA41C,
	(q15_t)0xA72B, (q15_t)0xA3D6, (q15_t)0xA774, (q15_t)0xA391,
	(q15_t)0xA7BD, (q15_t)0xA34B, (q15_t)0xA806, (q15_t)0xA306,
	(q15_t)0xA84F, (q15_t)0xA2C1, (q15_t)0xA898, (q15_t)0xA27C,
	(q15_t)0xA8E2, (q15_t)0xA238, (q15_t)0xA92B, (q15_t)0xA1F4,
	(q15_t)0xA975, (q15_t)0xA1AF, (q15_t)0xA9BF, (q15_t)0xA16C,
	(q15_t)0xAA0A, (q15_t)0xA128, (q15_t)0xAA54, (q15_t)0xA0E5,
	(q15_t)0xAA9F, (q15_t)0xA0A1, (q15_t)0xAAEA, (q15_t)0xA05F,
	(q15_t)0xAB35, (q15_t)0xA01C, (q15_t)0xAB81, (q15_t)0x9FD9,
	(q15_t)0xABCC, (q15_t)0x9F97, (q15_t)0xAC18, (q15_t)0x9F55,
	(q15_t)0xAC64, (q15_t)0x9F13, (q15_t)0xACB1, (q15_t)0x9ED2,
	(q15_t)0xACFD, (q15_t)0x9E90, (q15_t)0xAD4A, (q15_t)0x9E4F,
	(q15_t)0xAD96, (q15_t)0x9E0E, (q15_t)0xADE3, (q15_t)0x9DCE,
	(q15_t)0xAE31, (q15_t)0x9D8E, (q15_t)0xAE7E, (q15_t)0x9D4D,
	(q15_t)0xAECC, (q15_t)0x9D0D, (q15_t)0xAF1A, (q15_t)0x9CCE,
	(q15_t)0xAF68, (q15_t)0x9C8E, (q15_t)0xAFB6, (q15_t)0x9C4F,
	(q15_t)0xB004, (q15_t)0x9C10, (q15_t)0xB053, (q15_t)0x9BD2,
	(q15_t)0xB0A1, (q15_t)0x9B93, (q15_t)0xB0F0, (q15_t)0x9B55,
	(q15_t)0xB140, (q15_t)0x9B17, (q15_t)0xB18F, (q15_t)0x9AD9,
	(q15_t)0xB1DE, (q15_t)0x9A9C, (q15_t)0xB22E, (q15_t)0x9A5F,
	(q15_t)0xB27E, (q15_t)0x9A22, (q15_t)0xB2CE, (q15_t)0x99E5,
	(q15_t)0xB31E, (q15_t)0x99A8, (q15_t)0xB36F, (q15_t)0x996C,
	(q15_t)0xB3C0, (q15_t)0x9930, (q15_t)0xB410, (q15_t)0x98F4,
	(q15_t)0xB461, (q15_t)0x98B9, (q15_t)0xB4B3, (q15_t)0x987D,
	(q15_t)0xB504, (q15_t)0x9842, (q15_t)0xB556, (q15_t)0x9808,
	(q15_t)0xB5A7, (q15_t)0x97CD, (q15_t)0xB5F9, (q15_t)0x9793,
	(q15_t)0xB64B, (q15_t)0x9759, (q15_t)0xB69E, (q15_t)0x971F,
	(q15_t)0xB6F0, (q15_t)0x96E6, (q15_t)0xB743, (q15_t)0x96AC,
	(q15_t)0xB796, (q15_t)0x9673, (q15_t)0xB7E9, (q15_t)0x963B,
	(q15_t)0xB83C, (q15_t)0x9602, (q15_t)0xB88F, (q15_t)0x95CA,
	(q15_t)0xB8E3, (q15_t)0x9592, (q15_t)0xB936, (q15_t)0x955A,
	(q15_t)0xB98A, (q15_t)0x9523, (q15_t)0xB9DE, (q15_t)0x94EC,
	(q15_t)0xBA32, (q15_t)0x94B5, (q15_t)0xBA87, (q15_t)0x947E,
	(q15_t)0xBADB, (q15_t)0x9447, (q15_t)0xBB30, (q15_t)0x9411,
	(q15_t)0xBB85, (q15_t)0x93DB, (q15_t)0xBBDA, (q15_t)0x93A6,
	(q15_t)0xBC2F, (q15_t)0x9370, (q15_t)0xBC84, (q15_t)0x933B,
	(q15_t)0xBCDA, (q15_t)0x9306, (q15_t)0xBD2F, (q15_t)0x92D2,
	(q15_t)0xBD85, (q15_t)0x929D, (q15_t)0xBDDB, (q15_t)0x9269,
	(q15_t)0xBE31, (q15_t)0x9235, (q15_t)0xBE88, (q15_t)0x9202,
	(q15_t)0xBEDE, (q15_t)0x91CF, (q15_t)0xBF35, (q15_t)0x919C,
	(q15_t)0xBF8C, (q15_t)0x9169, (q15_t)0xBFE2, (q15_t)0x9136,
	(q15_t)0xC03A, (q15_t)0x9104, (q15_t)0xC091, (q15_t)0x90D2,
	(q15_t)0xC0E8, (q15_t)0x90A0, (q15_t)0xC140, (q15_t)0x906F,
	(q15_t)0xC197, (q15_t)0x903E, (q15_t)0xC1EF, (q15_t)0x900D,
	(q15_t)0xC247, (q15_t)0x8FDC, (q15_t)0xC29F, (q15_t)0x8FAC,
	(q15_t)0xC2F8, (q15_t)0x8F7C, (q15_t)0xC350, (q15_t)0x8F4C,
	(q15_t)0xC3A9, (q15_t)0x8F1D, (q15_t)0xC402, (q15_t)0x8EED,
	(q15_t)0xC45A, (q15_t)0x8EBE, (q15_t)0xC4B3, (q15_t)0x8E90,
	(q15_t)0xC50D, (q15_t)0x8E61, (q15_t)0xC566, (q15_t)0x8E33,
	(q15_t)0xC5BF, (q15_t)0x8E05, (q15_t)0xC619, (q15_t)0x8DD8,
	(q15_t)0xC673, (q15_t)0x8DAA, (q15_t)0xC6CD, (q15_t)0x8D7D,
	(q15_t)0xC727, (q15_t)0x8D50, (q15_t)0xC781, (q15_t)0x8D24,
	(q15_t)0xC7DB, (q15_t)0x8CF8, (q15_t)0xC835, (q15_t)0x8CCC,
	(q15_t)0xC890, (q15_t)0x8CA0, (q15_t)0xC8EB, (q15_t)0x8C75,
	(q15_t)0xC945, (q15_t)0x8C4A, (q15_t)0xC9A0, (q15_t)0x8C1F,
	(q15_t)0xC9FB, (q15_t)0x8BF4, (q15_t)0xCA57, (q15_t)0x8BCA,
	(q15_t)0xCAB2, (q15_t)0x8BA0, (q15_t)0xCB0D, (q15_t)0x8B76,
	(q15_t)0xCB69, (q15_t)0x8B4D, (q15_t)0xCBC5, (q15_t)0x8B24,
	(q15_t)0xCC21, (q15_t)0x8AFB, (q15_t)0xCC7D, (q15_t)0x8AD2,
	(q15_t)0xCCD9, (q15_t)0x8AAA, (q15_t)0xCD35, (q15_t)0x8A82,
	(q15_t)0xCD91, (q15_t)0x8A5A, (q15_t)0xCDEE, (q15_t)0x8A33,
	(q15_t)0xCE4A, (q15_t)0x8A0B, (q15_t)0xCEA7, (q15_t)0x89E4,
	(q15_t)0xCF04, (q15_t)0x89BE, (q15_t)0xCF61, (q15_t)0x8997,
	(q15_t)0xCFBE, (q15_t)0x8971, (q15_t)0xD01B, (q15_t)0x894C,
	(q15_t)0xD078, (q15_t)0x8926, (q15_t)0xD0D6, (q15_t)0x8901,
	(q15_t)0xD133, (q15_t)0x88DC, (q15_t)0xD191, (q15_t)0x88B8,
	(q15_t)0xD1EE, (q15_t)0x8893, (q15_t)0xD24C, (q15_t)0x886F,
	(q15_t)0xD2AA, (q15_t)0x884B, (q15_t)0xD308, (q15_t)0x8828,
	(q15_t)0xD367, (q15_t)0x8805, (q15_t)0xD3C5, (q15_t)0x87E2,
	(q15_t)0xD423, (q15_t)0x87BF, (q15_t)0xD482, (q15_t)0x879D,
	(q15_t)0xD4E0, (q15_t)0x877B, (q15_t)0xD53F, (q15_t)0x8759,
	(q15_t)0xD59E, (q15_t)0x8738, (q15_t)0xD5FD, (q15_t)0x8717,
	(q15_t)0xD65C, (q15_t)0x86F6, (q15_t)0xD6BB, (q15_t)0x86D5,
	(q15_t)0xD71A, (q15_t)0x86B5, (q15_t)0xD779, (q15_t)0x8695,
	(q15_t)0xD7D9, (q15_t)0x8675, (q15_t)0xD838, (q15_t)0x8656,
	(q15_t)0xD898, (q15_t)0x8637, (q15_t)0xD8F8, (q15_t)0x8618,
	(q15_t)0xD957, (q15_t)0x85FA, (q15_t)0xD9B7, (q15_t)0x85DB,
	(q15_t)0xDA17, (q15_t)0x85BD, (q15_t)0xDA77, (q15_t)0x85A0,
	(q15_t)0xDAD7, (q15_t)0x8582, (q15_t)0xDB38, (q15_t)0x8565,
	(q15_t)0xDB98, (q15_t)0x8549, (q15_t)0xDBF8, (q15_t)0x852C,
	(q15_t)0xDC59, (q15_t)0x8510, (q15_t)0xDCBA, (q15_t)0x84F4,
	(q15_t)0xDD1A, (q15_t)0x84D9, (q15_t)0xDD7B, (q15_t)0x84BD,
	(q15_t)0xDDDC, (q15_t)0x84A2, (q15_t)0xDE3D, (q15_t)0x8488,
	(q15_t)0xDE9E, (q15_t)0x846D, (q15_t)0xDEFF, (q15_t)0x8453,
	(q15_t)0xDF60, (q15_t)0x843A, (q15_t)0xDFC1, (q15_t)0x8420,
	(q15_t)0xE023, (q15_t)0x8407, (q15_t)0xE084, (q15_t)0x83EE,
	(q15_t)0xE0E6, (q15_t)0x83D6, (q15_t)0xE147, (q15_t)0x83BD,
	(q15_t)0xE1A9, (q15_t)0x83A5, (q15_t)0xE20A, (q15_t)0x838E,
	(q15_t)0xE26C, (q15_t)0x8376, (q15_t)0xE2CE, (q15_t)0x835F,
	(q15_t)0xE330, (q15_t)0x8348, (q15_t)0xE392, (q15_t)0x8332,
	(q15_t)0xE3F4, (q15_t)0x831C, (q15_t)0xE456, (q15_t)0x8306,
	(q15_t)0xE4B8, (q15_t)0x82F0, (q15_t)0xE51B, (q15_t)0x82DB,
	(q15_t)0xE57D, (q15_t)0x82C6, (q15_t)0xE5DF, (q15_t)0x82B1,
	(q15_t)0xE642, (q15_t)0x829D, (q15_t)0xE6A4, (q15_t)0x8289,
	(q15_t)0xE707, (q15_t)0x8275, (q15_t)0xE769, (q15_t)0x8262,
	(q15_t)0xE7CC, (q15_t)0x824F, (q15_t)0xE82F, (q15_t)0x823C,
	(q15_t)0xE892, (q15_t)0x8229, (q15_t)0xE8F5, (q15_t)0x8217,
	(q15_t)0xE957, (q15_t)0x8205, (q15_t)0xE9BA, (q15_t)0x81F3,
	(q15_t)0xEA1D, (q15_t)0x81E2, (q15_t)0xEA80, (q15_t)0x81D1,
	(q15_t)0xEAE4, (q15_t)0x81C0, (q15_t)0xEB47, (q15_t)0x81B0,
	(q15_t)0xEBAA, (q15_t)0x81A0, (q15_t)0xEC0D, (q15_t)0x8190,
	(q15_t)0xEC71, (q15_t)0x8180, (q15_t)0xECD4, (q15_t)0x8171,
	(q15_t)0xED37, (q15_t)0x8162, (q15_t)0xED9B, (q15_t)0x8154,
	(q15_t)0xEDFE, (q15_t)0x8145, (q15_t)0xEE62, (q15_t)0x8137,
	(q15_t)0xEEC6, (q15_t)0x812A, (q15_t)0xEF29, (q15_t)0x811C,
	(q15_t)0xEF8D, (q15_t)0x810F, (q15_t)0xEFF1, (q15_t)0x8102,
	(q15_t)0xF054, (q15_t)0x80F6, (q15_t)0xF0B8, (q15_t)0x80EA,
	(q15_t)0xF11C, (q15_t)0x80DE, (q15_t)0xF180, (q15_t)0x80D2,
	(q15_t)0xF1E4, (q15_t)0x80C7, (q15_t)0xF248, (q15_t)0x80BC,
	(q15_t)0xF2AC, (q15_t)0x80B2, (q15_t)0xF310, (q15_t)0x80A7,
	(q15_t)0xF374, (q15_t)0x809D, (q15_t)0xF3D8, (q15_t)0x8094,
	(q15_t)0xF43C, (q15_t)0x808A, (q15_t)0xF4A0, (q15_t)0x8081,
	(q15_t)0xF504, (q15_t)0x8078, (q15_t)0xF568, (q15_t)0x8070,
	(q15_t)0xF5CC, (q15_t)0x8068, (q15_t)0xF631, (q15_t)0x8060,
	(q15_t)0xF695, (q15_t)0x8058, (q15_t)0xF6F9, (q15_t)0x8051,
	(q15_t)0xF75D, (q15_t)0x804A, (q15_t)0xF7C2, (q15_t)0x8043,
	(q15_t)0xF826, (q15_t)0x803D, (q15_t)0xF88A, (q15_t)0x8037,
	(q15_t)0xF8EF, (q15_t)0x8031, (q15_t)0xF953, (q15_t)0x802C,
	(q15_t)0xF9B8, (q15_t)0x8027, (q15_t)0xFA1C, (q15_t)0x8022,
	(q15_t)0xFA80, (q15_t)0x801E, (q15_t)0xFAE5, (q15_t)0x801A,
	(q15_t)0xFB49, (q15_t)0x8016, (q15_t)0xFBAE, (q15_t)0x8012,
	(q15_t)0xFC12, (q15_t)0x800F, (q15_t)0xFC77, (q15_t)0x800C,
	(q15_t)0xFCDB, (q15_t)0x8009, (q15_t)0xFD40, (q15_t)0x8007,
	(q15_t)0xFDA4, (q15_t)0x8005, (q15_t)0xFE09, (q15_t)0x8003,
	(q15_t)0xFE6D, (q15_t)0x8002, (q15_t)0xFED2, (q15_t)0x8001,
	(q15_t)0xFF36, (q15_t)0x8000, (q15_t)0xFF9B, (q15_t)0x8000
};

/**
* \par
* Example code for q15 Twiddle factors Generation::
* \par
* <pre>for(i = 0; i< 3N/4; i++)
* {
*    twiddleCoefq15[2*i]= cos(i * 2*PI/(float)N);
*    twiddleCoefq15[2*i+1]= sin(i * 2*PI/(float)N);
* } </pre>
* \par
* where N = 4096	and PI = 3.14159265358979
* \par
* Cos and Sin values are interleaved fashion
* \par
* Convert Floating point to q15(Fixed point 1.15):
*	round(twiddleCoefq15(i) * pow(2, 15))
*
*/
const q15_t twiddleCoef_4096_q15[6144] =
{
	(q15_t)0x7FFF, (q15_t)0x0000, (q15_t)0x7FFF, (q15_t)0x0032,
	(q15_t)0x7FFF, (q15_t)0x0064, (q15_t)0x7FFF, (q15_t)0x0096,
	(q15_t)0x7FFF, (q15_t)0x00C9, (q15_t)0x7FFF, (q15_t)0x00FB,
	(q15_t)0x7FFE, (q15_t)0x012D, (q15_t)0x7FFE, (q15_t)0x015F,
	(q15_t)0x7FFD, (q15_t)0x0192, (q15_t)0x7FFC, (q15_t)0x01C4,
	(q15_t)0x7FFC, (q15_t)0x01F6, (q15_t)0x7FFB, (q15_t)0x0228,
	(q15_t)0x7FFA, (q15_t)0x025B, (q15_t)0x7FF9, (q15_t)0x028D,
	(q15_t)0x7FF8, (q15_t)0x02BF, (q15_t)0x7FF7, (q15_t)0x02F1,
	(q15_t)0x7FF6, (q15_t)0x0324, (q15_t)0x7FF4, (q15_t)0x0356,
	(q15_t)0x7FF3, (q15_t)0x0388, (q15_t)0x7FF2, (q15_t)0x03BA,
	(q15_t)0x7FF0, (q15_t)0x03ED, (q15_t)0x7FEE, (q15_t)0x041F,
	(q15_t)0x7FED, (q15_t)0x0451, (q15_t)0x7FEB, (q15_t)0x0483,
	(q15_t)0x7FE9, (q15_t)0x04B6, (q15_t)0x7FE7, (q15_t)0x04E8,
	(q15_t)0x7FE5, (q15_t)0x051A, (q15_t)0x7FE3, (q15_t)0x054C,
	(q15_t)0x7FE1, (q15_t)0x057F, (q15_t)0x7FDF, (q15_t)0x05B1,
	(q15_t)0x7FDD, (q15_t)0x05E3, (q15_t)0x7FDA, (q15_t)0x0615,
	(q15_t)0x7FD8, (q15_t)0x0647, (q15_t)0x7FD6, (q15_t)0x067A,
	(q15_t)0x7FD3, (q15_t)0x06AC, (q15_t)0x7FD0, (q15_t)0x06DE,
	(q15_t)0x7FCE, (q15_t)0x0710, (q15_t)0x7FCB, (q15_t)0x0742,
	(q15_t)0x7FC8, (q15_t)0x0775, (q15_t)0x7FC5, (q15_t)0x07A7,
	(q15_t)0x7FC2, (q15_t)0x07D9, (q15_t)0x7FBF, (q15_t)0x080B,
	(q15_t)0x7FBC, (q15_t)0x083D, (q15_t)0x7FB8, (q15_t)0x086F,
	(q15_t)0x7FB5, (q15_t)0x08A2, (q15_t)0x7FB1, (q15_t)0x08D4,
	(q15_t)0x7FAE, (q15_t)0x0906, (q15_t)0x7FAA, (q15_t)0x0938,
	(q15_t)0x7FA7, (q15_t)0x096A, (q15_t)0x7FA3, (q15_t)0x099C,
	(q15_t)0x7F9F, (q15_t)0x09CE, (q15_t)0x7F9B, (q15_t)0x0A00,
	(q15_t)0x7F97, (q15_t)0x0A33, (q15_t)0x7F93, (q15_t)0x0A65,
	(q15_t)0x7F8F, (q15_t)0x0A97, (q15_t)0x7F8B, (q15_t)0x0AC9,
	(q15_t)0x7F87, (q15_t)0x0AFB, (q15_t)0x7F82, (q15_t)0x0B2D,
	(q15_t)0x7F7E, (q15_t)0x0B5F, (q15_t)0x7F79, (q15_t)0x0B91,
	(q15_t)0x7F75, (q15_t)0x0BC3, (q15_t)0x7F70, (q15_t)0x0BF5,
	(q15_t)0x7F6B, (q15_t)0x0C27, (q15_t)0x7F67, (q15_t)0x0C59,
	(q15_t)0x7F62, (q15_t)0x0C8B, (q15_t)0x7F5D, (q15_t)0x0CBD,
	(q15_t)0x7F58, (q15_t)0x0CEF, (q15_t)0x7F53, (q15_t)0x0D21,
	(q15_t)0x7F4D, (q15_t)0x0D53, (q15_t)0x7F48, (q15_t)0x0D85,
	(q15_t)0x7F43, (q15_t)0x0DB7, (q15_t)0x7F3D, (q15_t)0x0DE9,
	(q15_t)0x7F38, (q15_t)0x0E1B, (q15_t)0x7F32, (q15_t)0x0E4D,
	(q15_t)0x7F2D, (q15_t)0x0E7F, (q15_t)0x7F27, (q15_t)0x0EB1,
	(q15_t)0x7F21, (q15_t)0x0EE3, (q15_t)0x7F1B, (q15_t)0x0F15,
	(q15_t)0x7F15, (q15_t)0x0F47, (q15_t)0x7F0F, (q15_t)0x0F79,
	(q15_t)0x7F09, (q15_t)0x0FAB, (q15_t)0x7F03, (q15_t)0x0FDD,
	(q15_t)0x7EFD, (q15_t)0x100E, (q15_t)0x7EF6, (q15_t)0x1040,
	(q15_t)0x7EF0, (q15_t)0x1072, (q15_t)0x7EE9, (q15_t)0x10A4,
	(q15_t)0x7EE3, (q15_t)0x10D6, (q15_t)0x7EDC, (q15_t)0x1108,
	(q15_t)0x7ED5, (q15_t)0x1139, (q15_t)0x7ECF, (q15_t)0x116B,
	(q15_t)0x7EC8, (q15_t)0x119D, (q15_t)0x7EC1, (q15_t)0x11CF,
	(q15_t)0x7EBA, (q15_t)0x1201, (q15_t)0x7EB3, (q15_t)0x1232,
	(q15_t)0x7EAB, (q15_t)0x1264, (q15_t)0x7EA4, (q15_t)0x1296,
	(q15_t)0x7E9D, (q15_t)0x12C8, (q15_t)0x7E95, (q15_t)0x12F9,
	(q15_t)0x7E8E, (q15_t)0x132B, (q15_t)0x7E86, (q15_t)0x135D,
	(q15_t)0x7E7F, (q15_t)0x138E, (q15_t)0x7E77, (q15_t)0x13C0,
	(q15_t)0x7E6F, (q15_t)0x13F2, (q15_t)0x7E67, (q15_t)0x1423,
	(q15_t)0x7E5F, (q15_t)0x1455, (q15_t)0x7E57, (q15_t)0x1487,
	(q15_t)0x7E4F, (q15_t)0x14B8, (q15_t)0x7E47, (q15_t)0x14EA,
	(q15_t)0x7E3F, (q15_t)0x151B, (q15_t)0x7E37, (q15_t)0x154D,
	(q15_t)0x7E2E, (q15_t)0x157F, (q15_t)0x7E26, (q15_t)0x15B0,
	(q15_t)0x7E1D, (q15_t)0x15E2, (q15_t)0x7E14, (q15_t)0x1613,
	(q15_t)0x7E0C, (q15_t)0x1645, (q15_t)0x7E03, (q15_t)0x1676,
	(q15_t)0x7DFA, (q15_t)0x16A8, (q15_t)0x7DF1, (q15_t)0x16D9,
	(q15_t)0x7DE8, (q15_t)0x170A, (q15_t)0x7DDF, (q15_t)0x173C,
	(q15_t)0x7DD6, (q15_t)0x176D, (q15_t)0x7DCD, (q15_t)0x179F,
	(q15_t)0x7DC3, (q15_t)0x17D0, (q15_t)0x7DBA, (q15_t)0x1802,
	(q15_t)0x7DB0, (q15_t)0x1833, (q15_t)0x7DA7, (q15_t)0x1864,
	(q15_t)0x7D9D, (q15_t)0x1896, (q15_t)0x7D94, (q15_t)0x18C7,
	(q15_t)0x7D8A, (q15_t)0x18F8, (q15_t)0x7D80, (q15_t)0x192A,
	(q15_t)0x7D76, (q15_t)0x195B, (q15_t)0x7D6C, (q15_t)0x198C,
	(q15_t)0x7D62, (q15_t)0x19BD, (q15_t)0x7D58, (q15_t)0x19EF,
	(q15_t)0x7D4E, (q15_t)0x1A20, (q15_t)0x7D43, (q15_t)0x1A51,
	(q15_t)0x7D39, (q15_t)0x1A82, (q15_t)0x7D2F, (q15_t)0x1AB3,
	(q15_t)0x7D24, (q15_t)0x1AE4, (q15_t)0x7D19, (q15_t)0x1B16,
	(q15_t)0x7D0F, (q15_t)0x1B47, (q15_t)0x7D04, (q15_t)0x1B78,
	(q15_t)0x7CF9, (q15_t)0x1BA9, (q15_t)0x7CEE, (q15_t)0x1BDA,
	(q15_t)0x7CE3, (q15_t)0x1C0B, (q15_t)0x7CD8, (q15_t)0x1C3C,
	(q15_t)0x7CCD, (q15_t)0x1C6D, (q15_t)0x7CC2, (q15_t)0x1C9E,
	(q15_t)0x7CB7, (q15_t)0x1CCF, (q15_t)0x7CAB, (q15_t)0x1D00,
	(q15_t)0x7CA0, (q15_t)0x1D31, (q15_t)0x7C94, (q15_t)0x1D62,
	(q15_t)0x7C89, (q15_t)0x1D93, (q15_t)0x7C7D, (q15_t)0x1DC4,
	(q15_t)0x7C71, (q15_t)0x1DF5, (q15_t)0x7C66, (q15_t)0x1E25,
	(q15_t)0x7C5A, (q15_t)0x1E56, (q15_t)0x7C4E, (q15_t)0x1E87,
	(q15_t)0x7C42, (q15_t)0x1EB8, (q15_t)0x7C36, (q15_t)0x1EE9,
	(q15_t)0x7C29, (q15_t)0x1F19, (q15_t)0x7C1D, (q15_t)0x1F4A,
	(q15_t)0x7C11, (q15_t)0x1F7B, (q15_t)0x7C05, (q15_t)0x1FAC,
	(q15_t)0x7BF8, (q15_t)0x1FDC, (q15_t)0x7BEB, (q15_t)0x200D,
	(q15_t)0x7BDF, (q15_t)0x203E, (q15_t)0x7BD2, (q15_t)0x206E,
	(q15_t)0x7BC5, (q15_t)0x209F, (q15_t)0x7BB9, (q15_t)0x20D0,
	(q15_t)0x7BAC, (q15_t)0x2100, (q15_t)0x7B9F, (q15_t)0x2131,
	(q15_t)0x7B92, (q15_t)0x2161, (q15_t)0x7B84, (q15_t)0x2192,
	(q15_t)0x7B77, (q15_t)0x21C2, (q15_t)0x7B6A, (q15_t)0x21F3,
	(q15_t)0x7B5D, (q15_t)0x2223, (q15_t)0x7B4F, (q15_t)0x2254,
	(q15_t)0x7B42, (q15_t)0x2284, (q15_t)0x7B34, (q15_t)0x22B4,
	(q15_t)0x7B26, (q15_t)0x22E5, (q15_t)0x7B19, (q15_t)0x2315,
	(q15_t)0x7B0B, (q15_t)0x2345, (q15_t)0x7AFD, (q15_t)0x2376,
	(q15_t)0x7AEF, (q15_t)0x23A6, (q15_t)0x7AE1, (q15_t)0x23D6,
	(q15_t)0x7AD3, (q15_t)0x2407, (q15_t)0x7AC5, (q15_t)0x2437,
	(q15_t)0x7AB6, (q15_t)0x2467, (q15_t)0x7AA8, (q15_t)0x2497,
	(q15_t)0x7A9A, (q15_t)0x24C7, (q15_t)0x7A8B, (q15_t)0x24F7,
	(q15_t)0x7A7D, (q15_t)0x2528, (q15_t)0x7A6E, (q15_t)0x2558,
	(q15_t)0x7A5F, (q15_t)0x2588, (q15_t)0x7A50, (q15_t)0x25B8,
	(q15_t)0x7A42, (q15_t)0x25E8, (q15_t)0x7A33, (q15_t)0x2618,
	(q15_t)0x7A24, (q15_t)0x2648, (q15_t)0x7A15, (q15_t)0x2678,
	(q15_t)0x7A05, (q15_t)0x26A8, (q15_t)0x79F6, (q15_t)0x26D8,
	(q15_t)0x79E7, (q15_t)0x2707, (q15_t)0x79D8, (q15_t)0x2737,
	(q15_t)0x79C8, (q15_t)0x2767, (q15_t)0x79B9, (q15_t)0x2797,
	(q15_t)0x79A9, (q15_t)0x27C7, (q15_t)0x7999, (q15_t)0x27F6,
	(q15_t)0x798A, (q15_t)0x2826, (q15_t)0x797A, (q15_t)0x2856,
	(q15_t)0x796A, (q15_t)0x2886, (q15_t)0x795A, (q15_t)0x28B5,
	(q15_t)0x794A, (q15_t)0x28E5, (q15_t)0x793A, (q15_t)0x2915,
	(q15_t)0x792A, (q15_t)0x2944, (q15_t)0x7919, (q15_t)0x2974,
	(q15_t)0x7909, (q15_t)0x29A3, (q15_t)0x78F9, (q15_t)0x29D3,
	(q15_t)0x78E8, (q15_t)0x2A02, (q15_t)0x78D8, (q15_t)0x2A32,
	(q15_t)0x78C7, (q15_t)0x2A61, (q15_t)0x78B6, (q15_t)0x2A91,
	(q15_t)0x78A6, (q15_t)0x2AC0, (q15_t)0x7895, (q15_t)0x2AEF,
	(q15_t)0x7884, (q15_t)0x2B1F, (q15_t)0x7873, (q15_t)0x2B4E,
	(q15_t)0x7862, (q15_t)0x2B7D, (q15_t)0x7851, (q15_t)0x2BAD,
	(q15_t)0x7840, (q15_t)0x2BDC, (q15_t)0x782E, (q15_t)0x2C0B,
	(q15_t)0x781D, (q15_t)0x2C3A, (q15_t)0x780C, (q15_t)0x2C69,
	(q15_t)0x77FA, (q15_t)0x2C98, (q15_t)0x77E9, (q15_t)0x2CC8,
	(q15_t)0x77D7, (q15_t)0x2CF7, (q15_t)0x77C5, (q15_t)0x2D26,
	(q15_t)0x77B4, (q15_t)0x2D55, (q15_t)0x77A2, (q15_t)0x2D84,
	(q15_t)0x7790, (q15_t)0x2DB3, (q15_t)0x777E, (q15_t)0x2DE2,
	(q15_t)0x776C, (q15_t)0x2E11, (q15_t)0x775A, (q15_t)0x2E3F,
	(q15_t)0x7747, (q15_t)0x2E6E, (q15_t)0x7735, (q15_t)0x2E9D,
	(q15_t)0x7723, (q15_t)0x2ECC, (q15_t)0x7710, (q15_t)0x2EFB,
	(q15_t)0x76FE, (q15_t)0x2F29, (q15_t)0x76EB, (q15_t)0x2F58,
	(q15_t)0x76D9, (q15_t)0x2F87, (q15_t)0x76C6, (q15_t)0x2FB5,
	(q15_t)0x76B3, (q15_t)0x2FE4, (q15_t)0x76A0, (q15_t)0x3013,
	(q15_t)0x768E, (q15_t)0x3041, (q15_t)0x767B, (q15_t)0x3070,
	(q15_t)0x7668, (q15_t)0x309E, (q15_t)0x7654, (q15_t)0x30CD,
	(q15_t)0x7641, (q15_t)0x30FB, (q15_t)0x762E, (q15_t)0x312A,
	(q15_t)0x761B, (q15_t)0x3158, (q15_t)0x7607, (q15_t)0x3186,
	(q15_t)0x75F4, (q15_t)0x31B5, (q15_t)0x75E0, (q15_t)0x31E3,
	(q15_t)0x75CC, (q15_t)0x3211, (q15_t)0x75B9, (q15_t)0x3240,
	(q15_t)0x75A5, (q15_t)0x326E, (q15_t)0x7591, (q15_t)0x329C,
	(q15_t)0x757D, (q15_t)0x32CA, (q15_t)0x7569, (q15_t)0x32F8,
	(q15_t)0x7555, (q15_t)0x3326, (q15_t)0x7541, (q15_t)0x3354,
	(q15_t)0x752D, (q15_t)0x3382, (q15_t)0x7519, (q15_t)0x33B0,
	(q15_t)0x7504, (q15_t)0x33DE, (q15_t)0x74F0, (q15_t)0x340C,
	(q15_t)0x74DB, (q15_t)0x343A, (q15_t)0x74C7, (q15_t)0x3468,
	(q15_t)0x74B2, (q15_t)0x3496, (q15_t)0x749E, (q15_t)0x34C4,
	(q15_t)0x7489, (q15_t)0x34F2, (q15_t)0x7474, (q15_t)0x351F,
	(q15_t)0x745F, (q15_t)0x354D, (q15_t)0x744A, (q15_t)0x357B,
	(q15_t)0x7435, (q15_t)0x35A8, (q15_t)0x7420, (q15_t)0x35D6,
	(q15_t)0x740B, (q15_t)0x3604, (q15_t)0x73F6, (q15_t)0x3631,
	(q15_t)0x73E0, (q15_t)0x365F, (q15_t)0x73CB, (q15_t)0x368C,
	(q15_t)0x73B5, (q15_t)0x36BA, (q15_t)0x73A0, (q15_t)0x36E7,
	(q15_t)0x738A, (q15_t)0x3714, (q15_t)0x7375, (q15_t)0x3742,
	(q15_t)0x735F, (q15_t)0x376F, (q15_t)0x7349, (q15_t)0x379C,
	(q15_t)0x7333, (q15_t)0x37CA, (q15_t)0x731D, (q15_t)0x37F7,
	(q15_t)0x7307, (q15_t)0x3824, (q15_t)0x72F1, (q15_t)0x3851,
	(q15_t)0x72DB, (q15_t)0x387E, (q15_t)0x72C5, (q15_t)0x38AB,
	(q15_t)0x72AF, (q15_t)0x38D8, (q15_t)0x7298, (q15_t)0x3906,
	(q15_t)0x7282, (q15_t)0x3932, (q15_t)0x726B, (q15_t)0x395F,
	(q15_t)0x7255, (q15_t)0x398C, (q15_t)0x723E, (q15_t)0x39B9,
	(q15_t)0x7227, (q15_t)0x39E6, (q15_t)0x7211, (q15_t)0x3A13,
	(q15_t)0x71FA, (q15_t)0x3A40, (q15_t)0x71E3, (q15_t)0x3A6C,
	(q15_t)0x71CC, (q15_t)0x3A99, (q15_t)0x71B5, (q15_t)0x3AC6,
	(q15_t)0x719E, (q15_t)0x3AF2, (q15_t)0x7186, (q15_t)0x3B1F,
	(q15_t)0x716F, (q15_t)0x3B4C, (q15_t)0x7158, (q15_t)0x3B78,
	(q15_t)0x7141, (q15_t)0x3BA5, (q15_t)0x7129, (q15_t)0x3BD1,
	(q15_t)0x7112, (q15_t)0x3BFD, (q15_t)0x70FA, (q15_t)0x3C2A,
	(q15_t)0x70E2, (q15_t)0x3C56, (q15_t)0x70CB, (q15_t)0x3C83,
	(q15_t)0x70B3, (q15_t)0x3CAF, (q15_t)0x709B, (q15_t)0x3CDB,
	(q15_t)0x7083, (q15_t)0x3D07, (q15_t)0x706B, (q15_t)0x3D33,
	(q15_t)0x7053, (q15_t)0x3D60, (q15_t)0x703B, (q15_t)0x3D8C,
	(q15_t)0x7023, (q15_t)0x3DB8, (q15_t)0x700A, (q15_t)0x3DE4,
	(q15_t)0x6FF2, (q15_t)0x3E10, (q15_t)0x6FDA, (q15_t)0x3E3C,
	(q15_t)0x6FC1, (q15_t)0x3E68, (q15_t)0x6FA9, (q15_t)0x3E93,
	(q15_t)0x6F90, (q15_t)0x3EBF, (q15_t)0x6F77, (q15_t)0x3EEB,
	(q15_t)0x6F5F, (q15_t)0x3F17, (q15_t)0x6F46, (q15_t)0x3F43,
	(q15_t)0x6F2D, (q15_t)0x3F6E, (q15_t)0x6F14, (q15_t)0x3F9A,
	(q15_t)0x6EFB, (q15_t)0x3FC5, (q15_t)0x6EE2, (q15_t)0x3FF1,
	(q15_t)0x6EC9, (q15_t)0x401D, (q15_t)0x6EAF, (q15_t)0x4048,
	(q15_t)0x6E96, (q15_t)0x4073, (q15_t)0x6E7D, (q15_t)0x409F,
	(q15_t)0x6E63, (q15_t)0x40CA, (q15_t)0x6E4A, (q15_t)0x40F6,
	(q15_t)0x6E30, (q15_t)0x4121, (q15_t)0x6E17, (q15_t)0x414C,
	(q15_t)0x6DFD, (q15_t)0x4177, (q15_t)0x6DE3, (q15_t)0x41A2,
	(q15_t)0x6DCA, (q15_t)0x41CE, (q15_t)0x6DB0, (q15_t)0x41F9,
	(q15_t)0x6D96, (q15_t)0x4224, (q15_t)0x6D7C, (q15_t)0x424F,
	(q15_t)0x6D62, (q15_t)0x427A, (q15_t)0x6D48, (q15_t)0x42A5,
	(q15_t)0x6D2D, (q15_t)0x42D0, (q15_t)0x6D13, (q15_t)0x42FA,
	(q15_t)0x6CF9, (q15_t)0x4325, (q15_t)0x6CDE, (q15_t)0x4350,
	(q15_t)0x6CC4, (q15_t)0x437B, (q15_t)0x6CA9, (q15_t)0x43A5,
	(q15_t)0x6C8F, (q15_t)0x43D0, (q15_t)0x6C74, (q15_t)0x43FB,
	(q15_t)0x6C59, (q15_t)0x4425, (q15_t)0x6C3F, (q15_t)0x4450,
	(q15_t)0x6C24, (q15_t)0x447A, (q15_t)0x6C09, (q15_t)0x44A5,
	(q15_t)0x6BEE, (q15_t)0x44CF, (q15_t)0x6BD3, (q15_t)0x44FA,
	(q15_t)0x6BB8, (q15_t)0x4524, (q15_t)0x6B9C, (q15_t)0x454E,
	(q15_t)0x6B81, (q15_t)0x4578, (q15_t)0x6B66, (q15_t)0x45A3,
	(q15_t)0x6B4A, (q15_t)0x45CD, (q15_t)0x6B2F, (q15_t)0x45F7,
	(q15_t)0x6B13, (q15_t)0x4621, (q15_t)0x6AF8, (q15_t)0x464B,
	(q15_t)0x6ADC, (q15_t)0x4675, (q15_t)0x6AC1, (q15_t)0x469F,
	(q15_t)0x6AA5, (q15_t)0x46C9, (q15_t)0x6A89, (q15_t)0x46F3,
	(q15_t)0x6A6D, (q15_t)0x471C, (q15_t)0x6A51, (q15_t)0x4746,
	(q15_t)0x6A35, (q15_t)0x4770, (q15_t)0x6A19, (q15_t)0x479A,
	(q15_t)0x69FD, (q15_t)0x47C3, (q15_t)0x69E1, (q15_t)0x47ED,
	(q15_t)0x69C4, (q15_t)0x4816, (q15_t)0x69A8, (q15_t)0x4840,
	(q15_t)0x698C, (q15_t)0x4869, (q15_t)0x696F, (q15_t)0x4893,
	(q15_t)0x6953, (q15_t)0x48BC, (q15_t)0x6936, (q15_t)0x48E6,
	(q15_t)0x6919, (q15_t)0x490F, (q15_t)0x68FD, (q15_t)0x4938,
	(q15_t)0x68E0, (q15_t)0x4961, (q15_t)0x68C3, (q15_t)0x498A,
	(q15_t)0x68A6, (q15_t)0x49B4, (q15_t)0x6889, (q15_t)0x49DD,
	(q15_t)0x686C, (q15_t)0x4A06, (q15_t)0x684F, (q15_t)0x4A2F,
	(q15_t)0x6832, (q15_t)0x4A58, (q15_t)0x6815, (q15_t)0x4A81,
	(q15_t)0x67F7, (q15_t)0x4AA9, (q15_t)0x67DA, (q15_t)0x4AD2,
	(q15_t)0x67BD, (q15_t)0x4AFB, (q15_t)0x679F, (q15_t)0x4B24,
	(q15_t)0x6782, (q15_t)0x4B4C, (q15_t)0x6764, (q15_t)0x4B75,
	(q15_t)0x6746, (q15_t)0x4B9E, (q15_t)0x6729, (q15_t)0x4BC6,
	(q15_t)0x670B, (q15_t)0x4BEF, (q15_t)0x66ED, (q15_t)0x4C17,
	(q15_t)0x66CF, (q15_t)0x4C3F, (q15_t)0x66B1, (q15_t)0x4C68,
	(q15_t)0x6693, (q15_t)0x4C90, (q15_t)0x6675, (q15_t)0x4CB8,
	(q15_t)0x6657, (q15_t)0x4CE1, (q15_t)0x6639, (q15_t)0x4D09,
	(q15_t)0x661A, (q15_t)0x4D31, (q15_t)0x65FC, (q15_t)0x4D59,
	(q15_t)0x65DD, (q15_t)0x4D81, (q15_t)0x65BF, (q15_t)0x4DA9,
	(q15_t)0x65A0, (q15_t)0x4DD1, (q15_t)0x6582, (q15_t)0x4DF9,
	(q15_t)0x6563, (q15_t)0x4E21, (q15_t)0x6545, (q15_t)0x4E48,
	(q15_t)0x6526, (q15_t)0x4E70, (q15_t)0x6507, (q15_t)0x4E98,
	(q15_t)0x64E8, (q15_t)0x4EBF, (q15_t)0x64C9, (q15_t)0x4EE7,
	(q15_t)0x64AA, (q15_t)0x4F0F, (q15_t)0x648B, (q15_t)0x4F36,
	(q15_t)0x646C, (q15_t)0x4F5E, (q15_t)0x644D, (q15_t)0x4F85,
	(q15_t)0x642D, (q15_t)0x4FAC, (q15_t)0x640E, (q15_t)0x4FD4,
	(q15_t)0x63EF, (q15_t)0x4FFB, (q15_t)0x63CF, (q15_t)0x5022,
	(q15_t)0x63B0, (q15_t)0x5049, (q15_t)0x6390, (q15_t)0x5070,
	(q15_t)0x6371, (q15_t)0x5097, (q15_t)0x6351, (q15_t)0x50BF,
	(q15_t)0x6331, (q15_t)0x50E5, (q15_t)0x6311, (q15_t)0x510C,
	(q15_t)0x62F2, (q15_t)0x5133, (q15_t)0x62D2, (q15_t)0x515A,
	(q15_t)0x62B2, (q15_t)0x5181, (q15_t)0x6292, (q15_t)0x51A8,
	(q15_t)0x6271, (q15_t)0x51CE, (q15_t)0x6251, (q15_t)0x51F5,
	(q15_t)0x6231, (q15_t)0x521C, (q15_t)0x6211, (q15_t)0x5242,
	(q15_t)0x61F1, (q15_t)0x5269, (q15_t)0x61D0, (q15_t)0x528F,
	(q15_t)0x61B0, (q15_t)0x52B5, (q15_t)0x618F, (q15_t)0x52DC,
	(q15_t)0x616F, (q15_t)0x5302, (q15_t)0x614E, (q15_t)0x5328,
	(q15_t)0x612D, (q15_t)0x534E, (q15_t)0x610D, (q15_t)0x5375,
	(q15_t)0x60EC, (q15_t)0x539B, (q15_t)0x60CB, (q15_t)0x53C1,
	(q15_t)0x60AA, (q15_t)0x53E7, (q15_t)0x6089, (q15_t)0x540D,
	(q15_t)0x6068, (q15_t)0x5433, (q15_t)0x6047, (q15_t)0x5458,
	(q15_t)0x6026, (q15_t)0x547E, (q15_t)0x6004, (q15_t)0x54A4,
	(q15_t)0x5FE3, (q15_t)0x54CA, (q15_t)0x5FC2, (q15_t)0x54EF,
	(q15_t)0x5FA0, (q15_t)0x5515, (q15_t)0x5F7F, (q15_t)0x553A,
	(q15_t)0x5F5E, (q15_t)0x5560, (q15_t)0x5F3C, (q15_t)0x5585,
	(q15_t)0x5F1A, (q15_t)0x55AB, (q15_t)0x5EF9, (q15_t)0x55D0,
	(q15_t)0x5ED7, (q15_t)0x55F5, (q15_t)0x5EB5, (q15_t)0x561A,
	(q15_t)0x5E93, (q15_t)0x5640, (q15_t)0x5E71, (q15_t)0x5665,
	(q15_t)0x5E50, (q15_t)0x568A, (q15_t)0x5E2D, (q15_t)0x56AF,
	(q15_t)0x5E0B, (q15_t)0x56D4, (q15_t)0x5DE9, (q15_t)0x56F9,
	(q15_t)0x5DC7, (q15_t)0x571D, (q15_t)0x5DA5, (q15_t)0x5742,
	(q15_t)0x5D83, (q15_t)0x5767, (q15_t)0x5D60, (q15_t)0x578C,
	(q15_t)0x5D3E, (q15_t)0x57B0, (q15_t)0x5D1B, (q15_t)0x57D5,
	(q15_t)0x5CF9, (q15_t)0x57F9, (q15_t)0x5CD6, (q15_t)0x581E,
	(q15_t)0x5CB4, (q15_t)0x5842, (q15_t)0x5C91, (q15_t)0x5867,
	(q15_t)0x5C6E, (q15_t)0x588B, (q15_t)0x5C4B, (q15_t)0x58AF,
	(q15_t)0x5C29, (q15_t)0x58D4, (q15_t)0x5C06, (q15_t)0x58F8,
	(q15_t)0x5BE3, (q15_t)0x591C, (q15_t)0x5BC0, (q15_t)0x5940,
	(q15_t)0x5B9D, (q15_t)0x5964, (q15_t)0x5B79, (q15_t)0x5988,
	(q15_t)0x5B56, (q15_t)0x59AC, (q15_t)0x5B33, (q15_t)0x59D0,
	(q15_t)0x5B10, (q15_t)0x59F3, (q15_t)0x5AEC, (q15_t)0x5A17,
	(q15_t)0x5AC9, (q15_t)0x5A3B, (q15_t)0x5AA5, (q15_t)0x5A5E,
	(q15_t)0x5A82, (q15_t)0x5A82, (q15_t)0x5A5E, (q15_t)0x5AA5,
	(q15_t)0x5A3B, (q15_t)0x5AC9, (q15_t)0x5A17, (q15_t)0x5AEC,
	(q15_t)0x59F3, (q15_t)0x5B10, (q15_t)0x59D0, (q15_t)0x5B33,
	(q15_t)0x59AC, (q15_t)0x5B56, (q15_t)0x5988, (q15_t)0x5B79,
	(q15_t)0x5964, (q15_t)0x5B9D, (q15_t)0x5940, (q15_t)0x5BC0,
	(q15_t)0x591C, (q15_t)0x5BE3, (q15_t)0x58F8, (q15_t)0x5C06,
	(q15_t)0x58D4, (q15_t)0x5C29, (q15_t)0x58AF, (q15_t)0x5C4B,
	(q15_t)0x588B, (q15_t)0x5C6E, (q15_t)0x5867, (q15_t)0x5C91,
	(q15_t)0x5842, (q15_t)0x5CB4, (q15_t)0x581E, (q15_t)0x5CD6,
	(q15_t)0x57F9, (q15_t)0x5CF9, (q15_t)0x57D5, (q15_t)0x5D1B,
	(q15_t)0x57B0, (q15_t)0x5D3E, (q15_t)0x578C, (q15_t)0x5D60,
	(q15_t)0x5767, (q15_t)0x5D83, (q15_t)0x5742, (q15_t)0x5DA5,
	(q15_t)0x571D, (q15_t)0x5DC7, (q15_t)0x56F9, (q15_t)0x5DE9,
	(q15_t)0x56D4, (q15_t)0x5E0B, (q15_t)0x56AF, (q15_t)0x5E2D,
	(q15_t)0x568A, (q15_t)0x5E50, (q15_t)0x5665, (q15_t)0x5E71,
	(q15_t)0x5640, (q15_t)0x5E93, (q15_t)0x561A, (q15_t)0x5EB5,
	(q15_t)0x55F5, (q15_t)0x5ED7, (q15_t)0x55D0, (q15_t)0x5EF9,
	(q15_t)0x55AB, (q15_t)0x5F1A, (q15_t)0x5585, (q15_t)0x5F3C,
	(q15_t)0x5560, (q15_t)0x5F5E, (q15_t)0x553A, (q15_t)0x5F7F,
	(q15_t)0x5515, (q15_t)0x5FA0, (q15_t)0x54EF, (q15_t)0x5FC2,
	(q15_t)0x54CA, (q15_t)0x5FE3, (q15_t)0x54A4, (q15_t)0x6004,
	(q15_t)0x547E, (q15_t)0x6026, (q15_t)0x5458, (q15_t)0x6047,
	(q15_t)0x5433, (q15_t)0x6068, (q15_t)0x540D, (q15_t)0x6089,
	(q15_t)0x53E7, (q15_t)0x60AA, (q15_t)0x53C1, (q15_t)0x60CB,
	(q15_t)0x539B, (q15_t)0x60EC, (q15_t)0x5375, (q15_t)0x610D,
	(q15_t)0x534E, (q15_t)0x612D, (q15_t)0x5328, (q15_t)0x614E,
	(q15_t)0x5302, (q15_t)0x616F, (q15_t)0x52DC, (q15_t)0x618F,
	(q15_t)0x52B5, (q15_t)0x61B0, (q15_t)0x528F, (q15_t)0x61D0,
	(q15_t)0x5269, (q15_t)0x61F1, (q15_t)0x5242, (q15_t)0x6211,
	(q15_t)0x521C, (q15_t)0x6231, (q15_t)0x51F5, (q15_t)0x6251,
	(q15_t)0x51CE, (q15_t)0x6271, (q15_t)0x51A8, (q15_t)0x6292,
	(q15_t)0x5181, (q15_t)0x62B2, (q15_t)0x515A, (q15_t)0x62D2,
	(q15_t)0x5133, (q15_t)0x62F2, (q15_t)0x510C, (q15_t)0x6311,
	(q15_t)0x50E5, (q15_t)0x6331, (q15_t)0x50BF, (q15_t)0x6351,
	(q15_t)0x5097, (q15_t)0x6371, (q15_t)0x5070, (q15_t)0x6390,
	(q15_t)0x5049, (q15_t)0x63B0, (q15_t)0x5022, (q15_t)0x63CF,
	(q15_t)0x4FFB, (q15_t)0x63EF, (q15_t)0x4FD4, (q15_t)0x640E,
	(q15_t)0x4FAC, (q15_t)0x642D, (q15_t)0x4F85, (q15_t)0x644D,
	(q15_t)0x4F5E, (q15_t)0x646C, (q15_t)0x4F36, (q15_t)0x648B,
	(q15_t)0x4F0F, (q15_t)0x64AA, (q15_t)0x4EE7, (q15_t)0x64C9,
	(q15_t)0x4EBF, (q15_t)0x64E8, (q15_t)0x4E98, (q15_t)0x6507,
	(q15_t)0x4E70, (q15_t)0x6526, (q15_t)0x4E48, (q15_t)0x6545,
	(q15_t)0x4E21, (q15_t)0x6563, (q15_t)0x4DF9, (q15_t)0x6582,
	(q15_t)0x4DD1, (q15_t)0x65A0, (q15_t)0x4DA9, (q15_t)0x65BF,
	(q15_t)0x4D81, (q15_t)0x65DD, (q15_t)0x4D59, (q15_t)0x65FC,
	(q15_t)0x4D31, (q15_t)0x661A, (q15_t)0x4D09, (q15_t)0x6639,
	(q15_t)0x4CE1, (q15_t)0x6657, (q15_t)0x4CB8, (q15_t)0x6675,
	(q15_t)0x4C90, (q15_t)0x6693, (q15_t)0x4C68, (q15_t)0x66B1,
	(q15_t)0x4C3F, (q15_t)0x66CF, (q15_t)0x4C17, (q15_t)0x66ED,
	(q15_t)0x4BEF, (q15_t)0x670B, (q15_t)0x4BC6, (q15_t)0x6729,
	(q15_t)0x4B9E, (q15_t)0x6746, (q15_t)0x4B75, (q15_t)0x6764,
	(q15_t)0x4B4C, (q15_t)0x6782, (q15_t)0x4B24, (q15_t)0x679F,
	(q15_t)0x4AFB, (q15_t)0x67BD, (q15_t)0x4AD2, (q15_t)0x67DA,
	(q15_t)0x4AA9, (q15_t)0x67F7, (q15_t)0x4A81, (q15_t)0x6815,
	(q15_t)0x4A58, (q15_t)0x6832, (q15_t)0x4A2F, (q15_t)0x684F,
	(q15_t)0x4A06, (q15_t)0x686C, (q15_t)0x49DD, (q15_t)0x6889,
	(q15_t)0x49B4, (q15_t)0x68A6, (q15_t)0x498A, (q15_t)0x68C3,
	(q15_t)0x4961, (q15_t)0x68E0, (q15_t)0x4938, (q15_t)0x68FD,
	(q15_t)0x490F, (q15_t)0x6919, (q15_t)0x48E6, (q15_t)0x6936,
	(q15_t)0x48BC, (q15_t)0x6953, (q15_t)0x4893, (q15_t)0x696F,
	(q15_t)0x4869, (q15_t)0x698C, (q15_t)0x4840, (q15_t)0x69A8,
	(q15_t)0x4816, (q15_t)0x69C4, (q15_t)0x47ED, (q15_t)0x69E1,
	(q15_t)0x47C3, (q15_t)0x69FD, (q15_t)0x479A, (q15_t)0x6A19,
	(q15_t)0x4770, (q15_t)0x6A35, (q15_t)0x4746, (q15_t)0x6A51,
	(q15_t)0x471C, (q15_t)0x6A6D, (q15_t)0x46F3, (q15_t)0x6A89,
	(q15_t)0x46C9, (q15_t)0x6AA5, (q15_t)0x469F, (q15_t)0x6AC1,
	(q15_t)0x4675, (q15_t)0x6ADC, (q15_t)0x464B, (q15_t)0x6AF8,
	(q15_t)0x4621, (q15_t)0x6B13, (q15_t)0x45F7, (q15_t)0x6B2F,
	(q15_t)0x45CD, (q15_t)0x6B4A, (q15_t)0x45A3, (q15_t)0x6B66,
	(q15_t)0x4578, (q15_t)0x6B81, (q15_t)0x454E, (q15_t)0x6B9C,
	(q15_t)0x4524, (q15_t)0x6BB8, (q15_t)0x44FA, (q15_t)0x6BD3,
	(q15_t)0x44CF, (q15_t)0x6BEE, (q15_t)0x44A5, (q15_t)0x6C09,
	(q15_t)0x447A, (q15_t)0x6C24, (q15_t)0x4450, (q15_t)0x6C3F,
	(q15_t)0x4425, (q15_t)0x6C59, (q15_t)0x43FB, (q15_t)0x6C74,
	(q15_t)0x43D0, (q15_t)0x6C8F, (q15_t)0x43A5, (q15_t)0x6CA9,
	(q15_t)0x437B, (q15_t)0x6CC4, (q15_t)0x4350, (q15_t)0x6CDE,
	(q15_t)0x4325, (q15_t)0x6CF9, (q15_t)0x42FA, (q15_t)0x6D13,
	(q15_t)0x42D0, (q15_t)0x6D2D, (q15_t)0x42A5, (q15_t)0x6D48,
	(q15_t)0x427A, (q15_t)0x6D62, (q15_t)0x424F, (q15_t)0x6D7C,
	(q15_t)0x4224, (q15_t)0x6D96, (q15_t)0x41F9, (q15_t)0x6DB0,
	(q15_t)0x41CE, (q15_t)0x6DCA, (q15_t)0x41A2, (q15_t)0x6DE3,
	(q15_t)0x4177, (q15_t)0x6DFD, (q15_t)0x414C, (q15_t)0x6E17,
	(q15_t)0x4121, (q15_t)0x6E30, (q15_t)0x40F6, (q15_t)0x6E4A,
	(q15_t)0x40CA, (q15_t)0x6E63, (q15_t)0x409F, (q15_t)0x6E7D,
	(q15_t)0x4073, (q15_t)0x6E96, (q15_t)0x4048, (q15_t)0x6EAF,
	(q15_t)0x401D, (q15_t)0x6EC9, (q15_t)0x3FF1, (q15_t)0x6EE2,
	(q15_t)0x3FC5, (q15_t)0x6EFB, (q15_t)0x3F9A, (q15_t)0x6F14,
	(q15_t)0x3F6E, (q15_t)0x6F2D, (q15_t)0x3F43, (q15_t)0x6F46,
	(q15_t)0x3F17, (q15_t)0x6F5F, (q15_t)0x3EEB, (q15_t)0x6F77,
	(q15_t)0x3EBF, (q15_t)0x6F90, (q15_t)0x3E93, (q15_t)0x6FA9,
	(q15_t)0x3E68, (q15_t)0x6FC1, (q15_t)0x3E3C, (q15_t)0x6FDA,
	(q15_t)0x3E10, (q15_t)0x6FF2, (q15_t)0x3DE4, (q15_t)0x700A,
	(q15_t)0x3DB8, (q15_t)0x7023, (q15_t)0x3D8C, (q15_t)0x703B,
	(q15_t)0x3D60, (q15_t)0x7053, (q15_t)0x3D33, (q15_t)0x706B,
	(q15_t)0x3D07, (q15_t)0x7083, (q15_t)0x3CDB, (q15_t)0x709B,
	(q15_t)0x3CAF, (q15_t)0x70B3, (q15_t)0x3C83, (q15_t)0x70CB,
	(q15_t)0x3C56, (q15_t)0x70E2, (q15_t)0x3C2A, (q15_t)0x70FA,
	(q15_t)0x3BFD, (q15_t)0x7112, (q15_t)0x3BD1, (q15_t)0x7129,
	(q15_t)0x3BA5, (q15_t)0x7141, (q15_t)0x3B78, (q15_t)0x7158,
	(q15_t)0x3B4C, (q15_t)0x716F, (q15_t)0x3B1F, (q15_t)0x7186,
	(q15_t)0x3AF2, (q15_t)0x719E, (q15_t)0x3AC6, (q15_t)0x71B5,
	(q15_t)0x3A99, (q15_t)0x71CC, (q15_t)0x3A6C, (q15_t)0x71E3,
	(q15_t)0x3A40, (q15_t)0x71FA, (q15_t)0x3A13, (q15_t)0x7211,
	(q15_t)0x39E6, (q15_t)0x7227, (q15_t)0x39B9, (q15_t)0x723E,
	(q15_t)0x398C, (q15_t)0x7255, (q15_t)0x395F, (q15_t)0x726B,
	(q15_t)0x3932, (q15_t)0x7282, (q15_t)0x3906, (q15_t)0x7298,
	(q15_t)0x38D8, (q15_t)0x72AF, (q15_t)0x38AB, (q15_t)0x72C5,
	(q15_t)0x387E, (q15_t)0x72DB, (q15_t)0x3851, (q15_t)0x72F1,
	(q15_t)0x3824, (q15_t)0x7307, (q15_t)0x37F7, (q15_t)0x731D,
	(q15_t)0x37CA, (q15_t)0x7333, (q15_t)0x379C, (q15_t)0x7349,
	(q15_t)0x376F, (q15_t)0x735F, (q15_t)0x3742, (q15_t)0x7375,
	(q15_t)0x3714, (q15_t)0x738A, (q15_t)0x36E7, (q15_t)0x73A0,
	(q15_t)0x36BA, (q15_t)0x73B5, (q15_t)0x368C, (q15_t)0x73CB,
	(q15_t)0x365F, (q15_t)0x73E0, (q15_t)0x3631, (q15_t)0x73F6,
	(q15_t)0x3604, (q15_t)0x740B, (q15_t)0x35D6, (q15_t)0x7420,
	(q15_t)0x35A8, (q15_t)0x7435, (q15_t)0x357B, (q15_t)0x744A,
	(q15_t)0x354D, (q15_t)0x745F, (q15_t)0x351F, (q15_t)0x7474,
	(q15_t)0x34F2, (q15_t)0x7489, (q15_t)0x34C4, (q15_t)0x749E,
	(q15_t)0x3496, (q15_t)0x74B2, (q15_t)0x3468, (q15_t)0x74C7,
	(q15_t)0x343A, (q15_t)0x74DB, (q15_t)0x340C, (q15_t)0x74F0,
	(q15_t)0x33DE, (q15_t)0x7504, (q15_t)0x33B0, (q15_t)0x7519,
	(q15_t)0x3382, (q15_t)0x752D, (q15_t)0x3354, (q15_t)0x7541,
	(q15_t)0x3326, (q15_t)0x7555, (q15_t)0x32F8, (q15_t)0x7569,
	(q15_t)0x32CA, (q15_t)0x757D, (q15_t)0x329C, (q15_t)0x7591,
	(q15_t)0x326E, (q15_t)0x75A5, (q15_t)0x3240, (q15_t)0x75B9,
	(q15_t)0x3211, (q15_t)0x75CC, (q15_t)0x31E3, (q15_t)0x75E0,
	(q15_t)0x31B5, (q15_t)0x75F4, (q15_t)0x3186, (q15_t)0x7607,
	(q15_t)0x3158, (q15_t)0x761B, (q15_t)0x312A, (q15_t)0x762E,
	(q15_t)0x30FB, (q15_t)0x7641, (q15_t)0x30CD, (q15_t)0x7654,
	(q15_t)0x309E, (q15_t)0x7668, (q15_t)0x3070, (q15_t)0x767B,
	(q15_t)0x3041, (q15_t)0x768E, (q15_t)0x3013, (q15_t)0x76A0,
	(q15_t)0x2FE4, (q15_t)0x76B3, (q15_t)0x2FB5, (q15_t)0x76C6,
	(q15_t)0x2F87, (q15_t)0x76D9, (q15_t)0x2F58, (q15_t)0x76EB,
	(q15_t)0x2F29, (q15_t)0x76FE, (q15_t)0x2EFB, (q15_t)0x7710,
	(q15_t)0x2ECC, (q15_t)0x7723, (q15_t)0x2E9D, (q15_t)0x7735,
	(q15_t)0x2E6E, (q15_t)0x7747, (q15_t)0x2E3F, (q15_t)0x775A,
	(q15_t)0x2E11, (q15_t)0x776C, (q15_t)0x2DE2, (q15_t)0x777E,
	(q15_t)0x2DB3, (q15_t)0x7790, (q15_t)0x2D84, (q15_t)0x77A2,
	(q15_t)0x2D55, (q15_t)0x77B4, (q15_t)0x2D26, (q15_t)0x77C5,
	(q15_t)0x2CF7, (q15_t)0x77D7, (q15_t)0x2CC8, (q15_t)0x77E9,
	(q15_t)0x2C98, (q15_t)0x77FA, (q15_t)0x2C69, (q15_t)0x780C,
	(q15_t)0x2C3A, (q15_t)0x781D, (q15_t)0x2C0B, (q15_t)0x782E,
	(q15_t)0x2BDC, (q15_t)0x7840, (q15_t)0x2BAD, (q15_t)0x7851,
	(q15_t)0x2B7D, (q15_t)0x7862, (q15_t)0x2B4E, (q15_t)0x7873,
	(q15_t)0x2B1F, (q15_t)0x7884, (q15_t)0x2AEF, (q15_t)0x7895,
	(q15_t)0x2AC0, (q15_t)0x78A6, (q15_t)0x2A91, (q15_t)0x78B6,
	(q15_t)0x2A61, (q15_t)0x78C7, (q15_t)0x2A32, (q15_t)0x78D8,
	(q15_t)0x2A02, (q15_t)0x78E8, (q15_t)0x29D3, (q15_t)0x78F9,
	(q15_t)0x29A3, (q15_t)0x7909, (q15_t)0x2974, (q15_t)0x7919,
	(q15_t)0x2944, (q15_t)0x792A, (q15_t)0x2915, (q15_t)0x793A,
	(q15_t)0x28E5, (q15_t)0x794A, (q15_t)0x28B5, (q15_t)0x795A,
	(q15_t)0x2886, (q15_t)0x796A, (q15_t)0x2856, (q15_t)0x797A,
	(q15_t)0x2826, (q15_t)0x798A, (q15_t)0x27F6, (q15_t)0x7999,
	(q15_t)0x27C7, (q15_t)0x79A9, (q15_t)0x2797, (q15_t)0x79B9,
	(q15_t)0x2767, (q15_t)0x79C8, (q15_t)0x2737, (q15_t)0x79D8,
	(q15_t)0x2707, (q15_t)0x79E7, (q15_t)0x26D8, (q15_t)0x79F6,
	(q15_t)0x26A8, (q15_t)0x7A05, (q15_t)0x2678, (q15_t)0x7A15,
	(q15_t)0x2648, (q15_t)0x7A24, (q15_t)0x2618, (q15_t)0x7A33,
	(q15_t)0x25E8, (q15_t)0x7A42, (q15_t)0x25B8, (q15_t)0x7A50,
	(q15_t)0x2588, (q15_t)0x7A5F, (q15_t)0x2558, (q15_t)0x7A6E,
	(q15_t)0x2528, (q15_t)0x7A7D, (q15_t)0x24F7, (q15_t)0x7A8B,
	(q15_t)0x24C7, (q15_t)0x7A9A, (q15_t)0x2497, (q15_t)0x7AA8,
	(q15_t)0x2467, (q15_t)0x7AB6, (q15_t)0x2437, (q15_t)0x7AC5,
	(q15_t)0x2407, (q15_t)0x7AD3, (q15_t)0x23D6, (q15_t)0x7AE1,
	(q15_t)0x23A6, (q15_t)0x7AEF, (q15_t)0x2376, (q15_t)0x7AFD,
	(q15_t)0x2345, (q15_t)0x7B0B, (q15_t)0x2315, (q15_t)0x7B19,
	(q15_t)0x22E5, (q15_t)0x7B26, (q15_t)0x22B4, (q15_t)0x7B34,
	(q15_t)0x2284, (q15_t)0x7B42, (q15_t)0x2254, (q15_t)0x7B4F,
	(q15_t)0x2223, (q15_t)0x7B5D, (q15_t)0x21F3, (q15_t)0x7B6A,
	(q15_t)0x21C2, (q15_t)0x7B77, (q15_t)0x2192, (q15_t)0x7B84,
	(q15_t)0x2161, (q15_t)0x7B92, (q15_t)0x2131, (q15_t)0x7B9F,
	(q15_t)0x2100, (q15_t)0x7BAC, (q15_t)0x20D0, (q15_t)0x7BB9,
	(q15_t)0x209F, (q15_t)0x7BC5, (q15_t)0x206E, (q15_t)0x7BD2,
	(q15_t)0x203E, (q15_t)0x7BDF, (q15_t)0x200D, (q15_t)0x7BEB,
	(q15_t)0x1FDC, (q15_t)0x7BF8, (q15_t)0x1FAC, (q15_t)0x7C05,
	(q15_t)0x1F7B, (q15_t)0x7C11, (q15_t)0x1F4A, (q15_t)0x7C1D,
	(q15_t)0x1F19, (q15_t)0x7C29, (q15_t)0x1EE9, (q15_t)0x7C36,
	(q15_t)0x1EB8, (q15_t)0x7C42, (q15_t)0x1E87, (q15_t)0x7C4E,
	(q15_t)0x1E56, (q15_t)0x7C5A, (q15_t)0x1E25, (q15_t)0x7C66,
	(q15_t)0x1DF5, (q15_t)0x7C71, (q15_t)0x1DC4, (q15_t)0x7C7D,
	(q15_t)0x1D93, (q15_t)0x7C89, (q15_t)0x1D62, (q15_t)0x7C94,
	(q15_t)0x1D31, (q15_t)0x7CA0, (q15_t)0x1D00, (q15_t)0x7CAB,
	(q15_t)0x1CCF, (q15_t)0x7CB7, (q15_t)0x1C9E, (q15_t)0x7CC2,
	(q15_t)0x1C6D, (q15_t)0x7CCD, (q15_t)0x1C3C, (q15_t)0x7CD8,
	(q15_t)0x1C0B, (q15_t)0x7CE3, (q15_t)0x1BDA, (q15_t)0x7CEE,
	(q15_t)0x1BA9, (q15_t)0x7CF9, (q15_t)0x1B78, (q15_t)0x7D04,
	(q15_t)0x1B47, (q15_t)0x7D0F, (q15_t)0x1B16, (q15_t)0x7D19,
	(q15_t)0x1AE4, (q15_t)0x7D24, (q15_t)0x1AB3, (q15_t)0x7D2F,
	(q15_t)0x1A82, (q15_t)0x7D39, (q15_t)0x1A51, (q15_t)0x7D43,
	(q15_t)0x1A20, (q15_t)0x7D4E, (q15_t)0x19EF, (q15_t)0x7D58,
	(q15_t)0x19BD, (q15_t)0x7D62, (q15_t)0x198C, (q15_t)0x7D6C,
	(q15_t)0x195B, (q15_t)0x7D76, (q15_t)0x192A, (q15_t)0x7D80,
	(q15_t)0x18F8, (q15_t)0x7D8A, (q15_t)0x18C7, (q15_t)0x7D94,
	(q15_t)0x1896, (q15_t)0x7D9D, (q15_t)0x1864, (q15_t)0x7DA7,
	(q15_t)0x1833, (q15_t)0x7DB0, (q15_t)0x1802, (q15_t)0x7DBA,
	(q15_t)0x17D0, (q15_t)0x7DC3, (q15_t)0x179F, (q15_t)0x7DCD,
	(q15_t)0x176D, (q15_t)0x7DD6, (q15_t)0x173C, (q15_t)0x7DDF,
	(q15_t)0x170A, (q15_t)0x7DE8, (q15_t)0x16D9, (q15_t)0x7DF1,
	(q15_t)0x16A8, (q15_t)0x7DFA, (q15_t)0x1676, (q15_t)0x7E03,
	(q15_t)0x1645, (q15_t)0x7E0C, (q15_t)0x1613, (q15_t)0x7E14,
	(q15_t)0x15E2, (q15_t)0x7E1D, (q15_t)0x15B0, (q15_t)0x7E26,
	(q15_t)0x157F, (q15_t)0x7E2E, (q15_t)0x154D, (q15_t)0x7E37,
	(q15_t)0x151B, (q15_t)0x7E3F, (q15_t)0x14EA, (q15_t)0x7E47,
	(q15_t)0x14B8, (q15_t)0x7E4F, (q15_t)0x1487, (q15_t)0x7E57,
	(q15_t)0x1455, (q15_t)0x7E5F, (q15_t)0x1423, (q15_t)0x7E67,
	(q15_t)0x13F2, (q15_t)0x7E6F, (q15_t)0x13C0, (q15_t)0x7E77,
	(q15_t)0x138E, (q15_t)0x7E7F, (q15_t)0x135D, (q15_t)0x7E86,
	(q15_t)0x132B, (q15_t)0x7E8E, (q15_t)0x12F9, (q15_t)0x7E95,
	(q15_t)0x12C8, (q15_t)0x7E9D, (q15_t)0x1296, (q15_t)0x7EA4,
	(q15_t)0x1264, (q15_t)0x7EAB, (q15_t)0x1232, (q15_t)0x7EB3,
	(q15_t)0x1201, (q15_t)0x7EBA, (q15_t)0x11CF, (q15_t)0x7EC1,
	(q15_t)0x119D, (q15_t)0x7EC8, (q15_t)0x116B, (q15_t)0x7ECF,
	(q15_t)0x1139, (q15_t)0x7ED5, (q15_t)0x1108, (q15_t)0x7EDC,
	(q15_t)0x10D6, (q15_t)0x7EE3, (q15_t)0x10A4, (q15_t)0x7EE9,
	(q15_t)0x1072, (q15_t)0x7EF0, (q15_t)0x1040, (q15_t)0x7EF6,
	(q15_t)0x100E, (q15_t)0x7EFD, (q15_t)0x0FDD, (q15_t)0x7F03,
	(q15_t)0x0FAB, (q15_t)0x7F09, (q15_t)0x0F79, (q15_t)0x7F0F,
	(q15_t)0x0F47, (q15_t)0x7F15, (q15_t)0x0F15, (q15_t)0x7F1B,
	(q15_t)0x0EE3, (q15_t)0x7F21, (q15_t)0x0EB1, (q15_t)0x7F27,
	(q15_t)0x0E7F, (q15_t)0x7F2D, (q15_t)0x0E4D, (q15_t)0x7F32,
	(q15_t)0x0E1B, (q15_t)0x7F38, (q15_t)0x0DE9, (q15_t)0x7F3D,
	(q15_t)0x0DB7, (q15_t)0x7F43, (q15_t)0x0D85, (q15_t)0x7F48,
	(q15_t)0x0D53, (q15_t)0x7F4D, (q15_t)0x0D21, (q15_t)0x7F53,
	(q15_t)0x0CEF, (q15_t)0x7F58, (q15_t)0x0CBD, (q15_t)0x7F5D,
	(q15_t)0x0C8B, (q15_t)0x7F62, (q15_t)0x0C59, (q15_t)0x7F67,
	(q15_t)0x0C27, (q15_t)0x7F6B, (q15_t)0x0BF5, (q15_t)0x7F70,
	(q15_t)0x0BC3, (q15_t)0x7F75, (q15_t)0x0B91, (q15_t)0x7F79,
	(q15_t)0x0B5F, (q15_t)0x7F7E, (q15_t)0x0B2D, (q15_t)0x7F82,
	(q15_t)0x0AFB, (q15_t)0x7F87, (q15_t)0x0AC9, (q15_t)0x7F8B,
	(q15_t)0x0A97, (q15_t)0x7F8F, (q15_t)0x0A65, (q15_t)0x7F93,
	(q15_t)0x0A33, (q15_t)0x7F97, (q15_t)0x0A00, (q15_t)0x7F9B,
	(q15_t)0x09CE, (q15_t)0x7F9F, (q15_t)0x099C, (q15_t)0x7FA3,
	(q15_t)0x096A, (q15_t)0x7FA7, (q15_t)0x0938, (q15_t)0x7FAA,
	(q15_t)0x0906, (q15_t)0x7FAE, (q15_t)0x08D4, (q15_t)0x7FB1,
	(q15_t)0x08A2, (q15_t)0x7FB5, (q15_t)0x086F, (q15_t)0x7FB8,
	(q15_t)0x083D, (q15_t)0x7FBC, (q15_t)0x080B, (q15_t)0x7FBF,
	(q15_t)0x07D9, (q15_t)0x7FC2, (q15_t)0x07A7, (q15_t)0x7FC5,
	(q15_t)0x0775, (q15_t)0x7FC8, (q15_t)0x0742, (q15_t)0x7FCB,
	(q15_t)0x0710, (q15_t)0x7FCE, (q15_t)0x06DE, (q15_t)0x7FD0,
	(q15_t)0x06AC, (q15_t)0x7FD3, (q15_t)0x067A, (q15_t)0x7FD6,
	(q15_t)0x0647, (q15_t)0x7FD8, (q15_t)0x0615, (q15_t)0x7FDA,
	(q15_t)0x05E3, (q15_t)0x7FDD, (q15_t)0x05B1, (q15_t)0x7FDF,
	(q15_t)0x057F, (q15_t)0x7FE1, (q15_t)0x054C, (q15_t)0x7FE3,
	(q15_t)0x051A, (q15_t)0x7FE5, (q15_t)0x04E8, (q15_t)0x7FE7,
	(q15_t)0x04B6, (q15_t)0x7FE9, (q15_t)0x0483, (q15_t)0x7FEB,
	(q15_t)0x0451, (q15_t)0x7FED, (q15_t)0x041F, (q15_t)0x7FEE,
	(q15_t)0x03ED, (q15_t)0x7FF0, (q15_t)0x03BA, (q15_t)0x7FF2,
	(q15_t)0x0388, (q15_t)0x7FF3, (q15_t)0x0356, (q15_t)0x7FF4,
	(q15_t)0x0324, (q15_t)0x7FF6, (q15_t)0x02F1, (q15_t)0x7FF7,
	(q15_t)0x02BF, (q15_t)0x7FF8, (q15_t)0x028D, (q15_t)0x7FF9,
	(q15_t)0x025B, (q15_t)0x7FFA, (q15_t)0x0228, (q15_t)0x7FFB,
	(q15_t)0x01F6, (q15_t)0x7FFC, (q15_t)0x01C4, (q15_t)0x7FFC,
	(q15_t)0x0192, (q15_t)0x7FFD, (q15_t)0x015F, (q15_t)0x7FFE,
	(q15_t)0x012D, (q15_t)0x7FFE, (q15_t)0x00FB, (q15_t)0x7FFF,
	(q15_t)0x00C9, (q15_t)0x7FFF, (q15_t)0x0096, (q15_t)0x7FFF,
	(q15_t)0x0064, (q15_t)0x7FFF, (q15_t)0x0032, (q15_t)0x7FFF,
	(q15_t)0x0000, (q15_t)0x7FFF, (q15_t)0xFFCD, (q15_t)0x7FFF,
	(q15_t)0xFF9B, (q15_t)0x7FFF, (q15_t)0xFF69, (q15_t)0x7FFF,
	(q15_t)0xFF36, (q15_t)0x7FFF, (q15_t)0xFF04, (q15_t)0x7FFF,
	(q15_t)0xFED2, (q15_t)0x7FFE, (q15_t)0xFEA0, (q15_t)0x7FFE,
	(q15_t)0xFE6D, (q15_t)0x7FFD, (q15_t)0xFE3B, (q15_t)0x7FFC,
	(q15_t)0xFE09, (q15_t)0x7FFC, (q15_t)0xFDD7, (q15_t)0x7FFB,
	(q15_t)0xFDA4, (q15_t)0x7FFA, (q15_t)0xFD72, (q15_t)0x7FF9,
	(q15_t)0xFD40, (q15_t)0x7FF8, (q15_t)0xFD0E, (q15_t)0x7FF7,
	(q15_t)0xFCDB, (q15_t)0x7FF6, (q15_t)0xFCA9, (q15_t)0x7FF4,
	(q15_t)0xFC77, (q15_t)0x7FF3, (q15_t)0xFC45, (q15_t)0x7FF2,
	(q15_t)0xFC12, (q15_t)0x7FF0, (q15_t)0xFBE0, (q15_t)0x7FEE,
	(q15_t)0xFBAE, (q15_t)0x7FED, (q15_t)0xFB7C, (q15_t)0x7FEB,
	(q15_t)0xFB49, (q15_t)0x7FE9, (q15_t)0xFB17, (q15_t)0x7FE7,
	(q15_t)0xFAE5, (q15_t)0x7FE5, (q15_t)0xFAB3, (q15_t)0x7FE3,
	(q15_t)0xFA80, (q15_t)0x7FE1, (q15_t)0xFA4E, (q15_t)0x7FDF,
	(q15_t)0xFA1C, (q15_t)0x7FDD, (q15_t)0xF9EA, (q15_t)0x7FDA,
	(q15_t)0xF9B8, (q15_t)0x7FD8, (q15_t)0xF985, (q15_t)0x7FD6,
	(q15_t)0xF953, (q15_t)0x7FD3, (q15_t)0xF921, (q15_t)0x7FD0,
	(q15_t)0xF8EF, (q15_t)0x7FCE, (q15_t)0xF8BD, (q15_t)0x7FCB,
	(q15_t)0xF88A, (q15_t)0x7FC8, (q15_t)0xF858, (q15_t)0x7FC5,
	(q15_t)0xF826, (q15_t)0x7FC2, (q15_t)0xF7F4, (q15_t)0x7FBF,
	(q15_t)0xF7C2, (q15_t)0x7FBC, (q15_t)0xF790, (q15_t)0x7FB8,
	(q15_t)0xF75D, (q15_t)0x7FB5, (q15_t)0xF72B, (q15_t)0x7FB1,
	(q15_t)0xF6F9, (q15_t)0x7FAE, (q15_t)0xF6C7, (q15_t)0x7FAA,
	(q15_t)0xF695, (q15_t)0x7FA7, (q15_t)0xF663, (q15_t)0x7FA3,
	(q15_t)0xF631, (q15_t)0x7F9F, (q15_t)0xF5FF, (q15_t)0x7F9B,
	(q15_t)0xF5CC, (q15_t)0x7F97, (q15_t)0xF59A, (q15_t)0x7F93,
	(q15_t)0xF568, (q15_t)0x7F8F, (q15_t)0xF536, (q15_t)0x7F8B,
	(q15_t)0xF504, (q15_t)0x7F87, (q15_t)0xF4D2, (q15_t)0x7F82,
	(q15_t)0xF4A0, (q15_t)0x7F7E, (q15_t)0xF46E, (q15_t)0x7F79,
	(q15_t)0xF43C, (q15_t)0x7F75, (q15_t)0xF40A, (q15_t)0x7F70,
	(q15_t)0xF3D8, (q15_t)0x7F6B, (q15_t)0xF3A6, (q15_t)0x7F67,
	(q15_t)0xF374, (q15_t)0x7F62, (q15_t)0xF342, (q15_t)0x7F5D,
	(q15_t)0xF310, (q15_t)0x7F58, (q15_t)0xF2DE, (q15_t)0x7F53,
	(q15_t)0xF2AC, (q15_t)0x7F4D, (q15_t)0xF27A, (q15_t)0x7F48,
	(q15_t)0xF248, (q15_t)0x7F43, (q15_t)0xF216, (q15_t)0x7F3D,
	(q15_t)0xF1E4, (q15_t)0x7F38, (q15_t)0xF1B2, (q15_t)0x7F32,
	(q15_t)0xF180, (q15_t)0x7F2D, (q15_t)0xF14E, (q15_t)0x7F27,
	(q15_t)0xF11C, (q15_t)0x7F21, (q15_t)0xF0EA, (q15_t)0x7F1B,
	(q15_t)0xF0B8, (q15_t)0x7F15, (q15_t)0xF086, (q15_t)0x7F0F,
	(q15_t)0xF054, (q15_t)0x7F09, (q15_t)0xF022, (q15_t)0x7F03,
	(q15_t)0xEFF1, (q15_t)0x7EFD, (q15_t)0xEFBF, (q15_t)0x7EF6,
	(q15_t)0xEF8D, (q15_t)0x7EF0, (q15_t)0xEF5B, (q15_t)0x7EE9,
	(q15_t)0xEF29, (q15_t)0x7EE3, (q15_t)0xEEF7, (q15_t)0x7EDC,
	(q15_t)0xEEC6, (q15_t)0x7ED5, (q15_t)0xEE94, (q15_t)0x7ECF,
	(q15_t)0xEE62, (q15_t)0x7EC8, (q15_t)0xEE30, (q15_t)0x7EC1,
	(q15_t)0xEDFE, (q15_t)0x7EBA, (q15_t)0xEDCD, (q15_t)0x7EB3,
	(q15_t)0xED9B, (q15_t)0x7EAB, (q15_t)0xED69, (q15_t)0x7EA4,
	(q15_t)0xED37, (q15_t)0x7E9D, (q15_t)0xED06, (q15_t)0x7E95,
	(q15_t)0xECD4, (q15_t)0x7E8E, (q15_t)0xECA2, (q15_t)0x7E86,
	(q15_t)0xEC71, (q15_t)0x7E7F, (q15_t)0xEC3F, (q15_t)0x7E77,
	(q15_t)0xEC0D, (q15_t)0x7E6F, (q15_t)0xEBDC, (q15_t)0x7E67,
	(q15_t)0xEBAA, (q15_t)0x7E5F, (q15_t)0xEB78, (q15_t)0x7E57,
	(q15_t)0xEB47, (q15_t)0x7E4F, (q15_t)0xEB15, (q15_t)0x7E47,
	(q15_t)0xEAE4, (q15_t)0x7E3F, (q15_t)0xEAB2, (q15_t)0x7E37,
	(q15_t)0xEA80, (q15_t)0x7E2E, (q15_t)0xEA4F, (q15_t)0x7E26,
	(q15_t)0xEA1D, (q15_t)0x7E1D, (q15_t)0xE9EC, (q15_t)0x7E14,
	(q15_t)0xE9BA, (q15_t)0x7E0C, (q15_t)0xE989, (q15_t)0x7E03,
	(q15_t)0xE957, (q15_t)0x7DFA, (q15_t)0xE926, (q15_t)0x7DF1,
	(q15_t)0xE8F5, (q15_t)0x7DE8, (q15_t)0xE8C3, (q15_t)0x7DDF,
	(q15_t)0xE892, (q15_t)0x7DD6, (q15_t)0xE860, (q15_t)0x7DCD,
	(q15_t)0xE82F, (q15_t)0x7DC3, (q15_t)0xE7FD, (q15_t)0x7DBA,
	(q15_t)0xE7CC, (q15_t)0x7DB0, (q15_t)0xE79B, (q15_t)0x7DA7,
	(q15_t)0xE769, (q15_t)0x7D9D, (q15_t)0xE738, (q15_t)0x7D94,
	(q15_t)0xE707, (q15_t)0x7D8A, (q15_t)0xE6D5, (q15_t)0x7D80,
	(q15_t)0xE6A4, (q15_t)0x7D76, (q15_t)0xE673, (q15_t)0x7D6C,
	(q15_t)0xE642, (q15_t)0x7D62, (q15_t)0xE610, (q15_t)0x7D58,
	(q15_t)0xE5DF, (q15_t)0x7D4E, (q15_t)0xE5AE, (q15_t)0x7D43,
	(q15_t)0xE57D, (q15_t)0x7D39, (q15_t)0xE54C, (q15_t)0x7D2F,
	(q15_t)0xE51B, (q15_t)0x7D24, (q15_t)0xE4E9, (q15_t)0x7D19,
	(q15_t)0xE4B8, (q15_t)0x7D0F, (q15_t)0xE487, (q15_t)0x7D04,
	(q15_t)0xE456, (q15_t)0x7CF9, (q15_t)0xE425, (q15_t)0x7CEE,
	(q15_t)0xE3F4, (q15_t)0x7CE3, (q15_t)0xE3C3, (q15_t)0x7CD8,
	(q15_t)0xE392, (q15_t)0x7CCD, (q15_t)0xE361, (q15_t)0x7CC2,
	(q15_t)0xE330, (q15_t)0x7CB7, (q15_t)0xE2FF, (q15_t)0x7CAB,
	(q15_t)0xE2CE, (q15_t)0x7CA0, (q15_t)0xE29D, (q15_t)0x7C94,
	(q15_t)0xE26C, (q15_t)0x7C89, (q15_t)0xE23B, (q15_t)0x7C7D,
	(q15_t)0xE20A, (q15_t)0x7C71, (q15_t)0xE1DA, (q15_t)0x7C66,
	(q15_t)0xE1A9, (q15_t)0x7C5A, (q15_t)0xE178, (q15_t)0x7C4E,
	(q15_t)0xE147, (q15_t)0x7C42, (q15_t)0xE116, (q15_t)0x7C36,
	(q15_t)0xE0E6, (q15_t)0x7C29, (q15_t)0xE0B5, (q15_t)0x7C1D,
	(q15_t)0xE084, (q15_t)0x7C11, (q15_t)0xE053, (q15_t)0x7C05,
	(q15_t)0xE023, (q15_t)0x7BF8, (q15_t)0xDFF2, (q15_t)0x7BEB,
	(q15_t)0xDFC1, (q15_t)0x7BDF, (q15_t)0xDF91, (q15_t)0x7BD2,
	(q15_t)0xDF60, (q15_t)0x7BC5, (q15_t)0xDF2F, (q15_t)0x7BB9,
	(q15_t)0xDEFF, (q15_t)0x7BAC, (q15_t)0xDECE, (q15_t)0x7B9F,
	(q15_t)0xDE9E, (q15_t)0x7B92, (q15_t)0xDE6D, (q15_t)0x7B84,
	(q15_t)0xDE3D, (q15_t)0x7B77, (q15_t)0xDE0C, (q15_t)0x7B6A,
	(q15_t)0xDDDC, (q15_t)0x7B5D, (q15_t)0xDDAB, (q15_t)0x7B4F,
	(q15_t)0xDD7B, (q15_t)0x7B42, (q15_t)0xDD4B, (q15_t)0x7B34,
	(q15_t)0xDD1A, (q15_t)0x7B26, (q15_t)0xDCEA, (q15_t)0x7B19,
	(q15_t)0xDCBA, (q15_t)0x7B0B, (q15_t)0xDC89, (q15_t)0x7AFD,
	(q15_t)0xDC59, (q15_t)0x7AEF, (q15_t)0xDC29, (q15_t)0x7AE1,
	(q15_t)0xDBF8, (q15_t)0x7AD3, (q15_t)0xDBC8, (q15_t)0x7AC5,
	(q15_t)0xDB98, (q15_t)0x7AB6, (q15_t)0xDB68, (q15_t)0x7AA8,
	(q15_t)0xDB38, (q15_t)0x7A9A, (q15_t)0xDB08, (q15_t)0x7A8B,
	(q15_t)0xDAD7, (q15_t)0x7A7D, (q15_t)0xDAA7, (q15_t)0x7A6E,
	(q15_t)0xDA77, (q15_t)0x7A5F, (q15_t)0xDA47, (q15_t)0x7A50,
	(q15_t)0xDA17, (q15_t)0x7A42, (q15_t)0xD9E7, (q15_t)0x7A33,
	(q15_t)0xD9B7, (q15_t)0x7A24, (q15_t)0xD987, (q15_t)0x7A15,
	(q15_t)0xD957, (q15_t)0x7A05, (q15_t)0xD927, (q15_t)0x79F6,
	(q15_t)0xD8F8, (q15_t)0x79E7, (q15_t)0xD8C8, (q15_t)0x79D8,
	(q15_t)0xD898, (q15_t)0x79C8, (q15_t)0xD868, (q15_t)0x79B9,
	(q15_t)0xD838, (q15_t)0x79A9, (q15_t)0xD809, (q15_t)0x7999,
	(q15_t)0xD7D9, (q15_t)0x798A, (q15_t)0xD7A9, (q15_t)0x797A,
	(q15_t)0xD779, (q15_t)0x796A, (q15_t)0xD74A, (q15_t)0x795A,
	(q15_t)0xD71A, (q15_t)0x794A, (q15_t)0xD6EA, (q15_t)0x793A,
	(q15_t)0xD6BB, (q15_t)0x792A, (q15_t)0xD68B, (q15_t)0x7919,
	(q15_t)0xD65C, (q15_t)0x7909, (q15_t)0xD62C, (q15_t)0x78F9,
	(q15_t)0xD5FD, (q15_t)0x78E8, (q15_t)0xD5CD, (q15_t)0x78D8,
	(q15_t)0xD59E, (q15_t)0x78C7, (q15_t)0xD56E, (q15_t)0x78B6,
	(q15_t)0xD53F, (q15_t)0x78A6, (q15_t)0xD510, (q15_t)0x7895,
	(q15_t)0xD4E0, (q15_t)0x7884, (q15_t)0xD4B1, (q15_t)0x7873,
	(q15_t)0xD482, (q15_t)0x7862, (q15_t)0xD452, (q15_t)0x7851,
	(q15_t)0xD423, (q15_t)0x7840, (q15_t)0xD3F4, (q15_t)0x782E,
	(q15_t)0xD3C5, (q15_t)0x781D, (q15_t)0xD396, (q15_t)0x780C,
	(q15_t)0xD367, (q15_t)0x77FA, (q15_t)0xD337, (q15_t)0x77E9,
	(q15_t)0xD308, (q15_t)0x77D7, (q15_t)0xD2D9, (q15_t)0x77C5,
	(q15_t)0xD2AA, (q15_t)0x77B4, (q15_t)0xD27B, (q15_t)0x77A2,
	(q15_t)0xD24C, (q15_t)0x7790, (q15_t)0xD21D, (q15_t)0x777E,
	(q15_t)0xD1EE, (q15_t)0x776C, (q15_t)0xD1C0, (q15_t)0x775A,
	(q15_t)0xD191, (q15_t)0x7747, (q15_t)0xD162, (q15_t)0x7735,
	(q15_t)0xD133, (q15_t)0x7723, (q15_t)0xD104, (q15_t)0x7710,
	(q15_t)0xD0D6, (q15_t)0x76FE, (q15_t)0xD0A7, (q15_t)0x76EB,
	(q15_t)0xD078, (q15_t)0x76D9, (q15_t)0xD04A, (q15_t)0x76C6,
	(q15_t)0xD01B, (q15_t)0x76B3, (q15_t)0xCFEC, (q15_t)0x76A0,
	(q15_t)0xCFBE, (q15_t)0x768E, (q15_t)0xCF8F, (q15_t)0x767B,
	(q15_t)0xCF61, (q15_t)0x7668, (q15_t)0xCF32, (q15_t)0x7654,
	(q15_t)0xCF04, (q15_t)0x7641, (q15_t)0xCED5, (q15_t)0x762E,
	(q15_t)0xCEA7, (q15_t)0x761B, (q15_t)0xCE79, (q15_t)0x7607,
	(q15_t)0xCE4A, (q15_t)0x75F4, (q15_t)0xCE1C, (q15_t)0x75E0,
	(q15_t)0xCDEE, (q15_t)0x75CC, (q15_t)0xCDBF, (q15_t)0x75B9,
	(q15_t)0xCD91, (q15_t)0x75A5, (q15_t)0xCD63, (q15_t)0x7591,
	(q15_t)0xCD35, (q15_t)0x757D, (q15_t)0xCD07, (q15_t)0x7569,
	(q15_t)0xCCD9, (q15_t)0x7555, (q15_t)0xCCAB, (q15_t)0x7541,
	(q15_t)0xCC7D, (q15_t)0x752D, (q15_t)0xCC4F, (q15_t)0x7519,
	(q15_t)0xCC21, (q15_t)0x7504, (q15_t)0xCBF3, (q15_t)0x74F0,
	(q15_t)0xCBC5, (q15_t)0x74DB, (q15_t)0xCB97, (q15_t)0x74C7,
	(q15_t)0xCB69, (q15_t)0x74B2, (q15_t)0xCB3B, (q15_t)0x749E,
	(q15_t)0xCB0D, (q15_t)0x7489, (q15_t)0xCAE0, (q15_t)0x7474,
	(q15_t)0xCAB2, (q15_t)0x745F, (q15_t)0xCA84, (q15_t)0x744A,
	(q15_t)0xCA57, (q15_t)0x7435, (q15_t)0xCA29, (q15_t)0x7420,
	(q15_t)0xC9FB, (q15_t)0x740B, (q15_t)0xC9CE, (q15_t)0x73F6,
	(q15_t)0xC9A0, (q15_t)0x73E0, (q15_t)0xC973, (q15_t)0x73CB,
	(q15_t)0xC945, (q15_t)0x73B5, (q15_t)0xC918, (q15_t)0x73A0,
	(q15_t)0xC8EB, (q15_t)0x738A, (q15_t)0xC8BD, (q15_t)0x7375,
	(q15_t)0xC890, (q15_t)0x735F, (q15_t)0xC863, (q15_t)0x7349,
	(q15_t)0xC835, (q15_t)0x7333, (q15_t)0xC808, (q15_t)0x731D,
	(q15_t)0xC7DB, (q15_t)0x7307, (q15_t)0xC7AE, (q15_t)0x72F1,
	(q15_t)0xC781, (q15_t)0x72DB, (q15_t)0xC754, (q15_t)0x72C5,
	(q15_t)0xC727, (q15_t)0x72AF, (q15_t)0xC6F9, (q15_t)0x7298,
	(q15_t)0xC6CD, (q15_t)0x7282, (q15_t)0xC6A0, (q15_t)0x726B,
	(q15_t)0xC673, (q15_t)0x7255, (q15_t)0xC646, (q15_t)0x723E,
	(q15_t)0xC619, (q15_t)0x7227, (q15_t)0xC5EC, (q15_t)0x7211,
	(q15_t)0xC5BF, (q15_t)0x71FA, (q15_t)0xC593, (q15_t)0x71E3,
	(q15_t)0xC566, (q15_t)0x71CC, (q15_t)0xC539, (q15_t)0x71B5,
	(q15_t)0xC50D, (q15_t)0x719E, (q15_t)0xC4E0, (q15_t)0x7186,
	(q15_t)0xC4B3, (q15_t)0x716F, (q15_t)0xC487, (q15_t)0x7158,
	(q15_t)0xC45A, (q15_t)0x7141, (q15_t)0xC42E, (q15_t)0x7129,
	(q15_t)0xC402, (q15_t)0x7112, (q15_t)0xC3D5, (q15_t)0x70FA,
	(q15_t)0xC3A9, (q15_t)0x70E2, (q15_t)0xC37C, (q15_t)0x70CB,
	(q15_t)0xC350, (q15_t)0x70B3, (q15_t)0xC324, (q15_t)0x709B,
	(q15_t)0xC2F8, (q15_t)0x7083, (q15_t)0xC2CC, (q15_t)0x706B,
	(q15_t)0xC29F, (q15_t)0x7053, (q15_t)0xC273, (q15_t)0x703B,
	(q15_t)0xC247, (q15_t)0x7023, (q15_t)0xC21B, (q15_t)0x700A,
	(q15_t)0xC1EF, (q15_t)0x6FF2, (q15_t)0xC1C3, (q15_t)0x6FDA,
	(q15_t)0xC197, (q15_t)0x6FC1, (q15_t)0xC16C, (q15_t)0x6FA9,
	(q15_t)0xC140, (q15_t)0x6F90, (q15_t)0xC114, (q15_t)0x6F77,
	(q15_t)0xC0E8, (q15_t)0x6F5F, (q15_t)0xC0BC, (q15_t)0x6F46,
	(q15_t)0xC091, (q15_t)0x6F2D, (q15_t)0xC065, (q15_t)0x6F14,
	(q15_t)0xC03A, (q15_t)0x6EFB, (q15_t)0xC00E, (q15_t)0x6EE2,
	(q15_t)0xBFE2, (q15_t)0x6EC9, (q15_t)0xBFB7, (q15_t)0x6EAF,
	(q15_t)0xBF8C, (q15_t)0x6E96, (q15_t)0xBF60, (q15_t)0x6E7D,
	(q15_t)0xBF35, (q15_t)0x6E63, (q15_t)0xBF09, (q15_t)0x6E4A,
	(q15_t)0xBEDE, (q15_t)0x6E30, (q15_t)0xBEB3, (q15_t)0x6E17,
	(q15_t)0xBE88, (q15_t)0x6DFD, (q15_t)0xBE5D, (q15_t)0x6DE3,
	(q15_t)0xBE31, (q15_t)0x6DCA, (q15_t)0xBE06, (q15_t)0x6DB0,
	(q15_t)0xBDDB, (q15_t)0x6D96, (q15_t)0xBDB0, (q15_t)0x6D7C,
	(q15_t)0xBD85, (q15_t)0x6D62, (q15_t)0xBD5A, (q15_t)0x6D48,
	(q15_t)0xBD2F, (q15_t)0x6D2D, (q15_t)0xBD05, (q15_t)0x6D13,
	(q15_t)0xBCDA, (q15_t)0x6CF9, (q15_t)0xBCAF, (q15_t)0x6CDE,
	(q15_t)0xBC84, (q15_t)0x6CC4, (q15_t)0xBC5A, (q15_t)0x6CA9,
	(q15_t)0xBC2F, (q15_t)0x6C8F, (q15_t)0xBC04, (q15_t)0x6C74,
	(q15_t)0xBBDA, (q15_t)0x6C59, (q15_t)0xBBAF, (q15_t)0x6C3F,
	(q15_t)0xBB85, (q15_t)0x6C24, (q15_t)0xBB5A, (q15_t)0x6C09,
	(q15_t)0xBB30, (q15_t)0x6BEE, (q15_t)0xBB05, (q15_t)0x6BD3,
	(q15_t)0xBADB, (q15_t)0x6BB8, (q15_t)0xBAB1, (q15_t)0x6B9C,
	(q15_t)0xBA87, (q15_t)0x6B81, (q15_t)0xBA5C, (q15_t)0x6B66,
	(q15_t)0xBA32, (q15_t)0x6B4A, (q15_t)0xBA08, (q15_t)0x6B2F,
	(q15_t)0xB9DE, (q15_t)0x6B13, (q15_t)0xB9B4, (q15_t)0x6AF8,
	(q15_t)0xB98A, (q15_t)0x6ADC, (q15_t)0xB960, (q15_t)0x6AC1,
	(q15_t)0xB936, (q15_t)0x6AA5, (q15_t)0xB90C, (q15_t)0x6A89,
	(q15_t)0xB8E3, (q15_t)0x6A6D, (q15_t)0xB8B9, (q15_t)0x6A51,
	(q15_t)0xB88F, (q15_t)0x6A35, (q15_t)0xB865, (q15_t)0x6A19,
	(q15_t)0xB83C, (q15_t)0x69FD, (q15_t)0xB812, (q15_t)0x69E1,
	(q15_t)0xB7E9, (q15_t)0x69C4, (q15_t)0xB7BF, (q15_t)0x69A8,
	(q15_t)0xB796, (q15_t)0x698C, (q15_t)0xB76C, (q15_t)0x696F,
	(q15_t)0xB743, (q15_t)0x6953, (q15_t)0xB719, (q15_t)0x6936,
	(q15_t)0xB6F0, (q15_t)0x6919, (q15_t)0xB6C7, (q15_t)0x68FD,
	(q15_t)0xB69E, (q15_t)0x68E0, (q15_t)0xB675, (q15_t)0x68C3,
	(q15_t)0xB64B, (q15_t)0x68A6, (q15_t)0xB622, (q15_t)0x6889,
	(q15_t)0xB5F9, (q15_t)0x686C, (q15_t)0xB5D0, (q15_t)0x684F,
	(q15_t)0xB5A7, (q15_t)0x6832, (q15_t)0xB57E, (q15_t)0x6815,
	(q15_t)0xB556, (q15_t)0x67F7, (q15_t)0xB52D, (q15_t)0x67DA,
	(q15_t)0xB504, (q15_t)0x67BD, (q15_t)0xB4DB, (q15_t)0x679F,
	(q15_t)0xB4B3, (q15_t)0x6782, (q15_t)0xB48A, (q15_t)0x6764,
	(q15_t)0xB461, (q15_t)0x6746, (q15_t)0xB439, (q15_t)0x6729,
	(q15_t)0xB410, (q15_t)0x670B, (q15_t)0xB3E8, (q15_t)0x66ED,
	(q15_t)0xB3C0, (q15_t)0x66CF, (q15_t)0xB397, (q15_t)0x66B1,
	(q15_t)0xB36F, (q15_t)0x6693, (q15_t)0xB347, (q15_t)0x6675,
	(q15_t)0xB31E, (q15_t)0x6657, (q15_t)0xB2F6, (q15_t)0x6639,
	(q15_t)0xB2CE, (q15_t)0x661A, (q15_t)0xB2A6, (q15_t)0x65FC,
	(q15_t)0xB27E, (q15_t)0x65DD, (q15_t)0xB256, (q15_t)0x65BF,
	(q15_t)0xB22E, (q15_t)0x65A0, (q15_t)0xB206, (q15_t)0x6582,
	(q15_t)0xB1DE, (q15_t)0x6563, (q15_t)0xB1B7, (q15_t)0x6545,
	(q15_t)0xB18F, (q15_t)0x6526, (q15_t)0xB167, (q15_t)0x6507,
	(q15_t)0xB140, (q15_t)0x64E8, (q15_t)0xB118, (q15_t)0x64C9,
	(q15_t)0xB0F0, (q15_t)0x64AA, (q15_t)0xB0C9, (q15_t)0x648B,
	(q15_t)0xB0A1, (q15_t)0x646C, (q15_t)0xB07A, (q15_t)0x644D,
	(q15_t)0xB053, (q15_t)0x642D, (q15_t)0xB02B, (q15_t)0x640E,
	(q15_t)0xB004, (q15_t)0x63EF, (q15_t)0xAFDD, (q15_t)0x63CF,
	(q15_t)0xAFB6, (q15_t)0x63B0, (q15_t)0xAF8F, (q15_t)0x6390,
	(q15_t)0xAF68, (q15_t)0x6371, (q15_t)0xAF40, (q15_t)0x6351,
	(q15_t)0xAF1A, (q15_t)0x6331, (q15_t)0xAEF3, (q15_t)0x6311,
	(q15_t)0xAECC, (q15_t)0x62F2, (q15_t)0xAEA5, (q15_t)0x62D2,
	(q15_t)0xAE7E, (q15_t)0x62B2, (q15_t)0xAE57, (q15_t)0x6292,
	(q15_t)0xAE31, (q15_t)0x6271, (q15_t)0xAE0A, (q15_t)0x6251,
	(q15_t)0xADE3, (q15_t)0x6231, (q15_t)0xADBD, (q15_t)0x6211,
	(q15_t)0xAD96, (q15_t)0x61F1, (q15_t)0xAD70, (q15_t)0x61D0,
	(q15_t)0xAD4A, (q15_t)0x61B0, (q15_t)0xAD23, (q15_t)0x618F,
	(q15_t)0xACFD, (q15_t)0x616F, (q15_t)0xACD7, (q15_t)0x614E,
	(q15_t)0xACB1, (q15_t)0x612D, (q15_t)0xAC8A, (q15_t)0x610D,
	(q15_t)0xAC64, (q15_t)0x60EC, (q15_t)0xAC3E, (q15_t)0x60CB,
	(q15_t)0xAC18, (q15_t)0x60AA, (q15_t)0xABF2, (q15_t)0x6089,
	(q15_t)0xABCC, (q15_t)0x6068, (q15_t)0xABA7, (q15_t)0x6047,
	(q15_t)0xAB81, (q15_t)0x6026, (q15_t)0xAB5B, (q15_t)0x6004,
	(q15_t)0xAB35, (q15_t)0x5FE3, (q15_t)0xAB10, (q15_t)0x5FC2,
	(q15_t)0xAAEA, (q15_t)0x5FA0, (q15_t)0xAAC5, (q15_t)0x5F7F,
	(q15_t)0xAA9F, (q15_t)0x5F5E, (q15_t)0xAA7A, (q15_t)0x5F3C,
	(q15_t)0xAA54, (q15_t)0x5F1A, (q15_t)0xAA2F, (q15_t)0x5EF9,
	(q15_t)0xAA0A, (q15_t)0x5ED7, (q15_t)0xA9E5, (q15_t)0x5EB5,
	(q15_t)0xA9BF, (q15_t)0x5E93, (q15_t)0xA99A, (q15_t)0x5E71,
	(q15_t)0xA975, (q15_t)0x5E50, (q15_t)0xA950, (q15_t)0x5E2D,
	(q15_t)0xA92B, (q15_t)0x5E0B, (q15_t)0xA906, (q15_t)0x5DE9,
	(q15_t)0xA8E2, (q15_t)0x5DC7, (q15_t)0xA8BD, (q15_t)0x5DA5,
	(q15_t)0xA898, (q15_t)0x5D83, (q15_t)0xA873, (q15_t)0x5D60,
	(q15_t)0xA84F, (q15_t)0x5D3E, (q15_t)0xA82A, (q15_t)0x5D1B,
	(q15_t)0xA806, (q15_t)0x5CF9, (q15_t)0xA7E1, (q15_t)0x5CD6,
	(q15_t)0xA7BD, (q15_t)0x5CB4, (q15_t)0xA798, (q15_t)0x5C91,
	(q15_t)0xA774, (q15_t)0x5C6E, (q15_t)0xA750, (q15_t)0x5C4B,
	(q15_t)0xA72B, (q15_t)0x5C29, (q15_t)0xA707, (q15_t)0x5C06,
	(q15_t)0xA6E3, (q15_t)0x5BE3, (q15_t)0xA6BF, (q15_t)0x5BC0,
	(q15_t)0xA69B, (q15_t)0x5B9D, (q15_t)0xA677, (q15_t)0x5B79,
	(q15_t)0xA653, (q15_t)0x5B56, (q15_t)0xA62F, (q15_t)0x5B33,
	(q15_t)0xA60C, (q15_t)0x5B10, (q15_t)0xA5E8, (q15_t)0x5AEC,
	(q15_t)0xA5C4, (q15_t)0x5AC9, (q15_t)0xA5A1, (q15_t)0x5AA5,
	(q15_t)0xA57D, (q15_t)0x5A82, (q15_t)0xA55A, (q15_t)0x5A5E,
	(q15_t)0xA536, (q15_t)0x5A3B, (q15_t)0xA513, (q15_t)0x5A17,
	(q15_t)0xA4EF, (q15_t)0x59F3, (q15_t)0xA4CC, (q15_t)0x59D0,
	(q15_t)0xA4A9, (q15_t)0x59AC, (q15_t)0xA486, (q15_t)0x5988,
	(q15_t)0xA462, (q15_t)0x5964, (q15_t)0xA43F, (q15_t)0x5940,
	(q15_t)0xA41C, (q15_t)0x591C, (q15_t)0xA3F9, (q15_t)0x58F8,
	(q15_t)0xA3D6, (q15_t)0x58D4, (q15_t)0xA3B4, (q15_t)0x58AF,
	(q15_t)0xA391, (q15_t)0x588B, (q15_t)0xA36E, (q15_t)0x5867,
	(q15_t)0xA34B, (q15_t)0x5842, (q15_t)0xA329, (q15_t)0x581E,
	(q15_t)0xA306, (q15_t)0x57F9, (q15_t)0xA2E4, (q15_t)0x57D5,
	(q15_t)0xA2C1, (q15_t)0x57B0, (q15_t)0xA29F, (q15_t)0x578C,
	(q15_t)0xA27C, (q15_t)0x5767, (q15_t)0xA25A, (q15_t)0x5742,
	(q15_t)0xA238, (q15_t)0x571D, (q15_t)0xA216, (q15_t)0x56F9,
	(q15_t)0xA1F4, (q15_t)0x56D4, (q15_t)0xA1D2, (q15_t)0x56AF,
	(q15_t)0xA1AF, (q15_t)0x568A, (q15_t)0xA18E, (q15_t)0x5665,
	(q15_t)0xA16C, (q15_t)0x5640, (q15_t)0xA14A, (q15_t)0x561A,
	(q15_t)0xA128, (q15_t)0x55F5, (q15_t)0xA106, (q15_t)0x55D0,
	(q15_t)0xA0E5, (q15_t)0x55AB, (q15_t)0xA0C3, (q15_t)0x5585,
	(q15_t)0xA0A1, (q15_t)0x5560, (q15_t)0xA080, (q15_t)0x553A,
	(q15_t)0xA05F, (q15_t)0x5515, (q15_t)0xA03D, (q15_t)0x54EF,
	(q15_t)0xA01C, (q15_t)0x54CA, (q15_t)0x9FFB, (q15_t)0x54A4,
	(q15_t)0x9FD9, (q15_t)0x547E, (q15_t)0x9FB8, (q15_t)0x5458,
	(q15_t)0x9F97, (q15_t)0x5433, (q15_t)0x9F76, (q15_t)0x540D,
	(q15_t)0x9F55, (q15_t)0x53E7, (q15_t)0x9F34, (q15_t)0x53C1,
	(q15_t)0x9F13, (q15_t)0x539B, (q15_t)0x9EF2, (q15_t)0x5375,
	(q15_t)0x9ED2, (q15_t)0x534E, (q15_t)0x9EB1, (q15_t)0x5328,
	(q15_t)0x9E90, (q15_t)0x5302, (q15_t)0x9E70, (q15_t)0x52DC,
	(q15_t)0x9E4F, (q15_t)0x52B5, (q15_t)0x9E2F, (q15_t)0x528F,
	(q15_t)0x9E0E, (q15_t)0x5269, (q15_t)0x9DEE, (q15_t)0x5242,
	(q15_t)0x9DCE, (q15_t)0x521C, (q15_t)0x9DAE, (q15_t)0x51F5,
	(q15_t)0x9D8E, (q15_t)0x51CE, (q15_t)0x9D6D, (q15_t)0x51A8,
	(q15_t)0x9D4D, (q15_t)0x5181, (q15_t)0x9D2D, (q15_t)0x515A,
	(q15_t)0x9D0D, (q15_t)0x5133, (q15_t)0x9CEE, (q15_t)0x510C,
	(q15_t)0x9CCE, (q15_t)0x50E5, (q15_t)0x9CAE, (q15_t)0x50BF,
	(q15_t)0x9C8E, (q15_t)0x5097, (q15_t)0x9C6F, (q15_t)0x5070,
	(q15_t)0x9C4F, (q15_t)0x5049, (q15_t)0x9C30, (q15_t)0x5022,
	(q15_t)0x9C10, (q15_t)0x4FFB, (q15_t)0x9BF1, (q15_t)0x4FD4,
	(q15_t)0x9BD2, (q15_t)0x4FAC, (q15_t)0x9BB2, (q15_t)0x4F85,
	(q15_t)0x9B93, (q15_t)0x4F5E, (q15_t)0x9B74, (q15_t)0x4F36,
	(q15_t)0x9B55, (q15_t)0x4F0F, (q15_t)0x9B36, (q15_t)0x4EE7,
	(q15_t)0x9B17, (q15_t)0x4EBF, (q15_t)0x9AF8, (q15_t)0x4E98,
	(q15_t)0x9AD9, (q15_t)0x4E70, (q15_t)0x9ABA, (q15_t)0x4E48,
	(q15_t)0x9A9C, (q15_t)0x4E21, (q15_t)0x9A7D, (q15_t)0x4DF9,
	(q15_t)0x9A5F, (q15_t)0x4DD1, (q15_t)0x9A40, (q15_t)0x4DA9,
	(q15_t)0x9A22, (q15_t)0x4D81, (q15_t)0x9A03, (q15_t)0x4D59,
	(q15_t)0x99E5, (q15_t)0x4D31, (q15_t)0x99C6, (q15_t)0x4D09,
	(q15_t)0x99A8, (q15_t)0x4CE1, (q15_t)0x998A, (q15_t)0x4CB8,
	(q15_t)0x996C, (q15_t)0x4C90, (q15_t)0x994E, (q15_t)0x4C68,
	(q15_t)0x9930, (q15_t)0x4C3F, (q15_t)0x9912, (q15_t)0x4C17,
	(q15_t)0x98F4, (q15_t)0x4BEF, (q15_t)0x98D6, (q15_t)0x4BC6,
	(q15_t)0x98B9, (q15_t)0x4B9E, (q15_t)0x989B, (q15_t)0x4B75,
	(q15_t)0x987D, (q15_t)0x4B4C, (q15_t)0x9860, (q15_t)0x4B24,
	(q15_t)0x9842, (q15_t)0x4AFB, (q15_t)0x9825, (q15_t)0x4AD2,
	(q15_t)0x9808, (q15_t)0x4AA9, (q15_t)0x97EA, (q15_t)0x4A81,
	(q15_t)0x97CD, (q15_t)0x4A58, (q15_t)0x97B0, (q15_t)0x4A2F,
	(q15_t)0x9793, (q15_t)0x4A06, (q15_t)0x9776, (q15_t)0x49DD,
	(q15_t)0x9759, (q15_t)0x49B4, (q15_t)0x973C, (q15_t)0x498A,
	(q15_t)0x971F, (q15_t)0x4961, (q15_t)0x9702, (q15_t)0x4938,
	(q15_t)0x96E6, (q15_t)0x490F, (q15_t)0x96C9, (q15_t)0x48E6,
	(q15_t)0x96AC, (q15_t)0x48BC, (q15_t)0x9690, (q15_t)0x4893,
	(q15_t)0x9673, (q15_t)0x4869, (q15_t)0x9657, (q15_t)0x4840,
	(q15_t)0x963B, (q15_t)0x4816, (q15_t)0x961E, (q15_t)0x47ED,
	(q15_t)0x9602, (q15_t)0x47C3, (q15_t)0x95E6, (q15_t)0x479A,
	(q15_t)0x95CA, (q15_t)0x4770, (q15_t)0x95AE, (q15_t)0x4746,
	(q15_t)0x9592, (q15_t)0x471C, (q15_t)0x9576, (q15_t)0x46F3,
	(q15_t)0x955A, (q15_t)0x46C9, (q15_t)0x953E, (q15_t)0x469F,
	(q15_t)0x9523, (q15_t)0x4675, (q15_t)0x9507, (q15_t)0x464B,
	(q15_t)0x94EC, (q15_t)0x4621, (q15_t)0x94D0, (q15_t)0x45F7,
	(q15_t)0x94B5, (q15_t)0x45CD, (q15_t)0x9499, (q15_t)0x45A3,
	(q15_t)0x947E, (q15_t)0x4578, (q15_t)0x9463, (q15_t)0x454E,
	(q15_t)0x9447, (q15_t)0x4524, (q15_t)0x942C, (q15_t)0x44FA,
	(q15_t)0x9411, (q15_t)0x44CF, (q15_t)0x93F6, (q15_t)0x44A5,
	(q15_t)0x93DB, (q15_t)0x447A, (q15_t)0x93C0, (q15_t)0x4450,
	(q15_t)0x93A6, (q15_t)0x4425, (q15_t)0x938B, (q15_t)0x43FB,
	(q15_t)0x9370, (q15_t)0x43D0, (q15_t)0x9356, (q15_t)0x43A5,
	(q15_t)0x933B, (q15_t)0x437B, (q15_t)0x9321, (q15_t)0x4350,
	(q15_t)0x9306, (q15_t)0x4325, (q15_t)0x92EC, (q15_t)0x42FA,
	(q15_t)0x92D2, (q15_t)0x42D0, (q15_t)0x92B7, (q15_t)0x42A5,
	(q15_t)0x929D, (q15_t)0x427A, (q15_t)0x9283, (q15_t)0x424F,
	(q15_t)0x9269, (q15_t)0x4224, (q15_t)0x924F, (q15_t)0x41F9,
	(q15_t)0x9235, (q15_t)0x41CE, (q15_t)0x921C, (q15_t)0x41A2,
	(q15_t)0x9202, (q15_t)0x4177, (q15_t)0x91E8, (q15_t)0x414C,
	(q15_t)0x91CF, (q15_t)0x4121, (q15_t)0x91B5, (q15_t)0x40F6,
	(q15_t)0x919C, (q15_t)0x40CA, (q15_t)0x9182, (q15_t)0x409F,
	(q15_t)0x9169, (q15_t)0x4073, (q15_t)0x9150, (q15_t)0x4048,
	(q15_t)0x9136, (q15_t)0x401D, (q15_t)0x911D, (q15_t)0x3FF1,
	(q15_t)0x9104, (q15_t)0x3FC5, (q15_t)0x90EB, (q15_t)0x3F9A,
	(q15_t)0x90D2, (q15_t)0x3F6E, (q15_t)0x90B9, (q15_t)0x3F43,
	(q15_t)0x90A0, (q15_t)0x3F17, (q15_t)0x9088, (q15_t)0x3EEB,
	(q15_t)0x906F, (q15_t)0x3EBF, (q15_t)0x9056, (q15_t)0x3E93,
	(q15_t)0x903E, (q15_t)0x3E68, (q15_t)0x9025, (q15_t)0x3E3C,
	(q15_t)0x900D, (q15_t)0x3E10, (q15_t)0x8FF5, (q15_t)0x3DE4,
	(q15_t)0x8FDC, (q15_t)0x3DB8, (q15_t)0x8FC4, (q15_t)0x3D8C,
	(q15_t)0x8FAC, (q15_t)0x3D60, (q15_t)0x8F94, (q15_t)0x3D33,
	(q15_t)0x8F7C, (q15_t)0x3D07, (q15_t)0x8F64, (q15_t)0x3CDB,
	(q15_t)0x8F4C, (q15_t)0x3CAF, (q15_t)0x8F34, (q15_t)0x3C83,
	(q15_t)0x8F1D, (q15_t)0x3C56, (q15_t)0x8F05, (q15_t)0x3C2A,
	(q15_t)0x8EED, (q15_t)0x3BFD, (q15_t)0x8ED6, (q15_t)0x3BD1,
	(q15_t)0x8EBE, (q15_t)0x3BA5, (q15_t)0x8EA7, (q15_t)0x3B78,
	(q15_t)0x8E90, (q15_t)0x3B4C, (q15_t)0x8E79, (q15_t)0x3B1F,
	(q15_t)0x8E61, (q15_t)0x3AF2, (q15_t)0x8E4A, (q15_t)0x3AC6,
	(q15_t)0x8E33, (q15_t)0x3A99, (q15_t)0x8E1C, (q15_t)0x3A6C,
	(q15_t)0x8E05, (q15_t)0x3A40, (q15_t)0x8DEE, (q15_t)0x3A13,
	(q15_t)0x8DD8, (q15_t)0x39E6, (q15_t)0x8DC1, (q15_t)0x39B9,
	(q15_t)0x8DAA, (q15_t)0x398C, (q15_t)0x8D94, (q15_t)0x395F,
	(q15_t)0x8D7D, (q15_t)0x3932, (q15_t)0x8D67, (q15_t)0x3906,
	(q15_t)0x8D50, (q15_t)0x38D8, (q15_t)0x8D3A, (q15_t)0x38AB,
	(q15_t)0x8D24, (q15_t)0x387E, (q15_t)0x8D0E, (q15_t)0x3851,
	(q15_t)0x8CF8, (q15_t)0x3824, (q15_t)0x8CE2, (q15_t)0x37F7,
	(q15_t)0x8CCC, (q15_t)0x37CA, (q15_t)0x8CB6, (q15_t)0x379C,
	(q15_t)0x8CA0, (q15_t)0x376F, (q15_t)0x8C8A, (q15_t)0x3742,
	(q15_t)0x8C75, (q15_t)0x3714, (q15_t)0x8C5F, (q15_t)0x36E7,
	(q15_t)0x8C4A, (q15_t)0x36BA, (q15_t)0x8C34, (q15_t)0x368C,
	(q15_t)0x8C1F, (q15_t)0x365F, (q15_t)0x8C09, (q15_t)0x3631,
	(q15_t)0x8BF4, (q15_t)0x3604, (q15_t)0x8BDF, (q15_t)0x35D6,
	(q15_t)0x8BCA, (q15_t)0x35A8, (q15_t)0x8BB5, (q15_t)0x357B,
	(q15_t)0x8BA0, (q15_t)0x354D, (q15_t)0x8B8B, (q15_t)0x351F,
	(q15_t)0x8B76, (q15_t)0x34F2, (q15_t)0x8B61, (q15_t)0x34C4,
	(q15_t)0x8B4D, (q15_t)0x3496, (q15_t)0x8B38, (q15_t)0x3468,
	(q15_t)0x8B24, (q15_t)0x343A, (q15_t)0x8B0F, (q15_t)0x340C,
	(q15_t)0x8AFB, (q15_t)0x33DE, (q15_t)0x8AE6, (q15_t)0x33B0,
	(q15_t)0x8AD2, (q15_t)0x3382, (q15_t)0x8ABE, (q15_t)0x3354,
	(q15_t)0x8AAA, (q15_t)0x3326, (q15_t)0x8A96, (q15_t)0x32F8,
	(q15_t)0x8A82, (q15_t)0x32CA, (q15_t)0x8A6E, (q15_t)0x329C,
	(q15_t)0x8A5A, (q15_t)0x326E, (q15_t)0x8A46, (q15_t)0x3240,
	(q15_t)0x8A33, (q15_t)0x3211, (q15_t)0x8A1F, (q15_t)0x31E3,
	(q15_t)0x8A0B, (q15_t)0x31B5, (q15_t)0x89F8, (q15_t)0x3186,
	(q15_t)0x89E4, (q15_t)0x3158, (q15_t)0x89D1, (q15_t)0x312A,
	(q15_t)0x89BE, (q15_t)0x30FB, (q15_t)0x89AB, (q15_t)0x30CD,
	(q15_t)0x8997, (q15_t)0x309E, (q15_t)0x8984, (q15_t)0x3070,
	(q15_t)0x8971, (q15_t)0x3041, (q15_t)0x895F, (q15_t)0x3013,
	(q15_t)0x894C, (q15_t)0x2FE4, (q15_t)0x8939, (q15_t)0x2FB5,
	(q15_t)0x8926, (q15_t)0x2F87, (q15_t)0x8914, (q15_t)0x2F58,
	(q15_t)0x8901, (q15_t)0x2F29, (q15_t)0x88EF, (q15_t)0x2EFB,
	(q15_t)0x88DC, (q15_t)0x2ECC, (q15_t)0x88CA, (q15_t)0x2E9D,
	(q15_t)0x88B8, (q15_t)0x2E6E, (q15_t)0x88A5, (q15_t)0x2E3F,
	(q15_t)0x8893, (q15_t)0x2E11, (q15_t)0x8881, (q15_t)0x2DE2,
	(q15_t)0x886F, (q15_t)0x2DB3, (q15_t)0x885D, (q15_t)0x2D84,
	(q15_t)0x884B, (q15_t)0x2D55, (q15_t)0x883A, (q15_t)0x2D26,
	(q15_t)0x8828, (q15_t)0x2CF7, (q15_t)0x8816, (q15_t)0x2CC8,
	(q15_t)0x8805, (q15_t)0x2C98, (q15_t)0x87F3, (q15_t)0x2C69,
	(q15_t)0x87E2, (q15_t)0x2C3A, (q15_t)0x87D1, (q15_t)0x2C0B,
	(q15_t)0x87BF, (q15_t)0x2BDC, (q15_t)0x87AE, (q15_t)0x2BAD,
	(q15_t)0x879D, (q15_t)0x2B7D, (q15_t)0x878C, (q15_t)0x2B4E,
	(q15_t)0x877B, (q15_t)0x2B1F, (q15_t)0x876A, (q15_t)0x2AEF,
	(q15_t)0x8759, (q15_t)0x2AC0, (q15_t)0x8749, (q15_t)0x2A91,
	(q15_t)0x8738, (q15_t)0x2A61, (q15_t)0x8727, (q15_t)0x2A32,
	(q15_t)0x8717, (q15_t)0x2A02, (q15_t)0x8706, (q15_t)0x29D3,
	(q15_t)0x86F6, (q15_t)0x29A3, (q15_t)0x86E6, (q15_t)0x2974,
	(q15_t)0x86D5, (q15_t)0x2944, (q15_t)0x86C5, (q15_t)0x2915,
	(q15_t)0x86B5, (q15_t)0x28E5, (q15_t)0x86A5, (q15_t)0x28B5,
	(q15_t)0x8695, (q15_t)0x2886, (q15_t)0x8685, (q15_t)0x2856,
	(q15_t)0x8675, (q15_t)0x2826, (q15_t)0x8666, (q15_t)0x27F6,
	(q15_t)0x8656, (q15_t)0x27C7, (q15_t)0x8646, (q15_t)0x2797,
	(q15_t)0x8637, (q15_t)0x2767, (q15_t)0x8627, (q15_t)0x2737,
	(q15_t)0x8618, (q15_t)0x2707, (q15_t)0x8609, (q15_t)0x26D8,
	(q15_t)0x85FA, (q15_t)0x26A8, (q15_t)0x85EA, (q15_t)0x2678,
	(q15_t)0x85DB, (q15_t)0x2648, (q15_t)0x85CC, (q15_t)0x2618,
	(q15_t)0x85BD, (q15_t)0x25E8, (q15_t)0x85AF, (q15_t)0x25B8,
	(q15_t)0x85A0, (q15_t)0x2588, (q15_t)0x8591, (q15_t)0x2558,
	(q15_t)0x8582, (q15_t)0x2528, (q15_t)0x8574, (q15_t)0x24F7,
	(q15_t)0x8565, (q15_t)0x24C7, (q15_t)0x8557, (q15_t)0x2497,
	(q15_t)0x8549, (q15_t)0x2467, (q15_t)0x853A, (q15_t)0x2437,
	(q15_t)0x852C, (q15_t)0x2407, (q15_t)0x851E, (q15_t)0x23D6,
	(q15_t)0x8510, (q15_t)0x23A6, (q15_t)0x8502, (q15_t)0x2376,
	(q15_t)0x84F4, (q15_t)0x2345, (q15_t)0x84E6, (q15_t)0x2315,
	(q15_t)0x84D9, (q15_t)0x22E5, (q15_t)0x84CB, (q15_t)0x22B4,
	(q15_t)0x84BD, (q15_t)0x2284, (q15_t)0x84B0, (q15_t)0x2254,
	(q15_t)0x84A2, (q15_t)0x2223, (q15_t)0x8495, (q15_t)0x21F3,
	(q15_t)0x8488, (q15_t)0x21C2, (q15_t)0x847B, (q15_t)0x2192,
	(q15_t)0x846D, (q15_t)0x2161, (q15_t)0x8460, (q15_t)0x2131,
	(q15_t)0x8453, (q15_t)0x2100, (q15_t)0x8446, (q15_t)0x20D0,
	(q15_t)0x843A, (q15_t)0x209F, (q15_t)0x842D, (q15_t)0x206E,
	(q15_t)0x8420, (q15_t)0x203E, (q15_t)0x8414, (q15_t)0x200D,
	(q15_t)0x8407, (q15_t)0x1FDC, (q15_t)0x83FA, (q15_t)0x1FAC,
	(q15_t)0x83EE, (q15_t)0x1F7B, (q15_t)0x83E2, (q15_t)0x1F4A,
	(q15_t)0x83D6, (q15_t)0x1F19, (q15_t)0x83C9, (q15_t)0x1EE9,
	(q15_t)0x83BD, (q15_t)0x1EB8, (q15_t)0x83B1, (q15_t)0x1E87,
	(q15_t)0x83A5, (q15_t)0x1E56, (q15_t)0x8399, (q15_t)0x1E25,
	(q15_t)0x838E, (q15_t)0x1DF5, (q15_t)0x8382, (q15_t)0x1DC4,
	(q15_t)0x8376, (q15_t)0x1D93, (q15_t)0x836B, (q15_t)0x1D62,
	(q15_t)0x835F, (q15_t)0x1D31, (q15_t)0x8354, (q15_t)0x1D00,
	(q15_t)0x8348, (q15_t)0x1CCF, (q15_t)0x833D, (q15_t)0x1C9E,
	(q15_t)0x8332, (q15_t)0x1C6D, (q15_t)0x8327, (q15_t)0x1C3C,
	(q15_t)0x831C, (q15_t)0x1C0B, (q15_t)0x8311, (q15_t)0x1BDA,
	(q15_t)0x8306, (q15_t)0x1BA9, (q15_t)0x82FB, (q15_t)0x1B78,
	(q15_t)0x82F0, (q15_t)0x1B47, (q15_t)0x82E6, (q15_t)0x1B16,
	(q15_t)0x82DB, (q15_t)0x1AE4, (q15_t)0x82D0, (q15_t)0x1AB3,
	(q15_t)0x82C6, (q15_t)0x1A82, (q15_t)0x82BC, (q15_t)0x1A51,
	(q15_t)0x82B1, (q15_t)0x1A20, (q15_t)0x82A7, (q15_t)0x19EF,
	(q15_t)0x829D, (q15_t)0x19BD, (q15_t)0x8293, (q15_t)0x198C,
	(q15_t)0x8289, (q15_t)0x195B, (q15_t)0x827F, (q15_t)0x192A,
	(q15_t)0x8275, (q15_t)0x18F8, (q15_t)0x826B, (q15_t)0x18C7,
	(q15_t)0x8262, (q15_t)0x1896, (q15_t)0x8258, (q15_t)0x1864,
	(q15_t)0x824F, (q15_t)0x1833, (q15_t)0x8245, (q15_t)0x1802,
	(q15_t)0x823C, (q15_t)0x17D0, (q15_t)0x8232, (q15_t)0x179F,
	(q15_t)0x8229, (q15_t)0x176D, (q15_t)0x8220, (q15_t)0x173C,
	(q15_t)0x8217, (q15_t)0x170A, (q15_t)0x820E, (q15_t)0x16D9,
	(q15_t)0x8205, (q15_t)0x16A8, (q15_t)0x81FC, (q15_t)0x1676,
	(q15_t)0x81F3, (q15_t)0x1645, (q15_t)0x81EB, (q15_t)0x1613,
	(q15_t)0x81E2, (q15_t)0x15E2, (q15_t)0x81D9, (q15_t)0x15B0,
	(q15_t)0x81D1, (q15_t)0x157F, (q15_t)0x81C8, (q15_t)0x154D,
	(q15_t)0x81C0, (q15_t)0x151B, (q15_t)0x81B8, (q15_t)0x14EA,
	(q15_t)0x81B0, (q15_t)0x14B8, (q15_t)0x81A8, (q15_t)0x1487,
	(q15_t)0x81A0, (q15_t)0x1455, (q15_t)0x8198, (q15_t)0x1423,
	(q15_t)0x8190, (q15_t)0x13F2, (q15_t)0x8188, (q15_t)0x13C0,
	(q15_t)0x8180, (q15_t)0x138E, (q15_t)0x8179, (q15_t)0x135D,
	(q15_t)0x8171, (q15_t)0x132B, (q15_t)0x816A, (q15_t)0x12F9,
	(q15_t)0x8162, (q15_t)0x12C8, (q15_t)0x815B, (q15_t)0x1296,
	(q15_t)0x8154, (q15_t)0x1264, (q15_t)0x814C, (q15_t)0x1232,
	(q15_t)0x8145, (q15_t)0x1201, (q15_t)0x813E, (q15_t)0x11CF,
	(q15_t)0x8137, (q15_t)0x119D, (q15_t)0x8130, (q15_t)0x116B,
	(q15_t)0x812A, (q15_t)0x1139, (q15_t)0x8123, (q15_t)0x1108,
	(q15_t)0x811C, (q15_t)0x10D6, (q15_t)0x8116, (q15_t)0x10A4,
	(q15_t)0x810F, (q15_t)0x1072, (q15_t)0x8109, (q15_t)0x1040,
	(q15_t)0x8102, (q15_t)0x100E, (q15_t)0x80FC, (q15_t)0x0FDD,
	(q15_t)0x80F6, (q15_t)0x0FAB, (q15_t)0x80F0, (q15_t)0x0F79,
	(q15_t)0x80EA, (q15_t)0x0F47, (q15_t)0x80E4, (q15_t)0x0F15,
	(q15_t)0x80DE, (q15_t)0x0EE3, (q15_t)0x80D8, (q15_t)0x0EB1,
	(q15_t)0x80D2, (q15_t)0x0E7F, (q15_t)0x80CD, (q15_t)0x0E4D,
	(q15_t)0x80C7, (q15_t)0x0E1B, (q15_t)0x80C2, (q15_t)0x0DE9,
	(q15_t)0x80BC, (q15_t)0x0DB7, (q15_t)0x80B7, (q15_t)0x0D85,
	(q15_t)0x80B2, (q15_t)0x0D53, (q15_t)0x80AC, (q15_t)0x0D21,
	(q15_t)0x80A7, (q15_t)0x0CEF, (q15_t)0x80A2, (q15_t)0x0CBD,
	(q15_t)0x809D, (q15_t)0x0C8B, (q15_t)0x8098, (q15_t)0x0C59,
	(q15_t)0x8094, (q15_t)0x0C27, (q15_t)0x808F, (q15_t)0x0BF5,
	(q15_t)0x808A, (q15_t)0x0BC3, (q15_t)0x8086, (q15_t)0x0B91,
	(q15_t)0x8081, (q15_t)0x0B5F, (q15_t)0x807D, (q15_t)0x0B2D,
	(q15_t)0x8078, (q15_t)0x0AFB, (q15_t)0x8074, (q15_t)0x0AC9,
	(q15_t)0x8070, (q15_t)0x0A97, (q15_t)0x806C, (q15_t)0x0A65,
	(q15_t)0x8068, (q15_t)0x0A33, (q15_t)0x8064, (q15_t)0x0A00,
	(q15_t)0x8060, (q15_t)0x09CE, (q15_t)0x805C, (q15_t)0x099C,
	(q15_t)0x8058, (q15_t)0x096A, (q15_t)0x8055, (q15_t)0x0938,
	(q15_t)0x8051, (q15_t)0x0906, (q15_t)0x804E, (q15_t)0x08D4,
	(q15_t)0x804A, (q15_t)0x08A2, (q15_t)0x8047, (q15_t)0x086F,
	(q15_t)0x8043, (q15_t)0x083D, (q15_t)0x8040, (q15_t)0x080B,
	(q15_t)0x803D, (q15_t)0x07D9, (q15_t)0x803A, (q15_t)0x07A7,
	(q15_t)0x8037, (q15_t)0x0775, (q15_t)0x8034, (q15_t)0x0742,
	(q15_t)0x8031, (q15_t)0x0710, (q15_t)0x802F, (q15_t)0x06DE,
	(q15_t)0x802C, (q15_t)0x06AC, (q15_t)0x8029, (q15_t)0x067A,
	(q15_t)0x8027, (q15_t)0x0647, (q15_t)0x8025, (q15_t)0x0615,
	(q15_t)0x8022, (q15_t)0x05E3, (q15_t)0x8020, (q15_t)0x05B1,
	(q15_t)0x801E, (q15_t)0x057F, (q15_t)0x801C, (q15_t)0x054C,
	(q15_t)0x801A, (q15_t)0x051A, (q15_t)0x8018, (q15_t)0x04E8,
	(q15_t)0x8016, (q15_t)0x04B6, (q15_t)0x8014, (q15_t)0x0483,
	(q15_t)0x8012, (q15_t)0x0451, (q15_t)0x8011, (q15_t)0x041F,
	(q15_t)0x800F, (q15_t)0x03ED, (q15_t)0x800D, (q15_t)0x03BA,
	(q15_t)0x800C, (q15_t)0x0388, (q15_t)0x800B, (q15_t)0x0356,
	(q15_t)0x8009, (q15_t)0x0324, (q15_t)0x8008, (q15_t)0x02F1,
	(q15_t)0x8007, (q15_t)0x02BF, (q15_t)0x8006, (q15_t)0x028D,
	(q15_t)0x8005, (q15_t)0x025B, (q15_t)0x8004, (q15_t)0x0228,
	(q15_t)0x8003, (q15_t)0x01F6, (q15_t)0x8003, (q15_t)0x01C4,
	(q15_t)0x8002, (q15_t)0x0192, (q15_t)0x8001, (q15_t)0x015F,
	(q15_t)0x8001, (q15_t)0x012D, (q15_t)0x8000, (q15_t)0x00FB,
	(q15_t)0x8000, (q15_t)0x00C9, (q15_t)0x8000, (q15_t)0x0096,
	(q15_t)0x8000, (q15_t)0x0064, (q15_t)0x8000, (q15_t)0x0032,
	(q15_t)0x8000, (q15_t)0x0000, (q15_t)0x8000, (q15_t)0xFFCD,
	(q15_t)0x8000, (q15_t)0xFF9B, (q15_t)0x8000, (q15_t)0xFF69,
	(q15_t)0x8000, (q15_t)0xFF36, (q15_t)0x8000, (q15_t)0xFF04,
	(q15_t)0x8001, (q15_t)0xFED2, (q15_t)0x8001, (q15_t)0xFEA0,
	(q15_t)0x8002, (q15_t)0xFE6D, (q15_t)0x8003, (q15_t)0xFE3B,
	(q15_t)0x8003, (q15_t)0xFE09, (q15_t)0x8004, (q15_t)0xFDD7,
	(q15_t)0x8005, (q15_t)0xFDA4, (q15_t)0x8006, (q15_t)0xFD72,
	(q15_t)0x8007, (q15_t)0xFD40, (q15_t)0x8008, (q15_t)0xFD0E,
	(q15_t)0x8009, (q15_t)0xFCDB, (q15_t)0x800B, (q15_t)0xFCA9,
	(q15_t)0x800C, (q15_t)0xFC77, (q15_t)0x800D, (q15_t)0xFC45,
	(q15_t)0x800F, (q15_t)0xFC12, (q15_t)0x8011, (q15_t)0xFBE0,
	(q15_t)0x8012, (q15_t)0xFBAE, (q15_t)0x8014, (q15_t)0xFB7C,
	(q15_t)0x8016, (q15_t)0xFB49, (q15_t)0x8018, (q15_t)0xFB17,
	(q15_t)0x801A, (q15_t)0xFAE5, (q15_t)0x801C, (q15_t)0xFAB3,
	(q15_t)0x801E, (q15_t)0xFA80, (q15_t)0x8020, (q15_t)0xFA4E,
	(q15_t)0x8022, (q15_t)0xFA1C, (q15_t)0x8025, (q15_t)0xF9EA,
	(q15_t)0x8027, (q15_t)0xF9B8, (q15_t)0x8029, (q15_t)0xF985,
	(q15_t)0x802C, (q15_t)0xF953, (q15_t)0x802F, (q15_t)0xF921,
	(q15_t)0x8031, (q15_t)0xF8EF, (q15_t)0x8034, (q15_t)0xF8BD,
	(q15_t)0x8037, (q15_t)0xF88A, (q15_t)0x803A, (q15_t)0xF858,
	(q15_t)0x803D, (q15_t)0xF826, (q15_t)0x8040, (q15_t)0xF7F4,
	(q15_t)0x8043, (q15_t)0xF7C2, (q15_t)0x8047, (q15_t)0xF790,
	(q15_t)0x804A, (q15_t)0xF75D, (q15_t)0x804E, (q15_t)0xF72B,
	(q15_t)0x8051, (q15_t)0xF6F9, (q15_t)0x8055, (q15_t)0xF6C7,
	(q15_t)0x8058, (q15_t)0xF695, (q15_t)0x805C, (q15_t)0xF663,
	(q15_t)0x8060, (q15_t)0xF631, (q15_t)0x8064, (q15_t)0xF5FF,
	(q15_t)0x8068, (q15_t)0xF5CC, (q15_t)0x806C, (q15_t)0xF59A,
	(q15_t)0x8070, (q15_t)0xF568, (q15_t)0x8074, (q15_t)0xF536,
	(q15_t)0x8078, (q15_t)0xF504, (q15_t)0x807D, (q15_t)0xF4D2,
	(q15_t)0x8081, (q15_t)0xF4A0, (q15_t)0x8086, (q15_t)0xF46E,
	(q15_t)0x808A, (q15_t)0xF43C, (q15_t)0x808F, (q15_t)0xF40A,
	(q15_t)0x8094, (q15_t)0xF3D8, (q15_t)0x8098, (q15_t)0xF3A6,
	(q15_t)0x809D, (q15_t)0xF374, (q15_t)0x80A2, (q15_t)0xF342,
	(q15_t)0x80A7, (q15_t)0xF310, (q15_t)0x80AC, (q15_t)0xF2DE,
	(q15_t)0x80B2, (q15_t)0xF2AC, (q15_t)0x80B7, (q15_t)0xF27A,
	(q15_t)0x80BC, (q15_t)0xF248, (q15_t)0x80C2, (q15_t)0xF216,
	(q15_t)0x80C7, (q15_t)0xF1E4, (q15_t)0x80CD, (q15_t)0xF1B2,
	(q15_t)0x80D2, (q15_t)0xF180, (q15_t)0x80D8, (q15_t)0xF14E,
	(q15_t)0x80DE, (q15_t)0xF11C, (q15_t)0x80E4, (q15_t)0xF0EA,
	(q15_t)0x80EA, (q15_t)0xF0B8, (q15_t)0x80F0, (q15_t)0xF086,
	(q15_t)0x80F6, (q15_t)0xF054, (q15_t)0x80FC, (q15_t)0xF022,
	(q15_t)0x8102, (q15_t)0xEFF1, (q15_t)0x8109, (q15_t)0xEFBF,
	(q15_t)0x810F, (q15_t)0xEF8D, (q15_t)0x8116, (q15_t)0xEF5B,
	(q15_t)0x811C, (q15_t)0xEF29, (q15_t)0x8123, (q15_t)0xEEF7,
	(q15_t)0x812A, (q15_t)0xEEC6, (q15_t)0x8130, (q15_t)0xEE94,
	(q15_t)0x8137, (q15_t)0xEE62, (q15_t)0x813E, (q15_t)0xEE30,
	(q15_t)0x8145, (q15_t)0xEDFE, (q15_t)0x814C, (q15_t)0xEDCD,
	(q15_t)0x8154, (q15_t)0xED9B, (q15_t)0x815B, (q15_t)0xED69,
	(q15_t)0x8162, (q15_t)0xED37, (q15_t)0x816A, (q15_t)0xED06,
	(q15_t)0x8171, (q15_t)0xECD4, (q15_t)0x8179, (q15_t)0xECA2,
	(q15_t)0x8180, (q15_t)0xEC71, (q15_t)0x8188, (q15_t)0xEC3F,
	(q15_t)0x8190, (q15_t)0xEC0D, (q15_t)0x8198, (q15_t)0xEBDC,
	(q15_t)0x81A0, (q15_t)0xEBAA, (q15_t)0x81A8, (q15_t)0xEB78,
	(q15_t)0x81B0, (q15_t)0xEB47, (q15_t)0x81B8, (q15_t)0xEB15,
	(q15_t)0x81C0, (q15_t)0xEAE4, (q15_t)0x81C8, (q15_t)0xEAB2,
	(q15_t)0x81D1, (q15_t)0xEA80, (q15_t)0x81D9, (q15_t)0xEA4F,
	(q15_t)0x81E2, (q15_t)0xEA1D, (q15_t)0x81EB, (q15_t)0xE9EC,
	(q15_t)0x81F3, (q15_t)0xE9BA, (q15_t)0x81FC, (q15_t)0xE989,
	(q15_t)0x8205, (q15_t)0xE957, (q15_t)0x820E, (q15_t)0xE926,
	(q15_t)0x8217, (q15_t)0xE8F5, (q15_t)0x8220, (q15_t)0xE8C3,
	(q15_t)0x8229, (q15_t)0xE892, (q15_t)0x8232, (q15_t)0xE860,
	(q15_t)0x823C, (q15_t)0xE82F, (q15_t)0x8245, (q15_t)0xE7FD,
	(q15_t)0x824F, (q15_t)0xE7CC, (q15_t)0x8258, (q15_t)0xE79B,
	(q15_t)0x8262, (q15_t)0xE769, (q15_t)0x826B, (q15_t)0xE738,
	(q15_t)0x8275, (q15_t)0xE707, (q15_t)0x827F, (q15_t)0xE6D5,
	(q15_t)0x8289, (q15_t)0xE6A4, (q15_t)0x8293, (q15_t)0xE673,
	(q15_t)0x829D, (q15_t)0xE642, (q15_t)0x82A7, (q15_t)0xE610,
	(q15_t)0x82B1, (q15_t)0xE5DF, (q15_t)0x82BC, (q15_t)0xE5AE,
	(q15_t)0x82C6, (q15_t)0xE57D, (q15_t)0x82D0, (q15_t)0xE54C,
	(q15_t)0x82DB, (q15_t)0xE51B, (q15_t)0x82E6, (q15_t)0xE4E9,
	(q15_t)0x82F0, (q15_t)0xE4B8, (q15_t)0x82FB, (q15_t)0xE487,
	(q15_t)0x8306, (q15_t)0xE456, (q15_t)0x8311, (q15_t)0xE425,
	(q15_t)0x831C, (q15_t)0xE3F4, (q15_t)0x8327, (q15_t)0xE3C3,
	(q15_t)0x8332, (q15_t)0xE392, (q15_t)0x833D, (q15_t)0xE361,
	(q15_t)0x8348, (q15_t)0xE330, (q15_t)0x8354, (q15_t)0xE2FF,
	(q15_t)0x835F, (q15_t)0xE2CE, (q15_t)0x836B, (q15_t)0xE29D,
	(q15_t)0x8376, (q15_t)0xE26C, (q15_t)0x8382, (q15_t)0xE23B,
	(q15_t)0x838E, (q15_t)0xE20A, (q15_t)0x8399, (q15_t)0xE1DA,
	(q15_t)0x83A5, (q15_t)0xE1A9, (q15_t)0x83B1, (q15_t)0xE178,
	(q15_t)0x83BD, (q15_t)0xE147, (q15_t)0x83C9, (q15_t)0xE116,
	(q15_t)0x83D6, (q15_t)0xE0E6, (q15_t)0x83E2, (q15_t)0xE0B5,
	(q15_t)0x83EE, (q15_t)0xE084, (q15_t)0x83FA, (q15_t)0xE053,
	(q15_t)0x8407, (q15_t)0xE023, (q15_t)0x8414, (q15_t)0xDFF2,
	(q15_t)0x8420, (q15_t)0xDFC1, (q15_t)0x842D, (q15_t)0xDF91,
	(q15_t)0x843A, (q15_t)0xDF60, (q15_t)0x8446, (q15_t)0xDF2F,
	(q15_t)0x8453, (q15_t)0xDEFF, (q15_t)0x8460, (q15_t)0xDECE,
	(q15_t)0x846D, (q15_t)0xDE9E, (q15_t)0x847B, (q15_t)0xDE6D,
	(q15_t)0x8488, (q15_t)0xDE3D, (q15_t)0x8495, (q15_t)0xDE0C,
	(q15_t)0x84A2, (q15_t)0xDDDC, (q15_t)0x84B0, (q15_t)0xDDAB,
	(q15_t)0x84BD, (q15_t)0xDD7B, (q15_t)0x84CB, (q15_t)0xDD4B,
	(q15_t)0x84D9, (q15_t)0xDD1A, (q15_t)0x84E6, (q15_t)0xDCEA,
	(q15_t)0x84F4, (q15_t)0xDCBA, (q15_t)0x8502, (q15_t)0xDC89,
	(q15_t)0x8510, (q15_t)0xDC59, (q15_t)0x851E, (q15_t)0xDC29,
	(q15_t)0x852C, (q15_t)0xDBF8, (q15_t)0x853A, (q15_t)0xDBC8,
	(q15_t)0x8549, (q15_t)0xDB98, (q15_t)0x8557, (q15_t)0xDB68,
	(q15_t)0x8565, (q15_t)0xDB38, (q15_t)0x8574, (q15_t)0xDB08,
	(q15_t)0x8582, (q15_t)0xDAD7, (q15_t)0x8591, (q15_t)0xDAA7,
	(q15_t)0x85A0, (q15_t)0xDA77, (q15_t)0x85AF, (q15_t)0xDA47,
	(q15_t)0x85BD, (q15_t)0xDA17, (q15_t)0x85CC, (q15_t)0xD9E7,
	(q15_t)0x85DB, (q15_t)0xD9B7, (q15_t)0x85EA, (q15_t)0xD987,
	(q15_t)0x85FA, (q15_t)0xD957, (q15_t)0x8609, (q15_t)0xD927,
	(q15_t)0x8618, (q15_t)0xD8F8, (q15_t)0x8627, (q15_t)0xD8C8,
	(q15_t)0x8637, (q15_t)0xD898, (q15_t)0x8646, (q15_t)0xD868,
	(q15_t)0x8656, (q15_t)0xD838, (q15_t)0x8666, (q15_t)0xD809,
	(q15_t)0x8675, (q15_t)0xD7D9, (q15_t)0x8685, (q15_t)0xD7A9,
	(q15_t)0x8695, (q15_t)0xD779, (q15_t)0x86A5, (q15_t)0xD74A,
	(q15_t)0x86B5, (q15_t)0xD71A, (q15_t)0x86C5, (q15_t)0xD6EA,
	(q15_t)0x86D5, (q15_t)0xD6BB, (q15_t)0x86E6, (q15_t)0xD68B,
	(q15_t)0x86F6, (q15_t)0xD65C, (q15_t)0x8706, (q15_t)0xD62C,
	(q15_t)0x8717, (q15_t)0xD5FD, (q15_t)0x8727, (q15_t)0xD5CD,
	(q15_t)0x8738, (q15_t)0xD59E, (q15_t)0x8749, (q15_t)0xD56E,
	(q15_t)0x8759, (q15_t)0xD53F, (q15_t)0x876A, (q15_t)0xD510,
	(q15_t)0x877B, (q15_t)0xD4E0, (q15_t)0x878C, (q15_t)0xD4B1,
	(q15_t)0x879D, (q15_t)0xD482, (q15_t)0x87AE, (q15_t)0xD452,
	(q15_t)0x87BF, (q15_t)0xD423, (q15_t)0x87D1, (q15_t)0xD3F4,
	(q15_t)0x87E2, (q15_t)0xD3C5, (q15_t)0x87F3, (q15_t)0xD396,
	(q15_t)0x8805, (q15_t)0xD367, (q15_t)0x8816, (q15_t)0xD337,
	(q15_t)0x8828, (q15_t)0xD308, (q15_t)0x883A, (q15_t)0xD2D9,
	(q15_t)0x884B, (q15_t)0xD2AA, (q15_t)0x885D, (q15_t)0xD27B,
	(q15_t)0x886F, (q15_t)0xD24C, (q15_t)0x8881, (q15_t)0xD21D,
	(q15_t)0x8893, (q15_t)0xD1EE, (q15_t)0x88A5, (q15_t)0xD1C0,
	(q15_t)0x88B8, (q15_t)0xD191, (q15_t)0x88CA, (q15_t)0xD162,
	(q15_t)0x88DC, (q15_t)0xD133, (q15_t)0x88EF, (q15_t)0xD104,
	(q15_t)0x8901, (q15_t)0xD0D6, (q15_t)0x8914, (q15_t)0xD0A7,
	(q15_t)0x8926, (q15_t)0xD078, (q15_t)0x8939, (q15_t)0xD04A,
	(q15_t)0x894C, (q15_t)0xD01B, (q15_t)0x895F, (q15_t)0xCFEC,
	(q15_t)0x8971, (q15_t)0xCFBE, (q15_t)0x8984, (q15_t)0xCF8F,
	(q15_t)0x8997, (q15_t)0xCF61, (q15_t)0x89AB, (q15_t)0xCF32,
	(q15_t)0x89BE, (q15_t)0xCF04, (q15_t)0x89D1, (q15_t)0xCED5,
	(q15_t)0x89E4, (q15_t)0xCEA7, (q15_t)0x89F8, (q15_t)0xCE79,
	(q15_t)0x8A0B, (q15_t)0xCE4A, (q15_t)0x8A1F, (q15_t)0xCE1C,
	(q15_t)0x8A33, (q15_t)0xCDEE, (q15_t)0x8A46, (q15_t)0xCDBF,
	(q15_t)0x8A5A, (q15_t)0xCD91, (q15_t)0x8A6E, (q15_t)0xCD63,
	(q15_t)0x8A82, (q15_t)0xCD35, (q15_t)0x8A96, (q15_t)0xCD07,
	(q15_t)0x8AAA, (q15_t)0xCCD9, (q15_t)0x8ABE, (q15_t)0xCCAB,
	(q15_t)0x8AD2, (q15_t)0xCC7D, (q15_t)0x8AE6, (q15_t)0xCC4F,
	(q15_t)0x8AFB, (q15_t)0xCC21, (q15_t)0x8B0F, (q15_t)0xCBF3,
	(q15_t)0x8B24, (q15_t)0xCBC5, (q15_t)0x8B38, (q15_t)0xCB97,
	(q15_t)0x8B4D, (q15_t)0xCB69, (q15_t)0x8B61, (q15_t)0xCB3B,
	(q15_t)0x8B76, (q15_t)0xCB0D, (q15_t)0x8B8B, (q15_t)0xCAE0,
	(q15_t)0x8BA0, (q15_t)0xCAB2, (q15_t)0x8BB5, (q15_t)0xCA84,
	(q15_t)0x8BCA, (q15_t)0xCA57, (q15_t)0x8BDF, (q15_t)0xCA29,
	(q15_t)0x8BF4, (q15_t)0xC9FB, (q15_t)0x8C09, (q15_t)0xC9CE,
	(q15_t)0x8C1F, (q15_t)0xC9A0, (q15_t)0x8C34, (q15_t)0xC973,
	(q15_t)0x8C4A, (q15_t)0xC945, (q15_t)0x8C5F, (q15_t)0xC918,
	(q15_t)0x8C75, (q15_t)0xC8EB, (q15_t)0x8C8A, (q15_t)0xC8BD,
	(q15_t)0x8CA0, (q15_t)0xC890, (q15_t)0x8CB6, (q15_t)0xC863,
	(q15_t)0x8CCC, (q15_t)0xC835, (q15_t)0x8CE2, (q15_t)0xC808,
	(q15_t)0x8CF8, (q15_t)0xC7DB, (q15_t)0x8D0E, (q15_t)0xC7AE,
	(q15_t)0x8D24, (q15_t)0xC781, (q15_t)0x8D3A, (q15_t)0xC754,
	(q15_t)0x8D50, (q15_t)0xC727, (q15_t)0x8D67, (q15_t)0xC6F9,
	(q15_t)0x8D7D, (q15_t)0xC6CD, (q15_t)0x8D94, (q15_t)0xC6A0,
	(q15_t)0x8DAA, (q15_t)0xC673, (q15_t)0x8DC1, (q15_t)0xC646,
	(q15_t)0x8DD8, (q15_t)0xC619, (q15_t)0x8DEE, (q15_t)0xC5EC,
	(q15_t)0x8E05, (q15_t)0xC5BF, (q15_t)0x8E1C, (q15_t)0xC593,
	(q15_t)0x8E33, (q15_t)0xC566, (q15_t)0x8E4A, (q15_t)0xC539,
	(q15_t)0x8E61, (q15_t)0xC50D, (q15_t)0x8E79, (q15_t)0xC4E0,
	(q15_t)0x8E90, (q15_t)0xC4B3, (q15_t)0x8EA7, (q15_t)0xC487,
	(q15_t)0x8EBE, (q15_t)0xC45A, (q15_t)0x8ED6, (q15_t)0xC42E,
	(q15_t)0x8EED, (q15_t)0xC402, (q15_t)0x8F05, (q15_t)0xC3D5,
	(q15_t)0x8F1D, (q15_t)0xC3A9, (q15_t)0x8F34, (q15_t)0xC37C,
	(q15_t)0x8F4C, (q15_t)0xC350, (q15_t)0x8F64, (q15_t)0xC324,
	(q15_t)0x8F7C, (q15_t)0xC2F8, (q15_t)0x8F94, (q15_t)0xC2CC,
	(q15_t)0x8FAC, (q15_t)0xC29F, (q15_t)0x8FC4, (q15_t)0xC273,
	(q15_t)0x8FDC, (q15_t)0xC247, (q15_t)0x8FF5, (q15_t)0xC21B,
	(q15_t)0x900D, (q15_t)0xC1EF, (q15_t)0x9025, (q15_t)0xC1C3,
	(q15_t)0x903E, (q15_t)0xC197, (q15_t)0x9056, (q15_t)0xC16C,
	(q15_t)0x906F, (q15_t)0xC140, (q15_t)0x9088, (q15_t)0xC114,
	(q15_t)0x90A0, (q15_t)0xC0E8, (q15_t)0x90B9, (q15_t)0xC0BC,
	(q15_t)0x90D2, (q15_t)0xC091, (q15_t)0x90EB, (q15_t)0xC065,
	(q15_t)0x9104, (q15_t)0xC03A, (q15_t)0x911D, (q15_t)0xC00E,
	(q15_t)0x9136, (q15_t)0xBFE2, (q15_t)0x9150, (q15_t)0xBFB7,
	(q15_t)0x9169, (q15_t)0xBF8C, (q15_t)0x9182, (q15_t)0xBF60,
	(q15_t)0x919C, (q15_t)0xBF35, (q15_t)0x91B5, (q15_t)0xBF09,
	(q15_t)0x91CF, (q15_t)0xBEDE, (q15_t)0x91E8, (q15_t)0xBEB3,
	(q15_t)0x9202, (q15_t)0xBE88, (q15_t)0x921C, (q15_t)0xBE5D,
	(q15_t)0x9235, (q15_t)0xBE31, (q15_t)0x924F, (q15_t)0xBE06,
	(q15_t)0x9269, (q15_t)0xBDDB, (q15_t)0x9283, (q15_t)0xBDB0,
	(q15_t)0x929D, (q15_t)0xBD85, (q15_t)0x92B7, (q15_t)0xBD5A,
	(q15_t)0x92D2, (q15_t)0xBD2F, (q15_t)0x92EC, (q15_t)0xBD05,
	(q15_t)0x9306, (q15_t)0xBCDA, (q15_t)0x9321, (q15_t)0xBCAF,
	(q15_t)0x933B, (q15_t)0xBC84, (q15_t)0x9356, (q15_t)0xBC5A,
	(q15_t)0x9370, (q15_t)0xBC2F, (q15_t)0x938B, (q15_t)0xBC04,
	(q15_t)0x93A6, (q15_t)0xBBDA, (q15_t)0x93C0, (q15_t)0xBBAF,
	(q15_t)0x93DB, (q15_t)0xBB85, (q15_t)0x93F6, (q15_t)0xBB5A,
	(q15_t)0x9411, (q15_t)0xBB30, (q15_t)0x942C, (q15_t)0xBB05,
	(q15_t)0x9447, (q15_t)0xBADB, (q15_t)0x9463, (q15_t)0xBAB1,
	(q15_t)0x947E, (q15_t)0xBA87, (q15_t)0x9499, (q15_t)0xBA5C,
	(q15_t)0x94B5, (q15_t)0xBA32, (q15_t)0x94D0, (q15_t)0xBA08,
	(q15_t)0x94EC, (q15_t)0xB9DE, (q15_t)0x9507, (q15_t)0xB9B4,
	(q15_t)0x9523, (q15_t)0xB98A, (q15_t)0x953E, (q15_t)0xB960,
	(q15_t)0x955A, (q15_t)0xB936, (q15_t)0x9576, (q15_t)0xB90C,
	(q15_t)0x9592, (q15_t)0xB8E3, (q15_t)0x95AE, (q15_t)0xB8B9,
	(q15_t)0x95CA, (q15_t)0xB88F, (q15_t)0x95E6, (q15_t)0xB865,
	(q15_t)0x9602, (q15_t)0xB83C, (q15_t)0x961E, (q15_t)0xB812,
	(q15_t)0x963B, (q15_t)0xB7E9, (q15_t)0x9657, (q15_t)0xB7BF,
	(q15_t)0x9673, (q15_t)0xB796, (q15_t)0x9690, (q15_t)0xB76C,
	(q15_t)0x96AC, (q15_t)0xB743, (q15_t)0x96C9, (q15_t)0xB719,
	(q15_t)0x96E6, (q15_t)0xB6F0, (q15_t)0x9702, (q15_t)0xB6C7,
	(q15_t)0x971F, (q15_t)0xB69E, (q15_t)0x973C, (q15_t)0xB675,
	(q15_t)0x9759, (q15_t)0xB64B, (q15_t)0x9776, (q15_t)0xB622,
	(q15_t)0x9793, (q15_t)0xB5F9, (q15_t)0x97B0, (q15_t)0xB5D0,
	(q15_t)0x97CD, (q15_t)0xB5A7, (q15_t)0x97EA, (q15_t)0xB57E,
	(q15_t)0x9808, (q15_t)0xB556, (q15_t)0x9825, (q15_t)0xB52D,
	(q15_t)0x9842, (q15_t)0xB504, (q15_t)0x9860, (q15_t)0xB4DB,
	(q15_t)0x987D, (q15_t)0xB4B3, (q15_t)0x989B, (q15_t)0xB48A,
	(q15_t)0x98B9, (q15_t)0xB461, (q15_t)0x98D6, (q15_t)0xB439,
	(q15_t)0x98F4, (q15_t)0xB410, (q15_t)0x9912, (q15_t)0xB3E8,
	(q15_t)0x9930, (q15_t)0xB3C0, (q15_t)0x994E, (q15_t)0xB397,
	(q15_t)0x996C, (q15_t)0xB36F, (q15_t)0x998A, (q15_t)0xB347,
	(q15_t)0x99A8, (q15_t)0xB31E, (q15_t)0x99C6, (q15_t)0xB2F6,
	(q15_t)0x99E5, (q15_t)0xB2CE, (q15_t)0x9A03, (q15_t)0xB2A6,
	(q15_t)0x9A22, (q15_t)0xB27E, (q15_t)0x9A40, (q15_t)0xB256,
	(q15_t)0x9A5F, (q15_t)0xB22E, (q15_t)0x9A7D, (q15_t)0xB206,
	(q15_t)0x9A9C, (q15_t)0xB1DE, (q15_t)0x9ABA, (q15_t)0xB1B7,
	(q15_t)0x9AD9, (q15_t)0xB18F, (q15_t)0x9AF8, (q15_t)0xB167,
	(q15_t)0x9B17, (q15_t)0xB140, (q15_t)0x9B36, (q15_t)0xB118,
	(q15_t)0x9B55, (q15_t)0xB0F0, (q15_t)0x9B74, (q15_t)0xB0C9,
	(q15_t)0x9B93, (q15_t)0xB0A1, (q15_t)0x9BB2, (q15_t)0xB07A,
	(q15_t)0x9BD2, (q15_t)0xB053, (q15_t)0x9BF1, (q15_t)0xB02B,
	(q15_t)0x9C10, (q15_t)0xB004, (q15_t)0x9C30, (q15_t)0xAFDD,
	(q15_t)0x9C4F, (q15_t)0xAFB6, (q15_t)0x9C6F, (q15_t)0xAF8F,
	(q15_t)0x9C8E, (q15_t)0xAF68, (q15_t)0x9CAE, (q15_t)0xAF40,
	(q15_t)0x9CCE, (q15_t)0xAF1A, (q15_t)0x9CEE, (q15_t)0xAEF3,
	(q15_t)0x9D0D, (q15_t)0xAECC, (q15_t)0x9D2D, (q15_t)0xAEA5,
	(q15_t)0x9D4D, (q15_t)0xAE7E, (q15_t)0x9D6D, (q15_t)0xAE57,
	(q15_t)0x9D8E, (q15_t)0xAE31, (q15_t)0x9DAE, (q15_t)0xAE0A,
	(q15_t)0x9DCE, (q15_t)0xADE3, (q15_t)0x9DEE, (q15_t)0xADBD,
	(q15_t)0x9E0E, (q15_t)0xAD96, (q15_t)0x9E2F, (q15_t)0xAD70,
	(q15_t)0x9E4F, (q15_t)0xAD4A, (q15_t)0x9E70, (q15_t)0xAD23,
	(q15_t)0x9E90, (q15_t)0xACFD, (q15_t)0x9EB1, (q15_t)0xACD7,
	(q15_t)0x9ED2, (q15_t)0xACB1, (q15_t)0x9EF2, (q15_t)0xAC8A,
	(q15_t)0x9F13, (q15_t)0xAC64, (q15_t)0x9F34, (q15_t)0xAC3E,
	(q15_t)0x9F55, (q15_t)0xAC18, (q15_t)0x9F76, (q15_t)0xABF2,
	(q15_t)0x9F97, (q15_t)0xABCC, (q15_t)0x9FB8, (q15_t)0xABA7,
	(q15_t)0x9FD9, (q15_t)0xAB81, (q15_t)0x9FFB, (q15_t)0xAB5B,
	(q15_t)0xA01C, (q15_t)0xAB35, (q15_t)0xA03D, (q15_t)0xAB10,
	(q15_t)0xA05F, (q15_t)0xAAEA, (q15_t)0xA080, (q15_t)0xAAC5,
	(q15_t)0xA0A1, (q15_t)0xAA9F, (q15_t)0xA0C3, (q15_t)0xAA7A,
	(q15_t)0xA0E5, (q15_t)0xAA54, (q15_t)0xA106, (q15_t)0xAA2F,
	(q15_t)0xA128, (q15_t)0xAA0A, (q15_t)0xA14A, (q15_t)0xA9E5,
	(q15_t)0xA16C, (q15_t)0xA9BF, (q15_t)0xA18E, (q15_t)0xA99A,
	(q15_t)0xA1AF, (q15_t)0xA975, (q15_t)0xA1D2, (q15_t)0xA950,
	(q15_t)0xA1F4, (q15_t)0xA92B, (q15_t)0xA216, (q15_t)0xA906,
	(q15_t)0xA238, (q15_t)0xA8E2, (q15_t)0xA25A, (q15_t)0xA8BD,
	(q15_t)0xA27C, (q15_t)0xA898, (q15_t)0xA29F, (q15_t)0xA873,
	(q15_t)0xA2C1, (q15_t)0xA84F, (q15_t)0xA2E4, (q15_t)0xA82A,
	(q15_t)0xA306, (q15_t)0xA806, (q15_t)0xA329, (q15_t)0xA7E1,
	(q15_t)0xA34B, (q15_t)0xA7BD, (q15_t)0xA36E, (q15_t)0xA798,
	(q15_t)0xA391, (q15_t)0xA774, (q15_t)0xA3B4, (q15_t)0xA750,
	(q15_t)0xA3D6, (q15_t)0xA72B, (q15_t)0xA3F9, (q15_t)0xA707,
	(q15_t)0xA41C, (q15_t)0xA6E3, (q15_t)0xA43F, (q15_t)0xA6BF,
	(q15_t)0xA462, (q15_t)0xA69B, (q15_t)0xA486, (q15_t)0xA677,
	(q15_t)0xA4A9, (q15_t)0xA653, (q15_t)0xA4CC, (q15_t)0xA62F,
	(q15_t)0xA4EF, (q15_t)0xA60C, (q15_t)0xA513, (q15_t)0xA5E8,
	(q15_t)0xA536, (q15_t)0xA5C4, (q15_t)0xA55A, (q15_t)0xA5A1,
	(q15_t)0xA57D, (q15_t)0xA57D, (q15_t)0xA5A1, (q15_t)0xA55A,
	(q15_t)0xA5C4, (q15_t)0xA536, (q15_t)0xA5E8, (q15_t)0xA513,
	(q15_t)0xA60C, (q15_t)0xA4EF, (q15_t)0xA62F, (q15_t)0xA4CC,
	(q15_t)0xA653, (q15_t)0xA4A9, (q15_t)0xA677, (q15_t)0xA486,
	(q15_t)0xA69B, (q15_t)0xA462, (q15_t)0xA6BF, (q15_t)0xA43F,
	(q15_t)0xA6E3, (q15_t)0xA41C, (q15_t)0xA707, (q15_t)0xA3F9,
	(q15_t)0xA72B, (q15_t)0xA3D6, (q15_t)0xA750, (q15_t)0xA3B4,
	(q15_t)0xA774, (q15_t)0xA391, (q15_t)0xA798, (q15_t)0xA36E,
	(q15_t)0xA7BD, (q15_t)0xA34B, (q15_t)0xA7E1, (q15_t)0xA329,
	(q15_t)0xA806, (q15_t)0xA306, (q15_t)0xA82A, (q15_t)0xA2E4,
	(q15_t)0xA84F, (q15_t)0xA2C1, (q15_t)0xA873, (q15_t)0xA29F,
	(q15_t)0xA898, (q15_t)0xA27C, (q15_t)0xA8BD, (q15_t)0xA25A,
	(q15_t)0xA8E2, (q15_t)0xA238, (q15_t)0xA906, (q15_t)0xA216,
	(q15_t)0xA92B, (q15_t)0xA1F4, (q15_t)0xA950, (q15_t)0xA1D2,
	(q15_t)0xA975, (q15_t)0xA1AF, (q15_t)0xA99A, (q15_t)0xA18E,
	(q15_t)0xA9BF, (q15_t)0xA16C, (q15_t)0xA9E5, (q15_t)0xA14A,
	(q15_t)0xAA0A, (q15_t)0xA128, (q15_t)0xAA2F, (q15_t)0xA106,
	(q15_t)0xAA54, (q15_t)0xA0E5, (q15_t)0xAA7A, (q15_t)0xA0C3,
	(q15_t)0xAA9F, (q15_t)0xA0A1, (q15_t)0xAAC5, (q15_t)0xA080,
	(q15_t)0xAAEA, (q15_t)0xA05F, (q15_t)0xAB10, (q15_t)0xA03D,
	(q15_t)0xAB35, (q15_t)0xA01C, (q15_t)0xAB5B, (q15_t)0x9FFB,
	(q15_t)0xAB81, (q15_t)0x9FD9, (q15_t)0xABA7, (q15_t)0x9FB8,
	(q15_t)0xABCC, (q15_t)0x9F97, (q15_t)0xABF2, (q15_t)0x9F76,
	(q15_t)0xAC18, (q15_t)0x9F55, (q15_t)0xAC3E, (q15_t)0x9F34,
	(q15_t)0xAC64, (q15_t)0x9F13, (q15_t)0xAC8A, (q15_t)0x9EF2,
	(q15_t)0xACB1, (q15_t)0x9ED2, (q15_t)0xACD7, (q15_t)0x9EB1,
	(q15_t)0xACFD, (q15_t)0x9E90, (q15_t)0xAD23, (q15_t)0x9E70,
	(q15_t)0xAD4A, (q15_t)0x9E4F, (q15_t)0xAD70, (q15_t)0x9E2F,
	(q15_t)0xAD96, (q15_t)0x9E0E, (q15_t)0xADBD, (q15_t)0x9DEE,
	(q15_t)0xADE3, (q15_t)0x9DCE, (q15_t)0xAE0A, (q15_t)0x9DAE,
	(q15_t)0xAE31, (q15_t)0x9D8E, (q15_t)0xAE57, (q15_t)0x9D6D,
	(q15_t)0xAE7E, (q15_t)0x9D4D, (q15_t)0xAEA5, (q15_t)0x9D2D,
	(q15_t)0xAECC, (q15_t)0x9D0D, (q15_t)0xAEF3, (q15_t)0x9CEE,
	(q15_t)0xAF1A, (q15_t)0x9CCE, (q15_t)0xAF40, (q15_t)0x9CAE,
	(q15_t)0xAF68, (q15_t)0x9C8E, (q15_t)0xAF8F, (q15_t)0x9C6F,
	(q15_t)0xAFB6, (q15_t)0x9C4F, (q15_t)0xAFDD, (q15_t)0x9C30,
	(q15_t)0xB004, (q15_t)0x9C10, (q15_t)0xB02B, (q15_t)0x9BF1,
	(q15_t)0xB053, (q15_t)0x9BD2, (q15_t)0xB07A, (q15_t)0x9BB2,
	(q15_t)0xB0A1, (q15_t)0x9B93, (q15_t)0xB0C9, (q15_t)0x9B74,
	(q15_t)0xB0F0, (q15_t)0x9B55, (q15_t)0xB118, (q15_t)0x9B36,
	(q15_t)0xB140, (q15_t)0x9B17, (q15_t)0xB167, (q15_t)0x9AF8,
	(q15_t)0xB18F, (q15_t)0x9AD9, (q15_t)0xB1B7, (q15_t)0x9ABA,
	(q15_t)0xB1DE, (q15_t)0x9A9C, (q15_t)0xB206, (q15_t)0x9A7D,
	(q15_t)0xB22E, (q15_t)0x9A5F, (q15_t)0xB256, (q15_t)0x9A40,
	(q15_t)0xB27E, (q15_t)0x9A22, (q15_t)0xB2A6, (q15_t)0x9A03,
	(q15_t)0xB2CE, (q15_t)0x99E5, (q15_t)0xB2F6, (q15_t)0x99C6,
	(q15_t)0xB31E, (q15_t)0x99A8, (q15_t)0xB347, (q15_t)0x998A,
	(q15_t)0xB36F, (q15_t)0x996C, (q15_t)0xB397, (q15_t)0x994E,
	(q15_t)0xB3C0, (q15_t)0x9930, (q15_t)0xB3E8, (q15_t)0x9912,
	(q15_t)0xB410, (q15_t)0x98F4, (q15_t)0xB439, (q15_t)0x98D6,
	(q15_t)0xB461, (q15_t)0x98B9, (q15_t)0xB48A, (q15_t)0x989B,
	(q15_t)0xB4B3, (q15_t)0x987D, (q15_t)0xB4DB, (q15_t)0x9860,
	(q15_t)0xB504, (q15_t)0x9842, (q15_t)0xB52D, (q15_t)0x9825,
	(q15_t)0xB556, (q15_t)0x9808, (q15_t)0xB57E, (q15_t)0x97EA,
	(q15_t)0xB5A7, (q15_t)0x97CD, (q15_t)0xB5D0, (q15_t)0x97B0,
	(q15_t)0xB5F9, (q15_t)0x9793, (q15_t)0xB622, (q15_t)0x9776,
	(q15_t)0xB64B, (q15_t)0x9759, (q15_t)0xB675, (q15_t)0x973C,
	(q15_t)0xB69E, (q15_t)0x971F, (q15_t)0xB6C7, (q15_t)0x9702,
	(q15_t)0xB6F0, (q15_t)0x96E6, (q15_t)0xB719, (q15_t)0x96C9,
	(q15_t)0xB743, (q15_t)0x96AC, (q15_t)0xB76C, (q15_t)0x9690,
	(q15_t)0xB796, (q15_t)0x9673, (q15_t)0xB7BF, (q15_t)0x9657,
	(q15_t)0xB7E9, (q15_t)0x963B, (q15_t)0xB812, (q15_t)0x961E,
	(q15_t)0xB83C, (q15_t)0x9602, (q15_t)0xB865, (q15_t)0x95E6,
	(q15_t)0xB88F, (q15_t)0x95CA, (q15_t)0xB8B9, (q15_t)0x95AE,
	(q15_t)0xB8E3, (q15_t)0x9592, (q15_t)0xB90C, (q15_t)0x9576,
	(q15_t)0xB936, (q15_t)0x955A, (q15_t)0xB960, (q15_t)0x953E,
	(q15_t)0xB98A, (q15_t)0x9523, (q15_t)0xB9B4, (q15_t)0x9507,
	(q15_t)0xB9DE, (q15_t)0x94EC, (q15_t)0xBA08, (q15_t)0x94D0,
	(q15_t)0xBA32, (q15_t)0x94B5, (q15_t)0xBA5C, (q15_t)0x9499,
	(q15_t)0xBA87, (q15_t)0x947E, (q15_t)0xBAB1, (q15_t)0x9463,
	(q15_t)0xBADB, (q15_t)0x9447, (q15_t)0xBB05, (q15_t)0x942C,
	(q15_t)0xBB30, (q15_t)0x9411, (q15_t)0xBB5A, (q15_t)0x93F6,
	(q15_t)0xBB85, (q15_t)0x93DB, (q15_t)0xBBAF, (q15_t)0x93C0,
	(q15_t)0xBBDA, (q15_t)0x93A6, (q15_t)0xBC04, (q15_t)0x938B,
	(q15_t)0xBC2F, (q15_t)0x9370, (q15_t)0xBC5A, (q15_t)0x9356,
	(q15_t)0xBC84, (q15_t)0x933B, (q15_t)0xBCAF, (q15_t)0x9321,
	(q15_t)0xBCDA, (q15_t)0x9306, (q15_t)0xBD05, (q15_t)0x92EC,
	(q15_t)0xBD2F, (q15_t)0x92D2, (q15_t)0xBD5A, (q15_t)0x92B7,
	(q15_t)0xBD85, (q15_t)0x929D, (q15_t)0xBDB0, (q15_t)0x9283,
	(q15_t)0xBDDB, (q15_t)0x9269, (q15_t)0xBE06, (q15_t)0x924F,
	(q15_t)0xBE31, (q15_t)0x9235, (q15_t)0xBE5D, (q15_t)0x921C,
	(q15_t)0xBE88, (q15_t)0x9202, (q15_t)0xBEB3, (q15_t)0x91E8,
	(q15_t)0xBEDE, (q15_t)0x91CF, (q15_t)0xBF09, (q15_t)0x91B5,
	(q15_t)0xBF35, (q15_t)0x919C, (q15_t)0xBF60, (q15_t)0x9182,
	(q15_t)0xBF8C, (q15_t)0x9169, (q15_t)0xBFB7, (q15_t)0x9150,
	(q15_t)0xBFE2, (q15_t)0x9136, (q15_t)0xC00E, (q15_t)0x911D,
	(q15_t)0xC03A, (q15_t)0x9104, (q15_t)0xC065, (q15_t)0x90EB,
	(q15_t)0xC091, (q15_t)0x90D2, (q15_t)0xC0BC, (q15_t)0x90B9,
	(q15_t)0xC0E8, (q15_t)0x90A0, (q15_t)0xC114, (q15_t)0x9088,
	(q15_t)0xC140, (q15_t)0x906F, (q15_t)0xC16C, (q15_t)0x9056,
	(q15_t)0xC197, (q15_t)0x903E, (q15_t)0xC1C3, (q15_t)0x9025,
	(q15_t)0xC1EF, (q15_t)0x900D, (q15_t)0xC21B, (q15_t)0x8FF5,
	(q15_t)0xC247, (q15_t)0x8FDC, (q15_t)0xC273, (q15_t)0x8FC4,
	(q15_t)0xC29F, (q15_t)0x8FAC, (q15_t)0xC2CC, (q15_t)0x8F94,
	(q15_t)0xC2F8, (q15_t)0x8F7C, (q15_t)0xC324, (q15_t)0x8F64,
	(q15_t)0xC350, (q15_t)0x8F4C, (q15_t)0xC37C, (q15_t)0x8F34,
	(q15_t)0xC3A9, (q15_t)0x8F1D, (q15_t)0xC3D5, (q15_t)0x8F05,
	(q15_t)0xC402, (q15_t)0x8EED, (q15_t)0xC42E, (q15_t)0x8ED6,
	(q15_t)0xC45A, (q15_t)0x8EBE, (q15_t)0xC487, (q15_t)0x8EA7,
	(q15_t)0xC4B3, (q15_t)0x8E90, (q15_t)0xC4E0, (q15_t)0x8E79,
	(q15_t)0xC50D, (q15_t)0x8E61, (q15_t)0xC539, (q15_t)0x8E4A,
	(q15_t)0xC566, (q15_t)0x8E33, (q15_t)0xC593, (q15_t)0x8E1C,
	(q15_t)0xC5BF, (q15_t)0x8E05, (q15_t)0xC5EC, (q15_t)0x8DEE,
	(q15_t)0xC619, (q15_t)0x8DD8, (q15_t)0xC646, (q15_t)0x8DC1,
	(q15_t)0xC673, (q15_t)0x8DAA, (q15_t)0xC6A0, (q15_t)0x8D94,
	(q15_t)0xC6CD, (q15_t)0x8D7D, (q15_t)0xC6F9, (q15_t)0x8D67,
	(q15_t)0xC727, (q15_t)0x8D50, (q15_t)0xC754, (q15_t)0x8D3A,
	(q15_t)0xC781, (q15_t)0x8D24, (q15_t)0xC7AE, (q15_t)0x8D0E,
	(q15_t)0xC7DB, (q15_t)0x8CF8, (q15_t)0xC808, (q15_t)0x8CE2,
	(q15_t)0xC835, (q15_t)0x8CCC, (q15_t)0xC863, (q15_t)0x8CB6,
	(q15_t)0xC890, (q15_t)0x8CA0, (q15_t)0xC8BD, (q15_t)0x8C8A,
	(q15_t)0xC8EB, (q15_t)0x8C75, (q15_t)0xC918, (q15_t)0x8C5F,
	(q15_t)0xC945, (q15_t)0x8C4A, (q15_t)0xC973, (q15_t)0x8C34,
	(q15_t)0xC9A0, (q15_t)0x8C1F, (q15_t)0xC9CE, (q15_t)0x8C09,
	(q15_t)0xC9FB, (q15_t)0x8BF4, (q15_t)0xCA29, (q15_t)0x8BDF,
	(q15_t)0xCA57, (q15_t)0x8BCA, (q15_t)0xCA84, (q15_t)0x8BB5,
	(q15_t)0xCAB2, (q15_t)0x8BA0, (q15_t)0xCAE0, (q15_t)0x8B8B,
	(q15_t)0xCB0D, (q15_t)0x8B76, (q15_t)0xCB3B, (q15_t)0x8B61,
	(q15_t)0xCB69, (q15_t)0x8B4D, (q15_t)0xCB97, (q15_t)0x8B38,
	(q15_t)0xCBC5, (q15_t)0x8B24, (q15_t)0xCBF3, (q15_t)0x8B0F,
	(q15_t)0xCC21, (q15_t)0x8AFB, (q15_t)0xCC4F, (q15_t)0x8AE6,
	(q15_t)0xCC7D, (q15_t)0x8AD2, (q15_t)0xCCAB, (q15_t)0x8ABE,
	(q15_t)0xCCD9, (q15_t)0x8AAA, (q15_t)0xCD07, (q15_t)0x8A96,
	(q15_t)0xCD35, (q15_t)0x8A82, (q15_t)0xCD63, (q15_t)0x8A6E,
	(q15_t)0xCD91, (q15_t)0x8A5A, (q15_t)0xCDBF, (q15_t)0x8A46,
	(q15_t)0xCDEE, (q15_t)0x8A33, (q15_t)0xCE1C, (q15_t)0x8A1F,
	(q15_t)0xCE4A, (q15_t)0x8A0B, (q15_t)0xCE79, (q15_t)0x89F8,
	(q15_t)0xCEA7, (q15_t)0x89E4, (q15_t)0xCED5, (q15_t)0x89D1,
	(q15_t)0xCF04, (q15_t)0x89BE, (q15_t)0xCF32, (q15_t)0x89AB,
	(q15_t)0xCF61, (q15_t)0x8997, (q15_t)0xCF8F, (q15_t)0x8984,
	(q15_t)0xCFBE, (q15_t)0x8971, (q15_t)0xCFEC, (q15_t)0x895F,
	(q15_t)0xD01B, (q15_t)0x894C, (q15_t)0xD04A, (q15_t)0x8939,
	(q15_t)0xD078, (q15_t)0x8926, (q15_t)0xD0A7, (q15_t)0x8914,
	(q15_t)0xD0D6, (q15_t)0x8901, (q15_t)0xD104, (q15_t)0x88EF,
	(q15_t)0xD133, (q15_t)0x88DC, (q15_t)0xD162, (q15_t)0x88CA,
	(q15_t)0xD191, (q15_t)0x88B8, (q15_t)0xD1C0, (q15_t)0x88A5,
	(q15_t)0xD1EE, (q15_t)0x8893, (q15_t)0xD21D, (q15_t)0x8881,
	(q15_t)0xD24C, (q15_t)0x886F, (q15_t)0xD27B, (q15_t)0x885D,
	(q15_t)0xD2AA, (q15_t)0x884B, (q15_t)0xD2D9, (q15_t)0x883A,
	(q15_t)0xD308, (q15_t)0x8828, (q15_t)0xD337, (q15_t)0x8816,
	(q15_t)0xD367, (q15_t)0x8805, (q15_t)0xD396, (q15_t)0x87F3,
	(q15_t)0xD3C5, (q15_t)0x87E2, (q15_t)0xD3F4, (q15_t)0x87D1,
	(q15_t)0xD423, (q15_t)0x87BF, (q15_t)0xD452, (q15_t)0x87AE,
	(q15_t)0xD482, (q15_t)0x879D, (q15_t)0xD4B1, (q15_t)0x878C,
	(q15_t)0xD4E0, (q15_t)0x877B, (q15_t)0xD510, (q15_t)0x876A,
	(q15_t)0xD53F, (q15_t)0x8759, (q15_t)0xD56E, (q15_t)0x8749,
	(q15_t)0xD59E, (q15_t)0x8738, (q15_t)0xD5CD, (q15_t)0x8727,
	(q15_t)0xD5FD, (q15_t)0x8717, (q15_t)0xD62C, (q15_t)0x8706,
	(q15_t)0xD65C, (q15_t)0x86F6, (q15_t)0xD68B, (q15_t)0x86E6,
	(q15_t)0xD6BB, (q15_t)0x86D5, (q15_t)0xD6EA, (q15_t)0x86C5,
	(q15_t)0xD71A, (q15_t)0x86B5, (q15_t)0xD74A, (q15_t)0x86A5,
	(q15_t)0xD779, (q15_t)0x8695, (q15_t)0xD7A9, (q15_t)0x8685,
	(q15_t)0xD7D9, (q15_t)0x8675, (q15_t)0xD809, (q15_t)0x8666,
	(q15_t)0xD838, (q15_t)0x8656, (q15_t)0xD868, (q15_t)0x8646,
	(q15_t)0xD898, (q15_t)0x8637, (q15_t)0xD8C8, (q15_t)0x8627,
	(q15_t)0xD8F8, (q15_t)0x8618, (q15_t)0xD927, (q15_t)0x8609,
	(q15_t)0xD957, (q15_t)0x85FA, (q15_t)0xD987, (q15_t)0x85EA,
	(q15_t)0xD9B7, (q15_t)0x85DB, (q15_t)0xD9E7, (q15_t)0x85CC,
	(q15_t)0xDA17, (q15_t)0x85BD, (q15_t)0xDA47, (q15_t)0x85AF,
	(q15_t)0xDA77, (q15_t)0x85A0, (q15_t)0xDAA7, (q15_t)0x8591,
	(q15_t)0xDAD7, (q15_t)0x8582, (q15_t)0xDB08, (q15_t)0x8574,
	(q15_t)0xDB38, (q15_t)0x8565, (q15_t)0xDB68, (q15_t)0x8557,
	(q15_t)0xDB98, (q15_t)0x8549, (q15_t)0xDBC8, (q15_t)0x853A,
	(q15_t)0xDBF8, (q15_t)0x852C, (q15_t)0xDC29, (q15_t)0x851E,
	(q15_t)0xDC59, (q15_t)0x8510, (q15_t)0xDC89, (q15_t)0x8502,
	(q15_t)0xDCBA, (q15_t)0x84F4, (q15_t)0xDCEA, (q15_t)0x84E6,
	(q15_t)0xDD1A, (q15_t)0x84D9, (q15_t)0xDD4B, (q15_t)0x84CB,
	(q15_t)0xDD7B, (q15_t)0x84BD, (q15_t)0xDDAB, (q15_t)0x84B0,
	(q15_t)0xDDDC, (q15_t)0x84A2, (q15_t)0xDE0C, (q15_t)0x8495,
	(q15_t)0xDE3D, (q15_t)0x8488, (q15_t)0xDE6D, (q15_t)0x847B,
	(q15_t)0xDE9E, (q15_t)0x846D, (q15_t)0xDECE, (q15_t)0x8460,
	(q15_t)0xDEFF, (q15_t)0x8453, (q15_t)0xDF2F, (q15_t)0x8446,
	(q15_t)0xDF60, (q15_t)0x843A, (q15_t)0xDF91, (q15_t)0x842D,
	(q15_t)0xDFC1, (q15_t)0x8420, (q15_t)0xDFF2, (q15_t)0x8414,
	(q15_t)0xE023, (q15_t)0x8407, (q15_t)0xE053, (q15_t)0x83FA,
	(q15_t)0xE084, (q15_t)0x83EE, (q15_t)0xE0B5, (q15_t)0x83E2,
	(q15_t)0xE0E6, (q15_t)0x83D6, (q15_t)0xE116, (q15_t)0x83C9,
	(q15_t)0xE147, (q15_t)0x83BD, (q15_t)0xE178, (q15_t)0x83B1,
	(q15_t)0xE1A9, (q15_t)0x83A5, (q15_t)0xE1DA, (q15_t)0x8399,
	(q15_t)0xE20A, (q15_t)0x838E, (q15_t)0xE23B, (q15_t)0x8382,
	(q15_t)0xE26C, (q15_t)0x8376, (q15_t)0xE29D, (q15_t)0x836B,
	(q15_t)0xE2CE, (q15_t)0x835F, (q15_t)0xE2FF, (q15_t)0x8354,
	(q15_t)0xE330, (q15_t)0x8348, (q15_t)0xE361, (q15_t)0x833D,
	(q15_t)0xE392, (q15_t)0x8332, (q15_t)0xE3C3, (q15_t)0x8327,
	(q15_t)0xE3F4, (q15_t)0x831C, (q15_t)0xE425, (q15_t)0x8311,
	(q15_t)0xE456, (q15_t)0x8306, (q15_t)0xE487, (q15_t)0x82FB,
	(q15_t)0xE4B8, (q15_t)0x82F0, (q15_t)0xE4E9, (q15_t)0x82E6,
	(q15_t)0xE51B, (q15_t)0x82DB, (q15_t)0xE54C, (q15_t)0x82D0,
	(q15_t)0xE57D, (q15_t)0x82C6, (q15_t)0xE5AE, (q15_t)0x82BC,
	(q15_t)0xE5DF, (q15_t)0x82B1, (q15_t)0xE610, (q15_t)0x82A7,
	(q15_t)0xE642, (q15_t)0x829D, (q15_t)0xE673, (q15_t)0x8293,
	(q15_t)0xE6A4, (q15_t)0x8289, (q15_t)0xE6D5, (q15_t)0x827F,
	(q15_t)0xE707, (q15_t)0x8275, (q15_t)0xE738, (q15_t)0x826B,
	(q15_t)0xE769, (q15_t)0x8262, (q15_t)0xE79B, (q15_t)0x8258,
	(q15_t)0xE7CC, (q15_t)0x824F, (q15_t)0xE7FD, (q15_t)0x8245,
	(q15_t)0xE82F, (q15_t)0x823C, (q15_t)0xE860, (q15_t)0x8232,
	(q15_t)0xE892, (q15_t)0x8229, (q15_t)0xE8C3, (q15_t)0x8220,
	(q15_t)0xE8F5, (q15_t)0x8217, (q15_t)0xE926, (q15_t)0x820E,
	(q15_t)0xE957, (q15_t)0x8205, (q15_t)0xE989, (q15_t)0x81FC,
	(q15_t)0xE9BA, (q15_t)0x81F3, (q15_t)0xE9EC, (q15_t)0x81EB,
	(q15_t)0xEA1D, (q15_t)0x81E2, (q15_t)0xEA4F, (q15_t)0x81D9,
	(q15_t)0xEA80, (q15_t)0x81D1, (q15_t)0xEAB2, (q15_t)0x81C8,
	(q15_t)0xEAE4, (q15_t)0x81C0, (q15_t)0xEB15, (q15_t)0x81B8,
	(q15_t)0xEB47, (q15_t)0x81B0, (q15_t)0xEB78, (q15_t)0x81A8,
	(q15_t)0xEBAA, (q15_t)0x81A0, (q15_t)0xEBDC, (q15_t)0x8198,
	(q15_t)0xEC0D, (q15_t)0x8190, (q15_t)0xEC3F, (q15_t)0x8188,
	(q15_t)0xEC71, (q15_t)0x8180, (q15_t)0xECA2, (q15_t)0x8179,
	(q15_t)0xECD4, (q15_t)0x8171, (q15_t)0xED06, (q15_t)0x816A,
	(q15_t)0xED37, (q15_t)0x8162, (q15_t)0xED69, (q15_t)0x815B,
	(q15_t)0xED9B, (q15_t)0x8154, (q15_t)0xEDCD, (q15_t)0x814C,
	(q15_t)0xEDFE, (q15_t)0x8145, (q15_t)0xEE30, (q15_t)0x813E,
	(q15_t)0xEE62, (q15_t)0x8137, (q15_t)0xEE94, (q15_t)0x8130,
	(q15_t)0xEEC6, (q15_t)0x812A, (q15_t)0xEEF7, (q15_t)0x8123,
	(q15_t)0xEF29, (q15_t)0x811C, (q15_t)0xEF5B, (q15_t)0x8116,
	(q15_t)0xEF8D, (q15_t)0x810F, (q15_t)0xEFBF, (q15_t)0x8109,
	(q15_t)0xEFF1, (q15_t)0x8102, (q15_t)0xF022, (q15_t)0x80FC,
	(q15_t)0xF054, (q15_t)0x80F6, (q15_t)0xF086, (q15_t)0x80F0,
	(q15_t)0xF0B8, (q15_t)0x80EA, (q15_t)0xF0EA, (q15_t)0x80E4,
	(q15_t)0xF11C, (q15_t)0x80DE, (q15_t)0xF14E, (q15_t)0x80D8,
	(q15_t)0xF180, (q15_t)0x80D2, (q15_t)0xF1B2, (q15_t)0x80CD,
	(q15_t)0xF1E4, (q15_t)0x80C7, (q15_t)0xF216, (q15_t)0x80C2,
	(q15_t)0xF248, (q15_t)0x80BC, (q15_t)0xF27A, (q15_t)0x80B7,
	(q15_t)0xF2AC, (q15_t)0x80B2, (q15_t)0xF2DE, (q15_t)0x80AC,
	(q15_t)0xF310, (q15_t)0x80A7, (q15_t)0xF342, (q15_t)0x80A2,
	(q15_t)0xF374, (q15_t)0x809D, (q15_t)0xF3A6, (q15_t)0x8098,
	(q15_t)0xF3D8, (q15_t)0x8094, (q15_t)0xF40A, (q15_t)0x808F,
	(q15_t)0xF43C, (q15_t)0x808A, (q15_t)0xF46E, (q15_t)0x8086,
	(q15_t)0xF4A0, (q15_t)0x8081, (q15_t)0xF4D2, (q15_t)0x807D,
	(q15_t)0xF504, (q15_t)0x8078, (q15_t)0xF536, (q15_t)0x8074,
	(q15_t)0xF568, (q15_t)0x8070, (q15_t)0xF59A, (q15_t)0x806C,
	(q15_t)0xF5CC, (q15_t)0x8068, (q15_t)0xF5FF, (q15_t)0x8064,
	(q15_t)0xF631, (q15_t)0x8060, (q15_t)0xF663, (q15_t)0x805C,
	(q15_t)0xF695, (q15_t)0x8058, (q15_t)0xF6C7, (q15_t)0x8055,
	(q15_t)0xF6F9, (q15_t)0x8051, (q15_t)0xF72B, (q15_t)0x804E,
	(q15_t)0xF75D, (q15_t)0x804A, (q15_t)0xF790, (q15_t)0x8047,
	(q15_t)0xF7C2, (q15_t)0x8043, (q15_t)0xF7F4, (q15_t)0x8040,
	(q15_t)0xF826, (q15_t)0x803D, (q15_t)0xF858, (q15_t)0x803A,
	(q15_t)0xF88A, (q15_t)0x8037, (q15_t)0xF8BD, (q15_t)0x8034,
	(q15_t)0xF8EF, (q15_t)0x8031, (q15_t)0xF921, (q15_t)0x802F,
	(q15_t)0xF953, (q15_t)0x802C, (q15_t)0xF985, (q15_t)0x8029,
	(q15_t)0xF9B8, (q15_t)0x8027, (q15_t)0xF9EA, (q15_t)0x8025,
	(q15_t)0xFA1C, (q15_t)0x8022, (q15_t)0xFA4E, (q15_t)0x8020,
	(q15_t)0xFA80, (q15_t)0x801E, (q15_t)0xFAB3, (q15_t)0x801C,
	(q15_t)0xFAE5, (q15_t)0x801A, (q15_t)0xFB17, (q15_t)0x8018,
	(q15_t)0xFB49, (q15_t)0x8016, (q15_t)0xFB7C, (q15_t)0x8014,
	(q15_t)0xFBAE, (q15_t)0x8012, (q15_t)0xFBE0, (q15_t)0x8011,
	(q15_t)0xFC12, (q15_t)0x800F, (q15_t)0xFC45, (q15_t)0x800D,
	(q15_t)0xFC77, (q15_t)0x800C, (q15_t)0xFCA9, (q15_t)0x800B,
	(q15_t)0xFCDB, (q15_t)0x8009, (q15_t)0xFD0E, (q15_t)0x8008,
	(q15_t)0xFD40, (q15_t)0x8007, (q15_t)0xFD72, (q15_t)0x8006,
	(q15_t)0xFDA4, (q15_t)0x8005, (q15_t)0xFDD7, (q15_t)0x8004,
	(q15_t)0xFE09, (q15_t)0x8003, (q15_t)0xFE3B, (q15_t)0x8003,
	(q15_t)0xFE6D, (q15_t)0x8002, (q15_t)0xFEA0, (q15_t)0x8001,
	(q15_t)0xFED2, (q15_t)0x8001, (q15_t)0xFF04, (q15_t)0x8000,
	(q15_t)0xFF36, (q15_t)0x8000, (q15_t)0xFF69, (q15_t)0x8000,
	(q15_t)0xFF9B, (q15_t)0x8000, (q15_t)0xFFCD, (q15_t)0x8000
};


/**
* @} end of CFFT_CIFFT group
*/

/*
* @brief  Q15 table for reciprocal
*/
const q15_t ALIGN4 armRecipTableQ15[64] = {
 0x7F03, 0x7D13, 0x7B31, 0x795E, 0x7798, 0x75E0,
 0x7434, 0x7294, 0x70FF, 0x6F76, 0x6DF6, 0x6C82,
 0x6B16, 0x69B5, 0x685C, 0x670C, 0x65C4, 0x6484,
 0x634C, 0x621C, 0x60F3, 0x5FD0, 0x5EB5, 0x5DA0,
 0x5C91, 0x5B88, 0x5A85, 0x5988, 0x5890, 0x579E,
 0x56B0, 0x55C8, 0x54E4, 0x5405, 0x532B, 0x5255,
 0x5183, 0x50B6, 0x4FEC, 0x4F26, 0x4E64, 0x4DA6,
 0x4CEC, 0x4C34, 0x4B81, 0x4AD0, 0x4A23, 0x4978,
 0x48D1, 0x482D, 0x478C, 0x46ED, 0x4651, 0x45B8,
 0x4521, 0x448D, 0x43FC, 0x436C, 0x42DF, 0x4255,
 0x41CC, 0x4146, 0x40C2, 0x4040
};

/*
* @brief  Q31 table for reciprocal
*/
const q31_t armRecipTableQ31[64] = {
  0x7F03F03F, 0x7D137420, 0x7B31E739, 0x795E9F94, 0x7798FD29, 0x75E06928,
  0x7434554D, 0x72943B4B, 0x70FF9C40, 0x6F760031, 0x6DF6F593, 0x6C8210E3,
  0x6B16EC3A, 0x69B526F6, 0x685C655F, 0x670C505D, 0x65C4952D, 0x6484E519,
  0x634CF53E, 0x621C7E4F, 0x60F33C61, 0x5FD0EEB3, 0x5EB55785, 0x5DA03BEB,
  0x5C9163A1, 0x5B8898E6, 0x5A85A85A, 0x598860DF, 0x58909373, 0x579E1318,
  0x56B0B4B8, 0x55C84F0B, 0x54E4BA80, 0x5405D124, 0x532B6E8F, 0x52556FD0,
  0x5183B35A, 0x50B618F3, 0x4FEC81A2, 0x4F26CFA2, 0x4E64E64E, 0x4DA6AA1D,
  0x4CEC008B, 0x4C34D010, 0x4B810016, 0x4AD078EF, 0x4A2323C4, 0x4978EA96,
  0x48D1B827, 0x482D77FE, 0x478C1657, 0x46ED801D, 0x4651A2E5, 0x45B86CE2,
  0x4521CCE1, 0x448DB244, 0x43FC0CFA, 0x436CCD78, 0x42DFE4B4, 0x42554426,
  0x41CCDDB6, 0x4146A3C6, 0x40C28923, 0x40408102
};

const uint16_t armBitRevIndexTable16[ARMBITREVINDEXTABLE_16_TABLE_LENGTH] =
{
   /* 8x2, size 20 */
   8,64, 24,72, 16,64, 40,80, 32,64, 56,88, 48,72, 88,104, 72,96, 104,112
};

const uint16_t armBitRevIndexTable32[ARMBITREVINDEXTABLE_32_TABLE_LENGTH] =
{
   /* 8x4, size 48 */
   8,64, 16,128, 24,192, 32,64, 40,72, 48,136, 56,200, 64,128, 72,80, 88,208,
   80,144, 96,192, 104,208, 112,152, 120,216, 136,192, 144,160, 168,208,
   152,224, 176,208, 184,232, 216,240, 200,224, 232,240
};

const uint16_t armBitRevIndexTable64[ARMBITREVINDEXTABLE_64_TABLE_LENGTH] =
{
   /* radix 8, size 56 */
   8,64, 16,128, 24,192, 32,256, 40,320, 48,384, 56,448, 80,136, 88,200,
   96,264, 104,328, 112,392, 120,456, 152,208, 160,272, 168,336, 176,400,
   184,464, 224,280, 232,344, 240,408, 248,472, 296,352, 304,416, 312,480,
   368,424, 376,488, 440,496
};

const uint16_t armBitRevIndexTable128[ARMBITREVINDEXTABLE_128_TABLE_LENGTH] =
{
   /* 8x2, size 208 */
   8,512, 16,64, 24,576, 32,128, 40,640, 48,192, 56,704, 64,256, 72,768,
   80,320, 88,832, 96,384, 104,896, 112,448, 120,960, 128,512, 136,520,
   144,768, 152,584, 160,520, 168,648, 176,200, 184,712, 192,264, 200,776,
   208,328, 216,840, 224,392, 232,904, 240,456, 248,968, 264,528, 272,320,
   280,592, 288,768, 296,656, 304,328, 312,720, 328,784, 344,848, 352,400,
   360,912, 368,464, 376,976, 384,576, 392,536, 400,832, 408,600, 416,584,
   424,664, 432,840, 440,728, 448,592, 456,792, 464,848, 472,856, 480,600,
   488,920, 496,856, 504,984, 520,544, 528,576, 536,608, 552,672, 560,608,
   568,736, 576,768, 584,800, 592,832, 600,864, 608,800, 616,928, 624,864,
   632,992, 648,672, 656,896, 664,928, 688,904, 696,744, 704,896, 712,808,
   720,912, 728,872, 736,928, 744,936, 752,920, 760,1000, 776,800, 784,832,
   792,864, 808,904, 816,864, 824,920, 840,864, 856,880, 872,944, 888,1008,
   904,928, 912,960, 920,992, 944,968, 952,1000, 968,992, 984,1008
};

const uint16_t armBitRevIndexTable256[ARMBITREVINDEXTABLE_256_TABLE_LENGTH] =
{
   /* 8x4, size 440 */
   8,512, 16,1024, 24,1536, 32,64, 40,576, 48,1088, 56,1600, 64,128, 72,640,
   80,1152, 88,1664, 96,192, 104,704, 112,1216, 120,1728, 128,256, 136,768,
   144,1280, 152,1792, 160,320, 168,832, 176,1344, 184,1856, 192,384,
   200,896, 208,1408, 216,1920, 224,448, 232,960, 240,1472, 248,1984,
   256,512, 264,520, 272,1032, 280,1544, 288,640, 296,584, 304,1096, 312,1608,
   320,768, 328,648, 336,1160, 344,1672, 352,896, 360,712, 368,1224, 376,1736,
   384,520, 392,776, 400,1288, 408,1800, 416,648, 424,840, 432,1352, 440,1864,
   448,776, 456,904, 464,1416, 472,1928, 480,904, 488,968, 496,1480, 504,1992,
   520,528, 512,1024, 528,1040, 536,1552, 544,1152, 552,592, 560,1104,
   568,1616, 576,1280, 584,656, 592,1168, 600,1680, 608,1408, 616,720,
   624,1232, 632,1744, 640,1032, 648,784, 656,1296, 664,1808, 672,1160,
   680,848, 688,1360, 696,1872, 704,1288, 712,912, 720,1424, 728,1936,
   736,1416, 744,976, 752,1488, 760,2000, 768,1536, 776,1552, 784,1048,
   792,1560, 800,1664, 808,1680, 816,1112, 824,1624, 832,1792, 840,1808,
   848,1176, 856,1688, 864,1920, 872,1936, 880,1240, 888,1752, 896,1544,
   904,1560, 912,1304, 920,1816, 928,1672, 936,1688, 944,1368, 952,1880,
   960,1800, 968,1816, 976,1432, 984,1944, 992,1928, 1000,1944, 1008,1496,
   1016,2008, 1032,1152, 1040,1056, 1048,1568, 1064,1408, 1072,1120,
   1080,1632, 1088,1536, 1096,1160, 1104,1184, 1112,1696, 1120,1552,
   1128,1416, 1136,1248, 1144,1760, 1160,1664, 1168,1312, 1176,1824,
   1184,1544, 1192,1920, 1200,1376, 1208,1888, 1216,1568, 1224,1672,
   1232,1440, 1240,1952, 1248,1560, 1256,1928, 1264,1504, 1272,2016,
   1288,1312, 1296,1408, 1304,1576, 1320,1424, 1328,1416, 1336,1640,
   1344,1792, 1352,1824, 1360,1920, 1368,1704, 1376,1800, 1384,1432,
   1392,1928, 1400,1768, 1416,1680, 1432,1832, 1440,1576, 1448,1936,
   1456,1832, 1464,1896, 1472,1808, 1480,1688, 1488,1936, 1496,1960,
   1504,1816, 1512,1944, 1520,1944, 1528,2024, 1560,1584, 1592,1648,
   1600,1792, 1608,1920, 1616,1800, 1624,1712, 1632,1808, 1640,1936,
   1648,1816, 1656,1776, 1672,1696, 1688,1840, 1704,1952, 1712,1928,
   1720,1904, 1728,1824, 1736,1952, 1744,1832, 1752,1968, 1760,1840,
   1768,1960, 1776,1944, 1784,2032, 1864,1872, 1848,1944, 1872,1888,
   1880,1904, 1888,1984, 1896,2000, 1912,2032, 1904,2016, 1976,2032,
   1960,1968, 2008,2032, 1992,2016, 2024,2032
};

const uint16_t armBitRevIndexTable512[ARMBITREVINDEXTABLE_512_TABLE_LENGTH] =
{
   /* radix 8, size 448 */
   8,512, 16,1024, 24,1536, 32,2048, 40,2560, 48,3072, 56,3584, 72,576,
   80,1088, 88,1600, 96,2112, 104,2624, 112,3136, 120,3648, 136,640, 144,1152,
   152,1664, 160,2176, 168,2688, 176,3200, 184,3712, 200,704, 208,1216,
   216,1728, 224,2240, 232,2752, 240,3264, 248,3776, 264,768, 272,1280,
   280,1792, 288,2304, 296,2816, 304,3328, 312,3840, 328,832, 336,1344,
   344,1856, 352,2368, 360,2880, 368,3392, 376,3904, 392,896, 400,1408,
   408,1920, 416,2432, 424,2944, 432,3456, 440,3968, 456,960, 464,1472,
   472,1984, 480,2496, 488,3008, 496,3520, 504,4032, 528,1032, 536,1544,
   544,2056, 552,2568, 560,3080, 568,3592, 592,1096, 600,1608, 608,2120,
   616,2632, 624,3144, 632,3656, 656,1160, 664,1672, 672,2184, 680,2696,
   688,3208, 696,3720, 720,1224, 728,1736, 736,2248, 744,2760, 752,3272,
   760,3784, 784,1288, 792,1800, 800,2312, 808,2824, 816,3336, 824,3848,
   848,1352, 856,1864, 864,2376, 872,2888, 880,3400, 888,3912, 912,1416,
   920,1928, 928,2440, 936,2952, 944,3464, 952,3976, 976,1480, 984,1992,
   992,2504, 1000,3016, 1008,3528, 1016,4040, 1048,1552, 1056,2064, 1064,2576,
   1072,3088, 1080,3600, 1112,1616, 1120,2128, 1128,2640, 1136,3152,
   1144,3664, 1176,1680, 1184,2192, 1192,2704, 1200,3216, 1208,3728,
   1240,1744, 1248,2256, 1256,2768, 1264,3280, 1272,3792, 1304,1808,
   1312,2320, 1320,2832, 1328,3344, 1336,3856, 1368,1872, 1376,2384,
   1384,2896, 1392,3408, 1400,3920, 1432,1936, 1440,2448, 1448,2960,
   1456,3472, 1464,3984, 1496,2000, 1504,2512, 1512,3024, 1520,3536,
   1528,4048, 1568,2072, 1576,2584, 1584,3096, 1592,3608, 1632,2136,
   1640,2648, 1648,3160, 1656,3672, 1696,2200, 1704,2712, 1712,3224,
   1720,3736, 1760,2264, 1768,2776, 1776,3288, 1784,3800, 1824,2328,
   1832,2840, 1840,3352, 1848,3864, 1888,2392, 1896,2904, 1904,3416,
   1912,3928, 1952,2456, 1960,2968, 1968,3480, 1976,3992, 2016,2520,
   2024,3032, 2032,3544, 2040,4056, 2088,2592, 2096,3104, 2104,3616,
   2152,2656, 2160,3168, 2168,3680, 2216,2720, 2224,3232, 2232,3744,
   2280,2784, 2288,3296, 2296,3808, 2344,2848, 2352,3360, 2360,3872,
   2408,2912, 2416,3424, 2424,3936, 2472,2976, 2480,3488, 2488,4000,
   2536,3040, 2544,3552, 2552,4064, 2608,3112, 2616,3624, 2672,3176,
   2680,3688, 2736,3240, 2744,3752, 2800,3304, 2808,3816, 2864,3368,
   2872,3880, 2928,3432, 2936,3944, 2992,3496, 3000,4008, 3056,3560,
   3064,4072, 3128,3632, 3192,3696, 3256,3760, 3320,3824, 3384,3888,
   3448,3952, 3512,4016, 3576,4080
};
#if 0  /* AmebaD already define this array in DROM */
const uint16_t armBitRevIndexTable1024[ARMBITREVINDEXTABLE_1024_TABLE_LENGTH] =
{
   /* 8x2, size 1800 */
   8,4096, 16,512, 24,4608, 32,1024, 40,5120, 48,1536, 56,5632, 64,2048,
   72,6144, 80,2560, 88,6656, 96,3072, 104,7168, 112,3584, 120,7680, 128,2048,
   136,4160, 144,576, 152,4672, 160,1088, 168,5184, 176,1600, 184,5696,
   192,2112, 200,6208, 208,2624, 216,6720, 224,3136, 232,7232, 240,3648,
   248,7744, 256,2048, 264,4224, 272,640, 280,4736, 288,1152, 296,5248,
   304,1664, 312,5760, 320,2176, 328,6272, 336,2688, 344,6784, 352,3200,
   360,7296, 368,3712, 376,7808, 384,2112, 392,4288, 400,704, 408,4800,
   416,1216, 424,5312, 432,1728, 440,5824, 448,2240, 456,6336, 464,2752,
   472,6848, 480,3264, 488,7360, 496,3776, 504,7872, 512,2048, 520,4352,
   528,768, 536,4864, 544,1280, 552,5376, 560,1792, 568,5888, 576,2304,
   584,6400, 592,2816, 600,6912, 608,3328, 616,7424, 624,3840, 632,7936,
   640,2176, 648,4416, 656,832, 664,4928, 672,1344, 680,5440, 688,1856,
   696,5952, 704,2368, 712,6464, 720,2880, 728,6976, 736,3392, 744,7488,
   752,3904, 760,8000, 768,2112, 776,4480, 784,896, 792,4992, 800,1408,
   808,5504, 816,1920, 824,6016, 832,2432, 840,6528, 848,2944, 856,7040,
   864,3456, 872,7552, 880,3968, 888,8064, 896,2240, 904,4544, 912,960,
   920,5056, 928,1472, 936,5568, 944,1984, 952,6080, 960,2496, 968,6592,
   976,3008, 984,7104, 992,3520, 1000,7616, 1008,4032, 1016,8128, 1024,4096,
   1032,4104, 1040,4352, 1048,4616, 1056,4104, 1064,5128, 1072,1544,
   1080,5640, 1088,2056, 1096,6152, 1104,2568, 1112,6664, 1120,3080,
   1128,7176, 1136,3592, 1144,7688, 1152,6144, 1160,4168, 1168,6400,
   1176,4680, 1184,6152, 1192,5192, 1200,1608, 1208,5704, 1216,2120,
   1224,6216, 1232,2632, 1240,6728, 1248,3144, 1256,7240, 1264,3656,
   1272,7752, 1280,4160, 1288,4232, 1296,4416, 1304,4744, 1312,4168,
   1320,5256, 1328,1672, 1336,5768, 1344,2184, 1352,6280, 1360,2696,
   1368,6792, 1376,3208, 1384,7304, 1392,3720, 1400,7816, 1408,6208,
   1416,4296, 1424,6464, 1432,4808, 1440,6216, 1448,5320, 1456,1736,
   1464,5832, 1472,2248, 1480,6344, 1488,2760, 1496,6856, 1504,3272,
   1512,7368, 1520,3784, 1528,7880, 1536,4224, 1544,4360, 1552,4480,
   1560,4872, 1568,4232, 1576,5384, 1584,1800, 1592,5896, 1600,2312,
   1608,6408, 1616,2824, 1624,6920, 1632,3336, 1640,7432, 1648,3848,
   1656,7944, 1664,6272, 1672,4424, 1680,6528, 1688,4936, 1696,6280,
   1704,5448, 1712,1864, 1720,5960, 1728,2376, 1736,6472, 1744,2888,
   1752,6984, 1760,3400, 1768,7496, 1776,3912, 1784,8008, 1792,4288,
   1800,4488, 1808,4544, 1816,5000, 1824,4296, 1832,5512, 1840,1928,
   1848,6024, 1856,2440, 1864,6536, 1872,2952, 1880,7048, 1888,3464,
   1896,7560, 1904,3976, 1912,8072, 1920,6336, 1928,4552, 1936,6592,
   1944,5064, 1952,6344, 1960,5576, 1968,1992, 1976,6088, 1984,2504,
   1992,6600, 2000,3016, 2008,7112, 2016,3528, 2024,7624, 2032,4040,
   2040,8136, 2056,4112, 2064,2112, 2072,4624, 2080,4352, 2088,5136,
   2096,4480, 2104,5648, 2120,6160, 2128,2576, 2136,6672, 2144,3088,
   2152,7184, 2160,3600, 2168,7696, 2176,2560, 2184,4176, 2192,2816,
   2200,4688, 2208,2568, 2216,5200, 2224,2824, 2232,5712, 2240,2576,
   2248,6224, 2256,2640, 2264,6736, 2272,3152, 2280,7248, 2288,3664,
   2296,7760, 2312,4240, 2320,2432, 2328,4752, 2336,6400, 2344,5264,
   2352,6528, 2360,5776, 2368,2816, 2376,6288, 2384,2704, 2392,6800,
   2400,3216, 2408,7312, 2416,3728, 2424,7824, 2432,2624, 2440,4304,
   2448,2880, 2456,4816, 2464,2632, 2472,5328, 2480,2888, 2488,5840,
   2496,2640, 2504,6352, 2512,2768, 2520,6864, 2528,3280, 2536,7376,
   2544,3792, 2552,7888, 2568,4368, 2584,4880, 2592,4416, 2600,5392,
   2608,4544, 2616,5904, 2632,6416, 2640,2832, 2648,6928, 2656,3344,
   2664,7440, 2672,3856, 2680,7952, 2696,4432, 2704,2944, 2712,4944,
   2720,4432, 2728,5456, 2736,2952, 2744,5968, 2752,2944, 2760,6480,
   2768,2896, 2776,6992, 2784,3408, 2792,7504, 2800,3920, 2808,8016,
   2824,4496, 2840,5008, 2848,6464, 2856,5520, 2864,6592, 2872,6032,
   2888,6544, 2896,2960, 2904,7056, 2912,3472, 2920,7568, 2928,3984,
   2936,8080, 2952,4560, 2960,3008, 2968,5072, 2976,6480, 2984,5584,
   2992,3016, 3000,6096, 3016,6608, 3032,7120, 3040,3536, 3048,7632,
   3056,4048, 3064,8144, 3072,4608, 3080,4120, 3088,4864, 3096,4632,
   3104,4616, 3112,5144, 3120,4872, 3128,5656, 3136,4624, 3144,6168,
   3152,4880, 3160,6680, 3168,4632, 3176,7192, 3184,3608, 3192,7704,
   3200,6656, 3208,4184, 3216,6912, 3224,4696, 3232,6664, 3240,5208,
   3248,6920, 3256,5720, 3264,6672, 3272,6232, 3280,6928, 3288,6744,
   3296,6680, 3304,7256, 3312,3672, 3320,7768, 3328,4672, 3336,4248,
   3344,4928, 3352,4760, 3360,4680, 3368,5272, 3376,4936, 3384,5784,
   3392,4688, 3400,6296, 3408,4944, 3416,6808, 3424,4696, 3432,7320,
   3440,3736, 3448,7832, 3456,6720, 3464,4312, 3472,6976, 3480,4824,
   3488,6728, 3496,5336, 3504,6984, 3512,5848, 3520,6736, 3528,6360,
   3536,6992, 3544,6872, 3552,6744, 3560,7384, 3568,3800, 3576,7896,
   3584,4736, 3592,4376, 3600,4992, 3608,4888, 3616,4744, 3624,5400,
   3632,5000, 3640,5912, 3648,4752, 3656,6424, 3664,5008, 3672,6936,
   3680,4760, 3688,7448, 3696,3864, 3704,7960, 3712,6784, 3720,4440,
   3728,7040, 3736,4952, 3744,6792, 3752,5464, 3760,7048, 3768,5976,
   3776,6800, 3784,6488, 3792,7056, 3800,7000, 3808,6808, 3816,7512,
   3824,3928, 3832,8024, 3840,4800, 3848,4504, 3856,5056, 3864,5016,
   3872,4808, 3880,5528, 3888,5064, 3896,6040, 3904,4816, 3912,6552,
   3920,5072, 3928,7064, 3936,4824, 3944,7576, 3952,3992, 3960,8088,
   3968,6848, 3976,4568, 3984,7104, 3992,5080, 4000,6856, 4008,5592,
   4016,7112, 4024,6104, 4032,6864, 4040,6616, 4048,7120, 4056,7128,
   4064,6872, 4072,7640, 4080,7128, 4088,8152, 4104,4128, 4112,4160,
   4120,4640, 4136,5152, 4144,4232, 4152,5664, 4160,4352, 4168,6176,
   4176,4416, 4184,6688, 4192,4616, 4200,7200, 4208,4744, 4216,7712,
   4224,4608, 4232,4616, 4240,4672, 4248,4704, 4256,4640, 4264,5216,
   4272,4704, 4280,5728, 4288,4864, 4296,6240, 4304,4928, 4312,6752,
   4320,4632, 4328,7264, 4336,4760, 4344,7776, 4360,4640, 4368,4416,
   4376,4768, 4384,6152, 4392,5280, 4400,6280, 4408,5792, 4424,6304,
   4440,6816, 4448,6664, 4456,7328, 4464,6792, 4472,7840, 4480,4624,
   4488,4632, 4496,4688, 4504,4832, 4512,6168, 4520,5344, 4528,6296,
   4536,5856, 4544,4880, 4552,6368, 4560,4944, 4568,6880, 4576,6680,
   4584,7392, 4592,6808, 4600,7904, 4608,6144, 4616,6152, 4624,6208,
   4632,4896, 4640,6176, 4648,5408, 4656,6240, 4664,5920, 4672,6400,
   4680,6432, 4688,6464, 4696,6944, 4704,6432, 4712,7456, 4720,4808,
   4728,7968, 4736,6656, 4744,6664, 4752,6720, 4760,4960, 4768,6688,
   4776,5472, 4784,6752, 4792,5984, 4800,6912, 4808,6496, 4816,6976,
   4824,7008, 4832,6944, 4840,7520, 4848,7008, 4856,8032, 4864,6160,
   4872,6168, 4880,6224, 4888,5024, 4896,6216, 4904,5536, 4912,6344,
   4920,6048, 4928,6416, 4936,6560, 4944,6480, 4952,7072, 4960,6728,
   4968,7584, 4976,6856, 4984,8096, 4992,6672, 5000,6680, 5008,6736,
   5016,5088, 5024,6232, 5032,5600, 5040,6360, 5048,6112, 5056,6928,
   5064,6624, 5072,6992, 5080,7136, 5088,6744, 5096,7648, 5104,6872,
   5112,8160, 5128,5152, 5136,5376, 5144,5408, 5168,5384, 5176,5672,
   5184,5376, 5192,6184, 5200,5392, 5208,6696, 5216,5408, 5224,7208,
   5232,5400, 5240,7720, 5248,7168, 5256,7200, 5264,7424, 5272,7456,
   5280,7176, 5288,7208, 5296,7432, 5304,5736, 5312,7184, 5320,6248,
   5328,7440, 5336,6760, 5344,7192, 5352,7272, 5360,7448, 5368,7784,
   5384,5408, 5392,5440, 5400,5472, 5408,6184, 5416,7208, 5424,5448,
   5432,5800, 5448,6312, 5464,6824, 5472,6696, 5480,7336, 5488,6824,
   5496,7848, 5504,7232, 5512,7264, 5520,7488, 5528,7520, 5536,7240,
   5544,7272, 5552,7496, 5560,5864, 5568,7248, 5576,6376, 5584,7504,
   5592,6888, 5600,7256, 5608,7400, 5616,7512, 5624,7912, 5632,7168,
   5640,7176, 5648,7232, 5656,7240, 5664,7200, 5672,7208, 5680,7264,
   5688,5928, 5696,7424, 5704,6440, 5712,7488, 5720,6952, 5728,7456,
   5736,7464, 5744,7520, 5752,7976, 5760,7296, 5768,7328, 5776,7552,
   5784,7584, 5792,7304, 5800,7336, 5808,7560, 5816,5992, 5824,7312,
   5832,6504, 5840,7568, 5848,7016, 5856,7320, 5864,7528, 5872,7576,
   5880,8040, 5888,7184, 5896,7192, 5904,7248, 5912,7256, 5920,6248,
   5928,7272, 5936,6376, 5944,6056, 5952,7440, 5960,6568, 5968,7504,
   5976,7080, 5984,6760, 5992,7592, 6000,6888, 6008,8104, 6016,7360,
   6024,7392, 6032,7616, 6040,7648, 6048,7368, 6056,7400, 6064,7624,
   6072,6120, 6080,7376, 6088,6632, 6096,7632, 6104,7144, 6112,7384,
   6120,7656, 6128,7640, 6136,8168, 6168,6240, 6192,6216, 6200,7264,
   6232,6704, 6248,7216, 6256,6680, 6264,7728, 6272,6656, 6280,6664,
   6288,6912, 6296,6496, 6304,6688, 6312,6696, 6320,6944, 6328,7520,
   6336,6672, 6344,6680, 6352,6928, 6360,6768, 6368,6704, 6376,7280,
   6384,6744, 6392,7792, 6408,6432, 6424,6752, 6440,7432, 6448,6536,
   6456,7560, 6472,6944, 6488,6832, 6496,6920, 6504,7344, 6512,7048,
   6520,7856, 6528,6720, 6536,6728, 6544,6976, 6552,7008, 6560,6752,
   6568,7448, 6576,7008, 6584,7576, 6592,6736, 6600,6744, 6608,6992,
   6616,6896, 6624,6936, 6632,7408, 6640,7064, 6648,7920, 6712,7280,
   6744,6960, 6760,7472, 6768,6936, 6776,7984, 6800,6848, 6808,6856,
   6832,6880, 6840,6888, 6848,7040, 6856,7048, 6864,7104, 6872,7024,
   6880,7072, 6888,7536, 6896,7136, 6904,8048, 6952,7496, 6968,7624,
   6984,7008, 7000,7088, 7016,7600, 7024,7112, 7032,8112, 7056,7104,
   7064,7112, 7080,7512, 7088,7136, 7096,7640, 7128,7152, 7144,7664,
   7160,8176, 7176,7200, 7192,7216, 7224,7272, 7240,7264, 7256,7280,
   7288,7736, 7296,7680, 7304,7712, 7312,7936, 7320,7968, 7328,7688,
   7336,7720, 7344,7944, 7352,7976, 7360,7696, 7368,7728, 7376,7952,
   7384,7984, 7392,7704, 7400,7736, 7408,7960, 7416,7800, 7432,7456,
   7448,7472, 7480,7592, 7496,7520, 7512,7536, 7528,7976, 7544,7864,
   7552,7744, 7560,7776, 7568,8000, 7576,8032, 7584,7752, 7592,7784,
   7600,8008, 7608,8040, 7616,7760, 7624,7792, 7632,8016, 7640,8048,
   7648,7768, 7656,7800, 7664,8024, 7672,7928, 7688,7712, 7704,7728,
   7752,7776, 7768,7792, 7800,7992, 7816,7840, 7824,8064, 7832,8096,
   7856,8072, 7864,8104, 7872,8064, 7880,8072, 7888,8080, 7896,8112,
   7904,8096, 7912,8104, 7920,8088, 7928,8056, 7944,7968, 7960,7984,
   8008,8032, 8024,8048, 8056,8120, 8072,8096, 8080,8128, 8088,8160,
   8112,8136, 8120,8168, 8136,8160, 8152,8176
};
#endif
const uint16_t armBitRevIndexTable2048[ARMBITREVINDEXTABLE_2048_TABLE_LENGTH] =
{
   /* 8x2, size 3808 */
   8,4096, 16,8192, 24,12288, 32,512, 40,4608, 48,8704, 56,12800, 64,1024,
   72,5120, 80,9216, 88,13312, 96,1536, 104,5632, 112,9728, 120,13824,
   128,2048, 136,6144, 144,10240, 152,14336, 160,2560, 168,6656, 176,10752,
   184,14848, 192,3072, 200,7168, 208,11264, 216,15360, 224,3584, 232,7680,
   240,11776, 248,15872, 256,1024, 264,4160, 272,8256, 280,12352, 288,576,
   296,4672, 304,8768, 312,12864, 320,1088, 328,5184, 336,9280, 344,13376,
   352,1600, 360,5696, 368,9792, 376,13888, 384,2112, 392,6208, 400,10304,
   408,14400, 416,2624, 424,6720, 432,10816, 440,14912, 448,3136, 456,7232,
   464,11328, 472,15424, 480,3648, 488,7744, 496,11840, 504,15936, 512,2048,
   520,4224, 528,8320, 536,12416, 544,640, 552,4736, 560,8832, 568,12928,
   576,1152, 584,5248, 592,9344, 600,13440, 608,1664, 616,5760, 624,9856,
   632,13952, 640,2176, 648,6272, 656,10368, 664,14464, 672,2688, 680,6784,
   688,10880, 696,14976, 704,3200, 712,7296, 720,11392, 728,15488, 736,3712,
   744,7808, 752,11904, 760,16000, 768,3072, 776,4288, 784,8384, 792,12480,
   800,3200, 808,4800, 816,8896, 824,12992, 832,1216, 840,5312, 848,9408,
   856,13504, 864,1728, 872,5824, 880,9920, 888,14016, 896,2240, 904,6336,
   912,10432, 920,14528, 928,2752, 936,6848, 944,10944, 952,15040, 960,3264,
   968,7360, 976,11456, 984,15552, 992,3776, 1000,7872, 1008,11968, 1016,16064,
   1032,4352, 1040,8448, 1048,12544, 1056,3072, 1064,4864, 1072,8960,
   1080,13056, 1088,1280, 1096,5376, 1104,9472, 1112,13568, 1120,1792,
   1128,5888, 1136,9984, 1144,14080, 1152,2304, 1160,6400, 1168,10496,
   1176,14592, 1184,2816, 1192,6912, 1200,11008, 1208,15104, 1216,3328,
   1224,7424, 1232,11520, 1240,15616, 1248,3840, 1256,7936, 1264,12032,
   1272,16128, 1288,4416, 1296,8512, 1304,12608, 1312,3328, 1320,4928,
   1328,9024, 1336,13120, 1352,5440, 1360,9536, 1368,13632, 1376,1856,
   1384,5952, 1392,10048, 1400,14144, 1408,2368, 1416,6464, 1424,10560,
   1432,14656, 1440,2880, 1448,6976, 1456,11072, 1464,15168, 1472,3392,
   1480,7488, 1488,11584, 1496,15680, 1504,3904, 1512,8000, 1520,12096,
   1528,16192, 1536,2112, 1544,4480, 1552,8576, 1560,12672, 1568,2240,
   1576,4992, 1584,9088, 1592,13184, 1600,2368, 1608,5504, 1616,9600,
   1624,13696, 1632,1920, 1640,6016, 1648,10112, 1656,14208, 1664,2432,
   1672,6528, 1680,10624, 1688,14720, 1696,2944, 1704,7040, 1712,11136,
   1720,15232, 1728,3456, 1736,7552, 1744,11648, 1752,15744, 1760,3968,
   1768,8064, 1776,12160, 1784,16256, 1792,3136, 1800,4544, 1808,8640,
   1816,12736, 1824,3264, 1832,5056, 1840,9152, 1848,13248, 1856,3392,
   1864,5568, 1872,9664, 1880,13760, 1888,1984, 1896,6080, 1904,10176,
   1912,14272, 1920,2496, 1928,6592, 1936,10688, 1944,14784, 1952,3008,
   1960,7104, 1968,11200, 1976,15296, 1984,3520, 1992,7616, 2000,11712,
   2008,15808, 2016,4032, 2024,8128, 2032,12224, 2040,16320, 2048,4096,
   2056,4104, 2064,8200, 2072,12296, 2080,4224, 2088,4616, 2096,8712,
   2104,12808, 2112,4352, 2120,5128, 2128,9224, 2136,13320, 2144,4480,
   2152,5640, 2160,9736, 2168,13832, 2176,4104, 2184,6152, 2192,10248,
   2200,14344, 2208,2568, 2216,6664, 2224,10760, 2232,14856, 2240,3080,
   2248,7176, 2256,11272, 2264,15368, 2272,3592, 2280,7688, 2288,11784,
   2296,15880, 2304,5120, 2312,4168, 2320,8264, 2328,12360, 2336,5248,
   2344,4680, 2352,8776, 2360,12872, 2368,5376, 2376,5192, 2384,9288,
   2392,13384, 2400,5504, 2408,5704, 2416,9800, 2424,13896, 2432,5128,
   2440,6216, 2448,10312, 2456,14408, 2464,2632, 2472,6728, 2480,10824,
   2488,14920, 2496,3144, 2504,7240, 2512,11336, 2520,15432, 2528,3656,
   2536,7752, 2544,11848, 2552,15944, 2560,6144, 2568,4232, 2576,8328,
   2584,12424, 2592,6272, 2600,4744, 2608,8840, 2616,12936, 2624,6400,
   2632,5256, 2640,9352, 2648,13448, 2656,6528, 2664,5768, 2672,9864,
   2680,13960, 2688,6152, 2696,6280, 2704,10376, 2712,14472, 2720,6280,
   2728,6792, 2736,10888, 2744,14984, 2752,3208, 2760,7304, 2768,11400,
   2776,15496, 2784,3720, 2792,7816, 2800,11912, 2808,16008, 2816,7168,
   2824,4296, 2832,8392, 2840,12488, 2848,7296, 2856,4808, 2864,8904,
   2872,13000, 2880,7424, 2888,5320, 2896,9416, 2904,13512, 2912,7552,
   2920,5832, 2928,9928, 2936,14024, 2944,7176, 2952,6344, 2960,10440,
   2968,14536, 2976,7304, 2984,6856, 2992,10952, 3000,15048, 3008,3272,
   3016,7368, 3024,11464, 3032,15560, 3040,3784, 3048,7880, 3056,11976,
   3064,16072, 3072,4160, 3080,4360, 3088,8456, 3096,12552, 3104,4288,
   3112,4872, 3120,8968, 3128,13064, 3136,4416, 3144,5384, 3152,9480,
   3160,13576, 3168,4544, 3176,5896, 3184,9992, 3192,14088, 3200,4168,
   3208,6408, 3216,10504, 3224,14600, 3232,4296, 3240,6920, 3248,11016,
   3256,15112, 3264,3336, 3272,7432, 3280,11528, 3288,15624, 3296,3848,
   3304,7944, 3312,12040, 3320,16136, 3328,5184, 3336,4424, 3344,8520,
   3352,12616, 3360,5312, 3368,4936, 3376,9032, 3384,13128, 3392,5440,
   3400,5448, 3408,9544, 3416,13640, 3424,5568, 3432,5960, 3440,10056,
   3448,14152, 3456,5192, 3464,6472, 3472,10568, 3480,14664, 3488,5320,
   3496,6984, 3504,11080, 3512,15176, 3520,5448, 3528,7496, 3536,11592,
   3544,15688, 3552,3912, 3560,8008, 3568,12104, 3576,16200, 3584,6208,
   3592,4488, 3600,8584, 3608,12680, 3616,6336, 3624,5000, 3632,9096,
   3640,13192, 3648,6464, 3656,5512, 3664,9608, 3672,13704, 3680,6592,
   3688,6024, 3696,10120, 3704,14216, 3712,6216, 3720,6536, 3728,10632,
   3736,14728, 3744,6344, 3752,7048, 3760,11144, 3768,15240, 3776,6472,
   3784,7560, 3792,11656, 3800,15752, 3808,3976, 3816,8072, 3824,12168,
   3832,16264, 3840,7232, 3848,4552, 3856,8648, 3864,12744, 3872,7360,
   3880,5064, 3888,9160, 3896,13256, 3904,7488, 3912,5576, 3920,9672,
   3928,13768, 3936,7616, 3944,6088, 3952,10184, 3960,14280, 3968,7240,
   3976,6600, 3984,10696, 3992,14792, 4000,7368, 4008,7112, 4016,11208,
   4024,15304, 4032,7496, 4040,7624, 4048,11720, 4056,15816, 4064,7624,
   4072,8136, 4080,12232, 4088,16328, 4096,8192, 4104,4112, 4112,8208,
   4120,12304, 4128,8320, 4136,4624, 4144,8720, 4152,12816, 4160,8448,
   4168,5136, 4176,9232, 4184,13328, 4192,8576, 4200,5648, 4208,9744,
   4216,13840, 4224,8200, 4232,6160, 4240,10256, 4248,14352, 4256,8328,
   4264,6672, 4272,10768, 4280,14864, 4288,8456, 4296,7184, 4304,11280,
   4312,15376, 4320,8584, 4328,7696, 4336,11792, 4344,15888, 4352,9216,
   4360,9232, 4368,8272, 4376,12368, 4384,9344, 4392,4688, 4400,8784,
   4408,12880, 4416,9472, 4424,5200, 4432,9296, 4440,13392, 4448,9600,
   4456,5712, 4464,9808, 4472,13904, 4480,9224, 4488,6224, 4496,10320,
   4504,14416, 4512,9352, 4520,6736, 4528,10832, 4536,14928, 4544,9480,
   4552,7248, 4560,11344, 4568,15440, 4576,9608, 4584,7760, 4592,11856,
   4600,15952, 4608,10240, 4616,10256, 4624,8336, 4632,12432, 4640,10368,
   4648,4752, 4656,8848, 4664,12944, 4672,10496, 4680,5264, 4688,9360,
   4696,13456, 4704,10624, 4712,5776, 4720,9872, 4728,13968, 4736,10248,
   4744,6288, 4752,10384, 4760,14480, 4768,10376, 4776,6800, 4784,10896,
   4792,14992, 4800,10504, 4808,7312, 4816,11408, 4824,15504, 4832,10632,
   4840,7824, 4848,11920, 4856,16016, 4864,11264, 4872,11280, 4880,8400,
   4888,12496, 4896,11392, 4904,11408, 4912,8912, 4920,13008, 4928,11520,
   4936,5328, 4944,9424, 4952,13520, 4960,11648, 4968,5840, 4976,9936,
   4984,14032, 4992,11272, 5000,6352, 5008,10448, 5016,14544, 5024,11400,
   5032,6864, 5040,10960, 5048,15056, 5056,11528, 5064,7376, 5072,11472,
   5080,15568, 5088,11656, 5096,7888, 5104,11984, 5112,16080, 5120,8256,
   5128,8272, 5136,8464, 5144,12560, 5152,8384, 5160,8400, 5168,8976,
   5176,13072, 5184,8512, 5192,5392, 5200,9488, 5208,13584, 5216,8640,
   5224,5904, 5232,10000, 5240,14096, 5248,8264, 5256,6416, 5264,10512,
   5272,14608, 5280,8392, 5288,6928, 5296,11024, 5304,15120, 5312,8520,
   5320,7440, 5328,11536, 5336,15632, 5344,8648, 5352,7952, 5360,12048,
   5368,16144, 5376,9280, 5384,9296, 5392,8528, 5400,12624, 5408,9408,
   5416,9424, 5424,9040, 5432,13136, 5440,9536, 5448,5456, 5456,9552,
   5464,13648, 5472,9664, 5480,5968, 5488,10064, 5496,14160, 5504,9288,
   5512,6480, 5520,10576, 5528,14672, 5536,9416, 5544,6992, 5552,11088,
   5560,15184, 5568,9544, 5576,7504, 5584,11600, 5592,15696, 5600,9672,
   5608,8016, 5616,12112, 5624,16208, 5632,10304, 5640,10320, 5648,8592,
   5656,12688, 5664,10432, 5672,10448, 5680,9104, 5688,13200, 5696,10560,
   5704,10576, 5712,9616, 5720,13712, 5728,10688, 5736,6032, 5744,10128,
   5752,14224, 5760,10312, 5768,6544, 5776,10640, 5784,14736, 5792,10440,
   5800,7056, 5808,11152, 5816,15248, 5824,10568, 5832,7568, 5840,11664,
   5848,15760, 5856,10696, 5864,8080, 5872,12176, 5880,16272, 5888,11328,
   5896,11344, 5904,8656, 5912,12752, 5920,11456, 5928,11472, 5936,9168,
   5944,13264, 5952,11584, 5960,11600, 5968,9680, 5976,13776, 5984,11712,
   5992,6096, 6000,10192, 6008,14288, 6016,11336, 6024,6608, 6032,10704,
   6040,14800, 6048,11464, 6056,7120, 6064,11216, 6072,15312, 6080,11592,
   6088,7632, 6096,11728, 6104,15824, 6112,11720, 6120,8144, 6128,12240,
   6136,16336, 6144,12288, 6152,12304, 6160,8216, 6168,12312, 6176,12416,
   6184,12432, 6192,8728, 6200,12824, 6208,12544, 6216,12560, 6224,9240,
   6232,13336, 6240,12672, 6248,12688, 6256,9752, 6264,13848, 6272,12296,
   6280,12312, 6288,10264, 6296,14360, 6304,12424, 6312,6680, 6320,10776,
   6328,14872, 6336,12552, 6344,7192, 6352,11288, 6360,15384, 6368,12680,
   6376,7704, 6384,11800, 6392,15896, 6400,13312, 6408,13328, 6416,8280,
   6424,12376, 6432,13440, 6440,13456, 6448,8792, 6456,12888, 6464,13568,
   6472,13584, 6480,9304, 6488,13400, 6496,13696, 6504,13712, 6512,9816,
   6520,13912, 6528,13320, 6536,13336, 6544,10328, 6552,14424, 6560,13448,
   6568,6744, 6576,10840, 6584,14936, 6592,13576, 6600,7256, 6608,11352,
   6616,15448, 6624,13704, 6632,7768, 6640,11864, 6648,15960, 6656,14336,
   6664,14352, 6672,8344, 6680,12440, 6688,14464, 6696,14480, 6704,8856,
   6712,12952, 6720,14592, 6728,14608, 6736,9368, 6744,13464, 6752,14720,
   6760,14736, 6768,9880, 6776,13976, 6784,14344, 6792,14360, 6800,10392,
   6808,14488, 6816,14472, 6824,14488, 6832,10904, 6840,15000, 6848,14600,
   6856,7320, 6864,11416, 6872,15512, 6880,14728, 6888,7832, 6896,11928,
   6904,16024, 6912,15360, 6920,15376, 6928,8408, 6936,12504, 6944,15488,
   6952,15504, 6960,8920, 6968,13016, 6976,15616, 6984,15632, 6992,9432,
   7000,13528, 7008,15744, 7016,15760, 7024,9944, 7032,14040, 7040,15368,
   7048,15384, 7056,10456, 7064,14552, 7072,15496, 7080,15512, 7088,10968,
   7096,15064, 7104,15624, 7112,7384, 7120,11480, 7128,15576, 7136,15752,
   7144,7896, 7152,11992, 7160,16088, 7168,12352, 7176,12368, 7184,8472,
   7192,12568, 7200,12480, 7208,12496, 7216,8984, 7224,13080, 7232,12608,
   7240,12624, 7248,9496, 7256,13592, 7264,12736, 7272,12752, 7280,10008,
   7288,14104, 7296,12360, 7304,12376, 7312,10520, 7320,14616, 7328,12488,
   7336,12504, 7344,11032, 7352,15128, 7360,12616, 7368,7448, 7376,11544,
   7384,15640, 7392,12744, 7400,7960, 7408,12056, 7416,16152, 7424,13376,
   7432,13392, 7440,8536, 7448,12632, 7456,13504, 7464,13520, 7472,9048,
   7480,13144, 7488,13632, 7496,13648, 7504,9560, 7512,13656, 7520,13760,
   7528,13776, 7536,10072, 7544,14168, 7552,13384, 7560,13400, 7568,10584,
   7576,14680, 7584,13512, 7592,13528, 7600,11096, 7608,15192, 7616,13640,
   7624,13656, 7632,11608, 7640,15704, 7648,13768, 7656,8024, 7664,12120,
   7672,16216, 7680,14400, 7688,14416, 7696,8600, 7704,12696, 7712,14528,
   7720,14544, 7728,9112, 7736,13208, 7744,14656, 7752,14672, 7760,9624,
   7768,13720, 7776,14784, 7784,14800, 7792,10136, 7800,14232, 7808,14408,
   7816,14424, 7824,10648, 7832,14744, 7840,14536, 7848,14552, 7856,11160,
   7864,15256, 7872,14664, 7880,14680, 7888,11672, 7896,15768, 7904,14792,
   7912,8088, 7920,12184, 7928,16280, 7936,15424, 7944,15440, 7952,8664,
   7960,12760, 7968,15552, 7976,15568, 7984,9176, 7992,13272, 8000,15680,
   8008,15696, 8016,9688, 8024,13784, 8032,15808, 8040,15824, 8048,10200,
   8056,14296, 8064,15432, 8072,15448, 8080,10712, 8088,14808, 8096,15560,
   8104,15576, 8112,11224, 8120,15320, 8128,15688, 8136,15704, 8144,11736,
   8152,15832, 8160,15816, 8168,15832, 8176,12248, 8184,16344, 8200,8320,
   8208,8224, 8216,12320, 8232,10368, 8240,8736, 8248,12832, 8256,8448,
   8264,8384, 8272,9248, 8280,13344, 8288,9232, 8296,10432, 8304,9760,
   8312,13856, 8328,12416, 8336,10272, 8344,14368, 8352,12296, 8360,14464,
   8368,10784, 8376,14880, 8384,8456, 8392,12480, 8400,11296, 8408,15392,
   8416,12552, 8424,14528, 8432,11808, 8440,15904, 8448,9216, 8456,8576,
   8464,9232, 8472,12384, 8480,9248, 8488,10624, 8496,8800, 8504,12896,
   8512,9472, 8520,8640, 8528,9312, 8536,13408, 8544,9296, 8552,10688,
   8560,9824, 8568,13920, 8576,9224, 8584,12672, 8592,10336, 8600,14432,
   8608,13320, 8616,14720, 8624,10848, 8632,14944, 8640,9480, 8648,12736,
   8656,11360, 8664,15456, 8672,13576, 8680,14784, 8688,11872, 8696,15968,
   8704,12288, 8712,12416, 8720,12296, 8728,12448, 8736,12304, 8744,10376,
   8752,8864, 8760,12960, 8768,12352, 8776,12480, 8784,9376, 8792,13472,
   8800,12368, 8808,10440, 8816,9888, 8824,13984, 8832,12320, 8840,12424,
   8848,10400, 8856,14496, 8864,12312, 8872,14472, 8880,10912, 8888,15008,
   8896,12384, 8904,12488, 8912,11424, 8920,15520, 8928,12568, 8936,14536,
   8944,11936, 8952,16032, 8960,12544, 8968,12672, 8976,12552, 8984,12512,
   8992,12560, 9000,10632, 9008,12568, 9016,13024, 9024,12608, 9032,12736,
   9040,9440, 9048,13536, 9056,12624, 9064,10696, 9072,9952, 9080,14048,
   9088,9240, 9096,12680, 9104,10464, 9112,14560, 9120,13336, 9128,14728,
   9136,10976, 9144,15072, 9152,9496, 9160,12744, 9168,11488, 9176,15584,
   9184,13592, 9192,14792, 9200,12000, 9208,16096, 9224,9344, 9232,9248,
   9240,12576, 9256,11392, 9264,12560, 9272,13088, 9280,9472, 9288,9408,
   9296,9504, 9304,13600, 9312,9488, 9320,11456, 9328,10016, 9336,14112,
   9352,13440, 9360,10528, 9368,14624, 9376,12360, 9384,15488, 9392,11040,
   9400,15136, 9408,9480, 9416,13504, 9424,11552, 9432,15648, 9440,12616,
   9448,15552, 9456,12064, 9464,16160, 9480,9600, 9488,9504, 9496,12640,
   9512,11648, 9520,12624, 9528,13152, 9544,9664, 9552,9568, 9560,13664,
   9576,11712, 9584,10080, 9592,14176, 9608,13696, 9616,10592, 9624,14688,
   9632,13384, 9640,15744, 9648,11104, 9656,15200, 9672,13760, 9680,11616,
   9688,15712, 9696,13640, 9704,15808, 9712,12128, 9720,16224, 9728,13312,
   9736,13440, 9744,13320, 9752,12704, 9760,13328, 9768,11400, 9776,13336,
   9784,13216, 9792,13376, 9800,13504, 9808,13384, 9816,13728, 9824,13392,
   9832,11464, 9840,10144, 9848,14240, 9856,13344, 9864,13448, 9872,10656,
   9880,14752, 9888,12376, 9896,15496, 9904,11168, 9912,15264, 9920,13408,
   9928,13512, 9936,11680, 9944,15776, 9952,12632, 9960,15560, 9968,12192,
   9976,16288, 9984,13568, 9992,13696, 10000,13576, 10008,12768, 10016,13584,
   10024,11656, 10032,13592, 10040,13280, 10048,13632, 10056,13760,
   10064,13640, 10072,13792, 10080,13648, 10088,11720, 10096,10208,
   10104,14304, 10112,13600, 10120,13704, 10128,10720, 10136,14816,
   10144,13400, 10152,15752, 10160,11232, 10168,15328, 10176,13664,
   10184,13768, 10192,11744, 10200,15840, 10208,13656, 10216,15816,
   10224,12256, 10232,16352, 10248,10272, 10256,10368, 10264,12328,
   10280,10384, 10288,10376, 10296,12840, 10304,11264, 10312,11296,
   10320,11392, 10328,13352, 10336,11272, 10344,10448, 10352,11400,
   10360,13864, 10376,12432, 10392,14376, 10400,12328, 10408,14480,
   10416,10792, 10424,14888, 10432,11280, 10440,12496, 10448,11304,
   10456,15400, 10464,11288, 10472,14544, 10480,11816, 10488,15912,
   10496,11264, 10504,11272, 10512,11280, 10520,12392, 10528,11296,
   10536,10640, 10544,12496, 10552,12904, 10560,11328, 10568,11360,
   10576,11456, 10584,13416, 10592,11336, 10600,10704, 10608,11464,
   10616,13928, 10624,11392, 10632,12688, 10640,11304, 10648,14440,
   10656,13352, 10664,14736, 10672,10856, 10680,14952, 10688,11344,
   10696,12752, 10704,11368, 10712,15464, 10720,11352, 10728,14800,
   10736,11880, 10744,15976, 10752,14336, 10760,14368, 10768,14464,
   10776,12456, 10784,14344, 10792,14376, 10800,14472, 10808,12968,
   10816,15360, 10824,15392, 10832,15488, 10840,13480, 10848,15368,
   10856,15400, 10864,15496, 10872,13992, 10880,14352, 10888,12440,
   10896,14480, 10904,14504, 10912,14360, 10920,14488, 10928,14488,
   10936,15016, 10944,15376, 10952,12504, 10960,11432, 10968,15528,
   10976,15384, 10984,14552, 10992,11944, 11000,16040, 11008,14400,
   11016,14432, 11024,14528, 11032,12520, 11040,14408, 11048,14440,
   11056,14536, 11064,13032, 11072,15424, 11080,15456, 11088,15552,
   11096,13544, 11104,15432, 11112,15464, 11120,15560, 11128,14056,
   11136,14416, 11144,12696, 11152,14544, 11160,14568, 11168,14424,
   11176,14744, 11184,14552, 11192,15080, 11200,15440, 11208,12760,
   11216,11496, 11224,15592, 11232,15448, 11240,14808, 11248,12008,
   11256,16104, 11272,11296, 11280,11392, 11288,12584, 11304,11408,
   11312,12688, 11320,13096, 11328,11520, 11336,11552, 11344,11648,
   11352,13608, 11360,11528, 11368,11472, 11376,11656, 11384,14120,
   11400,13456, 11416,14632, 11424,12392, 11432,15504, 11440,14440,
   11448,15144, 11456,11536, 11464,13520, 11472,11560, 11480,15656,
   11488,11544, 11496,15568, 11504,12072, 11512,16168, 11528,11552,
   11536,11648, 11544,12648, 11560,11664, 11568,12752, 11576,13160,
   11592,11616, 11600,11712, 11608,13672, 11624,11728, 11632,11720,
   11640,14184, 11656,13712, 11672,14696, 11680,13416, 11688,15760,
   11696,15464, 11704,15208, 11720,13776, 11736,15720, 11744,13672,
   11752,15824, 11760,12136, 11768,16232, 11776,14592, 11784,14624,
   11792,14720, 11800,12712, 11808,14600, 11816,14632, 11824,14728,
   11832,13224, 11840,15616, 11848,15648, 11856,15744, 11864,13736,
   11872,15624, 11880,15656, 11888,15752, 11896,14248, 11904,14608,
   11912,13464, 11920,14736, 11928,14760, 11936,14616, 11944,15512,
   11952,14744, 11960,15272, 11968,15632, 11976,13528, 11984,15760,
   11992,15784, 12000,15640, 12008,15576, 12016,12200, 12024,16296,
   12032,14656, 12040,14688, 12048,14784, 12056,12776, 12064,14664,
   12072,14696, 12080,14792, 12088,13288, 12096,15680, 12104,15712,
   12112,15808, 12120,13800, 12128,15688, 12136,15720, 12144,15816,
   12152,14312, 12160,14672, 12168,13720, 12176,14800, 12184,14824,
   12192,14680, 12200,15768, 12208,14808, 12216,15336, 12224,15696,
   12232,13784, 12240,15824, 12248,15848, 12256,15704, 12264,15832,
   12272,15832, 12280,16360, 12312,12336, 12344,12848, 12352,12544,
   12360,12552, 12368,12560, 12376,13360, 12384,12576, 12392,12584,
   12400,13336, 12408,13872, 12424,12448, 12440,14384, 12456,14496,
   12464,14472, 12472,14896, 12480,12672, 12488,12512, 12496,12688,
   12504,15408, 12512,12680, 12520,14560, 12528,14728, 12536,15920,
   12544,13312, 12552,13320, 12560,13328, 12568,13336, 12576,13344,
   12584,13352, 12592,13360, 12600,12912, 12608,13568, 12616,13576,
   12624,13584, 12632,13424, 12640,13600, 12648,13608, 12656,13400,
   12664,13936, 12672,13440, 12680,12704, 12688,13456, 12696,14448,
   12704,13448, 12712,14752, 12720,15496, 12728,14960, 12736,13696,
   12744,12768, 12752,13712, 12760,15472, 12768,13704, 12776,14816,
   12784,15752, 12792,15984, 12800,14336, 12808,14464, 12816,14344,
   12824,14472, 12832,14352, 12840,14480, 12848,14360, 12856,12976,
   12864,14400, 12872,14528, 12880,14408, 12888,13488, 12896,14416,
   12904,14544, 12912,14424, 12920,14000, 12928,14368, 12936,14496,
   12944,14376, 12952,14512, 12960,14384, 12968,14504, 12976,14488,
   12984,15024, 12992,14432, 13000,14560, 13008,14440, 13016,15536,
   13024,14448, 13032,14568, 13040,14744, 13048,16048, 13056,14592,
   13064,14720, 13072,14600, 13080,14728, 13088,14608, 13096,14736,
   13104,14616, 13112,14744, 13120,14656, 13128,14784, 13136,14664,
   13144,13552, 13152,14672, 13160,14800, 13168,14680, 13176,14064,
   13184,14624, 13192,14752, 13200,14632, 13208,14576, 13216,13464,
   13224,14760, 13232,15512, 13240,15088, 13248,14688, 13256,14816,
   13264,14696, 13272,15600, 13280,13720, 13288,14824, 13296,15768,
   13304,16112, 13336,13360, 13368,14616, 13376,13568, 13384,13576,
   13392,13584, 13400,13616, 13408,13600, 13416,13608, 13424,13592,
   13432,14128, 13448,13472, 13464,14640, 13480,15520, 13488,14536,
   13496,15152, 13504,13696, 13512,13536, 13520,13712, 13528,15664,
   13536,13704, 13544,15584, 13552,14792, 13560,16176, 13592,13616,
   13624,14680, 13656,13680, 13688,14192, 13704,13728, 13720,14704,
   13736,15776, 13744,15560, 13752,15216, 13768,13792, 13784,15728,
   13800,15840, 13808,15816, 13816,16240, 13824,15360, 13832,15488,
   13840,15368, 13848,15496, 13856,15376, 13864,15504, 13872,15384,
   13880,15512, 13888,15424, 13896,15552, 13904,15432, 13912,15560,
   13920,15440, 13928,15568, 13936,15448, 13944,14256, 13952,15392,
   13960,15520, 13968,15400, 13976,14768, 13984,15408, 13992,15528,
   14000,14552, 14008,15280, 14016,15456, 14024,15584, 14032,15464,
   14040,15792, 14048,15472, 14056,15592, 14064,14808, 14072,16304,
   14080,15616, 14088,15744, 14096,15624, 14104,15752, 14112,15632,
   14120,15760, 14128,15640, 14136,15768, 14144,15680, 14152,15808,
   14160,15688, 14168,15816, 14176,15696, 14184,15824, 14192,15704,
   14200,14320, 14208,15648, 14216,15776, 14224,15656, 14232,14832,
   14240,15664, 14248,15784, 14256,15576, 14264,15344, 14272,15712,
   14280,15840, 14288,15720, 14296,15856, 14304,15728, 14312,15848,
   14320,15832, 14328,16368, 14392,14488, 14400,14592, 14408,14600,
   14416,14608, 14424,14616, 14432,14624, 14440,14632, 14448,14640,
   14456,15512, 14504,14512, 14520,14904, 14528,14720, 14536,14728,
   14544,14736, 14552,15416, 14560,14752, 14568,14576, 14584,15928,
   14576,14760, 14592,15360, 14600,15368, 14608,15376, 14616,15384,
   14624,15392, 14632,15400, 14640,15408, 14648,15416, 14656,15616,
   14664,15624, 14672,15632, 14680,15640, 14688,15648, 14696,15656,
   14704,15664, 14712,15576, 14720,15488, 14728,15496, 14736,15504,
   14744,15512, 14752,15520, 14760,14768, 14776,14968, 14768,15528,
   14784,15744, 14792,15752, 14800,15760, 14808,15480, 14816,15776,
   14824,14832, 14840,15992, 14832,15784, 14856,14864, 14864,14880,
   14872,14896, 14880,14976, 14888,14992, 14896,15008, 14904,15024,
   14912,15104, 14920,15120, 14928,15136, 14936,15152, 14944,15232,
   14952,15248, 14960,15264, 14968,15280, 14984,15008, 15000,15024,
   15016,15024, 15040,15112, 15048,15128, 15056,15144, 15064,15544,
   15072,15240, 15080,15256, 15088,15272, 15096,16056, 15104,15872,
   15112,15888, 15120,15904, 15128,15920, 15136,16000, 15144,16016,
   15152,16032, 15160,16048, 15168,16128, 15176,16144, 15184,16160,
   15192,16176, 15200,16256, 15208,16272, 15216,16288, 15224,16304,
   15232,15880, 15240,15896, 15248,15912, 15256,15928, 15264,16008,
   15272,16024, 15280,16040, 15288,16056, 15296,16136, 15304,16152,
   15312,16168, 15320,15608, 15328,16264, 15336,16280, 15344,16296,
   15352,16120, 15416,15512, 15424,15616, 15432,15624, 15440,15632,
   15448,15640, 15456,15648, 15464,15656, 15472,15664, 15480,15768,
   15528,15536, 15544,16048, 15552,15744, 15560,15752, 15568,15760,
   15576,15672, 15584,15776, 15592,15600, 15600,15784, 15608,16184,
   15672,15768, 15736,15832, 15784,15792, 15800,16304, 15848,15856,
   15880,16000, 15864,16248, 15888,16000, 15896,16008, 15904,16000,
   15912,16016, 15920,16008, 15928,16024, 15936,16128, 15944,16160,
   15952,16256, 15960,16288, 15968,16136, 15976,16168, 15984,16264,
   15992,16296, 16008,16032, 16024,16040, 16064,16144, 16040,16048,
   16072,16176, 16080,16272, 16088,16304, 16096,16152, 16104,16184,
   16112,16280, 16136,16256, 16120,16312, 16144,16256, 16152,16264,
   16160,16256, 16168,16272, 16176,16264, 16184,16280, 16200,16208,
   16208,16224, 16216,16240, 16224,16320, 16232,16336, 16240,16352,
   16248,16368, 16264,16288, 16280,16296, 16296,16304, 16344,16368,
   16328,16352, 16360,16368
};

const uint16_t armBitRevIndexTable4096[ARMBITREVINDEXTABLE_4096_TABLE_LENGTH] =
{
   /* radix 8, size 4032 */
   8,4096, 16,8192, 24,12288, 32,16384, 40,20480, 48,24576, 56,28672, 64,512,
   72,4608, 80,8704, 88,12800, 96,16896, 104,20992, 112,25088, 120,29184,
   128,1024, 136,5120, 144,9216, 152,13312, 160,17408, 168,21504, 176,25600,
   184,29696, 192,1536, 200,5632, 208,9728, 216,13824, 224,17920, 232,22016,
   240,26112, 248,30208, 256,2048, 264,6144, 272,10240, 280,14336, 288,18432,
   296,22528, 304,26624, 312,30720, 320,2560, 328,6656, 336,10752, 344,14848,
   352,18944, 360,23040, 368,27136, 376,31232, 384,3072, 392,7168, 400,11264,
   408,15360, 416,19456, 424,23552, 432,27648, 440,31744, 448,3584, 456,7680,
   464,11776, 472,15872, 480,19968, 488,24064, 496,28160, 504,32256, 520,4160,
   528,8256, 536,12352, 544,16448, 552,20544, 560,24640, 568,28736, 584,4672,
   592,8768, 600,12864, 608,16960, 616,21056, 624,25152, 632,29248, 640,1088,
   648,5184, 656,9280, 664,13376, 672,17472, 680,21568, 688,25664, 696,29760,
   704,1600, 712,5696, 720,9792, 728,13888, 736,17984, 744,22080, 752,26176,
   760,30272, 768,2112, 776,6208, 784,10304, 792,14400, 800,18496, 808,22592,
   816,26688, 824,30784, 832,2624, 840,6720, 848,10816, 856,14912, 864,19008,
   872,23104, 880,27200, 888,31296, 896,3136, 904,7232, 912,11328, 920,15424,
   928,19520, 936,23616, 944,27712, 952,31808, 960,3648, 968,7744, 976,11840,
   984,15936, 992,20032, 1000,24128, 1008,28224, 1016,32320, 1032,4224,
   1040,8320, 1048,12416, 1056,16512, 1064,20608, 1072,24704, 1080,28800,
   1096,4736, 1104,8832, 1112,12928, 1120,17024, 1128,21120, 1136,25216,
   1144,29312, 1160,5248, 1168,9344, 1176,13440, 1184,17536, 1192,21632,
   1200,25728, 1208,29824, 1216,1664, 1224,5760, 1232,9856, 1240,13952,
   1248,18048, 1256,22144, 1264,26240, 1272,30336, 1280,2176, 1288,6272,
   1296,10368, 1304,14464, 1312,18560, 1320,22656, 1328,26752, 1336,30848,
   1344,2688, 1352,6784, 1360,10880, 1368,14976, 1376,19072, 1384,23168,
   1392,27264, 1400,31360, 1408,3200, 1416,7296, 1424,11392, 1432,15488,
   1440,19584, 1448,23680, 1456,27776, 1464,31872, 1472,3712, 1480,7808,
   1488,11904, 1496,16000, 1504,20096, 1512,24192, 1520,28288, 1528,32384,
   1544,4288, 1552,8384, 1560,12480, 1568,16576, 1576,20672, 1584,24768,
   1592,28864, 1608,4800, 1616,8896, 1624,12992, 1632,17088, 1640,21184,
   1648,25280, 1656,29376, 1672,5312, 1680,9408, 1688,13504, 1696,17600,
   1704,21696, 1712,25792, 1720,29888, 1736,5824, 1744,9920, 1752,14016,
   1760,18112, 1768,22208, 1776,26304, 1784,30400, 1792,2240, 1800,6336,
   1808,10432, 1816,14528, 1824,18624, 1832,22720, 1840,26816, 1848,30912,
   1856,2752, 1864,6848, 1872,10944, 1880,15040, 1888,19136, 1896,23232,
   1904,27328, 1912,31424, 1920,3264, 1928,7360, 1936,11456, 1944,15552,
   1952,19648, 1960,23744, 1968,27840, 1976,31936, 1984,3776, 1992,7872,
   2000,11968, 2008,16064, 2016,20160, 2024,24256, 2032,28352, 2040,32448,
   2056,4352, 2064,8448, 2072,12544, 2080,16640, 2088,20736, 2096,24832,
   2104,28928, 2120,4864, 2128,8960, 2136,13056, 2144,17152, 2152,21248,
   2160,25344, 2168,29440, 2184,5376, 2192,9472, 2200,13568, 2208,17664,
   2216,21760, 2224,25856, 2232,29952, 2248,5888, 2256,9984, 2264,14080,
   2272,18176, 2280,22272, 2288,26368, 2296,30464, 2312,6400, 2320,10496,
   2328,14592, 2336,18688, 2344,22784, 2352,26880, 2360,30976, 2368,2816,
   2376,6912, 2384,11008, 2392,15104, 2400,19200, 2408,23296, 2416,27392,
   2424,31488, 2432,3328, 2440,7424, 2448,11520, 2456,15616, 2464,19712,
   2472,23808, 2480,27904, 2488,32000, 2496,3840, 2504,7936, 2512,12032,
   2520,16128, 2528,20224, 2536,24320, 2544,28416, 2552,32512, 2568,4416,
   2576,8512, 2584,12608, 2592,16704, 2600,20800, 2608,24896, 2616,28992,
   2632,4928, 2640,9024, 2648,13120, 2656,17216, 2664,21312, 2672,25408,
   2680,29504, 2696,5440, 2704,9536, 2712,13632, 2720,17728, 2728,21824,
   2736,25920, 2744,30016, 2760,5952, 2768,10048, 2776,14144, 2784,18240,
   2792,22336, 2800,26432, 2808,30528, 2824,6464, 2832,10560, 2840,14656,
   2848,18752, 2856,22848, 2864,26944, 2872,31040, 2888,6976, 2896,11072,
   2904,15168, 2912,19264, 2920,23360, 2928,27456, 2936,31552, 2944,3392,
   2952,7488, 2960,11584, 2968,15680, 2976,19776, 2984,23872, 2992,27968,
   3000,32064, 3008,3904, 3016,8000, 3024,12096, 3032,16192, 3040,20288,
   3048,24384, 3056,28480, 3064,32576, 3080,4480, 3088,8576, 3096,12672,
   3104,16768, 3112,20864, 3120,24960, 3128,29056, 3144,4992, 3152,9088,
   3160,13184, 3168,17280, 3176,21376, 3184,25472, 3192,29568, 3208,5504,
   3216,9600, 3224,13696, 3232,17792, 3240,21888, 3248,25984, 3256,30080,
   3272,6016, 3280,10112, 3288,14208, 3296,18304, 3304,22400, 3312,26496,
   3320,30592, 3336,6528, 3344,10624, 3352,14720, 3360,18816, 3368,22912,
   3376,27008, 3384,31104, 3400,7040, 3408,11136, 3416,15232, 3424,19328,
   3432,23424, 3440,27520, 3448,31616, 3464,7552, 3472,11648, 3480,15744,
   3488,19840, 3496,23936, 3504,28032, 3512,32128, 3520,3968, 3528,8064,
   3536,12160, 3544,16256, 3552,20352, 3560,24448, 3568,28544, 3576,32640,
   3592,4544, 3600,8640, 3608,12736, 3616,16832, 3624,20928, 3632,25024,
   3640,29120, 3656,5056, 3664,9152, 3672,13248, 3680,17344, 3688,21440,
   3696,25536, 3704,29632, 3720,5568, 3728,9664, 3736,13760, 3744,17856,
   3752,21952, 3760,26048, 3768,30144, 3784,6080, 3792,10176, 3800,14272,
   3808,18368, 3816,22464, 3824,26560, 3832,30656, 3848,6592, 3856,10688,
   3864,14784, 3872,18880, 3880,22976, 3888,27072, 3896,31168, 3912,7104,
   3920,11200, 3928,15296, 3936,19392, 3944,23488, 3952,27584, 3960,31680,
   3976,7616, 3984,11712, 3992,15808, 4000,19904, 4008,24000, 4016,28096,
   4024,32192, 4040,8128, 4048,12224, 4056,16320, 4064,20416, 4072,24512,
   4080,28608, 4088,32704, 4112,8200, 4120,12296, 4128,16392, 4136,20488,
   4144,24584, 4152,28680, 4168,4616, 4176,8712, 4184,12808, 4192,16904,
   4200,21000, 4208,25096, 4216,29192, 4232,5128, 4240,9224, 4248,13320,
   4256,17416, 4264,21512, 4272,25608, 4280,29704, 4296,5640, 4304,9736,
   4312,13832, 4320,17928, 4328,22024, 4336,26120, 4344,30216, 4360,6152,
   4368,10248, 4376,14344, 4384,18440, 4392,22536, 4400,26632, 4408,30728,
   4424,6664, 4432,10760, 4440,14856, 4448,18952, 4456,23048, 4464,27144,
   4472,31240, 4488,7176, 4496,11272, 4504,15368, 4512,19464, 4520,23560,
   4528,27656, 4536,31752, 4552,7688, 4560,11784, 4568,15880, 4576,19976,
   4584,24072, 4592,28168, 4600,32264, 4624,8264, 4632,12360, 4640,16456,
   4648,20552, 4656,24648, 4664,28744, 4688,8776, 4696,12872, 4704,16968,
   4712,21064, 4720,25160, 4728,29256, 4744,5192, 4752,9288, 4760,13384,
   4768,17480, 4776,21576, 4784,25672, 4792,29768, 4808,5704, 4816,9800,
   4824,13896, 4832,17992, 4840,22088, 4848,26184, 4856,30280, 4872,6216,
   4880,10312, 4888,14408, 4896,18504, 4904,22600, 4912,26696, 4920,30792,
   4936,6728, 4944,10824, 4952,14920, 4960,19016, 4968,23112, 4976,27208,
   4984,31304, 5000,7240, 5008,11336, 5016,15432, 5024,19528, 5032,23624,
   5040,27720, 5048,31816, 5064,7752, 5072,11848, 5080,15944, 5088,20040,
   5096,24136, 5104,28232, 5112,32328, 5136,8328, 5144,12424, 5152,16520,
   5160,20616, 5168,24712, 5176,28808, 5200,8840, 5208,12936, 5216,17032,
   5224,21128, 5232,25224, 5240,29320, 5264,9352, 5272,13448, 5280,17544,
   5288,21640, 5296,25736, 5304,29832, 5320,5768, 5328,9864, 5336,13960,
   5344,18056, 5352,22152, 5360,26248, 5368,30344, 5384,6280, 5392,10376,
   5400,14472, 5408,18568, 5416,22664, 5424,26760, 5432,30856, 5448,6792,
   5456,10888, 5464,14984, 5472,19080, 5480,23176, 5488,27272, 5496,31368,
   5512,7304, 5520,11400, 5528,15496, 5536,19592, 5544,23688, 5552,27784,
   5560,31880, 5576,7816, 5584,11912, 5592,16008, 5600,20104, 5608,24200,
   5616,28296, 5624,32392, 5648,8392, 5656,12488, 5664,16584, 5672,20680,
   5680,24776, 5688,28872, 5712,8904, 5720,13000, 5728,17096, 5736,21192,
   5744,25288, 5752,29384, 5776,9416, 5784,13512, 5792,17608, 5800,21704,
   5808,25800, 5816,29896, 5840,9928, 5848,14024, 5856,18120, 5864,22216,
   5872,26312, 5880,30408, 5896,6344, 5904,10440, 5912,14536, 5920,18632,
   5928,22728, 5936,26824, 5944,30920, 5960,6856, 5968,10952, 5976,15048,
   5984,19144, 5992,23240, 6000,27336, 6008,31432, 6024,7368, 6032,11464,
   6040,15560, 6048,19656, 6056,23752, 6064,27848, 6072,31944, 6088,7880,
   6096,11976, 6104,16072, 6112,20168, 6120,24264, 6128,28360, 6136,32456,
   6160,8456, 6168,12552, 6176,16648, 6184,20744, 6192,24840, 6200,28936,
   6224,8968, 6232,13064, 6240,17160, 6248,21256, 6256,25352, 6264,29448,
   6288,9480, 6296,13576, 6304,17672, 6312,21768, 6320,25864, 6328,29960,
   6352,9992, 6360,14088, 6368,18184, 6376,22280, 6384,26376, 6392,30472,
   6416,10504, 6424,14600, 6432,18696, 6440,22792, 6448,26888, 6456,30984,
   6472,6920, 6480,11016, 6488,15112, 6496,19208, 6504,23304, 6512,27400,
   6520,31496, 6536,7432, 6544,11528, 6552,15624, 6560,19720, 6568,23816,
   6576,27912, 6584,32008, 6600,7944, 6608,12040, 6616,16136, 6624,20232,
   6632,24328, 6640,28424, 6648,32520, 6672,8520, 6680,12616, 6688,16712,
   6696,20808, 6704,24904, 6712,29000, 6736,9032, 6744,13128, 6752,17224,
   6760,21320, 6768,25416, 6776,29512, 6800,9544, 6808,13640, 6816,17736,
   6824,21832, 6832,25928, 6840,30024, 6864,10056, 6872,14152, 6880,18248,
   6888,22344, 6896,26440, 6904,30536, 6928,10568, 6936,14664, 6944,18760,
   6952,22856, 6960,26952, 6968,31048, 6992,11080, 7000,15176, 7008,19272,
   7016,23368, 7024,27464, 7032,31560, 7048,7496, 7056,11592, 7064,15688,
   7072,19784, 7080,23880, 7088,27976, 7096,32072, 7112,8008, 7120,12104,
   7128,16200, 7136,20296, 7144,24392, 7152,28488, 7160,32584, 7184,8584,
   7192,12680, 7200,16776, 7208,20872, 7216,24968, 7224,29064, 7248,9096,
   7256,13192, 7264,17288, 7272,21384, 7280,25480, 7288,29576, 7312,9608,
   7320,13704, 7328,17800, 7336,21896, 7344,25992, 7352,30088, 7376,10120,
   7384,14216, 7392,18312, 7400,22408, 7408,26504, 7416,30600, 7440,10632,
   7448,14728, 7456,18824, 7464,22920, 7472,27016, 7480,31112, 7504,11144,
   7512,15240, 7520,19336, 7528,23432, 7536,27528, 7544,31624, 7568,11656,
   7576,15752, 7584,19848, 7592,23944, 7600,28040, 7608,32136, 7624,8072,
   7632,12168, 7640,16264, 7648,20360, 7656,24456, 7664,28552, 7672,32648,
   7696,8648, 7704,12744, 7712,16840, 7720,20936, 7728,25032, 7736,29128,
   7760,9160, 7768,13256, 7776,17352, 7784,21448, 7792,25544, 7800,29640,
   7824,9672, 7832,13768, 7840,17864, 7848,21960, 7856,26056, 7864,30152,
   7888,10184, 7896,14280, 7904,18376, 7912,22472, 7920,26568, 7928,30664,
   7952,10696, 7960,14792, 7968,18888, 7976,22984, 7984,27080, 7992,31176,
   8016,11208, 8024,15304, 8032,19400, 8040,23496, 8048,27592, 8056,31688,
   8080,11720, 8088,15816, 8096,19912, 8104,24008, 8112,28104, 8120,32200,
   8144,12232, 8152,16328, 8160,20424, 8168,24520, 8176,28616, 8184,32712,
   8216,12304, 8224,16400, 8232,20496, 8240,24592, 8248,28688, 8272,8720,
   8280,12816, 8288,16912, 8296,21008, 8304,25104, 8312,29200, 8336,9232,
   8344,13328, 8352,17424, 8360,21520, 8368,25616, 8376,29712, 8400,9744,
   8408,13840, 8416,17936, 8424,22032, 8432,26128, 8440,30224, 8464,10256,
   8472,14352, 8480,18448, 8488,22544, 8496,26640, 8504,30736, 8528,10768,
   8536,14864, 8544,18960, 8552,23056, 8560,27152, 8568,31248, 8592,11280,
   8600,15376, 8608,19472, 8616,23568, 8624,27664, 8632,31760, 8656,11792,
   8664,15888, 8672,19984, 8680,24080, 8688,28176, 8696,32272, 8728,12368,
   8736,16464, 8744,20560, 8752,24656, 8760,28752, 8792,12880, 8800,16976,
   8808,21072, 8816,25168, 8824,29264, 8848,9296, 8856,13392, 8864,17488,
   8872,21584, 8880,25680, 8888,29776, 8912,9808, 8920,13904, 8928,18000,
   8936,22096, 8944,26192, 8952,30288, 8976,10320, 8984,14416, 8992,18512,
   9000,22608, 9008,26704, 9016,30800, 9040,10832, 9048,14928, 9056,19024,
   9064,23120, 9072,27216, 9080,31312, 9104,11344, 9112,15440, 9120,19536,
   9128,23632, 9136,27728, 9144,31824, 9168,11856, 9176,15952, 9184,20048,
   9192,24144, 9200,28240, 9208,32336, 9240,12432, 9248,16528, 9256,20624,
   9264,24720, 9272,28816, 9304,12944, 9312,17040, 9320,21136, 9328,25232,
   9336,29328, 9368,13456, 9376,17552, 9384,21648, 9392,25744, 9400,29840,
   9424,9872, 9432,13968, 9440,18064, 9448,22160, 9456,26256, 9464,30352,
   9488,10384, 9496,14480, 9504,18576, 9512,22672, 9520,26768, 9528,30864,
   9552,10896, 9560,14992, 9568,19088, 9576,23184, 9584,27280, 9592,31376,
   9616,11408, 9624,15504, 9632,19600, 9640,23696, 9648,27792, 9656,31888,
   9680,11920, 9688,16016, 9696,20112, 9704,24208, 9712,28304, 9720,32400,
   9752,12496, 9760,16592, 9768,20688, 9776,24784, 9784,28880, 9816,13008,
   9824,17104, 9832,21200, 9840,25296, 9848,29392, 9880,13520, 9888,17616,
   9896,21712, 9904,25808, 9912,29904, 9944,14032, 9952,18128, 9960,22224,
   9968,26320, 9976,30416, 10000,10448, 10008,14544, 10016,18640, 10024,22736,
   10032,26832, 10040,30928, 10064,10960, 10072,15056, 10080,19152,
   10088,23248, 10096,27344, 10104,31440, 10128,11472, 10136,15568,
   10144,19664, 10152,23760, 10160,27856, 10168,31952, 10192,11984,
   10200,16080, 10208,20176, 10216,24272, 10224,28368, 10232,32464,
   10264,12560, 10272,16656, 10280,20752, 10288,24848, 10296,28944,
   10328,13072, 10336,17168, 10344,21264, 10352,25360, 10360,29456,
   10392,13584, 10400,17680, 10408,21776, 10416,25872, 10424,29968,
   10456,14096, 10464,18192, 10472,22288, 10480,26384, 10488,30480,
   10520,14608, 10528,18704, 10536,22800, 10544,26896, 10552,30992,
   10576,11024, 10584,15120, 10592,19216, 10600,23312, 10608,27408,
   10616,31504, 10640,11536, 10648,15632, 10656,19728, 10664,23824,
   10672,27920, 10680,32016, 10704,12048, 10712,16144, 10720,20240,
   10728,24336, 10736,28432, 10744,32528, 10776,12624, 10784,16720,
   10792,20816, 10800,24912, 10808,29008, 10840,13136, 10848,17232,
   10856,21328, 10864,25424, 10872,29520, 10904,13648, 10912,17744,
   10920,21840, 10928,25936, 10936,30032, 10968,14160, 10976,18256,
   10984,22352, 10992,26448, 11000,30544, 11032,14672, 11040,18768,
   11048,22864, 11056,26960, 11064,31056, 11096,15184, 11104,19280,
   11112,23376, 11120,27472, 11128,31568, 11152,11600, 11160,15696,
   11168,19792, 11176,23888, 11184,27984, 11192,32080, 11216,12112,
   11224,16208, 11232,20304, 11240,24400, 11248,28496, 11256,32592,
   11288,12688, 11296,16784, 11304,20880, 11312,24976, 11320,29072,
   11352,13200, 11360,17296, 11368,21392, 11376,25488, 11384,29584,
   11416,13712, 11424,17808, 11432,21904, 11440,26000, 11448,30096,
   11480,14224, 11488,18320, 11496,22416, 11504,26512, 11512,30608,
   11544,14736, 11552,18832, 11560,22928, 11568,27024, 11576,31120,
   11608,15248, 11616,19344, 11624,23440, 11632,27536, 11640,31632,
   11672,15760, 11680,19856, 11688,23952, 11696,28048, 11704,32144,
   11728,12176, 11736,16272, 11744,20368, 11752,24464, 11760,28560,
   11768,32656, 11800,12752, 11808,16848, 11816,20944, 11824,25040,
   11832,29136, 11864,13264, 11872,17360, 11880,21456, 11888,25552,
   11896,29648, 11928,13776, 11936,17872, 11944,21968, 11952,26064,
   11960,30160, 11992,14288, 12000,18384, 12008,22480, 12016,26576,
   12024,30672, 12056,14800, 12064,18896, 12072,22992, 12080,27088,
   12088,31184, 12120,15312, 12128,19408, 12136,23504, 12144,27600,
   12152,31696, 12184,15824, 12192,19920, 12200,24016, 12208,28112,
   12216,32208, 12248,16336, 12256,20432, 12264,24528, 12272,28624,
   12280,32720, 12320,16408, 12328,20504, 12336,24600, 12344,28696,
   12376,12824, 12384,16920, 12392,21016, 12400,25112, 12408,29208,
   12440,13336, 12448,17432, 12456,21528, 12464,25624, 12472,29720,
   12504,13848, 12512,17944, 12520,22040, 12528,26136, 12536,30232,
   12568,14360, 12576,18456, 12584,22552, 12592,26648, 12600,30744,
   12632,14872, 12640,18968, 12648,23064, 12656,27160, 12664,31256,
   12696,15384, 12704,19480, 12712,23576, 12720,27672, 12728,31768,
   12760,15896, 12768,19992, 12776,24088, 12784,28184, 12792,32280,
   12832,16472, 12840,20568, 12848,24664, 12856,28760, 12896,16984,
   12904,21080, 12912,25176, 12920,29272, 12952,13400, 12960,17496,
   12968,21592, 12976,25688, 12984,29784, 13016,13912, 13024,18008,
   13032,22104, 13040,26200, 13048,30296, 13080,14424, 13088,18520,
   13096,22616, 13104,26712, 13112,30808, 13144,14936, 13152,19032,
   13160,23128, 13168,27224, 13176,31320, 13208,15448, 13216,19544,
   13224,23640, 13232,27736, 13240,31832, 13272,15960, 13280,20056,
   13288,24152, 13296,28248, 13304,32344, 13344,16536, 13352,20632,
   13360,24728, 13368,28824, 13408,17048, 13416,21144, 13424,25240,
   13432,29336, 13472,17560, 13480,21656, 13488,25752, 13496,29848,
   13528,13976, 13536,18072, 13544,22168, 13552,26264, 13560,30360,
   13592,14488, 13600,18584, 13608,22680, 13616,26776, 13624,30872,
   13656,15000, 13664,19096, 13672,23192, 13680,27288, 13688,31384,
   13720,15512, 13728,19608, 13736,23704, 13744,27800, 13752,31896,
   13784,16024, 13792,20120, 13800,24216, 13808,28312, 13816,32408,
   13856,16600, 13864,20696, 13872,24792, 13880,28888, 13920,17112,
   13928,21208, 13936,25304, 13944,29400, 13984,17624, 13992,21720,
   14000,25816, 14008,29912, 14048,18136, 14056,22232, 14064,26328,
   14072,30424, 14104,14552, 14112,18648, 14120,22744, 14128,26840,
   14136,30936, 14168,15064, 14176,19160, 14184,23256, 14192,27352,
   14200,31448, 14232,15576, 14240,19672, 14248,23768, 14256,27864,
   14264,31960, 14296,16088, 14304,20184, 14312,24280, 14320,28376,
   14328,32472, 14368,16664, 14376,20760, 14384,24856, 14392,28952,
   14432,17176, 14440,21272, 14448,25368, 14456,29464, 14496,17688,
   14504,21784, 14512,25880, 14520,29976, 14560,18200, 14568,22296,
   14576,26392, 14584,30488, 14624,18712, 14632,22808, 14640,26904,
   14648,31000, 14680,15128, 14688,19224, 14696,23320, 14704,27416,
   14712,31512, 14744,15640, 14752,19736, 14760,23832, 14768,27928,
   14776,32024, 14808,16152, 14816,20248, 14824,24344, 14832,28440,
   14840,32536, 14880,16728, 14888,20824, 14896,24920, 14904,29016,
   14944,17240, 14952,21336, 14960,25432, 14968,29528, 15008,17752,
   15016,21848, 15024,25944, 15032,30040, 15072,18264, 15080,22360,
   15088,26456, 15096,30552, 15136,18776, 15144,22872, 15152,26968,
   15160,31064, 15200,19288, 15208,23384, 15216,27480, 15224,31576,
   15256,15704, 15264,19800, 15272,23896, 15280,27992, 15288,32088,
   15320,16216, 15328,20312, 15336,24408, 15344,28504, 15352,32600,
   15392,16792, 15400,20888, 15408,24984, 15416,29080, 15456,17304,
   15464,21400, 15472,25496, 15480,29592, 15520,17816, 15528,21912,
   15536,26008, 15544,30104, 15584,18328, 15592,22424, 15600,26520,
   15608,30616, 15648,18840, 15656,22936, 15664,27032, 15672,31128,
   15712,19352, 15720,23448, 15728,27544, 15736,31640, 15776,19864,
   15784,23960, 15792,28056, 15800,32152, 15832,16280, 15840,20376,
   15848,24472, 15856,28568, 15864,32664, 15904,16856, 15912,20952,
   15920,25048, 15928,29144, 15968,17368, 15976,21464, 15984,25560,
   15992,29656, 16032,17880, 16040,21976, 16048,26072, 16056,30168,
   16096,18392, 16104,22488, 16112,26584, 16120,30680, 16160,18904,
   16168,23000, 16176,27096, 16184,31192, 16224,19416, 16232,23512,
   16240,27608, 16248,31704, 16288,19928, 16296,24024, 16304,28120,
   16312,32216, 16352,20440, 16360,24536, 16368,28632, 16376,32728,
   16424,20512, 16432,24608, 16440,28704, 16480,16928, 16488,21024,
   16496,25120, 16504,29216, 16544,17440, 16552,21536, 16560,25632,
   16568,29728, 16608,17952, 16616,22048, 16624,26144, 16632,30240,
   16672,18464, 16680,22560, 16688,26656, 16696,30752, 16736,18976,
   16744,23072, 16752,27168, 16760,31264, 16800,19488, 16808,23584,
   16816,27680, 16824,31776, 16864,20000, 16872,24096, 16880,28192,
   16888,32288, 16936,20576, 16944,24672, 16952,28768, 17000,21088,
   17008,25184, 17016,29280, 17056,17504, 17064,21600, 17072,25696,
   17080,29792, 17120,18016, 17128,22112, 17136,26208, 17144,30304,
   17184,18528, 17192,22624, 17200,26720, 17208,30816, 17248,19040,
   17256,23136, 17264,27232, 17272,31328, 17312,19552, 17320,23648,
   17328,27744, 17336,31840, 17376,20064, 17384,24160, 17392,28256,
   17400,32352, 17448,20640, 17456,24736, 17464,28832, 17512,21152,
   17520,25248, 17528,29344, 17576,21664, 17584,25760, 17592,29856,
   17632,18080, 17640,22176, 17648,26272, 17656,30368, 17696,18592,
   17704,22688, 17712,26784, 17720,30880, 17760,19104, 17768,23200,
   17776,27296, 17784,31392, 17824,19616, 17832,23712, 17840,27808,
   17848,31904, 17888,20128, 17896,24224, 17904,28320, 17912,32416,
   17960,20704, 17968,24800, 17976,28896, 18024,21216, 18032,25312,
   18040,29408, 18088,21728, 18096,25824, 18104,29920, 18152,22240,
   18160,26336, 18168,30432, 18208,18656, 18216,22752, 18224,26848,
   18232,30944, 18272,19168, 18280,23264, 18288,27360, 18296,31456,
   18336,19680, 18344,23776, 18352,27872, 18360,31968, 18400,20192,
   18408,24288, 18416,28384, 18424,32480, 18472,20768, 18480,24864,
   18488,28960, 18536,21280, 18544,25376, 18552,29472, 18600,21792,
   18608,25888, 18616,29984, 18664,22304, 18672,26400, 18680,30496,
   18728,22816, 18736,26912, 18744,31008, 18784,19232, 18792,23328,
   18800,27424, 18808,31520, 18848,19744, 18856,23840, 18864,27936,
   18872,32032, 18912,20256, 18920,24352, 18928,28448, 18936,32544,
   18984,20832, 18992,24928, 19000,29024, 19048,21344, 19056,25440,
   19064,29536, 19112,21856, 19120,25952, 19128,30048, 19176,22368,
   19184,26464, 19192,30560, 19240,22880, 19248,26976, 19256,31072,
   19304,23392, 19312,27488, 19320,31584, 19360,19808, 19368,23904,
   19376,28000, 19384,32096, 19424,20320, 19432,24416, 19440,28512,
   19448,32608, 19496,20896, 19504,24992, 19512,29088, 19560,21408,
   19568,25504, 19576,29600, 19624,21920, 19632,26016, 19640,30112,
   19688,22432, 19696,26528, 19704,30624, 19752,22944, 19760,27040,
   19768,31136, 19816,23456, 19824,27552, 19832,31648, 19880,23968,
   19888,28064, 19896,32160, 19936,20384, 19944,24480, 19952,28576,
   19960,32672, 20008,20960, 20016,25056, 20024,29152, 20072,21472,
   20080,25568, 20088,29664, 20136,21984, 20144,26080, 20152,30176,
   20200,22496, 20208,26592, 20216,30688, 20264,23008, 20272,27104,
   20280,31200, 20328,23520, 20336,27616, 20344,31712, 20392,24032,
   20400,28128, 20408,32224, 20456,24544, 20464,28640, 20472,32736,
   20528,24616, 20536,28712, 20584,21032, 20592,25128, 20600,29224,
   20648,21544, 20656,25640, 20664,29736, 20712,22056, 20720,26152,
   20728,30248, 20776,22568, 20784,26664, 20792,30760, 20840,23080,
   20848,27176, 20856,31272, 20904,23592, 20912,27688, 20920,31784,
   20968,24104, 20976,28200, 20984,32296, 21040,24680, 21048,28776,
   21104,25192, 21112,29288, 21160,21608, 21168,25704, 21176,29800,
   21224,22120, 21232,26216, 21240,30312, 21288,22632, 21296,26728,
   21304,30824, 21352,23144, 21360,27240, 21368,31336, 21416,23656,
   21424,27752, 21432,31848, 21480,24168, 21488,28264, 21496,32360,
   21552,24744, 21560,28840, 21616,25256, 21624,29352, 21680,25768,
   21688,29864, 21736,22184, 21744,26280, 21752,30376, 21800,22696,
   21808,26792, 21816,30888, 21864,23208, 21872,27304, 21880,31400,
   21928,23720, 21936,27816, 21944,31912, 21992,24232, 22000,28328,
   22008,32424, 22064,24808, 22072,28904, 22128,25320, 22136,29416,
   22192,25832, 22200,29928, 22256,26344, 22264,30440, 22312,22760,
   22320,26856, 22328,30952, 22376,23272, 22384,27368, 22392,31464,
   22440,23784, 22448,27880, 22456,31976, 22504,24296, 22512,28392,
   22520,32488, 22576,24872, 22584,28968, 22640,25384, 22648,29480,
   22704,25896, 22712,29992, 22768,26408, 22776,30504, 22832,26920,
   22840,31016, 22888,23336, 22896,27432, 22904,31528, 22952,23848,
   22960,27944, 22968,32040, 23016,24360, 23024,28456, 23032,32552,
   23088,24936, 23096,29032, 23152,25448, 23160,29544, 23216,25960,
   23224,30056, 23280,26472, 23288,30568, 23344,26984, 23352,31080,
   23408,27496, 23416,31592, 23464,23912, 23472,28008, 23480,32104,
   23528,24424, 23536,28520, 23544,32616, 23600,25000, 23608,29096,
   23664,25512, 23672,29608, 23728,26024, 23736,30120, 23792,26536,
   23800,30632, 23856,27048, 23864,31144, 23920,27560, 23928,31656,
   23984,28072, 23992,32168, 24040,24488, 24048,28584, 24056,32680,
   24112,25064, 24120,29160, 24176,25576, 24184,29672, 24240,26088,
   24248,30184, 24304,26600, 24312,30696, 24368,27112, 24376,31208,
   24432,27624, 24440,31720, 24496,28136, 24504,32232, 24560,28648,
   24568,32744, 24632,28720, 24688,25136, 24696,29232, 24752,25648,
   24760,29744, 24816,26160, 24824,30256, 24880,26672, 24888,30768,
   24944,27184, 24952,31280, 25008,27696, 25016,31792, 25072,28208,
   25080,32304, 25144,28784, 25208,29296, 25264,25712, 25272,29808,
   25328,26224, 25336,30320, 25392,26736, 25400,30832, 25456,27248,
   25464,31344, 25520,27760, 25528,31856, 25584,28272, 25592,32368,
   25656,28848, 25720,29360, 25784,29872, 25840,26288, 25848,30384,
   25904,26800, 25912,30896, 25968,27312, 25976,31408, 26032,27824,
   26040,31920, 26096,28336, 26104,32432, 26168,28912, 26232,29424,
   26296,29936, 26360,30448, 26416,26864, 26424,30960, 26480,27376,
   26488,31472, 26544,27888, 26552,31984, 26608,28400, 26616,32496,
   26680,28976, 26744,29488, 26808,30000, 26872,30512, 26936,31024,
   26992,27440, 27000,31536, 27056,27952, 27064,32048, 27120,28464,
   27128,32560, 27192,29040, 27256,29552, 27320,30064, 27384,30576,
   27448,31088, 27512,31600, 27568,28016, 27576,32112, 27632,28528,
   27640,32624, 27704,29104, 27768,29616, 27832,30128, 27896,30640,
   27960,31152, 28024,31664, 28088,32176, 28144,28592, 28152,32688,
   28216,29168, 28280,29680, 28344,30192, 28408,30704, 28472,31216,
   28536,31728, 28600,32240, 28664,32752, 28792,29240, 28856,29752,
   28920,30264, 28984,30776, 29048,31288, 29112,31800, 29176,32312,
   29368,29816, 29432,30328, 29496,30840, 29560,31352, 29624,31864,
   29688,32376, 29944,30392, 30008,30904, 30072,31416, 30136,31928,
   30200,32440, 30520,30968, 30584,31480, 30648,31992, 30712,32504,
   31096,31544, 31160,32056, 31224,32568, 31672,32120, 31736,32632,
   32248,32696
};


const uint16_t armBitRevIndexTable_fixed_16[ARMBITREVINDEXTABLE_FIXED_16_TABLE_LENGTH] =
{
   /* radix 4, size 12 */
   8,64, 16,32, 24,96, 40,80, 56,112, 88,104
};

const uint16_t armBitRevIndexTable_fixed_32[ARMBITREVINDEXTABLE_FIXED_32_TABLE_LENGTH] =
{
   /* 4x2, size 24 */
   8,128, 16,64, 24,192, 40,160, 48,96, 56,224, 72,144,
   88,208, 104,176, 120,240, 152,200, 184,232
};

const uint16_t armBitRevIndexTable_fixed_64[ARMBITREVINDEXTABLE_FIXED_64_TABLE_LENGTH] =
{
   /* radix 4, size 56 */
   8,256, 16,128, 24,384, 32,64, 40,320, 48,192, 56,448, 72,288, 80,160, 88,416, 104,352,
   112,224, 120,480, 136,272, 152,400, 168,336, 176,208, 184,464, 200,304, 216,432,
   232,368, 248,496, 280,392, 296,328, 312,456, 344,424, 376,488, 440,472
};

const uint16_t armBitRevIndexTable_fixed_128[ARMBITREVINDEXTABLE_FIXED_128_TABLE_LENGTH] =
{
   /* 4x2, size 112 */
   8,512, 16,256, 24,768, 32,128, 40,640, 48,384, 56,896, 72,576, 80,320, 88,832, 96,192,
   104,704, 112,448, 120,960, 136,544, 144,288, 152,800, 168,672, 176,416, 184,928, 200,608,
   208,352, 216,864, 232,736, 240,480, 248,992, 264,528, 280,784, 296,656, 304,400, 312,912,
   328,592, 344,848, 360,720, 368,464, 376,976, 392,560, 408,816, 424,688, 440,944, 456,624,
   472,880, 488,752, 504,1008, 536,776, 552,648, 568,904, 600,840, 616,712, 632,968,
   664,808, 696,936, 728,872, 760,1000, 824,920, 888,984
};

const uint16_t armBitRevIndexTable_fixed_256[ARMBITREVINDEXTABLE_FIXED_256_TABLE_LENGTH] =
{
   /* radix 4, size 240 */
   8,1024, 16,512, 24,1536, 32,256, 40,1280, 48,768, 56,1792, 64,128, 72,1152, 80,640,
   88,1664, 96,384, 104,1408, 112,896, 120,1920, 136,1088, 144,576, 152,1600, 160,320,
   168,1344, 176,832, 184,1856, 200,1216, 208,704, 216,1728, 224,448, 232,1472, 240,960,
   248,1984, 264,1056, 272,544, 280,1568, 296,1312, 304,800, 312,1824, 328,1184, 336,672,
   344,1696, 352,416, 360,1440, 368,928, 376,1952, 392,1120, 400,608, 408,1632, 424,1376,
   432,864, 440,1888, 456,1248, 464,736, 472,1760, 488,1504, 496,992, 504,2016, 520,1040,
   536,1552, 552,1296, 560,784, 568,1808, 584,1168, 592,656, 600,1680, 616,1424, 624,912,
   632,1936, 648,1104, 664,1616, 680,1360, 688,848, 696,1872, 712,1232, 728,1744, 744,1488,
   752,976, 760,2000, 776,1072, 792,1584, 808,1328, 824,1840, 840,1200, 856,1712, 872,1456,
   880,944, 888,1968, 904,1136, 920,1648, 936,1392, 952,1904, 968,1264, 984,1776, 1000,1520,
   1016,2032, 1048,1544, 1064,1288, 1080,1800, 1096,1160, 1112,1672, 1128,1416, 1144,1928,
   1176,1608, 1192,1352, 1208,1864, 1240,1736, 1256,1480, 1272,1992, 1304,1576, 1336,1832,
   1368,1704, 1384,1448, 1400,1960, 1432,1640, 1464,1896, 1496,1768, 1528,2024, 1592,1816,
   1624,1688, 1656,1944, 1720,1880, 1784,2008, 1912,1976
};

const uint16_t armBitRevIndexTable_fixed_512[ARMBITREVINDEXTABLE_FIXED_512_TABLE_LENGTH] =
{
   /* 4x2, size 480 */
   8,2048, 16,1024, 24,3072, 32,512, 40,2560, 48,1536, 56,3584, 64,256, 72,2304, 80,1280,
   88,3328, 96,768, 104,2816, 112,1792, 120,3840, 136,2176, 144,1152, 152,3200, 160,640,
   168,2688, 176,1664, 184,3712, 192,384, 200,2432, 208,1408, 216,3456, 224,896, 232,2944,
   240,1920, 248,3968, 264,2112, 272,1088, 280,3136, 288,576, 296,2624, 304,1600, 312,3648,
   328,2368, 336,1344, 344,3392, 352,832, 360,2880, 368,1856, 376,3904, 392,2240, 400,1216,
   408,3264, 416,704, 424,2752, 432,1728, 440,3776, 456,2496, 464,1472, 472,3520, 480,960,
   488,3008, 496,1984, 504,4032, 520,2080, 528,1056, 536,3104, 552,2592, 560,1568, 568,3616,
   584,2336, 592,1312, 600,3360, 608,800, 616,2848, 624,1824, 632,3872, 648,2208, 656,1184,
   664,3232, 680,2720, 688,1696, 696,3744, 712,2464, 720,1440, 728,3488, 736,928, 744,2976,
   752,1952, 760,4000, 776,2144, 784,1120, 792,3168, 808,2656, 816,1632, 824,3680, 840,2400,
   848,1376, 856,3424, 872,2912, 880,1888, 888,3936, 904,2272, 912,1248, 920,3296, 936,2784,
   944,1760, 952,3808, 968,2528, 976,1504, 984,3552, 1000,3040, 1008,2016, 1016,4064,
   1032,2064, 1048,3088, 1064,2576, 1072,1552, 1080,3600, 1096,2320, 1104,1296, 1112,3344,
   1128,2832, 1136,1808, 1144,3856, 1160,2192, 1176,3216, 1192,2704, 1200,1680, 1208,3728,
   1224,2448, 1232,1424, 1240,3472, 1256,2960, 1264,1936, 1272,3984, 1288,2128, 1304,3152,
   1320,2640, 1328,1616, 1336,3664, 1352,2384, 1368,3408, 1384,2896, 1392,1872, 1400,3920,
   1416,2256, 1432,3280, 1448,2768, 1456,1744, 1464,3792, 1480,2512, 1496,3536, 1512,3024,
   1520,2000, 1528,4048, 1544,2096, 1560,3120, 1576,2608, 1592,3632, 1608,2352, 1624,3376,
   1640,2864, 1648,1840, 1656,3888, 1672,2224, 1688,3248, 1704,2736, 1720,3760, 1736,2480,
   1752,3504, 1768,2992, 1776,1968, 1784,4016, 1800,2160, 1816,3184, 1832,2672, 1848,3696,
   1864,2416, 1880,3440, 1896,2928, 1912,3952, 1928,2288, 1944,3312, 1960,2800, 1976,3824,
   1992,2544, 2008,3568, 2024,3056, 2040,4080, 2072,3080, 2088,2568, 2104,3592, 2120,2312,
   2136,3336, 2152,2824, 2168,3848, 2200,3208, 2216,2696, 2232,3720, 2248,2440, 2264,3464,
   2280,2952, 2296,3976, 2328,3144, 2344,2632, 2360,3656, 2392,3400, 2408,2888, 2424,3912,
   2456,3272, 2472,2760, 2488,3784, 2520,3528, 2536,3016, 2552,4040, 2584,3112, 2616,3624,
   2648,3368, 2664,2856, 2680,3880, 2712,3240, 2744,3752, 2776,3496, 2792,2984, 2808,4008,
   2840,3176, 2872,3688, 2904,3432, 2936,3944, 2968,3304, 3000,3816, 3032,3560, 3064,4072,
   3128,3608, 3160,3352, 3192,3864, 3256,3736, 3288,3480, 3320,3992, 3384,3672, 3448,3928,
   3512,3800, 3576,4056, 3704,3896, 3832,4024
};

const uint16_t armBitRevIndexTable_fixed_1024[ARMBITREVINDEXTABLE_FIXED_1024_TABLE_LENGTH] =
{
    /* radix 4, size 992 */
    8,4096, 16,2048, 24,6144, 32,1024, 40,5120, 48,3072, 56,7168, 64,512, 72,4608,
    80,2560, 88,6656, 96,1536, 104,5632, 112,3584, 120,7680, 128,256, 136,4352,
    144,2304, 152,6400, 160,1280, 168,5376, 176,3328, 184,7424, 192,768, 200,4864,
    208,2816, 216,6912, 224,1792, 232,5888, 240,3840, 248,7936, 264,4224, 272,2176,
    280,6272, 288,1152, 296,5248, 304,3200, 312,7296, 320,640, 328,4736, 336,2688,
    344,6784, 352,1664, 360,5760, 368,3712, 376,7808, 392,4480, 400,2432, 408,6528,
    416,1408, 424,5504, 432,3456, 440,7552, 448,896, 456,4992, 464,2944, 472,7040,
    480,1920, 488,6016, 496,3968, 504,8064, 520,4160, 528,2112, 536,6208, 544,1088,
    552,5184, 560,3136, 568,7232, 584,4672, 592,2624, 600,6720, 608,1600, 616,5696,
    624,3648, 632,7744, 648,4416, 656,2368, 664,6464, 672,1344, 680,5440, 688,3392,
    696,7488, 704,832, 712,4928, 720,2880, 728,6976, 736,1856, 744,5952, 752,3904,
    760,8000, 776,4288, 784,2240, 792,6336, 800,1216, 808,5312, 816,3264, 824,7360,
    840,4800, 848,2752, 856,6848, 864,1728, 872,5824, 880,3776, 888,7872, 904,4544,
    912,2496, 920,6592, 928,1472, 936,5568, 944,3520, 952,7616, 968,5056, 976,3008,
    984,7104, 992,1984, 1000,6080, 1008,4032, 1016,8128, 1032,4128, 1040,2080,
    1048,6176, 1064,5152, 1072,3104, 1080,7200, 1096,4640, 1104,2592, 1112,6688,
    1120,1568, 1128,5664, 1136,3616, 1144,7712, 1160,4384, 1168,2336, 1176,6432,
    1184,1312, 1192,5408, 1200,3360, 1208,7456, 1224,4896, 1232,2848, 1240,6944,
    1248,1824, 1256,5920, 1264,3872, 1272,7968, 1288,4256, 1296,2208, 1304,6304,
    1320,5280, 1328,3232, 1336,7328, 1352,4768, 1360,2720, 1368,6816, 1376,1696,
    1384,5792, 1392,3744, 1400,7840, 1416,4512, 1424,2464, 1432,6560, 1448,5536,
    1456,3488, 1464,7584, 1480,5024, 1488,2976, 1496,7072, 1504,1952, 1512,6048,
    1520,4000, 1528,8096, 1544,4192, 1552,2144, 1560,6240, 1576,5216, 1584,3168,
    1592,7264, 1608,4704, 1616,2656, 1624,6752, 1640,5728, 1648,3680, 1656,7776,
    1672,4448, 1680,2400, 1688,6496, 1704,5472, 1712,3424, 1720,7520, 1736,4960,
    1744,2912, 1752,7008, 1760,1888, 1768,5984, 1776,3936, 1784,8032, 1800,4320,
    1808,2272, 1816,6368, 1832,5344, 1840,3296, 1848,7392, 1864,4832, 1872,2784,
    1880,6880, 1896,5856, 1904,3808, 1912,7904, 1928,4576, 1936,2528, 1944,6624,
    1960,5600, 1968,3552, 1976,7648, 1992,5088, 2000,3040, 2008,7136, 2024,6112,
    2032,4064, 2040,8160, 2056,4112, 2072,6160, 2088,5136, 2096,3088, 2104,7184,
    2120,4624, 2128,2576, 2136,6672, 2152,5648, 2160,3600, 2168,7696, 2184,4368,
    2192,2320, 2200,6416, 2216,5392, 2224,3344, 2232,7440, 2248,4880, 2256,2832,
    2264,6928, 2280,5904, 2288,3856, 2296,7952, 2312,4240, 2328,6288, 2344,5264,
    2352,3216, 2360,7312, 2376,4752, 2384,2704, 2392,6800, 2408,5776, 2416,3728,
    2424,7824, 2440,4496, 2456,6544, 2472,5520, 2480,3472, 2488,7568, 2504,5008,
    2512,2960, 2520,7056, 2536,6032, 2544,3984, 2552,8080, 2568,4176, 2584,6224,
    2600,5200, 2608,3152, 2616,7248, 2632,4688, 2648,6736, 2664,5712, 2672,3664,
    2680,7760, 2696,4432, 2712,6480, 2728,5456, 2736,3408, 2744,7504, 2760,4944,
    2768,2896, 2776,6992, 2792,5968, 2800,3920, 2808,8016, 2824,4304, 2840,6352,
    2856,5328, 2864,3280, 2872,7376, 2888,4816, 2904,6864, 2920,5840, 2928,3792,
    2936,7888, 2952,4560, 2968,6608, 2984,5584, 2992,3536, 3000,7632, 3016,5072,
    3032,7120, 3048,6096, 3056,4048, 3064,8144, 3080,4144, 3096,6192, 3112,5168,
    3128,7216, 3144,4656, 3160,6704, 3176,5680, 3184,3632, 3192,7728, 3208,4400,
    3224,6448, 3240,5424, 3248,3376, 3256,7472, 3272,4912, 3288,6960, 3304,5936,
    3312,3888, 3320,7984, 3336,4272, 3352,6320, 3368,5296, 3384,7344, 3400,4784,
    3416,6832, 3432,5808, 3440,3760, 3448,7856, 3464,4528, 3480,6576, 3496,5552,
    3512,7600, 3528,5040, 3544,7088, 3560,6064, 3568,4016, 3576,8112, 3592,4208,
    3608,6256, 3624,5232, 3640,7280, 3656,4720, 3672,6768, 3688,5744, 3704,7792,
    3720,4464, 3736,6512, 3752,5488, 3768,7536, 3784,4976, 3800,7024, 3816,6000,
    3824,3952, 3832,8048, 3848,4336, 3864,6384, 3880,5360, 3896,7408, 3912,4848,
    3928,6896, 3944,5872, 3960,7920, 3976,4592, 3992,6640, 4008,5616, 4024,7664,
    4040,5104, 4056,7152, 4072,6128, 4088,8176, 4120,6152, 4136,5128, 4152,7176,
    4168,4616, 4184,6664, 4200,5640, 4216,7688, 4232,4360, 4248,6408, 4264,5384,
    4280,7432, 4296,4872, 4312,6920, 4328,5896, 4344,7944, 4376,6280, 4392,5256,
    4408,7304, 4424,4744, 4440,6792, 4456,5768, 4472,7816, 4504,6536, 4520,5512,
    4536,7560, 4552,5000, 4568,7048, 4584,6024, 4600,8072, 4632,6216, 4648,5192,
    4664,7240, 4696,6728, 4712,5704, 4728,7752, 4760,6472, 4776,5448, 4792,7496,
    4808,4936, 4824,6984, 4840,5960, 4856,8008, 4888,6344, 4904,5320, 4920,7368,
    4952,6856, 4968,5832, 4984,7880, 5016,6600, 5032,5576, 5048,7624, 5080,7112,
    5096,6088, 5112,8136, 5144,6184, 5176,7208, 5208,6696, 5224,5672, 5240,7720,
    5272,6440, 5288,5416, 5304,7464, 5336,6952, 5352,5928, 5368,7976, 5400,6312,
    5432,7336, 5464,6824, 5480,5800, 5496,7848, 5528,6568, 5560,7592, 5592,7080,
    5608,6056, 5624,8104, 5656,6248, 5688,7272, 5720,6760, 5752,7784, 5784,6504,
    5816,7528, 5848,7016, 5864,5992, 5880,8040, 5912,6376, 5944,7400, 5976,6888,
    6008,7912, 6040,6632, 6072,7656, 6104,7144, 6136,8168, 6200,7192, 6232,6680,
    6264,7704, 6296,6424, 6328,7448, 6360,6936, 6392,7960, 6456,7320, 6488,6808,
    6520,7832, 6584,7576, 6616,7064, 6648,8088, 6712,7256, 6776,7768, 6840,7512,
    6872,7000, 6904,8024, 6968,7384, 7032,7896, 7096,7640, 7160,8152, 7288,7736,
    7352,7480, 7416,7992, 7544,7864, 7672,8120, 7928,8056
};

const uint16_t armBitRevIndexTable_fixed_2048[ARMBITREVINDEXTABLE_FIXED_2048_TABLE_LENGTH] =
{
    /* 4x2, size 1984 */
    8,8192, 16,4096, 24,12288, 32,2048, 40,10240, 48,6144, 56,14336, 64,1024,
    72,9216, 80,5120, 88,13312, 96,3072, 104,11264, 112,7168, 120,15360, 128,512,
    136,8704, 144,4608, 152,12800, 160,2560, 168,10752, 176,6656, 184,14848,
    192,1536, 200,9728, 208,5632, 216,13824, 224,3584, 232,11776, 240,7680,
    248,15872, 264,8448, 272,4352, 280,12544, 288,2304, 296,10496, 304,6400,
    312,14592, 320,1280, 328,9472, 336,5376, 344,13568, 352,3328, 360,11520,
    368,7424, 376,15616, 384,768, 392,8960, 400,4864, 408,13056, 416,2816,
    424,11008, 432,6912, 440,15104, 448,1792, 456,9984, 464,5888, 472,14080,
    480,3840, 488,12032, 496,7936, 504,16128, 520,8320, 528,4224, 536,12416,
    544,2176, 552,10368, 560,6272, 568,14464, 576,1152, 584,9344, 592,5248,
    600,13440, 608,3200, 616,11392, 624,7296, 632,15488, 648,8832, 656,4736,
    664,12928, 672,2688, 680,10880, 688,6784, 696,14976, 704,1664, 712,9856,
    720,5760, 728,13952, 736,3712, 744,11904, 752,7808, 760,16000, 776,8576,
    784,4480, 792,12672, 800,2432, 808,10624, 816,6528, 824,14720, 832,1408,
    840,9600, 848,5504, 856,13696, 864,3456, 872,11648, 880,7552, 888,15744,
    904,9088, 912,4992, 920,13184, 928,2944, 936,11136, 944,7040, 952,15232,
    960,1920, 968,10112, 976,6016, 984,14208, 992,3968, 1000,12160, 1008,8064,
    1016,16256, 1032,8256, 1040,4160, 1048,12352, 1056,2112, 1064,10304, 1072,6208,
    1080,14400, 1096,9280, 1104,5184, 1112,13376, 1120,3136, 1128,11328, 1136,7232,
    1144,15424, 1160,8768, 1168,4672, 1176,12864, 1184,2624, 1192,10816, 1200,6720,
    1208,14912, 1216,1600, 1224,9792, 1232,5696, 1240,13888, 1248,3648, 1256,11840,
    1264,7744, 1272,15936, 1288,8512, 1296,4416, 1304,12608, 1312,2368, 1320,10560,
    1328,6464, 1336,14656, 1352,9536, 1360,5440, 1368,13632, 1376,3392, 1384,11584,
    1392,7488, 1400,15680, 1416,9024, 1424,4928, 1432,13120, 1440,2880, 1448,11072,
    1456,6976, 1464,15168, 1472,1856, 1480,10048, 1488,5952, 1496,14144, 1504,3904,
    1512,12096, 1520,8000, 1528,16192, 1544,8384, 1552,4288, 1560,12480, 1568,2240,
    1576,10432, 1584,6336, 1592,14528, 1608,9408, 1616,5312, 1624,13504, 1632,3264,
    1640,11456, 1648,7360, 1656,15552, 1672,8896, 1680,4800, 1688,12992, 1696,2752,
    1704,10944, 1712,6848, 1720,15040, 1736,9920, 1744,5824, 1752,14016, 1760,3776,
    1768,11968, 1776,7872, 1784,16064, 1800,8640, 1808,4544, 1816,12736, 1824,2496,
    1832,10688, 1840,6592, 1848,14784, 1864,9664, 1872,5568, 1880,13760, 1888,3520,
    1896,11712, 1904,7616, 1912,15808, 1928,9152, 1936,5056, 1944,13248, 1952,3008,
    1960,11200, 1968,7104, 1976,15296, 1992,10176, 2000,6080, 2008,14272, 2016,4032,
    2024,12224, 2032,8128, 2040,16320, 2056,8224, 2064,4128, 2072,12320, 2088,10272,
    2096,6176, 2104,14368, 2120,9248, 2128,5152, 2136,13344, 2144,3104, 2152,11296,
    2160,7200, 2168,15392, 2184,8736, 2192,4640, 2200,12832, 2208,2592, 2216,10784,
    2224,6688, 2232,14880, 2248,9760, 2256,5664, 2264,13856, 2272,3616, 2280,11808,
    2288,7712, 2296,15904, 2312,8480, 2320,4384, 2328,12576, 2344,10528, 2352,6432,
    2360,14624, 2376,9504, 2384,5408, 2392,13600, 2400,3360, 2408,11552, 2416,7456,
    2424,15648, 2440,8992, 2448,4896, 2456,13088, 2464,2848, 2472,11040, 2480,6944,
    2488,15136, 2504,10016, 2512,5920, 2520,14112, 2528,3872, 2536,12064, 2544,7968,
    2552,16160, 2568,8352, 2576,4256, 2584,12448, 2600,10400, 2608,6304, 2616,14496,
    2632,9376, 2640,5280, 2648,13472, 2656,3232, 2664,11424, 2672,7328, 2680,15520,
    2696,8864, 2704,4768, 2712,12960, 2728,10912, 2736,6816, 2744,15008, 2760,9888,
    2768,5792, 2776,13984, 2784,3744, 2792,11936, 2800,7840, 2808,16032, 2824,8608,
    2832,4512, 2840,12704, 2856,10656, 2864,6560, 2872,14752, 2888,9632, 2896,5536,
    2904,13728, 2912,3488, 2920,11680, 2928,7584, 2936,15776, 2952,9120, 2960,5024,
    2968,13216, 2984,11168, 2992,7072, 3000,15264, 3016,10144, 3024,6048,
    3032,14240, 3040,4000, 3048,12192, 3056,8096, 3064,16288, 3080,8288, 3088,4192,
    3096,12384, 3112,10336, 3120,6240, 3128,14432, 3144,9312, 3152,5216, 3160,13408,
    3176,11360, 3184,7264, 3192,15456, 3208,8800, 3216,4704, 3224,12896, 3240,10848,
    3248,6752, 3256,14944, 3272,9824, 3280,5728, 3288,13920, 3296,3680, 3304,11872,
    3312,7776, 3320,15968, 3336,8544, 3344,4448, 3352,12640, 3368,10592, 3376,6496,
    3384,14688, 3400,9568, 3408,5472, 3416,13664, 3432,11616, 3440,7520, 3448,15712,
    3464,9056, 3472,4960, 3480,13152, 3496,11104, 3504,7008, 3512,15200, 3528,10080,
    3536,5984, 3544,14176, 3552,3936, 3560,12128, 3568,8032, 3576,16224, 3592,8416,
    3600,4320, 3608,12512, 3624,10464, 3632,6368, 3640,14560, 3656,9440, 3664,5344,
    3672,13536, 3688,11488, 3696,7392, 3704,15584, 3720,8928, 3728,4832, 3736,13024,
    3752,10976, 3760,6880, 3768,15072, 3784,9952, 3792,5856, 3800,14048, 3816,12000,
    3824,7904, 3832,16096, 3848,8672, 3856,4576, 3864,12768, 3880,10720, 3888,6624,
    3896,14816, 3912,9696, 3920,5600, 3928,13792, 3944,11744, 3952,7648, 3960,15840,
    3976,9184, 3984,5088, 3992,13280, 4008,11232, 4016,7136, 4024,15328, 4040,10208,
    4048,6112, 4056,14304, 4072,12256, 4080,8160, 4088,16352, 4104,8208, 4120,12304,
    4136,10256, 4144,6160, 4152,14352, 4168,9232, 4176,5136, 4184,13328, 4200,11280,
    4208,7184, 4216,15376, 4232,8720, 4240,4624, 4248,12816, 4264,10768, 4272,6672,
    4280,14864, 4296,9744, 4304,5648, 4312,13840, 4328,11792, 4336,7696, 4344,15888,
    4360,8464, 4376,12560, 4392,10512, 4400,6416, 4408,14608, 4424,9488, 4432,5392,
    4440,13584, 4456,11536, 4464,7440, 4472,15632, 4488,8976, 4496,4880, 4504,13072,
    4520,11024, 4528,6928, 4536,15120, 4552,10000, 4560,5904, 4568,14096,
    4584,12048, 4592,7952, 4600,16144, 4616,8336, 4632,12432, 4648,10384, 4656,6288,
    4664,14480, 4680,9360, 4688,5264, 4696,13456, 4712,11408, 4720,7312, 4728,15504,
    4744,8848, 4760,12944, 4776,10896, 4784,6800, 4792,14992, 4808,9872, 4816,5776,
    4824,13968, 4840,11920, 4848,7824, 4856,16016, 4872,8592, 4888,12688,
    4904,10640, 4912,6544, 4920,14736, 4936,9616, 4944,5520, 4952,13712, 4968,11664,
    4976,7568, 4984,15760, 5000,9104, 5016,13200, 5032,11152, 5040,7056, 5048,15248,
    5064,10128, 5072,6032, 5080,14224, 5096,12176, 5104,8080, 5112,16272, 5128,8272,
    5144,12368, 5160,10320, 5168,6224, 5176,14416, 5192,9296, 5208,13392,
    5224,11344, 5232,7248, 5240,15440, 5256,8784, 5272,12880, 5288,10832, 5296,6736,
    5304,14928, 5320,9808, 5328,5712, 5336,13904, 5352,11856, 5360,7760, 5368,15952,
    5384,8528, 5400,12624, 5416,10576, 5424,6480, 5432,14672, 5448,9552, 5464,13648,
    5480,11600, 5488,7504, 5496,15696, 5512,9040, 5528,13136, 5544,11088, 5552,6992,
    5560,15184, 5576,10064, 5584,5968, 5592,14160, 5608,12112, 5616,8016,
    5624,16208, 5640,8400, 5656,12496, 5672,10448, 5680,6352, 5688,14544, 5704,9424,
    5720,13520, 5736,11472, 5744,7376, 5752,15568, 5768,8912, 5784,13008,
    5800,10960, 5808,6864, 5816,15056, 5832,9936, 5848,14032, 5864,11984, 5872,7888,
    5880,16080, 5896,8656, 5912,12752, 5928,10704, 5936,6608, 5944,14800, 5960,9680,
    5976,13776, 5992,11728, 6000,7632, 6008,15824, 6024,9168, 6040,13264,
    6056,11216, 6064,7120, 6072,15312, 6088,10192, 6104,14288, 6120,12240,
    6128,8144, 6136,16336, 6152,8240, 6168,12336, 6184,10288, 6200,14384, 6216,9264,
    6232,13360, 6248,11312, 6256,7216, 6264,15408, 6280,8752, 6296,12848,
    6312,10800, 6320,6704, 6328,14896, 6344,9776, 6360,13872, 6376,11824, 6384,7728,
    6392,15920, 6408,8496, 6424,12592, 6440,10544, 6456,14640, 6472,9520,
    6488,13616, 6504,11568, 6512,7472, 6520,15664, 6536,9008, 6552,13104,
    6568,11056, 6576,6960, 6584,15152, 6600,10032, 6616,14128, 6632,12080,
    6640,7984, 6648,16176, 6664,8368, 6680,12464, 6696,10416, 6712,14512, 6728,9392,
    6744,13488, 6760,11440, 6768,7344, 6776,15536, 6792,8880, 6808,12976,
    6824,10928, 6840,15024, 6856,9904, 6872,14000, 6888,11952, 6896,7856,
    6904,16048, 6920,8624, 6936,12720, 6952,10672, 6968,14768, 6984,9648,
    7000,13744, 7016,11696, 7024,7600, 7032,15792, 7048,9136, 7064,13232,
    7080,11184, 7096,15280, 7112,10160, 7128,14256, 7144,12208, 7152,8112,
    7160,16304, 7176,8304, 7192,12400, 7208,10352, 7224,14448, 7240,9328,
    7256,13424, 7272,11376, 7288,15472, 7304,8816, 7320,12912, 7336,10864,
    7352,14960, 7368,9840, 7384,13936, 7400,11888, 7408,7792, 7416,15984, 7432,8560,
    7448,12656, 7464,10608, 7480,14704, 7496,9584, 7512,13680, 7528,11632,
    7544,15728, 7560,9072, 7576,13168, 7592,11120, 7608,15216, 7624,10096,
    7640,14192, 7656,12144, 7664,8048, 7672,16240, 7688,8432, 7704,12528,
    7720,10480, 7736,14576, 7752,9456, 7768,13552, 7784,11504, 7800,15600,
    7816,8944, 7832,13040, 7848,10992, 7864,15088, 7880,9968, 7896,14064,
    7912,12016, 7928,16112, 7944,8688, 7960,12784, 7976,10736, 7992,14832,
    8008,9712, 8024,13808, 8040,11760, 8056,15856, 8072,9200, 8088,13296,
    8104,11248, 8120,15344, 8136,10224, 8152,14320, 8168,12272, 8184,16368,
    8216,12296, 8232,10248, 8248,14344, 8264,9224, 8280,13320, 8296,11272,
    8312,15368, 8328,8712, 8344,12808, 8360,10760, 8376,14856, 8392,9736,
    8408,13832, 8424,11784, 8440,15880, 8472,12552, 8488,10504, 8504,14600,
    8520,9480, 8536,13576, 8552,11528, 8568,15624, 8584,8968, 8600,13064,
    8616,11016, 8632,15112, 8648,9992, 8664,14088, 8680,12040, 8696,16136,
    8728,12424, 8744,10376, 8760,14472, 8776,9352, 8792,13448, 8808,11400,
    8824,15496, 8856,12936, 8872,10888, 8888,14984, 8904,9864, 8920,13960,
    8936,11912, 8952,16008, 8984,12680, 9000,10632, 9016,14728, 9032,9608,
    9048,13704, 9064,11656, 9080,15752, 9112,13192, 9128,11144, 9144,15240,
    9160,10120, 9176,14216, 9192,12168, 9208,16264, 9240,12360, 9256,10312,
    9272,14408, 9304,13384, 9320,11336, 9336,15432, 9368,12872, 9384,10824,
    9400,14920, 9416,9800, 9432,13896, 9448,11848, 9464,15944, 9496,12616,
    9512,10568, 9528,14664, 9560,13640, 9576,11592, 9592,15688, 9624,13128,
    9640,11080, 9656,15176, 9672,10056, 9688,14152, 9704,12104, 9720,16200,
    9752,12488, 9768,10440, 9784,14536, 9816,13512, 9832,11464, 9848,15560,
    9880,13000, 9896,10952, 9912,15048, 9944,14024, 9960,11976, 9976,16072,
    10008,12744, 10024,10696, 10040,14792, 10072,13768, 10088,11720, 10104,15816,
    10136,13256, 10152,11208, 10168,15304, 10200,14280, 10216,12232, 10232,16328,
    10264,12328, 10296,14376, 10328,13352, 10344,11304, 10360,15400, 10392,12840,
    10408,10792, 10424,14888, 10456,13864, 10472,11816, 10488,15912, 10520,12584,
    10552,14632, 10584,13608, 10600,11560, 10616,15656, 10648,13096, 10664,11048,
    10680,15144, 10712,14120, 10728,12072, 10744,16168, 10776,12456, 10808,14504,
    10840,13480, 10856,11432, 10872,15528, 10904,12968, 10936,15016, 10968,13992,
    10984,11944, 11000,16040, 11032,12712, 11064,14760, 11096,13736, 11112,11688,
    11128,15784, 11160,13224, 11192,15272, 11224,14248, 11240,12200, 11256,16296,
    11288,12392, 11320,14440, 11352,13416, 11384,15464, 11416,12904, 11448,14952,
    11480,13928, 11496,11880, 11512,15976, 11544,12648, 11576,14696, 11608,13672,
    11640,15720, 11672,13160, 11704,15208, 11736,14184, 11752,12136, 11768,16232,
    11800,12520, 11832,14568, 11864,13544, 11896,15592, 11928,13032, 11960,15080,
    11992,14056, 12024,16104, 12056,12776, 12088,14824, 12120,13800, 12152,15848,
    12184,13288, 12216,15336, 12248,14312, 12280,16360, 12344,14360, 12376,13336,
    12408,15384, 12440,12824, 12472,14872, 12504,13848, 12536,15896, 12600,14616,
    12632,13592, 12664,15640, 12696,13080, 12728,15128, 12760,14104, 12792,16152,
    12856,14488, 12888,13464, 12920,15512, 12984,15000, 13016,13976, 13048,16024,
    13112,14744, 13144,13720, 13176,15768, 13240,15256, 13272,14232, 13304,16280,
    13368,14424, 13432,15448, 13496,14936, 13528,13912, 13560,15960, 13624,14680,
    13688,15704, 13752,15192, 13784,14168, 13816,16216, 13880,14552, 13944,15576,
    14008,15064, 14072,16088, 14136,14808, 14200,15832, 14264,15320, 14328,16344,
    14456,15416, 14520,14904, 14584,15928, 14712,15672, 14776,15160, 14840,16184,
    14968,15544, 15096,16056, 15224,15800, 15352,16312, 15608,15992, 15864,16248
};

const uint16_t armBitRevIndexTable_fixed_4096[ARMBITREVINDEXTABLE_FIXED_4096_TABLE_LENGTH] =
{
    /* radix 4, size 4032 */
    8,16384, 16,8192, 24,24576, 32,4096, 40,20480, 48,12288, 56,28672, 64,2048,
    72,18432, 80,10240, 88,26624, 96,6144, 104,22528, 112,14336, 120,30720,
    128,1024, 136,17408, 144,9216, 152,25600, 160,5120, 168,21504, 176,13312,
    184,29696, 192,3072, 200,19456, 208,11264, 216,27648, 224,7168, 232,23552,
    240,15360, 248,31744, 256,512, 264,16896, 272,8704, 280,25088, 288,4608,
    296,20992, 304,12800, 312,29184, 320,2560, 328,18944, 336,10752, 344,27136,
    352,6656, 360,23040, 368,14848, 376,31232, 384,1536, 392,17920, 400,9728,
    408,26112, 416,5632, 424,22016, 432,13824, 440,30208, 448,3584, 456,19968,
    464,11776, 472,28160, 480,7680, 488,24064, 496,15872, 504,32256, 520,16640,
    528,8448, 536,24832, 544,4352, 552,20736, 560,12544, 568,28928, 576,2304,
    584,18688, 592,10496, 600,26880, 608,6400, 616,22784, 624,14592, 632,30976,
    640,1280, 648,17664, 656,9472, 664,25856, 672,5376, 680,21760, 688,13568,
    696,29952, 704,3328, 712,19712, 720,11520, 728,27904, 736,7424, 744,23808,
    752,15616, 760,32000, 776,17152, 784,8960, 792,25344, 800,4864, 808,21248,
    816,13056, 824,29440, 832,2816, 840,19200, 848,11008, 856,27392, 864,6912,
    872,23296, 880,15104, 888,31488, 896,1792, 904,18176, 912,9984, 920,26368,
    928,5888, 936,22272, 944,14080, 952,30464, 960,3840, 968,20224, 976,12032,
    984,28416, 992,7936, 1000,24320, 1008,16128, 1016,32512, 1032,16512, 1040,8320,
    1048,24704, 1056,4224, 1064,20608, 1072,12416, 1080,28800, 1088,2176,
    1096,18560, 1104,10368, 1112,26752, 1120,6272, 1128,22656, 1136,14464,
    1144,30848, 1160,17536, 1168,9344, 1176,25728, 1184,5248, 1192,21632,
    1200,13440, 1208,29824, 1216,3200, 1224,19584, 1232,11392, 1240,27776,
    1248,7296, 1256,23680, 1264,15488, 1272,31872, 1288,17024, 1296,8832,
    1304,25216, 1312,4736, 1320,21120, 1328,12928, 1336,29312, 1344,2688,
    1352,19072, 1360,10880, 1368,27264, 1376,6784, 1384,23168, 1392,14976,
    1400,31360, 1408,1664, 1416,18048, 1424,9856, 1432,26240, 1440,5760, 1448,22144,
    1456,13952, 1464,30336, 1472,3712, 1480,20096, 1488,11904, 1496,28288,
    1504,7808, 1512,24192, 1520,16000, 1528,32384, 1544,16768, 1552,8576,
    1560,24960, 1568,4480, 1576,20864, 1584,12672, 1592,29056, 1600,2432,
    1608,18816, 1616,10624, 1624,27008, 1632,6528, 1640,22912, 1648,14720,
    1656,31104, 1672,17792, 1680,9600, 1688,25984, 1696,5504, 1704,21888,
    1712,13696, 1720,30080, 1728,3456, 1736,19840, 1744,11648, 1752,28032,
    1760,7552, 1768,23936, 1776,15744, 1784,32128, 1800,17280, 1808,9088,
    1816,25472, 1824,4992, 1832,21376, 1840,13184, 1848,29568, 1856,2944,
    1864,19328, 1872,11136, 1880,27520, 1888,7040, 1896,23424, 1904,15232,
    1912,31616, 1928,18304, 1936,10112, 1944,26496, 1952,6016, 1960,22400,
    1968,14208, 1976,30592, 1984,3968, 1992,20352, 2000,12160, 2008,28544,
    2016,8064, 2024,24448, 2032,16256, 2040,32640, 2056,16448, 2064,8256,
    2072,24640, 2080,4160, 2088,20544, 2096,12352, 2104,28736, 2120,18496,
    2128,10304, 2136,26688, 2144,6208, 2152,22592, 2160,14400, 2168,30784,
    2184,17472, 2192,9280, 2200,25664, 2208,5184, 2216,21568, 2224,13376,
    2232,29760, 2240,3136, 2248,19520, 2256,11328, 2264,27712, 2272,7232,
    2280,23616, 2288,15424, 2296,31808, 2312,16960, 2320,8768, 2328,25152,
    2336,4672, 2344,21056, 2352,12864, 2360,29248, 2368,2624, 2376,19008,
    2384,10816, 2392,27200, 2400,6720, 2408,23104, 2416,14912, 2424,31296,
    2440,17984, 2448,9792, 2456,26176, 2464,5696, 2472,22080, 2480,13888,
    2488,30272, 2496,3648, 2504,20032, 2512,11840, 2520,28224, 2528,7744,
    2536,24128, 2544,15936, 2552,32320, 2568,16704, 2576,8512, 2584,24896,
    2592,4416, 2600,20800, 2608,12608, 2616,28992, 2632,18752, 2640,10560,
    2648,26944, 2656,6464, 2664,22848, 2672,14656, 2680,31040, 2696,17728,
    2704,9536, 2712,25920, 2720,5440, 2728,21824, 2736,13632, 2744,30016, 2752,3392,
    2760,19776, 2768,11584, 2776,27968, 2784,7488, 2792,23872, 2800,15680,
    2808,32064, 2824,17216, 2832,9024, 2840,25408, 2848,4928, 2856,21312,
    2864,13120, 2872,29504, 2888,19264, 2896,11072, 2904,27456, 2912,6976,
    2920,23360, 2928,15168, 2936,31552, 2952,18240, 2960,10048, 2968,26432,
    2976,5952, 2984,22336, 2992,14144, 3000,30528, 3008,3904, 3016,20288,
    3024,12096, 3032,28480, 3040,8000, 3048,24384, 3056,16192, 3064,32576,
    3080,16576, 3088,8384, 3096,24768, 3104,4288, 3112,20672, 3120,12480,
    3128,28864, 3144,18624, 3152,10432, 3160,26816, 3168,6336, 3176,22720,
    3184,14528, 3192,30912, 3208,17600, 3216,9408, 3224,25792, 3232,5312,
    3240,21696, 3248,13504, 3256,29888, 3272,19648, 3280,11456, 3288,27840,
    3296,7360, 3304,23744, 3312,15552, 3320,31936, 3336,17088, 3344,8896,
    3352,25280, 3360,4800, 3368,21184, 3376,12992, 3384,29376, 3400,19136,
    3408,10944, 3416,27328, 3424,6848, 3432,23232, 3440,15040, 3448,31424,
    3464,18112, 3472,9920, 3480,26304, 3488,5824, 3496,22208, 3504,14016,
    3512,30400, 3520,3776, 3528,20160, 3536,11968, 3544,28352, 3552,7872,
    3560,24256, 3568,16064, 3576,32448, 3592,16832, 3600,8640, 3608,25024,
    3616,4544, 3624,20928, 3632,12736, 3640,29120, 3656,18880, 3664,10688,
    3672,27072, 3680,6592, 3688,22976, 3696,14784, 3704,31168, 3720,17856,
    3728,9664, 3736,26048, 3744,5568, 3752,21952, 3760,13760, 3768,30144,
    3784,19904, 3792,11712, 3800,28096, 3808,7616, 3816,24000, 3824,15808,
    3832,32192, 3848,17344, 3856,9152, 3864,25536, 3872,5056, 3880,21440,
    3888,13248, 3896,29632, 3912,19392, 3920,11200, 3928,27584, 3936,7104,
    3944,23488, 3952,15296, 3960,31680, 3976,18368, 3984,10176, 3992,26560,
    4000,6080, 4008,22464, 4016,14272, 4024,30656, 4040,20416, 4048,12224,
    4056,28608, 4064,8128, 4072,24512, 4080,16320, 4088,32704, 4104,16416,
    4112,8224, 4120,24608, 4136,20512, 4144,12320, 4152,28704, 4168,18464,
    4176,10272, 4184,26656, 4192,6176, 4200,22560, 4208,14368, 4216,30752,
    4232,17440, 4240,9248, 4248,25632, 4256,5152, 4264,21536, 4272,13344,
    4280,29728, 4296,19488, 4304,11296, 4312,27680, 4320,7200, 4328,23584,
    4336,15392, 4344,31776, 4360,16928, 4368,8736, 4376,25120, 4384,4640,
    4392,21024, 4400,12832, 4408,29216, 4424,18976, 4432,10784, 4440,27168,
    4448,6688, 4456,23072, 4464,14880, 4472,31264, 4488,17952, 4496,9760,
    4504,26144, 4512,5664, 4520,22048, 4528,13856, 4536,30240, 4552,20000,
    4560,11808, 4568,28192, 4576,7712, 4584,24096, 4592,15904, 4600,32288,
    4616,16672, 4624,8480, 4632,24864, 4648,20768, 4656,12576, 4664,28960,
    4680,18720, 4688,10528, 4696,26912, 4704,6432, 4712,22816, 4720,14624,
    4728,31008, 4744,17696, 4752,9504, 4760,25888, 4768,5408, 4776,21792,
    4784,13600, 4792,29984, 4808,19744, 4816,11552, 4824,27936, 4832,7456,
    4840,23840, 4848,15648, 4856,32032, 4872,17184, 4880,8992, 4888,25376,
    4904,21280, 4912,13088, 4920,29472, 4936,19232, 4944,11040, 4952,27424,
    4960,6944, 4968,23328, 4976,15136, 4984,31520, 5000,18208, 5008,10016,
    5016,26400, 5024,5920, 5032,22304, 5040,14112, 5048,30496, 5064,20256,
    5072,12064, 5080,28448, 5088,7968, 5096,24352, 5104,16160, 5112,32544,
    5128,16544, 5136,8352, 5144,24736, 5160,20640, 5168,12448, 5176,28832,
    5192,18592, 5200,10400, 5208,26784, 5216,6304, 5224,22688, 5232,14496,
    5240,30880, 5256,17568, 5264,9376, 5272,25760, 5288,21664, 5296,13472,
    5304,29856, 5320,19616, 5328,11424, 5336,27808, 5344,7328, 5352,23712,
    5360,15520, 5368,31904, 5384,17056, 5392,8864, 5400,25248, 5416,21152,
    5424,12960, 5432,29344, 5448,19104, 5456,10912, 5464,27296, 5472,6816,
    5480,23200, 5488,15008, 5496,31392, 5512,18080, 5520,9888, 5528,26272,
    5536,5792, 5544,22176, 5552,13984, 5560,30368, 5576,20128, 5584,11936,
    5592,28320, 5600,7840, 5608,24224, 5616,16032, 5624,32416, 5640,16800,
    5648,8608, 5656,24992, 5672,20896, 5680,12704, 5688,29088, 5704,18848,
    5712,10656, 5720,27040, 5728,6560, 5736,22944, 5744,14752, 5752,31136,
    5768,17824, 5776,9632, 5784,26016, 5800,21920, 5808,13728, 5816,30112,
    5832,19872, 5840,11680, 5848,28064, 5856,7584, 5864,23968, 5872,15776,
    5880,32160, 5896,17312, 5904,9120, 5912,25504, 5928,21408, 5936,13216,
    5944,29600, 5960,19360, 5968,11168, 5976,27552, 5984,7072, 5992,23456,
    6000,15264, 6008,31648, 6024,18336, 6032,10144, 6040,26528, 6056,22432,
    6064,14240, 6072,30624, 6088,20384, 6096,12192, 6104,28576, 6112,8096,
    6120,24480, 6128,16288, 6136,32672, 6152,16480, 6160,8288, 6168,24672,
    6184,20576, 6192,12384, 6200,28768, 6216,18528, 6224,10336, 6232,26720,
    6248,22624, 6256,14432, 6264,30816, 6280,17504, 6288,9312, 6296,25696,
    6312,21600, 6320,13408, 6328,29792, 6344,19552, 6352,11360, 6360,27744,
    6368,7264, 6376,23648, 6384,15456, 6392,31840, 6408,16992, 6416,8800,
    6424,25184, 6440,21088, 6448,12896, 6456,29280, 6472,19040, 6480,10848,
    6488,27232, 6496,6752, 6504,23136, 6512,14944, 6520,31328, 6536,18016,
    6544,9824, 6552,26208, 6568,22112, 6576,13920, 6584,30304, 6600,20064,
    6608,11872, 6616,28256, 6624,7776, 6632,24160, 6640,15968, 6648,32352,
    6664,16736, 6672,8544, 6680,24928, 6696,20832, 6704,12640, 6712,29024,
    6728,18784, 6736,10592, 6744,26976, 6760,22880, 6768,14688, 6776,31072,
    6792,17760, 6800,9568, 6808,25952, 6824,21856, 6832,13664, 6840,30048,
    6856,19808, 6864,11616, 6872,28000, 6880,7520, 6888,23904, 6896,15712,
    6904,32096, 6920,17248, 6928,9056, 6936,25440, 6952,21344, 6960,13152,
    6968,29536, 6984,19296, 6992,11104, 7000,27488, 7016,23392, 7024,15200,
    7032,31584, 7048,18272, 7056,10080, 7064,26464, 7080,22368, 7088,14176,
    7096,30560, 7112,20320, 7120,12128, 7128,28512, 7136,8032, 7144,24416,
    7152,16224, 7160,32608, 7176,16608, 7184,8416, 7192,24800, 7208,20704,
    7216,12512, 7224,28896, 7240,18656, 7248,10464, 7256,26848, 7272,22752,
    7280,14560, 7288,30944, 7304,17632, 7312,9440, 7320,25824, 7336,21728,
    7344,13536, 7352,29920, 7368,19680, 7376,11488, 7384,27872, 7400,23776,
    7408,15584, 7416,31968, 7432,17120, 7440,8928, 7448,25312, 7464,21216,
    7472,13024, 7480,29408, 7496,19168, 7504,10976, 7512,27360, 7528,23264,
    7536,15072, 7544,31456, 7560,18144, 7568,9952, 7576,26336, 7592,22240,
    7600,14048, 7608,30432, 7624,20192, 7632,12000, 7640,28384, 7648,7904,
    7656,24288, 7664,16096, 7672,32480, 7688,16864, 7696,8672, 7704,25056,
    7720,20960, 7728,12768, 7736,29152, 7752,18912, 7760,10720, 7768,27104,
    7784,23008, 7792,14816, 7800,31200, 7816,17888, 7824,9696, 7832,26080,
    7848,21984, 7856,13792, 7864,30176, 7880,19936, 7888,11744, 7896,28128,
    7912,24032, 7920,15840, 7928,32224, 7944,17376, 7952,9184, 7960,25568,
    7976,21472, 7984,13280, 7992,29664, 8008,19424, 8016,11232, 8024,27616,
    8040,23520, 8048,15328, 8056,31712, 8072,18400, 8080,10208, 8088,26592,
    8104,22496, 8112,14304, 8120,30688, 8136,20448, 8144,12256, 8152,28640,
    8168,24544, 8176,16352, 8184,32736, 8200,16400, 8216,24592, 8232,20496,
    8240,12304, 8248,28688, 8264,18448, 8272,10256, 8280,26640, 8296,22544,
    8304,14352, 8312,30736, 8328,17424, 8336,9232, 8344,25616, 8360,21520,
    8368,13328, 8376,29712, 8392,19472, 8400,11280, 8408,27664, 8424,23568,
    8432,15376, 8440,31760, 8456,16912, 8464,8720, 8472,25104, 8488,21008,
    8496,12816, 8504,29200, 8520,18960, 8528,10768, 8536,27152, 8552,23056,
    8560,14864, 8568,31248, 8584,17936, 8592,9744, 8600,26128, 8616,22032,
    8624,13840, 8632,30224, 8648,19984, 8656,11792, 8664,28176, 8680,24080,
    8688,15888, 8696,32272, 8712,16656, 8728,24848, 8744,20752, 8752,12560,
    8760,28944, 8776,18704, 8784,10512, 8792,26896, 8808,22800, 8816,14608,
    8824,30992, 8840,17680, 8848,9488, 8856,25872, 8872,21776, 8880,13584,
    8888,29968, 8904,19728, 8912,11536, 8920,27920, 8936,23824, 8944,15632,
    8952,32016, 8968,17168, 8984,25360, 9000,21264, 9008,13072, 9016,29456,
    9032,19216, 9040,11024, 9048,27408, 9064,23312, 9072,15120, 9080,31504,
    9096,18192, 9104,10000, 9112,26384, 9128,22288, 9136,14096, 9144,30480,
    9160,20240, 9168,12048, 9176,28432, 9192,24336, 9200,16144, 9208,32528,
    9224,16528, 9240,24720, 9256,20624, 9264,12432, 9272,28816, 9288,18576,
    9296,10384, 9304,26768, 9320,22672, 9328,14480, 9336,30864, 9352,17552,
    9368,25744, 9384,21648, 9392,13456, 9400,29840, 9416,19600, 9424,11408,
    9432,27792, 9448,23696, 9456,15504, 9464,31888, 9480,17040, 9496,25232,
    9512,21136, 9520,12944, 9528,29328, 9544,19088, 9552,10896, 9560,27280,
    9576,23184, 9584,14992, 9592,31376, 9608,18064, 9616,9872, 9624,26256,
    9640,22160, 9648,13968, 9656,30352, 9672,20112, 9680,11920, 9688,28304,
    9704,24208, 9712,16016, 9720,32400, 9736,16784, 9752,24976, 9768,20880,
    9776,12688, 9784,29072, 9800,18832, 9808,10640, 9816,27024, 9832,22928,
    9840,14736, 9848,31120, 9864,17808, 9880,26000, 9896,21904, 9904,13712,
    9912,30096, 9928,19856, 9936,11664, 9944,28048, 9960,23952, 9968,15760,
    9976,32144, 9992,17296, 10008,25488, 10024,21392, 10032,13200, 10040,29584,
    10056,19344, 10064,11152, 10072,27536, 10088,23440, 10096,15248, 10104,31632,
    10120,18320, 10136,26512, 10152,22416, 10160,14224, 10168,30608, 10184,20368,
    10192,12176, 10200,28560, 10216,24464, 10224,16272, 10232,32656, 10248,16464,
    10264,24656, 10280,20560, 10288,12368, 10296,28752, 10312,18512, 10328,26704,
    10344,22608, 10352,14416, 10360,30800, 10376,17488, 10392,25680, 10408,21584,
    10416,13392, 10424,29776, 10440,19536, 10448,11344, 10456,27728, 10472,23632,
    10480,15440, 10488,31824, 10504,16976, 10520,25168, 10536,21072, 10544,12880,
    10552,29264, 10568,19024, 10576,10832, 10584,27216, 10600,23120, 10608,14928,
    10616,31312, 10632,18000, 10648,26192, 10664,22096, 10672,13904, 10680,30288,
    10696,20048, 10704,11856, 10712,28240, 10728,24144, 10736,15952, 10744,32336,
    10760,16720, 10776,24912, 10792,20816, 10800,12624, 10808,29008, 10824,18768,
    10840,26960, 10856,22864, 10864,14672, 10872,31056, 10888,17744, 10904,25936,
    10920,21840, 10928,13648, 10936,30032, 10952,19792, 10960,11600, 10968,27984,
    10984,23888, 10992,15696, 11000,32080, 11016,17232, 11032,25424, 11048,21328,
    11056,13136, 11064,29520, 11080,19280, 11096,27472, 11112,23376, 11120,15184,
    11128,31568, 11144,18256, 11160,26448, 11176,22352, 11184,14160, 11192,30544,
    11208,20304, 11216,12112, 11224,28496, 11240,24400, 11248,16208, 11256,32592,
    11272,16592, 11288,24784, 11304,20688, 11312,12496, 11320,28880, 11336,18640,
    11352,26832, 11368,22736, 11376,14544, 11384,30928, 11400,17616, 11416,25808,
    11432,21712, 11440,13520, 11448,29904, 11464,19664, 11480,27856, 11496,23760,
    11504,15568, 11512,31952, 11528,17104, 11544,25296, 11560,21200, 11568,13008,
    11576,29392, 11592,19152, 11608,27344, 11624,23248, 11632,15056, 11640,31440,
    11656,18128, 11672,26320, 11688,22224, 11696,14032, 11704,30416, 11720,20176,
    11728,11984, 11736,28368, 11752,24272, 11760,16080, 11768,32464, 11784,16848,
    11800,25040, 11816,20944, 11824,12752, 11832,29136, 11848,18896, 11864,27088,
    11880,22992, 11888,14800, 11896,31184, 11912,17872, 11928,26064, 11944,21968,
    11952,13776, 11960,30160, 11976,19920, 11992,28112, 12008,24016, 12016,15824,
    12024,32208, 12040,17360, 12056,25552, 12072,21456, 12080,13264, 12088,29648,
    12104,19408, 12120,27600, 12136,23504, 12144,15312, 12152,31696, 12168,18384,
    12184,26576, 12200,22480, 12208,14288, 12216,30672, 12232,20432, 12248,28624,
    12264,24528, 12272,16336, 12280,32720, 12296,16432, 12312,24624, 12328,20528,
    12344,28720, 12360,18480, 12376,26672, 12392,22576, 12400,14384, 12408,30768,
    12424,17456, 12440,25648, 12456,21552, 12464,13360, 12472,29744, 12488,19504,
    12504,27696, 12520,23600, 12528,15408, 12536,31792, 12552,16944, 12568,25136,
    12584,21040, 12592,12848, 12600,29232, 12616,18992, 12632,27184, 12648,23088,
    12656,14896, 12664,31280, 12680,17968, 12696,26160, 12712,22064, 12720,13872,
    12728,30256, 12744,20016, 12760,28208, 12776,24112, 12784,15920, 12792,32304,
    12808,16688, 12824,24880, 12840,20784, 12856,28976, 12872,18736, 12888,26928,
    12904,22832, 12912,14640, 12920,31024, 12936,17712, 12952,25904, 12968,21808,
    12976,13616, 12984,30000, 13000,19760, 13016,27952, 13032,23856, 13040,15664,
    13048,32048, 13064,17200, 13080,25392, 13096,21296, 13112,29488, 13128,19248,
    13144,27440, 13160,23344, 13168,15152, 13176,31536, 13192,18224, 13208,26416,
    13224,22320, 13232,14128, 13240,30512, 13256,20272, 13272,28464, 13288,24368,
    13296,16176, 13304,32560, 13320,16560, 13336,24752, 13352,20656, 13368,28848,
    13384,18608, 13400,26800, 13416,22704, 13424,14512, 13432,30896, 13448,17584,
    13464,25776, 13480,21680, 13496,29872, 13512,19632, 13528,27824, 13544,23728,
    13552,15536, 13560,31920, 13576,17072, 13592,25264, 13608,21168, 13624,29360,
    13640,19120, 13656,27312, 13672,23216, 13680,15024, 13688,31408, 13704,18096,
    13720,26288, 13736,22192, 13744,14000, 13752,30384, 13768,20144, 13784,28336,
    13800,24240, 13808,16048, 13816,32432, 13832,16816, 13848,25008, 13864,20912,
    13880,29104, 13896,18864, 13912,27056, 13928,22960, 13936,14768, 13944,31152,
    13960,17840, 13976,26032, 13992,21936, 14008,30128, 14024,19888, 14040,28080,
    14056,23984, 14064,15792, 14072,32176, 14088,17328, 14104,25520, 14120,21424,
    14136,29616, 14152,19376, 14168,27568, 14184,23472, 14192,15280, 14200,31664,
    14216,18352, 14232,26544, 14248,22448, 14264,30640, 14280,20400, 14296,28592,
    14312,24496, 14320,16304, 14328,32688, 14344,16496, 14360,24688, 14376,20592,
    14392,28784, 14408,18544, 14424,26736, 14440,22640, 14456,30832, 14472,17520,
    14488,25712, 14504,21616, 14520,29808, 14536,19568, 14552,27760, 14568,23664,
    14576,15472, 14584,31856, 14600,17008, 14616,25200, 14632,21104, 14648,29296,
    14664,19056, 14680,27248, 14696,23152, 14704,14960, 14712,31344, 14728,18032,
    14744,26224, 14760,22128, 14776,30320, 14792,20080, 14808,28272, 14824,24176,
    14832,15984, 14840,32368, 14856,16752, 14872,24944, 14888,20848, 14904,29040,
    14920,18800, 14936,26992, 14952,22896, 14968,31088, 14984,17776, 15000,25968,
    15016,21872, 15032,30064, 15048,19824, 15064,28016, 15080,23920, 15088,15728,
    15096,32112, 15112,17264, 15128,25456, 15144,21360, 15160,29552, 15176,19312,
    15192,27504, 15208,23408, 15224,31600, 15240,18288, 15256,26480, 15272,22384,
    15288,30576, 15304,20336, 15320,28528, 15336,24432, 15344,16240, 15352,32624,
    15368,16624, 15384,24816, 15400,20720, 15416,28912, 15432,18672, 15448,26864,
    15464,22768, 15480,30960, 15496,17648, 15512,25840, 15528,21744, 15544,29936,
    15560,19696, 15576,27888, 15592,23792, 15608,31984, 15624,17136, 15640,25328,
    15656,21232, 15672,29424, 15688,19184, 15704,27376, 15720,23280, 15736,31472,
    15752,18160, 15768,26352, 15784,22256, 15800,30448, 15816,20208, 15832,28400,
    15848,24304, 15856,16112, 15864,32496, 15880,16880, 15896,25072, 15912,20976,
    15928,29168, 15944,18928, 15960,27120, 15976,23024, 15992,31216, 16008,17904,
    16024,26096, 16040,22000, 16056,30192, 16072,19952, 16088,28144, 16104,24048,
    16120,32240, 16136,17392, 16152,25584, 16168,21488, 16184,29680, 16200,19440,
    16216,27632, 16232,23536, 16248,31728, 16264,18416, 16280,26608, 16296,22512,
    16312,30704, 16328,20464, 16344,28656, 16360,24560, 16376,32752, 16408,24584,
    16424,20488, 16440,28680, 16456,18440, 16472,26632, 16488,22536, 16504,30728,
    16520,17416, 16536,25608, 16552,21512, 16568,29704, 16584,19464, 16600,27656,
    16616,23560, 16632,31752, 16648,16904, 16664,25096, 16680,21000, 16696,29192,
    16712,18952, 16728,27144, 16744,23048, 16760,31240, 16776,17928, 16792,26120,
    16808,22024, 16824,30216, 16840,19976, 16856,28168, 16872,24072, 16888,32264,
    16920,24840, 16936,20744, 16952,28936, 16968,18696, 16984,26888, 17000,22792,
    17016,30984, 17032,17672, 17048,25864, 17064,21768, 17080,29960, 17096,19720,
    17112,27912, 17128,23816, 17144,32008, 17176,25352, 17192,21256, 17208,29448,
    17224,19208, 17240,27400, 17256,23304, 17272,31496, 17288,18184, 17304,26376,
    17320,22280, 17336,30472, 17352,20232, 17368,28424, 17384,24328, 17400,32520,
    17432,24712, 17448,20616, 17464,28808, 17480,18568, 17496,26760, 17512,22664,
    17528,30856, 17560,25736, 17576,21640, 17592,29832, 17608,19592, 17624,27784,
    17640,23688, 17656,31880, 17688,25224, 17704,21128, 17720,29320, 17736,19080,
    17752,27272, 17768,23176, 17784,31368, 17800,18056, 17816,26248, 17832,22152,
    17848,30344, 17864,20104, 17880,28296, 17896,24200, 17912,32392, 17944,24968,
    17960,20872, 17976,29064, 17992,18824, 18008,27016, 18024,22920, 18040,31112,
    18072,25992, 18088,21896, 18104,30088, 18120,19848, 18136,28040, 18152,23944,
    18168,32136, 18200,25480, 18216,21384, 18232,29576, 18248,19336, 18264,27528,
    18280,23432, 18296,31624, 18328,26504, 18344,22408, 18360,30600, 18376,20360,
    18392,28552, 18408,24456, 18424,32648, 18456,24648, 18472,20552, 18488,28744,
    18520,26696, 18536,22600, 18552,30792, 18584,25672, 18600,21576, 18616,29768,
    18632,19528, 18648,27720, 18664,23624, 18680,31816, 18712,25160, 18728,21064,
    18744,29256, 18760,19016, 18776,27208, 18792,23112, 18808,31304, 18840,26184,
    18856,22088, 18872,30280, 18888,20040, 18904,28232, 18920,24136, 18936,32328,
    18968,24904, 18984,20808, 19000,29000, 19032,26952, 19048,22856, 19064,31048,
    19096,25928, 19112,21832, 19128,30024, 19144,19784, 19160,27976, 19176,23880,
    19192,32072, 19224,25416, 19240,21320, 19256,29512, 19288,27464, 19304,23368,
    19320,31560, 19352,26440, 19368,22344, 19384,30536, 19400,20296, 19416,28488,
    19432,24392, 19448,32584, 19480,24776, 19496,20680, 19512,28872, 19544,26824,
    19560,22728, 19576,30920, 19608,25800, 19624,21704, 19640,29896, 19672,27848,
    19688,23752, 19704,31944, 19736,25288, 19752,21192, 19768,29384, 19800,27336,
    19816,23240, 19832,31432, 19864,26312, 19880,22216, 19896,30408, 19912,20168,
    19928,28360, 19944,24264, 19960,32456, 19992,25032, 20008,20936, 20024,29128,
    20056,27080, 20072,22984, 20088,31176, 20120,26056, 20136,21960, 20152,30152,
    20184,28104, 20200,24008, 20216,32200, 20248,25544, 20264,21448, 20280,29640,
    20312,27592, 20328,23496, 20344,31688, 20376,26568, 20392,22472, 20408,30664,
    20440,28616, 20456,24520, 20472,32712, 20504,24616, 20536,28712, 20568,26664,
    20584,22568, 20600,30760, 20632,25640, 20648,21544, 20664,29736, 20696,27688,
    20712,23592, 20728,31784, 20760,25128, 20776,21032, 20792,29224, 20824,27176,
    20840,23080, 20856,31272, 20888,26152, 20904,22056, 20920,30248, 20952,28200,
    20968,24104, 20984,32296, 21016,24872, 21048,28968, 21080,26920, 21096,22824,
    21112,31016, 21144,25896, 21160,21800, 21176,29992, 21208,27944, 21224,23848,
    21240,32040, 21272,25384, 21304,29480, 21336,27432, 21352,23336, 21368,31528,
    21400,26408, 21416,22312, 21432,30504, 21464,28456, 21480,24360, 21496,32552,
    21528,24744, 21560,28840, 21592,26792, 21608,22696, 21624,30888, 21656,25768,
    21688,29864, 21720,27816, 21736,23720, 21752,31912, 21784,25256, 21816,29352,
    21848,27304, 21864,23208, 21880,31400, 21912,26280, 21928,22184, 21944,30376,
    21976,28328, 21992,24232, 22008,32424, 22040,25000, 22072,29096, 22104,27048,
    22120,22952, 22136,31144, 22168,26024, 22200,30120, 22232,28072, 22248,23976,
    22264,32168, 22296,25512, 22328,29608, 22360,27560, 22376,23464, 22392,31656,
    22424,26536, 22456,30632, 22488,28584, 22504,24488, 22520,32680, 22552,24680,
    22584,28776, 22616,26728, 22648,30824, 22680,25704, 22712,29800, 22744,27752,
    22760,23656, 22776,31848, 22808,25192, 22840,29288, 22872,27240, 22888,23144,
    22904,31336, 22936,26216, 22968,30312, 23000,28264, 23016,24168, 23032,32360,
    23064,24936, 23096,29032, 23128,26984, 23160,31080, 23192,25960, 23224,30056,
    23256,28008, 23272,23912, 23288,32104, 23320,25448, 23352,29544, 23384,27496,
    23416,31592, 23448,26472, 23480,30568, 23512,28520, 23528,24424, 23544,32616,
    23576,24808, 23608,28904, 23640,26856, 23672,30952, 23704,25832, 23736,29928,
    23768,27880, 23800,31976, 23832,25320, 23864,29416, 23896,27368, 23928,31464,
    23960,26344, 23992,30440, 24024,28392, 24040,24296, 24056,32488, 24088,25064,
    24120,29160, 24152,27112, 24184,31208, 24216,26088, 24248,30184, 24280,28136,
    24312,32232, 24344,25576, 24376,29672, 24408,27624, 24440,31720, 24472,26600,
    24504,30696, 24536,28648, 24568,32744, 24632,28696, 24664,26648, 24696,30744,
    24728,25624, 24760,29720, 24792,27672, 24824,31768, 24856,25112, 24888,29208,
    24920,27160, 24952,31256, 24984,26136, 25016,30232, 25048,28184, 25080,32280,
    25144,28952, 25176,26904, 25208,31000, 25240,25880, 25272,29976, 25304,27928,
    25336,32024, 25400,29464, 25432,27416, 25464,31512, 25496,26392, 25528,30488,
    25560,28440, 25592,32536, 25656,28824, 25688,26776, 25720,30872, 25784,29848,
    25816,27800, 25848,31896, 25912,29336, 25944,27288, 25976,31384, 26008,26264,
    26040,30360, 26072,28312, 26104,32408, 26168,29080, 26200,27032, 26232,31128,
    26296,30104, 26328,28056, 26360,32152, 26424,29592, 26456,27544, 26488,31640,
    26552,30616, 26584,28568, 26616,32664, 26680,28760, 26744,30808, 26808,29784,
    26840,27736, 26872,31832, 26936,29272, 26968,27224, 27000,31320, 27064,30296,
    27096,28248, 27128,32344, 27192,29016, 27256,31064, 27320,30040, 27352,27992,
    27384,32088, 27448,29528, 27512,31576, 27576,30552, 27608,28504, 27640,32600,
    27704,28888, 27768,30936, 27832,29912, 27896,31960, 27960,29400, 28024,31448,
    28088,30424, 28120,28376, 28152,32472, 28216,29144, 28280,31192, 28344,30168,
    28408,32216, 28472,29656, 28536,31704, 28600,30680, 28664,32728, 28792,30776,
    28856,29752, 28920,31800, 28984,29240, 29048,31288, 29112,30264, 29176,32312,
    29304,31032, 29368,30008, 29432,32056, 29560,31544, 29624,30520, 29688,32568,
    29816,30904, 29944,31928, 30072,31416, 30136,30392, 30200,32440, 30328,31160,
    30456,32184, 30584,31672, 30712,32696, 30968,31864, 31096,31352, 31224,32376,
    31480,32120, 31736,32632, 32248,32504
};

/**
* \par
* Example code for Floating-point RFFT Twiddle factors Generation:
* \par
* <pre>TW = exp(2*pi*i*[0:L/2-1]/L - pi/2*i).' </pre>
* \par
* Real and Imag values are in interleaved fashion
*/
const float32_t twiddleCoef_rfft_32[32] = {
    0.000000000f,  1.000000000f,
    0.195090322f,  0.980785280f,
    0.382683432f,  0.923879533f,
    0.555570233f,  0.831469612f,
    0.707106781f,  0.707106781f,
    0.831469612f,  0.555570233f,
    0.923879533f,  0.382683432f,
    0.980785280f,  0.195090322f,
    1.000000000f,  0.000000000f,
    0.980785280f, -0.195090322f,
    0.923879533f, -0.382683432f,
    0.831469612f, -0.555570233f,
    0.707106781f, -0.707106781f,
    0.555570233f, -0.831469612f,
    0.382683432f, -0.923879533f,
    0.195090322f, -0.980785280f
};

const float32_t twiddleCoef_rfft_64[64] = {
    0.000000000000000f,  1.000000000000000f,
    0.098017140329561f,  0.995184726672197f,
    0.195090322016128f,  0.980785280403230f,
    0.290284677254462f,  0.956940335732209f,
    0.382683432365090f,  0.923879532511287f,
    0.471396736825998f,  0.881921264348355f,
    0.555570233019602f,  0.831469612302545f,
    0.634393284163645f,  0.773010453362737f,
    0.707106781186547f,  0.707106781186548f,
    0.773010453362737f,  0.634393284163645f,
    0.831469612302545f,  0.555570233019602f,
    0.881921264348355f,  0.471396736825998f,
    0.923879532511287f,  0.382683432365090f,
    0.956940335732209f,  0.290284677254462f,
    0.980785280403230f,  0.195090322016128f,
    0.995184726672197f,  0.098017140329561f,
    1.000000000000000f,  0.000000000000000f,
    0.995184726672197f, -0.098017140329561f,
    0.980785280403230f, -0.195090322016128f,
    0.956940335732209f, -0.290284677254462f,
    0.923879532511287f, -0.382683432365090f,
    0.881921264348355f, -0.471396736825998f,
    0.831469612302545f, -0.555570233019602f,
    0.773010453362737f, -0.634393284163645f,
    0.707106781186548f, -0.707106781186547f,
    0.634393284163645f, -0.773010453362737f,
    0.555570233019602f, -0.831469612302545f,
    0.471396736825998f, -0.881921264348355f,
    0.382683432365090f, -0.923879532511287f,
    0.290284677254462f, -0.956940335732209f,
    0.195090322016129f, -0.980785280403230f,
    0.098017140329561f, -0.995184726672197f
};

const float32_t twiddleCoef_rfft_128[128] = {
    0.000000000f,  1.000000000f,
    0.049067674f,  0.998795456f,
    0.098017140f,  0.995184727f,
    0.146730474f,  0.989176510f,
    0.195090322f,  0.980785280f,
    0.242980180f,  0.970031253f,
    0.290284677f,  0.956940336f,
    0.336889853f,  0.941544065f,
    0.382683432f,  0.923879533f,
    0.427555093f,  0.903989293f,
    0.471396737f,  0.881921264f,
    0.514102744f,  0.857728610f,
    0.555570233f,  0.831469612f,
    0.595699304f,  0.803207531f,
    0.634393284f,  0.773010453f,
    0.671558955f,  0.740951125f,
    0.707106781f,  0.707106781f,
    0.740951125f,  0.671558955f,
    0.773010453f,  0.634393284f,
    0.803207531f,  0.595699304f,
    0.831469612f,  0.555570233f,
    0.857728610f,  0.514102744f,
    0.881921264f,  0.471396737f,
    0.903989293f,  0.427555093f,
    0.923879533f,  0.382683432f,
    0.941544065f,  0.336889853f,
    0.956940336f,  0.290284677f,
    0.970031253f,  0.242980180f,
    0.980785280f,  0.195090322f,
    0.989176510f,  0.146730474f,
    0.995184727f,  0.098017140f,
    0.998795456f,  0.049067674f,
    1.000000000f,  0.000000000f,
    0.998795456f, -0.049067674f,
    0.995184727f, -0.098017140f,
    0.989176510f, -0.146730474f,
    0.980785280f, -0.195090322f,
    0.970031253f, -0.242980180f,
    0.956940336f, -0.290284677f,
    0.941544065f, -0.336889853f,
    0.923879533f, -0.382683432f,
    0.903989293f, -0.427555093f,
    0.881921264f, -0.471396737f,
    0.857728610f, -0.514102744f,
    0.831469612f, -0.555570233f,
    0.803207531f, -0.595699304f,
    0.773010453f, -0.634393284f,
    0.740951125f, -0.671558955f,
    0.707106781f, -0.707106781f,
    0.671558955f, -0.740951125f,
    0.634393284f, -0.773010453f,
    0.595699304f, -0.803207531f,
    0.555570233f, -0.831469612f,
    0.514102744f, -0.857728610f,
    0.471396737f, -0.881921264f,
    0.427555093f, -0.903989293f,
    0.382683432f, -0.923879533f,
    0.336889853f, -0.941544065f,
    0.290284677f, -0.956940336f,
    0.242980180f, -0.970031253f,
    0.195090322f, -0.980785280f,
    0.146730474f, -0.989176510f,
    0.098017140f, -0.995184727f,
    0.049067674f, -0.998795456f
};

const float32_t twiddleCoef_rfft_256[256] = {
    0.000000000f,  1.000000000f,
    0.024541229f,  0.999698819f,
    0.049067674f,  0.998795456f,
    0.073564564f,  0.997290457f,
    0.098017140f,  0.995184727f,
    0.122410675f,  0.992479535f,
    0.146730474f,  0.989176510f,
    0.170961889f,  0.985277642f,
    0.195090322f,  0.980785280f,
    0.219101240f,  0.975702130f,
    0.242980180f,  0.970031253f,
    0.266712757f,  0.963776066f,
    0.290284677f,  0.956940336f,
    0.313681740f,  0.949528181f,
    0.336889853f,  0.941544065f,
    0.359895037f,  0.932992799f,
    0.382683432f,  0.923879533f,
    0.405241314f,  0.914209756f,
    0.427555093f,  0.903989293f,
    0.449611330f,  0.893224301f,
    0.471396737f,  0.881921264f,
    0.492898192f,  0.870086991f,
    0.514102744f,  0.857728610f,
    0.534997620f,  0.844853565f,
    0.555570233f,  0.831469612f,
    0.575808191f,  0.817584813f,
    0.595699304f,  0.803207531f,
    0.615231591f,  0.788346428f,
    0.634393284f,  0.773010453f,
    0.653172843f,  0.757208847f,
    0.671558955f,  0.740951125f,
    0.689540545f,  0.724247083f,
    0.707106781f,  0.707106781f,
    0.724247083f,  0.689540545f,
    0.740951125f,  0.671558955f,
    0.757208847f,  0.653172843f,
    0.773010453f,  0.634393284f,
    0.788346428f,  0.615231591f,
    0.803207531f,  0.595699304f,
    0.817584813f,  0.575808191f,
    0.831469612f,  0.555570233f,
    0.844853565f,  0.534997620f,
    0.857728610f,  0.514102744f,
    0.870086991f,  0.492898192f,
    0.881921264f,  0.471396737f,
    0.893224301f,  0.449611330f,
    0.903989293f,  0.427555093f,
    0.914209756f,  0.405241314f,
    0.923879533f,  0.382683432f,
    0.932992799f,  0.359895037f,
    0.941544065f,  0.336889853f,
    0.949528181f,  0.313681740f,
    0.956940336f,  0.290284677f,
    0.963776066f,  0.266712757f,
    0.970031253f,  0.242980180f,
    0.975702130f,  0.219101240f,
    0.980785280f,  0.195090322f,
    0.985277642f,  0.170961889f,
    0.989176510f,  0.146730474f,
    0.992479535f,  0.122410675f,
    0.995184727f,  0.098017140f,
    0.997290457f,  0.073564564f,
    0.998795456f,  0.049067674f,
    0.999698819f,  0.024541229f,
    1.000000000f,  0.000000000f,
    0.999698819f, -0.024541229f,
    0.998795456f, -0.049067674f,
    0.997290457f, -0.073564564f,
    0.995184727f, -0.098017140f,
    0.992479535f, -0.122410675f,
    0.989176510f, -0.146730474f,
    0.985277642f, -0.170961889f,
    0.980785280f, -0.195090322f,
    0.975702130f, -0.219101240f,
    0.970031253f, -0.242980180f,
    0.963776066f, -0.266712757f,
    0.956940336f, -0.290284677f,
    0.949528181f, -0.313681740f,
    0.941544065f, -0.336889853f,
    0.932992799f, -0.359895037f,
    0.923879533f, -0.382683432f,
    0.914209756f, -0.405241314f,
    0.903989293f, -0.427555093f,
    0.893224301f, -0.449611330f,
    0.881921264f, -0.471396737f,
    0.870086991f, -0.492898192f,
    0.857728610f, -0.514102744f,
    0.844853565f, -0.534997620f,
    0.831469612f, -0.555570233f,
    0.817584813f, -0.575808191f,
    0.803207531f, -0.595699304f,
    0.788346428f, -0.615231591f,
    0.773010453f, -0.634393284f,
    0.757208847f, -0.653172843f,
    0.740951125f, -0.671558955f,
    0.724247083f, -0.689540545f,
    0.707106781f, -0.707106781f,
    0.689540545f, -0.724247083f,
    0.671558955f, -0.740951125f,
    0.653172843f, -0.757208847f,
    0.634393284f, -0.773010453f,
    0.615231591f, -0.788346428f,
    0.595699304f, -0.803207531f,
    0.575808191f, -0.817584813f,
    0.555570233f, -0.831469612f,
    0.534997620f, -0.844853565f,
    0.514102744f, -0.857728610f,
    0.492898192f, -0.870086991f,
    0.471396737f, -0.881921264f,
    0.449611330f, -0.893224301f,
    0.427555093f, -0.903989293f,
    0.405241314f, -0.914209756f,
    0.382683432f, -0.923879533f,
    0.359895037f, -0.932992799f,
    0.336889853f, -0.941544065f,
    0.313681740f, -0.949528181f,
    0.290284677f, -0.956940336f,
    0.266712757f, -0.963776066f,
    0.242980180f, -0.970031253f,
    0.219101240f, -0.975702130f,
    0.195090322f, -0.980785280f,
    0.170961889f, -0.985277642f,
    0.146730474f, -0.989176510f,
    0.122410675f, -0.992479535f,
    0.098017140f, -0.995184727f,
    0.073564564f, -0.997290457f,
    0.049067674f, -0.998795456f,
    0.024541229f, -0.999698819f
};

const float32_t twiddleCoef_rfft_512[512] = {
    0.000000000f,  1.000000000f,
    0.012271538f,  0.999924702f,
    0.024541229f,  0.999698819f,
    0.036807223f,  0.999322385f,
    0.049067674f,  0.998795456f,
    0.061320736f,  0.998118113f,
    0.073564564f,  0.997290457f,
    0.085797312f,  0.996312612f,
    0.098017140f,  0.995184727f,
    0.110222207f,  0.993906970f,
    0.122410675f,  0.992479535f,
    0.134580709f,  0.990902635f,
    0.146730474f,  0.989176510f,
    0.158858143f,  0.987301418f,
    0.170961889f,  0.985277642f,
    0.183039888f,  0.983105487f,
    0.195090322f,  0.980785280f,
    0.207111376f,  0.978317371f,
    0.219101240f,  0.975702130f,
    0.231058108f,  0.972939952f,
    0.242980180f,  0.970031253f,
    0.254865660f,  0.966976471f,
    0.266712757f,  0.963776066f,
    0.278519689f,  0.960430519f,
    0.290284677f,  0.956940336f,
    0.302005949f,  0.953306040f,
    0.313681740f,  0.949528181f,
    0.325310292f,  0.945607325f,
    0.336889853f,  0.941544065f,
    0.348418680f,  0.937339012f,
    0.359895037f,  0.932992799f,
    0.371317194f,  0.928506080f,
    0.382683432f,  0.923879533f,
    0.393992040f,  0.919113852f,
    0.405241314f,  0.914209756f,
    0.416429560f,  0.909167983f,
    0.427555093f,  0.903989293f,
    0.438616239f,  0.898674466f,
    0.449611330f,  0.893224301f,
    0.460538711f,  0.887639620f,
    0.471396737f,  0.881921264f,
    0.482183772f,  0.876070094f,
    0.492898192f,  0.870086991f,
    0.503538384f,  0.863972856f,
    0.514102744f,  0.857728610f,
    0.524589683f,  0.851355193f,
    0.534997620f,  0.844853565f,
    0.545324988f,  0.838224706f,
    0.555570233f,  0.831469612f,
    0.565731811f,  0.824589303f,
    0.575808191f,  0.817584813f,
    0.585797857f,  0.810457198f,
    0.595699304f,  0.803207531f,
    0.605511041f,  0.795836905f,
    0.615231591f,  0.788346428f,
    0.624859488f,  0.780737229f,
    0.634393284f,  0.773010453f,
    0.643831543f,  0.765167266f,
    0.653172843f,  0.757208847f,
    0.662415778f,  0.749136395f,
    0.671558955f,  0.740951125f,
    0.680600998f,  0.732654272f,
    0.689540545f,  0.724247083f,
    0.698376249f,  0.715730825f,
    0.707106781f,  0.707106781f,
    0.715730825f,  0.698376249f,
    0.724247083f,  0.689540545f,
    0.732654272f,  0.680600998f,
    0.740951125f,  0.671558955f,
    0.749136395f,  0.662415778f,
    0.757208847f,  0.653172843f,
    0.765167266f,  0.643831543f,
    0.773010453f,  0.634393284f,
    0.780737229f,  0.624859488f,
    0.788346428f,  0.615231591f,
    0.795836905f,  0.605511041f,
    0.803207531f,  0.595699304f,
    0.810457198f,  0.585797857f,
    0.817584813f,  0.575808191f,
    0.824589303f,  0.565731811f,
    0.831469612f,  0.555570233f,
    0.838224706f,  0.545324988f,
    0.844853565f,  0.534997620f,
    0.851355193f,  0.524589683f,
    0.857728610f,  0.514102744f,
    0.863972856f,  0.503538384f,
    0.870086991f,  0.492898192f,
    0.876070094f,  0.482183772f,
    0.881921264f,  0.471396737f,
    0.887639620f,  0.460538711f,
    0.893224301f,  0.449611330f,
    0.898674466f,  0.438616239f,
    0.903989293f,  0.427555093f,
    0.909167983f,  0.416429560f,
    0.914209756f,  0.405241314f,
    0.919113852f,  0.393992040f,
    0.923879533f,  0.382683432f,
    0.928506080f,  0.371317194f,
    0.932992799f,  0.359895037f,
    0.937339012f,  0.348418680f,
    0.941544065f,  0.336889853f,
    0.945607325f,  0.325310292f,
    0.949528181f,  0.313681740f,
    0.953306040f,  0.302005949f,
    0.956940336f,  0.290284677f,
    0.960430519f,  0.278519689f,
    0.963776066f,  0.266712757f,
    0.966976471f,  0.254865660f,
    0.970031253f,  0.242980180f,
    0.972939952f,  0.231058108f,
    0.975702130f,  0.219101240f,
    0.978317371f,  0.207111376f,
    0.980785280f,  0.195090322f,
    0.983105487f,  0.183039888f,
    0.985277642f,  0.170961889f,
    0.987301418f,  0.158858143f,
    0.989176510f,  0.146730474f,
    0.990902635f,  0.134580709f,
    0.992479535f,  0.122410675f,
    0.993906970f,  0.110222207f,
    0.995184727f,  0.098017140f,
    0.996312612f,  0.085797312f,
    0.997290457f,  0.073564564f,
    0.998118113f,  0.061320736f,
    0.998795456f,  0.049067674f,
    0.999322385f,  0.036807223f,
    0.999698819f,  0.024541229f,
    0.999924702f,  0.012271538f,
    1.000000000f,  0.000000000f,
    0.999924702f, -0.012271538f,
    0.999698819f, -0.024541229f,
    0.999322385f, -0.036807223f,
    0.998795456f, -0.049067674f,
    0.998118113f, -0.061320736f,
    0.997290457f, -0.073564564f,
    0.996312612f, -0.085797312f,
    0.995184727f, -0.098017140f,
    0.993906970f, -0.110222207f,
    0.992479535f, -0.122410675f,
    0.990902635f, -0.134580709f,
    0.989176510f, -0.146730474f,
    0.987301418f, -0.158858143f,
    0.985277642f, -0.170961889f,
    0.983105487f, -0.183039888f,
    0.980785280f, -0.195090322f,
    0.978317371f, -0.207111376f,
    0.975702130f, -0.219101240f,
    0.972939952f, -0.231058108f,
    0.970031253f, -0.242980180f,
    0.966976471f, -0.254865660f,
    0.963776066f, -0.266712757f,
    0.960430519f, -0.278519689f,
    0.956940336f, -0.290284677f,
    0.953306040f, -0.302005949f,
    0.949528181f, -0.313681740f,
    0.945607325f, -0.325310292f,
    0.941544065f, -0.336889853f,
    0.937339012f, -0.348418680f,
    0.932992799f, -0.359895037f,
    0.928506080f, -0.371317194f,
    0.923879533f, -0.382683432f,
    0.919113852f, -0.393992040f,
    0.914209756f, -0.405241314f,
    0.909167983f, -0.416429560f,
    0.903989293f, -0.427555093f,
    0.898674466f, -0.438616239f,
    0.893224301f, -0.449611330f,
    0.887639620f, -0.460538711f,
    0.881921264f, -0.471396737f,
    0.876070094f, -0.482183772f,
    0.870086991f, -0.492898192f,
    0.863972856f, -0.503538384f,
    0.857728610f, -0.514102744f,
    0.851355193f, -0.524589683f,
    0.844853565f, -0.534997620f,
    0.838224706f, -0.545324988f,
    0.831469612f, -0.555570233f,
    0.824589303f, -0.565731811f,
    0.817584813f, -0.575808191f,
    0.810457198f, -0.585797857f,
    0.803207531f, -0.595699304f,
    0.795836905f, -0.605511041f,
    0.788346428f, -0.615231591f,
    0.780737229f, -0.624859488f,
    0.773010453f, -0.634393284f,
    0.765167266f, -0.643831543f,
    0.757208847f, -0.653172843f,
    0.749136395f, -0.662415778f,
    0.740951125f, -0.671558955f,
    0.732654272f, -0.680600998f,
    0.724247083f, -0.689540545f,
    0.715730825f, -0.698376249f,
    0.707106781f, -0.707106781f,
    0.698376249f, -0.715730825f,
    0.689540545f, -0.724247083f,
    0.680600998f, -0.732654272f,
    0.671558955f, -0.740951125f,
    0.662415778f, -0.749136395f,
    0.653172843f, -0.757208847f,
    0.643831543f, -0.765167266f,
    0.634393284f, -0.773010453f,
    0.624859488f, -0.780737229f,
    0.615231591f, -0.788346428f,
    0.605511041f, -0.795836905f,
    0.595699304f, -0.803207531f,
    0.585797857f, -0.810457198f,
    0.575808191f, -0.817584813f,
    0.565731811f, -0.824589303f,
    0.555570233f, -0.831469612f,
    0.545324988f, -0.838224706f,
    0.534997620f, -0.844853565f,
    0.524589683f, -0.851355193f,
    0.514102744f, -0.857728610f,
    0.503538384f, -0.863972856f,
    0.492898192f, -0.870086991f,
    0.482183772f, -0.876070094f,
    0.471396737f, -0.881921264f,
    0.460538711f, -0.887639620f,
    0.449611330f, -0.893224301f,
    0.438616239f, -0.898674466f,
    0.427555093f, -0.903989293f,
    0.416429560f, -0.909167983f,
    0.405241314f, -0.914209756f,
    0.393992040f, -0.919113852f,
    0.382683432f, -0.923879533f,
    0.371317194f, -0.928506080f,
    0.359895037f, -0.932992799f,
    0.348418680f, -0.937339012f,
    0.336889853f, -0.941544065f,
    0.325310292f, -0.945607325f,
    0.313681740f, -0.949528181f,
    0.302005949f, -0.953306040f,
    0.290284677f, -0.956940336f,
    0.278519689f, -0.960430519f,
    0.266712757f, -0.963776066f,
    0.254865660f, -0.966976471f,
    0.242980180f, -0.970031253f,
    0.231058108f, -0.972939952f,
    0.219101240f, -0.975702130f,
    0.207111376f, -0.978317371f,
    0.195090322f, -0.980785280f,
    0.183039888f, -0.983105487f,
    0.170961889f, -0.985277642f,
    0.158858143f, -0.987301418f,
    0.146730474f, -0.989176510f,
    0.134580709f, -0.990902635f,
    0.122410675f, -0.992479535f,
    0.110222207f, -0.993906970f,
    0.098017140f, -0.995184727f,
    0.085797312f, -0.996312612f,
    0.073564564f, -0.997290457f,
    0.061320736f, -0.998118113f,
    0.049067674f, -0.998795456f,
    0.036807223f, -0.999322385f,
    0.024541229f, -0.999698819f,
    0.012271538f, -0.999924702f
};

const float32_t twiddleCoef_rfft_1024[1024] = {
    0.000000000f,  1.000000000f,
    0.006135885f,  0.999981175f,
    0.012271538f,  0.999924702f,
    0.018406730f,  0.999830582f,
    0.024541229f,  0.999698819f,
    0.030674803f,  0.999529418f,
    0.036807223f,  0.999322385f,
    0.042938257f,  0.999077728f,
    0.049067674f,  0.998795456f,
    0.055195244f,  0.998475581f,
    0.061320736f,  0.998118113f,
    0.067443920f,  0.997723067f,
    0.073564564f,  0.997290457f,
    0.079682438f,  0.996820299f,
    0.085797312f,  0.996312612f,
    0.091908956f,  0.995767414f,
    0.098017140f,  0.995184727f,
    0.104121634f,  0.994564571f,
    0.110222207f,  0.993906970f,
    0.116318631f,  0.993211949f,
    0.122410675f,  0.992479535f,
    0.128498111f,  0.991709754f,
    0.134580709f,  0.990902635f,
    0.140658239f,  0.990058210f,
    0.146730474f,  0.989176510f,
    0.152797185f,  0.988257568f,
    0.158858143f,  0.987301418f,
    0.164913120f,  0.986308097f,
    0.170961889f,  0.985277642f,
    0.177004220f,  0.984210092f,
    0.183039888f,  0.983105487f,
    0.189068664f,  0.981963869f,
    0.195090322f,  0.980785280f,
    0.201104635f,  0.979569766f,
    0.207111376f,  0.978317371f,
    0.213110320f,  0.977028143f,
    0.219101240f,  0.975702130f,
    0.225083911f,  0.974339383f,
    0.231058108f,  0.972939952f,
    0.237023606f,  0.971503891f,
    0.242980180f,  0.970031253f,
    0.248927606f,  0.968522094f,
    0.254865660f,  0.966976471f,
    0.260794118f,  0.965394442f,
    0.266712757f,  0.963776066f,
    0.272621355f,  0.962121404f,
    0.278519689f,  0.960430519f,
    0.284407537f,  0.958703475f,
    0.290284677f,  0.956940336f,
    0.296150888f,  0.955141168f,
    0.302005949f,  0.953306040f,
    0.307849640f,  0.951435021f,
    0.313681740f,  0.949528181f,
    0.319502031f,  0.947585591f,
    0.325310292f,  0.945607325f,
    0.331106306f,  0.943593458f,
    0.336889853f,  0.941544065f,
    0.342660717f,  0.939459224f,
    0.348418680f,  0.937339012f,
    0.354163525f,  0.935183510f,
    0.359895037f,  0.932992799f,
    0.365612998f,  0.930766961f,
    0.371317194f,  0.928506080f,
    0.377007410f,  0.926210242f,
    0.382683432f,  0.923879533f,
    0.388345047f,  0.921514039f,
    0.393992040f,  0.919113852f,
    0.399624200f,  0.916679060f,
    0.405241314f,  0.914209756f,
    0.410843171f,  0.911706032f,
    0.416429560f,  0.909167983f,
    0.422000271f,  0.906595705f,
    0.427555093f,  0.903989293f,
    0.433093819f,  0.901348847f,
    0.438616239f,  0.898674466f,
    0.444122145f,  0.895966250f,
    0.449611330f,  0.893224301f,
    0.455083587f,  0.890448723f,
    0.460538711f,  0.887639620f,
    0.465976496f,  0.884797098f,
    0.471396737f,  0.881921264f,
    0.476799230f,  0.879012226f,
    0.482183772f,  0.876070094f,
    0.487550160f,  0.873094978f,
    0.492898192f,  0.870086991f,
    0.498227667f,  0.867046246f,
    0.503538384f,  0.863972856f,
    0.508830143f,  0.860866939f,
    0.514102744f,  0.857728610f,
    0.519355990f,  0.854557988f,
    0.524589683f,  0.851355193f,
    0.529803625f,  0.848120345f,
    0.534997620f,  0.844853565f,
    0.540171473f,  0.841554977f,
    0.545324988f,  0.838224706f,
    0.550457973f,  0.834862875f,
    0.555570233f,  0.831469612f,
    0.560661576f,  0.828045045f,
    0.565731811f,  0.824589303f,
    0.570780746f,  0.821102515f,
    0.575808191f,  0.817584813f,
    0.580813958f,  0.814036330f,
    0.585797857f,  0.810457198f,
    0.590759702f,  0.806847554f,
    0.595699304f,  0.803207531f,
    0.600616479f,  0.799537269f,
    0.605511041f,  0.795836905f,
    0.610382806f,  0.792106577f,
    0.615231591f,  0.788346428f,
    0.620057212f,  0.784556597f,
    0.624859488f,  0.780737229f,
    0.629638239f,  0.776888466f,
    0.634393284f,  0.773010453f,
    0.639124445f,  0.769103338f,
    0.643831543f,  0.765167266f,
    0.648514401f,  0.761202385f,
    0.653172843f,  0.757208847f,
    0.657806693f,  0.753186799f,
    0.662415778f,  0.749136395f,
    0.666999922f,  0.745057785f,
    0.671558955f,  0.740951125f,
    0.676092704f,  0.736816569f,
    0.680600998f,  0.732654272f,
    0.685083668f,  0.728464390f,
    0.689540545f,  0.724247083f,
    0.693971461f,  0.720002508f,
    0.698376249f,  0.715730825f,
    0.702754744f,  0.711432196f,
    0.707106781f,  0.707106781f,
    0.711432196f,  0.702754744f,
    0.715730825f,  0.698376249f,
    0.720002508f,  0.693971461f,
    0.724247083f,  0.689540545f,
    0.728464390f,  0.685083668f,
    0.732654272f,  0.680600998f,
    0.736816569f,  0.676092704f,
    0.740951125f,  0.671558955f,
    0.745057785f,  0.666999922f,
    0.749136395f,  0.662415778f,
    0.753186799f,  0.657806693f,
    0.757208847f,  0.653172843f,
    0.761202385f,  0.648514401f,
    0.765167266f,  0.643831543f,
    0.769103338f,  0.639124445f,
    0.773010453f,  0.634393284f,
    0.776888466f,  0.629638239f,
    0.780737229f,  0.624859488f,
    0.784556597f,  0.620057212f,
    0.788346428f,  0.615231591f,
    0.792106577f,  0.610382806f,
    0.795836905f,  0.605511041f,
    0.799537269f,  0.600616479f,
    0.803207531f,  0.595699304f,
    0.806847554f,  0.590759702f,
    0.810457198f,  0.585797857f,
    0.814036330f,  0.580813958f,
    0.817584813f,  0.575808191f,
    0.821102515f,  0.570780746f,
    0.824589303f,  0.565731811f,
    0.828045045f,  0.560661576f,
    0.831469612f,  0.555570233f,
    0.834862875f,  0.550457973f,
    0.838224706f,  0.545324988f,
    0.841554977f,  0.540171473f,
    0.844853565f,  0.534997620f,
    0.848120345f,  0.529803625f,
    0.851355193f,  0.524589683f,
    0.854557988f,  0.519355990f,
    0.857728610f,  0.514102744f,
    0.860866939f,  0.508830143f,
    0.863972856f,  0.503538384f,
    0.867046246f,  0.498227667f,
    0.870086991f,  0.492898192f,
    0.873094978f,  0.487550160f,
    0.876070094f,  0.482183772f,
    0.879012226f,  0.476799230f,
    0.881921264f,  0.471396737f,
    0.884797098f,  0.465976496f,
    0.887639620f,  0.460538711f,
    0.890448723f,  0.455083587f,
    0.893224301f,  0.449611330f,
    0.895966250f,  0.444122145f,
    0.898674466f,  0.438616239f,
    0.901348847f,  0.433093819f,
    0.903989293f,  0.427555093f,
    0.906595705f,  0.422000271f,
    0.909167983f,  0.416429560f,
    0.911706032f,  0.410843171f,
    0.914209756f,  0.405241314f,
    0.916679060f,  0.399624200f,
    0.919113852f,  0.393992040f,
    0.921514039f,  0.388345047f,
    0.923879533f,  0.382683432f,
    0.926210242f,  0.377007410f,
    0.928506080f,  0.371317194f,
    0.930766961f,  0.365612998f,
    0.932992799f,  0.359895037f,
    0.935183510f,  0.354163525f,
    0.937339012f,  0.348418680f,
    0.939459224f,  0.342660717f,
    0.941544065f,  0.336889853f,
    0.943593458f,  0.331106306f,
    0.945607325f,  0.325310292f,
    0.947585591f,  0.319502031f,
    0.949528181f,  0.313681740f,
    0.951435021f,  0.307849640f,
    0.953306040f,  0.302005949f,
    0.955141168f,  0.296150888f,
    0.956940336f,  0.290284677f,
    0.958703475f,  0.284407537f,
    0.960430519f,  0.278519689f,
    0.962121404f,  0.272621355f,
    0.963776066f,  0.266712757f,
    0.965394442f,  0.260794118f,
    0.966976471f,  0.254865660f,
    0.968522094f,  0.248927606f,
    0.970031253f,  0.242980180f,
    0.971503891f,  0.237023606f,
    0.972939952f,  0.231058108f,
    0.974339383f,  0.225083911f,
    0.975702130f,  0.219101240f,
    0.977028143f,  0.213110320f,
    0.978317371f,  0.207111376f,
    0.979569766f,  0.201104635f,
    0.980785280f,  0.195090322f,
    0.981963869f,  0.189068664f,
    0.983105487f,  0.183039888f,
    0.984210092f,  0.177004220f,
    0.985277642f,  0.170961889f,
    0.986308097f,  0.164913120f,
    0.987301418f,  0.158858143f,
    0.988257568f,  0.152797185f,
    0.989176510f,  0.146730474f,
    0.990058210f,  0.140658239f,
    0.990902635f,  0.134580709f,
    0.991709754f,  0.128498111f,
    0.992479535f,  0.122410675f,
    0.993211949f,  0.116318631f,
    0.993906970f,  0.110222207f,
    0.994564571f,  0.104121634f,
    0.995184727f,  0.098017140f,
    0.995767414f,  0.091908956f,
    0.996312612f,  0.085797312f,
    0.996820299f,  0.079682438f,
    0.997290457f,  0.073564564f,
    0.997723067f,  0.067443920f,
    0.998118113f,  0.061320736f,
    0.998475581f,  0.055195244f,
    0.998795456f,  0.049067674f,
    0.999077728f,  0.042938257f,
    0.999322385f,  0.036807223f,
    0.999529418f,  0.030674803f,
    0.999698819f,  0.024541229f,
    0.999830582f,  0.018406730f,
    0.999924702f,  0.012271538f,
    0.999981175f,  0.006135885f,
    1.000000000f,  0.000000000f,
    0.999981175f, -0.006135885f,
    0.999924702f, -0.012271538f,
    0.999830582f, -0.018406730f,
    0.999698819f, -0.024541229f,
    0.999529418f, -0.030674803f,
    0.999322385f, -0.036807223f,
    0.999077728f, -0.042938257f,
    0.998795456f, -0.049067674f,
    0.998475581f, -0.055195244f,
    0.998118113f, -0.061320736f,
    0.997723067f, -0.067443920f,
    0.997290457f, -0.073564564f,
    0.996820299f, -0.079682438f,
    0.996312612f, -0.085797312f,
    0.995767414f, -0.091908956f,
    0.995184727f, -0.098017140f,
    0.994564571f, -0.104121634f,
    0.993906970f, -0.110222207f,
    0.993211949f, -0.116318631f,
    0.992479535f, -0.122410675f,
    0.991709754f, -0.128498111f,
    0.990902635f, -0.134580709f,
    0.990058210f, -0.140658239f,
    0.989176510f, -0.146730474f,
    0.988257568f, -0.152797185f,
    0.987301418f, -0.158858143f,
    0.986308097f, -0.164913120f,
    0.985277642f, -0.170961889f,
    0.984210092f, -0.177004220f,
    0.983105487f, -0.183039888f,
    0.981963869f, -0.189068664f,
    0.980785280f, -0.195090322f,
    0.979569766f, -0.201104635f,
    0.978317371f, -0.207111376f,
    0.977028143f, -0.213110320f,
    0.975702130f, -0.219101240f,
    0.974339383f, -0.225083911f,
    0.972939952f, -0.231058108f,
    0.971503891f, -0.237023606f,
    0.970031253f, -0.242980180f,
    0.968522094f, -0.248927606f,
    0.966976471f, -0.254865660f,
    0.965394442f, -0.260794118f,
    0.963776066f, -0.266712757f,
    0.962121404f, -0.272621355f,
    0.960430519f, -0.278519689f,
    0.958703475f, -0.284407537f,
    0.956940336f, -0.290284677f,
    0.955141168f, -0.296150888f,
    0.953306040f, -0.302005949f,
    0.951435021f, -0.307849640f,
    0.949528181f, -0.313681740f,
    0.947585591f, -0.319502031f,
    0.945607325f, -0.325310292f,
    0.943593458f, -0.331106306f,
    0.941544065f, -0.336889853f,
    0.939459224f, -0.342660717f,
    0.937339012f, -0.348418680f,
    0.935183510f, -0.354163525f,
    0.932992799f, -0.359895037f,
    0.930766961f, -0.365612998f,
    0.928506080f, -0.371317194f,
    0.926210242f, -0.377007410f,
    0.923879533f, -0.382683432f,
    0.921514039f, -0.388345047f,
    0.919113852f, -0.393992040f,
    0.916679060f, -0.399624200f,
    0.914209756f, -0.405241314f,
    0.911706032f, -0.410843171f,
    0.909167983f, -0.416429560f,
    0.906595705f, -0.422000271f,
    0.903989293f, -0.427555093f,
    0.901348847f, -0.433093819f,
    0.898674466f, -0.438616239f,
    0.895966250f, -0.444122145f,
    0.893224301f, -0.449611330f,
    0.890448723f, -0.455083587f,
    0.887639620f, -0.460538711f,
    0.884797098f, -0.465976496f,
    0.881921264f, -0.471396737f,
    0.879012226f, -0.476799230f,
    0.876070094f, -0.482183772f,
    0.873094978f, -0.487550160f,
    0.870086991f, -0.492898192f,
    0.867046246f, -0.498227667f,
    0.863972856f, -0.503538384f,
    0.860866939f, -0.508830143f,
    0.857728610f, -0.514102744f,
    0.854557988f, -0.519355990f,
    0.851355193f, -0.524589683f,
    0.848120345f, -0.529803625f,
    0.844853565f, -0.534997620f,
    0.841554977f, -0.540171473f,
    0.838224706f, -0.545324988f,
    0.834862875f, -0.550457973f,
    0.831469612f, -0.555570233f,
    0.828045045f, -0.560661576f,
    0.824589303f, -0.565731811f,
    0.821102515f, -0.570780746f,
    0.817584813f, -0.575808191f,
    0.814036330f, -0.580813958f,
    0.810457198f, -0.585797857f,
    0.806847554f, -0.590759702f,
    0.803207531f, -0.595699304f,
    0.799537269f, -0.600616479f,
    0.795836905f, -0.605511041f,
    0.792106577f, -0.610382806f,
    0.788346428f, -0.615231591f,
    0.784556597f, -0.620057212f,
    0.780737229f, -0.624859488f,
    0.776888466f, -0.629638239f,
    0.773010453f, -0.634393284f,
    0.769103338f, -0.639124445f,
    0.765167266f, -0.643831543f,
    0.761202385f, -0.648514401f,
    0.757208847f, -0.653172843f,
    0.753186799f, -0.657806693f,
    0.749136395f, -0.662415778f,
    0.745057785f, -0.666999922f,
    0.740951125f, -0.671558955f,
    0.736816569f, -0.676092704f,
    0.732654272f, -0.680600998f,
    0.728464390f, -0.685083668f,
    0.724247083f, -0.689540545f,
    0.720002508f, -0.693971461f,
    0.715730825f, -0.698376249f,
    0.711432196f, -0.702754744f,
    0.707106781f, -0.707106781f,
    0.702754744f, -0.711432196f,
    0.698376249f, -0.715730825f,
    0.693971461f, -0.720002508f,
    0.689540545f, -0.724247083f,
    0.685083668f, -0.728464390f,
    0.680600998f, -0.732654272f,
    0.676092704f, -0.736816569f,
    0.671558955f, -0.740951125f,
    0.666999922f, -0.745057785f,
    0.662415778f, -0.749136395f,
    0.657806693f, -0.753186799f,
    0.653172843f, -0.757208847f,
    0.648514401f, -0.761202385f,
    0.643831543f, -0.765167266f,
    0.639124445f, -0.769103338f,
    0.634393284f, -0.773010453f,
    0.629638239f, -0.776888466f,
    0.624859488f, -0.780737229f,
    0.620057212f, -0.784556597f,
    0.615231591f, -0.788346428f,
    0.610382806f, -0.792106577f,
    0.605511041f, -0.795836905f,
    0.600616479f, -0.799537269f,
    0.595699304f, -0.803207531f,
    0.590759702f, -0.806847554f,
    0.585797857f, -0.810457198f,
    0.580813958f, -0.814036330f,
    0.575808191f, -0.817584813f,
    0.570780746f, -0.821102515f,
    0.565731811f, -0.824589303f,
    0.560661576f, -0.828045045f,
    0.555570233f, -0.831469612f,
    0.550457973f, -0.834862875f,
    0.545324988f, -0.838224706f,
    0.540171473f, -0.841554977f,
    0.534997620f, -0.844853565f,
    0.529803625f, -0.848120345f,
    0.524589683f, -0.851355193f,
    0.519355990f, -0.854557988f,
    0.514102744f, -0.857728610f,
    0.508830143f, -0.860866939f,
    0.503538384f, -0.863972856f,
    0.498227667f, -0.867046246f,
    0.492898192f, -0.870086991f,
    0.487550160f, -0.873094978f,
    0.482183772f, -0.876070094f,
    0.476799230f, -0.879012226f,
    0.471396737f, -0.881921264f,
    0.465976496f, -0.884797098f,
    0.460538711f, -0.887639620f,
    0.455083587f, -0.890448723f,
    0.449611330f, -0.893224301f,
    0.444122145f, -0.895966250f,
    0.438616239f, -0.898674466f,
    0.433093819f, -0.901348847f,
    0.427555093f, -0.903989293f,
    0.422000271f, -0.906595705f,
    0.416429560f, -0.909167983f,
    0.410843171f, -0.911706032f,
    0.405241314f, -0.914209756f,
    0.399624200f, -0.916679060f,
    0.393992040f, -0.919113852f,
    0.388345047f, -0.921514039f,
    0.382683432f, -0.923879533f,
    0.377007410f, -0.926210242f,
    0.371317194f, -0.928506080f,
    0.365612998f, -0.930766961f,
    0.359895037f, -0.932992799f,
    0.354163525f, -0.935183510f,
    0.348418680f, -0.937339012f,
    0.342660717f, -0.939459224f,
    0.336889853f, -0.941544065f,
    0.331106306f, -0.943593458f,
    0.325310292f, -0.945607325f,
    0.319502031f, -0.947585591f,
    0.313681740f, -0.949528181f,
    0.307849640f, -0.951435021f,
    0.302005949f, -0.953306040f,
    0.296150888f, -0.955141168f,
    0.290284677f, -0.956940336f,
    0.284407537f, -0.958703475f,
    0.278519689f, -0.960430519f,
    0.272621355f, -0.962121404f,
    0.266712757f, -0.963776066f,
    0.260794118f, -0.965394442f,
    0.254865660f, -0.966976471f,
    0.248927606f, -0.968522094f,
    0.242980180f, -0.970031253f,
    0.237023606f, -0.971503891f,
    0.231058108f, -0.972939952f,
    0.225083911f, -0.974339383f,
    0.219101240f, -0.975702130f,
    0.213110320f, -0.977028143f,
    0.207111376f, -0.978317371f,
    0.201104635f, -0.979569766f,
    0.195090322f, -0.980785280f,
    0.189068664f, -0.981963869f,
    0.183039888f, -0.983105487f,
    0.177004220f, -0.984210092f,
    0.170961889f, -0.985277642f,
    0.164913120f, -0.986308097f,
    0.158858143f, -0.987301418f,
    0.152797185f, -0.988257568f,
    0.146730474f, -0.989176510f,
    0.140658239f, -0.990058210f,
    0.134580709f, -0.990902635f,
    0.128498111f, -0.991709754f,
    0.122410675f, -0.992479535f,
    0.116318631f, -0.993211949f,
    0.110222207f, -0.993906970f,
    0.104121634f, -0.994564571f,
    0.098017140f, -0.995184727f,
    0.091908956f, -0.995767414f,
    0.085797312f, -0.996312612f,
    0.079682438f, -0.996820299f,
    0.073564564f, -0.997290457f,
    0.067443920f, -0.997723067f,
    0.061320736f, -0.998118113f,
    0.055195244f, -0.998475581f,
    0.049067674f, -0.998795456f,
    0.042938257f, -0.999077728f,
    0.036807223f, -0.999322385f,
    0.030674803f, -0.999529418f,
    0.024541229f, -0.999698819f,
    0.018406730f, -0.999830582f,
    0.012271538f, -0.999924702f,
    0.006135885f, -0.999981175f
};

const float32_t twiddleCoef_rfft_2048[2048] = {
    0.000000000f,  1.000000000f,
    0.003067957f,  0.999995294f,
    0.006135885f,  0.999981175f,
    0.009203755f,  0.999957645f,
    0.012271538f,  0.999924702f,
    0.015339206f,  0.999882347f,
    0.018406730f,  0.999830582f,
    0.021474080f,  0.999769405f,
    0.024541229f,  0.999698819f,
    0.027608146f,  0.999618822f,
    0.030674803f,  0.999529418f,
    0.033741172f,  0.999430605f,
    0.036807223f,  0.999322385f,
    0.039872928f,  0.999204759f,
    0.042938257f,  0.999077728f,
    0.046003182f,  0.998941293f,
    0.049067674f,  0.998795456f,
    0.052131705f,  0.998640218f,
    0.055195244f,  0.998475581f,
    0.058258265f,  0.998301545f,
    0.061320736f,  0.998118113f,
    0.064382631f,  0.997925286f,
    0.067443920f,  0.997723067f,
    0.070504573f,  0.997511456f,
    0.073564564f,  0.997290457f,
    0.076623861f,  0.997060070f,
    0.079682438f,  0.996820299f,
    0.082740265f,  0.996571146f,
    0.085797312f,  0.996312612f,
    0.088853553f,  0.996044701f,
    0.091908956f,  0.995767414f,
    0.094963495f,  0.995480755f,
    0.098017140f,  0.995184727f,
    0.101069863f,  0.994879331f,
    0.104121634f,  0.994564571f,
    0.107172425f,  0.994240449f,
    0.110222207f,  0.993906970f,
    0.113270952f,  0.993564136f,
    0.116318631f,  0.993211949f,
    0.119365215f,  0.992850414f,
    0.122410675f,  0.992479535f,
    0.125454983f,  0.992099313f,
    0.128498111f,  0.991709754f,
    0.131540029f,  0.991310860f,
    0.134580709f,  0.990902635f,
    0.137620122f,  0.990485084f,
    0.140658239f,  0.990058210f,
    0.143695033f,  0.989622017f,
    0.146730474f,  0.989176510f,
    0.149764535f,  0.988721692f,
    0.152797185f,  0.988257568f,
    0.155828398f,  0.987784142f,
    0.158858143f,  0.987301418f,
    0.161886394f,  0.986809402f,
    0.164913120f,  0.986308097f,
    0.167938295f,  0.985797509f,
    0.170961889f,  0.985277642f,
    0.173983873f,  0.984748502f,
    0.177004220f,  0.984210092f,
    0.180022901f,  0.983662419f,
    0.183039888f,  0.983105487f,
    0.186055152f,  0.982539302f,
    0.189068664f,  0.981963869f,
    0.192080397f,  0.981379193f,
    0.195090322f,  0.980785280f,
    0.198098411f,  0.980182136f,
    0.201104635f,  0.979569766f,
    0.204108966f,  0.978948175f,
    0.207111376f,  0.978317371f,
    0.210111837f,  0.977677358f,
    0.213110320f,  0.977028143f,
    0.216106797f,  0.976369731f,
    0.219101240f,  0.975702130f,
    0.222093621f,  0.975025345f,
    0.225083911f,  0.974339383f,
    0.228072083f,  0.973644250f,
    0.231058108f,  0.972939952f,
    0.234041959f,  0.972226497f,
    0.237023606f,  0.971503891f,
    0.240003022f,  0.970772141f,
    0.242980180f,  0.970031253f,
    0.245955050f,  0.969281235f,
    0.248927606f,  0.968522094f,
    0.251897818f,  0.967753837f,
    0.254865660f,  0.966976471f,
    0.257831102f,  0.966190003f,
    0.260794118f,  0.965394442f,
    0.263754679f,  0.964589793f,
    0.266712757f,  0.963776066f,
    0.269668326f,  0.962953267f,
    0.272621355f,  0.962121404f,
    0.275571819f,  0.961280486f,
    0.278519689f,  0.960430519f,
    0.281464938f,  0.959571513f,
    0.284407537f,  0.958703475f,
    0.287347460f,  0.957826413f,
    0.290284677f,  0.956940336f,
    0.293219163f,  0.956045251f,
    0.296150888f,  0.955141168f,
    0.299079826f,  0.954228095f,
    0.302005949f,  0.953306040f,
    0.304929230f,  0.952375013f,
    0.307849640f,  0.951435021f,
    0.310767153f,  0.950486074f,
    0.313681740f,  0.949528181f,
    0.316593376f,  0.948561350f,
    0.319502031f,  0.947585591f,
    0.322407679f,  0.946600913f,
    0.325310292f,  0.945607325f,
    0.328209844f,  0.944604837f,
    0.331106306f,  0.943593458f,
    0.333999651f,  0.942573198f,
    0.336889853f,  0.941544065f,
    0.339776884f,  0.940506071f,
    0.342660717f,  0.939459224f,
    0.345541325f,  0.938403534f,
    0.348418680f,  0.937339012f,
    0.351292756f,  0.936265667f,
    0.354163525f,  0.935183510f,
    0.357030961f,  0.934092550f,
    0.359895037f,  0.932992799f,
    0.362755724f,  0.931884266f,
    0.365612998f,  0.930766961f,
    0.368466830f,  0.929640896f,
    0.371317194f,  0.928506080f,
    0.374164063f,  0.927362526f,
    0.377007410f,  0.926210242f,
    0.379847209f,  0.925049241f,
    0.382683432f,  0.923879533f,
    0.385516054f,  0.922701128f,
    0.388345047f,  0.921514039f,
    0.391170384f,  0.920318277f,
    0.393992040f,  0.919113852f,
    0.396809987f,  0.917900776f,
    0.399624200f,  0.916679060f,
    0.402434651f,  0.915448716f,
    0.405241314f,  0.914209756f,
    0.408044163f,  0.912962190f,
    0.410843171f,  0.911706032f,
    0.413638312f,  0.910441292f,
    0.416429560f,  0.909167983f,
    0.419216888f,  0.907886116f,
    0.422000271f,  0.906595705f,
    0.424779681f,  0.905296759f,
    0.427555093f,  0.903989293f,
    0.430326481f,  0.902673318f,
    0.433093819f,  0.901348847f,
    0.435857080f,  0.900015892f,
    0.438616239f,  0.898674466f,
    0.441371269f,  0.897324581f,
    0.444122145f,  0.895966250f,
    0.446868840f,  0.894599486f,
    0.449611330f,  0.893224301f,
    0.452349587f,  0.891840709f,
    0.455083587f,  0.890448723f,
    0.457813304f,  0.889048356f,
    0.460538711f,  0.887639620f,
    0.463259784f,  0.886222530f,
    0.465976496f,  0.884797098f,
    0.468688822f,  0.883363339f,
    0.471396737f,  0.881921264f,
    0.474100215f,  0.880470889f,
    0.476799230f,  0.879012226f,
    0.479493758f,  0.877545290f,
    0.482183772f,  0.876070094f,
    0.484869248f,  0.874586652f,
    0.487550160f,  0.873094978f,
    0.490226483f,  0.871595087f,
    0.492898192f,  0.870086991f,
    0.495565262f,  0.868570706f,
    0.498227667f,  0.867046246f,
    0.500885383f,  0.865513624f,
    0.503538384f,  0.863972856f,
    0.506186645f,  0.862423956f,
    0.508830143f,  0.860866939f,
    0.511468850f,  0.859301818f,
    0.514102744f,  0.857728610f,
    0.516731799f,  0.856147328f,
    0.519355990f,  0.854557988f,
    0.521975293f,  0.852960605f,
    0.524589683f,  0.851355193f,
    0.527199135f,  0.849741768f,
    0.529803625f,  0.848120345f,
    0.532403128f,  0.846490939f,
    0.534997620f,  0.844853565f,
    0.537587076f,  0.843208240f,
    0.540171473f,  0.841554977f,
    0.542750785f,  0.839893794f,
    0.545324988f,  0.838224706f,
    0.547894059f,  0.836547727f,
    0.550457973f,  0.834862875f,
    0.553016706f,  0.833170165f,
    0.555570233f,  0.831469612f,
    0.558118531f,  0.829761234f,
    0.560661576f,  0.828045045f,
    0.563199344f,  0.826321063f,
    0.565731811f,  0.824589303f,
    0.568258953f,  0.822849781f,
    0.570780746f,  0.821102515f,
    0.573297167f,  0.819347520f,
    0.575808191f,  0.817584813f,
    0.578313796f,  0.815814411f,
    0.580813958f,  0.814036330f,
    0.583308653f,  0.812250587f,
    0.585797857f,  0.810457198f,
    0.588281548f,  0.808656182f,
    0.590759702f,  0.806847554f,
    0.593232295f,  0.805031331f,
    0.595699304f,  0.803207531f,
    0.598160707f,  0.801376172f,
    0.600616479f,  0.799537269f,
    0.603066599f,  0.797690841f,
    0.605511041f,  0.795836905f,
    0.607949785f,  0.793975478f,
    0.610382806f,  0.792106577f,
    0.612810082f,  0.790230221f,
    0.615231591f,  0.788346428f,
    0.617647308f,  0.786455214f,
    0.620057212f,  0.784556597f,
    0.622461279f,  0.782650596f,
    0.624859488f,  0.780737229f,
    0.627251815f,  0.778816512f,
    0.629638239f,  0.776888466f,
    0.632018736f,  0.774953107f,
    0.634393284f,  0.773010453f,
    0.636761861f,  0.771060524f,
    0.639124445f,  0.769103338f,
    0.641481013f,  0.767138912f,
    0.643831543f,  0.765167266f,
    0.646176013f,  0.763188417f,
    0.648514401f,  0.761202385f,
    0.650846685f,  0.759209189f,
    0.653172843f,  0.757208847f,
    0.655492853f,  0.755201377f,
    0.657806693f,  0.753186799f,
    0.660114342f,  0.751165132f,
    0.662415778f,  0.749136395f,
    0.664710978f,  0.747100606f,
    0.666999922f,  0.745057785f,
    0.669282588f,  0.743007952f,
    0.671558955f,  0.740951125f,
    0.673829000f,  0.738887324f,
    0.676092704f,  0.736816569f,
    0.678350043f,  0.734738878f,
    0.680600998f,  0.732654272f,
    0.682845546f,  0.730562769f,
    0.685083668f,  0.728464390f,
    0.687315341f,  0.726359155f,
    0.689540545f,  0.724247083f,
    0.691759258f,  0.722128194f,
    0.693971461f,  0.720002508f,
    0.696177131f,  0.717870045f,
    0.698376249f,  0.715730825f,
    0.700568794f,  0.713584869f,
    0.702754744f,  0.711432196f,
    0.704934080f,  0.709272826f,
    0.707106781f,  0.707106781f,
    0.709272826f,  0.704934080f,
    0.711432196f,  0.702754744f,
    0.713584869f,  0.700568794f,
    0.715730825f,  0.698376249f,
    0.717870045f,  0.696177131f,
    0.720002508f,  0.693971461f,
    0.722128194f,  0.691759258f,
    0.724247083f,  0.689540545f,
    0.726359155f,  0.687315341f,
    0.728464390f,  0.685083668f,
    0.730562769f,  0.682845546f,
    0.732654272f,  0.680600998f,
    0.734738878f,  0.678350043f,
    0.736816569f,  0.676092704f,
    0.738887324f,  0.673829000f,
    0.740951125f,  0.671558955f,
    0.743007952f,  0.669282588f,
    0.745057785f,  0.666999922f,
    0.747100606f,  0.664710978f,
    0.749136395f,  0.662415778f,
    0.751165132f,  0.660114342f,
    0.753186799f,  0.657806693f,
    0.755201377f,  0.655492853f,
    0.757208847f,  0.653172843f,
    0.759209189f,  0.650846685f,
    0.761202385f,  0.648514401f,
    0.763188417f,  0.646176013f,
    0.765167266f,  0.643831543f,
    0.767138912f,  0.641481013f,
    0.769103338f,  0.639124445f,
    0.771060524f,  0.636761861f,
    0.773010453f,  0.634393284f,
    0.774953107f,  0.632018736f,
    0.776888466f,  0.629638239f,
    0.778816512f,  0.627251815f,
    0.780737229f,  0.624859488f,
    0.782650596f,  0.622461279f,
    0.784556597f,  0.620057212f,
    0.786455214f,  0.617647308f,
    0.788346428f,  0.615231591f,
    0.790230221f,  0.612810082f,
    0.792106577f,  0.610382806f,
    0.793975478f,  0.607949785f,
    0.795836905f,  0.605511041f,
    0.797690841f,  0.603066599f,
    0.799537269f,  0.600616479f,
    0.801376172f,  0.598160707f,
    0.803207531f,  0.595699304f,
    0.805031331f,  0.593232295f,
    0.806847554f,  0.590759702f,
    0.808656182f,  0.588281548f,
    0.810457198f,  0.585797857f,
    0.812250587f,  0.583308653f,
    0.814036330f,  0.580813958f,
    0.815814411f,  0.578313796f,
    0.817584813f,  0.575808191f,
    0.819347520f,  0.573297167f,
    0.821102515f,  0.570780746f,
    0.822849781f,  0.568258953f,
    0.824589303f,  0.565731811f,
    0.826321063f,  0.563199344f,
    0.828045045f,  0.560661576f,
    0.829761234f,  0.558118531f,
    0.831469612f,  0.555570233f,
    0.833170165f,  0.553016706f,
    0.834862875f,  0.550457973f,
    0.836547727f,  0.547894059f,
    0.838224706f,  0.545324988f,
    0.839893794f,  0.542750785f,
    0.841554977f,  0.540171473f,
    0.843208240f,  0.537587076f,
    0.844853565f,  0.534997620f,
    0.846490939f,  0.532403128f,
    0.848120345f,  0.529803625f,
    0.849741768f,  0.527199135f,
    0.851355193f,  0.524589683f,
    0.852960605f,  0.521975293f,
    0.854557988f,  0.519355990f,
    0.856147328f,  0.516731799f,
    0.857728610f,  0.514102744f,
    0.859301818f,  0.511468850f,
    0.860866939f,  0.508830143f,
    0.862423956f,  0.506186645f,
    0.863972856f,  0.503538384f,
    0.865513624f,  0.500885383f,
    0.867046246f,  0.498227667f,
    0.868570706f,  0.495565262f,
    0.870086991f,  0.492898192f,
    0.871595087f,  0.490226483f,
    0.873094978f,  0.487550160f,
    0.874586652f,  0.484869248f,
    0.876070094f,  0.482183772f,
    0.877545290f,  0.479493758f,
    0.879012226f,  0.476799230f,
    0.880470889f,  0.474100215f,
    0.881921264f,  0.471396737f,
    0.883363339f,  0.468688822f,
    0.884797098f,  0.465976496f,
    0.886222530f,  0.463259784f,
    0.887639620f,  0.460538711f,
    0.889048356f,  0.457813304f,
    0.890448723f,  0.455083587f,
    0.891840709f,  0.452349587f,
    0.893224301f,  0.449611330f,
    0.894599486f,  0.446868840f,
    0.895966250f,  0.444122145f,
    0.897324581f,  0.441371269f,
    0.898674466f,  0.438616239f,
    0.900015892f,  0.435857080f,
    0.901348847f,  0.433093819f,
    0.902673318f,  0.430326481f,
    0.903989293f,  0.427555093f,
    0.905296759f,  0.424779681f,
    0.906595705f,  0.422000271f,
    0.907886116f,  0.419216888f,
    0.909167983f,  0.416429560f,
    0.910441292f,  0.413638312f,
    0.911706032f,  0.410843171f,
    0.912962190f,  0.408044163f,
    0.914209756f,  0.405241314f,
    0.915448716f,  0.402434651f,
    0.916679060f,  0.399624200f,
    0.917900776f,  0.396809987f,
    0.919113852f,  0.393992040f,
    0.920318277f,  0.391170384f,
    0.921514039f,  0.388345047f,
    0.922701128f,  0.385516054f,
    0.923879533f,  0.382683432f,
    0.925049241f,  0.379847209f,
    0.926210242f,  0.377007410f,
    0.927362526f,  0.374164063f,
    0.928506080f,  0.371317194f,
    0.929640896f,  0.368466830f,
    0.930766961f,  0.365612998f,
    0.931884266f,  0.362755724f,
    0.932992799f,  0.359895037f,
    0.934092550f,  0.357030961f,
    0.935183510f,  0.354163525f,
    0.936265667f,  0.351292756f,
    0.937339012f,  0.348418680f,
    0.938403534f,  0.345541325f,
    0.939459224f,  0.342660717f,
    0.940506071f,  0.339776884f,
    0.941544065f,  0.336889853f,
    0.942573198f,  0.333999651f,
    0.943593458f,  0.331106306f,
    0.944604837f,  0.328209844f,
    0.945607325f,  0.325310292f,
    0.946600913f,  0.322407679f,
    0.947585591f,  0.319502031f,
    0.948561350f,  0.316593376f,
    0.949528181f,  0.313681740f,
    0.950486074f,  0.310767153f,
    0.951435021f,  0.307849640f,
    0.952375013f,  0.304929230f,
    0.953306040f,  0.302005949f,
    0.954228095f,  0.299079826f,
    0.955141168f,  0.296150888f,
    0.956045251f,  0.293219163f,
    0.956940336f,  0.290284677f,
    0.957826413f,  0.287347460f,
    0.958703475f,  0.284407537f,
    0.959571513f,  0.281464938f,
    0.960430519f,  0.278519689f,
    0.961280486f,  0.275571819f,
    0.962121404f,  0.272621355f,
    0.962953267f,  0.269668326f,
    0.963776066f,  0.266712757f,
    0.964589793f,  0.263754679f,
    0.965394442f,  0.260794118f,
    0.966190003f,  0.257831102f,
    0.966976471f,  0.254865660f,
    0.967753837f,  0.251897818f,
    0.968522094f,  0.248927606f,
    0.969281235f,  0.245955050f,
    0.970031253f,  0.242980180f,
    0.970772141f,  0.240003022f,
    0.971503891f,  0.237023606f,
    0.972226497f,  0.234041959f,
    0.972939952f,  0.231058108f,
    0.973644250f,  0.228072083f,
    0.974339383f,  0.225083911f,
    0.975025345f,  0.222093621f,
    0.975702130f,  0.219101240f,
    0.976369731f,  0.216106797f,
    0.977028143f,  0.213110320f,
    0.977677358f,  0.210111837f,
    0.978317371f,  0.207111376f,
    0.978948175f,  0.204108966f,
    0.979569766f,  0.201104635f,
    0.980182136f,  0.198098411f,
    0.980785280f,  0.195090322f,
    0.981379193f,  0.192080397f,
    0.981963869f,  0.189068664f,
    0.982539302f,  0.186055152f,
    0.983105487f,  0.183039888f,
    0.983662419f,  0.180022901f,
    0.984210092f,  0.177004220f,
    0.984748502f,  0.173983873f,
    0.985277642f,  0.170961889f,
    0.985797509f,  0.167938295f,
    0.986308097f,  0.164913120f,
    0.986809402f,  0.161886394f,
    0.987301418f,  0.158858143f,
    0.987784142f,  0.155828398f,
    0.988257568f,  0.152797185f,
    0.988721692f,  0.149764535f,
    0.989176510f,  0.146730474f,
    0.989622017f,  0.143695033f,
    0.990058210f,  0.140658239f,
    0.990485084f,  0.137620122f,
    0.990902635f,  0.134580709f,
    0.991310860f,  0.131540029f,
    0.991709754f,  0.128498111f,
    0.992099313f,  0.125454983f,
    0.992479535f,  0.122410675f,
    0.992850414f,  0.119365215f,
    0.993211949f,  0.116318631f,
    0.993564136f,  0.113270952f,
    0.993906970f,  0.110222207f,
    0.994240449f,  0.107172425f,
    0.994564571f,  0.104121634f,
    0.994879331f,  0.101069863f,
    0.995184727f,  0.098017140f,
    0.995480755f,  0.094963495f,
    0.995767414f,  0.091908956f,
    0.996044701f,  0.088853553f,
    0.996312612f,  0.085797312f,
    0.996571146f,  0.082740265f,
    0.996820299f,  0.079682438f,
    0.997060070f,  0.076623861f,
    0.997290457f,  0.073564564f,
    0.997511456f,  0.070504573f,
    0.997723067f,  0.067443920f,
    0.997925286f,  0.064382631f,
    0.998118113f,  0.061320736f,
    0.998301545f,  0.058258265f,
    0.998475581f,  0.055195244f,
    0.998640218f,  0.052131705f,
    0.998795456f,  0.049067674f,
    0.998941293f,  0.046003182f,
    0.999077728f,  0.042938257f,
    0.999204759f,  0.039872928f,
    0.999322385f,  0.036807223f,
    0.999430605f,  0.033741172f,
    0.999529418f,  0.030674803f,
    0.999618822f,  0.027608146f,
    0.999698819f,  0.024541229f,
    0.999769405f,  0.021474080f,
    0.999830582f,  0.018406730f,
    0.999882347f,  0.015339206f,
    0.999924702f,  0.012271538f,
    0.999957645f,  0.009203755f,
    0.999981175f,  0.006135885f,
    0.999995294f,  0.003067957f,
    1.000000000f,  0.000000000f,
    0.999995294f, -0.003067957f,
    0.999981175f, -0.006135885f,
    0.999957645f, -0.009203755f,
    0.999924702f, -0.012271538f,
    0.999882347f, -0.015339206f,
    0.999830582f, -0.018406730f,
    0.999769405f, -0.021474080f,
    0.999698819f, -0.024541229f,
    0.999618822f, -0.027608146f,
    0.999529418f, -0.030674803f,
    0.999430605f, -0.033741172f,
    0.999322385f, -0.036807223f,
    0.999204759f, -0.039872928f,
    0.999077728f, -0.042938257f,
    0.998941293f, -0.046003182f,
    0.998795456f, -0.049067674f,
    0.998640218f, -0.052131705f,
    0.998475581f, -0.055195244f,
    0.998301545f, -0.058258265f,
    0.998118113f, -0.061320736f,
    0.997925286f, -0.064382631f,
    0.997723067f, -0.067443920f,
    0.997511456f, -0.070504573f,
    0.997290457f, -0.073564564f,
    0.997060070f, -0.076623861f,
    0.996820299f, -0.079682438f,
    0.996571146f, -0.082740265f,
    0.996312612f, -0.085797312f,
    0.996044701f, -0.088853553f,
    0.995767414f, -0.091908956f,
    0.995480755f, -0.094963495f,
    0.995184727f, -0.098017140f,
    0.994879331f, -0.101069863f,
    0.994564571f, -0.104121634f,
    0.994240449f, -0.107172425f,
    0.993906970f, -0.110222207f,
    0.993564136f, -0.113270952f,
    0.993211949f, -0.116318631f,
    0.992850414f, -0.119365215f,
    0.992479535f, -0.122410675f,
    0.992099313f, -0.125454983f,
    0.991709754f, -0.128498111f,
    0.991310860f, -0.131540029f,
    0.990902635f, -0.134580709f,
    0.990485084f, -0.137620122f,
    0.990058210f, -0.140658239f,
    0.989622017f, -0.143695033f,
    0.989176510f, -0.146730474f,
    0.988721692f, -0.149764535f,
    0.988257568f, -0.152797185f,
    0.987784142f, -0.155828398f,
    0.987301418f, -0.158858143f,
    0.986809402f, -0.161886394f,
    0.986308097f, -0.164913120f,
    0.985797509f, -0.167938295f,
    0.985277642f, -0.170961889f,
    0.984748502f, -0.173983873f,
    0.984210092f, -0.177004220f,
    0.983662419f, -0.180022901f,
    0.983105487f, -0.183039888f,
    0.982539302f, -0.186055152f,
    0.981963869f, -0.189068664f,
    0.981379193f, -0.192080397f,
    0.980785280f, -0.195090322f,
    0.980182136f, -0.198098411f,
    0.979569766f, -0.201104635f,
    0.978948175f, -0.204108966f,
    0.978317371f, -0.207111376f,
    0.977677358f, -0.210111837f,
    0.977028143f, -0.213110320f,
    0.976369731f, -0.216106797f,
    0.975702130f, -0.219101240f,
    0.975025345f, -0.222093621f,
    0.974339383f, -0.225083911f,
    0.973644250f, -0.228072083f,
    0.972939952f, -0.231058108f,
    0.972226497f, -0.234041959f,
    0.971503891f, -0.237023606f,
    0.970772141f, -0.240003022f,
    0.970031253f, -0.242980180f,
    0.969281235f, -0.245955050f,
    0.968522094f, -0.248927606f,
    0.967753837f, -0.251897818f,
    0.966976471f, -0.254865660f,
    0.966190003f, -0.257831102f,
    0.965394442f, -0.260794118f,
    0.964589793f, -0.263754679f,
    0.963776066f, -0.266712757f,
    0.962953267f, -0.269668326f,
    0.962121404f, -0.272621355f,
    0.961280486f, -0.275571819f,
    0.960430519f, -0.278519689f,
    0.959571513f, -0.281464938f,
    0.958703475f, -0.284407537f,
    0.957826413f, -0.287347460f,
    0.956940336f, -0.290284677f,
    0.956045251f, -0.293219163f,
    0.955141168f, -0.296150888f,
    0.954228095f, -0.299079826f,
    0.953306040f, -0.302005949f,
    0.952375013f, -0.304929230f,
    0.951435021f, -0.307849640f,
    0.950486074f, -0.310767153f,
    0.949528181f, -0.313681740f,
    0.948561350f, -0.316593376f,
    0.947585591f, -0.319502031f,
    0.946600913f, -0.322407679f,
    0.945607325f, -0.325310292f,
    0.944604837f, -0.328209844f,
    0.943593458f, -0.331106306f,
    0.942573198f, -0.333999651f,
    0.941544065f, -0.336889853f,
    0.940506071f, -0.339776884f,
    0.939459224f, -0.342660717f,
    0.938403534f, -0.345541325f,
    0.937339012f, -0.348418680f,
    0.936265667f, -0.351292756f,
    0.935183510f, -0.354163525f,
    0.934092550f, -0.357030961f,
    0.932992799f, -0.359895037f,
    0.931884266f, -0.362755724f,
    0.930766961f, -0.365612998f,
    0.929640896f, -0.368466830f,
    0.928506080f, -0.371317194f,
    0.927362526f, -0.374164063f,
    0.926210242f, -0.377007410f,
    0.925049241f, -0.379847209f,
    0.923879533f, -0.382683432f,
    0.922701128f, -0.385516054f,
    0.921514039f, -0.388345047f,
    0.920318277f, -0.391170384f,
    0.919113852f, -0.393992040f,
    0.917900776f, -0.396809987f,
    0.916679060f, -0.399624200f,
    0.915448716f, -0.402434651f,
    0.914209756f, -0.405241314f,
    0.912962190f, -0.408044163f,
    0.911706032f, -0.410843171f,
    0.910441292f, -0.413638312f,
    0.909167983f, -0.416429560f,
    0.907886116f, -0.419216888f,
    0.906595705f, -0.422000271f,
    0.905296759f, -0.424779681f,
    0.903989293f, -0.427555093f,
    0.902673318f, -0.430326481f,
    0.901348847f, -0.433093819f,
    0.900015892f, -0.435857080f,
    0.898674466f, -0.438616239f,
    0.897324581f, -0.441371269f,
    0.895966250f, -0.444122145f,
    0.894599486f, -0.446868840f,
    0.893224301f, -0.449611330f,
    0.891840709f, -0.452349587f,
    0.890448723f, -0.455083587f,
    0.889048356f, -0.457813304f,
    0.887639620f, -0.460538711f,
    0.886222530f, -0.463259784f,
    0.884797098f, -0.465976496f,
    0.883363339f, -0.468688822f,
    0.881921264f, -0.471396737f,
    0.880470889f, -0.474100215f,
    0.879012226f, -0.476799230f,
    0.877545290f, -0.479493758f,
    0.876070094f, -0.482183772f,
    0.874586652f, -0.484869248f,
    0.873094978f, -0.487550160f,
    0.871595087f, -0.490226483f,
    0.870086991f, -0.492898192f,
    0.868570706f, -0.495565262f,
    0.867046246f, -0.498227667f,
    0.865513624f, -0.500885383f,
    0.863972856f, -0.503538384f,
    0.862423956f, -0.506186645f,
    0.860866939f, -0.508830143f,
    0.859301818f, -0.511468850f,
    0.857728610f, -0.514102744f,
    0.856147328f, -0.516731799f,
    0.854557988f, -0.519355990f,
    0.852960605f, -0.521975293f,
    0.851355193f, -0.524589683f,
    0.849741768f, -0.527199135f,
    0.848120345f, -0.529803625f,
    0.846490939f, -0.532403128f,
    0.844853565f, -0.534997620f,
    0.843208240f, -0.537587076f,
    0.841554977f, -0.540171473f,
    0.839893794f, -0.542750785f,
    0.838224706f, -0.545324988f,
    0.836547727f, -0.547894059f,
    0.834862875f, -0.550457973f,
    0.833170165f, -0.553016706f,
    0.831469612f, -0.555570233f,
    0.829761234f, -0.558118531f,
    0.828045045f, -0.560661576f,
    0.826321063f, -0.563199344f,
    0.824589303f, -0.565731811f,
    0.822849781f, -0.568258953f,
    0.821102515f, -0.570780746f,
    0.819347520f, -0.573297167f,
    0.817584813f, -0.575808191f,
    0.815814411f, -0.578313796f,
    0.814036330f, -0.580813958f,
    0.812250587f, -0.583308653f,
    0.810457198f, -0.585797857f,
    0.808656182f, -0.588281548f,
    0.806847554f, -0.590759702f,
    0.805031331f, -0.593232295f,
    0.803207531f, -0.595699304f,
    0.801376172f, -0.598160707f,
    0.799537269f, -0.600616479f,
    0.797690841f, -0.603066599f,
    0.795836905f, -0.605511041f,
    0.793975478f, -0.607949785f,
    0.792106577f, -0.610382806f,
    0.790230221f, -0.612810082f,
    0.788346428f, -0.615231591f,
    0.786455214f, -0.617647308f,
    0.784556597f, -0.620057212f,
    0.782650596f, -0.622461279f,
    0.780737229f, -0.624859488f,
    0.778816512f, -0.627251815f,
    0.776888466f, -0.629638239f,
    0.774953107f, -0.632018736f,
    0.773010453f, -0.634393284f,
    0.771060524f, -0.636761861f,
    0.769103338f, -0.639124445f,
    0.767138912f, -0.641481013f,
    0.765167266f, -0.643831543f,
    0.763188417f, -0.646176013f,
    0.761202385f, -0.648514401f,
    0.759209189f, -0.650846685f,
    0.757208847f, -0.653172843f,
    0.755201377f, -0.655492853f,
    0.753186799f, -0.657806693f,
    0.751165132f, -0.660114342f,
    0.749136395f, -0.662415778f,
    0.747100606f, -0.664710978f,
    0.745057785f, -0.666999922f,
    0.743007952f, -0.669282588f,
    0.740951125f, -0.671558955f,
    0.738887324f, -0.673829000f,
    0.736816569f, -0.676092704f,
    0.734738878f, -0.678350043f,
    0.732654272f, -0.680600998f,
    0.730562769f, -0.682845546f,
    0.728464390f, -0.685083668f,
    0.726359155f, -0.687315341f,
    0.724247083f, -0.689540545f,
    0.722128194f, -0.691759258f,
    0.720002508f, -0.693971461f,
    0.717870045f, -0.696177131f,
    0.715730825f, -0.698376249f,
    0.713584869f, -0.700568794f,
    0.711432196f, -0.702754744f,
    0.709272826f, -0.704934080f,
    0.707106781f, -0.707106781f,
    0.704934080f, -0.709272826f,
    0.702754744f, -0.711432196f,
    0.700568794f, -0.713584869f,
    0.698376249f, -0.715730825f,
    0.696177131f, -0.717870045f,
    0.693971461f, -0.720002508f,
    0.691759258f, -0.722128194f,
    0.689540545f, -0.724247083f,
    0.687315341f, -0.726359155f,
    0.685083668f, -0.728464390f,
    0.682845546f, -0.730562769f,
    0.680600998f, -0.732654272f,
    0.678350043f, -0.734738878f,
    0.676092704f, -0.736816569f,
    0.673829000f, -0.738887324f,
    0.671558955f, -0.740951125f,
    0.669282588f, -0.743007952f,
    0.666999922f, -0.745057785f,
    0.664710978f, -0.747100606f,
    0.662415778f, -0.749136395f,
    0.660114342f, -0.751165132f,
    0.657806693f, -0.753186799f,
    0.655492853f, -0.755201377f,
    0.653172843f, -0.757208847f,
    0.650846685f, -0.759209189f,
    0.648514401f, -0.761202385f,
    0.646176013f, -0.763188417f,
    0.643831543f, -0.765167266f,
    0.641481013f, -0.767138912f,
    0.639124445f, -0.769103338f,
    0.636761861f, -0.771060524f,
    0.634393284f, -0.773010453f,
    0.632018736f, -0.774953107f,
    0.629638239f, -0.776888466f,
    0.627251815f, -0.778816512f,
    0.624859488f, -0.780737229f,
    0.622461279f, -0.782650596f,
    0.620057212f, -0.784556597f,
    0.617647308f, -0.786455214f,
    0.615231591f, -0.788346428f,
    0.612810082f, -0.790230221f,
    0.610382806f, -0.792106577f,
    0.607949785f, -0.793975478f,
    0.605511041f, -0.795836905f,
    0.603066599f, -0.797690841f,
    0.600616479f, -0.799537269f,
    0.598160707f, -0.801376172f,
    0.595699304f, -0.803207531f,
    0.593232295f, -0.805031331f,
    0.590759702f, -0.806847554f,
    0.588281548f, -0.808656182f,
    0.585797857f, -0.810457198f,
    0.583308653f, -0.812250587f,
    0.580813958f, -0.814036330f,
    0.578313796f, -0.815814411f,
    0.575808191f, -0.817584813f,
    0.573297167f, -0.819347520f,
    0.570780746f, -0.821102515f,
    0.568258953f, -0.822849781f,
    0.565731811f, -0.824589303f,
    0.563199344f, -0.826321063f,
    0.560661576f, -0.828045045f,
    0.558118531f, -0.829761234f,
    0.555570233f, -0.831469612f,
    0.553016706f, -0.833170165f,
    0.550457973f, -0.834862875f,
    0.547894059f, -0.836547727f,
    0.545324988f, -0.838224706f,
    0.542750785f, -0.839893794f,
    0.540171473f, -0.841554977f,
    0.537587076f, -0.843208240f,
    0.534997620f, -0.844853565f,
    0.532403128f, -0.846490939f,
    0.529803625f, -0.848120345f,
    0.527199135f, -0.849741768f,
    0.524589683f, -0.851355193f,
    0.521975293f, -0.852960605f,
    0.519355990f, -0.854557988f,
    0.516731799f, -0.856147328f,
    0.514102744f, -0.857728610f,
    0.511468850f, -0.859301818f,
    0.508830143f, -0.860866939f,
    0.506186645f, -0.862423956f,
    0.503538384f, -0.863972856f,
    0.500885383f, -0.865513624f,
    0.498227667f, -0.867046246f,
    0.495565262f, -0.868570706f,
    0.492898192f, -0.870086991f,
    0.490226483f, -0.871595087f,
    0.487550160f, -0.873094978f,
    0.484869248f, -0.874586652f,
    0.482183772f, -0.876070094f,
    0.479493758f, -0.877545290f,
    0.476799230f, -0.879012226f,
    0.474100215f, -0.880470889f,
    0.471396737f, -0.881921264f,
    0.468688822f, -0.883363339f,
    0.465976496f, -0.884797098f,
    0.463259784f, -0.886222530f,
    0.460538711f, -0.887639620f,
    0.457813304f, -0.889048356f,
    0.455083587f, -0.890448723f,
    0.452349587f, -0.891840709f,
    0.449611330f, -0.893224301f,
    0.446868840f, -0.894599486f,
    0.444122145f, -0.895966250f,
    0.441371269f, -0.897324581f,
    0.438616239f, -0.898674466f,
    0.435857080f, -0.900015892f,
    0.433093819f, -0.901348847f,
    0.430326481f, -0.902673318f,
    0.427555093f, -0.903989293f,
    0.424779681f, -0.905296759f,
    0.422000271f, -0.906595705f,
    0.419216888f, -0.907886116f,
    0.416429560f, -0.909167983f,
    0.413638312f, -0.910441292f,
    0.410843171f, -0.911706032f,
    0.408044163f, -0.912962190f,
    0.405241314f, -0.914209756f,
    0.402434651f, -0.915448716f,
    0.399624200f, -0.916679060f,
    0.396809987f, -0.917900776f,
    0.393992040f, -0.919113852f,
    0.391170384f, -0.920318277f,
    0.388345047f, -0.921514039f,
    0.385516054f, -0.922701128f,
    0.382683432f, -0.923879533f,
    0.379847209f, -0.925049241f,
    0.377007410f, -0.926210242f,
    0.374164063f, -0.927362526f,
    0.371317194f, -0.928506080f,
    0.368466830f, -0.929640896f,
    0.365612998f, -0.930766961f,
    0.362755724f, -0.931884266f,
    0.359895037f, -0.932992799f,
    0.357030961f, -0.934092550f,
    0.354163525f, -0.935183510f,
    0.351292756f, -0.936265667f,
    0.348418680f, -0.937339012f,
    0.345541325f, -0.938403534f,
    0.342660717f, -0.939459224f,
    0.339776884f, -0.940506071f,
    0.336889853f, -0.941544065f,
    0.333999651f, -0.942573198f,
    0.331106306f, -0.943593458f,
    0.328209844f, -0.944604837f,
    0.325310292f, -0.945607325f,
    0.322407679f, -0.946600913f,
    0.319502031f, -0.947585591f,
    0.316593376f, -0.948561350f,
    0.313681740f, -0.949528181f,
    0.310767153f, -0.950486074f,
    0.307849640f, -0.951435021f,
    0.304929230f, -0.952375013f,
    0.302005949f, -0.953306040f,
    0.299079826f, -0.954228095f,
    0.296150888f, -0.955141168f,
    0.293219163f, -0.956045251f,
    0.290284677f, -0.956940336f,
    0.287347460f, -0.957826413f,
    0.284407537f, -0.958703475f,
    0.281464938f, -0.959571513f,
    0.278519689f, -0.960430519f,
    0.275571819f, -0.961280486f,
    0.272621355f, -0.962121404f,
    0.269668326f, -0.962953267f,
    0.266712757f, -0.963776066f,
    0.263754679f, -0.964589793f,
    0.260794118f, -0.965394442f,
    0.257831102f, -0.966190003f,
    0.254865660f, -0.966976471f,
    0.251897818f, -0.967753837f,
    0.248927606f, -0.968522094f,
    0.245955050f, -0.969281235f,
    0.242980180f, -0.970031253f,
    0.240003022f, -0.970772141f,
    0.237023606f, -0.971503891f,
    0.234041959f, -0.972226497f,
    0.231058108f, -0.972939952f,
    0.228072083f, -0.973644250f,
    0.225083911f, -0.974339383f,
    0.222093621f, -0.975025345f,
    0.219101240f, -0.975702130f,
    0.216106797f, -0.976369731f,
    0.213110320f, -0.977028143f,
    0.210111837f, -0.977677358f,
    0.207111376f, -0.978317371f,
    0.204108966f, -0.978948175f,
    0.201104635f, -0.979569766f,
    0.198098411f, -0.980182136f,
    0.195090322f, -0.980785280f,
    0.192080397f, -0.981379193f,
    0.189068664f, -0.981963869f,
    0.186055152f, -0.982539302f,
    0.183039888f, -0.983105487f,
    0.180022901f, -0.983662419f,
    0.177004220f, -0.984210092f,
    0.173983873f, -0.984748502f,
    0.170961889f, -0.985277642f,
    0.167938295f, -0.985797509f,
    0.164913120f, -0.986308097f,
    0.161886394f, -0.986809402f,
    0.158858143f, -0.987301418f,
    0.155828398f, -0.987784142f,
    0.152797185f, -0.988257568f,
    0.149764535f, -0.988721692f,
    0.146730474f, -0.989176510f,
    0.143695033f, -0.989622017f,
    0.140658239f, -0.990058210f,
    0.137620122f, -0.990485084f,
    0.134580709f, -0.990902635f,
    0.131540029f, -0.991310860f,
    0.128498111f, -0.991709754f,
    0.125454983f, -0.992099313f,
    0.122410675f, -0.992479535f,
    0.119365215f, -0.992850414f,
    0.116318631f, -0.993211949f,
    0.113270952f, -0.993564136f,
    0.110222207f, -0.993906970f,
    0.107172425f, -0.994240449f,
    0.104121634f, -0.994564571f,
    0.101069863f, -0.994879331f,
    0.098017140f, -0.995184727f,
    0.094963495f, -0.995480755f,
    0.091908956f, -0.995767414f,
    0.088853553f, -0.996044701f,
    0.085797312f, -0.996312612f,
    0.082740265f, -0.996571146f,
    0.079682438f, -0.996820299f,
    0.076623861f, -0.997060070f,
    0.073564564f, -0.997290457f,
    0.070504573f, -0.997511456f,
    0.067443920f, -0.997723067f,
    0.064382631f, -0.997925286f,
    0.061320736f, -0.998118113f,
    0.058258265f, -0.998301545f,
    0.055195244f, -0.998475581f,
    0.052131705f, -0.998640218f,
    0.049067674f, -0.998795456f,
    0.046003182f, -0.998941293f,
    0.042938257f, -0.999077728f,
    0.039872928f, -0.999204759f,
    0.036807223f, -0.999322385f,
    0.033741172f, -0.999430605f,
    0.030674803f, -0.999529418f,
    0.027608146f, -0.999618822f,
    0.024541229f, -0.999698819f,
    0.021474080f, -0.999769405f,
    0.018406730f, -0.999830582f,
    0.015339206f, -0.999882347f,
    0.012271538f, -0.999924702f,
    0.009203755f, -0.999957645f,
    0.006135885f, -0.999981175f,
    0.003067957f, -0.999995294f
};

const float32_t twiddleCoef_rfft_4096[4096] = {
    0.000000000f,  1.000000000f,
    0.001533980f,  0.999998823f,
    0.003067957f,  0.999995294f,
    0.004601926f,  0.999989411f,
    0.006135885f,  0.999981175f,
    0.007669829f,  0.999970586f,
    0.009203755f,  0.999957645f,
    0.010737659f,  0.999942350f,
    0.012271538f,  0.999924702f,
    0.013805389f,  0.999904701f,
    0.015339206f,  0.999882347f,
    0.016872988f,  0.999857641f,
    0.018406730f,  0.999830582f,
    0.019940429f,  0.999801170f,
    0.021474080f,  0.999769405f,
    0.023007681f,  0.999735288f,
    0.024541229f,  0.999698819f,
    0.026074718f,  0.999659997f,
    0.027608146f,  0.999618822f,
    0.029141509f,  0.999575296f,
    0.030674803f,  0.999529418f,
    0.032208025f,  0.999481187f,
    0.033741172f,  0.999430605f,
    0.035274239f,  0.999377670f,
    0.036807223f,  0.999322385f,
    0.038340120f,  0.999264747f,
    0.039872928f,  0.999204759f,
    0.041405641f,  0.999142419f,
    0.042938257f,  0.999077728f,
    0.044470772f,  0.999010686f,
    0.046003182f,  0.998941293f,
    0.047535484f,  0.998869550f,
    0.049067674f,  0.998795456f,
    0.050599749f,  0.998719012f,
    0.052131705f,  0.998640218f,
    0.053663538f,  0.998559074f,
    0.055195244f,  0.998475581f,
    0.056726821f,  0.998389737f,
    0.058258265f,  0.998301545f,
    0.059789571f,  0.998211003f,
    0.061320736f,  0.998118113f,
    0.062851758f,  0.998022874f,
    0.064382631f,  0.997925286f,
    0.065913353f,  0.997825350f,
    0.067443920f,  0.997723067f,
    0.068974328f,  0.997618435f,
    0.070504573f,  0.997511456f,
    0.072034653f,  0.997402130f,
    0.073564564f,  0.997290457f,
    0.075094301f,  0.997176437f,
    0.076623861f,  0.997060070f,
    0.078153242f,  0.996941358f,
    0.079682438f,  0.996820299f,
    0.081211447f,  0.996696895f,
    0.082740265f,  0.996571146f,
    0.084268888f,  0.996443051f,
    0.085797312f,  0.996312612f,
    0.087325535f,  0.996179829f,
    0.088853553f,  0.996044701f,
    0.090381361f,  0.995907229f,
    0.091908956f,  0.995767414f,
    0.093436336f,  0.995625256f,
    0.094963495f,  0.995480755f,
    0.096490431f,  0.995333912f,
    0.098017140f,  0.995184727f,
    0.099543619f,  0.995033199f,
    0.101069863f,  0.994879331f,
    0.102595869f,  0.994723121f,
    0.104121634f,  0.994564571f,
    0.105647154f,  0.994403680f,
    0.107172425f,  0.994240449f,
    0.108697444f,  0.994074879f,
    0.110222207f,  0.993906970f,
    0.111746711f,  0.993736722f,
    0.113270952f,  0.993564136f,
    0.114794927f,  0.993389211f,
    0.116318631f,  0.993211949f,
    0.117842062f,  0.993032350f,
    0.119365215f,  0.992850414f,
    0.120888087f,  0.992666142f,
    0.122410675f,  0.992479535f,
    0.123932975f,  0.992290591f,
    0.125454983f,  0.992099313f,
    0.126976696f,  0.991905700f,
    0.128498111f,  0.991709754f,
    0.130019223f,  0.991511473f,
    0.131540029f,  0.991310860f,
    0.133060525f,  0.991107914f,
    0.134580709f,  0.990902635f,
    0.136100575f,  0.990695025f,
    0.137620122f,  0.990485084f,
    0.139139344f,  0.990272812f,
    0.140658239f,  0.990058210f,
    0.142176804f,  0.989841278f,
    0.143695033f,  0.989622017f,
    0.145212925f,  0.989400428f,
    0.146730474f,  0.989176510f,
    0.148247679f,  0.988950265f,
    0.149764535f,  0.988721692f,
    0.151281038f,  0.988490793f,
    0.152797185f,  0.988257568f,
    0.154312973f,  0.988022017f,
    0.155828398f,  0.987784142f,
    0.157343456f,  0.987543942f,
    0.158858143f,  0.987301418f,
    0.160372457f,  0.987056571f,
    0.161886394f,  0.986809402f,
    0.163399949f,  0.986559910f,
    0.164913120f,  0.986308097f,
    0.166425904f,  0.986053963f,
    0.167938295f,  0.985797509f,
    0.169450291f,  0.985538735f,
    0.170961889f,  0.985277642f,
    0.172473084f,  0.985014231f,
    0.173983873f,  0.984748502f,
    0.175494253f,  0.984480455f,
    0.177004220f,  0.984210092f,
    0.178513771f,  0.983937413f,
    0.180022901f,  0.983662419f,
    0.181531608f,  0.983385110f,
    0.183039888f,  0.983105487f,
    0.184547737f,  0.982823551f,
    0.186055152f,  0.982539302f,
    0.187562129f,  0.982252741f,
    0.189068664f,  0.981963869f,
    0.190574755f,  0.981672686f,
    0.192080397f,  0.981379193f,
    0.193585587f,  0.981083391f,
    0.195090322f,  0.980785280f,
    0.196594598f,  0.980484862f,
    0.198098411f,  0.980182136f,
    0.199601758f,  0.979877104f,
    0.201104635f,  0.979569766f,
    0.202607039f,  0.979260123f,
    0.204108966f,  0.978948175f,
    0.205610413f,  0.978633924f,
    0.207111376f,  0.978317371f,
    0.208611852f,  0.977998515f,
    0.210111837f,  0.977677358f,
    0.211611327f,  0.977353900f,
    0.213110320f,  0.977028143f,
    0.214608811f,  0.976700086f,
    0.216106797f,  0.976369731f,
    0.217604275f,  0.976037079f,
    0.219101240f,  0.975702130f,
    0.220597690f,  0.975364885f,
    0.222093621f,  0.975025345f,
    0.223589029f,  0.974683511f,
    0.225083911f,  0.974339383f,
    0.226578264f,  0.973992962f,
    0.228072083f,  0.973644250f,
    0.229565366f,  0.973293246f,
    0.231058108f,  0.972939952f,
    0.232550307f,  0.972584369f,
    0.234041959f,  0.972226497f,
    0.235533059f,  0.971866337f,
    0.237023606f,  0.971503891f,
    0.238513595f,  0.971139158f,
    0.240003022f,  0.970772141f,
    0.241491885f,  0.970402839f,
    0.242980180f,  0.970031253f,
    0.244467903f,  0.969657385f,
    0.245955050f,  0.969281235f,
    0.247441619f,  0.968902805f,
    0.248927606f,  0.968522094f,
    0.250413007f,  0.968139105f,
    0.251897818f,  0.967753837f,
    0.253382037f,  0.967366292f,
    0.254865660f,  0.966976471f,
    0.256348682f,  0.966584374f,
    0.257831102f,  0.966190003f,
    0.259312915f,  0.965793359f,
    0.260794118f,  0.965394442f,
    0.262274707f,  0.964993253f,
    0.263754679f,  0.964589793f,
    0.265234030f,  0.964184064f,
    0.266712757f,  0.963776066f,
    0.268190857f,  0.963365800f,
    0.269668326f,  0.962953267f,
    0.271145160f,  0.962538468f,
    0.272621355f,  0.962121404f,
    0.274096910f,  0.961702077f,
    0.275571819f,  0.961280486f,
    0.277046080f,  0.960856633f,
    0.278519689f,  0.960430519f,
    0.279992643f,  0.960002146f,
    0.281464938f,  0.959571513f,
    0.282936570f,  0.959138622f,
    0.284407537f,  0.958703475f,
    0.285877835f,  0.958266071f,
    0.287347460f,  0.957826413f,
    0.288816408f,  0.957384501f,
    0.290284677f,  0.956940336f,
    0.291752263f,  0.956493919f,
    0.293219163f,  0.956045251f,
    0.294685372f,  0.955594334f,
    0.296150888f,  0.955141168f,
    0.297615707f,  0.954685755f,
    0.299079826f,  0.954228095f,
    0.300543241f,  0.953768190f,
    0.302005949f,  0.953306040f,
    0.303467947f,  0.952841648f,
    0.304929230f,  0.952375013f,
    0.306389795f,  0.951906137f,
    0.307849640f,  0.951435021f,
    0.309308760f,  0.950961666f,
    0.310767153f,  0.950486074f,
    0.312224814f,  0.950008245f,
    0.313681740f,  0.949528181f,
    0.315137929f,  0.949045882f,
    0.316593376f,  0.948561350f,
    0.318048077f,  0.948074586f,
    0.319502031f,  0.947585591f,
    0.320955232f,  0.947094366f,
    0.322407679f,  0.946600913f,
    0.323859367f,  0.946105232f,
    0.325310292f,  0.945607325f,
    0.326760452f,  0.945107193f,
    0.328209844f,  0.944604837f,
    0.329658463f,  0.944100258f,
    0.331106306f,  0.943593458f,
    0.332553370f,  0.943084437f,
    0.333999651f,  0.942573198f,
    0.335445147f,  0.942059740f,
    0.336889853f,  0.941544065f,
    0.338333767f,  0.941026175f,
    0.339776884f,  0.940506071f,
    0.341219202f,  0.939983753f,
    0.342660717f,  0.939459224f,
    0.344101426f,  0.938932484f,
    0.345541325f,  0.938403534f,
    0.346980411f,  0.937872376f,
    0.348418680f,  0.937339012f,
    0.349856130f,  0.936803442f,
    0.351292756f,  0.936265667f,
    0.352728556f,  0.935725689f,
    0.354163525f,  0.935183510f,
    0.355597662f,  0.934639130f,
    0.357030961f,  0.934092550f,
    0.358463421f,  0.933543773f,
    0.359895037f,  0.932992799f,
    0.361325806f,  0.932439629f,
    0.362755724f,  0.931884266f,
    0.364184790f,  0.931326709f,
    0.365612998f,  0.930766961f,
    0.367040346f,  0.930205023f,
    0.368466830f,  0.929640896f,
    0.369892447f,  0.929074581f,
    0.371317194f,  0.928506080f,
    0.372741067f,  0.927935395f,
    0.374164063f,  0.927362526f,
    0.375586178f,  0.926787474f,
    0.377007410f,  0.926210242f,
    0.378427755f,  0.925630831f,
    0.379847209f,  0.925049241f,
    0.381265769f,  0.924465474f,
    0.382683432f,  0.923879533f,
    0.384100195f,  0.923291417f,
    0.385516054f,  0.922701128f,
    0.386931006f,  0.922108669f,
    0.388345047f,  0.921514039f,
    0.389758174f,  0.920917242f,
    0.391170384f,  0.920318277f,
    0.392581674f,  0.919717146f,
    0.393992040f,  0.919113852f,
    0.395401479f,  0.918508394f,
    0.396809987f,  0.917900776f,
    0.398217562f,  0.917290997f,
    0.399624200f,  0.916679060f,
    0.401029897f,  0.916064966f,
    0.402434651f,  0.915448716f,
    0.403838458f,  0.914830312f,
    0.405241314f,  0.914209756f,
    0.406643217f,  0.913587048f,
    0.408044163f,  0.912962190f,
    0.409444149f,  0.912335185f,
    0.410843171f,  0.911706032f,
    0.412241227f,  0.911074734f,
    0.413638312f,  0.910441292f,
    0.415034424f,  0.909805708f,
    0.416429560f,  0.909167983f,
    0.417823716f,  0.908528119f,
    0.419216888f,  0.907886116f,
    0.420609074f,  0.907241978f,
    0.422000271f,  0.906595705f,
    0.423390474f,  0.905947298f,
    0.424779681f,  0.905296759f,
    0.426167889f,  0.904644091f,
    0.427555093f,  0.903989293f,
    0.428941292f,  0.903332368f,
    0.430326481f,  0.902673318f,
    0.431710658f,  0.902012144f,
    0.433093819f,  0.901348847f,
    0.434475961f,  0.900683429f,
    0.435857080f,  0.900015892f,
    0.437237174f,  0.899346237f,
    0.438616239f,  0.898674466f,
    0.439994271f,  0.898000580f,
    0.441371269f,  0.897324581f,
    0.442747228f,  0.896646470f,
    0.444122145f,  0.895966250f,
    0.445496017f,  0.895283921f,
    0.446868840f,  0.894599486f,
    0.448240612f,  0.893912945f,
    0.449611330f,  0.893224301f,
    0.450980989f,  0.892533555f,
    0.452349587f,  0.891840709f,
    0.453717121f,  0.891145765f,
    0.455083587f,  0.890448723f,
    0.456448982f,  0.889749586f,
    0.457813304f,  0.889048356f,
    0.459176548f,  0.888345033f,
    0.460538711f,  0.887639620f,
    0.461899791f,  0.886932119f,
    0.463259784f,  0.886222530f,
    0.464618686f,  0.885510856f,
    0.465976496f,  0.884797098f,
    0.467333209f,  0.884081259f,
    0.468688822f,  0.883363339f,
    0.470043332f,  0.882643340f,
    0.471396737f,  0.881921264f,
    0.472749032f,  0.881197113f,
    0.474100215f,  0.880470889f,
    0.475450282f,  0.879742593f,
    0.476799230f,  0.879012226f,
    0.478147056f,  0.878279792f,
    0.479493758f,  0.877545290f,
    0.480839331f,  0.876808724f,
    0.482183772f,  0.876070094f,
    0.483527079f,  0.875329403f,
    0.484869248f,  0.874586652f,
    0.486210276f,  0.873841843f,
    0.487550160f,  0.873094978f,
    0.488888897f,  0.872346059f,
    0.490226483f,  0.871595087f,
    0.491562916f,  0.870842063f,
    0.492898192f,  0.870086991f,
    0.494232309f,  0.869329871f,
    0.495565262f,  0.868570706f,
    0.496897049f,  0.867809497f,
    0.498227667f,  0.867046246f,
    0.499557113f,  0.866280954f,
    0.500885383f,  0.865513624f,
    0.502212474f,  0.864744258f,
    0.503538384f,  0.863972856f,
    0.504863109f,  0.863199422f,
    0.506186645f,  0.862423956f,
    0.507508991f,  0.861646461f,
    0.508830143f,  0.860866939f,
    0.510150097f,  0.860085390f,
    0.511468850f,  0.859301818f,
    0.512786401f,  0.858516224f,
    0.514102744f,  0.857728610f,
    0.515417878f,  0.856938977f,
    0.516731799f,  0.856147328f,
    0.518044504f,  0.855353665f,
    0.519355990f,  0.854557988f,
    0.520666254f,  0.853760301f,
    0.521975293f,  0.852960605f,
    0.523283103f,  0.852158902f,
    0.524589683f,  0.851355193f,
    0.525895027f,  0.850549481f,
    0.527199135f,  0.849741768f,
    0.528502002f,  0.848932055f,
    0.529803625f,  0.848120345f,
    0.531104001f,  0.847306639f,
    0.532403128f,  0.846490939f,
    0.533701002f,  0.845673247f,
    0.534997620f,  0.844853565f,
    0.536292979f,  0.844031895f,
    0.537587076f,  0.843208240f,
    0.538879909f,  0.842382600f,
    0.540171473f,  0.841554977f,
    0.541461766f,  0.840725375f,
    0.542750785f,  0.839893794f,
    0.544038527f,  0.839060237f,
    0.545324988f,  0.838224706f,
    0.546610167f,  0.837387202f,
    0.547894059f,  0.836547727f,
    0.549176662f,  0.835706284f,
    0.550457973f,  0.834862875f,
    0.551737988f,  0.834017501f,
    0.553016706f,  0.833170165f,
    0.554294121f,  0.832320868f,
    0.555570233f,  0.831469612f,
    0.556845037f,  0.830616400f,
    0.558118531f,  0.829761234f,
    0.559390712f,  0.828904115f,
    0.560661576f,  0.828045045f,
    0.561931121f,  0.827184027f,
    0.563199344f,  0.826321063f,
    0.564466242f,  0.825456154f,
    0.565731811f,  0.824589303f,
    0.566996049f,  0.823720511f,
    0.568258953f,  0.822849781f,
    0.569520519f,  0.821977115f,
    0.570780746f,  0.821102515f,
    0.572039629f,  0.820225983f,
    0.573297167f,  0.819347520f,
    0.574553355f,  0.818467130f,
    0.575808191f,  0.817584813f,
    0.577061673f,  0.816700573f,
    0.578313796f,  0.815814411f,
    0.579564559f,  0.814926329f,
    0.580813958f,  0.814036330f,
    0.582061990f,  0.813144415f,
    0.583308653f,  0.812250587f,
    0.584553943f,  0.811354847f,
    0.585797857f,  0.810457198f,
    0.587040394f,  0.809557642f,
    0.588281548f,  0.808656182f,
    0.589521319f,  0.807752818f,
    0.590759702f,  0.806847554f,
    0.591996695f,  0.805940391f,
    0.593232295f,  0.805031331f,
    0.594466499f,  0.804120377f,
    0.595699304f,  0.803207531f,
    0.596930708f,  0.802292796f,
    0.598160707f,  0.801376172f,
    0.599389298f,  0.800457662f,
    0.600616479f,  0.799537269f,
    0.601842247f,  0.798614995f,
    0.603066599f,  0.797690841f,
    0.604289531f,  0.796764810f,
    0.605511041f,  0.795836905f,
    0.606731127f,  0.794907126f,
    0.607949785f,  0.793975478f,
    0.609167012f,  0.793041960f,
    0.610382806f,  0.792106577f,
    0.611597164f,  0.791169330f,
    0.612810082f,  0.790230221f,
    0.614021559f,  0.789289253f,
    0.615231591f,  0.788346428f,
    0.616440175f,  0.787401747f,
    0.617647308f,  0.786455214f,
    0.618852988f,  0.785506830f,
    0.620057212f,  0.784556597f,
    0.621259977f,  0.783604519f,
    0.622461279f,  0.782650596f,
    0.623661118f,  0.781694832f,
    0.624859488f,  0.780737229f,
    0.626056388f,  0.779777788f,
    0.627251815f,  0.778816512f,
    0.628445767f,  0.777853404f,
    0.629638239f,  0.776888466f,
    0.630829230f,  0.775921699f,
    0.632018736f,  0.774953107f,
    0.633206755f,  0.773982691f,
    0.634393284f,  0.773010453f,
    0.635578320f,  0.772036397f,
    0.636761861f,  0.771060524f,
    0.637943904f,  0.770082837f,
    0.639124445f,  0.769103338f,
    0.640303482f,  0.768122029f,
    0.641481013f,  0.767138912f,
    0.642657034f,  0.766153990f,
    0.643831543f,  0.765167266f,
    0.645004537f,  0.764178741f,
    0.646176013f,  0.763188417f,
    0.647345969f,  0.762196298f,
    0.648514401f,  0.761202385f,
    0.649681307f,  0.760206682f,
    0.650846685f,  0.759209189f,
    0.652010531f,  0.758209910f,
    0.653172843f,  0.757208847f,
    0.654333618f,  0.756206001f,
    0.655492853f,  0.755201377f,
    0.656650546f,  0.754194975f,
    0.657806693f,  0.753186799f,
    0.658961293f,  0.752176850f,
    0.660114342f,  0.751165132f,
    0.661265838f,  0.750151646f,
    0.662415778f,  0.749136395f,
    0.663564159f,  0.748119380f,
    0.664710978f,  0.747100606f,
    0.665856234f,  0.746080074f,
    0.666999922f,  0.745057785f,
    0.668142041f,  0.744033744f,
    0.669282588f,  0.743007952f,
    0.670421560f,  0.741980412f,
    0.671558955f,  0.740951125f,
    0.672694769f,  0.739920095f,
    0.673829000f,  0.738887324f,
    0.674961646f,  0.737852815f,
    0.676092704f,  0.736816569f,
    0.677222170f,  0.735778589f,
    0.678350043f,  0.734738878f,
    0.679476320f,  0.733697438f,
    0.680600998f,  0.732654272f,
    0.681724074f,  0.731609381f,
    0.682845546f,  0.730562769f,
    0.683965412f,  0.729514438f,
    0.685083668f,  0.728464390f,
    0.686200312f,  0.727412629f,
    0.687315341f,  0.726359155f,
    0.688428753f,  0.725303972f,
    0.689540545f,  0.724247083f,
    0.690650714f,  0.723188489f,
    0.691759258f,  0.722128194f,
    0.692866175f,  0.721066199f,
    0.693971461f,  0.720002508f,
    0.695075114f,  0.718937122f,
    0.696177131f,  0.717870045f,
    0.697277511f,  0.716801279f,
    0.698376249f,  0.715730825f,
    0.699473345f,  0.714658688f,
    0.700568794f,  0.713584869f,
    0.701662595f,  0.712509371f,
    0.702754744f,  0.711432196f,
    0.703845241f,  0.710353347f,
    0.704934080f,  0.709272826f,
    0.706021261f,  0.708190637f,
    0.707106781f,  0.707106781f,
    0.708190637f,  0.706021261f,
    0.709272826f,  0.704934080f,
    0.710353347f,  0.703845241f,
    0.711432196f,  0.702754744f,
    0.712509371f,  0.701662595f,
    0.713584869f,  0.700568794f,
    0.714658688f,  0.699473345f,
    0.715730825f,  0.698376249f,
    0.716801279f,  0.697277511f,
    0.717870045f,  0.696177131f,
    0.718937122f,  0.695075114f,
    0.720002508f,  0.693971461f,
    0.721066199f,  0.692866175f,
    0.722128194f,  0.691759258f,
    0.723188489f,  0.690650714f,
    0.724247083f,  0.689540545f,
    0.725303972f,  0.688428753f,
    0.726359155f,  0.687315341f,
    0.727412629f,  0.686200312f,
    0.728464390f,  0.685083668f,
    0.729514438f,  0.683965412f,
    0.730562769f,  0.682845546f,
    0.731609381f,  0.681724074f,
    0.732654272f,  0.680600998f,
    0.733697438f,  0.679476320f,
    0.734738878f,  0.678350043f,
    0.735778589f,  0.677222170f,
    0.736816569f,  0.676092704f,
    0.737852815f,  0.674961646f,
    0.738887324f,  0.673829000f,
    0.739920095f,  0.672694769f,
    0.740951125f,  0.671558955f,
    0.741980412f,  0.670421560f,
    0.743007952f,  0.669282588f,
    0.744033744f,  0.668142041f,
    0.745057785f,  0.666999922f,
    0.746080074f,  0.665856234f,
    0.747100606f,  0.664710978f,
    0.748119380f,  0.663564159f,
    0.749136395f,  0.662415778f,
    0.750151646f,  0.661265838f,
    0.751165132f,  0.660114342f,
    0.752176850f,  0.658961293f,
    0.753186799f,  0.657806693f,
    0.754194975f,  0.656650546f,
    0.755201377f,  0.655492853f,
    0.756206001f,  0.654333618f,
    0.757208847f,  0.653172843f,
    0.758209910f,  0.652010531f,
    0.759209189f,  0.650846685f,
    0.760206682f,  0.649681307f,
    0.761202385f,  0.648514401f,
    0.762196298f,  0.647345969f,
    0.763188417f,  0.646176013f,
    0.764178741f,  0.645004537f,
    0.765167266f,  0.643831543f,
    0.766153990f,  0.642657034f,
    0.767138912f,  0.641481013f,
    0.768122029f,  0.640303482f,
    0.769103338f,  0.639124445f,
    0.770082837f,  0.637943904f,
    0.771060524f,  0.636761861f,
    0.772036397f,  0.635578320f,
    0.773010453f,  0.634393284f,
    0.773982691f,  0.633206755f,
    0.774953107f,  0.632018736f,
    0.775921699f,  0.630829230f,
    0.776888466f,  0.629638239f,
    0.777853404f,  0.628445767f,
    0.778816512f,  0.627251815f,
    0.779777788f,  0.626056388f,
    0.780737229f,  0.624859488f,
    0.781694832f,  0.623661118f,
    0.782650596f,  0.622461279f,
    0.783604519f,  0.621259977f,
    0.784556597f,  0.620057212f,
    0.785506830f,  0.618852988f,
    0.786455214f,  0.617647308f,
    0.787401747f,  0.616440175f,
    0.788346428f,  0.615231591f,
    0.789289253f,  0.614021559f,
    0.790230221f,  0.612810082f,
    0.791169330f,  0.611597164f,
    0.792106577f,  0.610382806f,
    0.793041960f,  0.609167012f,
    0.793975478f,  0.607949785f,
    0.794907126f,  0.606731127f,
    0.795836905f,  0.605511041f,
    0.796764810f,  0.604289531f,
    0.797690841f,  0.603066599f,
    0.798614995f,  0.601842247f,
    0.799537269f,  0.600616479f,
    0.800457662f,  0.599389298f,
    0.801376172f,  0.598160707f,
    0.802292796f,  0.596930708f,
    0.803207531f,  0.595699304f,
    0.804120377f,  0.594466499f,
    0.805031331f,  0.593232295f,
    0.805940391f,  0.591996695f,
    0.806847554f,  0.590759702f,
    0.807752818f,  0.589521319f,
    0.808656182f,  0.588281548f,
    0.809557642f,  0.587040394f,
    0.810457198f,  0.585797857f,
    0.811354847f,  0.584553943f,
    0.812250587f,  0.583308653f,
    0.813144415f,  0.582061990f,
    0.814036330f,  0.580813958f,
    0.814926329f,  0.579564559f,
    0.815814411f,  0.578313796f,
    0.816700573f,  0.577061673f,
    0.817584813f,  0.575808191f,
    0.818467130f,  0.574553355f,
    0.819347520f,  0.573297167f,
    0.820225983f,  0.572039629f,
    0.821102515f,  0.570780746f,
    0.821977115f,  0.569520519f,
    0.822849781f,  0.568258953f,
    0.823720511f,  0.566996049f,
    0.824589303f,  0.565731811f,
    0.825456154f,  0.564466242f,
    0.826321063f,  0.563199344f,
    0.827184027f,  0.561931121f,
    0.828045045f,  0.560661576f,
    0.828904115f,  0.559390712f,
    0.829761234f,  0.558118531f,
    0.830616400f,  0.556845037f,
    0.831469612f,  0.555570233f,
    0.832320868f,  0.554294121f,
    0.833170165f,  0.553016706f,
    0.834017501f,  0.551737988f,
    0.834862875f,  0.550457973f,
    0.835706284f,  0.549176662f,
    0.836547727f,  0.547894059f,
    0.837387202f,  0.546610167f,
    0.838224706f,  0.545324988f,
    0.839060237f,  0.544038527f,
    0.839893794f,  0.542750785f,
    0.840725375f,  0.541461766f,
    0.841554977f,  0.540171473f,
    0.842382600f,  0.538879909f,
    0.843208240f,  0.537587076f,
    0.844031895f,  0.536292979f,
    0.844853565f,  0.534997620f,
    0.845673247f,  0.533701002f,
    0.846490939f,  0.532403128f,
    0.847306639f,  0.531104001f,
    0.848120345f,  0.529803625f,
    0.848932055f,  0.528502002f,
    0.849741768f,  0.527199135f,
    0.850549481f,  0.525895027f,
    0.851355193f,  0.524589683f,
    0.852158902f,  0.523283103f,
    0.852960605f,  0.521975293f,
    0.853760301f,  0.520666254f,
    0.854557988f,  0.519355990f,
    0.855353665f,  0.518044504f,
    0.856147328f,  0.516731799f,
    0.856938977f,  0.515417878f,
    0.857728610f,  0.514102744f,
    0.858516224f,  0.512786401f,
    0.859301818f,  0.511468850f,
    0.860085390f,  0.510150097f,
    0.860866939f,  0.508830143f,
    0.861646461f,  0.507508991f,
    0.862423956f,  0.506186645f,
    0.863199422f,  0.504863109f,
    0.863972856f,  0.503538384f,
    0.864744258f,  0.502212474f,
    0.865513624f,  0.500885383f,
    0.866280954f,  0.499557113f,
    0.867046246f,  0.498227667f,
    0.867809497f,  0.496897049f,
    0.868570706f,  0.495565262f,
    0.869329871f,  0.494232309f,
    0.870086991f,  0.492898192f,
    0.870842063f,  0.491562916f,
    0.871595087f,  0.490226483f,
    0.872346059f,  0.488888897f,
    0.873094978f,  0.487550160f,
    0.873841843f,  0.486210276f,
    0.874586652f,  0.484869248f,
    0.875329403f,  0.483527079f,
    0.876070094f,  0.482183772f,
    0.876808724f,  0.480839331f,
    0.877545290f,  0.479493758f,
    0.878279792f,  0.478147056f,
    0.879012226f,  0.476799230f,
    0.879742593f,  0.475450282f,
    0.880470889f,  0.474100215f,
    0.881197113f,  0.472749032f,
    0.881921264f,  0.471396737f,
    0.882643340f,  0.470043332f,
    0.883363339f,  0.468688822f,
    0.884081259f,  0.467333209f,
    0.884797098f,  0.465976496f,
    0.885510856f,  0.464618686f,
    0.886222530f,  0.463259784f,
    0.886932119f,  0.461899791f,
    0.887639620f,  0.460538711f,
    0.888345033f,  0.459176548f,
    0.889048356f,  0.457813304f,
    0.889749586f,  0.456448982f,
    0.890448723f,  0.455083587f,
    0.891145765f,  0.453717121f,
    0.891840709f,  0.452349587f,
    0.892533555f,  0.450980989f,
    0.893224301f,  0.449611330f,
    0.893912945f,  0.448240612f,
    0.894599486f,  0.446868840f,
    0.895283921f,  0.445496017f,
    0.895966250f,  0.444122145f,
    0.896646470f,  0.442747228f,
    0.897324581f,  0.441371269f,
    0.898000580f,  0.439994271f,
    0.898674466f,  0.438616239f,
    0.899346237f,  0.437237174f,
    0.900015892f,  0.435857080f,
    0.900683429f,  0.434475961f,
    0.901348847f,  0.433093819f,
    0.902012144f,  0.431710658f,
    0.902673318f,  0.430326481f,
    0.903332368f,  0.428941292f,
    0.903989293f,  0.427555093f,
    0.904644091f,  0.426167889f,
    0.905296759f,  0.424779681f,
    0.905947298f,  0.423390474f,
    0.906595705f,  0.422000271f,
    0.907241978f,  0.420609074f,
    0.907886116f,  0.419216888f,
    0.908528119f,  0.417823716f,
    0.909167983f,  0.416429560f,
    0.909805708f,  0.415034424f,
    0.910441292f,  0.413638312f,
    0.911074734f,  0.412241227f,
    0.911706032f,  0.410843171f,
    0.912335185f,  0.409444149f,
    0.912962190f,  0.408044163f,
    0.913587048f,  0.406643217f,
    0.914209756f,  0.405241314f,
    0.914830312f,  0.403838458f,
    0.915448716f,  0.402434651f,
    0.916064966f,  0.401029897f,
    0.916679060f,  0.399624200f,
    0.917290997f,  0.398217562f,
    0.917900776f,  0.396809987f,
    0.918508394f,  0.395401479f,
    0.919113852f,  0.393992040f,
    0.919717146f,  0.392581674f,
    0.920318277f,  0.391170384f,
    0.920917242f,  0.389758174f,
    0.921514039f,  0.388345047f,
    0.922108669f,  0.386931006f,
    0.922701128f,  0.385516054f,
    0.923291417f,  0.384100195f,
    0.923879533f,  0.382683432f,
    0.924465474f,  0.381265769f,
    0.925049241f,  0.379847209f,
    0.925630831f,  0.378427755f,
    0.926210242f,  0.377007410f,
    0.926787474f,  0.375586178f,
    0.927362526f,  0.374164063f,
    0.927935395f,  0.372741067f,
    0.928506080f,  0.371317194f,
    0.929074581f,  0.369892447f,
    0.929640896f,  0.368466830f,
    0.930205023f,  0.367040346f,
    0.930766961f,  0.365612998f,
    0.931326709f,  0.364184790f,
    0.931884266f,  0.362755724f,
    0.932439629f,  0.361325806f,
    0.932992799f,  0.359895037f,
    0.933543773f,  0.358463421f,
    0.934092550f,  0.357030961f,
    0.934639130f,  0.355597662f,
    0.935183510f,  0.354163525f,
    0.935725689f,  0.352728556f,
    0.936265667f,  0.351292756f,
    0.936803442f,  0.349856130f,
    0.937339012f,  0.348418680f,
    0.937872376f,  0.346980411f,
    0.938403534f,  0.345541325f,
    0.938932484f,  0.344101426f,
    0.939459224f,  0.342660717f,
    0.939983753f,  0.341219202f,
    0.940506071f,  0.339776884f,
    0.941026175f,  0.338333767f,
    0.941544065f,  0.336889853f,
    0.942059740f,  0.335445147f,
    0.942573198f,  0.333999651f,
    0.943084437f,  0.332553370f,
    0.943593458f,  0.331106306f,
    0.944100258f,  0.329658463f,
    0.944604837f,  0.328209844f,
    0.945107193f,  0.326760452f,
    0.945607325f,  0.325310292f,
    0.946105232f,  0.323859367f,
    0.946600913f,  0.322407679f,
    0.947094366f,  0.320955232f,
    0.947585591f,  0.319502031f,
    0.948074586f,  0.318048077f,
    0.948561350f,  0.316593376f,
    0.949045882f,  0.315137929f,
    0.949528181f,  0.313681740f,
    0.950008245f,  0.312224814f,
    0.950486074f,  0.310767153f,
    0.950961666f,  0.309308760f,
    0.951435021f,  0.307849640f,
    0.951906137f,  0.306389795f,
    0.952375013f,  0.304929230f,
    0.952841648f,  0.303467947f,
    0.953306040f,  0.302005949f,
    0.953768190f,  0.300543241f,
    0.954228095f,  0.299079826f,
    0.954685755f,  0.297615707f,
    0.955141168f,  0.296150888f,
    0.955594334f,  0.294685372f,
    0.956045251f,  0.293219163f,
    0.956493919f,  0.291752263f,
    0.956940336f,  0.290284677f,
    0.957384501f,  0.288816408f,
    0.957826413f,  0.287347460f,
    0.958266071f,  0.285877835f,
    0.958703475f,  0.284407537f,
    0.959138622f,  0.282936570f,
    0.959571513f,  0.281464938f,
    0.960002146f,  0.279992643f,
    0.960430519f,  0.278519689f,
    0.960856633f,  0.277046080f,
    0.961280486f,  0.275571819f,
    0.961702077f,  0.274096910f,
    0.962121404f,  0.272621355f,
    0.962538468f,  0.271145160f,
    0.962953267f,  0.269668326f,
    0.963365800f,  0.268190857f,
    0.963776066f,  0.266712757f,
    0.964184064f,  0.265234030f,
    0.964589793f,  0.263754679f,
    0.964993253f,  0.262274707f,
    0.965394442f,  0.260794118f,
    0.965793359f,  0.259312915f,
    0.966190003f,  0.257831102f,
    0.966584374f,  0.256348682f,
    0.966976471f,  0.254865660f,
    0.967366292f,  0.253382037f,
    0.967753837f,  0.251897818f,
    0.968139105f,  0.250413007f,
    0.968522094f,  0.248927606f,
    0.968902805f,  0.247441619f,
    0.969281235f,  0.245955050f,
    0.969657385f,  0.244467903f,
    0.970031253f,  0.242980180f,
    0.970402839f,  0.241491885f,
    0.970772141f,  0.240003022f,
    0.971139158f,  0.238513595f,
    0.971503891f,  0.237023606f,
    0.971866337f,  0.235533059f,
    0.972226497f,  0.234041959f,
    0.972584369f,  0.232550307f,
    0.972939952f,  0.231058108f,
    0.973293246f,  0.229565366f,
    0.973644250f,  0.228072083f,
    0.973992962f,  0.226578264f,
    0.974339383f,  0.225083911f,
    0.974683511f,  0.223589029f,
    0.975025345f,  0.222093621f,
    0.975364885f,  0.220597690f,
    0.975702130f,  0.219101240f,
    0.976037079f,  0.217604275f,
    0.976369731f,  0.216106797f,
    0.976700086f,  0.214608811f,
    0.977028143f,  0.213110320f,
    0.977353900f,  0.211611327f,
    0.977677358f,  0.210111837f,
    0.977998515f,  0.208611852f,
    0.978317371f,  0.207111376f,
    0.978633924f,  0.205610413f,
    0.978948175f,  0.204108966f,
    0.979260123f,  0.202607039f,
    0.979569766f,  0.201104635f,
    0.979877104f,  0.199601758f,
    0.980182136f,  0.198098411f,
    0.980484862f,  0.196594598f,
    0.980785280f,  0.195090322f,
    0.981083391f,  0.193585587f,
    0.981379193f,  0.192080397f,
    0.981672686f,  0.190574755f,
    0.981963869f,  0.189068664f,
    0.982252741f,  0.187562129f,
    0.982539302f,  0.186055152f,
    0.982823551f,  0.184547737f,
    0.983105487f,  0.183039888f,
    0.983385110f,  0.181531608f,
    0.983662419f,  0.180022901f,
    0.983937413f,  0.178513771f,
    0.984210092f,  0.177004220f,
    0.984480455f,  0.175494253f,
    0.984748502f,  0.173983873f,
    0.985014231f,  0.172473084f,
    0.985277642f,  0.170961889f,
    0.985538735f,  0.169450291f,
    0.985797509f,  0.167938295f,
    0.986053963f,  0.166425904f,
    0.986308097f,  0.164913120f,
    0.986559910f,  0.163399949f,
    0.986809402f,  0.161886394f,
    0.987056571f,  0.160372457f,
    0.987301418f,  0.158858143f,
    0.987543942f,  0.157343456f,
    0.987784142f,  0.155828398f,
    0.988022017f,  0.154312973f,
    0.988257568f,  0.152797185f,
    0.988490793f,  0.151281038f,
    0.988721692f,  0.149764535f,
    0.988950265f,  0.148247679f,
    0.989176510f,  0.146730474f,
    0.989400428f,  0.145212925f,
    0.989622017f,  0.143695033f,
    0.989841278f,  0.142176804f,
    0.990058210f,  0.140658239f,
    0.990272812f,  0.139139344f,
    0.990485084f,  0.137620122f,
    0.990695025f,  0.136100575f,
    0.990902635f,  0.134580709f,
    0.991107914f,  0.133060525f,
    0.991310860f,  0.131540029f,
    0.991511473f,  0.130019223f,
    0.991709754f,  0.128498111f,
    0.991905700f,  0.126976696f,
    0.992099313f,  0.125454983f,
    0.992290591f,  0.123932975f,
    0.992479535f,  0.122410675f,
    0.992666142f,  0.120888087f,
    0.992850414f,  0.119365215f,
    0.993032350f,  0.117842062f,
    0.993211949f,  0.116318631f,
    0.993389211f,  0.114794927f,
    0.993564136f,  0.113270952f,
    0.993736722f,  0.111746711f,
    0.993906970f,  0.110222207f,
    0.994074879f,  0.108697444f,
    0.994240449f,  0.107172425f,
    0.994403680f,  0.105647154f,
    0.994564571f,  0.104121634f,
    0.994723121f,  0.102595869f,
    0.994879331f,  0.101069863f,
    0.995033199f,  0.099543619f,
    0.995184727f,  0.098017140f,
    0.995333912f,  0.096490431f,
    0.995480755f,  0.094963495f,
    0.995625256f,  0.093436336f,
    0.995767414f,  0.091908956f,
    0.995907229f,  0.090381361f,
    0.996044701f,  0.088853553f,
    0.996179829f,  0.087325535f,
    0.996312612f,  0.085797312f,
    0.996443051f,  0.084268888f,
    0.996571146f,  0.082740265f,
    0.996696895f,  0.081211447f,
    0.996820299f,  0.079682438f,
    0.996941358f,  0.078153242f,
    0.997060070f,  0.076623861f,
    0.997176437f,  0.075094301f,
    0.997290457f,  0.073564564f,
    0.997402130f,  0.072034653f,
    0.997511456f,  0.070504573f,
    0.997618435f,  0.068974328f,
    0.997723067f,  0.067443920f,
    0.997825350f,  0.065913353f,
    0.997925286f,  0.064382631f,
    0.998022874f,  0.062851758f,
    0.998118113f,  0.061320736f,
    0.998211003f,  0.059789571f,
    0.998301545f,  0.058258265f,
    0.998389737f,  0.056726821f,
    0.998475581f,  0.055195244f,
    0.998559074f,  0.053663538f,
    0.998640218f,  0.052131705f,
    0.998719012f,  0.050599749f,
    0.998795456f,  0.049067674f,
    0.998869550f,  0.047535484f,
    0.998941293f,  0.046003182f,
    0.999010686f,  0.044470772f,
    0.999077728f,  0.042938257f,
    0.999142419f,  0.041405641f,
    0.999204759f,  0.039872928f,
    0.999264747f,  0.038340120f,
    0.999322385f,  0.036807223f,
    0.999377670f,  0.035274239f,
    0.999430605f,  0.033741172f,
    0.999481187f,  0.032208025f,
    0.999529418f,  0.030674803f,
    0.999575296f,  0.029141509f,
    0.999618822f,  0.027608146f,
    0.999659997f,  0.026074718f,
    0.999698819f,  0.024541229f,
    0.999735288f,  0.023007681f,
    0.999769405f,  0.021474080f,
    0.999801170f,  0.019940429f,
    0.999830582f,  0.018406730f,
    0.999857641f,  0.016872988f,
    0.999882347f,  0.015339206f,
    0.999904701f,  0.013805389f,
    0.999924702f,  0.012271538f,
    0.999942350f,  0.010737659f,
    0.999957645f,  0.009203755f,
    0.999970586f,  0.007669829f,
    0.999981175f,  0.006135885f,
    0.999989411f,  0.004601926f,
    0.999995294f,  0.003067957f,
    0.999998823f,  0.001533980f,
    1.000000000f,  0.000000000f,
    0.999998823f, -0.001533980f,
    0.999995294f, -0.003067957f,
    0.999989411f, -0.004601926f,
    0.999981175f, -0.006135885f,
    0.999970586f, -0.007669829f,
    0.999957645f, -0.009203755f,
    0.999942350f, -0.010737659f,
    0.999924702f, -0.012271538f,
    0.999904701f, -0.013805389f,
    0.999882347f, -0.015339206f,
    0.999857641f, -0.016872988f,
    0.999830582f, -0.018406730f,
    0.999801170f, -0.019940429f,
    0.999769405f, -0.021474080f,
    0.999735288f, -0.023007681f,
    0.999698819f, -0.024541229f,
    0.999659997f, -0.026074718f,
    0.999618822f, -0.027608146f,
    0.999575296f, -0.029141509f,
    0.999529418f, -0.030674803f,
    0.999481187f, -0.032208025f,
    0.999430605f, -0.033741172f,
    0.999377670f, -0.035274239f,
    0.999322385f, -0.036807223f,
    0.999264747f, -0.038340120f,
    0.999204759f, -0.039872928f,
    0.999142419f, -0.041405641f,
    0.999077728f, -0.042938257f,
    0.999010686f, -0.044470772f,
    0.998941293f, -0.046003182f,
    0.998869550f, -0.047535484f,
    0.998795456f, -0.049067674f,
    0.998719012f, -0.050599749f,
    0.998640218f, -0.052131705f,
    0.998559074f, -0.053663538f,
    0.998475581f, -0.055195244f,
    0.998389737f, -0.056726821f,
    0.998301545f, -0.058258265f,
    0.998211003f, -0.059789571f,
    0.998118113f, -0.061320736f,
    0.998022874f, -0.062851758f,
    0.997925286f, -0.064382631f,
    0.997825350f, -0.065913353f,
    0.997723067f, -0.067443920f,
    0.997618435f, -0.068974328f,
    0.997511456f, -0.070504573f,
    0.997402130f, -0.072034653f,
    0.997290457f, -0.073564564f,
    0.997176437f, -0.075094301f,
    0.997060070f, -0.076623861f,
    0.996941358f, -0.078153242f,
    0.996820299f, -0.079682438f,
    0.996696895f, -0.081211447f,
    0.996571146f, -0.082740265f,
    0.996443051f, -0.084268888f,
    0.996312612f, -0.085797312f,
    0.996179829f, -0.087325535f,
    0.996044701f, -0.088853553f,
    0.995907229f, -0.090381361f,
    0.995767414f, -0.091908956f,
    0.995625256f, -0.093436336f,
    0.995480755f, -0.094963495f,
    0.995333912f, -0.096490431f,
    0.995184727f, -0.098017140f,
    0.995033199f, -0.099543619f,
    0.994879331f, -0.101069863f,
    0.994723121f, -0.102595869f,
    0.994564571f, -0.104121634f,
    0.994403680f, -0.105647154f,
    0.994240449f, -0.107172425f,
    0.994074879f, -0.108697444f,
    0.993906970f, -0.110222207f,
    0.993736722f, -0.111746711f,
    0.993564136f, -0.113270952f,
    0.993389211f, -0.114794927f,
    0.993211949f, -0.116318631f,
    0.993032350f, -0.117842062f,
    0.992850414f, -0.119365215f,
    0.992666142f, -0.120888087f,
    0.992479535f, -0.122410675f,
    0.992290591f, -0.123932975f,
    0.992099313f, -0.125454983f,
    0.991905700f, -0.126976696f,
    0.991709754f, -0.128498111f,
    0.991511473f, -0.130019223f,
    0.991310860f, -0.131540029f,
    0.991107914f, -0.133060525f,
    0.990902635f, -0.134580709f,
    0.990695025f, -0.136100575f,
    0.990485084f, -0.137620122f,
    0.990272812f, -0.139139344f,
    0.990058210f, -0.140658239f,
    0.989841278f, -0.142176804f,
    0.989622017f, -0.143695033f,
    0.989400428f, -0.145212925f,
    0.989176510f, -0.146730474f,
    0.988950265f, -0.148247679f,
    0.988721692f, -0.149764535f,
    0.988490793f, -0.151281038f,
    0.988257568f, -0.152797185f,
    0.988022017f, -0.154312973f,
    0.987784142f, -0.155828398f,
    0.987543942f, -0.157343456f,
    0.987301418f, -0.158858143f,
    0.987056571f, -0.160372457f,
    0.986809402f, -0.161886394f,
    0.986559910f, -0.163399949f,
    0.986308097f, -0.164913120f,
    0.986053963f, -0.166425904f,
    0.985797509f, -0.167938295f,
    0.985538735f, -0.169450291f,
    0.985277642f, -0.170961889f,
    0.985014231f, -0.172473084f,
    0.984748502f, -0.173983873f,
    0.984480455f, -0.175494253f,
    0.984210092f, -0.177004220f,
    0.983937413f, -0.178513771f,
    0.983662419f, -0.180022901f,
    0.983385110f, -0.181531608f,
    0.983105487f, -0.183039888f,
    0.982823551f, -0.184547737f,
    0.982539302f, -0.186055152f,
    0.982252741f, -0.187562129f,
    0.981963869f, -0.189068664f,
    0.981672686f, -0.190574755f,
    0.981379193f, -0.192080397f,
    0.981083391f, -0.193585587f,
    0.980785280f, -0.195090322f,
    0.980484862f, -0.196594598f,
    0.980182136f, -0.198098411f,
    0.979877104f, -0.199601758f,
    0.979569766f, -0.201104635f,
    0.979260123f, -0.202607039f,
    0.978948175f, -0.204108966f,
    0.978633924f, -0.205610413f,
    0.978317371f, -0.207111376f,
    0.977998515f, -0.208611852f,
    0.977677358f, -0.210111837f,
    0.977353900f, -0.211611327f,
    0.977028143f, -0.213110320f,
    0.976700086f, -0.214608811f,
    0.976369731f, -0.216106797f,
    0.976037079f, -0.217604275f,
    0.975702130f, -0.219101240f,
    0.975364885f, -0.220597690f,
    0.975025345f, -0.222093621f,
    0.974683511f, -0.223589029f,
    0.974339383f, -0.225083911f,
    0.973992962f, -0.226578264f,
    0.973644250f, -0.228072083f,
    0.973293246f, -0.229565366f,
    0.972939952f, -0.231058108f,
    0.972584369f, -0.232550307f,
    0.972226497f, -0.234041959f,
    0.971866337f, -0.235533059f,
    0.971503891f, -0.237023606f,
    0.971139158f, -0.238513595f,
    0.970772141f, -0.240003022f,
    0.970402839f, -0.241491885f,
    0.970031253f, -0.242980180f,
    0.969657385f, -0.244467903f,
    0.969281235f, -0.245955050f,
    0.968902805f, -0.247441619f,
    0.968522094f, -0.248927606f,
    0.968139105f, -0.250413007f,
    0.967753837f, -0.251897818f,
    0.967366292f, -0.253382037f,
    0.966976471f, -0.254865660f,
    0.966584374f, -0.256348682f,
    0.966190003f, -0.257831102f,
    0.965793359f, -0.259312915f,
    0.965394442f, -0.260794118f,
    0.964993253f, -0.262274707f,
    0.964589793f, -0.263754679f,
    0.964184064f, -0.265234030f,
    0.963776066f, -0.266712757f,
    0.963365800f, -0.268190857f,
    0.962953267f, -0.269668326f,
    0.962538468f, -0.271145160f,
    0.962121404f, -0.272621355f,
    0.961702077f, -0.274096910f,
    0.961280486f, -0.275571819f,
    0.960856633f, -0.277046080f,
    0.960430519f, -0.278519689f,
    0.960002146f, -0.279992643f,
    0.959571513f, -0.281464938f,
    0.959138622f, -0.282936570f,
    0.958703475f, -0.284407537f,
    0.958266071f, -0.285877835f,
    0.957826413f, -0.287347460f,
    0.957384501f, -0.288816408f,
    0.956940336f, -0.290284677f,
    0.956493919f, -0.291752263f,
    0.956045251f, -0.293219163f,
    0.955594334f, -0.294685372f,
    0.955141168f, -0.296150888f,
    0.954685755f, -0.297615707f,
    0.954228095f, -0.299079826f,
    0.953768190f, -0.300543241f,
    0.953306040f, -0.302005949f,
    0.952841648f, -0.303467947f,
    0.952375013f, -0.304929230f,
    0.951906137f, -0.306389795f,
    0.951435021f, -0.307849640f,
    0.950961666f, -0.309308760f,
    0.950486074f, -0.310767153f,
    0.950008245f, -0.312224814f,
    0.949528181f, -0.313681740f,
    0.949045882f, -0.315137929f,
    0.948561350f, -0.316593376f,
    0.948074586f, -0.318048077f,
    0.947585591f, -0.319502031f,
    0.947094366f, -0.320955232f,
    0.946600913f, -0.322407679f,
    0.946105232f, -0.323859367f,
    0.945607325f, -0.325310292f,
    0.945107193f, -0.326760452f,
    0.944604837f, -0.328209844f,
    0.944100258f, -0.329658463f,
    0.943593458f, -0.331106306f,
    0.943084437f, -0.332553370f,
    0.942573198f, -0.333999651f,
    0.942059740f, -0.335445147f,
    0.941544065f, -0.336889853f,
    0.941026175f, -0.338333767f,
    0.940506071f, -0.339776884f,
    0.939983753f, -0.341219202f,
    0.939459224f, -0.342660717f,
    0.938932484f, -0.344101426f,
    0.938403534f, -0.345541325f,
    0.937872376f, -0.346980411f,
    0.937339012f, -0.348418680f,
    0.936803442f, -0.349856130f,
    0.936265667f, -0.351292756f,
    0.935725689f, -0.352728556f,
    0.935183510f, -0.354163525f,
    0.934639130f, -0.355597662f,
    0.934092550f, -0.357030961f,
    0.933543773f, -0.358463421f,
    0.932992799f, -0.359895037f,
    0.932439629f, -0.361325806f,
    0.931884266f, -0.362755724f,
    0.931326709f, -0.364184790f,
    0.930766961f, -0.365612998f,
    0.930205023f, -0.367040346f,
    0.929640896f, -0.368466830f,
    0.929074581f, -0.369892447f,
    0.928506080f, -0.371317194f,
    0.927935395f, -0.372741067f,
    0.927362526f, -0.374164063f,
    0.926787474f, -0.375586178f,
    0.926210242f, -0.377007410f,
    0.925630831f, -0.378427755f,
    0.925049241f, -0.379847209f,
    0.924465474f, -0.381265769f,
    0.923879533f, -0.382683432f,
    0.923291417f, -0.384100195f,
    0.922701128f, -0.385516054f,
    0.922108669f, -0.386931006f,
    0.921514039f, -0.388345047f,
    0.920917242f, -0.389758174f,
    0.920318277f, -0.391170384f,
    0.919717146f, -0.392581674f,
    0.919113852f, -0.393992040f,
    0.918508394f, -0.395401479f,
    0.917900776f, -0.396809987f,
    0.917290997f, -0.398217562f,
    0.916679060f, -0.399624200f,
    0.916064966f, -0.401029897f,
    0.915448716f, -0.402434651f,
    0.914830312f, -0.403838458f,
    0.914209756f, -0.405241314f,
    0.913587048f, -0.406643217f,
    0.912962190f, -0.408044163f,
    0.912335185f, -0.409444149f,
    0.911706032f, -0.410843171f,
    0.911074734f, -0.412241227f,
    0.910441292f, -0.413638312f,
    0.909805708f, -0.415034424f,
    0.909167983f, -0.416429560f,
    0.908528119f, -0.417823716f,
    0.907886116f, -0.419216888f,
    0.907241978f, -0.420609074f,
    0.906595705f, -0.422000271f,
    0.905947298f, -0.423390474f,
    0.905296759f, -0.424779681f,
    0.904644091f, -0.426167889f,
    0.903989293f, -0.427555093f,
    0.903332368f, -0.428941292f,
    0.902673318f, -0.430326481f,
    0.902012144f, -0.431710658f,
    0.901348847f, -0.433093819f,
    0.900683429f, -0.434475961f,
    0.900015892f, -0.435857080f,
    0.899346237f, -0.437237174f,
    0.898674466f, -0.438616239f,
    0.898000580f, -0.439994271f,
    0.897324581f, -0.441371269f,
    0.896646470f, -0.442747228f,
    0.895966250f, -0.444122145f,
    0.895283921f, -0.445496017f,
    0.894599486f, -0.446868840f,
    0.893912945f, -0.448240612f,
    0.893224301f, -0.449611330f,
    0.892533555f, -0.450980989f,
    0.891840709f, -0.452349587f,
    0.891145765f, -0.453717121f,
    0.890448723f, -0.455083587f,
    0.889749586f, -0.456448982f,
    0.889048356f, -0.457813304f,
    0.888345033f, -0.459176548f,
    0.887639620f, -0.460538711f,
    0.886932119f, -0.461899791f,
    0.886222530f, -0.463259784f,
    0.885510856f, -0.464618686f,
    0.884797098f, -0.465976496f,
    0.884081259f, -0.467333209f,
    0.883363339f, -0.468688822f,
    0.882643340f, -0.470043332f,
    0.881921264f, -0.471396737f,
    0.881197113f, -0.472749032f,
    0.880470889f, -0.474100215f,
    0.879742593f, -0.475450282f,
    0.879012226f, -0.476799230f,
    0.878279792f, -0.478147056f,
    0.877545290f, -0.479493758f,
    0.876808724f, -0.480839331f,
    0.876070094f, -0.482183772f,
    0.875329403f, -0.483527079f,
    0.874586652f, -0.484869248f,
    0.873841843f, -0.486210276f,
    0.873094978f, -0.487550160f,
    0.872346059f, -0.488888897f,
    0.871595087f, -0.490226483f,
    0.870842063f, -0.491562916f,
    0.870086991f, -0.492898192f,
    0.869329871f, -0.494232309f,
    0.868570706f, -0.495565262f,
    0.867809497f, -0.496897049f,
    0.867046246f, -0.498227667f,
    0.866280954f, -0.499557113f,
    0.865513624f, -0.500885383f,
    0.864744258f, -0.502212474f,
    0.863972856f, -0.503538384f,
    0.863199422f, -0.504863109f,
    0.862423956f, -0.506186645f,
    0.861646461f, -0.507508991f,
    0.860866939f, -0.508830143f,
    0.860085390f, -0.510150097f,
    0.859301818f, -0.511468850f,
    0.858516224f, -0.512786401f,
    0.857728610f, -0.514102744f,
    0.856938977f, -0.515417878f,
    0.856147328f, -0.516731799f,
    0.855353665f, -0.518044504f,
    0.854557988f, -0.519355990f,
    0.853760301f, -0.520666254f,
    0.852960605f, -0.521975293f,
    0.852158902f, -0.523283103f,
    0.851355193f, -0.524589683f,
    0.850549481f, -0.525895027f,
    0.849741768f, -0.527199135f,
    0.848932055f, -0.528502002f,
    0.848120345f, -0.529803625f,
    0.847306639f, -0.531104001f,
    0.846490939f, -0.532403128f,
    0.845673247f, -0.533701002f,
    0.844853565f, -0.534997620f,
    0.844031895f, -0.536292979f,
    0.843208240f, -0.537587076f,
    0.842382600f, -0.538879909f,
    0.841554977f, -0.540171473f,
    0.840725375f, -0.541461766f,
    0.839893794f, -0.542750785f,
    0.839060237f, -0.544038527f,
    0.838224706f, -0.545324988f,
    0.837387202f, -0.546610167f,
    0.836547727f, -0.547894059f,
    0.835706284f, -0.549176662f,
    0.834862875f, -0.550457973f,
    0.834017501f, -0.551737988f,
    0.833170165f, -0.553016706f,
    0.832320868f, -0.554294121f,
    0.831469612f, -0.555570233f,
    0.830616400f, -0.556845037f,
    0.829761234f, -0.558118531f,
    0.828904115f, -0.559390712f,
    0.828045045f, -0.560661576f,
    0.827184027f, -0.561931121f,
    0.826321063f, -0.563199344f,
    0.825456154f, -0.564466242f,
    0.824589303f, -0.565731811f,
    0.823720511f, -0.566996049f,
    0.822849781f, -0.568258953f,
    0.821977115f, -0.569520519f,
    0.821102515f, -0.570780746f,
    0.820225983f, -0.572039629f,
    0.819347520f, -0.573297167f,
    0.818467130f, -0.574553355f,
    0.817584813f, -0.575808191f,
    0.816700573f, -0.577061673f,
    0.815814411f, -0.578313796f,
    0.814926329f, -0.579564559f,
    0.814036330f, -0.580813958f,
    0.813144415f, -0.582061990f,
    0.812250587f, -0.583308653f,
    0.811354847f, -0.584553943f,
    0.810457198f, -0.585797857f,
    0.809557642f, -0.587040394f,
    0.808656182f, -0.588281548f,
    0.807752818f, -0.589521319f,
    0.806847554f, -0.590759702f,
    0.805940391f, -0.591996695f,
    0.805031331f, -0.593232295f,
    0.804120377f, -0.594466499f,
    0.803207531f, -0.595699304f,
    0.802292796f, -0.596930708f,
    0.801376172f, -0.598160707f,
    0.800457662f, -0.599389298f,
    0.799537269f, -0.600616479f,
    0.798614995f, -0.601842247f,
    0.797690841f, -0.603066599f,
    0.796764810f, -0.604289531f,
    0.795836905f, -0.605511041f,
    0.794907126f, -0.606731127f,
    0.793975478f, -0.607949785f,
    0.793041960f, -0.609167012f,
    0.792106577f, -0.610382806f,
    0.791169330f, -0.611597164f,
    0.790230221f, -0.612810082f,
    0.789289253f, -0.614021559f,
    0.788346428f, -0.615231591f,
    0.787401747f, -0.616440175f,
    0.786455214f, -0.617647308f,
    0.785506830f, -0.618852988f,
    0.784556597f, -0.620057212f,
    0.783604519f, -0.621259977f,
    0.782650596f, -0.622461279f,
    0.781694832f, -0.623661118f,
    0.780737229f, -0.624859488f,
    0.779777788f, -0.626056388f,
    0.778816512f, -0.627251815f,
    0.777853404f, -0.628445767f,
    0.776888466f, -0.629638239f,
    0.775921699f, -0.630829230f,
    0.774953107f, -0.632018736f,
    0.773982691f, -0.633206755f,
    0.773010453f, -0.634393284f,
    0.772036397f, -0.635578320f,
    0.771060524f, -0.636761861f,
    0.770082837f, -0.637943904f,
    0.769103338f, -0.639124445f,
    0.768122029f, -0.640303482f,
    0.767138912f, -0.641481013f,
    0.766153990f, -0.642657034f,
    0.765167266f, -0.643831543f,
    0.764178741f, -0.645004537f,
    0.763188417f, -0.646176013f,
    0.762196298f, -0.647345969f,
    0.761202385f, -0.648514401f,
    0.760206682f, -0.649681307f,
    0.759209189f, -0.650846685f,
    0.758209910f, -0.652010531f,
    0.757208847f, -0.653172843f,
    0.756206001f, -0.654333618f,
    0.755201377f, -0.655492853f,
    0.754194975f, -0.656650546f,
    0.753186799f, -0.657806693f,
    0.752176850f, -0.658961293f,
    0.751165132f, -0.660114342f,
    0.750151646f, -0.661265838f,
    0.749136395f, -0.662415778f,
    0.748119380f, -0.663564159f,
    0.747100606f, -0.664710978f,
    0.746080074f, -0.665856234f,
    0.745057785f, -0.666999922f,
    0.744033744f, -0.668142041f,
    0.743007952f, -0.669282588f,
    0.741980412f, -0.670421560f,
    0.740951125f, -0.671558955f,
    0.739920095f, -0.672694769f,
    0.738887324f, -0.673829000f,
    0.737852815f, -0.674961646f,
    0.736816569f, -0.676092704f,
    0.735778589f, -0.677222170f,
    0.734738878f, -0.678350043f,
    0.733697438f, -0.679476320f,
    0.732654272f, -0.680600998f,
    0.731609381f, -0.681724074f,
    0.730562769f, -0.682845546f,
    0.729514438f, -0.683965412f,
    0.728464390f, -0.685083668f,
    0.727412629f, -0.686200312f,
    0.726359155f, -0.687315341f,
    0.725303972f, -0.688428753f,
    0.724247083f, -0.689540545f,
    0.723188489f, -0.690650714f,
    0.722128194f, -0.691759258f,
    0.721066199f, -0.692866175f,
    0.720002508f, -0.693971461f,
    0.718937122f, -0.695075114f,
    0.717870045f, -0.696177131f,
    0.716801279f, -0.697277511f,
    0.715730825f, -0.698376249f,
    0.714658688f, -0.699473345f,
    0.713584869f, -0.700568794f,
    0.712509371f, -0.701662595f,
    0.711432196f, -0.702754744f,
    0.710353347f, -0.703845241f,
    0.709272826f, -0.704934080f,
    0.708190637f, -0.706021261f,
    0.707106781f, -0.707106781f,
    0.706021261f, -0.708190637f,
    0.704934080f, -0.709272826f,
    0.703845241f, -0.710353347f,
    0.702754744f, -0.711432196f,
    0.701662595f, -0.712509371f,
    0.700568794f, -0.713584869f,
    0.699473345f, -0.714658688f,
    0.698376249f, -0.715730825f,
    0.697277511f, -0.716801279f,
    0.696177131f, -0.717870045f,
    0.695075114f, -0.718937122f,
    0.693971461f, -0.720002508f,
    0.692866175f, -0.721066199f,
    0.691759258f, -0.722128194f,
    0.690650714f, -0.723188489f,
    0.689540545f, -0.724247083f,
    0.688428753f, -0.725303972f,
    0.687315341f, -0.726359155f,
    0.686200312f, -0.727412629f,
    0.685083668f, -0.728464390f,
    0.683965412f, -0.729514438f,
    0.682845546f, -0.730562769f,
    0.681724074f, -0.731609381f,
    0.680600998f, -0.732654272f,
    0.679476320f, -0.733697438f,
    0.678350043f, -0.734738878f,
    0.677222170f, -0.735778589f,
    0.676092704f, -0.736816569f,
    0.674961646f, -0.737852815f,
    0.673829000f, -0.738887324f,
    0.672694769f, -0.739920095f,
    0.671558955f, -0.740951125f,
    0.670421560f, -0.741980412f,
    0.669282588f, -0.743007952f,
    0.668142041f, -0.744033744f,
    0.666999922f, -0.745057785f,
    0.665856234f, -0.746080074f,
    0.664710978f, -0.747100606f,
    0.663564159f, -0.748119380f,
    0.662415778f, -0.749136395f,
    0.661265838f, -0.750151646f,
    0.660114342f, -0.751165132f,
    0.658961293f, -0.752176850f,
    0.657806693f, -0.753186799f,
    0.656650546f, -0.754194975f,
    0.655492853f, -0.755201377f,
    0.654333618f, -0.756206001f,
    0.653172843f, -0.757208847f,
    0.652010531f, -0.758209910f,
    0.650846685f, -0.759209189f,
    0.649681307f, -0.760206682f,
    0.648514401f, -0.761202385f,
    0.647345969f, -0.762196298f,
    0.646176013f, -0.763188417f,
    0.645004537f, -0.764178741f,
    0.643831543f, -0.765167266f,
    0.642657034f, -0.766153990f,
    0.641481013f, -0.767138912f,
    0.640303482f, -0.768122029f,
    0.639124445f, -0.769103338f,
    0.637943904f, -0.770082837f,
    0.636761861f, -0.771060524f,
    0.635578320f, -0.772036397f,
    0.634393284f, -0.773010453f,
    0.633206755f, -0.773982691f,
    0.632018736f, -0.774953107f,
    0.630829230f, -0.775921699f,
    0.629638239f, -0.776888466f,
    0.628445767f, -0.777853404f,
    0.627251815f, -0.778816512f,
    0.626056388f, -0.779777788f,
    0.624859488f, -0.780737229f,
    0.623661118f, -0.781694832f,
    0.622461279f, -0.782650596f,
    0.621259977f, -0.783604519f,
    0.620057212f, -0.784556597f,
    0.618852988f, -0.785506830f,
    0.617647308f, -0.786455214f,
    0.616440175f, -0.787401747f,
    0.615231591f, -0.788346428f,
    0.614021559f, -0.789289253f,
    0.612810082f, -0.790230221f,
    0.611597164f, -0.791169330f,
    0.610382806f, -0.792106577f,
    0.609167012f, -0.793041960f,
    0.607949785f, -0.793975478f,
    0.606731127f, -0.794907126f,
    0.605511041f, -0.795836905f,
    0.604289531f, -0.796764810f,
    0.603066599f, -0.797690841f,
    0.601842247f, -0.798614995f,
    0.600616479f, -0.799537269f,
    0.599389298f, -0.800457662f,
    0.598160707f, -0.801376172f,
    0.596930708f, -0.802292796f,
    0.595699304f, -0.803207531f,
    0.594466499f, -0.804120377f,
    0.593232295f, -0.805031331f,
    0.591996695f, -0.805940391f,
    0.590759702f, -0.806847554f,
    0.589521319f, -0.807752818f,
    0.588281548f, -0.808656182f,
    0.587040394f, -0.809557642f,
    0.585797857f, -0.810457198f,
    0.584553943f, -0.811354847f,
    0.583308653f, -0.812250587f,
    0.582061990f, -0.813144415f,
    0.580813958f, -0.814036330f,
    0.579564559f, -0.814926329f,
    0.578313796f, -0.815814411f,
    0.577061673f, -0.816700573f,
    0.575808191f, -0.817584813f,
    0.574553355f, -0.818467130f,
    0.573297167f, -0.819347520f,
    0.572039629f, -0.820225983f,
    0.570780746f, -0.821102515f,
    0.569520519f, -0.821977115f,
    0.568258953f, -0.822849781f,
    0.566996049f, -0.823720511f,
    0.565731811f, -0.824589303f,
    0.564466242f, -0.825456154f,
    0.563199344f, -0.826321063f,
    0.561931121f, -0.827184027f,
    0.560661576f, -0.828045045f,
    0.559390712f, -0.828904115f,
    0.558118531f, -0.829761234f,
    0.556845037f, -0.830616400f,
    0.555570233f, -0.831469612f,
    0.554294121f, -0.832320868f,
    0.553016706f, -0.833170165f,
    0.551737988f, -0.834017501f,
    0.550457973f, -0.834862875f,
    0.549176662f, -0.835706284f,
    0.547894059f, -0.836547727f,
    0.546610167f, -0.837387202f,
    0.545324988f, -0.838224706f,
    0.544038527f, -0.839060237f,
    0.542750785f, -0.839893794f,
    0.541461766f, -0.840725375f,
    0.540171473f, -0.841554977f,
    0.538879909f, -0.842382600f,
    0.537587076f, -0.843208240f,
    0.536292979f, -0.844031895f,
    0.534997620f, -0.844853565f,
    0.533701002f, -0.845673247f,
    0.532403128f, -0.846490939f,
    0.531104001f, -0.847306639f,
    0.529803625f, -0.848120345f,
    0.528502002f, -0.848932055f,
    0.527199135f, -0.849741768f,
    0.525895027f, -0.850549481f,
    0.524589683f, -0.851355193f,
    0.523283103f, -0.852158902f,
    0.521975293f, -0.852960605f,
    0.520666254f, -0.853760301f,
    0.519355990f, -0.854557988f,
    0.518044504f, -0.855353665f,
    0.516731799f, -0.856147328f,
    0.515417878f, -0.856938977f,
    0.514102744f, -0.857728610f,
    0.512786401f, -0.858516224f,
    0.511468850f, -0.859301818f,
    0.510150097f, -0.860085390f,
    0.508830143f, -0.860866939f,
    0.507508991f, -0.861646461f,
    0.506186645f, -0.862423956f,
    0.504863109f, -0.863199422f,
    0.503538384f, -0.863972856f,
    0.502212474f, -0.864744258f,
    0.500885383f, -0.865513624f,
    0.499557113f, -0.866280954f,
    0.498227667f, -0.867046246f,
    0.496897049f, -0.867809497f,
    0.495565262f, -0.868570706f,
    0.494232309f, -0.869329871f,
    0.492898192f, -0.870086991f,
    0.491562916f, -0.870842063f,
    0.490226483f, -0.871595087f,
    0.488888897f, -0.872346059f,
    0.487550160f, -0.873094978f,
    0.486210276f, -0.873841843f,
    0.484869248f, -0.874586652f,
    0.483527079f, -0.875329403f,
    0.482183772f, -0.876070094f,
    0.480839331f, -0.876808724f,
    0.479493758f, -0.877545290f,
    0.478147056f, -0.878279792f,
    0.476799230f, -0.879012226f,
    0.475450282f, -0.879742593f,
    0.474100215f, -0.880470889f,
    0.472749032f, -0.881197113f,
    0.471396737f, -0.881921264f,
    0.470043332f, -0.882643340f,
    0.468688822f, -0.883363339f,
    0.467333209f, -0.884081259f,
    0.465976496f, -0.884797098f,
    0.464618686f, -0.885510856f,
    0.463259784f, -0.886222530f,
    0.461899791f, -0.886932119f,
    0.460538711f, -0.887639620f,
    0.459176548f, -0.888345033f,
    0.457813304f, -0.889048356f,
    0.456448982f, -0.889749586f,
    0.455083587f, -0.890448723f,
    0.453717121f, -0.891145765f,
    0.452349587f, -0.891840709f,
    0.450980989f, -0.892533555f,
    0.449611330f, -0.893224301f,
    0.448240612f, -0.893912945f,
    0.446868840f, -0.894599486f,
    0.445496017f, -0.895283921f,
    0.444122145f, -0.895966250f,
    0.442747228f, -0.896646470f,
    0.441371269f, -0.897324581f,
    0.439994271f, -0.898000580f,
    0.438616239f, -0.898674466f,
    0.437237174f, -0.899346237f,
    0.435857080f, -0.900015892f,
    0.434475961f, -0.900683429f,
    0.433093819f, -0.901348847f,
    0.431710658f, -0.902012144f,
    0.430326481f, -0.902673318f,
    0.428941292f, -0.903332368f,
    0.427555093f, -0.903989293f,
    0.426167889f, -0.904644091f,
    0.424779681f, -0.905296759f,
    0.423390474f, -0.905947298f,
    0.422000271f, -0.906595705f,
    0.420609074f, -0.907241978f,
    0.419216888f, -0.907886116f,
    0.417823716f, -0.908528119f,
    0.416429560f, -0.909167983f,
    0.415034424f, -0.909805708f,
    0.413638312f, -0.910441292f,
    0.412241227f, -0.911074734f,
    0.410843171f, -0.911706032f,
    0.409444149f, -0.912335185f,
    0.408044163f, -0.912962190f,
    0.406643217f, -0.913587048f,
    0.405241314f, -0.914209756f,
    0.403838458f, -0.914830312f,
    0.402434651f, -0.915448716f,
    0.401029897f, -0.916064966f,
    0.399624200f, -0.916679060f,
    0.398217562f, -0.917290997f,
    0.396809987f, -0.917900776f,
    0.395401479f, -0.918508394f,
    0.393992040f, -0.919113852f,
    0.392581674f, -0.919717146f,
    0.391170384f, -0.920318277f,
    0.389758174f, -0.920917242f,
    0.388345047f, -0.921514039f,
    0.386931006f, -0.922108669f,
    0.385516054f, -0.922701128f,
    0.384100195f, -0.923291417f,
    0.382683432f, -0.923879533f,
    0.381265769f, -0.924465474f,
    0.379847209f, -0.925049241f,
    0.378427755f, -0.925630831f,
    0.377007410f, -0.926210242f,
    0.375586178f, -0.926787474f,
    0.374164063f, -0.927362526f,
    0.372741067f, -0.927935395f,
    0.371317194f, -0.928506080f,
    0.369892447f, -0.929074581f,
    0.368466830f, -0.929640896f,
    0.367040346f, -0.930205023f,
    0.365612998f, -0.930766961f,
    0.364184790f, -0.931326709f,
    0.362755724f, -0.931884266f,
    0.361325806f, -0.932439629f,
    0.359895037f, -0.932992799f,
    0.358463421f, -0.933543773f,
    0.357030961f, -0.934092550f,
    0.355597662f, -0.934639130f,
    0.354163525f, -0.935183510f,
    0.352728556f, -0.935725689f,
    0.351292756f, -0.936265667f,
    0.349856130f, -0.936803442f,
    0.348418680f, -0.937339012f,
    0.346980411f, -0.937872376f,
    0.345541325f, -0.938403534f,
    0.344101426f, -0.938932484f,
    0.342660717f, -0.939459224f,
    0.341219202f, -0.939983753f,
    0.339776884f, -0.940506071f,
    0.338333767f, -0.941026175f,
    0.336889853f, -0.941544065f,
    0.335445147f, -0.942059740f,
    0.333999651f, -0.942573198f,
    0.332553370f, -0.943084437f,
    0.331106306f, -0.943593458f,
    0.329658463f, -0.944100258f,
    0.328209844f, -0.944604837f,
    0.326760452f, -0.945107193f,
    0.325310292f, -0.945607325f,
    0.323859367f, -0.946105232f,
    0.322407679f, -0.946600913f,
    0.320955232f, -0.947094366f,
    0.319502031f, -0.947585591f,
    0.318048077f, -0.948074586f,
    0.316593376f, -0.948561350f,
    0.315137929f, -0.949045882f,
    0.313681740f, -0.949528181f,
    0.312224814f, -0.950008245f,
    0.310767153f, -0.950486074f,
    0.309308760f, -0.950961666f,
    0.307849640f, -0.951435021f,
    0.306389795f, -0.951906137f,
    0.304929230f, -0.952375013f,
    0.303467947f, -0.952841648f,
    0.302005949f, -0.953306040f,
    0.300543241f, -0.953768190f,
    0.299079826f, -0.954228095f,
    0.297615707f, -0.954685755f,
    0.296150888f, -0.955141168f,
    0.294685372f, -0.955594334f,
    0.293219163f, -0.956045251f,
    0.291752263f, -0.956493919f,
    0.290284677f, -0.956940336f,
    0.288816408f, -0.957384501f,
    0.287347460f, -0.957826413f,
    0.285877835f, -0.958266071f,
    0.284407537f, -0.958703475f,
    0.282936570f, -0.959138622f,
    0.281464938f, -0.959571513f,
    0.279992643f, -0.960002146f,
    0.278519689f, -0.960430519f,
    0.277046080f, -0.960856633f,
    0.275571819f, -0.961280486f,
    0.274096910f, -0.961702077f,
    0.272621355f, -0.962121404f,
    0.271145160f, -0.962538468f,
    0.269668326f, -0.962953267f,
    0.268190857f, -0.963365800f,
    0.266712757f, -0.963776066f,
    0.265234030f, -0.964184064f,
    0.263754679f, -0.964589793f,
    0.262274707f, -0.964993253f,
    0.260794118f, -0.965394442f,
    0.259312915f, -0.965793359f,
    0.257831102f, -0.966190003f,
    0.256348682f, -0.966584374f,
    0.254865660f, -0.966976471f,
    0.253382037f, -0.967366292f,
    0.251897818f, -0.967753837f,
    0.250413007f, -0.968139105f,
    0.248927606f, -0.968522094f,
    0.247441619f, -0.968902805f,
    0.245955050f, -0.969281235f,
    0.244467903f, -0.969657385f,
    0.242980180f, -0.970031253f,
    0.241491885f, -0.970402839f,
    0.240003022f, -0.970772141f,
    0.238513595f, -0.971139158f,
    0.237023606f, -0.971503891f,
    0.235533059f, -0.971866337f,
    0.234041959f, -0.972226497f,
    0.232550307f, -0.972584369f,
    0.231058108f, -0.972939952f,
    0.229565366f, -0.973293246f,
    0.228072083f, -0.973644250f,
    0.226578264f, -0.973992962f,
    0.225083911f, -0.974339383f,
    0.223589029f, -0.974683511f,
    0.222093621f, -0.975025345f,
    0.220597690f, -0.975364885f,
    0.219101240f, -0.975702130f,
    0.217604275f, -0.976037079f,
    0.216106797f, -0.976369731f,
    0.214608811f, -0.976700086f,
    0.213110320f, -0.977028143f,
    0.211611327f, -0.977353900f,
    0.210111837f, -0.977677358f,
    0.208611852f, -0.977998515f,
    0.207111376f, -0.978317371f,
    0.205610413f, -0.978633924f,
    0.204108966f, -0.978948175f,
    0.202607039f, -0.979260123f,
    0.201104635f, -0.979569766f,
    0.199601758f, -0.979877104f,
    0.198098411f, -0.980182136f,
    0.196594598f, -0.980484862f,
    0.195090322f, -0.980785280f,
    0.193585587f, -0.981083391f,
    0.192080397f, -0.981379193f,
    0.190574755f, -0.981672686f,
    0.189068664f, -0.981963869f,
    0.187562129f, -0.982252741f,
    0.186055152f, -0.982539302f,
    0.184547737f, -0.982823551f,
    0.183039888f, -0.983105487f,
    0.181531608f, -0.983385110f,
    0.180022901f, -0.983662419f,
    0.178513771f, -0.983937413f,
    0.177004220f, -0.984210092f,
    0.175494253f, -0.984480455f,
    0.173983873f, -0.984748502f,
    0.172473084f, -0.985014231f,
    0.170961889f, -0.985277642f,
    0.169450291f, -0.985538735f,
    0.167938295f, -0.985797509f,
    0.166425904f, -0.986053963f,
    0.164913120f, -0.986308097f,
    0.163399949f, -0.986559910f,
    0.161886394f, -0.986809402f,
    0.160372457f, -0.987056571f,
    0.158858143f, -0.987301418f,
    0.157343456f, -0.987543942f,
    0.155828398f, -0.987784142f,
    0.154312973f, -0.988022017f,
    0.152797185f, -0.988257568f,
    0.151281038f, -0.988490793f,
    0.149764535f, -0.988721692f,
    0.148247679f, -0.988950265f,
    0.146730474f, -0.989176510f,
    0.145212925f, -0.989400428f,
    0.143695033f, -0.989622017f,
    0.142176804f, -0.989841278f,
    0.140658239f, -0.990058210f,
    0.139139344f, -0.990272812f,
    0.137620122f, -0.990485084f,
    0.136100575f, -0.990695025f,
    0.134580709f, -0.990902635f,
    0.133060525f, -0.991107914f,
    0.131540029f, -0.991310860f,
    0.130019223f, -0.991511473f,
    0.128498111f, -0.991709754f,
    0.126976696f, -0.991905700f,
    0.125454983f, -0.992099313f,
    0.123932975f, -0.992290591f,
    0.122410675f, -0.992479535f,
    0.120888087f, -0.992666142f,
    0.119365215f, -0.992850414f,
    0.117842062f, -0.993032350f,
    0.116318631f, -0.993211949f,
    0.114794927f, -0.993389211f,
    0.113270952f, -0.993564136f,
    0.111746711f, -0.993736722f,
    0.110222207f, -0.993906970f,
    0.108697444f, -0.994074879f,
    0.107172425f, -0.994240449f,
    0.105647154f, -0.994403680f,
    0.104121634f, -0.994564571f,
    0.102595869f, -0.994723121f,
    0.101069863f, -0.994879331f,
    0.099543619f, -0.995033199f,
    0.098017140f, -0.995184727f,
    0.096490431f, -0.995333912f,
    0.094963495f, -0.995480755f,
    0.093436336f, -0.995625256f,
    0.091908956f, -0.995767414f,
    0.090381361f, -0.995907229f,
    0.088853553f, -0.996044701f,
    0.087325535f, -0.996179829f,
    0.085797312f, -0.996312612f,
    0.084268888f, -0.996443051f,
    0.082740265f, -0.996571146f,
    0.081211447f, -0.996696895f,
    0.079682438f, -0.996820299f,
    0.078153242f, -0.996941358f,
    0.076623861f, -0.997060070f,
    0.075094301f, -0.997176437f,
    0.073564564f, -0.997290457f,
    0.072034653f, -0.997402130f,
    0.070504573f, -0.997511456f,
    0.068974328f, -0.997618435f,
    0.067443920f, -0.997723067f,
    0.065913353f, -0.997825350f,
    0.064382631f, -0.997925286f,
    0.062851758f, -0.998022874f,
    0.061320736f, -0.998118113f,
    0.059789571f, -0.998211003f,
    0.058258265f, -0.998301545f,
    0.056726821f, -0.998389737f,
    0.055195244f, -0.998475581f,
    0.053663538f, -0.998559074f,
    0.052131705f, -0.998640218f,
    0.050599749f, -0.998719012f,
    0.049067674f, -0.998795456f,
    0.047535484f, -0.998869550f,
    0.046003182f, -0.998941293f,
    0.044470772f, -0.999010686f,
    0.042938257f, -0.999077728f,
    0.041405641f, -0.999142419f,
    0.039872928f, -0.999204759f,
    0.038340120f, -0.999264747f,
    0.036807223f, -0.999322385f,
    0.035274239f, -0.999377670f,
    0.033741172f, -0.999430605f,
    0.032208025f, -0.999481187f,
    0.030674803f, -0.999529418f,
    0.029141509f, -0.999575296f,
    0.027608146f, -0.999618822f,
    0.026074718f, -0.999659997f,
    0.024541229f, -0.999698819f,
    0.023007681f, -0.999735288f,
    0.021474080f, -0.999769405f,
    0.019940429f, -0.999801170f,
    0.018406730f, -0.999830582f,
    0.016872988f, -0.999857641f,
    0.015339206f, -0.999882347f,
    0.013805389f, -0.999904701f,
    0.012271538f, -0.999924702f,
    0.010737659f, -0.999942350f,
    0.009203755f, -0.999957645f,
    0.007669829f, -0.999970586f,
    0.006135885f, -0.999981175f,
    0.004601926f, -0.999989411f,
    0.003067957f, -0.999995294f,
    0.001533980f, -0.999998823f
};


/**
 * \par
 * Example code for the generation of the floating-point sine table:
 * <pre>
 * tableSize = 512;
 * for(n = 0; n < (tableSize + 1); n++)
 * {
 *	sinTable[n]=sin(2*pi*n/tableSize);
 * }</pre>
 * \par
 * where pi value is  3.14159265358979
 */

const float32_t sinTable_f32[FAST_MATH_TABLE_SIZE + 1] = {
   0.00000000f, 0.01227154f, 0.02454123f, 0.03680722f, 0.04906767f, 0.06132074f,
   0.07356456f, 0.08579731f, 0.09801714f, 0.11022221f, 0.12241068f, 0.13458071f,
   0.14673047f, 0.15885814f, 0.17096189f, 0.18303989f, 0.19509032f, 0.20711138f,
   0.21910124f, 0.23105811f, 0.24298018f, 0.25486566f, 0.26671276f, 0.27851969f,
   0.29028468f, 0.30200595f, 0.31368174f, 0.32531029f, 0.33688985f, 0.34841868f,
   0.35989504f, 0.37131719f, 0.38268343f, 0.39399204f, 0.40524131f, 0.41642956f,
   0.42755509f, 0.43861624f, 0.44961133f, 0.46053871f, 0.47139674f, 0.48218377f,
   0.49289819f, 0.50353838f, 0.51410274f, 0.52458968f, 0.53499762f, 0.54532499f,
   0.55557023f, 0.56573181f, 0.57580819f, 0.58579786f, 0.59569930f, 0.60551104f,
   0.61523159f, 0.62485949f, 0.63439328f, 0.64383154f, 0.65317284f, 0.66241578f,
   0.67155895f, 0.68060100f, 0.68954054f, 0.69837625f, 0.70710678f, 0.71573083f,
   0.72424708f, 0.73265427f, 0.74095113f, 0.74913639f, 0.75720885f, 0.76516727f,
   0.77301045f, 0.78073723f, 0.78834643f, 0.79583690f, 0.80320753f, 0.81045720f,
   0.81758481f, 0.82458930f, 0.83146961f, 0.83822471f, 0.84485357f, 0.85135519f,
   0.85772861f, 0.86397286f, 0.87008699f, 0.87607009f, 0.88192126f, 0.88763962f,
   0.89322430f, 0.89867447f, 0.90398929f, 0.90916798f, 0.91420976f, 0.91911385f,
   0.92387953f, 0.92850608f, 0.93299280f, 0.93733901f, 0.94154407f, 0.94560733f,
   0.94952818f, 0.95330604f, 0.95694034f, 0.96043052f, 0.96377607f, 0.96697647f,
   0.97003125f, 0.97293995f, 0.97570213f, 0.97831737f, 0.98078528f, 0.98310549f,
   0.98527764f, 0.98730142f, 0.98917651f, 0.99090264f, 0.99247953f, 0.99390697f,
   0.99518473f, 0.99631261f, 0.99729046f, 0.99811811f, 0.99879546f, 0.99932238f,
   0.99969882f, 0.99992470f, 1.00000000f, 0.99992470f, 0.99969882f, 0.99932238f,
   0.99879546f, 0.99811811f, 0.99729046f, 0.99631261f, 0.99518473f, 0.99390697f,
   0.99247953f, 0.99090264f, 0.98917651f, 0.98730142f, 0.98527764f, 0.98310549f,
   0.98078528f, 0.97831737f, 0.97570213f, 0.97293995f, 0.97003125f, 0.96697647f,
   0.96377607f, 0.96043052f, 0.95694034f, 0.95330604f, 0.94952818f, 0.94560733f,
   0.94154407f, 0.93733901f, 0.93299280f, 0.92850608f, 0.92387953f, 0.91911385f,
   0.91420976f, 0.90916798f, 0.90398929f, 0.89867447f, 0.89322430f, 0.88763962f,
   0.88192126f, 0.87607009f, 0.87008699f, 0.86397286f, 0.85772861f, 0.85135519f,
   0.84485357f, 0.83822471f, 0.83146961f, 0.82458930f, 0.81758481f, 0.81045720f,
   0.80320753f, 0.79583690f, 0.78834643f, 0.78073723f, 0.77301045f, 0.76516727f,
   0.75720885f, 0.74913639f, 0.74095113f, 0.73265427f, 0.72424708f, 0.71573083f,
   0.70710678f, 0.69837625f, 0.68954054f, 0.68060100f, 0.67155895f, 0.66241578f,
   0.65317284f, 0.64383154f, 0.63439328f, 0.62485949f, 0.61523159f, 0.60551104f,
   0.59569930f, 0.58579786f, 0.57580819f, 0.56573181f, 0.55557023f, 0.54532499f,
   0.53499762f, 0.52458968f, 0.51410274f, 0.50353838f, 0.49289819f, 0.48218377f,
   0.47139674f, 0.46053871f, 0.44961133f, 0.43861624f, 0.42755509f, 0.41642956f,
   0.40524131f, 0.39399204f, 0.38268343f, 0.37131719f, 0.35989504f, 0.34841868f,
   0.33688985f, 0.32531029f, 0.31368174f, 0.30200595f, 0.29028468f, 0.27851969f,
   0.26671276f, 0.25486566f, 0.24298018f, 0.23105811f, 0.21910124f, 0.20711138f,
   0.19509032f, 0.18303989f, 0.17096189f, 0.15885814f, 0.14673047f, 0.13458071f,
   0.12241068f, 0.11022221f, 0.09801714f, 0.08579731f, 0.07356456f, 0.06132074f,
   0.04906767f, 0.03680722f, 0.02454123f, 0.01227154f, 0.00000000f, -0.01227154f,
   -0.02454123f, -0.03680722f, -0.04906767f, -0.06132074f, -0.07356456f,
   -0.08579731f, -0.09801714f, -0.11022221f, -0.12241068f, -0.13458071f,
   -0.14673047f, -0.15885814f, -0.17096189f, -0.18303989f, -0.19509032f,
   -0.20711138f, -0.21910124f, -0.23105811f, -0.24298018f, -0.25486566f,
   -0.26671276f, -0.27851969f, -0.29028468f, -0.30200595f, -0.31368174f,
   -0.32531029f, -0.33688985f, -0.34841868f, -0.35989504f, -0.37131719f,
   -0.38268343f, -0.39399204f, -0.40524131f, -0.41642956f, -0.42755509f,
   -0.43861624f, -0.44961133f, -0.46053871f, -0.47139674f, -0.48218377f,
   -0.49289819f, -0.50353838f, -0.51410274f, -0.52458968f, -0.53499762f,
   -0.54532499f, -0.55557023f, -0.56573181f, -0.57580819f, -0.58579786f,
   -0.59569930f, -0.60551104f, -0.61523159f, -0.62485949f, -0.63439328f,
   -0.64383154f, -0.65317284f, -0.66241578f, -0.67155895f, -0.68060100f,
   -0.68954054f, -0.69837625f, -0.70710678f, -0.71573083f, -0.72424708f,
   -0.73265427f, -0.74095113f, -0.74913639f, -0.75720885f, -0.76516727f,
   -0.77301045f, -0.78073723f, -0.78834643f, -0.79583690f, -0.80320753f,
   -0.81045720f, -0.81758481f, -0.82458930f, -0.83146961f, -0.83822471f,
   -0.84485357f, -0.85135519f, -0.85772861f, -0.86397286f, -0.87008699f,
   -0.87607009f, -0.88192126f, -0.88763962f, -0.89322430f, -0.89867447f,
   -0.90398929f, -0.90916798f, -0.91420976f, -0.91911385f, -0.92387953f,
   -0.92850608f, -0.93299280f, -0.93733901f, -0.94154407f, -0.94560733f,
   -0.94952818f, -0.95330604f, -0.95694034f, -0.96043052f, -0.96377607f,
   -0.96697647f, -0.97003125f, -0.97293995f, -0.97570213f, -0.97831737f,
   -0.98078528f, -0.98310549f, -0.98527764f, -0.98730142f, -0.98917651f,
   -0.99090264f, -0.99247953f, -0.99390697f, -0.99518473f, -0.99631261f,
   -0.99729046f, -0.99811811f, -0.99879546f, -0.99932238f, -0.99969882f,
   -0.99992470f, -1.00000000f, -0.99992470f, -0.99969882f, -0.99932238f,
   -0.99879546f, -0.99811811f, -0.99729046f, -0.99631261f, -0.99518473f,
   -0.99390697f, -0.99247953f, -0.99090264f, -0.98917651f, -0.98730142f,
   -0.98527764f, -0.98310549f, -0.98078528f, -0.97831737f, -0.97570213f,
   -0.97293995f, -0.97003125f, -0.96697647f, -0.96377607f, -0.96043052f,
   -0.95694034f, -0.95330604f, -0.94952818f, -0.94560733f, -0.94154407f,
   -0.93733901f, -0.93299280f, -0.92850608f, -0.92387953f, -0.91911385f,
   -0.91420976f, -0.90916798f, -0.90398929f, -0.89867447f, -0.89322430f,
   -0.88763962f, -0.88192126f, -0.87607009f, -0.87008699f, -0.86397286f,
   -0.85772861f, -0.85135519f, -0.84485357f, -0.83822471f, -0.83146961f,
   -0.82458930f, -0.81758481f, -0.81045720f, -0.80320753f, -0.79583690f,
   -0.78834643f, -0.78073723f, -0.77301045f, -0.76516727f, -0.75720885f,
   -0.74913639f, -0.74095113f, -0.73265427f, -0.72424708f, -0.71573083f,
   -0.70710678f, -0.69837625f, -0.68954054f, -0.68060100f, -0.67155895f,
   -0.66241578f, -0.65317284f, -0.64383154f, -0.63439328f, -0.62485949f,
   -0.61523159f, -0.60551104f, -0.59569930f, -0.58579786f, -0.57580819f,
   -0.56573181f, -0.55557023f, -0.54532499f, -0.53499762f, -0.52458968f,
   -0.51410274f, -0.50353838f, -0.49289819f, -0.48218377f, -0.47139674f,
   -0.46053871f, -0.44961133f, -0.43861624f, -0.42755509f, -0.41642956f,
   -0.40524131f, -0.39399204f, -0.38268343f, -0.37131719f, -0.35989504f,
   -0.34841868f, -0.33688985f, -0.32531029f, -0.31368174f, -0.30200595f,
   -0.29028468f, -0.27851969f, -0.26671276f, -0.25486566f, -0.24298018f,
   -0.23105811f, -0.21910124f, -0.20711138f, -0.19509032f, -0.18303989f,
   -0.17096189f, -0.15885814f, -0.14673047f, -0.13458071f, -0.12241068f,
   -0.11022221f, -0.09801714f, -0.08579731f, -0.07356456f, -0.06132074f,
   -0.04906767f, -0.03680722f, -0.02454123f, -0.01227154f, -0.00000000f
};

/**
 * \par
 * Table values are in Q31 (1.31 fixed-point format) and generation is done in
 * three steps.  First,  generate sin values in floating point:
 * <pre>
 * tableSize = 512;
 * for(n = 0; n < (tableSize + 1); n++)
 * {
 *	sinTable[n]= sin(2*pi*n/tableSize);
 * } </pre>
 * where pi value is  3.14159265358979
 * \par
 * Second, convert floating-point to Q31 (Fixed point):
 *	(sinTable[i] * pow(2, 31))
 * \par
 * Finally, round to the nearest integer value:
 * 	sinTable[i] += (sinTable[i] > 0 ? 0.5 :-0.5);
 */
const q31_t sinTable_q31[FAST_MATH_TABLE_SIZE + 1] = {
	0L, 26352928L, 52701887L, 79042909L, 105372028L, 131685278L, 157978697L,
	184248325L, 210490206L, 236700388L, 262874923L, 289009871L, 315101295L,
	341145265L, 367137861L, 393075166L, 418953276L, 444768294L, 470516330L,
	496193509L, 521795963L, 547319836L, 572761285L, 598116479L, 623381598L,
	648552838L, 673626408L, 698598533L, 723465451L, 748223418L, 772868706L,
	797397602L, 821806413L, 846091463L, 870249095L, 894275671L, 918167572L,
	941921200L, 965532978L, 988999351L, 1012316784L, 1035481766L, 1058490808L,
	1081340445L, 1104027237L, 1126547765L, 1148898640L, 1171076495L, 1193077991L,
	1214899813L, 1236538675L, 1257991320L, 1279254516L, 1300325060L, 1321199781L,
	1341875533L, 1362349204L, 1382617710L, 1402678000L, 1422527051L, 1442161874L,
	1461579514L, 1480777044L, 1499751576L, 1518500250L, 1537020244L, 1555308768L,
	1573363068L, 1591180426L, 1608758157L, 1626093616L, 1643184191L, 1660027308L,
	1676620432L, 1692961062L, 1709046739L, 1724875040L, 1740443581L, 1755750017L,
	1770792044L, 1785567396L, 1800073849L, 1814309216L, 1828271356L, 1841958164L,
	1855367581L, 1868497586L, 1881346202L, 1893911494L, 1906191570L, 1918184581L,
	1929888720L, 1941302225L, 1952423377L, 1963250501L, 1973781967L, 1984016189L,
	1993951625L, 2003586779L, 2012920201L, 2021950484L, 2030676269L, 2039096241L,
	2047209133L, 2055013723L, 2062508835L, 2069693342L, 2076566160L, 2083126254L,
	2089372638L, 2095304370L, 2100920556L, 2106220352L, 2111202959L, 2115867626L,
	2120213651L, 2124240380L, 2127947206L, 2131333572L, 2134398966L, 2137142927L,
	2139565043L, 2141664948L, 2143442326L, 2144896910L, 2146028480L, 2146836866L,
	2147321946L, 2147483647L, 2147321946L, 2146836866L, 2146028480L, 2144896910L,
	2143442326L, 2141664948L, 2139565043L, 2137142927L, 2134398966L, 2131333572L,
	2127947206L, 2124240380L, 2120213651L, 2115867626L, 2111202959L, 2106220352L,
	2100920556L, 2095304370L, 2089372638L, 2083126254L, 2076566160L, 2069693342L,
	2062508835L, 2055013723L, 2047209133L, 2039096241L, 2030676269L, 2021950484L,
	2012920201L, 2003586779L, 1993951625L, 1984016189L, 1973781967L, 1963250501L,
	1952423377L, 1941302225L, 1929888720L, 1918184581L, 1906191570L, 1893911494L,
	1881346202L, 1868497586L, 1855367581L, 1841958164L, 1828271356L, 1814309216L,
	1800073849L, 1785567396L, 1770792044L, 1755750017L, 1740443581L, 1724875040L,
	1709046739L, 1692961062L, 1676620432L, 1660027308L, 1643184191L, 1626093616L,
	1608758157L, 1591180426L, 1573363068L, 1555308768L, 1537020244L, 1518500250L,
	1499751576L, 1480777044L, 1461579514L, 1442161874L, 1422527051L, 1402678000L,
	1382617710L, 1362349204L, 1341875533L, 1321199781L, 1300325060L, 1279254516L,
	1257991320L, 1236538675L, 1214899813L, 1193077991L, 1171076495L, 1148898640L,
	1126547765L, 1104027237L, 1081340445L, 1058490808L, 1035481766L, 1012316784L,
	988999351L, 965532978L, 941921200L, 918167572L, 894275671L, 870249095L,
	846091463L, 821806413L, 797397602L, 772868706L, 748223418L, 723465451L,
	698598533L, 673626408L, 648552838L, 623381598L, 598116479L, 572761285L,
	547319836L, 521795963L, 496193509L, 470516330L, 444768294L, 418953276L,
	393075166L, 367137861L, 341145265L, 315101295L, 289009871L, 262874923L,
	236700388L, 210490206L, 184248325L, 157978697L, 131685278L, 105372028L,
	79042909L, 52701887L, 26352928L, 0L, -26352928L, -52701887L, -79042909L,
	-105372028L, -131685278L, -157978697L, -184248325L, -210490206L, -236700388L,
	-262874923L, -289009871L, -315101295L, -341145265L, -367137861L, -393075166L,
	-418953276L, -444768294L, -470516330L, -496193509L, -521795963L, -547319836L,
	-572761285L, -598116479L, -623381598L, -648552838L, -673626408L, -698598533L,
	-723465451L, -748223418L, -772868706L, -797397602L, -821806413L, -846091463L,
	-870249095L, -894275671L, -918167572L, -941921200L, -965532978L, -988999351L,
	-1012316784L, -1035481766L, -1058490808L, -1081340445L, -1104027237L,
	-1126547765L, -1148898640L, -1171076495L, -1193077991L, -1214899813L,
	-1236538675L, -1257991320L, -1279254516L, -1300325060L, -1321199781L,
	-1341875533L, -1362349204L, -1382617710L, -1402678000L, -1422527051L,
	-1442161874L, -1461579514L, -1480777044L, -1499751576L, -1518500250L,
	-1537020244L, -1555308768L, -1573363068L, -1591180426L, -1608758157L,
	-1626093616L, -1643184191L, -1660027308L, -1676620432L, -1692961062L,
	-1709046739L, -1724875040L, -1740443581L, -1755750017L, -1770792044L,
	-1785567396L, -1800073849L, -1814309216L, -1828271356L, -1841958164L,
	-1855367581L, -1868497586L, -1881346202L, -1893911494L, -1906191570L,
	-1918184581L, -1929888720L, -1941302225L, -1952423377L, -1963250501L,
	-1973781967L, -1984016189L, -1993951625L, -2003586779L, -2012920201L,
	-2021950484L, -2030676269L, -2039096241L, -2047209133L, -2055013723L,
	-2062508835L, -2069693342L, -2076566160L, -2083126254L, -2089372638L,
	-2095304370L, -2100920556L, -2106220352L, -2111202959L, -2115867626L,
	-2120213651L, -2124240380L, -2127947206L, -2131333572L, -2134398966L,
	-2137142927L, -2139565043L, -2141664948L, -2143442326L, -2144896910L,
	-2146028480L, -2146836866L, -2147321946L, (q31_t)0x80000000, -2147321946L,
	-2146836866L, -2146028480L, -2144896910L, -2143442326L, -2141664948L,
	-2139565043L, -2137142927L, -2134398966L, -2131333572L, -2127947206L,
	-2124240380L, -2120213651L, -2115867626L, -2111202959L, -2106220352L,
	-2100920556L, -2095304370L, -2089372638L, -2083126254L, -2076566160L,
	-2069693342L, -2062508835L, -2055013723L, -2047209133L, -2039096241L,
	-2030676269L, -2021950484L, -2012920201L, -2003586779L, -1993951625L,
	-1984016189L, -1973781967L, -1963250501L, -1952423377L, -1941302225L,
	-1929888720L, -1918184581L, -1906191570L, -1893911494L, -1881346202L,
	-1868497586L, -1855367581L, -1841958164L, -1828271356L, -1814309216L,
	-1800073849L, -1785567396L, -1770792044L, -1755750017L, -1740443581L,
	-1724875040L, -1709046739L, -1692961062L, -1676620432L, -1660027308L,
	-1643184191L, -1626093616L, -1608758157L, -1591180426L, -1573363068L,
	-1555308768L, -1537020244L, -1518500250L, -1499751576L, -1480777044L,
	-1461579514L, -1442161874L, -1422527051L, -1402678000L, -1382617710L,
	-1362349204L, -1341875533L, -1321199781L, -1300325060L, -1279254516L,
	-1257991320L, -1236538675L, -1214899813L, -1193077991L, -1171076495L,
	-1148898640L, -1126547765L, -1104027237L, -1081340445L, -1058490808L,
	-1035481766L, -1012316784L, -988999351L, -965532978L, -941921200L,
	-918167572L, -894275671L, -870249095L, -846091463L, -821806413L, -797397602L,
	-772868706L, -748223418L, -723465451L, -698598533L, -673626408L, -648552838L,
	-623381598L, -598116479L, -572761285L, -547319836L, -521795963L, -496193509L,
	-470516330L, -444768294L, -418953276L, -393075166L, -367137861L, -341145265L,
	-315101295L, -289009871L, -262874923L, -236700388L, -210490206L, -184248325L,
	-157978697L, -131685278L, -105372028L, -79042909L, -52701887L, -26352928L, 0
};

/**
 * \par
 * Table values are in Q15 (1.15 fixed-point format) and generation is done in
 * three steps.  First,  generate sin values in floating point:
 * <pre>
 * tableSize = 512;
 * for(n = 0; n < (tableSize + 1); n++)
 * {
 *	sinTable[n]= sin(2*pi*n/tableSize);
 * } </pre>
 * where pi value is  3.14159265358979
 * \par
 * Second, convert floating-point to Q15 (Fixed point):
 *	(sinTable[i] * pow(2, 15))
 * \par
 * Finally, round to the nearest integer value:
 * 	sinTable[i] += (sinTable[i] > 0 ? 0.5 :-0.5);
 */
const q15_t sinTable_q15[FAST_MATH_TABLE_SIZE + 1] = {
	0, 402, 804, 1206, 1608, 2009, 2411, 2811, 3212, 3612, 4011, 4410, 4808,
	5205, 5602, 5998, 6393, 6787, 7180, 7571, 7962, 8351, 8740, 9127, 9512,
	9896, 10279, 10660, 11039, 11417, 11793, 12167, 12540, 12910, 13279,
	13646, 14010, 14373, 14733, 15091, 15447, 15800, 16151, 16500, 16846,
	17190, 17531, 17869, 18205, 18538, 18868, 19195, 19520, 19841, 20160,
	20475, 20788, 21097, 21403, 21706, 22006, 22302, 22595, 22884, 23170,
	23453, 23732, 24008, 24279, 24548, 24812, 25073, 25330, 25583, 25833,
	26078, 26320, 26557, 26791, 27020, 27246, 27467, 27684, 27897, 28106,
	28311, 28511, 28707, 28899, 29086, 29269, 29448, 29622, 29792, 29957,
	30118, 30274, 30425, 30572, 30715, 30853, 30986, 31114, 31238, 31357,
	31471, 31581, 31686, 31786, 31881, 31972, 32058, 32138, 32214, 32286,
	32352, 32413, 32470, 32522, 32568, 32610, 32647, 32679, 32706, 32729,
	32746, 32758, 32766, 32767, 32766, 32758, 32746, 32729, 32706, 32679,
	32647, 32610, 32568, 32522, 32470, 32413, 32352, 32286, 32214, 32138,
	32058, 31972, 31881, 31786, 31686, 31581, 31471, 31357, 31238, 31114,
	30986, 30853, 30715, 30572, 30425, 30274, 30118, 29957, 29792, 29622,
	29448, 29269, 29086, 28899, 28707, 28511, 28311, 28106, 27897, 27684,
	27467, 27246, 27020, 26791, 26557, 26320, 26078, 25833, 25583, 25330,
	25073, 24812, 24548, 24279, 24008, 23732, 23453, 23170, 22884, 22595,
	22302, 22006, 21706, 21403, 21097, 20788, 20475, 20160, 19841, 19520,
	19195, 18868, 18538, 18205, 17869, 17531, 17190, 16846, 16500, 16151,
	15800, 15447, 15091, 14733, 14373, 14010, 13646, 13279, 12910, 12540,
	12167, 11793, 11417, 11039, 10660, 10279, 9896, 9512, 9127, 8740, 8351,
	7962, 7571, 7180, 6787, 6393, 5998, 5602, 5205, 4808, 4410, 4011, 3612,
	3212, 2811, 2411, 2009, 1608, 1206, 804, 402, 0, -402, -804, -1206,
	-1608, -2009, -2411, -2811, -3212, -3612, -4011, -4410, -4808, -5205,
	-5602, -5998, -6393, -6787, -7180, -7571, -7962, -8351, -8740, -9127,
	-9512, -9896, -10279, -10660, -11039, -11417, -11793, -12167, -12540,
	-12910, -13279, -13646, -14010, -14373, -14733, -15091, -15447, -15800,
	-16151, -16500, -16846, -17190, -17531, -17869, -18205, -18538, -18868,
	-19195, -19520, -19841, -20160, -20475, -20788, -21097, -21403, -21706,
	-22006, -22302, -22595, -22884, -23170, -23453, -23732, -24008, -24279,
	-24548, -24812, -25073, -25330, -25583, -25833, -26078, -26320, -26557,
	-26791, -27020, -27246, -27467, -27684, -27897, -28106, -28311, -28511,
	-28707, -28899, -29086, -29269, -29448, -29622, -29792, -29957, -30118,
	-30274, -30425, -30572, -30715, -30853, -30986, -31114, -31238, -31357,
	-31471, -31581, -31686, -31786, -31881, -31972, -32058, -32138, -32214,
	-32286, -32352, -32413, -32470, -32522, -32568, -32610, -32647, -32679,
	-32706, -32729, -32746, -32758, -32766, -32768, -32766, -32758, -32746,
	-32729, -32706, -32679, -32647, -32610, -32568, -32522, -32470, -32413,
	-32352, -32286, -32214, -32138, -32058, -31972, -31881, -31786, -31686,
	-31581, -31471, -31357, -31238, -31114, -30986, -30853, -30715, -30572,
	-30425, -30274, -30118, -29957, -29792, -29622, -29448, -29269, -29086,
	-28899, -28707, -28511, -28311, -28106, -27897, -27684, -27467, -27246,
	-27020, -26791, -26557, -26320, -26078, -25833, -25583, -25330, -25073,
	-24812, -24548, -24279, -24008, -23732, -23453, -23170, -22884, -22595,
	-22302, -22006, -21706, -21403, -21097, -20788, -20475, -20160, -19841,
	-19520, -19195, -18868, -18538, -18205, -17869, -17531, -17190, -16846,
	-16500, -16151, -15800, -15447, -15091, -14733, -14373, -14010, -13646,
	-13279, -12910, -12540, -12167, -11793, -11417, -11039, -10660, -10279,
	-9896, -9512, -9127, -8740, -8351, -7962, -7571, -7180, -6787, -6393,
	-5998, -5602, -5205, -4808, -4410, -4011, -3612, -3212, -2811, -2411,
	-2009, -1608, -1206, -804, -402, 0
};
