#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/kref.h>
#include <asm/uaccess.h>
#include <linux/usb.h>
#include <linux/version.h>
       #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,37)
#include <linux/smp_lock.h>
#else
#include <linux/spinlock.h>
#endif

/* Define these values to match your devices */
#define USB_CH341_VENDOR_ID_NEW		0x1a86
#define USB_CH341_VENDOR_ID_OLD		0x4348
#define USB_CH341_PRODUCT_ID			0x5512
#define DRIVER_VERSION "ch341_epp driver version 0.1"

/* table of devices that work with this driver */
static struct usb_device_id ch341_p_table [] = {
	{ USB_DEVICE(USB_CH341_VENDOR_ID_NEW,USB_CH341_PRODUCT_ID) },
	{ USB_DEVICE(USB_CH341_VENDOR_ID_OLD,USB_CH341_PRODUCT_ID) },
	{ }					/* Terminating entry */
};
MODULE_DEVICE_TABLE (usb, ch341_p_table);

/* Get a minor range for your devices from the usb maintainer */
#define USB_CH341_P_MINOR_BASE	200

/* Structure to hold all of our device specific stuff */
struct usb_ch341_p 
{
	struct usb_device *	udev;			/* the usb device for this device */
	struct usb_interface *	interface;		/* the interface for this device */
	unsigned char *		bulk_in_buffer;		/* the buffer to receive data */
	size_t			bulk_in_size;		/* the size of the receive buffer */
	__u8			bulk_in_endpointAddr;	/* the address of the bulk in endpoint */
	__u8			bulk_out_endpointAddr;	/* the address of the bulk out endpoint */
	struct kref		kref;
};

#define to_ch341_p_dev(d) 	container_of(d, struct usb_ch341_p, kref)

#define MIN(x,y)   ((x)<(y)?(x):(y))

static struct usb_driver ch341_p_driver;

static void ch341_p_delete(struct kref *kref)
{	
	struct usb_ch341_p *dev = to_ch341_p_dev(kref);
  printk("CH341 to dev ok!\n");
	usb_put_dev(dev->udev);

  printk("CH341 put dev ok!\n");
	
	if(dev->bulk_in_buffer != NULL)
	{
	  kfree (dev->bulk_in_buffer);
    printk("CH341 free buffer ok!\n");
  }
	kfree (dev);
  printk("CH341 free dev ok!\n");
}

static int ch341_p_open(struct inode *inode, struct file *file)
{
	struct usb_ch341_p *dev;
	struct usb_interface *interface;
	int subminor;
	int retval = 0;
	subminor = iminor(inode);

	interface = usb_find_interface(&ch341_p_driver, subminor);
	if (!interface) {
		pr_err ("%s - error, can't find device for minor %d",
		     __FUNCTION__, subminor);
		retval = -ENODEV;
		goto exit;
	}

	dev = usb_get_intfdata(interface);
	if (!dev) {
		retval = -ENODEV;
		goto exit;
	}

	/* increment our usage count for the device */
	kref_get(&dev->kref);

	/* save our object in the file's private structure */
	file->private_data = dev;

	usb_control_msg(dev->udev,usb_sndctrlpipe(dev->udev,0),0x9a,0x40,0x2525,0,NULL,0,0);
	
exit:
	return retval;
}


static int ch341_p_release(struct inode *inode, struct file *file)
{
	struct usb_ch341_p *dev;
	dev = (struct usb_ch341_p *)file->private_data;
	if (dev == NULL)
		return -ENODEV;

	/* decrement the count on our device */
	kref_put(&dev->kref, ch341_p_delete);
	return 0;
}

static ssize_t ch341_p_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	struct usb_ch341_p *dev;
	int retval = 0;
	int bytes_read;

	dev = (struct usb_ch341_p *)file->private_data;
	
	/* do a blocking bulk read to get data from the device */
	retval = usb_bulk_msg(dev->udev,
			      usb_rcvbulkpipe(dev->udev, dev->bulk_in_endpointAddr),
			      dev->bulk_in_buffer,
			      min(dev->bulk_in_size, count),
			      &bytes_read, 10000);

	/* if the read was successful, copy the data to userspace */
	if (!retval) {
		if (copy_to_user(buffer, dev->bulk_in_buffer, bytes_read))
			retval = -EFAULT;
		else
			retval = bytes_read;
	}

	return retval;
}

static void ch341_p_write_bulk_callback(struct urb *urb, struct pt_regs *regs)
{
	struct usb_ch341_p *dev;

	dev = (struct usb_ch341_p *)urb->context;

	/* sync/async unlink faults aren't errors */
	if (urb->status && 
	    !(urb->status == -ENOENT || 
	      urb->status == -ECONNRESET ||
	      urb->status == -ESHUTDOWN)) 
	{
		pr_dbg("%s - nonzero write bulk status received: %d",
		    __FUNCTION__, urb->status);
	}


	/* free up our allocated buffer */
       #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,37)
	usb_buffer_free(urb->dev, urb->transfer_buffer_length, 
			urb->transfer_buffer, urb->transfer_dma);
			#else
	usb_free_coherent(urb->dev, urb->transfer_buffer_length, 
			urb->transfer_buffer, urb->transfer_dma);
			
       #endif
}

static ssize_t ch341_p_write(struct file *file, const char *user_buffer, size_t count, loff_t *ppos)
{
	struct usb_ch341_p *dev;
	int retval = 0;
	struct urb *urb = NULL;
	char *buf = NULL;

	dev = (struct usb_ch341_p *)file->private_data;

	/* verify that we actually have some data to write */
	if (count == 0)
		goto exit;

	/* create a urb, and a buffer for it, and copy the data to the urb */
	urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!urb) 
	{
		retval = -ENOMEM;
		goto error;
	}
        #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,37)
	buf = usb_buffer_alloc(dev->udev, count, GFP_KERNEL, &urb->transfer_dma);
        #else
	urb->transfer_buffer = usb_alloc_coherent(dev->udev, count, GFP_KERNEL, &urb->transfer_dma);
        #endif
        #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,37)
	if (!buf)
        #else
        if(urb->transfer_buffer == NULL)
        #endif 
	{
		retval = -ENOMEM;
               // printk(KERN_INFO "Enomem\n");
		goto error;
	}
        #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,37)
	if (copy_from_user(buf, user_buffer, count))
        #else 
	if (copy_from_user(urb->transfer_buffer, user_buffer, count)) 
        #endif
	{
		retval = -EFAULT;
             //   printk(KERN_INFO "EFAULT\n");
		goto error;
	}
        #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,37)
	usb_fill_bulk_urb(urb, dev->udev,
			  usb_sndbulkpipe(dev->udev, dev->bulk_out_endpointAddr),
			  buf, count, (usb_complete_t)ch341_p_write_bulk_callback, dev);
        #else
	/* initialize the urb properly */
	usb_fill_bulk_urb(urb, dev->udev,
			  usb_sndbulkpipe(dev->udev, dev->bulk_out_endpointAddr),
			  //buf, count, ch341_p_write_bulk_callback, dev);
			  urb->transfer_buffer, count, (usb_complete_t)ch341_p_write_bulk_callback, dev);
        #endif
	urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;

	/* send the data out the bulk port */
	retval = usb_submit_urb(urb, GFP_KERNEL);
	if (retval) 
	{
		pr_err("%s - failed submitting write urb, error %d", __FUNCTION__, retval);
		goto error;
	}

	/* release our reference to this urb, the USB core will eventually free it entirely */
	usb_free_urb(urb);

exit:
	return count;

error:
       #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,37)
	usb_buffer_free(dev->udev, count, buf, urb->transfer_dma);
	#else
	usb_free_coherent(dev->udev, count, buf, urb->transfer_dma);
       #endif
	usb_free_urb(urb);
	return retval;
}

static struct file_operations ch341_p_fops = {
	.owner =	THIS_MODULE,
	.read  =	ch341_p_read,
	.write =	ch341_p_write,
	.open  =	ch341_p_open,
	.release =	ch341_p_release,
};

/* 
 * usb class driver info in order to get a minor number from the usb core,
 * and to have the device registered with devfs and the driver core
 */
static struct usb_class_driver ch341_p_class = 
{
	.name =		"usb/ch341_p%d",
	.fops =		&ch341_p_fops,
	.minor_base =	USB_CH341_P_MINOR_BASE,
};

static int ch341_p_probe(struct usb_interface *interface, const struct usb_device_id *id)
{
	struct usb_ch341_p *dev;
	struct usb_host_interface *iface_desc;
	struct usb_endpoint_descriptor *endpoint;
	size_t buffer_size;
	int i;
	int retval = -ENOMEM;
	/* allocate memory for our device state and initialize it */
	dev = kmalloc(sizeof(*dev), GFP_KERNEL);
	if (dev == NULL) {
		pr_err("Out of memory");
		goto error;
	}
	memset(dev, 0x00, sizeof(*dev));
	kref_init(&dev->kref);

	dev->udev = usb_get_dev(interface_to_usbdev(interface));
	dev->interface = interface;

	/* set up the endpoint information */
	/* use only the first bulk-in and bulk-out endpoints */
	iface_desc = interface->cur_altsetting;
	for (i = 0; i < iface_desc->desc.bNumEndpoints; ++i) 
	{
		endpoint = &iface_desc->endpoint[i].desc;

		if (!dev->bulk_in_endpointAddr &&
		    (endpoint->bEndpointAddress & 0x80) && (!(endpoint->bEndpointAddress & 0x01)) &&
		    ((endpoint->bmAttributes & 3)
					== 0x02)) 
		{
			/* we found a bulk in endpoint */
			buffer_size = le16_to_cpu(endpoint->wMaxPacketSize);
			dev->bulk_in_size = buffer_size;
			dev->bulk_in_endpointAddr = endpoint->bEndpointAddress;
			dev->bulk_in_buffer = kmalloc(4096, GFP_KERNEL);
			if (!dev->bulk_in_buffer) 
			{
				goto error;
			}

		}

		if (!dev->bulk_out_endpointAddr &&
		    !(endpoint->bEndpointAddress & 0x81) &&
		    ((endpoint->bmAttributes & 3)
					== 0x02)) 
		{
			/* we found a bulk out endpoint */
			dev->bulk_out_endpointAddr = endpoint->bEndpointAddress;
		}
	}
	if (!(dev->bulk_in_endpointAddr && dev->bulk_out_endpointAddr)) 
	{
		pr_err("Could not find both bulk-in and bulk-out endpoints");
		goto error;
	}

	/* save our data pointer in this interface device */
	usb_set_intfdata(interface, dev);

	/* we can register the device now, as it is ready */
	retval = usb_register_dev(interface, &ch341_p_class);
	if (retval) 
	{
		/* something prevented us from registering this driver */
		pr_err("Not able to get a minor for this device.");
		usb_set_intfdata(interface, NULL);
		goto error;
	}

	/* let the user know what node this device is now attached to */
       dev_info(&dev->udev->dev, "USB ch341_p device now attached to wch_ch341_p-%d", interface->minor);
	return 0;

error:
	if (dev)
		kref_put(&dev->kref, ch341_p_delete);
	return retval;
}

static void ch341_p_disconnect(struct usb_interface *interface)
{
	struct usb_ch341_p *dev;
	//int minor = interface->minor;
       #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,37)
       #else
         #if LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0)
         spinlock_t ch341_lock = SPIN_LOCK_UNLOCKED;
         unsigned long flags;
         #endif
       #endif


	/* prevent ch341_p_open() from racing ch341_p_disconnect() */
       #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,37)
	lock_kernel();
       #else
         #if LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0)
         spin_lock_irqsave(&ch341_lock,flags);
         #endif
       #endif
	dev = usb_get_intfdata(interface);
	usb_set_intfdata(interface, NULL);

	/* give back our minor */
	usb_deregister_dev(interface, &ch341_p_class);
       #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,37)
	unlock_kernel();
       #else
         #if LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0)
        spin_unlock_irqrestore(&ch341_lock,flags);
        #endif
       #endif

	/* decrement our usage count */
	kref_put(&dev->kref, ch341_p_delete);

        //dev_info(&dev->udev->dev, "USB ch341_p #%d now disconnected", minor);
        //info("USB ch341_p #%d now disconnected", minor);
}

static struct usb_driver ch341_p_driver = {
	.name =		"ch341_p",
	.probe =	ch341_p_probe,
	.disconnect =	ch341_p_disconnect,
	.id_table =	ch341_p_table,
};

static int __init usb_ch341_p_init(void)
{
	int result;
	/* register this driver with the USB subsystem */
	result = usb_register(&ch341_p_driver);
	if (result)
		pr_err("usb_register failed. Error number %d", result);
	return result;
}

static void __exit usb_ch341_p_exit(void)
{
	//int result;
	/* deregister this driver with the USB subsystem */
	//result = 
  usb_deregister(&ch341_p_driver);
	//if (result)
	//	err("usb_register failed. Error number %d", result);
}

module_init (usb_ch341_p_init);
module_exit (usb_ch341_p_exit);

MODULE_LICENSE("GPL");
