/*
 * $Id$
 */

package edu.jas.ufd;

import java.io.Serializable;
import java.util.List;
import java.util.SortedMap;

import edu.jas.poly.GenPolynomial;
import edu.jas.structure.GcdRingElem;

/**
 * Factorization algorithms interface.
 *
 * <p><b>Usage:</b> To create objects that implement the <code>Factorization</code> interface use
 * the <code>FactorFactory</code>. It will select an appropriate implementation based on the types
 * of polynomial coefficients C. To obtain an implementation use <code>getImplementation()</code>,
 * it returns an object of a class which extends the <code>FactorAbstract</code> class which
 * implements the <code>Factorization</code> interface.
 *
 * <p>
 *
 * <pre>
 * Factorization&lt;CT&gt; engine;
 * engine = FactorFactory.&lt;CT&gt; getImplementation(cofac);
 * c = engine.factors(a);
 * </pre>
 *
 * <p>For example, if the coefficient type is BigInteger, the usage looks like
 *
 * <pre>
 * BigInteger cofac = new BigInteger();
 * Factorization&lt;BigInteger&gt; engine;
 * engine = FactorFactory.getImplementation(cofac);
 * Sm = engine.factors(poly);
 * </pre>
 *
 * @author Heinz Kredel
 * @see edu.jas.ufd.FactorFactory#getImplementation
 */
public interface Factorization<C extends GcdRingElem<C>> extends Serializable {

  /**
   * GenPolynomial test if is irreducible.
   *
   * @param P GenPolynomial.
   * @return true if P is irreducible, else false.
   */
  public boolean isIrreducible(GenPolynomial<C> P);

  /**
   * GenPolynomial test if a non trivial factorization exsists.
   *
   * @param P GenPolynomial.
   * @return true if P is reducible, else false.
   */
  public boolean isReducible(GenPolynomial<C> P);

  /**
   * GenPolynomial test if is squarefree.
   *
   * @param P GenPolynomial.
   * @return true if P is squarefree, else false.
   */
  public boolean isSquarefree(GenPolynomial<C> P);

  /**
   * GenPolynomial factorization of a squarefree polynomial.
   *
   * @param P squarefree and primitive! or monic! GenPolynomial.
   * @return [p_1,...,p_k] with P = prod_{i=1,...,r} p_i.
   */
  public List<GenPolynomial<C>> factorsSquarefree(GenPolynomial<C> P);

  /**
   * GenPolynomial factorization.
   *
   * @param P GenPolynomial.
   * @return [p_1 -&gt; e_1, ..., p_k -&gt; e_k] with P = prod_{i=1,...,k} p_i**e_i.
   */
  public SortedMap<GenPolynomial<C>, Long> factors(GenPolynomial<C> P);

  /**
   * GenPolynomial factorization ignoring multiplicities.
   *
   * @param P GenPolynomial.
   * @return [p_1, ..., p_k] with P = prod_{i=1,...,k} p_i**{e_i} for some e_i.
   */
  public List<GenPolynomial<C>> factorsRadical(GenPolynomial<C> P);

  /**
   * GenPolynomial greatest squarefree divisor.
   *
   * @param P GenPolynomial.
   * @return squarefree(P).
   */
  public GenPolynomial<C> squarefreePart(GenPolynomial<C> P);

  /**
   * GenPolynomial squarefree factorization.
   *
   * @param P GenPolynomial.
   * @return [p_1 -&gt; e_1, ..., p_k -&gt; e_k] with P = prod_{i=1,...,k} p_i^{e_i} and p_i
   *     squarefree.
   */
  public SortedMap<GenPolynomial<C>, Long> squarefreeFactors(GenPolynomial<C> P);

  /**
   * GenPolynomial is factorization.
   *
   * @param P GenPolynomial
   * @param F = [p_1,...,p_k].
   * @return true if P = prod_{i=1,...,r} p_i, else false.
   */
  public boolean isFactorization(GenPolynomial<C> P, List<GenPolynomial<C>> F);

  /**
   * GenPolynomial is factorization.
   *
   * @param P GenPolynomial.
   * @param F = [p_1 -&gt; e_1, ..., p_k -&gt; e_k].
   * @return true if P = prod_{i=1,...,k} p_i**e_i , else false.
   */
  public boolean isFactorization(GenPolynomial<C> P, SortedMap<GenPolynomial<C>, Long> F);
}
