/*
 * java-math-library is a Java library focused on number theory, but not necessarily limited to it. It is based on the PSIQS 4.0 factoring project.
 * Copyright (C) 2018 Tilman Neumann (www.tilman-neumann.de)
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program;
 * if not, see <http://www.gnu.org/licenses/>.
 */
package de.tilman_neumann.jml.roots;

import static de.tilman_neumann.jml.base.BigIntConstants.*;

import java.math.BigInteger;
import java.util.HashSet;

/**
 * Fast recognition of exact integer squares, using the algorithm explained in class SqrtExactTest.
 *
 * <p>Actually this is a copy of class SqrtExact03_12_v02, based on dataset 3 and using the square
 * bit pattern test proposed by Graeme Willoughby, instead of a fastMod test.
 *
 * <p>This is the fastest version so far.
 *
 * @author Tilman Neumann
 */
public class SqrtExact {

  private static final int MOD_COUNT = 10;

  /** mod list starting with FASTMOD = 2^12 */
  private static int[] mods = new int[] {4096, 4095, 3553, 3901, 3937, 2479, 3841, 2623, 3161, 97};

  /** square rests modulo m indexed by m */
  @SuppressWarnings("unchecked")
  private static HashSet<Integer>[] squareRests = new HashSet[MOD_COUNT];

  // fastModExp = 12 -> max mod = 2^12 = 4096:
  // square bit pattern test -> 16.655146% possible squares
  static HashSet<Integer> squareRestsMod4095() {
    return toHashSet(
        new int[] {
          0, 1, 4, 9, 16, 25, 36, 49, 64, 79, 81, 91, 100, 121, 126, 130, 144, 169, 196, 205, 211,
          225, 235, 256, 259, 261, 270, 274, 289, 295, 315, 316, 324, 351, 361, 364, 394, 400, 406,
          415, 441, 445, 459, 469, 484, 490, 504, 511, 520, 529, 550, 571, 576, 585, 589, 595, 610,
          625, 634, 646, 666, 676, 679, 711, 715, 729, 751, 781, 784, 805, 819, 820, 835, 841, 844,
          855, 900, 910, 919, 940, 945, 946, 949, 961, 991, 1015, 1024, 1026, 1030, 1036, 1044,
          1054, 1066, 1075, 1080, 1089, 1096, 1114, 1134, 1135, 1141, 1156, 1170, 1171, 1180, 1206,
          1219, 1225, 1234, 1260, 1261, 1264, 1296, 1309, 1330, 1339, 1351, 1366, 1369, 1381, 1390,
          1395, 1404, 1414, 1429, 1444, 1456, 1465, 1485, 1486, 1495, 1521, 1530, 1534, 1561, 1570,
          1576, 1600, 1611, 1621, 1624, 1639, 1654, 1660, 1674, 1681, 1719, 1726, 1729, 1759, 1764,
          1765, 1771, 1780, 1810, 1834, 1836, 1845, 1849, 1855, 1876, 1885, 1894, 1899, 1915, 1936,
          1954, 1960, 1975, 1989, 1990, 1999, 2011, 2016, 2025, 2041, 2044, 2079, 2080, 2115, 2116,
          2146, 2149, 2170, 2185, 2200, 2206, 2209, 2214, 2275, 2284, 2304, 2305, 2311, 2314, 2326,
          2331, 2340, 2349, 2356, 2380, 2389, 2395, 2401, 2419, 2430, 2431, 2440, 2461, 2466, 2479,
          2500, 2506, 2521, 2536, 2545, 2584, 2590, 2599, 2601, 2626, 2629, 2655, 2664, 2674, 2695,
          2704, 2716, 2731, 2734, 2746, 2755, 2779, 2781, 2794, 2809, 2821, 2830, 2835, 2844, 2851,
          2860, 2899, 2916, 2926, 2934, 2935, 2941, 2961, 2965, 2986, 2989, 3004, 3019, 3025, 3046,
          3091, 3094, 3124, 3130, 3136, 3145, 3150, 3159, 3175, 3186, 3199, 3214, 3220, 3241, 3249,
          3250, 3259, 3276, 3280, 3285, 3301, 3319, 3325, 3340, 3355, 3364, 3376, 3406, 3409, 3420,
          3445, 3474, 3481, 3501, 3511, 3514, 3535, 3546, 3550, 3565, 3571, 3574, 3591, 3600, 3640,
          3649, 3654, 3670, 3676, 3679, 3691, 3721, 3735, 3745, 3754, 3760, 3766, 3780, 3784, 3796,
          3805, 3826, 3844, 3861, 3865, 3871, 3886, 3901, 3910, 3949, 3955, 3964, 3969, 3991, 3994,
          4005, 4039, 4060, 4069, 4081
        });
  }

  static {
    squareRests[1] = squareRestsMod4095();
  }
  // add mod 4095 (336 square rests) -> ~ 1.3693% possible squares
  static HashSet<Integer> squareRestsMod3553() {
    return toHashSet(
        new int[] {
          0, 1, 4, 9, 16, 25, 26, 36, 38, 42, 47, 49, 55, 64, 66, 77, 81, 93, 100, 102, 104, 111,
          115, 119, 121, 137, 144, 152, 157, 168, 169, 187, 188, 191, 196, 213, 220, 225, 229, 234,
          247, 251, 253, 254, 256, 264, 273, 289, 290, 291, 302, 308, 323, 324, 339, 342, 353, 361,
          366, 372, 378, 389, 399, 400, 408, 410, 416, 423, 427, 429, 434, 438, 441, 443, 444, 460,
          463, 467, 476, 484, 495, 510, 511, 518, 529, 543, 548, 562, 570, 576, 577, 586, 587, 594,
          595, 608, 614, 625, 627, 628, 631, 638, 647, 650, 663, 665, 671, 672, 676, 682, 693, 727,
          729, 731, 746, 748, 752, 757, 764, 784, 786, 790, 795, 803, 807, 814, 815, 817, 818, 834,
          837, 841, 852, 859, 880, 883, 885, 900, 902, 916, 917, 918, 935, 936, 950, 951, 961, 969,
          973, 988, 994, 995, 999, 1004, 1005, 1012, 1016, 1024, 1035, 1037, 1045, 1046, 1050, 1054,
          1056, 1070, 1071, 1087, 1089, 1090, 1092, 1103, 1109, 1122, 1126, 1137, 1138, 1147, 1156,
          1160, 1164, 1175, 1182, 1189, 1203, 1208, 1222, 1225, 1232, 1233, 1241, 1254, 1258, 1259,
          1274, 1277, 1279, 1290, 1292, 1296, 1301, 1309, 1318, 1334, 1335, 1347, 1356, 1358, 1368,
          1369, 1373, 1375, 1379, 1398, 1411, 1412, 1413, 1430, 1441, 1444, 1445, 1453, 1461, 1463,
          1464, 1479, 1483, 1488, 1505, 1512, 1521, 1529, 1543, 1545, 1555, 1556, 1562, 1563, 1565,
          1577, 1582, 1596, 1600, 1607, 1615, 1631, 1632, 1640, 1650, 1651, 1653, 1662, 1664, 1670,
          1681, 1683, 1692, 1698, 1708, 1716, 1717, 1719, 1721, 1730, 1736, 1738, 1749, 1752, 1753,
          1764, 1772, 1776, 1783, 1786, 1787, 1793, 1802, 1835, 1840, 1849, 1852, 1862, 1868, 1871,
          1879, 1885, 1886, 1904, 1906, 1917, 1919, 1923, 1925, 1930, 1936, 1939, 1947, 1963, 1973,
          1974, 1980, 1981, 1985, 2002, 2006, 2014, 2025, 2038, 2039, 2040, 2044, 2049, 2057, 2058,
          2061, 2072, 2082, 2090, 2091, 2095, 2099, 2106, 2110, 2116, 2126, 2134, 2167, 2172, 2189,
          2192, 2194, 2201, 2209, 2211, 2223, 2227, 2242, 2248, 2253, 2259, 2270, 2277, 2278, 2280,
          2286, 2291, 2297, 2299, 2303, 2304, 2308, 2310, 2325, 2337, 2344, 2346, 2348, 2354, 2363,
          2365, 2376, 2379, 2380, 2381, 2398, 2399, 2401, 2410, 2413, 2414, 2418, 2429, 2432, 2456,
          2457, 2467, 2486, 2490, 2495, 2498, 2500, 2508, 2512, 2517, 2524, 2531, 2533, 2534, 2546,
          2550, 2552, 2563, 2566, 2585, 2588, 2600, 2601, 2610, 2619, 2622, 2627, 2633, 2652, 2660,
          2665, 2667, 2671, 2677, 2684, 2685, 2688, 2695, 2699, 2704, 2707, 2718, 2721, 2722, 2728,
          2733, 2737, 2753, 2755, 2762, 2772, 2775, 2797, 2809, 2821, 2831, 2838, 2847, 2854, 2869,
          2874, 2875, 2886, 2894, 2905, 2907, 2908, 2916, 2924, 2926, 2937, 2942, 2949, 2962, 2971,
          2973, 2975, 2984, 2990, 2992, 3007, 3008, 3018, 3025, 3026, 3028, 3030, 3041, 3045, 3047,
          3051, 3056, 3059, 3078, 3085, 3094, 3095, 3102, 3103, 3113, 3127, 3136, 3144, 3146, 3158,
          3160, 3161, 3171, 3177, 3179, 3180, 3212, 3215, 3217, 3228, 3234, 3239, 3246, 3249, 3256,
          3260, 3265, 3268, 3272, 3279, 3294, 3298, 3311, 3315, 3323, 3331, 3334, 3336, 3345, 3348,
          3349, 3353, 3364, 3367, 3370, 3382, 3391, 3399, 3402, 3408, 3421, 3425, 3426, 3436, 3443,
          3455, 3459, 3469, 3477, 3481, 3501, 3502, 3503, 3520, 3521, 3532, 3534, 3535, 3540, 3545,
          3551
        });
  }

  static {
    squareRests[2] = squareRestsMod3553();
  }
  // add mod 3553 (540 square rests) -> ~ 0.20738% possible squares
  static HashSet<Integer> squareRestsMod3901() {
    return toHashSet(
        new int[] {
          0, 1, 3, 4, 7, 9, 12, 16, 17, 21, 25, 27, 28, 36, 37, 48, 49, 51, 59, 61, 63, 64, 65, 68,
          75, 81, 83, 84, 94, 95, 100, 106, 108, 110, 111, 112, 119, 121, 131, 142, 144, 147, 148,
          153, 158, 166, 169, 173, 175, 177, 178, 183, 189, 191, 192, 194, 195, 196, 197, 202, 204,
          206, 215, 225, 230, 235, 236, 241, 243, 244, 247, 249, 252, 253, 256, 259, 260, 272, 277,
          282, 285, 286, 289, 290, 298, 300, 310, 314, 318, 319, 324, 330, 332, 333, 335, 336, 341,
          343, 353, 357, 361, 363, 365, 376, 380, 383, 393, 397, 400, 401, 410, 413, 418, 424, 425,
          426, 427, 431, 432, 440, 441, 444, 448, 451, 455, 459, 474, 476, 478, 479, 484, 498, 502,
          507, 519, 521, 523, 524, 525, 526, 529, 531, 534, 535, 538, 542, 549, 559, 566, 567, 568,
          573, 576, 581, 582, 585, 588, 591, 592, 598, 606, 611, 612, 614, 617, 618, 619, 625, 629,
          632, 645, 658, 659, 662, 664, 665, 667, 674, 675, 676, 685, 690, 692, 694, 695, 700, 705,
          708, 712, 713, 723, 729, 732, 733, 739, 741, 742, 747, 754, 756, 758, 759, 764, 768, 770,
          773, 776, 777, 780, 784, 788, 806, 808, 811, 815, 816, 817, 824, 831, 833, 841, 846, 847,
          853, 855, 858, 860, 863, 867, 870, 871, 874, 878, 893, 894, 895, 899, 900, 905, 907, 911,
          914, 917, 920, 925, 929, 930, 940, 941, 942, 943, 944, 946, 949, 954, 957, 961, 964, 972,
          974, 976, 977, 982, 988, 990, 991, 994, 996, 999, 1003, 1005, 1008, 1012, 1019, 1021,
          1023, 1024, 1029, 1034, 1036, 1037, 1040, 1055, 1059, 1061, 1066, 1071, 1082, 1083, 1088,
          1089, 1090, 1095, 1102, 1105, 1106, 1108, 1109, 1115, 1117, 1123, 1128, 1130, 1140, 1142,
          1144, 1149, 1156, 1160, 1162, 1165, 1178, 1179, 1183, 1187, 1189, 1191, 1192, 1193, 1199,
          1200, 1202, 1203, 1211, 1223, 1225, 1226, 1230, 1231, 1239, 1240, 1243, 1246, 1249, 1254,
          1256, 1270, 1271, 1272, 1273, 1275, 1276, 1278, 1281, 1283, 1285, 1286, 1293, 1294, 1296,
          1306, 1320, 1322, 1323, 1328, 1332, 1337, 1340, 1344, 1353, 1358, 1364, 1365, 1366, 1369,
          1372, 1377, 1379, 1387, 1391, 1397, 1405, 1411, 1412, 1414, 1418, 1422, 1427, 1428, 1434,
          1437, 1438, 1442, 1444, 1447, 1452, 1459, 1460, 1474, 1475, 1481, 1489, 1494, 1504, 1505,
          1506, 1510, 1511, 1520, 1521, 1522, 1525, 1531, 1532, 1538, 1553, 1555, 1557, 1558, 1559,
          1563, 1569, 1572, 1575, 1578, 1587, 1588, 1593, 1598, 1600, 1602, 1604, 1605, 1606, 1607,
          1610, 1614, 1615, 1625, 1626, 1640, 1645, 1646, 1647, 1652, 1654, 1661, 1663, 1669, 1670,
          1672, 1677, 1681, 1687, 1693, 1696, 1698, 1700, 1701, 1704, 1708, 1709, 1719, 1724, 1728,
          1729, 1741, 1743, 1746, 1747, 1753, 1755, 1760, 1764, 1766, 1771, 1773, 1776, 1781, 1787,
          1792, 1794, 1802, 1804, 1807, 1811, 1813, 1818, 1820, 1833, 1835, 1836, 1837, 1842, 1847,
          1849, 1851, 1854, 1857, 1867, 1870, 1875, 1887, 1889, 1894, 1896, 1901, 1904, 1907, 1912,
          1916, 1930, 1934, 1935, 1936, 1939, 1945, 1974, 1977, 1978, 1986, 1990, 1992, 1995, 1999,
          2001, 2002, 2008, 2021, 2022, 2023, 2025, 2028, 2029, 2030, 2033, 2053, 2055, 2057, 2069,
          2070, 2075, 2076, 2082, 2084, 2085, 2086, 2092, 2096, 2100, 2102, 2104, 2105, 2115, 2116,
          2119, 2123, 2124, 2136, 2139, 2140, 2143, 2152, 2162, 2165, 2168, 2169, 2170, 2174, 2179,
          2183, 2186, 2187, 2189, 2194, 2196, 2198, 2199, 2209, 2217, 2221, 2223, 2226, 2227, 2233,
          2236, 2241, 2245, 2251, 2257, 2258, 2262, 2264, 2268, 2270, 2272, 2274, 2277, 2281, 2290,
          2292, 2304, 2305, 2306, 2309, 2310, 2311, 2319, 2324, 2327, 2328, 2331, 2335, 2340, 2345,
          2350, 2351, 2352, 2353, 2354, 2357, 2362, 2364, 2368, 2375, 2387, 2392, 2399, 2401, 2405,
          2411, 2414, 2418, 2424, 2433, 2434, 2444, 2445, 2447, 2448, 2451, 2456, 2458, 2468, 2471,
          2472, 2476, 2491, 2493, 2494, 2497, 2499, 2500, 2507, 2515, 2516, 2518, 2519, 2523, 2527,
          2528, 2538, 2539, 2541, 2554, 2555, 2559, 2565, 2574, 2580, 2585, 2589, 2594, 2599, 2601,
          2602, 2603, 2606, 2609, 2610, 2613, 2617, 2621, 2622, 2632, 2634, 2636, 2638, 2641, 2648,
          2650, 2656, 2657, 2659, 2660, 2666, 2668, 2679, 2681, 2682, 2683, 2685, 2686, 2687, 2693,
          2696, 2697, 2700, 2704, 2707, 2715, 2721, 2726, 2733, 2734, 2740, 2742, 2743, 2750, 2751,
          2760, 2762, 2768, 2775, 2776, 2777, 2779, 2780, 2787, 2790, 2798, 2800, 2807, 2809, 2820,
          2822, 2823, 2826, 2829, 2832, 2834, 2838, 2845, 2847, 2848, 2852, 2862, 2870, 2871, 2873,
          2881, 2883, 2885, 2891, 2892, 2899, 2903, 2909, 2914, 2915, 2916, 2917, 2921, 2922, 2926,
          2928, 2930, 2931, 2932, 2935, 2938, 2941, 2942, 2946, 2956, 2964, 2968, 2969, 2970, 2973,
          2975, 2982, 2986, 2988, 2989, 2995, 2997, 2998, 3009, 3011, 3014, 3015, 3016, 3017, 3024,
          3025, 3026, 3029, 3032, 3036, 3056, 3057, 3058, 3063, 3069, 3071, 3072, 3080, 3082, 3083,
          3087, 3092, 3097, 3102, 3104, 3108, 3109, 3111, 3119, 3120, 3130, 3134, 3136, 3139, 3149,
          3152, 3155, 3157, 3158, 3161, 3163, 3165, 3166, 3170, 3177, 3181, 3183, 3185, 3191, 3198,
          3202, 3203, 3205, 3213, 3217, 3223, 3224, 3232, 3238, 3244, 3246, 3247, 3249, 3260, 3264,
          3267, 3268, 3270, 3275, 3277, 3285, 3296, 3298, 3302, 3306, 3307, 3314, 3315, 3318, 3324,
          3327, 3332, 3337, 3341, 3343, 3345, 3346, 3349, 3351, 3353, 3358, 3361, 3364, 3369, 3371,
          3379, 3384, 3385, 3388, 3390, 3398, 3401, 3412, 3420, 3426, 3431, 3432, 3433, 3434, 3439,
          3440, 3443, 3447, 3452, 3467, 3468, 3473, 3478, 3480, 3481, 3484, 3486, 3487, 3490, 3495,
          3496, 3502, 3503, 3512, 3514, 3515, 3526, 3527, 3534, 3537, 3549, 3550, 3561, 3567, 3572,
          3573, 3576, 3578, 3579, 3580, 3581, 3586, 3590, 3596, 3597, 3599, 3600, 3606, 3609, 3620,
          3628, 3633, 3637, 3644, 3646, 3647, 3653, 3655, 3656, 3661, 3668, 3669, 3673, 3675, 3678,
          3680, 3682, 3683, 3690, 3693, 3700, 3703, 3713, 3715, 3716, 3717, 3720, 3721, 3722, 3727,
          3729, 3730, 3738, 3745, 3747, 3760, 3761, 3762, 3763, 3764, 3766, 3768, 3772, 3776, 3784,
          3794, 3796, 3810, 3813, 3816, 3819, 3821, 3825, 3828, 3834, 3835, 3839, 3841, 3843, 3844,
          3849, 3854, 3855, 3856, 3858, 3862, 3866, 3879, 3881, 3882, 3886, 3888, 3896
        });
  }

  static {
    squareRests[3] = squareRestsMod3901();
  }
  // add mod 3901 (1008 square rests) -> ~ 0.05282% possible squares
  static HashSet<Integer> squareRestsMod3937() {
    return toHashSet(
        new int[] {
          0, 1, 2, 4, 8, 9, 16, 18, 19, 25, 31, 32, 35, 36, 38, 41, 47, 49, 50, 62, 64, 69, 70, 71,
          72, 76, 81, 82, 87, 94, 98, 100, 103, 107, 113, 121, 124, 128, 129, 131, 138, 140, 142,
          143, 144, 149, 152, 157, 159, 162, 163, 164, 165, 169, 171, 174, 187, 188, 191, 195, 196,
          200, 206, 211, 214, 221, 225, 226, 227, 231, 242, 248, 249, 255, 256, 258, 262, 267, 273,
          276, 279, 280, 284, 286, 288, 289, 295, 298, 304, 314, 315, 318, 324, 326, 328, 330, 335,
          338, 341, 342, 348, 357, 361, 369, 374, 376, 381, 382, 390, 392, 397, 400, 403, 407, 411,
          412, 413, 417, 419, 422, 423, 428, 431, 441, 442, 443, 450, 452, 453, 454, 462, 465, 469,
          475, 479, 481, 484, 485, 496, 498, 501, 503, 505, 510, 512, 516, 521, 524, 527, 529, 534,
          543, 545, 546, 552, 555, 558, 560, 568, 572, 576, 577, 578, 589, 590, 596, 607, 608, 621,
          625, 628, 629, 630, 636, 639, 648, 651, 652, 653, 656, 660, 661, 665, 667, 669, 670, 671,
          676, 679, 682, 684, 687, 696, 707, 714, 717, 722, 723, 729, 733, 738, 748, 752, 762, 763,
          764, 775, 777, 779, 780, 783, 784, 793, 794, 800, 803, 806, 811, 814, 822, 824, 826, 831,
          834, 838, 841, 844, 846, 856, 862, 865, 869, 875, 877, 882, 884, 886, 893, 900, 904, 906,
          907, 908, 915, 919, 924, 927, 930, 931, 938, 939, 949, 950, 958, 961, 962, 963, 965, 968,
          970, 971, 977, 989, 992, 993, 996, 1002, 1006, 1010, 1011, 1017, 1020, 1024, 1025, 1027,
          1031, 1032, 1033, 1037, 1041, 1042, 1048, 1051, 1054, 1058, 1063, 1068, 1085, 1086, 1087,
          1089, 1090, 1092, 1095, 1103, 1104, 1110, 1116, 1120, 1123, 1136, 1144, 1147, 1151, 1152,
          1154, 1156, 1161, 1165, 1175, 1178, 1179, 1180, 1185, 1187, 1192, 1203, 1211, 1213, 1214,
          1216, 1217, 1219, 1225, 1227, 1237, 1241, 1242, 1247, 1250, 1256, 1258, 1260, 1265, 1271,
          1272, 1278, 1279, 1281, 1285, 1287, 1289, 1291, 1296, 1302, 1304, 1306, 1307, 1311, 1312,
          1320, 1322, 1330, 1334, 1338, 1340, 1341, 1342, 1343, 1349, 1351, 1352, 1358, 1364, 1368,
          1369, 1373, 1374, 1383, 1392, 1397, 1399, 1405, 1413, 1414, 1415, 1423, 1427, 1428, 1431,
          1433, 1434, 1435, 1444, 1446, 1457, 1458, 1459, 1461, 1465, 1466, 1467, 1471, 1473, 1476,
          1485, 1495, 1496, 1497, 1504, 1519, 1521, 1524, 1526, 1528, 1533, 1535, 1537, 1539, 1550,
          1554, 1555, 1558, 1559, 1560, 1566, 1568, 1585, 1586, 1588, 1595, 1597, 1600, 1606, 1612,
          1622, 1628, 1631, 1637, 1644, 1645, 1648, 1651, 1652, 1653, 1659, 1662, 1668, 1676, 1681,
          1682, 1683, 1688, 1692, 1693, 1712, 1713, 1715, 1719, 1721, 1723, 1724, 1725, 1730, 1733,
          1738, 1745, 1750, 1754, 1755, 1764, 1768, 1771, 1772, 1775, 1786, 1787, 1795, 1799, 1800,
          1803, 1808, 1812, 1814, 1816, 1830, 1838, 1839, 1847, 1848, 1849, 1854, 1857, 1860, 1862,
          1865, 1876, 1878, 1885, 1891, 1893, 1895, 1898, 1899, 1900, 1905, 1907, 1909, 1916, 1922,
          1923, 1924, 1926, 1927, 1930, 1931, 1936, 1940, 1941, 1942, 1947, 1954, 1955, 1957, 1967,
          1969, 1973, 1978, 1981, 1984, 1986, 1989, 1992, 1993, 2003, 2004, 2009, 2012, 2020, 2022,
          2025, 2029, 2033, 2034, 2040, 2043, 2047, 2048, 2050, 2051, 2053, 2054, 2062, 2064, 2066,
          2074, 2079, 2081, 2082, 2084, 2093, 2096, 2102, 2105, 2108, 2113, 2116, 2126, 2136, 2139,
          2147, 2149, 2153, 2159, 2167, 2170, 2172, 2174, 2175, 2177, 2178, 2180, 2184, 2189, 2190,
          2195, 2201, 2203, 2206, 2208, 2209, 2211, 2219, 2220, 2221, 2229, 2232, 2233, 2240, 2241,
          2246, 2257, 2263, 2272, 2279, 2281, 2283, 2288, 2294, 2295, 2299, 2301, 2302, 2303, 2304,
          2308, 2312, 2322, 2327, 2330, 2333, 2335, 2350, 2356, 2357, 2358, 2360, 2365, 2370, 2374,
          2384, 2389, 2401, 2403, 2406, 2407, 2415, 2422, 2426, 2428, 2432, 2434, 2438, 2443, 2449,
          2450, 2451, 2454, 2457, 2463, 2465, 2474, 2477, 2481, 2482, 2484, 2485, 2487, 2489, 2494,
          2500, 2511, 2512, 2513, 2516, 2520, 2530, 2542, 2544, 2549, 2551, 2556, 2558, 2561, 2562,
          2570, 2574, 2575, 2577, 2578, 2581, 2582, 2587, 2589, 2592, 2601, 2604, 2608, 2609, 2611,
          2612, 2613, 2614, 2622, 2624, 2639, 2640, 2643, 2644, 2653, 2655, 2660, 2667, 2668, 2671,
          2675, 2676, 2680, 2682, 2684, 2685, 2686, 2697, 2698, 2699, 2701, 2702, 2704, 2705, 2711,
          2716, 2717, 2728, 2729, 2735, 2736, 2737, 2738, 2746, 2748, 2761, 2766, 2767, 2784, 2787,
          2791, 2794, 2795, 2798, 2809, 2810, 2815, 2825, 2826, 2828, 2829, 2830, 2831, 2835, 2841,
          2846, 2854, 2856, 2862, 2866, 2868, 2870, 2888, 2892, 2893, 2897, 2901, 2911, 2914, 2915,
          2916, 2918, 2921, 2922, 2923, 2930, 2932, 2934, 2939, 2942, 2946, 2947, 2952, 2953, 2955,
          2959, 2963, 2965, 2970, 2973, 2981, 2983, 2985, 2990, 2992, 2994, 2995, 3008, 3009, 3015,
          3021, 3025, 3038, 3042, 3043, 3045, 3048, 3052, 3056, 3057, 3063, 3066, 3069, 3070, 3073,
          3074, 3078, 3079, 3083, 3085, 3089, 3097, 3100, 3108, 3109, 3110, 3116, 3118, 3119, 3120,
          3132, 3135, 3136, 3147, 3151, 3163, 3169, 3170, 3172, 3176, 3190, 3193, 3194, 3197, 3200,
          3201, 3207, 3209, 3211, 3212, 3213, 3224, 3225, 3243, 3244, 3249, 3256, 3257, 3259, 3262,
          3263, 3269, 3273, 3274, 3275, 3288, 3290, 3295, 3296, 3302, 3304, 3306, 3311, 3317, 3318,
          3319, 3321, 3324, 3327, 3333, 3336, 3337, 3349, 3352, 3362, 3364, 3366, 3373, 3376, 3381,
          3383, 3384, 3386, 3389, 3415, 3417, 3419, 3424, 3426, 3429, 3430, 3438, 3442, 3445, 3446,
          3448, 3450, 3451, 3455, 3459, 3460, 3461, 3466, 3473, 3476, 3479, 3481, 3490, 3491, 3497,
          3500, 3503, 3505, 3508, 3510, 3511, 3513, 3517, 3523, 3528, 3536, 3542, 3544, 3550, 3553,
          3565, 3567, 3569, 3572, 3573, 3574, 3575, 3581, 3590, 3593, 3597, 3598, 3600, 3603, 3605,
          3606, 3616, 3624, 3627, 3628, 3629, 3632, 3635, 3637, 3643, 3655, 3659, 3660, 3663, 3676,
          3677, 3678, 3683, 3691, 3694, 3696, 3698, 3699, 3705, 3708, 3709, 3714, 3717, 3720, 3721,
          3724, 3725, 3727, 3730, 3745, 3751, 3752, 3753, 3755, 3756, 3759, 3765, 3767, 3770, 3771,
          3782, 3783, 3786, 3787, 3790, 3796, 3798, 3800, 3807, 3810, 3814, 3818, 3821, 3823, 3827,
          3829, 3831, 3832, 3841, 3844, 3845, 3846, 3848, 3851, 3852, 3854, 3860, 3862, 3872, 3879,
          3880, 3882, 3883, 3884, 3889, 3891, 3894, 3908, 3910, 3913, 3914, 3925, 3931, 3934
        });
  }

  static {
    squareRests[4] = squareRestsMod3937();
  }
  // add mod 3937 (1024 square rests) -> ~ 0.013225% possible squares
  static HashSet<Integer> squareRestsMod2479() {
    return toHashSet(
        new int[] {
          0, 1, 4, 9, 10, 16, 21, 25, 26, 33, 36, 37, 40, 47, 49, 62, 64, 65, 67, 71, 73, 77, 81,
          83, 84, 86, 90, 100, 102, 104, 107, 114, 121, 122, 123, 127, 132, 138, 144, 148, 149, 151,
          155, 157, 158, 159, 160, 169, 173, 174, 181, 188, 189, 194, 196, 201, 210, 211, 215, 218,
          222, 223, 225, 226, 234, 238, 248, 250, 255, 256, 260, 263, 266, 268, 269, 284, 285, 287,
          289, 292, 293, 297, 303, 305, 307, 308, 317, 322, 323, 324, 330, 332, 333, 336, 344, 345,
          349, 354, 358, 359, 360, 361, 370, 371, 374, 382, 391, 395, 397, 400, 403, 406, 408, 411,
          416, 417, 418, 419, 423, 428, 435, 437, 441, 451, 456, 469, 470, 478, 484, 485, 488, 490,
          491, 492, 493, 502, 506, 508, 509, 518, 525, 528, 529, 534, 545, 546, 551, 552, 555, 558,
          559, 562, 565, 571, 576, 583, 585, 591, 592, 595, 596, 601, 603, 604, 613, 617, 618, 619,
          620, 622, 625, 626, 628, 629, 632, 636, 638, 639, 640, 650, 657, 659, 662, 663, 665, 667,
          670, 676, 687, 691, 692, 693, 694, 696, 699, 703, 706, 707, 710, 719, 724, 729, 730, 737,
          741, 743, 747, 751, 752, 756, 761, 766, 770, 773, 774, 776, 777, 784, 786, 793, 802, 804,
          805, 810, 813, 814, 818, 821, 823, 825, 826, 830, 839, 840, 841, 844, 851, 858, 860, 863,
          872, 877, 881, 885, 887, 888, 892, 895, 897, 900, 904, 918, 925, 926, 935, 936, 952, 953,
          955, 959, 961, 962, 963, 971, 973, 974, 978, 987, 992, 998, 1000, 1002, 1003, 1006, 1009,
          1011, 1015, 1020, 1024, 1026, 1027, 1029, 1040, 1045, 1052, 1061, 1064, 1069, 1070, 1072,
          1073, 1076, 1082, 1089, 1094, 1098, 1101, 1107, 1109, 1111, 1119, 1121, 1126, 1131, 1136,
          1137, 1140, 1143, 1148, 1154, 1156, 1158, 1163, 1168, 1172, 1174, 1175, 1188, 1193, 1194,
          1195, 1210, 1212, 1220, 1221, 1222, 1225, 1228, 1230, 1231, 1232, 1242, 1246, 1255, 1261,
          1262, 1265, 1268, 1270, 1274, 1279, 1283, 1288, 1292, 1294, 1295, 1296, 1298, 1299, 1302,
          1306, 1320, 1322, 1328, 1329, 1332, 1333, 1335, 1341, 1344, 1357, 1359, 1362, 1365, 1366,
          1369, 1373, 1376, 1379, 1380, 1394, 1395, 1396, 1399, 1402, 1405, 1407, 1413, 1416, 1417,
          1422, 1431, 1432, 1433, 1436, 1440, 1442, 1443, 1444, 1446, 1447, 1454, 1469, 1471, 1480,
          1483, 1484, 1489, 1490, 1491, 1496, 1507, 1510, 1513, 1514, 1521, 1528, 1529, 1533, 1538,
          1542, 1545, 1547, 1550, 1551, 1555, 1557, 1558, 1563, 1564, 1565, 1566, 1570, 1580, 1581,
          1588, 1590, 1595, 1600, 1601, 1603, 1612, 1617, 1618, 1624, 1625, 1627, 1629, 1631, 1632,
          1637, 1644, 1655, 1662, 1664, 1668, 1672, 1675, 1676, 1681, 1690, 1691, 1692, 1698, 1699,
          1701, 1711, 1712, 1714, 1729, 1730, 1735, 1739, 1740, 1742, 1743, 1746, 1748, 1751, 1764,
          1765, 1766, 1767, 1775, 1777, 1779, 1797, 1801, 1802, 1804, 1806, 1809, 1810, 1813, 1823,
          1824, 1825, 1834, 1838, 1846, 1849, 1871, 1876, 1877, 1880, 1886, 1890, 1891, 1897, 1898,
          1899, 1912, 1913, 1915, 1923, 1925, 1931, 1935, 1936, 1940, 1949, 1952, 1957, 1958, 1960,
          1962, 1964, 1965, 1968, 1972, 1982, 1997, 1998, 1999, 2002, 2005, 2007, 2008, 2010, 2014,
          2019, 2024, 2025, 2026, 2031, 2032, 2034, 2035, 2036, 2039, 2045, 2046, 2047, 2065, 2069,
          2072, 2075, 2081, 2083, 2093, 2098, 2099, 2100, 2102, 2106, 2110, 2112, 2113, 2116, 2136,
          2137, 2139, 2142, 2145, 2150, 2153, 2158, 2167, 2173, 2179, 2180, 2183, 2184, 2193, 2199,
          2204, 2208, 2209, 2211, 2217, 2220, 2221, 2227, 2230, 2232, 2236, 2246, 2247, 2248, 2250,
          2258, 2260, 2266, 2267, 2273, 2278, 2282, 2284, 2287, 2293, 2294, 2295, 2297, 2301, 2303,
          2304, 2315, 2327, 2332, 2334, 2338, 2340, 2342, 2343, 2359, 2361, 2364, 2367, 2368, 2369,
          2371, 2378, 2380, 2384, 2394, 2401, 2404, 2405, 2409, 2412, 2416, 2421, 2426, 2431, 2433,
          2435, 2438, 2441, 2445, 2449, 2451, 2452, 2467, 2468, 2472, 2476
        });
  }

  static {
    squareRests[5] = squareRestsMod2479();
  }
  // add mod 2479 (646 square rests) -> ~ 0.003205% possible squares
  static HashSet<Integer> squareRestsMod3841() {
    return toHashSet(
        new int[] {
          0, 1, 2, 3, 4, 6, 8, 9, 12, 16, 18, 24, 25, 27, 29, 31, 32, 36, 47, 48, 49, 50, 54, 58,
          62, 64, 72, 75, 77, 81, 85, 87, 93, 94, 96, 98, 100, 108, 115, 116, 121, 124, 127, 128,
          133, 141, 144, 147, 150, 154, 162, 167, 169, 170, 173, 174, 179, 185, 186, 188, 192, 196,
          200, 209, 211, 215, 216, 223, 225, 230, 231, 232, 233, 239, 242, 243, 248, 254, 255, 256,
          261, 265, 266, 279, 282, 288, 289, 294, 299, 300, 308, 311, 317, 324, 334, 335, 338, 340,
          345, 346, 348, 353, 358, 361, 363, 370, 372, 376, 381, 384, 391, 392, 395, 397, 399, 400,
          409, 415, 418, 422, 423, 427, 430, 432, 441, 446, 449, 450, 455, 460, 461, 462, 464, 466,
          478, 484, 486, 491, 496, 501, 507, 508, 509, 510, 512, 515, 519, 522, 529, 530, 532, 533,
          537, 545, 555, 558, 564, 565, 576, 577, 578, 588, 598, 599, 600, 601, 616, 622, 623, 625,
          627, 629, 633, 634, 645, 648, 653, 668, 669, 670, 671, 675, 676, 679, 680, 690, 692, 693,
          696, 699, 706, 715, 716, 717, 722, 725, 726, 729, 731, 740, 744, 745, 749, 752, 761, 762,
          765, 767, 768, 775, 782, 783, 784, 790, 794, 795, 798, 800, 805, 809, 818, 830, 836, 837,
          841, 844, 846, 851, 853, 854, 857, 859, 860, 863, 864, 867, 877, 882, 883, 892, 897, 898,
          899, 900, 901, 910, 920, 922, 923, 924, 928, 929, 932, 933, 943, 947, 949, 951, 956, 959,
          961, 967, 968, 972, 979, 982, 989, 992, 997, 1002, 1005, 1013, 1014, 1016, 1018, 1020,
          1021, 1024, 1030, 1035, 1038, 1044, 1051, 1058, 1059, 1060, 1064, 1066, 1067, 1074, 1083,
          1087, 1089, 1090, 1099, 1110, 1116, 1117, 1128, 1129, 1130, 1135, 1139, 1143, 1152, 1154,
          1156, 1159, 1173, 1175, 1176, 1177, 1181, 1185, 1191, 1196, 1197, 1198, 1200, 1202, 1205,
          1219, 1223, 1225, 1227, 1231, 1232, 1235, 1244, 1245, 1246, 1250, 1254, 1258, 1265, 1266,
          1267, 1268, 1269, 1277, 1281, 1283, 1290, 1291, 1296, 1297, 1301, 1306, 1313, 1319, 1323,
          1336, 1337, 1338, 1340, 1342, 1343, 1347, 1350, 1352, 1357, 1358, 1360, 1361, 1363, 1365,
          1369, 1380, 1383, 1384, 1386, 1392, 1393, 1398, 1411, 1412, 1421, 1429, 1430, 1432, 1434,
          1435, 1444, 1450, 1451, 1452, 1457, 1458, 1462, 1473, 1480, 1488, 1490, 1498, 1503, 1504,
          1507, 1511, 1519, 1521, 1522, 1524, 1527, 1530, 1531, 1534, 1536, 1541, 1545, 1547, 1550,
          1553, 1557, 1559, 1564, 1565, 1566, 1567, 1568, 1580, 1587, 1588, 1590, 1591, 1596, 1599,
          1600, 1603, 1610, 1611, 1618, 1619, 1633, 1635, 1636, 1657, 1660, 1665, 1672, 1674, 1679,
          1681, 1682, 1688, 1691, 1692, 1695, 1697, 1702, 1703, 1706, 1708, 1714, 1718, 1720, 1726,
          1727, 1728, 1731, 1733, 1734, 1751, 1754, 1757, 1764, 1766, 1777, 1784, 1794, 1796, 1797,
          1798, 1800, 1802, 1803, 1807, 1817, 1820, 1840, 1841, 1843, 1844, 1846, 1848, 1849, 1853,
          1856, 1858, 1864, 1865, 1866, 1869, 1875, 1879, 1881, 1886, 1887, 1894, 1895, 1898, 1899,
          1902, 1909, 1912, 1913, 1918, 1921, 1922, 1925, 1933, 1934, 1935, 1936, 1944, 1945, 1958,
          1959, 1961, 1963, 1964, 1967, 1978, 1981, 1984, 1987, 1991, 1994, 2004, 2005, 2007, 2010,
          2013, 2025, 2026, 2028, 2032, 2033, 2036, 2037, 2040, 2042, 2048, 2051, 2053, 2060, 2065,
          2070, 2076, 2079, 2088, 2093, 2097, 2101, 2102, 2111, 2116, 2118, 2119, 2120, 2125, 2128,
          2132, 2134, 2141, 2145, 2148, 2151, 2166, 2171, 2174, 2175, 2178, 2180, 2185, 2187, 2189,
          2193, 2198, 2203, 2209, 2220, 2221, 2232, 2233, 2234, 2235, 2237, 2243, 2247, 2255, 2256,
          2258, 2260, 2267, 2270, 2278, 2279, 2283, 2285, 2286, 2293, 2295, 2301, 2303, 2304, 2308,
          2312, 2318, 2323, 2325, 2339, 2341, 2346, 2347, 2349, 2350, 2352, 2354, 2359, 2362, 2369,
          2370, 2371, 2382, 2385, 2387, 2392, 2394, 2395, 2396, 2400, 2401, 2404, 2410, 2415, 2419,
          2423, 2427, 2431, 2438, 2446, 2450, 2454, 2462, 2464, 2465, 2470, 2479, 2485, 2488, 2490,
          2492, 2500, 2507, 2508, 2509, 2511, 2513, 2516, 2519, 2523, 2530, 2532, 2533, 2534, 2536,
          2538, 2543, 2553, 2554, 2555, 2559, 2561, 2562, 2566, 2569, 2571, 2577, 2580, 2582, 2589,
          2592, 2594, 2599, 2601, 2602, 2603, 2605, 2612, 2617, 2626, 2631, 2635, 2638, 2646, 2649,
          2657, 2672, 2674, 2676, 2680, 2681, 2684, 2686, 2691, 2693, 2694, 2697, 2699, 2700, 2703,
          2704, 2714, 2716, 2720, 2722, 2726, 2730, 2737, 2738, 2749, 2753, 2760, 2761, 2766, 2768,
          2769, 2772, 2784, 2786, 2787, 2796, 2799, 2809, 2819, 2822, 2824, 2829, 2841, 2842, 2845,
          2847, 2853, 2855, 2858, 2860, 2861, 2864, 2868, 2870, 2875, 2877, 2881, 2883, 2887, 2888,
          2893, 2900, 2901, 2902, 2904, 2911, 2914, 2916, 2923, 2924, 2927, 2933, 2937, 2939, 2946,
          2947, 2953, 2960, 2967, 2969, 2971, 2976, 2980, 2983, 2991, 2993, 2996, 3006, 3008, 3013,
          3014, 3015, 3017, 3022, 3025, 3031, 3037, 3038, 3039, 3042, 3044, 3048, 3054, 3060, 3062,
          3063, 3067, 3068, 3071, 3072, 3082, 3083, 3090, 3091, 3094, 3095, 3100, 3105, 3106, 3113,
          3114, 3118, 3121, 3128, 3130, 3132, 3134, 3136, 3153, 3160, 3163, 3174, 3175, 3176, 3177,
          3180, 3182, 3187, 3192, 3197, 3198, 3200, 3201, 3205, 3206, 3209, 3215, 3220, 3221, 3222,
          3223, 3229, 3236, 3238, 3245, 3249, 3261, 3266, 3267, 3269, 3270, 3272, 3289, 3295, 3297,
          3301, 3305, 3314, 3320, 3325, 3330, 3335, 3341, 3343, 3344, 3347, 3348, 3351, 3358, 3359,
          3361, 3362, 3364, 3367, 3371, 3376, 3382, 3384, 3387, 3389, 3390, 3394, 3397, 3404, 3405,
          3406, 3412, 3416, 3417, 3427, 3428, 3429, 3433, 3436, 3439, 3440, 3452, 3454, 3456, 3462,
          3466, 3468, 3473, 3477, 3481, 3497, 3502, 3508, 3509, 3514, 3519, 3521, 3523, 3525, 3528,
          3531, 3532, 3535, 3543, 3545, 3551, 3554, 3555, 3565, 3568, 3569, 3571, 3573, 3583, 3588,
          3591, 3592, 3594, 3596, 3600, 3601, 3604, 3606, 3614, 3615, 3619, 3623, 3629, 3634, 3635,
          3637, 3640, 3657, 3659, 3661, 3669, 3675, 3680, 3681, 3682, 3683, 3686, 3688, 3692, 3693,
          3696, 3698, 3703, 3705, 3706, 3707, 3712, 3716, 3721, 3728, 3730, 3732, 3735, 3738, 3739,
          3749, 3750, 3751, 3755, 3758, 3761, 3762, 3767, 3772, 3773, 3774, 3781, 3788, 3790, 3795,
          3796, 3798, 3801, 3804, 3807, 3811, 3818, 3821, 3824, 3826, 3831, 3836
        });
  }

  static {
    squareRests[6] = squareRestsMod3841();
  }
  // add mod 3841 (1008 square rests) -> ~ 7.6E-4% possible squares
  static HashSet<Integer> squareRestsMod2623() {
    return toHashSet(
        new int[] {
          0, 1, 4, 9, 13, 14, 15, 16, 25, 36, 41, 47, 49, 52, 56, 57, 58, 60, 64, 66, 74, 81, 83,
          86, 95, 97, 100, 102, 103, 107, 109, 110, 117, 121, 122, 126, 127, 135, 138, 142, 144,
          164, 167, 169, 170, 178, 182, 183, 186, 187, 188, 195, 196, 197, 203, 208, 210, 219, 224,
          225, 228, 229, 230, 231, 232, 239, 240, 253, 256, 258, 259, 264, 269, 271, 283, 289, 293,
          296, 301, 302, 305, 310, 314, 317, 318, 324, 325, 332, 339, 341, 344, 350, 353, 354, 357,
          361, 365, 367, 369, 375, 379, 380, 382, 385, 388, 391, 393, 400, 402, 408, 411, 412, 418,
          422, 423, 427, 428, 430, 431, 436, 439, 440, 441, 443, 446, 447, 454, 461, 466, 468, 473,
          474, 479, 483, 484, 487, 488, 489, 497, 504, 508, 513, 522, 527, 529, 530, 533, 537, 540,
          552, 554, 563, 565, 568, 569, 574, 576, 583, 590, 594, 595, 597, 606, 611, 613, 615, 619,
          623, 625, 626, 637, 646, 649, 651, 655, 656, 658, 659, 662, 666, 668, 670, 676, 680, 683,
          685, 686, 698, 705, 712, 713, 719, 723, 728, 729, 731, 732, 735, 737, 741, 744, 745, 746,
          747, 748, 752, 754, 766, 771, 774, 778, 780, 784, 788, 789, 790, 797, 798, 805, 809, 812,
          815, 818, 827, 832, 834, 838, 840, 841, 842, 853, 855, 857, 858, 866, 869, 870, 873, 874,
          876, 881, 895, 896, 900, 901, 903, 912, 914, 916, 918, 919, 920, 924, 927, 928, 934, 956,
          957, 960, 961, 962, 963, 967, 971, 977, 981, 989, 990, 995, 998, 1003, 1010, 1012, 1024,
          1025, 1032, 1033, 1036, 1038, 1041, 1042, 1046, 1049, 1053, 1056, 1057, 1073, 1076, 1079,
          1084, 1085, 1086, 1089, 1098, 1099, 1110, 1111, 1113, 1118, 1132, 1134, 1139, 1143, 1154,
          1156, 1158, 1159, 1162, 1171, 1172, 1174, 1175, 1178, 1184, 1186, 1201, 1204, 1205, 1208,
          1215, 1217, 1219, 1220, 1221, 1225, 1229, 1235, 1239, 1240, 1242, 1245, 1247, 1256, 1261,
          1262, 1268, 1272, 1278, 1282, 1285, 1290, 1294, 1296, 1300, 1301, 1303, 1306, 1315, 1326,
          1328, 1330, 1333, 1337, 1339, 1342, 1343, 1346, 1347, 1354, 1356, 1357, 1358, 1364, 1369,
          1376, 1387, 1389, 1390, 1391, 1399, 1400, 1407, 1412, 1416, 1417, 1419, 1423, 1425, 1428,
          1430, 1442, 1444, 1450, 1455, 1459, 1460, 1463, 1468, 1473, 1476, 1477, 1478, 1479, 1483,
          1486, 1498, 1500, 1503, 1505, 1506, 1509, 1511, 1516, 1520, 1521, 1522, 1526, 1528, 1529,
          1530, 1540, 1541, 1545, 1552, 1559, 1561, 1564, 1571, 1572, 1573, 1583, 1586, 1589, 1591,
          1595, 1600, 1601, 1602, 1605, 1606, 1608, 1622, 1627, 1631, 1632, 1634, 1635, 1638, 1643,
          1644, 1647, 1648, 1650, 1651, 1659, 1669, 1672, 1674, 1681, 1683, 1686, 1688, 1692, 1693,
          1694, 1708, 1712, 1713, 1717, 1720, 1721, 1724, 1730, 1733, 1735, 1744, 1755, 1756, 1760,
          1764, 1769, 1772, 1773, 1774, 1778, 1784, 1788, 1794, 1803, 1810, 1815, 1816, 1817, 1821,
          1827, 1829, 1830, 1831, 1842, 1844, 1846, 1849, 1850, 1855, 1864, 1866, 1872, 1887, 1890,
          1892, 1896, 1903, 1905, 1906, 1907, 1913, 1916, 1927, 1930, 1932, 1933, 1936, 1939, 1948,
          1949, 1951, 1952, 1956, 1966, 1971, 1979, 1988, 1991, 1993, 1994, 1999, 2001, 2009, 2013,
          2014, 2016, 2018, 2022, 2025, 2027, 2032, 2035, 2038, 2052, 2059, 2061, 2062, 2065, 2070,
          2073, 2074, 2075, 2077, 2078, 2079, 2087, 2088, 2089, 2099, 2108, 2113, 2116, 2120, 2121,
          2122, 2123, 2130, 2131, 2132, 2138, 2147, 2148, 2150, 2151, 2154, 2160, 2171, 2174, 2181,
          2191, 2193, 2197, 2199, 2208, 2209, 2210, 2216, 2218, 2237, 2242, 2245, 2252, 2253, 2257,
          2260, 2261, 2271, 2272, 2276, 2277, 2279, 2293, 2296, 2302, 2303, 2304, 2314, 2315, 2317,
          2319, 2322, 2323, 2331, 2332, 2333, 2337, 2338, 2343, 2345, 2357, 2360, 2363, 2365, 2366,
          2374, 2375, 2376, 2378, 2379, 2380, 2382, 2388, 2401, 2406, 2418, 2421, 2424, 2425, 2431,
          2439, 2443, 2444, 2449, 2452, 2455, 2460, 2462, 2465, 2467, 2474, 2476, 2482, 2486, 2487,
          2489, 2492, 2498, 2500, 2504, 2505, 2510, 2515, 2517, 2535, 2537, 2543, 2546, 2547, 2548,
          2550, 2553, 2558, 2561, 2562, 2575, 2577, 2578, 2581, 2584, 2589, 2596, 2601, 2603, 2604,
          2611, 2618, 2620
        });
  }

  static {
    squareRests[7] = squareRestsMod2623();
  }
  // add mod 2623 (682 square rests) -> ~ 1.6499999E-4% possible squares
  static HashSet<Integer> squareRestsMod3161() {
    return toHashSet(
        new int[] {
          0, 1, 4, 5, 7, 9, 16, 20, 22, 25, 28, 29, 34, 35, 36, 38, 45, 49, 63, 64, 71, 74, 78, 80,
          81, 82, 83, 87, 88, 93, 94, 100, 109, 110, 112, 116, 121, 125, 129, 136, 138, 140, 144,
          145, 152, 154, 158, 169, 170, 173, 175, 180, 183, 187, 190, 196, 197, 198, 202, 203, 209,
          219, 223, 225, 227, 233, 238, 239, 245, 252, 254, 256, 261, 266, 267, 281, 284, 289, 291,
          296, 299, 306, 312, 315, 318, 320, 323, 324, 326, 328, 332, 339, 342, 343, 347, 348, 349,
          352, 353, 354, 355, 361, 370, 372, 373, 376, 390, 393, 400, 401, 402, 405, 407, 410, 411,
          415, 429, 431, 435, 436, 439, 440, 441, 448, 451, 457, 458, 463, 464, 465, 470, 471, 484,
          497, 499, 500, 502, 509, 516, 517, 518, 523, 529, 538, 542, 544, 545, 546, 550, 552, 557,
          560, 567, 571, 573, 574, 576, 579, 580, 581, 593, 605, 608, 609, 616, 618, 625, 629, 632,
          633, 634, 638, 639, 642, 645, 647, 651, 654, 658, 661, 663, 666, 674, 676, 680, 683, 689,
          690, 692, 697, 700, 702, 703, 718, 720, 725, 729, 732, 734, 738, 741, 747, 748, 754, 758,
          759, 760, 763, 767, 770, 778, 779, 783, 784, 788, 789, 790, 792, 799, 806, 808, 811, 812,
          834, 836, 837, 841, 845, 846, 847, 850, 857, 863, 865, 869, 871, 875, 876, 877, 879, 892,
          893, 894, 898, 899, 900, 903, 906, 908, 915, 921, 932, 933, 935, 950, 952, 953, 956, 961,
          966, 977, 980, 981, 982, 985, 986, 990, 993, 1002, 1006, 1008, 1009, 1010, 1015, 1016,
          1019, 1024, 1044, 1045, 1064, 1068, 1069, 1074, 1078, 1086, 1089, 1093, 1095, 1097, 1102,
          1106, 1111, 1115, 1118, 1124, 1125, 1126, 1135, 1136, 1138, 1151, 1153, 1154, 1156, 1161,
          1164, 1165, 1173, 1183, 1184, 1190, 1194, 1195, 1196, 1198, 1202, 1211, 1214, 1219, 1224,
          1225, 1227, 1234, 1242, 1247, 1248, 1260, 1263, 1270, 1272, 1277, 1280, 1281, 1282, 1283,
          1292, 1296, 1299, 1301, 1304, 1305, 1309, 1311, 1312, 1328, 1329, 1330, 1333, 1334, 1335,
          1339, 1343, 1354, 1356, 1357, 1368, 1369, 1372, 1379, 1383, 1386, 1388, 1391, 1392, 1396,
          1397, 1401, 1405, 1408, 1412, 1414, 1416, 1417, 1420, 1421, 1422, 1426, 1437, 1443, 1444,
          1445, 1446, 1451, 1455, 1463, 1466, 1478, 1480, 1483, 1488, 1492, 1495, 1499, 1501, 1504,
          1514, 1517, 1521, 1530, 1531, 1533, 1538, 1541, 1542, 1546, 1553, 1557, 1560, 1561, 1562,
          1571, 1572, 1575, 1586, 1589, 1590, 1599, 1600, 1601, 1604, 1608, 1615, 1619, 1620, 1623,
          1628, 1630, 1631, 1640, 1644, 1647, 1657, 1660, 1662, 1666, 1669, 1673, 1678, 1681, 1683,
          1695, 1698, 1706, 1710, 1715, 1716, 1717, 1718, 1724, 1735, 1739, 1740, 1741, 1744, 1745,
          1747, 1749, 1753, 1756, 1760, 1764, 1765, 1769, 1770, 1773, 1775, 1778, 1782, 1789, 1792,
          1793, 1804, 1805, 1807, 1818, 1822, 1826, 1827, 1828, 1831, 1832, 1833, 1849, 1850, 1852,
          1856, 1857, 1860, 1862, 1865, 1869, 1878, 1879, 1880, 1881, 1884, 1889, 1891, 1898, 1901,
          1913, 1914, 1919, 1927, 1934, 1936, 1937, 1942, 1947, 1950, 1959, 1963, 1965, 1966, 1967,
          1971, 1977, 1978, 1988, 1996, 1997, 2000, 2005, 2007, 2008, 2010, 2023, 2025, 2026, 2035,
          2036, 2037, 2043, 2046, 2050, 2055, 2059, 2064, 2066, 2068, 2072, 2075, 2083, 2087, 2092,
          2093, 2097, 2116, 2117, 2137, 2142, 2145, 2146, 2151, 2152, 2153, 2155, 2159, 2168, 2171,
          2175, 2176, 2179, 2180, 2181, 2184, 2195, 2200, 2205, 2208, 2209, 2211, 2226, 2228, 2229,
          2240, 2246, 2253, 2255, 2258, 2261, 2262, 2263, 2267, 2268, 2269, 2282, 2284, 2285, 2286,
          2290, 2292, 2296, 2298, 2304, 2311, 2314, 2315, 2316, 2320, 2324, 2325, 2327, 2349, 2350,
          2353, 2355, 2362, 2369, 2371, 2372, 2373, 2377, 2378, 2382, 2383, 2391, 2394, 2398, 2401,
          2402, 2403, 2407, 2413, 2414, 2420, 2423, 2427, 2429, 2432, 2436, 2441, 2443, 2458, 2459,
          2461, 2464, 2469, 2471, 2472, 2478, 2481, 2485, 2487, 2495, 2498, 2500, 2503, 2507, 2510,
          2514, 2516, 2519, 2522, 2523, 2527, 2528, 2529, 2532, 2536, 2543, 2545, 2552, 2553, 2556,
          2568, 2580, 2581, 2582, 2585, 2587, 2588, 2590, 2594, 2601, 2604, 2609, 2611, 2615, 2616,
          2617, 2619, 2623, 2632, 2638, 2643, 2644, 2645, 2652, 2659, 2661, 2662, 2664, 2677, 2690,
          2691, 2696, 2697, 2698, 2703, 2704, 2710, 2713, 2720, 2721, 2722, 2725, 2726, 2730, 2732,
          2746, 2750, 2751, 2754, 2756, 2759, 2760, 2761, 2768, 2771, 2785, 2788, 2789, 2791, 2800,
          2806, 2807, 2808, 2809, 2812, 2813, 2814, 2818, 2819, 2822, 2829, 2833, 2835, 2837, 2838,
          2841, 2843, 2846, 2849, 2855, 2862, 2865, 2870, 2872, 2877, 2880, 2894, 2895, 2900, 2905,
          2907, 2909, 2916, 2922, 2923, 2928, 2934, 2936, 2938, 2942, 2952, 2958, 2959, 2963, 2964,
          2965, 2971, 2974, 2978, 2981, 2986, 2988, 2991, 2992, 3003, 3007, 3009, 3016, 3017, 3021,
          3023, 3025, 3032, 3036, 3040, 3045, 3049, 3051, 3052, 3061, 3067, 3068, 3073, 3074, 3078,
          3079, 3080, 3081, 3083, 3087, 3090, 3097, 3098, 3112, 3116, 3123, 3125, 3126, 3127, 3132,
          3133, 3136, 3139, 3141, 3145, 3152, 3154, 3156, 3157, 3160
        });
  }

  static {
    squareRests[8] = squareRestsMod3161();
  }
  // add mod 3161 (825 square rests) -> ~ 2.5E-5% possible squares
  static HashSet<Integer> squareRestsMod97() {
    return toHashSet(
        new int[] {
          0, 1, 2, 3, 4, 6, 8, 9, 11, 12, 16, 18, 22, 24, 25, 27, 31, 32, 33, 35, 36, 43, 44, 47,
          48, 49, 50, 53, 54, 61, 62, 64, 65, 66, 70, 72, 73, 75, 79, 81, 85, 86, 88, 89, 91, 93,
          94, 95, 96
        });
  }

  static {
    squareRests[9] = squareRestsMod97();
  }
  // add mod 97 (49 square rests) -> ~ 0.0% possible squares

  private static HashSet<Integer> toHashSet(int[] arr) {
    HashSet<Integer> hashset =
        new HashSet<Integer>(
            arr.length << 1); // sparse hashset -> less entries per list -> faster lookup!
    for (int i : arr) hashset.add(i);
    return hashset;
  }

  /**
   * Return sqrt(n) if n is a square of an integer, null otherwise.
   *
   * @param n
   * @return sqrt(n) or null
   */
  public static BigInteger exactSqrt(BigInteger n) {
    if (n.equals(I_0)) return I_0;
    // check contained power of 2
    int lsb = n.getLowestSetBit();
    if ((lsb & 1) == 1) return null; // n has an odd power of 2 -> no square

    int bits = n.bitLength();
    if (bits > 63) {
      // Improvement over "fastMod" (modulus to a power of 2) test, thanks to Graeme Willoughby:
      // Write some number as (2^m)k ,where k is the odd number left after factoring out all the
      // multiples of 2.
      // The square of this number is (2 ^(2m))k^2. Now we know that the square of an odd number is
      // of the form 8N+1.
      // So the general form of any square number is (2^(2m))(8N+1) , with m,N=0,1,2...
      // -> If n is square, then it must have the binary representation n = <something>001<2m 0's>
      // -> If n is square, then it must satisfy (n/2^lsb) % 8 == 1.
      // This is a very fast test, and 83.3% of input values are determined to be non-square by this
      // method,
      // which is everything you can get from examining bit patterns.
      if ((n.shiftRight(lsb).intValue() & 7) != 1)
        return null; // we need only the lowest three bits

      // For N <= 106 bit SqrtInt is faster than that the mod-tests
      if (bits > 106) {
        // test list of subsequent mods
        for (int i = 1; i < MOD_COUNT; i++) {
          int nextMod = mods[i];
          int rest = n.mod(BigInteger.valueOf(nextMod)).intValue();
          if (!squareRests[i].contains(rest)) return null; // rest = n % nextMod is no square rest
        }
      }
    }

    // Newton sqrt() required
    BigInteger[] isqrt = SqrtInt.iSqrt(n);
    return isqrt[0].equals(isqrt[1]) ? isqrt[0] : null;
  }
}
