/*
 * java-math-library is a Java library focused on number theory, but not necessarily limited to it. It is based on the PSIQS 4.0 factoring project.
 * Copyright (C) 2018 Tilman Neumann (www.tilman-neumann.de)
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program;
 * if not, see <http://www.gnu.org/licenses/>.
 */
package de.tilman_neumann.jml.factor.psiqs;

import java.math.BigInteger;

import de.tilman_neumann.jml.factor.base.congruence.CongruenceCollectorParallel;
import de.tilman_neumann.jml.factor.base.matrixSolver.MatrixSolver;
import de.tilman_neumann.jml.factor.siqs.data.BaseArrays;
import de.tilman_neumann.jml.factor.siqs.poly.AParamGenerator;
import de.tilman_neumann.jml.factor.siqs.poly.AParamGenerator01;
import de.tilman_neumann.jml.factor.siqs.powers.PowerFinder;
import de.tilman_neumann.jml.factor.siqs.sieve.SieveParams;

/**
 * Multi-threaded SIQS using the single block hybrid sieve.
 *
 * @author Tilman Neumann
 */
public class PSIQS_SBH_U extends PSIQSBase {
  private int blockSize;

  /**
   * Standard constructor.
   *
   * @param Cmult multiplier for prime base size
   * @param Mmult multiplier for sieve array size
   * @param wantedQCount hypercube dimension (null for automatic selection)
   * @param maxQRestExponent A Q with unfactored rest QRest is considered smooth if QRest <=
   *     N^maxQRestExponent. Good values are 0.16..0.19; null means that it is determined
   *     automatically.
   * @param blockSize wanted sieve block size in byte
   * @param numberOfThreads
   * @param powerFinder algorithm to add powers to the primes used for sieving
   * @param matrixSolver solver for smooth congruences matrix
   */
  public PSIQS_SBH_U(
      float Cmult,
      float Mmult,
      Integer wantedQCount,
      Float maxQRestExponent,
      int blockSize,
      int numberOfThreads,
      PowerFinder powerFinder,
      MatrixSolver matrixSolver) {

    super(
        Cmult,
        Mmult,
        maxQRestExponent,
        numberOfThreads,
        null,
        powerFinder,
        matrixSolver,
        new AParamGenerator01(wantedQCount));
    this.blockSize = blockSize;
  }

  @Override
  public String getName() {
    String maxQRestExponentStr = "maxQRestExponent=" + String.format("%.3f", maxQRestExponent);
    return "PSIQS_SBH_U(Cmult="
        + Cmult
        + ", Mmult="
        + Mmult
        + ", qCount="
        + apg.getQCount()
        + ", "
        + maxQRestExponentStr
        + ", blockSize="
        + blockSize
        + ", "
        + powerFinder.getName()
        + ", "
        + matrixSolver.getName()
        + ", "
        + numberOfThreads
        + " threads)";
  }

  protected PSIQSThreadBase createThread(
      int k,
      BigInteger N,
      BigInteger kN,
      int d,
      SieveParams sieveParams,
      BaseArrays baseArrays,
      AParamGenerator apg,
      CongruenceCollectorParallel cc,
      int threadIndex) {

    return new PSIQSThread_SBH_U(
        k, N, kN, d, sieveParams, baseArrays, blockSize, apg, cc, threadIndex);
  }
}
