/*
 * java-math-library is a Java library focused on number theory, but not necessarily limited to it. It is based on the PSIQS 4.0 factoring project.
 * Copyright (C) 2018 Tilman Neumann (www.tilman-neumann.de)
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program;
 * if not, see <http://www.gnu.org/licenses/>.
 */
package de.tilman_neumann.jml.factor.lehman;

import java.math.BigInteger;

import org.apache.log4j.Logger;

import de.tilman_neumann.jml.factor.FactorAlgorithm;
import de.tilman_neumann.jml.gcd.Gcd63;
import de.tilman_neumann.util.ConfigUtil;
import de.tilman_neumann.jml.factor.tdiv.TDiv63Inverse;

/**
 * An attempt to reproduce Warren D. Smith's Lehman implementation used by YaFu, using fast inverse
 * trial division. This attempt may not match the original accurately.
 *
 * @author Tilman Neumann
 */
public class Lehman_Smith extends FactorAlgorithm {
  private static final Logger LOG = Logger.getLogger(Lehman_Smith.class);

  /** This is a constant that is below 1 for rounding up double values to long. */
  private static final double ROUND_UP_DOUBLE = 0.9999999665;

  private long fourN;
  private double sqrt4N;
  private boolean doTDivFirst;
  private final Gcd63 gcdEngine = new Gcd63();
  private final TDiv63Inverse tdiv = new TDiv63Inverse(1 << 21);

  /**
   * Full constructor.
   *
   * @param doTDivFirst If true then trial division is done before the Lehman loop. This is
   *     recommended if arguments N are known to have factors < cbrt(N) frequently.
   */
  public Lehman_Smith(boolean doTDivFirst) {
    this.doTDivFirst = doTDivFirst;
  }

  @Override
  public String getName() {
    return "Lehman_Smith(" + doTDivFirst + ")";
  }

  @Override
  public BigInteger findSingleFactor(BigInteger N) {
    return BigInteger.valueOf(findSingleFactor(N.longValue()));
  }

  public long findSingleFactor(long N) {
    final int cbrt = (int) Math.cbrt(N);

    // do trial division before Lehman loop ?
    long factor;
    tdiv.setTestLimit(cbrt);
    if (doTDivFirst && (factor = tdiv.findSingleFactor(N)) > 1) return factor;

    fourN = N << 2;
    sqrt4N = Math.sqrt(fourN);

    final int kLimit = cbrt;
    final double sixthRootTerm =
        0.25 * Math.pow(N, 1 / 6.0); // double precision is required for stability
    for (int k = 1; k <= kLimit; k++) {
      double sqrtK = Math.sqrt(k);
      final double sqrt4kN = sqrt4N * sqrtK;
      // only use long values
      final long aStart = (long) (sqrt4kN + ROUND_UP_DOUBLE); // much faster than ceil() !
      long aLimit = (long) (sqrt4kN + sixthRootTerm / sqrtK);
      long aStep;
      if ((k & 1) == 0) {
        // k even -> make sure aLimit is odd
        aLimit |= 1L;
        aStep = 2;
      } else {
        final long kPlusN = k + N;
        if ((kPlusN & 3) == 0) {
          aStep = 8;
          aLimit += ((kPlusN - aLimit) & 7);
        } else {
          aStep = 4;
          aLimit += ((kPlusN - aLimit) & 3);
        }
      }

      // processing the a-loop top-down is faster than bottom-up
      final long fourkN = k * fourN;
      for (long a = aLimit; a >= aStart; a -= aStep) {
        final long test = a * a - fourkN;
        // Here test<0 is possible because of double to long cast errors in the 'a'-computation.
        // But then b = Math.sqrt(test) gives 0 (sic!) => 0*0 != test => no errors.
        final long b = (long) Math.sqrt(test);
        if (b * b == test) {
          return gcdEngine.gcd(a + b, N);
        }
      }
    }

    // do trial division after Lehman loop ?
    if (!doTDivFirst && (factor = tdiv.findSingleFactor(N)) > 1) return factor;

    // If sqrt(4kN) is very near to an exact integer then the fast ceil() in the
    // 'aStart'-computation
    // may have failed. Then we need a "correction loop":
    final int kTwoA = (((cbrt >> 6) + 6) / 6) * 6;
    for (int k = kTwoA + 1; k <= kLimit; k++) {
      long a = (long) (sqrt4N * Math.sqrt(k) + ROUND_UP_DOUBLE) - 1;
      long test = a * a - k * fourN;
      long b = (long) Math.sqrt(test);
      if (b * b == test) {
        return gcdEngine.gcd(a + b, N);
      }
    }

    return 0; // fail
  }

  /**
   * Test.
   *
   * @param args ignored
   */
  public static void main(String[] args) {
    ConfigUtil.initProject();

    // These test number were too hard for previous versions:
    long[] testNumbers =
        new long[] {
          5640012124823L,
          7336014366011L,
          19699548984827L,
          52199161732031L,
          73891306919159L,
          112454098638991L,
          32427229648727L,
          87008511088033L,
          92295512906873L,
          338719143795073L,
          346425669865991L,
          1058244082458461L,
          1773019201473077L,
          6150742154616377L,
          44843649362329L,
          67954151927287L,
          134170056884573L,
          198589283218993L,
          737091621253457L,
          1112268234497993L,
          2986396307326613L,
          26275638086419L,
          62246008190941L,
          209195243701823L,
          290236682491211L,
          485069046631849L,
          1239671094365611L,
          2815471543494793L,
          5682546780292609L,
        };

    Lehman_Smith lehman = new Lehman_Smith(true);
    for (long N : testNumbers) {
      long factor = lehman.findSingleFactor(N);
      LOG.info("N=" + N + " has factor " + factor);
    }
  }
}
