/*
 * java-math-library is a Java library focused on number theory, but not necessarily limited to it. It is based on the PSIQS 4.0 factoring project.
 * Copyright (C) 2018 Tilman Neumann (www.tilman-neumann.de)
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program;
 * if not, see <http://www.gnu.org/licenses/>.
 */
package de.tilman_neumann.jml.factor.lehman;

import java.math.BigInteger;

import de.tilman_neumann.jml.gcd.Gcd63;
import de.tilman_neumann.jml.factor.FactorAlgorithm;
import de.tilman_neumann.jml.factor.tdiv.TDiv63Inverse;

/**
 * Simple implementation of Lehmans factor algorithm, following
 * https://programmingpraxis.com/2017/08/22/lehmans-factoring-algorithm/, using fast inverse trial
 * division.
 *
 * <p>This implementation is pretty slow, but useful for illustrating the basic algorithm. It may
 * fail for N>=47 bit where 4*k*N produces a long-overflow.
 *
 * @author Tilman Neumann
 */
public class Lehman_Simple extends FactorAlgorithm {
  private final Gcd63 gcdEngine = new Gcd63();

  private boolean doTDivFirst;

  private static final TDiv63Inverse tdiv = new TDiv63Inverse(1 << 21);

  /**
   * Full constructor.
   *
   * @param doTDivFirst If true then trial division is done before the Lehman loop. This is
   *     recommended if arguments N are known to have factors < cbrt(N) frequently.
   */
  public Lehman_Simple(boolean doTDivFirst) {
    this.doTDivFirst = doTDivFirst;
  }

  @Override
  public String getName() {
    return "Lehman_Simple(" + doTDivFirst + ")";
  }

  @Override
  public BigInteger findSingleFactor(BigInteger N) {
    return BigInteger.valueOf(findSingleFactor(N.longValue()));
  }

  public long findSingleFactor(long N) {
    int cbrt = (int) Math.ceil(Math.cbrt(N));

    // do trial division before Lehman loop ?
    long factor;
    tdiv.setTestLimit(cbrt);
    if (doTDivFirst && (factor = tdiv.findSingleFactor(N)) > 1) return factor;

    // Lehman loop
    double sixthRoot = Math.pow(N, 1 / 6.0); // double precision is required for stability
    for (int k = 1; k <= cbrt; k++) {
      long fourKN = k * N << 2; // long overflow possible for N>=47 bit
      double fourSqrtK = Math.sqrt(k << 4);
      long sqrt4kN = (long) Math.ceil(Math.sqrt(fourKN)); // ceil() is required for stability
      long limit = (long) (sqrt4kN + sixthRoot / fourSqrtK);
      for (long a = sqrt4kN; a <= limit; a++) {
        long test = a * a - fourKN;
        long b = (long) Math.sqrt(test);
        if (b * b == test) {
          long gcd = gcdEngine.gcd(a + b, N);
          if (gcd > 1 && gcd < N) return gcd;
        }
      }
    }

    // do trial division after Lehman loop ?
    if (!doTDivFirst && (factor = tdiv.findSingleFactor(N)) > 1) return factor;

    return 0; // Fail
  }
}
