package org.matheclipse.core.builtin;

import static org.matheclipse.core.expression.F.ArcCos;
import static org.matheclipse.core.expression.F.C0;
import static org.matheclipse.core.expression.F.C1;
import static org.matheclipse.core.expression.F.C1D2;
import static org.matheclipse.core.expression.F.C4;
import static org.matheclipse.core.expression.F.CN1;
import static org.matheclipse.core.expression.F.CN1D2;
import static org.matheclipse.core.expression.F.CN2;
import static org.matheclipse.core.expression.F.Divide;
import static org.matheclipse.core.expression.F.Dot;
import static org.matheclipse.core.expression.F.Function;
import static org.matheclipse.core.expression.F.List;
import static org.matheclipse.core.expression.F.Map;
import static org.matheclipse.core.expression.F.MapThread;
import static org.matheclipse.core.expression.F.Most;
import static org.matheclipse.core.expression.F.Negate;
import static org.matheclipse.core.expression.F.Norm;
import static org.matheclipse.core.expression.F.Plus;
import static org.matheclipse.core.expression.F.Power;
import static org.matheclipse.core.expression.F.Prepend;
import static org.matheclipse.core.expression.F.ReplaceAll;
import static org.matheclipse.core.expression.F.Rule;
import static org.matheclipse.core.expression.F.Slot1;
import static org.matheclipse.core.expression.F.Sqr;
import static org.matheclipse.core.expression.F.Sqrt;
import static org.matheclipse.core.expression.F.Subtract;
import static org.matheclipse.core.expression.F.Times;

import java.util.ArrayList;
import java.util.List;

import org.hipparchus.exception.MathIllegalArgumentException;
import org.hipparchus.exception.MathRuntimeException;
import org.hipparchus.linear.BlockFieldMatrix;
import org.hipparchus.linear.DecompositionSolver;
import org.hipparchus.linear.EigenDecomposition;
import org.hipparchus.linear.FieldDecompositionSolver;
import org.hipparchus.linear.FieldLUDecomposition;
import org.hipparchus.linear.FieldMatrix;
import org.hipparchus.linear.FieldVector;
import org.hipparchus.linear.MatrixUtils;
import org.hipparchus.linear.RealMatrix;
import org.hipparchus.linear.RealVector;
import org.hipparchus.linear.RiccatiEquationSolver;
import org.hipparchus.linear.RiccatiEquationSolverImpl;
import org.matheclipse.core.convert.Convert;
import org.matheclipse.core.eval.EvalAttributes;
import org.matheclipse.core.eval.EvalEngine;
import org.matheclipse.core.eval.exception.IterationLimitExceeded;
import org.matheclipse.core.eval.exception.LimitException;
import org.matheclipse.core.eval.exception.Validate;
import org.matheclipse.core.eval.exception.ValidateException;
import org.matheclipse.core.eval.interfaces.AbstractEvaluator;
import org.matheclipse.core.eval.interfaces.AbstractFunctionEvaluator;
import org.matheclipse.core.eval.interfaces.AbstractMatrix1Expr;
import org.matheclipse.core.eval.interfaces.AbstractMatrix1Matrix;
import org.matheclipse.core.eval.interfaces.AbstractNonOrderlessArgMultiple;
import org.matheclipse.core.eval.util.IndexFunctionDiagonal;
import org.matheclipse.core.eval.util.IndexTableGenerator;
import org.matheclipse.core.expression.ASTRealMatrix;
import org.matheclipse.core.expression.ASTRealVector;
import org.matheclipse.core.expression.ExprField;
import org.matheclipse.core.expression.F;
import org.matheclipse.core.expression.S;
import org.matheclipse.core.generic.Comparators;
import org.matheclipse.core.interfaces.IAST;
import org.matheclipse.core.interfaces.IASTAppendable;
import org.matheclipse.core.interfaces.IASTMutable;
import org.matheclipse.core.interfaces.IBuiltInSymbol;
import org.matheclipse.core.interfaces.IEvalStepListener;
import org.matheclipse.core.interfaces.IExpr;
import org.matheclipse.core.interfaces.INumber;
import org.matheclipse.core.interfaces.INumericArray;
import org.matheclipse.core.interfaces.ISparseArray;
import org.matheclipse.core.interfaces.ISymbol;
import org.matheclipse.parser.client.FEConfig;

public final class LinearAlgebra {

  /**
   * See <a href="https://pangin.pro/posts/computation-in-static-initializer">Beware of computation
   * in static initializer</a>
   */
  private static class Initializer {

    private static void init() {
      S.ArrayDepth.setEvaluator(new ArrayDepth());
      S.CharacteristicPolynomial.setEvaluator(new CharacteristicPolynomial());
      S.CholeskyDecomposition.setEvaluator(new CholeskyDecomposition());
      S.ConjugateTranspose.setEvaluator(new ConjugateTranspose());
      S.Cross.setEvaluator(new Cross());
      S.DesignMatrix.setEvaluator(new DesignMatrix());
      S.Det.setEvaluator(new Det());
      S.Diagonal.setEvaluator(new Diagonal());
      S.DiagonalMatrix.setEvaluator(new DiagonalMatrix());
      S.Dimensions.setEvaluator(new Dimensions());
      S.Dot.setEvaluator(new Dot());
      S.Eigenvalues.setEvaluator(new Eigenvalues());
      S.Eigenvectors.setEvaluator(new Eigenvectors());
      S.FourierMatrix.setEvaluator(new FourierMatrix());
      S.FromPolarCoordinates.setEvaluator(new FromPolarCoordinates());
      S.HilbertMatrix.setEvaluator(new HilbertMatrix());
      S.IdentityMatrix.setEvaluator(new IdentityMatrix());
      S.Inner.setEvaluator(new Inner());
      S.Inverse.setEvaluator(new Inverse());
      S.JacobiMatrix.setEvaluator(new JacobiMatrix());
      S.LeastSquares.setEvaluator(new LeastSquares());
      S.LinearSolve.setEvaluator(new LinearSolve());
      S.LowerTriangularize.setEvaluator(new LowerTriangularize());
      S.LUDecomposition.setEvaluator(new LUDecomposition());
      S.MatrixMinimalPolynomial.setEvaluator(new MatrixMinimalPolynomial());
      S.MatrixExp.setEvaluator(new MatrixExp());
      S.MatrixPower.setEvaluator(new MatrixPower());
      S.MatrixRank.setEvaluator(new MatrixRank());
      S.Norm.setEvaluator(new Norm());
      S.Normalize.setEvaluator(new Normalize());
      S.NullSpace.setEvaluator(new NullSpace());
      S.Orthogonalize.setEvaluator(new Orthogonalize());
      S.PseudoInverse.setEvaluator(PseudoInverse.CONST);
      S.Projection.setEvaluator(new Projection());
      S.QRDecomposition.setEvaluator(new QRDecomposition());
      S.RiccatiSolve.setEvaluator(new RiccatiSolve());
      S.RowReduce.setEvaluator(new RowReduce());
      S.SingularValueDecomposition.setEvaluator(new SingularValueDecomposition());
      S.ToeplitzMatrix.setEvaluator(new ToeplitzMatrix());
      S.ToPolarCoordinates.setEvaluator(new ToPolarCoordinates());
      S.Tr.setEvaluator(new Tr());
      S.Transpose.setEvaluator(new Transpose());
      S.UpperTriangularize.setEvaluator(new UpperTriangularize());
      S.UnitVector.setEvaluator(new UnitVector());
      S.VandermondeMatrix.setEvaluator(new VandermondeMatrix());
      S.VectorAngle.setEvaluator(new VectorAngle());
    }
  }

  /**
   * Matrix class that wraps a <code>FieldMatrix&lt;T&gt;</code> matrix, which is transformed to
   * reduced row echelon format.
   *
   * <p>See: <a href="http://en.wikipedia.org/wiki/Row_echelon_form">Wikipedia - Row echelon
   * form</a>.
   *
   * <p>The code was adapted from: <a
   * href="http://rosettacode.org/wiki/Reduced_row_echelon_form#Java">Rosetta Code - Reduced row
   * echelon form</a>
   */
  private static final class FieldReducedRowEchelonForm {
    /** Wrapper for a row and column index. */
    private static class RowColIndex {
      int row;
      int col;

      RowColIndex(int r, int c) {
        row = r;
        col = c;
      }

      @Override
      public String toString() {
        return "(" + row + ", " + col + ")";
      }
    }

    private final FieldMatrix<IExpr> originalMatrix;
    private final FieldMatrix<IExpr> rowReducedMatrix;
    private FieldMatrix<IExpr> nullSpaceCache;
    private int matrixRankCache;

    /** Number of rows. */
    private final int numRows;

    /** Number of columns. */
    private final int numCols;

    /**
     * Constructor which creates row reduced echelon matrix from the given <code>
     * FieldMatrix&lt;T&gt;</code> matrix.
     *
     * @param matrix matrix which will be transformed to a row reduced echelon matrix.
     * @see #rowReduce()
     */
    public FieldReducedRowEchelonForm(FieldMatrix<IExpr> matrix) {
      this.originalMatrix = matrix;
      this.rowReducedMatrix = matrix.copy();
      this.numRows = matrix.getRowDimension();
      this.numCols = matrix.getColumnDimension();
      this.matrixRankCache = -1;
      this.nullSpaceCache = null;
      rowReduce();
    }

    /**
     * Test if <code>expr</code> equals the zero element.
     *
     * @param expr
     * @return
     */
    protected boolean isZero(IExpr expr) {
      return expr.isZero();
    }

    /**
     * Test if <code>expr</code> equals the one element.
     *
     * @param expr
     * @return
     */
    protected boolean isOne(IExpr expr) {
      return expr.isOne();
    }

    private RowColIndex findPivot(RowColIndex a) {
      int first_row = a.row;
      RowColIndex pivot = new RowColIndex(a.row, a.col);
      RowColIndex current = new RowColIndex(a.row, a.col);

      for (int i = a.row; i < (numRows - first_row); i++) {
        current.row = i;
        if (isOne(rowReducedMatrix.getEntry(current.row, current.col))) {
          swapRow(current, a);
        }
      }

      current.row = a.row;
      for (int i = current.row; i < (numRows - first_row); i++) {
        current.row = i;
        if (!isZero(rowReducedMatrix.getEntry(current.row, current.col))) {
          pivot.row = i;
          break;
        }
      }

      return pivot;
    }

    private IExpr getCoordinate(RowColIndex a) {
      return rowReducedMatrix.getEntry(a.row, a.col);
    }

    /**
     * Get the row reduced echelon form of the matrix.
     *
     * <p>See: <a href="http://en.wikipedia.org/wiki/Row_echelon_form">Wikipedia - Row echelon
     * form</a>.
     *
     * @return
     */
    public FieldMatrix<IExpr> getRowReducedMatrix() {
      return rowReducedMatrix;
    }

    /**
     * Swap the rows <code>a.row</code> and <code>b.row</code> in the matrix and swap the <code>row
     * </code> values in the corresponding <code>RowColIndex</code> objects.
     *
     * @param a
     * @param b
     */
    private void swapRow(RowColIndex a, RowColIndex b) {
      IExpr[] temp = rowReducedMatrix.getRow(a.row);
      rowReducedMatrix.setRow(a.row, rowReducedMatrix.getRow(b.row));
      rowReducedMatrix.setRow(b.row, temp);

      int t = a.row;
      a.row = b.row;
      b.row = t;
    }

    /**
     * Test if the column <code>a.col</code> of the matrix contains only zero-elements starting with
     * the <code>a.row</code> element.
     *
     * @param a
     * @return
     */
    private boolean isColumnZeroFromRow(RowColIndex a) {
      for (int i = 0; i < numRows; i++) {
        if (!isZero(rowReducedMatrix.getEntry(i, a.col))) {
          return false;
        }
      }

      return true;
    }

    /**
     * Test if the <code>row</code> of the matrix contains only zero-elements.
     *
     * @param row
     * @return
     */
    private boolean isRowZeroes(int row) {
      IExpr[] temp = rowReducedMatrix.getRow(row);
      for (int i = 0; i < numCols; i++) {
        if (!isZero(temp[i])) {
          return false;
        }
      }

      return true;
    }

    /**
     * Add the values of the row <code>to.row</code> to the product of the values of the row <code>
     * from.row * factor</code> and assign the result values back to the row <code>to.row</code>.
     *
     * @param to
     * @param from
     * @param factor
     */
    private void multiplyAdd(RowColIndex to, RowColIndex from, IExpr factor) {
      IExpr[] row = rowReducedMatrix.getRow(to.row);
      IExpr[] rowMultiplied = rowReducedMatrix.getRow(from.row);

      for (int i = 0; i < numCols; i++) {
        rowReducedMatrix.setEntry(to.row, i, row[i].plus((rowMultiplied[i].times(factor))));
      }
    }

    /**
     * Get the nullspace of the row reduced matrix.
     *
     * <p>See: <a href="http://en.wikipedia.org/wiki/Kernel_%28linear_algebra%29">Wikipedia - Kernel
     * (linear algebra)</a>. <a href="http://en.wikibooks.org/wiki/Linear_Algebra/Null_Spaces">
     * Wikibooks - Null Spaces</a>.
     *
     * @param minusOneFactor factor <code>-1</code> for multiplying all elements of the free part of
     *     the reduced row echelon form matrix
     * @return <code>null</code> if the input matrix has full rank, otherwise return the nullspaace.
     */
    public FieldMatrix<IExpr> getNullSpace(IExpr minusOneFactor) {
      int rank = getMatrixRank();
      int newRowDimension = rowReducedMatrix.getColumnDimension() - rank;
      if (newRowDimension == 0) {
        return null;
      }
      int newColumnDimension = rowReducedMatrix.getColumnDimension();
      if (nullSpaceCache != null) {
        return nullSpaceCache;
      }
      nullSpaceCache = rowReducedMatrix.createMatrix(newRowDimension, newColumnDimension);
      getResultOfNullspace(minusOneFactor, rank);
      return nullSpaceCache;
    }

    private void getResultOfNullspace(IExpr minusOneFactor, int rank) {
      // search free columns
      boolean[] columns = new boolean[nullSpaceCache.getColumnDimension()];
      int numberOfFreeColumns = 0;
      for (int i = 0; i < rank; i++) {
        if (!columns[i]) {
          for (int k = i; k < rowReducedMatrix.getColumnDimension(); k++) {
            if (isZero(rowReducedMatrix.getEntry(i, k))) {
              columns[k] = true;
              // free column
              int offset = 0;
              for (int j = 0; j < rank; j++) {
                if (columns[j]) {
                  offset++;
                }
                nullSpaceCache.setEntry(
                    numberOfFreeColumns, j + offset, rowReducedMatrix.getEntry(j, i));
              }
              numberOfFreeColumns++;
            } else {
              break;
            }
          }
        }
      }

      // Let's take the rest of the 'free part' of the reduced row echelon
      // form
      int start = rank + numberOfFreeColumns;
      int row = numberOfFreeColumns;
      for (int i = start; i < nullSpaceCache.getColumnDimension(); i++) {
        int offset = 0;
        for (int j = 0; j < rank; j++) {
          if (columns[j]) {
            offset++;
          }
          nullSpaceCache.setEntry(row, j + offset, rowReducedMatrix.getEntry(j, i));
        }
        row++;
      }
      for (int i = start; i < nullSpaceCache.getColumnDimension(); i++) {
        columns[i] = true;
      }

      // multiply matrix with scalar -1
      nullSpaceCache = nullSpaceCache.scalarMultiply(minusOneFactor);

      // append the 'one element' (typically as identity matrix)
      row = 0;
      for (int i = 0; i < columns.length; i++) {
        if (columns[i]) {
          nullSpaceCache.setEntry(row++, i, F.C1);
        }
      }
    }

    /**
     * Create the &quot;reduced row echelon form&quot; of a matrix.
     *
     * <p>See: <a href="http://en.wikipedia.org/wiki/Row_echelon_form">Wikipedia - Row echelon
     * form</a>.
     *
     * @return
     */
    private FieldMatrix<IExpr> rowReduce() {
      RowColIndex pivot = new RowColIndex(0, 0);
      int submatrix = 0;
      for (int x = 0; x < numCols; x++) {
        pivot = new RowColIndex(pivot.row, x);
        // Step 1
        // Begin with the leftmost nonzero column. This is a pivot column.
        // The pivot position is at the top.
        for (int i = x; i < numCols; i++) {
          if (isColumnZeroFromRow(pivot) == false) {
            break;
          } else {
            pivot.col = i;
          }
        }
        // Step 2
        // Select a nonzero entry in the pivot column with the highest
        // absolute value as a pivot.
        pivot = findPivot(pivot);

        if (isZero(getCoordinate(pivot))) {
          pivot.row++;
          if (pivot.row + 1 > numRows) {
            pivot.row--;
            continue;
          }
        }

        // If necessary, interchange rows to move this entry into the pivot
        // position.
        // move this row to the top of the submatrix
        if (pivot.row != submatrix && !isRowZeroes(pivot.row)) {
          swapRow(new RowColIndex(submatrix, pivot.col), pivot);
        }

        if (!(isZero(getCoordinate(pivot)))) {
          // Force pivot to be 1
          if (!isOne(getCoordinate(pivot))) {
            IExpr scalar = getCoordinate(pivot).inverse();
            scaleRow(pivot, scalar);
          }
          // Step 3
          // Use row replacement operations to create zeroes in all positions
          // below the pivot.
          // belowPivot = belowPivot + (Pivot * -belowPivot)
          for (int i = pivot.row + 1; i < numRows; i++) {
            RowColIndex belowPivot = new RowColIndex(i, pivot.col);
            IExpr complement = getCoordinate(belowPivot).negate().divide(getCoordinate(pivot));
            multiplyAdd(belowPivot, pivot, complement);
          }
          // Step 5
          // Beginning with the rightmost pivot and working upward and to the
          // left, create zeroes above each pivot.
          // If a pivot is not 1, make it 1 by a scaling operation.
          // Use row replacement operations to create zeroes in all positions
          // above the pivot
          for (int i = pivot.row; i >= 0; i--) {
            if (i == pivot.row) {
              if (!isOne(getCoordinate(pivot))) {
                scaleRow(pivot, getCoordinate(pivot).inverse());
              }
              continue;
            }

            RowColIndex abovePivot = new RowColIndex(i, pivot.col);
            IExpr complement = getCoordinate(abovePivot).negate().divide(getCoordinate(pivot));
            multiplyAdd(abovePivot, pivot, complement);
          }
          submatrix++;
        }
        // Step 4
        // Ignore the row containing the pivot position and cover all rows,
        // if any, above it.
        // Apply steps 1-3 to the remaining submatrix. Repeat until there
        // are no more nonzero entries.
        if ((pivot.row + 1) >= numRows) { // || isRowZeroes(pivot.row + 1)) {
          break;
        }

        pivot.row++;
      }
      EvalEngine engine = EvalEngine.get();
      IEvalStepListener listener = engine.getStepListener();
      if (listener != null) {
        listener.add(
            Convert.matrix2List(originalMatrix),
            Convert.matrix2List(rowReducedMatrix),
            engine.getRecursionCounter(),
            -1,
            "ReducedRowEchelonForm");
      }
      return rowReducedMatrix;
    }

    /**
     * Get the rank of the row reduced matrix.
     *
     * <p>See: <a href="http://en.wikipedia.org/wiki/Rank_%28linear_algebra%29">Wikipedia - Rank
     * (linear algebra)</a>.
     *
     * @return the rank of the matrix.
     */
    public int getMatrixRank() {
      if (rowReducedMatrix.getRowDimension() == 0 || rowReducedMatrix.getColumnDimension() == 0) {
        return 0;
      }
      if (matrixRankCache < 0) {
        matrixRankCache = 0;
        int rows = rowReducedMatrix.getRowDimension() - 1;
        for (int i = rows; i >= 0; i--) {
          if (!isRowZeroes(i)) {
            matrixRankCache = i + 1;
            return matrixRankCache;
          }
        }
      }
      return matrixRankCache;
    }

    /**
     * Multiply all <code>x.row</code> elements with the scalar <code>factor</code>.
     *
     * @param x
     * @param factor
     */
    private void scaleRow(RowColIndex x, IExpr factor) {
      for (int i = 0; i < numCols; i++) {
        rowReducedMatrix.multiplyEntry(x.row, i, factor);
      }
    }
  }

  /**
   *
   *
   * <pre>
   * ArrayDepth(a)
   * </pre>
   *
   * <blockquote>
   *
   * <p>returns the depth of the non-ragged array <code>a</code>, defined as <code>
   * Length(Dimensions(a))</code>.<br>
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; ArrayDepth({{a,b},{c,d}})
   * 2
   *
   * &gt;&gt; ArrayDepth(x)
   * 0
   * </pre>
   */
  private static final class ArrayDepth extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      IExpr arg1 = ast.arg1();
      if (arg1.isAST()) {
        IAST list = (IAST) arg1;
        IExpr header = list.head();
        //        ArrayList<Integer> dims = new ArrayList<Integer>();
        //        arrayDepthRecursive(list, header, dims);
        ArrayList<Integer> dims = LinearAlgebra.dimensions(list, header);
        return F.ZZ(dims.size());
      }
      if (arg1.isSparseArray()) {
        int[] dims = ((ISparseArray) arg1).getDimension();
        return F.ZZ(dims.length);
      }
      if (arg1.isNumericArray()) {
        int[] dims = ((INumericArray) arg1).getDimension();
        return F.ZZ(dims.length);
      }
      return F.C0;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_1;
    }

    @Override
    public void setUp(final ISymbol newSymbol) {
      setOptions(
          newSymbol, //
          F.List(F.Rule(S.AllowedHeads, S.Automatic)));
    }
  }

  /**
   *
   *
   * <pre>
   * CharacteristicPolynomial(matrix, var)
   * </pre>
   *
   * <blockquote>
   *
   * <p>computes the characteristic polynomial of a <code>matrix</code> for the variable <code>var
   * </code>.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href= "https://en.wikipedia.org/wiki/Characteristic_polynomial">Wikipedia -
   *       Characteristic polynomial</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; CharacteristicPolynomial({{1, 2}, {42, 43}}, x)
   * -41-44*x+x^2
   * </pre>
   */
  private static final class CharacteristicPolynomial extends AbstractFunctionEvaluator {

    /**
     * Generate the characteristic polynomial of a square matrix.
     *
     * @param dim dimension of the square matrix
     * @param matrix the square matrix
     * @param variable the variable which should be used in the resulting characteristic polynomial
     * @return
     */
    public static IAST generateCharacteristicPolynomial(int dim, IAST matrix, IExpr variable) {
      final IExpr[] valuesForIdentityMatrix = {F.C0, variable};
      return F.Det(F.Subtract(matrix, diagonalMatrix(valuesForIdentityMatrix, dim)));
    }

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      int[] dimensions = ast.arg1().isMatrix();
      if (dimensions != null && dimensions[0] == dimensions[1]) {
        // a matrix with square dimensions
        IAST matrix = (IAST) ast.arg1().normal(false);
        IExpr variable = ast.arg2();
        if (!variable.isVariable()) {
          // `1` is not a valid variable.
          return IOFunctions.printMessage(ast.topHead(), "ivar", F.List(variable), engine);
        }
        return generateCharacteristicPolynomial(dimensions[0], matrix, variable);
      }

      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_2_2;
    }
  }

  /**
   * Calculates the Cholesky decomposition of a matrix.
   *
   * <p>The Cholesky decomposition of a real symmetric positive-definite matrix A consists of a
   * lower triangular matrix L with same size such that: A = LL<sup>T</sup>. In a sense, this is the
   * square root of A.
   */
  private static final class CholeskyDecomposition extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      RealMatrix matrix;
      try {

        matrix = ast.arg1().toRealMatrix();
        if (matrix != null) {
          final org.hipparchus.linear.CholeskyDecomposition dcomposition =
              new org.hipparchus.linear.CholeskyDecomposition(matrix);
          // Returns the transpose of the matrix L of the decomposition.
          return new ASTRealMatrix(dcomposition.getLT(), false);
        }
      } catch (final MathRuntimeException mre) {
        // org.hipparchus.exception.MathIllegalArgumentException: inconsistent dimensions: 0 != 3
        return engine.printMessage(ast.topHead(), mre);
      } catch (final ValidateException ve) {
        if (FEConfig.SHOW_STACKTRACE) {
          ve.printStackTrace();
        }
        return engine.printMessage(ast.topHead(), ve);
      } catch (final IndexOutOfBoundsException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
      }

      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_1;
    }
  }

  /**
   *
   *
   * <pre>
   * ConjugateTranspose(matrix)
   * </pre>
   *
   * <blockquote>
   *
   * <p>get the transposed <code>matrix</code> with conjugated matrix elements.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href="http://en.wikipedia.org/wiki/Transpose">Wikipedia - Transpose</a>
   *   <li><a href="http://en.wikipedia.org/wiki/Complex_conjugation">Wikipedia - Complex
   *       conjugation</a>
   * </ul>
   */
  private static final class ConjugateTranspose extends Transpose {

    @Override
    protected IExpr transform(final IExpr expr) {
      return expr.conjugate();
    }
  }

  /**
   *
   *
   * <pre>
   * Cross(a, b)
   * </pre>
   *
   * <blockquote>
   *
   * <p>computes the vector cross product of <code>a</code> and <code>b</code>.
   *
   * </blockquote>
   *
   * <p>See:
   *
   * <ul>
   *   <li><a href="https://en.wikipedia.org/wiki/Cross_product">Wikipedia: Cross product</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; Cross({x1, y1, z1}, {x2, y2, z2})
   * {-y2*z1+y1*z2,x2*z1-x1*z2,-x2*y1+x1*y2}
   *
   * &gt;&gt; Cross({x, y})
   * {-y,x}
   * </pre>
   *
   * <p>The arguments are expected to be vectors of equal length, and the number of arguments is
   * expected to be 1 less than their length.
   *
   * <pre>
   * &gt;&gt; Cross({1, 2}, {3, 4, 5})
   * Cross({1, 2}, {3, 4, 5})
   * </pre>
   */
  private static final class Cross extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      IExpr arg1 = ast.arg1();
      if (ast.isAST2()) {
        IExpr arg2 = ast.arg2();
        int dim1 = arg1.isVector();
        int dim2 = arg2.isVector();
        if (dim1 == 2 && dim2 == 2) {
          final IAST v1 = (IAST) arg1.normal(false);
          final IAST v2 = (IAST) arg2.normal(false);

          if ((v1.isAST2()) || (v2.isAST2())) {
            // Cross({a,b}, {c,d})", "a*d-b*c
            return F.Subtract(Times(v1.arg1(), v2.arg2()), Times(v1.arg2(), v2.arg1()));
          }
        } else if (dim1 == 3 && dim2 == 3) {
          final IAST v1 = (IAST) arg1.normal(false);
          final IAST v2 = (IAST) arg2.normal(false);

          if ((v1.isAST3()) || (v2.isAST3())) {
            return List(
                Plus(Times(v1.arg2(), v2.arg3()), Times(CN1, v1.arg3(), v2.arg2())),
                Plus(Times(v1.arg3(), v2.arg1()), Times(CN1, v1.arg1(), v2.arg3())),
                Plus(Times(v1.arg1(), v2.arg2()), Times(CN1, v1.arg2(), v2.arg1())));
          }
        }
      } else if (ast.isAST1()) {
        int dim1 = arg1.isVector();
        if (dim1 == 2) {
          final IAST v1 = (IAST) arg1.normal(false);
          return List(Negate(v1.arg2()), v1.arg1());
        }
      } else if (ast.size() > 3) {
        int dim1 = arg1.isVector();
        if (dim1 == ast.size()) {
          for (int i = 2; i < ast.size(); i++) {
            if (ast.get(i).isVector() != dim1) {
              return F.NIL;
            }
          }
          // TODO implement for more than 2 vector arguments
        }
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_INFINITY;
    }
  }

  /**
   *
   *
   * <pre>
   * DesignMatrix(m, f, x)
   * </pre>
   *
   * <blockquote>
   *
   * <p>returns the design matrix.
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; DesignMatrix({{2, 1}, {3, 4}, {5, 3}, {7, 6}}, x, x)
   * {{1,2},{1,3},{1,5},{1,7}}
   *
   * &gt;&gt; DesignMatrix({{2, 1}, {3, 4}, {5, 3}, {7, 6}}, f(x), x)
   * {{1,f(2)},{1,f(3)},{1,f(5)},{1,f(7)}}
   * </pre>
   */
  private static class DesignMatrix extends AbstractEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      IExpr m = ast.arg1();
      IExpr f = ast.arg2();
      IExpr x = ast.arg3();
      if (f.isList()) {
        if (x.isAtom()) {
          // DesignMatrix(m_, f_List, x_?AtomQ) :=
          // DesignMatrix(m, {f}, ConstantArray(x, Length(f)))
          return F.DesignMatrix(m, F.List(f), F.ConstantArray(x, F.ZZ(((IAST) f).argSize())));
        } else if (x.isList()) {
          // DesignMatrix(m_, f_List, x_List) :=
          // Prepend(MapThread(Function({g, y, r}, g /. y -> r), {f, x, Most(#)}), 1)& /@ m
          return Map(
              Function(
                  Prepend(
                      MapThread(
                          Function(List(S.g, S.y, S.r), ReplaceAll(S.g, Rule(S.y, S.r))),
                          List(f, x, Most(Slot1))),
                      C1)),
              m);
        }
      } else {
        if (x.isAtom()) {
          // DesignMatrix(m_, f_, x_?AtomQ) := DesignMatrix(m, {f}, {x})
          return F.DesignMatrix(m, F.List(f), F.List(x));
        }
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_3_3;
    }

    @Override
    public void setUp(final ISymbol newSymbol) {}
  }

  /**
   *
   *
   * <pre>
   * Det(matrix)
   * </pre>
   *
   * <blockquote>
   *
   * <p>computes the determinant of the <code>matrix</code>.
   *
   * </blockquote>
   *
   * <p>See:
   *
   * <ul>
   *   <li><a href="https://en.wikipedia.org/wiki/Determinant">Wikipedia: Determinant</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; Det({{1, 1, 0}, {1, 0, 1}, {0, 1, 1}})
   * -2
   * </pre>
   *
   * <p>Symbolic determinant:
   *
   * <pre>
   * &gt;&gt; Det({{a, b, c}, {d, e, f}, {g, h, i}})
   * -c*e*g+b*f*g+c*d*h-a*f*h-b*d*i+a*e*i
   * </pre>
   */
  private static class Det extends AbstractMatrix1Expr {

    @Override
    public int[] checkMatrixDimensions(IExpr arg1) {
      return Convert.checkNonEmptySquareMatrix(S.Det, arg1);
    }

    @Override
    public IExpr matrixEval(final FieldMatrix<IExpr> matrix) {
      if (matrix.getRowDimension() == 2 && matrix.getColumnDimension() == 2) {
        return determinant2x2(matrix);
      }
      if (matrix.getRowDimension() == 3 && matrix.getColumnDimension() == 3) {
        return determinant3x3(matrix);
      }
      final FieldLUDecomposition<IExpr> lu = new FieldLUDecomposition<IExpr>(matrix);
      return F.evalExpand(lu.getDeterminant());
    }

    @Override
    public IExpr realMatrixEval(RealMatrix matrix) {
      final org.hipparchus.linear.LUDecomposition lu =
          new org.hipparchus.linear.LUDecomposition(matrix);
      return F.num(lu.getDeterminant());
    }
  }

  private static class Diagonal extends AbstractEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      IExpr arg1 = ast.arg1();
      final int[] dim = arg1.isMatrix();
      if (dim != null) {
        int diff = 0;
        if (ast.size() > 2) {
          diff = ast.arg2().toIntDefault(Integer.MIN_VALUE);
          if (diff == Integer.MIN_VALUE) {
            return F.NIL;
          }
        }
        if (arg1.isAST()) {
          final IAST matrix = (IAST) arg1;
          int rowLength = dim[0];
          int columnLength = dim[1];
          IAST row;
          IASTAppendable result = F.ListAlloc(rowLength);
          for (int i = 1; i <= rowLength; i++) {
            row = (IAST) matrix.get(i);
            int indx = i + diff;
            if (indx > 0 && indx <= columnLength) {
              result.append(row.get(indx));
            }
          }
          return result;
        } else {
          FieldMatrix<IExpr> matrix = Convert.list2Matrix(arg1);
          if (matrix != null) {
            IASTAppendable result = F.ListAlloc(dim[0]);
            for (int i = 0; i < dim[0]; i++) {
              int indx = i + diff;
              if (indx >= 0 && indx < dim[1]) {
                result.append(matrix.getEntry(i, indx));
              }
            }
            return result;
          }
        }
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }

    @Override
    public void setUp(final ISymbol newSymbol) {}
  }

  /**
   *
   *
   * <pre>
   * DiagonalMatrix(list)
   * </pre>
   *
   * <blockquote>
   *
   * <p>gives a matrix with the values in $list$ on its diagonal and zeroes elsewhere.
   *
   * </blockquote>
   *
   * <pre>
   * &gt;&gt; DiagonalMatrix({1, 2, 3})
   * {{1, 0, 0}, {0, 2, 0}, {0, 0, 3}}
   *
   * &gt;&gt; MatrixForm(%)
   *  1   0   0
   *  0   2   0
   *  0   0   3
   *
   * &gt;&gt; DiagonalMatrix(a + b)
   * DiagonalMatrix(a + b)
   * </pre>
   */
  private static class DiagonalMatrix extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      try {
        IExpr arg1 = ast.arg1();
        int dimension = arg1.isVector();
        final IAST vector;
        if (dimension >= 0) {
          IExpr normal = arg1.normal(false);
          if (normal.isAST()) {
            vector = (IAST) normal;
          } else {
            vector = F.NIL;
          }
        } else if (arg1.isAST()) {
          vector = (IAST) arg1;
        } else {
          vector = F.NIL;
        }
        if (vector.isPresent()) {
          int m = vector.size();
          final int offset = ast.isAST2() ? Validate.checkIntType(ast, 2, Integer.MIN_VALUE) : 0;
          return F.matrix((i, j) -> (i + offset) == j ? vector.get(i + 1) : F.C0, m - 1, m - 1);
        }
      } catch (final ValidateException ve) {
        // int number validation
        return engine.printMessage(ast.topHead(), ve);
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }

    @Override
    public void setUp(final ISymbol newSymbol) {}
  }

  /**
   *
   *
   * <pre>
   * Dimensions(expr)
   * </pre>
   *
   * <blockquote>
   *
   * <p>returns a list of the dimensions of the expression <code>expr</code>.
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <p>A vector of length 3:
   *
   * <pre>
   * &gt;&gt; Dimensions({a, b, c})
   *  = {3}
   * </pre>
   *
   * <p>A 3x2 matrix:
   *
   * <pre>
   * &gt;&gt; Dimensions({{a, b}, {c, d}, {e, f}})
   *  = {3, 2}
   * </pre>
   *
   * <p>Ragged arrays are not taken into account:
   *
   * <pre>
   * &gt;&gt; Dimensions({{a, b}, {b, c}, {c, d, e}})
   * {3}
   * </pre>
   *
   * <p>The expression can have any head:
   *
   * <pre>
   * &gt;&gt; Dimensions[f[f[a, b, c]]]
   * {1, 3}
   * &gt;&gt; Dimensions({})
   * {0}
   * &gt;&gt; Dimensions({{}})
   * {1, 0}
   * </pre>
   */
  private static class Dimensions extends AbstractFunctionEvaluator {

    private static IAST getDimensions(final IAST ast, int maximumLevel) {
      IAST list = (IAST) ast.arg1();
      IExpr header = list.head();
      ArrayList<Integer> dims = dimensions(list, header, maximumLevel - 1);
      int dimsSize = dims.size();
      IASTAppendable res = F.ListAlloc(dimsSize);
      return res.appendArgs(0, dimsSize, i -> F.ZZ(dims.get(i).intValue()));
    }

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      int maximumLevel = Integer.MAX_VALUE;
      if (ast.isAST2() && ast.arg2().isInteger()) {
        maximumLevel = ast.arg2().toIntDefault(Integer.MIN_VALUE);
        if (maximumLevel < 0) {
          // Positive integer (less equal 2147483647) expected at position `2` in `1`.
          return IOFunctions.printMessage(S.Dimensions, "intpm", F.List(ast, F.C2), engine);
        }
      }
      IExpr arg1 = ast.arg1();
      if (arg1.isAST()) {
        if (maximumLevel > 0) {
          return getDimensions(ast, maximumLevel);
        }
      } else if (arg1.isSparseArray()) {
        if (maximumLevel > 0) {
          return getDimensions(((ISparseArray) arg1).getDimension(), maximumLevel);
        }
      } else if (arg1.isNumericArray()) {
        if (maximumLevel > 0) {
          return getDimensions(((INumericArray) arg1).getDimension(), maximumLevel);
        }
      }

      return F.CEmptyList;
    }

    private IExpr getDimensions(int[] dims, int maximumLevel) {
      if (dims.length > maximumLevel) {
        int[] dest = new int[maximumLevel];
        System.arraycopy(dims, 0, dest, 0, maximumLevel);
        return F.ast(S.List, dest);
      }
      return F.ast(S.List, dims);
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }

    @Override
    public void setUp(final ISymbol newSymbol) {
      setOptions(
          newSymbol, //
          F.List(F.Rule(S.AllowedHeads, S.Automatic)));
    }
  }

  /**
   *
   *
   * <pre>
   * Dot(x, y) or x . y
   * </pre>
   *
   * <blockquote>
   *
   * <p><code>x . y</code> computes the vector dot product or matrix product <code>x . y</code>.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href="https://en.wikipedia.org/wiki/Matrix_multiplication">Wikipedia - Matrix
   *       multiplication</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <p>Scalar product of vectors:
   *
   * <pre>
   * &gt;&gt; {a, b, c} . {x, y, z}
   * a*x+b*y+c*z
   * </pre>
   *
   * <p>Product of matrices and vectors:
   *
   * <pre>
   * &gt;&gt; {{a, b}, {c, d}} . {x, y}
   * {a*x+b*y,c*x+d*y}
   * </pre>
   *
   * <p>Matrix product:
   *
   * <pre>
   * &gt;&gt; {{a, b}, {c, d}} . {{r, s}, {t, u}}
   * {{a*r+b*t,a*s+b*u}, {c*r+d*t,c*s+d*u}}
   *
   * &gt;&gt; a . b
   * a.b
   * </pre>
   */
  private static class Dot extends AbstractNonOrderlessArgMultiple {

    @Override
    public IExpr evaluateAST1(final IAST ast, EvalEngine engine) {
      return ast.arg1();
    }

    @Override
    public IExpr e2ObjArg(IAST ast, final IExpr arg1, final IExpr arg2) {

      if (arg1.size() == 1 && arg2.size() == 1) {
        if (arg1.isList() && arg2.isList()) {
          return F.C0;
        }
        return F.NIL;
      }

      EvalEngine engine = EvalEngine.get();
      boolean togetherMode = engine.isTogetherMode();
      engine.setTogetherMode(true);
      try {
        IExpr temp = numericalDot(arg1, arg2);
        if (temp.isPresent()) {
          return temp;
        }

        FieldMatrix<IExpr> matrix0;
        FieldMatrix<IExpr> matrix1;
        FieldVector<IExpr> vector0;
        FieldVector<IExpr> vector1;

        int[] dim1 = arg1.isMatrix();
        if (dim1 != null && dim1[1] != 0) {

          matrix0 = Convert.list2Matrix(arg1);
          if (matrix0 != null) {
            if (arg2.isMatrix() != null) {

              matrix1 = Convert.list2Matrix(arg2);
              if (matrix1 != null) {
                return Convert.matrix2Expr(matrix0.multiply(matrix1));
              }
            } else if (arg2.isVector() != (-1)) {

              vector1 = Convert.list2Vector(arg2);
              if (vector1 != null) {
                return Convert.vector2Expr(matrix0.operate(vector1));
              }
            }
          }
          return F.NIL;
          // return engine.printMessage(ast.topHead() + ": Error in matrix");
        } else {
          int dim = arg1.isVector();
          if (dim != (-1)) {
            if (dim == 0) {
              if (arg2.isVector() == 0) {
                return F.C0;
              }
            } else {
              vector0 = Convert.list2Vector(arg1);
              if (vector0 != null) {
                if (arg2.isMatrix() != null) {
                  matrix1 = Convert.list2Matrix(arg2);
                  if (matrix1 != null) {
                    return Convert.vector2Expr(matrix1.preMultiply(vector0));
                  }
                } else if (arg2.isVector() != (-1)) {
                  vector1 = Convert.list2Vector(arg2);
                  if (vector1 != null) {
                    return vector0.dotProduct(vector1);
                  }
                }
              }
              return engine.printMessage(ast.topHead() + ": Error in vector");
            }
          }
        }

        // } catch (final ClassCastException e) {
        // if (Config.SHOW_STACKTRACE) {
        // e.printStackTrace();
        // }
        // } catch (final IndexOutOfBoundsException e) {
        // if (Config.SHOW_STACKTRACE) {
        // e.printStackTrace();
        // }
      } catch (final MathRuntimeException mre) {
        // org.hipparchus.exception.MathIllegalArgumentException: inconsistent dimensions: 0 != 3
        return engine.printMessage(ast.topHead(), mre);
      } catch (final RuntimeException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
        engine.printMessage(ast.topHead() + ": " + e.getMessage());
        // } catch (final org.hipparchus.exception.MathRuntimeException e) {
        // if (Config.SHOW_STACKTRACE) {
        // e.printStackTrace();
        // }
      } finally {
        engine.setTogetherMode(togetherMode);
      }
      return F.NIL;
    }

    private IExpr numericalDot(final IExpr o0, final IExpr o1) throws MathIllegalArgumentException {
      if (o0.isRealMatrix()) {
        if (o1.isMatrix() != null) {
          RealMatrix m1 = o1.toRealMatrix();
          if (m1 != null) {
            RealMatrix m0 = o0.toRealMatrix();
            return new ASTRealMatrix(m0.multiply(m1), false);
          }
        } else if (o1.isVector() != (-1)) {
          RealVector m1 = o1.toRealVector();
          if (m1 != null) {
            RealMatrix m0 = o0.toRealMatrix();
            return new ASTRealVector(m0.operate(m1), false);
          }
        }
      } else if (o0.isRealVector()) {
        if (o1.isMatrix() != null) {
          RealMatrix m1 = o1.toRealMatrix();
          if (m1 != null) {
            RealVector v0 = o0.toRealVector();
            return new ASTRealVector(m1.preMultiply(v0), false);
          }
        } else if (o1.isVector() != (-1)) {
          RealVector v1 = o1.toRealVector();
          if (v1 != null) {
            RealVector v0 = o0.toRealVector();
            return F.num(v0.dotProduct(v1));
          }
        }
      }

      if (o1.isRealMatrix()) {
        if (o0.isMatrix() != null) {
          RealMatrix m0 = o0.toRealMatrix();
          if (m0 != null) {
            RealMatrix m1 = o1.toRealMatrix();
            return new ASTRealMatrix(m0.multiply(m1), false);
          }
        } else if (o0.isVector() != (-1)) {
          RealVector v0 = o0.toRealVector();
          if (v0 != null) {
            RealMatrix m1 = o1.toRealMatrix();
            return new ASTRealVector(m1.preMultiply(v0), false);
          }
        }
      } else if (o1.isRealVector()) {
        if (o0.isMatrix() != null) {
          RealMatrix m0 = o0.toRealMatrix();
          if (m0 != null) {
            RealVector m1 = o1.toRealVector();
            return new ASTRealVector(m0.operate(m1), false);
          }
        } else if (o0.isVector() != (-1)) {
          RealVector v0 = o0.toRealVector();
          if (v0 != null) {
            RealVector v1 = o1.toRealVector();
            if (v0.getDimension() == v1.getDimension()) {
              return F.num(v0.dotProduct(v1));
            }
          }
        }
      }
      return F.NIL;
    }

    @Override
    public IExpr numericEval(final IAST ast, EvalEngine engine) {
      return evaluate(ast, engine);
    }

    @Override
    public void setUp(final ISymbol newSymbol) {
      newSymbol.setAttributes(ISymbol.FLAT | ISymbol.ONEIDENTITY);
    }
  }

  /**
   *
   *
   * <pre>
   * Eigenvalues(matrix)
   * </pre>
   *
   * <blockquote>
   *
   * <p>get the numerical eigenvalues of the <code>matrix</code>.
   *
   * </blockquote>
   *
   * <p>See:
   *
   * <ul>
   *   <li><a href="http://en.wikipedia.org/wiki/Eigenvalue">Wikipedia - Eigenvalue</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt;&gt; Eigenvalues({{1,0,0},{0,1,0},{0,0,1}})
   * {1.0,1.0,1.0}
   * </pre>
   */
  private static class Eigenvalues extends AbstractMatrix1Expr {

    @Override
    public int[] checkMatrixDimensions(IExpr arg1) {
      return Convert.checkNonEmptySquareMatrix(S.Eigenvalues, arg1);
    }

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.size() == 2) {
        FieldMatrix<IExpr> matrix;
        try {

          IExpr arg1 = ast.arg1();
          int[] dim = arg1.isMatrix();
          if (dim != null) {
            if (dim[0] == 1 && dim[1] == 1) {
              // Eigenvalues({{a}})
              return List(((IAST) arg1).getPart(1, 1));
            }
            if (dim[0] == 2 && dim[1] == 2) {
              matrix = Convert.list2Matrix(arg1);
              if (matrix != null) {
                // Eigenvalues({{a, b}, {c, d}}) =>
                // {
                // 1/2*(a + d - Sqrt(a^2 + 4*b*c - 2*a*d + d^2)),
                // 1/2*(a + d + Sqrt(a^2 + 4*b*c - 2*a*d + d^2))
                // }
                IExpr sqrtExpr =
                    Sqrt(
                        Plus(
                            Sqr(matrix.getEntry(0, 0)),
                            Times(C4, matrix.getEntry(0, 1), matrix.getEntry(1, 0)),
                            Times(CN2, matrix.getEntry(0, 0), matrix.getEntry(1, 1)),
                            Sqr(matrix.getEntry(1, 1))));
                return List(
                    Times(
                        C1D2, Plus(Negate(sqrtExpr), matrix.getEntry(0, 0), matrix.getEntry(1, 1))),
                    Times(C1D2, Plus(sqrtExpr, matrix.getEntry(0, 0), matrix.getEntry(1, 1))));
              }
            }
            // if (((IAST) arg1).forAllLeaves(x->x.isExactNumber(), 1)) {
            // ISymbol x = F.Dummy("x");
            // IExpr m = engine.evaluate(F.CharacteristicPolynomial(arg1, x));
            // IAST list = PolynomialFunctions.roots(m, false, F.List(x), engine);
            // if (list.isPresent()) {
            // return F.Reverse(list);
            // }
            // }
          }

        } catch (final RuntimeException e) {
          if (FEConfig.SHOW_STACKTRACE) {
            e.printStackTrace();
          }
        }
      }
      // switch to numeric calculation
      return numericEval(ast, engine);
    }

    @Override
    public IExpr matrixEval(FieldMatrix<IExpr> matrix) {
      return F.NIL;
    }

    @Override
    public IAST realMatrixEval(RealMatrix matrix) {
      EigenDecomposition ed = new EigenDecomposition(matrix);
      double[] realValues = ed.getRealEigenvalues();
      double[] imagValues = ed.getImagEigenvalues();
      int size = realValues.length;
      IASTAppendable list = F.ListAlloc(size);
      return list.appendArgs(
          0,
          size,
          (int i) -> {
            if (F.isZero(imagValues[i])) {
              return F.num(realValues[i]);
            }
            return F.complexNum(realValues[i], imagValues[i]);
          });
    }
  }

  /**
   * Eigenvectors(matrix) </pre>
   *
   * <blockquote>
   *
   * <p>get the numerical eigenvectors of the <code>matrix</code>.
   *
   * </blockquote>
   *
   * <p>See:
   *
   * <ul>
   *   <li><a href="http://en.wikipedia.org/wiki/Eigenvalue">Wikipedia - Eigenvalue</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt;&gt; Eigenvectors({{1,0,0},{0,1,0},{0,0,1}})
   * {{1.0,0.0,0.0},{0.0,1.0,0.0},{0.0,0.0,1.0}}
   * </pre>
   */
  private static class Eigenvectors extends AbstractMatrix1Expr {

    @Override
    public int[] checkMatrixDimensions(IExpr arg1) {
      return Convert.checkNonEmptySquareMatrix(S.Eigenvectors, arg1);
    }

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.size() == 2) {
        FieldMatrix<IExpr> matrix;
        try {

          int[] dim = ast.arg1().isMatrix();
          if (dim != null) {
            if (dim[0] == 1 && dim[1] == 1) {
              // Eigenvectors({{a}})
              return C1;
            }
            if (dim[0] == 2 && dim[1] == 2) {
              matrix = Convert.list2Matrix(ast.arg1());
              if (matrix != null) {
                if (matrix.getEntry(1, 0).isZero()) {
                  if (matrix.getEntry(0, 0).equals(matrix.getEntry(1, 1))) {
                    // Eigenvectors({{a, b}, {0, a}})
                    return List(List(C1, C0), List(C0, C0));
                  } else {
                    // Eigenvectors({{a, b}, {0, d}})
                    return List(
                        List(C1, C0),
                        List(
                            Divide(
                                Negate(matrix.getEntry(0, 1)),
                                Subtract(matrix.getEntry(0, 0), matrix.getEntry(1, 1))),
                            C1));
                  }
                } else {
                  // Eigenvectors({{a, b}, {c, d}}) =>
                  // {
                  // { - (1/(2*c)) * (-a + d + Sqrt(a^2 + 4*b*c - 2*a*d + d^2)), 1},
                  // { - (1/(2*c)) * (-a + d - Sqrt(a^2 + 4*b*c - 2*a*d + d^2)), 1}
                  // }
                  IExpr sqrtExpr =
                      Sqrt(
                          Plus(
                              Sqr(matrix.getEntry(0, 0)),
                              Times(C4, matrix.getEntry(0, 1), matrix.getEntry(1, 0)),
                              Times(CN2, matrix.getEntry(0, 0), matrix.getEntry(1, 1)),
                              Sqr(matrix.getEntry(1, 1))));
                  return List(
                      List(
                          Times(
                              CN1D2,
                              Power(matrix.getEntry(1, 0), CN1),
                              Plus(sqrtExpr, Negate(matrix.getEntry(0, 0)), matrix.getEntry(1, 1))),
                          C1),
                      List(
                          Times(
                              CN1D2,
                              Power(matrix.getEntry(1, 0), CN1),
                              Plus(
                                  Negate(sqrtExpr),
                                  Negate(matrix.getEntry(0, 0)),
                                  matrix.getEntry(1, 1))),
                          C1));
                }
              }
            }
          }
        } catch (final MathRuntimeException mre) {
          // org.hipparchus.exception.MathIllegalArgumentException: inconsistent dimensions: 0 != 3
          return engine.printMessage(ast.topHead(), mre);
        } catch (final ClassCastException e) {
          if (FEConfig.SHOW_STACKTRACE) {
            e.printStackTrace();
          }
        } catch (final IndexOutOfBoundsException e) {
          if (FEConfig.SHOW_STACKTRACE) {
            e.printStackTrace();
          }
        }
      }

      // switch to numeric calculation
      return numericEval(ast, engine);
    }

    @Override
    public IExpr matrixEval(FieldMatrix<IExpr> matrix) {
      return F.NIL;
    }

    @Override
    public IAST realMatrixEval(RealMatrix matrix) {
      // TODO
      //      ComplexEigenDecomposition ced = new ComplexEigenDecomposition(matrix);
      //      int size = matrix.getColumnDimension();
      //      IASTAppendable list = F.ListAlloc(size);
      //      return list.appendArgs(
      //          0,
      //          size,
      //          i -> {
      //            FieldVector<Complex> rv = ced.getEigenvector(i);
      //            return Convert.complexVector2List(rv);
      //          });
      EigenDecomposition ed = new EigenDecomposition(matrix);
      int size = matrix.getColumnDimension();
      IASTAppendable list = F.ListAlloc(size);
      return list.appendArgs(
          0,
          size,
          i -> {
            RealVector rv = ed.getEigenvector(i);
            return Convert.vector2List(rv);
          });
    }
  }

  private static class FourierMatrix extends AbstractFunctionEvaluator {

    /**
     * Complex number on unit circle with given argument.
     *
     * @param arg
     * @return complex number on unit circle with given argument
     */
    private static IExpr unit(IExpr arg) {
      return F.Plus(F.Cos(arg), F.Times(F.CI, F.Sin(arg)));
    }

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.arg1().isInteger()) {
        final int m = ast.arg1().toIntDefault(Integer.MIN_VALUE);
        if (m <= 0) {
          // Positive integer (less equal 2147483647) expected at position `2` in `1`.
          return IOFunctions.printMessage(S.FourierMatrix, "intpm", F.List(ast, F.C1), engine);
        }
        IAST scalar = F.Sqrt(F.QQ(1, m));
        return F.matrix(
            (i, j) -> unit(F.QQ(2L * ((long) i) * ((long) j), m).times(S.Pi)).times(scalar), m, m);
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_1;
    }
  }

  /**
   *
   *
   * <pre>
   * FromPolarCoordinates({r, t})
   * </pre>
   *
   * <blockquote>
   *
   * <p>return the cartesian coordinates for the polar coordinates <code>{r, t}</code>.
   *
   * </blockquote>
   *
   * <pre>
   * FromPolarCoordinates({r, t, p})
   * </pre>
   *
   * <blockquote>
   *
   * <p>return the cartesian coordinates for the polar coordinates <code>{r, t, p}</code>.
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; FromPolarCoordinates({r, t})
   * {r*Cos(t),r*Sin(t)}
   *
   * &gt;&gt; FromPolarCoordinates({r, t, p})
   * {r*Cos(t),r*Cos(p)*Sin(t),r*Sin(p)*Sin(t)}
   * </pre>
   */
  private static final class FromPolarCoordinates extends AbstractEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      IExpr arg1 = ast.arg1();
      int dim = arg1.isVector();
      if (dim > 0) {
        if (arg1.isAST()) {
          IAST list = (IAST) arg1;
          if (dim == 2) {
            IExpr r = list.arg1();
            IExpr theta = list.arg2();
            return F.list(F.Times(r, F.Cos(theta)), F.Times(r, F.Sin(theta)));
          } else if (dim == 3) {
            IExpr r = list.arg1();
            IExpr theta = list.arg2();
            IExpr phi = list.arg3();
            return F.list(
                F.Times(r, F.Cos(theta)),
                F.Times(r, F.Cos(phi), F.Sin(theta)),
                F.Times(r, F.Sin(theta), F.Sin(phi)));
          }
        } else {
          FieldVector<IExpr> vector = Convert.list2Vector(arg1);
          if (dim == 2) {
            IExpr r = vector.getEntry(0);
            IExpr theta = vector.getEntry(1);
            return F.list(F.Times(r, F.Cos(theta)), F.Times(r, F.Sin(theta)));
          } else if (dim == 3) {
            IExpr r = vector.getEntry(0);
            IExpr theta = vector.getEntry(1);
            IExpr phi = vector.getEntry(2);
            return F.list(
                F.Times(r, F.Cos(theta)),
                F.Times(r, F.Cos(phi), F.Sin(theta)),
                F.Times(r, F.Sin(theta), F.Sin(phi)));
          }
        }
      } else if (arg1.isList()) {
        return ((IAST) arg1).mapThreadEvaled(engine, F.ListAlloc(ast.size()), ast, 1);
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_1;
    }

    @Override
    public void setUp(final ISymbol newSymbol) {}
  }

  /**
   *
   *
   * <pre>
   * HilbertMatrix(n)
   * </pre>
   *
   * <blockquote>
   *
   * <p>gives the hilbert matrix with <code>n</code> rows and columns.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href="http://en.wikipedia.org/wiki/Hilbert_matrix">Wikipedia - Hilbert matrix</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; HilbertMatrix(2)
   * {{1,1/2},
   *  {1/2,1/3}}
   * </pre>
   */
  private static class HilbertMatrix extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      int rowSize = 0;
      int columnSize = 0;
      if (ast.isAST1() && ast.arg1().isInteger()) {
        rowSize = ast.arg1().toIntDefault(Integer.MIN_VALUE);
        if (rowSize < 0) {
          // Positive integer (less equal 2147483647) expected at position `2` in `1`.
          return IOFunctions.printMessage(S.HilbertMatrix, "intpm", F.List(ast, F.C1), engine);
        }
        columnSize = rowSize;
      } else if (ast.isAST2() && ast.arg1().isInteger() && ast.arg2().isInteger()) {
        rowSize = ast.arg1().toIntDefault(Integer.MIN_VALUE);
        if (rowSize < 0) {
          // Positive integer (less equal 2147483647) expected at position `2` in `1`.
          return IOFunctions.printMessage(S.HilbertMatrix, "intpm", F.List(ast, F.C1), engine);
        }
        columnSize = ast.arg2().toIntDefault(Integer.MIN_VALUE);
        if (columnSize < 0) {
          // Positive integer (less equal 2147483647) expected at position `2` in `1`.
          return IOFunctions.printMessage(S.HilbertMatrix, "intpm", F.List(ast, F.C2), engine);
        }
      } else {
        return F.NIL;
      }
      return F.matrix((i, j) -> F.QQ(1, i + j + 1), rowSize, columnSize);
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }
  }

  /**
   *
   *
   * <pre>
   * IdentityMatrix(n)
   * </pre>
   *
   * <blockquote>
   *
   * <p>gives the identity matrix with <code>n</code> rows and columns.
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; IdentityMatrix(3)
   * {{1, 0, 0}, {0, 1, 0}, {0, 0, 1}}
   * </pre>
   */
  private static class IdentityMatrix extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.arg1().isInteger()) {
        int m = ast.arg1().toIntDefault(Integer.MIN_VALUE);
        if (m < 0) {
          // Positive integer (less equal 2147483647) expected at position `2` in `1`.
          return IOFunctions.printMessage(S.IdentityMatrix, "intpm", F.List(ast, F.C1), engine);
        }
        return F.matrix((i, j) -> i.equals(j) ? F.C1 : F.C0, m, m);
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_1;
    }
  }

  /**
   *
   *
   * <pre>
   * Inner(f, x, y, g)
   * </pre>
   *
   * <blockquote>
   *
   * <p>computes a generalised inner product of <code>x</code> and <code>y</code>, using a
   * multiplication function <code>f</code> and an addition function <code>g</code>.
   *
   * </blockquote>
   *
   * <pre>
   * &gt;&gt; Inner(f, {a, b}, {x, y}, g)
   * g(f(a, x), f(b, y))
   * </pre>
   *
   * <p>'Inner' can be used to compute a dot product:
   *
   * <pre>
   * &gt;&gt; Inner(Times, {a, b}, {c, d}, Plus) == {a, b} . {c, d}
   * </pre>
   *
   * <p>The inner product of two boolean matrices:
   *
   * <pre>
   * &gt;&gt; Inner(And, {{False, False}, {False, True}}, {{True, False}, {True, True}}, Or)
   * {{False, False}, {True, True}}
   * </pre>
   */
  private static class Inner extends AbstractFunctionEvaluator {

    private static class InnerAlgorithm {
      final IExpr f;
      final IExpr g;
      final IExpr head;
      final IAST list1;
      final IAST list2;
      int list2Dim0;

      private InnerAlgorithm(final IExpr f, final IAST list1, final IAST list2, final IExpr g) {
        this.f = f;
        this.list1 = list1;
        this.list2 = list2;
        this.g = g;
        this.head = list2.head();
      }

      private IAST inner() {
        ArrayList<Integer> list1Dimensions = dimensions(list1, list1.head(), Integer.MAX_VALUE);
        ArrayList<Integer> list2Dimensions = dimensions(list2, list2.head(), Integer.MAX_VALUE);
        list2Dim0 = list2Dimensions.get(0);
        return recursionInner(
            new ArrayList<Integer>(),
            new ArrayList<Integer>(),
            list1Dimensions.subList(0, list1Dimensions.size() - 1),
            list2Dimensions.subList(1, list2Dimensions.size()));
      }

      @SuppressWarnings("unchecked")
      private IAST recursionInner(
          ArrayList<Integer> list1Cur,
          ArrayList<Integer> list2Cur,
          List<Integer> list1RestDimensions,
          List<Integer> list2RestDimensions) {
        if (list1RestDimensions.size() > 0) {
          int size = list1RestDimensions.get(0) + 1;
          IASTAppendable newResult = F.ast(head, size, false);
          for (int i = 1; i < size; i++) {
            ArrayList<Integer> list1CurClone = (ArrayList<Integer>) list1Cur.clone();
            list1CurClone.add(i);
            IAST recursionInner =
                recursionInner(
                    list1CurClone,
                    list2Cur,
                    list1RestDimensions.subList(1, list1RestDimensions.size()),
                    list2RestDimensions);
            if (recursionInner.isPresent()) {
              newResult.append(recursionInner);
            } else {
              return F.NIL;
            }
          }
          return newResult;
        } else if (list2RestDimensions.size() > 0) {
          int size = list2RestDimensions.get(0) + 1;
          IASTAppendable newResult = F.ast(head, size, false);
          for (int i = 1; i < size; i++) {
            ArrayList<Integer> list2CurClone = (ArrayList<Integer>) list2Cur.clone();
            list2CurClone.add(i);
            IAST recursionInner =
                recursionInner(
                    list1Cur,
                    list2CurClone,
                    list1RestDimensions,
                    list2RestDimensions.subList(1, list2RestDimensions.size()));
            if (recursionInner.isPresent()) {
              newResult.append(recursionInner);
            } else {
              return F.NIL;
            }
          }
          return newResult;
        } else {
          try {
            int size = list2Dim0 + 1;
            IASTAppendable part = F.ast(g, size, false);
            return part.appendArgs(size, i -> summand(list1Cur, list2Cur, i));
            // for (int i = 1; i < size; i++) {
            // part.append(summand(list1Cur, list2Cur, i));
            // }
            // return part;
          } catch (IndexOutOfBoundsException ioobe) {
            // TODO Length `1` of dimension `2` in `3` is incommensurate with length `4` of
            // dimension `5` in `6`.
            return IOFunctions.printMessage(S.Inner, "incom", F.List(), EvalEngine.get());
          }
        }
      }

      @SuppressWarnings("unchecked")
      private IAST summand(ArrayList<Integer> list1Cur, ArrayList<Integer> list2Cur, final int i) {
        IASTAppendable result = F.ast(f, 2, false);
        ArrayList<Integer> list1CurClone = (ArrayList<Integer>) list1Cur.clone();
        list1CurClone.add(i);
        result.append(list1.getPart(list1CurClone));
        ArrayList<Integer> list2CurClone = (ArrayList<Integer>) list2Cur.clone();
        list2CurClone.add(0, i);
        result.append(list2.getPart(list2CurClone));
        return result;
      }
    }

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.arg2().isAST() && ast.arg3().isAST()) {
        IExpr f = ast.arg1();
        IAST list1 = (IAST) ast.arg2().normal(false);
        IAST list2 = (IAST) ast.arg3().normal(false);
        IExpr g;
        if (ast.isAST3()) {
          g = S.Plus;
        } else {
          g = ast.arg4();
        }
        IExpr head2 = list2.head();
        if (!list1.head().equals(head2)) {
          return F.NIL;
        }
        ArrayList<Integer> dim1 = dimensions(list1);
        ArrayList<Integer> dim2 = dimensions(list2);
        if (dim1.size() == 0) {
          // Nonatomic expression expected at position `1` in `2`.
          return IOFunctions.printMessage(S.Inner, "normal", F.List(F.C2, list1), EvalEngine.get());
        }
        if (dim2.size() == 0) {
          // Nonatomic expression expected at position `1` in `2`.
          return IOFunctions.printMessage(S.Inner, "normal", F.List(F.C3, list2), EvalEngine.get());
        }
        int dimSize1 = dim1.get(dim1.size() - 1);
        int dimSize2 = dim2.get(0);
        if (dimSize1 != dimSize2) {
          // Length `1` of dimension `2` in `3` is incommensurate with length `4` of dimension `5`
          // in `6`.
          return IOFunctions.printMessage(
              S.Inner,
              "incom",
              F.List(F.ZZ(dimSize1), F.ZZ(dim1.size()), list1, F.C1, F.ZZ(dimSize2), list2),
              EvalEngine.get());
        }
        if (list1.size() == list2.size()) {
          InnerAlgorithm ic = new InnerAlgorithm(f, list1, list2, g);
          return ic.inner();
        }
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_3_4;
    }
  }

  /**
   *
   *
   * <pre>
   * Inverse(matrix)
   * </pre>
   *
   * <blockquote>
   *
   * <p>computes the inverse of the <code>matrix</code>.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href="https://en.wikipedia.org/wiki/Invertible_matrix">Wikipedia - Invertible
   *       matrix</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; Inverse({{1, 2, 0}, {2, 3, 0}, {3, 4, 1}})
   * {{-3,2,0},
   *  {2,-1,0},
   *  {1,-2,1}}
   * </pre>
   *
   * <p>The matrix <code>{{1, 0}, {0, 0}}</code> is singular.
   *
   * <pre>
   * &gt;&gt; Inverse({{1, 0}, {0, 0}})
   * Inverse({{1, 0}, {0, 0}})
   *
   * &gt;&gt; Inverse({{1, 0, 0}, {0, Sqrt(3)/2, 1/2}, {0,-1 / 2, Sqrt(3)/2}})
   * {{1,0,0},
   *  {0,Sqrt(3)/2,-1/2},
   *  {0,1/2,1/(1/(2*Sqrt(3))+Sqrt(3)/2)}}
   * </pre>
   */
  private static final class Inverse extends AbstractMatrix1Matrix {

    @Override
    public int[] checkMatrixDimensions(IExpr arg1) {
      return Convert.checkNonEmptySquareMatrix(S.Inverse, arg1);
    }

    public static FieldMatrix<IExpr> inverseMatrix(FieldMatrix<IExpr> matrix) {
      final FieldLUDecomposition<IExpr> lu = new FieldLUDecomposition<IExpr>(matrix);
      FieldDecompositionSolver<IExpr> solver = lu.getSolver();
      if (!solver.isNonSingular()) {
        // Matrix `1` is singular.
        IOFunctions.printMessage(
            S.Inverse, "sing", F.List(Convert.matrix2List(matrix, false)), EvalEngine.get());
        return null;
      }
      return solver.getInverse();
    }

    @Override
    public FieldMatrix<IExpr> matrixEval(FieldMatrix<IExpr> matrix) {
      return inverseMatrix(matrix);
    }

    @Override
    public RealMatrix realMatrixEval(RealMatrix matrix) {
      final org.hipparchus.linear.LUDecomposition lu =
          new org.hipparchus.linear.LUDecomposition(matrix);
      DecompositionSolver solver = lu.getSolver();
      if (!solver.isNonSingular()) {
        // Matrix `1` is singular.
        IOFunctions.printMessage(
            S.Inverse, "sing", F.List(Convert.matrix2List(matrix, false)), EvalEngine.get());
        return null;
      }
      return solver.getInverse();
    }
  }

  /**
   *
   *
   * <pre>
   * JacobiMatrix(matrix, var)
   * </pre>
   *
   * <blockquote>
   *
   * <p>creates a Jacobian matrix.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href="http://en.wikipedia.org/wiki/Jacobian">Wikipedia - Jacobian</a>
   * </ul>
   */
  private static class JacobiMatrix extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.arg1().isVector() >= 0) {
        IAST variables = F.NIL;
        if (ast.arg2().isSymbol()) {
          variables = F.List();
        } else if (ast.arg2().isList() && ast.arg2().size() > 1) {
          variables = (IAST) ast.arg2();
        }
        if (variables.isPresent()) {
          int variablesSize = variables.size();
          if (ast.arg1().isAST()) {
            IAST vector = (IAST) ast.arg1();
            int vectorSize = vector.size();
            IASTAppendable jacobiMatrix = F.ListAlloc(vectorSize);
            final IAST vars = variables;
            return jacobiMatrix.appendArgs(
                vectorSize,
                i -> {
                  IASTAppendable jacobiRow = F.ListAlloc(variablesSize);
                  return jacobiRow.appendArgs(variablesSize, j -> F.D(vector.get(i), vars.get(j)));
                });
          } else {
            FieldVector<IExpr> vector = Convert.list2Vector(ast.arg1());
            if (vector != null) {
              int vectorSize = vector.getDimension();
              IASTAppendable jacobiMatrix = F.ListAlloc(vectorSize);
              final IAST vars = variables;
              return jacobiMatrix.appendArgs(
                  vectorSize,
                  i -> {
                    IASTAppendable jacobiRow = F.ListAlloc(variablesSize);
                    return jacobiRow.appendArgs(
                        variablesSize, j -> F.D(vector.getEntry(i), vars.get(j)));
                  });
            }
          }
        }
      }

      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_2_2;
    }
  }

  /**
   *
   *
   * <pre>
   * LeastSquares(matrix, right)
   * </pre>
   *
   * <blockquote>
   *
   * <p>solves the linear least-squares problem 'matrix . x = right'.
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; LeastSquares(Table(Complex(i,Rational(2 * i + 2 + j, 1 + 9 * i + j)),{i,0,3},{j,0,2}), {1,1,1,1})
   * {-1577780898195/827587904419-I*11087326045520/827587904419,35583840059240/5793115330933+I*275839049310660/5793115330933,-3352155369084/827587904419-I*2832105547140/827587904419}
   * </pre>
   */
  private static class LeastSquares extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.arg1().isMatrix() != null && ast.arg2().isVector() >= 0) {
        IAST matrix = (IAST) ast.arg1().normal(false);
        IAST vector = (IAST) ast.arg2().normal(false);
        if (matrix.isList() && vector.isList()) {
          try {
            if (matrix.isNumericMode() || vector.isNumericMode()) {
              RealMatrix realMatrix = ast.arg1().toRealMatrix();
              if (realMatrix != null) {
                RealVector realVector = ast.arg2().toRealVector();
                if (realVector != null) {
                  // for numerical stability use: Dot(PseudoInverse(matrix), vector)
                  realMatrix = PseudoInverse.CONST.realMatrixEval(realMatrix);
                  if (realMatrix != null) {
                    return new ASTRealVector(realMatrix.operate(realVector), false);
                  }
                }
              }
              return F.NIL;
            }
          } catch (final MathRuntimeException mre) {
            // org.hipparchus.exception.MathIllegalArgumentException: inconsistent dimensions: 0 !=
            // 3
            return engine.printMessage(ast.topHead(), mre);
          } catch (final ClassCastException e) {
            if (FEConfig.SHOW_STACKTRACE) {
              e.printStackTrace();
            }
          } catch (final IndexOutOfBoundsException e) {
            if (FEConfig.SHOW_STACKTRACE) {
              e.printStackTrace();
            }
          }
          try {
            IAST matrixTransposed = (IAST) S.ConjugateTranspose.of(engine, matrix);
            return F.Expand(
                F.LinearSolve(
                    F.ConjugateTranspose(F.Dot(matrixTransposed, matrix)),
                    F.Dot(matrixTransposed, vector)));
          } catch (final ClassCastException e) {
            if (FEConfig.SHOW_STACKTRACE) {
              e.printStackTrace();
            }
          } catch (final IndexOutOfBoundsException e) {
            if (FEConfig.SHOW_STACKTRACE) {
              e.printStackTrace();
            }
          }
        }
      }

      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_2_2;
    }
  }

  /**
   *
   *
   * <pre>
   * LinearSolve(matrix, right)
   * </pre>
   *
   * <blockquote>
   *
   * <p>solves the linear equation system 'matrix . x = right' and returns one corresponding
   * solution <code>x</code>.
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; LinearSolve({{1, 1, 0}, {1, 0, 1}, {0, 1, 1}}, {1, 2, 3})
   * {0,1,2}
   * </pre>
   *
   * <p>Test the solution:
   *
   * <pre>
   * &gt;&gt; {{1, 1, 0}, {1, 0, 1}, {0, 1, 1}} . {0, 1, 2}
   * {1,2,3}
   * </pre>
   *
   * <p>If there are several solutions, one arbitrary solution is returned:
   *
   * <pre>
   * &gt;&gt; LinearSolve({{1, 2, 3}, {4, 5, 6}, {7, 8, 9}}, {1, 1, 1})
   * {-1,1,0}
   * </pre>
   *
   * <p>Infeasible systems are reported:
   *
   * <pre>
   * &gt;&gt; LinearSolve({{1, 2, 3}, {4, 5, 6}, {7, 8, 9}}, {1, -2, 3})
   *  : Linear equation encountered that has no solution.
   * LinearSolve({{1, 2, 3}, {4, 5, 6}, {7, 8, 9}}, {1, -2, 3})
   * </pre>
   *
   * <p>Argument {1, {2}} at position 1 is not a non-empty rectangular matrix.
   *
   * <pre>
   * &gt;&gt; LinearSolve({1, {2}}, {1, 2})
   * LinearSolve({1, {2}}, {1, 2})
   * &gt;&gt; LinearSolve({{1, 2}, {3, 4}}, {1, {2}})
   * LinearSolve({{1, 2}, {3, 4}}, {1, {2}})
   * </pre>
   */
  private static class LinearSolve extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      final int[] matrixDims = ast.arg1().isMatrix();
      if (matrixDims != null && ast.arg2().isVector() >= 0) {
        try {
          final FieldMatrix<IExpr> matrix = Convert.list2Matrix(ast.arg1());
          final FieldVector<IExpr> vector = Convert.list2Vector(ast.arg2());
          if (matrix != null && vector != null) {
            if (matrixDims[0] > matrixDims[1]) {
              if (vector.getDimension() == matrix.getRowDimension()
                  && vector.getDimension() <= matrix.getColumnDimension()) {
                return underdeterminedSystem(matrix, vector, engine);
              }
              return engine.printMessage("LinearSolve: first argument is not a square matrix.");
            }
            if (vector.getDimension() != matrix.getRowDimension()) {
              return engine.printMessage(
                  "LinearSolve: matrix row and vector have different dimensions.");
            }
            if (matrixDims[0] == 1 && matrixDims[1] >= 1) {
              IExpr temp = eval1x1Matrix(matrix, vector, engine);
              if (temp.isPresent()) {
                return temp;
              }
              return underdeterminedSystem(matrix, vector, engine);
            }
            if (matrixDims[0] == 2 && matrixDims[1] == 2) {
              IExpr temp = eval2x2Matrix(matrix, vector, engine);
              if (temp.isPresent()) {
                return temp;
              }
              return underdeterminedSystem(matrix, vector, engine);
            }
            if (matrixDims[0] == 3 && matrixDims[1] == 3) {
              IExpr temp = eval3x3Matrix(matrix, vector, engine);
              if (temp.isPresent()) {
                return temp;
              }
              return underdeterminedSystem(matrix, vector, engine);
            }
            if (matrixDims[0] != matrixDims[1]) {
              return underdeterminedSystem(matrix, vector, engine);
            }
            FieldDecompositionSolver<IExpr> solver =
                new FieldLUDecomposition<IExpr>(matrix).getSolver();
            if (solver.isNonSingular()) {
              FieldVector<IExpr> resultVector = solver.solve(vector);
              for (int i = 0; i < resultVector.getDimension(); i++) {
                if (resultVector.getEntry(i).isIndeterminate()
                    || //
                    resultVector.getEntry(i).isDirectedInfinity()) {
                  return underdeterminedSystem(matrix, vector, engine);
                }
              }
              return Convert.vector2List(resultVector);
            } else {
              return underdeterminedSystem(matrix, vector, engine);
            }
          }
        } catch (LimitException le) {
          throw le;
        } catch (final RuntimeException e) {
          if (FEConfig.SHOW_STACKTRACE) {
            e.printStackTrace();
          }
        }
      }

      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_2_2;
    }

    /**
     * For a underdetermined system, return one of the possible solutions through a row reduced
     * matrix.
     *
     * @param matrix
     * @param vector
     * @param engine
     * @return
     */
    private IExpr underdeterminedSystem(
        final FieldMatrix<IExpr> matrix, FieldVector<IExpr> vector, EvalEngine engine) {
      FieldMatrix<IExpr> augmentedMatrix = Convert.augmentedFieldMatrix(matrix, vector);
      if (augmentedMatrix != null) {
        return LinearAlgebra.rowReduced2List(augmentedMatrix, true, engine);
      }
      return F.NIL;
    }

    /**
     * Solve <code>matrix . vector</code> for a <code>1 x 1</code> matrix
     *
     * @param matrix
     * @param vector
     * @param engine
     * @return
     */
    private IExpr eval1x1Matrix(
        FieldMatrix<IExpr> matrix, FieldVector<IExpr> vector, EvalEngine engine) {
      IASTAppendable result = F.ListAlloc(matrix.getColumnDimension());
      IExpr a = matrix.getEntry(0, 0);
      IExpr x = vector.getEntry(0);
      // x/a
      IExpr row = F.Divide(x, a);
      result.append(row);
      if (matrix.getColumnDimension() > 1) {
        for (int i = 1; i < matrix.getColumnDimension(); i++) {
          result.append(F.C0);
        }
      }
      IExpr temp = engine.evalQuiet(result);
      if (temp.isAST()) {
        IAST ast = (IAST) temp;
        for (int k = 1; k < ast.size(); k++) {
          if (ast.get(k).isIndeterminate()
              || //
              ast.get(k).isDirectedInfinity()) {
            return F.NIL;
          }
        }
      }
      return temp;
    }

    /**
     * Solve <code>matrix . vector</code> for a <code>2 x 2</code> matrix
     *
     * @param matrix
     * @param vector
     * @param engine
     * @return
     */
    private IExpr eval2x2Matrix(
        FieldMatrix<IExpr> matrix, FieldVector<IExpr> vector, EvalEngine engine) {
      IASTAppendable result = F.ListAlloc(matrix.getColumnDimension());
      IExpr a = matrix.getEntry(0, 0);
      IExpr b = matrix.getEntry(0, 1);
      IExpr c = matrix.getEntry(1, 0);
      IExpr d = matrix.getEntry(1, 1);

      IExpr x = vector.getEntry(0);
      IExpr y = vector.getEntry(1);
      // (d*x-b*y)/((-b)*c+a*d)
      IExpr row =
          F.Times(
              F.Power(F.Plus(F.Times(F.CN1, b, c), F.Times(a, d)), -1),
              F.Plus(F.Times(d, x), F.Times(F.CN1, b, y)));
      result.append(row);
      // (c*x-a*y)/(b*c-a*d)
      row =
          F.Times(
              F.Power(F.Plus(F.Times(b, c), F.Times(F.CN1, a, d)), -1),
              F.Plus(F.Times(c, x), F.Times(F.CN1, a, y)));
      result.append(row);

      if (matrix.getColumnDimension() > 2) {
        for (int i = 2; i < matrix.getColumnDimension(); i++) {
          result.append(F.C0);
        }
      }
      IExpr temp = engine.evalQuiet(result);
      if (temp.isAST()) {
        IAST ast = (IAST) temp;
        for (int k = 1; k < ast.size(); k++) {
          if (ast.get(k).isIndeterminate()
              || //
              ast.get(k).isDirectedInfinity()) {
            return F.NIL;
          }
        }
      }
      return temp;
    }

    /**
     * Solve <code>matrix . vector</code> for a <code>3 x 3</code> matrix
     *
     * @param matrix
     * @param vector
     * @param engine
     * @return
     */
    private IExpr eval3x3Matrix(
        FieldMatrix<IExpr> matrix, FieldVector<IExpr> vector, EvalEngine engine) {
      IASTAppendable result = F.ListAlloc(matrix.getColumnDimension());
      IExpr a = matrix.getEntry(0, 0);
      IExpr b = matrix.getEntry(0, 1);
      IExpr c = matrix.getEntry(0, 2);
      IExpr d = matrix.getEntry(1, 0);
      IExpr e = matrix.getEntry(1, 1);
      IExpr f = matrix.getEntry(1, 2);
      IExpr g = matrix.getEntry(2, 0);
      IExpr h = matrix.getEntry(2, 1);
      IExpr i = matrix.getEntry(2, 2);

      IExpr x = vector.getEntry(0);
      IExpr y = vector.getEntry(1);
      IExpr z = vector.getEntry(2);
      // (f*h*x-e*i*x-c*h*y+b*i*y+c*e*z-b*f*z)/(c*e*g-b*f*g-c*d*h+a*f*h+b*d*i-a*e*i)
      IExpr row =
          F.Times(
              F.Power(
                  F.Plus(
                      F.Times(c, e, g),
                      F.Times(F.CN1, b, f, g),
                      F.Times(F.CN1, c, d, h),
                      F.Times(a, f, h),
                      F.Times(b, d, i),
                      F.Times(F.CN1, a, e, i)),
                  -1),
              F.Plus(
                  F.Times(f, h, x),
                  F.Times(F.CN1, e, i, x),
                  F.Times(F.CN1, c, h, y),
                  F.Times(b, i, y),
                  F.Times(c, e, z),
                  F.Times(F.CN1, b, f, z)));
      result.append(row);
      // ((-f)*g*x+d*i*x+c*g*y-a*i*y-c*d*z+a*f*z)/(c*e*g-b*f*g-c*d*h+a*f*h+b*d*i-a*e*i)
      row =
          F.Times(
              F.Power(
                  F.Plus(
                      F.Times(c, e, g),
                      F.Times(F.CN1, b, f, g),
                      F.Times(F.CN1, c, d, h),
                      F.Times(a, f, h),
                      F.Times(b, d, i),
                      F.Times(F.CN1, a, e, i)),
                  -1),
              F.Plus(
                  F.Times(F.CN1, f, g, x),
                  F.Times(d, i, x),
                  F.Times(c, g, y),
                  F.Times(F.CN1, a, i, y),
                  F.Times(F.CN1, c, d, z),
                  F.Times(a, f, z)));
      result.append(row);
      // (e*g*x-d*h*x-b*g*y+a*h*y+b*d*z-a*e*z)/(c*e*g-b*f*g-c*d*h+a*f*h+b*d*i-a*e*i)
      row =
          F.Times(
              F.Power(
                  F.Plus(
                      F.Times(c, e, g),
                      F.Times(F.CN1, b, f, g),
                      F.Times(F.CN1, c, d, h),
                      F.Times(a, f, h),
                      F.Times(b, d, i),
                      F.Times(F.CN1, a, e, i)),
                  -1),
              F.Plus(
                  F.Times(e, g, x),
                  F.Times(F.CN1, d, h, x),
                  F.Times(F.CN1, b, g, y),
                  F.Times(a, h, y),
                  F.Times(b, d, z),
                  F.Times(F.CN1, a, e, z)));
      result.append(row);

      if (matrix.getColumnDimension() > 3) {
        for (int k = 3; k < matrix.getColumnDimension(); k++) {
          result.append(F.C0);
        }
      }

      IExpr temp = engine.evalQuiet(result);
      if (temp.isAST()) {
        IAST ast = (IAST) temp;
        for (int k = 1; k < ast.size(); k++) {
          if (ast.get(k).isIndeterminate()
              || //
              ast.get(k).isDirectedInfinity()) {
            return F.NIL;
          }
        }
      }
      return temp;
    }
  }

  private static class LowerTriangularize extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      try {
        int[] dim = ast.arg1().isMatrix(false);
        if (dim == null) {
          // Argument `1` at position `2` is not a non-empty rectangular matrix.
          return IOFunctions.printMessage(
              ast.topHead(), "matrix", F.List(ast.arg1(), F.C1), engine);
        }

        final int k;
        if (ast.size() == 3) {
          k = Validate.checkIntType(ast, 2, Integer.MIN_VALUE);
        } else {
          k = 0;
        }
        int m = dim[0];
        int n = dim[1];
        FieldMatrix<IExpr> matrix = Convert.list2Matrix(ast.arg1());
        return F.matrix((i, j) -> i >= j - k ? matrix.getEntry(i, j) : F.C0, m, n);

      } catch (final ValidateException ve) {
        // int number validation
        return engine.printMessage(ast.topHead(), ve);
      }
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }
  }

  /**
   *
   *
   * <pre>
   * LUDecomposition(matrix)
   * </pre>
   *
   * <blockquote>
   *
   * <p>calculate the LUP-decomposition of a square <code>matrix</code>.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href="https://en.wikipedia.org/wiki/LU_decomposition">Wikipedia - LU decomposition</a>
   *   <li><a href=
   *       "http://commons.apache.org/proper/commons-math/apidocs/org/apache/commons/math3/linear/FieldLUDecomposition.html">Commons
   *       Math - Class FieldLUDecomposition</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt;&gt; LUDecomposition[{{1, 2, 3}, {3, 4, 11}, {13, 7, 8}}]
   * {{{1,0,0},
   *   {3,1,0},
   *   {13,19/2,1}},
   *  {{1,2,3},
   *   {0,-2,2},
   *   {0,0,-50}},{1,2,3}}
   * </pre>
   */
  private static class LUDecomposition extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      FieldMatrix<IExpr> matrix;
      boolean togetherMode = engine.isTogetherMode();
      try {
        engine.setTogetherMode(true);
        int[] dim = ast.arg1().isMatrix();
        if (dim != null && dim[0] > 0 && dim[1] > 0) {
          if (dim[0] != dim[1]) {
            // Argument `1` at position `2` is not a non-empty square matrix.
            return IOFunctions.printMessage(ast.topHead(), "matsq", F.List(), engine);
          }
          matrix = Convert.list2Matrix(ast.arg1());
          if (matrix != null) {
            final FieldLUDecomposition<IExpr> lu = new FieldLUDecomposition<IExpr>(matrix);
            final FieldMatrix<IExpr> lMatrix = lu.getL();
            final FieldMatrix<IExpr> uMatrix = lu.getU();
            final int[] iArr = lu.getPivot();
            // final int permutationCount = lu.getPermutationCount();
            int size = iArr.length;
            final IASTAppendable iList = F.ListAlloc(size);
            // +1 because in Symja the offset is +1 compared to java arrays
            iList.appendArgs(0, size, i -> F.ZZ(iArr[i] + 1));
            return F.List(Convert.matrix2List(lMatrix), Convert.matrix2List(uMatrix), iList);
          }
        }
      } catch (final ClassCastException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
      } catch (final IndexOutOfBoundsException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
      } finally {
        engine.setTogetherMode(togetherMode);
      }

      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_1;
    }
  }

  private static class MatrixExp extends AbstractFunctionEvaluator {
    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.size() == 2) {
        int[] dim = ast.arg1().isMatrix();
        if (dim != null && dim[0] == dim[1] && dim[0] > 0) {
          RealMatrix matrix = ast.arg1().toRealMatrix();
          if (matrix != null) {
            RealMatrix result = MatrixUtils.matrixExponential(matrix);
            return new ASTRealMatrix(result, false);
          }
        }
      }
      return F.NIL;
    }
  }

  /**
   *
   *
   * <pre>
   * MatrixMinimalPolynomial(matrix, var)
   * </pre>
   *
   * <blockquote>
   *
   * <p>computes the matrix minimal polynomial of a <code>matrix</code> for the variable <code>var
   * </code>.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href= "https://en.wikipedia.org/wiki/Minimal_polynomial_(linear_algebra)">Wikipedia -
   *       Minimal polynomial (linear algebra)</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; MatrixMinimalPolynomial({{1, -1, -1}, {1, -2, 1}, {0, 1, -3}}, x)
   * -1+x+4*x^2+x^3
   * </pre>
   */
  private static class MatrixMinimalPolynomial extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      int[] dimensions = ast.arg1().isMatrix(false);
      if (dimensions != null && dimensions[0] == dimensions[1] && dimensions[0] > 0) {
        // a matrix with square dimensions
        IExpr matrix = ast.arg1();
        IExpr variable = ast.arg2();
        if (!variable.isVariable()) {
          // `1` is not a valid variable.
          return IOFunctions.printMessage(ast.topHead(), "ivar", F.List(variable), engine);
        }
        ISymbol i = F.Dummy("i"); // new Symbol("§i", Context.SYSTEM);
        int n = 1;
        IAST qu = F.List();
        IAST mnm =
            (IAST)
                engine.evaluate(
                    F.List(F.Flatten(diagonalMatrix(new IExpr[] {F.C0, F.C1}, dimensions[0]))));
        if (!(mnm instanceof IASTAppendable)) {
          mnm = mnm.copyAppendable();
        }
        while (qu.isEmpty()) {
          ((IASTAppendable) mnm).append(engine.evaluate(F.Flatten(F.MatrixPower(matrix, F.ZZ(n)))));
          qu = (IAST) S.NullSpace.of(engine, F.Transpose(mnm));
          n++;
        }
        return S.Dot.of(
            engine, qu.arg1(), F.Table(F.Power(variable, i), F.List(i, F.C0, F.ZZ(--n))));
      }

      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_2_2;
    }
  }

  /**
   *
   *
   * <pre>
   * MatrixPower(matrix, n)
   * </pre>
   *
   * <blockquote>
   *
   * <p>computes the <code>n</code>th power of a <code>matrix</code>
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; MatrixPower({{1, 2}, {1, 1}}, 10)
   * {{3363,4756},
   *  {2378,3363}}
   *
   * &gt;&gt; MatrixPower({{1, 2}, {2, 5}}, -3)
   * {{169,-70},
   *  {-70,29}}
   * </pre>
   *
   * <p>Argument {{1, 0}, {0}} at position 1 is not a non-empty rectangular matrix.
   *
   * <pre>
   * &gt;&gt; MatrixPower({{1, 0}, {0}}, 2)
   * MatrixPower({{1, 0}, {0}}, 2)
   * </pre>
   */
  private static final class MatrixPower extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      FieldMatrix<IExpr> matrix;
      FieldMatrix<IExpr> resultMatrix;
      boolean togetherMode = engine.isTogetherMode();
      try {
        engine.setTogetherMode(true);
        int[] dimensions = ast.arg1().isMatrix(false);
        if (dimensions != null && dimensions[1] > 0 && dimensions[0] > 0) {
          matrix = Convert.list2Matrix(ast.arg1());
          if (matrix == null) {
            return F.NIL;
          }
          int p = ast.arg2().toIntDefault(Integer.MIN_VALUE);
          if (p == Integer.MIN_VALUE) {
            return F.NIL;
          }
          if (p == 1) {
            ((IAST) ast.arg1()).addEvalFlags(IAST.IS_MATRIX);
            return ast.arg1();
          }
          if (p == 0) {
            resultMatrix =
                new BlockFieldMatrix<IExpr>(
                    ExprField.CONST, matrix.getRowDimension(), matrix.getColumnDimension());
            int min = matrix.getRowDimension();
            if (min > matrix.getColumnDimension()) {
              min = matrix.getColumnDimension();
            }
            for (int i = 0; i < min; i++) {
              resultMatrix.setEntry(i, i, F.C1);
            }

            return Convert.matrix2List(resultMatrix);
          }
          int iterationLimit = engine.getIterationLimit();
          if (iterationLimit >= 0 && iterationLimit <= p) {
            IterationLimitExceeded.throwIt(p, ast);
          }

          if (p < 0) {
            resultMatrix = Inverse.inverseMatrix(matrix);
            matrix = resultMatrix;
            p *= (-1);
          } else {
            resultMatrix = matrix;
          }
          for (int i = 1; i < p; i++) {
            resultMatrix = resultMatrix.multiply(matrix);
          }
          return Convert.matrix2List(resultMatrix);
        }
        return F.NIL;
      } catch (final RuntimeException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
        return engine.printMessage(ast.topHead(), e);
      } finally {
        engine.setTogetherMode(togetherMode);
      }
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_2_2;
    }
  }

  /**
   *
   *
   * <pre>
   * MatrixRank(matrix)
   * </pre>
   *
   * <blockquote>
   *
   * <p>returns the rank of <code>matrix</code>.
   *
   * </blockquote>
   *
   * <p>See:
   *
   * <ul>
   *   <li><a href= "https://en.wikipedia.org/wiki/Rank_%28linear_algebra%29">Wikipedia - Rank
   *       (linear algebra</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; MatrixRank({{1, 2, 3}, {4, 5, 6}, {7, 8, 9}})
   * 2
   * &gt;&gt; MatrixRank({{1, 1, 0}, {1, 0, 1}, {0, 1, 1}})
   * 3
   * &gt;&gt; MatrixRank({{a, b}, {3 a, 3 b}})
   * 1
   * </pre>
   *
   * <p>Argument <code>{{1, 0}, {0}}</code> at position <code>1</code> is not a non-empty
   * rectangular matrix.
   *
   * <pre>
   * &gt;&gt; MatrixRank({{1, 0}, {0}})
   * MatrixRank({{1, 0}, {0}})
   * </pre>
   */
  private static final class MatrixRank extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      FieldMatrix<IExpr> matrix;
      try {
        IExpr arg1 = engine.evaluate(ast.arg1());
        if (arg1.isMatrix() != null) {
          matrix = Convert.list2Matrix(arg1);
          if (matrix != null) {
            FieldReducedRowEchelonForm fmw = new FieldReducedRowEchelonForm(matrix);
            return F.ZZ(fmw.getMatrixRank());
          }
        }

      } catch (final ClassCastException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
      } catch (final IndexOutOfBoundsException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
      }

      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_1;
    }

    @Override
    public void setUp(final ISymbol newSymbol) {
      newSymbol.setAttributes(ISymbol.HOLDALL);
    }
  }

  /**
   *
   *
   * <pre>
   * Norm(m, l)
   * </pre>
   *
   * <blockquote>
   *
   * <p>computes the <code>l</code>-norm of matrix <code>m</code> (currently only works for
   * vectors!).<br>
   *
   * </blockquote>
   *
   * <pre>
   * Norm(m)
   * </pre>
   *
   * <blockquote>
   *
   * <p>computes the 2-norm of matrix <code>m</code> (currently only works for vectors!).<br>
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; Norm({1, 2, 3, 4}, 2)
   * Sqrt(30)
   *
   * &gt;&gt; Norm({10, 100, 200}, 1)
   * 310
   *
   * &gt;&gt; Norm({a, b, c})
   * Sqrt(Abs(a)^2+Abs(b)^2+Abs(c)^2)
   *
   * &gt;&gt; Norm({-100, 2, 3, 4}, Infinity)
   * 100
   *
   * &gt;&gt; Norm(1 + I)
   * Sqrt(2)
   * </pre>
   *
   * <p>The first Norm argument should be a number, vector, or matrix.<br>
   *
   * <pre>
   * &gt;&gt; Norm({1, {2, 3}})
   * Norm({1, {2, 3}})
   *
   * &gt;&gt; Norm({x, y})
   * Sqrt(Abs(x)^2+Abs(y)^2)
   *
   * &gt;&gt; Norm({x, y}, p)
   * (Abs(x) ^ p + Abs(y) ^ p) ^ (1 / p)
   * </pre>
   *
   * <p>The second argument of Norm, 0, should be a symbol, Infinity, or an integer or real number
   * not less than 1 for vector p-norms; or 1, 2, Infinity, or &ldquo;Frobenius&rdquo; for matrix
   * norms.<br>
   *
   * <pre>
   * &gt;&gt; Norm({x, y}, 0)
   * Norm({x, y}, 0)
   * </pre>
   *
   * <p>The second argument of Norm, 0.5, should be a symbol, Infinity, or an integer or real number
   * not less than 1 for vector p-norms; or 1, 2, Infinity, or &ldquo;Frobenius&rdquo; for matrix
   * norms.
   *
   * <pre>
   * &gt;&gt; Norm({x, y}, 0.5)
   * Norm({x, y}, 0.5)
   *
   * &gt;&gt; Norm({})
   * Norm({})
   *
   * &gt;&gt; Norm(0)
   * 0
   * </pre>
   */
  private static final class Norm extends AbstractEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      IExpr arg1 = ast.arg1();
      int dim = arg1.isVector();
      if (dim > (-1)) {
        if (dim == 0) {
          // The first Norm argument should be a scalar, vector or matrix.
          return IOFunctions.printMessage(ast.topHead(), "nvm", F.List(), engine);
        }
        arg1 = arg1.normal(false);
        if (arg1.isList()) {
          IAST vector = (IAST) arg1;
          if (ast.isAST2()) {
            IExpr p = ast.arg2();
            if (p.isInfinity()) {
              return vector.map(S.Max, x -> F.Abs(x));
            } else {
              if (p.isSymbol() || p.isReal()) {
                if (p.isZero()) {
                  engine.printMessage("Norm: 0 not allowed as second argument!");
                  return F.NIL;
                }
                if (p.isReal() && p.lessThan(F.C1).isTrue()) {
                  engine.printMessage("Norm: Second argument is < 1!");
                  return F.NIL;
                }
                return F.Power(vector.map(S.Plus, x -> F.Power(F.Abs(x), p)), p.inverse());
              }
            }
            return F.NIL;
          }
          return F.Sqrt(vector.map(S.Plus, x -> F.Sqr(F.Abs(x))));
        }
        return F.NIL;
      }
      int[] matrixDim = arg1.isMatrix();
      if (matrixDim != null) {
        if (matrixDim[1] == 0) {
          // The first Norm argument should be a scalar, vector or matrix.
          return IOFunctions.printMessage(ast.topHead(), "nvm", F.List(), engine);
        }
        RealMatrix matrix;
        try {
          matrix = arg1.toRealMatrix();
          if (matrix != null) {
            if (ast.size() > 2 && ast.arg2().isString("Frobenius")) {
              return F.Norm(F.Flatten(arg1));
            }
            if (matrixDim[0] < matrixDim[1]) {
              int d = matrixDim[0];
              matrixDim[0] = matrixDim[1];
              matrixDim[1] = d;
              matrix = matrix.transpose();
            }
            final org.hipparchus.linear.SingularValueDecomposition svd =
                new org.hipparchus.linear.SingularValueDecomposition(matrix);
            RealMatrix sSVD = svd.getS();
            IASTAppendable result = F.ast(S.Max, matrixDim[1], false);
            for (int i = 0; i < matrixDim[1]; i++) {
              result.append(F.num(sSVD.getEntry(i, i)));
            }
            return result;
          }

        } catch (final ValidateException ve) {
          if (FEConfig.SHOW_STACKTRACE) {
            ve.printStackTrace();
          }
          return engine.printMessage(ast.topHead(), ve);
        } catch (final IndexOutOfBoundsException e) {
          if (FEConfig.SHOW_STACKTRACE) {
            e.printStackTrace();
          }
        }
      }
      if (arg1.isNumber()) {
        if (ast.isAST2()) {
          return F.NIL;
        }
        // absolute Value of a number
        return ((INumber) arg1).abs();
      }
      if (arg1.isNumericFunction(true) && !arg1.isList()) {
        if (ast.isAST2()) {
          return F.NIL;
        }
        // absolute Value
        return F.Abs(arg1);
      }
      // The first Norm argument should be a scalar, vector or matrix.
      return IOFunctions.printMessage(ast.topHead(), "nvm", F.List(), engine);
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }
  }

  /**
   *
   *
   * <pre>
   * Normalize(v)
   * </pre>
   *
   * <blockquote>
   *
   * <p>calculates the normalized vector <code>v</code>.
   *
   * </blockquote>
   *
   * <pre>
   * Normalize(z)
   * </pre>
   *
   * <blockquote>
   *
   * <p>calculates the normalized complex number <code>z</code>.
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; Normalize({1, 1, 1, 1})
   * {1/2,1/2,1/2,1/2}
   *
   * &gt;&gt; Normalize(1 + I)
   * (1+I)/Sqrt(2)
   *
   * &gt;&gt; Normalize(0)
   * 0
   *
   * &gt;&gt; Normalize({0})
   * {0}
   *
   * &gt;&gt; Normalize({})
   * {}
   * </pre>
   */
  private static final class Normalize extends AbstractEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      IExpr normFunction = S.Norm;
      if (ast.isAST2()) {
        normFunction = ast.arg2();
      }
      IExpr arg1 = ast.arg1();
      if (arg1.isEmptyList() && ast.isAST1()) {
        return arg1;
      }
      IExpr norm = engine.evaluate(F.unaryAST1(normFunction, ast.arg1()));
      if (norm.isZero()) {
        return arg1;
      }
      return F.Divide(ast.arg1(), norm);
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }
  }

  /**
   *
   *
   * <pre>
   * NullSpace(matrix)
   * </pre>
   *
   * <blockquote>
   *
   * <p>returns a list of vectors that span the nullspace of the <code>matrix</code>.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href= "http://en.wikipedia.org/wiki/Kernel_%28linear_algebra%29">Wikipedia - Kernel
   *       (linear algebra)</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; NullSpace({{1,0,-3,0,2,-8},{0,1,5,0,-1,4},{0,0,0,1,7,-9},{0,0,0,0,0,0}})
   * {{3,-5,1,0,0,0},
   *  {-2,1,0,-7,1,0},
   *  {8,-4,0,9,0,1}}
   * </pre>
   *
   * <pre>
   * ```
   * &gt;&gt; NullSpace({{1, 2, 3}, {4, 5, 6}, {7, 8, 9}})
   * {{1,-2,1}}
   *
   * &gt;&gt; A = {{1, 1, 0}, {1, 0, 1}, {0, 1, 1}}
   * &gt;&gt; NullSpace(A)
   * {}
   *
   * &gt;&gt; MatrixRank(A)
   * 3
   * </pre>
   *
   * <p>Argument {1, {2}} at position 1 is not a non-empty rectangular matrix.<br>
   *
   * <pre>
   * &gt;&gt; NullSpace({1, {2}})
   * NullSpace({1, {2}})
   * </pre>
   */
  private static class NullSpace extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      FieldMatrix<IExpr> matrix;
      boolean togetherMode = engine.isTogetherMode();
      try {
        engine.setTogetherMode(true);
        int[] dims = ast.arg1().isMatrix();
        if (dims != null) {
          matrix = Convert.list2Matrix(ast.arg1());
          if (matrix != null) {
            FieldReducedRowEchelonForm fmw = new FieldReducedRowEchelonForm(matrix);
            FieldMatrix<IExpr> nullspace = fmw.getNullSpace(F.CN1);
            if (nullspace == null) {
              return F.List();
            }

            IASTMutable list2 = Convert.matrix2List(nullspace);
            // rows in descending orders
            EvalAttributes.sort(list2, Comparators.REVERSE_CANONICAL_COMPARATOR);
            return list2;
          }
        }
      } catch (final MathRuntimeException mrex) {
        // org.hipparchus.exception.MathIllegalArgumentException
        if (FEConfig.SHOW_STACKTRACE) {
          mrex.printStackTrace();
        }
      } catch (final ClassCastException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
      } catch (final IndexOutOfBoundsException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
      } finally {
        engine.setTogetherMode(togetherMode);
      }

      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_1;
    }
  }

  private static class Orthogonalize extends AbstractEvaluator {

    static IBuiltInSymbol oneStep =
        F.localBiFunction(
            "oneStep",
            (vec, vecmat) -> {
              if (vecmat.isEmptyList()) {
                return vec;
              }
              IExpr function = // [$ (#1-(vec.#2)/(#2.#2)*#2)& $]
                  F.Function(
                      F.Plus(
                          F.Slot1,
                          F.Times(
                              F.CN1,
                              F.Dot(vec, F.Slot2),
                              F.Power(F.Dot(F.Slot2, F.Slot2), F.CN1),
                              F.Slot2))); // $$;
              return F.eval(F.Fold(function, vec, vecmat));
            });

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      IExpr arg1 = ast.arg1();
      int[] dim = arg1.isMatrix();
      if (dim != null) {
        // Gram-Schmidt orthogonalization
        IExpr result =
            F.Map(
                F.Function(F.Normalize(F.Slot1)), //
                F.Fold(
                    F.Function(F.Append(F.Slot1, F.binaryAST2(oneStep, F.Slot2, F.Slot1))),
                    F.List(),
                    arg1));

        return engine.evaluate(result);
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }

    @Override
    public void setUp(final ISymbol newSymbol) {}
  }

  /**
   *
   *
   * <pre>
   * Projection(vector1, vector2)
   * </pre>
   *
   * <blockquote>
   *
   * <p>Find the orthogonal projection of <code>vector1</code> onto another <code>vector2</code>.
   *
   * </blockquote>
   *
   * <pre>
   * Projection(vector1, vector2, ipf)
   * </pre>
   *
   * <blockquote>
   *
   * <p>Find the orthogonal projection of <code>vector1</code> onto another <code>vector2</code>
   * using the inner product function <code>ipf</code>.
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; Projection({5, I, 7}, {1, 1, 1})
   * {4+I*1/3,4+I*1/3,4+I*1/3}
   * </pre>
   */
  private static class Projection extends AbstractEvaluator {

    /**
     * Create a binary function &quot"dot product&quot; <code>head(u,v)</code> and evaluate it.
     *
     * @param head the header of the binary function
     * @param u the first argument of the function
     * @param v the second argument of the function
     * @param engine the evaluation engine
     * @return the evaluated <code>head(u,v)</code> AST.
     */
    private static IExpr dotProduct(IExpr head, IExpr u, IExpr v, EvalEngine engine) {
      return engine.evaluate(F.binaryAST2(head, u, v));
    }

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      int dim1 = ast.arg1().isVector();
      int dim2 = ast.arg2().isVector();
      try {
        if (ast.size() == 4) {
          IExpr head = ast.arg3();
          if (dim1 >= 0 && dim1 == dim2) {
            if (dim1 == 0) {
              return F.CEmptyList;
            }
            if (head.equals(S.Dot)) {
              FieldVector<IExpr> u = Convert.list2Vector(ast.arg1());
              FieldVector<IExpr> v = Convert.list2Vector(ast.arg2());
              if (u != null && v != null) {
                return Convert.vector2List(u.projection(v));
              }
            }
          }
          IExpr u = ast.arg1();
          IExpr v = ast.arg2();
          return v.times(dotProduct(head, u, v, engine).divide(dotProduct(head, v, v, engine)));
        }
        if (dim1 >= 0 && dim1 == dim2) {
          if (dim1 == 0) {
            return F.CEmptyList;
          }
          FieldVector<IExpr> u = Convert.list2Vector(ast.arg1());
          FieldVector<IExpr> v = Convert.list2Vector(ast.arg2());
          FieldVector<IExpr> vConjugate = v.copy();
          for (int i = 0; i < dim2; i++) {
            vConjugate.setEntry(i, vConjugate.getEntry(i).conjugate());
          }

          return Convert.vector2List(
              v.mapMultiply(u.dotProduct(vConjugate).divide(v.dotProduct(vConjugate))));
        }
      } catch (ValidateException ve) {
        return engine.printMessage(ast.topHead(), ve);
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_2_3;
    }

    @Override
    public void setUp(final ISymbol newSymbol) {}
  }

  /**
   *
   *
   * <pre>
   * PseudoInverse(matrix)
   * </pre>
   *
   * <blockquote>
   *
   * <p>computes the Moore-Penrose pseudoinverse of the <code>matrix</code>. If <code>matrix</code>
   * is invertible, the pseudoinverse equals the inverse.
   *
   * </blockquote>
   *
   * <p>See:
   *
   * <ul>
   *   <li><a href= "https://en.wikipedia.org/wiki/Moore%E2%80%93Penrose_pseudoinverse">Wikipedia:
   *       Moore-Penrose pseudoinverse</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; PseudoInverse({{1, 2}, {2, 3}, {3, 4}})
   * {{1.0000000000000002,2.000000000000001},
   *  {1.9999999999999976,2.999999999999996},
   *  {3.000000000000001,4.0}}
   * &gt;&gt; PseudoInverse({{1, 2, 0}, {2, 3, 0}, {3, 4, 1}})
   * {{-2.999999999999998,1.9999999999999967,4.440892098500626E-16},
   *  {1.999999999999999,-0.9999999999999982,-2.7755575615628914E-16},
   *  {0.9999999999999999,-1.9999999999999991,1.0}}
   * &gt;&gt; PseudoInverse({{1.0, 2.5}, {2.5, 1.0}})
   * {{-0.19047619047619038,0.47619047619047616},
   *  {0.47619047619047616,-0.1904761904761904}}
   * </pre>
   *
   * <p>Argument {1, {2}} at position 1 is not a non-empty rectangular matrix.
   *
   * <pre>
   * &gt;&gt; PseudoInverse({1, {2}})
   * PseudoInverse({1, {2}})
   * </pre>
   */
  private static final class PseudoInverse extends AbstractMatrix1Matrix {
    protected static final PseudoInverse CONST = new PseudoInverse();

    @Override
    public int[] checkMatrixDimensions(IExpr arg1) {
      return Convert.checkNonEmptyRectangularMatrix(S.PseudoInverse, arg1);
    }

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      return numericEval(ast, engine);
    }

    @Override
    public FieldMatrix<IExpr> matrixEval(FieldMatrix<IExpr> matrix) {
      return null;
    }

    @Override
    public RealMatrix realMatrixEval(RealMatrix matrix) {
      final org.hipparchus.linear.SingularValueDecomposition lu =
          new org.hipparchus.linear.SingularValueDecomposition(matrix);
      DecompositionSolver solver = lu.getSolver();
      return solver.getInverse();
    }
  }

  /**
   *
   *
   * <pre>
   * QRDecomposition(matrix)
   * </pre>
   *
   * <blockquote>
   *
   * <p>computes the QR decomposition of the <code>matrix</code>.
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; QRDecomposition({{1, 2}, {3, 4}, {5, 6}})
   * {
   * {{-0.16903085094570325,0.8970852271450604,0.4082482904638636},
   *  {-0.50709255283711,0.2760262237369421,-0.8164965809277258},
   *  {-0.8451542547285165,-0.3450327796711773,0.40824829046386274}},
   * {{-5.916079783099616,-7.437357441610944},
   *  {0.0,0.828078671210824},
   *  {0.0,0.0}}}
   * </pre>
   */
  private static class QRDecomposition extends AbstractMatrix1Expr {

    @Override
    public int[] checkMatrixDimensions(IExpr arg1) {
      return Convert.checkNonEmptyRectangularMatrix(S.QRDecomposition, arg1);
    }

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      // FieldMatrix<IExpr> matrix;
      try {

        int[] dim = ast.arg1().isMatrix();
        if (dim != null) {
          // if (dim[0] == 1 && dim[1] == 1) {
          // }

          // if (dim[0] == 2 && dim[1] == 2) {
          // matrix = Convert.list2Matrix((IAST) ast.arg1());
          // if (matrix != null) {
          // }
          // }

        }

      } catch (final ClassCastException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
      } catch (final IndexOutOfBoundsException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
      }

      // switch to numeric calculation
      return numericEval(ast, engine);
    }

    @Override
    public IExpr matrixEval(FieldMatrix<IExpr> matrix) {
      return F.NIL;
    }

    @Override
    public IAST realMatrixEval(RealMatrix matrix) {
      org.hipparchus.linear.QRDecomposition ed = new org.hipparchus.linear.QRDecomposition(matrix);
      if (Convert.realMatrix2List(ed.getQ()) != null
          && Convert.realMatrix2List(ed.getR()) != null) {
        return F.List(Convert.realMatrix2List(ed.getQ()), Convert.realMatrix2List(ed.getR()));
      }
      return F.NIL;
    }
  }

  private static class RiccatiSolve extends AbstractEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.arg1().argSize() == 2
          && ast.arg1().isListOfMatrices()
          && //
          ast.arg2().argSize() == 2
          && ast.arg2().isListOfMatrices()) {
        try {
          IAST list1 = (IAST) ast.arg1();
          IAST list2 = (IAST) ast.arg2();
          RealMatrix A = list1.arg1().toRealMatrix();
          if (A != null) {
            RealMatrix B = list1.arg2().toRealMatrix();
            if (B != null) {
              RealMatrix Q = list2.arg1().toRealMatrix();
              if (Q != null) {
                RealMatrix R = list2.arg2().toRealMatrix();
                if (R != null) {
                  RiccatiEquationSolver solver = new RiccatiEquationSolverImpl(A, B, Q, R);
                  RealMatrix result = solver.getP();
                  return new ASTRealMatrix(result, false);
                }
              }
            }
          }
        } catch (MathRuntimeException mrex) {
          IOFunctions.printMessage(ast.topHead(), mrex, engine);
        }
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_2_2;
    }
  }

  /**
   *
   *
   * <pre>
   * RowReduce(matrix)
   * </pre>
   *
   * <blockquote>
   *
   * <p>returns the reduced row-echelon form of <code>matrix</code>.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href="http://en.wikipedia.org/wiki/Row_echelon_form">Wikipedia - Row echelon form</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; RowReduce({{1,1,0,1,5},{1,0,0,2,2},{0,0,1,4,-1},{0,0,0,0,0}})
   * {{1,0,0,2,2},
   *  {0,1,0,-1,3},
   *  {0,0,1,4,-1},
   *  {0,0,0,0,0}}
   *
   * &gt;&gt; RowReduce({{1, 0, a}, {1, 1, b}})
   * {{1,0,a},
   *  {0,1,-a+b}}
   *
   * &gt;&gt; RowReduce({{1, 2, 3}, {4, 5, 6}, {7, 8, 9}})
   * {{1,0,-1},
   *  {0,1,2},
   *  {0,0,0}}
   * </pre>
   *
   * <p>Argument {{1, 0}, {0}} at position 1 is not a non-empty rectangular matrix.<br>
   *
   * <pre>
   * &gt;&gt; RowReduce({{1, 0}, {0}})
   * RowReduce({{1, 0}, {0}})
   * </pre>
   */
  private static class RowReduce extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      FieldMatrix<IExpr> matrix;
      boolean togetherMode = engine.isTogetherMode();
      try {
        engine.setTogetherMode(true);
        int[] dims = ast.arg1().isMatrix();
        if (dims != null) {
          matrix = Convert.list2Matrix(ast.arg1());
          if (matrix != null) {
            FieldReducedRowEchelonForm fmw = new FieldReducedRowEchelonForm(matrix);
            return Convert.matrix2List(fmw.getRowReducedMatrix());
          }
        }
      } catch (final ClassCastException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
      } catch (final IndexOutOfBoundsException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
      } finally {
        engine.setTogetherMode(togetherMode);
      }

      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_1;
    }
  }

  /**
   *
   *
   * <pre>
   * SingularValueDecomposition(matrix)
   * </pre>
   *
   * <blockquote>
   *
   * <p>calculates the singular value decomposition for the <code>matrix</code>.
   *
   * </blockquote>
   *
   * <p>'SingularValueDecomposition' returns <code>u</code>, <code>s</code>, <code>w</code> such
   * that <code>matrix =u s v</code>, <code>u' u</code>=1, <code>v' v</code>=1, and <code>s</code>
   * is diagonal.
   *
   * <p>See:
   *
   * <ul>
   *   <li><a href= "https://en.wikipedia.org/wiki/Singular_value_decomposition">Wikipedia: Singular
   *       value decomposition</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; SingularValueDecomposition({{1.5, 2.0}, {2.5, 3.0}})
   * {
   * {{0.5389535334972082,0.8423354965397538},
   *  {0.8423354965397537,-0.5389535334972083}},
   * {{4.635554529660638,0.0},
   *  {0.0,0.10786196059193007}},
   * {{0.6286775450376476,-0.7776660879615599},
   *  {0.7776660879615599,0.6286775450376476}}}
   * </pre>
   *
   * <p>Symbolic SVD is not implemented, performing numerically.
   *
   * <pre>
   * &gt;&gt; SingularValueDecomposition({{3/2, 2}, {5/2, 3}})
   * {
   * {{0.5389535334972082,0.8423354965397538},
   *  {0.8423354965397537,-0.5389535334972083}},
   * {{4.635554529660638,0.0},
   *  {0.0,0.10786196059193007}},
   * {{0.6286775450376476,-0.7776660879615599},
   *  {0.7776660879615599,0.6286775450376476}}}
   * </pre>
   *
   * <p>Argument {1, {2}} at position 1 is not a non-empty rectangular matrix.
   *
   * <pre>
   * &gt;&gt; SingularValueDecomposition({1, {2}})
   * SingularValueDecomposition({1, {2}})
   * </pre>
   */
  private static final class SingularValueDecomposition extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      RealMatrix matrix;
      try {

        matrix = ast.arg1().toRealMatrix();
        if (matrix != null) {
          final org.hipparchus.linear.SingularValueDecomposition svd =
              new org.hipparchus.linear.SingularValueDecomposition(matrix);
          return F.List(
              new ASTRealMatrix(svd.getU(), false),
              new ASTRealMatrix(svd.getS(), false),
              new ASTRealMatrix(svd.getV(), false));
        }

      } catch (final ValidateException ve) {
        if (FEConfig.SHOW_STACKTRACE) {
          ve.printStackTrace();
        }
        return engine.printMessage(ast.topHead(), ve);
      } catch (final IndexOutOfBoundsException e) {
        if (FEConfig.SHOW_STACKTRACE) {
          e.printStackTrace();
        }
      }

      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_1;
    }
  }

  private static class ToeplitzMatrix extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.isAST2()) {
        if (ast.arg1().isList() && ast.arg2().isList()) {
          IAST vector1 = (IAST) ast.arg1();
          IAST vector2 = (IAST) ast.arg2();
          int numberOfRows = vector1.argSize();
          int numberOfColumns = vector2.argSize();
          return F.matrix(
              (i, j) -> i <= j ? vector2.get(j - i + 1) : vector1.get(i - j + 1),
              numberOfRows,
              numberOfColumns);
        }

        return F.NIL;
      }
      if (ast.arg1().isList()) {
        IAST vector = (IAST) ast.arg1();
        int m = vector.argSize();
        return F.matrix((i, j) -> i <= j ? vector.get(j - i + 1) : vector.get(i - j + 1), m, m);
      }

      if (ast.arg1().isInteger()) {
        int m = ast.arg1().toIntDefault(Integer.MIN_VALUE);
        if (m < 0) {
          // Positive integer (less equal 2147483647) expected at position `2` in `1`.
          return IOFunctions.printMessage(S.ToeplitzMatrix, "intpm", F.List(ast, F.C1), engine);
        }
        int[] count = new int[1];
        count[0] = 1;
        return F.matrix((i, j) -> i <= j ? F.ZZ(j - i + 1) : F.ZZ(i - j + 1), m, m);
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }
  }

  /**
   *
   *
   * <pre>
   * ToPolarCoordinates({x, y})
   * </pre>
   *
   * <blockquote>
   *
   * <p>return the polar coordinates for the cartesian coordinates <code>{x, y}</code>.
   *
   * </blockquote>
   *
   * <pre>
   * ToPolarCoordinates({x, y, z})
   * </pre>
   *
   * <blockquote>
   *
   * <p>return the polar coordinates for the cartesian coordinates <code>{x, y, z}</code>.
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; ToPolarCoordinates({x, y})
   * {Sqrt(x^2+y^2),ArcTan(x,y)}
   *
   * &gt;&gt; ToPolarCoordinates({x, y, z})
   * {Sqrt(x^2+y^2+z^2),ArcCos(x/Sqrt(x^2+y^2+z^2)),ArcTan(y,z)}
   * </pre>
   */
  private static final class ToPolarCoordinates extends AbstractEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      final IExpr arg1 = ast.arg1();
      int dim = arg1.isVector();
      if (dim > 0) {
        if (arg1.isAST()) {
          IAST list = (IAST) arg1;
          if (dim == 2) {
            IExpr x = list.arg1();
            IExpr y = list.arg2();
            return F.list(F.Sqrt(F.Plus(F.Sqr(x), F.Sqr(y))), F.ArcTan(x, y));
          } else if (dim == 3) {
            IExpr x = list.arg1();
            IExpr y = list.arg2();
            IExpr z = list.arg3();
            IAST sqrtExpr = F.Sqrt(F.Plus(F.Sqr(x), F.Sqr(y), F.Sqr(z)));
            return F.list(sqrtExpr, F.ArcCos(F.Divide(x, sqrtExpr)), F.ArcTan(y, z));
          }
        } else {
          FieldVector<IExpr> vector = Convert.list2Vector(arg1);
          if (dim == 2) {
            IExpr x = vector.getEntry(0);
            IExpr y = vector.getEntry(1);
            return F.list(F.Sqrt(F.Plus(F.Sqr(x), F.Sqr(y))), F.ArcTan(x, y));
          } else if (dim == 3) {
            IExpr x = vector.getEntry(0);
            IExpr y = vector.getEntry(1);
            IExpr z = vector.getEntry(2);
            IAST sqrtExpr = F.Sqrt(F.Plus(F.Sqr(x), F.Sqr(y), F.Sqr(z)));
            return F.list(sqrtExpr, F.ArcCos(F.Divide(x, sqrtExpr)), F.ArcTan(y, z));
          }
        }
      } else if (arg1.isList()) {
        IAST list = (IAST) arg1;
        return list.mapThreadEvaled(engine, F.ListAlloc(list.size()), ast, 1);
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_1;
    }

    @Override
    public void setUp(final ISymbol newSymbol) {}
  }

  /**
   *
   *
   * <pre>
   * Tr(matrix)
   * </pre>
   *
   * <blockquote>
   *
   * <p>computes the trace of the <code>matrix</code>.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href="http://en.wikipedia.org/wiki/Trace_matrix">Wikipedia - Trace (linear
   *       algebra)</a><br>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; Tr({{1, 2, 3}, {4, 5, 6}, {7, 8, 9}})
   * 15
   * </pre>
   *
   * <p>Symbolic trace:
   *
   * <pre>
   * &gt;&gt; Tr({{a, b, c}, {d, e, f}, {g, h, i}})
   * a+e+i
   * </pre>
   */
  private static class Tr extends AbstractEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      // TODO generalize for tensors
      IExpr arg1 = ast.arg1();
      IExpr header = S.Plus;
      if (ast.size() > 2) {
        header = ast.arg2();
      }

      final int[] dim = arg1.isMatrix();
      if (dim != null) {
        if (arg1.isAST()) {
          final IAST mat = (IAST) arg1;
          int len = dim[0] < dim[1] ? dim[0] : dim[1];
          IASTAppendable tr = F.ast(header, len, true);
          mat.forEach(1, len + 1, (x, i) -> tr.set(i, ((IAST) x).get(i)));
          return tr;
        } else {
          FieldMatrix<IExpr> matrix = Convert.list2Matrix(arg1);
          if (matrix != null) {
            int len = dim[0] < dim[1] ? dim[0] : dim[1];
            IASTAppendable tr = F.ast(header, len, true);
            for (int i = 0; i < len; i++) {
              tr.set(i + 1, matrix.getEntry(i, i));
            }
            return tr;
          }
        }
      }

      final int len = arg1.isVector();
      if (len >= 0) {
        if (arg1.isAST()) {
          IASTAppendable tr = F.ast(header, len, true);
          ((IAST) arg1).forEach(1, len + 1, (x, i) -> tr.set(i, x));
          return tr;
        } else {
          FieldVector<IExpr> vector = Convert.list2Vector(arg1);
          if (vector != null) {
            IASTAppendable tr = F.ast(header, len, true);
            for (int i = 0; i < len; i++) {
              tr.set(i + 1, vector.getEntry(i));
            }
            return tr;
          }
        }
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }

    @Override
    public void setUp(final ISymbol newSymbol) {}
  }

  /**
   *
   *
   * <pre>
   * Transpose(m)
   * </pre>
   *
   * <blockquote>
   *
   * <p>transposes rows and columns in the matrix <code>m</code>.
   *
   * </blockquote>
   *
   * <p>See:
   *
   * <ul>
   *   <li><a href="https://en.wikipedia.org/wiki/Transpose">Wikipedia - Transpose</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; Transpose({{1, 2, 3}, {4, 5, 6}})
   * {{1, 4}, {2, 5}, {3, 6}}
   * &gt;&gt; MatrixForm(%)
   * 1   4
   * 2   5
   * 3   6
   *
   * &gt;&gt; Transpose(x)
   * Transpose(x)
   * </pre>
   */
  private static class Transpose extends AbstractEvaluator {

    private static class TransposePermute {
      /** The current tensor. */
      final IAST tensor;
      /** The dimensions of the current tensor. */
      final int[] dimensions;
      /** The permutation of the result tensor */
      final int[] permutation;
      /** The position from which to extract the current element */
      int[] positions;

      private TransposePermute(
          IAST tensor, ArrayList<Integer> tensorDimensions, int[] permutation) {
        this.tensor = tensor;
        this.dimensions = new int[tensorDimensions.size()];
        for (int i = 0; i < tensorDimensions.size(); i++) {
          dimensions[i] = tensorDimensions.get(i);
        }
        this.permutation = permutation;
        this.positions = new int[dimensions.length];
      }

      private IAST transposeRecursive() {
        return transposeRecursive(0, null);
      }

      /**
       * @param permutationIndex the current permutation index, which should be used to get the
       *     element from permutation array
       * @param resultList the parent list or <code>null</code> if the root-list should be created.
       * @return
       */
      private IAST transposeRecursive(int permutationIndex, IASTAppendable resultList) {
        if (permutationIndex >= permutation.length) {
          if (resultList != null) {
            resultList.append(tensor.getPart(positions));
          }
        } else {
          int size = dimensions[permutation[permutationIndex] - 1];
          IASTAppendable list = F.ListAlloc(size);
          if (resultList != null) {
            resultList.append(list);
          }
          for (int i = 0; i < size; i++) {
            positions[permutation[permutationIndex] - 1] = i + 1;
            transposeRecursive(permutationIndex + 1, list);
          }
          return list;
        }
        return F.NIL;
      }
    }

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.size() == 3) {
        if (ast.arg1().isList() && ast.arg2().isList()) {
          IAST tensor = (IAST) ast.arg1();
          ArrayList<Integer> dims = dimensions(tensor, tensor.head(), Integer.MAX_VALUE);
          int[] permutation = Validate.checkListOfInts(ast, ast.arg2(), 1, dims.size(), engine);
          if (permutation == null) {
            return F.NIL;
          }
          return new TransposePermute(tensor, dims, permutation).transposeRecursive();
        }
        return F.NIL;
      }
      final int[] dim = ast.arg1().isMatrix();
      if (dim != null) {
        // TODO improve for sparse arrays
        // final IAST originalMatrix = (IAST) ast.arg1().normal(false);
        final FieldMatrix<IExpr> matrix = Convert.list2Matrix(ast.arg1());
        if (matrix != null) {
          final FieldMatrix<IExpr> transposed = matrix.transpose();
          IExpr transposedMatrix = Convert.matrix2Expr(transposed).mapExpr(x -> transform(x));
          // because the rows can contain sub lists the IAST.IS_MATRIX flag cannot be set directly.
          // isMatrix()
          // must be used!
          transposedMatrix.isMatrix(true);
          return transposedMatrix;
          // return transpose(originalMatrix, dim[0], dim[1]);
        }
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }

    protected IExpr transform(final IExpr expr) {
      return expr;
    }

    /**
     * Transpose the given matrix.
     *
     * @param matrix the matrix which should be transposed
     * @param rows number of rows of the matrix
     * @param cols number of columns of the matrix
     * @return
     */
    private IAST transpose(final IAST matrix, int rows, int cols) {
      final IASTAppendable transposedMatrix = F.ast(S.List, cols, true);
      transposedMatrix.setArgs(cols + 1, i -> F.ast(S.List, rows, true));
      // for (int i = 1; i <= cols; i++) {
      // transposedMatrix.set(i, F.ast(F.List, rows, true));
      // }

      IAST originalRow;
      IASTMutable transposedResultRow;
      for (int i = 1; i <= rows; i++) {
        originalRow = (IAST) matrix.get(i);
        for (int j = 1; j <= cols; j++) {
          transposedResultRow = (IASTMutable) transposedMatrix.get(j);
          transposedResultRow.set(i, transform(originalRow.get(j)));
        }
      }
      // because the rows can contain sub lists the IAST.IS_MATRIX flag cannot be set directly.
      // isMatrix() must be
      // used!
      transposedMatrix.isMatrix(true);
      return transposedMatrix;
    }
  }

  /**
   *
   *
   * <pre>
   * UnitVector(position)
   * </pre>
   *
   * <blockquote>
   *
   * <p>returns a unit vector with element <code>1</code> at the given <code>position</code>.
   *
   * </blockquote>
   *
   * <pre>
   * UnitVector(dimension, position)
   * </pre>
   *
   * <blockquote>
   *
   * <p>returns a unit vector with dimension <code>dimension</code> and an element <code>1</code> at
   * the given <code>position</code>.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href="http://en.wikipedia.org/wiki/Unit_vector">Wikipedia - Unit vector</a><br>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; UnitVector(4,3)
   * {0,0,1,0}
   * </pre>
   */
  private static final class UnitVector extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.isAST2()) {
        int n = ast.arg1().toIntDefault(Integer.MIN_VALUE);
        if (n < 0) {
          return F.NIL;
          // Positive integer (less equal 2147483647) expected at position `2` in `1`.
          // return IOFunctions.printMessage(F.UnitVector, "intpm", F.List(ast, F.C1), engine);
        }
        int k = ast.arg2().toIntDefault(Integer.MIN_VALUE);
        if (k < 0) {
          return F.NIL;
          // Positive integer (less equal 2147483647) expected at position `2` in `1`.
          // return IOFunctions.printMessage(F.UnitVector, "intpm", F.List(ast, F.C2), engine);
        }

        if (k <= n) {
          IASTAppendable vector = F.ListAlloc(n);
          vector.appendArgs(0, n, i -> F.C0);
          vector.set(k, F.C1);
          return vector;
        }
        return F.NIL;
      }

      if (ast.arg1().isInteger()) {
        int k = ast.arg1().toIntDefault(Integer.MIN_VALUE);
        if (k < 0) {
          // Positive integer (less equal 2147483647) expected at position `2` in `1`.
          return IOFunctions.printMessage(S.UnitVector, "intpm", F.List(ast, F.C1), engine);
        }
        if (k == 1) {
          return F.List(F.C1, F.C0);
        }
        if (k == 2) {
          return F.List(F.C0, F.C1);
        }
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }
  }

  private static class UpperTriangularize extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      int[] dim = ast.arg1().isMatrix(false);
      if (dim == null) {
        // Argument `1` at position `2` is not a non-empty rectangular matrix.
        return IOFunctions.printMessage(ast.topHead(), "matrix", F.List(ast.arg1(), F.C1), engine);
      }
      try {
        final int k;
        if (ast.size() == 3) {
          k = Validate.checkIntType(ast, 2, Integer.MIN_VALUE);
        } else {
          k = 0;
        }
        int m = dim[0];
        int n = dim[1];
        FieldMatrix<IExpr> matrix = Convert.list2Matrix(ast.arg1());
        return F.matrix((i, j) -> i <= j - k ? matrix.getEntry(i, j) : F.C0, m, n);
      } catch (final ValidateException ve) {
        // int number validation
        return engine.printMessage(ast.topHead(), ve);
      }
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_2;
    }
  }

  /**
   *
   *
   * <pre>
   * VandermondeMatrix(n)
   * </pre>
   *
   * <blockquote>
   *
   * <p>gives the Vandermonde matrix with <code>n</code> rows and columns.
   *
   * </blockquote>
   *
   * <p>See:<br>
   *
   * <ul>
   *   <li><a href="http://en.wikipedia.org/wiki/Vandermonde_matrix">Wikipedia - Vandermonde
   *       matrix</a>
   * </ul>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; VandermondeMatrix({a,b,c})
   * {{1,a,a^2},
   *  {1,b,b^2},
   *  {1,c,c^2}}
   * </pre>
   */
  private static class VandermondeMatrix extends AbstractFunctionEvaluator {
    public VandermondeMatrix() {}

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      if (ast.arg1().isList()) {
        final IAST lst = (IAST) ast.arg1();
        final int len0 = lst.argSize();

        final int[] indexArray = new int[2];
        indexArray[0] = len0;
        indexArray[1] = len0;

        // final IIndexFunction<IExpr> function = new IIndexFunction<IExpr>() {
        // @Override
        // public IExpr evaluate(int[] index) {
        // return Power(lst.get(index[0] + 1), F.integer(index[1]));
        // }
        // };
        final IndexTableGenerator generator =
            new IndexTableGenerator(
                indexArray,
                S.List, //
                indx -> Power(lst.get(indx[0] + 1), F.ZZ(indx[1])));
        final IAST matrix = (IAST) generator.table();
        // because the rows can contain sub lists the IAST.IS_MATRIX flag cannot be set directly.
        // isMatrix()
        // must be used!
        matrix.isMatrix(true);
        return matrix;
      }

      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_1_1;
    }
  }

  /**
   *
   *
   * <pre>
   * VectorAngle(u, v)
   * </pre>
   *
   * <blockquote>
   *
   * <p>gives the angles between vectors <code>u</code> and <code>v</code>
   *
   * </blockquote>
   *
   * <h3>Examples</h3>
   *
   * <pre>
   * &gt;&gt; VectorAngle({1, 0}, {0, 1})
   * Pi/2
   *
   * &gt;&gt; VectorAngle({1, 2}, {3, 1})
   * Pi/4
   *
   * &gt;&gt; VectorAngle({1, 1, 0}, {1, 0, 1})
   * Pi/3
   *
   * &gt;&gt; VectorAngle({0, 1}, {0, 1})
   * 0
   * </pre>
   */
  private static class VectorAngle extends AbstractFunctionEvaluator {

    @Override
    public IExpr evaluate(final IAST ast, EvalEngine engine) {
      IExpr arg1 = ast.arg1();
      IExpr arg2 = ast.arg2();

      int dim1 = arg1.isVector();
      int dim2 = arg2.isVector();
      if (dim1 > (-1) && dim2 > (-1)) {
        return ArcCos(Divide(Dot(arg1, F.Conjugate(arg2)), Times(Norm(arg1), Norm(arg2))));
      }
      return F.NIL;
    }

    @Override
    public int[] expectedArgSize(IAST ast) {
      return ARGS_2_2;
    }
  }

  /**
   * Use cramer's rule to solve linear equations represented by a <code>2 x 3</code> augmented
   * matrix which represents the system <code>M.x == b</code>, where the columns of the <code>2 x 2
   * </code> matrix <code>M</code> are augmented by the vector <code>b</code>. This method assumes
   * that the dimensions of the matrix are already checked by the caller. See: <a
   * href="https://en.wikipedia.org/wiki/Cramer's_rule">Wikipedia Cramer's rule</a>
   *
   * @param matrix the <code>2 x 3</code> augmented matrix
   * @param quiet show no message if there is no solution
   * @param engine the evaluation engine
   * @return a list of values which solve the equations or <code>F#NIL</code>, if the equations have
   *     no solution.
   */
  public static IAST cramersRule2x3(FieldMatrix<IExpr> matrix, boolean quiet, EvalEngine engine) {
    IASTAppendable list = F.ListAlloc(2);
    // a1 * b2 - b1 * a2
    IExpr denominator = determinant2x2(matrix);
    if (denominator.isZero()) {
      if (!quiet) {
        return engine.printMessage("Row reduced linear equations have no solution.");
      }
      return F.NIL;
    }
    // c1 * b2 - b1 * c2
    IExpr xNumerator =
        F.Subtract(
            F.Times(matrix.getEntry(0, 2), matrix.getEntry(1, 1)),
            F.Times(matrix.getEntry(0, 1), matrix.getEntry(1, 2)));
    list.append(F.Divide(xNumerator, denominator));
    // a1 * c2 - c1*a2
    IExpr yNumerator =
        F.Subtract(
            F.Times(matrix.getEntry(0, 0), matrix.getEntry(1, 2)),
            F.Times(matrix.getEntry(0, 2), matrix.getEntry(1, 0)));
    list.append(F.Divide(yNumerator, denominator));
    return list;
  }

  /**
   * Use cramer's rule to solve linear equations represented by a <code>3 x 4</code> augmented
   * matrix which represents the system <code>M.x == b</code>, where the columns of the <code>3 x 3
   * </code> matrix <code>M</code> are augmented by the vector <code>b</code>. This method assumes
   * that the dimensions of the matrix are already checked by the caller. See: <a
   * href="https://en.wikipedia.org/wiki/Cramer's_rule">Wikipedia Cramer's rule</a>
   *
   * @param matrix the <code>3 x 4</code> augmented matrix
   * @param quiet show no message if there is no solution
   * @param engine the evaluation engine
   * @return a list of values which solve the equations or <code>F#NIL</code>, if the equations have
   *     no solution.
   */
  public static IAST cramersRule3x4(FieldMatrix<IExpr> matrix, boolean quiet, EvalEngine engine) {
    IASTAppendable list = F.ListAlloc(3);
    FieldMatrix<IExpr> denominatorMatrix = matrix.getSubMatrix(0, 2, 0, 2);
    IExpr denominator = determinant3x3(denominatorMatrix);
    if (denominator.isZero()) {
      if (!quiet) {
        return engine.printMessage("Row reduced linear equations have no solution.");
      }
      return F.NIL;
    }

    FieldMatrix<IExpr> xMatrix = denominatorMatrix.copy();
    xMatrix.setColumn(
        0, new IExpr[] {matrix.getEntry(0, 3), matrix.getEntry(1, 3), matrix.getEntry(2, 3)});
    IExpr xNumerator = determinant3x3(xMatrix);

    list.append(F.Divide(xNumerator, denominator));

    FieldMatrix<IExpr> yMatrix = denominatorMatrix.copy();
    yMatrix.setColumn(
        1, new IExpr[] {matrix.getEntry(0, 3), matrix.getEntry(1, 3), matrix.getEntry(2, 3)});
    IExpr yNumerator = determinant3x3(yMatrix);

    list.append(F.Divide(yNumerator, denominator));

    FieldMatrix<IExpr> zMatrix = denominatorMatrix.copy();
    zMatrix.setColumn(
        2, new IExpr[] {matrix.getEntry(0, 3), matrix.getEntry(1, 3), matrix.getEntry(2, 3)});
    IExpr zNumerator = determinant3x3(zMatrix);

    list.append(F.Divide(zNumerator, denominator));

    return list;
  }

  /**
   * Get the determinant of a <code>2 x 2</code> matrix. This method assumes that the dimensions of
   * the matrix are already checked by the caller.
   *
   * @param matrix a 2x2 matrix
   * @return
   */
  public static IExpr determinant2x2(final FieldMatrix<IExpr> matrix) {
    // 2x2 matrix
    IExpr[] row1 = matrix.getRow(0);
    IExpr[] row2 = matrix.getRow(1);
    return F.evalExpand(row1[0].times(row2[1]).subtract((row1[1].times(row2[0]))));
  }

  /**
   * Get the determinant of a <code>3 x 3</code> matrix. This method assumes that the dimensions of
   * the matrix are already checked by the caller.
   *
   * @param matrix a 3x3 matrix
   * @return
   */
  public static IExpr determinant3x3(final FieldMatrix<IExpr> matrix) {
    // 3x3 matrix
    IExpr[] row1 = matrix.getRow(0);
    IExpr[] row2 = matrix.getRow(1);
    IExpr[] row3 = matrix.getRow(2);
    return F.evalExpand(
        row1[0]
            .times(row2[1].times(row3[2]))
            .subtract((row1[0].times(row2[2].times(row3[1]))))
            .subtract((row1[1].times(row2[0].times(row3[2]))))
            .plus((row1[1].times(row2[2].times(row3[0]))))
            .plus((row1[2].times(row2[0].times(row3[1]))))
            .subtract((row1[2].times(row2[1].times(row3[0])))));
  }

  /**
   * Create a diagonal matrix from <code>valueArray[0]</code> (non-diagonal elements) and <code>
   * valueArray[1]</code> (diagonal elements).
   *
   * @param valueArray 2 values for non-diagonal and diagonal elemnets of the matrix.
   * @param dimension of the square matrix
   * @return
   */
  public static IAST diagonalMatrix(final IExpr[] valueArray, int dimension) {
    final int[] indexArray = new int[2];
    indexArray[0] = dimension;
    indexArray[1] = dimension;
    final IndexTableGenerator generator =
        new IndexTableGenerator(indexArray, S.List, new IndexFunctionDiagonal(valueArray));
    final IAST matrix = (IAST) generator.table();
    // because the rows can contain sub lists the IAST.IS_MATRIX flag cannot be set directly.
    // isMatrix() must be
    // used!
    matrix.isMatrix(true);
    return matrix;
  }

  public static ArrayList<Integer> dimensions(IAST ast) {
    return dimensionsRecursive(ast, ast.head(), Integer.MAX_VALUE, new ArrayList<Integer>());
  }

  public static ArrayList<Integer> dimensions(IAST ast, IExpr header) {
    return dimensions(ast, header, Integer.MAX_VALUE);
  }

  public static ArrayList<Integer> dimensions(IAST ast, IExpr header, int maxLevel) {
    return dimensionsRecursive(ast, header, maxLevel, new ArrayList<Integer>());
  }

  private static ArrayList<Integer> dimensionsRecursive(
      IAST ast, IExpr header, int maxLevel, ArrayList<Integer> dims) {

    int size = ast.size();
    dims.add(size - 1);
    if (size > 1 && ast.arg1().isAST()) {
      IAST arg1AST = (IAST) ast.arg1();
      int arg1Size = arg1AST.size();
      if (header.equals(S.List)) {
        if (!arg1AST.isSparseArray() && !header.equals(arg1AST.head())) {
          return dims;
        }
      } else if (!header.equals(arg1AST.head())) {
        return dims;
      }
      if (maxLevel > 0) {
        for (int i = 2; i < size; i++) {
          IExpr arg = ast.get(i);
          if (header.equals(S.List)) {
            if (!arg.isSparseArray() && !header.equals(arg.head())) {
              return dims;
            }
          } else if (!header.equals(arg.head())) {
            return dims;
          }
          if (arg.isAST() || arg.isSparseArray()) {
            if (arg1Size == arg.size()) {
              continue;
            }
          }
          return dims;
        }
        dimensionsRecursive(arg1AST, header, maxLevel - 1, dims);
      }
    }
    return dims;
  }

  //  private static ArrayList<Integer> arrayDepthRecursive(
  //      IAST ast, IExpr header, ArrayList<Integer> dims) {
  //    int size = ast.size();
  //    dims.add(size - 1);
  //    if (size > 1) {
  //      int arg1Size = ast.arg1().size();
  //      for (int i = 2; i < size; i++) {
  //        IExpr arg = ast.get(i);
  //        if (arg1Size != arg.size()) {
  //          return dims;
  //        }
  //      }
  //      IExpr arg1 = ast.arg1();
  //      if (arg1.isAST()) {
  //        arrayDepthRecursive((IAST) arg1, header, dims);
  //      }
  //      if (arg1.isSparseArray() || arg1.isNumericArray()) {
  //        arg1 = arg1.normal(false);
  //        if (arg1.isAST()) {
  //          arrayDepthRecursive((IAST) arg1, header, dims);
  //        }
  //      }
  //    }
  //
  //    return dims;
  //  }

  public static void initialize() {
    Initializer.init();
  }

  /**
   * Return the solution of the given (augmented-)matrix interpreted as a system of linear
   * equations.
   *
   * @param matrix
   * @param quiet suppress warning messages if <code>true</code>
   * @param engine the evaluation engine
   * @return <code>F.NIL</code> if the linear system is inconsistent and has no solution
   */
  public static IAST rowReduced2List(FieldMatrix<IExpr> matrix, boolean quiet, EvalEngine engine) {

    int rows = matrix.getRowDimension();
    int cols = matrix.getColumnDimension();
    if (rows == 2 && cols == 3) {
      IAST list = cramersRule2x3(matrix, quiet, engine);
      if (list.isPresent()) {
        return list;
      }
    } else if (rows == 3 && cols == 4) {
      IAST list = cramersRule3x4(matrix, quiet, engine);
      if (list.isPresent()) {
        return list;
      }
    }
    FieldReducedRowEchelonForm ref = new FieldReducedRowEchelonForm(matrix);
    FieldMatrix<IExpr> rowReduced = ref.getRowReducedMatrix();
    // System.out.println(rowReduced.toString());
    IExpr lastVarCoefficient = rowReduced.getEntry(rows - 1, cols - 2);
    if (lastVarCoefficient.isZero()) {
      if (!rowReduced.getEntry(rows - 1, cols - 1).isZero()) {
        return engine.printMessage("Row reduced linear equations have no solution.");
      }
    }
    IASTAppendable list = F.ListAlloc(rows < cols - 1 ? cols - 1 : rows);
    list.appendArgs(0, rows, j -> S.Together.of(engine, rowReduced.getEntry(j, cols - 1)));
    if (rows < cols - 1) {
      list.appendArgs(rows, cols - 1, i -> F.C0);
    }
    return list;
  }

  /**
   * Row reduce the given <code>(augmented-)matrix</code> and append the result as rules for the
   * given <code>variableList</code>.
   *
   * @param matrix a (augmented-)matrix
   * @param listOfVariables list of variable symbols
   * @param resultList a list to which the rules should be appended
   * @param engine the evaluation engine
   * @return resultList with the appended results as list of rules
   */
  public static IAST rowReduced2RulesList(
      FieldMatrix<IExpr> matrix,
      IAST listOfVariables,
      IASTAppendable resultList,
      EvalEngine engine) {
    int rows = matrix.getRowDimension();
    int cols = matrix.getColumnDimension();
    IAST smallList = null;
    if (rows == 2 && cols == 3) {
      smallList = cramersRule2x3(matrix, true, engine);
    } else if (rows == 3 && cols == 4) {
      smallList = cramersRule3x4(matrix, true, engine);
    }
    if (smallList != null) {
      if (!smallList.isPresent()) {
        // no solution
        return F.List();
      }
      final IAST sList = smallList;
      int size = smallList.size();
      IASTAppendable list = F.ListAlloc(size);
      list.appendArgs(size, j -> F.Rule(listOfVariables.get(j), engine.evaluate(sList.get(j))));

      resultList.append(list);
      return resultList;
    }
    FieldReducedRowEchelonForm ref = new FieldReducedRowEchelonForm(matrix);
    FieldMatrix<IExpr> rowReduced = ref.getRowReducedMatrix();
    int size = listOfVariables.argSize();

    IExpr lastVarCoefficient = rowReduced.getEntry(rows - 1, cols - 2);

    if (lastVarCoefficient.isZero()) {
      if (!rowReduced.getEntry(rows - 1, cols - 1).isZero()) {
        // no solution
        return F.List();
      }
    }
    IAST rule;
    IASTAppendable list = F.ListAlloc(rows);
    for (int j = 1; j < rows + 1; j++) {
      if (j < size + 1) {
        IExpr diagonal = rowReduced.getEntry(j - 1, j - 1);
        if (!diagonal.isZero()) {
          IASTAppendable plus = F.PlusAlloc(cols);
          plus.append(rowReduced.getEntry(j - 1, cols - 1));
          for (int i = j; i < cols - 1; i++) {
            if (!rowReduced.getEntry(j - 1, i).isZero()) {
              plus.append(
                  F.Times(rowReduced.getEntry(j - 1, i).negate(), listOfVariables.get(i + 1)));
            }
          }
          rule = F.Rule(listOfVariables.get(j), S.Together.of(engine, plus.oneIdentity0()));
          list.append(rule);
        }
      }
    }
    resultList.append(list);
    return resultList;
  }

  private LinearAlgebra() {}
}
