/*
 *
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

import { LoadingBtn } from '@/components/CallBackButton/LoadingBtn';
import { FormContextValue } from '@/components/Context/FormContext';
import ConfigurationForm from '@/pages/RegCenter/Cluster/Configuration/components/ConfigurationModal/ConfigurationForm';
import { Cluster } from '@/types/RegCenter/data';
import { l } from '@/utils/intl';
import { ModalForm } from '@ant-design/pro-components';
import { Button, Form } from 'antd';
import React, { useEffect } from 'react';

type ConfigurationModalProps = {
  visible: boolean;
  onClose: () => void;
  value: Partial<Cluster.Config>;
  onSubmit: (values: Partial<Cluster.Config>) => void;
  onHeartBeat: (values: Partial<Cluster.Config>) => void;
};
export default (props: ConfigurationModalProps) => {
  const { visible, onClose, onSubmit, value, onHeartBeat } = props;

  /**
   * init form
   */
  const [form] = Form.useForm();
  /**
   * init form context
   */
  const formContext = React.useMemo<FormContextValue>(
    () => ({
      resetForm: () => form.resetFields() // 定义 resetForm 方法
    }),
    [form]
  );

  const [submitting, setSubmitting] = React.useState<boolean>(false);

  /**
   * when modalVisible or values changed, set form values
   */
  useEffect(() => {
    form.setFieldsValue(value);
  }, [visible, value, form]);

  /**
   * handle cancel
   */
  const handleCancel = () => {
    onClose();
    formContext.resetForm();
    setSubmitting(false);
  };
  /**
   * submit form
   */
  const submitForm = async () => {
    const fieldsValue = await form.validateFields();
    setSubmitting(true);
    onSubmit(fieldsValue);
    handleCancel();
  };

  /**
   * handle test connect
   * */
  const handleTestConnect = async () => {
    const fieldsValue = await form.validateFields();
    onHeartBeat(fieldsValue);
  };

  /**
   * render footer
   */
  const renderFooter = () => {
    return [
      <Button key={'cancel'} onClick={() => handleCancel()}>
        {l('button.cancel')}
      </Button>,
      <LoadingBtn
        key={'test'}
        props={{
          size: 'middle',
          type: 'primary',
          style: { background: '#52c41a' }
        }}
        click={handleTestConnect}
        title={l('button.test.connection')}
      />,
      <Button
        key={'finish'}
        loading={submitting}
        type='primary'
        htmlType={'submit'}
        autoFocus
        onClick={() => submitForm()}
      >
        {l('button.save')}
      </Button>
    ];
  };

  return (
    <>
      <ModalForm
        width={'80%'}
        open={visible}
        modalProps={{
          onCancel: handleCancel,
          styles: {
            body: {
              maxHeight: '70vh',
              overflowY: 'auto',
              overflowX: 'hidden'
            }
          },
          maskClosable: false
        }}
        title={value.id ? l('rc.cc.modify') : l('rc.cc.create')}
        submitter={{ render: () => [...renderFooter()] }}
        initialValues={value}
        form={form}
      >
        <ConfigurationForm form={form} value={value} />
      </ModalForm>
    </>
  );
};
