/*
 *
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

import { l } from '@/utils/intl';
import { WarningMessage } from '@/utils/messages';
import { Transfer, Tree } from 'antd';
import type { TransferDirection, TransferItem } from 'antd/es/transfer';
import type { DataNode } from 'antd/es/tree';
import React, { Key } from 'react';

const { DirectoryTree } = Tree;
interface TreeTransferProps {
  dataSource: DataNode[];
  targetKeys: Key[];
  onChange?: (targetKeys: Key[], direction: TransferDirection, moveKeys: string[]) => void;
}

const isChecked = (selectedKeys: React.Key[], eventKey: React.Key) =>
  selectedKeys.includes(eventKey);
const generateTree = (treeNodes: DataNode[] = [], checkedKeys: Key[] = []): DataNode[] =>
  treeNodes.map(({ children, ...props }) => ({
    ...props,
    disabled: checkedKeys.includes(props.key as string),
    children: generateTree(children, checkedKeys)
  }));
export const TreeTransfer: React.FC<TreeTransferProps> = ({
  dataSource,
  targetKeys,
  ...restProps
}) => {
  const transferDataSource: TransferItem[] = [];

  function flatten(list: DataNode[] = []) {
    list.forEach((item) => {
      transferDataSource.push(item as TransferItem);
      flatten(item.children);
    });
  }

  flatten(dataSource);

  return (
    <Transfer
      {...restProps}
      rowKey={(record) => record.id as string}
      targetKeys={targetKeys as string[]}
      dataSource={transferDataSource}
      className={'treeList'}
      render={(item) => item.path}
      showSelectAll={true}
    >
      {({ direction, onItemSelect, selectedKeys }) => {
        if (direction === 'left') {
          const checkedKeys = [...selectedKeys, ...targetKeys];
          return (
            <DirectoryTree
              blockNode
              checkable
              className={'treeList'}
              checkStrictly
              defaultExpandAll
              checkedKeys={checkedKeys}
              treeData={generateTree(dataSource, targetKeys)}
              onCheck={async (_, { node: { key, isLeaf } }) => {
                if (isLeaf) {
                  onItemSelect(key as string, !isChecked(checkedKeys, key));
                } else {
                  WarningMessage(l('rc.udf.register.select'));
                }
              }}
              onSelect={async (_, { node: { key, isLeaf } }) => {
                if (isLeaf) {
                  onItemSelect(key as string, !isChecked(checkedKeys, key));
                } else {
                  WarningMessage(l('rc.udf.register.select'));
                }
              }}
            />
          );
        }
      }}
    </Transfer>
  );
};
