/*
 *
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package org.dinky.data.model;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * LineageResult
 *
 * @since 2022/8/20 21:09
 */
public class LineageRel {

    private final String sourceCatalog;

    private final String sourceDatabase;

    private final String sourceTable;

    private final String sourceColumn;

    private final String targetCatalog;

    private final String targetDatabase;

    private final String targetTable;

    private final String targetColumn;

    /**
     * Stores the expression for data conversion,
     * which source table fields are transformed by which expression the target field
     */
    private String transform;

    private static final String DELIMITER = ".";

    public LineageRel(
            String sourceCatalog,
            String sourceDatabase,
            String sourceTable,
            String sourceColumn,
            String targetCatalog,
            String targetDatabase,
            String targetTable,
            String targetColumn,
            String transform) {
        this.sourceCatalog = sourceCatalog;
        this.sourceDatabase = sourceDatabase;
        this.sourceTable = sourceTable;
        this.sourceColumn = sourceColumn;
        this.targetCatalog = targetCatalog;
        this.targetDatabase = targetDatabase;
        this.targetTable = targetTable;
        this.targetColumn = targetColumn;
        this.transform = transform;
    }

    public LineageRel(
            String catalog,
            String database,
            String sourceTable,
            String sourceColumn,
            String targetTable,
            String targetColumn) {
        this.sourceCatalog = catalog;
        this.sourceDatabase = database;
        this.sourceTable = sourceTable;
        this.sourceColumn = sourceColumn;
        this.targetCatalog = catalog;
        this.targetDatabase = database;
        this.targetTable = targetTable;
        this.targetColumn = targetColumn;
    }

    public static LineageRel build(
            String sourceTablePath,
            String sourceColumn,
            String targetTablePath,
            String targetColumn,
            String transform) {
        String[] sourceItems = sourceTablePath.split("\\.");
        String[] targetItems = targetTablePath.split("\\.");

        return new LineageRel(
                sourceItems[0],
                sourceItems[1],
                sourceItems[2],
                sourceColumn,
                targetItems[0],
                targetItems[1],
                targetItems[2],
                targetColumn,
                transform);
    }

    public static List<LineageRel> build(String catalog, String database, String[][] expectedArray) {
        return Stream.of(expectedArray)
                .map(e -> {
                    LineageRel lineageRel = new LineageRel(catalog, database, e[0], e[1], e[2], e[3]);
                    // transform field is optional
                    if (e.length == 5) {
                        lineageRel.setTransform(e[4]);
                    }
                    return lineageRel;
                })
                .collect(Collectors.toList());
    }

    public String getSourceCatalog() {
        return sourceCatalog;
    }

    public String getSourceDatabase() {
        return sourceDatabase;
    }

    public String getSourceTable() {
        return sourceTable;
    }

    public String getSourceColumn() {
        return sourceColumn;
    }

    public String getTargetCatalog() {
        return targetCatalog;
    }

    public String getTargetDatabase() {
        return targetDatabase;
    }

    public String getTargetTable() {
        return targetTable;
    }

    public String getTargetColumn() {
        return targetColumn;
    }

    public String getSourceTablePath() {
        return sourceCatalog + DELIMITER + sourceDatabase + DELIMITER + sourceTable;
    }

    public String getTargetTablePath() {
        return targetCatalog + DELIMITER + targetDatabase + DELIMITER + targetTable;
    }

    public String getTransform() {
        return transform;
    }

    public void setTransform(String transform) {
        this.transform = transform;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        LineageRel that = (LineageRel) o;

        if (!sourceCatalog.equals(that.sourceCatalog)) return false;
        if (!sourceDatabase.equals(that.sourceDatabase)) return false;
        if (!sourceTable.equals(that.sourceTable)) return false;
        if (!sourceColumn.equals(that.sourceColumn)) return false;
        if (!targetCatalog.equals(that.targetCatalog)) return false;
        if (!targetDatabase.equals(that.targetDatabase)) return false;
        if (!targetTable.equals(that.targetTable)) return false;
        if (!targetColumn.equals(that.targetColumn)) return false;
        return Objects.equals(transform, that.transform);
    }

    @Override
    public int hashCode() {
        int result = sourceCatalog.hashCode();
        result = 31 * result + sourceDatabase.hashCode();
        result = 31 * result + sourceTable.hashCode();
        result = 31 * result + sourceColumn.hashCode();
        result = 31 * result + targetCatalog.hashCode();
        result = 31 * result + targetDatabase.hashCode();
        result = 31 * result + targetTable.hashCode();
        result = 31 * result + targetColumn.hashCode();
        result = 31 * result + (transform != null ? transform.hashCode() : 0);
        return result;
    }
}
