/*
 *
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package org.dinky.utils;

import org.assertj.core.api.Assertions;
import org.junit.Ignore;
import org.junit.Test;

/** DirUtilTest */
@Ignore
public class SqlUtilTest {

    @Test
    public void testRemoveNote() {
        String testSql = "/**\n"
                + "test1\n"
                + "*/\n"
                + "//test2\n"
                + "-- test3\n"
                + "--test4\n"
                + "select 1 --test5\n"
                + " from test # test9\n"
                + " where '1'  <> '-- ::.' //test6\n"
                + " and 1=1 --test7\n"
                + " and 'zz' <> null; /**test8*/";

        String removedNoteSql = SqlUtil.removeNote(testSql);
        Assertions.assertThat(removedNoteSql).isNotNull();
        Assertions.assertThat(removedNoteSql)
                .isEqualTo("//test2\n" + "\n"
                        + "\n"
                        + "select 1 \n"
                        + " from test # test9\n"
                        + " where '1'  <> '-- ::.' //test6\n"
                        + " and 1=1 \n"
                        + " and 'zz' <> null;");
    }

    @Test
    public void getStatements() {
        String sql = "set 'state.savepoints.dir' = 'hdfs://namenode:9000/tmp/checkpoint'; --ddd\n"
                + "set 'state.checkpoints.dir' = 'hdfs://namenode:9000/tmp/checkpoint'; --dd \n"
                + "create table abc ;\n";
        String[] statements = SqlUtil.getStatements(sql, ";\\s*(?:\\n|--.*)");
        Assertions.assertThat(statements.length).isEqualTo(3);
    }

    @Test
    public void testGetStatementsWithMultiComment() {
        String sql = "set 'key1' =  'value1';-- 同行注释1\n" + "-- 多行注释1\n"
                + " -- 多行注释2\n"
                + "-- 多行注释3\n"
                + "set 'key2' =  'value2'; -- 同行注释2\n"
                + "  -- 注释1\n"
                + "set 'key3' =  'value4'; -- 同行注释3 --同行注释4\n"
                + "-- 注释2\n"
                + ";\n"
                + "\n"
                + ";;\n"
                + "\n"
                + "\n"
                + "select \n"
                + "  * \n"
                + "   -- sql内注释1\n"
                + "   -- sql内注释2\n"
                + "  from table1 ; \n"
                + " select ';--' as `test1`,';  --' as `test2`,'; \n"
                + " --' as `test2`,* from table2;\n"
                + "select ' select * from tb1 ; \n"
                + "\n"
                + " select * from tb2; ' as sql_stmt;\n"
                + "-- 注释3\n"
                + "--注释4\n"
                + " --注释5\n"
                + "EXECUTE JAR WITH (\n"
                + "'uri'='a.jar',\n"
                + "'main-class'='realtime.app.DwdKafkaToDelay',\n"
                + "'args'='base64@LS1vZmZzZXQuaW5pdCBsYXRlc3Q=',\n"
                + "'allowNonRestoredState'='false'\n"
                + ");\n";
        String expected0 = "set 'key1' =  'value1'";
        String expected1 = "set 'key2' =  'value2'";
        String expected2 = "set 'key3' =  'value4'";
        String expected3 = "select \n" + "  * \n" + "   \n" + "   \n" + "  from table1";
        String expected4 = "select ';--' as `test1`,';  --' as `test2`,'; \n" + " --' as `test2`,* from table2";
        String expected5 = "select ' select * from tb1 ; \n" + " select * from tb2; ' as sql_stmt";
        String expected6 = "EXECUTE JAR WITH (\n" + "'uri'='a.jar',\n"
                + "'main-class'='realtime.app.DwdKafkaToDelay',\n"
                + "'args'='base64@LS1vZmZzZXQuaW5pdCBsYXRlc3Q=',\n"
                + "'allowNonRestoredState'='false'\n"
                + ")";
        String[] statements = SqlUtil.getStatements(sql);
        Assertions.assertThat(statements.length).isEqualTo(7);
        Assertions.assertThat(statements[0]).isEqualTo(expected0);
        Assertions.assertThat(statements[1]).isEqualTo(expected1);
        Assertions.assertThat(statements[2]).isEqualTo(expected2);
        Assertions.assertThat(statements[3]).isEqualTo(expected3);
        Assertions.assertThat(statements[4]).isEqualTo(expected4);
        Assertions.assertThat(statements[5]).isEqualTo(expected5);
        Assertions.assertThat(statements[6]).isEqualTo(expected6);
    }

    @Test
    public void testGetStatementsByMysql2Doris() {
        String sql = "set 'execution.checkpointing.interval'='5 s';\n"
                + "ADD CUSTOMJAR 'rs:/flink-sql-connector-mysql-cdc.jar';\n"
                + "ADD CUSTOMJAR 'rs:/mysql-connector-java.jar';\n"
                + "EXECUTE JAR WITH (\n"
                + "'uri'='rs:/flink-doris-connector.jar',\n"
                + "'main-class'='org.apache.doris.flink.tools.cdc.CdcTools',\n"
                + "'args'='base64@bXlzcWwtc3luYy1kYXRhYmFzZSAgICAgLS1kYXRhYmFzZSBkaW5reSAgICAgLS1teXNxbC1jb25mIGhvc3RuYW1lPW15c3FsICAgICAtLW15c3FsLWNvbmYgcG9ydD0zMzA2ICAgICAtLW15c3FsLWNvbmYgdXNlcm5hbWU9cm9vdCAgICAgLS1teXNxbC1jb25mIHBhc3N3b3JkPWRpbmt5ICAgICAtLW15c3FsLWNvbmYgZGF0YWJhc2UtbmFtZT1kaW5reSAgICAgLS1teXNxbC1jb25mIHNlcnZlci10aW1lLXpvbmU9QXNpYS9TaGFuZ2hhaSAgICAgLS1pbmNsdWRpbmctdGFibGVzICJkaW5reV90YXNrIiAgICAgLS1zaW5rLWNvbmYgZmVub2Rlcz1kb3Jpcy1mZTo4MDMwICAgICAtLXNpbmstY29uZiB1c2VybmFtZT1yb290ICAgICAtLXNpbmstY29uZiBqZGJjLXVybD1qZGJjOm15c3FsOi8vZG9yaXMtZmU6OTAzMCAgICAgLS1zaW5rLWNvbmYgc2luay5sYWJlbC1wcmVmaXg9bGFiZWwtMSAgICAgLS10YWJsZS1jb25mIHJlcGxpY2F0aW9uX251bT0xIA==',\n"
                + "'allowNonRestoredState'='false'\n"
                + ");\n";
        String expected0 = "set 'execution.checkpointing.interval'='5 s'";
        String expected1 = "ADD CUSTOMJAR 'rs:/flink-sql-connector-mysql-cdc.jar'";
        String expected2 = "ADD CUSTOMJAR 'rs:/mysql-connector-java.jar'";
        String expected3 = "EXECUTE JAR WITH (\n" + "'uri'='rs:/flink-doris-connector.jar',\n"
                + "'main-class'='org.apache.doris.flink.tools.cdc.CdcTools',\n"
                + "'args'='base64@bXlzcWwtc3luYy1kYXRhYmFzZSAgICAgLS1kYXRhYmFzZSBkaW5reSAgICAgLS1teXNxbC1jb25mIGhvc3RuYW1lPW15c3FsICAgICAtLW15c3FsLWNvbmYgcG9ydD0zMzA2ICAgICAtLW15c3FsLWNvbmYgdXNlcm5hbWU9cm9vdCAgICAgLS1teXNxbC1jb25mIHBhc3N3b3JkPWRpbmt5ICAgICAtLW15c3FsLWNvbmYgZGF0YWJhc2UtbmFtZT1kaW5reSAgICAgLS1teXNxbC1jb25mIHNlcnZlci10aW1lLXpvbmU9QXNpYS9TaGFuZ2hhaSAgICAgLS1pbmNsdWRpbmctdGFibGVzICJkaW5reV90YXNrIiAgICAgLS1zaW5rLWNvbmYgZmVub2Rlcz1kb3Jpcy1mZTo4MDMwICAgICAtLXNpbmstY29uZiB1c2VybmFtZT1yb290ICAgICAtLXNpbmstY29uZiBqZGJjLXVybD1qZGJjOm15c3FsOi8vZG9yaXMtZmU6OTAzMCAgICAgLS1zaW5rLWNvbmYgc2luay5sYWJlbC1wcmVmaXg9bGFiZWwtMSAgICAgLS10YWJsZS1jb25mIHJlcGxpY2F0aW9uX251bT0xIA==',\n"
                + "'allowNonRestoredState'='false'\n"
                + ")";
        String[] statements = SqlUtil.getStatements(sql);
        Assertions.assertThat(statements.length).isEqualTo(4);
        Assertions.assertThat(statements[0]).isEqualTo(expected0);
        Assertions.assertThat(statements[1]).isEqualTo(expected1);
        Assertions.assertThat(statements[2]).isEqualTo(expected2);
        Assertions.assertThat(statements[3]).isEqualTo(expected3);
    }

    @Test
    public void testGetStatementsByLineageTestSql() {
        String sql = "create table ods_order_sale_goods_snapshot_r\n" + "(\n"
                + "    store_code string,\n"
                + "    --import_code string,\n"
                + "    good_code string,\n"
                + "    create_time string,\n"
                + "    category_code string,\n"
                + "    purchase_kind_code string,\n"
                + "    modify_time string,\n"
                + "    is_unfree_goods char,\n"
                + "    is_unmember_goods char,\n"
                + "    member_price decimal(22,4),\n"
                + "    retail_price decimal(22,4),\n"
                + "    purchase_kind_des string,\n"
                + "    sale_order_code string,\n"
                + "    id string,\n"
                + "    category_des string,\n"
                + "    proc_time as proctime(),\n"
                + "    sale_time TIMESTAMP(3)  ,\n"
                + "    origin_table STRING   ,\n"
                + "    PRIMARY KEY (id) NOT ENFORCED\n"
                + ") with (\n"
                + "      'connector'='datagen'\n"
                + "      );\n"
                + "\n"
                + "create table ods_order_sale_order_details_r\n"
                + "(\n"
                + "    --import_code string,\n"
                + "    member_coupon_type string,\n"
                + "    lot_num string,\n"
                + "    item_type string,\n"
                + "    modify_time string,\n"
                + "    perform_price_type string,\n"
                + "    row_num int,\n"
                + "    account_price decimal(22,4),\n"
                + "    perform_price decimal(22,4),\n"
                + "    sale_order_code string,\n"
                + "    refer_bill_code string,\n"
                + "    id string,\n"
                + "    cost_price decimal(22,4),\n"
                + "    store_code string,\n"
                + "    account_money decimal(22,4),\n"
                + "    good_code string,\n"
                + "    quantity decimal(22,4),\n"
                + "    apportion_price decimal(22,4),\n"
                + "    create_time string,\n"
                + "    --is_preorder string,\n"
                + "    perform_profit_margin decimal(22,4),\n"
                + "    --promotionbill_plan_code string,\n"
                + "    service_charge decimal(22,4),\n"
                + "    promotionbill_plan_type string,\n"
                + "    account_price_gross_money decimal(22,4),\n"
                + "    refer_bill_type string,\n"
                + "    refer_bill_row_num int,\n"
                + "    output_tax string,\n"
                + "    --useful int,\n"
                + "    proc_time as proctime(),\n"
                + "    sale_time TIMESTAMP(3) ,\n"
                + "    PRIMARY KEY (sale_order_code,row_num,lot_num,good_code) NOT ENFORCED\n"
                + ") with (\n"
                + "      'connector'='datagen'\n"
                + "      );\n"
                + "\n"
                + "create table ods_order_sales_ordersalesman_r\n"
                + "(\n"
                + "    store_code       string,\n"
                + "    create_time      string,\n"
                + "    edit_time        string,\n"
                + "    sale_order_code  string,\n"
                + "    sales_job_number     string,\n"
                + "    id string,\n"
                + "    row_num int,\n"
                + "    sale_details_id string,\n"
                + "    sale_time TIMESTAMP(3) ,\n"
                + "    PRIMARY KEY (id) NOT ENFORCED\n"
                + ") with (\n"
                + "      'connector'='datagen'\n"
                + "      );\n"
                + "\n"
                + "create table ods_order_sale_order_r\n"
                + "(\n"
                + "    --import_code string,\n"
                + "    modify_time string,\n"
                + "    --around_money decimal(22,4),\n"
                + "    --odd_change_money decimal,\n"
                + "    --customer_pay_money decimal,\n"
                + "    sale_order_code string,\n"
                + "    other_free_money decimal,\n"
                + "    gift_free_money decimal,\n"
                + "    bill_type string,\n"
                + "    refer_bill_code string,\n"
                + "    pair_count decimal,\n"
                + "    store_code string,\n"
                + "    member_id string,\n"
                + "    cashdesk_id string,\n"
                + "    point_number decimal,\n"
                + "    coupon_code string,\n"
                + "    create_time string,\n"
                + "    casher_code string,\n"
                + "    bill_kind string,\n"
                + "    order_code string,\n"
                + "    order_from string,\n"
                + "    refer_bill_type string,\n"
                + "    coupon_plan_code string,\n"
                + "    cash_free_money decimal,\n"
                + "    coupon_type string,\n"
                + "    proc_time as proctime(),\n"
                + "    sale_time TIMESTAMP(3) ,\n"
                + "    PRIMARY KEY (sale_order_code) NOT ENFORCED\n"
                + ") with (\n"
                + "      'connector'='datagen'\n"
                + "      );\n"
                + "\n"
                + "\n"
                + "CREATE TABLE IF NOT EXISTS ods_zt_ord_order_r (\n"
                + "                                                  id                        string,\n"
                + "                                                  parent_order_code         string,\n"
                + "                                                  order_code                string,\n"
                + "                                                  customer_code             string,\n"
                + "                                                  top_channel_code          string,\n"
                + "                                                  goods_channel_code        string,\n"
                + "                                                  pos_code                  string,\n"
                + "                                                  channel_code              string,\n"
                + "                                                  warehouse_code            string,\n"
                + "                                                  state                     string,\n"
                + "                                                  need_receive_amount       decimal(22,4),\n"
                + "    customer_need_pay_amount   decimal(22,4),\n"
                + "    goods_total_amount         decimal(22,4),\n"
                + "    expense_total_amount       decimal(22,4),\n"
                + "    preferential_total_amount  decimal(22,4),\n"
                + "    subsidy_total_amount       decimal(22,4),\n"
                + "    decucted_total_amount      decimal(22,4),\n"
                + "    exercise_total_amount      decimal(22,4),\n"
                + "    ordonnance_id             string,\n"
                + "    splited                   string,\n"
                + "    delivery_provider         string,\n"
                + "    payee                     string,\n"
                + "    referrer_id               string,\n"
                + "    ticket_num                string,\n"
                + "    referrer_name             string,\n"
                + "    out_trade_code            string,\n"
                + "    remark                    string,\n"
                + "    creater_id                string,\n"
                + "    create_time               string,\n"
                + "    modify_time               string,\n"
                + "    del_flag                  string,\n"
                + "    from_warehouse_code       string,\n"
                + "    order_type                int,\n"
                + "    PRIMARY KEY (`id`) NOT ENFORCED\n"
                + "    ) WITH (\n"
                + "          'connector'='datagen'\n"
                + "          );\n"
                + "\n"
                + "CREATE TABLE IF NOT EXISTS dwd_sd_so_ordr_detl_r (\n"
                + "                                                     stsc_date string comment '统计日期-天',\n"
                + "                                                     sale_ordr_doc string comment '销售订单编号',\n"
                + "    --store_code string comment '门店编码',\n"
                + "                                                     ordr_sour_code string comment '订单来源编码',\n"
                + "                                                     sale_order_proc_time timestamp comment '',\n"
                + "                                                     ordr_sale_time string comment '',\n"
                + "                                                     prmn_prog_type_code string comment '促销方案类型编码',\n"
                + "                                                     ordr_type_code string comment '订单类型编码(销售单/退货单)',\n"
                + "                                                     ordr_cate_code string comment '订单类别编码(标准/团购单/赠品记账(积分兑奖)/财务记账(订金))',\n"
                + "                                                     prmn_prog_code string comment '促销方案编码',\n"
                + "                                                     coup_type string comment '',\n"
                + "                                                     coup_code string comment '',\n"
                + "                                                     line_item INT comment '行项目',\n"
                + "                                                     refn_ordr_type_code string comment '参考单据类型编码(销售单/团购单/处方单/订金退货申请单)',\n"
                + "                                                     refn_ordr_doc string comment '参考单据编号',\n"
                + "                                                     refn_ordr_item INT comment '参考单据行项目号',\n"
                + "                                                     memb_disc_mode string comment '会员优惠方式(会员折扣/会员价/VIP会员价)',\n"
                + "    --proj_cate string comment '',\n"
                + "                                                     goods_code string comment '商品编码',\n"
                + "                                                     sale_order_details_proc_time timestamp comment '',\n"
                + "                                                     cate_clas_code string comment '',\n"
                + "                                                     cate_clas_name string comment '',\n"
                + "                                                     purc_clas_code_new string comment '',\n"
                + "                                                     purc_clas_name_new string comment '',\n"
                + "                                                     lotn string comment '批号',\n"
                + "                                                     sale_tax string comment '',\n"
                + "                                                     memb_id string comment '会员ID',\n"
                + "                                                     memb_point decimal comment '',\n"
                + "                                                     chk_out_id string comment '',\n"
                + "                                                     casr_id string comment '',\n"
                + "                                                     is_memb_goods string comment '非会员商品标识',\n"
                + "                                                     retail_pric decimal(22,4) comment '零售单价',\n"
                + "    memb_pric decimal(22,4) comment '会员单价',\n"
                + "    exec_pric_type string comment '执行价类型编码',\n"
                + "    --befo_divid_pric decimal(22,4) comment '',\n"
                + "    divid_pric decimal(22,4) comment '医保单价',\n"
                + "    acnt_pric decimal(22,4) comment '记账单价',\n"
                + "    exec_pric decimal(22,4) comment '执行单价',\n"
                + "    dct_amt decimal comment '',\n"
                + "    gift_dct_amt decimal comment '',\n"
                + "    other_free_amt decimal comment '',\n"
                + "    amt1 decimal(22,4) comment '',\n"
                + "    amt2 decimal(22,4) comment '',\n"
                + "    sale_amt decimal(22,4) comment '销售金额',\n"
                + "    sale_qty decimal(22,4) comment '销售数量',\n"
                + "    china_med_qty decimal comment '',\n"
                + "    etl_time timestamp comment '',\n"
                + "    is_n_surd_prof string comment '不让利商品标识',\n"
                + "    cost_pric decimal(22,4) comment '成本单价',\n"
                + "    cost_amt decimal(22,4) comment '',\n"
                + "    --sale_cost_amt decimal(22,4) comment '销售成本额',\n"
                + "    is_effe_ordr string comment '是否有效订单',\n"
                + "    order_code string comment '',\n"
                + "    is_ecp_self_dstn_ordr string comment '是否ECP自配送订单',\n"
                + "    stat_date string comment '',\n"
                + "    proj_cate_code string comment '项目类别编码',\n"
                + "    purchase_kind_code string comment '项目类别编码',\n"
                + "    sale_goods_snapshot_proc_time timestamp comment '',\n"
                + "    casr_code string comment '收银员编码',\n"
                + "    service_charge decimal(22,4)  comment '',\n"
                + "    sale_pers_id string comment '营销员ID',\n"
                + "    phmc_code string comment '门店编码',\n"
                + "    out_phmc_code  string comment '出货门店',\n"
                + "    is_ydch_flag   string comment '是否异店出货',\n"
                + "    coup_prog_code string comment '券方案编码',\n"
                + "    PRIMARY KEY (stsc_date,sale_ordr_doc,goods_code,line_item,lotn) NOT ENFORCED\n"
                + "    ) WITH (\n"
                + "          'connector'='print'\n"
                + "          );\n"
                + "\n"
                + "\n"
                + "insert into dwd_sd_so_ordr_detl_r\n"
                + "select ifnull(date_format(cast(t1.sale_time as timestamp), 'yyyyMMdd'), 'NA')  as stsc_date\n"
                + "     , t1.sale_order_code                                                                  as sale_ordr_doc\n"
                + "     --, t1.store_code                                                                       as store_code\n"
                + "     , (case when coalesce(t.order_from, '') = '' then '9999' else t.order_from end)       as ordr_sour_code\n"
                + "     , t.proc_time                                               as sale_order_proc_time\n"
                + "     , date_format(cast(t1.sale_time as timestamp),'yyyy-MM-dd HH:mm:ss') as ordr_sale_time\n"
                + "     , t1.promotionbill_plan_type                                as prmn_prog_type_code\n"
                + "     , t.bill_type                                               as ordr_type_code\n"
                + "     , t.bill_kind                                               as ordr_cate_code\n"
                + "     , t1.promotionbill_plan_type                                as prmn_prog_code\n"
                + "     , t.coupon_type                                             as coup_type\n"
                + "     , t.coupon_code                                             as coup_code\n"
                + "     , t1.row_num                                                as line_item\n"
                + "     , t1.refer_bill_type                                        as refn_ordr_type_code\n"
                + "     , t1.refer_bill_code                                        as refn_ordr_doc\n"
                + "     , t1.refer_bill_row_num                                     as refn_ordr_item\n"
                + "     , t1.member_coupon_type                                     as memb_disc_mode\n"
                + "     --, t1.item_type                                              as proj_cate\n"
                + "     , t1.good_code                                              as goods_code\n"
                + "     , t1.proc_time                                              as sale_order_details_proc_time\n"
                + "     , t2.category_code                                          as cate_clas_code\n"
                + "     , t2.category_des                                           as cate_clas_name\n"
                + "     , t2.purchase_kind_code                                     as purc_clas_code_new\n"
                + "     , t2.purchase_kind_des                                      as purc_clas_name_new\n"
                + "     , t1.lot_num                                                as lotn\n"
                + "     , t1.output_tax                                             as sale_tax\n"
                + "     , t.member_id                                               as memb_id\n"
                + "     , t.point_number                                            as memb_point\n"
                + "     , t.cashdesk_id                                             as chk_out_id\n"
                + "     ,case when coalesce(t.casher_code , '') = '' then 'NA' else LPAD(CAST(t.casher_code  as string), 8, '0') end as casr_id\n"
                + "     , t2.is_unmember_goods                                      as is_memb_goods\n"
                + "     , t2.retail_price                                           as retail_pric\n"
                + "     , t2.member_price                                           as memb_pric\n"
                + "     , t1.perform_price_type                                     as exec_pric_type\n"
                + "     --, t1.perform_price                                          as befo_divid_pric\n"
                + "     , t1.apportion_price                                        as divid_pric\n"
                + "     , t1.account_price                                          as acnt_pric\n"
                + "     , t1.perform_price                                          as exec_pric\n"
                + "     , t.cash_free_money                                         as dct_amt\n"
                + "     , t.gift_free_money                                         as gift_dct_amt\n"
                + "     , t.other_free_money                                        as other_free_amt\n"
                + "     , t1.perform_profit_margin                                  as amt1\n"
                + "     , t1.account_price_gross_money                              as amt2\n"
                + "     , t1.account_money                                          as sale_amt\n"
                + "     , t1.quantity                                               as sale_qty\n"
                + "     , t.pair_count                                              as china_med_qty\n"
                + "     , current_timestamp                                         as etl_time\n"
                + "     , t2.is_unfree_goods                                        as is_n_surd_prof\n"
                + "     , t1.cost_price                                             as cost_pric\n"
                + "     , t1.cost_price * t1.quantity                               as cost_amt\n"
                + "     --, t1.cost_price * t1.quantity                               as sale_cost_amt\n"
                + "     , case when t.bill_kind in ('3', '4', '15') then 'N' else 'Y' end as is_effe_ordr\n"
                + "     , coalesce(t.order_code, t.sale_order_code)                 as order_code\n"
                + "     , 'N'                                                       as is_ecp_self_dstn_ordr\n"
                + "     , date_format(cast(t1.sale_time as timestamp), 'yyyyMMdd')        as stat_date\n"
                + "     , t1.item_type                                              as proj_cate_code\n"
                + "     , t2.purchase_kind_code                                     as purchase_kind_code\n"
                + "     , t2.proc_time                                              as sale_goods_snapshot_proc_time\n"
                + "     , t.casher_code                                             as casr_code\n"
                + "     , t1.service_charge                                         as service_charge\n"
                + "     ,case when coalesce(t3.sales_job_number , '') = '' then 'NA' else LPAD(CAST(t3.sales_job_number  as string), 8, '0') end as sale_pers_id\n"
                + "     ,case when t4.from_warehouse_code is not null then t4.from_warehouse_code\n"
                + "           when coalesce(trim(t1.store_code),'')='' then 'NA'\n"
                + "           else trim(t1.store_code) end                               as phmc_code\n"
                + "     ,case when t4.warehouse_code is not null then t4.warehouse_code\n"
                + "           when coalesce(trim(t1.store_code),'')='' then 'NA'\n"
                + "           else trim(t1.store_code) end                               as out_phmc_code\n"
                + "     ,if(t4.pos_code is not null, 'Y', 'N')                      as is_ydch_flag\n"
                + "     ,t.coupon_plan_code                                         as coup_prog_code\n"
                + "from ods_order_sale_order_r t\n"
                + "         inner join ods_order_sale_order_details_r t1\n"
                + "                    on t.sale_order_code = t1.sale_order_code\n"
                + "                        and date_format(cast(t1.sale_time as timestamp), 'yyyyMMdd')=date_format(localtimestamp,'yyyyMMdd')\n"
                + "         left join ods_order_sale_goods_snapshot_r t2\n"
                + "                   on t1.sale_order_code = t2.sale_order_code\n"
                + "                       and t1.good_code = t2.good_code\n"
                + "                       and t2.origin_table='sale_goods_snapshot'\n"
                + "                       and date_format(cast(t2.sale_time as timestamp), 'yyyyMMdd')=date_format(localtimestamp,'yyyyMMdd')\n"
                + "         left join ods_order_sales_ordersalesman_r t3\n"
                + "                   on t1.sale_order_code = t3.sale_order_code\n"
                + "                       and t1.row_num = t3.row_num\n"
                + "                       and date_format(cast(t3.sale_time as timestamp), 'yyyyMMdd')=date_format(localtimestamp,'yyyyMMdd')\n"
                + "         left join ods_zt_ord_order_r t4\n"
                + "                   on t.sale_order_code = t4.pos_code\n"
                + "                       and t4.order_type =1\n"
                + "where date_format(cast(t.sale_time as timestamp), 'yyyyMMdd')=date_format(localtimestamp,'yyyyMMdd');";
        String expected0 = "create table ods_order_sale_goods_snapshot_r\n" + "(\n"
                + "    store_code string,\n"
                + "    \n"
                + "    good_code string,\n"
                + "    create_time string,\n"
                + "    category_code string,\n"
                + "    purchase_kind_code string,\n"
                + "    modify_time string,\n"
                + "    is_unfree_goods char,\n"
                + "    is_unmember_goods char,\n"
                + "    member_price decimal(22,4),\n"
                + "    retail_price decimal(22,4),\n"
                + "    purchase_kind_des string,\n"
                + "    sale_order_code string,\n"
                + "    id string,\n"
                + "    category_des string,\n"
                + "    proc_time as proctime(),\n"
                + "    sale_time TIMESTAMP(3)  ,\n"
                + "    origin_table STRING   ,\n"
                + "    PRIMARY KEY (id) NOT ENFORCED\n"
                + ") with (\n"
                + "      'connector'='datagen'\n"
                + "      )";
        String expected1 = "create table ods_order_sale_order_details_r\n" + "(\n"
                + "    \n"
                + "    member_coupon_type string,\n"
                + "    lot_num string,\n"
                + "    item_type string,\n"
                + "    modify_time string,\n"
                + "    perform_price_type string,\n"
                + "    row_num int,\n"
                + "    account_price decimal(22,4),\n"
                + "    perform_price decimal(22,4),\n"
                + "    sale_order_code string,\n"
                + "    refer_bill_code string,\n"
                + "    id string,\n"
                + "    cost_price decimal(22,4),\n"
                + "    store_code string,\n"
                + "    account_money decimal(22,4),\n"
                + "    good_code string,\n"
                + "    quantity decimal(22,4),\n"
                + "    apportion_price decimal(22,4),\n"
                + "    create_time string,\n"
                + "    \n"
                + "    perform_profit_margin decimal(22,4),\n"
                + "    \n"
                + "    service_charge decimal(22,4),\n"
                + "    promotionbill_plan_type string,\n"
                + "    account_price_gross_money decimal(22,4),\n"
                + "    refer_bill_type string,\n"
                + "    refer_bill_row_num int,\n"
                + "    output_tax string,\n"
                + "    \n"
                + "    proc_time as proctime(),\n"
                + "    sale_time TIMESTAMP(3) ,\n"
                + "    PRIMARY KEY (sale_order_code,row_num,lot_num,good_code) NOT ENFORCED\n"
                + ") with (\n"
                + "      'connector'='datagen'\n"
                + "      )";
        String expected2 = "create table ods_order_sales_ordersalesman_r\n" + "(\n"
                + "    store_code       string,\n"
                + "    create_time      string,\n"
                + "    edit_time        string,\n"
                + "    sale_order_code  string,\n"
                + "    sales_job_number     string,\n"
                + "    id string,\n"
                + "    row_num int,\n"
                + "    sale_details_id string,\n"
                + "    sale_time TIMESTAMP(3) ,\n"
                + "    PRIMARY KEY (id) NOT ENFORCED\n"
                + ") with (\n"
                + "      'connector'='datagen'\n"
                + "      )";

        String expected3 = "create table ods_order_sale_order_r\n" + "(\n"
                + "    \n"
                + "    modify_time string,\n"
                + "    \n"
                + "    \n"
                + "    \n"
                + "    sale_order_code string,\n"
                + "    other_free_money decimal,\n"
                + "    gift_free_money decimal,\n"
                + "    bill_type string,\n"
                + "    refer_bill_code string,\n"
                + "    pair_count decimal,\n"
                + "    store_code string,\n"
                + "    member_id string,\n"
                + "    cashdesk_id string,\n"
                + "    point_number decimal,\n"
                + "    coupon_code string,\n"
                + "    create_time string,\n"
                + "    casher_code string,\n"
                + "    bill_kind string,\n"
                + "    order_code string,\n"
                + "    order_from string,\n"
                + "    refer_bill_type string,\n"
                + "    coupon_plan_code string,\n"
                + "    cash_free_money decimal,\n"
                + "    coupon_type string,\n"
                + "    proc_time as proctime(),\n"
                + "    sale_time TIMESTAMP(3) ,\n"
                + "    PRIMARY KEY (sale_order_code) NOT ENFORCED\n"
                + ") with (\n"
                + "      'connector'='datagen'\n"
                + "      )";
        String expected4 = "CREATE TABLE IF NOT EXISTS ods_zt_ord_order_r (\n"
                + "                                                  id                        string,\n"
                + "                                                  parent_order_code         string,\n"
                + "                                                  order_code                string,\n"
                + "                                                  customer_code             string,\n"
                + "                                                  top_channel_code          string,\n"
                + "                                                  goods_channel_code        string,\n"
                + "                                                  pos_code                  string,\n"
                + "                                                  channel_code              string,\n"
                + "                                                  warehouse_code            string,\n"
                + "                                                  state                     string,\n"
                + "                                                  need_receive_amount       decimal(22,4),\n"
                + "    customer_need_pay_amount   decimal(22,4),\n"
                + "    goods_total_amount         decimal(22,4),\n"
                + "    expense_total_amount       decimal(22,4),\n"
                + "    preferential_total_amount  decimal(22,4),\n"
                + "    subsidy_total_amount       decimal(22,4),\n"
                + "    decucted_total_amount      decimal(22,4),\n"
                + "    exercise_total_amount      decimal(22,4),\n"
                + "    ordonnance_id             string,\n"
                + "    splited                   string,\n"
                + "    delivery_provider         string,\n"
                + "    payee                     string,\n"
                + "    referrer_id               string,\n"
                + "    ticket_num                string,\n"
                + "    referrer_name             string,\n"
                + "    out_trade_code            string,\n"
                + "    remark                    string,\n"
                + "    creater_id                string,\n"
                + "    create_time               string,\n"
                + "    modify_time               string,\n"
                + "    del_flag                  string,\n"
                + "    from_warehouse_code       string,\n"
                + "    order_type                int,\n"
                + "    PRIMARY KEY (`id`) NOT ENFORCED\n"
                + "    ) WITH (\n"
                + "          'connector'='datagen'\n"
                + "          )";
        String expected5 = "CREATE TABLE IF NOT EXISTS dwd_sd_so_ordr_detl_r (\n"
                + "                                                     stsc_date string comment '统计日期-天',\n"
                + "                                                     sale_ordr_doc string comment '销售订单编号',\n"
                + "    \n"
                + "                                                     ordr_sour_code string comment '订单来源编码',\n"
                + "                                                     sale_order_proc_time timestamp comment '',\n"
                + "                                                     ordr_sale_time string comment '',\n"
                + "                                                     prmn_prog_type_code string comment '促销方案类型编码',\n"
                + "                                                     ordr_type_code string comment '订单类型编码(销售单/退货单)',\n"
                + "                                                     ordr_cate_code string comment '订单类别编码(标准/团购单/赠品记账(积分兑奖)/财务记账(订金))',\n"
                + "                                                     prmn_prog_code string comment '促销方案编码',\n"
                + "                                                     coup_type string comment '',\n"
                + "                                                     coup_code string comment '',\n"
                + "                                                     line_item INT comment '行项目',\n"
                + "                                                     refn_ordr_type_code string comment '参考单据类型编码(销售单/团购单/处方单/订金退货申请单)',\n"
                + "                                                     refn_ordr_doc string comment '参考单据编号',\n"
                + "                                                     refn_ordr_item INT comment '参考单据行项目号',\n"
                + "                                                     memb_disc_mode string comment '会员优惠方式(会员折扣/会员价/VIP会员价)',\n"
                + "    \n"
                + "                                                     goods_code string comment '商品编码',\n"
                + "                                                     sale_order_details_proc_time timestamp comment '',\n"
                + "                                                     cate_clas_code string comment '',\n"
                + "                                                     cate_clas_name string comment '',\n"
                + "                                                     purc_clas_code_new string comment '',\n"
                + "                                                     purc_clas_name_new string comment '',\n"
                + "                                                     lotn string comment '批号',\n"
                + "                                                     sale_tax string comment '',\n"
                + "                                                     memb_id string comment '会员ID',\n"
                + "                                                     memb_point decimal comment '',\n"
                + "                                                     chk_out_id string comment '',\n"
                + "                                                     casr_id string comment '',\n"
                + "                                                     is_memb_goods string comment '非会员商品标识',\n"
                + "                                                     retail_pric decimal(22,4) comment '零售单价',\n"
                + "    memb_pric decimal(22,4) comment '会员单价',\n"
                + "    exec_pric_type string comment '执行价类型编码',\n"
                + "    \n"
                + "    divid_pric decimal(22,4) comment '医保单价',\n"
                + "    acnt_pric decimal(22,4) comment '记账单价',\n"
                + "    exec_pric decimal(22,4) comment '执行单价',\n"
                + "    dct_amt decimal comment '',\n"
                + "    gift_dct_amt decimal comment '',\n"
                + "    other_free_amt decimal comment '',\n"
                + "    amt1 decimal(22,4) comment '',\n"
                + "    amt2 decimal(22,4) comment '',\n"
                + "    sale_amt decimal(22,4) comment '销售金额',\n"
                + "    sale_qty decimal(22,4) comment '销售数量',\n"
                + "    china_med_qty decimal comment '',\n"
                + "    etl_time timestamp comment '',\n"
                + "    is_n_surd_prof string comment '不让利商品标识',\n"
                + "    cost_pric decimal(22,4) comment '成本单价',\n"
                + "    cost_amt decimal(22,4) comment '',\n"
                + "    \n"
                + "    is_effe_ordr string comment '是否有效订单',\n"
                + "    order_code string comment '',\n"
                + "    is_ecp_self_dstn_ordr string comment '是否ECP自配送订单',\n"
                + "    stat_date string comment '',\n"
                + "    proj_cate_code string comment '项目类别编码',\n"
                + "    purchase_kind_code string comment '项目类别编码',\n"
                + "    sale_goods_snapshot_proc_time timestamp comment '',\n"
                + "    casr_code string comment '收银员编码',\n"
                + "    service_charge decimal(22,4)  comment '',\n"
                + "    sale_pers_id string comment '营销员ID',\n"
                + "    phmc_code string comment '门店编码',\n"
                + "    out_phmc_code  string comment '出货门店',\n"
                + "    is_ydch_flag   string comment '是否异店出货',\n"
                + "    coup_prog_code string comment '券方案编码',\n"
                + "    PRIMARY KEY (stsc_date,sale_ordr_doc,goods_code,line_item,lotn) NOT ENFORCED\n"
                + "    ) WITH (\n"
                + "          'connector'='print'\n"
                + "          )";
        String expected6 = "insert into dwd_sd_so_ordr_detl_r\n"
                + "select ifnull(date_format(cast(t1.sale_time as timestamp), 'yyyyMMdd'), 'NA')  as stsc_date\n"
                + "     , t1.sale_order_code                                                                  as sale_ordr_doc\n"
                + "     \n"
                + "     , (case when coalesce(t.order_from, '') = '' then '9999' else t.order_from end)       as ordr_sour_code\n"
                + "     , t.proc_time                                               as sale_order_proc_time\n"
                + "     , date_format(cast(t1.sale_time as timestamp),'yyyy-MM-dd HH:mm:ss') as ordr_sale_time\n"
                + "     , t1.promotionbill_plan_type                                as prmn_prog_type_code\n"
                + "     , t.bill_type                                               as ordr_type_code\n"
                + "     , t.bill_kind                                               as ordr_cate_code\n"
                + "     , t1.promotionbill_plan_type                                as prmn_prog_code\n"
                + "     , t.coupon_type                                             as coup_type\n"
                + "     , t.coupon_code                                             as coup_code\n"
                + "     , t1.row_num                                                as line_item\n"
                + "     , t1.refer_bill_type                                        as refn_ordr_type_code\n"
                + "     , t1.refer_bill_code                                        as refn_ordr_doc\n"
                + "     , t1.refer_bill_row_num                                     as refn_ordr_item\n"
                + "     , t1.member_coupon_type                                     as memb_disc_mode\n"
                + "     \n"
                + "     , t1.good_code                                              as goods_code\n"
                + "     , t1.proc_time                                              as sale_order_details_proc_time\n"
                + "     , t2.category_code                                          as cate_clas_code\n"
                + "     , t2.category_des                                           as cate_clas_name\n"
                + "     , t2.purchase_kind_code                                     as purc_clas_code_new\n"
                + "     , t2.purchase_kind_des                                      as purc_clas_name_new\n"
                + "     , t1.lot_num                                                as lotn\n"
                + "     , t1.output_tax                                             as sale_tax\n"
                + "     , t.member_id                                               as memb_id\n"
                + "     , t.point_number                                            as memb_point\n"
                + "     , t.cashdesk_id                                             as chk_out_id\n"
                + "     ,case when coalesce(t.casher_code , '') = '' then 'NA' else LPAD(CAST(t.casher_code  as string), 8, '0') end as casr_id\n"
                + "     , t2.is_unmember_goods                                      as is_memb_goods\n"
                + "     , t2.retail_price                                           as retail_pric\n"
                + "     , t2.member_price                                           as memb_pric\n"
                + "     , t1.perform_price_type                                     as exec_pric_type\n"
                + "     \n"
                + "     , t1.apportion_price                                        as divid_pric\n"
                + "     , t1.account_price                                          as acnt_pric\n"
                + "     , t1.perform_price                                          as exec_pric\n"
                + "     , t.cash_free_money                                         as dct_amt\n"
                + "     , t.gift_free_money                                         as gift_dct_amt\n"
                + "     , t.other_free_money                                        as other_free_amt\n"
                + "     , t1.perform_profit_margin                                  as amt1\n"
                + "     , t1.account_price_gross_money                              as amt2\n"
                + "     , t1.account_money                                          as sale_amt\n"
                + "     , t1.quantity                                               as sale_qty\n"
                + "     , t.pair_count                                              as china_med_qty\n"
                + "     , current_timestamp                                         as etl_time\n"
                + "     , t2.is_unfree_goods                                        as is_n_surd_prof\n"
                + "     , t1.cost_price                                             as cost_pric\n"
                + "     , t1.cost_price * t1.quantity                               as cost_amt\n"
                + "     \n"
                + "     , case when t.bill_kind in ('3', '4', '15') then 'N' else 'Y' end as is_effe_ordr\n"
                + "     , coalesce(t.order_code, t.sale_order_code)                 as order_code\n"
                + "     , 'N'                                                       as is_ecp_self_dstn_ordr\n"
                + "     , date_format(cast(t1.sale_time as timestamp), 'yyyyMMdd')        as stat_date\n"
                + "     , t1.item_type                                              as proj_cate_code\n"
                + "     , t2.purchase_kind_code                                     as purchase_kind_code\n"
                + "     , t2.proc_time                                              as sale_goods_snapshot_proc_time\n"
                + "     , t.casher_code                                             as casr_code\n"
                + "     , t1.service_charge                                         as service_charge\n"
                + "     ,case when coalesce(t3.sales_job_number , '') = '' then 'NA' else LPAD(CAST(t3.sales_job_number  as string), 8, '0') end as sale_pers_id\n"
                + "     ,case when t4.from_warehouse_code is not null then t4.from_warehouse_code\n"
                + "           when coalesce(trim(t1.store_code),'')='' then 'NA'\n"
                + "           else trim(t1.store_code) end                               as phmc_code\n"
                + "     ,case when t4.warehouse_code is not null then t4.warehouse_code\n"
                + "           when coalesce(trim(t1.store_code),'')='' then 'NA'\n"
                + "           else trim(t1.store_code) end                               as out_phmc_code\n"
                + "     ,if(t4.pos_code is not null, 'Y', 'N')                      as is_ydch_flag\n"
                + "     ,t.coupon_plan_code                                         as coup_prog_code\n"
                + "from ods_order_sale_order_r t\n"
                + "         inner join ods_order_sale_order_details_r t1\n"
                + "                    on t.sale_order_code = t1.sale_order_code\n"
                + "                        and date_format(cast(t1.sale_time as timestamp), 'yyyyMMdd')=date_format(localtimestamp,'yyyyMMdd')\n"
                + "         left join ods_order_sale_goods_snapshot_r t2\n"
                + "                   on t1.sale_order_code = t2.sale_order_code\n"
                + "                       and t1.good_code = t2.good_code\n"
                + "                       and t2.origin_table='sale_goods_snapshot'\n"
                + "                       and date_format(cast(t2.sale_time as timestamp), 'yyyyMMdd')=date_format(localtimestamp,'yyyyMMdd')\n"
                + "         left join ods_order_sales_ordersalesman_r t3\n"
                + "                   on t1.sale_order_code = t3.sale_order_code\n"
                + "                       and t1.row_num = t3.row_num\n"
                + "                       and date_format(cast(t3.sale_time as timestamp), 'yyyyMMdd')=date_format(localtimestamp,'yyyyMMdd')\n"
                + "         left join ods_zt_ord_order_r t4\n"
                + "                   on t.sale_order_code = t4.pos_code\n"
                + "                       and t4.order_type =1\n"
                + "where date_format(cast(t.sale_time as timestamp), 'yyyyMMdd')=date_format(localtimestamp,'yyyyMMdd')";
        String[] statements = SqlUtil.getStatements(sql);
        Assertions.assertThat(statements.length).isEqualTo(7);
        Assertions.assertThat(statements[0]).isEqualTo(expected0);
        Assertions.assertThat(statements[1]).isEqualTo(expected1);
        Assertions.assertThat(statements[2]).isEqualTo(expected2);
        Assertions.assertThat(statements[3]).isEqualTo(expected3);
        Assertions.assertThat(statements[4]).isEqualTo(expected4);
        Assertions.assertThat(statements[5]).isEqualTo(expected5);
        Assertions.assertThat(statements[6]).isEqualTo(expected6);
    }

    @Test
    public void testGetStatementsByDatagenTest() {
        String sql = "DROP TABLE IF EXISTS source_table3;\n" + "\n"
                + "CREATE TABLE IF NOT EXISTS source_table3 (\n"
                + "  -- 订单id\n"
                + "  `order_id` BIGINT,\n"
                + "  --产品\n"
                + "  `product` BIGINT,\n"
                + "  --金额\n"
                + "  `amount` BIGINT,\n"
                + "  -- 支付时间\n"
                + "  `order_time` as CAST(CURRENT_TIMESTAMP AS TIMESTAMP(3)), -- `在这里插入代码片`\n"
                + "  --WATERMARK\n"
                + "  WATERMARK FOR order_time AS order_time - INTERVAL '2' SECOND\n"
                + ")\n"
                + "WITH\n"
                + "  (\n"
                + "    'connector' = 'datagen',\n"
                + "    'rows-per-second' = '1',\n"
                + "    'fields.order_id.min' = '1',\n"
                + "    'fields.order_id.max' = '2',\n"
                + "    'fields.amount.min' = '1',\n"
                + "    'fields.amount.max' = '10',\n"
                + "    'fields.product.min' = '1',\n"
                + "    'fields.product.max' = '2'\n"
                + "  );\n"
                + "\n"
                + "-- SELECT * FROM source_table3 LIMIT 10;\n"
                + "DROP TABLE IF EXISTS sink_table5;\n"
                + "\n"
                + "CREATE TABLE IF NOT EXISTS sink_table5 (\n"
                + "  --产品\n"
                + "  `product` BIGINT,\n"
                + "  --金额\n"
                + "  `amount` BIGINT,\n"
                + "  --支付时间\n"
                + "  `order_time` TIMESTAMP(3),\n"
                + "  -- 1分钟时间聚合总数\n"
                + "  `one_minute_sum` BIGINT\n"
                + ")\n"
                + "WITH\n"
                + "  ('connector' = 'print');\n"
                + "\n"
                + "INSERT INTO\n"
                + "  sink_table5\n"
                + "SELECT\n"
                + "  product,\n"
                + "  amount,\n"
                + "  order_time,\n"
                + "  SUM(amount) OVER (\n"
                + "    PARTITION BY\n"
                + "      product\n"
                + "    ORDER BY\n"
                + "      order_time\n"
                + "      -- 标识统计范围是1个 product 的最近 1 分钟的数据\n"
                + "      RANGE BETWEEN INTERVAL '1' MINUTE PRECEDING\n"
                + "      AND CURRENT ROW\n"
                + "  ) as one_minute_sum\n"
                + "FROM\n"
                + "  source_table3;";
        String expected0 = "DROP TABLE IF EXISTS source_table3";
        String expected1 = "CREATE TABLE IF NOT EXISTS source_table3 (\n" + "  \n"
                + "  `order_id` BIGINT,\n"
                + "  \n"
                + "  `product` BIGINT,\n"
                + "  \n"
                + "  `amount` BIGINT,\n"
                + "  \n"
                + "  `order_time` as CAST(CURRENT_TIMESTAMP AS TIMESTAMP(3)), \n"
                + "  \n"
                + "  WATERMARK FOR order_time AS order_time - INTERVAL '2' SECOND\n"
                + ")\n"
                + "WITH\n"
                + "  (\n"
                + "    'connector' = 'datagen',\n"
                + "    'rows-per-second' = '1',\n"
                + "    'fields.order_id.min' = '1',\n"
                + "    'fields.order_id.max' = '2',\n"
                + "    'fields.amount.min' = '1',\n"
                + "    'fields.amount.max' = '10',\n"
                + "    'fields.product.min' = '1',\n"
                + "    'fields.product.max' = '2'\n"
                + "  )";
        String expected2 = "DROP TABLE IF EXISTS sink_table5";
        String expected3 = "CREATE TABLE IF NOT EXISTS sink_table5 (\n" + "  \n"
                + "  `product` BIGINT,\n"
                + "  \n"
                + "  `amount` BIGINT,\n"
                + "  \n"
                + "  `order_time` TIMESTAMP(3),\n"
                + "  \n"
                + "  `one_minute_sum` BIGINT\n"
                + ")\n"
                + "WITH\n"
                + "  ('connector' = 'print')";
        String expected4 = "INSERT INTO\n" + "  sink_table5\n"
                + "SELECT\n"
                + "  product,\n"
                + "  amount,\n"
                + "  order_time,\n"
                + "  SUM(amount) OVER (\n"
                + "    PARTITION BY\n"
                + "      product\n"
                + "    ORDER BY\n"
                + "      order_time\n"
                + "      \n"
                + "      RANGE BETWEEN INTERVAL '1' MINUTE PRECEDING\n"
                + "      AND CURRENT ROW\n"
                + "  ) as one_minute_sum\n"
                + "FROM\n"
                + "  source_table3";
        String[] statements = SqlUtil.getStatements(sql);
        Assertions.assertThat(statements.length).isEqualTo(5);
        Assertions.assertThat(statements[0]).isEqualTo(expected0);
        Assertions.assertThat(statements[1]).isEqualTo(expected1);
        Assertions.assertThat(statements[2]).isEqualTo(expected2);
        Assertions.assertThat(statements[3]).isEqualTo(expected3);
        Assertions.assertThat(statements[4]).isEqualTo(expected4);
    }
}
