"""
smbios_data.py: SMBIOS Dictionary for model data
"""

"""
Terms:
  AAPL: AppleInternal Model (ie. not released to public)
  Board ID: The board ID is a unique identifier for the motherboard.
  Firmware Features: Hex bitmask denoting supported abilities of firmware. (ie. APFS, Large BaseSystem, etc.)
  Secure Boot Model: T2/Apple Silicon Model Identifier
  CPU Generation: Stock CPU supported by the board (generally lowest generation)
  Wireless Model: Driver used for wireless networking
  Bluetooth Model: Chipset model
  Screen Size: Size of the screen in inches (generally lowest size if multiple in same model)
  UGA Graphics: If model needs UGA to GOP conversion
  Ethernet Chipset: Vendor of the ethernet chipset (if multiple unique chipset within Vendor, chipset name is used)
  nForce Chipset: If model uses nForce chipset
  Switchable GPUs: If model uses a GMUX
  Stock GPUs: GPUs variations shipped

Reference:
  https://github.com/acidanthera/OpenCorePkg/blob/master/Library/OcMacInfoLib/AutoGenerated.c
"""

from ..detections import device_probe

from . import (
    cpu_data,
    os_data,
    bluetooth_data
)


smbios_dictionary = {
    "MacBook1,1": {
        "Marketing Name": "MacBook (13-inch)",
        "Board ID": "Mac-F4208CC8",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.yonah.value,
        "Max OS Supported": os_data.os_data.snow_leopard,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "Screen Size": 13,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Legacy iSight": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.GMA_950
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "MacBook2,1": {
        "Marketing Name": "MacBook (13-inch Late 2006)",
        "Board ID": "Mac-F4208CA9",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "Screen Size": 13,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Legacy iSight": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.GMA_950
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "MacBook3,1": {
        "Marketing Name": "MacBook (13-inch Late 2007)",
        "Board ID": "Mac-F22788C8",
        "FirmwareFeatures": "0xC0001407",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "Screen Size": 13,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Legacy iSight": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.GMA_X3100
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "MacBook4,1": {
        "Marketing Name": "MacBook (13-inch, Early 2008)",
        "Board ID": "Mac-F22788A9",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 13,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Legacy iSight": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.GMA_X3100
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "MacBook5,1": {
        "Marketing Name": "MacBook (13-inch, Aluminum, Late 2008)",
        "Board ID": "Mac-F42D89C8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 13,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBook5,1_v2": {
        "Board ID": "Mac-F42D89A9",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 13,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBook5,2": {
        "Marketing Name": "MacBook (13-inch, Mid 2009)",
        "Board ID": "Mac-F22788AA",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 13,
        "Ethernet Chipset": "Nvidia",
        "Legacy iSight": True,
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBook6,1": {
        "Marketing Name": "MacBook (13-inch, Late 2009)",
        "Board ID": "Mac-F22C8AC8",
        "FirmwareFeatures": "0xFC0FE13F",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2070,
        "Screen Size": 13,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBook7,1": {
        "Marketing Name": "MacBook (13-inch, Mid 2010)",
        "Board ID": "Mac-F22C89C8",
        "FirmwareFeatures": "0xFC0FE13F",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2070,
        "Screen Size": 13,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBook8,1": {
        "Marketing Name": "MacBook (Retina, 12-inch, Early 2015)",
        "Board ID": "Mac-BE0E8AC46FE800CC",
        "FirmwareFeatures": "0xFC0FE13F",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.broadwell.value,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703_UART,
        "Screen Size": 12,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Broadwell
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBook8,1_v2": {
        "Board ID": "Mac-F305150B0C7DEEEF",
        "FirmwareFeatures": "0xFC0FE13F",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.broadwell.value,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703_UART,
        "Screen Size": 12,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Broadwell
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBook9,1": {
        "Marketing Name": "MacBook (Retina, 12-inch, Early 2016)",
        "Board ID": "Mac-9AE82516C7C6B903",
        "FirmwareFeatures": "0x8FC0FE13F",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.skylake.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703_UART,
        "Screen Size": 12,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Skylake
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBook10,1": {
        "Marketing Name": "MacBook (Retina, 12-inch, 2017)",
        "Board ID": "Mac-EE2EBD4B90B839A8",
        "FirmwareFeatures": "0x8FC0FE13F",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.kaby_lake.value,
        "Max OS Supported": os_data.os_data.ventura,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703_UART,
        "Screen Size": 12,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Kaby_Lake
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookAir1,1": {
        "Marketing Name": "MacBook Air (Original)",
        "Board ID": "Mac-F42C8CC8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.GMA_X3100
        ],
        "Stock Storage": [
            "SATA 1.8",
        ],
    },
    "MacBookAir2,1": {
        "Marketing Name": "MacBook Air (Mid 2009)",
        "Board ID": "Mac-F42D88C8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 13,
        "nForce Chipset": True,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 1.8",
        ],
    },
    "MacBookAir3,1": {
        "Marketing Name": "MacBook Air (11-inch, Late 2010)",
        "Board ID": "Mac-942452F5819B1C1B",
        "FirmwareFeatures": "0xD00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 11,
        "nForce Chipset": True,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "mSATA",
        ],
    },
    "MacBookAir3,2": {
        "Marketing Name": "MacBook Air (13-inch, Late 2010)",
        "Board ID": "Mac-942C5DF58193131B",
        "FirmwareFeatures": "0xD00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 13,
        "nForce Chipset": True,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "mSATA",
        ],
    },
    "MacBookAir4,1": {
        "Marketing Name": "MacBook Air (11-inch, Mid 2011)",
        "Board ID": "Mac-C08A6BB70A942AC2",
        "FirmwareFeatures": "0xD00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.sandy_bridge.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Screen Size": 11,
        "Ethernet Chipset": "Broadcom",  # Set for Apple Thunderbolt Adapter
        "Stock GPUs": [
            device_probe.Intel.Archs.Sandy_Bridge
        ],
        "Stock Storage": [
            "mSATA",
        ],
    },
    "MacBookAir4,2": {
        "Marketing Name": "MacBook Air (13-inch, Mid 2011)",
        "Board ID": "Mac-742912EFDBEE19B3",
        "FirmwareFeatures": "0xD00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.sandy_bridge.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Screen Size": 13,
        "Ethernet Chipset": "Broadcom",  # Set for Apple Thunderbolt Adapter
        "Stock GPUs": [
            device_probe.Intel.Archs.Sandy_Bridge
        ],
        "Stock Storage": [
            "mSATA",
        ],
    },
    "MacBookAir5,1": {
        "Marketing Name": "MacBook Air (11-inch, Mid 2012)",
        "Board ID": "Mac-66F35F19FE2A0D05",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.ivy_bridge.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Screen Size": 11,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Ivy_Bridge
        ],
        "Stock Storage": [
            "mSATA",
        ],
    },
    "MacBookAir5,2": {
        "Marketing Name": "MacBook Air (13-inch, Mid 2012)",
        "Board ID": "Mac-2E6FAB96566FE58C",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.ivy_bridge.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Ivy_Bridge
        ],
        "Stock Storage": [
            "mSATA",
        ],
    },
    "MacBookAir6,1": {
        "Marketing Name": "MacBook Air (11-inch, Mid 2013)",
        "Board ID": "Mac-35C1E88140C3E6CF",
        "FirmwareFeatures": "0xE00FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Screen Size": 11,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell
        ],
        "Stock Storage": [
            "mSATA",
        ],
    },
    "MacBookAir6,2": {
        "Marketing Name": "MacBook Air (13-inch, Mid 2013)",
        "Board ID": "Mac-7DF21CB3ED6977E5",
        "FirmwareFeatures": "0xE00FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookAir7,1": {
        "Marketing Name": "MacBook Air (11-inch, Early 2015)",
        "Board ID": "Mac-9F18E312C5C2BF0B",
        "FirmwareFeatures": "0x8FF0FF576",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.broadwell.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Screen Size": 11,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Broadwell
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookAir7,2": {
        "Marketing Name": "MacBook Air (13-inch, Early 2015)",
        "Board ID": "Mac-937CB26E2E02BB01",
        "FirmwareFeatures": "0x8FF0FF576",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.broadwell.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Broadwell
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookAir8,1": {
        "Marketing Name": "MacBook Air (Retina, 13-inch, 2018)",
        "Board ID": "Mac-827FAC58A8FDFA22",
        "FirmwareFeatures": "0x8FD8FF42E",
        "SecureBootModel": "j140k",
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.sonoma,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookAir8,1_v2": {
        "Board ID": "Mac-112818653D3AABFC",
        "FirmwareFeatures": "0x8FD8FF42E",
        "SecureBootModel": "j140k",  # TODO: Verify
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.sonoma,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "AAPL_MacBookAir8,1": {
        # AppleInternal MacBookAir8,1
        # True Model unknown
        "Board ID": "Mac-827FAC58A8FDFA22",
        "FirmwareFeatures": "0x8FD8FF42E",
        "SecureBootModel": "x589amlu",
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.sonoma,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookAir8,2": {
        "Marketing Name": "MacBook Air (Retina, 13-inch, 2019)",
        "Board ID": "Mac-226CB3C6A851A671",
        "FirmwareFeatures": "0x8FD8FF42E",
        "SecureBootModel": "j140a",
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.sonoma,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookAir9,1": {
        "Marketing Name": "MacBook Air (Retina, 13-inch, 2020)",
        "Board ID": "Mac-0CFF9C7C2B63DF8D",
        "FirmwareFeatures": "0x8FFAFF06E",
        "SecureBootModel": "j230k",
        "CPU Generation": cpu_data.CPUGen.ice_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Ice_Lake
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "AAPL_MacBookAir9,1": {
        # AppleInternal MacBookAir9,1
        # True Model unknown
        "Board ID": "Mac-0CFF9C7C2B63DF8D",
        "FirmwareFeatures": "0x8FFAFF06E",
        "SecureBootModel": "x589icly",
        "CPU Generation": cpu_data.CPUGen.ice_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Ice_Lake
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookAir10,1": {
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j313",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Ethernet Chipset": None,
        "Stock GPUs": [], # TODO: Add Apple Silicon GPU
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookAir10,1": {
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j313",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Ethernet Chipset": None,
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro1,1": {
        "Marketing Name": "MacBook Pro (original)",
        "Board ID": "Mac-F425BEC8",
        "FirmwareFeatures": "",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.yonah.value,
        "Max OS Supported": os_data.os_data.snow_leopard,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "Screen Size": 15,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Stock GPUs": [
            device_probe.AMD.Archs.R500
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "MacBookPro1,2": {
        "Marketing Name": "MacBook Pro (17-inch)",
        "Board ID": "Mac-F42DBEC8",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.yonah.value,
        "Max OS Supported": os_data.os_data.snow_leopard,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "Screen Size": 17,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Stock GPUs": [
            device_probe.AMD.Archs.R500
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "MacBookPro2,1": {
        "Marketing Name": "MacBook Pro (15-inch Core 2 Duo)",
        "Board ID": "Mac-F42189C8",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "Screen Size": 17,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Stock GPUs": [
            device_probe.AMD.Archs.R500
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "MacBookPro2,2": {
        "Marketing Name": "MacBook Pro (15-inch Core 2 Duo)",
        "Board ID": "Mac-F42187C8",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "Screen Size": 15,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Stock GPUs": [
            device_probe.AMD.Archs.R500
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "MacBookPro3,1": {
        "Marketing Name": "MacBook Pro (15-inch,  2.4 2.2GHz)",
        "Board ID": "Mac-F4238BC8",
        "FirmwareFeatures": "0xC0001407",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "Screen Size": 15,  # Shipped with 17 as well
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "MacBookPro3,1_v2": {
        "Board ID": "Mac-F42388C8",
        "FirmwareFeatures": "0xC0001407",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "Screen Size": 15,  # Shipped with 17 as well
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "MacBookPro4,1": {
        "Marketing Name": "MacBook Pro (17-inch, Early 2008)",
        "Board ID": "Mac-F42C89C8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 15,  # Shipped with 17 as well
        "Switchable GPUs": True,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "MacBookPro4,1_v2": {
        "Board ID": "Mac-F42C86C8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 15,  # Shipped with 17 as well
        "Switchable GPUs": True,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "MacBookPro5,1": {
        "Marketing Name": "MacBook Pro (15-inch, Late 2008)",
        "Board ID": "Mac-F42D86C8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro5,1_v2": {
        "Board ID": "Mac-F42D86A9",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro5,2": {
        "Marketing Name": "MacBook Pro (17-inch, Early 2009)",
        "Board ID": "Mac-F2268EC8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 17,
        "Switchable GPUs": True,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro5,3": {
        "Marketing Name": "MacBook Pro (15-inch, 2.53GHz, Mid 2009)",
        "Board ID": "Mac-F22587C8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro5,4": {
        "Marketing Name": "MacBook Pro (15-inch, 2.53GHz, Mid 2009)",
        "Board ID": "Mac-F22587A1",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro5,5": {
        "Marketing Name": "MacBook Pro (13-inch, Mid 2009)",
        "Board ID": "Mac-F2268AC8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 13,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro6,1": {
        "Marketing Name": "MacBook Pro (17-inch, Mid 2010)",
        "Board ID": "Mac-F22589C8",
        "FirmwareFeatures": "0xC00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.nehalem.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2070,
        "Screen Size": 17,
        "Switchable GPUs": True,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Iron_Lake,
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro6,2": {
        "Marketing Name": "MacBook Pro (15-inch, Mid 2010)",
        "Board ID": "Mac-F22586C8",
        "FirmwareFeatures": "0xC00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.nehalem.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2070,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Iron_Lake,
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro7,1": {
        "Marketing Name": "MacBook Pro (13-inch, Mid 2010)",
        "Board ID": "Mac-F222BEC8",
        "FirmwareFeatures": "0xC00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Screen Size": 13,
        "Ethernet Chipset": "Broadcom",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro8,1": {
        "Marketing Name": "MacBook Pro (13-inch, Early 2011)",
        "Board ID": "Mac-94245B3640C91C81",
        "FirmwareFeatures": "0xC00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.sandy_bridge.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2070,
        "Screen Size": 13,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Sandy_Bridge,
            device_probe.AMD.Archs.TeraScale_2
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro8,2": {
        "Marketing Name": "MacBook Pro (15-inch, Early 2011)",
        "Board ID": "Mac-94245A3940C91C80",
        "FirmwareFeatures": "0xC00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.sandy_bridge.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2070,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Sandy_Bridge,
            device_probe.AMD.Archs.TeraScale_2
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro8,3": {
        "Marketing Name": "MacBook Pro (17-inch, Early 2011)",
        "Board ID": "Mac-942459F5819B171B",
        "FirmwareFeatures": "0xC00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.sandy_bridge.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2070,
        "Screen Size": 17,
        "Switchable GPUs": True,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Sandy_Bridge,
            device_probe.AMD.Archs.TeraScale_2
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "AAPL_MacBookPro8,3": {
        # AppleInternal MacBookPro8,3
        # True Model unknown
        "Board ID": "Mac-94245AF5819B141B",
        "FirmwareFeatures": "0xC00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.sandy_bridge.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2070,
        "Screen Size": 17,
        "Switchable GPUs": True,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Sandy_Bridge,
            device_probe.AMD.Archs.TeraScale_2
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro9,1": {
        "Marketing Name": "MacBook Pro (15-inch, Mid 2012)",
        "Board ID": "Mac-4B7AC7E43945597E",
        "FirmwareFeatures": "0xC00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.ivy_bridge.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Ivy_Bridge,
            device_probe.NVIDIA.Archs.Kepler
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro9,2": {
        "Marketing Name": "MacBook Pro (13-inch, Mid 2012)",
        "Board ID": "Mac-6F01561E16C75D06",
        "FirmwareFeatures": "0xC10DF577",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.ivy_bridge.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Screen Size": 13,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Ivy_Bridge,
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "MacBookPro10,1": {
        "Marketing Name": "MacBook Pro (Retina, Mid 2012)",
        "Board ID": "Mac-C3EC7CD22292981F",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.ivy_bridge.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Ivy_Bridge,
            device_probe.NVIDIA.Archs.Kepler
        ],
        "Stock Storage": [
            "mSATA",
        ],
    },
    "MacBookPro10,2": {
        "Marketing Name": "MacBook Pro (Retina, 13-inch, Late 2012)",
        "Board ID": "Mac-AFD8A9D944EA4843",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.ivy_bridge.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Ivy_Bridge,
        ],
        "Stock Storage": [
            "mSATA",
        ],
    },
    "MacBookPro11,1": {
        "Marketing Name": "MacBook Pro (Retina, 13-inch, Late 2013)",
        "Board ID": "Mac-189A3D4F975D5FFC",
        "FirmwareFeatures": "0xEB0FF577",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro11,2": {
        "Marketing Name": "MacBook Pro (Retina, 15-inch, Mid 2014)",
        "Board ID": "Mac-3CBD00234E554E41",
        "FirmwareFeatures": "0xEB0FF577",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Screen Size": 15,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro11,3": {
        "Marketing Name": "MacBook Pro (Retina, 15-inch, Late 2013)",
        "Board ID": "Mac-2BD1B31983FE1663",
        "FirmwareFeatures": "0xEB0FF577",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
            device_probe.NVIDIA.Archs.Kepler,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro11,4": {
        "Marketing Name": "MacBook Pro (Retina, 15-inch, Mid 2015)",
        "Board ID": "Mac-06F11FD93F0323C5",
        "FirmwareFeatures": "0x8EB0FF577",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Screen Size": 15,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro11,5": {
        "Marketing Name": "MacBook Pro (Retina, 15-inch, Mid 2014)",
        "Board ID": "Mac-06F11F11946D27C5",
        "FirmwareFeatures": "0x8EB0FF577",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
            device_probe.AMD.Archs.Legacy_GCN_7000
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro12,1": {
        "Marketing Name": "MacBook Pro (Retina, 13-inch, Early 2015)",
        "Board ID": "Mac-E43C1C25D4880AD6",
        "FirmwareFeatures": "0x8FD0FF576",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.broadwell.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Broadwell,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro13,1": {
        "Marketing Name": "MacBook Pro (13-inch, 2016, Two Thunderbolt 3 ports)",
        "Board ID": "Mac-473D31EABEB93F9B",
        "FirmwareFeatures": "0x8FC0FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.skylake.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703_UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Skylake,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro13,2": {
        "Marketing Name": "MacBook Pro (13-inch, 2016, Four Thunderbolt 3 Ports)",
        "Board ID": "Mac-66E35819EE2D0D05",
        "FirmwareFeatures": "0x8FC0FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.skylake.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703_UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Skylake,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro13,3": {
        "Marketing Name": "MacBook Pro (15-inch, 2016)",
        "Board ID": "Mac-A5C67F76ED83108C",
        "FirmwareFeatures": "0x8FC0FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.skylake.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703_UART,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Skylake,
            device_probe.AMD.Archs.Polaris
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro14,1": {
        "Marketing Name": "MacBook Pro (13-inch, 2017, Two Thunderbolt 3 ports)",
        "Board ID": "Mac-B4831CEBD52A0C4C",
        "FirmwareFeatures": "0x8FF0FF57E",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.kaby_lake.value,
        "Max OS Supported": os_data.os_data.ventura,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703_UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Kaby_Lake,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro14,2": {
        "Marketing Name": "MacBook Pro (13-inch, 2017, Four Thunderbolt 3 Ports)",
        "Board ID": "Mac-CAD6701F7CEA0921",
        "FirmwareFeatures": "0x8FF0FF57E",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.kaby_lake.value,
        "Max OS Supported": os_data.os_data.ventura,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703_UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Kaby_Lake,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro14,3": {
        "Marketing Name": "MacBook Pro (15-inch, 2017)",
        "Board ID": "Mac-551B86E5744E2388",
        "FirmwareFeatures": "0x8FF0FF57E",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.kaby_lake.value,
        "Max OS Supported": os_data.os_data.ventura,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703_UART,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Kaby_Lake,
            device_probe.AMD.Archs.Polaris
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro15,1": {
        "Marketing Name": "MacBook Pro (15-inch, 2018)",
        "Board ID": "Mac-937A206F2EE63C01",
        "FirmwareFeatures": "0x8FD8FF426",
        "SecureBootModel": "j680",
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake,
            device_probe.AMD.Archs.Polaris
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro15,2": {
        "Marketing Name": "MacBook Pro (13-inch, 2018, Four Thunderbolt 3 Ports)",
        "Board ID": "Mac-827FB448E656EC26",
        "FirmwareFeatures": "0x8FD8FF426",
        "SecureBootModel": "j132",
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro15,3": {
        "Marketing Name": "MacBook Pro (15-inch, 2019)",
        "Board ID": "Mac-1E7E29AD0135F9BC",
        "FirmwareFeatures": "0x8FD8FF426",
        "SecureBootModel": "j780",
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 15,
        "Switchable GPUs": True,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake,
            device_probe.AMD.Archs.Vega
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro15,4": {
        "Marketing Name": "MacBook Pro (13-inch, 2019, Two Thunderbolt 3 ports)",
        "Board ID": "Mac-53FDB3D8DB8CA971",
        "FirmwareFeatures": "0x8FD8FF426",
        "SecureBootModel": "j213",
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro16,1": {
        "Marketing Name": "MacBook Pro (16-inch, 2019)",
        "Board ID": "Mac-E1008331FDC96864",
        "FirmwareFeatures": "0x8FDAFF066",
        "SecureBootModel": "j152f",
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 16,
        "Switchable GPUs": True,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake,
            device_probe.AMD.Archs.Navi
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro16,2": {
        "Marketing Name": "MacBook Pro (13-inch, 2020, Four Thunderbolt 3 ports)",
        "Board ID": "Mac-5F9802EFE386AA28",
        "FirmwareFeatures": "0x8FFFFFF7F",
        "SecureBootModel": "j214k",
        "CPU Generation": cpu_data.CPUGen.ice_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Ice_Lake,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro16,3": {
        "Marketing Name": "MacBook Pro (13-inch, 2020, Two Thunderbolt 3 ports)",
        "Board ID": "Mac-E7203C0F68AA0004",
        "FirmwareFeatures": "0x8FDAFF066",
        "SecureBootModel": "j223",
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro16,4": {
        "Marketing Name": "MacBook Pro (16-inch, 2019)",
        "Board ID": "Mac-A61BADE1FDAD7B05",
        "FirmwareFeatures": "0x8FDAFF066",
        "SecureBootModel": "j215",
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Screen Size": 16,
        "Switchable GPUs": True,
        "Ethernet Chipset": None,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake,
            device_probe.AMD.Archs.Navi
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro17,1": {
        "Marketing Name": "MacBook Pro (13-inch, M1, 2020)",
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j293",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Screen Size": 13,
        "Ethernet Chipset": None,
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro18,1": {
        "Marketing Name": "MacBook Pro (16-inch, 2021)",
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j316s",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Screen Size": 16,
        "Ethernet Chipset": None,
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro18,2": {
        "Marketing Name": "MacBook Pro (16-inch, 2021)",
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j316c",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Screen Size": 16,
        "Ethernet Chipset": None,
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro18,3": {
        "Marketing Name": "MacBook Pro (14-inch, 2021)",
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j314s",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Screen Size": 14,
        "Ethernet Chipset": None,
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacBookPro18,4": {
        "Marketing Name": "MacBook Pro (14-inch, 2021)",
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j314c",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Screen Size": 14,
        "Ethernet Chipset": None,
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "Mac14,7": {
        "Marketing Name": "MacBook Pro (13-inch, M2, 2022)",
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "J493AP",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Screen Size": 14,
        "Ethernet Chipset": None,
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "Macmini1,1": {
        "Marketing Name": "Mac mini (Early 2006)",
        "Board ID": "Mac-F4208EC8",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.yonah.value,
        "Max OS Supported": os_data.os_data.snow_leopard,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "Ethernet Chipset": "Marvell",
        "Stock GPUs": [
            device_probe.Intel.Archs.GMA_950
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "Macmini2,1": {
        "Marketing Name": "Mac mini (Mid 2007)",
        "Board ID": "Mac-F4208EAA",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "Ethernet Chipset": "Marvell",
        "Stock GPUs": [
            device_probe.Intel.Archs.GMA_950
        ],
        "Stock Storage": [
            "SATA 2.5",
            "PATA",
        ],
    },
    "Macmini3,1": {
        "Marketing Name": "Mac mini (Early 2009)",
        "Board ID": "Mac-F22C86C8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "Macmini4,1": {
        "Marketing Name": "Mac mini (Mid 2010)",
        "Board ID": "Mac-F2208EC8",
        "FirmwareFeatures": "0xC00C9423",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2070,
        "Ethernet Chipset": "Broadcom",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "Macmini5,1": {
        "Marketing Name": "Mac mini (Mid 2011)",
        "Board ID": "Mac-8ED6AF5B48C039E1",
        "FirmwareFeatures": "0xD00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.sandy_bridge.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Sandy_Bridge
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "Macmini5,2": {
        "Marketing Name": "Mac mini (Mid 2011)",
        "Board ID": "Mac-4BC72D62AD45599E",
        "FirmwareFeatures": "0xD00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.sandy_bridge.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Sandy_Bridge,
            device_probe.AMD.Archs.TeraScale_2
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "Macmini5,3": {
        "Marketing Name": "Mac mini Server (Mid 2011)",
        "Board ID": "Mac-7BA5B2794B2CDB12",
        "FirmwareFeatures": "0xD00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.sandy_bridge.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Sandy_Bridge,
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "Macmini6,1": {
        "Marketing Name": "Mac mini (Late 2012)",
        "Board ID": "Mac-031AEE4D24BFF0B1",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.ivy_bridge.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Ivy_Bridge,
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "Macmini6,2": {
        "Marketing Name": "Mac mini Server (Late 2012)",
        "Board ID": "Mac-F65AE981FFA204ED",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.ivy_bridge.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Ivy_Bridge,
        ],
        "Stock Storage": [
            "SATA 2.5",
        ],
    },
    "Macmini7,1": {
        "Marketing Name": "Mac mini (Late 2014)",
        "Board ID": "Mac-35C5E08120C7EEAF",
        "FirmwareFeatures": "0x8E00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
        ],
        "Stock Storage": [
            "SATA 2.5",
            "NVMe",
        ],
    },
    "Macmini8,1": {
        "Marketing Name": "Mac mini (2018)",
        "Board ID": "Mac-7BA5B2DFE22DDD8C",
        "FirmwareFeatures": "0x8FD8FF466",
        "SecureBootModel": "j174",
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake,
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "Macmini9,1": {
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j274",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "iMac4,1": {
        "Marketing Name": "iMac (20-inch, Early 2006)",
        "Board ID": "Mac-F42786C8",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.yonah.value,
        "Max OS Supported": os_data.os_data.snow_leopard,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Legacy iSight": True,
        "Stock GPUs": [
            device_probe.AMD.Archs.R500,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "iMac4,2": {
        "Marketing Name": "iMac (17-inch, Mid 2006)",
        "Board ID": "Mac-F4218EC8",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.yonah.value,
        "Max OS Supported": os_data.os_data.snow_leopard,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Legacy iSight": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.GMA_950,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "iMac5,1": {
        "Marketing Name": "iMac (17-inch, Late 2006)",
        "Board ID": "Mac-F4228EC8",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Legacy iSight": True,
        "Stock GPUs": [
            device_probe.AMD.Archs.R500,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "iMac5,2": {
        "Marketing Name": "iMac (17-inch, Late 2006 CD)",
        "Board ID": "Mac-F4218EC8",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Legacy iSight": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.GMA_950,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "iMac6,1": {
        "Marketing Name": "iMac (24-inch, Late 2006)",
        "Board ID": "Mac-F4218FC8",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Legacy iSight": True,
        "Stock GPUs": [
            device_probe.AMD.Archs.R500,
            device_probe.NVIDIA.Archs.Curie
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "iMac7,1": {
        "Marketing Name": "iMac (20-inch, Mid 2007)",
        "Board ID": "Mac-F42386C8",
        "FirmwareFeatures": "0xC0001407",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn,  # Stock models shipped with Conroe
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.AMD.Archs.TeraScale_1,
            device_probe.NVIDIA.Archs.Tesla,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "iMac7,1_v2": {
        "Board ID": "Mac-F4238CC8",
        "FirmwareFeatures": "0xC0001407",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn,  # Stock models shipped with Conroe
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.AMD.Archs.TeraScale_1,
            device_probe.NVIDIA.Archs.Tesla,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "iMac8,1": {
        "Marketing Name": "iMac (20-inch, Early 2008)",
        "Board ID": "Mac-F227BEC8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.AMD.Archs.TeraScale_1,
            device_probe.NVIDIA.Archs.Tesla,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "iMac8,1_v2": {
        "Board ID": "Mac-F226BEC8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "UGA Graphics": True,
        "Ethernet Chipset": "Marvell",
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.AMD.Archs.TeraScale_1,
            device_probe.NVIDIA.Archs.Tesla,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "iMac9,1": {
        "Marketing Name": "iMac (20-inch, Mid 2009)",
        "Board ID": "Mac-F2218FA9",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla,
            device_probe.AMD.Archs.TeraScale_1,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "iMac9,1_v2": {
        "Board ID": "Mac-F2218EA9",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "iMac9,1_v3": {
        "Board ID": "Mac-F2218EC8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "iMac9,1_v4": {
        "Board ID": "Mac-F2218FC8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "iMac10,1": {
        "Marketing Name": "iMac (27-inch, Late 2009)",
        "Board ID": "Mac-F221DCC8",
        # "Board ID": "Mac-F2268CC8",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.AMD.Archs.TeraScale_1,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "iMac10,1_v2": {
        "Board ID": "Mac-F2268CC8",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.AMD.Archs.TeraScale_1,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "iMac10,1_v3": {
        "Board ID": "Mac-F2268DC8",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Nvidia",
        "nForce Chipset": True,
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.AMD.Archs.TeraScale_1,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "iMac11,1": {
        "Marketing Name": "iMac (27-inch, Late 2009)",
        "Board ID": "Mac-F2268DAE",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.nehalem.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Broadcom",
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.AMD.Archs.TeraScale_1,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "iMac11,2": {
        "Marketing Name": "iMac (21.5-inch, Mid 2010)",
        "Board ID": "Mac-F2238AC8",
        "FirmwareFeatures": "0xC00C9423",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.nehalem.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Broadcom",
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.AMD.Archs.TeraScale_1,
            device_probe.AMD.Archs.TeraScale_2,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "iMac11,3": {
        "Marketing Name": "iMac (27-inch, Mid 2010)",
        "Board ID": "Mac-F2238BAE",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.nehalem.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Broadcom",
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.AMD.Archs.TeraScale_1,
            device_probe.AMD.Archs.TeraScale_2,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "iMac12,1": {
        "Marketing Name": "iMac (21.5-inch, Mid 2011)",
        "Board ID": "Mac-942B5BF58194151B",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.sandy_bridge.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Broadcom",
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.Intel.Archs.Sandy_Bridge,
            device_probe.AMD.Archs.TeraScale_2,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "iMac12,2": {
        "Marketing Name": "iMac (27-inch, Mid 2011)",
        "Board ID": "Mac-942B59F58194171B",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.sandy_bridge.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Broadcom",
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.Intel.Archs.Sandy_Bridge,
            device_probe.AMD.Archs.TeraScale_2,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "AAPL_iMac12,2": {
        # AppleInternal iMac12,2
        # True Model unknown
        "Board ID": "Mac-942B5B3A40C91381",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.sandy_bridge.value,
        "Max OS Supported": os_data.os_data.high_sierra,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Broadcom",
        "Socketed GPUs": "MXM",
        "Stock GPUs": [
            device_probe.Intel.Archs.Sandy_Bridge,
            device_probe.AMD.Archs.TeraScale_2,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "iMac13,1": {
        "Marketing Name": "iMac (21.5-inch, Late 2012)",
        "Board ID": "Mac-00BE6ED71E35EB86",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.ivy_bridge.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Ivy_Bridge,
            device_probe.NVIDIA.Archs.Kepler,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "mSATA"
        ],
    },
    "iMac13,2": {
        "Marketing Name": "iMac (27-inch, Late 2012)",
        "Board ID": "Mac-FC02E91DDD3FA6A4",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.ivy_bridge.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Ivy_Bridge,
            device_probe.NVIDIA.Archs.Kepler,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "mSATA"
        ],
    },
    "iMac13,3": {
        "Marketing Name": "iMac (21.5-inch, Early 2013)",
        "Board ID": "Mac-7DF2A3B5E5D671ED",
        "FirmwareFeatures": "0xE00DE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.ivy_bridge.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4360,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v1,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Ivy_Bridge,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "mSATA"
        ],
    },
    "iMac14,1": {
        "Marketing Name": "iMac (21.5-inch, Late 2013)",
        "Board ID": "Mac-031B6874CF7F642A",
        "FirmwareFeatures": "0xFB0FF577",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac14,2": {
        "Marketing Name": "iMac (27-inch, Late 2013)",
        "Board ID": "Mac-27ADBB7B4CEE8E61",
        "FirmwareFeatures": "0xE00FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
            device_probe.NVIDIA.Archs.Kepler,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac14,3": {
        "Marketing Name": "iMac (21.5-inch, Late 2013)",
        "Board ID": "Mac-77EB7D7DAF985301",
        "FirmwareFeatures": "0xE00FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.catalina,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
            device_probe.NVIDIA.Archs.Kepler,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac14,4": {
        "Marketing Name": "iMac (21.5-inch, Mid 2014)",
        "Board ID": "Mac-81E3E92DD6088272",
        "FirmwareFeatures": "0xF00FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac15,1": {
        "Marketing Name": "iMac (Retina 5K, 27-inch, Mid 2015)",
        "Board ID": "Mac-42FD25EABCABB274",
        "FirmwareFeatures": "0xF80FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
            device_probe.AMD.Archs.Legacy_GCN_7000,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac15,1_v2": {
        "Board ID": "Mac-FA842E06C61E91C5",
        "FirmwareFeatures": "0xF80FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
            device_probe.AMD.Archs.Legacy_GCN_7000,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac16,1": {
        "Marketing Name": "iMac (21.5-inch, Late 2015)",
        "Board ID": "Mac-A369DDC4E67F1C45",
        "FirmwareFeatures": "0x8FC0FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.broadwell.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Broadwell,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac16,2": {
        "Marketing Name": "iMac (Retina 4K, 21.5-inch, Late 2015)",
        "Board ID": "Mac-FFE5EF870D7BA81A",
        "FirmwareFeatures": "0x8FC0FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.broadwell.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Broadwell,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac17,1": {
        "Marketing Name": "iMac (Retina 5K, 27-inch, Late 2015)",
        "Board ID": "Mac-DB15BD556843C820",
        "FirmwareFeatures": "0x8FC0FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.skylake.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.Skylake,
            device_probe.AMD.Archs.Legacy_GCN_8000,
            device_probe.AMD.Archs.Legacy_GCN_9000,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac17,1_v2": {
        "Board ID": "Mac-65CE76090165799A",
        "FirmwareFeatures": "0x8FC0FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.skylake.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.Skylake,
            device_probe.AMD.Archs.Legacy_GCN_8000,
            device_probe.AMD.Archs.Legacy_GCN_9000,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac17,1_v3": {
        "Board ID": "Mac-B809C3757DA9BB8D",
        "FirmwareFeatures": "0x8FC0FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.skylake.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.Skylake,
            device_probe.AMD.Archs.Legacy_GCN_8000,
            device_probe.AMD.Archs.Legacy_GCN_9000,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac18,1": {
        "Marketing Name": "iMac (21.5-inch, 2017)",
        "Board ID": "Mac-4B682C642B45593E",
        "FirmwareFeatures": "0x8FD0FF576",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.kaby_lake.value,
        "Max OS Supported": os_data.os_data.ventura,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.Intel.Archs.Kaby_Lake,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac18,2": {
        "Marketing Name": "iMac (Retina 4K, 21.5-inch, 2017)",
        "Board ID": "Mac-77F17D7DA9285301",
        "FirmwareFeatures": "0x8FD0FF576",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.kaby_lake.value,
        "Max OS Supported": os_data.os_data.ventura,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703,
        "Ethernet Chipset": "Broadcom",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.Kaby_Lake,
            device_probe.AMD.Archs.Polaris,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac18,3": {
        "Marketing Name": "iMac (Retina 5K, 27-inch, 2017)",
        "Board ID": "Mac-BE088AF8C5EB4FA2",
        "FirmwareFeatures": "0x8FD0FF576",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.kaby_lake.value,
        "Max OS Supported": os_data.os_data.ventura,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20703,
        "Ethernet Chipset": "Broadcom",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.Kaby_Lake,
            device_probe.AMD.Archs.Polaris,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac19,1": {
        "Marketing Name": "iMac (Retina 5K, 27-inch, 2019)",
        "Board ID": "Mac-AA95B1DDAB278B95",
        "FirmwareFeatures": "0x8FD8FF576",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Ethernet Chipset": "Broadcom",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake,
            device_probe.AMD.Archs.Polaris,
            device_probe.AMD.Archs.Vega,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "AAPL_iMac19,1": {
        # AppleInternal iMac19,1 unit
        "Board ID": "Mac-CF21D135A7D34AA6",
        "FirmwareFeatures": "0x8FD8FF576",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Ethernet Chipset": "Broadcom",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake,
            device_probe.AMD.Archs.Polaris,
            device_probe.AMD.Archs.Vega,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac19,2": {
        "Marketing Name": "iMac (Retina 4K, 21.5-inch, 2019)",
        "Board ID": "Mac-63001698E7A34814",
        "FirmwareFeatures": "0x8FD8FF576",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Ethernet Chipset": "Broadcom",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.Coffee_Lake,
            device_probe.AMD.Archs.Polaris,
            device_probe.AMD.Archs.Vega,
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe"
        ],
    },
    "iMac20,1": {
        "Marketing Name": "iMac (Retina 5K, 27-inch, 2020)",
        "Board ID": "Mac-CFF7D910A743CAAF",
        "FirmwareFeatures": "0x8FFB3F066",
        "SecureBootModel": "j185",
        "CPU Generation": cpu_data.CPUGen.comet_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Ethernet Chipset": "Broadcom",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.Comet_Lake,
            device_probe.AMD.Archs.Navi,
        ],
        "Stock Storage": [
            "NVMe"
        ],
    },
    "iMac20,2": {
        "Marketing Name": "iMac (Retina 5K, 27-inch, 2020)",
        "Board ID": "Mac-AF89B6D9451A490B",
        "FirmwareFeatures": "0x8FFB3F066",
        "SecureBootModel": "j185f",
        "CPU Generation": cpu_data.CPUGen.comet_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Ethernet Chipset": "Broadcom",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.Intel.Archs.Comet_Lake,
            device_probe.AMD.Archs.Navi,
        ],
        "Stock Storage": [
            "NVMe"
        ],
    },
    "iMac21,1": {
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j456",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe"
        ],
    },
    "iMac21,2": {
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j457",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe"
        ],
    },
    "iMacPro1,1": {
        "Marketing Name": "iMac Pro (2017)",
        "Board ID": "Mac-7BA5B2D9E42DDD94",
        "FirmwareFeatures": "0xFD8FF53E",
        "SecureBootModel": "j137",
        "CPU Generation": cpu_data.CPUGen.skylake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Ethernet Chipset": "Aquantia",
        "Dual DisplayPort Display": True,
        "Stock GPUs": [
            device_probe.AMD.Archs.Vega,
        ],
        "Stock Storage": [
            "NVMe"
        ],
    },
    "MacPro1,1": {
        "Marketing Name": "Mac Pro",
        "Board ID": "Mac-F4208DC8",
        "FirmwareFeatures": "0x80000015",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "UGA Graphics": True,
        "Ethernet Chipset": "Intel 80003ES2LAN",
        "Socketed GPUs": "PCIe",
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Curie
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "MacPro2,1": {
        "Marketing Name": "Mac Pro",
        "Board ID": "Mac-F4208DA9",
        "FirmwareFeatures": "0xC0000015",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "UGA Graphics": True,
        "Ethernet Chipset": "Intel 80003ES2LAN",
        "Socketed GPUs": "PCIe",
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Curie
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "MacPro3,1": {
        "Marketing Name": "Mac Pro (Early 2008)",
        "Board ID": "Mac-F42C88C8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm43224,
        "Bluetooth Model": bluetooth_data.bluetooth_data.APPLE_CSR,
        "UGA Graphics": True,
        "Ethernet Chipset": "Intel 80003ES2LAN",
        "Socketed GPUs": "PCIe",
        "Stock GPUs": [
            device_probe.AMD.Archs.TeraScale_1,
            device_probe.AMD.Archs.Polaris,

        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "MacPro4,1": {
        "Marketing Name": "Mac Pro (Early 2009)",
        "Board ID": "Mac-F221BEC8",
        "FirmwareFeatures": "0xE001F537",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.nehalem.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": device_probe.Atheros.Chipsets.AirPortAtheros40,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Intel 82574L",
        "Socketed GPUs": "PCIe",
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla,
            device_probe.AMD.Archs.Polaris,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "MacPro5,1": {
        "Marketing Name": "Mac Pro (Mid 2010)",
        "Board ID": "Mac-F221BEC8",
        "FirmwareFeatures": "0xE80FE137",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.nehalem.value,
        "Max OS Supported": os_data.os_data.mojave,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirPortBrcm4331,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM2046,
        "Ethernet Chipset": "Intel 82574L",
        "Socketed GPUs": "PCIe",
        "Stock GPUs": [
            device_probe.AMD.Archs.TeraScale_2,
            device_probe.AMD.Archs.Polaris,
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    "MacPro6,1": {
        "Marketing Name": "Mac Pro (Late 2013)",
        "Board ID": "Mac-F60DEB81FF30ACF6",
        "FirmwareFeatures": "0x8E90FF576",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.ivy_bridge.value,
        "Max OS Supported": os_data.os_data.monterey,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [
            device_probe.AMD.Archs.Legacy_GCN_7000
        ],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "MacPro7,1": {
        "Marketing Name": "Mac Pro (2019)",
        "Board ID": "Mac-27AD2F918AE68F61",
        "FirmwareFeatures": "0x8FDAFF066",
        "SecureBootModel": "j160",
        "CPU Generation": cpu_data.CPUGen.coffee_lake.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.UART,
        "Ethernet Chipset": "Aquantia",
        "Socketed GPUs": "PCIe",
        "Stock GPUs": [
            device_probe.AMD.Archs.Polaris,
            device_probe.AMD.Archs.Vega,
            device_probe.AMD.Archs.Navi
        ],
        "Stock Storage": [
            "SATA 3.5",
            "NVMe",
        ],
    },
    "Xserve1,1": {
        "Marketing Name": "Xserve (Late 2006)",
        "Board ID": "Mac-F4208AC8",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.conroe.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "UGA Graphics": True,
        "Ethernet Chipset": "Intel 80003ES2LAN",
        "Socketed GPUs": "PCIe",
        "Stock GPUs": [
            device_probe.AMD.Archs.R500
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "Xserve2,1": {
        "Marketing Name": "Xserve (Early 2008)",
        "Board ID": "Mac-F42289C8",
        "FirmwareFeatures": "0xC0001403",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.lion,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "UGA Graphics": True,
        "Ethernet Chipset": "Intel 80003ES2LAN",
        "Socketed GPUs": "PCIe",
        "Stock GPUs": [
            device_probe.AMD.Archs.R500
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "Xserve3,1": {
        "Marketing Name": "Xserve (Early 2009)",
        "Board ID": "Mac-F223BEC8",
        "FirmwareFeatures": "0xE001F537",
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.nehalem.value,
        "Max OS Supported": os_data.os_data.el_capitan,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Ethernet Chipset": "Intel 82574L",
        "Socketed GPUs": "PCIe",
        "Stock GPUs": [
            device_probe.NVIDIA.Archs.Tesla
        ],
        "Stock Storage": [
            "SATA 3.5",
        ],
    },
    # Mac Studio M1 Max
    "Mac13,1": {
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j375c",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Ethernet Chipset": "Aquantia",
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe",
        ],
    },
    # Mac Studio M1 Ultra
    "Mac13,2": {
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j375d",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Ethernet Chipset": "Aquantia",
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "ADP2,1": {
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.pentium_4.value,
        "Max OS Supported": os_data.os_data.leopard,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Socketed GPUs": "PCIe",
        "Stock GPUs": [
            device_probe.Intel.Archs.GMA_950
        ],
        "Stock Storage": [
            "SATA 3.5",
            "PATA",
        ],
    },
    "ADP3,2": {
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j273a",
        "CPU Generation": cpu_data.CPUGen.apple_dtk.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": device_probe.Broadcom.Chipsets.AppleBCMWLANBusInterfacePCIe,
        "Bluetooth Model": bluetooth_data.bluetooth_data.PCIe,
        "Ethernet Chipset": "Broadcom",
        "Stock GPUs": [],
        "Stock Storage": [
            "NVMe",
        ],
    },
    "AAPLJ53,1": {
        # AppleInternal MacBookPro11,4
        "Board ID": "Mac-C08A65A66A9A3BA2",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": cpu_data.CPUGen.haswell.value,
        "Max OS Supported": os_data.os_data.mavericks,
        "Wireless Model": device_probe.Broadcom.Chipsets.AirportBrcmNIC,
        "Bluetooth Model": bluetooth_data.bluetooth_data.BRCM20702_v2,
        "Stock GPUs": [
            device_probe.Intel.Archs.Haswell,
        ],
        "Stock Storage": [
            "NVMe",
        ],

    },
    "Intel Virtual Machine": {
        "Board ID": "VMM-x86_64",
        "FirmwareFeatures": None,
        "SecureBootModel": "x86legacy",
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "VirtualMac1,1": {
        # Apple Silicon Virtual Machine
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "vma1",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "VirtualMac2,1": {
        # Apple Silicon Virtual Machine
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "vma2macos",
        "CPU Generation": cpu_data.CPUGen.apple_silicon.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "VMware7,1": {
        # VMware Virtual Machine
        "Board ID": "440BX Desktop Reference Platform",
        "FirmwareFeatures": None,
        "SecureBootModel": "x86legacy",
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "Parallels17,1": {
        # Parallels Virtual Machine
        "Board ID": "Parallels Virtual Platform",
        "FirmwareFeatures": None,
        "SecureBootModel": "x86legacy",
        "CPU Generation": cpu_data.CPUGen.penryn.value,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "iBridge2,11": {
        # Unknown ID, Intel based, present in Monterey
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j230",
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "iBridge2,13": {
        # Unknown ID, Intel based, present in Monterey
        "Board ID": None,
        "FirmwareFeatures": None,
        "SecureBootModel": "j214",
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "APPL_GPU_Test_Board": {
        # AppleInternal GPU Test Board
        # System is used for testing Intel iGPUs
        # Model has been configured with the following:
        # - Broadwell iGPU (HD Graphics 5300)
        # - Skylake iGPU (HD Graphics 5xx)
        # - Kaby Lake iGPU (HD Graphics 6xx)
        # - Ice Lake iGPU (G4/G7)
        "Board ID": "Mac-50619A408DB004DA",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "APPL_UNKNOWN_MODEL_1": {
        # Laptop model
        # No mentions in SLE outside X86PlatformPlugin
        "Board ID": "Mac-112B0A653D3AAB9C",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "APPL_UNKNOWN_MODEL_2": {
        # Laptop model
        # Only mentioned in ApplePlatformEnabler and X86PlatformPlugin
        "Board ID": "Mac-4BFBC784B845591E",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "APPL_UNKNOWN_MODEL_3": {
        # Laptop model, 15"/17" with dGPU and GMUX
        # Only mentioned in AppleGraphicsPowerManagement, AppleMuxControl2, AGDCBacklightControl, ApplePlatformEnabler and X86PlatformPlugin
        "Board ID": "Mac-564FBA6031E5946A",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
     "APPL_UNKNOWN_MODEL_4": {
        # Laptop model, iGPU only
        # Only mentioned in AppleGraphicsPowerManagement, ApplePlatformEnabler and X86PlatformPlugin
        "Board ID": "Mac-6FEBD60817C77D8A",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "APPL_UNKNOWN_MODEL_5": {
        # Laptop model, iGPU only, Iris Plus Graphics G7
        # Only mentioned in AppleGraphicsPowerManagement, ApplePlatformEnabler and X86PlatformPlugin
        "Board ID": "Mac-747B1AEFF11738BE",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "APPL_UNKNOWN_MODEL_6": {
        # Laptop model, iGPU only
        # Only mentioned in AppleGraphicsPowerManagement, ApplePlatformEnabler and X86PlatformPlugin
        "Board ID": "Mac-87DCB00F4AD77EEA",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "APPL_UNKNOWN_MODEL_7": {
        # Laptop model
        # Only mentioned in X86PlatformPlugin, AppleGVA
        "Board ID": "Mac-90BE64C3CB5A9AEB",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "APPL_UNKNOWN_MODEL_7": {
        # Laptop model, iGPU only
        # Only mentioned in AppleGraphicsPowerManagement and X86PlatformPlugin
        "Board ID": "Mac-9394BDF4BF862EE7",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.max_os,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "APPL_UNKNOWN_MODEL_8": {
        # Unknown, only mentioned in supported models for IAs
        "Board ID": "Mac-C6F71043CEAA02A6",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "APPL_UNKNOWN_MODEL_9": {
        # Unknown, only mentioned in supported models for IAs
        "Board ID": "Mac-90BE64C3CB5A9AEB",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "APPL_UNKNOWN_MODEL_10": {
        # Unknown, only mentioned in supported models for IAs
        "Board ID": "Mac-36B6B6DA9CFCD881",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
    "APPL_UNKNOWN_MODEL_11": {
        # Unknown, only mentioned in supported models for IAs
        "Board ID": "Mac-5A49A77366F81C72",
        "FirmwareFeatures": None,
        "SecureBootModel": None,
        "CPU Generation": None,
        "Max OS Supported": os_data.os_data.big_sur,
        "Wireless Model": None,
        "Bluetooth Model": bluetooth_data.bluetooth_data.NonApplicable,
        "Stock GPUs": [],
        "Stock Storage": [],
    },
}
