/**
 * @license
 * Copyright (c) 2020, 2025, Oracle and/or its affiliates.
 * The Universal Permissive License (UPL), Version 1.0
 * @ignore
 */

'use strict';

define([
    'ojs/ojcore',
    'knockout',
    'wrc-frontend/common/keyup-focuser',
    'wrc-frontend/microservices/perspective/perspective-memory-manager',
    'wrc-frontend/microservices/pages-history/pages-history-manager',
    'wrc-frontend/microservices/pages-history/pages-bookmark-manager',
    'wrc-frontend/common/page-definition-helper',
    'wrc-frontend/integration/viewModels/utils',
    'wrc-frontend/core/runtime',
    'wrc-frontend/core/types',
    'wrc-frontend/core/utils',
    'ojs/ojlogger',
    'ojs/ojknockout',
    'ojs/ojcheckboxset'
  ],
  function (
    oj,
    ko,
    KeyUpFocuser,
    PerspectiveMemoryManager,
    PagesHistoryManager,
    PagesBookmarkManager,
    PageDefinitionHelper,
    ViewModelUtils,
    Runtime,
    CoreTypes,
    CoreUtils,
    Logger
  ) {
    function TableToolbar(viewParams) {
      const self = this;

      // Need perspective declared here, because
      // expression in table.html refers to it
      this.perspective = viewParams.perspective;

      this.i18n = {
        ariaLabel: {
          icons: {
            landing: {value: oj.Translations.getTranslatedString('wrc-common.ariaLabel.icons.landing.value')}
          }
        },
        buttons: {
          'new': { id: 'new', iconFile: 'new-icon-blk_24x24', disabled: false, visible: ko.observable(true),
            label: oj.Translations.getTranslatedString('wrc-table-toolbar.buttons.new.label')
          },
          'write': { id: 'write', iconFile: 'write-wdt-model-blk_24x24', disabled: false, visible: ko.observable(true),
            label: ko.observable(oj.Translations.getTranslatedString('wrc-common.buttons.write.label'))
          },
          'clone': { id: 'clone', iconFile: 'clone-icon-blk_24x24', disabled: true,
            label: oj.Translations.getTranslatedString('wrc-table-toolbar.buttons.clone.label')
          },
          'delete': { id: 'delete', iconFile: 'delete-icon-blk_24x24', disabled: true,
            label: oj.Translations.getTranslatedString('wrc-table-toolbar.buttons.delete.label')
          },
          'customize': { id: 'customize', iconFile: 'table-customizer-icon-blk_24x24', visible: ko.observable(false),
            label: oj.Translations.getTranslatedString('wrc-table-toolbar.buttons.customize.label')
          },
          'dashboard': { id: 'dashboard', iconFile: 'custom-view-icon-blk_24x24', disabled: false, visible: ko.observable(false),
            label: ko.observable()
          }
        },
        icons: {
          'pagesHistory': {
            'visible': ko.observable(!Runtime.getProperty('features.pagesHistory.disabled')),
            'back': {
              iconFile: ko.observable('pages-history-back-gry_24x24'), disabled: true,
              tooltip: oj.Translations.getTranslatedString('wrc-common.tooltips.pagesHistory.back.value')
            },
            'next': {
              iconFile: ko.observable('pages-history-next-gry_24x24'), disabled: true,
              tooltip: oj.Translations.getTranslatedString('wrc-common.tooltips.pagesHistory.next.value')
            },
            'launch': {
              iconFile: ko.observable('pages-history-icon-gry_24x24'), disabled: true,
              tooltip: oj.Translations.getTranslatedString('wrc-common.tooltips.pagesHistory.launch.value')
            },
            'star': {
              id: 'pages-history-star-icon',
              iconFile: ko.observable('pages-bookmark-off_24x24'), disabled: true, visible: ko.observable(!Runtime.getProperty('features.bookmarks.disabled')),
              tooltip: oj.Translations.getTranslatedString('wrc-common.tooltips.pagesHistory.star.value')
            }
          },
          'landing': { iconFile: 'landing-page-icon-blk_24x24', visible: ko.observable(Runtime.getRole() === CoreTypes.Console.RuntimeRole.APP.name),
            tooltip: oj.Translations.getTranslatedString('wrc-table-toolbar.icons.landing.tooltip')
          },
          'history': { iconFile: 'beanpath-history-icon-blk_24x24',
            tooltip: oj.Translations.getTranslatedString('wrc-table-toolbar.icons.history.tooltip')
          },
          'instructions': { iconFile: 'toggle-instructions-on-blk_24x24',
            tooltip: oj.Translations.getTranslatedString('wrc-table-toolbar.icons.instructions.tooltip')
          },
          'help': { iconFile: 'toggle-help-on-blk_24x24',
            tooltip: oj.Translations.getTranslatedString('wrc-table-toolbar.icons.help.tooltip')
          },
          'reset': { iconFile: 'action-reset-icon-blk_24x24', visible: ko.observable(false),
            tooltip: oj.Translations.getTranslatedString('wrc-common.ariaLabel.icons.reset.value')
          },
          'sync': { iconFile: ko.observable('sync-off-icon-blk_24x24'),
            tooltip: oj.Translations.getTranslatedString('wrc-table-toolbar.icons.sync.tooltip'),
            tooltipOn: oj.Translations.getTranslatedString('wrc-table-toolbar.icons.sync.tooltipOn')
          },
          'syncInterval': { iconFile: 'sync-interval-icon-blk_24x24',
            tooltip: oj.Translations.getTranslatedString('wrc-table-toolbar.icons.syncInterval.tooltip')
          },
          'separator': {
            iconFile: 'separator-vertical_10x24',
            tooltip: oj.Translations.getTranslatedString('wrc-perspective.icons.separator.tooltip')
          }
        },
        menus: {
          history: {
            'clear': {
              id: 'clear-history',
              iconFile: 'erase-icon-blk_24x24',
              disabled: false,
              value: oj.Translations.getTranslatedString('wrc-perspective.menus.history.clear.value'),
              label: oj.Translations.getTranslatedString('wrc-perspective.menus.history.clear.label')
            }
          },
          bookmarks: {
            'add': {
              id: 'add-bookmark',
              disabled: ko.observable(false),
              value: 'addBookmark',
              label: oj.Translations.getTranslatedString('wrc-pages-bookmark.menus.bookmark.add.label')
            },
            'show': {
              id: 'show-bookmarks',
              disabled: false,
              value: 'showBookmarks',
              label: oj.Translations.getTranslatedString('wrc-pages-bookmark.menus.bookmark.show.label')
            }
          }
        },
        instructions: {
          'selectItems': {
            value: oj.Translations.getTranslatedString('wrc-table-toolbar.instructions.selectItems.value', '{0}')
          }
        }
      };

      this.showHelp = ko.observable(false);

      // This instance-scope variable is used to determine which
      // sync-<state>-icon-blk_24x24.png is assigned to the
      // <img id="sync-icon">
      this.autoSyncEnabled = ko.observable(false);
      this.showAutoSyncIcons = ko.observable(true);

      // This instance-scope variable is used to remember and
      // recall the value of the auto-sync interval
      this.perspectiveMemory = PerspectiveMemoryManager.getPerspectiveMemory(self.perspective.id);
      this.showInstructions = ko.observable(true);
      this.showBeanPathHistory = ko.observable(this.perspectiveMemory.beanPathHistory.visibility);

      this.signalBindings=[];
      this.readonly = ko.observable();

      this.connected = function () {
        function setIsReadOnlyRuntimeProperty() {
          const rdjData = viewParams.parentRouter.data.rdjData();
          const isNonCreatableCollection = rdjData.self.kind === 'nonCreatableCollection';
          Runtime.setProperty(Runtime.PropertyName.CFE_IS_READONLY, isNonCreatableCollection || !['configuration','modeling','properties','security'].includes(viewParams.perspective.id));
          self.readonly(Runtime.isReadOnly());
        }

        setIsReadOnlyRuntimeProperty();

        const label = oj.Translations.getTranslatedString(`wrc-common.buttons.${ViewModelUtils.isElectronApiAvailable() ? 'savenow' : 'write'}.label`);
        self.i18n.buttons.write.label(label);
        self.i18n.buttons.write.visible(Runtime.getRole() === CoreTypes.Console.RuntimeRole.APP.name && ['modeling','properties'].includes(self.perspective.id));
        self.i18n.buttons.new.visible(['configuration','modeling','security','properties'].includes(self.perspective.id));

        let binding = viewParams.signaling.readonlyChanged.add((newRO) => {
          self.readonly(newRO);
        });

        self.signalBindings.push(binding);

        // Get auto-sync interval from perspectiveMemory
        const syncInterval = self.perspectiveMemory.contentPage.syncInterval;
        if (syncInterval !== null) {
          const ele = document.getElementById('sync-icon');
          if (ele !== null) ele.setAttribute('data-interval', syncInterval);
        }

        binding = viewParams.signaling.autoSyncCancelled.add(autoSyncCancelledCallback);

        self.signalBindings.push(binding);

        binding = viewParams.signaling.pagesHistoryChanged.add((changeType) => {
          self.refreshPagesHistoryIcons();
        });

        self.signalBindings.push(binding);

        binding = viewParams.signaling.pagesBookmarkChanged.add((changeType) => {
          if (changeType === 'bookmark-added' || changeType === 'bookmarks-removed') {
            self.refreshPagesHistoryIcons();
          }
        });

        self.signalBindings.push(binding);

        this.renderToolbarButtons();

        self.refreshPagesHistoryIcons();
      };

      this.disconnected = function () {
        KeyUpFocuser.unregister('.cfe-content-area-body-iconbar-icon');
        // autoSyncCancelled is for stopping auto-sync when the user presses the
        // "Disconnect" icon in the toolbar icon of domain.js, or
        // the CFE notices that the CBE process is stopped.
        viewParams.signaling.autoSyncCancelled.remove(autoSyncCancelledCallback);

        self.signalBindings.forEach(function (binding) {
          binding.detach();
        });
        self.signalBindings = [];
      };

      this.registerTableIconbarIconsKeyUpFocuser = (id) => {
        let result = KeyUpFocuser.getKeyUpCallback(id);

        if (!result.alreadyRegistered) {
          result = KeyUpFocuser.register(
            id,
            {
              Enter: {key: 'Enter', action: 'select', callback: onKeyUpFocuserActionSelect}
            },
            {}
          );
        }

        return result.keyUpCallback;
      };

      this.resetIconKeyUp = (event) => {
        if (event.key === 'Enter') {
          simulateSyncIconClickEvent(event.currentTarget.firstElementChild);
        }
      };

      this.refreshPagesHistoryIcons = function () {
        const data = PagesHistoryManager.getPagesHistoryData();

        self.i18n.icons.pagesHistory.back.iconFile(data.canBack ? 'pages-history-back-blk_24x24' : 'pages-history-back-gry_24x24');
        self.i18n.icons.pagesHistory.back.disabled = !data.canBack;
        self.i18n.icons.pagesHistory.next.iconFile(data.canNext ? 'pages-history-next-blk_24x24' : 'pages-history-next-gry_24x24');
        self.i18n.icons.pagesHistory.next.disabled = !data.canNext;
        self.i18n.icons.pagesHistory.launch.iconFile(data.canLaunch ? 'pages-history-icon-blk_24x24' : 'pages-history-icon-gry_24x24');
        self.i18n.icons.pagesHistory.launch.disabled = !data.canLaunch;

        const isBookmarked = PagesBookmarkManager.hasPagesBookmark(data.current);
        self.i18n.icons.pagesHistory.star.iconFile(data.canBookmark ? (isBookmarked ? 'pages-bookmark-on_24x24' : 'pages-bookmark-off_24x24') : 'pages-bookmark-gry_24x24');
        self.i18n.icons.pagesHistory.star.disabled = !data.canBookmark;
      };

      this.launchPagesBookmarkMenu = function (event) {
        const pagesHistoryViewModel = ko.dataFor(document.getElementById('beanpath-history-container'));
        pagesHistoryViewModel.launchPagesBookmarkMenu(event);
      }.bind(this);

      this.pagesBookmarkMenuClickListener = function (event) {
        const pagesHistoryViewModel = ko.dataFor(document.getElementById('beanpath-history-container'));
        pagesHistoryViewModel.pagesBookmarkMenuClickListener(event);
      }.bind(this);

      function simulateSyncIconClickEvent(node) {
        event.preventDefault();
        const attr = node.attributes['data-interval'];
        if (CoreUtils.isNotUndefinedNorNull(attr)) {
          handleSyncIconEvent(node);
        }
      }

      function onKeyUpFocuserActionSelect(event) {
        function simulateHelpIconClickEvent(event) {
          const link = document.querySelector(`#${event.target.id} > a`);
          if (link !== null) {
            event.preventDefault();
            const event1 = new Event('click', {bubbles: true});
            link.dispatchEvent(event1);
          }
        }

        const firstElementChild = event.target.firstElementChild.firstElementChild;
        const iconId = firstElementChild.getAttribute('id');

        switch (iconId) {
          case 'landing-page-icon':
            self.landingPageClick(event);
            break;
          case 'toggle-history':
            self.toggleHistoryClick(event);
            break;
          case 'page-help-icon':
            simulateHelpIconClickEvent(event);
            break;
          case 'sync-icon':
            simulateSyncIconClickEvent(firstElementChild);
            break;
          case 'sync-interval-icon':
            self.syncIntervalClick(event);
            break;
        }
      }

      function showResetPageIcon(value) {
        self.i18n.icons.reset.visible(value);
      }

      function resetIconsVisibleState(state) {
        self.showAutoSyncIcons(state);
      }

      this.isHistoryVisible = function() {
        return viewParams.isHistoryVisible();
      };

      this.renderToolbarButtons =  function () {
        const rdjData = viewParams.parentRouter?.data?.rdjData();
        const isDashboard = (self.perspective.id === 'monitoring' && CoreUtils.isNotUndefinedNorNull(rdjData?.dashboardCreateForm));
        if (isDashboard) {
          self.i18n.buttons.dashboard.label(rdjData?.dashboardCreateForm?.label);
        }
        self.i18n.buttons.dashboard.visible(isDashboard);
        resetIconsVisibleState(self.perspective.id === 'monitoring');
        showResetPageIcon(['configuration', 'security'].includes(self.perspective.id));
        const pdjData = viewParams.parentRouter?.data?.pdjData();
        this.i18n.buttons.customize.visible(PageDefinitionHelper.hasTable(pdjData));
      }.bind(this);

      this.toggleHistoryClick = function (event) {
        const withHistoryVisible = !self.showBeanPathHistory();
        // Call function in table.js with negation of value
        // assigned to the knockout observable.
        viewParams.onBeanPathHistoryToggled(withHistoryVisible);
        // Set knockout observable to value returned from
        // onBeanPathHistoryToggled() function in table.js
        self.showBeanPathHistory(withHistoryVisible);
      };

      this.pagesHistoryIconClick = function (event) {
        const pagesHistoryViewModel = ko.dataFor(document.getElementById('beanpath-history-container'));
        pagesHistoryViewModel.pagesHistoryIconClicked(event);
      };

      this.landingPageClick = function (event) {
        viewParams.onLandingPageSelected();
      };

      this.toggleInstructionsClick = function (event) {
        const instructionsVisible = !self.showInstructions();
        self.showInstructions(instructionsVisible);
        viewParams.onInstructionsToggled(instructionsVisible, self.showBeanPathHistory());
      };

      this.helpPageClick = function (event) {
        const helpVisible = !self.showHelp();
        self.showHelp(helpVisible);
        viewParams.onHelpPageToggled(helpVisible);
      };

      function handleSyncIconEvent(node) {
        const attr = node.attributes['data-interval'];
        if (CoreUtils.isNotUndefinedNorNull(attr)) {
          const autoSyncEnabled = self.autoSyncEnabled();
          let syncInterval = 0;
          if (CoreUtils.isNotUndefinedNorNull(attr.value)) {
            // Get sync interval from the "data-interval" attribute
            syncInterval = parseInt(attr.value);
          }
          if (syncInterval === 0) {
            // Just reload and ensure the sync state is not running
            if (autoSyncEnabled) {
              self.autoSyncEnabled(false);
            }
          }
          else {
            // Toggle the sync state and no interval when currently enabled
            self.autoSyncEnabled(!autoSyncEnabled);
            if (autoSyncEnabled) syncInterval = 0;
          }
          PagesHistoryManager.setPagesHistoryCurrentAction('bypass');
          setAutoSyncIcon();
          viewParams.onSyncClicked(syncInterval);
        }
      }

      this.syncClick = function (event) {
        const attr = event.target.attributes['data-interval'];
        if (CoreUtils.isNotUndefinedNorNull(attr)) {
          handleSyncIconEvent(event.target);
        }
      };

      this.syncIntervalClick = function (event) {
        // Get <img id="sync-icon"> DOM element
        const ele = document.getElementById('sync-icon');
        // Get numeric representation of "data-interval"
        // attribute of DOM element.
        const currentValue = parseInt(ele.getAttribute('data-interval'));
        // Call function defined in table.js that returns a Promise, passing
        // in the numeric representation of current sync interval.
        viewParams.onSyncIntervalClicked(currentValue)
          .then((result) => {
            self.perspectiveMemory.contentPage.syncInterval = result.interval;
            // Update attributes of <img id="sync-icon"> DOM element
            ele.setAttribute('data-interval', result.interval);
            let syncInterval = parseInt(result.interval);
            if (syncInterval > 0) {
              self.autoSyncEnabled(true);
            }
            else {
              self.autoSyncEnabled(false);
            }
            setAutoSyncIcon();
          })
          .catch((cancel) => {
            // Change of the interval value was cancelled
            setAutoSyncIcon();
          });
      };

      function setAutoSyncIcon() {
        // Change tooltip to let end user know the state
        let syncIconElement = document.getElementById('sync-icon');
        if (syncIconElement !== null) {
          syncIconElement.setAttribute('title', (self.autoSyncEnabled() ? self.i18n.icons.sync.tooltipOn : self.i18n.icons.sync.tooltip));
          self.i18n.icons.sync.iconFile(self.autoSyncEnabled() ? 'bouncing-icon-blk_24x38' : 'sync-off-icon-blk_24x38');
        }
      }

      this.cancelAutoSync = function () {
        self.autoSyncEnabled(false);
        setAutoSyncIcon();
      }.bind(this);

      function autoSyncCancelledCallback(source, newSyncInterval) {
        if (self.autoSyncEnabled()) {
          viewParams.onSyncClicked(0);
        }
        self.autoSyncEnabled(false);
      }

      this.newAction = function (event) {
        if (self.showBeanPathHistory()) {
          const withHistoryVisible = viewParams.onBeanPathHistoryToggled(false);
          self.showBeanPathHistory(withHistoryVisible);
        }
        viewParams.newAction(event);
      };

      this.writeContentFileAction = function (event) {
        viewParams.onWriteContentFile('download');
      };

      this.customizeAction = (event) => {
        if (self.i18n.buttons.customize.visible()) {
          viewParams.onCustomizeButtonClicked(event);
        }
      };

      this.dashboardAction = (event) => {
        if (self.showBeanPathHistory()) {
          const withHistoryVisible = viewParams.onBeanPathHistoryToggled(false);
          self.showBeanPathHistory(withHistoryVisible);
        }
        viewParams.onDashboardButtonClicked(event);
      };
    }

    return TableToolbar;
  }
);