/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.config;

import io.helidon.config.ConfigException;
import io.helidon.config.ConfigThreadFactory;
import io.helidon.config.ConfigUtils;
import io.helidon.config.spi.ChangeEventType;
import io.helidon.config.spi.PollingStrategy;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiFunction;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class ScheduledPollingStrategy
implements PollingStrategy {
    private static final Logger LOGGER = Logger.getLogger(ScheduledPollingStrategy.class.getName());
    private final RecurringPolicy recurringPolicy;
    private final boolean defaultExecutor;
    private ScheduledExecutorService executor;
    private ScheduledFuture<?> scheduledFuture;
    private PollingStrategy.Polled polled;

    private ScheduledPollingStrategy(Builder builder) {
        this.recurringPolicy = builder.recurringPolicy;
        ScheduledExecutorService executor = builder.executor;
        if (executor == null) {
            this.executor = Executors.newSingleThreadScheduledExecutor(new ConfigThreadFactory("file-watch-polling"));
            this.defaultExecutor = true;
        } else {
            this.executor = executor;
            this.defaultExecutor = false;
        }
    }

    public static ScheduledPollingStrategy create(RecurringPolicy recurringPolicy, ScheduledExecutorService executor) {
        return ScheduledPollingStrategy.builder().recurringPolicy(recurringPolicy).executor(executor).build();
    }

    public static Builder builder() {
        return new Builder();
    }

    @Override
    public synchronized void start(PollingStrategy.Polled polled) {
        if (this.defaultExecutor && this.executor.isShutdown()) {
            this.executor = Executors.newSingleThreadScheduledExecutor(new ConfigThreadFactory("file-watch-polling"));
        }
        if (this.executor.isShutdown()) {
            throw new ConfigException("Cannot start a scheduled polling strategy, as the executor service is shutdown");
        }
        this.polled = polled;
        this.scheduleNext();
    }

    @Override
    public synchronized void stop() {
        if (this.scheduledFuture != null) {
            this.scheduledFuture.cancel(true);
        }
        if (this.defaultExecutor) {
            ConfigUtils.shutdownExecutor(this.executor);
        }
    }

    private void scheduleNext() {
        try {
            this.scheduledFuture = this.executor.schedule(this::fireEvent, this.recurringPolicy.interval().toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (RejectedExecutionException e) {
            if (this.executor.isShutdown()) {
                LOGGER.log(Level.FINEST, "Executor service is shut down, polling is terminated for " + String.valueOf(this), e);
            }
            LOGGER.log(Level.SEVERE, "Failed to schedule next polling for " + String.valueOf(this) + ", polling will stop", e);
        }
    }

    private synchronized void fireEvent() {
        ChangeEventType event = this.polled.poll(Instant.now());
        switch (event) {
            case CHANGED: 
            case DELETED: {
                this.recurringPolicy.shorten();
                break;
            }
            case UNCHANGED: {
                this.recurringPolicy.lengthen();
                break;
            }
        }
        this.scheduleNext();
    }

    ScheduledExecutorService executor() {
        return this.executor;
    }

    public String toString() {
        return "ScheduledPollingStrategy{recurringPolicy=" + String.valueOf(this.recurringPolicy) + "}";
    }

    @FunctionalInterface
    public static interface RecurringPolicy {
        public Duration interval();

        default public void shorten() {
        }

        default public void lengthen() {
        }

        public static AdaptiveBuilder adaptiveBuilder(Duration initialInterval) {
            return new AdaptiveBuilder(initialInterval);
        }

        public static final class AdaptiveBuilder {
            private final Duration interval;
            private Duration min;
            private Duration max;
            private BiFunction<Duration, Integer, Duration> shortenFunction;
            private BiFunction<Duration, Integer, Duration> lengthenFunction;
            private static final BiFunction<Duration, Integer, Duration> DEFAULT_SHORTEN = (currentDuration, changesFactor) -> currentDuration.dividedBy(2L);
            private static final BiFunction<Duration, Integer, Duration> DEFAULT_LENGTHEN = (currentDuration, changesFactor) -> currentDuration.multipliedBy(2L);

            private AdaptiveBuilder(Duration initialInterval) {
                this.interval = initialInterval;
            }

            public AdaptiveBuilder min(Duration min) {
                this.min = min;
                return this;
            }

            public AdaptiveBuilder max(Duration max) {
                this.max = max;
                return this;
            }

            public AdaptiveBuilder shorten(BiFunction<Duration, Integer, Duration> shortenFunction) {
                this.shortenFunction = shortenFunction;
                return this;
            }

            public AdaptiveBuilder lengthen(BiFunction<Duration, Integer, Duration> lengthenFunction) {
                this.lengthenFunction = lengthenFunction;
                return this;
            }

            public RecurringPolicy build() {
                Duration min = this.min == null ? this.interval.dividedBy(10L) : this.min;
                Duration max = this.max == null ? this.interval.multipliedBy(5L) : this.max;
                BiFunction<Duration, Integer, Duration> lengthenFunction = this.lengthenFunction == null ? DEFAULT_LENGTHEN : this.lengthenFunction;
                BiFunction<Duration, Integer, Duration> shortenFunction = this.shortenFunction == null ? DEFAULT_SHORTEN : this.shortenFunction;
                return new AdaptiveRecurringPolicy(min, this.interval, max, shortenFunction, lengthenFunction);
            }
        }
    }

    static class AdaptiveRecurringPolicy
    implements RecurringPolicy {
        private final AtomicInteger prolongationFactor = new AtomicInteger(0);
        private final Duration min;
        private final Duration max;
        private final BiFunction<Duration, Integer, Duration> shortenFunction;
        private final BiFunction<Duration, Integer, Duration> lengthenFunction;
        private Duration delay;

        AdaptiveRecurringPolicy(Duration min, Duration initialDelay, Duration max, BiFunction<Duration, Integer, Duration> shortenFunction, BiFunction<Duration, Integer, Duration> lengthenFunction) {
            this.min = min;
            this.max = max;
            this.delay = initialDelay;
            this.shortenFunction = shortenFunction;
            this.lengthenFunction = lengthenFunction;
        }

        @Override
        public Duration interval() {
            return this.delay;
        }

        @Override
        public void shorten() {
            int factor = this.prolongationFactor.updateAndGet(i -> {
                if (i < 0) {
                    return --i;
                }
                return -1;
            });
            Duration candidate = this.shortenFunction.apply(this.delay, -factor);
            this.delay = this.min.compareTo(candidate) > 0 ? this.min : candidate;
        }

        @Override
        public void lengthen() {
            int factor = this.prolongationFactor.updateAndGet(i -> {
                if (i > 0) {
                    return ++i;
                }
                return 1;
            });
            Duration candidate = this.lengthenFunction.apply(this.delay, factor);
            this.delay = this.max.compareTo(candidate) > 0 ? candidate : this.max;
        }

        Duration delay() {
            return this.delay;
        }
    }

    public static class RegularRecurringPolicy
    implements RecurringPolicy {
        private final Duration interval;

        public RegularRecurringPolicy(Duration interval) {
            this.interval = interval;
        }

        @Override
        public Duration interval() {
            return this.interval;
        }

        public String toString() {
            return "RegularRecurringPolicy{interval=" + String.valueOf(this.interval) + "}";
        }
    }

    public static final class Builder
    implements io.helidon.common.Builder<ScheduledPollingStrategy> {
        private RecurringPolicy recurringPolicy;
        private ScheduledExecutorService executor;

        private Builder() {
        }

        public ScheduledPollingStrategy build() {
            return new ScheduledPollingStrategy(this);
        }

        public Builder recurringPolicy(RecurringPolicy recurringPolicy) {
            this.recurringPolicy = recurringPolicy;
            return this;
        }

        public Builder executor(ScheduledExecutorService executor) {
            this.executor = executor;
            return this;
        }
    }
}

