/*********************************************************************
 This is an example for our nRF51822 based Bluefruit LE modules

 Pick one up today in the adafruit shop!

 Adafruit invests time and resources providing this open source code,
 please support Adafruit and open-source hardware by purchasing
 products from Adafruit!

 MIT license, check LICENSE for more information
 All text above, and the splash screen below must be included in
 any redistribution
*********************************************************************/

#include <Arduino.h>
#include <SPI.h>
#include "Adafruit_BLE.h"
#include "Adafruit_BluefruitLE_SPI.h"
#include "Adafruit_BluefruitLE_UART.h"
#if SOFTWARE_SERIAL_AVAILABLE
  #include <SoftwareSerial.h>
#endif

#include "BluefruitConfig.h"

// Create the bluefruit object, either software serial...uncomment these lines
/*
SoftwareSerial bluefruitSS = SoftwareSerial(BLUEFRUIT_SWUART_TXD_PIN, BLUEFRUIT_SWUART_RXD_PIN);

Adafruit_BluefruitLE_UART ble(bluefruitSS, BLUEFRUIT_UART_MODE_PIN,
                      BLUEFRUIT_UART_CTS_PIN, BLUEFRUIT_UART_RTS_PIN);
*/

/* ...or hardware serial, which does not need the RTS/CTS pins. Uncomment this line */
// Adafruit_BluefruitLE_UART ble(BLUEFRUIT_HWSERIAL_NAME, BLUEFRUIT_UART_MODE_PIN);

/* ...hardware SPI, using SCK/MOSI/MISO hardware SPI pins and then user selected CS/IRQ/RST */
Adafruit_BluefruitLE_SPI ble(BLUEFRUIT_SPI_CS, BLUEFRUIT_SPI_IRQ, BLUEFRUIT_SPI_RST);

/* ...software SPI, using SCK/MOSI/MISO user-defined SPI pins and then user selected CS/IRQ/RST */
//Adafruit_BluefruitLE_SPI ble(BLUEFRUIT_SPI_SCK, BLUEFRUIT_SPI_MISO,
//                             BLUEFRUIT_SPI_MOSI, BLUEFRUIT_SPI_CS,
//                             BLUEFRUIT_SPI_IRQ, BLUEFRUIT_SPI_RST);


// A small helper
void error(const __FlashStringHelper*err) {
  logger::println(err);
  while (1);
}

/* This demo perform factory reset on the first run then
 * - Write a defined magic number at offset 0
 * - Write a DATE string at offset 16
 * - Dump the whole NVM section, User's NVM is started at 0x0640
 */

#define MAGIC_NUMBER  0xC0FFEE
#define MAGIC_STRING  __DATE__

/**************************************************************************/
/*!
    @brief  Sets up the HW an the BLE module (this function is called
            automatically on startup)
*/
/**************************************************************************/
void setup(void)
{
  while (!Serial);  // required for Flora & Micro
  delay(500);

  Serial.begin(115200);
  logger::println(F("Adafruit Bluefruit NVM DATA Example"));
  logger::println(F("-------------------------------------"));

  /* Initialise the module */
  logger::print(F("Initialising the Bluefruit LE module: "));

  if ( !ble.begin(VERBOSE_MODE) )
  {
    error(F("Couldn't find Bluefruit, make sure it's in CoMmanD mode & check wiring?"));
  }
  logger::println( F("OK!") );

  /* Disable command echo from Bluefruit */
//  ble.echo(false);

  logger::println("Requesting Bluefruit info:");
  /* Print Bluefruit information */
  ble.info();

  int32_t magic_number;
  ble.readNVM(0, &magic_number);

  if ( magic_number != MAGIC_NUMBER )
  {
    /* Perform a factory reset to make sure everything is in a known state */
    logger::println(F("Magic not found: performing a factory reset: "));
    if ( ! ble.factoryReset() ){
      error(F("Couldn't factory reset"));
    }

    // Write data to NVM
    logger::println( F("Write defined data to NVM") );
    ble.writeNVM(0 , MAGIC_NUMBER);
    ble.writeNVM(16, MAGIC_STRING);
  }else
  {
    logger::println(F("Magic found"));
  }

  // Read from NVM and print out
  ble.readNVM(0, &magic_number);
  logger::print( F("Magic Number: ") );
  logger::println( magic_number );

  char magic_str[32];
  ble.readNVM(16, magic_str, sizeof(magic_str));
  logger::print( F("Magic String: ") );
  logger::println( magic_str );

  // Dump the whole NVM section
  logger::println();
  logger::println("Dumping the whole NVM contents");
  ble.atcommand(F("AT+DBGNVMRD"));
  logger::println("User NVM data is at offset 0x0640");
}

/**************************************************************************/
/*!
    @brief  Constantly poll for new command or response data
*/
/**************************************************************************/
void loop(void)
{
  // Display command prompt
  logger::print(F("AT > "));

  // Check for user input and echo it back if anything was found
  char command[BUFSIZE+1];
  getUserInput(command, BUFSIZE);

  // Send command
  ble.atcommand(command);
}

/**************************************************************************/
/*!
    @brief  Checks for user input (via the Serial Monitor)
*/
/**************************************************************************/
void getUserInput(char buffer[], uint8_t maxSize)
{
  memset(buffer, 0, maxSize);
  while( Serial.available() == 0 ) {
    delay(1);
  }

  uint8_t count=0;

  do
  {
    count += Serial.readBytes(buffer+count, maxSize);
    delay(2);
  } while( (count < maxSize) && !(Serial.available() == 0) );
}
