package plugin_test

import (
	"testing"
	"time"

	"github.com/stretchr/testify/require"

	"github.com/prometheus/prometheus/model/rulefmt"
	"github.com/slok/sloth/pkg/common/model"
	plugintesting "github.com/slok/sloth/pkg/prometheus/plugin/k8stransform/testing"
)

func baseSLOGroupResult() model.PromSLOGroupResult {
	return model.PromSLOGroupResult{
		SLOResults: []model.PromSLOResult{
			{
				SLO: model.PromSLO{
					Name: "test",
				},
				PrometheusRules: model.PromSLORules{
					SLIErrorRecRules: model.PromRuleGroup{
						Name:     "slo-test-sli-error-rules",
						Interval: 60 * time.Minute,
						Rules: []rulefmt.Rule{
							{Record: "rec1", Expr: "exp1"},
							{Record: "rec2", Expr: "exp2", Labels: map[string]string{"team": "team-a"}},
							{Record: "rec3", Expr: "exp3"},
							{Record: "rec4", Expr: "exp4"},
						},
					},
					MetadataRecRules: model.PromRuleGroup{
						Name:     "slo-test-metadata-rules",
						Interval: 40 * time.Minute,
						Rules: []rulefmt.Rule{
							{Record: "rec5", Expr: "exp5", Annotations: map[string]string{"info": "metadata"}},
							{Record: "rec6", Expr: "exp6"},
							{Record: "rec7", Expr: "exp7"},
						},
					},
					AlertRules: model.PromRuleGroup{
						Name: "slo-test-alert-rules",
						Rules: []rulefmt.Rule{
							{Alert: "alert1", Expr: "expA", Labels: map[string]string{"severity": "page"}, Annotations: map[string]string{"summary": "High error rate"}},
							{Alert: "alert2", Expr: "expB"},
						},
					},
					ExtraRules: []model.PromRuleGroup{
						{
							Name:     "slo-test-extra-rules-1",
							Interval: 120 * time.Minute,
							Rules: []rulefmt.Rule{
								{Record: "rec8", Expr: "exp8"},
							},
						},
						{
							Name: "slo-test-extra-rules-2",
							Rules: []rulefmt.Rule{
								{Alert: "alert3", Expr: "expC"},
							},
						},
					},
				},
			},
		},
	}
}

func TestPlugin(t *testing.T) {
	tests := map[string]struct {
		kmeta   model.K8sMeta
		slos    model.PromSLOGroupResult
		expYAML string
	}{
		"Test plugin": {
			kmeta: model.K8sMeta{
				Namespace:   "test-ns",
				Name:        "test01",
				Labels:      map[string]string{"app": "sloth"},
				Annotations: map[string]string{"annotation1": "value1"},
			},
			slos: baseSLOGroupResult(),
			expYAML: `
---
# Code generated by Sloth (dev): https://github.com/slok/sloth.
# DO NOT EDIT.

apiVersion: monitoring.coreos.com/v1
kind: PrometheusRule
metadata:
  annotations:
    annotation1: value1
  labels:
    app: sloth
    app.kubernetes.io/component: SLO
    app.kubernetes.io/managed-by: sloth
  name: test01
  namespace: test-ns
spec:
  groups:
  - interval: 1h
    name: slo-test-sli-error-rules
    rules:
    - expr: exp1
      record: rec1
    - expr: exp2
      labels:
        team: team-a
      record: rec2
    - expr: exp3
      record: rec3
    - expr: exp4
      record: rec4
  - interval: 40m
    name: slo-test-metadata-rules
    rules:
    - annotations:
        info: metadata
      expr: exp5
      record: rec5
    - expr: exp6
      record: rec6
    - expr: exp7
      record: rec7
  - name: slo-test-alert-rules
    rules:
    - alert: alert1
      annotations:
        summary: High error rate
      expr: expA
      labels:
        severity: page
    - alert: alert2
      expr: expB
  - interval: 2h
    name: slo-test-extra-rules-1
    rules:
    - expr: exp8
      record: rec8
  - name: slo-test-extra-rules-2
    rules:
    - alert: alert3
      expr: expC
`,
		},
	}

	for name, test := range tests {
		t.Run(name, func(t *testing.T) {
			pt, err := plugintesting.NewPluginTester("")
			require.NoError(t, err)
			pt.AssertYAML(t, test.expYAML, test.kmeta, test.slos)
		})
	}
}
