/*
 * Decompiled with CFR 0.152.
 */
package forge.gui;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import forge.card.CardEdition;
import forge.item.IPaperCard;
import forge.item.PaperCard;
import forge.localinstance.properties.ForgeConstants;
import forge.model.FModel;
import forge.util.FileUtil;
import java.io.File;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

public class ImportSourceAnalyzer {
    private final File source;
    private final AnalysisCallback cb;
    private final int numFilesToAnalyze;
    private int numFilesAnalyzed;
    private Map<String, String> defaultPicNames;
    private Map<String, String> defaultPicOldNameToCurrentName;
    Map<String, Map<String, String>> cardFileNamesBySet;
    Map<String, String> nameUpdates;
    Map<String, String> iconFileNames;
    Map<String, String> tokenFileNames;
    Map<String, String> questTokenFileNames;
    private final Map<String, Map<String, String>> fileNameDb = new HashMap<String, Map<String, String>>();

    public ImportSourceAnalyzer(String source, AnalysisCallback cb) {
        this.source = new File(source);
        this.cb = cb;
        this.numFilesToAnalyze = this.countFiles(this.source);
    }

    public int getNumFilesToAnalyze() {
        return this.numFilesToAnalyze;
    }

    public int getNumFilesAnalyzed() {
        return this.numFilesAnalyzed;
    }

    public void doAnalysis() {
        this.identifyAndAnalyze(this.source);
    }

    private void identifyAndAnalyze(File root) {
        String dirname = root.getName();
        if ("res".equalsIgnoreCase(dirname)) {
            this.analyzeOldResDir(root);
        } else if ("constructed".equalsIgnoreCase(dirname)) {
            this.analyzeConstructedDeckDir(root);
        } else if ("draft".equalsIgnoreCase(dirname)) {
            this.analyzeDraftDeckDir(root);
        } else if ("plane".equalsIgnoreCase(dirname) || "planar".equalsIgnoreCase(dirname)) {
            this.analyzePlanarDeckDir(root);
        } else if ("scheme".equalsIgnoreCase(dirname)) {
            this.analyzeSchemeDeckDir(root);
        } else if ("sealed".equalsIgnoreCase(dirname)) {
            this.analyzeSealedDeckDir(root);
        } else if (StringUtils.containsIgnoreCase(dirname, "deck")) {
            this.analyzeDecksDir(root);
        } else if ("gauntlet".equalsIgnoreCase(dirname)) {
            this.analyzeGauntletDataDir(root);
        } else if ("layouts".equalsIgnoreCase(dirname)) {
            this.analyzeLayoutsDir(root);
        } else if ("pics".equalsIgnoreCase(dirname)) {
            this.analyzeCardPicsDir(root);
        } else if ("pics_product".equalsIgnoreCase(dirname)) {
            this.analyzeProductPicsDir(root);
        } else if ("preferences".equalsIgnoreCase(dirname)) {
            this.analyzePreferencesDir(root);
        } else if ("quest".equalsIgnoreCase(dirname)) {
            this.analyzeQuestDir(root);
        } else if (null != FModel.getMagicDb().getEditions().get(dirname)) {
            this.analyzeCardPicsSetDir(root);
        } else {
            int numUnhandledFiles = 0;
            File[] files = root.listFiles();
            assert (files != null);
            for (File file : files) {
                if (this.cb.checkCancel()) {
                    return;
                }
                if (file.isFile()) {
                    String filename = file.getName();
                    if (StringUtils.endsWithIgnoreCase(filename, ".dck")) {
                        this.analyzeDecksDir(root);
                        numUnhandledFiles = 0;
                        break;
                    }
                    if (StringUtils.endsWithIgnoreCase(filename, ".jpg")) {
                        this.analyzeCardPicsDir(root);
                        numUnhandledFiles = 0;
                        break;
                    }
                    ++numUnhandledFiles;
                    continue;
                }
                if (!file.isDirectory()) continue;
                this.identifyAndAnalyze(file);
            }
            this.numFilesAnalyzed += numUnhandledFiles;
        }
    }

    private void analyzeOldResDir(File root) {
        this.analyzeDir(root, new Analyzer(){

            @Override
            boolean onDir(File dir) {
                String dirname = dir.getName();
                if ("decks".equalsIgnoreCase(dirname)) {
                    ImportSourceAnalyzer.this.analyzeDecksDir(dir);
                } else if ("gauntlet".equalsIgnoreCase(dirname)) {
                    ImportSourceAnalyzer.this.analyzeGauntletDataDir(dir);
                } else if ("layouts".equalsIgnoreCase(dirname)) {
                    ImportSourceAnalyzer.this.analyzeLayoutsDir(dir);
                } else if ("pics".equalsIgnoreCase(dirname)) {
                    ImportSourceAnalyzer.this.analyzeCardPicsDir(dir);
                } else if ("pics_product".equalsIgnoreCase(dirname)) {
                    ImportSourceAnalyzer.this.analyzeProductPicsDir(dir);
                } else if ("preferences".equalsIgnoreCase(dirname)) {
                    ImportSourceAnalyzer.this.analyzePreferencesDir(dir);
                } else if ("quest".equalsIgnoreCase(dirname)) {
                    ImportSourceAnalyzer.this.analyzeQuestDir(dir);
                } else {
                    return false;
                }
                return true;
            }
        });
    }

    private void analyzeDecksDir(File root) {
        this.analyzeDir(root, new Analyzer(){

            @Override
            void onFile(File file) {
                String filename = file.getName();
                if (StringUtils.endsWithIgnoreCase(filename, ".dck")) {
                    File targetFile = new File(ImportSourceAnalyzer.lcaseExt(filename));
                    ImportSourceAnalyzer.this.cb.addOp(OpType.UNKNOWN_DECK, file, targetFile);
                }
            }

            @Override
            boolean onDir(File dir) {
                String dirname = dir.getName();
                if ("constructed".equalsIgnoreCase(dirname)) {
                    ImportSourceAnalyzer.this.analyzeConstructedDeckDir(dir);
                } else {
                    if ("cube".equalsIgnoreCase(dirname)) {
                        return false;
                    }
                    if ("draft".equalsIgnoreCase(dirname)) {
                        ImportSourceAnalyzer.this.analyzeDraftDeckDir(dir);
                    } else if ("plane".equalsIgnoreCase(dirname) || "planar".equalsIgnoreCase(dirname)) {
                        ImportSourceAnalyzer.this.analyzePlanarDeckDir(dir);
                    } else if ("scheme".equalsIgnoreCase(dirname)) {
                        ImportSourceAnalyzer.this.analyzeSchemeDeckDir(dir);
                    } else if ("sealed".equalsIgnoreCase(dirname)) {
                        ImportSourceAnalyzer.this.analyzeSealedDeckDir(dir);
                    } else {
                        ImportSourceAnalyzer.this.analyzeKnownDeckDir(dir, null, OpType.UNKNOWN_DECK);
                    }
                }
                return true;
            }
        });
    }

    private void analyzeConstructedDeckDir(File root) {
        this.analyzeKnownDeckDir(root, ForgeConstants.DECK_CONSTRUCTED_DIR, OpType.CONSTRUCTED_DECK);
    }

    private void analyzeDraftDeckDir(File root) {
        this.analyzeKnownDeckDir(root, ForgeConstants.DECK_DRAFT_DIR, OpType.DRAFT_DECK);
    }

    private void analyzePlanarDeckDir(File root) {
        this.analyzeKnownDeckDir(root, ForgeConstants.DECK_PLANE_DIR, OpType.PLANAR_DECK);
    }

    private void analyzeSchemeDeckDir(File root) {
        this.analyzeKnownDeckDir(root, ForgeConstants.DECK_SCHEME_DIR, OpType.SCHEME_DECK);
    }

    private void analyzeSealedDeckDir(File root) {
        this.analyzeKnownDeckDir(root, ForgeConstants.DECK_SEALED_DIR, OpType.SEALED_DECK);
    }

    private void analyzeKnownDeckDir(File root, final String targetDir, final OpType opType) {
        this.analyzeDir(root, new Analyzer(){

            @Override
            void onFile(File file) {
                File targetFile;
                String filename = file.getName();
                if (StringUtils.endsWithIgnoreCase(filename, ".dck") && !file.equals(targetFile = new File(targetDir, ImportSourceAnalyzer.lcaseExt(filename)))) {
                    ImportSourceAnalyzer.this.cb.addOp(opType, file, targetFile);
                }
            }

            @Override
            boolean onDir(File dir) {
                ImportSourceAnalyzer.this.analyzeKnownDeckDir(dir, targetDir, opType);
                return true;
            }
        });
    }

    private void analyzeGauntletDataDir(File root) {
        this.analyzeDir(root, new Analyzer(){

            @Override
            void onFile(File file) {
                File targetFile;
                String filename = file.getName();
                if (StringUtils.endsWithIgnoreCase(filename, ".dat") && !filename.startsWith("LOCKED_") && !file.equals(targetFile = new File(ForgeConstants.GAUNTLET_DIR.userPrefLoc, ImportSourceAnalyzer.lcaseExt(filename)))) {
                    ImportSourceAnalyzer.this.cb.addOp(OpType.GAUNTLET_DATA, file, targetFile);
                }
            }
        });
    }

    private void analyzeLayoutsDir(File root) {
        this.analyzeDir(root, new Analyzer(){

            @Override
            void onFile(File file) {
                String filename = file.getName();
                if (StringUtils.endsWithIgnoreCase(filename, "_preferred.xml")) {
                    File targetFile = new File(ForgeConstants.USER_PREFS_DIR, file.getName().toLowerCase(Locale.ENGLISH).replace("_preferred", ""));
                    ImportSourceAnalyzer.this.cb.addOp(OpType.PREFERENCE_FILE, file, targetFile);
                }
            }
        });
    }

    private static String oldCleanString(String in) {
        StringBuilder out = new StringBuilder();
        for (int i = 0; i < in.length(); ++i) {
            char c = in.charAt(i);
            if (c == ' ' || c == '-') {
                out.append('_');
                continue;
            }
            if (!Character.isLetterOrDigit(c) && c != '_') continue;
            out.append(c);
        }
        return out.toString().toLowerCase();
    }

    @Deprecated
    private void addDefaultPicNames(PaperCard c, boolean backFace) {
    }

    private void analyzeCardPicsDir(File root) {
        if (null == this.defaultPicNames) {
            this.defaultPicNames = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);
            this.defaultPicOldNameToCurrentName = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);
            for (PaperCard c : FModel.getMagicDb().getCommonCards().getAllCards()) {
                this.addDefaultPicNames(c, false);
                if (!c.hasBackFace()) continue;
                this.addDefaultPicNames(c, true);
            }
            for (PaperCard c : FModel.getMagicDb().getVariantCards().getAllCards()) {
                this.addDefaultPicNames(c, false);
            }
        }
        this.analyzeListedDir(root, ForgeConstants.CACHE_CARD_PICS_DIR, new ListedAnalyzer(){

            @Override
            public String map(String filename) {
                if (ImportSourceAnalyzer.this.defaultPicOldNameToCurrentName.containsKey(filename)) {
                    return (String)ImportSourceAnalyzer.this.defaultPicOldNameToCurrentName.get(filename);
                }
                return (String)ImportSourceAnalyzer.this.defaultPicNames.get(filename);
            }

            @Override
            public OpType getOpType(String filename) {
                return OpType.DEFAULT_CARD_PIC;
            }

            @Override
            boolean onDir(File dir) {
                if ("icons".equalsIgnoreCase(dir.getName())) {
                    ImportSourceAnalyzer.this.analyzeIconsPicsDir(dir);
                } else if ("tokens".equalsIgnoreCase(dir.getName())) {
                    ImportSourceAnalyzer.this.analyzeTokenPicsDir(dir);
                } else {
                    ImportSourceAnalyzer.this.analyzeCardPicsSetDir(dir);
                }
                return true;
            }
        });
    }

    private static void addSetCards(Map<String, String> cardFileNames, Iterable<PaperCard> library, Predicate<PaperCard> filter) {
        for (PaperCard c : Iterables.filter(library, filter)) {
            String filename = c.getCardImageKey() + ".jpg";
            cardFileNames.put(filename, filename);
            if (!c.hasBackFace()) continue;
            filename = c.getCardAltImageKey() + ".jpg";
            cardFileNames.put(filename, filename);
        }
    }

    private void analyzeCardPicsSetDir(File root) {
        String editionCode;
        CardEdition.Collection editions;
        CardEdition edition;
        if (null == this.cardFileNamesBySet) {
            this.cardFileNamesBySet = new TreeMap<String, Map<String, String>>(String.CASE_INSENSITIVE_ORDER);
            for (Object ce : FModel.getMagicDb().getEditions()) {
                TreeMap<String, String> cardFileNames = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);
                Predicate<PaperCard> filter = IPaperCard.Predicates.printedInSet(((CardEdition)ce).getCode());
                ImportSourceAnalyzer.addSetCards(cardFileNames, FModel.getMagicDb().getCommonCards().getAllCards(), filter);
                ImportSourceAnalyzer.addSetCards(cardFileNames, FModel.getMagicDb().getVariantCards().getAllCards(), filter);
                this.cardFileNamesBySet.put(((CardEdition)ce).getCode2(), cardFileNames);
            }
            this.nameUpdates = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);
            Predicate predPlanes = arg0 -> arg0.getRules().getType().isPlane() || arg0.getRules().getType().isPhenomenon();
            for (PaperCard c : Iterables.filter(FModel.getMagicDb().getVariantCards().getAllCards(), predPlanes)) {
                String baseName = c.getCardImageKey();
                this.nameUpdates.put(baseName + ".full.jpg", baseName + ".jpg");
                if (!c.hasBackFace()) continue;
                baseName = c.getCardAltImageKey();
                this.nameUpdates.put(baseName + ".full.jpg", baseName + ".jpg");
            }
        }
        if (null == (edition = (editions = FModel.getMagicDb().getEditions()).get(editionCode = root.getName()))) {
            this.numFilesAnalyzed += this.countFiles(root);
            return;
        }
        final String editionCode2 = edition.getCode2();
        final Map<String, String> validFilenames = this.cardFileNamesBySet.get(editionCode2);
        this.analyzeListedDir(root, ForgeConstants.CACHE_CARD_PICS_DIR, new ListedAnalyzer(){

            @Override
            public String map(String filename) {
                filename = editionCode2 + "/" + filename;
                if (ImportSourceAnalyzer.this.nameUpdates.containsKey(filename)) {
                    filename = ImportSourceAnalyzer.this.nameUpdates.get(filename);
                }
                if (validFilenames.containsKey(filename)) {
                    return (String)validFilenames.get(filename);
                }
                if (StringUtils.endsWithIgnoreCase(filename, ".jpg") || StringUtils.endsWithIgnoreCase(filename, ".png")) {
                    return filename;
                }
                return null;
            }

            @Override
            public OpType getOpType(String filename) {
                return validFilenames.containsKey(filename) ? OpType.SET_CARD_PIC : OpType.POSSIBLE_SET_CARD_PIC;
            }
        });
    }

    private void analyzeIconsPicsDir(File root) {
        if (null == this.iconFileNames) {
            this.iconFileNames = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);
            for (Pair<String, String> nameurl : FileUtil.readNameUrlFile(ForgeConstants.IMAGE_LIST_QUEST_OPPONENT_ICONS_FILE)) {
                this.iconFileNames.put(nameurl.getLeft(), nameurl.getLeft());
            }
        }
        this.analyzeListedDir(root, ForgeConstants.CACHE_ICON_PICS_DIR, new ListedAnalyzer(){

            @Override
            public String map(String filename) {
                return ImportSourceAnalyzer.this.iconFileNames.getOrDefault(filename, null);
            }

            @Override
            public OpType getOpType(String filename) {
                return OpType.QUEST_PIC;
            }
        });
    }

    private void analyzeTokenPicsDir(File root) {
        if (null == this.tokenFileNames) {
            this.tokenFileNames = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);
            this.questTokenFileNames = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);
            for (Pair<String, String> nameurl : FileUtil.readNameUrlFile(ForgeConstants.IMAGE_LIST_TOKENS_FILE)) {
                this.tokenFileNames.put(nameurl.getLeft(), nameurl.getLeft());
            }
            for (Pair<String, String> nameurl : FileUtil.readNameUrlFile(ForgeConstants.IMAGE_LIST_QUEST_TOKENS_FILE)) {
                this.questTokenFileNames.put(nameurl.getLeft(), nameurl.getLeft());
            }
        }
        this.analyzeListedDir(root, ForgeConstants.CACHE_TOKEN_PICS_DIR, new ListedAnalyzer(){

            @Override
            public String map(String filename) {
                if (ImportSourceAnalyzer.this.questTokenFileNames.containsKey(filename)) {
                    return ImportSourceAnalyzer.this.questTokenFileNames.get(filename);
                }
                if (ImportSourceAnalyzer.this.tokenFileNames.containsKey(filename)) {
                    return ImportSourceAnalyzer.this.tokenFileNames.get(filename);
                }
                return null;
            }

            @Override
            public OpType getOpType(String filename) {
                return ImportSourceAnalyzer.this.questTokenFileNames.containsKey(filename) ? OpType.QUEST_PIC : OpType.TOKEN_PIC;
            }
        });
    }

    private void analyzeProductPicsDir(File root) {
        this.analyzeDir(root, new Analyzer(){

            @Override
            boolean onDir(File dir) {
                String dirName = dir.getName();
                if ("booster".equalsIgnoreCase(dirName)) {
                    ImportSourceAnalyzer.this.analyzeSimpleListedDir(dir, ForgeConstants.IMAGE_LIST_QUEST_BOOSTERS_FILE, ForgeConstants.CACHE_BOOSTER_PICS_DIR, OpType.QUEST_PIC);
                } else if ("fatpacks".equalsIgnoreCase(dirName)) {
                    ImportSourceAnalyzer.this.analyzeSimpleListedDir(dir, ForgeConstants.IMAGE_LIST_QUEST_FATPACKS_FILE, ForgeConstants.CACHE_FATPACK_PICS_DIR, OpType.QUEST_PIC);
                } else if ("boosterboxes".equalsIgnoreCase(dirName)) {
                    ImportSourceAnalyzer.this.analyzeSimpleListedDir(dir, ForgeConstants.IMAGE_LIST_QUEST_BOOSTERBOXES_FILE, ForgeConstants.CACHE_BOOSTERBOX_PICS_DIR, OpType.QUEST_PIC);
                } else if ("precons".equalsIgnoreCase(dirName)) {
                    ImportSourceAnalyzer.this.analyzeSimpleListedDir(dir, ForgeConstants.IMAGE_LIST_QUEST_PRECONS_FILE, ForgeConstants.CACHE_PRECON_PICS_DIR, OpType.QUEST_PIC);
                } else if ("tournamentpacks".equalsIgnoreCase(dirName)) {
                    ImportSourceAnalyzer.this.analyzeSimpleListedDir(dir, ForgeConstants.IMAGE_LIST_QUEST_TOURNAMENTPACKS_FILE, ForgeConstants.CACHE_TOURNAMENTPACK_PICS_DIR, OpType.QUEST_PIC);
                } else {
                    return false;
                }
                return true;
            }
        });
    }

    private void analyzePreferencesDir(File root) {
        this.analyzeDir(root, new Analyzer(){

            @Override
            void onFile(File file) {
                File targetFile;
                String filename = file.getName();
                if (("editor.preferences".equalsIgnoreCase(filename) || "forge.preferences".equalsIgnoreCase(filename)) && !file.equals(targetFile = new File(ForgeConstants.USER_PREFS_DIR, filename.toLowerCase(Locale.ENGLISH)))) {
                    ImportSourceAnalyzer.this.cb.addOp(OpType.PREFERENCE_FILE, file, targetFile);
                }
            }
        });
    }

    private void analyzeQuestDir(File root) {
        this.analyzeDir(root, new Analyzer(){

            @Override
            void onFile(File file) {
                File targetFile;
                String filename = file.getName();
                if ("all-prices.txt".equalsIgnoreCase(filename) && !file.equals(targetFile = new File(ForgeConstants.DB_DIR, filename.toLowerCase(Locale.ENGLISH)))) {
                    ImportSourceAnalyzer.this.cb.addOp(OpType.DB_FILE, file, targetFile);
                }
            }

            @Override
            boolean onDir(File dir) {
                if ("data".equalsIgnoreCase(dir.getName())) {
                    ImportSourceAnalyzer.this.analyzeQuestDataDir(dir);
                    return true;
                }
                return false;
            }
        });
    }

    private void analyzeQuestDataDir(File root) {
        this.analyzeDir(root, new Analyzer(){

            @Override
            void onFile(File file) {
                File targetFile;
                String filename = file.getName();
                if (StringUtils.endsWithIgnoreCase(filename, ".dat") && !file.equals(targetFile = new File(ForgeConstants.QUEST_SAVE_DIR, ImportSourceAnalyzer.lcaseExt(filename)))) {
                    ImportSourceAnalyzer.this.cb.addOp(OpType.QUEST_DATA, file, targetFile);
                }
            }
        });
    }

    private void analyzeDir(File root, Analyzer analyzer) {
        File[] files = root.listFiles();
        assert (files != null);
        for (File file : files) {
            if (this.cb.checkCancel()) {
                return;
            }
            if (file.isFile()) {
                ++this.numFilesAnalyzed;
                analyzer.onFile(file);
                continue;
            }
            if (!file.isDirectory() || analyzer.onDir(file)) continue;
            this.numFilesAnalyzed += this.countFiles(file);
        }
    }

    private void analyzeSimpleListedDir(File root, String listFile, String targetDir, final OpType opType) {
        if (!this.fileNameDb.containsKey(listFile)) {
            TreeMap<String, String> fileNames = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);
            for (Pair<String, String> nameurl : FileUtil.readNameUrlFile(listFile)) {
                fileNames.put(nameurl.getLeft(), nameurl.getLeft());
            }
            this.fileNameDb.put(listFile, fileNames);
        }
        final Map<String, String> fileDb = this.fileNameDb.get(listFile);
        this.analyzeListedDir(root, targetDir, new ListedAnalyzer(){

            @Override
            public String map(String filename) {
                return fileDb.getOrDefault(filename, null);
            }

            @Override
            public OpType getOpType(String filename) {
                return opType;
            }
        });
    }

    private void analyzeListedDir(File root, final String targetDir, final ListedAnalyzer listedAnalyzer) {
        this.analyzeDir(root, new Analyzer(){

            @Override
            void onFile(File file) {
                File targetFile;
                String filename = listedAnalyzer.map(file.getName());
                if (null != filename && !file.equals(targetFile = new File(targetDir, filename))) {
                    ImportSourceAnalyzer.this.cb.addOp(listedAnalyzer.getOpType(filename), file, targetFile);
                }
            }

            @Override
            boolean onDir(File dir) {
                return listedAnalyzer.onDir(dir);
            }
        });
    }

    private int countFiles(File root) {
        int count = 0;
        File[] files = root.listFiles();
        assert (files != null);
        for (File file : files) {
            if (this.cb.checkCancel()) {
                return 0;
            }
            if (file.isFile()) {
                ++count;
                continue;
            }
            if (!file.isDirectory()) continue;
            count += this.countFiles(file);
        }
        return count;
    }

    private static String lcaseExt(String filename) {
        int lastDotIdx = filename.lastIndexOf(46);
        if (0 > lastDotIdx) {
            return filename;
        }
        String basename = filename.substring(0, lastDotIdx);
        String ext = filename.substring(lastDotIdx).toLowerCase(Locale.ENGLISH);
        if (filename.endsWith(ext)) {
            return filename;
        }
        return basename + ext;
    }

    private abstract class ListedAnalyzer {
        private ListedAnalyzer() {
        }

        abstract String map(String var1);

        abstract OpType getOpType(String var1);

        boolean onDir(File dir) {
            return false;
        }
    }

    private class Analyzer {
        private Analyzer() {
        }

        void onFile(File file) {
        }

        boolean onDir(File dir) {
            return false;
        }
    }

    public static interface AnalysisCallback {
        public boolean checkCancel();

        public void addOp(OpType var1, File var2, File var3);
    }

    public static enum OpType {
        CONSTRUCTED_DECK,
        DRAFT_DECK,
        PLANAR_DECK,
        SCHEME_DECK,
        SEALED_DECK,
        UNKNOWN_DECK,
        DEFAULT_CARD_PIC,
        SET_CARD_PIC,
        POSSIBLE_SET_CARD_PIC,
        TOKEN_PIC,
        QUEST_PIC,
        GAUNTLET_DATA,
        QUEST_DATA,
        PREFERENCE_FILE,
        DB_FILE;

    }
}

