/*
 * Decompiled with CFR 0.152.
 */
package forge.card;

import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimaps;
import forge.StaticData;
import forge.card.CardEdition;
import forge.card.CardFace;
import forge.card.CardRarity;
import forge.card.CardRules;
import forge.card.CardSplitType;
import forge.card.CardType;
import forge.card.ICardDatabase;
import forge.card.ICardFace;
import forge.deck.generation.IDeckGenPool;
import forge.item.PaperCard;
import forge.util.CollectionSuppliers;
import forge.util.Lang;
import forge.util.TextUtil;
import forge.util.lang.LangEnglish;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

public final class CardDb
implements ICardDatabase,
IDeckGenPool {
    public static final String foilSuffix = "+";
    public static final char NameSetSeparator = '|';
    private final String exlcudedCardName = "Concentrate";
    private final String exlcudedCardSet = "DS0";
    private final ListMultimap<String, PaperCard> allCardsByName = Multimaps.newListMultimap(new TreeMap(String.CASE_INSENSITIVE_ORDER), CollectionSuppliers.arrayLists());
    private final Map<String, PaperCard> uniqueCardsByName = Maps.newTreeMap(String.CASE_INSENSITIVE_ORDER);
    private final Map<String, CardRules> rulesByName;
    private final Map<String, ICardFace> facesByName = Maps.newTreeMap(String.CASE_INSENSITIVE_ORDER);
    private final Map<String, String> normalizedNames = Maps.newTreeMap(String.CASE_INSENSITIVE_ORDER);
    private static Map<String, String> artPrefs = Maps.newHashMap();
    private final Map<String, String> alternateName = Maps.newTreeMap(String.CASE_INSENSITIVE_ORDER);
    private final Map<String, Integer> artIds = Maps.newHashMap();
    private final CardEdition.Collection editions;
    private List<String> filtered;
    private Map<String, Boolean> nonLegendaryCreatureNames = Maps.newHashMap();
    private CardArtPreference defaultCardArtPreference;
    private final Editor editor = new Editor();

    public CardDb(Map<String, CardRules> rules, CardEdition.Collection editions0, List<String> filteredCards, String cardArtPreference) {
        this.filtered = filteredCards;
        this.rulesByName = rules;
        this.editions = editions0;
        for (CardRules rule : rules.values()) {
            if (filteredCards.contains(rule.getName()) && !"Concentrate".equalsIgnoreCase(rule.getName())) continue;
            for (ICardFace face : rule.getAllFaces()) {
                this.addFaceToDbNames(face);
            }
        }
        this.setCardArtPreference(cardArtPreference);
    }

    private void addFaceToDbNames(ICardFace face) {
        String altName;
        if (face == null) {
            return;
        }
        String name = face.getName();
        this.facesByName.put(name, face);
        String normalName = StringUtils.stripAccents(name);
        if (!normalName.equals(name)) {
            this.normalizedNames.put(normalName, name);
        }
        if ((altName = face.getAltName()) != null) {
            this.alternateName.put(altName, face.getName());
            String normalAltName = StringUtils.stripAccents(altName);
            if (!normalAltName.equals(altName)) {
                this.normalizedNames.put(normalAltName, altName);
            }
        }
    }

    private void addSetCard(CardEdition e, CardEdition.CardInSet cis, CardRules cr) {
        int artIdx = 1;
        String key = e.getCode() + "/" + cis.name;
        if (this.artIds.containsKey(key)) {
            artIdx = this.artIds.get(key) + 1;
        }
        this.artIds.put(key, artIdx);
        this.addCard(new PaperCard(cr, e.getCode(), cis.rarity, artIdx, false, cis.collectorNumber, cis.artistName, cis.functionalVariantName));
    }

    private boolean addFromSetByName(String cardName, CardEdition ed, CardRules cr) {
        List<CardEdition.CardInSet> cardsInSet = ed.getCardInSet(cardName);
        if (cr.hasFunctionalVariants()) {
            cardsInSet = cardsInSet.stream().filter(c -> StringUtils.isEmpty(c.functionalVariantName) || cr.getSupportedFunctionalVariants().contains(c.functionalVariantName)).collect(Collectors.toList());
        }
        if (cardsInSet.isEmpty()) {
            return false;
        }
        for (CardEdition.CardInSet cis : cardsInSet) {
            this.addSetCard(ed, cis, cr);
        }
        return true;
    }

    public void loadCard(String cardName, String setCode, CardRules cr) {
        this.rulesByName.put(cardName, cr);
        boolean reIndexNecessary = false;
        CardEdition ed = this.editions.get(setCode);
        if (ed == null || ed.equals(CardEdition.UNKNOWN)) {
            for (CardEdition e : this.editions) {
                reIndexNecessary |= this.addFromSetByName(cardName, e, cr);
            }
        } else {
            reIndexNecessary |= this.addFromSetByName(cardName, ed, cr);
        }
        if (reIndexNecessary) {
            this.reIndex();
        }
    }

    public void initialize(boolean logMissingPerEdition, boolean logMissingSummary, boolean enableUnknownCards) {
        LinkedHashSet allMissingCards = new LinkedHashSet();
        ArrayList<String> missingCards = new ArrayList<String>();
        CardEdition upcomingSet = null;
        Date today = new Date();
        this.buildRenamedCards();
        for (CardEdition e : this.editions.getOrderedEditions()) {
            boolean isCoreExpSet;
            boolean coreOrExpSet = e.getType() == CardEdition.Type.CORE || e.getType() == CardEdition.Type.EXPANSION;
            boolean bl = isCoreExpSet = coreOrExpSet || e.getType() == CardEdition.Type.REPRINT;
            if (logMissingPerEdition && isCoreExpSet) {
                System.out.print(e.getName() + " (" + e.getAllCardsInSet().size() + " cards)");
            }
            if (coreOrExpSet && e.getDate().after(today) && upcomingSet == null) {
                upcomingSet = e;
            }
            for (CardEdition.CardInSet cis : e.getAllCardsInSet()) {
                CardRules cr = this.rulesByName.get(cis.name);
                if (cr == null) {
                    missingCards.add(cis.name);
                    continue;
                }
                if (cr.hasFunctionalVariants() && StringUtils.isNotEmpty(cis.functionalVariantName) && !cr.getSupportedFunctionalVariants().contains(cis.functionalVariantName)) continue;
                this.addSetCard(e, cis, cr);
            }
            if (isCoreExpSet && logMissingPerEdition) {
                if (missingCards.isEmpty()) {
                    System.out.println(" ... 100% ");
                } else {
                    int missing = (e.getAllCardsInSet().size() - missingCards.size()) * 10000 / e.getAllCardsInSet().size();
                    System.out.printf(" ... %.2f%% (%s missing: %s)%n", Float.valueOf((float)missing * 0.01f), Lang.nounWithAmount(missingCards.size(), "card"), StringUtils.join(missingCards, " | "));
                }
            }
            if (isCoreExpSet && logMissingSummary) {
                allMissingCards.addAll(missingCards);
            }
            missingCards.clear();
            this.artIds.clear();
        }
        if (logMissingSummary) {
            System.out.printf("Totally %d cards not implemented: %s\n", allMissingCards.size(), StringUtils.join(allMissingCards, " | "));
        }
        if (upcomingSet != null) {
            System.err.println("Upcoming set " + upcomingSet + " dated in the future. All unaccounted cards will be added to this set with unknown rarity.");
        }
        for (CardRules cr : this.rulesByName.values()) {
            if (this.contains(cr.getName())) continue;
            if (!cr.isCustom()) {
                if (upcomingSet != null) {
                    this.addCard(new PaperCard(cr, upcomingSet.getCode(), CardRarity.Unknown));
                    continue;
                }
                if (!enableUnknownCards || this.filtered.contains(cr.getName())) continue;
                System.err.println("The card " + cr.getName() + " was not assigned to any set. Adding it to UNKNOWN set... to fix see res/editions/ folder. ");
                this.addCard(new PaperCard(cr, CardEdition.UNKNOWN.getCode(), CardRarity.Special));
                continue;
            }
            System.err.println("The custom card " + cr.getName() + " was not assigned to any set. Adding it to custom USER set, and will try to load custom art from USER edition.");
            this.addCard(new PaperCard(cr, "USER", CardRarity.Special));
        }
        this.reIndex();
    }

    private void buildRenamedCards() {
        Lang lang = Lang.getInstance();
        if (lang == null) {
            lang = new LangEnglish();
        }
        for (CardEdition.CardInSet cis : this.editions.get("SLX").getCards()) {
            String orgName = this.alternateName.get(cis.name);
            if (orgName == null) continue;
            CardRules org = this.getRules(orgName);
            CardFace renamedMain = (CardFace)((CardFace)org.getMainPart()).clone();
            renamedMain.setName(renamedMain.getAltName());
            renamedMain.setAltName(null);
            renamedMain.setOracleText(renamedMain.getOracleText().replace(orgName, renamedMain.getName()).replace(lang.getNickName(orgName), lang.getNickName(renamedMain.getName())));
            this.facesByName.put(renamedMain.getName(), renamedMain);
            CardFace renamedOther = null;
            if (org.getOtherPart() != null) {
                renamedOther = (CardFace)((CardFace)org.getOtherPart()).clone();
                orgName = renamedOther.getName();
                renamedOther.setName(renamedOther.getAltName());
                renamedOther.setAltName(null);
                renamedOther.setOracleText(renamedOther.getOracleText().replace(orgName, renamedOther.getName()).replace(lang.getNickName(orgName), lang.getNickName(renamedOther.getName())));
                this.facesByName.put(renamedOther.getName(), renamedOther);
            }
            CardRules within = new CardRules(new ICardFace[]{renamedMain, renamedOther, null, null, null, null, null}, org.getSplitType(), org.getAiHints());
            within.setNormalizedName(org.getNormalizedName());
            this.rulesByName.put(cis.name, within);
        }
    }

    public void addCard(PaperCard paperCard) {
        if (this.excludeCard(paperCard.getName(), paperCard.getEdition())) {
            return;
        }
        this.allCardsByName.put(paperCard.getName(), paperCard);
        if (paperCard.getRules().getSplitType() == CardSplitType.None) {
            return;
        }
        if (paperCard.getRules().getOtherPart() != null) {
            this.allCardsByName.put(paperCard.getRules().getOtherPart().getName(), paperCard);
        }
        if (paperCard.getRules().getSplitType() == CardSplitType.Split) {
            this.allCardsByName.put(paperCard.getRules().getMainPart().getName(), paperCard);
        } else if (paperCard.getRules().getSplitType() == CardSplitType.Specialize) {
            for (ICardFace face : paperCard.getRules().getSpecializeParts().values()) {
                this.allCardsByName.put(face.getName(), paperCard);
            }
        }
    }

    private boolean excludeCard(String cardName, String cardEdition) {
        if (this.filtered.isEmpty()) {
            return false;
        }
        if (this.filtered.contains(cardName)) {
            if ("DS0".equalsIgnoreCase(cardEdition) && "Concentrate".equalsIgnoreCase(cardName)) {
                return true;
            }
            return !"Concentrate".equalsIgnoreCase(cardName);
        }
        return false;
    }

    private void reIndex() {
        this.uniqueCardsByName.clear();
        for (Map.Entry<String, Collection<PaperCard>> kv : this.allCardsByName.asMap().entrySet()) {
            PaperCard pc = CardDb.getFirstWithImage(kv.getValue());
            this.uniqueCardsByName.put(kv.getKey(), pc);
        }
    }

    private static PaperCard getFirstWithImage(Collection<PaperCard> cards) {
        Iterator<PaperCard> iterator = cards.iterator();
        PaperCard pc = iterator.next();
        while (iterator.hasNext()) {
            if (pc.hasImage()) {
                return pc;
            }
            pc = iterator.next();
        }
        return pc;
    }

    public boolean setPreferredArt(String cardName, String setCode, int artIndex) {
        String cardRequestForPreferredArt = CardRequest.compose(cardName, setCode, artIndex);
        PaperCard pc = this.getCard(cardRequestForPreferredArt);
        if (pc != null) {
            artPrefs.put(cardName, cardRequestForPreferredArt);
            this.uniqueCardsByName.put(cardName, pc);
            return true;
        }
        return false;
    }

    public boolean hasPreferredArt(String cardName) {
        return artPrefs.getOrDefault(cardName, null) != null;
    }

    public CardRules getRules(String cardName) {
        CardRules result = this.rulesByName.get(cardName);
        if (result != null) {
            return result;
        }
        return CardRules.getUnsupportedCardNamed(cardName);
    }

    public CardArtPreference getCardArtPreference() {
        return this.defaultCardArtPreference;
    }

    public void setCardArtPreference(boolean latestArt, boolean coreExpansionOnly) {
        this.defaultCardArtPreference = coreExpansionOnly ? (latestArt ? CardArtPreference.LATEST_ART_CORE_EXPANSIONS_REPRINT_ONLY : CardArtPreference.ORIGINAL_ART_CORE_EXPANSIONS_REPRINT_ONLY) : (latestArt ? CardArtPreference.LATEST_ART_ALL_EDITIONS : CardArtPreference.ORIGINAL_ART_ALL_EDITIONS);
    }

    public void setCardArtPreference(String artPreference) {
        artPreference = artPreference.toLowerCase().trim();
        boolean isLatest = artPreference.contains("latest");
        if (!artPreference.contains("original") && !artPreference.contains("earliest")) {
            isLatest = true;
        }
        boolean hasFilter = artPreference.contains("core");
        this.setCardArtPreference(isLatest, hasFilter);
    }

    @Override
    public PaperCard getCard(String cardName) {
        CardRequest request = CardRequest.fromString(cardName);
        return this.tryGetCard(request);
    }

    @Override
    public PaperCard getCard(String cardName, String setCode) {
        CardRequest request = CardRequest.fromString(CardRequest.compose(cardName, setCode));
        return this.tryGetCard(request);
    }

    @Override
    public PaperCard getCard(String cardName, String setCode, int artIndex) {
        String reqInfo = CardRequest.compose(cardName, setCode, artIndex);
        CardRequest request = CardRequest.fromString(reqInfo);
        return this.tryGetCard(request);
    }

    @Override
    public PaperCard getCard(String cardName, String setCode, String collectorNumber) {
        String reqInfo = CardRequest.compose(cardName, setCode, collectorNumber);
        CardRequest request = CardRequest.fromString(reqInfo);
        return this.tryGetCard(request);
    }

    @Override
    public PaperCard getCard(String cardName, String setCode, int artIndex, String collectorNumber) {
        String reqInfo = CardRequest.compose(cardName, setCode, artIndex, collectorNumber);
        CardRequest request = CardRequest.fromString(reqInfo);
        return this.tryGetCard(request);
    }

    private PaperCard tryGetCard(CardRequest request) {
        if (request == null) {
            return null;
        }
        String reqEditionCode = request.edition;
        if (reqEditionCode != null && reqEditionCode.length() > 0) {
            CardEdition edition = this.editions.get(reqEditionCode.toUpperCase());
            return this.getCardFromSet(request.cardName, edition, request.artIndex, request.collectorNumber, request.isFoil);
        }
        Collection cards = this.getAllCards(request.cardName);
        if (cards.isEmpty()) {
            return null;
        }
        String cardRequest = CardRequest.compose(request.cardName, request.isFoil);
        return this.getCardFromEditions(cardRequest, this.defaultCardArtPreference, request.artIndex);
    }

    @Override
    public PaperCard getCardFromSet(String cardName, CardEdition edition, boolean isFoil) {
        return this.getCardFromSet(cardName, edition, -1, "N.A.", isFoil);
    }

    @Override
    public PaperCard getCardFromSet(String cardName, CardEdition edition, int artIndex, boolean isFoil) {
        return this.getCardFromSet(cardName, edition, artIndex, "N.A.", isFoil);
    }

    @Override
    public PaperCard getCardFromSet(String cardName, CardEdition edition, String collectorNumber, boolean isFoil) {
        return this.getCardFromSet(cardName, edition, -1, collectorNumber, isFoil);
    }

    @Override
    public PaperCard getCardFromSet(String cardName, CardEdition edition, int artIndex, String collectorNumber, boolean isFoil) {
        PaperCard candidate;
        if (edition == null || cardName == null) {
            return null;
        }
        CardRequest cardNameRequest = CardRequest.fromString(cardName);
        cardName = cardNameRequest.cardName;
        isFoil = isFoil || cardNameRequest.isFoil;
        Collection candidates = this.getAllCards(cardName, c -> {
            boolean setFilter;
            boolean artIndexFilter = true;
            boolean collectorNumberFilter = true;
            boolean bl = setFilter = c.getEdition().equalsIgnoreCase(edition.getCode()) || c.getEdition().equalsIgnoreCase(edition.getCode2());
            if (artIndex > 0) {
                boolean bl2 = artIndexFilter = c.getArtIndex() == artIndex;
            }
            if (collectorNumber != null && collectorNumber.length() > 0 && !collectorNumber.equals("N.A.")) {
                collectorNumberFilter = c.getCollectorNumber().equals(collectorNumber);
            }
            return setFilter && artIndexFilter && collectorNumberFilter;
        });
        if (candidates.isEmpty()) {
            return null;
        }
        Iterator candidatesIterator = candidates.iterator();
        PaperCard firstCandidate = candidate = (PaperCard)candidatesIterator.next();
        while (!candidate.hasImage() && candidatesIterator.hasNext()) {
            candidate = (PaperCard)candidatesIterator.next();
        }
        candidate = candidate.hasImage() ? candidate : firstCandidate;
        return isFoil ? candidate.getFoiled() : candidate;
    }

    public PaperCard getCardFromEditions(String cardName) {
        return this.getCardFromEditions(cardName, this.defaultCardArtPreference);
    }

    public PaperCard getCardFromEditions(String cardName, Predicate<PaperCard> filter) {
        return this.getCardFromEditions(cardName, this.defaultCardArtPreference, filter);
    }

    @Override
    public PaperCard getCardFromEditions(String cardName, CardArtPreference artPreference) {
        return this.getCardFromEditions(cardName, artPreference, -1);
    }

    @Override
    public PaperCard getCardFromEditions(String cardName, CardArtPreference artPreference, Predicate<PaperCard> filter) {
        return this.getCardFromEditions(cardName, artPreference, -1, filter);
    }

    @Override
    public PaperCard getCardFromEditions(String cardInfo, CardArtPreference artPreference, int artIndex) {
        return this.tryToGetCardFromEditions(cardInfo, artPreference, artIndex, null);
    }

    @Override
    public PaperCard getCardFromEditions(String cardInfo, CardArtPreference artPreference, int artIndex, Predicate<PaperCard> filter) {
        return this.tryToGetCardFromEditions(cardInfo, artPreference, artIndex, filter);
    }

    public PaperCard getCardFromEditionsReleasedBefore(String cardName, Date releaseDate) {
        return this.getCardFromEditionsReleasedBefore(cardName, this.defaultCardArtPreference, 1, releaseDate);
    }

    public PaperCard getCardFromEditionsReleasedBefore(String cardName, int artIndex, Date releaseDate) {
        return this.getCardFromEditionsReleasedBefore(cardName, this.defaultCardArtPreference, artIndex, releaseDate);
    }

    public PaperCard getCardFromEditionsReleasedBefore(String cardName, Date releaseDate, Predicate<PaperCard> filter) {
        return this.getCardFromEditionsReleasedBefore(cardName, this.defaultCardArtPreference, releaseDate, filter);
    }

    @Override
    public PaperCard getCardFromEditionsReleasedBefore(String cardName, CardArtPreference artPreference, Date releaseDate) {
        return this.getCardFromEditionsReleasedBefore(cardName, artPreference, 1, releaseDate);
    }

    @Override
    public PaperCard getCardFromEditionsReleasedBefore(String cardName, CardArtPreference artPreference, Date releaseDate, Predicate<PaperCard> filter) {
        return this.getCardFromEditionsReleasedBefore(cardName, artPreference, 1, releaseDate, filter);
    }

    @Override
    public PaperCard getCardFromEditionsReleasedBefore(String cardName, CardArtPreference artPreference, int artIndex, Date releaseDate) {
        return this.tryToGetCardFromEditions(cardName, artPreference, artIndex, releaseDate, true, null);
    }

    @Override
    public PaperCard getCardFromEditionsReleasedBefore(String cardName, CardArtPreference artPreference, int artIndex, Date releaseDate, Predicate<PaperCard> filter) {
        return this.tryToGetCardFromEditions(cardName, artPreference, artIndex, releaseDate, true, filter);
    }

    public PaperCard getCardFromEditionsReleasedAfter(String cardName, Date releaseDate) {
        return this.getCardFromEditionsReleasedAfter(cardName, this.defaultCardArtPreference, 1, releaseDate);
    }

    public PaperCard getCardFromEditionsReleasedAfter(String cardName, int artIndex, Date releaseDate) {
        return this.getCardFromEditionsReleasedAfter(cardName, this.defaultCardArtPreference, artIndex, releaseDate);
    }

    @Override
    public PaperCard getCardFromEditionsReleasedAfter(String cardName, CardArtPreference artPreference, Date releaseDate) {
        return this.getCardFromEditionsReleasedAfter(cardName, artPreference, 1, releaseDate);
    }

    @Override
    public PaperCard getCardFromEditionsReleasedAfter(String cardName, CardArtPreference artPreference, Date releaseDate, Predicate<PaperCard> filter) {
        return this.getCardFromEditionsReleasedAfter(cardName, artPreference, 1, releaseDate, filter);
    }

    @Override
    public PaperCard getCardFromEditionsReleasedAfter(String cardName, CardArtPreference artPreference, int artIndex, Date releaseDate) {
        return this.tryToGetCardFromEditions(cardName, artPreference, artIndex, releaseDate, false, null);
    }

    @Override
    public PaperCard getCardFromEditionsReleasedAfter(String cardName, CardArtPreference artPreference, int artIndex, Date releaseDate, Predicate<PaperCard> filter) {
        return this.tryToGetCardFromEditions(cardName, artPreference, artIndex, releaseDate, false, filter);
    }

    private PaperCard tryToGetCardFromEditions(String cardInfo, CardArtPreference artPreference, int artIndex, Predicate<PaperCard> filter) {
        return this.tryToGetCardFromEditions(cardInfo, artPreference, artIndex, null, false, filter);
    }

    private PaperCard tryToGetCardFromEditions(String cardInfo, CardArtPreference artPreference, int artIndex, Date releaseDate, boolean releasedBeforeFlag, Predicate<PaperCard> filter) {
        PaperCard candidate;
        if (cardInfo == null) {
            return null;
        }
        CardRequest cr = CardRequest.fromString(cardInfo);
        CardArtPreference artPref = artPreference != null ? artPreference : this.defaultCardArtPreference;
        cr.artIndex = Math.max(cr.artIndex, 1);
        if (cr.artIndex != artIndex && artIndex > 1) {
            cr.artIndex = artIndex;
        }
        filter = filter != null ? filter : Predicates.alwaysTrue();
        Predicate<PaperCard> cardQueryFilter = releaseDate != null ? c -> {
            if (c.getArtIndex() != cr.artIndex) {
                return false;
            }
            CardEdition ed = this.editions.get(c.getEdition());
            if (ed == null) {
                return false;
            }
            if (releasedBeforeFlag) {
                return ed.getDate().before(releaseDate);
            }
            return ed.getDate().after(releaseDate);
        } : card -> card.getArtIndex() == cr.artIndex;
        Collection cards = this.getAllCards(cr.cardName, (Predicate)(cardQueryFilter = Predicates.and(cardQueryFilter, filter)));
        if (cards.size() == 1) {
            return cr.isFoil ? ((PaperCard)cards.get(0)).getFoiled() : (PaperCard)cards.get(0);
        }
        ArrayList<CardEdition> cardEditions = new ArrayList<CardEdition>();
        HashMap<String, PaperCard> candidatesCard = new HashMap<String, PaperCard>();
        for (PaperCard card2 : cards) {
            String setCode = card2.getEdition();
            CardEdition ed = setCode.equals(CardEdition.UNKNOWN.getCode()) ? CardEdition.UNKNOWN : this.editions.get(card2.getEdition());
            if (ed == null) continue;
            cardEditions.add(ed);
            candidatesCard.put(setCode, card2);
        }
        if (cardEditions.isEmpty()) {
            return null;
        }
        ArrayList acceptedEditions = Lists.newArrayList(Iterables.filter(cardEditions, artPref::accept));
        if (acceptedEditions.isEmpty()) {
            acceptedEditions.addAll(cardEditions);
        }
        if (acceptedEditions.size() > 1) {
            Collections.sort(acceptedEditions);
            if (artPref.latestFirst) {
                Collections.reverse(acceptedEditions);
            }
        }
        Iterator editionIterator = acceptedEditions.iterator();
        CardEdition ed = (CardEdition)editionIterator.next();
        PaperCard firstCandidate = candidate = (PaperCard)candidatesCard.get(ed.getCode());
        while (!candidate.hasImage() && editionIterator.hasNext()) {
            ed = (CardEdition)editionIterator.next();
            candidate = (PaperCard)candidatesCard.get(ed.getCode());
        }
        candidate = candidate.hasImage() ? candidate : firstCandidate;
        return cr.isFoil ? candidate.getFoiled() : candidate;
    }

    @Override
    public int getMaxArtIndex(String cardName) {
        if (cardName == null) {
            return -1;
        }
        int max = -1;
        for (PaperCard pc : this.getAllCards(cardName)) {
            if (max >= pc.getArtIndex()) continue;
            max = pc.getArtIndex();
        }
        return max;
    }

    @Override
    public int getArtCount(String cardName, String setCode) {
        return this.getArtCount(cardName, setCode, null);
    }

    public int getArtCount(String cardName, String setCode, String functionalVariantName) {
        if (cardName == null || setCode == null) {
            return 0;
        }
        Predicate predicate = card -> card.getEdition().equalsIgnoreCase(setCode);
        if (functionalVariantName != null && !functionalVariantName.equals("")) {
            predicate = Predicates.and(predicate, card -> functionalVariantName.equals(card.getFunctionalVariant()));
        }
        Collection cardsInSet = this.getAllCards(cardName, predicate);
        return cardsInSet.size();
    }

    @Override
    public Collection<PaperCard> getUniqueCards() {
        return this.uniqueCardsByName.values();
    }

    public Collection<PaperCard> getUniqueCardsNoAlt() {
        return Maps.filterEntries(this.uniqueCardsByName, e -> {
            if (e == null) {
                return false;
            }
            return ((String)e.getKey()).equals(((PaperCard)e.getValue()).getName());
        }).values();
    }

    public PaperCard getUniqueByName(String name) {
        return this.uniqueCardsByName.get(this.getName(name));
    }

    public Collection<ICardFace> getAllFaces() {
        return this.facesByName.values();
    }

    public ICardFace getFaceByName(String name) {
        return this.facesByName.get(this.getName(name));
    }

    public boolean isNonLegendaryCreatureName(String name) {
        Boolean bool = this.nonLegendaryCreatureNames.get(name);
        if (bool != null) {
            return bool;
        }
        ICardFace face = StaticData.instance().getCommonCards().getFaceByName(name);
        if (face == null) {
            this.nonLegendaryCreatureNames.put(name, false);
            return false;
        }
        CardType type = face.getType();
        bool = type.isCreature() && !type.isLegendary();
        this.nonLegendaryCreatureNames.put(name, bool);
        return bool;
    }

    @Override
    public Collection<PaperCard> getAllCards() {
        return Collections.unmodifiableCollection(this.allCardsByName.values());
    }

    public Collection<PaperCard> getAllCardsNoAlt() {
        return Multimaps.filterEntries(this.allCardsByName, entry -> ((String)entry.getKey()).equals(((PaperCard)entry.getValue()).getName())).values();
    }

    public Collection<PaperCard> getAllNonPromoCards() {
        return Lists.newArrayList(Iterables.filter(this.getAllCards(), paperCard -> {
            CardEdition edition = null;
            try {
                edition = this.editions.getEditionByCodeOrThrow(paperCard.getEdition());
            }
            catch (Exception ex) {
                return false;
            }
            return edition != null && edition.getType() != CardEdition.Type.PROMO;
        }));
    }

    public Collection<PaperCard> getUniqueCardsNoAltNoOnline() {
        return Lists.newArrayList(Iterables.filter(this.getUniqueCardsNoAlt(), paperCard -> {
            CardEdition edition = null;
            try {
                edition = this.editions.getEditionByCodeOrThrow(paperCard.getEdition());
                if (edition.getType() == CardEdition.Type.ONLINE || edition.getType() == CardEdition.Type.FUNNY) {
                    return false;
                }
            }
            catch (Exception ex) {
                return false;
            }
            return true;
        }));
    }

    public Collection<PaperCard> getAllNonPromosNonReprintsNoAlt() {
        return Lists.newArrayList(Iterables.filter(this.getAllCardsNoAlt(), paperCard -> {
            CardEdition edition = null;
            try {
                edition = this.editions.getEditionByCodeOrThrow(paperCard.getEdition());
                if (edition.getType() == CardEdition.Type.PROMO || edition.getType() == CardEdition.Type.REPRINT || edition.getType() == CardEdition.Type.COLLECTOR_EDITION) {
                    return false;
                }
            }
            catch (Exception ex) {
                return false;
            }
            return true;
        }));
    }

    public String getName(String cardName) {
        return this.getName(cardName, false);
    }

    public String getName(String cardName, boolean engine) {
        if (this.alternateName.containsKey(cardName = this.normalizedNames.getOrDefault(cardName, cardName)) && engine) {
            return this.alternateName.get(cardName);
        }
        return cardName;
    }

    public List<PaperCard> getAllCards(String cardName) {
        return this.allCardsByName.get((Object)this.getName(cardName));
    }

    public List<PaperCard> getAllCardsNoAlt(String cardName) {
        return Lists.newArrayList(Multimaps.filterEntries(this.allCardsByName, entry -> ((String)entry.getKey()).equals(((PaperCard)entry.getValue()).getName())).get(this.getName(cardName)));
    }

    public List<PaperCard> getAllCards(Predicate<PaperCard> predicate) {
        return Lists.newArrayList(Iterables.filter(this.getAllCards(), predicate));
    }

    public List<PaperCard> getAllCards(String cardName, Predicate<PaperCard> predicate) {
        return Lists.newArrayList(Iterables.filter(this.getAllCards(cardName), predicate));
    }

    public List<PaperCard> getAllCardsNoAlt(Predicate<PaperCard> predicate) {
        return Lists.newArrayList(Iterables.filter(this.getAllCardsNoAlt(), predicate));
    }

    @Override
    public Collection<PaperCard> getAllCards(CardEdition edition) {
        ArrayList<PaperCard> cards = Lists.newArrayList();
        for (CardEdition.CardInSet cis : edition.getAllCardsInSet()) {
            PaperCard card = this.getCard(cis.name, edition.getCode());
            if (card == null) continue;
            cards.add(card);
        }
        return cards;
    }

    @Override
    public boolean contains(String name) {
        return this.allCardsByName.containsKey(this.getName(name));
    }

    @Override
    public Iterator<PaperCard> iterator() {
        return this.getAllCards().iterator();
    }

    @Override
    public Predicate<? super PaperCard> wasPrintedInSets(List<String> setCodes) {
        return new PredicateExistsInSets(setCodes);
    }

    @Override
    public Predicate<? super PaperCard> isLegal(List<String> allowedSetCodes) {
        return new PredicateLegalInSets(allowedSetCodes);
    }

    @Override
    public Predicate<? super PaperCard> wasPrintedAtRarity(CardRarity rarity) {
        return new PredicatePrintedAtRarity(rarity);
    }

    public StringBuilder appendCardToStringBuilder(PaperCard card, StringBuilder sb) {
        boolean hasBadSetInfo = card.getEdition().equals(CardEdition.UNKNOWN.getCode()) || StringUtils.isBlank(card.getEdition());
        sb.append(card.getName());
        if (card.isFoil()) {
            sb.append(foilSuffix);
        }
        if (!hasBadSetInfo) {
            int artCount = this.getArtCount(card.getName(), card.getEdition(), card.getFunctionalVariant());
            sb.append('|').append(card.getEdition());
            if (artCount >= 1) {
                sb.append('|').append(card.getArtIndex());
            }
        }
        return sb;
    }

    public PaperCard createUnsupportedCard(String cardRequest) {
        CardRequest request = CardRequest.fromString(cardRequest);
        CardEdition cardEdition = CardEdition.UNKNOWN;
        CardRarity cardRarity = CardRarity.Unknown;
        if (StringUtils.isBlank(request.edition)) {
            for (CardEdition edition : this.editions) {
                for (CardEdition.CardInSet cardInSet : edition.getAllCardsInSet()) {
                    if (!cardInSet.name.equals(request.cardName)) continue;
                    cardEdition = edition;
                    cardRarity = cardInSet.rarity;
                    break;
                }
                if (cardEdition == CardEdition.UNKNOWN) continue;
                break;
            }
        } else {
            cardEdition = this.editions.get(request.edition);
            if (cardEdition != null) {
                for (CardEdition.CardInSet cardInSet : cardEdition.getAllCardsInSet()) {
                    if (!cardInSet.name.equals(request.cardName)) continue;
                    cardRarity = cardInSet.rarity;
                    break;
                }
            } else {
                cardEdition = CardEdition.UNKNOWN;
            }
        }
        if (cardRarity == CardRarity.Unknown) {
            System.err.println("Forge could not find this card in the Database. Any chance you might have mistyped the card name?");
        } else {
            System.err.println("We're sorry, but this card is not supported yet.");
        }
        return new PaperCard(CardRules.getUnsupportedCardNamed(request.cardName), cardEdition.getCode(), cardRarity);
    }

    public Editor getEditor() {
        return this.editor;
    }

    public class Editor {
        private boolean immediateReindex = true;

        public CardRules putCard(CardRules rules) {
            return this.putCard(rules, null);
        }

        public CardRules putCard(CardRules rules, List<Pair<String, CardRarity>> whenItWasPrinted) {
            String cardName = rules.getName();
            CardRules result = (CardRules)CardDb.this.rulesByName.get(cardName);
            if (result != null && result.getName().equals(cardName)) {
                result.reinitializeFromRules(rules);
                return result;
            }
            result = CardDb.this.rulesByName.put(cardName, rules);
            ArrayList<PaperCard> paperCards = new ArrayList<PaperCard>();
            if (null == whenItWasPrinted || whenItWasPrinted.isEmpty()) {
                for (CardEdition e : CardDb.this.editions.getOrderedEditions()) {
                    int artIdx = 1;
                    for (CardEdition.CardInSet cis : e.getCardInSet(cardName)) {
                        paperCards.add(new PaperCard(rules, e.getCode(), cis.rarity, artIdx++, false, cis.collectorNumber, cis.artistName, cis.functionalVariantName));
                    }
                }
            } else {
                String lastEdition = null;
                int artIdx = 0;
                for (Pair<String, CardRarity> tuple : whenItWasPrinted) {
                    List<CardEdition.CardInSet> cardsInSet;
                    CardEdition ed;
                    if (!tuple.getKey().equals(lastEdition)) {
                        artIdx = 1;
                        lastEdition = tuple.getKey();
                    }
                    if ((ed = CardDb.this.editions.get(lastEdition)) == null || (cardsInSet = ed.getCardInSet(cardName)).isEmpty()) continue;
                    int cardInSetIndex = Math.max(artIdx - 1, 0);
                    CardEdition.CardInSet cds = cardsInSet.get(cardInSetIndex);
                    paperCards.add(new PaperCard(rules, lastEdition, tuple.getValue(), artIdx++, false, cds.collectorNumber, cds.artistName, cds.functionalVariantName));
                }
            }
            if (paperCards.isEmpty()) {
                paperCards.add(new PaperCard(rules, CardEdition.UNKNOWN.getCode(), CardRarity.Special));
            }
            for (PaperCard paperCard : paperCards) {
                CardDb.this.addCard(paperCard);
            }
            if (this.immediateReindex) {
                CardDb.this.reIndex();
            }
            return result;
        }

        public boolean isImmediateReindex() {
            return this.immediateReindex;
        }

        public void setImmediateReindex(boolean immediateReindex) {
            this.immediateReindex = immediateReindex;
        }
    }

    private class PredicatePrintedAtRarity
    implements Predicate<PaperCard> {
        private final Set<String> matchingCards = new HashSet<String>();

        public PredicatePrintedAtRarity(CardRarity rarity) {
            for (PaperCard c : CardDb.this.getAllCards()) {
                if (c.getRarity() != rarity) continue;
                this.matchingCards.add(c.getName());
            }
        }

        @Override
        public boolean apply(PaperCard subject) {
            return this.matchingCards.contains(subject.getName());
        }
    }

    private class PredicateLegalInSets
    implements Predicate<PaperCard> {
        private final List<String> sets = new ArrayList<String>();

        public PredicateLegalInSets(List<String> allowedSets) {
            this.sets.addAll(allowedSets);
        }

        @Override
        public boolean apply(PaperCard card) {
            if (card == null) {
                return false;
            }
            return this.sets.contains(card.getEdition());
        }
    }

    private class PredicateExistsInSets
    implements Predicate<PaperCard> {
        private final List<String> sets;

        public PredicateExistsInSets(List<String> wantSets) {
            this.sets = wantSets;
        }

        @Override
        public boolean apply(PaperCard subject) {
            for (PaperCard c : CardDb.this.getAllCards(subject.getName())) {
                if (!this.sets.contains(c.getEdition())) continue;
                return true;
            }
            return false;
        }
    }

    public static class CardRequest {
        public String cardName;
        public String edition;
        public int artIndex;
        public boolean isFoil;
        public String collectorNumber;

        private CardRequest(String name, String edition, int artIndex, boolean isFoil, String collectorNumber) {
            this.cardName = name;
            this.edition = edition;
            this.artIndex = artIndex;
            this.isFoil = isFoil;
            this.collectorNumber = collectorNumber;
        }

        public static boolean isFoilCardName(String cardName) {
            return cardName.trim().endsWith(CardDb.foilSuffix);
        }

        public static String compose(String cardName, boolean isFoil) {
            if (isFoil) {
                return CardRequest.isFoilCardName(cardName) ? cardName : cardName + CardDb.foilSuffix;
            }
            return CardRequest.isFoilCardName(cardName) ? cardName.substring(0, cardName.length() - CardDb.foilSuffix.length()) : cardName;
        }

        public static String compose(String cardName, String setCode) {
            setCode = setCode != null ? setCode : "";
            String string = cardName = cardName != null ? cardName : "";
            if (cardName.indexOf(124) != -1) {
                cardName = CardRequest.fromString((String)cardName).cardName;
            }
            return cardName + '|' + setCode;
        }

        public static String compose(String cardName, String setCode, int artIndex) {
            String requestInfo = CardRequest.compose(cardName, setCode);
            artIndex = Math.max(artIndex, 1);
            return requestInfo + '|' + artIndex;
        }

        public static String compose(String cardName, String setCode, String collectorNumber) {
            String requestInfo = CardRequest.compose(cardName, setCode);
            collectorNumber = CardRequest.preprocessCollectorNumber(collectorNumber);
            return requestInfo + '|' + collectorNumber;
        }

        private static String preprocessCollectorNumber(String collectorNumber) {
            if (collectorNumber == null) {
                return "";
            }
            if (!(collectorNumber = collectorNumber.trim()).startsWith("[")) {
                collectorNumber = "[" + collectorNumber;
            }
            if (!collectorNumber.endsWith("]")) {
                collectorNumber = collectorNumber + "]";
            }
            return collectorNumber;
        }

        public static String compose(String cardName, String setCode, int artIndex, String collectorNumber) {
            String requestInfo = CardRequest.compose(cardName, setCode, artIndex);
            collectorNumber = CardRequest.preprocessCollectorNumber(collectorNumber);
            return requestInfo + '|' + collectorNumber;
        }

        private static boolean isCollectorNumber(String s2) {
            return s2.startsWith("[") && s2.endsWith("]");
        }

        private static boolean isArtIndex(String s2) {
            return StringUtils.isNumeric(s2) && s2.length() <= 2;
        }

        private static boolean isSetCode(String s2) {
            return !StringUtils.isNumeric(s2);
        }

        private static CardRequest fromPreferredArtEntry(String preferredArt, boolean isFoil) {
            String[] info = TextUtil.split(preferredArt, '|');
            if (info.length != 3) {
                return null;
            }
            try {
                String cardName = info[0];
                String setCode = info[1];
                int artIndex = Integer.parseInt(info[2]);
                return new CardRequest(cardName, setCode, artIndex, isFoil, "N.A.");
            }
            catch (NumberFormatException ex) {
                return null;
            }
        }

        public static CardRequest fromString(String reqInfo) {
            String preferredArt;
            String setCode;
            int cNrPos;
            int artPos;
            int setPos;
            if (reqInfo == null) {
                return null;
            }
            String[] info = TextUtil.split(reqInfo, '|');
            if (info.length >= 4) {
                setPos = CardRequest.isSetCode(info[1]) ? 1 : -1;
                artPos = CardRequest.isArtIndex(info[2]) ? 2 : -1;
                cNrPos = CardRequest.isCollectorNumber(info[3]) ? 3 : -1;
            } else if (info.length == 3) {
                setPos = CardRequest.isSetCode(info[1]) ? 1 : -1;
                artPos = CardRequest.isArtIndex(info[2]) ? 2 : -1;
                cNrPos = CardRequest.isCollectorNumber(info[2]) ? 2 : -1;
            } else if (info.length == 2) {
                setPos = CardRequest.isSetCode(info[1]) ? 1 : -1;
                artPos = CardRequest.isArtIndex(info[1]) ? 1 : -1;
                cNrPos = -1;
            } else {
                setPos = -1;
                artPos = -1;
                cNrPos = -1;
            }
            String cardName = info[0];
            boolean isFoil = false;
            if (CardRequest.isFoilCardName(cardName)) {
                cardName = cardName.substring(0, cardName.length() - CardDb.foilSuffix.length());
                isFoil = true;
            }
            int artIndex = artPos > 0 ? Integer.parseInt(info[artPos]) : -1;
            String collectorNumber = cNrPos > 0 ? info[cNrPos].substring(1, info[cNrPos].length() - 1) : "N.A.";
            String string = setCode = setPos > 0 ? info[setPos] : null;
            if (setCode != null && setCode.equals(CardEdition.UNKNOWN.getCode())) {
                setCode = null;
            }
            if (setCode == null && (preferredArt = (String)artPrefs.get(cardName)) != null) {
                CardRequest request = CardRequest.fromPreferredArtEntry(preferredArt, isFoil);
                if (request != null) {
                    return request;
                }
                System.err.printf("[LOG]: Faulty Entry in Preferred Art for Card %s - Please check!%n", cardName);
            }
            if (collectorNumber.equals("N.A.") && artIndex == -1) {
                artIndex = 1;
            }
            return new CardRequest(cardName, setCode, artIndex, isFoil, collectorNumber);
        }
    }

    public static enum CardArtPreference {
        LATEST_ART_ALL_EDITIONS(false, true),
        LATEST_ART_CORE_EXPANSIONS_REPRINT_ONLY(true, true),
        ORIGINAL_ART_ALL_EDITIONS(false, false),
        ORIGINAL_ART_CORE_EXPANSIONS_REPRINT_ONLY(true, false);

        public final boolean filterSets;
        public final boolean latestFirst;

        private CardArtPreference(boolean filterIrregularSets, boolean latestSetFirst) {
            this.filterSets = filterIrregularSets;
            this.latestFirst = latestSetFirst;
        }

        public boolean accept(CardEdition ed) {
            if (ed == null) {
                return false;
            }
            return !this.filterSets || ed.getType() == CardEdition.Type.CORE || ed.getType() == CardEdition.Type.EXPANSION || ed.getType() == CardEdition.Type.REPRINT;
        }
    }
}

