/*
 * Decompiled with CFR 0.152.
 */
package forge.deck;

import com.google.common.collect.Lists;
import forge.StaticData;
import forge.card.CardDb;
import forge.card.CardEdition;
import forge.card.CardType;
import forge.card.MagicColor;
import forge.deck.DeckFormat;
import forge.deck.DeckSection;
import forge.item.PaperCard;
import forge.util.Localizer;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

public class DeckRecognizer {
    private static final Pattern SEARCH_SINGLE_SLASH = Pattern.compile("(?<=[^/])\\s*/\\s*(?=[^/])");
    private static final String DOUBLE_SLASH = "//";
    private static final String LINE_COMMENT_DELIMITER_OR_MD_HEADER = "#";
    private static final String ASTERISK = "* ";
    public static final String REGRP_DECKNAME = "deckName";
    public static final String REX_DECK_NAME = String.format("^(\\/\\/\\s*)?(?<pre>(deck|name(\\s)?))(\\:|=)\\s*(?<%s>([a-zA-Z0-9',\\/\\-\\s\\)\\]\\(\\[\\#]+))\\s*(.*)$", "deckName");
    public static final Pattern DECK_NAME_PATTERN = Pattern.compile(REX_DECK_NAME, 2);
    public static final String REGRP_TOKEN = "token";
    public static final String REGRP_COLR1 = "colr1";
    public static final String REGRP_COLR2 = "colr2";
    public static final String REGRP_MANA = "mana";
    public static final String REX_NOCARD = String.format("^(?<pre>[^a-zA-Z]*)\\s*(?<title>(\\w+[:]\\s*))?(?<%s>[a-zA-Z]+)(?<post>[^a-zA-Z]*)?$", "token");
    public static final String REX_CMC = String.format("^(?<pre>[^a-zA-Z]*)\\s*(?<%s>(C(M)?C(\\s)?\\d{1,2}))(?<post>[^\\d]*)?$", "token");
    public static final String REX_RARITY = String.format("^(?<pre>[^a-zA-Z]*)\\s*(?<%s>((un)?common|(mythic)?\\s*(rare)?|land|special))(?<post>[^a-zA-Z]*)?$", "token");
    public static final String MANA_SYMBOLS = "w|u|b|r|g|c|m|wu|ub|br|rg|gw|wb|ur|bg|rw|gu";
    public static final String REX_MANA_SYMBOLS = String.format("\\{(?<%s>(%s))\\}", "mana", "w|u|b|r|g|c|m|wu|ub|br|rg|gw|wb|ur|bg|rw|gu");
    public static final String REX_MANA_COLOURS = String.format("(\\{(%s)\\})|(white|blue|black|red|green|colo(u)?rless|multicolo(u)?r)", "w|u|b|r|g|c|m|wu|ub|br|rg|gw|wb|ur|bg|rw|gu");
    public static final String REX_MANA = String.format("^(?<pre>[^a-zA-Z]*)\\s*(?<%s>(%s))((\\s|-|\\|)(?<%s>(%s)))?(?<post>[^a-zA-Z]*)?$", "colr1", REX_MANA_COLOURS, "colr2", REX_MANA_COLOURS);
    public static final Pattern NONCARD_PATTERN = Pattern.compile(REX_NOCARD, 2);
    public static final Pattern CMC_PATTERN = Pattern.compile(REX_CMC, 2);
    public static final Pattern CARD_RARITY_PATTERN = Pattern.compile(REX_RARITY, 2);
    public static final Pattern MANA_PATTERN = Pattern.compile(REX_MANA, 2);
    public static final Pattern MANA_SYMBOL_PATTERN = Pattern.compile(REX_MANA_SYMBOLS, 2);
    public static final String REGRP_SET = "setcode";
    public static final String REGRP_COLLNR = "collnr";
    public static final String REGRP_CARD = "cardname";
    public static final String REGRP_CARDNO = "count";
    public static final String REX_CARD_NAME = String.format("(\\[)?(?<%s>[a-zA-Z0-9\u00e0-\u00ff\u00c0-\u0178&',\\.:!\\+\\\"\\/\\-\\s]+)(\\])?", "cardname");
    public static final String REX_SET_CODE = String.format("(?<%s>[a-zA-Z0-9_]{2,7})", "setcode");
    public static final String REX_COLL_NUMBER = String.format("(?<%s>\\*?[0-9A-Z]+\\S?[A-Z]*)", "collnr");
    public static final String REX_CARD_COUNT = String.format("(?<%s>[\\d]{1,2})(?<mult>x)?", "count");
    public static final String REGRP_FOIL_GFISH = "foil";
    private static final String REX_FOIL_MTGGOLDFISH = String.format("(?<%s>\\(F\\))?", "foil");
    public static final String REGRP_DECK_SEC_XMAGE_STYLE = "decsec";
    private static final String REX_DECKSEC_XMAGE = String.format("(?<%s>(MB|SB|CM))", "decsec");
    public static final String REX_CARD_SET_REQUEST = String.format("(%s\\s*:\\s*)?(%s\\s)?\\s*%s\\s*(\\s|\\||\\(|\\[|\\{)\\s?%s(\\s|\\)|\\]|\\})?\\s*%s", REX_DECKSEC_XMAGE, REX_CARD_COUNT, REX_CARD_NAME, REX_SET_CODE, REX_FOIL_MTGGOLDFISH);
    public static final Pattern CARD_SET_PATTERN = Pattern.compile(REX_CARD_SET_REQUEST);
    public static final String REX_SET_CARD_REQUEST = String.format("(%s\\s*:\\s*)?(%s\\s)?\\s*(\\(|\\[|\\{)?%s(\\s+|\\)|\\]|\\}|\\|)\\s*%s\\s*%s\\s*", REX_DECKSEC_XMAGE, REX_CARD_COUNT, REX_SET_CODE, REX_CARD_NAME, REX_FOIL_MTGGOLDFISH);
    public static final Pattern SET_CARD_PATTERN = Pattern.compile(REX_SET_CARD_REQUEST);
    public static final String REX_FULL_REQUEST_CARD_SET = String.format("(%s\\s*:\\s*)?(%s\\s)?\\s*%s\\s*(\\||\\(|\\[|\\{|\\s)%s(\\s|\\)|\\]|\\})?(\\s+|\\|\\s*)%s\\s*%s\\s*", REX_DECKSEC_XMAGE, REX_CARD_COUNT, REX_CARD_NAME, REX_SET_CODE, REX_COLL_NUMBER, REX_FOIL_MTGGOLDFISH);
    public static final Pattern CARD_SET_COLLNO_PATTERN = Pattern.compile(REX_FULL_REQUEST_CARD_SET);
    public static final String REX_FULL_REQUEST_SET_CARD = String.format("^(%s\\s*:\\s*)?(%s\\s)?\\s*(\\(|\\[|\\{)?%s(\\s+|\\)|\\]|\\}|\\|)\\s*%s(\\s+|\\|\\s*)%s\\s*%s$", REX_DECKSEC_XMAGE, REX_CARD_COUNT, REX_SET_CODE, REX_CARD_NAME, REX_COLL_NUMBER, REX_FOIL_MTGGOLDFISH);
    public static final Pattern SET_CARD_COLLNO_PATTERN = Pattern.compile(REX_FULL_REQUEST_SET_CARD);
    public static final String REX_FULL_REQUEST_CARD_COLLNO_SET = String.format("^(%s\\s*:\\s*)?(%s\\s)?\\s*%s\\s+(\\<%s\\>)\\s*(\\(|\\[|\\{)?%s(\\s+|\\)|\\]|\\}|\\|)\\s*%s$", REX_DECKSEC_XMAGE, REX_CARD_COUNT, REX_CARD_NAME, REX_COLL_NUMBER, REX_SET_CODE, REX_FOIL_MTGGOLDFISH);
    public static final Pattern CARD_COLLNO_SET_PATTERN = Pattern.compile(REX_FULL_REQUEST_CARD_COLLNO_SET);
    public static final String REX_FULL_REQUEST_XMAGE = String.format("^(%s\\s*:\\s*)?(%s\\s)?\\s*(\\[)?%s:%s(\\])\\s+%s\\s*%s$", REX_DECKSEC_XMAGE, REX_CARD_COUNT, REX_SET_CODE, REX_COLL_NUMBER, REX_CARD_NAME, REX_FOIL_MTGGOLDFISH);
    public static final Pattern SET_COLLNO_CARD_XMAGE_PATTERN = Pattern.compile(REX_FULL_REQUEST_XMAGE);
    public static final String REX_CARDONLY = String.format("(%s\\s*:\\s*)?(%s\\s)?\\s*%s\\s*%s", REX_DECKSEC_XMAGE, REX_CARD_COUNT, REX_CARD_NAME, REX_FOIL_MTGGOLDFISH);
    public static final Pattern CARD_ONLY_PATTERN = Pattern.compile(REX_CARDONLY);
    private static final CharSequence[] CARD_TYPES = DeckRecognizer.allCardTypes();
    private static final CharSequence[] DECK_SECTION_NAMES = new CharSequence[]{"side", "sideboard", "sb", "main", "card", "mainboard", "avatar", "commander", "schemes", "conspiracy", "planes", "deck", "dungeon"};
    private Date releaseDateConstraint = null;
    private List<String> allowedSetCodes = null;
    private List<String> gameFormatBannedCards = null;
    private List<String> gameFormatRestrictedCards = null;
    private List<DeckSection> allowedDeckSections = null;
    private boolean includeBannedAndRestricted = false;
    private DeckFormat deckFormat = null;
    private CardDb.CardArtPreference artPreference = StaticData.instance().getCardArtPreference();
    private static final HashMap<Integer, String> manaSymbolsMap = new HashMap<Integer, String>(){
        {
            this.put(3, "WU");
            this.put(6, "UB");
            this.put(12, "BR");
            this.put(24, "RG");
            this.put(17, "GW");
            this.put(5, "WB");
            this.put(10, "UR");
            this.put(20, "BG");
            this.put(9, "RW");
            this.put(18, "GU");
        }
    };

    private static CharSequence[] allCardTypes() {
        ArrayList<String> cardTypesList = new ArrayList<String>();
        ArrayList<CardType.CoreType> coreTypes = Lists.newArrayList(CardType.CoreType.values());
        for (CardType.CoreType coreType : coreTypes) {
            cardTypesList.add(coreType.name().toLowerCase());
        }
        cardTypesList.add("sorceries");
        cardTypesList.add("aura");
        cardTypesList.add(REGRP_MANA);
        cardTypesList.add("spell");
        cardTypesList.add("other spell");
        cardTypesList.add("planeswalker");
        return cardTypesList.toArray(new CharSequence[0]);
    }

    public List<Token> parseCardList(String[] cardList) {
        ArrayList<Token> tokens = new ArrayList<Token>();
        DeckSection referenceDeckSectionInParsing = null;
        for (String line : cardList) {
            Token token = this.recognizeLine(line, referenceDeckSectionInParsing);
            if (token == null) continue;
            TokenType tokenType = token.getType();
            if (!token.isTokenForDeck() && tokenType != TokenType.DECK_SECTION_NAME || tokenType == TokenType.LIMITED_CARD && !this.includeBannedAndRestricted) {
                tokens.add(token);
                continue;
            }
            if (token.getType() == TokenType.DECK_NAME) {
                tokens.add(0, token);
                continue;
            }
            if (token.getType() == TokenType.DECK_SECTION_NAME) {
                referenceDeckSectionInParsing = DeckSection.valueOf(token.getText());
                tokens.add(token);
                continue;
            }
            DeckSection tokenSection = token.getTokenSection();
            PaperCard tokenCard = token.getCard();
            if (this.isAllowed(tokenSection)) {
                if (!tokenSection.equals((Object)referenceDeckSectionInParsing)) {
                    Token sectionToken = Token.DeckSection(tokenSection.name(), this.allowedDeckSections);
                    if (!tokens.isEmpty() && ((Token)tokens.get(tokens.size() - 1)).isCardPlaceholder()) {
                        tokens.add(tokens.size() - 1, sectionToken);
                    } else {
                        tokens.add(sectionToken);
                    }
                    referenceDeckSectionInParsing = tokenSection;
                }
                tokens.add(token);
                continue;
            }
            Token unsupportedCard = Token.UnsupportedCard(tokenCard.getName(), tokenCard.getEdition(), token.getQuantity());
            tokens.add(unsupportedCard);
        }
        return tokens;
    }

    private boolean isAllowed(DeckSection tokenSection) {
        return this.allowedDeckSections == null || this.allowedDeckSections.contains((Object)tokenSection);
    }

    public Token recognizeLine(String rawLine, DeckSection referenceSection) {
        if (rawLine == null) {
            return null;
        }
        if (StringUtils.isBlank(rawLine.trim())) {
            return null;
        }
        int smartQuote = 8217;
        String refLine = rawLine.trim().replace('\u2019', '\'');
        String line = StringUtils.startsWith(refLine = DeckRecognizer.purgeAllLinks(refLine), LINE_COMMENT_DELIMITER_OR_MD_HEADER) ? refLine.replaceAll(LINE_COMMENT_DELIMITER_OR_MD_HEADER, "") : refLine.trim();
        if (StringUtils.startsWith(line = SEARCH_SINGLE_SLASH.matcher(line).replaceFirst(" // "), ASTERISK)) {
            line = line.substring(2);
        }
        if (line.endsWith("#!Commander")) {
            line = line.replaceAll("#!Commander", "");
            line = String.format("CM:%s", line.trim());
        } else if (line.trim().equals("[Conspiracy]")) {
            line = String.format("/ %s", line);
        }
        Token result = this.recogniseCardToken(line, referenceSection);
        if (result == null) {
            result = this.recogniseNonCardToken(line);
        }
        return result != null ? result : (StringUtils.startsWith(refLine, DOUBLE_SLASH) || StringUtils.startsWith(refLine, LINE_COMMENT_DELIMITER_OR_MD_HEADER) ? new Token(TokenType.COMMENT, 0, refLine) : new Token(TokenType.UNKNOWN_TEXT, 0, refLine));
    }

    public static String purgeAllLinks(String line) {
        String urlPattern = "(?<protocol>((https|ftp|file|http):))(?<sep>((//|\\\\)+))(?<url>([\\w\\d:#@%/;$~_?+-=\\\\.&]*))";
        Pattern p = Pattern.compile(urlPattern, 2);
        Matcher m4 = p.matcher(line);
        while (m4.find()) {
            line = line.replaceAll(m4.group(), "").trim();
        }
        if (StringUtils.endsWith(line, "()")) {
            return line.substring(0, line.length() - 2);
        }
        return line;
    }

    public Token recogniseCardToken(String text, DeckSection currentDeckSection) {
        String line = text.trim();
        Token unknownCardToken = null;
        StaticData data = StaticData.instance();
        List<Matcher> cardMatchers = this.getRegExMatchers(line);
        for (Matcher matcher : cardMatchers) {
            int artIndex;
            int cardCount;
            String cardName = this.getRexGroup(matcher, REGRP_CARD);
            if (cardName == null) continue;
            if (!data.isMTGCard(cardName = cardName.trim())) {
                cardName = this.checkDoubleSidedCard(cardName);
            }
            String ccount = this.getRexGroup(matcher, REGRP_CARDNO);
            String setCode = this.getRexGroup(matcher, REGRP_SET);
            String collNo = this.getRexGroup(matcher, REGRP_COLLNR);
            String foilGr = this.getRexGroup(matcher, REGRP_FOIL_GFISH);
            String deckSecFromCardLine = this.getRexGroup(matcher, REGRP_DECK_SEC_XMAGE_STYLE);
            boolean isFoil = foilGr != null;
            int n = cardCount = ccount != null ? Integer.parseInt(ccount) : 1;
            if (cardName == null) {
                if (ccount == null) continue;
                unknownCardToken = Token.UnknownCard(text, null, 0);
                continue;
            }
            String collectorNumber = collNo != null ? collNo : "N.A.";
            try {
                artIndex = Integer.parseInt(collectorNumber);
            }
            catch (NumberFormatException ex) {
                artIndex = -1;
            }
            if (setCode != null) {
                CardEdition edition = StaticData.instance().getEditions().get(setCode);
                if (edition == null) {
                    unknownCardToken = Token.UnknownCard(cardName, setCode, cardCount);
                    continue;
                }
                PaperCard pc = data.getCardFromSet(cardName, edition, collectorNumber, artIndex, isFoil);
                if (pc != null) {
                    return this.checkAndSetCardToken(pc, edition, cardCount, deckSecFromCardLine, currentDeckSection, true);
                }
                return Token.UnknownCard(cardName, setCode, cardCount);
            }
            PaperCard pc = null;
            if (this.hasGameFormatConstraints()) {
                pc = data.getCardFromSupportedEditions(cardName, isFoil, this.artPreference, this.allowedSetCodes, this.releaseDateConstraint);
            }
            if (pc == null) {
                pc = data.getCardFromSupportedEditions(cardName, isFoil, this.artPreference, null, this.releaseDateConstraint);
            }
            if (pc == null) continue;
            CardEdition edition = StaticData.instance().getCardEdition(pc.getEdition());
            return this.checkAndSetCardToken(pc, edition, cardCount, deckSecFromCardLine, currentDeckSection, false);
        }
        return unknownCardToken;
    }

    private String checkDoubleSidedCard(String cardName) {
        if (!cardName.contains(DOUBLE_SLASH)) {
            return null;
        }
        String cardRequest = cardName.trim();
        String[] sides = cardRequest.split(DOUBLE_SLASH);
        if (sides.length != 2) {
            return null;
        }
        String leftSide = sides[0].trim();
        String rightSide = sides[1].trim();
        StaticData data = StaticData.instance();
        if (data.isMTGCard(leftSide)) {
            return leftSide;
        }
        if (data.isMTGCard(rightSide)) {
            return rightSide;
        }
        return null;
    }

    private Token checkAndSetCardToken(PaperCard pc, CardEdition edition, int cardCount, String deckSecFromCardLine, DeckSection referenceSection, boolean cardRequestHasSetCode) {
        if (this.IsIllegalInFormat(edition.getCode())) {
            return Token.NotAllowedCard(pc, cardCount, cardRequestHasSetCode);
        }
        if (this.isNotCompliantWithReleaseDateRestrictions(edition)) {
            return Token.CardInInvalidSet(pc, cardCount, cardRequestHasSetCode);
        }
        DeckSection tokenSection = this.getTokenSection(deckSecFromCardLine, referenceSection, pc);
        if (this.isBannedInFormat(pc)) {
            return Token.LimitedCard(pc, cardCount, tokenSection, LimitedCardType.BANNED, cardRequestHasSetCode);
        }
        if (this.isRestrictedInFormat(pc, cardCount)) {
            return Token.LimitedCard(pc, cardCount, tokenSection, LimitedCardType.RESTRICTED, cardRequestHasSetCode);
        }
        return Token.LegalCard(pc, cardCount, tokenSection, cardRequestHasSetCode);
    }

    private DeckSection getTokenSection(String deckSec, DeckSection currentDeckSection, PaperCard card) {
        if (deckSec != null) {
            DeckSection cardSection;
            switch (deckSec.toUpperCase().trim()) {
                case "MB": {
                    cardSection = DeckSection.Main;
                    break;
                }
                case "SB": {
                    cardSection = DeckSection.Sideboard;
                    break;
                }
                case "CM": {
                    cardSection = DeckSection.Commander;
                    break;
                }
                default: {
                    cardSection = DeckSection.matchingSection(card);
                }
            }
            if (cardSection.validate(card)) {
                return cardSection;
            }
        }
        if (currentDeckSection != null) {
            if (currentDeckSection.validate(card)) {
                return currentDeckSection;
            }
            return DeckSection.matchingSection(card);
        }
        DeckSection matchedSection = DeckSection.matchingSection(card);
        if (matchedSection == DeckSection.Main && this.isAllowed(DeckSection.Commander) && DeckSection.Commander.validate(card)) {
            return DeckSection.Commander;
        }
        if (this.isAllowed(matchedSection)) {
            return matchedSection;
        }
        return DeckSection.Main.validate(card) ? DeckSection.Main : matchedSection;
    }

    private boolean hasGameFormatConstraints() {
        return this.allowedSetCodes != null && !this.allowedSetCodes.isEmpty() || this.gameFormatBannedCards != null && !this.gameFormatBannedCards.isEmpty() || this.gameFormatRestrictedCards != null && !this.gameFormatRestrictedCards.isEmpty();
    }

    private String getRexGroup(Matcher matcher, String groupName) {
        String rexGroup;
        try {
            rexGroup = matcher.group(groupName);
        }
        catch (IllegalArgumentException ex) {
            rexGroup = null;
        }
        return rexGroup;
    }

    private boolean isBannedInFormat(PaperCard pc) {
        return this.gameFormatBannedCards != null && this.gameFormatBannedCards.contains(pc.getName()) || this.deckFormat != null && !this.deckFormat.isLegalCard(pc);
    }

    private boolean isRestrictedInFormat(PaperCard pc, int cardCount) {
        return this.gameFormatRestrictedCards != null && this.gameFormatRestrictedCards.contains(pc.getName()) && cardCount > 1;
    }

    private boolean IsIllegalInFormat(String setCode) {
        return this.allowedSetCodes != null && !this.allowedSetCodes.contains(setCode);
    }

    private boolean isNotCompliantWithReleaseDateRestrictions(CardEdition edition) {
        return this.releaseDateConstraint != null && edition.getDate().compareTo(this.releaseDateConstraint) >= 0;
    }

    private List<Matcher> getRegExMatchers(String line) {
        Pattern[] OtherPatterns;
        Pattern[] patternsWithCollNumber;
        ArrayList<Matcher> matchers = new ArrayList<Matcher>();
        for (Pattern pattern : patternsWithCollNumber = new Pattern[]{CARD_SET_COLLNO_PATTERN, SET_CARD_COLLNO_PATTERN, CARD_COLLNO_SET_PATTERN, SET_COLLNO_CARD_XMAGE_PATTERN}) {
            Matcher matcher = pattern.matcher(line);
            if (!matcher.matches() || this.getRexGroup(matcher, REGRP_SET) == null || this.getRexGroup(matcher, REGRP_COLLNR) == null) continue;
            matchers.add(matcher);
        }
        for (Pattern pattern : OtherPatterns = new Pattern[]{CARD_SET_PATTERN, SET_CARD_PATTERN, CARD_ONLY_PATTERN}) {
            Matcher matcher = pattern.matcher(line);
            if (!matcher.matches()) continue;
            matchers.add(matcher);
        }
        return matchers;
    }

    public Token recogniseNonCardToken(String text) {
        if (DeckRecognizer.isDeckSectionName(text)) {
            String tokenText = DeckRecognizer.nonCardTokenMatch(text);
            return Token.DeckSection(tokenText, this.allowedDeckSections);
        }
        if (DeckRecognizer.isCardCMC(text)) {
            String tokenText = this.getCardCMCMatch(text);
            return new Token(TokenType.CARD_CMC, tokenText);
        }
        if (DeckRecognizer.isCardRarity(text)) {
            String tokenText = DeckRecognizer.cardRarityTokenMatch(text);
            if (tokenText != null && !tokenText.trim().isEmpty()) {
                return new Token(TokenType.CARD_RARITY, tokenText);
            }
            return null;
        }
        if (DeckRecognizer.isCardType(text)) {
            String tokenText = DeckRecognizer.nonCardTokenMatch(text);
            return new Token(TokenType.CARD_TYPE, tokenText);
        }
        if (DeckRecognizer.isManaToken(text)) {
            String tokenText = DeckRecognizer.getManaTokenMatch(text);
            return new Token(TokenType.MANA_COLOUR, tokenText);
        }
        if (DeckRecognizer.isDeckName(text)) {
            String deckName = DeckRecognizer.deckNameMatch(text);
            return new Token(TokenType.DECK_NAME, deckName.trim());
        }
        return null;
    }

    public static boolean isCardType(String lineAsIs) {
        String nonCardToken = DeckRecognizer.nonCardTokenMatch(lineAsIs);
        if (nonCardToken == null) {
            return false;
        }
        return StringUtils.startsWithAny(nonCardToken.toLowerCase(), CARD_TYPES);
    }

    public static boolean isCardRarity(String lineAsIs) {
        return DeckRecognizer.cardRarityTokenMatch(lineAsIs) != null;
    }

    public static boolean isCardCMC(String lineAsIs) {
        return DeckRecognizer.cardCMCTokenMatch(lineAsIs) != null;
    }

    public static boolean isManaToken(String lineAsIs) {
        return DeckRecognizer.manaTokenMatch(lineAsIs) != null;
    }

    public static boolean isDeckSectionName(String lineAsIs) {
        String nonCardToken = DeckRecognizer.nonCardTokenMatch(lineAsIs);
        if (nonCardToken == null) {
            return false;
        }
        return StringUtils.equalsAnyIgnoreCase(nonCardToken, DECK_SECTION_NAMES);
    }

    private static String nonCardTokenMatch(String lineAsIs) {
        if (lineAsIs == null) {
            return null;
        }
        String line = lineAsIs.trim();
        Matcher noncardMatcher = NONCARD_PATTERN.matcher(line);
        if (!noncardMatcher.matches()) {
            return null;
        }
        return noncardMatcher.group(REGRP_TOKEN);
    }

    private static String cardRarityTokenMatch(String lineAsIs) {
        if (lineAsIs == null) {
            return null;
        }
        String line = lineAsIs.trim();
        Matcher cardRarityMatcher = CARD_RARITY_PATTERN.matcher(line);
        if (!cardRarityMatcher.matches()) {
            return null;
        }
        return cardRarityMatcher.group(REGRP_TOKEN);
    }

    private static String cardCMCTokenMatch(String lineAsIs) {
        if (lineAsIs == null) {
            return null;
        }
        String line = lineAsIs.trim();
        Matcher cardCMCmatcher = CMC_PATTERN.matcher(line);
        if (!cardCMCmatcher.matches()) {
            return null;
        }
        return cardCMCmatcher.group(REGRP_TOKEN);
    }

    private String getCardCMCMatch(String lineAsIs) {
        String tokenMatch = DeckRecognizer.cardCMCTokenMatch(lineAsIs);
        tokenMatch = (tokenMatch = tokenMatch.toUpperCase()).contains("CC") ? tokenMatch.replaceAll("CC", "").trim() : tokenMatch.replaceAll("CMC", "").trim();
        return String.format("CMC: %s", tokenMatch);
    }

    private static Pair<String, String> manaTokenMatch(String lineAsIs) {
        if (lineAsIs == null) {
            return null;
        }
        String line = lineAsIs.trim();
        Matcher manaMatcher = MANA_PATTERN.matcher(line);
        if (!manaMatcher.matches()) {
            return null;
        }
        String firstMana = manaMatcher.group(REGRP_COLR1);
        String secondMana = manaMatcher.group(REGRP_COLR2);
        firstMana = DeckRecognizer.matchAnyManaSymbolIn(firstMana);
        secondMana = DeckRecognizer.matchAnyManaSymbolIn(secondMana);
        return Pair.of(firstMana, secondMana);
    }

    private static String matchAnyManaSymbolIn(String manaToken) {
        if (manaToken == null) {
            return null;
        }
        Matcher matchManaSymbol = MANA_SYMBOL_PATTERN.matcher(manaToken);
        if (matchManaSymbol.matches()) {
            return matchManaSymbol.group(REGRP_MANA);
        }
        return manaToken;
    }

    private static String getManaTokenMatch(String lineAsIs) {
        Pair<String, String> matchedMana = DeckRecognizer.manaTokenMatch(lineAsIs);
        String color1name = matchedMana.getLeft();
        String color2name = matchedMana.getRight();
        if (color1name.length() == 2) {
            MagicColor.Color magicColor = DeckRecognizer.getMagicColor(color1name.substring(0, 1));
            MagicColor.Color magicColor2 = DeckRecognizer.getMagicColor(color1name.substring(1));
            return DeckRecognizer.getMagicColourLabel(magicColor, magicColor2);
        }
        MagicColor.Color magicColor = DeckRecognizer.getMagicColor(color1name);
        if (color2name == null) {
            return DeckRecognizer.getMagicColourLabel(magicColor);
        }
        MagicColor.Color magicColor2 = DeckRecognizer.getMagicColor(color2name);
        if (magicColor2 == magicColor) {
            return DeckRecognizer.getMagicColourLabel(magicColor);
        }
        return DeckRecognizer.getMagicColourLabel(magicColor, magicColor2);
    }

    private static String getMagicColourLabel(MagicColor.Color magicColor) {
        if (magicColor == null) {
            return String.format("%s {W}{U}{B}{R}{G}", DeckRecognizer.getLocalisedMagicColorName("Multicolour"));
        }
        return String.format("%s %s", DeckRecognizer.getLocalisedMagicColorName(magicColor.getName()), magicColor.getSymbol());
    }

    private static String getMagicColourLabel(MagicColor.Color magicColor1, MagicColor.Color magicColor2) {
        if (magicColor2 == null || magicColor2 == MagicColor.Color.COLORLESS || magicColor1 == MagicColor.Color.COLORLESS) {
            return String.format("%s // %s", DeckRecognizer.getMagicColourLabel(magicColor1), DeckRecognizer.getMagicColourLabel(magicColor2));
        }
        String localisedName1 = DeckRecognizer.getLocalisedMagicColorName(magicColor1.getName());
        String localisedName2 = DeckRecognizer.getLocalisedMagicColorName(magicColor2.getName());
        String comboManaSymbol = manaSymbolsMap.get(magicColor1.getColormask() | magicColor2.getColormask());
        return String.format("%s/%s {%s}", localisedName1, localisedName2, comboManaSymbol);
    }

    private static MagicColor.Color getMagicColor(String colorName) {
        if (colorName.toLowerCase().startsWith("multi") || colorName.equalsIgnoreCase("m")) {
            return null;
        }
        byte color = MagicColor.fromName(colorName.toLowerCase());
        switch (color) {
            case 1: {
                return MagicColor.Color.WHITE;
            }
            case 2: {
                return MagicColor.Color.BLUE;
            }
            case 4: {
                return MagicColor.Color.BLACK;
            }
            case 8: {
                return MagicColor.Color.RED;
            }
            case 16: {
                return MagicColor.Color.GREEN;
            }
        }
        return MagicColor.Color.COLORLESS;
    }

    public static String getLocalisedMagicColorName(String colorName) {
        Localizer localizer = Localizer.getInstance();
        switch (colorName.toLowerCase()) {
            case "white": {
                return localizer.getMessage("lblWhite", new Object[0]);
            }
            case "blue": {
                return localizer.getMessage("lblBlue", new Object[0]);
            }
            case "black": {
                return localizer.getMessage("lblBlack", new Object[0]);
            }
            case "red": {
                return localizer.getMessage("lblRed", new Object[0]);
            }
            case "green": {
                return localizer.getMessage("lblGreen", new Object[0]);
            }
            case "colorless": {
                return localizer.getMessage("lblColorless", new Object[0]);
            }
            case "multicolour": 
            case "multicolor": {
                return localizer.getMessage("lblMulticolor", new Object[0]);
            }
        }
        return "";
    }

    public static String getColorNameByLocalisedName(String localisedName) {
        Localizer localizer = Localizer.getInstance();
        if (localisedName.equals(localizer.getMessage("lblWhite", new Object[0]))) {
            return "white";
        }
        if (localisedName.equals(localizer.getMessage("lblBlue", new Object[0]))) {
            return "blue";
        }
        if (localisedName.equals(localizer.getMessage("lblBlack", new Object[0]))) {
            return "black";
        }
        if (localisedName.equals(localizer.getMessage("lblRed", new Object[0]))) {
            return "red";
        }
        if (localisedName.equals(localizer.getMessage("lblGreen", new Object[0]))) {
            return "green";
        }
        return "";
    }

    private static Pair<String, String> getManaNameAndSymbol(String matchedMana) {
        if (matchedMana == null) {
            return null;
        }
        Localizer localizer = Localizer.getInstance();
        switch (matchedMana.toLowerCase()) {
            case "white": 
            case "w": {
                return Pair.of(localizer.getMessage("lblWhite", new Object[0]), MagicColor.Color.WHITE.getSymbol());
            }
            case "blue": 
            case "u": {
                return Pair.of(localizer.getMessage("lblBlue", new Object[0]), MagicColor.Color.BLUE.getSymbol());
            }
            case "black": 
            case "b": {
                return Pair.of(localizer.getMessage("lblBlack", new Object[0]), MagicColor.Color.BLACK.getSymbol());
            }
            case "red": 
            case "r": {
                return Pair.of(localizer.getMessage("lblRed", new Object[0]), MagicColor.Color.RED.getSymbol());
            }
            case "green": 
            case "g": {
                return Pair.of(localizer.getMessage("lblGreen", new Object[0]), MagicColor.Color.GREEN.getSymbol());
            }
            case "colorless": 
            case "c": {
                return Pair.of(localizer.getMessage("lblColorless", new Object[0]), MagicColor.Color.COLORLESS.getSymbol());
            }
        }
        return Pair.of(localizer.getMessage("lblMulticolor", new Object[0]), "");
    }

    public static boolean isDeckName(String lineAsIs) {
        if (lineAsIs == null) {
            return false;
        }
        String line = lineAsIs.trim();
        Matcher deckNameMatcher = DECK_NAME_PATTERN.matcher(line);
        return deckNameMatcher.matches();
    }

    public static String deckNameMatch(String text) {
        if (text == null) {
            return "";
        }
        String line = text.trim();
        Matcher deckNamePattern = DECK_NAME_PATTERN.matcher(line);
        if (deckNamePattern.matches()) {
            return deckNamePattern.group(REGRP_DECKNAME);
        }
        return "";
    }

    public void setDateConstraint(int year, int month) {
        Calendar ca = Calendar.getInstance();
        ca.set(year, month, 1);
        this.releaseDateConstraint = ca.getTime();
    }

    public void setGameFormatConstraint(List<String> allowedSetCodes, List<String> bannedCards, List<String> restrictedCards) {
        this.allowedSetCodes = allowedSetCodes != null && !allowedSetCodes.isEmpty() ? allowedSetCodes : null;
        this.gameFormatBannedCards = bannedCards != null && !bannedCards.isEmpty() ? bannedCards : null;
        this.gameFormatRestrictedCards = restrictedCards != null && !restrictedCards.isEmpty() ? restrictedCards : null;
    }

    public void setDeckFormatConstraint(DeckFormat deckFormat0) {
        this.deckFormat = deckFormat0;
    }

    public void setArtPreference(CardDb.CardArtPreference artPref) {
        this.artPreference = artPref;
    }

    public void setAllowedDeckSections(List<DeckSection> deckSections) {
        this.allowedDeckSections = deckSections;
    }

    public void forceImportBannedAndRestrictedCards() {
        this.includeBannedAndRestricted = true;
    }

    public static class Token {
        private final TokenType type;
        private final int number;
        private final String text;
        private LimitedCardType limitedCardType = null;
        private PaperCard card = null;
        private DeckSection tokenSection = null;
        private boolean cardRequestHasSetCode = true;

        public static Token LegalCard(PaperCard card, int count, DeckSection section, boolean cardRequestHasSetCode) {
            return new Token(TokenType.LEGAL_CARD, count, card, section, cardRequestHasSetCode);
        }

        public static Token LimitedCard(PaperCard card, int count, DeckSection section, LimitedCardType limitedType, boolean cardRequestHasSetCode) {
            return new Token(TokenType.LIMITED_CARD, count, card, section, limitedType, cardRequestHasSetCode);
        }

        public static Token NotAllowedCard(PaperCard card, int count, boolean cardRequestHasSetCode) {
            return new Token(TokenType.CARD_FROM_NOT_ALLOWED_SET, count, card, cardRequestHasSetCode);
        }

        public static Token CardInInvalidSet(PaperCard card, int count, boolean cardRequestHasSetCode) {
            return new Token(TokenType.CARD_FROM_INVALID_SET, count, card, cardRequestHasSetCode);
        }

        public static Token UnknownCard(String cardName, String setCode, int count) {
            String ttext = setCode == null || setCode.isEmpty() ? cardName : String.format("%s [%s]", cardName, setCode);
            return new Token(TokenType.UNKNOWN_CARD, count, ttext);
        }

        public static Token UnsupportedCard(String cardName, String setCode, int count) {
            String ttext = setCode == null || setCode.isEmpty() ? cardName : String.format("%s [%s]", cardName, setCode);
            return new Token(TokenType.UNSUPPORTED_CARD, count, ttext);
        }

        public static Token WarningMessage(String msg) {
            return new Token(TokenType.WARNING_MESSAGE, msg);
        }

        private static Token UnsupportedDeckSection(String sectionName) {
            return new Token(TokenType.UNSUPPORTED_DECK_SECTION, sectionName);
        }

        public static Token DeckSection(String sectionName0, List<DeckSection> allowedDeckSections) {
            String sectionName = sectionName0.toLowerCase().trim();
            DeckSection matchedSection = null;
            if (sectionName.equals("side") || sectionName.contains("sideboard") || sectionName.equals("sb")) {
                matchedSection = DeckSection.Sideboard;
            } else if (sectionName.equals("main") || sectionName.contains("card") || sectionName.equals("mainboard") || sectionName.equals("deck")) {
                matchedSection = DeckSection.Main;
            } else if (sectionName.equals("avatar")) {
                matchedSection = DeckSection.Avatar;
            } else if (sectionName.equals("commander")) {
                matchedSection = DeckSection.Commander;
            } else if (sectionName.equals("schemes")) {
                matchedSection = DeckSection.Schemes;
            } else if (sectionName.equals("conspiracy")) {
                matchedSection = DeckSection.Conspiracy;
            } else if (sectionName.equals("planes")) {
                matchedSection = DeckSection.Planes;
            } else if (sectionName.equals("attractions")) {
                matchedSection = DeckSection.Attractions;
            }
            if (matchedSection == null) {
                return null;
            }
            if (allowedDeckSections != null && !allowedDeckSections.contains((Object)matchedSection)) {
                return Token.UnsupportedDeckSection(sectionName0);
            }
            return new Token(TokenType.DECK_SECTION_NAME, matchedSection.name());
        }

        private Token(TokenType type1, int count, PaperCard tokenCard, boolean cardRequestHasSetCode) {
            this.number = count;
            this.type = type1;
            this.text = "";
            this.card = tokenCard;
            this.tokenSection = null;
            this.limitedCardType = null;
            this.cardRequestHasSetCode = cardRequestHasSetCode;
        }

        private Token(TokenType type1, int count, PaperCard tokenCard, DeckSection section, boolean cardRequestHasSetCode) {
            this(type1, count, tokenCard, cardRequestHasSetCode);
            this.tokenSection = section;
            this.limitedCardType = null;
        }

        private Token(TokenType type1, int count, PaperCard tokenCard, DeckSection section, LimitedCardType limitedCardType1, boolean cardRequestHasSetCode) {
            this(type1, count, tokenCard, cardRequestHasSetCode);
            this.tokenSection = section;
            this.limitedCardType = limitedCardType1;
        }

        public Token(TokenType type1, int count, String message) {
            this.number = count;
            this.type = type1;
            this.text = message;
        }

        public Token(TokenType type1, String message) {
            this(type1, 0, message);
        }

        public final String getText() {
            if (this.isCardToken()) {
                return String.format("%s [%s] #%s", this.card.getName(), this.card.getEdition(), this.card.getCollectorNumber());
            }
            return this.text;
        }

        public final PaperCard getCard() {
            return this.card;
        }

        public final TokenType getType() {
            return this.type;
        }

        public final int getQuantity() {
            return this.number;
        }

        public final boolean cardRequestHasNoCode() {
            return !this.cardRequestHasSetCode;
        }

        public final DeckSection getTokenSection() {
            return this.tokenSection;
        }

        public void resetTokenSection(DeckSection referenceDeckSection) {
            this.tokenSection = referenceDeckSection != null ? referenceDeckSection : DeckSection.Main;
        }

        public void replaceTokenCard(PaperCard replacementCard) {
            if (!this.isCardToken()) {
                return;
            }
            this.card = replacementCard;
        }

        public final LimitedCardType getLimitedCardType() {
            return this.limitedCardType;
        }

        public boolean isCardToken() {
            return this.type == TokenType.LEGAL_CARD || this.type == TokenType.LIMITED_CARD || this.type == TokenType.CARD_FROM_NOT_ALLOWED_SET || this.type == TokenType.CARD_FROM_INVALID_SET;
        }

        public boolean isTokenForDeck() {
            return this.type == TokenType.LEGAL_CARD || this.type == TokenType.LIMITED_CARD || this.type == TokenType.DECK_NAME;
        }

        public boolean isCardTokenForDeck() {
            return this.type == TokenType.LEGAL_CARD || this.type == TokenType.LIMITED_CARD;
        }

        public boolean isCardPlaceholder() {
            return this.type == TokenType.CARD_RARITY || this.type == TokenType.CARD_CMC || this.type == TokenType.MANA_COLOUR || this.type == TokenType.CARD_TYPE;
        }

        public boolean isDeckSection() {
            return this.type == TokenType.DECK_SECTION_NAME;
        }

        public TokenKey getKey() {
            return TokenKey.fromToken(this);
        }

        public static class TokenKey {
            private static final String KEYSEP = "|";
            public String cardName;
            public String setCode;
            public String collectorNumber;
            public DeckSection deckSection;
            public TokenType tokenType;
            public LimitedCardType limitedType;

            public static TokenKey fromToken(Token token) {
                if (!token.isCardToken()) {
                    return null;
                }
                TokenKey key = new TokenKey();
                key.cardName = CardDb.CardRequest.compose(token.card.getName(), token.getCard().isFoil());
                key.setCode = token.card.getEdition();
                key.collectorNumber = token.card.getCollectorNumber();
                key.tokenType = token.getType();
                if (token.tokenSection != null) {
                    key.deckSection = token.tokenSection;
                }
                if (token.limitedCardType != null) {
                    key.limitedType = token.limitedCardType;
                }
                return key;
            }

            public String toString() {
                StringBuilder keyString = new StringBuilder();
                keyString.append(String.format("%s%s%s%s%s", this.cardName, KEYSEP, this.setCode, KEYSEP, this.collectorNumber));
                if (this.deckSection != null) {
                    keyString.append(String.format("%sD%s", KEYSEP, this.deckSection.name()));
                }
                keyString.append(String.format("%sT%s", KEYSEP, this.tokenType.name()));
                if (this.limitedType != null) {
                    keyString.append(String.format("%sL%s", KEYSEP, this.limitedType.name()));
                }
                return keyString.toString();
            }

            public static TokenKey fromString(String keyString) {
                TokenType tokenType;
                String[] keyInfo = StringUtils.split(keyString, KEYSEP);
                if (keyInfo.length < 4) {
                    return null;
                }
                TokenKey tokenKey = new TokenKey();
                tokenKey.cardName = keyInfo[0];
                tokenKey.setCode = keyInfo[1];
                tokenKey.collectorNumber = keyInfo[2];
                int nxtInfoIdx = 3;
                if (keyInfo[nxtInfoIdx].startsWith("D")) {
                    tokenKey.deckSection = DeckSection.valueOf(keyInfo[nxtInfoIdx].substring(1));
                    ++nxtInfoIdx;
                }
                tokenKey.tokenType = tokenType = TokenType.valueOf(keyInfo[nxtInfoIdx].substring(1));
                if (tokenType == TokenType.LIMITED_CARD) {
                    tokenKey.limitedType = LimitedCardType.valueOf(keyInfo[nxtInfoIdx + 1].substring(1));
                }
                return tokenKey;
            }
        }
    }

    public static enum LimitedCardType {
        BANNED,
        RESTRICTED;

    }

    public static enum TokenType {
        LEGAL_CARD,
        LIMITED_CARD,
        CARD_FROM_NOT_ALLOWED_SET,
        CARD_FROM_INVALID_SET,
        WARNING_MESSAGE,
        UNKNOWN_CARD,
        UNSUPPORTED_CARD,
        UNSUPPORTED_DECK_SECTION,
        UNKNOWN_TEXT,
        COMMENT,
        DECK_NAME,
        DECK_SECTION_NAME,
        CARD_TYPE,
        CARD_RARITY,
        CARD_CMC,
        MANA_COLOUR;

    }
}

