"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.tokens = exports.regExpToken = void 0;

var _postcssValueParser = require("postcss-value-parser");

var _cssColorKeywords = _interopRequireDefault(require("css-color-keywords"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

var matchString = function matchString(node) {
  if (node.type !== "string") return null;
  return node.value.replace(/\\([0-9a-f]{1,6})(?:\s|$)/gi, function (match, charCode) {
    return String.fromCharCode(parseInt(charCode, 16));
  }).replace(/\\/g, "");
};

var hexColorRe = /^(#(?:[0-9a-f]{3,4}){1,2})$/i;
var cssFunctionNameRe = /^(rgba?|hsla?|hwb|lab|lch|gray|color)$/;

var matchColor = function matchColor(node) {
  if (node.type === "word" && (hexColorRe.test(node.value) || node.value in _cssColorKeywords["default"] || node.value === "transparent")) {
    return node.value;
  } else if (node.type === "function" && cssFunctionNameRe.test(node.value)) {
    return (0, _postcssValueParser.stringify)(node);
  }

  return null;
};

var noneRe = /^(none)$/i;
var autoRe = /^(auto)$/i;
var identRe = /(^-?[_a-z][_a-z0-9-]*$)/i; // Note if these are wrong, you'll need to change index.js too

var numberRe = /^([+-]?(?:\d*\.)?\d+(?:[Ee][+-]?\d+)?)$/; // Note lengthRe is sneaky: you can omit units for 0

var lengthRe = /^(0$|(?:[+-]?(?:\d*\.)?\d+(?:[Ee][+-]?\d+)?)((?=px$)|(?=Px$)|(?=PX$)|(?=pX$)))/;
var unsupportedUnitRe = /^([+-]?(?:\d*\.)?\d+(?:[Ee][+-]?\d+)?(ch|em|ex|rem|vh|vw|vmin|vmax|cm|mm|in|pc|pt))$/;
var angleRe = /^([+-]?(?:\d*\.)?\d+(?:[Ee][+-]?\d+)?(?:deg|rad))$/;
var percentRe = /^([+-]?(?:\d*\.)?\d+(?:[Ee][+-]?\d+)?%)$/;

var noopToken = function noopToken(predicate) {
  return function (node) {
    return predicate(node) ? "<token>" : null;
  };
};

var valueForTypeToken = function valueForTypeToken(type) {
  return function (node) {
    return node.type === type ? node.value : null;
  };
};

var regExpToken = function regExpToken(regExp) {
  var transform = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : String;
  return function (node) {
    if (node.type !== "word") return null;
    var match = node.value.match(regExp);
    if (match === null) return null;
    var value = transform(match[1]);

    if (/px/.test(node.value)) {
      return "scalePx2dp(".concat(value, ")");
    } else {
      return value;
    }
  };
};

exports.regExpToken = regExpToken;
var tokens = {
  SPACE: noopToken(function (node) {
    return node.type === "space";
  }),
  SLASH: noopToken(function (node) {
    return node.type === "div" && node.value === "/";
  }),
  COMMA: noopToken(function (node) {
    return node.type === "div" && node.value === ",";
  }),
  WORD: valueForTypeToken("word"),
  NONE: regExpToken(noneRe),
  AUTO: regExpToken(autoRe),
  NUMBER: regExpToken(numberRe, Number),
  LENGTH: regExpToken(lengthRe, Number),
  UNSUPPORTED_LENGTH_UNIT: regExpToken(unsupportedUnitRe),
  ANGLE: regExpToken(angleRe),
  PERCENT: regExpToken(percentRe),
  IDENT: regExpToken(identRe),
  STRING: matchString,
  COLOR: matchColor,
  LINE: regExpToken(/^(none|underline|line-through)$/i)
};
exports.tokens = tokens;