"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseScript = exports.replaceMemberExpression = exports.replaceIdentifier = void 0;
const t = require("babel-types");
const babel_traverse_1 = require("babel-traverse");
const utils_1 = require("./utils");
const global_1 = require("./global");
const defaultClassName = '_C';
const buildDecorator = (id) => t.decorator(t.callExpression(t.identifier('withWeapp'), [id]));
function replaceIdentifier(callee) {
    if (callee.isIdentifier()) {
        const name = callee.node.name;
        if (name === 'getApp' || name === 'getCurrentPages') {
            callee.replaceWith(t.memberExpression(t.identifier('Taro'), callee.node));
        }
    }
}
exports.replaceIdentifier = replaceIdentifier;
function replaceMemberExpression(callee) {
    if (callee.isMemberExpression()) {
        const object = callee.get('object');
        if (object.isIdentifier({ name: 'wx' })) {
            object.replaceWith(t.identifier('Taro'));
        }
    }
}
exports.replaceMemberExpression = replaceMemberExpression;
function parseScript(script, returned, wxses = [], refId) {
    script = script || 'Page({})';
    if (t.isJSXText(returned)) {
        const block = utils_1.buildBlockElement();
        block.children = [returned];
        returned = block;
    }
    let ast = utils_1.parseCode(script);
    let classDecl;
    let foundWXInstance = false;
    const vistor = {
        BlockStatement(path) {
            path.scope.rename('wx', 'Taro');
        },
        Identifier(path) {
            if (path.isReferenced() && path.node.name === 'wx') {
                path.replaceWith(t.identifier('Taro'));
            }
        },
        CallExpression(path) {
            const callee = path.get('callee');
            replaceIdentifier(callee);
            replaceMemberExpression(callee);
            if (callee.isIdentifier({ name: 'Page' }) ||
                callee.isIdentifier({ name: 'Component' }) ||
                callee.isIdentifier({ name: 'App' })) {
                foundWXInstance = true;
                const componentType = callee.node.name;
                classDecl = parsePage(path, returned || t.nullLiteral(), componentType, refId, wxses);
                ast.program.body.push(classDecl, t.exportDefaultDeclaration(t.identifier(componentType !== 'App' ? defaultClassName : 'App')));
                // path.insertAfter(t.exportDefaultDeclaration(t.identifier(defaultClassName)))
                path.remove();
            }
        }
    };
    babel_traverse_1.default(ast, vistor);
    if (!foundWXInstance) {
        ast = utils_1.parseCode(script + ';Component({})');
        babel_traverse_1.default(ast, vistor);
    }
    const taroComponentsImport = utils_1.buildImportStatement('@tarojs/components', [
        ...global_1.usedComponents
    ]);
    const taroImport = utils_1.buildImportStatement('@tarojs/taro', [], 'Taro');
    const reactImport = utils_1.buildImportStatement('react', [], 'React');
    const withWeappImport = utils_1.buildImportStatement('@tarojs/with-weapp', [], 'withWeapp');
    ast.program.body.unshift(taroComponentsImport, reactImport, taroImport, withWeappImport, ...wxses.filter(wxs => !wxs.src.startsWith('./wxs__')).map(wxs => utils_1.buildImportStatement(wxs.src, [], wxs.module)));
    return ast;
}
exports.parseScript = parseScript;
function parsePage(pagePath, returned, componentType, refId, wxses) {
    const stateKeys = [];
    pagePath.traverse({
        CallExpression(path) {
            const callee = path.get('callee');
            replaceIdentifier(callee);
            replaceMemberExpression(callee);
        }
    });
    if (refId) {
        refId.forEach(id => {
            if (!stateKeys.includes(id)) {
                stateKeys.push(id);
            }
        });
    }
    const propsKeys = [];
    const arg = pagePath.get('arguments')[0];
    const classBody = [];
    if (arg.isObjectExpression() || arg.isIdentifier()) {
        //
    }
    else {
        throw utils_1.codeFrameError(arg.node, `${componentType || '组件'} 的第一个参数必须是一个对象或变量才能转换。`);
    }
    const wxsNames = new Set(wxses ? wxses.map(w => w.module) : []);
    const renderFunc = utils_1.buildRender(componentType === 'App'
        ? t.memberExpression(t.memberExpression(t.thisExpression(), t.identifier('props')), t.identifier('children'))
        : returned, stateKeys.filter(s => !wxsNames.has(s)), propsKeys);
    const classDecl = t.classDeclaration(t.identifier(componentType === 'App' ? 'App' : defaultClassName), t.memberExpression(t.identifier('React'), t.identifier('Component')), t.classBody(classBody.concat(renderFunc)), []);
    classDecl.decorators = [buildDecorator(arg.node)];
    return classDecl;
}
//# sourceMappingURL=script.js.map