"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseModule = exports.parseTemplate = exports.generateVueFile = exports.parseWXML = exports.parseVue = void 0;
/* eslint-disable camelcase */
const himalaya_wxml_1 = require("himalaya-wxml");
const t = require("babel-types");
const babel_traverse_1 = require("babel-traverse");
const wxml_1 = require("./wxml");
const template_1 = require("./template");
const fs = require("fs");
const path_1 = require("path");
const utils_1 = require("./utils");
const script_1 = require("./script");
const lodash_1 = require("lodash");
const { prettyPrint } = require('html');
function parseVue(dirPath, wxml, jsCode = '') {
    let ast = utils_1.parseCode(jsCode);
    let foundWXInstance = false;
    const vistor = {
        BlockStatement(path) {
            path.scope.rename('wx', 'Taro');
        },
        Identifier(path) {
            if (path.isReferenced() && path.node.name === 'wx') {
                path.replaceWith(t.identifier('Taro'));
            }
        },
        CallExpression(path) {
            const callee = path.get('callee');
            script_1.replaceIdentifier(callee);
            script_1.replaceMemberExpression(callee);
            if (callee.isIdentifier({ name: 'Page' }) ||
                callee.isIdentifier({ name: 'Component' }) ||
                callee.isIdentifier({ name: 'App' })) {
                foundWXInstance = true;
                const componentType = callee.node.name;
                ast.program.body.push(t.exportDefaultDeclaration(t.callExpression(t.identifier('withWeapp'), [path.node.arguments[0], t.stringLiteral(componentType)])));
                // path.insertAfter(t.exportDefaultDeclaration(t.identifier(defaultClassName)))
                path.remove();
            }
        }
    };
    babel_traverse_1.default(ast, vistor);
    if (!foundWXInstance) {
        ast = utils_1.parseCode(jsCode + ';Component({})');
        babel_traverse_1.default(ast, vistor);
    }
    const taroImport = utils_1.buildImportStatement('@tarojs/taro', [], 'Taro');
    const withWeappImport = utils_1.buildImportStatement('@tarojs/with-weapp', [], 'withWeapp');
    ast.program.body.unshift(taroImport, withWeappImport
    // ...wxses.filter(wxs => !wxs.src.startsWith('./wxs__')).map(wxs => buildImportStatement(wxs.src, [], wxs.module))
    );
    const { imports, template } = parseWXML(dirPath, wxml, []);
    return {
        ast,
        imports,
        template
    };
}
exports.parseVue = parseVue;
function parseWXML(dirPath, wxml, imports) {
    // const parseResult = getCacheWxml(dirPath)
    // if (parseResult) {
    //   return parseResult
    // }
    try {
        wxml = prettyPrint(wxml, {
            max_char: 0,
            indent_char: 0,
            unformatted: ['text', 'wxs']
        });
    }
    catch (error) {
        //
    }
    const nodes = himalaya_wxml_1.parse(wxml.trim()).map(node => parseNode(node, dirPath, imports));
    const template = generateVueFile(nodes);
    return {
        nodes,
        template,
        imports
    };
}
exports.parseWXML = parseWXML;
function parseElement(element, dirPath, imports) {
    let forItem = 'item';
    let forIndex = 'index';
    switch (element.tagName) {
        case 'template':
            parseTemplate(element, imports);
            break;
        case 'wxs':
            parseWXS(element, imports);
            break;
        case 'import':
        case 'include':
            parseModule(element, dirPath, imports);
            return {
                type: wxml_1.NodeType.Text,
                content: ''
            };
        default:
            break;
    }
    return {
        tagName: element.tagName,
        type: element.type,
        children: element.children.map(child => parseNode(child, dirPath, imports)),
        attributes: element.attributes
            .filter(a => {
            let match = true;
            if (a.key === wxml_1.WX_FOR_ITEM) {
                match = false;
                // 这里用 `||` 不用 `??` 是因为用户可能会填一个空字符串
                forItem = a.value || forItem;
            }
            else if (a.key === wxml_1.WX_FOR_INDEX) {
                match = false;
                forIndex = a.value || forIndex;
            }
            return match;
        }).map(a => {
            return parseAttribute(a, forItem, forIndex);
        })
    };
}
function parseNode(node, dirPath, imports) {
    if (node.type === wxml_1.NodeType.Text) {
        return node;
    }
    else if (node.type === wxml_1.NodeType.Comment) {
        return node;
    }
    return parseElement(node, dirPath, imports);
}
const VUE_IF = 'v-if';
const VUE_ELSE_IF = 'v-else-if';
const VUE_FOR = 'v-for';
const VUE_ELSE = 'v-else';
const VUE_KEY = 'key';
const WX_FOR_2 = 'wx:for-items';
function parseAttribute(attr, forItem, forIndex) {
    let { key, value } = attr;
    let isVueDirectives = true;
    const isBind = key.startsWith('bind');
    const isCatch = key.startsWith('catch');
    const isEvent = isBind || isCatch;
    switch (key) {
        case wxml_1.WX_IF:
            key = VUE_IF;
            break;
        case wxml_1.WX_ELSE_IF:
            key = VUE_ELSE_IF;
            break;
        case wxml_1.WX_ELSE:
            key = VUE_ELSE;
            break;
        case wxml_1.WX_FOR:
        case WX_FOR_2:
            key = VUE_FOR;
            break;
        case wxml_1.WX_KEY:
            key = VUE_KEY;
            value = value || forIndex;
            isVueDirectives = false;
            break;
        default:
            isVueDirectives = false;
            break;
    }
    const { type, content } = wxml_1.parseContent(value !== null && value !== void 0 ? value : '', true);
    if (type === 'expression') {
        if ((content === null || content === void 0 ? void 0 : content.startsWith('(')) && content.endsWith(')')) {
            value = content.slice(1, content.length - 1);
        }
        else {
            value = content;
        }
        if (key === VUE_FOR) {
            value = `(${forItem}, ${forIndex}) in ${value}`;
        }
        if (key === 'data') {
            if (value.includes(':') || (value.includes('...') && value.includes(','))) {
                value = `{ ${value} }`;
            }
        }
        if (!isVueDirectives && !isEvent) {
            key = ':' + key;
        }
    }
    if (key === VUE_KEY) {
        // 微信小程序真的太多没在文档里的用法了
        if ((value === null || value === void 0 ? void 0 : value.includes('this')) || (value === null || value === void 0 ? void 0 : value.includes('*item'))) {
            value = forItem;
        }
        if (value !== forItem || value !== forIndex) {
            value = `${forItem}.${value} || ${value} || ${forIndex}`;
        }
        key = ':' + key;
    }
    if (value === null && key !== VUE_ELSE) {
        value = 'true';
    }
    if (isBind) {
        key = key.replace(/^bind/g, '@');
    }
    if (isCatch) {
        key = key.replace(/^catch/g, '@') + '.stop';
    }
    if (isEvent && value === 'true') {
        value = 'emptyHandler';
    }
    if (key.startsWith(':')) {
        key = ':' + lodash_1.kebabCase(key.slice(1));
    }
    else {
        key = lodash_1.kebabCase(key);
    }
    return {
        key,
        value
    };
}
function createElement(tagName) {
    return {
        tagName,
        type: wxml_1.NodeType.Element,
        children: [],
        attributes: []
    };
}
function generateVueFile(children) {
    const template = createElement('template');
    const container = createElement('block');
    container.children = children;
    template.children = [container];
    return himalaya_wxml_1.stringify([template]);
}
exports.generateVueFile = generateVueFile;
function parseTemplate(element, imports) {
    const { attributes, children } = element;
    const is = attributes.find(a => a.key === 'is');
    const data = attributes.find(a => a.key === 'data');
    const name = attributes.find(a => a.key === 'name');
    if (name) {
        const value = name.value;
        const { type } = wxml_1.parseContent(value !== null && value !== void 0 ? value : '');
        if (type === 'expression') {
            console.warn('template 的属性 name 只能是一个字符串，考虑更改以下源码逻辑：\n', himalaya_wxml_1.stringify(element));
        }
        const componentName = template_1.buildTemplateName(name.key);
        const component = parseWXML('', himalaya_wxml_1.stringify(children), imports);
        imports.push({
            name: componentName,
            template: component.template
        });
    }
    else if (is) {
        const value = is.value;
        if (!value) {
            console.warn('template 的 `is` 属性不能为空', himalaya_wxml_1.stringify(element));
        }
        const { type } = wxml_1.parseContent(value !== null && value !== void 0 ? value : '');
        if (type === 'expression') {
            console.warn('template 的属性 is 只能是一个字符串，考虑更改以下源码逻辑：\n', himalaya_wxml_1.stringify(element));
        }
        element.tagName = template_1.buildTemplateName(value, false);
        element.attributes = [];
        if (data) {
            element.attributes.push({
                key: 'data',
                value: data.value
            });
        }
    }
    else {
        throw new Error('template 标签必须指名 `is` 或 `name` 任意一个标签:\n' + himalaya_wxml_1.stringify(element));
    }
    return element;
}
exports.parseTemplate = parseTemplate;
function parseModule(element, dirPath, imports) {
    var _a, _b;
    const { attributes, tagName } = element;
    const src = attributes.find(a => a.key === 'src');
    if (!src) {
        throw new Error(`${tagName} 标签必须包含 \`src\` 属性` + '\n' + himalaya_wxml_1.stringify(element));
    }
    let srcValue = (_a = src.value) !== null && _a !== void 0 ? _a : '';
    const { type } = wxml_1.parseContent(srcValue);
    if (type === 'expression') {
        console.warn(tagName + '的属性 src 只能是一个字符串，考虑更改以下源码逻辑：\n', himalaya_wxml_1.stringify(element));
    }
    if (srcValue.startsWith('/')) {
        const vpath = path_1.resolve(utils_1.setting.rootPath, srcValue.substr(1));
        if (!fs.existsSync(vpath)) {
            throw new Error(`import/include 的 src 请填入相对路径再进行转换：src="${srcValue}"`);
        }
        let relativePath = path_1.relative(dirPath, vpath);
        relativePath = relativePath.replace(/\\/g, '/');
        if (relativePath.indexOf('.') !== 0) {
            srcValue = './' + relativePath;
        }
        srcValue = relativePath;
    }
    if (tagName === 'import') {
        const wxml = template_1.getWXMLsource(dirPath, srcValue, tagName);
        const mods = (_b = parseWXML(path_1.resolve(dirPath, srcValue), wxml, imports || [])) === null || _b === void 0 ? void 0 : _b.imports;
        imports.push(...(mods || []));
    }
    else {
        console.warn(`暂不支持 ${tagName} 标签的转换`, '考虑修改源码使用 import 替代\n' + himalaya_wxml_1.stringify(element));
    }
}
exports.parseModule = parseModule;
function parseWXS(element, imports) {
    let moduleName = null;
    let src = null;
    const { attributes } = element;
    for (const attr of attributes) {
        const { key, value } = attr;
        if (key === 'module') {
            moduleName = value;
        }
        if (key === 'src') {
            src = value;
        }
    }
    if (!src) {
        const script = element.children.reduce((acc, node) => {
            if (node.type === wxml_1.NodeType.Text) {
                return acc + node.content;
            }
            return acc;
        }, '');
        src = './wxs__' + moduleName;
        const ast = utils_1.parseCode(script);
        babel_traverse_1.default(ast, {
            CallExpression(path) {
                if (t.isIdentifier(path.node.callee, { name: 'getRegExp' })) {
                    console.warn(utils_1.codeFrameError(path.node, '请使用 JavaScript 标准正则表达式把这个 getRegExp 函数重构。'));
                }
            }
        });
        imports.push({
            ast,
            name: moduleName,
            wxs: true
        });
    }
    if (!moduleName || !src) {
        throw new Error('一个 WXS 需要同时存在两个属性：`wxs`, `src`');
    }
}
//# sourceMappingURL=vue.js.map