#include <stdio.h>
#include <U8g2lib.h>

// https://github.com/sarafong/T-Rex-Runner
// Upgraded u8g2 version of the classic Google Chrome T-Rex Runner game.

// You may reference Drivers/drv_gpio.c for pinout
// In u8x8.h #define U8X8_USE_PINS 
#define U8G2_PIN_UP                          3      // PA3
#define U8G2_PIN_DOWN                        4      // PA4
#define U8G2_PIN_LEFT                        6      // PA6
#define U8G2_PIN_RIGHT                      24      // PB8
#define U8G2_PIN_SELECT                     21      // PB5
#define U8G2_PIN_HOME                       17      // PB1

#define OLED_SPI_PIN_CLK                     5      // PA5
#define OLED_SPI_PIN_MOSI                    7      // PA7
#define OLED_SPI_PIN_RES                     2      // PA2
#define OLED_SPI_PIN_DC                      1      // PA1
#define OLED_SPI_PIN_CS                      0      // PA0

// Check https://github.com/olikraus/u8g2/wiki/u8g2setupcpp for all supported devices
// U8G2_SSD1306_128X64_NONAME_F_HW_I2C u8g2(U8G2_R0, U8X8_PIN_NONE);
static U8G2_SSD1306_128X64_NONAME_F_4W_SW_SPI u8g2(U8G2_R0, 
                                            /* clock=*/ OLED_SPI_PIN_CLK, 
                                            /* data=*/ OLED_SPI_PIN_MOSI, 
                                            /* cs=*/ OLED_SPI_PIN_CS, 
                                            /* dc=*/ OLED_SPI_PIN_DC, 
                                            /* reset=*/ OLED_SPI_PIN_RES);

static int random(int low, int high)
{
  return rand() % (high - low + 1) + low;
}

static unsigned long millis() {
    return ((unsigned long)rt_tick_get() / RT_TICK_PER_SECOND * 1000);
}

//Multiple Dino Bitmaps

//--------------------Bitmaps--------------------//

//----------Dimensions----------//
#define smallcactus_width 8
#define smallcactus_height 17
#define standing_width 21
#define standing_height 20
#define frontLeg_width 21
#define frontLeg_height 20
#define backLeg_width 21
#define backLeg_height 20
#define gameover_width 99
#define gameover_height 35
#define easy_width 32
#define easy_height 15
#define medium_width 30
#define medium_height 20
#define hard_width 32
#define hard_height 15
#define insane_width 30
#define insane_height 20
#define menu_width 32
#define menu_height 15
#define replay_width 32
#define replay_height 15

//----------Bitmap Arrays----------//

static const unsigned char smallcactus_bits[] =  {
  0x18, 0x18, 0x98, 0x98, 0x99, 0x99, 0x99, 0x99, 0x79, 0x19, 0x1e, 0x18,
  0x18, 0x18, 0x18, 0x18, 0x18
};

static const unsigned char standing_bits[] = {
  0x00, 0xf8, 0x0f, 0x00, 0xcc, 0x1f, 0x00, 0xf8, 0x1f, 0x00, 0xfc, 0x1f,
  0x00, 0xf8, 0x1f, 0x00, 0xfc, 0x00, 0x00, 0xf8, 0x07, 0x01, 0x3e, 0x00,
  0x01, 0x7f, 0x00, 0xc3, 0xff, 0x01, 0xef, 0x7f, 0x00, 0xff, 0x7f, 0x00,
  0xfe, 0x3f, 0x00, 0xfc, 0x3f, 0x00, 0xf8, 0x1f, 0x00, 0xf0, 0x07, 0x00,
  0xe0, 0x0e, 0x00, 0x60, 0x04, 0x00, 0x20, 0x08, 0x00, 0x60, 0x08, 0x00
};

static const unsigned char frontLeg_bits[] = {
  0x00, 0xf0, 0x0f, 0x00, 0xdc, 0x1f, 0x00, 0xf8, 0x1f, 0x00, 0xf8, 0x1f,
  0x00, 0xf8, 0x1f, 0x00, 0xfc, 0x00, 0x00, 0xf8, 0x07, 0x01, 0x7e, 0x00,
  0x01, 0x7f, 0x00, 0xc3, 0xff, 0x01, 0xef, 0x7f, 0x01, 0xff, 0x7f, 0x00,
  0xfe, 0x7f, 0x00, 0xfc, 0x3f, 0x00, 0xf8, 0x1f, 0x00, 0xf0, 0x0f, 0x00,
  0xe0, 0x18, 0x00, 0xe0, 0x00, 0x00, 0x20, 0x00, 0x00, 0x60, 0x00, 0x00
};

static const unsigned char backLeg_bits[] = {
  0x00, 0xf8, 0x0f, 0x00, 0xd8, 0x1f, 0x00, 0xfc, 0x1f, 0x00, 0xf8, 0x1f,
  0x00, 0xfc, 0x1f, 0x00, 0xf8, 0x00, 0x00, 0xfc, 0x07, 0x01, 0x3e, 0x00,
  0x01, 0x7f, 0x00, 0xc3, 0xff, 0x01, 0xef, 0x7f, 0x00, 0xff, 0x7f, 0x00,
  0xfe, 0x3f, 0x00, 0xfc, 0x3f, 0x00, 0xf8, 0x1f, 0x00, 0xf0, 0x07, 0x00,
  0x60, 0x0e, 0x00, 0xc0, 0x04, 0x00, 0x00, 0x08, 0x00, 0x00, 0x0c, 0x00
};

static const unsigned char gameover_bits[] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xc0, 0x03, 0x0e, 0xd8, 0xc0, 0x07, 0x80, 0x07, 0x24,
  0xe0, 0x01, 0x0f, 0x00, 0x20, 0x00, 0x09, 0xf8, 0x40, 0x00, 0x80, 0x04,
  0x66, 0x30, 0x80, 0x09, 0x00, 0x20, 0x03, 0x19, 0xe8, 0xc0, 0x03, 0x80,
  0x04, 0x3c, 0xe0, 0x01, 0x0d, 0x00, 0x60, 0x02, 0x1b, 0x88, 0x40, 0x00,
  0xc0, 0x0c, 0x18, 0x20, 0x00, 0x07, 0x00, 0xc0, 0x03, 0x11, 0x98, 0xc0,
  0x07, 0x80, 0x07, 0x08, 0xe0, 0x01, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54,
  0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xfe, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x1e, 0xe6, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x4e, 0xee, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x6e, 0xef, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xce, 0xef, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xee, 0xef, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0xe0, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9e, 0xf4, 0x01,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xff,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe,
  0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xfe, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const unsigned char easy_bits[] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
  0x00, 0x00, 0x00, 0xc0, 0xf0, 0x00, 0x00, 0xf0, 0x10, 0x00, 0x00, 0xfc,
  0x50, 0x6d, 0x09, 0xff, 0x10, 0x2c, 0xc5, 0xff, 0x10, 0x4b, 0x06, 0xff,
  0xf0, 0x7e, 0x02, 0xfc, 0x00, 0x00, 0x02, 0xf0, 0x00, 0x00, 0x00, 0xc0,
  0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const unsigned char medium_bits[] = {
  0x00, 0x00, 0x00, 0x00, 0x22, 0x40, 0x01, 0x00, 0x66, 0x40, 0x00, 0x00,
  0xaa, 0x75, 0xd5, 0x0f, 0xda, 0x4f, 0x61, 0x08, 0x82, 0x48, 0x45, 0x09,
  0x42, 0x77, 0x5d, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0xe0, 0x00, 0x00,
  0x00, 0xe0, 0x00, 0x00, 0x00, 0xf0, 0x01, 0x00, 0x00, 0xf0, 0x01, 0x00,
  0x00, 0xf8, 0x03, 0x00, 0x00, 0xf8, 0x03, 0x00, 0x00, 0xfc, 0x07, 0x00,
  0x00, 0xfc, 0x07, 0x00, 0x00, 0xfe, 0x0f, 0x00
};

static const unsigned char hard_bits[] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x07, 0x00, 0x00, 0x00, 0x1f, 0x40, 0x00, 0x20, 0x7f, 0x40, 0x04, 0x20,
  0xff, 0x41, 0xbb, 0x3b, 0xff, 0x43, 0x62, 0x24, 0xff, 0x41, 0x8c, 0x24,
  0x7f, 0x40, 0xb8, 0x38, 0x1f, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const unsigned char insane_bits[] = {
  0x00, 0xfe, 0x0f, 0x00, 0x00, 0xfc, 0x07, 0x00, 0x00, 0xfc, 0x07, 0x00,
  0x00, 0xf8, 0x03, 0x00, 0x00, 0xf8, 0x03, 0x00, 0x00, 0xf0, 0x01, 0x00,
  0x00, 0xf0, 0x01, 0x00, 0x00, 0xe0, 0x00, 0x00, 0x00, 0xe0, 0x00, 0x00,
  0x00, 0x40, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xc0, 0xdd, 0x9d, 0x03,
  0x50, 0x84, 0xa5, 0x03, 0x50, 0x52, 0x65, 0x00, 0x58, 0xdc, 0xa5, 0x03,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const unsigned char menu_bits[] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x07, 0x00, 0x00, 0x00, 0x1f, 0x10, 0x01, 0x00, 0x7f, 0x30, 0x01, 0x00,
  0xff, 0xb1, 0xdd, 0x25, 0xff, 0x53, 0x5d, 0x26, 0xff, 0x11, 0x44, 0x4a,
  0x7f, 0x10, 0x5d, 0x3a, 0x1f, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const unsigned char replay_bits[] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
  0x00, 0x00, 0x00, 0xc0, 0x07, 0x20, 0x00, 0xf0, 0x09, 0x20, 0x00, 0xfc,
  0xe9, 0xae, 0x4b, 0xff, 0xe7, 0x32, 0xeb, 0xff, 0x15, 0xb2, 0x34, 0xff,
  0xe9, 0xae, 0x13, 0xfc, 0x00, 0x02, 0x10, 0xf0, 0x00, 0x00, 0x00, 0xc0,
  0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};


//----------Constructor from U8GLIB----------//
//U8GLIB_NHD_C12864 u8g(13, 11, 10, 9, 8);  // SPI Com: SCK = 13, MOSI = 11, CS = 10, A0 = 9, RST = 8
//U8G2_ST7567_JLX12864_F_4W_SW_SPI u8g(U8G2_R0, 13, 11, 10, 9, 8);

//----------Initializing Variables----------//
static unsigned long dinoTimeToMove = 0; //Used to space out time(millis) between draws/moves
//static unsigned long dinoTimeToDraw = 0;

static int score;
static int highScore = 0;
static int multiplier; //Score multiplier

//bool ignoreRepeat = false; //Avoid Switch Bouncing
static long compensation; //To compensate for the amount of time the Arduino has been running for after each run (essentially "resets" the time)
static bool gameOver;
static bool menu = true;

static bool down, jump; //Boolean values to check the state of the dinosaur
static int dinoSwitch; //Switch between dino bitmaps to simulate movement
static int dinoY, cactiX1, cactiX2; //Positions of objects
static int velocity; //Speed of entire game
static int difficulty; //How fast the velocity increases


//--------------------Functions--------------------//

//Set/Reset Loop for Playing Again
static void reset() {
  compensation = millis();
  gameOver = false;
  down = false;
  jump = false;
  dinoSwitch = true;
  dinoY = 44;
  cactiX1 = 130;
  cactiX2 = cactiX1 + random(70, 120);
  velocity = 2;
}

//Deteching Joystick Interaction
static void keyPress(rt_uint8_t event) {
  //if (event != 0 )rt_kprintf("Key Event%d\n", event);
  if ( menu && event!=0 /* && ignoreRepeat */) { //Any Input
    //rt_kprintf("Key Pressed %d\n", event);
    //ignoreRepeat = false;
    reset();
    menu = false;

    if ( event == U8X8_MSG_GPIO_MENU_PREV ) difficulty = 10000; //Left
    else if ( event == U8X8_MSG_GPIO_MENU_SELECT ) difficulty = random( 2500, 10000 ); //Click
    else if ( event == U8X8_MSG_GPIO_MENU_DOWN ) difficulty = 2500; //Down
    else if ( event == U8X8_MSG_GPIO_MENU_NEXT ) difficulty = 5000; // Right
    else if ( event == U8X8_MSG_GPIO_MENU_UP )difficulty = 7500; //Up

    multiplier = 15000 / difficulty;
  }
  else if ( gameOver && ( event == U8X8_MSG_GPIO_MENU_PREV || event == U8X8_MSG_GPIO_MENU_NEXT ) ) { //Left -> PlayAgain // Right -> Menu
    //ignoreRepeat = false;
    reset();
    if ( event == U8X8_MSG_GPIO_MENU_NEXT ) menu = true; //Pushing Right
  }
  else if ( dinoY == 44 && event == U8X8_MSG_GPIO_MENU_UP ) { //Corresponds to pushing up on the joystick
    jump = true;
    down = true;
  }
  //else if ( input > 1000 ) ignoreRepeat = true; //Avoid switch bouncing
}

//Collision Detecting and Setting HighScore
static void collision() {
  if ( cactiX1 <= 23 && cactiX1 + smallcactus_width >= 15 && dinoY + 20 >= 50 ) {
    if ( (millis() - compensation)*multiplier / 250 > highScore && !gameOver ) highScore = (millis() - compensation) * multiplier / 250; //Changes highscore if current score is greater
    gameOver = true;
  }
}

//-------Move Functions-------//

static void moveDino() {
  if ( dinoY > 44 ) dinoY = 44; //Resets dinosaur to it passes it
  if ( jump || dinoY <= 43 ) { //Allows jumping if on the ground
    jump = false;
    if ( dinoY >= 16 && down ) dinoY -= velocity; //Going up
    else { //Going down
      down = false;
      dinoY += velocity;
    }
  }
}

static void moveCactus() {
  cactiX1 -= velocity;
  cactiX2 -= velocity;
  if ( cactiX1 <= -15 ) { //Once cacti reaches left side of screen, resets to right side of screen
    cactiX1 = cactiX2;
    cactiX2 = cactiX1 + random(70, 150);
  }
}

static void moveObjects() {
  if ( millis() > dinoTimeToMove + 50 ) { //Updates every 50 milliseconds
    velocity = (millis() - compensation) / difficulty + 2; //Increases velocity as game progresses
    moveDino();
    moveCactus();
    dinoTimeToMove = millis();
  }
}

static void drawDinoCactus( ) {
  if ( dinoY < 44 ) u8g2.drawXBM(10, dinoY, standing_width, standing_height, standing_bits); //While jumping don't change bitmap
  else if ( (millis() - compensation) % 16 > 9 ) u8g2.drawXBM(10, dinoY, frontLeg_width, frontLeg_height, frontLeg_bits); //Switch between bitmaps every 9 millis
  else u8g2.drawXBM(10, dinoY, backLeg_width, backLeg_height, backLeg_bits);
  u8g2.drawXBM(cactiX1, 47, smallcactus_width, smallcactus_height, smallcactus_bits); //Draw cacti
  u8g2.drawXBM(cactiX2, 47, smallcactus_width, smallcactus_height, smallcactus_bits);
}

static void drawScore() {
  char scoreBuff[50];
  if ( menu ) { //Print highscore on menu screen
    score = highScore;
    snprintf(scoreBuff, 50, "HighS: %d", score);
  } else if ( !gameOver ) { //Increments score ONLY while playing
    score = (millis() - compensation) * multiplier / 250;
    snprintf(scoreBuff, 50, "Score: %d", score);
  }
  u8g2.setFont(u8g2_font_t0_11_mf );
  u8g2.drawStr( 55, 8, scoreBuff );
}

//Draws Game Over Screen
static void playAgain() {
  u8g2.drawXBM(14, 14, gameover_width, gameover_height, gameover_bits);
  u8g2.drawXBM(23, 30, replay_width, replay_height, replay_bits);
  u8g2.drawXBM(70, 30, menu_width, menu_height, menu_bits);
}

//Draws Menu Screen
static void menuScreen() {
  u8g2.drawXBM(26, 30, easy_width, easy_height, easy_bits);
  u8g2.drawXBM(49, 12, medium_width, medium_height, medium_bits);
  u8g2.drawXBM(69, 30, hard_width, hard_height, hard_bits);
  u8g2.drawXBM(49, 43, insane_width, insane_height, insane_bits);
}

static void draw() {
  u8g2.clearBuffer();
  drawDinoCactus();
  drawScore();;
  if ( menu ) menuScreen(); //Draw Menu Screen
  else if ( gameOver ) playAgain(); //Draw Game Over Screen
  u8g2.sendBuffer();
}

static void u8g2_trex_entry(void *parameter)
{
  reset();
  u8g2.begin(/*Select=*/ U8G2_PIN_SELECT, /*Right/Next=*/ U8G2_PIN_RIGHT, /*Left/Prev=*/ U8G2_PIN_LEFT, /*Up=*/ U8G2_PIN_UP, /*Down=*/ U8G2_PIN_DOWN, /*Home/Cancel=*/ U8G2_PIN_HOME);

  for(;;)
  {
    keyPress(u8g2.getMenuEvent());
    if ( !gameOver && !menu ) moveObjects(); //Only move objects while not on menu or game over screen
    draw();
    collision();
  }
}

#define THREAD_PRIORITY         25
#define THREAD_STACK_SIZE       1024
#define THREAD_TIMESLICE        5

static rt_thread_t tid1 = RT_NULL;

static void u8g2_game_trex(int argc,char *argv[])
{
  tid1 = rt_thread_create("tu8g250",
                          u8g2_trex_entry, RT_NULL,
                          THREAD_STACK_SIZE,
                          THREAD_PRIORITY, THREAD_TIMESLICE);
  if (tid1 != RT_NULL)
    rt_thread_startup(tid1);
}
MSH_CMD_EXPORT(u8g2_game_trex, u8g2 game T-Rex sample);
