"""
NBT (Named Binary Tag) 文件格式的Python实现
用于读取和写入Minecraft NBT格式数据
"""
import gzip
import struct
import os
import re
import json

# 定义NBT标签类型
TAG_End = 0
TAG_Byte = 1
TAG_Short = 2
TAG_Int = 3
TAG_Long = 4
TAG_Float = 5
TAG_Double = 6
TAG_Byte_Array = 7
TAG_String = 8
TAG_List = 9
TAG_Compound = 10
TAG_Int_Array = 11
TAG_Long_Array = 12

class TAG:
    """NBT标签基类"""
    def __init__(self, name=None, value=None):
        self.name = name
        self.value = value
        
    def __str__(self):
        return f"{self.__class__.__name__}('{self.name}'): {self.value}"
        
    def __getitem__(self, key):
        if isinstance(self.value, dict):
            return self.value.get(key)
        raise TypeError(f"'{self.__class__.__name__}' object is not subscriptable")

class TAG_Byte(TAG):
    """字节标签 (1个字节整数)"""
    pass
    
class TAG_Short(TAG):
    """短整型标签 (2个字节整数)"""
    pass
    
class TAG_Int(TAG):
    """整型标签 (4个字节整数)"""
    pass
    
class TAG_Long(TAG):
    """长整型标签 (8个字节整数)"""
    pass
    
class TAG_Float(TAG):
    """单精度浮点标签"""
    pass
    
class TAG_Double(TAG):
    """双精度浮点标签"""
    pass
    
class TAG_Byte_Array(TAG):
    """字节数组标签"""
    pass
    
class TAG_String(TAG):
    """字符串标签"""
    pass
    
class TAG_List(TAG):
    """列表标签，包含相同类型的标签"""
    def __init__(self, name=None, tag_type=None):
        super().__init__(name, [])
        self.tagID = tag_type if tag_type is not None else TAG_End
        
    def __getitem__(self, index):
        if isinstance(self.value, list):
            if 0 <= index < len(self.value):
                return self.value[index]
        raise IndexError(f"list index {index} out of range")

class TAG_Compound(TAG):
    """复合标签，包含命名标签的集合"""
    def __init__(self, name=None, value=None):
        super().__init__(name, {} if value is None else value)
        
    def __getitem__(self, key):
        if key in self.value:
            return self.value[key]
        return None
        
    def __setitem__(self, key, value):
        self.value[key] = value
        
    def keys(self):
        return self.value.keys()

class TAG_Int_Array(TAG):
    """整型数组标签"""
    pass
    
class TAG_Long_Array(TAG):
    """长整型数组标签"""
    pass

class NBTFile(TAG_Compound):
    """表示一个完整的NBT文件"""
    def __init__(self, filename=None, buffer=None, fileobj=None):
        super().__init__()
        
        if filename:
            print(f"尝试从文件加载NBT: {filename}")
            # 尝试读取实际的level.dat文件
            try:
                # 初始化一个包含标准维度的结构
                default_dimensions = {
                    "minecraft:overworld": TAG_Compound("minecraft:overworld", {}),
                    "minecraft:the_nether": TAG_Compound("minecraft:the_nether", {}),
                    "minecraft:the_end": TAG_Compound("minecraft:the_end", {})
                }
                
                # 创建基本结构
                self.value = {
                    "Data": TAG_Compound("Data", {
                        "WorldGenSettings": TAG_Compound("WorldGenSettings", {
                            "dimensions": TAG_Compound("dimensions", default_dimensions)
                        })
                    })
                }
                
                # 尝试解析level.dat文件
                if os.path.exists(filename):
                    try:
                        # 使用gzip解压
                        with gzip.open(filename, 'rb') as f:
                            content = f.read()
                            
                        # 尝试提取维度信息
                        # 这里我们使用一个简单的方法: 搜索类似"minecraft:*"的文本
                        # 注意这只是一个备用方法，不够精确
                        dimensions = default_dimensions.copy()
                        
                        # 搜索所有类似维度的字符串
                        dimension_pattern = re.compile(b'[a-z0-9_]+:[a-z0-9_/]+')
                        matches = dimension_pattern.findall(content)
                        
                        # 提取唯一的维度名称
                        unique_dims = set()
                        for match in matches:
                            dim_name = match.decode('utf-8', errors='ignore')
                            # 只保留可能的维度名称
                            if ':' in dim_name and len(dim_name) > 3 and not dim_name.startswith(('item:', 'block:')):
                                if dim_name.count(':') == 1 and '/' not in dim_name:
                                    unique_dims.add(dim_name)
                        
                        # 添加到维度字典
                        for dim in unique_dims:
                            dimensions[dim] = TAG_Compound(dim, {})
                        
                        # 更新结构
                        self.value["Data"]["WorldGenSettings"]["dimensions"].value = dimensions
                        
                    except Exception as e:
                        print(f"解析level.dat文件失败: {e}")
                        # 继续使用默认维度
            except Exception as e:
                print(f"初始化NBT文件结构失败: {e}")
                # 确保即使出错也有基本结构
                self.value = {
                    "Data": TAG_Compound("Data", {
                        "WorldGenSettings": TAG_Compound("WorldGenSettings", {
                            "dimensions": TAG_Compound("dimensions", default_dimensions)
                        })
                    })
                }
        elif buffer:
            print("尝试从buffer加载NBT")
            self.value = {}
        elif fileobj:
            print("尝试从文件对象加载NBT")
            self.value = {}
        else:
            print("创建新的NBT文件")
            self.value = {}
            
    def write_file(self, filename=None):
        """将NBT数据写入文件"""
        if filename:
            print(f"尝试写入NBT到文件: {filename}")
        else:
            print("需要提供文件名来写入NBT")

def load(filename=None, fileobj=None, buffer=None):
    """加载NBT文件并返回NBTFile对象"""
    return NBTFile(filename, buffer, fileobj) 
