/*
 * Decompiled with CFR 0.152.
 */
package com.minecraft.selector.core;

import com.minecraft.selector.core.MapRenderer;
import com.minecraft.selector.utils.LogManager;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

public class DynamicRegionManager {
    private final LogManager logManager;
    private final MapRenderer renderer;
    private final String savePath;
    private final Map<Point, RegionTile> loadedRegions = new ConcurrentHashMap<Point, RegionTile>();
    private final Map<Point, Future<RegionTile>> loadingRegions = new ConcurrentHashMap<Point, Future<RegionTile>>();
    private final Set<Point> availableRegions = new HashSet<Point>();
    private final ExecutorService loadingExecutor;
    private static final int REGION_SIZE = 512;
    private static final int PRELOAD_RADIUS = 1;
    private final List<RegionLoadListener> listeners = new CopyOnWriteArrayList<RegionLoadListener>();

    public DynamicRegionManager(String savePath) {
        this.savePath = savePath;
        this.logManager = LogManager.getInstance();
        this.renderer = new MapRenderer(2, null);
        this.loadingExecutor = Executors.newFixedThreadPool(3);
        this.scanAvailableRegions();
        this.logManager.info("\u52a8\u6001\u533a\u57df\u7ba1\u7406\u5668\u5df2\u521d\u59cb\u5316\uff0c\u53d1\u73b0 " + this.availableRegions.size() + " \u4e2a\u53ef\u7528\u533a\u57df", "DynamicRegionManager");
    }

    private void scanAvailableRegions() {
        File regionDir = new File(this.savePath, "region");
        if (!regionDir.exists()) {
            this.logManager.warn("\u533a\u57df\u76ee\u5f55\u4e0d\u5b58\u5728: " + regionDir.getAbsolutePath(), "DynamicRegionManager");
            return;
        }
        File[] mcaFiles = regionDir.listFiles((dir, name) -> name.endsWith(".mca"));
        if (mcaFiles == null) {
            return;
        }
        for (File mcaFile : mcaFiles) {
            String fileName = mcaFile.getName();
            if (!fileName.matches("r\\.-?\\d+\\.-?\\d+\\.mca")) continue;
            String[] parts = fileName.replace(".mca", "").split("\\.");
            try {
                int regionX = Integer.parseInt(parts[1]);
                int regionZ = Integer.parseInt(parts[2]);
                this.availableRegions.add(new Point(regionX, regionZ));
            }
            catch (NumberFormatException e) {
                this.logManager.warn("\u65e0\u6cd5\u89e3\u6790\u533a\u57df\u6587\u4ef6\u540d: " + fileName, "DynamicRegionManager");
            }
        }
        this.logManager.info("\u626b\u63cf\u5b8c\u6210\uff0c\u53d1\u73b0 " + this.availableRegions.size() + " \u4e2a\u533a\u57df\u6587\u4ef6", "DynamicRegionManager");
    }

    public void updateViewport(Rectangle worldViewport) {
        int minRegionX = (int)Math.floor((double)worldViewport.x / 512.0);
        int maxRegionX = (int)Math.floor((double)(worldViewport.x + worldViewport.width) / 512.0);
        int minRegionZ = (int)Math.floor((double)worldViewport.y / 512.0);
        int maxRegionZ = (int)Math.floor((double)(worldViewport.y + worldViewport.height) / 512.0);
        this.logManager.debug(String.format("\u539f\u59cb\u89c6\u53e3: x=%d, y=%d, w=%d, h=%d", worldViewport.x, worldViewport.y, worldViewport.width, worldViewport.height), "DynamicRegionManager");
        this.logManager.debug(String.format("\u8ba1\u7b97\u533a\u57df\u8303\u56f4: X[%d,%d], Z[%d,%d]", minRegionX, maxRegionX, minRegionZ, maxRegionZ), "DynamicRegionManager");
        ++maxRegionX;
        --minRegionZ;
        ++maxRegionZ;
        HashSet<Point> requiredRegions = new HashSet<Point>();
        for (int x = --minRegionX; x <= maxRegionX; ++x) {
            for (int z = minRegionZ; z <= maxRegionZ; ++z) {
                Point regionCoord = new Point(x, z);
                if (!this.availableRegions.contains(regionCoord)) continue;
                requiredRegions.add(regionCoord);
            }
        }
        this.logManager.debug(String.format("\u89c6\u53e3\u66f4\u65b0: \u9700\u8981 %d \u4e2a\u533a\u57df [%d,%d] \u5230 [%d,%d]", requiredRegions.size(), minRegionX, minRegionZ, maxRegionX, maxRegionZ), "DynamicRegionManager");
        this.loadRequiredRegions(requiredRegions);
    }

    private void unloadUnneededRegions(Set<Point> requiredRegions) {
        ArrayList<Point> toUnload = new ArrayList<Point>();
        for (Point regionCoord : this.loadedRegions.keySet()) {
            if (requiredRegions.contains(regionCoord)) continue;
            boolean isTooFar = true;
            for (Point required : requiredRegions) {
                int distance = Math.max(Math.abs(regionCoord.x - required.x), Math.abs(regionCoord.y - required.y));
                if (distance > 2) continue;
                isTooFar = false;
                break;
            }
            if (!isTooFar) continue;
            toUnload.add(regionCoord);
        }
        for (Point regionCoord : toUnload) {
            RegionTile tile = this.loadedRegions.remove(regionCoord);
            if (tile == null) continue;
            this.logManager.debug("\u5378\u8f7d\u533a\u57df: r." + regionCoord.x + "." + regionCoord.y + ".mca", "DynamicRegionManager");
            this.notifyRegionUnloaded(regionCoord);
        }
        ArrayList<Point> toCancel = new ArrayList<Point>();
        for (Point regionCoord : this.loadingRegions.keySet()) {
            if (requiredRegions.contains(regionCoord)) continue;
            toCancel.add(regionCoord);
        }
        for (Point regionCoord : toCancel) {
            Future<RegionTile> future = this.loadingRegions.get(regionCoord);
            if (future == null) continue;
            future.cancel(true);
            this.loadingRegions.remove(regionCoord);
            this.logManager.debug("\u53d6\u6d88\u52a0\u8f7d\u533a\u57df: r." + regionCoord.x + "." + regionCoord.y + ".mca", "DynamicRegionManager");
        }
    }

    private void loadRequiredRegions(Set<Point> requiredRegions) {
        for (Point regionCoord : requiredRegions) {
            if (this.loadedRegions.containsKey(regionCoord) || this.loadingRegions.containsKey(regionCoord)) continue;
            this.startLoadingRegion(regionCoord);
        }
    }

    private void unloadLeastRecentlyUsed() {
        RegionTile tile;
        Point oldestRegion = null;
        long oldestTime = Long.MAX_VALUE;
        for (Map.Entry<Point, RegionTile> entry : this.loadedRegions.entrySet()) {
            long lastAccess = entry.getValue().getLastAccessTime();
            if (lastAccess >= oldestTime) continue;
            oldestTime = lastAccess;
            oldestRegion = entry.getKey();
        }
        if (oldestRegion != null && (tile = this.loadedRegions.remove(oldestRegion)) != null) {
            this.logManager.debug("LRU\u5378\u8f7d\u533a\u57df: r." + oldestRegion.x + "." + oldestRegion.y + ".mca", "DynamicRegionManager");
            this.notifyRegionUnloaded(oldestRegion);
        }
    }

    private void startLoadingRegion(Point regionCoord) {
        this.logManager.debug("\u5f00\u59cb\u52a0\u8f7d\u533a\u57df: r." + regionCoord.x + "." + regionCoord.y + ".mca", "DynamicRegionManager");
        this.notifyLoadingStarted(regionCoord);
        Future<RegionTile> future = this.loadingExecutor.submit(() -> {
            try {
                BufferedImage regionImage;
                String mcaFile = new File(this.savePath, String.format("region/r.%d.%d.mca", regionCoord.x, regionCoord.y)).getAbsolutePath();
                MapRenderer.BlockInfo[][] topBlocks = this.renderer.getTopBlocks(mcaFile, 32, 1);
                if (topBlocks != null && (regionImage = this.renderer.renderToPng(topBlocks, 1)) != null) {
                    RegionTile tile = new RegionTile(regionCoord, regionImage);
                    this.logManager.debug("\u533a\u57df\u52a0\u8f7d\u5b8c\u6210: r." + regionCoord.x + "." + regionCoord.y + ".mca", "DynamicRegionManager");
                    return tile;
                }
                this.logManager.warn("\u533a\u57df\u6e32\u67d3\u5931\u8d25: r." + regionCoord.x + "." + regionCoord.y + ".mca", "DynamicRegionManager");
                return null;
            }
            catch (Exception e) {
                this.logManager.error("\u533a\u57df\u52a0\u8f7d\u51fa\u9519: r." + regionCoord.x + "." + regionCoord.y + ".mca", "DynamicRegionManager", e);
                throw e;
            }
        });
        this.loadingRegions.put(regionCoord, future);
        CompletableFuture.supplyAsync(() -> {
            try {
                return (RegionTile)future.get();
            }
            catch (CancellationException e) {
                this.logManager.debug("\u533a\u57df\u52a0\u8f7d\u88ab\u53d6\u6d88: r." + regionCoord.x + "." + regionCoord.y + ".mca", "DynamicRegionManager");
                return null;
            }
            catch (Exception e) {
                this.logManager.error("\u83b7\u53d6\u533a\u57df\u52a0\u8f7d\u7ed3\u679c\u5931\u8d25", "DynamicRegionManager", e);
                this.notifyLoadingFailed(regionCoord, e);
                return null;
            }
        }).thenAccept(tile -> {
            this.loadingRegions.remove(regionCoord);
            if (tile != null) {
                this.loadedRegions.put(regionCoord, (RegionTile)tile);
                this.notifyRegionLoaded(regionCoord, (RegionTile)tile);
            }
        });
    }

    public RegionTile getLoadedRegion(Point regionCoord) {
        return this.loadedRegions.get(regionCoord);
    }

    public boolean isRegionLoading(Point regionCoord) {
        return this.loadingRegions.containsKey(regionCoord);
    }

    public Map<Point, RegionTile> getAllLoadedRegions() {
        return new HashMap<Point, RegionTile>(this.loadedRegions);
    }

    public Set<Point> getAvailableRegions() {
        return new HashSet<Point>(this.availableRegions);
    }

    public void addRegionLoadListener(RegionLoadListener listener) {
        this.listeners.add(listener);
    }

    public void removeRegionLoadListener(RegionLoadListener listener) {
        this.listeners.remove(listener);
    }

    private void notifyRegionLoaded(Point regionCoord, RegionTile tile) {
        for (RegionLoadListener listener : this.listeners) {
            try {
                listener.onRegionLoaded(regionCoord, tile);
            }
            catch (Exception e) {
                this.logManager.error("\u533a\u57df\u52a0\u8f7d\u76d1\u542c\u5668\u51fa\u9519", "DynamicRegionManager", e);
            }
        }
    }

    private void notifyRegionUnloaded(Point regionCoord) {
        for (RegionLoadListener listener : this.listeners) {
            try {
                listener.onRegionUnloaded(regionCoord);
            }
            catch (Exception e) {
                this.logManager.error("\u533a\u57df\u5378\u8f7d\u76d1\u542c\u5668\u51fa\u9519", "DynamicRegionManager", e);
            }
        }
    }

    private void notifyLoadingStarted(Point regionCoord) {
        for (RegionLoadListener listener : this.listeners) {
            try {
                listener.onLoadingStarted(regionCoord);
            }
            catch (Exception e) {
                this.logManager.error("\u533a\u57df\u52a0\u8f7d\u5f00\u59cb\u76d1\u542c\u5668\u51fa\u9519", "DynamicRegionManager", e);
            }
        }
    }

    private void notifyLoadingFailed(Point regionCoord, Exception error) {
        for (RegionLoadListener listener : this.listeners) {
            try {
                listener.onLoadingFailed(regionCoord, error);
            }
            catch (Exception e) {
                this.logManager.error("\u533a\u57df\u52a0\u8f7d\u5931\u8d25\u76d1\u542c\u5668\u51fa\u9519", "DynamicRegionManager", e);
            }
        }
    }

    public void clearAllLoadedRegions() {
        this.logManager.info("\u624b\u52a8\u6e05\u7406\u6240\u6709\u5df2\u52a0\u8f7d\u533a\u57df\uff0c\u91ca\u653e\u5185\u5b58", "DynamicRegionManager");
        for (Map.Entry<Point, Future<RegionTile>> entry : this.loadingRegions.entrySet()) {
            entry.getValue().cancel(true);
            this.notifyRegionUnloaded(entry.getKey());
        }
        this.loadingRegions.clear();
        for (Point regionCoord : this.loadedRegions.keySet()) {
            this.notifyRegionUnloaded(regionCoord);
        }
        this.loadedRegions.clear();
        this.logManager.info("\u5185\u5b58\u6e05\u7406\u5b8c\u6210", "DynamicRegionManager");
    }

    public String getStatusInfo() {
        return String.format("\u5df2\u52a0\u8f7d: %d, \u52a0\u8f7d\u4e2d: %d, \u53ef\u7528: %d", this.loadedRegions.size(), this.loadingRegions.size(), this.availableRegions.size());
    }

    public void shutdown() {
        this.loadingExecutor.shutdown();
        try {
            if (!this.loadingExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.loadingExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.loadingExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
        this.loadedRegions.clear();
        this.loadingRegions.clear();
        this.renderer.shutdown();
        this.logManager.info("\u52a8\u6001\u533a\u57df\u7ba1\u7406\u5668\u5df2\u5173\u95ed", "DynamicRegionManager");
    }

    public static class RegionTile {
        private final Point regionCoord;
        private final BufferedImage image;
        private final long loadTime;
        private long lastAccessTime;

        public RegionTile(Point regionCoord, BufferedImage image) {
            this.regionCoord = regionCoord;
            this.image = image;
            this.lastAccessTime = this.loadTime = System.currentTimeMillis();
        }

        public Point getRegionCoord() {
            return this.regionCoord;
        }

        public BufferedImage getImage() {
            this.lastAccessTime = System.currentTimeMillis();
            return this.image;
        }

        public long getLoadTime() {
            return this.loadTime;
        }

        public long getLastAccessTime() {
            return this.lastAccessTime;
        }
    }

    public static interface RegionLoadListener {
        public void onRegionLoaded(Point var1, RegionTile var2);

        public void onRegionUnloaded(Point var1);

        public void onLoadingStarted(Point var1);

        public void onLoadingFailed(Point var1, Exception var2);
    }
}

