!**********************************************************************************************************************************
! LICENSING
! Copyright (C) 2013-2015  National Renewable Energy Laboratory
!
!    This file is part of HydroDyn.
!
! Licensed under the Apache License, Version 2.0 (the "License");
! you may not use this file except in compliance with the License.
! You may obtain a copy of the License at
!
!     http://www.apache.org/licenses/LICENSE-2.0
!
! Unless required by applicable law or agreed to in writing, software
! distributed under the License is distributed on an "AS IS" BASIS,
! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
! See the License for the specific language governing permissions and
! limitations under the License.
!    
!**********************************************************************************************************************************
MODULE HydroDyn_Input

      ! This MODULE stores variables used for input.

   USE                              NWTC_Library
   USE                              HydroDyn_Types
   USE                              HydroDyn_Output
   USE                              SeaState
   USE                              Morison
   USE                              Morison_Output
   USE                              NWTC_RandomNumber
   IMPLICIT                         NONE

CONTAINS
   


!====================================================================================================
SUBROUTINE PrintBadChannelWarning(NUserOutputs, UserOutputs , foundMask, ErrStat, ErrMsg )
!     The routine prints out warning messages if the user has requested invalid output channel names
!     The errstat is set to ErrID_Warning if any element in foundMask is .FALSE.
!----------------------------------------------------------------------------------------------------  
   INTEGER,                       INTENT( IN    ) :: NUserOutputs         ! Number of user-specified output channels
   CHARACTER(ChanLen),            INTENT( IN    ) :: UserOutputs (:)      ! An array holding the names of the requested output channels. 
   LOGICAL,                       INTENT( IN    ) :: foundMask (:)        ! A mask indicating whether a user requested channel belongs to a module's output channels.
   INTEGER,                       INTENT(   OUT ) :: ErrStat              ! returns a non-zero value when an error occurs  
   CHARACTER(*),                  INTENT(   OUT ) :: ErrMsg               ! Error message if ErrStat /= ErrID_None
   INTEGER                                        :: I
   
   ErrStat = ErrID_None
   ErrMsg  = ''
   
   DO I = 1, NUserOutputs
      IF (.NOT. foundMask(I)) THEN
         ErrMsg  = ' A requested output channel is invalid'         
         CALL ProgWarn( 'The requested output channel is invalid: ' // UserOutputs(I) )
         ErrStat = ErrID_Warn
      END IF
   END DO
   
END SUBROUTINE PrintBadChannelWarning



!====================================================================================================
SUBROUTINE HydroDyn_ParseInput( InputFileName, OutRootName, FileInfo_In, InputFileData, ErrStat, ErrMsg )
!     This public subroutine reads the input required for HydroDyn from the file whose name is an
!     input parameter.
!----------------------------------------------------------------------------------------------------

      ! Passed variables
   CHARACTER(*),                  intent(in   ) :: InputFileName        !< The name of the input file, for putting in echo file.
   CHARACTER(*),                  intent(in   ) :: OutRootName          !< The rootname of the echo file, possibly opened in this routine
   TYPE(FileInfoType),            INTENT(IN   ) :: FileInfo_In          !< The derived type for holding the file information
   TYPE(HydroDyn_InputFile),      INTENT(INOUT) :: InputFileData        ! the hydrodyn input file data
   INTEGER,                       INTENT(  OUT) :: ErrStat              ! returns a non-zero value when an error occurs
   CHARACTER(*),                  INTENT(  OUT) :: ErrMsg               ! Error message if ErrStat /= ErrID_None

      ! Local variables
   INTEGER                                      :: I, j                 ! generic integer for counting
   CHARACTER(   2)                              :: strI                 ! string version of the loop counter
   INTEGER                                      :: UnEc                 ! The local unit number for this module's echo file
   CHARACTER(1024)                              :: EchoFile             ! Name of HydroDyn echo file
   CHARACTER(MaxFileInfoLineLen)                :: Line                 ! String to temporarially hold value of read line
   real(ReKi), ALLOCATABLE                      :: tmpVec1(:), tmpVec2(:) ! Temporary arrays for WAMIT data
   integer(IntKi)                               :: startIndx, endIndx   ! indices into working arrays
   INTEGER, ALLOCATABLE                         :: tmpArray(:)          ! Temporary array storage of the joint output list
   REAL(ReKi), ALLOCATABLE                      :: tmpReArray(:)        ! Temporary array storage of the joint output list
   INTEGER(IntKi)                               :: CurLine              !< Current entry in FileInfo_In%Lines array
   INTEGER(IntKi)                               :: ErrStat2
   CHARACTER(ErrMsgLen)                         :: ErrMsg2
   CHARACTER(*),  PARAMETER                     :: RoutineName = 'HydroDyn_ParaseInput'

   
      ! Initialize local data
   UnEc     = -1
   ErrStat  =  ErrID_None         
   ErrMsg   =  ""   
   InputFileData%Echo = .FALSE.  ! initialize for error handling (cleanup() routine)
   

   !-------------------------------------------------------------------------------------------------
   ! General settings 
   !-------------------------------------------------------------------------------------------------

   CurLine = 3    ! Skip the first three lines as they are known to be header lines and separators
   call ParseVar( FileInfo_In, CurLine, 'Echo', InputFileData%Echo, ErrStat2, ErrMsg2 )
         if (Failed()) return;

   if ( InputFileData%Echo ) then
      EchoFile = TRIM(OutRootName)//'.ech'
      CALL OpenEcho ( UnEc, TRIM(EchoFile), ErrStat2, ErrMsg2 )
         if (Failed())  return;
      WRITE(UnEc, '(A)') 'Echo file for AeroDyn 15 primary input file: '//trim(InputFileName)
      ! Write the first three lines into the echo file
      WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(1))
      WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(2))

      CurLine = 3
      call ParseVar( FileInfo_In, CurLine, 'Echo', InputFileData%Echo, ErrStat2, ErrMsg2, UnEc )
         if (Failed()) return
   endif

   !-------------------------------------------------------------------------------------------------
   ! Data section for floating platform
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! PotMod - State indicating potential flow model used in the simulation. 0=none, 1=WAMIT, 2=FIT
   call ParseVar( FileInfo_In, CurLine, 'PotMod', InputFileData%PotMod, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! ExctnMod  - Wave Excitation model {0: None, 1: DFT, 2: state-space} (switch)
      ! [STATE-SPACE REQUIRES *.ssexctn INPUT FILE]
   call ParseVar( FileInfo_In, CurLine, 'ExctnMod', InputFileData%WAMIT%ExctnMod, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! ExctnDisp  - Use body displacements to compute Wave Excitations {0: use undisplaced position, 1: use displaced position, 2: use low-pass filtered displaced position) [only used when PotMod=1 and ExctnMod>0]} (switch)
   call ParseVar( FileInfo_In, CurLine, 'ExctnDisp', InputFileData%WAMIT%ExctnDisp, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;
      
      ! ExctnCutOff  - Cutoff (corner) frequency of the low-pass time-filtered displaced position (Hz) [>0.0] [used only when PotMod=1, ExctnMod>0, and ExctnDisp=2])
      ! [STATE-SPACE REQUIRES *.ssexctn INPUT FILE]
   call ParseVar( FileInfo_In, CurLine, 'ExctnCutOff', InputFileData%WAMIT%ExctnCutOff, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! PtfmYMod - Model for large platform yaw offset {0: Static reference yaw offset based on PtfmRefY, 1: dynamic reference yaw offset based on low-pass filtering the PRP yaw motion with cutoff frequency PtfmYCutOff} (switch)
   call ParseVar( FileInfo_In, CurLine, 'PtfmYMod', InputFileData%PtfmYMod, ErrStat2, ErrMsg2, UnEc )
      if (Failed()) return

      ! PtfmRefY - Constant or initial platform reference yaw offset (deg)
   call ParseVar( FileInfo_In, CurLine, 'PtfmRefY', InputFileData%PtfmRefY, ErrStat2, ErrMsg2, UnEc )
      if (Failed()) return
   InputFileData%PtfmRefY = InputFileData%PtfmRefY * D2R

      ! PtfmYCutOff - Cutoff frequency for the low-pass filtering of PRP yaw motion when PtfmYMod=1 [unused when PtfmYMod=0] (Hz)
   call ParseVar( FileInfo_In, CurLine, 'PtfmYCutOff', InputFileData%PtfmYCutOff, ErrStat2, ErrMsg2, UnEc )
      if (Failed()) return

      ! NExctnHdg  - Number of PRP headings/yaw offset evenly distributed in the range of [-180, 180) deg to precompute [used only when PtfmYMod = 1]
   call ParseVar( FileInfo_In, CurLine, 'NExctnHdg', InputFileData%WAMIT%NExctnHdg, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;
   InputFileData%WAMIT2%NExctnHdg = InputFileData%WAMIT%NExctnHdg

      ! RdtnMod  - Radiation memory-effect model {1: convolution, 2: state-space} (switch)
      ! [STATE-SPACE REQUIRES *.ss INPUT FILE]
   call ParseVar( FileInfo_In, CurLine, 'RdtnMod', InputFileData%WAMIT%RdtnMod, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! RdtnTMax - Analysis time for wave radiation kernel calculations
      ! NOTE: Use RdtnTMax = 0.0 to eliminate wave radiation damping
   call ParseVar( FileInfo_In, CurLine, 'RdtnTMax', InputFileData%WAMIT%RdtnTMax, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! RdtnDT - Time step for wave radiation kernel calculations
   call ParseVar( FileInfo_In, CurLine, 'RdtnDT', InputFileData%WAMIT%Conv_Rdtn%RdtnDTChr, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! NBody - Number of WAMIT bodies to be used (-) [>=1; only used when PotMod=1. If NBodyMod=1, the WAMIT data 
      !         contains a vector of size 6*NBody x 1 and matrices of size 6*NBody x 6*NBody; if NBodyMod>1, there 
      !         are NBody sets of WAMIT data each with a vector of size 6 x 1 and matrices of size 6 x 6]
   call ParseVar( FileInfo_In, CurLine, 'NBody', InputFileData%NBody, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! NBodyMod - Body coupling model {1: include coupling terms between each body and NBody in HydroDyn equals NBODY in WAMIT, 
      !            2: neglect coupling terms between each body and NBODY=1 with XBODY=0 in WAMIT, 3: Neglect coupling terms 
      !            between each body and NBODY=1 with XBODY=/0 in WAMIT} (switch) [only used when PotMod=1]
   call ParseVar( FileInfo_In, CurLine, 'NBodyMod', InputFileData%NBodyMod, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;
      
         ! allocate space for the WAMIT-related data arrays:
   if ( InputFileData%NBodyMod == 1 )  then
      InputFileData%nWAMITObj = 1  ! Special case where all data in a single WAMIT input file as opposed to  InputFileData%NBody number of separate input files.
      InputFileData%vecMultiplier = InputFileData%NBody
   else
      InputFileData%nWAMITObj    = InputFileData%NBody
      InputFileData%vecMultiplier = 1
   end if
   
   CALL AllocAry( InputFileData%PotFile      , InputFileData%nWAMITObj, 'PotFile'      , ErrStat2, ErrMsg2);   if (Failed())  return;
   CALL AllocAry( InputFileData%WAMITULEN    , InputFileData%nWAMITObj, 'WAMITULEN'    , ErrStat2, ErrMsg2);   if (Failed())  return;
   CALL AllocAry( InputFileData%PtfmRefxt    , InputFileData%NBody,     'PtfmRefxt'    , ErrStat2, ErrMsg2);   if (Failed())  return;
   CALL AllocAry( InputFileData%PtfmRefyt    , InputFileData%NBody,     'PtfmRefyt'    , ErrStat2, ErrMsg2);   if (Failed())  return;
   CALL AllocAry( InputFileData%PtfmRefzt    , InputFileData%NBody,     'PtfmRefzt'    , ErrStat2, ErrMsg2);   if (Failed())  return;
   CALL AllocAry( InputFileData%PtfmRefztRot , InputFileData%NBody,     'PtfmRefztRot' , ErrStat2, ErrMsg2);   if (Failed())  return;
   CALL AllocAry( InputFileData%PtfmVol0     , InputFileData%NBody,     'PtfmVol0'     , ErrStat2, ErrMsg2);   if (Failed())  return;
   CALL AllocAry( InputFileData%PtfmCOBxt    , InputFileData%NBody,     'PtfmCOBxt'    , ErrStat2, ErrMsg2);   if (Failed())  return;
   CALL AllocAry( InputFileData%PtfmCOByt    , InputFileData%NBody,     'PtfmCOByt'    , ErrStat2, ErrMsg2);   if (Failed())  return;


      ! PotFile - Root name of Potential flow data files (Could be WAMIT files or the FIT input file)
   call ParseAry( FileInfo_In, CurLine, 'PotFile', InputFileData%PotFile, InputFileData%nWAMITObj, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! WAMITULEN - WAMIT characteristic body length scale
   call ParseAry( FileInfo_In, CurLine, 'WAMITULEN', InputFileData%WAMITULEN, InputFileData%nWAMITObj, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! PtfmRefxt  - The xt offset of the body reference point(s) from (0,0,0) (meters)
   call ParseAry( FileInfo_In, CurLine, 'PtfmRefxt', InputFileData%PtfmRefxt, InputFileData%NBody, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! PtfmRefyt  - The yt offset of the body reference point(s) from (0,0,0) (meters)
   call ParseAry( FileInfo_In, CurLine, 'PtfmRefyt', InputFileData%PtfmRefyt, InputFileData%NBody, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! PtfmRefzt  - The zt offset of the body reference point(s) from (0,0,0) (meters)
   call ParseAry( FileInfo_In, CurLine, 'PtfmRefzt', InputFileData%PtfmRefzt, InputFileData%NBody, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! PtfmRefztRot  - The rotation about zt of the body reference frame(s) from xt/yt (deg)
   call ParseAry( FileInfo_In, CurLine, 'PtfmRefztRot', InputFileData%PtfmRefztRot, InputFileData%NBody, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;
   InputFileData%PtfmRefztRot = InputFileData%PtfmRefztRot*D2R_D ! Convert to radians
   
      ! PtfmVol0 - Displaced volume of water when the platform is in its undisplaced position
   call ParseAry( FileInfo_In, CurLine, 'PtfmVol0', InputFileData%PtfmVol0, InputFileData%NBody, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! PtfmCOBxt  - The xt offset of the center of buoyancy (COB) from the WAMIT reference point
   call ParseAry( FileInfo_In, CurLine, 'PtfmCOBxt', InputFileData%PtfmCOBxt, InputFileData%NBody, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! PtfmCOByt - The yt offset of the center of buoyancy (COB) from the WAMIT reference point
   call ParseAry( FileInfo_In, CurLine, 'PtfmCOByt', InputFileData%PtfmCOByt, InputFileData%NBody, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;
   

   !-------------------------------------------------------------------------------------------------
   ! Data section for 2nd order WAMIT forces
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

        ! MnDrift    -- Mean drift forces computed from WAMIT file: {0: No mean drift, [7, 8, 9, 10, 11, or 12]: WAMIT file to use}
   call ParseVar( FileInfo_In, CurLine, 'MnDrift', InputFileData%WAMIT2%MnDrift, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

        ! NewmanApp  -- Slow drift forces computed with Newman's approximation from  WAMIT file: {0: No mean drift, [7, 8, 9, 10, 11, or 12]: WAMIT file to use}
   call ParseVar( FileInfo_In, CurLine, 'NewmanApp', InputFileData%WAMIT2%NewmanApp, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

        ! DiffQTF    -- Full Difference-Frequency forces computed with full QTFs from WAMIT file: {0: No difference-frequency forces, [10, 11, or 12]: WAMIT file to use} -- Only one of MnDrift, NewmanApp, or DiffQYT can be non-zero
   call ParseVar( FileInfo_In, CurLine, 'DiffQTF', InputFileData%WAMIT2%DiffQTF, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

        ! SumQTF     -- Full        Sum-Frequency forces computed with full QTFs from WAMIT file: {0: No        Sum-frequency forces, [10, 11, or 12]: WAMIT file to use}
   call ParseVar( FileInfo_In, CurLine, 'SumQTF', InputFileData%WAMIT2%SumQTF, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;


   !-------------------------------------------------------------------------------------------------
   ! Floating Platform Additional Stiffness and Damping Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

   ! If NBodyMod = 1 then vecMultiplier = NBody and nWAMITObj = 1
   ! Else                 vecMultiplier = 1     and nWAMITObj = NBody
   CALL AllocAry( InputFileData%AddF0,     InputFileData%vecMultiplier*6, InputFileData%nWAMITObj, 'InputFileData%AddF0'    , ErrStat2, ErrMsg2);                                   if (Failed())  return; 
   CALL AllocAry( InputFileData%AddCLin,   InputFileData%vecMultiplier*6, InputFileData%vecMultiplier*6, InputFileData%nWAMITObj, 'InputFileData%AddCLin'  , ErrStat2, ErrMsg2);    if (Failed())  return; 
   CALL AllocAry( InputFileData%AddBLin,   InputFileData%vecMultiplier*6, InputFileData%vecMultiplier*6, InputFileData%nWAMITObj, 'InputFileData%AddBLin'  , ErrStat2, ErrMsg2);    if (Failed())  return;
   CALL AllocAry( InputFileData%AddBQuad,  InputFileData%vecMultiplier*6, InputFileData%vecMultiplier*6, InputFileData%nWAMITObj, 'InputFileData%AddBQuad' , ErrStat2, ErrMsg2);    if (Failed())  return;
   CALL AllocAry( tmpVec1, InputFileData%nWAMITObj, 'tmpVec1', ErrStat2, ErrMsg2);  if (Failed())  return;
   CALL AllocAry( tmpVec2, 6*InputFileData%NBody,   'tmpVec2', ErrStat2, ErrMsg2);  if (Failed())  return;

      ! AddF0 - Additional preload
   do i = 1,6*InputFileData%vecMultiplier   
      call ParseAry( FileInfo_In, CurLine, 'AddF0', tmpVec1, InputFileData%nWAMITObj, ErrStat2, ErrMsg2, UnEc )
         if (Failed())  return;

      do j = 1, InputFileData%nWAMITObj
         InputFileData%AddF0(i,j) = tmpVec1(j)
      end do
   end do
   
      ! AddCLin
   do i=1,6*InputFileData%vecMultiplier

      write(strI,'(I1)') i
      call ParseAry( FileInfo_In, CurLine, ' Row '//strI//' of the additional linear stiffness matrix', &
                     tmpVec2, 6*InputFileData%NBody, ErrStat2, ErrMsg2, UnEc )
         if (Failed())  return;

      do j = 1, InputFileData%nWAMITObj
         startIndx = 6*InputFileData%vecMultiplier*(j-1) + 1
         endIndx   = startIndx + 6*InputFileData%vecMultiplier - 1
         InputFileData%AddCLin(i,:,j) = tmpVec2(startIndx:endIndx)
      end do
   end do


       ! AddBLin
   DO I=1,6*InputFileData%vecMultiplier

      call ParseAry( FileInfo_In, CurLine, ' Row '//strI//' of the additional linear damping matrix', &
                     tmpVec2, 6*InputFileData%NBody, ErrStat2, ErrMsg2, UnEc )
         if (Failed())  return;

      do j = 1, InputFileData%nWAMITObj
         startIndx = 6*InputFileData%vecMultiplier*(j-1) + 1
         endIndx   = startIndx + 6*InputFileData%vecMultiplier - 1
         InputFileData%AddBLin(I,:,j) = tmpVec2(startIndx:endIndx)
      end do
   END DO


       ! AddBQuad
   DO I=1,6*InputFileData%vecMultiplier

      call ParseAry( FileInfo_In, CurLine, ' Row '//strI//' of the additional quadratic damping matrix', &
                     tmpVec2, 6*InputFileData%NBody, ErrStat2, ErrMsg2, UnEc )
         if (Failed())  return;

      do j = 1, InputFileData%nWAMITObj
         startIndx = 6*InputFileData%vecMultiplier*(j-1) + 1
         endIndx   = startIndx + 6*InputFileData%vecMultiplier - 1
         InputFileData%AddBQuad(I,:,j) = tmpVec2(startIndx:endIndx)
      end do
   END DO

   !-------------------------------------------------------------------------------------------------
   !  Strip Theory Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1
   
   ! WaveDisp  - Method of computing Wave Kinematics {0: use undisplaced position, 1: use displaced position) } (switch)
   call ParseVar( FileInfo_In, CurLine, 'WaveDisp', InputFileData%Morison%WaveDisp, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;
      
   ! AMMod - Method of computing distributed added-mass force. {0: nodes below SWL when undisplaced. 1: Up to the free surface} (switch)
   call ParseVar( FileInfo_In, CurLine, 'AMMod', InputFileData%Morison%AMMod, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

   !-------------------------------------------------------------------------------------------------
   !  Axial Coefficients Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1
   
      ! NAxCoef - Number of axial coefficients
   call ParseVar( FileInfo_In, CurLine, 'NAxCoef', InputFileData%Morison%NAxCoefs, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;
   
      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Axial coefficient table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Axial coefficient table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
  
   IF ( InputFileData%Morison%NAxCoefs > 0 ) THEN
      CALL AllocAry( tmpReArray, 7, 'temporary array for AxialCoefs', ErrStat2, ErrMsg2 )
         if (Failed())  return;
      
         ! Allocate memory for Axial Coef-related arrays
      ALLOCATE ( InputFileData%Morison%AxialCoefs(InputFileData%Morison%NAxCoefs), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for AxialCoefs array.'
         if (Failed())  return;
      END IF
          
      DO I = 1,InputFileData%Morison%NAxCoefs
         ! read the table entries AxCoefID, AxCd, AxCa, AxCp, AxFdMod, AxVnCOff, AxFDLoFSc in the HydroDyn input file
         ! Try reading in 7 entries first
         call ParseAry( FileInfo_In, CurLine, ' axial coefficients line '//trim( Int2LStr(I)), tmpReArray, size(tmpReArray), ErrStat2, ErrMsg2, UnEc )
         if ( ErrStat2 /= ErrID_None ) then ! Try reading in 5 entries
            tmpReArray(6) = -1.0  ! AxVnCoff
            tmpReArray(7) =  1.0  ! AxFDLoFSc
            call ParseAry( FileInfo_In, CurLine, ' axial coefficients line '//trim( Int2LStr(I)), tmpReArray(1:5), 5, ErrStat2, ErrMsg2, UnEc )
            if ( ErrStat2 /= ErrID_None ) then ! Try reading in 4 entries
               tmpReArray(5) =  0.0  ! AxFdMod
               call ParseAry( FileInfo_In, CurLine, ' axial coefficients line '//trim( Int2LStr(I)), tmpReArray(1:4), 4, ErrStat2, ErrMsg2, UnEc )
               if (Failed())  return;
            end if
         end if
         InputFileData%Morison%AxialCoefs(I)%AxCoefID = NINT(tmpReArray(1))
         InputFileData%Morison%AxialCoefs(I)%AxCd     =      tmpReArray(2)
         InputFileData%Morison%AxialCoefs(I)%AxCa     =      tmpReArray(3)
         InputFileData%Morison%AxialCoefs(I)%AxCp     =      tmpReArray(4)
         InputFileData%Morison%AxialCoefs(I)%AxFDMod  = NINT(tmpReArray(5))
         InputFileData%Morison%AxialCoefs(I)%AxVnCOff =      tmpReArray(6) 
         InputFileData%Morison%AxialCoefs(I)%AxFDLoFSc =     tmpReArray(7)
      END DO

      if (allocated(tmpReArray))      deallocate(tmpReArray)
   END IF

   
   !-------------------------------------------------------------------------------------------------
   ! Member Joints Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! NJoints - Number of member joints
   call ParseVar( FileInfo_In, CurLine, 'NJoints', InputFileData%Morison%NJoints, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Joints table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Joints table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1

   IF ( InputFileData%Morison%NJoints > 0 ) THEN
      CALL AllocAry( tmpReArray, 6, 'temporary array for InpJoints', ErrStat2, ErrMsg2 )
         if (Failed())  return;

         ! Allocate memory for Joint-related arrays
      ALLOCATE ( InputFileData%Morison%InpJoints(InputFileData%Morison%NJoints), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for InpJoints array.'
         if (Failed())  return;
      END IF

      DO I = 1,InputFileData%Morison%NJoints
            ! read the table entries   JointID   Jointxi     Jointyi    Jointzi      JointAxID   JointOvrlp    in the HydroDyn input file
         call ParseAry( FileInfo_In, CurLine, ' joints table line '//trim( Int2LStr(I)), tmpReArray, size(tmpReArray), ErrStat2, ErrMsg2, UnEc )
            if (Failed())  return;
         InputFileData%Morison%InpJoints(I)%JointID      =  NINT(tmpReArray(1))
         InputFileData%Morison%InpJoints(I)%Position(1)  =       tmpReArray(2)
         InputFileData%Morison%InpJoints(I)%Position(2)  =       tmpReArray(3)
         InputFileData%Morison%InpJoints(I)%Position(3)  =       tmpReArray(4)
         InputFileData%Morison%InpJoints(I)%JointAxID    =  NINT(tmpReArray(5))
         InputFileData%Morison%InpJoints(I)%JointOvrlp   =  NINT(tmpReArray(6))
      END DO

      if (allocated(tmpReArray))      deallocate(tmpReArray)
   END IF


   !-------------------------------------------------------------------------------------------------
   ! Cylindrical Member Cross-section Properties Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! NPropSetsCyl - Number of cylindrical member cross-section property sets
   call ParseVar( FileInfo_In, CurLine, 'NPropSetsCyl', InputFileData%Morison%NPropSetsCyl, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'MPropSetsCyl table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'MPropSetsCyl table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1

   IF ( InputFileData%Morison%NPropSetsCyl > 0 ) THEN

      CALL AllocAry( tmpReArray, 3, 'temporary array for MPropSetsCyl', ErrStat2, ErrMsg2 )
         if (Failed())  return;

         ! Allocate memory for Member cross-section property set-related arrays
      ALLOCATE ( InputFileData%Morison%MPropSetsCyl(InputFileData%Morison%NPropSetsCyl), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for MPropSetsCyl array.'
         if (Failed())  return;
      END IF

      DO I = 1,InputFileData%Morison%NPropSetsCyl
         call ParseAry( FileInfo_In, CurLine, ' MPropSetsCyl line '//trim( Int2LStr(I)), tmpReArray, size(tmpReArray), ErrStat2, ErrMsg2, UnEc )
            if (Failed())  return;
         InputFileData%Morison%MPropSetsCyl(I)%PropSetID = NINT(tmpReArray(1))
         InputFileData%Morison%MPropSetsCyl(I)%PropD     =      tmpReArray(2)
         InputFileData%Morison%MPropSetsCyl(I)%PropThck  =      tmpReArray(3)
      END DO

      if (allocated(tmpReArray))      deallocate(tmpReArray)
   END IF

   !-------------------------------------------------------------------------------------------------
   ! Rectangular Member Cross-section Properties Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! NPropSetsRec - Number of rectangular member cross-section property sets
   call ParseVar( FileInfo_In, CurLine, 'NPropSetsRec', InputFileData%Morison%NPropSetsRec, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'MPropSetsRec table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'MPropSetsRec table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1

   IF ( InputFileData%Morison%NPropSetsRec > 0 ) THEN

      CALL AllocAry( tmpReArray, 4, 'temporary array for MPropSetsRec', ErrStat2, ErrMsg2 )
         if (Failed())  return;

         ! Allocate memory for Member cross-section property set-related arrays
      ALLOCATE ( InputFileData%Morison%MPropSetsRec(InputFileData%Morison%NPropSetsRec), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for MPropSetsRec array.'
         if (Failed())  return;
      END IF

      DO I = 1,InputFileData%Morison%NPropSetsRec
         call ParseAry( FileInfo_In, CurLine, ' MPropSetsRec line '//trim( Int2LStr(I)), tmpReArray, size(tmpReArray), ErrStat2, ErrMsg2, UnEc )
            if (Failed())  return;
         InputFileData%Morison%MPropSetsRec(I)%PropSetID = NINT(tmpReArray(1))
         InputFileData%Morison%MPropSetsRec(I)%PropA     =      tmpReArray(2)
         InputFileData%Morison%MPropSetsRec(I)%PropB     =      tmpReArray(3)
         InputFileData%Morison%MPropSetsRec(I)%PropThck  =      tmpReArray(4)
      END DO

      if (allocated(tmpReArray))      deallocate(tmpReArray)
   END IF

   !-------------------------------------------------------------------------------------------------
   ! Simple cylindrical member hydrodynamic coefficients Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Simple cylindrical member hydrodynamic coefficients table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Simple cylindrical member hydrodynamic coefficients table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1


   CALL AllocAry( tmpReArray, 14, 'temporary array for Simple cylindrical member hydrodynamic coefficients', ErrStat2, ErrMsg2 )
      if (Failed())  return
   ! call ParseAry( FileInfo_In, CurLine, 'Simple cylindrical member hydrodynamic coefficients table row '//trim( Int2LStr(I)), tmpReArray, size(tmpReArray), ErrStat2, ErrMsg2, UnEc )
   !    if (Failed())  return;
   CALL ParseRAryWKywrd( FileInfo_In, CurLine, 'Simple cylindrical member hydrodynamic coefficients table row '//trim( Int2LStr(1_IntKi)), tmpReArray, size(tmpReArray), &
                         'MCF', 1.0_ReKi, (/5,6/), InputFileData%Morison%SimplMCF, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return

   InputFileData%Morison%SimplCd       = tmpReArray( 1)
   InputFileData%Morison%SimplCdMG     = tmpReArray( 2)
   InputFileData%Morison%SimplCa       = tmpReArray( 3)
   InputFileData%Morison%SimplCaMG     = tmpReArray( 4)
   InputFileData%Morison%SimplCp       = tmpReArray( 5)
   InputFileData%Morison%SimplCpMG     = tmpReArray( 6)
   InputFileData%Morison%SimplAxCd     = tmpReArray( 7)
   InputFileData%Morison%SimplAxCdMG   = tmpReArray( 8)
   InputFileData%Morison%SimplAxCa     = tmpReArray( 9)
   InputFileData%Morison%SimplAxCaMG   = tmpReArray(10)
   InputFileData%Morison%SimplAxCp     = tmpReArray(11)
   InputFileData%Morison%SimplAxCpMG   = tmpReArray(12)
   InputFileData%Morison%SimplCb       = tmpReArray(13)
   InputFileData%Morison%SimplCbMG     = tmpReArray(14)

   if (allocated(tmpReArray))      deallocate(tmpReArray)

   !-------------------------------------------------------------------------------------------------
   ! Simple rectangular member hydrodynamic coefficients Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Simple rectangular member hydrodynamic coefficients table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Simple rectangular member hydrodynamic coefficients table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1


   CALL AllocAry( tmpReArray, 18, 'temporary array for Simple rectangular member hydrodynamic coefficients', ErrStat2, ErrMsg2 )
      if (Failed())  return
   CALL ParseRAryWKywrd( FileInfo_In, CurLine, 'Simple rectangular member hydrodynamic coefficients table row '//trim( Int2LStr(1_IntKi)), tmpReArray, size(tmpReArray), &
                         'MCF', 1.0_ReKi, (/9,10/), InputFileData%Morison%SimplRecMCF, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return

   InputFileData%Morison%SimplRecCdA      = tmpReArray( 1)
   InputFileData%Morison%SimplRecCdAMG    = tmpReArray( 2)
   InputFileData%Morison%SimplRecCdB      = tmpReArray( 3)
   InputFileData%Morison%SimplRecCdBMG    = tmpReArray( 4)
   InputFileData%Morison%SimplRecCaA      = tmpReArray( 5)
   InputFileData%Morison%SimplRecCaAMG    = tmpReArray( 6)
   InputFileData%Morison%SimplRecCaB      = tmpReArray( 7)
   InputFileData%Morison%SimplRecCaBMG    = tmpReArray( 8)
   InputFileData%Morison%SimplRecCp       = tmpReArray( 9)
   InputFileData%Morison%SimplRecCpMG     = tmpReArray(10)
   InputFileData%Morison%SimplRecAxCd     = tmpReArray(11)
   InputFileData%Morison%SimplRecAxCdMG   = tmpReArray(12)
   InputFileData%Morison%SimplRecAxCa     = tmpReArray(13)
   InputFileData%Morison%SimplRecAxCaMG   = tmpReArray(14)
   InputFileData%Morison%SimplRecAxCp     = tmpReArray(15)
   InputFileData%Morison%SimplRecAxCpMG   = tmpReArray(16)
   InputFileData%Morison%SimplRecCb       = tmpReArray(17)
   InputFileData%Morison%SimplRecCbMG     = tmpReArray(18)

   if (allocated(tmpReArray))      deallocate(tmpReArray)

   !-------------------------------------------------------------------------------------------------
   ! Depth-based Cylindrical Member Hydrodynamic Coefficients Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! NCoefDpthCyl - Number of depth-based cylindrical member hydrodynamic coefficient property sets
   call ParseVar( FileInfo_In, CurLine, 'NCoefDpthCyl', InputFileData%Morison%NCoefDpthCyl, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Depth-based cylindrical member hydrodynamic coefficients table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Depth-based cylindrical member hydrodynamic coefficients table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1

   IF ( InputFileData%Morison%NCoefDpthCyl > 0 ) THEN

      CALL AllocAry( tmpReArray, 15, 'temporary array for CoefDpthsCyl', ErrStat2, ErrMsg2 )
         if (Failed())  return;

         ! Allocate memory for depth-based coefficient arrays
      ALLOCATE ( InputFileData%Morison%CoefDpthsCyl(InputFileData%Morison%NCoefDpthCyl), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for CoefDpthsCyl array.'
         if (Failed())  return;
      END IF
                  
      DO I = 1,InputFileData%Morison%NCoefDpthCyl
         CALL ParseRAryWKywrd( FileInfo_In, CurLine, ' CoefDpthsCyl coefficients table row '//trim( Int2LStr(I)), tmpReArray, size(tmpReArray), &
                         'MCF', 1.0_ReKi, (/6,7/), InputFileData%Morison%CoefDpthsCyl(I)%DpthMCF, ErrStat2, ErrMsg2, UnEc )
            if (Failed())  return

         InputFileData%Morison%CoefDpthsCyl(I)%Dpth         = tmpReArray( 1)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthCd       = tmpReArray( 2)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthCdMG     = tmpReArray( 3)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthCa       = tmpReArray( 4)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthCaMG     = tmpReArray( 5)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthCp       = tmpReArray( 6)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthCpMG     = tmpReArray( 7)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthAxCd     = tmpReArray( 8)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthAxCdMG   = tmpReArray( 9)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthAxCa     = tmpReArray(10)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthAxCaMG   = tmpReArray(11)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthAxCp     = tmpReArray(12)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthAxCpMG   = tmpReArray(13)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthCb       = tmpReArray(14)
         InputFileData%Morison%CoefDpthsCyl(I)%DpthCbMG     = tmpReArray(15)
      END DO
      
      DO I = 2,InputFileData%Morison%NCoefDpthCyl
         IF (InputFileData%Morison%CoefDpthsCyl(I)%DpthMCF .NEQV. InputFileData%Morison%CoefDpthsCyl(1)%DpthMCF) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2 = 'In the depth-based cylindrical member hydrodynamic coefficients, MCF is specified for some depth but not others.'
            if (Failed()) RETURN
         END IF
      END DO

      if (allocated(tmpReArray))      deallocate(tmpReArray)
   END IF

   !-------------------------------------------------------------------------------------------------
   ! Depth-based Rectangular Member Hydrodynamic Coefficients Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! NCoefDpthRec - Number of depth-based rectangular member hydrodynamic coefficient property sets
   call ParseVar( FileInfo_In, CurLine, 'NCoefDpthRec', InputFileData%Morison%NCoefDpthRec, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Depth-based rectangular member hydrodynamic coefficients table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Depth-based rectangular member hydrodynamic coefficients table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1

   IF ( InputFileData%Morison%NCoefDpthRec > 0 ) THEN

      CALL AllocAry( tmpReArray, 19, 'temporary array for CoefDpthsRec', ErrStat2, ErrMsg2 )
         if (Failed())  return;

         ! Allocate memory for depth-based coefficient arrays
      ALLOCATE ( InputFileData%Morison%CoefDpthsRec(InputFileData%Morison%NCoefDpthRec), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for CoefDpthsRec array.'
         if (Failed())  return;
      END IF
                  
      DO I = 1,InputFileData%Morison%NCoefDpthRec
         CALL ParseRAryWKywrd( FileInfo_In, CurLine, ' CoefDpthsRec coefficients table row '//trim( Int2LStr(I)), tmpReArray, size(tmpReArray), &
                         'MCF', 1.0_ReKi, (/10,11/), InputFileData%Morison%CoefDpthsRec(I)%DpthMCF, ErrStat2, ErrMsg2, UnEc )
            if (Failed())  return


         InputFileData%Morison%CoefDpthsRec(I)%Dpth         = tmpReArray( 1)
         InputFileData%Morison%CoefDpthsRec(I)%DpthCdA      = tmpReArray( 2)
         InputFileData%Morison%CoefDpthsRec(I)%DpthCdAMG    = tmpReArray( 3)
         InputFileData%Morison%CoefDpthsRec(I)%DpthCdB      = tmpReArray( 4)
         InputFileData%Morison%CoefDpthsRec(I)%DpthCdBMG    = tmpReArray( 5)
         InputFileData%Morison%CoefDpthsRec(I)%DpthCaA      = tmpReArray( 6)
         InputFileData%Morison%CoefDpthsRec(I)%DpthCaAMG    = tmpReArray( 7)
         InputFileData%Morison%CoefDpthsRec(I)%DpthCaB      = tmpReArray( 8)
         InputFileData%Morison%CoefDpthsRec(I)%DpthCaBMG    = tmpReArray( 9)
         InputFileData%Morison%CoefDpthsRec(I)%DpthCp       = tmpReArray(10)
         InputFileData%Morison%CoefDpthsRec(I)%DpthCpMG     = tmpReArray(11)
         InputFileData%Morison%CoefDpthsRec(I)%DpthAxCd     = tmpReArray(12)
         InputFileData%Morison%CoefDpthsRec(I)%DpthAxCdMG   = tmpReArray(13)
         InputFileData%Morison%CoefDpthsRec(I)%DpthAxCa     = tmpReArray(14)
         InputFileData%Morison%CoefDpthsRec(I)%DpthAxCaMG   = tmpReArray(15)
         InputFileData%Morison%CoefDpthsRec(I)%DpthAxCp     = tmpReArray(16)
         InputFileData%Morison%CoefDpthsRec(I)%DpthAxCpMG   = tmpReArray(17)
         InputFileData%Morison%CoefDpthsRec(I)%DpthCb       = tmpReArray(18)
         InputFileData%Morison%CoefDpthsRec(I)%DpthCbMG     = tmpReArray(19)
      END DO
      
      DO I = 2,InputFileData%Morison%NCoefDpthRec
         IF (InputFileData%Morison%CoefDpthsRec(I)%DpthMCF .NEQV. InputFileData%Morison%CoefDpthsRec(1)%DpthMCF) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2 = 'In the depth-based rectangular member hydrodynamic coefficients, MCF is specified for some depth but not others.'
            if (Failed()) RETURN
         END IF
      END DO

      if (allocated(tmpReArray))      deallocate(tmpReArray)
   END IF

   !-------------------------------------------------------------------------------------------------
   ! Member-based Cylindrical Member Hydrodynamic Coefficients Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! NCoefMembersCyl - Number of member-based cylindrical member hydrodynamic coefficient property sets
   call ParseVar( FileInfo_In, CurLine, 'NCoefMembersCyl', InputFileData%Morison%NCoefMembersCyl, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Member-based cylindrical member  hydrodynamic coefficients table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Member-based cylindrical member  hydrodynamic coefficients table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1

   IF ( InputFileData%Morison%NCoefMembersCyl > 0 ) THEN

      CALL AllocAry( tmpReArray, 29, 'temporary array for CoefMembersCyl', ErrStat2, ErrMsg2 )
         if (Failed())  return;

         ! Allocate memory for Member-based coefficient arrays
      ALLOCATE ( InputFileData%Morison%CoefMembersCyl(InputFileData%Morison%NCoefMembersCyl), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for CoefMembersCyl array.'
         if (Failed())  return;
      END IF

      DO I = 1,InputFileData%Morison%NCoefMembersCyl
            
         CALL ParseRAryWKywrd( FileInfo_In, CurLine, 'Member-based cylindrical member  hydrodynamic coefficients table row '//trim( Int2LStr(I)), tmpReArray, size(tmpReArray), &
                      'MCF', 1.0_ReKi, (/10,11,12,13/), InputFileData%Morison%CoefMembersCyl(I)%MemberMCF, ErrStat2, ErrMsg2, UnEc )
            if (Failed())  return

         InputFileData%Morison%CoefMembersCyl(I)%MemberID         = NINT(tmpReArray( 1))
         InputFileData%Morison%CoefMembersCyl(I)%MemberCd1        =      tmpReArray( 2)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCd2        =      tmpReArray( 3)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCdMG1      =      tmpReArray( 4)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCdMG2      =      tmpReArray( 5)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCa1        =      tmpReArray( 6)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCa2        =      tmpReArray( 7)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCaMG1      =      tmpReArray( 8)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCaMG2      =      tmpReArray( 9)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCp1        =      tmpReArray(10)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCp2        =      tmpReArray(11)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCpMG1      =      tmpReArray(12)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCpMG2      =      tmpReArray(13)
         InputFileData%Morison%CoefMembersCyl(I)%MemberAxCd1      =      tmpReArray(14)
         InputFileData%Morison%CoefMembersCyl(I)%MemberAxCd2      =      tmpReArray(15)
         InputFileData%Morison%CoefMembersCyl(I)%MemberAxCdMG1    =      tmpReArray(16)
         InputFileData%Morison%CoefMembersCyl(I)%MemberAxCdMG2    =      tmpReArray(17)
         InputFileData%Morison%CoefMembersCyl(I)%MemberAxCa1      =      tmpReArray(18)
         InputFileData%Morison%CoefMembersCyl(I)%MemberAxCa2      =      tmpReArray(19)
         InputFileData%Morison%CoefMembersCyl(I)%MemberAxCaMG1    =      tmpReArray(20)
         InputFileData%Morison%CoefMembersCyl(I)%MemberAxCaMG2    =      tmpReArray(21)
         InputFileData%Morison%CoefMembersCyl(I)%MemberAxCp1      =      tmpReArray(22)
         InputFileData%Morison%CoefMembersCyl(I)%MemberAxCp2      =      tmpReArray(23)
         InputFileData%Morison%CoefMembersCyl(I)%MemberAxCpMG1    =      tmpReArray(24)
         InputFileData%Morison%CoefMembersCyl(I)%MemberAxCpMG2    =      tmpReArray(25)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCb1        =      tmpReArray(26)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCb2        =      tmpReArray(27)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCbMG1      =      tmpReArray(28)
         InputFileData%Morison%CoefMembersCyl(I)%MemberCbMG2      =      tmpReArray(29)
      END DO

      if (allocated(tmpReArray))      deallocate(tmpReArray)
   END IF

   !-------------------------------------------------------------------------------------------------
   ! Member-based Rectangular Member Hydrodynamic Coefficients Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! NCoefMembersRec - Number of member-based rectangular member hydrodynamic coefficient property sets
   call ParseVar( FileInfo_In, CurLine, 'NCoefMembersRec', InputFileData%Morison%NCoefMembersRec, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Member-based rectangular member hydrodynamic coefficients table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Member-based rectangular member hydrodynamic coefficients table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1

   IF ( InputFileData%Morison%NCoefMembersRec > 0 ) THEN

      CALL AllocAry( tmpReArray, 37, 'temporary array for CoefMembersRec', ErrStat2, ErrMsg2 )
         if (Failed())  return;

         ! Allocate memory for Member-based coefficient arrays
      ALLOCATE ( InputFileData%Morison%CoefMembersRec(InputFileData%Morison%NCoefMembersRec), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for CoefMembersRec array.'
         if (Failed())  return;
      END IF

      DO I = 1,InputFileData%Morison%NCoefMembersRec
            
         CALL ParseRAryWKywrd( FileInfo_In, CurLine, 'Member-based rectangular member hydrodynamic coefficients table row '//trim( Int2LStr(I)), tmpReArray, size(tmpReArray), &
                      'MCF', 1.0_ReKi, (/18,19,20,21/), InputFileData%Morison%CoefMembersRec(I)%MemberMCF, ErrStat2, ErrMsg2, UnEc )
            if (Failed())  return

         InputFileData%Morison%CoefMembersRec(I)%MemberID         = NINT(tmpReArray( 1))
         InputFileData%Morison%CoefMembersRec(I)%MemberCdA1       =      tmpReArray( 2)
         InputFileData%Morison%CoefMembersRec(I)%MemberCdA2       =      tmpReArray( 3)
         InputFileData%Morison%CoefMembersRec(I)%MemberCdAMG1     =      tmpReArray( 4)
         InputFileData%Morison%CoefMembersRec(I)%MemberCdAMG2     =      tmpReArray( 5)
         InputFileData%Morison%CoefMembersRec(I)%MemberCdB1       =      tmpReArray( 6)
         InputFileData%Morison%CoefMembersRec(I)%MemberCdB2       =      tmpReArray( 7)
         InputFileData%Morison%CoefMembersRec(I)%MemberCdBMG1     =      tmpReArray( 8)
         InputFileData%Morison%CoefMembersRec(I)%MemberCdBMG2     =      tmpReArray( 9)
         InputFileData%Morison%CoefMembersRec(I)%MemberCaA1       =      tmpReArray(10)
         InputFileData%Morison%CoefMembersRec(I)%MemberCaA2       =      tmpReArray(11)
         InputFileData%Morison%CoefMembersRec(I)%MemberCaAMG1     =      tmpReArray(12)
         InputFileData%Morison%CoefMembersRec(I)%MemberCaAMG2     =      tmpReArray(13)
         InputFileData%Morison%CoefMembersRec(I)%MemberCaB1       =      tmpReArray(14)
         InputFileData%Morison%CoefMembersRec(I)%MemberCaB2       =      tmpReArray(15)
         InputFileData%Morison%CoefMembersRec(I)%MemberCaBMG1     =      tmpReArray(16)
         InputFileData%Morison%CoefMembersRec(I)%MemberCaBMG2     =      tmpReArray(17)
         InputFileData%Morison%CoefMembersRec(I)%MemberCp1        =      tmpReArray(18)
         InputFileData%Morison%CoefMembersRec(I)%MemberCp2        =      tmpReArray(19)
         InputFileData%Morison%CoefMembersRec(I)%MemberCpMG1      =      tmpReArray(20)
         InputFileData%Morison%CoefMembersRec(I)%MemberCpMG2      =      tmpReArray(21)
         InputFileData%Morison%CoefMembersRec(I)%MemberAxCd1      =      tmpReArray(22)
         InputFileData%Morison%CoefMembersRec(I)%MemberAxCd2      =      tmpReArray(23)
         InputFileData%Morison%CoefMembersRec(I)%MemberAxCdMG1    =      tmpReArray(24)
         InputFileData%Morison%CoefMembersRec(I)%MemberAxCdMG2    =      tmpReArray(25)
         InputFileData%Morison%CoefMembersRec(I)%MemberAxCa1      =      tmpReArray(26)
         InputFileData%Morison%CoefMembersRec(I)%MemberAxCa2      =      tmpReArray(27)
         InputFileData%Morison%CoefMembersRec(I)%MemberAxCaMG1    =      tmpReArray(28)
         InputFileData%Morison%CoefMembersRec(I)%MemberAxCaMG2    =      tmpReArray(29)
         InputFileData%Morison%CoefMembersRec(I)%MemberAxCp1      =      tmpReArray(30)
         InputFileData%Morison%CoefMembersRec(I)%MemberAxCp2      =      tmpReArray(31)
         InputFileData%Morison%CoefMembersRec(I)%MemberAxCpMG1    =      tmpReArray(32)
         InputFileData%Morison%CoefMembersRec(I)%MemberAxCpMG2    =      tmpReArray(33)
         InputFileData%Morison%CoefMembersRec(I)%MemberCb1        =      tmpReArray(34)
         InputFileData%Morison%CoefMembersRec(I)%MemberCb2        =      tmpReArray(35)
         InputFileData%Morison%CoefMembersRec(I)%MemberCbMG1      =      tmpReArray(36)
         InputFileData%Morison%CoefMembersRec(I)%MemberCbMG2      =      tmpReArray(37)
      END DO

      if (allocated(tmpReArray))      deallocate(tmpReArray)
   END IF

   !-------------------------------------------------------------------------------------------------
   ! Members Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))   ! Write section break to echo
   CurLine = CurLine + 1

      ! NMembers - Number of members in the input file
   call ParseVar( FileInfo_In, CurLine, 'NMembers', InputFileData%Morison%NMembers, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Members table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Members table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1

   IF ( InputFileData%Morison%NMembers > 0 ) THEN

         ! Allocate memory for Members arrays
      ALLOCATE ( InputFileData%Morison%InpMembers(InputFileData%Morison%NMembers), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN         
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for InpMembers array.'
         if (Failed())  return;
      END IF

      DO I = 1,InputFileData%Morison%NMembers
         ! We can't use the ParseAry here since PropPot is a logical
         Line = FileInfo_In%Lines(CurLine)
         READ(Line,*,IOSTAT=ErrStat2) InputFileData%Morison%InpMembers(I)%MemberID,    InputFileData%Morison%InpMembers(I)%MJointID1,    &
                                      InputFileData%Morison%InpMembers(I)%MJointID2,   InputFileData%Morison%InpMembers(I)%MPropSetID1,  &
                                      InputFileData%Morison%InpMembers(I)%MPropSetID2, InputFileData%Morison%InpMembers(I)%MSecGeom,     &
                                      InputFileData%Morison%InpMembers(I)%MSpinOrient, InputFileData%Morison%InpMembers(I)%MDivSize,     &
                                      InputFileData%Morison%InpMembers(I)%MCoefMod,    InputFileData%Morison%InpMembers(I)%MHstLMod,     &
                                      InputFileData%Morison%InpMembers(I)%PropPot
         IF ( ErrStat2 /= 0 ) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2  = 'Error reading members table row '//trim( Int2LStr(I))//', line '  &
                        //trim( Int2LStr(FileInfo_In%FileLine(CurLine)))//' of file '//trim(FileInfo_In%FileList(FileInfo_In%FileIndx(CurLine)))
            if (Failed())  return;
         END IF
         InputFileData%Morison%InpMembers(I)%MSpinOrient = InputFileData%Morison%InpMembers(I)%MSpinOrient * D2R

         if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))     ! Echo this line
         CurLine = CurLine+1
      END DO

   END IF


   !-------------------------------------------------------------------------------------------------
   ! Filled Members Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))   ! Write section break to echo
   CurLine = CurLine + 1

      ! NFillGroups - Number of fill groups
   call ParseVar( FileInfo_In, CurLine, 'NFillGroups', InputFileData%Morison%NFillGroups, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Fill groups table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Fill groups table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1

   IF ( InputFileData%Morison%NFillGroups > 0 ) THEN

         ! Allocate memory for filled group arrays
      ALLOCATE ( InputFileData%Morison%FilledGroups(InputFileData%Morison%NFillGroups), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for FilledGroups array.'
         if (Failed())  return;
      END IF

      DO I = 1,InputFileData%Morison%NFillGroups
         ! We can't use the ParseAry here since the number of entries is indicated by the first entry 
         Line = FileInfo_In%Lines(CurLine)

         READ(Line,*,IOSTAT=ErrStat2) InputFileData%Morison%FilledGroups(I)%FillNumM
         IF ( ErrStat2 /= 0 ) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2  = 'Failed to read FillNumM.'
            if (Failed())  return;
         END IF

         ALLOCATE ( InputFileData%Morison%FilledGroups(I)%FillMList(InputFileData%Morison%FilledGroups(I)%FillNumM), STAT = ErrStat2 )
         IF ( ErrStat2 /= 0 ) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2  = 'Error allocating space for FillMList array.'
            if (Failed())  return;
         END IF

         READ(Line,*,IOSTAT=ErrStat2) InputFileData%Morison%FilledGroups(I)%FillNumM,  InputFileData%Morison%FilledGroups(I)%FillMList,   &
                                      InputFileData%Morison%FilledGroups(I)%FillFSLoc, InputFileData%Morison%FilledGroups(I)%FillDensChr

         IF ( ErrStat2 /= 0 ) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2  = 'Failed to read filled group properties.'
            if (Failed())  return;
         END IF

         if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))     ! Echo this line
         CurLine = CurLine+1
      END DO

   END IF


   !-------------------------------------------------------------------------------------------------
   ! Marine Growth by Depth Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! NMGDepths - Number marine growth depths
   call ParseVar( FileInfo_In, CurLine, 'NMGDepths', InputFileData%Morison%NMGDepths, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Marine growth by depth table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Marine growth by depth table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1

   IF ( InputFileData%Morison%NMGDepths > 0 ) THEN
      CALL AllocAry( tmpReArray, 3, 'temporary array for marine growth table', ErrStat2, ErrMsg2 )
         if (Failed())  return;

         ! Allocate memory for marine growth depths array
      ALLOCATE ( InputFileData%Morison%MGDepths(InputFileData%Morison%NMGDepths), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for MGDepths array.'
         if (Failed())  return;
      END IF

      DO I = 1,InputFileData%Morison%NMGDepths
         call ParseAry( FileInfo_In, CurLine, ' Marine growth table row '//trim( Int2LStr(I)), tmpReArray, size(tmpReArray), ErrStat2, ErrMsg2, UnEc )
         InputFileData%Morison%MGDepths(I)%MGDpth  = tmpReArray(1)
         InputFileData%Morison%MGDepths(I)%MGThck  = tmpReArray(2)
         InputFileData%Morison%MGDepths(I)%MGDens  = tmpReArray(3)
      END DO

      if (allocated(tmpReArray))      deallocate(tmpReArray)
   END IF


   !-------------------------------------------------------------------------------------------------
   ! Member Output List Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! NMOutputs - Number of members to output
   call ParseVar( FileInfo_In, CurLine, 'NMOutputs', InputFileData%Morison%NMOutputs, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

      ! Table header
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Member output list table header line 1: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') 'Member output list table header line 2: '//NewLine//trim(FileInfo_In%Lines(CurLine))
   CurLine = CurLine + 1

   IF ( InputFileData%Morison%NMOutputs > 0 ) THEN

         ! Allocate memory for filled group arrays
      ALLOCATE ( InputFileData%Morison%MOutLst(InputFileData%Morison%NMOutputs), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for MOutLst array.'
         if (Failed())  return;
      END IF      


      DO I = 1,InputFileData%Morison%NMOutputs

         ! We can't use the ParseAry here since the number of entries is indicated by the first entry 
         Line = FileInfo_In%Lines(CurLine)

         READ(Line,*,IOSTAT=ErrStat2) InputFileData%Morison%MOutLst(I)%MemberID, InputFileData%Morison%MOutLst(I)%NOutLoc
         IF ( ErrStat2 /= 0 ) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2  = 'Failed to read NOutLoc.'
            if (Failed())  return;
         END IF      
         
         ALLOCATE ( InputFileData%Morison%MOutLst(I)%NodeLocs(InputFileData%Morison%MOutLst(I)%NOutLoc), STAT = ErrStat2 )
         IF ( ErrStat2 /= 0 ) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2  = 'Error allocating space for NodeLocs array.'
            if (Failed())  return;
         END IF      

         ALLOCATE ( InputFileData%Morison%MOutLst(I)%MeshIndx1(InputFileData%Morison%MOutLst(I)%NOutLoc), STAT = ErrStat2 )
         IF ( ErrStat2 /= 0 ) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2  = 'Error allocating space for %MeshIndx1 array.'
            if (Failed())  return;
         END IF

         ALLOCATE ( InputFileData%Morison%MOutLst(I)%MemberIndx1(InputFileData%Morison%MOutLst(I)%NOutLoc), STAT = ErrStat2 )
         IF ( ErrStat2 /= 0 ) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2  = 'Error allocating space for %MemberIndx1 array.'
            if (Failed())  return;
         END IF

         ALLOCATE ( InputFileData%Morison%MOutLst(I)%MeshIndx2(InputFileData%Morison%MOutLst(I)%NOutLoc), STAT = ErrStat2 )
         IF ( ErrStat2 /= 0 ) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2  = 'Error allocating space for %MeshIndx2 array.'
            if (Failed())  return;
         END IF      

         ALLOCATE ( InputFileData%Morison%MOutLst(I)%MemberIndx2(InputFileData%Morison%MOutLst(I)%NOutLoc), STAT = ErrStat2 )
         IF ( ErrStat2 /= 0 ) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2  = 'Error allocating space for %MemberIndx2 array.'
            if (Failed())  return;
         END IF    

         ALLOCATE ( InputFileData%Morison%MOutLst(I)%s(InputFileData%Morison%MOutLst(I)%NOutLoc), STAT = ErrStat2 )
         IF ( ErrStat2 /= 0 ) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2  = 'Error allocating space for s array.'
            if (Failed())  return;
         END IF      

         READ(Line,*,IOSTAT=ErrStat2) InputFileData%Morison%MOutLst(I)%MemberID,  InputFileData%Morison%MOutLst(I)%NOutLoc,  &
                                      InputFileData%Morison%MOutLst(I)%NodeLocs

         IF ( ErrStat2 /= 0 ) THEN
            ErrStat2 = ErrID_Fatal
            ErrMsg2  = 'Failed to read member output list properties.'
            if (Failed())  return;
         END IF

         if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))     ! Echo this line
         CurLine = CurLine+1
      END DO

   END IF


   !-------------------------------------------------------------------------------------------------
   ! Joint Output List Section
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

      ! NJOutputs - Number of joints to output
   call ParseVar( FileInfo_In, CurLine, 'NJOutputs', InputFileData%Morison%NJOutputs, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

   IF ( InputFileData%Morison%NJOutputs > 0 ) THEN

      ALLOCATE ( InputFileData%Morison%JOutLst(InputFileData%Morison%NJOutputs), STAT = ErrStat2 )
      IF ( ErrStat2 /= 0 ) THEN
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = 'Error allocating space for JOutLst data structures.'
         if (Failed())  return;
      END IF      

      CALL AllocAry( tmpArray, InputFileData%Morison%NJOutputs, 'temporary array for Joint outputs', ErrStat2, ErrMsg2 )
         if (Failed())  return;

      call ParseAry( FileInfo_In, CurLine, 'JOutLst table row '//trim( Int2LStr(I)), tmpArray, InputFileData%Morison%NJOutputs, ErrStat2, ErrMsg2, UnEc )
         if (Failed())  return;

      DO I = 1,InputFileData%Morison%NJOutputs
         InputFileData%Morison%JOutLst(I)%JointID = tmpArray(I)
      END DO

      DEALLOCATE(tmpArray)   
      
   ELSE
      
      ! There are no Joint Outputs, but there is a line here.  We don't parse it since we don't want to error on it
      if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write line to echo
      CurLine = CurLine + 1

   END IF
   

   !-------------------------------------------------------------------------------------------------
   ! Data section for OUTPUT
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1

         ! HDSum - Whether or not to generate a summary file
   call ParseVar( FileInfo_In, CurLine, 'HDSum', InputFileData%HDSum, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

         ! OutAll - Whether or not to output information for every member and joint
   call ParseVar( FileInfo_In, CurLine, 'OutAll', InputFileData%OutAll, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

         ! OutSwtch - Specify how to write to an output file
   call ParseVar( FileInfo_In, CurLine, 'OutSwtch', InputFileData%OutSwtch, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

        ! OutFmt - Format for numerical outputs
   call ParseVar( FileInfo_In, CurLine, 'OutFmt', InputFileData%OutFmt, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;

         ! OutSFmt - Format for output column headers
   call ParseVar( FileInfo_In, CurLine, 'OutSFmt', InputFileData%OutSFmt, ErrStat2, ErrMsg2, UnEc )
      if (Failed())  return;


   !-------------------------------------------------------------------------------------------------
   ! Data section for FLOATING PLATFORM OUTPUTS
   !-------------------------------------------------------------------------------------------------
   if ( InputFileData%Echo )   WRITE(UnEc, '(A)') trim(FileInfo_In%Lines(CurLine))    ! Write section break to echo
   CurLine = CurLine + 1
      
      ! OutList - list of requested parameters to output to a file
   call AllocAry( InputFileData%UserOutputs, MaxUserOutputs, 'InputFileData%UserOutputs', ErrStat2, ErrMsg2 )  ! MaxUserOutputs is set in registry 
      if (Failed())  return;
   
   call ReadOutputListFromFileInfo( FileInfo_In, CurLine, InputFileData%UserOutputs, InputFileData%NUserOutputs, ErrStat2, ErrMsg2, UnEc )
         if (Failed()) return;

   
   !-------------------------------------------------------------------------------------------------
   ! This is the end of the input file
   !-------------------------------------------------------------------------------------------------
   
   CALL Cleanup()

   RETURN

CONTAINS
   
   SUBROUTINE ParseRAryWKywrd( FileInfo, LineNum, AryName, Ary, AryLen, Kywrd, KywrdVal, KywrdEntry, HasKywrd, ErrStat, ErrMsg, UnEc )
    
      ! Arguments declarations.
      INTEGER,             INTENT(IN)             :: AryLen                        !< The length of the array to parse.
      TYPE (FileInfoType), INTENT(IN)             :: FileInfo                      !< The derived type for holding the file information.
      INTEGER(IntKi),      INTENT(INOUT)          :: LineNum                       !< The number of the line to parse.
      CHARACTER(*),        INTENT(IN)             :: AryName                       !< The array name we are trying to fill.
      REAL(ReKi),          INTENT(OUT)            :: Ary(AryLen)                   !< The array to receive the input values.
      CHARACTER(*),        INTENT(IN)             :: Kywrd                         !< The keyword to look for
      REAL(ReKi),          INTENT(IN)             :: KywrdVal                      !< Value to be used when the keyword is encountered
      INTEGER(IntKi),      INTENT(IN)             :: KywrdEntry(:)                 !< Entries where the provided keyword is allowed
      LOGICAL,             INTENT(OUT)            :: HasKywrd                      !< T/F to indicate whether keyword is present
      INTEGER(IntKi),      INTENT(OUT)            :: ErrStat                       !< The error status.
      CHARACTER(*),        INTENT(OUT)            :: ErrMsg                        !< The error message, if ErrStat /= 0.
      INTEGER,             INTENT(IN), OPTIONAL   :: UnEc                          !< I/O unit for echo file. If present and > 0, write to UnEc.

      ! Local declarations.
      INTEGER(IntKi)                         :: i,j                           ! Local counter.
      CHARACTER(25), ALLOCATABLE             :: tmpChrArray(:)                ! Temporary character array storage
      
      CHARACTER(*), PARAMETER                :: RoutineName = 'ParseRAryWKywrd'

      hasKywrd = .FALSE.
      ErrStat = ErrID_None
      ErrMsg  = ""
       
      CALL AllocAry( tmpChrArray, AryLen, 'temporary array for ParseRAryWKywrd', ErrStat, ErrMsg )
         IF (ErrStat /= 0) THEN
            ErrStat = ErrID_Fatal
            ErrMsg = 'Error allocating temporary array for ParseRAryWKywrd ' // ' when parsing ' // AryName
            RETURN
         END IF
      
      CALL ParseAry( FileInfo, LineNum, AryName, tmpChrArray, size(tmpChrArray), ErrStat, ErrMsg, UnEc )
         IF (ErrStat /= 0) THEN
            ErrStat = ErrID_Fatal
            ErrMsg = 'Error parsing ' // AryName
            RETURN
         END IF
      
      DO j = 1,size(KywrdEntry)
         i = KywrdEntry(j)
         IF ( TRIM(tmpChrArray(i)) == Kywrd ) THEN
            hasKywrd = .TRUE.
         END IF
      END DO
      
      IF ( hasKywrd ) THEN
         DO j = 1,size(KywrdEntry)
            i = KywrdEntry(j)  
            IF ( TRIM(tmpChrArray(i)) == Kywrd ) THEN
               tmpChrArray(i) = Num2Lstr(KywrdVal)
            ELSE 
               ErrStat = ErrID_Fatal
               ErrMsg  = 'When parsing ' // AryName // ', ' // kywrd // ' is used at some but not all relevant places.'
               RETURN
            END IF
         END DO
      END IF
      
      DO i=1,AryLen
         READ(tmpChrArray(i),*,IOSTAT=ErrStat)   Ary(i)
         IF (ErrStat /= 0) THEN
            ErrStat = ErrID_Fatal
            ErrMsg  = 'When parsing ' // AryName // ', nonnumerical entry is encountered where numerical entry is expected.'
            RETURN;
         END IF
      END DO
      
      IF (ALLOCATED(tmpChrArray))   DEALLOCATE(tmpChrArray)
      
   END SUBROUTINE ParseRAryWKywrd
   
   
   !..............................
   logical function Failed()
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
      Failed = ErrStat >= AbortErrLev
      if (Failed)    call Cleanup()
   end function Failed
   SUBROUTINE Cleanup()
      IF (ALLOCATED(tmpArray  )) DEALLOCATE(tmpArray  )
      IF (ALLOCATED(tmpReArray)) DEALLOCATE(tmpReArray)
      IF (ALLOCATED(tmpVec1   )) DEALLOCATE(tmpVec1   )
      IF (ALLOCATED(tmpVec2   )) DEALLOCATE(tmpVec2   )
         ! Cleanup the Echo file and global variables
      if (UnEc > 0)  close ( UnEc )
   END SUBROUTINE Cleanup
END SUBROUTINE HydroDyn_ParseInput



  

!====================================================================================================
SUBROUTINE HydroDynInput_ProcessInitData( InitInp, Interval, InputFileData, ErrStat, ErrMsg )
!     This private subroutine verifies the input required for HydroDyn is correctly specified.
!----------------------------------------------------------------------------------------------------


      ! Passed variables

   TYPE(HydroDyn_InitInputType),  INTENT( IN    )   :: InitInp              ! the hydrodyn data
   REAL(DbKi),                    INTENT( IN    )   :: Interval             ! The DT supplied by the glue code/driver
   TYPE(HydroDyn_InputFile),      INTENT( INOUT )   :: InputFileData        ! the hydrodyn input file data
   INTEGER,                       INTENT(   OUT )   :: ErrStat              ! returns a non-zero value when an error occurs
   CHARACTER(*),                  INTENT(   OUT )   :: ErrMsg               ! Error message if ErrStat /= ErrID_None

   INTEGER                                          :: I                    ! Generic loop counter index
   INTEGER                                          :: J                    ! Generic loop counter index
   INTEGER                                          :: K                    ! Generic loop counter index
   CHARACTER(1024)                                  :: TmpPath              ! Temporary storage for relative path name
   LOGICAL                                          :: FoundID              ! Boolean flag indicating whether an ID from one tables is found in one of the other input table
   REAL(ReKi)                                       :: MinCylDepth          ! The minimum depth entry in the Depth-based cylindrical-member Hydrodynamic coefficents table
   REAL(ReKi)                                       :: MaxCylDepth          ! The maximum depth entry in the Depth-based cylindrical-member Hydrodynamic coefficents table
   REAL(ReKi)                                       :: MinRecDepth          ! The minimum depth entry in the Depth-based rectangular-member Hydrodynamic coefficents table
   REAL(ReKi)                                       :: MaxRecDepth          ! The maximum depth entry in the Depth-based rectangular-member Hydrodynamic coefficents table
   REAL(ReKi)                                       :: z1
   REAL(ReKi)                                       :: z2
   REAL(ReKi)                                       :: MinMembrDpth
   REAL(ReKi)                                       :: MaxMembrDpth
!   CHARACTER(ChanLen), ALLOCATABLE                       :: tmpOutLst(:)         !
   CHARACTER(3)                                     :: TmpExtension         ! Temporary variable for holding the file extension for 10d, 11d, 12d, 10s, 11s, 12s WAMIT files
   LOGICAL                                          :: TmpFileExist         ! Temporary variable in checking the existance of an input file.
   LOGICAL                                          :: JointUsed
   REAL(ReKi)                                       :: l
   REAL(ReKi)                                       :: lvec(3)
   LOGICAL, ALLOCATABLE                             :: foundMask(:)
   
   INTEGER(IntKi)                                   :: ErrStat2, IOS
   CHARACTER(ErrMsgLen)                             :: ErrMsg2
   CHARACTER(*), PARAMETER                          :: RoutineName = 'HydroDynInput_ProcessInitData'
      ! Initialize ErrStat
         
   ErrStat = ErrID_None         
   ErrStat2 = ErrID_None
   ErrMsg  = ""    
   ErrMsg2  = ""
      
     
      !-------------------------------------------------------------------------
      ! Check environmental conditions
      !-------------------------------------------------------------------------
   if (.not. associated(InitInp%WaveField)) then
      call SetErrStat( ErrID_Fatal,' No SeaState information available.',ErrStat,ErrMsg,RoutineName)
      return
   endif
   
   if (InitInp%WaveField%NStepWave == 0) then
      call SetErrStat( ErrID_Fatal,' No SeaState information available.',ErrStat,ErrMsg,RoutineName)
      return
   endif

      ! MSL2SWL - Mean sea level to still water level
   IF ( InputFileData%PotMod == 1 .AND. .NOT. EqualRealNos(InitInp%WaveField%MSL2SWL, 0.0_ReKi) ) THEN
      CALL SetErrStat( ErrID_Fatal,'SeaState MSL2SWL must be 0 when PotMod = 1 (WAMIT).',ErrStat,ErrMsg,RoutineName)        
      RETURN
   END IF
   

      ! WaveMod - Wave kinematics model switch.
   IF ( InputFileData%PotMod > 0 .and. InitInp%WaveField%WaveMod == WaveMod_ExtFull ) THEN
         CALL SetErrStat( ErrID_Fatal,'WaveMod cannot be 6 when PotMod is not 0.',ErrStat,ErrMsg,RoutineName)
         RETURN
   END IF

      ! Linearization Checks
   ! LIN-TODO:
   !errors if:
   !if (                                                                   &
   !     (InputFileData%PotMod /= 0 .or. InputFileData%PotMod /=1)                .or. &
   !     (InputFileData%WAMIT%ExctnMod /=0 .or. InputFileData%WAMIT%ExctnMod /=2) .or. &
   !     (InputFileData%WAMIT%RdtnMod  /=0 .or. InputFileData%WAMIT%RdtnMod  /=2) .or. &
   !     (InputFileData%WAMIT2%MnDrift /=0)                                 .or. &
   !     (InputFileData%WAMIT2%NewmanApp /= 0)                              .or. &
   !     (InputFileData%WAMIT2%SumQTF /= 0 )                                     ) then
   !   
   !end if

       ! PotFile - Root name of potential flow files

   IF ( InputFileData%PotMod > 0 ) THEN
      do i = 1,InputFileData%nWAMITObj
          IF ( LEN_TRIM( InputFileData%PotFile(i) ) == 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'PotFile must not be an empty string.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

            ! if this is a relative path, let's make it relative to the location of the main input file
            ! tell the WAMIT and WAMIT2 modules what the filename is

         IF ( PathIsRelative( InputFileData%PotFile(i) ) ) THEN
            CALL GetPath( TRIM(InitInp%InputFile), TmpPath )
            InputFileData%PotFile(i)            = TRIM(TmpPath)//TRIM(InputFileData%PotFile(i))
         END IF
      end do

   !TODO: Move this to where the WAMIT modules are initialized
      InputFileData%WAMIT%WAMITFile    = InputFileData%PotFile(1)
      InputFileData%WAMIT2%WAMITFile   = InputFileData%PotFile(1)
      
   ELSE
      InputFileData%PotFile            = ""
      InputFileData%WAMIT%WAMITFile    = ""
      InputFileData%WAMIT2%WAMITFile   = ""  
      ! These can be set to zero because they are only used if PotMod = 1
      InputFileData%WAMIT%ExctnMod     = 0  
      InputFileData%WAMIT%RdtnMod      = 0
   END IF

      ! Set the WAMIT file name on the Convolution module
   InputFileData%WAMIT%Conv_Rdtn%WAMITFile = InputFileData%WAMIT%WAMITFile

      ! WAMITULEN - WAMIT characteristic body length scale

   IF ( InputFileData%PotMod == 1 ) THEN
!TODO: Deal with WAMIT2 and check each WAMITULEN not just the first
      InputFileData%WAMIT2%WAMITULEN = InputFileData%WAMITULEN(1)    ! Copy to the WAMIT2 module info
      do i = 1,InputFileData%nWAMITObj
         IF ( InputFileData%WAMITULEN(i) < 0.0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'WAMITULEN must be positive.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
      end do
   ELSE

      InputFileData%WAMITULEN = 1.0
      InputFileData%WAMIT2%WAMITULEN = 1.0

   END IF
   
      ! ExctnDisp - Method of computing Wave Excitation
   if ( InputFileData%PotMod /= 1 .or. InputFileData%WAMIT%ExctnMod == 0 .or. InitInp%WaveField%WaveMod == WaveMod_None) then
      InputFileData%WAMIT%ExctnDisp    = 0  !Force ExctnDisp = 0, so that the Grid of Wave Excitation forces is not computed (saves time and memory)
   end if
   
   ! ExctnCutOff
   if ( InputFileData%PotMod == 1 .and. InputFileData%WAMIT%ExctnMod  > 0 .and. InputFileData%WAMIT%ExctnDisp == 2 .and. InputFileData%WAMIT%ExctnCutOff <= 0.0 ) then
      CALL SetErrStat( ErrID_Fatal,'ExctnCutOff must be greater than zero.',ErrStat,ErrMsg,RoutineName)
   end if   
      
      ! PtfmVol0 - Displaced volume of water when the platform is in its undisplaced position

   IF ( InputFileData%PotMod == 1 ) THEN
      do i = 1,InputFileData%nWAMITObj
         IF ( InputFileData%PtfmVol0(i) < 0.0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'PtfmVol0 must not be negative.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
      end do
   ELSE

      InputFileData%PtfmVol0 = 0.0

   END IF

      ! PtfmRefzt - The zt offset of the body reference point(s) from (0,0,0) (meters) [1 to NBody] 
      ! NOTE: only used when PotMod=1. If NBodyMod=2,PtfmRefzt=0.0

   IF ( InputFileData%PotMod == 1 .and. InputFileData%NBodyMod == 2) THEN
      do i = 1,InputFileData%NBody
         IF ( .not. EqualRealNos( InputFileData%PtfmRefzt(i), 0.0_ReKi ) )THEN
            CALL SetErrStat( ErrID_Fatal,'PtfmRefzt must be 0.0 for all WAMIT bodies when NBodyMod=2.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
      end do
   END IF

      ! RdtnTMax - Analysis time for wave radiation kernel calculations
      ! NOTE: Use RdtnTMax = 0.0 to eliminate wave radiation damping

   IF ( InputFileData%PotMod == 1 ) THEN

      IF ( InputFileData%WAMIT%RdtnTMax < 0.0 ) THEN
         CALL SetErrStat( ErrID_Fatal,'RdtnTMax must not be negative.',ErrStat,ErrMsg,RoutineName)
         RETURN
      END IF

   ELSE

      InputFileData%WAMIT%RdtnTMax = 0.0

   END IF

       ! RdtnDT - Time step for wave radiation kernel calculations

   IF ( InputFileData%PotMod == 1 ) THEN

      CALL Conv2UC( InputFileData%WAMIT%Conv_Rdtn%RdtnDTChr )    ! Convert Line to upper case.
      
      IF ( TRIM(InputFileData%WAMIT%Conv_Rdtn%RdtnDTChr) == 'DEFAULT' )  THEN   ! .TRUE. when one wants to use the default value timestep provided by the glue code.

         InputFileData%WAMIT%Conv_Rdtn%RdtnDT = Interval 

      ELSE                                   ! The input must have been specified numerically.

         READ (InputFileData%WAMIT%Conv_Rdtn%RdtnDTChr,*,IOSTAT=IOS)  InputFileData%WAMIT%Conv_Rdtn%RdtnDT
            CALL CheckIOS ( IOS, "", 'RdtnDT', NumType, ErrStat2, ErrMsg2 )
            CALL SetErrStat(ErrStat2, ErrMsg2,ErrStat,ErrMsg,RoutineName)
            IF ( ErrStat >= AbortErrLev ) RETURN
         
      END IF

      IF ( InputFileData%WAMIT%Conv_Rdtn%RdtnDT <= 0.0 ) THEN
         CALL SetErrStat( ErrID_Fatal,'RdtnDT must be greater than zero.',ErrStat,ErrMsg,RoutineName)
         RETURN
      END IF

      if ( (.not. ( EqualRealNos(Interval, InputFileData%WAMIT%Conv_Rdtn%RdtnDT) ) ) .and. ( (InputFileData%WAMIT%ExctnMod > 1) .or. (InputFileData%WAMIT%RdtnMod > 0) ) ) then
         call SetErrStat( ErrID_Fatal,'RdtnDT must be equal to the glue-code DT if PotMod = 1 and using RdtnMod > 0 or ExctnMod > 1.',ErrStat,ErrMsg,RoutineName)
         return
      end if
      
   ELSE

      InputFileData%WAMIT%Conv_Rdtn%RdtnDT = 0.0

   END IF


   !-------------------------------------------------------------------------------------------------
   ! Second order Forces due to Waves section (WAMIT2 Module)
   !-------------------------------------------------------------------------------------------------
   
      ! Check that we only specified one of MnDrift, NewmanApp, or DiffQTF
      !        (compared pairwise -- if any two are both true, we have a problem)
   IF ( ( InputFileData%WAMIT2%MnDrift /= 0 .AND. InputFileData%WAMIT2%NewmanApp /= 0 ) .OR. &
        ( InputFileData%WAMIT2%DiffQTF /= 0 .AND. InputFileData%WAMIT2%NewmanApp /= 0 ) .OR. &
        ( InputFileData%WAMIT2%MnDrift /= 0 .AND. InputFileData%WAMIT2%DiffQTF   /= 0 ) ) THEN
      CALL SetErrStat( ErrID_Fatal,'Only one of MnDrift, NewmanApp, or DiffQTF can be non-zero.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF


   if ( InputFileData%NBody > 1 .and. InputFileData%WAMIT2%MnDrift == 8 ) then
      call SetErrStat( ErrID_Fatal,'MnDrift cannot equal 8 when NBody > 1.',ErrStat,ErrMsg,RoutineName)
      return
   end if

   if ( InputFileData%NBody > 1 .and. InputFileData%WAMIT2%NewmanApp == 8 ) then
      call SetErrStat( ErrID_Fatal,'NewmanApp cannot equal 8 when NBody > 1.',ErrStat,ErrMsg,RoutineName)
      return
   end if
   

      ! Check MnDrift and set the flag indicating WAMIT2 should perform the mean drift calculation.
      ! Also make sure we have a valid input value for the file extension

   IF ( InputFileData%WAMIT2%MnDrift == 0 ) THEN      ! not using MnDrift
      InputFileData%WAMIT2%MnDriftF = .FALSE.
   ELSE IF ( InputFileData%WAMIT2%MnDrift == 7  .OR. InputFileData%WAMIT2%MnDrift == 8  .OR. InputFileData%WAMIT2%MnDrift == 9 .OR. &
             InputFileData%WAMIT2%MnDrift == 10 .OR. InputFileData%WAMIT2%MnDrift == 11 .OR. InputFileData%WAMIT2%MnDrift == 12 ) THEN   ! Valid values for MnDrift
      IF ( InputFileData%PotMod /= 1 ) THEN
         CALL SetErrStat( ErrID_warn,'MnDrift can only be used with PotMod==1.  Turning off',ErrStat,ErrMsg,RoutineName)
         InputFileData%WAMIT2%MnDriftF = .FALSE.
      ELSE
         InputFileData%WAMIT2%MnDriftF = .TRUE.
      ENDIF
   ELSE     ! Must have received an invalid value
      CALL SetErrStat( ErrID_Fatal,'MnDrift can only have values of 0, 7, 8, 9, 10, 11, or 12.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF


      ! Check NewmanApp and set the flag indicating WAMIT2 should perform the mean drift calculation.
      ! Also make sure we have a valid input value for the file extension

   IF ( InputFileData%WAMIT2%NewmanApp == 0 ) THEN    ! not using NewmanApp
      InputFileData%WAMIT2%NewmanAppF = .FALSE.
   ELSE IF ( InputFileData%WAMIT2%NewmanApp == 7  .OR. InputFileData%WAMIT2%NewmanApp == 8  .OR. InputFileData%WAMIT2%NewmanApp == 9 .OR. &
             InputFileData%WAMIT2%NewmanApp == 10 .OR. InputFileData%WAMIT2%NewmanApp == 11 .OR. InputFileData%WAMIT2%NewmanApp == 12 ) THEN ! Valid values for NewmanApp
      IF ( InputFileData%PotMod /= 1 ) THEN
         CALL SetErrStat( ErrID_warn,'NewmanApp can only be used with PotMod==1.  Turning off',ErrStat,ErrMsg,RoutineName)
         InputFileData%WAMIT2%NewmanAppF = .FALSE.
      ELSE
         InputFileData%WAMIT2%NewmanAppF = .TRUE.
      ENDIF
   ELSE     ! Must have received an invalid value
      CALL SetErrStat( ErrID_Fatal,'NewmanApp can only have values of 0, 7, 8, 9, 10, 11, or 12.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF


      ! Check DiffQTF and set the flag indicating WAMIT2 should perform the mean drift calculation.
      ! Also make sure we have a valid input value for the file extension

   IF ( InputFileData%WAMIT2%DiffQTF == 0 ) THEN      ! not using DiffQTF method
       InputFileData%WAMIT2%DiffQTFF = .FALSE.
   ELSE IF ( InputFileData%WAMIT2%DiffQTF == 10 .OR. InputFileData%WAMIT2%DiffQTF == 11 .OR. InputFileData%WAMIT2%DiffQTF == 12 ) THEN    ! Valid values for DiffQTF
      IF ( InputFileData%PotMod /= 1 ) THEN
         CALL SetErrStat( ErrID_warn,'DiffQTF can only be used with PotMod==1.  Turning off',ErrStat,ErrMsg,RoutineName)
         InputFileData%WAMIT2%DiffQTFF = .FALSE.
      ELSE
         InputFileData%WAMIT2%DiffQTFF = .TRUE.
      ENDIF
   ELSE
      CALL SetErrStat( ErrID_Fatal,'DiffQTF can only have values of 0, 10, 11, or 12.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF


      ! Check SumQTF and set the flag indicating WAMIT2 should perform the mean drift calculation.
      ! Also make sure we have a valid input value for the file extension

   IF ( InputFileData%WAMIT2%SumQTF == 0 ) THEN       ! not using SumQTF method
      InputFileData%WAMIT2%SumQTFF = .FALSE.
   ELSE IF ( InputFileData%WAMIT2%SumQTF == 10 .OR. InputFileData%WAMIT2%SumQTF == 11 .OR. InputFileData%WAMIT2%SumQTF == 12 ) THEN       ! Valid values for SumQTF
      IF ( InputFileData%PotMod /= 1 ) THEN
         CALL SetErrStat( ErrID_warn,'SumQTF can only be used with PotMod==1.  Turning off',ErrStat,ErrMsg,RoutineName)
         InputFileData%WAMIT2%SumQTFF = .FALSE.
      ELSE
         InputFileData%WAMIT2%SumQTFF = .TRUE.
      ENDIF
   ELSE
      CALL SetErrStat( ErrID_Fatal,'SumQTF can only have values of 0, 10, 11, or 12.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF



      ! now that it has been established that the input parameters for second order are good, we check to make sure that the WAMIT files actually exist.
      ! Check MnDrift file
   IF ( InputFileData%WAMIT2%MnDriftF ) THEN
      ! Check if using QTF file types (10d, 11d, 12d) or not (7,8,9)
      IF ( InputFileData%WAMIT2%MnDrift <= 9 ) THEN
         TmpExtension = TRIM(Num2LStr(InputFileData%WAMIT2%MnDrift))
         INQUIRE( file=TRIM(InputFileData%WAMIT2%WAMITFile)//'.'//TRIM(TmpExtension), exist=TmpFileExist )
      ELSE  ! 10, 11, 12
         TmpExtension = TRIM(Num2LStr(InputFileData%WAMIT2%MnDrift))//'d'
         INQUIRE( file=TRIM(InputFileData%WAMIT2%WAMITFile)//'.'//TRIM(TmpExtension), exist=TmpFileExist )
      ENDIF
      IF ( .not. TmpFileExist ) THEN
         CALL SetErrStat( ErrID_Fatal,'Cannot find the WAMIT file '//TRIM(InputFileData%WAMIT2%WAMITFile)//'.'//TRIM(TmpExtension)// &
                    ' required by the MnDrift option.',ErrStat,ErrMsg,RoutineName)
         RETURN
      END IF
   END IF

      ! Check existence of NewmanApp file
   IF ( InputFileData%WAMIT2%NewmanAppF ) THEN
      ! Check if using QTF file types (10d, 11d, 12d) or not (7,8,9)
      IF ( InputFileData%WAMIT2%NewmanApp <= 9 ) THEN
         TmpExtension = TRIM(Num2LStr(InputFileData%WAMIT2%NewmanApp))
         INQUIRE( file=TRIM(InputFileData%WAMIT2%WAMITFile)//'.'//TRIM(TmpExtension), exist=TmpFileExist )
      ELSE  ! 10, 11, 12
         TmpExtension = TRIM(Num2LStr(InputFileData%WAMIT2%NewmanApp))//'d'
         INQUIRE( file=TRIM(InputFileData%WAMIT2%WAMITFile)//'.'//TRIM(TmpExtension), exist=TmpFileExist )
      ENDIF
      IF ( .not. TmpFileExist ) THEN
         CALL SetErrStat( ErrID_Fatal,'Cannot find the WAMIT file '//TRIM(InputFileData%WAMIT2%WAMITFile)//'.'//TRIM(TmpExtension)// &
                    ' required by the NewmanApp option.',ErrStat,ErrMsg,RoutineName)
         RETURN
      END IF
   END IF

   IF ( InputFileData%WAMIT2%DiffQTFF ) THEN
      TmpExtension = TRIM(Num2LStr(InputFileData%WAMIT2%DiffQTF))//'d'
      INQUIRE( file=TRIM(InputFileData%WAMIT2%WAMITFile)//'.'//TRIM(TmpExtension), exist=TmpFileExist )
      IF ( .not. TmpFileExist ) THEN
         CALL SetErrStat( ErrID_Fatal,'Cannot find the WAMIT file '//TRIM(InputFileData%WAMIT2%WAMITFile)//'.'//TRIM(TmpExtension)// &
                    ' required by the DiffQTF option.',ErrStat,ErrMsg,RoutineName)
         RETURN
      END IF
   END IF

   IF ( InputFileData%WAMIT2%SumQTFF ) THEN
      TmpExtension = TRIM(Num2LStr(InputFileData%WAMIT2%SumQTF))//'s'
      INQUIRE( file=TRIM(InputFileData%WAMIT2%WAMITFile)//'.'//TRIM(TmpExtension), exist=TmpFileExist )
      IF ( .not. TmpFileExist ) THEN
         CALL SetErrStat( ErrID_Fatal,'Cannot find the WAMIT file '//TRIM(InputFileData%WAMIT2%WAMITFile)//'.'//TRIM(TmpExtension)// &
                    ' required by the SumQTF option.',ErrStat,ErrMsg,RoutineName)
         RETURN
      END IF
   END IF

   !..................
   ! check for ExctnMod = 2 requirements
   !..................
   if ( (InputFileData%WAMIT%ExctnMod == 2) ) then

      if ( InitInp%InvalidWithSSExctn ) then
         call SetErrStat( ErrID_Fatal, 'Given SeaState conditions cannot be used with state-space wave excitations. In SeaState, WaveMod cannot be 6; WaveDirMod must be 0; WvDiffQTF must be FALSE; and WvSumQTF must be FALSE. Or in HydroDyn set ExctnMod to 0 or 1.', ErrStat, ErrMsg, RoutineName )
      end if

      if ( InputFileData%PotMod /= 1 ) then
         call SetErrStat( ErrID_Fatal, 'Potential-flow model via WAMIT must be used with state-space wave excitations. Set PotMod= 1.', ErrStat, ErrMsg, RoutineName )
      end if
      
      if ( InputFileData%WAMIT2%MnDrift /= 0 ) then
         call SetErrStat( ErrID_Fatal, 'Mean-drift 2nd-order forces cannot be used with state-space wave excitations. Set MnDrift=0.', ErrStat, ErrMsg, RoutineName )
      end if

      if ( InputFileData%WAMIT2%NewmanApp /= 0 ) then
         call SetErrStat( ErrID_Fatal, "Mean- and slow-drift 2nd-order forces computed with Newman's approximation cannot be used with state-space wave excitations. Set NewmanApp=0.", ErrStat, ErrMsg, RoutineName )
      end if
      
      if ( InputFileData%WAMIT2%DiffQTF /= 0 ) then
         call SetErrStat( ErrID_Fatal, 'Full difference-frequency 2nd-order forces computed with full QTF cannot be used with state-space wave excitations. Set DiffQTF=0.', ErrStat, ErrMsg, RoutineName )
      end if

      if ( InputFileData%WAMIT2%SumQTF /= 0 ) then
         call SetErrStat( ErrID_Fatal, 'Full summation-frequency 2nd-order forces computed with full QTF cannot be used with State-space wave excitations. Set SumQTF=0.', ErrStat, ErrMsg, RoutineName )
      end if

   end if

   !..................
   ! check for linearization
   !..................
   if (InitInp%Linearize) then
      
      if ( InputFileData%PotMod > 1 ) then
         call SetErrStat( ErrID_Fatal, 'Potential-flow model cannot be set to FIT for linearization. Set PotMod= 0 or 1.', ErrStat, ErrMsg, RoutineName )
      end if
      
      if ( (InputFileData%WAMIT%ExctnMod == 1) ) then
         call SetErrStat( ErrID_Fatal, 'Cannot set wave excitation model to DFT for linearization. Set ExctnMod=0 or 2.', ErrStat, ErrMsg, RoutineName )
      end if

      if ( InputFileData%WAMIT%RdtnMod == 1 ) then
         call SetErrStat( ErrID_Fatal, 'Cannot set wave radiation model to convolution for linearization. Set RdtnMod=0 or 2.', ErrStat, ErrMsg, RoutineName )
      end if
      
      if ( InputFileData%WAMIT2%MnDrift /= 0 ) then
         call SetErrStat( ErrID_Fatal, 'Mean-drift 2nd-order forces cannot be used for linearization. Set MnDrift=0.', ErrStat, ErrMsg, RoutineName )
      end if

      if ( InputFileData%WAMIT2%NewmanApp /= 0 ) then
         call SetErrStat( ErrID_Fatal, "Mean- and slow-drift 2nd-order forces computed with Newman's approximation cannot be used for linearization. Set NewmanApp=0.", ErrStat, ErrMsg, RoutineName )
      end if
      
      if ( InputFileData%WAMIT2%DiffQTF /= 0 ) then
         call SetErrStat( ErrID_Fatal, 'Full difference-frequency 2nd-order forces computed with full QTF cannot be used for linearization. Set DiffQTF=0.', ErrStat, ErrMsg, RoutineName )
      end if

      if ( InputFileData%WAMIT2%SumQTF /= 0 ) then
         call SetErrStat( ErrID_Fatal, 'Full summation-frequency 2nd-order forces computed with full QTF cannot be used for linearization. Set SumQTF=0.', ErrStat, ErrMsg, RoutineName )
      end if

   end if

   !-------------------------------------------------------------------------------------------------
   ! Strip Theory Options Section
   !-------------------------------------------------------------------------------------------------

   IF ( InputFileData%Morison%WaveDisp /= 0 .AND. InputFileData%Morison%WaveDisp /= 1) THEN
      CALL SetErrStat( ErrID_Fatal,'WaveDisp must be 0 or 1',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF
   IF ( InputFileData%Morison%AMMod /= 0 .AND. InputFileData%Morison%AMMod /= 1) THEN
      CALL SetErrStat( ErrID_Fatal,'AMMod must be 0 or 1',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF
  

   !-------------------------------------------------------------------------------------------------
   ! Member Joints Section
   !-------------------------------------------------------------------------------------------------

   IF ( InputFileData%Morison%NJoints < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'NJoints parameter cannot be negative.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

   IF ( InputFileData%Morison%NJoints == 1 ) THEN
      CALL SetErrStat( ErrID_Fatal,'NJoints parameter cannot be set to 1.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

     
     
      ! Check the axial coefs are >= 0 and IDs are unique
   IF ( InputFileData%Morison%NAxCoefs > 0 ) THEN
   
      DO I = 1,InputFileData%Morison%NAxCoefs 

         IF (  InputFileData%Morison%AxialCoefs(I)%AxCd < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'AxCd must be greater than or equal to zero.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF   
         IF (  InputFileData%Morison%AxialCoefs(I)%AxCa < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'AxCa must be greater than or equal to zero.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF   
         IF (  InputFileData%Morison%AxialCoefs(I)%AxCp < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'AxCp must be greater than or equal to zero.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
         IF (  InputFileData%Morison%AxialCoefs(I)%AxFDMod /= 0_IntKi .AND. InputFileData%Morison%AxialCoefs(I)%AxFDMod /= 1_IntKi ) THEN
            CALL SetErrStat( ErrID_Fatal,'AxFDMod must be 0 or 1.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
         IF (  InputFileData%Morison%AxialCoefs(I)%AxFDLoFSc < 0_ReKi .OR. InputFileData%Morison%AxialCoefs(I)%AxFDLoFSc > 1_ReKi ) THEN
            CALL SetErrStat( ErrID_Fatal,'AxFDLoFSc must be between 0 and 1 inclusive.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
         
            ! Make sure that the current AxCoefID is not used elsewhere in the table.
         DO J = I+1,InputFileData%Morison%NAxCoefs
            IF ( InputFileData%Morison%AxialCoefs(I)%AxCoefID == InputFileData%Morison%AxialCoefs(J)%AxCoefID ) THEN
               CALL SetErrStat( ErrID_Fatal,'Duplicate AxCoefIDs were found in the Axial Coefficients table.',ErrStat,ErrMsg,RoutineName)
               RETURN
            END IF
         END DO
   
      END DO
      
   END IF


      ! Check JointOvrlp values
  !NOTE: This is ignored in the current version of Morison.  3/15/2020 GJH
   
   IF ( InputFileData%Morison%NJoints > 1 ) THEN

      ! Initialize Joints
      DO I = 1,InputFileData%Morison%NJoints
         InputFileData%Morison%InpJoints(I)%NConnections   = 0
      END DO

      
      
      
      DO I = 1,InputFileData%Morison%NJoints

            ! Make sure that the current JointID is not used elsewhere in the table.
         DO J = I+1,InputFileData%Morison%NJoints
            IF ( InputFileData%Morison%InpJoints(I)%JointID == InputFileData%Morison%InpJoints(J)%JointID ) THEN
               CALL SetErrStat( ErrID_Fatal,'Duplicate JointIDs were found in the Member Joints table.',ErrStat,ErrMsg,RoutineName)
               RETURN
            END IF
         END DO

            ! Add up total number of joints flagged with JoinOvrlp = 1 option
         !IF ( InputFileData%Morison%InpJoints(I)%JointOvrlp == 1 ) THEN
         !   InputFileData%Morison%TotalPossibleSuperMembers = InputFileData%Morison%TotalPossibleSuperMembers + 1
         !END IF

            ! Check that every joint id is used at least once in the members table
         JointUsed = .FALSE.
         DO J = 1, InputFileData%Morison%NMembers
         
            IF ( InputFileData%Morison%InpMembers(J)%MJointID1 == InputFileData%Morison%InpJoints(I)%JointID ) THEN
               JointUsed = .TRUE.
               EXIT
            END IF
            IF ( InputFileData%Morison%InpMembers(J)%MJointID2 == InputFileData%Morison%InpJoints(I)%JointID ) THEN
               JointUsed = .TRUE.
               EXIT
            END IF
         END DO
         
         IF ( .NOT. JointUsed ) THEN
            CALL SetErrStat( ErrID_Fatal,'Every JointID in the Joints table must appear once in the Members table.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF  
   ! TODO : Implement Super member elements. GJH 7/24/13
   
         IF ( InputFileData%Morison%InpJoints(I)%JointOvrlp /= 0  ) THEN
            CALL SetErrStat( ErrID_Fatal,'JointOvrlp parameter must be set to 0.  Future versions of HydroDyn will support vales of 0 or 1.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
         !IF ( ( InputFileData%Morison%InpJoints(I)%JointOvrlp < 0 ) .OR. ( InputFileData%Morison%InpJoints(I)%JointOvrlp > 1 ) ) THEN
         !   ErrMsg  = ' JointOvrlp parameter must be set to 0 or 1.'
         !   ErrStat = ErrID_Fatal
         !   RETURN
         !END IF
         
            ! Make sure the axial coef id appears in the Ax table
         IF ( InputFileData%Morison%NAxCoefs > 0 ) THEN
            InputFileData%Morison%InpJoints(I)%JointAxIDIndx = -1
            DO J = 1,InputFileData%Morison%NAxCoefs         
               IF ( InputFileData%Morison%InpJoints(I)%JointAxID == InputFileData%Morison%AxialCoefs(J)%AxCoefID ) &
                  InputFileData%Morison%InpJoints(I)%JointAxIDIndx = J   
            END DO
            IF ( InputFileData%Morison%InpJoints(I)%JointAxIDIndx == -1 ) THEN
               CALL SetErrStat( ErrID_Fatal,'The specified JointAxID in the Joints Table does not appear in the Axial Coefficients table.',ErrStat,ErrMsg,RoutineName)
               RETURN
            END IF
         ELSE
            ! TODO: Issue error because we need at least one Axial coef table entry GJH  8/1/31
         END IF
      
      END DO
   END IF


   !-------------------------------------------------------------------------------------------------
   ! Member Cross-section Properties Section
   !-------------------------------------------------------------------------------------------------

   IF ( InputFileData%Morison%NPropSetsCyl < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'Number of cylindrical member cross-section property sets must be greater than zero.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

   IF ( InputFileData%Morison%NPropSetsCyl > 0 ) THEN

      DO I = 1,InputFileData%Morison%NPropSetsCyl

            ! Make sure that the current JointID is not used elsewhere in the table.
         DO J = I+1,InputFileData%Morison%NPropSetsCyl
            IF ( InputFileData%Morison%MPropSetsCyl(I)%PropSetID == InputFileData%Morison%MPropSetsCyl(J)%PropSetID ) THEN
               CALL SetErrStat( ErrID_Fatal,'Duplicate PropSetIDs were found in the Cylindrical Member Cross-section Properties table.',ErrStat,ErrMsg,RoutineName)
               RETURN
            END IF
         END DO

         IF ( ( InputFileData%Morison%MPropSetsCyl(I)%PropD < 0 ) .OR.  ( InputFileData%Morison%MPropSetsCyl(I)%PropThck < 0 ) .OR. ( ( InputFileData%Morison%MPropSetsCyl(I)%PropD - 2.0 * InputFileData%Morison%MPropSetsCyl(I)%PropThck ) < 0) ) THEN
            CALL SetErrStat( ErrID_Fatal,'For cylindrical members, PropD and PropThck must be greater than zero and (PropD - 2*PropThck) must be greater than zero.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
      END DO

   END IF

   IF ( InputFileData%Morison%NPropSetsRec < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'Number of rectangular member cross-section property sets must be greater than zero.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

   IF ( InputFileData%Morison%NPropSetsRec > 0 ) THEN

      DO I = 1,InputFileData%Morison%NPropSetsRec

            ! Make sure that the current JointID is not used elsewhere in the table.
         DO J = I+1,InputFileData%Morison%NPropSetsRec
            IF ( InputFileData%Morison%MPropSetsRec(I)%PropSetID == InputFileData%Morison%MPropSetsRec(J)%PropSetID ) THEN
               CALL SetErrStat( ErrID_Fatal,'Duplicate PropSetIDs were found in the Rectangular Member Cross-section Properties table.',ErrStat,ErrMsg,RoutineName)
               RETURN
            END IF
         END DO

         IF ( ( InputFileData%Morison%MPropSetsRec(I)%PropA < 0 ) .OR. ( InputFileData%Morison%MPropSetsRec(I)%PropB < 0 ) .OR.  ( InputFileData%Morison%MPropSetsRec(I)%PropThck < 0 ) .OR. ( ( InputFileData%Morison%MPropSetsRec(I)%PropA - 2.0 * InputFileData%Morison%MPropSetsRec(I)%PropThck ) < 0) .OR. ( ( InputFileData%Morison%MPropSetsRec(I)%PropB - 2.0 * InputFileData%Morison%MPropSetsRec(I)%PropThck ) < 0) ) THEN
            CALL SetErrStat( ErrID_Fatal,'For rectangular members, PropA, PropB, and PropThck must be greater than zero and (PropA - 2*PropThck) and (PropB - 2*PropThck) must be greater than zero.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
      END DO

   END IF

   !-------------------------------------------------------------------------------------------------
   ! Simple hydrodynamic coefficients Section
   !-------------------------------------------------------------------------------------------------

   IF ( InputFileData%Morison%SimplCd < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplCd must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplCdMG   < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplCdMG must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplCa     < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplCa must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplCaMG   < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplCaMG must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplAxCd   < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplAxCd must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplAxCdMG < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplAxCdMG must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplAxCa   < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplAxCa must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplAxCaMG < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplAxCaMG must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplCb     < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplCb must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplCbMG   < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplCbMG must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
   END IF
   IF (ErrStat >= AbortErrLev) RETURN

   IF ( InputFileData%Morison%SimplRecCdA < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecCdA must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecCdAMG  < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecCdAMG must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecCdB    < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecCdB must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecCdBMG  < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecCdBMG must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecCaA    < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecCaA must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecCaAMG  < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecCaAMG must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecCaB    < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecCaB must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecCaBMG  < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecCaBMG must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecAxCd   < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecAxCd must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecAxCdMG < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecAxCdMG must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecAxCa   < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecAxCa must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecAxCaMG < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecAxCaMG must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecCb     < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecCb must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
   ELSE IF ( InputFileData%Morison%SimplRecCbMG   < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'SimplRecCbMG must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
   END IF
   IF (ErrStat >= AbortErrLev) RETURN

   !TODO: Do we need a test for AxCp

   !-------------------------------------------------------------------------------------------------
   ! Depth-based Hydrodynamic Coefficients Section
   !-------------------------------------------------------------------------------------------------

   IF ( InputFileData%Morison%NCoefDpthCyl < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'NCoefDpthCyl must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

   IF ( InputFileData%Morison%NCoefDpthCyl > 0 ) THEN
      MinCylDepth =  99999999.0
      MaxCylDepth = -99999999.0
      DO I = 1,InputFileData%Morison%NCoefDpthCyl

            ! Record the minimum and maximum depths covered by this table.  This will be used as part of a consistency check
            ! in the members table, below.
         IF (  InputFileData%Morison%CoefDpthsCyl(I)%Dpth < MinCylDepth ) THEN
            MinCylDepth = InputFileData%Morison%CoefDpthsCyl(I)%Dpth
         ELSE
            CALL SetErrStat( ErrID_Fatal,'The rows of the Depth-based Hydrodynamic Coefficients table must be ordered with increasing depth (decreasing Z).',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
         IF ( InputFileData%Morison%CoefDpthsCyl(I)%Dpth > MaxCylDepth ) THEN
            MaxCylDepth = InputFileData%Morison%CoefDpthsCyl(I)%Dpth
         END IF

            ! Make sure that the current Dpth is not used elsewhere in the table.
         DO J = I+1,InputFileData%Morison%NCoefDpthCyl
            IF ( EqualRealNos( InputFileData%Morison%CoefDpthsCyl(I)%Dpth, InputFileData%Morison%CoefDpthsCyl(J)%Dpth ) ) THEN
               CALL SetErrStat( ErrID_Fatal,'Duplicate Dpths were found in the Depth-based Hydrodynamic Coefficients table for cylindrical members.',ErrStat,ErrMsg,RoutineName)
               RETURN
            END IF
         END DO

         IF ( InputFileData%Morison%CoefDpthsCyl(I)%DpthCd < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for cylindrical members, DpthCd must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsCyl(I)%DpthCdMG   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for cylindrical members, DpthCdMG must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsCyl(I)%DpthCa     < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for cylindrical members, DpthCa must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsCyl(I)%DpthCaMG   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for cylindrical members, DpthCaMG must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsCyl(I)%DpthAxCd   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for cylindrical members, DpthAxCd must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsCyl(I)%DpthAxCdMG < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for cylindrical members, DpthAxCdMG must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsCyl(I)%DpthAxCa   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for cylindrical members, DpthAxCa must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsCyl(I)%DpthAxCaMG < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for cylindrical members, DpthAxCaMG must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsCyl(I)%DpthAxCp   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for cylindrical members, DpthAxCp must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsCyl(I)%DpthAxCpMG < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for cylindrical members, DpthAxCpMG must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsCyl(I)%DpthCb     < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for cylindrical members, DpthCb must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsCyl(I)%DpthCbMG   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for cylindrical members, DpthCbMG must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         END IF
         IF (ErrStat >= AbortErrLev) RETURN

      END DO

      ! TODO: Sort the table based on depth so that a linear interpolation can be easily performed between entries.

   END IF

   IF ( InputFileData%Morison%NCoefDpthRec < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'NCoefDpthRec must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

   IF ( InputFileData%Morison%NCoefDpthRec > 0 ) THEN
      MinRecDepth =  99999999.0
      MaxRecDepth = -99999999.0
      DO I = 1,InputFileData%Morison%NCoefDpthRec

            ! Record the minimum and maximum depths covered by this table.  This will be used as part of a consistency check
            ! in the members table, below.
         IF (  InputFileData%Morison%CoefDpthsRec(I)%Dpth < MinRecDepth ) THEN
            MinRecDepth = InputFileData%Morison%CoefDpthsRec(I)%Dpth
         ELSE
            CALL SetErrStat( ErrID_Fatal,'The rows of the Depth-based Hydrodynamic Coefficients table must be ordered with increasing depth (decreasing Z).',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
         IF ( InputFileData%Morison%CoefDpthsRec(I)%Dpth > MaxRecDepth ) THEN
            MaxRecDepth = InputFileData%Morison%CoefDpthsRec(I)%Dpth
         END IF

            ! Make sure that the current Dpth is not used elsewhere in the table.
         DO J = I+1,InputFileData%Morison%NCoefDpthRec
            IF ( EqualRealNos( InputFileData%Morison%CoefDpthsRec(I)%Dpth, InputFileData%Morison%CoefDpthsRec(J)%Dpth ) ) THEN
               CALL SetErrStat( ErrID_Fatal,'Duplicate Dpths were found in the Depth-based Hydrodynamic Coefficients table for rectangular members.',ErrStat,ErrMsg,RoutineName)
               RETURN
            END IF
         END DO

         IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthCdA < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthCdA must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthCdAMG  < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthCdAMG must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthCdB    < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthCdB must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthCdBMG  < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthCdBMG must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthCaA    < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthCa must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthCaAMG  < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthCaMG must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthCaB    < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthCa must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthCaBMG  < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthCaMG must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthAxCd   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthAxCd must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthAxCdMG < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthAxCdMG must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthAxCa   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthAxCa must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthAxCaMG < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthAxCaMG must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthAxCp   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthAxCp must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthAxCpMG < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthAxCpMG must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthCb     < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthCb must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefDpthsRec(I)%DpthCbMG   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the Depth-based hydrodynamic coefficients table for rectangular members, DpthCbMG must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         END IF
         IF (ErrStat >= AbortErrLev) RETURN

      END DO

      ! TODO: Sort the table based on depth so that a linear interpolation can be easily performed between entries.

   END IF

   !-------------------------------------------------------------------------------------------------
   ! Member-based Hydrodynamic Coefficients Section
   !-------------------------------------------------------------------------------------------------

   IF ( InputFileData%Morison%NCoefMembersCyl < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'NCoefMembersCyl must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

   IF ( InputFileData%Morison%NCoefMembersCyl > 0 ) THEN

      DO I = 1,InputFileData%Morison%NCoefMembersCyl

            ! Make sure that the current MemberID is not used elsewhere in the table.
         DO J = I+1,InputFileData%Morison%NCoefMembersCyl
            IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberID == InputFileData%Morison%CoefMembersCyl(J)%MemberID ) THEN
               CALL SetErrStat( ErrID_Fatal,'Duplicate MemberIDs were found in the Member-based Hydrodynamic coefficients table for cylindrical members.',ErrStat,ErrMsg,RoutineName)
               RETURN
            END IF
         END DO

         IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberCd1 < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberCd1 must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberCd2     < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberCd2 must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberCdMG1   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberCdMG1 must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberCdMG2   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberCdMG2 must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberCa1     < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberCa1 must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberCa2     < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberCa2 must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberCaMG1   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberCaMG1 must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberCaMG2   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberCaMG2 must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberAxCa1   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberAxCa1 must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberAxCa2   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberAxCa2 must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberAxCaMG1 < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberAxCaMG1 must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberAxCaMG2 < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberAxCaMG2 must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberCb1     < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberCb1 must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberCb2     < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberCb2 must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberCbMG1   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberCbMG1 must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersCyl(I)%MemberCbMG2   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for cylindrical members, MemberCbMG2 must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         END IF
         IF (ErrStat >= AbortErrLev) RETURN
      END DO

   END IF

   IF ( InputFileData%Morison%NCoefMembersRec < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'NCoefMembersRec must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

   IF ( InputFileData%Morison%NCoefMembersRec > 0 ) THEN

      DO I = 1,InputFileData%Morison%NCoefMembersRec

            ! Make sure that the current MemberID is not used elsewhere in the table.
         DO J = I+1,InputFileData%Morison%NCoefMembersRec
            IF ( InputFileData%Morison%CoefMembersRec(I)%MemberID == InputFileData%Morison%CoefMembersRec(J)%MemberID ) THEN
               CALL SetErrStat( ErrID_Fatal,'Duplicate MemberIDs were found in the Member-based Hydrodynamic coefficients table for rectangular members.',ErrStat,ErrMsg,RoutineName)
               RETURN
            END IF
         END DO

         IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCdA1 < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCdA1 must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCdA2    < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCdA2 must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCdAMG1  < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCdAMG1 must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCdAMG2  < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCdAMG2 must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCdB1    < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCdB1 must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCdB2    < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCdB2 must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCdBMG1  < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCdBMG1 must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCdBMG2  < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCdBMG2 must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCaA1    < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCaA1 must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCaA2    < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCaA2 must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCaAMG1  < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCaAMG1 must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCaAMG2  < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCaAMG2 must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCaB1    < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCaB1 must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCaB2    < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCaB2 must be greater or equal to zero.'   ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCaBMG1  < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCaBMG1 must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCaBMG2  < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCaBMG2 must be greater or equal to zero.' ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberAxCa1   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberAxCa1 must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberAxCa2   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberAxCa2 must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberAxCaMG1 < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberAxCaMG1 must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberAxCaMG2 < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberAxCaMG2 must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCb1     < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCb1 must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCb2     < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCb2 must be greater or equal to zero.'    ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCbMG1   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCbMG1 must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         ELSE IF ( InputFileData%Morison%CoefMembersRec(I)%MemberCbMG2   < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'In the member-based hydrodynamic coefficients table for rectangular members, MemberCbMG2 must be greater or equal to zero.'  ,ErrStat,ErrMsg,RoutineName)
         END IF
         IF (ErrStat >= AbortErrLev) RETURN
      END DO

   END IF

   !-------------------------------------------------------------------------------------------------
   ! Members Section
   !-------------------------------------------------------------------------------------------------

   IF ( InputFileData%Morison%NMembers < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'NMembers in the Members table must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

   IF ( InputFileData%Morison%NMembers > 0 ) THEN

         ! Initialize all member data
      DO I = 1,InputFileData%Morison%NMembers
         InputFileData%Morison%InpMembers(I)%MJointID1Indx    = -1
         InputFileData%Morison%InpMembers(I)%MJointID2Indx    = -1
         InputFileData%Morison%InpMembers(I)%MPropSetID1Indx  = -1
         InputFileData%Morison%InpMembers(I)%MPropSetID2Indx  = -1
         InputFileData%Morison%InpMembers(I)%MmbrFilledIDIndx = -1
         InputFileData%Morison%InpMembers(I)%MmbrCoefIDIndx   = -1
      END DO

      DO I = 1,InputFileData%Morison%NMembers

            ! Make sure that the current MemberID is not used elsewhere in the table.
         DO J = I+1,InputFileData%Morison%NMembers
            IF ( InputFileData%Morison%InpMembers(I)%MemberID == InputFileData%Morison%InpMembers(J)%MemberID ) THEN
               CALL SetErrStat( ErrID_Fatal,'Duplicate MemberIDs were found in the Members table.',ErrStat,ErrMsg,RoutineName)
               RETURN
            END IF
         END DO

            ! Find JointID1 and JointID2 in the Joint table and then record their index locations in the Joint table
         DO J = 1,InputFileData%Morison%NJoints
            IF ( InputFileData%Morison%InpMembers(I)%MJointID1 == InputFileData%Morison%InpJoints(J)%JointID ) THEN
               InputFileData%Morison%InpMembers(I)%MJointID1Indx = J
               InputFileData%Morison%InpJoints(J)%NConnections = InputFileData%Morison%InpJoints(J)%NConnections + 1
               InputFileData%Morison%InpJoints(J)%ConnectionList(InputFileData%Morison%InpJoints(J)%NConnections) = I
            END IF
            IF ( InputFileData%Morison%InpMembers(I)%MJointID2 == InputFileData%Morison%InpJoints(J)%JointID ) THEN
               InputFileData%Morison%InpMembers(I)%MJointID2Indx = J
               InputFileData%Morison%InpJoints(J)%NConnections = InputFileData%Morison%InpJoints(J)%NConnections + 1
               InputFileData%Morison%InpJoints(J)%ConnectionList(InputFileData%Morison%InpJoints(J)%NConnections) = -I !TODO: Come up with a better method for this work GJH 4/6/20
            END IF
         END DO
         
            ! Make sure that a JointID entry in the Joints table was found
         IF ( InputFileData%Morison%InpMembers(I)%MJointID1Indx == -1 ) THEN
            CALL SetErrStat( ErrID_Fatal,'JointID1 in the Members table does not appear in the Joints table.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
         IF ( InputFileData%Morison%InpMembers(I)%MJointID2Indx == -1 ) THEN
            CALL SetErrStat( ErrID_Fatal,'JointID2 in the Members table does not appear in the Joints table.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

            ! Make sure we do not have any zero length members
         lvec = InputFileData%Morison%InpJoints(InputFileData%Morison%InpMembers(I)%MJointID1Indx)%Position - InputFileData%Morison%InpJoints(InputFileData%Morison%InpMembers(I)%MJointID2Indx)%Position
         l = sqrt( lvec(1)*lvec(1) + lvec(2)*lvec(2) + lvec(3)*lvec(3) )
         IF ( EqualRealNos(0.0_ReKi, l) ) THEN
            CALL SetErrStat( ErrID_Fatal,'A member cannot have zero length.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

            ! Find MPropSetID1 and MPropSetID2 in the Member cross-section properties table and then record their index locations
         IF ( InputFileData%Morison%InpMembers(I)%MSecGeom == MSecGeom_Cyl ) THEN
            DO J = 1,InputFileData%Morison%NPropSetsCyl
               IF ( InputFileData%Morison%InpMembers(I)%MPropSetID1 == InputFileData%Morison%MPropSetsCyl(J)%PropSetID ) THEN
                  InputFileData%Morison%InpMembers(I)%MPropSetID1Indx = J
               END IF
               IF ( InputFileData%Morison%InpMembers(I)%MPropSetID2 == InputFileData%Morison%MPropSetsCyl(J)%PropSetID ) THEN
                  InputFileData%Morison%InpMembers(I)%MPropSetID2Indx = J
               END IF
            END DO
         ELSE IF ( InputFileData%Morison%InpMembers(I)%MSecGeom == MSecGeom_Rec ) THEN
            DO J = 1,InputFileData%Morison%NPropSetsRec
               IF ( InputFileData%Morison%InpMembers(I)%MPropSetID1 == InputFileData%Morison%MPropSetsRec(J)%PropSetID ) THEN
                  InputFileData%Morison%InpMembers(I)%MPropSetID1Indx = J
               END IF
               IF ( InputFileData%Morison%InpMembers(I)%MPropSetID2 == InputFileData%Morison%MPropSetsRec(J)%PropSetID ) THEN
                  InputFileData%Morison%InpMembers(I)%MPropSetID2Indx = J
               END IF
            END DO
         END IF

            ! Make sure that a PropSetID entry in the Member cross-section properties table was found
         IF ( InputFileData%Morison%InpMembers(I)%MPropSetID1Indx == -1 ) THEN
            CALL SetErrStat( ErrID_Fatal,'MPropSetID1 in the Members table does not appear in the Member cross-section properties table.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
         IF ( InputFileData%Morison%InpMembers(I)%MPropSetID2Indx == -1 ) THEN
            CALL SetErrStat( ErrID_Fatal,'MPropSetID2 in the Members table does not appear in the Member cross-section properties table.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF


         ! NOTE: We cannot test that MDivSize > MemberLength yet because there may be a joint overlap which is going to alter the final length of this member

         IF ( InputFileData%Morison%InpMembers(I)%MDivSize <= 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'MDivSize must be greater than zero.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF


         IF ( ( InputFileData%Morison%InpMembers(I)%MCoefMod /= 1 ) .AND. ( InputFileData%Morison%InpMembers(I)%MCoefMod /= 2 ) .AND. ( InputFileData%Morison%InpMembers(I)%MCoefMod /= 3 ) )  THEN
            CALL SetErrStat( ErrID_Fatal,'MCoefMod must be 1, 2, or 3.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

         IF ( InputFileData%Morison%InpMembers(I)%MCoefMod == 2 ) THEN
            ! We will not extrapolate depth-based coefficient values, so make sure that the depth-based table has values that are outside the depth range of this member
            ! NOTE: This is actually potentially overly conservative because the final member may be shorter due to joint overlap handling.
            z1 = InputFileData%Morison%InpJoints( InputFileData%Morison%InpMembers(I)%MJointID1Indx )%Position(3)
            z2 = InputFileData%Morison%InpJoints( InputFileData%Morison%InpMembers(I)%MJointID2Indx )%Position(3)
            MinMembrDpth = min( z1, z2 )
            MaxMembrDpth = max( z1, z2 )
            IF ( InputFileData%Morison%InpMembers(I)%MSecGeom == MSecGeom_Cyl ) THEN
               IF ( InputFileData%Morison%NCoefDpthCyl == 0 ) THEN
                  CALL SetErrStat( ErrID_Fatal,'NCoefDpthCyl for cylindrical members must be greater than zero when a member is using a depth-based coefficient model.',ErrStat,ErrMsg,RoutineName)
                  RETURN
               END IF
               IF ( ( MinMembrDpth < MinCylDepth ) .OR. ( MaxMembrDpth > MaxCylDepth ) ) THEN
                  CALL SetErrStat( ErrID_Fatal,'This cylindrical member uses a depth-based coefficient model, but the member depth is outside the range of values provided in the depth-based hydrodynamic coefficients table.',ErrStat,ErrMsg,RoutineName)
                  RETURN
               END IF
            ELSE IF ( InputFileData%Morison%InpMembers(I)%MSecGeom == MSecGeom_Rec ) THEN
               IF ( InputFileData%Morison%NCoefDpthRec == 0 ) THEN
                  CALL SetErrStat( ErrID_Fatal,'NCoefDpthRec for rectangular members must be greater than zero when a member is using a depth-based coefficient model.',ErrStat,ErrMsg,RoutineName)
                  RETURN
               END IF
               IF ( ( MinMembrDpth < MinRecDepth ) .OR. ( MaxMembrDpth > MaxRecDepth ) ) THEN
                  CALL SetErrStat( ErrID_Fatal,'This rectangular member uses a depth-based coefficient model, but the member depth is outside the range of values provided in the depth-based hydrodynamic coefficients table.',ErrStat,ErrMsg,RoutineName)
                  RETURN
               END IF
            END IF
         END IF

         IF ( InputFileData%Morison%InpMembers(I)%MCoefMod == 3 ) THEN

            IF (InputFileData%Morison%InpMembers(I)%MSecGeom == MSecGeom_Cyl) THEN

               IF ( InputFileData%Morison%NCoefMembersCyl == 0 ) THEN
                  CALL SetErrStat( ErrID_Fatal,'NCoefMembersCyl must be greater than zero when a member is using a member-based coefficient model.',ErrStat,ErrMsg,RoutineName)
                  RETURN
               END IF
               ! Make sure this id appears in the Members table and mark it's location for future use
               FoundID = .FALSE.
               DO J = 1,InputFileData%Morison%NCoefMembersCyl
                  IF ( InputFileData%Morison%CoefMembersCyl(J)%MemberID == InputFileData%Morison%InpMembers(I)%MemberID ) THEN
                     FoundID = .TRUE.
                     InputFileData%Morison%InpMembers(I)%MmbrCoefIDIndx = J
                  END IF
               END DO
               IF ( .NOT. FoundID ) THEN
                  CALL SetErrStat( ErrID_Fatal,'Could not locate the MemberID referenced in the Members table in the associated Member-based Hydrodynamic coefficients table for cylindrical members.',ErrStat,ErrMsg,RoutineName)
                  RETURN
               END IF

            ELSE IF (InputFileData%Morison%InpMembers(I)%MSecGeom == MSecGeom_Rec) THEN

               IF ( InputFileData%Morison%NCoefMembersRec == 0 ) THEN
                  CALL SetErrStat( ErrID_Fatal,'NCoefMembersRec must be greater than zero when a member is using a member-based coefficient model.',ErrStat,ErrMsg,RoutineName)
                  RETURN
               END IF
               ! Make sure this id appears in the Members table and mark it's location for future use
               FoundID = .FALSE.
               DO J = 1,InputFileData%Morison%NCoefMembersRec
                  IF ( InputFileData%Morison%CoefMembersRec(J)%MemberID == InputFileData%Morison%InpMembers(I)%MemberID ) THEN
                     FoundID = .TRUE.
                     InputFileData%Morison%InpMembers(I)%MmbrCoefIDIndx = J
                  END IF
               END DO
               IF ( .NOT. FoundID ) THEN
                  CALL SetErrStat( ErrID_Fatal,'Could not locate the MemberID referenced in the Members table in the associated Member-based Hydrodynamic coefficients table for rectangular members.',ErrStat,ErrMsg,RoutineName)
                  RETURN
               END IF

            END IF

         END IF

         IF ( InputFileData%Morison%InpMembers(I)%MSecGeom /= MSecGeom_Cyl .AND. InputFileData%Morison%InpMembers(I)%MSecGeom /= MSecGeom_Rec ) THEN
            CALL SetErrStat( ErrID_Fatal,'MSecGeom must be 1 for cylindrical members or 2 for rectangular members.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

         IF ( InputFileData%Morison%InpMembers(I)%MHstLMod /= 0 .AND. InputFileData%Morison%InpMembers(I)%MHstLMod /= 1 .AND. InputFileData%Morison%InpMembers(I)%MHstLMod /= 2 ) THEN
            CALL SetErrStat( ErrID_Fatal,'MHstLMod must be 1 for column-type hydrostatic load calculation or 2 for ship-like calculation.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

         IF ( InputFileData%Morison%InpMembers(I)%MSecGeom == MSecGeom_Rec .AND. InputFileData%Morison%InpMembers(I)%MHstLMod /= 2 ) THEN
            CALL SetErrStat( ErrID_Fatal,'MHstLMod must be 2 for rectangular members.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

         IF ( InputFileData%Morison%InpMembers(I)%PropPot .AND. InputFileData%PotMod == 0  ) THEN
            CALL SetErrStat( ErrID_Fatal,'A member cannot have PropPot set to TRUE if PotMod = 0 in the FLOATING PLATFORM section.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

      END DO

   END IF


   !-------------------------------------------------------------------------------------------------
   ! Filled Members Section
   !-------------------------------------------------------------------------------------------------

   IF ( InputFileData%Morison%NFillGroups < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'NFillGroups in the Filled-members table must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

   IF ( InputFileData%Morison%NFillGroups > 0 ) THEN

      DO I = 1,InputFileData%Morison%NFillGroups

         IF ( InputFileData%Morison%FilledGroups(I)%FillNumM < 1 ) THEN
            CALL SetErrStat( ErrID_Fatal,'FillNumM in the Filled-members table must be greater than zero.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

         DO J = 1,InputFileData%Morison%FilledGroups(I)%FillNumM

            DO K=1,InputFileData%Morison%NMembers
               IF ( InputFileData%Morison%FilledGroups(I)%FillMList(J) == InputFileData%Morison%InpMembers(K)%MemberID ) THEN
                  FoundID = .TRUE.
                     ! Check to make sure this member is not already part of another fill group!
                  IF ( InputFileData%Morison%InpMembers(K)%MmbrFilledIDIndx /= -1 ) THEN
                     CALL SetErrStat( ErrID_Fatal,'A member cannot be a part of more than one fill group!',ErrStat,ErrMsg,RoutineName)
                  END IF

                  InputFileData%Morison%InpMembers(k)%MmbrFilledIDIndx = I

                  ! Set entries of member list of the filled group to the member index instead of ID for quick lookup later
                  InputFileData%Morison%FilledGroups(I)%FillMList(J) = K

               END IF
            END DO

         END DO



            ! Make sure that the filled group members are connected
            ! NOTE: This would be easier if the input mesh was already a FAST Framework mesh because then you could use the mesh routines to determine connectivity.

            !InputFileData%Morison%FilledGroups(I)%FillMList(J)

            ! Make sure the FillFSLoc is within one of the group members
            !InputFileData%Morison%FilledGroups(I)%FillFSLoc


               ! Deal with DEFAULT or create a REAL from the string

         CALL Conv2UC( InputFileData%Morison%FilledGroups(I)%FillDensChr )
         IF ( TRIM(InputFileData%Morison%FilledGroups(I)%FillDensChr) /= 'DEFAULT' )  THEN

            READ (InputFileData%Morison%FilledGroups(I)%FillDensChr,*,IOSTAT=IOS)  InputFileData%Morison%FilledGroups(I)%FillDens
               CALL CheckIOS ( IOS, "", 'FillDens', NumType, ErrStat2, ErrMsg2 )
               CALL SetErrStat(ErrStat2, ErrMsg2,ErrStat,ErrMsg,RoutineName)
               IF ( ErrStat >= AbortErrLev ) RETURN
         ELSE
            InputFileData%Morison%FilledGroups(I)%FillDens = InitInp%WaveField%WtrDens
         END IF

      END DO

   END IF


   !-------------------------------------------------------------------------------------------------
   ! Marine Growth by Depth Section
   !-------------------------------------------------------------------------------------------------

   IF ( InputFileData%Morison%NMGDepths < 0 ) THEN
      CALL SetErrStat( ErrID_Fatal,'NMGDepths in the Marine growth table must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF


   IF ( InputFileData%Morison%NMGDepths > 0 ) THEN

      InputFileData%Morison%MGTop    = -999999.0
      InputFileData%Morison%MGBottom =  999999.0
      
      DO I = 1,InputFileData%Morison%NMGDepths
            ! Store the boundaries of the marine growth zone
         IF ( InputFileData%Morison%MGDepths(I)%MGDpth > InputFileData%Morison%MGTop ) THEN
            InputFileData%Morison%MGTop    = InputFileData%Morison%MGDepths(I)%MGDpth
         END IF
         IF ( InputFileData%Morison%MGDepths(I)%MGDpth < InputFileData%Morison%MGBottom ) THEN
            InputFileData%Morison%MGBottom = InputFileData%Morison%MGDepths(I)%MGDpth
         ELSE
            CALL SetErrStat( ErrID_Fatal,'The rows of the marine growth table must be ordered with increasing depth (decreasing Z).',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

            ! Make sure that the current MGDpth is not used elsewhere in the table.
         DO J = I+1,InputFileData%Morison%NMGDepths
            IF ( EqualRealNos( InputFileData%Morison%MGDepths(I)%MGDpth, InputFileData%Morison%MGDepths(J)%MGDpth ) ) THEN
               CALL SetErrStat( ErrID_Fatal,'Duplicate MGDpth were found in the Marine Growth table.',ErrStat,ErrMsg,RoutineName)
               RETURN
            END IF
         END DO

         IF ( InputFileData%Morison%MGDepths(I)%MGThck < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'MGThck in the Marine growth table must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
         IF ( InputFileData%Morison%MGDepths(I)%MGDens < 0 ) THEN
            CALL SetErrStat( ErrID_Fatal,'MGDens in the Marine growth table must be greater or equal to zero.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

      END DO

   END IF


   !-------------------------------------------------------------------------------------------------
   ! Large yaw offset
   !-------------------------------------------------------------------------------------------------
   if (InputFileData%PtfmYMod /= 0 .AND. InputFileData%PtfmYMod /= 1) then
      call SetErrStat( ErrID_Fatal,'PtfmYMod must be 0 (static platform reference yaw offset) or 1 (dynamic platform reference yaw offset).',ErrStat,ErrMsg,RoutineName)
      return
   end if
   IF ( InputFileData%PtfmYMod .EQ. 1_IntKi ) THEN
      if ( InputFileData%PtfmYCutOff <= 0.0_ReKi ) then
         CALL SetErrStat( ErrID_Fatal, 'PtfmYCutOff must be greater than 0 Hz.',ErrStat,ErrMsg,RoutineName)
      end if
      if ( InputFileData%Morison%WaveDisp == 0 .AND. InputFileData%Morison%NMembers > 0 ) then
         call SetErrStat( ErrID_Fatal,'Dynamic reference yaw offset (PtfmYMod=1) cannot be used with WaveDisp=0. Set WaveDisp=1.',ErrStat,ErrMsg,RoutineName)
         return
      end if
      if ( InputFileData%PotMod > 0 .AND. InputFileData%WAMIT%ExctnMod == 2 ) then
         call SetErrStat( ErrID_Fatal,'Dynamic reference yaw offset (PtfmYMod=1) cannot be used with state-space wave excitations. Set ExctnMod=0 or 1.', ErrStat, ErrMsg, RoutineName )
         return
      end if
      if ( InputFileData%PotMod > 0 .AND. InputFileData%WAMIT%NExctnHdg < 2 ) then
         call SetErrStat( ErrID_Fatal, 'NExctnHdg must be greater than or equal to 2.', ErrStat, ErrMsg, RoutineName )
         return
      end if
      if ( InputFileData%WAMIT2%SumQTFF .OR. InputFileData%WAMIT2%DiffQTFF ) then
         call SetErrStat( ErrID_Fatal, 'Dynamic reference yaw offset (PtfmYMod=1) cannot be used with full sum-frequency or difference-frequency QTFs. Set SumQTF and DiffQTF to 0.', ErrStat, ErrMsg, RoutineName )
         return
      end if
   END IF

   !-------------------------------------------------------------------------------------------------
   ! Member Output List Section
   !-------------------------------------------------------------------------------------------------

   IF ( ( InputFileData%Morison%NMOutputs < 0 ) .OR. ( InputFileData%Morison%NMOutputs > 9 ) ) THEN
      CALL SetErrStat( ErrID_Fatal,'NMOutputs in the Member output list must be greater or equal to zero and less than 10.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

   IF ( InputFileData%Morison%NMOutputs > 0 ) THEN


      DO I = 1,InputFileData%Morison%NMOutputs

         InputFileData%Morison%MOutLst(I)%MemberIDIndx = -1

            ! Find MemberID in this Member output list table in the Members table
         DO J = 1,InputFileData%Morison%NMembers
            IF ( InputFileData%Morison%InpMembers(J)%MemberID == InputFileData%Morison%MOutLst(I)%MemberID ) THEN
               InputFileData%Morison%MOutLst(I)%MemberIDIndx = J
            END IF
         END DO

            ! Make sure that a PropSetID entry in the Member cross-section properties table was found
         IF ( InputFileData%Morison%MOutLst(I)%MemberIDIndx == -1 ) THEN
            CALL SetErrStat( ErrID_Fatal,'MemberID in the Member output list table does not appear in the Members table.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

         IF ( ( InputFileData%Morison%MOutLst(I)%NOutLoc < 1 ) .OR. ( InputFileData%Morison%MOutLst(I)%NOutLoc > 9) ) THEN
            CALL SetErrStat( ErrID_Fatal,'NOutLoc in the Member output list must be greater than zero and less than 10.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF

         DO J = 1,InputFileData%Morison%MOutLst(I)%NOutLoc
            IF ( ( InputFileData%Morison%MOutLst(I)%NodeLocs(J) < 0.0 ) .OR. ( InputFileData%Morison%MOutLst(I)%NodeLocs(J) > 1.0 ) ) THEN
               CALL SetErrStat( ErrID_Fatal,'NodeLocs in the Member output list must be greater or equal to 0.0 and less than or equal to 1.0.',ErrStat,ErrMsg,RoutineName)
               RETURN
            END IF
         END DO


      END DO

   END IF

   !-------------------------------------------------------------------------------------------------
   ! Joint Output List Section
   !-------------------------------------------------------------------------------------------------

   IF ( ( InputFileData%Morison%NJOutputs < 0 ) .OR. ( InputFileData%Morison%NMOutputs > 9 ) ) THEN
      CALL SetErrStat( ErrID_Fatal,'NJOutputs in the Joint output list must be greater or equal to zero and less than 10.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

   IF ( InputFileData%Morison%NJOutputs > 0 ) THEN


      DO I=1,InputFileData%Morison%NJOutputs
           
           InputFileData%Morison%JOutLst(I)%JointIDIndx = -1
         ! Find MemberID in this Member output list table in the Members table
         DO J = 1,InputFileData%Morison%NJoints
            IF ( InputFileData%Morison%InpJoints(J)%JointID == InputFileData%Morison%JOutLst(I)%JointID ) THEN
               InputFileData%Morison%JOutLst(I)%JointIDIndx = J
               EXIT
            END IF 
         END DO
         
            ! Make sure that a Joint Output ID found in the JOutLst is in the Joints table
         IF ( InputFileData%Morison%JOutLst(I)%JointIDIndx == -1 ) THEN
            CALL SetErrStat( ErrID_Fatal,'JointID in the Joint output list table does not appear in the Joints table.',ErrStat,ErrMsg,RoutineName)
            RETURN
         END IF
      END DO
   END IF
   
   !-------------------------------------------------------------------------------------------------
   ! Data section for OUTPUT
   !-------------------------------------------------------------------------------------------------


      ! OutAll - output all member and joint data

   IF ( InputFileData%OutAll ) THEN    !TODO: Alter this check once OutAll is supported
         CALL SetErrStat( ErrID_Fatal,'OutAll must be FALSE. Future versions of HydroDyn will once again support values of either TRUE or FALSE.',ErrStat,ErrMsg,RoutineName)
         RETURN
   END IF


      ! OutSwtch - output file switch

   IF ( InputFileData%OutSwtch /= 1 .AND. InputFileData%OutSwtch /= 2 .AND. InputFileData%OutSwtch /= 3 ) THEN
      CALL SetErrStat( ErrID_Fatal,'OutSwitch must be set to 1, 2, or 3.',ErrStat,ErrMsg,RoutineName)
      RETURN
   END IF

   !InputFileData%OutFmt
   !InputFileData%OutSFmt


         ! OutList - list of requested parameters to output to a file


   !----------------------------------------------------------
   !  Output List
   !----------------------------------------------------------

      ! First we need to extract module-specific output lists from the user-input list.
      ! Any unidentified channels will be attached to the HydroDyn module's output list.
   IF (  InputFileData%NUserOutputs > 0 ) THEN
      ALLOCATE ( foundMask(InputFileData%NUserOutputs) , STAT = ErrStat2 )
      IF ( ErrStat2 /= ErrID_None ) THEN
         CALL SetErrStat( ErrID_Fatal,'Error allocating space for temporary array: foundMask in the HydroDynInput_GetInput subroutine.',ErrStat,ErrMsg,RoutineName)
         
         RETURN
      END IF
      foundMask = .FALSE.
      
         ! Extract Morison list
      InputFileData%Morison%NumOuts = GetMorisonChannels  ( InputFileData%NUserOutputs, InputFileData%UserOutputs, InputFileData%Morison%OutList, foundMask, ErrStat2, ErrMsg2 ); CALL SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)
   
         ! Attach remaining items to the HydroDyn list
      InputFileData%NumOuts        = HDOut_GetChannels ( InputFileData%NUserOutputs, InputFileData%UserOutputs, InputFileData%OutList, foundMask, ErrStat2, ErrMsg2 ); CALL SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)
      
      CALL PrintBadChannelWarning(InputFileData%NUserOutputs, InputFileData%UserOutputs , foundMask, ErrStat2, ErrMsg2 ); CALL SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)
      
      IF (ErrStat >= AbortErrLev ) RETURN

      DEALLOCATE(foundMask)
   ELSE

      ! Set number of outputs to zero
      InputFileData%NumOuts = 0
      InputFileData%Morison%NumOuts = 0

      ! Allocate outlist with zero length
      call AllocAry(InputFileData%OutList, 0, "InputFileData%OutList", ErrStat2, ErrMsg2); 
      call SetErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)

   END IF
      ! Now that we have the sub-lists organized, lets do some additional validation.
   
   !----------------------------------------------------------
   ! Populate data in sub-types from parent or other module types
   !----------------------------------------------------------

      ! WAMIT
      InputFileData%WAMIT%HasWAMIT     = InputFileData%PotMod == 1
      ! WAMIT2
      InputFileData%WAMIT2%HasWAMIT    = InputFileData%PotMod == 1
      ! Morison
      InputFileData%Morison%UnSum      = InputFileData%UnSum
      InputFileData%Morison%Gravity    = InitInp%Gravity

         ! Process the input geometry and generate the simulation mesh representation
      call Morison_GenerateSimulationNodes( InitInp%WaveField%MSL2SWL, InputFileData%Morison%NJoints, InputFileData%Morison%InpJoints, InputFileData%Morison%NMembers, InputFileData%Morison%InpMembers, InputFileData%Morison%NNodes, InputFileData%Morison%Nodes, errStat2, errMsg2 )
      !CALL Morison_ProcessMorisonGeometry( InputFileData%Morison, ErrStat2, ErrMsg2 )
      CALL SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'HydroDynInput_GetInput' )
      IF ( ErrStat >= AbortErrLev ) RETURN


END SUBROUTINE HydroDynInput_ProcessInitData

END MODULE HydroDyn_Input
