!**********************************************************************************************************************************
!> ## FAST_Farm
!! The FAST_Farm, FAST_Farm_Subs, and FAST_Farm_Types modules make up a driver for the multi-turbine FAST.Farm code. 
!! FAST_Farms_Types will be auto-generated by the FAST registry program, based on the variables specified in the
!! FAST_Farm_Registry.txt file.
!!
! ..................................................................................................................................
!! ## LICENSING
!! Copyright (C) 2017  Bonnie Jonkman, independent contributor
!! Copyright (C) 2017  National Renewable Energy Laboratory
!!
!!    This file is part of FAST_Farm.
!!
!! Licensed under the Apache License, Version 2.0 (the "License");
!! you may not use this file except in compliance with the License.
!! You may obtain a copy of the License at
!!
!!     http://www.apache.org/licenses/LICENSE-2.0
!!
!! Unless required by applicable law or agreed to in writing, software
!! distributed under the License is distributed on an "AS IS" BASIS,
!! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
!! See the License for the specific language governing permissions and
!! limitations under the License.
!**********************************************************************************************************************************
MODULE FAST_Farm_Subs

   USE FAST_Farm_Types
   USE NWTC_Library
   USE WakeDynamics
   USE AWAE
   USE FAST_Farm_IO
   USE FAST_Subs
   USE FASTWrapper
   
#ifdef _OPENMP
   USE OMP_LIB 
#endif

   IMPLICIT NONE
   
CONTAINS

   subroutine TrilinearInterpRegGrid(V, pt, dims, val)
   
      real(SiKi),     intent(in   ) :: V(:,0:,0:,0:)      !< The volume data being sampled
      real(ReKi),     intent(in   ) :: pt(3)           !< The point, in grid coordinates where we want to sample the data
      integer(IntKi), intent(in   ) :: dims(3)         !< The grid dimensions
      real(ReKi),     intent(  out) :: val(3)          !< The interpolated value of V at location, pt
   
      integer(IntKi) :: x0,x1,y0,y1,z0,z1, i
      real(ReKi) :: xd,yd,zd,c00(3),c01(3),c10(3),c11(3),c0(3),c1(3)
      REAL(ReKi)      :: N(8)           ! array for holding scaling factors for the interpolation algorithm
      REAL(ReKi)      :: u(8)           ! array for holding the corner values for the interpolation algorithm across a cubic volume
      real(ReKi)      :: val2(3)
      
      x0 = min(max(floor(pt(1)),0),dims(1)-1)
      x1 = x0 + 1
      if (x0 == (dims(1)-1)) x1 = x0  ! Handle case where x0 is the last index in the grid, in this case xd = 0.0, so the 2nd term in the interpolation will not contribute
      xd = 2.0_ReKi * (pt(1) - REAL(x0, ReKi)) - 1.0_ReKi

      y0 = min(max(floor(pt(2)),0),dims(2)-1)
      y1 = y0 + 1
      if (y0 == (dims(2)-1)) y1 = y0  ! Handle case where y0 is the last index in the grid, in this case yd = 0.0, so the 2nd term in the interpolation will not contribute
      yd = 2.0_ReKi * (pt(2) - REAL(y0, ReKi)) - 1.0_ReKi

      z0 = min(max(floor(pt(3)),0),dims(3)-1)
      z1 = z0 + 1
      if (z0 == (dims(3)-1)) z1 = z0  ! Handle case where z0 is the last index in the grid, in this case zd = 0.0, so the 2nd term in the interpolation will not contribute
      zd = 2.0_ReKi * (pt(3) - REAL(z0, ReKi)) - 1.0_ReKi
      
      !-------------------------------------------------------------------------------------------------
      ! Interpolate on the grid
      !-------------------------------------------------------------------------------------------------

      N(1)  = ( 1.0_ReKi + zd )*( 1.0_ReKi - yd )*( 1.0_ReKi - xd )
      N(2)  = ( 1.0_ReKi + zd )*( 1.0_ReKi + yd )*( 1.0_ReKi - xd )
      N(3)  = ( 1.0_ReKi - zd )*( 1.0_ReKi + yd )*( 1.0_ReKi - xd )
      N(4)  = ( 1.0_ReKi - zd )*( 1.0_ReKi - yd )*( 1.0_ReKi - xd )
      N(5)  = ( 1.0_ReKi + zd )*( 1.0_ReKi - yd )*( 1.0_ReKi + xd )
      N(6)  = ( 1.0_ReKi + zd )*( 1.0_ReKi + yd )*( 1.0_ReKi + xd )
      N(7)  = ( 1.0_ReKi - zd )*( 1.0_ReKi + yd )*( 1.0_ReKi + xd )
      N(8)  = ( 1.0_ReKi - zd )*( 1.0_ReKi - yd )*( 1.0_ReKi + xd )
      N     = N / real( size(N), ReKi )  ! normalize

      do i=1,3
         u(1)  = real(V( i, x0, y0, z1 ), ReKi)
         u(2)  = real(V( i, x0, y1, z1 ), ReKi)
         u(3)  = real(V( i, x0, y1, z0 ), ReKi)
         u(4)  = real(V( i, x0, y0, z0 ), ReKi)
         u(5)  = real(V( i, x1, y0, z1 ), ReKi)
         u(6)  = real(V( i, x1, y1, z1 ), ReKi)
         u(7)  = real(V( i, x1, y1, z0 ), ReKi)
         u(8)  = real(V( i, x1, y0, z0 ), ReKi)
            
         val(i)  =  SUM ( N * u ) 
      end do
      
      !
      !
      !xd = pt(1) - x0
      !yd = pt(2) - y0
      !zd = pt(3) - z0
      !c00 = V(:,x0,y0,z0)*(1.0_ReKi-xd) + V(:,x1,y0,z0)*xd
      !c01 = V(:,x0,y0,z1)*(1.0_ReKi-xd) + V(:,x1,y0,z1)*xd
      !c10 = V(:,x0,y1,z0)*(1.0_ReKi-xd) + V(:,x1,y1,z0)*xd
      !c11 = V(:,x0,y1,z1)*(1.0_ReKi-xd) + V(:,x1,y1,z1)*xd
      !
      !c0  = c00*(1.0_ReKi-yd) + c10*yd
      !c1  = c01*(1.0_ReKi-yd) + c11*yd
      !
      !val2 = c0 *(1.0_ReKi-zd) + c1 *zd
      !do i = 1,3
      !   if ( .not. EqualRealNos(val(i),val2(i)) ) then
      !      write(*,*) "Different inpolated wind values: "//trim(Num2LStr(val(1)))//", "//trim(Num2LStr(val(2)))//", "//trim(Num2LStr(val(3)))//", "//trim(Num2LStr(val2(1)))//", "//trim(Num2LStr(val2(2)))//", "//trim(Num2LStr(val2(3)))
      !      return
      !   end if
      !end do
   end subroutine TrilinearInterpRegGrid

   
!----------------------------------------------------------------------------------------------------------------------------------
!> Routine to call Init routine for each module. This routine sets all of the init input data for each module. The initialization algorithm is: \n
!!   -  Read-In Input File
!!   -  Check Inputs and Set Parameters
!!   -  In parallel:
!!      1.  CALL AWAE_Init
!!      2.  CALL WD_Init
!!   -  Transfer y_AWAE_Init to u_F_Init and CALL F_Init
!!   -  Open Output File
!!   -  n=0
!!   -  t=0
SUBROUTINE Farm_Initialize( farm, InputFile, ErrStat, ErrMsg )

   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                !< FAST.Farm data
      
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat             !< Error status of the operation
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg              !< Error message if ErrStat /= ErrID_None
   CHARACTER(*),             INTENT(IN   ) :: InputFile           !< A CHARACTER string containing the name of the primary FAST.Farm input file
   
   
   ! local variables 
   type(AWAE_InitInputType)                :: AWAE_InitInput
   type(AWAE_InitOutputType)               :: AWAE_InitOutput
   
   INTEGER(IntKi)                          :: ErrStat2   
   CHARACTER(ErrMsgLen)                    :: ErrMsg2
   TYPE(WD_InitInputType)                  :: WD_InitInput            ! init-input data for WakeDynamics module
   CHARACTER(*), PARAMETER                 :: RoutineName = 'Farm_Initialize'       
   CHARACTER(ChanLen)                      :: OutList(Farm_MaxOutPts) ! list of user-requested output channels
   INTEGER(IntKi)                          :: i
   !..........
   ErrStat = ErrID_None
   ErrMsg  = ""         
   AbortErrLev  = ErrID_Fatal                                 ! Until we read otherwise from the FAST input file, we abort only on FATAL errors
      
   
      ! ... Open and read input files, initialize global parameters. ...
      
   IF (LEN_TRIM(InputFile) == 0) THEN ! no input file was specified
      CALL SetErrStat( ErrID_Fatal, 'The required input file was not specified on the command line.', ErrStat, ErrMsg, RoutineName )
      CALL NWTC_DisplaySyntax( InputFile, 'FAST.Farm.exe' )
      RETURN
   END IF            
                        
      ! Determine the root name of the primary file (will be used for output files)
   CALL GetRoot( InputFile, farm%p%OutFileRoot )      
    
   DO i=1,NumFFModules
      farm%p%Module_Ver(i)%Date = 'unknown date'
      farm%p%Module_Ver(i)%Ver  = 'unknown version'
   END DO       
   farm%p%Module_Ver( ModuleFF_FWrap )%Name = 'FAST Wrapper'
   farm%p%Module_Ver( ModuleFF_WD    )%Name = 'Wake Dynamics'
   farm%p%Module_Ver( ModuleFF_AWAE  )%Name = 'Ambient Wind and Array Effects'
   
   !...............................................................................................................................  
   ! step 1: read input file
   !...............................................................................................................................  
      
   call Farm_ReadPrimaryFile( InputFile, farm%p, WD_InitInput%InputFileData, AWAE_InitInput%InputFileData, OutList, ErrStat2, ErrMsg2 );  if(Failed()) return;

   !...............................................................................................................................  
   ! step 2: validate input & set parameters
   !...............................................................................................................................  
      
   call Farm_ValidateInput( farm%p, WD_InitInput%InputFileData, AWAE_InitInput%InputFileData, ErrStat2, ErrMsg2 );  if(Failed()) return;
   
   farm%p%NOutTurb = min(farm%p%NumTurbines,9)  ! We only support output for the first 9 turbines, even if the farm has more than 9 
   
   farm%p%n_high_low = NINT( farm%p%dt_low / farm%p%dt_high )
            
         ! let's make sure the FAST.Farm DT_low is an exact multiple of dt_high 
         ! (i'm doing this outside of Farm_ValidateInput so we know that dt_low/=0 before computing n_high_low):
      IF ( .NOT. EqualRealNos( real(farm%p%DT_low,SiKi), real(farm%p%DT_high,SiKi) * farm%p%n_high_low )  ) THEN
         CALL SetErrStat(ErrID_Fatal, "DT_high ("//TRIM(Num2LStr(farm%p%dt_high))//" s) must be an integer divisor of DT_low (" &
                        //TRIM(Num2LStr(farm%p%dt_low))//" s).", ErrStat, ErrMsg, RoutineName ) 
      END IF
      
   farm%p%TChanLen = max( 10, int(log10(farm%p%TMax))+7 )
   farm%p%OutFmt_t = 'F'//trim(num2lstr( farm%p%TChanLen ))//'.4' ! 'F10.4'    
   farm%p%n_TMax  = FLOOR( ( farm%p%TMax / farm%p%DT_low ) ) + 1  ! We're going to go from step 0 to (n_TMax - 1)
                   ! [note that FAST uses the ceiling function, so it might think we're doing one more step than FAST.Farm; 
                   ! This difference will be a problem only if FAST thinks it's doing FEWER timesteps than FAST.Farm does.]
   
   IF ( WD_InitInput%InputFileData%NumPlanes > farm%p%n_TMax ) THEN
      WD_InitInput%InputFileData%NumPlanes = max( 2, min( WD_InitInput%InputFileData%NumPlanes, farm%p%n_TMax ) )
      call SetErrStat(ErrID_Warn, "For efficiency, NumPlanes has been reduced to the number of time steps ("//TRIM(Num2LStr(WD_InitInput%InputFileData%NumPlanes))//").", ErrStat, ErrMsg, RoutineName )
   ENDIF
   
   !...............................................................................................................................  
   ! step 3: initialize WAT, AWAE, and WD (b, c, and d can be done in parallel)
   !...............................................................................................................................  

      !-------------------
      ! a. read WAT input files using InflowWind
   if (farm%p%WAT /= Mod_WAT_None) then
      call WAT_init( farm%p, farm%WAT_IfW, AWAE_InitInput, ErrStat2, ErrMsg2 )
      if(Failed()) return;
   endif

      !-------------------
      ! b. CALL AWAE_Init

   if (farm%p%WAT /= Mod_WAT_None) AWAE_InitInput%WAT_Enabled = .true.
   AWAE_InitInput%InputFileData%dr           = WD_InitInput%InputFileData%dr
   AWAE_InitInput%InputFileData%dt_low       = farm%p%dt_low
   AWAE_InitInput%InputFileData%NumTurbines  = farm%p%NumTurbines
   AWAE_InitInput%InputFileData%NumRadii     = WD_InitInput%InputFileData%NumRadii
   AWAE_InitInput%InputFileData%NumPlanes    = WD_InitInput%InputFileData%NumPlanes
   AWAE_InitInput%InputFileData%WindFilePath = farm%p%WindFilePath
   AWAE_InitInput%n_high_low                 = farm%p%n_high_low
   AWAE_InitInput%NumDT                      = farm%p%n_TMax
   AWAE_InitInput%OutFileRoot                = farm%p%OutFileRoot
   if (farm%p%WAT /= Mod_WAT_None .and. associated(farm%WAT_IfW%p%FlowField)) then
      AWAE_InitInput%WAT_FlowField => farm%WAT_IfW%p%FlowField
   endif
   call AWAE_Init( AWAE_InitInput, farm%AWAE%u, farm%AWAE%p, farm%AWAE%x, farm%AWAE%xd, farm%AWAE%z, farm%AWAE%OtherSt, farm%AWAE%y, &
                   farm%AWAE%m, farm%p%DT_low, AWAE_InitOutput, ErrStat2, ErrMsg2 )
   if(Failed()) return;
      
   farm%AWAE%IsInitialized = .true.

   farm%p%X0_Low = AWAE_InitOutput%X0_Low
   farm%p%Y0_low = AWAE_InitOutput%Y0_low
   farm%p%Z0_low = AWAE_InitOutput%Z0_low
   farm%p%nX_Low = AWAE_InitOutput%nX_Low
   farm%p%nY_low = AWAE_InitOutput%nY_low
   farm%p%nZ_low = AWAE_InitOutput%nZ_low
   farm%p%dX_low = AWAE_InitOutput%dX_low
   farm%p%dY_low = AWAE_InitOutput%dY_low
   farm%p%dZ_low = AWAE_InitOutput%dZ_low
   farm%p%Module_Ver( ModuleFF_AWAE  ) = AWAE_InitOutput%Ver
   
      !-------------------
      ! c. initialize WD (one instance per turbine, each can be done in parallel, too)
      
   call Farm_InitWD( farm, WD_InitInput, ErrStat2, ErrMsg2 );  if(Failed()) return;
      
      
   !...............................................................................................................................  
   ! step 4: initialize FAST (each instance of FAST can also be done in parallel)
   !...............................................................................................................................  

   CALL Farm_InitFAST( farm, WD_InitInput%InputFileData, AWAE_InitOutput, ErrStat2, ErrMsg2);  if(Failed()) return;
      
   !...............................................................................................................................  
   ! step 4.5: initialize farm-level MoorDyn if applicable
   !...............................................................................................................................  
   
   if (farm%p%MooringMod == 3) then
      CALL Farm_InitMD( farm, ErrStat2, ErrMsg2);  if(Failed()) return;  ! FAST instances must be initialized first so that turbine initial positions are known
   end if

   !...............................................................................................................................  
   ! step 5: Open output file (or set up output file handling)      
   !...............................................................................................................................  
   
      ! Set parameters for output channels:
   CALL Farm_SetOutParam(OutList, farm, ErrStat2, ErrMsg2 );  if(Failed()) return; ! requires: p%NumOuts, sets: p%OutParam.
      
   call Farm_InitOutput( farm, ErrStat2, ErrMsg2 );  if(Failed()) return;

      ! Print the summary file if requested:
   IF (farm%p%SumPrint) THEN
      CALL Farm_PrintSum( farm, WD_InitInput%InputFileData, ErrStat2, ErrMsg2 );  if(Failed()) return;
   END IF
   
   !...............................................................................................................................
   ! Destroy initializion data
   !...............................................................................................................................      
   CALL Cleanup()
   
CONTAINS
   SUBROUTINE Cleanup()
      call WD_DestroyInitInput(WD_InitInput, ErrStat2, ErrMsg2)
      call AWAE_DestroyInitInput(AWAE_InitInput, ErrStat2, ErrMsg2)
      call AWAE_DestroyInitOutput(AWAE_InitOutput, ErrStat2, ErrMsg2)
   END SUBROUTINE Cleanup

   logical function Failed()
      call SetErrStat(errStat2, errMsg2, errStat, errMsg, RoutineName)
      Failed = errStat >= AbortErrLev
      if (Failed) call cleanup()
   end function Failed
END SUBROUTINE Farm_Initialize



!----------------------------------------------------------------------------------------------------------------------------------
!> This routine sets the WAT InflowWind data storage.  Rather than initialize all of InflowWind, we just call the HAWC wind init.
SUBROUTINE WAT_init( p, WAT_IfW, AWAE_InitInput, ErrStat, ErrMsg )
   USE   InflowWind_IO, only: IfW_HAWC_Init
   type(farm_ParameterType),  intent(inout) :: p                     !< farm parameters data
   type(WAT_IfW_data),        intent(inout) :: WAT_IfW               !< InflowWind data
   type(AWAE_InitInputType),  intent(inout) :: AWAE_InitInput        !< for error checking, and temporary to pass IfW
   integer(IntKi),            intent(  out) :: ErrStat               !< Error status of the operation
   character(*),              intent(  out) :: ErrMsg                !< Error message if ErrStat /= ErrID_None

   type(HAWC_InitInputType)                 :: HAWC_InitInput
   type(WindFileDat)                        :: FileDat
   character(1024)                          :: BoxFileRoot, BoxFile_u, BoxFile_v, BoxFile_w
   character(1024)                          :: sDummy
   character(6)                             :: FileEnding(3)
   integer(IntKi)                           :: i,j,k,n
   integer(IntKi)                           :: ErrStat2
   character(ErrMsgLen)                     :: ErrMsg2
   character(*), parameter                  :: RoutineName = 'WAT_init'

   ErrStat  = ErrID_None
   ErrMsg   = ""

   ! If flowfield is allocated, deallocate and allocate again to clear old data
   if (associated(WAT_IfW%p%FlowField))   deallocate(WAT_IfW%p%FlowField)
   allocate(WAT_IfW%p%FlowField)

   ! HAWC file names
   call SplitFileName (p%WAT_BoxFile, BoxFileRoot, FileEnding, ErrStat2, ErrMsg2);  if (Failed()) return
   HAWC_InitInput%WindFileName(1) = trim(BoxFileRoot)//trim(FileEnding(1))
   HAWC_InitInput%WindFileName(2) = trim(BoxFileRoot)//trim(FileEnding(2))
   HAWC_InitInput%WindFileName(3) = trim(BoxFileRoot)//trim(FileEnding(3))

   ! HAWC spatial grid
   if (p%WAT == Mod_WAT_PreDef) then       ! from library of WAT files, set the NxNyNz and DxDyDz terms
      call MannLibDims(BoxFileRoot, p%RotorDiamRef, p%WAT_NxNyNz, p%WAT_DxDyDz, ErrStat2, ErrMsg2);  if (Failed()) return
      write(sDummy, '(3(I8,1X))') p%WAT_NxNyNz
      call WrScr('  WAT: NxNyNz set to: '//trim(sDummy)//' (inferred from filename)')
      write(sDummy, '(3(F8.3,1X))') p%WAT_DxDyDz
      call WrScr('  WAT: DxDyDz set to: '//trim(sDummy)//' (based on rotor diameter)')
   endif
   ! Sanity check
   if (any(p%WAT_NxNyNz<2)) then
      call SetErrStat(ErrID_Fatal, "Values of WAT_NxNyNz should be above 2", ErrStat, ErrMsg, RoutineName)
      return
   endif
   if (any(p%WAT_DxDyDz<=0)) then
      call SetErrStat(ErrID_Fatal, "Values of WAT_DxDyDz should be strictly positive", ErrStat, ErrMsg, RoutineName)
      return
   endif
   ! NOTE: We don't check for the dimensions of of the grid here compared to high res because we don't know it for VTKs
   !       See AWAE_IO_InitGridInfo  

   HAWC_InitInput%nx = p%WAT_NxNyNz(1)
   HAWC_InitInput%ny = p%WAT_NxNyNz(2)
   HAWC_InitInput%nz = p%WAT_NxNyNz(3)
   HAWC_InitInput%dx = p%WAT_DxDyDz(1)
   HAWC_InitInput%dy = p%WAT_DxDyDz(2)
   HAWC_InitInput%dz = p%WAT_DxDyDz(3)
   HAWC_InitInput%G3D%RefHt            = 0.5_ReKi * p%WAT_NxNyNz(3)*p%WAT_DxDyDz(3)          ! reference height; the height (in meters) of the vertical center of the grid (m)
   HAWC_InitInput%G3D%URef             = 1.0_ReKi    ! Set to 1.0 so that dX = DTime (this affects data storage)
   HAWC_InitInput%G3D%WindProfileType  = 0           ! Wind profile type (0=constant;1=logarithmic,2=power law)
   HAWC_InitInput%G3D%PLExp            = 0.0_ReKi
   HAWC_InitInput%G3D%ScaleMethod      = 0           ! NOTE: setting this to 2 doesn't do the same as what we do below with ScaleBox
   HAWC_InitInput%G3D%SF               = 1.0_ReKi    ! Turbulence scaling factor for the x direction (-)   [ScaleMethod=1]
   HAWC_InitInput%G3D%SigmaF           = 1.0_ReKi    ! Turbulence standard deviation to calculate scaling from in x direction (m/s)    [ScaleMethod=2] 
   HAWC_InitInput%G3D%Z0               = 0.3_ReKi    ! Surface roughness (not used)
   HAWC_InitInput%G3D%XOffset          = 0.0_ReKi    ! Initial offset in +x direction (shift of wind box)

   WAT_IfW%p%FlowField%PropagationDir  = 0.0_ReKi
   WAT_IfW%p%FlowField%VFlowAngle      = 0.0_ReKi
   WAT_IfW%p%FlowField%RotateWindBox   = .false.

   WAT_IfW%p%FlowField%FieldType = Grid3D_FieldType
   call IfW_HAWC_Init(HAWC_InitInput, -1, WAT_IfW%p%FlowField%Grid3D, FileDat, ErrStat2, ErrMsg2);  if (Failed()) return   ! summary file unit set to -1

   if (p%WAT_ScaleBox) then
      call WrScr('   WAT: Scaling Box for unit standard deviation and zero mean')
      call Grid3D_ZeroMean_UnitStd(WAT_IfW%p%FlowField%Grid3D%Vel)
   endif

   ! Reference position for wind rotation (not used here, but should be set)
   WAT_IfW%p%FlowField%RefPosition = [0.0_ReKi, 0.0_ReKi, WAT_IfW%p%FlowField%Grid3D%RefHeight]

   WAT_IfW%IsInitialized = .true.

   call Cleanup()
   return

contains
   logical function Failed()
      call SetErrStat( ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      Failed =  ErrStat >= AbortErrLev
      if (Failed) call Cleanup()
   end function Failed
   subroutine Cleanup()
      ! nothing to clean up
   end subroutine Cleanup

   !  Split out the ending of .u or u.bin from the filename.
   !  If none given, then append ending and check for file existance
   subroutine SplitFileName(FileName, BaseName, Ending, ErrStat3, ErrMsg3)
      character(1024),  intent(in   ) :: FileName
      character(1024),  intent(  out) :: BaseName
      character(6),     intent(  out) :: Ending(3)
      integer(IntKi),   intent(  out) :: ErrStat3
      character(*),     intent(  out) :: ErrMsg3
      integer(IntKi)                  :: i,l
      logical                         :: foundFile
      ErrStat3 = ErrID_None
      ErrMsg3  = ""
      ! check if passed filename ends in .u or u.bin
      l=len_trim(FileName)
      if (index(FileName,'.u')>0) then
         BaseName = FileName(1:l-2)
         Ending(1)= '.u'
         inquire(file=trim(BaseName)//trim(Ending(1)), exist=foundfile)
         if (.not. foundFile) then
            ErrStat3 = ErrID_Fatal
            ErrMsg3  = 'Cannot find wake added turbulence Mann box file with name '//trim(BaseName)//trim(Ending(1))
         endif
         Ending(2)= '.v'
         Ending(3)= '.w'
         return
      elseif (index(FileName,'u.bin') > 0) then
         BaseName = FileName(1:l-4)
         Ending(1)= 'u.bin'
         inquire(file=trim(BaseName)//trim(Ending(1)), exist=foundfile)
         if (.not. foundFile) then
            ErrStat3 = ErrID_Fatal
            ErrMsg3  = 'Cannot find wake added turbulence Mann box file with name '//trim(BaseName)//trim(Ending(1))
         endif
         Ending(2)= 'v.bin'
         Ending(3)= 'w.bin'
         return
      else  ! Ending not included in filename, so try figure it out
         BaseName = trim(FileName)
         ! is it .u for file ending
         Ending(1)= '.u'
         Ending(2)= '.v'
         Ending(3)= '.w'
         inquire(file=trim(BaseName)//trim(Ending(1)),  exist=foundFile)
         if (foundFile) return
         ! is it u.bin for file ending
         Ending(1)= 'u.bin'
         Ending(2)= 'v.bin'
         Ending(3)= 'w.bin'
         inquire(file=trim(BaseName)//trim(Ending(1)),  exist=foundFile)
         if (foundFile) return
         ! didn't find file, so error out
         ErrStat3 = ErrID_Fatal
         ErrMsg3  = 'Cannot find wake added turbulence Mann box file with name '//trim(BaseName)//'.u '//' or '//trim(BaseName)//'u.bin '
      endif
   end subroutine SplitFileName
   !> If it is a filename of a library, expect following format: FFDB_512x512x64.u where:
   !!     512x512x64  -- Number of grid points in X,Y,Z -- Nx, Ny, Nz
   subroutine MannLibDims(BoxFileRoot,RotorDiamRef,Nxyz,Dxyz,ErrStat3,ErrMsg3)
      character(1024),  intent(in   ) :: BoxFileroot
      real(ReKi),       intent(in   ) :: RotorDiamRef    ! reference rotordiam
      integer(IntKi),   intent(  out) :: Nxyz(3)
      real(ReKi),       intent(  out) :: Dxyz(3)         ! derived based on rotor diameter
      integer(IntKi),   intent(  out) :: ErrStat3
      character(*),     intent(  out) :: ErrMsg3
      integer(IntKi)                  :: i,iLast, n      ! generic indexing stuff
      character(1)                    :: C0              ! characters for testing
      real(ReKi), parameter           :: ScaleFact=0.03  ! scale ifactor for Dx,Dy,Dz based on rotor diameter
      character(1024)                 :: sDigitsX        ! String made of digits and "x"
      character(11)                   :: CharNums="1234567890X"
      character(1024), allocatable :: StrArray(:) ! Array of strings extracted from line
      Nxyz(:)=-1

      ErrStat3 = ErrID_None
      ErrMsg3  = ""

      ! Set Dxyz
      Dxyz=real(RotorDiamRef,ReKi)*ScaleFact

      ! --- Create a string made of digits and "x" only, starting from the end of the filename
      n = len_trim(BoxFileRoot)
      iLast = n
      do i=n,1,-1
         C0 = BoxFileRoot(i:i)
         call Conv2UC(C0)
         if ((index(CharNums,C0)==0)) then
            exit
         endif
         iLast=i
      enddo
      sDigitsX=BoxFileRoot(iLast:n)
      call Conv2UC(sDigitsX)

      ! --- Splitting string according to character "x"
      call strsplit(sDigitsX, StrArray, 'X')
      if (size(StrArray)/=3) then
         ErrStat3 = ErrID_Fatal
         ErrMsg3  = "Could not find three substrings delimited by 'x' in filename "//trim(BoxFileRoot)// &
                    ".  Expecting filename to include something like '512x512x64' for 512 by 512 by 64 points"
         return
      endif
      do i=1,3
         if (.not.(is_integer(StrArray(i), Nxyz(i)))) then
            ! NOTE: should not happen, unless we have "xx"
            ErrStat3 = ErrID_Fatal
            ErrMsg3  = "Could not convert substring `"//trim(StrArray(i))//"` to an integer in filename "//trim(BoxFileRoot)// &
                       ".  Expecting filename to include something like '512x512x64' for 512 by 512 by 64 points"
            return
         endif
      enddo
      ErrStat3=ErrID_None
      ErrMsg3 =""
   end subroutine MannLibDims
end subroutine WAT_init

!> Remove mean from all grid nodes and set standard deviation to 1 at all nodes
! See Grid3D_ScaleTurbulence and ScaleMethod in InflowWind as well
subroutine Grid3D_ZeroMean_UnitStd(Vel)
   real(SiKi),  dimension(:,:,:,:), intent(inout) :: Vel !< Array of field velocities 3 x ny x nz x nt
   integer(IntKi) :: i,j,k
   real(SiKi)     :: vmean, vstd
   real(SiKi)     :: nt
   nt = real(size(Vel, 4), SiKi)
   do i=1,size(Vel, 2)
      do j=1,size(Vel, 3)
         do k=1,3
            vmean = sum(Vel(k,i,j,:))/nt
            vstd  = sqrt(sum((Vel(k,i,j,:) - vmean)**2)/nt)
            if ( EqualRealNos( vstd, 0.0_SiKi) ) then
               vstd = 1.0_SiKi
            endif
            Vel(k,i,j,:) = (Vel(k,i,j,:) - vmean)/vstd
         enddo
      enddo
   enddo
end subroutine Grid3D_ZeroMean_UnitStd


!----------------------------------------------------------------------------------------------------------------------------------
!> This routine initializes all instances of WakeDynamics
SUBROUTINE Farm_InitWD( farm, WD_InitInp, ErrStat, ErrMsg )


      ! Passed variables
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data
   TYPE(WD_InitInputType),   INTENT(INOUT) :: WD_InitInp                      !< init input for WakeDynamics module; input file data already filled in
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

   ! local variables
   type(WD_InitOutputType)                 :: WD_InitOut

   INTEGER(IntKi)                          :: nt                          ! loop counter for rotor number
   INTEGER(IntKi)                          :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'Farm_InitWD'
         
   ErrStat = ErrID_None
   ErrMsg = ""
   
   ALLOCATE(farm%WD(farm%p%NumTurbines),STAT=ErrStat2);  if (Failed0('Wake Dynamics data')) return;
            
      !.................
      ! Initialize each instance of WD
      !................                  
      
      DO nt = 1,farm%p%NumTurbines
         !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         ! initialization can be done in parallel (careful for FWrap_InitInp, though)
         !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++         
         
         WD_InitInp%TurbNum     = nt
         WD_InitInp%OutFileRoot = farm%p%OutFileRoot
         
            ! note that WD_Init has Interval as INTENT(IN) so, we don't need to worry about overwriting farm%p%dt_low here:
         call WD_Init( WD_InitInp, farm%WD(nt)%u, farm%WD(nt)%p, farm%WD(nt)%x, farm%WD(nt)%xd, farm%WD(nt)%z, &
                          farm%WD(nt)%OtherSt, farm%WD(nt)%y, farm%WD(nt)%m, farm%p%dt_low, WD_InitOut, ErrStat2, ErrMsg2 )
         
         farm%WD(nt)%IsInitialized = .true.
            CALL SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)
            if (ErrStat >= AbortErrLev) then
               call cleanup()
               return
            end if
            
      END DO   
      
      farm%p%Module_Ver( ModuleFF_WD ) = WD_InitOut%Ver
      
      call cleanup()
      
contains
   subroutine cleanup()
      call WD_DestroyInitOutput( WD_InitOut, ErrStat2, ErrMsg2 )
   end subroutine cleanup

   ! check for failed where /= 0 is fatal
   logical function Failed0(txt)
      character(*), intent(in) :: txt
      if (ErrStat2 /= 0) then
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = "Could not allocate memory for "//trim(txt)
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      endif
      Failed0 = ErrStat >= AbortErrLev
      if(Failed0) call cleanUp()
   end function Failed0
END SUBROUTINE Farm_InitWD
!----------------------------------------------------------------------------------------------------------------------------------
!> This routine initializes all instances of FAST using the FASTWrapper module
SUBROUTINE Farm_InitFAST( farm, WD_InitInp, AWAE_InitOutput, ErrStat, ErrMsg )


      ! Passed variables
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data
   TYPE(WD_InputFileType),   INTENT(IN   ) :: WD_InitInp                      !< input-file data for WakeDynamics module
   TYPE(AWAE_InitOutputType),INTENT(IN   ) :: AWAE_InitOutput                 !< initialization output from AWAE
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

   ! local variables
   type(FWrap_InitInputType)               :: FWrap_InitInp
   type(FWrap_InitOutputType)              :: FWrap_InitOut   
   REAL(DbKi)                              :: FWrap_Interval                  !< Coupling interval that FWrap is called at (affected by MooringMod)

   INTEGER(IntKi)                          :: nt                          ! loop counter for rotor number
   INTEGER(IntKi)                          :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'Farm_InitFAST'
   
   
   ErrStat = ErrID_None
   ErrMsg = ""
   
   ALLOCATE(farm%FWrap(farm%p%NumTurbines),STAT=ErrStat2);  if (Failed0('FAST Wrapper data')) return;
            
      !.................
      ! Initialize each instance of FAST
      !................            
      FWrap_InitInp%nr            = WD_InitInp%NumRadii
      FWrap_InitInp%dr            = WD_InitInp%dr
      FWrap_InitInp%tmax          = farm%p%TMax
      FWrap_InitInp%n_high_low    = farm%p%n_high_low + 1   ! Add 1 because the FAST wrapper uses an index that starts at 1
      FWrap_InitInp%dt_high       = farm%p%dt_high
     
      FWrap_InitInp%nX_high       = AWAE_InitOutput%nX_high
      FWrap_InitInp%nY_high       = AWAE_InitOutput%nY_high
      FWrap_InitInp%nZ_high       = AWAE_InitOutput%nZ_high
      
      if (farm%p%MooringMod > 0) then
         FWrap_Interval = farm%p%dt_mooring    ! when there is a farm-level mooring model, FASTWrapper will be called at the mooring coupling time step
      else
         FWrap_Interval = farm%p%dt_low        ! otherwise FASTWrapper will be called at the regular FAST.Farm time step
      end if
      
     !OMP PARALLEL DO default(shared) PRIVATE(nt, FWrap_InitOut, ErrStat2, ErrMsg2) schedule(runtime)
      DO nt = 1,farm%p%NumTurbines
         !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         ! initialization can be done in parallel (careful for FWrap_InitInp, though)
         !+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++         
         
         FWrap_InitInp%FASTInFile    = farm%p%WT_FASTInFile(nt)
         FWrap_InitInp%p_ref_Turbine = farm%p%WT_Position(:,nt)
         FWrap_InitInp%WaveFieldMod  = farm%p%WaveFieldMod
         FWrap_InitInp%TurbNum       = nt
         FWrap_InitInp%RootName      = trim(farm%p%OutFileRoot)//'.T'//num2lstr(nt)
         
         
         FWrap_InitInp%p_ref_high(1) = AWAE_InitOutput%X0_high(nt)
         FWrap_InitInp%p_ref_high(2) = AWAE_InitOutput%Y0_high(nt)
         FWrap_InitInp%p_ref_high(3) = AWAE_InitOutput%Z0_high(nt)

         FWrap_InitInp%dX_high       = AWAE_InitOutput%dX_high(nt)
         FWrap_InitInp%dY_high       = AWAE_InitOutput%dY_high(nt)
         FWrap_InitInp%dZ_high       = AWAE_InitOutput%dZ_high(nt)

         FWrap_InitInp%Vdist_High   => AWAE_InitOutput%Vdist_High(nt)%data

            ! note that FWrap_Init has Interval as INTENT(IN) so, we don't need to worry about overwriting farm%p%dt_low here:
            ! NOTE: FWrap_interval, and FWrap_InitOut appear unused
         call FWrap_Init( FWrap_InitInp, farm%FWrap(nt)%u, farm%FWrap(nt)%p, farm%FWrap(nt)%x, farm%FWrap(nt)%xd, farm%FWrap(nt)%z, &
                          farm%FWrap(nt)%OtherSt, farm%FWrap(nt)%y, farm%FWrap(nt)%m, FWrap_Interval, FWrap_InitOut, ErrStat2, ErrMsg2 )
         
         farm%FWrap(nt)%IsInitialized = .true.
         
         if (ErrStat2 >= AbortErrLev) then
            !OMP CRITICAL  ! Needed to avoid data race on ErrStat and ErrMsg
            CALL SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)
            !OMP END CRITICAL
         endif
            
      END DO   
      !OMP END PARALLEL DO  

      if (ErrStat >= AbortErrLev) then
         call cleanup()
         return
      end if
   
      farm%p%Module_Ver( ModuleFF_FWrap ) = FWrap_InitOut%Ver
      
      call cleanup()
      
contains
   subroutine cleanup()
      call FWrap_DestroyInitInput( FWrap_InitInp, ErrStat2, ErrMsg2 )
      call FWrap_DestroyInitOutput( FWrap_InitOut, ErrStat2, ErrMsg2 )
   end subroutine cleanup
   ! check for failed where /= 0 is fatal
   logical function Failed0(txt)
      character(*), intent(in) :: txt
      if (ErrStat2 /= 0) then
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = "Could not allocate memory for "//trim(txt)
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      endif
      Failed0 = ErrStat >= AbortErrLev
      if(Failed0) call cleanUp()
   end function Failed0
END SUBROUTINE Farm_InitFAST
!----------------------------------------------------------------------------------------------------------------------------------
!> This routine initializes a farm-level instance of MoorDyn if applicable
SUBROUTINE Farm_InitMD( farm, ErrStat, ErrMsg )

   ! Passed variables
   type(All_FastFarm_Data), TARGET, INTENT(INOUT) :: farm                            !< FAST.Farm data
   INTEGER(IntKi),                  INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),                    INTENT(  OUT) :: ErrMsg                          !< Error message

   ! local variables
   type(MD_InitInputType)                         :: MD_InitInp
   type(MD_InitOutputType)                        :: MD_InitOut

   character(1025)                                :: Path, FileRoot                  ! for vtk outputs
   INTEGER(IntKi)                                 :: nt                              ! loop counter for rotor number
   INTEGER(IntKi)                                 :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                           :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER                      :: RoutineName = 'Farm_InitMD'
   TYPE(MeshType), POINTER                        :: SubstructureMotion
   
   ErrStat = ErrID_None
   ErrMsg = ""
   
   CALL WrScr(" --------- in FARM_InitMD, to initiailze farm-level MoorDyn ------- ")
   
   
   ! sort out how many times FASt and MoorDyn will be called per FAST.Farm time step based on DT_low and DT_mooring
   IF ( EqualRealNos( farm%p%dt_mooring, farm%p%DT_low ) ) THEN
      farm%p%n_mooring = 1
   ELSE
      IF ( farm%p%dt_mooring > farm%p%DT_low ) THEN
         ErrStat = ErrID_Fatal
         ErrMsg = "The farm mooring coupling time step ("//TRIM(Num2LStr(farm%p%dt_mooring))// &
                    " s) cannot be larger than FAST.Farm time step ("//TRIM(Num2LStr(farm%p%DT_low))//" s)."
      ELSE
            ! calculate the number of FAST-MoorDyn subcycles:
         farm%p%n_mooring = NINT( farm%p%DT_low / farm%p%dt_mooring )
            
            ! let's make sure the FAST DT is an exact integer divisor of the global (FAST.Farm) time step:
         IF ( .NOT. EqualRealNos( farm%p%DT_low, farm%p%dt_mooring * farm%p%n_mooring )  ) THEN
            ErrStat = ErrID_Fatal
            ErrMsg  = "The MoorDyn coupling time step, DT_mooring ("//TRIM(Num2LStr(farm%p%dt_mooring))// &
                      " s) must be an integer divisor of the FAST.Farm time step ("//TRIM(Num2LStr(farm%p%DT_low))//" s)."
         END IF
            
      END IF
   END IF     
   

   !.................
   ! MoorDyn initialization inputs...
   !................            
   !FWrap_InitInp%tmax          = farm%p%TMax
   !FWrap_InitInp%n_high_low    = farm%p%n_high_low + 1   ! Add 1 because the FAST wrapper uses an index that starts at 1
   !FWrap_InitInp%dt_high       = farm%p%dt_high
   

   MD_InitInp%FileName  = farm%p%MD_FileName                    ! input file name and path
   MD_InitInp%RootName  = trim(farm%p%OutFileRoot)//'.FarmMD'   ! root of output files
   MD_InitInp%FarmSize  = farm%p%NumTurbines                    ! number of turbines in the array. >0 tells MoorDyn to operate in farm mode
   
   ALLOCATE( MD_InitInp%PtfmInit(6,farm%p%NumTurbines), MD_InitInp%TurbineRefPos(3,farm%p%NumTurbines), STAT = ErrStat2 )
   if (Failed0("MoorDyn PtfmInit and TurbineRefPos initialization inputs in FAST.Farm.")) return;
   
   ! gather spatial initialization inputs for Farm-level MoorDyn (platform locations in their respective coordinate systems and locations of the turbines in the farm global coordinate system)
   DO nt = 1,farm%p%NumTurbines              
      MD_InitInp%PtfmInit(:,nt) = farm%FWrap(nt)%m%Turbine%p_FAST%PlatformPosInit ! platform initial positions in their respective coordinate systems from each FAST/ED instance
      MD_InitInp%TurbineRefPos(:,nt) = farm%p%WT_Position(:,nt)            ! reference positions of each turbine in the farm global coordinate system
   END DO 
    
   ! These aren't currently handled at the FAST.Farm level, so just give the farm's MoorDyn default values, which can be overwridden by its input file
   MD_InitInp%g         =    9.81
   MD_InitInp%rhoW      = 1025.0
   MD_InitInp%WtrDepth  =    0.0   !TODO: eventually connect this to a global depth input variable <<<

   ! Visualization of shared moorings
   if (farm%p%WrMooringVis) then
      MD_InitInp%VisMeshes=.true.
      farm%MD%VTK_Count = 0
      call GetPath ( MD_InitInp%RootName, Path, FileRoot ) ! the returned DVR_Outs%VTK_OutFileRoot includes a file separator character at the end
      farm%MD%VTK_OutFileRoot = trim(Path)//PathSep//'vtk'//PathSep//trim(FileRoot)
      farm%MD%VTK_TWidth = 5        !FIXME: this should be set based on sim length
   endif

   ! allocate MoorDyn inputs (assuming size 2 for linear interpolation/extrapolation... >
   ALLOCATE( farm%MD%Input( 2 ), farm%MD%InputTimes( 2 ), STAT = ErrStat2 )
   if (Failed0("MD%Input and MD%InputTimes.")) return;

   ! Assign the SS pointer of the first SS instance (turbine 1) to MD. Because MD in FF mode will only pull frequency info, instance of SS doesn't matter (error will be thrown by MD if user asks for SS grid).
   MD_InitInp%WaveField => farm%FWrap(1)%m%Turbine%SeaSt%p%WaveField ! this is the same wave field as Init%OutData_SeaSt%WaveField in FAST_subs.f90 (as set by line 278 in SeaSt_Init). Cant use Init%OutData_SeaSt%WaveField because Init is a local variable to FAST_InitializeAll
   
   ! initialize MoorDyn
   CALL MD_Init( MD_InitInp, farm%MD%Input(1), farm%MD%p, farm%MD%x, farm%MD%xd, farm%MD%z, &
                 farm%MD%OtherSt, farm%MD%y, farm%MD%m, farm%p%DT_mooring, MD_InitOut, ErrStat2, ErrMsg2 )
   if (Failed()) return;
   
   farm%MD%IsInitialized = .true.

   
   ! Copy MD inputs over into the 2nd entry of the input array, to allow the first extrapolation in FARM_MD_Increment
   CALL MD_CopyInput (farm%MD%Input(1),  farm%MD%Input(2),  MESH_NEWCOPY, Errstat2, ErrMsg2);  if (Failed()) return;
   farm%MD%InputTimes(2) = -0.1_DbKi
   
   CALL MD_CopyInput (farm%MD%Input(1), farm%MD%u,  MESH_NEWCOPY, Errstat2, ErrMsg2);  if (Failed()) return; ! do this to initialize meshes/allocatable arrays for output of ExtrapInterp routine
   
   
   ! Set up mesh maps between MoorDyn and floating platforms (or substructure).
   ! allocate mesh mappings for coupling farm-level MoorDyn with OpenFAST instances
   ALLOCATE( farm%m%MD_2_FWrap(farm%p%NumTurbines), farm%m%FWrap_2_MD(farm%p%NumTurbines), STAT = ErrStat2 )
   if (Failed0("MD_2_FWrap and FWrap_2_MD.")) return;
   
   ! MoorDyn point mesh to/from ElastoDyn (or SubDyn) point mesh
   do nt = 1,farm%p%NumTurbines
      !if (farm%MD%p%NFairs(nt) > 0 ) then   ! only set up a mesh map if MoorDyn has connections to this turbine
      
      ! loads
      CALL MeshMapCreate( farm%MD%y%CoupledLoads(nt), farm%FWrap(nt)%m%Turbine%MeshMapData%SubstructureLoads_Tmp_Farm, farm%m%MD_2_FWrap(nt), ErrStat2, ErrMsg2 )
      if (Failed()) return;
     
      ! kinematics
      IF (farm%FWrap(nt)%m%Turbine%p_FAST%CompSub == Module_SD) then
         SubstructureMotion => farm%FWrap(nt)%m%Turbine%SD%y%y3Mesh
      ELSE
         SubstructureMotion => farm%FWrap(nt)%m%Turbine%ED%y%PlatformPtMesh
      END IF
   
      CALL MeshMapCreate( SubstructureMotion, farm%MD%Input(1)%CoupledKinematics(nt), farm%m%FWrap_2_MD(nt), ErrStat2, ErrMsg2 )
      if (Failed()) return;

   end do
   

   farm%p%Module_Ver( ModuleFF_MD) = MD_InitOut%Ver
   
   call cleanup()
      
contains
   subroutine cleanup()
      call MD_DestroyInitInput(  MD_InitInp, ErrStat2, ErrMsg2 )
      call MD_DestroyInitOutput( MD_InitOut, ErrStat2, ErrMsg2 )
   end subroutine cleanup

   logical function Failed()
      call SetErrStat(errStat2, errMsg2, errStat, errMsg, RoutineName)
      Failed = errStat >= AbortErrLev
      if (Failed) call cleanup()
   end function Failed

   ! check for failed where /= 0 is fatal
   logical function Failed0(txt)
      character(*), intent(in) :: txt
      if (ErrStat2 /= 0) then
         ErrStat2 = ErrID_Fatal
         ErrMsg2  = "Could not allocate memory for "//trim(txt)
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
      endif
      Failed0 = ErrStat >= AbortErrLev
      if(Failed0) call cleanUp()
   end function Failed0
END SUBROUTINE Farm_InitMD
!----------------------------------------------------------------------------------------------------------------------------------
!> This routine moves a farm-level MoorDyn simulation one step forward, to catch up with FWrap_Increment
subroutine FARM_MD_Increment(t, n, farm, ErrStat, ErrMsg)
   REAL(DbKi),                      INTENT(IN   ) :: t                               !< Current simulation time in seconds
   INTEGER(IntKi),                  INTENT(IN   ) :: n                               !< Current step of the simulation in FARM MoorDyn terms
   type(All_FastFarm_Data), TARGET, INTENT(INOUT) :: farm                            !< FAST.Farm data  
   INTEGER(IntKi),                  INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),                    INTENT(  OUT) :: ErrMsg                          !< Error message

   INTEGER(IntKi)                                 :: nt
   INTEGER(IntKi)                                 :: n_ss
   INTEGER(IntKi)                                 :: n_FMD
   REAL(DbKi)                                     :: t_next        ! time at next step after this one (s)  
   INTEGER(IntKi)                                 :: ErrStat2 
   CHARACTER(ErrMsgLen)                           :: ErrMsg2
   CHARACTER(*),   PARAMETER                      :: RoutineName = 'FARM_MD_Increment'
   TYPE(MeshType), POINTER                        :: SubstructureMotion

   ErrStat = ErrID_None
   ErrMsg = ""
   
   ! ----- extrapolate MD inputs -----
   t_next = t + farm%p%DT_mooring

   ! Do a linear extrapolation to estimate MoorDyn inputs at time n_ss+1
   CALL MD_Input_ExtrapInterp(farm%MD%Input, farm%MD%InputTimes, farm%MD%u, t_next, ErrStat2, ErrMsg2)
   if (Failed()) return;
   
   ! Shift "window" of MD%Input: move values of Input and InputTimes from index 1 to index 2
   CALL MD_CopyInput (farm%MD%Input(1),  farm%MD%Input(2),  MESH_UPDATECOPY, Errstat2, ErrMsg2)
   if (Failed()) return;
   farm%MD%InputTimes(2) = farm%MD%InputTimes(1)

   ! update index 1 entries with the new extrapolated values
   CALL MD_CopyInput (farm%MD%u,  farm%MD%Input(1),  MESH_UPDATECOPY, Errstat2, ErrMsg2)
   if (Failed()) return;
   farm%MD%InputTimes(1) = t_next  


   ! ----- map substructure kinematics to MoorDyn inputs -----      (from mapping called at start of CalcOutputs Solve INputs)

   do nt = 1,farm%p%NumTurbines
      !if (farm%MD%p%NFairs(nt) > 0 ) then   
   
         IF (farm%FWrap(nt)%m%Turbine%p_FAST%CompSub == Module_SD) then
            SubstructureMotion => farm%FWrap(nt)%m%Turbine%SD%y%y3Mesh
         ELSE
            SubstructureMotion => farm%FWrap(nt)%m%Turbine%ED%y%PlatformPtMesh
         END IF
   
         CALL Transfer_Point_to_Point( SubstructureMotion, farm%MD%Input(1)%CoupledKinematics(nt), farm%m%FWrap_2_MD(nt), ErrStat2, ErrMsg2 )
         if (Failed()) return;

      !end if
   end do

   
   ! ----- update states and calculate outputs -----
   
   CALL MD_UpdateStates( t, n_FMD, farm%MD%Input, farm%MD%InputTimes, farm%MD%p, farm%MD%x,  &
                         farm%MD%xd, farm%MD%z, farm%MD%OtherSt, farm%MD%m, ErrStat2, ErrMsg2 )
   if (Failed()) return;
      
   
   CALL MD_CalcOutput( t, farm%MD%Input(1), farm%MD%p, farm%MD%x, farm%MD%xd, farm%MD%z,  &
                       farm%MD%OtherSt, farm%MD%y, farm%MD%m, ErrStat2, ErrMsg2 )
   if (Failed()) return;
   
   
   ! ----- map MD load outputs to each turbine's substructure -----   (taken from U FullOpt1...)
   do nt = 1,farm%p%NumTurbines
   
      if (farm%MD%p%nCpldPoints(nt) > 0 ) then   ! only map loads if MoorDyn has connections to this turbine (currently considering only Point connections <<< )
         
         IF (farm%FWrap(nt)%m%Turbine%p_FAST%CompSub == Module_SD) then
            SubstructureMotion => farm%FWrap(nt)%m%Turbine%SD%y%y3Mesh
         ELSE
            SubstructureMotion => farm%FWrap(nt)%m%Turbine%ED%y%PlatformPtMesh
         END IF
      
         ! mapping; Note: SubstructureLoads_Tmp_Farm contains loads from the farm-level (at a previous step); gets integrated into individual turbines inside FWrap_Increment()
         CALL Transfer_Point_to_Point( farm%MD%y%CoupledLoads(nt), farm%FWrap(nt)%m%Turbine%MeshMapData%SubstructureLoads_Tmp_Farm,  &
                                       farm%m%MD_2_FWrap(nt), ErrStat2, ErrMsg2,  &
                                       farm%MD%Input(1)%CoupledKinematics(nt), SubstructureMotion ) !u_MD and y_ED contain the displacements needed for moment calculations
         if (Failed()) return;
         
      end if
   end do


contains
   logical function Failed()
      call SetErrStat(errStat2, errMsg2, errStat, errMsg, RoutineName)
      Failed = errStat >= AbortErrLev
   end function Failed
end subroutine Farm_MD_Increment
!----------------------------------------------------------------------------------------------------------------------------------
!> This routine performs the initial call to calculate outputs (at t=0).
!! The Initial Calculate Output algorithm: \n
!!    -  Set u_AWAE=0, CALL AWAE_CO, and transfer y_AWAE to u_F and u_WD
!!    -  CALL F_t0
!!    -  Transfer y_F to and u_WD
!!    -  CALL WD_CO
!!    -  Transfer y_WD to u_AWAE
!!    -  CALL AWAE_CO
!!    -  Transfer y_AWAE to u_F and u_WD
!!    -  Write Output to File
subroutine FARM_InitialCO(farm, ErrStat, ErrMsg)
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

   INTEGER(IntKi)                          :: nt                    
   INTEGER(IntKi)                          :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'FARM_InitialCO'
   
   
   ErrStat = ErrID_None
   ErrMsg = ""
   
   

   
   !.......................................................................................
   ! Initial calls to AWAE module 
   !.......................................................................................
   
      !--------------------
      ! 1a. u_AWAE=0         
   farm%AWAE%u%xhat_plane = 0.0_ReKi     ! Orientations of wake planes, normal to wake planes, for each turbine
   farm%AWAE%u%p_plane    = 0.0_ReKi     ! Center positions of wake planes for each turbine
   farm%AWAE%u%Vx_wake    = 0.0_ReKi     ! Axial wake velocity deficit at wake planes, distributed radially, for each turbine
   farm%AWAE%u%Vy_wake    = 0.0_ReKi     ! Horizontal wake velocity deficit at wake planes, distributed radially, for each turbine
   farm%AWAE%u%Vz_wake    = 0.0_ReKi     ! "Vertical" wake velocity deficit at wake planes, distributed radially, for each turbine
   farm%AWAE%u%D_wake     = 0.0_ReKi     ! Wake diameters at wake planes for each turbine      
   
      !--------------------
      ! 1b. CALL AWAE_CO      
   call AWAE_CalcOutput( 0.0_DbKi, farm%AWAE%u, farm%AWAE%p, farm%AWAE%x, farm%AWAE%xd, farm%AWAE%z, &
                     farm%AWAE%OtherSt, farm%AWAE%y, farm%AWAE%m, ErrStat2, ErrMsg2 )         
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         if (ErrStat >= AbortErrLev) return
      !--------------------
      ! 1c. transfer y_AWAE to u_F and u_WD         
   
   call Transfer_AWAE_to_WD(farm)   

   !.......................................................................................
   ! CALL F_t0 (can be done in parallel)
   !.......................................................................................
         
   DO nt = 1,farm%p%NumTurbines
      
      call FWrap_t0( farm%FWrap(nt)%u, farm%FWrap(nt)%p, farm%FWrap(nt)%x, farm%FWrap(nt)%xd, farm%FWrap(nt)%z, &
                     farm%FWrap(nt)%OtherSt, farm%FWrap(nt)%y, farm%FWrap(nt)%m, ErrStat2, ErrMsg2 )         
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)
               
   END DO
   if (ErrStat >= AbortErrLev) return
   
   !.......................................................................................
   ! Transfer y_F to u_WD
   !.......................................................................................
      
      !--------------------
      !  Transfer y_F to u_WD     
   
   call Transfer_FAST_to_WD(farm)
      
   !.......................................................................................
   ! CALL WD_CO (can be done in parallel)
   !.......................................................................................
   
   DO nt = 1,farm%p%NumTurbines
      
      call WD_CalcOutput( 0.0_DbKi, farm%WD(nt)%u, farm%WD(nt)%p, farm%WD(nt)%x, farm%WD(nt)%xd, farm%WD(nt)%z, &
                     farm%WD(nt)%OtherSt, farm%WD(nt)%y, farm%WD(nt)%m, ErrStat2, ErrMsg2 )         
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)
               
   END DO
   if (ErrStat >= AbortErrLev) return
   
   !.......................................................................................
   ! Transfer y_WD to u_AWAE
   !.......................................................................................
   
   call Transfer_WD_to_AWAE(farm)
   
   !.......................................................................................
   ! CALL AWAE_CO
   !.......................................................................................
   
   call AWAE_CalcOutput( 0.0_DbKi, farm%AWAE%u, farm%AWAE%p, farm%AWAE%x, farm%AWAE%xd, farm%AWAE%z, &
                     farm%AWAE%OtherSt, farm%AWAE%y, farm%AWAE%m, ErrStat2, ErrMsg2 )         
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
   if (ErrStat >= AbortErrLev) return
   
   !.......................................................................................
   ! Transfer y_AWAE to u_F and u_WD
   !.......................................................................................

   call Transfer_AWAE_to_WD(farm)   
   
   !.......................................................................................
   ! Write Output to File
   !.......................................................................................
   
   call Farm_WriteOutput(0, 0.0_DbKi, farm, ErrStat2, ErrMsg2)
      call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
   
end subroutine FARM_InitialCO
!---------------------------------------------------------------------------------------------------------------------------------- 
!> This routine updates states each time increment. 
!! The update states algorithm: \n 
!!    -  In parallel:  
!!       1. call WD_US 
!!       2. call F_Increment 
!!       3. call AWAE_UpdateStates 
!!    -  \f$ n = n + 1 \f$ 
!!    -  \f$ t = t + \Delta t \f$ 
subroutine FARM_UpdateStates(t, n, farm, ErrStat, ErrMsg)
   REAL(DbKi),               INTENT(IN   ) :: t                               !< Current simulation time in seconds
   INTEGER(IntKi),           INTENT(IN   ) :: n                               !< Current step of the simulation: t = n*Interval
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

   INTEGER(IntKi)                          :: nt                      
   INTEGER(IntKi)                          :: n_ss                      
   INTEGER(IntKi)                          :: n_FMD   
   REAL(DbKi)                              :: t2                              ! time within the FAST-MoorDyn substepping loop for shared moorings
   INTEGER(IntKi)                          :: ErrStatAWAE, ErrStatMD, ErrStat2 
   CHARACTER(ErrMsgLen)                    :: ErrMsg2
   CHARACTER(ErrMsgLen)                    :: ErrMsgAWAE
   CHARACTER(ErrMsgLen)                    :: ErrMsgMD
   INTEGER(IntKi), ALLOCATABLE             :: ErrStatF(:)                     ! Temporary Error status for FAST
   CHARACTER(ErrMsgLen), ALLOCATABLE       :: ErrMsgF (:)                     ! Temporary Error message for FAST
   CHARACTER(*),   PARAMETER               :: RoutineName = 'FARM_UpdateStates'
   REAL(DbKi)                              :: tm1,tm2,tm3, tm01, tm02, tm03, tmSF, tmSM  ! timer variables
   
   ErrStat = ErrID_None
   ErrMsg = ""

   allocate ( ErrStatF ( farm%p%NumTurbines ), STAT=errStat2 )
       if (errStat2 /= 0) call SetErrStat ( ErrID_Fatal, 'Could not allocate memory for ErrStatF.', errStat, errMsg, RoutineName )
   allocate ( ErrMsgF ( farm%p%NumTurbines ), STAT=errStat2 )
       if (errStat2 /= 0) call SetErrStat ( ErrID_Fatal, 'Could not allocate memory for ErrMsgF.', errStat, errMsg, RoutineName )
   if (ErrStat >= AbortErrLev) return
   
   

   
   
   !.......................................................................................
   ! update module states (steps 1. and 2. and 3. and 4. can be done in parallel)
   !.......................................................................................
   
      !--------------------
      ! 1. CALL WD_US         
  
   !$OMP PARALLEL default(shared)
   !$OMP do private(nt, ErrStat2, ErrMsg2) schedule(runtime)
   DO nt = 1,farm%p%NumTurbines
      
      call WD_UpdateStates( t, n, farm%WD(nt)%u, farm%WD(nt)%p, farm%WD(nt)%x, farm%WD(nt)%xd, farm%WD(nt)%z, &
                     farm%WD(nt)%OtherSt, farm%WD(nt)%m, ErrStat2, ErrMsg2 )         


      ! Error handling
      if (errStat2 /= ErrID_None) then
         !$OMP CRITICAL  ! Needed to avoid data race on ErrStat and ErrMsg
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':FARM_UpdateStates')
         !$OMP END CRITICAL
      endif
         
   END DO
   !$OMP END DO 
   !$OMP END PARALLEL
   
   if (ErrStat >= AbortErrLev) return
   
      !--------------------
      ! 2. CALL F_Increment (and FARM_MD_Increment) and 4. CALL AWAE_UpdateStates  
      
      
   ! set the inputs needed for FAST (these are slow-varying so can just be done once per farm time step)
   do nt = 1,farm%p%NumTurbines
      call FWrap_SetInputs(farm%FWrap(nt)%u, farm%FWrap(nt)%m, t)
   end do
   
   
   !#ifdef printthreads
   !   tm1 = omp_get_wtime()  
   !   tmSF = 0.0_DbKi 
   !   tmSM = 0.0_DbKi 
   !#endif     
   ! Original case: no shared moorings 
   if (farm%p%MooringMod == 0) then     

      !$OMP PARALLEL DO DEFAULT(Shared) Private(nt)
      DO nt = 1,farm%p%NumTurbines
         call FWrap_Increment( t, n, farm%FWrap(nt)%u, farm%FWrap(nt)%p, farm%FWrap(nt)%x, farm%FWrap(nt)%xd, farm%FWrap(nt)%z, &
                     farm%FWrap(nt)%OtherSt, farm%FWrap(nt)%y, farm%FWrap(nt)%m, ErrStatF(nt), ErrMsgF(nt) )         
      END DO
      !$OMP END PARALLEL DO  
   
   ! Farm-level moorings case using MoorDyn
   else if (farm%p%MooringMod == 3) then
      
      
      ! This is the FAST-MoorDyn farm-level substepping loop        
      do n_ss = 1, farm%p%n_mooring                   ! do n_mooring substeps (number of FAST/FarmMD steps per Farm time step)
      
         n_FMD = n*farm%p%n_mooring  + n_ss - 1       ! number of the current time step of the call to FAST and MoorDyn         
         t2   = t + farm%p%DT_mooring*(n_ss - 1)      ! current time in the loop

         !#ifdef printthreads
         !   tm01 = omp_get_wtime()  
         !#endif
         
         ! A nested parallel for loop to call each instance of OpenFAST in parallel
         !$OMP PARALLEL DO DEFAULT(Shared) Private(nt)
         DO nt = 1,farm%p%NumTurbines
            call FWrap_Increment( t2, n_FMD, farm%FWrap(nt)%u, farm%FWrap(nt)%p, farm%FWrap(nt)%x, farm%FWrap(nt)%xd, farm%FWrap(nt)%z, &
                        farm%FWrap(nt)%OtherSt, farm%FWrap(nt)%y, farm%FWrap(nt)%m, ErrStatF(nt), ErrMsgF(nt) )         
         END DO              
         !$OMP END PARALLEL DO
         
         !#ifdef printthreads
         !   tm02 = omp_get_wtime()  
         !#endif  
      
         ! call farm-level MoorDyn time step here (can't multithread this with FAST since it needs inputs from all FAST instances)
         call Farm_MD_Increment( t2, n_FMD, farm, ErrStatMD, ErrMsgMD)
         call SetErrStat(ErrStatMD, ErrMsgMD, ErrStat, ErrMsg, 'FARM_UpdateStates')  ! MD error status <<<<<
         
         !#ifdef printthreads
         !   tm03 = omp_get_wtime()
         !   tmSF = tmSF + tm02-tm01
         !   tmSM = tmSM + tm03-tm02
         !#endif           
         
      end do    ! n_ss substepping
   
      !#ifdef printthreads
      !   tm2 = omp_get_wtime() 
      !   write(*,*)  '       Time on FAST sims: '//trim(num2lstr(tmSF))//' s.  Time on Farm MoorDyn: '//trim(num2lstr(tmSM))//' seconds'
      !#endif
      
      
   else
      CALL SetErrStat( ErrID_Fatal, 'MooringMod must be 0 or 3.', ErrStat, ErrMsg, RoutineName )
   end if
   !#ifdef printthreads   
   !  tm2 = omp_get_wtime()
   !  write(*,*) 'Total FAST and Moordyn for FF_US took '//trim(num2lstr(tm2-tm1))//' seconds.'
   !#endif 

   call AWAE_UpdateStates( t, n, farm%AWAE%u, farm%AWAE%p, farm%AWAE%x, farm%AWAE%xd, farm%AWAE%z, &
                     farm%AWAE%OtherSt, farm%AWAE%m, ErrStatAWAE, ErrMsgAWAE )       

   !#ifdef printthreads   
   !  tm3 = omp_get_wtime()
   !  write(*,*) 'AWAE_US took '//trim(num2lstr(tm3-tm2))//' seconds.'
   !  write(*,*) 'Total Farm_US took '//trim(num2lstr(tm3-tm1))//' seconds.'
   !#endif 
   
   ! update error messages from FAST's and AWAE's time steps
   DO nt = 1,farm%p%NumTurbines 
      call SetErrStat(ErrStatF(nt), ErrMsgF(nt), ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':FARM_UpdateStates') ! FAST error status
   END DO
   
   call SetErrStat(ErrStatAWAE, ErrMsgAWAE, ErrStat, ErrMsg, 'FARM_UpdateStates')  ! AWAE error status
   
   ! calculate outputs from FAST as needed by FAST.Farm
   do nt = 1,farm%p%NumTurbines
      call FWrap_CalcOutput(farm%FWrap(nt)%p, farm%FWrap(nt)%u, farm%FWrap(nt)%y, farm%FWrap(nt)%m, ErrStat2, ErrMsg2)  
         call setErrStat(ErrStat2,ErrMsg2,ErrStat,ErrMsg,RoutineName)
   end do

   
   if (ErrStat >= AbortErrLev) return

   
end subroutine FARM_UpdateStates
!---------------------------------------------------------------------------------------------------------------------------------- 
subroutine Farm_WriteOutput(n, t, farm, ErrStat, ErrMsg)
   INTEGER(IntKi),           INTENT(IN   ) :: n                               !< Time step increment number
   REAL(DbKi),               INTENT(IN   ) :: t                               !< Current simulation time in seconds
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message
                
   INTEGER(IntKi)                          :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'FARM_WriteOutput'
   INTEGER(IntKi)                          :: nt, ir, iOutDist, np, iVelPt  ! Loop counters
   REAL(ReKi)                              :: vel(3), pt(3)
   REAL(ReKi)                              :: vec_interp(3)
   REAL(ReKi)                              :: norm2_vec, delta, deltad
   
   
   ErrStat = ErrID_None
   ErrMsg = ""
   
      ! If requested write output channel data
   if ( farm%p%NumOuts > 0 ) then
    
      
         ! Define the output channel specifying the current simulation time:
      farm%m%AllOuts(  Farm_Time_Indx) = REAL( t, ReKi )


      do nt = 1, farm%p%NOutTurb
         
         !.......................................................................................
         ! Wind Turbine and its Inflow
         !.......................................................................................

            ! Orientation of rotor centerline, normal to disk
         farm%m%AllOuts(RtAxsXT(nt)) = farm%FWrap(nt)%y%xHat_Disk(1)
         farm%m%AllOuts(RtAxsYT(nt)) = farm%FWrap(nt)%y%xHat_Disk(2)
         farm%m%AllOuts(RtAxsZT(nt)) = farm%FWrap(nt)%y%xHat_Disk(3)
         
            ! Center position of hub, m
         farm%m%AllOuts(RtPosXT(nt)) = farm%FWrap(nt)%y%p_hub(1)
         farm%m%AllOuts(RtPosYT(nt)) = farm%FWrap(nt)%y%p_hub(2)
         farm%m%AllOuts(RtPosZT(nt)) = farm%FWrap(nt)%y%p_hub(3)
         
            ! Rotor diameter, m
         farm%m%AllOuts(RtDiamT(nt)) = farm%FWrap(nt)%y%D_rotor
         
            ! Nacelle-yaw error at the wake planes, deg 
         farm%m%AllOuts(YawErrT(nt)) = farm%FWrap(nt)%y%YawErr*R2D
         
            ! Ambient turbulence intensity of the wind at the rotor disk, percent
         farm%m%AllOuts(TIAmbT(nt))  = farm%AWAE%y%TI_amb(nt)*100.0_ReKi 
         
            ! Rotor-disk-averaged ambient wind speed (normal to disk, not including structural motion, local induction or wakes from upstream turbines), m/s
         farm%m%AllOuts(RtVAmbT(nt)) = farm%AWAE%y%Vx_wind_disk(nt)
         
            ! Time-filtered rotor-disk-averaged ambient wind speed (normal to disk, not including structural motion, local induction or wakes from upstream turbines), m/s
         farm%m%AllOuts(RtVAmbFiltT(nt)) = farm%WD(nt)%xd%Vx_wind_disk_filt(0) ! NOTE: filtered value will be 0 at t=0

            ! Rotor-disk-averaged relative wind speed (normal to disk, including structural motion and wakes from upstream turbines, but not including local induction), m/s
         farm%m%AllOuts(RtVRelT(nt)) = farm%FWrap(nt)%y%DiskAvg_Vx_Rel

            ! Skew azimuth angle (instantaneous)
         farm%m%AllOuts(AziSkewT(nt)) = farm%FWrap(nt)%y%psi_skew* R2D

            ! Skew azimuth angle (time-filtered)
         farm%m%AllOuts(AziSkewFiltT(nt)) = farm%WD(nt)%xd%psi_skew_filt*R2D ! NOTE: filtered value will be 0 at t=0

            ! Skew angle (instantaneous)
         farm%m%AllOuts(RtSkewT(nt)) = farm%FWrap(nt)%y%chi_skew * R2D

            ! Skew angle (time-filtered)
         farm%m%AllOuts(RtSkewFiltT(nt)) = farm%WD(nt)%xd%chi_skew_filt*R2D ! NOTE: filtered value will be 0 at t=0

            ! Rotor circulation for curled-wake model
         farm%m%AllOuts(RtGamCurlT(nt)) = farm%WD(nt)%m%GammaCurl

            !Rotor-disk averaged thrust coefficient
         farm%m%AllOuts(RtCtAvgT(nt)) = farm%WD(nt)%m%Ct_avg
         
            ! Azimuthally averaged thrust force coefficient (normal to disk), distributed radially, -
         do ir = 1, farm%p%NOutRadii
            farm%m%AllOuts(CtTN(ir, nt)) = farm%FWrap(nt)%y%AzimAvg_Ct(farm%p%OutRadii(ir)+1)  ! y%AzimAvg_Ct is a 1-based array but the user specifies 0-based node indices, so we need to add 1
         end do

         
         !.......................................................................................
         ! Wake (for an Individual Rotor)
         !.......................................................................................
         
            ! Loop over user-requested, downstream distances (OutDist), m   
         do iOutDist = 1, farm%p%NOutDist
            
            if (  farm%p%OutDist(iOutDist) >= maxval( farm%WD(nt)%y%x_plane(0:min(farm%WD(nt)%p%NumPlanes-1,n+1)) ) ) then
               
               farm%m%AllOuts(WkAxsXTD(iOutDist,nt)) = 0.0_ReKi
               farm%m%AllOuts(WkAxsYTD(iOutDist,nt)) = 0.0_ReKi
               farm%m%AllOuts(WkAxsZTD(iOutDist,nt)) = 0.0_ReKi
                                                           
                  ! Center position of the wake centerline 
               farm%m%AllOuts(WkPosXTD(iOutDist,nt)) = 0.0_ReKi
               farm%m%AllOuts(WkPosYTD(iOutDist,nt)) = 0.0_ReKi
               farm%m%AllOuts(WkPosZTD(iOutDist,nt)) = 0.0_ReKi
                                                           
                  ! Advection, deflection, and meandering  
                  !  of the wake for downstream wake volum 
               farm%m%AllOuts(WkVelXTD(iOutDist,nt)) = 0.0_ReKi
               farm%m%AllOuts(WkVelYTD(iOutDist,nt)) = 0.0_ReKi
               farm%m%AllOuts(WkVelZTD(iOutDist,nt)) = 0.0_ReKi
                                                           
                  ! Wake diameter for downstream wake volu 
               farm%m%AllOuts(WkDiamTD(iOutDist,nt)) = 0.0_ReKi
               
               do ir = 1, farm%p%NOutRadii
                  
                     ! Axial and radial wake velocity deficits for radial node, OutRadii(ir), and downstream wake volume, np, of turbine, nt, m/s
                  farm%m%AllOuts(WkDfVxTND(ir,iOutDist,nt)) = 0.0_ReKi
                  farm%m%AllOuts(WkDfVrTND(ir,iOutDist,nt)) = 0.0_ReKi
               
                     ! Total eddy viscosity, and individual contributions to the eddy viscosity from ambient turbulence and the shear layer, 
                     !  or radial node, OutRadii(ir), and downstream wake volume, np, of turbine, nt, m/s
                  farm%m%AllOuts(EddVisTND(ir,iOutDist,nt)) = 0.0_ReKi
                  farm%m%AllOuts(EddAmbTND(ir,iOutDist,nt)) = 0.0_ReKi
                  farm%m%AllOuts(EddShrTND(ir,iOutDist,nt)) = 0.0_ReKi
                  
               end do  

            else
               
                  ! Find wake volume which contains the user-requested downstream location.
               do np = 0, min(farm%WD(nt)%p%NumPlanes-2 , n)

                  if ( ( farm%p%OutDist(iOutDist) >= farm%WD(nt)%y%x_plane(np) ) .and. ( farm%p%OutDist(iOutDist) < farm%WD(nt)%y%x_plane(np+1) ) ) then   ! A wake volume has been found

                     delta = ( farm%p%OutDist(iOutDist) - farm%WD(nt)%y%x_plane(np) ) / ( farm%WD(nt)%y%x_plane(np+1) - farm%WD(nt)%y%x_plane(np) )
                     deltad = (1.0_ReKi-delta)

                     vec_interp       =  delta*farm%WD(nt)%y%xhat_plane(:, np+1) + deltad*farm%WD(nt)%y%xhat_plane(:, np)
                     norm2_vec        =  TwoNorm( vec_interp ) 
                        ! Orientation of the wake centerline for downstream wake volume, np, of turbine, nt, in the global coordinate system, -
                     farm%m%AllOuts(WkAxsXTD(iOutDist,nt)) = vec_interp(1)/norm2_vec
                     farm%m%AllOuts(WkAxsYTD(iOutDist,nt)) = vec_interp(2)/norm2_vec
                     farm%m%AllOuts(WkAxsZTD(iOutDist,nt)) = vec_interp(3)/norm2_vec 

                     if ( farm%AWAE%m%parallelFlag(np,nt) ) then
                        vec_interp       =  delta*farm%WD(nt)%y%p_plane(:, np+1) + deltad*farm%WD(nt)%y%p_plane(:, np)
                     else
                        vec_interp = delta*farm%AWAE%m%rhat_e(:,np,nt) + deltad*farm%AWAE%m%rhat_s(:,np,nt)
                        vec_interp = delta*farm%AWAE%m%pvec_ce(:,np,nt) + deltad*farm%AWAE%m%pvec_cs(:,np,nt) + ( delta*farm%AWAE%m%r_e(np,nt) + deltad*farm%AWAE%m%r_s(np,nt) )* vec_interp / TwoNorm(vec_interp)
                     end if
               
                        ! Center position of the wake centerline for downstream wake volume, np, of turbine, nt, in the global coordinate system, m
                     farm%m%AllOuts(WkPosXTD(iOutDist,nt)) = vec_interp(1)
                     farm%m%AllOuts(WkPosYTD(iOutDist,nt)) = vec_interp(2)
                     farm%m%AllOuts(WkPosZTD(iOutDist,nt)) = vec_interp(3)

                        ! Advection, deflection, and meandering velocity (not including the horizontal wake-deflection correction) 
                        !  of the wake for downstream wake volume, np, of turbine, nt, in the global coordinate system, m/s
                     farm%m%AllOuts(WkVelXTD(iOutDist,nt)) = delta*farm%AWAE%y%V_plane(1,np+1,nt) + deltad*farm%AWAE%y%V_plane(1,np,nt)
                     farm%m%AllOuts(WkVelYTD(iOutDist,nt)) = delta*farm%AWAE%y%V_plane(2,np+1,nt) + deltad*farm%AWAE%y%V_plane(2,np,nt)
                     farm%m%AllOuts(WkVelZTD(iOutDist,nt)) = delta*farm%AWAE%y%V_plane(3,np+1,nt) + deltad*farm%AWAE%y%V_plane(3,np,nt)

                        ! Wake diameter for downstream wake volume, np, of turbine, nt, m
                     farm%m%AllOuts(WkDiamTD(iOutDist,nt)) = delta*farm%WD(nt)%y%D_wake(np+1) + deltad*farm%WD(nt)%y%D_wake(np)  !farm%AWAE%u%D_wake(np,nt)
            
                     if (farm%WD(nt)%p%Mod_Wake == Mod_Wake_Polar) then
                        do ir = 1, farm%p%NOutRadii
                     
                              ! Axial and radial wake velocity deficits for radial node, OutRadii(ir), and downstream wake volume, np, of turbine, nt, m/s
                           farm%m%AllOuts(WkDfVxTND(ir,iOutDist,nt)) = delta*farm%WD(nt)%y%Vx_wake(farm%p%OutRadii(ir),np+1) + deltad*farm%WD(nt)%y%Vx_wake(farm%p%OutRadii(ir),np)
                           farm%m%AllOuts(WkDfVrTND(ir,iOutDist,nt)) = delta*farm%WD(nt)%y%Vr_wake(farm%p%OutRadii(ir),np+1) + deltad*farm%WD(nt)%y%Vr_wake(farm%p%OutRadii(ir),np)
                  
                              ! Total eddy viscosity, and individual contributions to the eddy viscosity from ambient turbulence and the shear layer, 
                              !  or radial node, OutRadii(ir), and downstream wake volume, np, of turbine, nt, m/s
                           farm%m%AllOuts(EddVisTND(ir,iOutDist,nt)) = delta*farm%WD(nt)%m%vt_tot(farm%p%OutRadii(ir),np+1) + deltad*farm%WD(nt)%m%vt_tot(farm%p%OutRadii(ir),np)
                           farm%m%AllOuts(EddAmbTND(ir,iOutDist,nt)) = delta*farm%WD(nt)%m%vt_amb(farm%p%OutRadii(ir),np+1) + deltad*farm%WD(nt)%m%vt_amb(farm%p%OutRadii(ir),np)
                           farm%m%AllOuts(EddShrTND(ir,iOutDist,nt)) = delta*farm%WD(nt)%m%vt_shr(farm%p%OutRadii(ir),np+1) + deltad*farm%WD(nt)%m%vt_shr(farm%p%OutRadii(ir),np)
                     
                        end do  
                     else
                         ! These outputs are invalid for Curl and Cartesian
                     endif

                  else if ( ( farm%p%OutDist(iOutDist) >= farm%WD(nt)%y%x_plane(np+1) ) .and. ( farm%p%OutDist(iOutDist) < farm%WD(nt)%y%x_plane(np) ) ) then   ! Overlapping wake volumes result in invalid output
               
                     farm%m%AllOuts(WkAxsXTD(iOutDist,nt)) = 0.0_ReKi
                     farm%m%AllOuts(WkAxsYTD(iOutDist,nt)) = 0.0_ReKi
                     farm%m%AllOuts(WkAxsZTD(iOutDist,nt)) = 0.0_ReKi
                                                           
                        ! Center position of the wake centerline 
                     farm%m%AllOuts(WkPosXTD(iOutDist,nt)) = 0.0_ReKi
                     farm%m%AllOuts(WkPosYTD(iOutDist,nt)) = 0.0_ReKi
                     farm%m%AllOuts(WkPosZTD(iOutDist,nt)) = 0.0_ReKi
                                                           
                        ! Advection, deflection, and meandering  
                        !  of the wake for downstream wake volum 
                     farm%m%AllOuts(WkVelXTD(iOutDist,nt)) = 0.0_ReKi
                     farm%m%AllOuts(WkVelYTD(iOutDist,nt)) = 0.0_ReKi
                     farm%m%AllOuts(WkVelZTD(iOutDist,nt)) = 0.0_ReKi
                                                           
                        ! Wake diameter for downstream wake volu 
                     farm%m%AllOuts(WkDiamTD(iOutDist,nt)) = 0.0_ReKi
               
                     do ir = 1, farm%p%NOutRadii
                  
                           ! Axial and radial wake velocity deficits for radial node, OutRadii(ir), and downstream wake volume, np, of turbine, nt, m/s
                        farm%m%AllOuts(WkDfVxTND(ir,iOutDist,nt)) = 0.0_ReKi
                        farm%m%AllOuts(WkDfVrTND(ir,iOutDist,nt)) = 0.0_ReKi
               
                           ! Total eddy viscosity, and individual contributions to the eddy viscosity from ambient turbulence and the shear layer, 
                           !  or radial node, OutRadii(ir), and downstream wake volume, np, of turbine, nt, m/s
                        farm%m%AllOuts(EddVisTND(ir,iOutDist,nt)) = 0.0_ReKi
                        farm%m%AllOuts(EddAmbTND(ir,iOutDist,nt)) = 0.0_ReKi
                        farm%m%AllOuts(EddShrTND(ir,iOutDist,nt)) = 0.0_ReKi
                  
                     end do  
              
                     exit

                  end if
                  
               end do  

            end if

         end do

      end do
      
      !.......................................................................................
      ! Ambient Wind and Array Effects
      !.......................................................................................
      
         ! Loop over user-requested, velocity locations  
      do iVelPt = 1, farm%p%NWindVel        

            ! Determine the requested pt in grid coordinates
         pt = (/farm%p%WindVelX(iVelPt), farm%p%WindVelY(iVelPt),farm%p%WindVelZ(iVelPt)/)
         pt(1) = (pt(1) - farm%p%X0_low)/ farm%p%dX_low
         pt(2) = (pt(2) - farm%p%Y0_low)/ farm%p%dY_low
         pt(3) = (pt(3) - farm%p%Z0_low)/ farm%p%dZ_low
         
            ! Ambient wind velocity (not including wakes) for point, pt,  in global coordinates (from the low-resolution domain), m/s
         call TrilinearInterpRegGrid(farm%AWAE%m%Vamb_low, pt, (/farm%p%nX_low,farm%p%nY_low,farm%p%nZ_low/), vel)
         farm%m%AllOuts(WVAmbX(iVelPt)) = vel(1)
         farm%m%AllOuts(WVAmbY(iVelPt)) = vel(2)
         farm%m%AllOuts(WVAmbZ(iVelPt)) = vel(3)
         
            ! Disturbed wind velocity (including wakes) for point, pt,  in global coordinates (from the low-resolution domain), m/s
         call TrilinearInterpRegGrid(farm%AWAE%m%Vdist_low_full, pt, (/farm%p%nX_low,farm%p%nY_low,farm%p%nZ_low/), vel)
         farm%m%AllOuts(WVDisX(iVelPt)) = vel(1)
         farm%m%AllOuts(WVDisY(iVelPt)) = vel(2)
         farm%m%AllOuts(WVDisZ(iVelPt)) = vel(3)
            
              
      end do
      
      

      
      call WriteFarmOutputToFile(t, farm, ErrStat2, ErrMsg2)
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         
   end if
end subroutine Farm_WriteOutput
!---------------------------------------------------------------------------------------------------------------------------------- 
!> This routine calculates outputs at each time increment and solves for the inputs at the next step. 
!! The calculate output algorithm: \n 
!!    -  In parallel:  
!!       1. call WD_CO and transfer y_WD to u_AWAE 
!!       2. Transfer y_F to u_WD 
!!    -  CALL AWAE_CO 
!!    -  Transfer y_AWAE to u_F and u_WD 
!!    -  Write Output to File
subroutine FARM_CalcOutput(t, farm, ErrStat, ErrMsg)
   REAL(DbKi),               INTENT(IN   ) :: t                               !< Current simulation time in seconds
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

   INTEGER(IntKi)                          :: nt,j
   INTEGER(IntKi)                          :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'FARM_CalcOutput'
   INTEGER(IntKi)                          :: n                               ! time step increment number
!   REAL(DbKi)                              :: tm1
   ErrStat = ErrID_None
   ErrMsg = ""
   
  ! tm1 = omp_get_wtime()
   
   !.......................................................................................
   ! calculate module outputs and perform some input-output solves (steps 1. and 2. and 3. can be done in parallel,
   !  but be careful that step 3 doesn't modify the inputs to steps 1 or 2)
   !.......................................................................................
   
      !--------------------
      ! 1. call WD_CO and transfer y_WD to u_AWAE        
   
   !$OMP PARALLEL DO DEFAULT (shared) PRIVATE(nt, ErrStat2, ErrMsg2) schedule(runtime)
   DO nt = 1,farm%p%NumTurbines
      
      call WD_CalcOutput( t, farm%WD(nt)%u, farm%WD(nt)%p, farm%WD(nt)%x, farm%WD(nt)%xd, farm%WD(nt)%z, &
                     farm%WD(nt)%OtherSt, farm%WD(nt)%y, farm%WD(nt)%m, ErrStat2, ErrMsg2 )         
      if (ErrStat2 >= AbortErrLev) then
         !$OMP CRITICAL  ! Needed to avoid data race on ErrStat and ErrMsg
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)       
         !$OMP END CRITICAL
      endif
   END DO
   !$OMP END PARALLEL DO  
   if (ErrStat >= AbortErrLev) return

   ! IO operation, not done using OpenMP
   DO nt = 1,farm%p%NumTurbines
      call WD_WritePlaneOutputs( t, farm%WD(nt)%u, farm%WD(nt)%p, farm%WD(nt)%x, farm%WD(nt)%xd, farm%WD(nt)%z, &
                     farm%WD(nt)%OtherSt, farm%WD(nt)%y, farm%WD(nt)%m, ErrStat2, ErrMsg2 )         
      if (ErrStat2 >= AbortErrLev) then
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)       
      endif
   END DO
   if (ErrStat >= AbortErrLev) return


   call Transfer_WD_to_AWAE(farm)
   
      !--------------------
      ! 2. Transfer y_F to u_WD         
         
   call Transfer_FAST_to_WD(farm)
         
   !.......................................................................................
   ! calculate AWAE outputs and perform rest of input-output solves
   !.......................................................................................
   
      !--------------------
      ! 1. call AWAE_CO 
   call AWAE_CalcOutput( t, farm%AWAE%u, farm%AWAE%p, farm%AWAE%x, farm%AWAE%xd, farm%AWAE%z, &
                     farm%AWAE%OtherSt, farm%AWAE%y, farm%AWAE%m, ErrStat2, ErrMsg2 )         
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)

      !--------------------
      ! 2. Transfer y_AWAE to u_F  and u_WD   
   call Transfer_AWAE_to_WD(farm)   
   
   
   !.......................................................................................
   ! Write Output to File
   !.......................................................................................
      ! NOTE: Visualization data is output via the AWAE module
   n = nint(t/farm%p%DT_low)
   call Farm_WriteOutput(n, t, farm, ErrStat2, ErrMsg2)
      call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
   
   !.......................................................................................
   ! Write shared moorings visualization
   !.......................................................................................

   ! Write visualization meshes
   if (farm%p%MooringMod == 3) then
      if (farm%MD%p%VisMeshes) then
         if (allocated(farm%MD%y%VisLinesMesh)) then
            do j=1,size(farm%MD%y%VisLinesMesh)
               if (farm%MD%y%VisLinesMesh(j)%Committed) then
                  call MeshWrVTK((/0.0_SiKi,0.0_SiKi,0.0_SiKi/), farm%MD%y%VisLinesMesh(j), trim(farm%MD%VTK_OutFileRoot)//'.MD_Line'//trim(Num2LStr(j)), farm%MD%VTK_count, .false., ErrSTat2, ErrMsg2, farm%MD%VTK_tWidth )
               endif
            enddo
         endif
         if (allocated(farm%MD%y%VisRodsMesh)) then
            do j=1,size(farm%MD%y%VisRodsMesh)
               if (farm%MD%y%VisRodsMesh(j)%Committed) then
                  call MeshWrVTK((/0.0_SiKi,0.0_SiKi,0.0_SiKi/), farm%MD%y%VisRodsMesh(j), trim(farm%MD%VTK_OutFileRoot)//'.MD_Rod'//trim(Num2LStr(j)), farm%MD%VTK_count, .false., ErrSTat2, ErrMsg2, farm%MD%VTK_tWidth )
               endif
            enddo
         endif
         farm%MD%VTK_Count = farm%MD%VTK_Count + 1
      endif
   endif

 !  write(*,*) 'Total Farm_CO-serial took '//trim(num2lstr(omp_get_wtime()-tm1))//' seconds.' 
   
end subroutine FARM_CalcOutput
!----------------------------------------------------------------------------------------------------------------------------------
!> This routine ends the modules used in this simulation. It does not exit the program.
!!    -  In parallel:
!!       1. CALL WAT_End 
!!       2. CALL AWAE_End
!!       3. CALL WD_End
!!       4. CALL FWrap_End
!!       5. CALL MD_End
!!    -  Close Output File   
subroutine FARM_End(farm, ErrStat, ErrMsg)
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm  
   INTEGER(IntKi),           INTENT(  OUT) :: ErrStat                         !< Error status
   CHARACTER(*),             INTENT(  OUT) :: ErrMsg                          !< Error message

   INTEGER(IntKi)                          :: nt                    
   INTEGER(IntKi)                          :: ErrStat2                        ! Temporary Error status
   CHARACTER(ErrMsgLen)                    :: ErrMsg2                         ! Temporary Error message
   CHARACTER(*),   PARAMETER               :: RoutineName = 'FARM_End'
   
   
   
   ErrStat = ErrID_None
   ErrMsg = ""
   
   !.......................................................................................
   ! end all modules (1-4 can be done in parallel) 
   !.......................................................................................
   
      !--------------
      ! 1. end AWAE
   if (farm%WAT_IfW%IsInitialized) then
      call InflowWind_End(farm%WAT_IfW%u, farm%WAT_IfW%p, farm%WAT_IfW%x, farm%WAT_IfW%xd, farm%WAT_IfW%z, &
                     farm%WAT_IfW%OtherSt, farm%WAT_IfW%y, farm%WAT_IfW%m, ErrStat2, ErrMsg2 )
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         farm%WAT_IfW%IsInitialized = .false.
   endif

      !--------------
      ! 2. end AWAE
   if (farm%AWAE%IsInitialized) then      
      call AWAE_End( farm%AWAE%u, farm%AWAE%p, farm%AWAE%x, farm%AWAE%xd, farm%AWAE%z, &
                     farm%AWAE%OtherSt, farm%AWAE%y, farm%AWAE%m, ErrStat2, ErrMsg2 )
         call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
         farm%AWAE%IsInitialized = .false.
   end if      
      
      !--------------
      ! 3. end WakeDynamics
   if (allocated(farm%WD)) then
      DO nt = 1,farm%p%NumTurbines
         if (farm%WD(nt)%IsInitialized) then      
            call WD_End( farm%WD(nt)%u, farm%WD(nt)%p, farm%WD(nt)%x, farm%WD(nt)%xd, farm%WD(nt)%z, &
                         farm%WD(nt)%OtherSt, farm%WD(nt)%y, farm%WD(nt)%m, ErrStat2, ErrMsg2 )
               call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)
            farm%WD(nt)%IsInitialized = .false.
         end if      
      END DO
   end if
   
      !--------------
      ! 5. End each instance of FAST (each instance of FAST can be done in parallel, too)   
   if (allocated(farm%FWrap)) then
      DO nt = 1,farm%p%NumTurbines
         if (farm%FWrap(nt)%IsInitialized) then
            CALL FWrap_End( farm%FWrap(nt)%u, farm%FWrap(nt)%p, farm%FWrap(nt)%x, farm%FWrap(nt)%xd, farm%FWrap(nt)%z, &
                            farm%FWrap(nt)%OtherSt, farm%FWrap(nt)%y, farm%FWrap(nt)%m, ErrStat2, ErrMsg2 )
            call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, 'T'//trim(num2lstr(nt))//':'//RoutineName)
            farm%FWrap(nt)%IsInitialized = .false.
         end if
      END DO
   end if   
   
      !--------------
      ! 6. End farm-level MoorDyn
   if (farm%p%MooringMod == 3 .and. allocated(farm%MD%Input)) then
      call MD_End(farm%MD%Input(1), farm%MD%p, farm%MD%x, farm%MD%xd, farm%MD%z, farm%MD%OtherSt, farm%MD%y, farm%MD%m, ErrStat2, ErrMsg2)
      call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName )
      !TODO: any related items need to be cleared?
   end if
   
   
   !.......................................................................................
   ! close output file
   !.......................................................................................
   call Farm_EndOutput( farm, ErrStat2, ErrMsg2 )
      call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg,RoutineName)
      
      
   !.......................................................................................
   ! clear all data from 'farm' structure
   !.......................................................................................
   call Farm_DestroyAll_FastFarm_Data( farm, ErrStat2, ErrMsg2 )
      call SetErrStat(ErrStat2, ErrMsg2, ErrStat, ErrMsg, RoutineName)
   
end subroutine FARM_End
!----------------------------------------------------------------------------------------------------------------------------------
SUBROUTINE Transfer_FAST_to_WD(farm)
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  

   integer(intKi)  :: nt
   
   DO nt = 1,farm%p%NumTurbines   
      farm%WD(nt)%u%xhat_disk      = farm%FWrap(nt)%y%xHat_Disk       ! Orientation of rotor centerline, normal to disk
      farm%WD(nt)%u%psi_skew       = farm%FWrap(nt)%y%psi_skew        ! Azimuth angle from the nominally vertical axis in the disk plane to the vector about which the inflow skew angle is defined
      farm%WD(nt)%u%chi_skew       = farm%FWrap(nt)%y%chi_skew        ! Inflow skew angle
      farm%WD(nt)%u%p_hub          = farm%FWrap(nt)%y%p_hub           ! Center position of hub, m
      farm%WD(nt)%u%D_rotor        = farm%FWrap(nt)%y%D_rotor         ! Rotor diameter, m
      farm%WD(nt)%u%Vx_rel_disk    = farm%FWrap(nt)%y%DiskAvg_Vx_Rel  ! Rotor-disk-averaged relative wind speed (ambient + deficits + motion), normal to disk, m/s
      farm%WD(nt)%u%Ct_azavg       = farm%FWrap(nt)%y%AzimAvg_Ct      ! Azimuthally averaged thrust force coefficient (normal to disk), distributed radially, -
      farm%WD(nt)%u%Cq_azavg       = farm%FWrap(nt)%y%AzimAvg_Cq      ! Azimuthally averaged torque force coefficient (normal to disk), distributed radially, -
      farm%WD(nt)%u%YawErr         = farm%FWrap(nt)%y%YawErr          ! Nacelle-yaw error at the wake planes, rad   
   END DO
   
END SUBROUTINE Transfer_FAST_to_WD
!----------------------------------------------------------------------------------------------------------------------------------
SUBROUTINE Transfer_AWAE_to_WD(farm)
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  

   integer(intKi)  :: nt
   
   DO nt = 1,farm%p%NumTurbines
      farm%WD(nt)%u%V_plane      = farm%AWAE%y%V_plane(:,:,nt)   ! Advection, deflection, and meandering velocity of wake planes, m/s
      farm%WD(nt)%u%Vx_wind_disk = farm%AWAE%y%Vx_wind_disk(nt)  ! Rotor-disk-averaged ambient wind speed, normal to planes, m/s
      farm%WD(nt)%u%TI_amb       = farm%AWAE%y%TI_amb(nt)        ! Ambient turbulence intensity of wind at rotor disk
   END DO
   
END SUBROUTINE Transfer_AWAE_to_WD
!----------------------------------------------------------------------------------------------------------------------------------
SUBROUTINE Transfer_WD_to_AWAE(farm)
   type(All_FastFarm_Data),  INTENT(INOUT) :: farm                            !< FAST.Farm data  

   integer(intKi)  :: nt
   
   DO nt = 1,farm%p%NumTurbines   
      farm%AWAE%u%xhat_plane(:,:,nt) = farm%WD(nt)%y%xhat_plane     ! Orientations of wake planes, normal to wake planes, for each turbine
      farm%AWAE%u%p_plane(:,:,nt)    = farm%WD(nt)%y%p_plane        ! Center positions of wake planes for each turbine
      farm%AWAE%u%Vx_wake(:,:,:,nt)  = farm%WD(nt)%y%Vx_wake2       ! Axial wake velocity deficit at wake planes, distributed radially, for each turbine
      farm%AWAE%u%Vy_wake(:,:,:,nt)  = farm%WD(nt)%y%Vy_wake2       ! Horizontal wake velocity deficit at wake planes, distributed radially, for each turbine
      farm%AWAE%u%Vz_wake(:,:,:,nt)  = farm%WD(nt)%y%Vz_wake2       ! "Vertical" wake velocity deficit at wake planes, distributed radially, for each turbine
      farm%AWAE%u%D_wake(:,nt)       = farm%WD(nt)%y%D_wake         ! Wake diameters at wake planes for each turbine
      if (farm%p%WAT /= Mod_WAT_None) then
         farm%AWAE%u%WAT_k(:,:,:,nt) = farm%WD(nt)%y%WAT_k          ! scaling factor for each wake plane for WAT
      endif
   END DO
   
END SUBROUTINE Transfer_WD_to_AWAE
!----------------------------------------------------------------------------------------------------------------------------------
END MODULE FAST_Farm_Subs
!**********************************************************************************************************************************
