/*
--------------------------------------------------------------------------------

	Revelation Shaders

	Copyright (C) 2024 HaringPro
	Apache License 2.0

	Pass: Deferred lighting and sky combination
		  Compute specular reflections

--------------------------------------------------------------------------------
*/

#define PASS_DEFERRED_LIGHTING

//======// Utility //=============================================================================//

#include "/lib/Utility.glsl"

//======// Output //==============================================================================//

/* RENDERTARGETS: 0 */
out vec3 sceneOut;

//======// Uniform //=============================================================================//

writeonly uniform image2D colorimg8;

uniform sampler2D cloudOriginTex;

#include "/lib/universal/Uniform.glsl"

//======// SSBO //================================================================================//

#include "/lib/universal/SSBO.glsl"

//======// Struct //==============================================================================//

#include "/lib/universal/Material.glsl"

//======// Function //============================================================================//

#include "/lib/universal/Transform.glsl"
#include "/lib/universal/Fetch.glsl"
#include "/lib/universal/Random.glsl"

#include "/lib/atmosphere/Common.glsl"
#include "/lib/atmosphere/PrecomputedAtmosphericScattering.glsl"
#include "/lib/atmosphere/Celestial.glsl"

#ifdef CLOUD_SHADOWS
	#include "/lib/atmosphere/clouds/Shadows.glsl"
#endif

#include "/lib/lighting/Common.glsl"
#include "/lib/lighting/shadow/Render.glsl"

#if AO_ENABLED > 0 && !defined SSILVB_ENABLED
	#include "/lib/lighting/SSAO.glsl"
	#include "/lib/lighting/GTAO.glsl"
#endif

#include "/lib/SpatialUpscale.glsl"

#ifdef RAIN_PUDDLES
	#include "/lib/surface/RainPuddle.glsl"
#endif

//======// Main //================================================================================//
void main() {
	ivec2 screenTexel = ivec2(gl_FragCoord.xy);
    vec2 screenCoord = gl_FragCoord.xy * viewPixelSize;

	vec3 screenPos = vec3(screenCoord, FetchDepthFix(screenTexel));
	vec3 viewPos = ScreenToViewSpace(screenPos);

	#if defined DISTANT_HORIZONS
		bool dhTerrainMask = screenPos.z > 1.0 - EPS;
		if (dhTerrainMask) {
			screenPos.z = loadDepth0DH(screenTexel);
			viewPos = ScreenToViewSpaceDH(screenPos);
		}
	#endif

	vec3 worldPos = mat3(gbufferModelViewInverse) * viewPos;
	vec3 worldDir = normalize(worldPos);
	uvec4 gbufferData0 = loadGbufferData0(screenTexel);

	uint materialID = gbufferData0.y;

	vec3 albedo = sRGBtoLinear(loadAlbedo(screenTexel));

	float dither = BlueNoiseTemporal(screenTexel);

	sceneOut = vec3(0.0);

	if (screenPos.z > 1.0 - EPS + float(materialID)) {
		vec2 skyViewCoord = FromSkyViewLutParams(worldDir);
		sceneOut = textureBicubic(skyViewTex, skyViewCoord).rgb;

		if (!RayIntersectsGround(viewerHeight, worldDir.y)) {
			vec3 celestial = RenderSun(worldDir, worldSunVector);
			vec3 vanillaMoon = albedo;

			#ifdef GALAXY
				celestial += mix(RenderGalaxy(worldDir), vanillaMoon, step(0.06, vanillaMoon.g));
			#else
				celestial += mix(RenderStars(worldDir), vanillaMoon, step(0.06, vanillaMoon.g));
			#endif

			vec3 transmittance = GetTransmittanceToTopAtmosphereBoundary(viewerHeight, worldDir.y);
			sceneOut += celestial * mix(vec3(1.0), transmittance, step(viewerHeight, atmosphereModel.top_radius));
		}

		#ifdef CLOUDS
			#ifdef CLOUD_TAAU_ENABLED
				vec4 cloudData = texture(cloudReconstructTex, screenCoord);
			#else
				// Dither offset
				screenCoord += viewPixelSize * (dither - 0.5);
				vec4 cloudData = textureBicubic(cloudOriginTex, screenCoord);
			#endif
			sceneOut = sceneOut * cloudData.a + cloudData.rgb;
		#endif

		imageStore(colorimg8, screenTexel, vec4(0.0));
	} else {
		worldPos += gbufferModelViewInverse[3].xyz;

		vec3 flatNormal = FetchFlatNormal(gbufferData0);
		#ifdef NORMAL_MAPPING
			vec3 worldNormal = FetchWorldNormal(gbufferData0);
		#else
			vec3 worldNormal = flatNormal;
		#endif
		vec3 viewNormal = mat3(gbufferModelView) * worldNormal;

		vec2 lightmap = Unpack2x8U(gbufferData0.x);

		#if defined SPECULAR_MAPPING && defined MC_SPECULAR_MAP
			vec4 specularTex = loadGbufferData1(screenTexel);

			// Compute rain puddles
			#ifdef RAIN_PUDDLES
				if (wetnessCustom > 1e-2) {
					if (clamp(materialID, 1000u, 1002u) != materialID && materialID != 20u && materialID != 40u) {
						CalculateRainPuddles(albedo, worldNormal, specularTex.rgb, worldPos, flatNormal, lightmap.y);
					}
				}
			#endif

			Material material = GetMaterialData(specularTex);
			imageStore(colorimg8, screenTexel, specularTex);
		#else
			Material material = Material(1.0, 0.0, 0.0, false, false);
		#endif

		float sssAmount = 0.0;
		#if SUBSURFACE_SCATTERING_MODE < 2
			// Hard-coded sss amount for certain materials
			switch (materialID) {
				case 1000u: case 1001u: case 1002u: case 1003u: case 27u: case 28u: // Plants
					sssAmount = 0.6;
					break;
				case 13u: // Leaves
					sssAmount = 0.85;
					break;
				case 37u: case 39u: // Weak SSS
					sssAmount = 0.5;
					break;
				case 38u: case 51u: // Strong SSS
					sssAmount = 0.8;
					break;
				case 40u: // Particles
					sssAmount = 0.35;
					break;
			}
		#endif
		#if TEXTURE_FORMAT == 0 && SUBSURFACE_SCATTERING_MODE > 0 && defined SPECULAR_MAPPING
			sssAmount = max(sssAmount, specularTex.b * step(64.5 * r255, specularTex.b));
		#endif

		// Remap sss amount to [0, 1] range
		sssAmount = remap(64.0 * r255, 1.0, sssAmount) * eyeSkylightSmooth * SUBSURFACE_SCATTERING_STRENGTH;

		// Ambient occlusion
		#if AO_ENABLED > 0 && !defined SSILVB_ENABLED
			vec3 ao = vec3(1.0);
			#if AO_ENABLED == 1
				ao.x = CalculateSSAO(screenCoord, viewPos, viewNormal, SampleStbnUnitvec2(screenTexel, frameCounter));
			#else
				ao.x = CalculateGTAO(screenCoord, viewPos, viewNormal, SampleStbnVec2(screenTexel, frameCounter));
			#endif

			#ifdef AO_MULTI_BOUNCE
				ao = ApproxMultiBounce(ao.x, albedo);
			#else
				ao = vec3(ao.x);
			#endif
		#else
			const float ao = 1.0;
		#endif

		// Cloud shadows
		#ifdef CLOUD_SHADOWS
			// float cloudShadow = CalculateCloudShadows(worldPos);
			vec2 cloudShadowCoord = WorldToCloudShadowScreenPos(worldPos).xy + (dither - 0.5) / textureSize(cloudShadowTex, 0);
			float cloudShadow = textureBicubic(cloudShadowTex, saturate(cloudShadowCoord)).x;
		#else
			float cloudShadow = 1.0 - wetness * 0.96;
		#endif

		// Sunlight
		vec3 sunlightMult = cloudShadow * saturate(lightmap.y * 1e6 + float(isEyeInWater)) * global.light.directIlluminance;
		vec3 specularHighlight = vec3(0.0);

		float worldDistSquared = sdot(worldPos);
		float distanceFade = sqr(pow16(0.64 * rcp(shadowDistance * shadowDistance) * sdot(worldPos.xz)));
		#if defined DISTANT_HORIZONS
			distanceFade = saturate(distanceFade + float(dhTerrainMask));
		#endif

		float LdotV = dot(worldLightVector, -worldDir);
		float NdotL = dot(worldNormal, worldLightVector);

		bool doShadows = NdotL > 1e-3;
		bool doSss = sssAmount > 1e-3;
		bool inShadowMapRange = distanceFade < EPS;

		// Shadows and SSS
        if (doShadows || doSss) {
			vec3 shadow = sunlightMult;

			// Apply shadowmap
        	if (inShadowMapRange) {
				float distortionFactor;
				vec3 normalOffset = flatNormal * (worldDistSquared * 1e-4 + 3e-2) * (2.0 - saturate(NdotL));
				vec3 shadowScreenPos = WorldToShadowScreenSpace(worldPos + normalOffset, distortionFactor);

				if (saturate(shadowScreenPos) == shadowScreenPos) {
					vec2 blockerSearch;
					// Sub-surface scattering
					if (doSss) {
						blockerSearch = BlockerSearchSSS(shadowScreenPos, dither, 0.5 * distortionFactor);
						vec3 subsurfaceScattering = CalculateSubsurfaceScattering(albedo, sssAmount, blockerSearch.y, LdotV);

						// Formula from https://www.alanzucconi.com/2017/08/30/fast-subsurface-scattering-1/
						// float bssrdf = sqr(saturate(dot(worldDir, worldLightVector + 0.2 * worldNormal))) * 4.0;
						sceneOut += subsurfaceScattering * sunlightMult * ao;
					} else {
						blockerSearch.x = BlockerSearch(shadowScreenPos, dither, 0.5 * distortionFactor);
					}

					// Shadows
					if (doShadows) {
						shadowScreenPos.z -= (worldDistSquared * 1e-9 + 3e-6) * (1.0 + dither) / distortionFactor * shadowDistance;

						shadow *= PercentageCloserFilter(shadowScreenPos, worldPos, dither, 0.5 * blockerSearch.x * distortionFactor);
					}
				}
			}

			// Process diffuse and specular highlights
			if (doShadows && dot(shadow, vec3(1.0)) > EPS || doSss && !inShadowMapRange) {
				#ifdef SCREEN_SPACE_SHADOWS
					#if defined NORMAL_MAPPING
						vec3 viewFlatNormal = mat3(gbufferModelView) * flatNormal;
					#else
						#define viewFlatNormal viewNormal
					#endif

					shadow *= materialID == 39u ? 1.0 : ScreenSpaceShadow(viewPos, viewFlatNormal, dither, sssAmount);
				#endif

				// Apply parallax shadows
				#ifdef PARALLAX_SHADOW
					#if defined PARALLAX && !defined PARALLAX_DEPTH_WRITE
						shadow *= oms(loadSceneColor(screenTexel).x);
					#endif
				#endif

				vec3 halfway = normalize(worldLightVector - worldDir);
				float NdotV = abs(dot(worldNormal, worldDir));
				float NdotH = dot(worldNormal, halfway);
				float LdotH = dot(worldLightVector, halfway);

				// Sunlight diffuse
				vec3 sunlightDiffuse = DiffuseHammon(LdotV, NdotV, NdotL, NdotH, material.roughness, albedo);
				sunlightDiffuse += PI * SUBSURFACE_SCATTERING_BRIGHTNESS * uniformPhase * sssAmount * distanceFade;
				sceneOut += shadow * saturate(sunlightDiffuse);

				#if defined SPECULAR_MAPPING && defined MC_SPECULAR_MAP
					vec3 f0 = GetMaterialF0(material.metalness, albedo);
				#else
					const vec3 f0 = vec3(DEFAULT_DIELECTRIC_F0);
				#endif

				specularHighlight = shadow * SpecularGGX(LdotH, NdotV, NdotL, NdotH, material.roughness, f0);
			}
		}

		// Skylight and bounced sunlight
		#ifndef SSILVB_ENABLED
			if (lightmap.y > EPS) {
				// Skylight
				vec3 skylight = lightningShading;
				skylight *= 0.02 * (worldNormal.y * 0.5 + 0.5);

				// Spherical harmonics skylight
				skylight += ConvolvedReconstructSH3(global.light.skySH, worldNormal);

				sceneOut += skylight * cube(lightmap.y) * ao;

				// Bounced sunlight
				float bounce = CalculateApproxBouncedLight(worldNormal);
				bounce *= pow5(lightmap.y);
				sceneOut += bounce * sunlightMult * ao;
			}
		#endif

		// Emissive & Blocklight
		vec3 blocklightColor = blackbody(float(BLOCKLIGHT_TEMPERATURE));
		#if EMISSIVE_MODE > 0 && defined SPECULAR_MAPPING
			sceneOut += material.emissiveness * dot(albedo, vec3(0.75));
		#endif
		#if EMISSIVE_MODE < 2
			// Hard-coded emissive
			vec4 emissive = HardCodeEmissive(materialID, albedo, worldPos, blocklightColor);
			#ifndef SSILVB_ENABLED
				if (emissive.a * lightmap.x > 1e-5) {
					lightmap.x = CalculateBlocklightFalloff(lightmap.x);
					sceneOut += lightmap.x * emissive.a * (ao * oms(lightmap.x) + lightmap.x) * blocklightColor;
				}
			#endif

			sceneOut += emissive.rgb * EMISSIVE_BRIGHTNESS;
		#elif !defined SSILVB_ENABLED
			lightmap.x = CalculateBlocklightFalloff(lightmap.x);
			sceneOut += lightmap.x * (ao * oms(lightmap.x) + lightmap.x) * blocklightColor;
		#endif

		// Handheld light
		#ifdef HANDHELD_LIGHTING
			if (heldBlockLightValue + heldBlockLightValue2 > 1e-4) {
				float NdotL = saturate(dot(worldNormal, -worldDir)) * 0.8 + 0.2;
				float attenuation = rcp(1.0 + worldDistSquared) * NdotL;
				float irradiance = attenuation * max(heldBlockLightValue, heldBlockLightValue2) * HELD_LIGHT_BRIGHTNESS;

				sceneOut += irradiance * (ao - oms(ao) * sqr(attenuation)) * blocklightColor;
			}
		#endif

		// Indirect diffuse lighting
		#ifdef SSILVB_ENABLED
			#ifdef SVGF_ENABLED
				float NdotV = abs(dot(worldNormal, worldDir));
				sceneOut += SpatialUpscale(screenTexel >> 1, worldNormal, length(viewPos), NdotV);
			#else
				sceneOut += texelFetch(colortex3, screenTexel >> 1, 0).rgb;
			#endif
		#endif

		// Minimal ambient light
		sceneOut += (worldNormal.y * 0.4 + 0.6) * max(MINIMUM_AMBIENT_BRIGHTNESS, 5e-3 * nightVision) * ao;

		// Apply albedo
		sceneOut *= albedo;

		// Metallic diffuse elimination
		material.metalness *= 0.2 * lightmap.y + 0.8;
		sceneOut *= oms(material.metalness);

		// Specular highlights
		sceneOut += specularHighlight;

		// Output clamp
		sceneOut = satU16f(sceneOut);
	}
}
