/* Screen-Space Path Tracing */

#define SSPT_SPP 2 // [1 2 3 4 5 6 7 8 9 10 11 12 14 16 18 20 22 24]
#define SSPT_BOUNCES 1 // [1]
#define SSPT_RT_STEPS 16 // [1 2 3 4 5 6 7 8 9 10 11 12 14 16 18 20 22 24 26 28 30 32 34 36 38 40 42 44 46 48 50 52 54 56 58 60 62 64]

#define SSPT_RR_MIN_BOUNCES 1 // [1 2 3 4 5 6 7 8 9 10 11 12 14 16 18 20 22 24]

//================================================================================================//

vec3 SampleRaytrace(in vec3 viewPos, in vec3 viewDir, in float dither, in vec3 rayPos) {
	if (viewDir.z > max0(-viewPos.z)) return vec3(1e6);

	vec3 endPos = ViewToScreenSpace(viewDir + viewPos);
	vec3 rayDir = normalize(endPos - rayPos);

	float stepLength = minOf((step(0.0, rayDir) - rayPos) / rayDir) * rcp(float(SSPT_RT_STEPS));

	vec3 rayStep = rayDir * stepLength;
	rayPos += rayStep * dither;

	rayPos.xy *= viewSize;
	rayStep.xy *= viewSize;

	for (uint i = 0u; i < SSPT_RT_STEPS; ++i, rayPos += rayStep) {
		if (clamp(rayPos.xy, vec2(0.0), viewSize) != rayPos.xy) break;
		float sampleDepth = loadDepth0(ivec2(rayPos.xy));

		if (sampleDepth < rayPos.z) {
			float sampleDepthLinear = ScreenToViewDepth(sampleDepth);
			float traceDepthLinear = ScreenToViewDepth(rayPos.z);
			#if defined DISTANT_HORIZONS
				if (sampleDepth > 1.0 - EPS) sampleDepthLinear = ScreenToViewDepthDH(loadDepth0DH(ivec2(rayPos.xy)));
			#endif

			if (traceDepthLinear - sampleDepthLinear > 0.2 * traceDepthLinear) return vec3(rayPos.xy, sampleDepth);
		}
	}

	return vec3(1e6);
}

// struct TracingData {
// 	vec3 rayPos;
//     vec3 rayDir;
//     vec3 viewNormal;
// 	vec3 contribution;
// };

vec3 CalculateSSPT(in vec3 screenPos, in vec3 viewPos, in vec3 worldNormal, in vec2 lightmap) {
	lightmap.y *= lightmap.y * lightmap.y;

    vec3 viewNormal = mat3(gbufferModelView) * worldNormal;

    NoiseGenerator noiseGenerator = initNoiseGenerator(gl_GlobalInvocationID.xy, uint(frameCounter));

    #if defined DISTANT_HORIZONS
        float screenDepthMax = ViewToScreenDepth(ScreenToViewDepthDH(1.0));
    #else
        #define screenDepthMax 1.0
    #endif

	vec3 sum = vec3(0.0);

    for (uint spp = 0u; spp < SSPT_SPP; ++spp) {
		// Initialize tracing data.
		// TracingData target = TracingData(screenPos, vec3(0.0), viewNormal, vec3(1.0));

		// for (uint bounce = 1u; bounce <= SSPT_BOUNCES; ++bounce) {
			vec3 rayDir = SampleCosineHemisphere(viewNormal, nextVec2(noiseGenerator));

			float dither = nextFloat(noiseGenerator);
			vec3 hitPos = SampleRaytrace(viewPos + viewNormal * 1e-2, rayDir, dither, screenPos);

			if (hitPos.z < screenDepthMax) {
				vec3 sampleRadiance = texelFetch(colortex4, ivec2(hitPos.xy) >> 1, 0).rgb;
				sum += sampleRadiance;
			} else if (dot(lightmap, vec2(1.0)) > EPS) {
				// vec3 rayDirWorld = mat3(gbufferModelViewInverse) * rayDir;
				// vec3 skyRadiance = texture(skyViewTex, FromSkyViewLutParams(rayDirWorld) + vec2(0.0, 0.5)).rgb;

				float occulusion = saturate(dot(viewNormal, rayDir));
				vec3 skyRadiance = ConvolvedReconstructSH3(global.light.skySH, worldNormal);
				sum += skyRadiance * lightmap.y * occulusion;
				// break;
			}

            // Russian roulette
			// if (bounce >= SSPT_RR_MIN_BOUNCES) {
			// 	float probability = saturate(luminance(target.contribution));
			// 	if (probability < dither) break;
			// 	target.contribution *= rcp(probability);
			// }
		// }
	}

	return sum * rcp(float(SSPT_SPP));
}