/**
 * Marlin 3D Printer Firmware
 * Copyright (c) 2024 MarlinFirmware [https://github.com/MarlinFirmware/Marlin]
 *
 * Based on Sprinter and grbl.
 * Copyright (c) 2011 Camiel Gubbels / Erik van der Zalm
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 */
#pragma once

#include <Arduino.h>
#include "HAL.h"

#ifndef NUM_DIGITAL_PINS
   #error "Expected NUM_DIGITAL_PINS not found"
#endif

/**
 *  Life gets complicated if you want an easy to use 'M43 I' output (in port/pin order)
 *  because the variants in this platform do not always define all the I/O port/pins
 *  that a CPU has.
 *
 *  VARIABLES:
 *     Ard_num - Arduino pin number - defined by the platform. It is used by digitalRead and
 *               digitalWrite commands and by M42.
 *             - does not contain port/pin info
 *             - is not in port/pin order
 *             - typically a variant will only assign Ard_num to port/pins that are actually used
 *     Index - M43 counter - only used to get Ard_num
 *     x - a parameter/argument used to search the pin_array to try to find a signal name
 *         associated with a Ard_num
 *     Port_pin - port number and pin number for use with CPU registers and printing reports
 *
 *  Since M43 uses digitalRead and digitalWrite commands, only the Port_pins with an Ard_num
 *  are accessed and/or displayed.
 *
 *  Three arrays are used.
 *
 *  digitalPin[] is provided by the platform.  It consists of the Port_pin numbers in
 *  Arduino pin number order.
 *
 *  pin_array is a structure generated by the pins/pinsDebug.h header file.  It is generated by
 *  the preprocessor. Only the signals associated with enabled options are in this table.
 *  It contains:
 *    - name of the signal
 *    - the Ard_num assigned by the pins_YOUR_BOARD.h file using the platform defines.
 *        EXAMPLE:  "#define KILL_PIN  PB1" results in Ard_num of 57.  57 is then used as the
 *                  argument to digitalPinToPinName(IO) to get the Port_pin number
 *    - if it is a digital or analog signal.  PWMs are considered digital here.
 *
 *  pin_xref is a structure generated by this header file.  It is generated by the
 *  preprocessor. It is in port/pin order.  It contains just the port/pin numbers defined by the
 *  platform for this variant.
 *    - Ard_num
 *    - printable version of Port_pin
 *
 *  Routines with an "x" as a parameter/argument are used to search the pin_array to try to
 *  find a signal name associated with a port/pin.
 *
 *  NOTE -  the Arduino pin number is what is used by the M42 command, NOT the port/pin for that
 *          signal.  The Arduino pin number is listed by the M43 I command.
 */

#define NUM_ANALOG_FIRST A0

#define MODE_PIN_INPUT  0 // Input mode (reset state)
#define MODE_PIN_OUTPUT 1 // General purpose output mode
#define MODE_PIN_ALT    2 // Alternate function mode
#define MODE_PIN_ANALOG 3 // Analog mode

#define PIN_NUM(P) (P & 0x000F)
#define PIN_NUM_ALPHA_LEFT(P) (((P & 0x000F) < 10) ? ('0' + (P & 0x000F)) : '1')
#define PIN_NUM_ALPHA_RIGHT(P) (((P & 0x000F) > 9)  ? ('0' + (P & 0x000F) - 10) : 0 )
#define PORT_NUM(P) ((P  >> 4) & 0x0007)
#define PORT_ALPHA(P) ('A' + (P  >> 4))

/**
 * Translation of routines & variables used by pinsDebug.h
 */
#define NUMBER_PINS_TOTAL NUM_DIGITAL_PINS
#define VALID_PIN(ANUM) (pin_t(ANUM) >= 0 && pin_t(ANUM) < NUMBER_PINS_TOTAL)
#define digitalRead_mod(Ard_num) extDigitalRead(Ard_num)  // must use Arduino pin numbers when doing reads
#define PRINT_PIN(Q)
#define PRINT_PIN_ANALOG(p) do{ sprintf_P(buffer, PSTR(" (A%2d)  "), DIGITAL_PIN_TO_ANALOG_PIN(pin)); SERIAL_ECHO(buffer); }while(0)
#define DIGITAL_PIN_TO_ANALOG_PIN(ANUM) -1  // will report analog pin number in the print port routine

// x is a variable used to search pin_array
#define GET_ARRAY_IS_DIGITAL(x) ((bool) pin_array[x].is_digital)
#define GET_ARRAY_PIN(x) ((pin_t) pin_array[x].pin)
#define PRINT_ARRAY_NAME(x) do{ sprintf_P(buffer, PSTR("%-" STRINGIFY(MAX_NAME_LENGTH) "s"), pin_array[x].name); SERIAL_ECHO(buffer); }while(0)
#define MULTI_NAME_PAD 33 // space needed to be pretty if not first name assigned to a pin

uint8_t get_pin_mode(const pin_t Ard_num) {

  uint dir = gpio_get_dir( Ard_num);

  if (dir) return MODE_PIN_OUTPUT;
  else return MODE_PIN_INPUT;

}

bool getValidPinMode(const pin_t Ard_num) {
  const uint8_t pin_mode = get_pin_mode(Ard_num);
  return pin_mode == MODE_PIN_OUTPUT || pin_mode == MODE_PIN_ALT;  // assume all alt definitions are PWM
}

int8_t digital_pin_to_analog_pin(pin_t Ard_num) {
  Ard_num -= NUM_ANALOG_FIRST;
  return (Ard_num >= 0 && Ard_num < NUM_ANALOG_INPUTS) ? Ard_num : -1;
}

bool isAnalogPin(const pin_t Ard_num) {
  return digital_pin_to_analog_pin(Ard_num) != -1;
}

bool is_digital(const pin_t x) {
  const uint8_t pin_mode = get_pin_mode(x);
  return pin_mode == MODE_PIN_INPUT || pin_mode == MODE_PIN_OUTPUT;
}

void printPinPort(const pin_t Ard_num) {
  SERIAL_ECHOPGM("Pin: ");
  SERIAL_ECHO(Ard_num);
}

bool pwm_status(const pin_t Ard_num) {
  return get_pin_mode(Ard_num) == MODE_PIN_ALT;
}

void printPinPWM(const pin_t Ard_num) {
  if (PWM_PIN(Ard_num)) {
  }
}
