/*
 * This file is part of Cleanflight.
 *
 * Cleanflight is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cleanflight is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cleanflight.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <stdbool.h>
#include <time.h>

#include "config/parameter_group.h"

#include "common/time.h"

#define GPS_DBHZ_MIN 0
#define GPS_DBHZ_MAX 55

#define LAT 0
#define LON 1

#define GPS_DEGREES_DIVIDER 10000000L

typedef enum {
    GPS_UBLOX = 0,
    GPS_MSP,
    GPS_FAKE,
    GPS_PROVIDER_COUNT
} gpsProvider_e;

typedef enum {
    SBAS_AUTO = 0,
    SBAS_EGNOS,
    SBAS_WAAS,
    SBAS_MSAS,
    SBAS_GAGAN,
    SBAS_SPAN,
    SBAS_NONE
} sbasMode_e;

#define SBAS_MODE_MAX SBAS_GAGAN

typedef enum {
    GPS_BAUDRATE_115200 = 0,
    GPS_BAUDRATE_57600,
    GPS_BAUDRATE_38400,
    GPS_BAUDRATE_19200,
    GPS_BAUDRATE_9600,
    GPS_BAUDRATE_230400,
    GPS_BAUDRATE_460800,
    GPS_BAUDRATE_921600,
    GPS_BAUDRATE_COUNT
} gpsBaudRate_e;

typedef enum {
    GPS_AUTOCONFIG_OFF = 0,
    GPS_AUTOCONFIG_ON,
} gpsAutoConfig_e;

typedef enum {
    GPS_AUTOBAUD_OFF = 0,
    GPS_AUTOBAUD_ON
} gpsAutoBaud_e;

typedef enum {
    GPS_DYNMODEL_PEDESTRIAN = 0,
    GPS_DYNMODEL_AUTOMOTIVE,
    GPS_DYNMODEL_AIR_1G,
    GPS_DYNMODEL_AIR_2G,
    GPS_DYNMODEL_AIR_4G,
    GPS_DYNMODEL_SEA,
    GPS_DYNMODEL_MOWER,
} gpsDynModel_e;

typedef enum {
    GPS_NO_FIX = 0,
    GPS_FIX_2D,
    GPS_FIX_3D
} gpsFixType_e;

#define GPS_BAUDRATE_MAX GPS_BAUDRATE_9600

typedef struct gpsConfig_s {
    gpsProvider_e provider;
    sbasMode_e sbasMode;
    gpsAutoConfig_e autoConfig;
    gpsAutoBaud_e autoBaud;
    gpsDynModel_e dynModel;
    bool ubloxUseGalileo;
    bool ubloxUseBeidou;
    bool ubloxUseGlonass;
    uint8_t gpsMinSats;
    uint8_t ubloxNavHz;
    gpsBaudRate_e autoBaudMax;
} gpsConfig_t;

PG_DECLARE(gpsConfig_t, gpsConfig);

typedef struct gpsCoordinateDDDMMmmmm_s {
    int16_t dddmm;
    int16_t mmmm;
} gpsCoordinateDDDMMmmmm_t;

/* LLH Location in NEU axis system */
typedef struct gpsLocation_s {
    int32_t lat;    // Latitude * 1e+7
    int32_t lon;    // Longitude * 1e+7
    int32_t alt;    // Altitude in centimeters (meters * 100)
} gpsLocation_t;

#define HDOP_SCALE (100)

typedef struct gpsSolutionData_s {
    struct {
        bool hasNewData;
        bool gpsHeartbeat;  // Toggle each update
        bool validVelNE;
        bool validVelD;
        bool validEPE;      // EPH/EPV values are valid - actual accuracy
        bool validTime;
    } flags;

    gpsFixType_e fixType;
    uint8_t numSat;

    gpsLocation_t llh;
    int16_t       velNED[3];

    int16_t groundSpeed;
    int16_t groundCourse;

    uint16_t eph;   // horizontal accuracy (cm)
    uint16_t epv;   // vertical accuracy (cm)

    uint16_t hdop;  // generic HDOP value (*HDOP_SCALE)

    dateTime_t time; // GPS time in UTC

} gpsSolutionData_t;

typedef struct {
    uint16_t    lastMessageDt;
    uint32_t    errors;                // gps error counter - crc error/lost of data/sync etc..
    uint32_t    timeouts;
    uint32_t    packetCount;
} gpsStatistics_t;

extern gpsSolutionData_t gpsSol;
extern gpsStatistics_t   gpsStats;

struct magDev_s;
void gpsPreInit(void);
void gpsInit(void);
// Called periodically from GPS task. Returns true iff the GPS
// information was updated.
bool gpsUpdate(void);
void updateGpsIndicator(timeUs_t currentTimeUs);
bool isGPSHealthy(void);
bool isGPSHeadingValid(void);
struct serialPort_s;
void gpsEnablePassthrough(struct serialPort_s *gpsPassthroughPort);
void mspGPSReceiveNewData(const uint8_t * bufferPtr);

const char *getGpsHwVersion(void);
uint8_t getGpsProtoMajorVersion(void);
uint8_t getGpsProtoMinorVersion(void);

int getGpsBaudrate(void);
int gpsBaudRateToInt(gpsBaudRate_e baudrate);

#if defined(USE_GPS_FAKE)
void gpsFakeSet(
    gpsFixType_e fixType,
    uint8_t numSat,
    int32_t lat, 
    int32_t lon, 
    int32_t alt, 
    int16_t groundSpeed, 
    int16_t groundCourse, 
    int16_t velNED_X,  
    int16_t velNED_Y,  
    int16_t velNED_Z,
    time_t time);
#endif
