/*
 * This file is part of Cleanflight.
 *
 * Cleanflight is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cleanflight is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cleanflight.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdbool.h>
#include <stdint.h>

#include <platform.h>

#include "build/build_config.h"

#include "common/maths.h"
#include "common/utils.h"

#include "drivers/pitotmeter/pitotmeter.h"
#include "drivers/pitotmeter/pitotmeter_adc.h"
#include "drivers/adc.h"

#if defined(USE_ADC) && defined(USE_PITOT_ADC)

/*
 * NXP MPXV7002DP differential pressure sensor
 *
 */

#define PITOT_ADC_VOLTAGE_SCALER        (2.0f / 1.0f)       // MPXV7002DP is 5V device, assumed resistive divider 1K:1K
#define PITOT_ADC_VOLTAGE_ZERO          (2.5f)              // Pressure offset is 2.5V
#define PITOT_ADC_VOLTAGE_TO_PRESSURE   (1000.0f)           // 1V/kPa = 1000 Pa/V

static bool adcPitotStart(pitotDev_t *pitot)
{
    UNUSED(pitot);
    return true;
}

static bool adcPitotRead(pitotDev_t *pitot)
{
    UNUSED(pitot);
    return true;
}

static void adcPitotCalculate(pitotDev_t *pitot, float *pressure, float *temperature)
{
    UNUSED(pitot);
    uint16_t adcRaw = adcGetChannel(ADC_AIRSPEED);
    float voltage = (float)adcRaw * (3.3f / 4095.0f);  // 12 bit ADC with 3.3V VREF

    if (pressure)
        *pressure = (voltage * PITOT_ADC_VOLTAGE_SCALER - PITOT_ADC_VOLTAGE_ZERO) * PITOT_ADC_VOLTAGE_TO_PRESSURE;
    if (temperature)
        *temperature = SSL_AIR_TEMPERATURE; // Temperature at standard sea level
}

bool adcPitotDetect(pitotDev_t *pitot)
{
    pitot->delay = 10000;
    pitot->calibThreshold = 0.00001f;   // TODO :: should be tested !!!
    pitot->start = adcPitotStart;
    pitot->get = adcPitotRead;
    pitot->calculate = adcPitotCalculate;
    return adcIsFunctionAssigned(ADC_AIRSPEED);
}
#endif
