/*
 * This file is part of Cleanflight.
 *
 * Cleanflight is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cleanflight is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cleanflight.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <stdint.h>
#include <stdbool.h>

#include "common/time.h"

typedef enum FlightLogFieldCondition {
    FLIGHT_LOG_FIELD_CONDITION_ALWAYS = 0,
    FLIGHT_LOG_FIELD_CONDITION_MOTORS,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_MOTORS_1,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_MOTORS_2,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_MOTORS_3,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_MOTORS_4,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_MOTORS_5,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_MOTORS_6,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_MOTORS_7,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_MOTORS_8,

    FLIGHT_LOG_FIELD_CONDITION_SERVOS,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_1,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_2,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_3,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_4,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_5,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_6,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_7,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_8,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_9,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_10,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_11,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_12,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_13,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_14,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_15,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_16,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_17,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_18,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_19,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_20,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_21,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_22,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_23,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_24,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_25,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_26,
    /*
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_27,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_28,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_29,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_30,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_31,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_32,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_33,
    FLIGHT_LOG_FIELD_CONDITION_AT_LEAST_SERVOS_34,
    */

    FLIGHT_LOG_FIELD_CONDITION_MAG,
    FLIGHT_LOG_FIELD_CONDITION_BARO,
    FLIGHT_LOG_FIELD_CONDITION_PITOT,
    FLIGHT_LOG_FIELD_CONDITION_VBAT,
    FLIGHT_LOG_FIELD_CONDITION_AMPERAGE,
    FLIGHT_LOG_FIELD_CONDITION_SURFACE,
    FLIGHT_LOG_FIELD_CONDITION_FIXED_WING_NAV,
    FLIGHT_LOG_FIELD_CONDITION_MC_NAV,
    FLIGHT_LOG_FIELD_CONDITION_RSSI,

    FLIGHT_LOG_FIELD_CONDITION_NONZERO_PID_D_0,
    FLIGHT_LOG_FIELD_CONDITION_NONZERO_PID_D_1,
    FLIGHT_LOG_FIELD_CONDITION_NONZERO_PID_D_2,

    FLIGHT_LOG_FIELD_CONDITION_NOT_LOGGING_EVERY_FRAME,

    FLIGHT_LOG_FIELD_CONDITION_DEBUG,

    FLIGHT_LOG_FIELD_CONDITION_NAV_ACC,
    FLIGHT_LOG_FIELD_CONDITION_NAV_POS,
    FLIGHT_LOG_FIELD_CONDITION_NAV_PID,
    FLIGHT_LOG_FIELD_CONDITION_ACC,
    FLIGHT_LOG_FIELD_CONDITION_ATTITUDE,
    FLIGHT_LOG_FIELD_CONDITION_RC_DATA,
    FLIGHT_LOG_FIELD_CONDITION_RC_COMMAND,
    FLIGHT_LOG_FIELD_CONDITION_GYRO_RAW,

    FLIGHT_LOG_FIELD_CONDITION_GYRO_PEAKS_ROLL,
    FLIGHT_LOG_FIELD_CONDITION_GYRO_PEAKS_PITCH,
    FLIGHT_LOG_FIELD_CONDITION_GYRO_PEAKS_YAW,

    FLIGHT_LOG_FIELD_CONDITION_NEVER,

    FLIGHT_LOG_FIELD_CONDITION_FIRST = FLIGHT_LOG_FIELD_CONDITION_ALWAYS,
    FLIGHT_LOG_FIELD_CONDITION_LAST = FLIGHT_LOG_FIELD_CONDITION_NEVER
} FlightLogFieldCondition;

typedef enum FlightLogFieldPredictor {
    //No prediction:
    FLIGHT_LOG_FIELD_PREDICTOR_0              = 0,

    //Predict that the field is the same as last frame:
    FLIGHT_LOG_FIELD_PREDICTOR_PREVIOUS       = 1,

    //Predict that the slope between this field and the previous item is the same as that between the past two history items:
    FLIGHT_LOG_FIELD_PREDICTOR_STRAIGHT_LINE  = 2,

    //Predict that this field is the same as the average of the last two history items:
    FLIGHT_LOG_FIELD_PREDICTOR_AVERAGE_2      = 3,

    //Predict that this field is minthrottle
    FLIGHT_LOG_FIELD_PREDICTOR_MINTHROTTLE    = 4,

    //Predict that this field is the same as motor 0
    FLIGHT_LOG_FIELD_PREDICTOR_MOTOR_0        = 5,

    //This field always increments
    FLIGHT_LOG_FIELD_PREDICTOR_INC            = 6,

    //Predict this GPS co-ordinate is the GPS home co-ordinate (or no prediction if that coordinate is not set)
    FLIGHT_LOG_FIELD_PREDICTOR_HOME_COORD     = 7,

    //Predict 1500
    FLIGHT_LOG_FIELD_PREDICTOR_1500           = 8,

    //Predict vbatref, the reference ADC level stored in the header
    FLIGHT_LOG_FIELD_PREDICTOR_VBATREF        = 9,

    //Predict the last time value written in the main stream
    FLIGHT_LOG_FIELD_PREDICTOR_LAST_MAIN_FRAME_TIME = 10

} FlightLogFieldPredictor;

typedef enum FlightLogFieldEncoding {
    FLIGHT_LOG_FIELD_ENCODING_SIGNED_VB       = 0, // Signed variable-byte
    FLIGHT_LOG_FIELD_ENCODING_UNSIGNED_VB     = 1, // Unsigned variable-byte
    FLIGHT_LOG_FIELD_ENCODING_NEG_14BIT       = 3, // Unsigned variable-byte but we negate the value before storing, value is 14 bits
    FLIGHT_LOG_FIELD_ENCODING_TAG8_8SVB       = 6,
    FLIGHT_LOG_FIELD_ENCODING_TAG2_3S32       = 7,
    FLIGHT_LOG_FIELD_ENCODING_TAG8_4S16       = 8,
    FLIGHT_LOG_FIELD_ENCODING_NULL            = 9 // Nothing is written to the file, take value to be zero
} FlightLogFieldEncoding;

typedef enum FlightLogFieldSign {
    FLIGHT_LOG_FIELD_UNSIGNED = 0,
    FLIGHT_LOG_FIELD_SIGNED   = 1
} FlightLogFieldSign;

typedef enum FlightLogEvent {
    FLIGHT_LOG_EVENT_SYNC_BEEP = 0,
    FLIGHT_LOG_EVENT_INFLIGHT_ADJUSTMENT = 13,
    FLIGHT_LOG_EVENT_LOGGING_RESUME = 14,
    FLIGHT_LOG_EVENT_FLIGHTMODE = 30, // Add new event type for flight mode status.
    FLIGHT_LOG_EVENT_IMU_FAILURE = 40,
    FLIGHT_LOG_EVENT_LOG_END = 255
} FlightLogEvent;

typedef struct flightLogEvent_syncBeep_s {
    uint32_t time;
} flightLogEvent_syncBeep_t;

typedef struct flightLogEvent_flightMode_s { // New Event Data type
    uint32_t flags;
    uint32_t lastFlags;
} flightLogEvent_flightMode_t;

typedef struct flightLogEvent_inflightAdjustment_s {
    uint8_t adjustmentFunction;
    bool floatFlag;
    int32_t newValue;
    float newFloatValue;
} flightLogEvent_inflightAdjustment_t;

typedef struct flightLogEvent_loggingResume_s {
    uint32_t logIteration;
    timeUs_t currentTimeUs;
} flightLogEvent_loggingResume_t;

typedef struct flightLogEvent_IMUError_s {
    uint32_t errorCode;
} flightLogEvent_IMUError_t;

#define FLIGHT_LOG_EVENT_INFLIGHT_ADJUSTMENT_FUNCTION_FLOAT_VALUE_FLAG 128

typedef union flightLogEventData_u {
    flightLogEvent_syncBeep_t syncBeep;
    flightLogEvent_flightMode_t flightMode; // New event data
    flightLogEvent_inflightAdjustment_t inflightAdjustment;
    flightLogEvent_loggingResume_t loggingResume;
    flightLogEvent_IMUError_t imuError;
} flightLogEventData_t;

typedef struct flightLogEvent_s {
    FlightLogEvent event;
    flightLogEventData_t data;
} flightLogEvent_t;
