#nullable enable

using System;
using System.Collections.Generic;
using RimWorld;
using Verse;

namespace rjw
{
	/// <summary>
	/// Affects stats based on the pawn's body type.
	/// </summary>
	public class StatPart_BodyType : StatPart
	{
		public enum ValueMode { Factor, Offset }

		/// <summary>
		/// Whether to apply the values as factors or offsets.
		/// </summary>
		public ValueMode mode = ValueMode.Factor;

		/// <summary>
		/// The dictionary of body-types to values.
		/// </summary>
		public Dictionary<BodyTypeDef, float> bodyToValue = new();

		public override void TransformValue(StatRequest req, ref float val)
		{
			if (!req.HasThing) return;
			if (req.Thing is not Pawn pawn) return;
			if (pawn.story?.bodyType is not { } bodyType) return;
			if (!bodyToValue.TryGetValue(bodyType, out var value)) return;

			val = mode switch
			{
				ValueMode.Factor => val * value,
				ValueMode.Offset => val + value,
				_ => throw new InvalidOperationException("bad value for `mode`")
			};
		}

		public override string? ExplanationPart(StatRequest req)
		{
			if (!req.HasThing) return null;
			if (req.Thing is not Pawn pawn) return null;
			if (pawn.story?.bodyType is not { } bodyType) return null;
			if (!bodyToValue.TryGetValue(bodyType, out var value)) return null;

			// These do not affect the value, so no explanation needed.
			if (mode == ValueMode.Factor && value == 1f) return null;
			if (mode == ValueMode.Offset && value == 0f) return null;

			var numSense = mode switch
			{
				ValueMode.Factor => ToStringNumberSense.Factor,
				ValueMode.Offset => ToStringNumberSense.Offset,
				_ => throw new InvalidOperationException("bad value for `mode`")
			};

			var label = "StatPart_BodyType".Translate();
			var text = value.ToStringByStyle(ToStringStyle.FloatMaxThree, numSense);
			return $"{label}: {text}";
		}
	}
}