namespace rjw.Modules.Attraction.StandardPreferences
{
	/// <summary>
	/// <para>Handles factions between two pawns.</para>
	/// <para>This is also responsible for adding a bias to animals to breed
	/// within their own faction.</para>
	/// </summary>
	public sealed class S_Faction : PredicatedAttractionPreference
	{
		[StandardPreference]
		public static void ApplyTo(ref AttractionRequest request)
		{
			// Observer must have a faction.
			if (request.Pawn.Faction is null) return;

			request.SetPreference(new S_Faction());
		}

		private S_Faction() : base(
			AttractionMode.Social,
			nameof(S_Faction),
			FactorOperation.Add,
			HasDifferingFaction,
			whenTrue: WithDifferingFaction,
			whenFalse: WithSameFaction)
		{ }

		private static bool HasDifferingFaction(ref AttractionRequest request)
		{
			// Should not happen, but if something adds this preference
			// incorrectly, handle it by treating them as different to everything.
			if (request.Pawn.Faction is not { } pFaction) return true;
			if (request.Target.Faction is not { } tFaction) return true;
			// Special case for two non-humanlike factions.  Treat them as the same.
			if (!pFaction.def.humanlikeFaction && !tFaction.def.humanlikeFaction) return false;
			return pFaction != tFaction;
		}

		private static float WithDifferingFaction(ref AttractionRequest request, float factor)
		{
			var target = request.Target;
			
			// Unless raping, pawns don't generally care about visitors.
			// TODO: Unless their ideology suggests otherwise, I guess.
			if (!request.Purpose.IsForRape()) return 0f;

			// Animals don't consider factions when raping.
			if (request.Category.AnimalObserver()) return 0f;
			// Human rapists have no consideration for factionless animals.
			if (request.Category.AnimalTarget() && target.Faction is null) return 0f;

			return 0.2f;
		}
		
		// Bias rapists toward raping pawns of other factions.
		private static float WithSameFaction(ref AttractionRequest request, float factor)
		{
			// Small bias towards pawns being attracted to their faction members more.
			if (!request.Purpose.IsForRape()) return 0.1f;

			// Animals don't consider factions when raping.
			if (request.Category.AnimalObserver()) return 0f;

			return -0.2f;
		}
	}
}