using Verse;
using RimWorld;

namespace rjw.Modules.Attraction.StandardPreferences
{
	/// <summary>
	/// Handles the pawn's romantic relationships.
	/// </summary>
	public sealed class R_Relationship : PredicatedAttractionPreference
	{
		const float MinimumCheatFactor = 0.25f;

		[StandardPreference]
		public static void ApplyTo(ref AttractionRequest request)
		{
			// Disabled by certain settings.
			if (RJWSettings.WildMode || RJWSettings.HippieMode) return;
			// Psychopaths don't care about the nuances of romantic relationships.
			if (xxx.is_psychopath(request.Pawn)) return;

			request.SetPreference(new R_Relationship());
		}

		private R_Relationship() : base(
			AttractionMode.Romantic,
			nameof(R_Relationship),
			FactorOperation.Multiply,
			IsInRelationship,
			whenTrue: WhenInRelationship,
			whenFalse: WhenNotInRelationship)
		{ }

		private static bool IsInRelationship(ref AttractionRequest request) =>
			request.HasLoveRelationWithTarget;

		private static float WhenInRelationship(ref AttractionRequest request, float factor)
		{
			var pawn = request.Pawn;
			if (request.Purpose.IsForRape() && pawn.story?.traits is { } traits)
			{
				return request.Target.gender switch
				{
					// Let's check for spousal abuse traits.
					Gender.Male when traits.HasTrait(TraitDefOf.DislikesMen) => 1.5f,
					Gender.Female when traits.HasTrait(TraitDefOf.DislikesWomen) => 1.5f,
					// Abraisive pawns more likely to rub someone in a wrong way (and place).
					_ when traits.HasTrait(TraitDefOf.Abrasive) => 1f,
					// Otherwise, the pawn is more likely to give their angry-love to someone else.
					_ => 0.5f
				};
			}
			return 1.5f;
		}

		private static float WhenNotInRelationship(ref AttractionRequest request, float factor)
		{
			var pawn = request.Pawn;
			// At the moment, animals do not have any special handling here,
			// but who knows about the future.
			if (pawn.IsAnimal()) return 1f;
			// Pawns can always gawk and social consequences are disregarded for rape.
			if (request.Purpose is not AttractionPurpose.ForFucking) return 1f;
			// The pawn might be okay with random sex with another.
			if (AttractionUtility.CanFoolAround(ref request)) return 1f;

			// Nymphos can be configured to cheat when sexually frustrated.
			if (RJWHookupSettings.NymphosCanCheat)
				if (xxx.is_nympho(pawn) && xxx.is_frustrated(pawn))
					return 1f;

			// If the pawn is drunk, they may cheat based on the severity.
			if (pawn.HasBeerGoggles(out var severity))
				return GenMath.LerpDouble(0f, 1f, MinimumCheatFactor, 1f, severity);

			// Otherwise, apply a significant penalty.
			return MinimumCheatFactor;
		}
	}
}