using RimWorld;

namespace rjw.Modules.Attraction.StandardPreferences
{
	/// <summary>
	/// <para>Handles an animal versus their natural predators.</para>
	/// <para>Prey avoid mating with their natural predators, but predators are
	/// happy to fuck potential prey.  Also, if two predators would prey on each
	/// other, this cancels out.</para>
	/// </summary>
	public sealed class P_AnimalToPredator : PredicatedAttractionPreference
	{
		[StandardPreference]
		public static void ApplyTo(ref AttractionRequest request)
		{
			// Request must be between animals.
			if (request.Category is not AttractionCategory.BetweenAnimals) return;
			// Target must be a predator.
			if (!request.Target.RaceProps.predator) return;
			// Target must not be the same species.
			if (request.Target.IsSameSpecies(request.Pawn)) return;
			// Ignored when both animals are tame; they should be used to living
			// side by side and can identify other tamed animals.
			if (request.Pawn.IsTameAnimal() && request.Target.IsTameAnimal()) return;

			request.SetPreference(new P_AnimalToPredator());
		}

		private P_AnimalToPredator() : base(
			AttractionMode.Physical,
			nameof(P_AnimalToPredator),
			FactorOperation.Multiply,
			IsPotentialPrey,
			whenTrue: FixedValue(0.4f))
		{ }

		private static bool IsPotentialPrey(ref AttractionRequest request)
		{
			var pawn = request.Pawn;
			var target = request.Target;
			var pPred = pawn.RaceProps.predator && FoodUtility.IsAcceptablePreyFor(pawn, target);
			var tPred = target.RaceProps.predator && FoodUtility.IsAcceptablePreyFor(target, pawn);
			return pPred != tPred && tPred;
		}
	}
}