using System;

namespace rjw.Modules.Attraction
{
	/// <summary>
	/// <para>How an interaction between the two pawns of an attraction request
	/// is categorized.</para>
	/// <para>The use of "human" here is a bit of a misnomer; it will include pawns
	/// of tool-user intelligence, but RJW does not really make a differentiation.</para>
	/// <para>Test to see if a pawn is humanlike if it is important.</para>
	/// </summary>
	public enum AttractionCategory
	{
		/// <summary>
		/// Both pawns are humanlike (or tool-users).
		/// </summary>
		BetweenHumans,
		/// <summary>
		/// Both pawns are animals.
		/// </summary>
		BetweenAnimals,
		/// <summary>
		/// The observing pawn is an animal but the target is a human.
		/// </summary>
		AnimalToHuman,
		/// <summary>
		/// The observing pawn is a human but the target is an animal.
		/// </summary>
		HumanToAnimal
	}

	public static class AttractionCategoryExtensions
	{
		/// <summary>
		/// Convenience extension to determine if the category involves bestiality.
		/// </summary>
		/// <param name="category">The category to inspect.</param>
		/// <returns>Whether the category implies bestiality.</returns>
		public static bool IsBestiality(this AttractionCategory category) =>
			category is AttractionCategory.AnimalToHuman or AttractionCategory.HumanToAnimal;

		/// <summary>
		/// Convenience extension to determine if the category involves animals.
		/// </summary>
		/// <param name="category">The category to inspect.</param>
		/// <returns>Whether the category involves animals.</returns>
		public static bool InvolvesAnimals(this AttractionCategory category) =>
			category is not AttractionCategory.BetweenHumans;

		/// <summary>
		/// Convenience extension to determine if the category has an animal observer.
		/// </summary>
		/// <param name="category">The category to inspect.</param>
		/// <returns>Whether the observer is an animal.</returns>
		public static bool AnimalObserver(this AttractionCategory category) =>
			category is AttractionCategory.BetweenAnimals or AttractionCategory.AnimalToHuman;

		/// <summary>
		/// Convenience extension to determine if the category has an animal target.
		/// </summary>
		/// <param name="category">The category to inspect.</param>
		/// <returns>Whether the target is an animal.</returns>
		public static bool AnimalTarget(this AttractionCategory category) =>
			category is AttractionCategory.BetweenAnimals or AttractionCategory.HumanToAnimal;
	}
}