﻿using RimWorld;
using System.Collections.Generic;
using System.Linq;
using Verse;
using Multiplayer.API;

namespace rjw
{
	/// <summary>
	/// Generator of RMB categories for socializing
	/// </summary>
	static class RMB_Socialize
	{
		/// <summary>
		/// Create socialize option
		/// </summary>
		/// <param name="pawn">Selected pawn</param>
		/// <param name="target">Target of right click</param>
		public static IEnumerable<FloatMenuOption> GetOptions(Pawn pawn, Pawn target)
		{
			AcceptanceReport canCreateEntries = DoBasicChecks(pawn, target);

			if (!canCreateEntries)
			{
				if (RJWSettings.DevMode && !canCreateEntries.Reason.NullOrEmpty())
					yield return new FloatMenuOption(canCreateEntries.Reason, null);

				yield break;
			}

			FloatMenuOption opt = GenerateCategoryOption(pawn, target);
			if (opt != null)
				yield return opt;
		}

		/// <summary>
		/// Check for the things that should be obvious to the player or does not change for particular pawns.
		/// </summary>
		/// <returns>
		/// AcceptanceReport. Reason is an untranslated string and should only be shown in the DevMode
		/// </returns>
		private static AcceptanceReport DoBasicChecks(Pawn pawn, Pawn target)
		{
			if (!pawn.interactions.CanInteractNowWith(target))
			{
				return $"No socialize: cannot interact with {target.NameShortColored} now";
			}

			return true;
		}

		private static FloatMenuOption GenerateCategoryOption(Pawn pawn, Pawn target)
		{
			return FloatMenuUtility.DecoratePrioritizedTask(new FloatMenuOption("RJW_RMB_Social".Translate(target), delegate ()
			{
				FloatMenuUtility.MakeMenu(GenerateSocialOptions(pawn, target).Where(x => x.action != null), (FloatMenuOption opt) => opt.Label, (FloatMenuOption opt) => opt.action);
			}, MenuOptionPriority.High), pawn, target);
		}

		/// <summary>
		/// Generates sub-menu options for socializing
		/// </summary>
		/// <param name="pawn">Initiator pawn</param>
		/// <param name="target">Target pawn</param>
		/// <returns>A list of menu options, where each item represents a possible interaction</returns>
		private static IEnumerable<FloatMenuOption> GenerateSocialOptions(Pawn pawn, LocalTargetInfo target)
		{
			FloatMenuOption option = new("RJW_RMB_Insult".Translate(), delegate ()
			{
				Socialize(pawn, target, InteractionDefOf.Insult);
			}, MenuOptionPriority.High);
			yield return FloatMenuUtility.DecoratePrioritizedTask(option, pawn, target);

			if (!pawn.HostileTo(target.Pawn))
			{
				option = new FloatMenuOption("RJW_RMB_SocialFight".Translate(), delegate ()
				{
					SocializeFight(pawn, target);
				}, MenuOptionPriority.High);
				yield return FloatMenuUtility.DecoratePrioritizedTask(option, pawn, target);

				option = new FloatMenuOption("RJW_RMB_Chat".Translate(), delegate ()
				{
					Socialize(pawn, target, InteractionDefOf.Chitchat);
				}, MenuOptionPriority.High);
				yield return FloatMenuUtility.DecoratePrioritizedTask(option, pawn, target);

				// OP shit +12 relations, enable in cheat menu
				if (RJWSettings.Allow_RMB_DeepTalk)
				{
					option = new FloatMenuOption("RJW_RMB_DeepTalk".Translate(), delegate ()
					{
						pawn.interactions.TryInteractWith(target.Pawn, InteractionDefOf.DeepTalk);
					}, MenuOptionPriority.High);
					yield return FloatMenuUtility.DecoratePrioritizedTask(option, pawn, target);
				}

				if (!LovePartnerRelationUtility.LovePartnerRelationExists(pawn, target.Pawn))
				{
					option = new FloatMenuOption("RJW_RMB_RomanceAttempt".Translate(), delegate ()
					{
						Socialize(pawn, target, InteractionDefOf.RomanceAttempt);
					}, MenuOptionPriority.High);
					yield return FloatMenuUtility.DecoratePrioritizedTask(option, pawn, target);
				}

				if (pawn.relations.DirectRelationExists(PawnRelationDefOf.Lover, target.Pawn) || pawn.relations.DirectRelationExists(PawnRelationDefOf.Fiance, target.Pawn))
				{
					option = new FloatMenuOption("RJW_RMB_MarriageProposal".Translate(), delegate ()
					{
						Socialize(pawn, target, InteractionDefOf.MarriageProposal);
					}, MenuOptionPriority.High);
					yield return FloatMenuUtility.DecoratePrioritizedTask(option, pawn, target);
				}
			}
		}

		//multiplayer synch actions

		/// <summary>
		/// Stops current job of the <paramref name="pawn"/> and starts a social fight
		/// </summary>
		/// <param name="pawn">Initiator</param>
		/// <param name="target">Target pawn</param>
		[SyncMethod]
		static void SocializeFight(Pawn pawn, LocalTargetInfo target)
		{
			pawn.interactions.StartSocialFight(target.Pawn);
		}

		/// <summary>
		/// Stops current job of the <paramref name="pawn"/> and starts an interaction
		/// </summary>
		/// <param name="pawn">Initiator</param>
		/// <param name="target">Target pawn</param>
		/// <param name="interaction">Interaction def</param>
		[SyncMethod]
		static void Socialize(Pawn pawn, LocalTargetInfo target, InteractionDef interaction)
		{
			pawn.interactions.TryInteractWith(target.Pawn, interaction);
		}
	}
}