﻿using Verse;
using Verse.AI;

namespace rjw
{
	public static class Pather_Utility
	{
		/// <summary>
		/// Checks the distance from the pawn to a target, using the casual sex
		/// distance configuration.  This check is relatively quick, ignoring walls.
		/// </summary>
		/// <param name="pawn">The pawn acting as the origin position.</param>
		/// <param name="target">The target position.</param>
		/// <returns>Whether the target is close enough.</returns>
		public static bool cells_to_target_casual(Pawn pawn, LocalTargetInfo target)
		{
			if (!target.IsValid) return false;
			return pawn.Position.DistanceTo(target.Cell) < RJWSettings.maxDistanceCellsCasual;
		}

		/// <summary>
		/// Checks the distance from the pawn to a target, using the rape
		/// distance configuration.  This check is relatively quick, ignoring walls.
		/// </summary>
		/// <param name="pawn">The pawn acting as the origin position.</param>
		/// <param name="target">The target position.</param>
		/// <returns>Whether the target is close enough.</returns>
		public static bool cells_to_target_rape(Pawn pawn, LocalTargetInfo target)
		{
			if (!target.IsValid) return false;
			return pawn.Position.DistanceTo(target.Cell) < RJWSettings.maxDistanceCellsRape;
		}

		/// <summary>
		/// Checks if a pawn can reach a target.  This actually checks pathing, so
		/// is rather expensive.  Try to make this your last check when evaluating
		/// multiple pawns.
		/// </summary>
		/// <param name="pawn">The pawn acting as the origin position.</param>
		/// <param name="target">The target position.</param>
		/// <param name="maxDanger">The maximum danger allowed on the path.</param>
		/// <param name="peMode">What end goal to use for the pathfinder.</param>
		/// <returns>Whether the target is reachable.</returns>
		public static bool can_path_to_target(
			Pawn pawn,
			LocalTargetInfo target,
			Danger maxDanger = Danger.Some,
			PathEndMode peMode = PathEndMode.Touch)
		{
			if (!target.IsValid) return false;

			var startPos = pawn.Position;
			var traversalParams = TraverseParms.For(pawn, maxDanger);

			// If unlimited, just let RimWorld optimize this using its usual
			// reachablility checks.
			if (RJWSettings.maxDistancePathCost <= 0)
				return pawn.Map.reachability.CanReach(startPos, target, peMode, traversalParams);

			// Otherwise, we will need to actually check the cost of the full path.
			using var pawnPath = pawn.Map.pathFinder.FindPathNow(startPos, target, traversalParams, peMode: peMode);
			return pawnPath.TotalCost <= RJWSettings.maxDistancePathCost;
		}
	}
}