//#define TESTMODE // Uncomment to enable logging.

using Multiplayer.API;
using RimWorld;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Verse;
using Verse.AI;
using static rjw.CasualSex_Helper;

namespace rjw
{
	/// <summary>
	/// Helper for sex with animals
	/// </summary>
	public class BreederHelper
	{
		public const int max_animals_at_once = 1; // lets not forget that the purpose is procreation, not sodomy

		const float MinimumAttraction = 0.1f;

		[Conditional("TESTMODE")]
		private static void DebugText(string msg)
		{
			ModLog.Message(msg);
		}

		public static bool IsBreedableFor(Pawn breeder, Pawn breedee)
		{
			// Basic eligibility checks.
			if (breedee == breeder) return false;
			if (breedee.Map != breeder.Map) return false;
			if (breedee.Suspended) return false;
			if (breedee.IsForbidden(breeder)) return false;
			if (breedee.HostileTo(breeder)) return false;
			if (!xxx.is_not_dying(breedee)) return false;
			if (!xxx.can_get_raped(breedee)) return false;
			if (!Pather_Utility.cells_to_target_rape(breeder, breedee)) return false;
			if (!breeder.CanReserve(breedee, max_animals_at_once)) return false;

			// Make sure the pairing follows user configuration.
			if (!xxx.can_do_animalsex(breeder, breedee)) return false;

			return true;
		}

		public static IEnumerable<Pawn> DesignatedBreedeesFor(Pawn animal)
		{
			DebugText($" Count of designated breeders: {DesignatorsData.rjwBreeding.Count}");

			if (!animal.IsAnimal())
			{
				DebugText($"BreederHelper::find_designated_breeder( {xxx.get_pawnname(animal)} ) was not an animal");
				yield break;
			}

			foreach (var breedee in DesignatorsData.rjwBreeding)
			{
				// Basic eligibility checks.
				if (!IsBreedableFor(animal, breedee)) continue;

				// Animals that are already pregnant are passed over.
				// However, pregnant humans will continue to be bred because FUN.
				if (breedee.IsPregnant() && breedee.IsAnimal()) continue;

				yield return breedee;
			}
		}

		/// <summary>
		/// For an animal, finds a suitable designated pawn to breed.
		/// </summary>
		/// <param name="pawn">The breeding pawn.</param>
		/// <returns>The selected breedee, or null if none found.</returns>
		public static Pawn FindDesignatedBreeder(Pawn pawn)
		{
			DebugText($"BreederHelper::FindDesignatedBreeder( {xxx.get_pawnname(pawn)} ) called");
			DebugText($" Pawn faction: {pawn.Faction}");
			
			var targets = DesignatedBreedeesFor(pawn);

			DebugText($" Initial target count: {targets.Count()}");

			// TODO for consideration --
			// Currently, designated breeding is always treated as rape.  It feels
			// like this should be more nuanced.  A better approach might be:
			// - Animal-on-animal is always consensual.
			// - Animal-on-human is consensual when the human is attracted to the animal
			//   (attraction grater than 1).  This would mostly only be the case for
			//   zoophiles, but would essentially leave it to the attraction system to
			//   make the determination.
			//   - An interaction here might be that you can just get the designated
			//     breedee drunk enough to accept being a breeder until zoophile kicks in.
			// - And otherwise, it is treated as rape.
			//
			// The actual job would need to also stick to these rules and make sure
			// that zoophiles don't suffer from "breakage" when they're actually into it.

			var ignoreOrientation = Genital_Helper.has_male_bits(pawn) || xxx.is_insect(pawn);
			foreach (var result in SexAppraiser.FindVictims(pawn, targets, true, ignoreOrientation))
			{
				var target = result.Target;
				DebugText($" Checking target: {xxx.get_pawnname(target)}");

				if (result.ObserverAttraction < MinimumAttraction) continue;
				if (!Pather_Utility.can_path_to_target(pawn, target)) continue;

				DebugText($" Using target: {xxx.get_pawnname(target)}");
				DebugText($" Race of target: {target.kindDef.race.defName.ToLower()}");
				return target;
			}

			DebugText(" Found no available targets");
			return null;
		}

		public static IEnumerable<Pawn> AnimalBreedeesFor(Pawn pawn)
		{
			DebugText($" Count of map pawns: {pawn.Map.mapPawns.AllPawnsSpawned.Count}");

			foreach (var breedee in pawn.Map.mapPawns.AllPawnsSpawned)
			{
				// Basic eligibility checks.
				if (!breedee.IsAnimal()) continue;
				if (!IsBreedableFor(pawn, breedee)) continue;

				yield return breedee;
			}
		}

		/// <summary>
		/// <para>For an animal or human, finds a suitable animal to breed.</para>
		/// <para>The target animal may or may not be designated for breeding.</para>
		/// </summary>
		/// <param name="pawn">The breeding pawn.</param>
		/// <returns>The selected breedee, or null if none found.</returns>
		[SyncMethod]
		public static Pawn FindAnimalToBreed(Pawn pawn)
		{
			DebugText("BreederHelper::FindAnimalToBreed( " + xxx.get_pawnname(pawn) + " ) called");

			var maxDanger = Danger.None;

			// We will be pruning this initial pawn list.
			var targets = AnimalBreedeesFor(pawn);

			DebugText($" Initial target count: {targets.Count()}");

			// Just skip to the failure case if we already have nothing.
			if (!targets.Any()) goto notFound;

			var parts = pawn.GetGenitalsList();
			if (!Genital_Helper.has_male_bits(pawn, parts) && (Genital_Helper.has_vagina(pawn, parts) || Genital_Helper.has_anus(pawn)))
			{
				// The pawn can only be penetrated, so find one that is able.
				targets = targets.Where(xxx.can_fuck);
			}

			if (xxx.need_some_sex(pawn) >= 3.0f)
			{
				// If horny, allow targets that may be outside of safe temperatures.
				maxDanger = Danger.Some;
			}

			if (pawn.IsAnimal())
			{
				// Animals will only go for interspecies targets they can see.
				// Same-species targets can still find each other across distances
				// helping low animal populations to find each other.
				targets = targets.Where(x => pawn.IsSameSpecies(x) || pawn.CanSee(x));

				// Animals always ignore temperature danger.
				maxDanger = Danger.Some;
			}

			DebugText($" Filtered target count: {targets.Count()}");

			foreach (var result in SexAppraiser.FindPartners(pawn, targets))
			{
				var target = result.Target;
				DebugText($" Checking target: {xxx.get_pawnname(target)}");

				if (result.ObserverAttraction < MinimumAttraction) continue;
				if (!Pather_Utility.can_path_to_target(pawn, target, maxDanger)) continue;
				if (GetCloseness(pawn, target) is QuickieCloseness.OutOfRange) continue;

				DebugText($" Using target: {xxx.get_pawnname(target)}");
				DebugText($" Race of target: {target.kindDef.race.defName.ToLower()}");
				return target;
			}

		notFound:
			DebugText(" Found no available targets");
			return null;
		}
	}
}