using System.Linq;
using RimWorld;
using Verse;

namespace rjw
{
	/// <summary>
	/// <para>Stat worker that presents the calculated value as its inverse.</para>
	/// <para>So, if the calculated value is `2`, it will produce `1/2`.</para>
	/// <para>Best used with stats with a minimum value of `1`.</para>
	/// <para>Negative values are not supported.  If the value ever turns out
	/// to be zero or less, it will use the stat's maximum value as its inverse.</para>
	/// </summary>
	public class StatWorker_Inverse : StatWorker
	{
		public override void FinalizeValue(StatRequest req, ref float val, bool applyPostProcess)
		{
			base.FinalizeValue(req, ref val, applyPostProcess);
			val = DoInvert(val);
		}

		public override string GetExplanationFinalizePart(StatRequest req, ToStringNumberSense numberSense, float finalVal)
		{
			// Wish they would just yield the parts in a generator...
			var baseText = base.GetExplanationFinalizePart(req, numberSense, finalVal);
			var oldLines = baseText.Split('\n');
			
			// A bit confusing: `finalVal` is already inverted, so we have to un-invert it
			// to recover the pre-inverted value.
			var preInvertText = stat.ValueToString(DoInvert(finalVal), stat.toStringNumberSense, true);
			var postInvertText = stat.ValueToString(finalVal, stat.toStringNumberSense, true);
			var label = "StatWorker_Invert".Translate(preInvertText);

			var newLines = oldLines.Take(-1)
				.Append($"{label}: {postInvertText}")
				.Concat(oldLines.TakeLast(1));
			return string.Join("\n", newLines);
		}

		private float DoInvert(float val)
		{
			if (val >= stat.maxValue) return 0f;
			if (val <= 0f) return stat.maxValue;
			return 1f / val;
		}
	}
}