#nullable enable

using System;
using System.Collections.Generic;
using System.Linq;

namespace rjw.Modules.Shared.Extensions
{
	public static class EnumerableExtensions
	{
		/// <summary>
		/// <para>A function that may or may not apply to a value of `T`.</para>
		/// <para>The `output` is considered to only have meaning if it returns
		/// `true`; it is undefined behavior to use this value otherwise.</para>
		/// </summary>
		/// <typeparam name="T">The input type.</typeparam>
		/// <typeparam name="U">The output type.</typeparam>
		/// <param name="val">The input value.</param>
		/// <param name="output">A variable to store the output value.</param>
		/// <returns>Whether the function was applicable to `val`.</returns>
		public delegate bool PartialFunction<T, U>(T val, out U output);

		/// <summary>
		/// Filters to distinct values using a `keySelector` function.
		/// </summary>
		/// <typeparam name="TSource">The type of the value.</typeparam>
		/// <typeparam name="TKey">The type of the key.</typeparam>
		/// <param name="source">The source enumerable.</param>
		/// <param name="keySelector">The key selector.</param>
		/// <returns>The filtered enumerable.</returns>
		public static IEnumerable<TSource> DistinctBy<TSource, TKey>(
			this IEnumerable<TSource> source,
			Func<TSource, TKey> keySelector)
		{
			HashSet<TKey> seenKeys = new();
			foreach (TSource element in source)
				if (seenKeys.Add(keySelector(element)))
					yield return element;
		}

		/// <summary>
		/// <para>LINQ's `First` method, but using the try pattern.</para>
		/// <para>If the sequence was empty, `result` will be the default value of `T`.</para>
		/// </summary>
		/// <typeparam name="T">Any type.</typeparam>
		/// <param name="source">The source enumerable.</param>
		/// <param name="margin">A variable to store the result into.</param>
		/// <returns>Whether or not an element was found.</returns>
		public static bool TryFirst<T>(this IEnumerable<T> source, out T result)
		{
			foreach (var item in source)
			{
				result = item;
				return true;
			}

			// Empty enumerable, so use the default.
			result = default!;
			return false;
		}

		/// <summary>
		/// <para>Gets items of the enumerable with their index as a tuple.</para>
		/// <para>This works for enumerables of any kind (even those that are not
		/// indexable) and the index represents the order it was yielded.</para>
		/// </summary>
		/// <typeparam name="T">Any type.</typeparam>
		/// <param name="self">The enumerable.</param>
		/// <returns>An enumerable of tuples, with the item and index.</returns>
		public static IEnumerable<(T item, int index)> WithIndex<T>(this IEnumerable<T> self) =>
			self.Select((item, index) => (item, index));

		/// <summary>
		/// <para>A combination filter and map function, using a partial function.</para>
		/// </summary>
		/// <typeparam name="T">The input type, from the enumerable.</typeparam>
		/// <typeparam name="U">The output type.</typeparam>
		/// <param name="self">This enumerable instance.</param>
		/// <param name="fn">The partial function to apply.</param>
		/// <returns>A new enumerable where the function was applicable.</returns>
		public static IEnumerable<U> Collect<T, U>(this IEnumerable<T> self, PartialFunction<T, U> fn) {
			foreach (var val in self)
				if (fn(val, out var output))
					yield return output;
		}

		/// <summary>
		/// <para>Searches this enumerable for the first value that can be applied
		/// to the given partial function and provides the transformed value, if so.</para>
		/// </summary>
		/// <typeparam name="T">The input type, from the enumerable.</typeparam>
		/// <typeparam name="U">The output type.</typeparam>
		/// <param name="self">This enumerable instance.</param>
		/// <param name="fn">The partial function to apply.</param>
		/// <param name="result">A variable to place the result into.</param>
		/// <returns>If an applicable value was located and the result meaningful.</returns>
		public static bool TryCollectFirst<T, U>(this IEnumerable<T> self, PartialFunction<T, U> fn, out U result) {
			foreach (var val in self)
				if (fn(val, out result))
					return true;

			result = default!;
			return false;
		}
	}
}
