using System.Collections.Generic;
using Verse;
using RimWorld;

namespace rjw.Modules.Attraction.StandardPreferences
{
	/// <summary>
	/// <para>Handles body-size preference.  This is controlled by a pair of
	/// stats which are manipulated by various other systems.</para>
	/// <para>Check the XML data to see what all influences them.</para>
	/// </summary>
	public sealed class P_BodySize : CurveAttractionPreference
	{
		[StandardPreference]
		public static void ApplyTo(ref AttractionRequest request)
		{
			// Ignored for pawns with specialized rape mechanics.  We'll just say
			// their anatomy allows them to handle a pawn of any size.
			if (xxx.is_mechanoid(request.Pawn)) return;
			if (xxx.is_insect(request.Pawn)) return;

			var points = GetPoints(request.Pawn, request.Target);
			request.SetPreference(new P_BodySize(points));
		}

		private P_BodySize(IEnumerable<CurvePoint> points) : base(
			AttractionMode.Physical,
			nameof(P_BodySize),
			GetRelativeBodySize,
			points)
		{ }

		private static IEnumerable<CurvePoint> GetPoints(Pawn pawn, Pawn target)
		{
			var min = pawn.GetStatValueForPawn(RJWStatDefOf.SexMinSizePreference, target);
			var max = pawn.GetStatValueForPawn(RJWStatDefOf.SexMaxSizePreference, target);

			yield return new(min * 0.5f, 0.4f);
			yield return new(min, 1f);
			yield return new(max, 1f);
			yield return new(max * 2f, 0.4f);
		}

		private static float GetRelativeBodySize(ref AttractionRequest request)
		{
			var pSize = request.Pawn.BodySize;
			var tSize = request.Target.BodySize;

			// Mass needs to make sense.
			if (pSize <= 0f || tSize <= 0f) return 1f;

			// Values below 1.0 indicate the target is less massive than the observer.
			return tSize / pSize;
		}
	}
}